"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.invokeOpenApiGenerator = exports.NonClientGeneratorDirectory = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const fs = require("fs");
const path = require("path");
const util_1 = require("projen/lib/util");
/**
 * Enum for generator directories for non-client generators
 */
var NonClientGeneratorDirectory;
(function (NonClientGeneratorDirectory) {
    NonClientGeneratorDirectory["DOCS"] = "docs";
})(NonClientGeneratorDirectory = exports.NonClientGeneratorDirectory || (exports.NonClientGeneratorDirectory = {}));
const serializeProperties = (properties) => Object.entries(properties)
    .map(([key, value]) => `${key}=${value}`)
    .join(",");
/**
 * Clean up any files that have already been generated by the openapi generator
 * @param generatedOutputPath output path for the generator
 */
const cleanPreviouslyGeneratedFiles = (generatedOutputPath) => {
    // OpenAPI generator writes a manifest called FILES which lists the files it generated.
    const openApiGeneratedFilesManifestPath = path.join(generatedOutputPath, ".openapi-generator", "FILES");
    // If the manifest exists, delete the files it lists
    if (fs.existsSync(openApiGeneratedFilesManifestPath)) {
        const previouslyGeneratedFiles = new Set(fs
            .readFileSync(openApiGeneratedFilesManifestPath, { encoding: "utf-8" })
            .split("\n")
            .filter((x) => x));
        previouslyGeneratedFiles.forEach((previouslyGeneratedFile) => {
            fs.unlinkSync(path.join(generatedOutputPath, previouslyGeneratedFile));
        });
    }
};
/**
 * Generate client code or docs by invoking the root generate script
 */
exports.invokeOpenApiGenerator = (options) => {
    // Prior to generating, clean up any previously generated files so that we don't have stale generated code from
    // previous executions.
    cleanPreviouslyGeneratedFiles(options.outputPath);
    const additionalProperties = options.additionalProperties
        ? ` --additional-properties "${serializeProperties(options.additionalProperties)}"`
        : "";
    util_1.exec(`./generate --generator ${options.generator} --spec-path ${options.specPath} --output-path ${options.outputPath} --generator-dir ${options.generatorDirectory}${additionalProperties}`, {
        cwd: path.resolve(__dirname, "..", "..", "..", "..", "scripts", "generators"),
    });
};
//# sourceMappingURL=data:application/json;base64,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