"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayWebAcl = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_wafv2_1 = require("aws-cdk-lib/aws-wafv2");
const constructs_1 = require("constructs");
/**
 * Associate an AWS WAF v2 Web ACL with the given api
 */
class OpenApiGatewayWebAcl extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const aclName = `${pdk_nag_1.PDKNag.getStackPrefix(aws_cdk_lib_1.Stack.of(this))
            .split("/")
            .join("-")}-${id}-WebAcl`;
        const ipSetName = `${aclName}-IPSet`;
        // Create the IP Set if requested
        this.ipSet = props.cidrAllowList
            ? new aws_wafv2_1.CfnIPSet(this, "IPSet", {
                name: ipSetName,
                addresses: props.cidrAllowList.cidrRanges,
                ipAddressVersion: props.cidrAllowList.cidrType,
                scope: "REGIONAL",
            })
            : undefined;
        const managedRules = props.managedRules ?? [
            { vendor: "AWS", name: "AWSManagedRulesCommonRuleSet" },
        ];
        const rules = [
            // Add a rule for the IP Set if specified
            ...(this.ipSet
                ? [
                    {
                        name: ipSetName,
                        priority: 1,
                        visibilityConfig: {
                            metricName: ipSetName,
                            cloudWatchMetricsEnabled: true,
                            sampledRequestsEnabled: true,
                        },
                        action: {
                            block: {},
                        },
                        statement: {
                            notStatement: {
                                statement: {
                                    ipSetReferenceStatement: {
                                        arn: this.ipSet.attrArn,
                                    },
                                },
                            },
                        },
                    },
                ]
                : []),
            // Add the managed rules
            ...managedRules.map(({ vendor, name }, i) => ({
                name: `${vendor}-${name}`,
                priority: i + 2,
                statement: { managedRuleGroupStatement: { vendorName: vendor, name } },
                overrideAction: { none: {} },
                visibilityConfig: {
                    metricName: `${aclName}-${vendor}-${name}`,
                    cloudWatchMetricsEnabled: true,
                    sampledRequestsEnabled: true,
                },
            })),
        ];
        this.webAcl = new aws_wafv2_1.CfnWebACL(this, "WebACL", {
            name: aclName,
            defaultAction: {
                // Allow by default, and use rules to deny unwanted requests
                allow: {},
            },
            scope: "REGIONAL",
            visibilityConfig: {
                cloudWatchMetricsEnabled: true,
                sampledRequestsEnabled: true,
                metricName: aclName,
            },
            rules,
        });
        this.webAclAssociation = new aws_wafv2_1.CfnWebACLAssociation(this, "WebACLAssociation", {
            resourceArn: props.apiDeploymentStageArn,
            webAclArn: this.webAcl.attrArn,
        });
    }
}
exports.OpenApiGatewayWebAcl = OpenApiGatewayWebAcl;
//# sourceMappingURL=data:application/json;base64,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