"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareApiSpec = exports.concatMethodAndPath = void 0;
const constants_1 = require("./constants");
/**
 * Serialise a method and path into a single string
 */
exports.concatMethodAndPath = ({ method, path }) => `${method.toLowerCase()}||${path.toLowerCase()}`;
// Add to methods to ensure no auth is added
const NO_AUTH_SPEC_SNIPPET = {
    security: [],
    "x-amazon-apigateway-auth": {
        type: "NONE",
    },
};
/**
 * Create the OpenAPI definition with api gateway extensions for the given authorizer
 * @param methodAuthorizer the authorizer used for the method
 */
const applyMethodAuthorizer = (methodAuthorizer) => {
    if (methodAuthorizer) {
        if (methodAuthorizer.authorizerId === constants_1.DefaultAuthorizerIds.NONE) {
            return NO_AUTH_SPEC_SNIPPET;
        }
        else {
            return {
                security: [
                    {
                        [methodAuthorizer.authorizerId]: methodAuthorizer.authorizationScopes || [],
                    },
                ],
            };
        }
    }
    return {};
};
/**
 * Adds API Gateway integrations and auth to the given operation
 */
const applyMethodIntegration = (path, method, { integrations, corsOptions }, operation, getOperationName) => {
    const operationName = getOperationName({ method, path });
    if (!(operationName in integrations)) {
        throw new Error(`Missing required integration for operation ${operationName} (${method} ${path})`);
    }
    const { methodAuthorizer, integration } = integrations[operationName];
    validateAuthorizerReference(methodAuthorizer, operation.security, operationName);
    return {
        ...operation,
        responses: Object.fromEntries(Object.entries(operation.responses).map(([statusCode, response]) => [
            statusCode,
            {
                ...response,
                headers: {
                    ...(corsOptions ? getCorsHeaderDefinitions() : {}),
                    // TODO: Consider following response header references
                    ...response.headers,
                },
            },
        ])),
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
        "x-amazon-apigateway-integration": integration,
        ...applyMethodAuthorizer(methodAuthorizer),
    };
};
const getCorsHeaderDefinitions = () => ({
    "Access-Control-Allow-Origin": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Methods": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Headers": {
        schema: { type: "string" },
    },
});
const generateCorsResponseHeaders = (corsOptions) => ({
    "Access-Control-Allow-Headers": `'${corsOptions.allowHeaders.join(",")}'`,
    "Access-Control-Allow-Methods": `'${corsOptions.allowMethods.join(",")}'`,
    "Access-Control-Allow-Origin": `'${corsOptions.allowOrigins.join(",")}'`,
});
const generateCorsResponseParameters = (corsOptions, prefix = "method.response.header") => Object.fromEntries(Object.entries(generateCorsResponseHeaders(corsOptions)).map(([header, value]) => [`${prefix}.${header}`, value]));
/**
 * Generates an "options" method with no auth to respond with the appropriate headers if cors is enabled
 */
const generateCorsOptionsMethod = (pathItem, { corsOptions }) => {
    // Do not generate if already manually defined, or cors not enabled
    if (constants_1.HttpMethods.OPTIONS in pathItem || !corsOptions) {
        return {};
    }
    const statusCode = corsOptions.statusCode;
    return {
        [constants_1.HttpMethods.OPTIONS]: {
            summary: "CORS Support",
            description: "Enable CORS by returning the correct headers",
            responses: {
                [`${statusCode}`]: {
                    description: "Default response for CORS method",
                    headers: getCorsHeaderDefinitions(),
                    content: {},
                },
            },
            // @ts-ignore Ignore apigateway extensions which are not part of default openapi spec type
            "x-amazon-apigateway-integration": {
                type: "mock",
                requestTemplates: {
                    "application/json": `{"statusCode": ${statusCode}}`,
                },
                responses: {
                    default: {
                        statusCode: `${statusCode}`,
                        responseParameters: generateCorsResponseParameters(corsOptions),
                        responseTemplates: {
                            "application/json": "{}",
                        },
                    },
                },
            },
            // No auth for CORS options requests
            ...NO_AUTH_SPEC_SNIPPET,
        },
    };
};
/**
 * Prepares a given api path by adding integrations, configuring auth
 */
const preparePathSpec = (path, pathItem, options, getOperationName) => {
    const supportedHttpMethods = new Set(Object.values(constants_1.HttpMethods));
    const unsupportedMethodsInSpec = Object.keys(pathItem).filter((method) => !supportedHttpMethods.has(method));
    if (unsupportedMethodsInSpec.length > 0) {
        throw new Error(`Path ${path} contains unsupported method${unsupportedMethodsInSpec.length > 1 ? "s" : ""} ${unsupportedMethodsInSpec.join(", ")}. Supported methods are ${Object.values(constants_1.HttpMethods).join(", ")}.`);
    }
    return {
        ...pathItem,
        ...Object.fromEntries(Object.values(constants_1.HttpMethods)
            .filter((method) => pathItem[method])
            .map((method) => [
            method,
            applyMethodIntegration(path, method, options, pathItem[method], getOperationName),
        ])),
        // Generate an 'options' method required for CORS preflight requests if cors is enabled
        ...generateCorsOptionsMethod(pathItem, options),
    };
};
/**
 * Return whether the given OpenAPI object is a reference object
 */
const isRef = (obj) => "$ref" in obj;
/**
 * Validate the construct security schemes against the security schemes in the original spec.
 * Construct-defined authorizers always override those in the spec if they have the same ID, however we validate that
 * we are not overriding an authorizer of a different type to avoid mistakes/mismatches between the spec and the
 * construct.
 * @param constructSecuritySchemes security schemes generated from the construct authorizers
 * @param existingSpecSecuritySchemes security schemes already defined in the spec
 */
const validateSecuritySchemes = (constructSecuritySchemes, existingSpecSecuritySchemes) => {
    if (existingSpecSecuritySchemes) {
        const constructSecuritySchemeIds = new Set(Object.keys(constructSecuritySchemes));
        const existingSecuritySchemeIds = new Set(Object.keys(existingSpecSecuritySchemes));
        const overlappingSecuritySchemeIds = [...constructSecuritySchemeIds].filter((id) => existingSecuritySchemeIds.has(id));
        // Any overlapping security schemes (defined in both the spec (or source smithy model) and the construct) must be of the same type.
        // The one defined in the construct will take precedence since a custom/cognito authorizer can have a resolved arn in the construct,
        // and we allow usage in the model as a forward definition with blank arn.
        overlappingSecuritySchemeIds.forEach((schemeId) => {
            if (!isRef(existingSpecSecuritySchemes[schemeId])) {
                const existingScheme = existingSpecSecuritySchemes[schemeId];
                if (constructSecuritySchemes[schemeId].type !== existingScheme.type) {
                    throw new Error(`Security scheme with id ${schemeId} was of type ${constructSecuritySchemes[schemeId].type} in construct but ${existingScheme.type} in OpenAPI spec or Smithy model.`);
                }
                const constructApiGatewayAuthType = constructSecuritySchemes[schemeId]["x-amazon-apigateway-authtype"];
                const existingApiGatewayAuthType = existingScheme["x-amazon-apigateway-authtype"];
                if (constructApiGatewayAuthType !== existingApiGatewayAuthType) {
                    throw new Error(`Security scheme with id ${schemeId} was of type ${constructApiGatewayAuthType} in construct but ${existingApiGatewayAuthType} in OpenAPI spec or Smithy model.`);
                }
            }
            else {
                throw new Error(`Security scheme with id ${schemeId} is a reference in the OpenAPI spec or Smithy model which is not supported.`);
            }
        });
    }
};
/**
 * Validate the given authorizer reference (either default or at an operation level) defined in the construct against
 * those already in the spec.
 * @param constructAuthorizer the authorizer defined in the construct
 * @param existingSpecAuthorizers the authorizers already defined in the spec
 * @param operation the operation we are validating (for clearer error messages)
 */
const validateAuthorizerReference = (constructAuthorizer, existingSpecAuthorizers, operation = "Default") => {
    // Only need to validate if defined in both - if just one we'll use that.
    if (constructAuthorizer && existingSpecAuthorizers) {
        const mergedSpecAuthorizers = Object.fromEntries(existingSpecAuthorizers.flatMap((securityRequirement) => Object.keys(securityRequirement).map((id) => [
            id,
            securityRequirement[id],
        ])));
        const specAuthorizerIds = Object.keys(mergedSpecAuthorizers);
        if (specAuthorizerIds.length > 1) {
            // Spec defined multiple authorizers but the construct can only specify one
            throw new Error(`${operation} authorizers ${specAuthorizerIds
                .sort()
                .join(", ")} defined in the OpenAPI Spec or Smithy Model would be overridden by single construct authorizer ${constructAuthorizer.authorizerId}`);
        }
        else if (specAuthorizerIds.length === 1) {
            // Single authorizer - check that they have the same id
            if (specAuthorizerIds[0] !== constructAuthorizer.authorizerId) {
                throw new Error(`${operation} authorizer ${specAuthorizerIds[0]} defined in the OpenAPI Spec or Smithy Model would be overridden by construct authorizer ${constructAuthorizer.authorizerId}`);
            }
            // Check that there are no differing scopes between the construct and spec
            const specScopes = new Set(mergedSpecAuthorizers[specAuthorizerIds[0]]);
            const constructScopes = new Set(constructAuthorizer.authorizationScopes);
            const differingScopes = [
                ...[...specScopes].filter((scope) => !constructScopes.has(scope)),
                ...[...constructScopes].filter((scope) => !specScopes.has(scope)),
            ];
            if (differingScopes.length > 0) {
                throw new Error(`${operation} authorizer scopes ${[...specScopes].join(", ")} defined in the OpenAPI Spec or Smithy Model differ from those in the construct (${[
                    ...constructScopes,
                ].join(", ")})`);
            }
        }
        else if (constructAuthorizer.authorizerId !== constants_1.DefaultAuthorizerIds.NONE) {
            // "security" section of spec is [] which means no auth, but the authorizer in the construct is not the "none" authorizer.
            throw new Error(`${operation} explicitly defines no auth in the OpenAPI Spec or Smithy Model which would be overridden by construct authorizer ${constructAuthorizer.authorizerId}`);
        }
    }
};
/**
 * Prepares the api spec for deployment by adding integrations, configuring auth, etc
 */
exports.prepareApiSpec = (spec, options) => {
    // Reverse lookup for the operation name given a method and path
    const operationNameByPath = Object.fromEntries(Object.entries(options.operationLookup).map(([operationName, methodAndPath]) => [
        exports.concatMethodAndPath(methodAndPath),
        operationName,
    ]));
    const getOperationName = (methodAndPath) => operationNameByPath[exports.concatMethodAndPath(methodAndPath)];
    validateSecuritySchemes(options.securitySchemes, spec.components?.securitySchemes);
    validateAuthorizerReference(options.defaultAuthorizerReference, spec.security);
    return {
        ...spec,
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-request-validators.html
        "x-amazon-apigateway-request-validators": {
            all: {
                validateRequestBody: true,
                validateRequestParameters: true,
            },
        },
        "x-amazon-apigateway-request-validator": "all",
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-gateway-responses.html
        "x-amazon-apigateway-gateway-responses": {
            BAD_REQUEST_BODY: {
                statusCode: 400,
                responseTemplates: {
                    "application/json": '{"message": "$context.error.validationErrorString"}',
                },
                ...(options.corsOptions
                    ? {
                        responseParameters: generateCorsResponseParameters(options.corsOptions, "gatewayresponse.header"),
                    }
                    : {}),
            },
        },
        paths: {
            ...Object.fromEntries(Object.entries(spec.paths).map(([path, pathDetails]) => [
                path,
                preparePathSpec(path, pathDetails, options, getOperationName),
            ])),
        },
        components: {
            ...spec.components,
            securitySchemes: {
                // Apply any security schemes that already exist in the spec
                ...spec.components?.securitySchemes,
                // Construct security schemes override any in the spec with the same id
                ...options.securitySchemes,
            },
        },
        // Apply the default authorizer at the top level
        ...(options.defaultAuthorizerReference
            ? applyMethodAuthorizer(options.defaultAuthorizerReference)
            : {}),
    };
};
//# sourceMappingURL=data:application/json;base64,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