"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const core_1 = require("@aws-solutions-constructs/core");
const lambda = require("aws-cdk-lib/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
// Note: All integration tests for alb are for HTTP APIs, as certificates require
// validation through DNS and email. This validation is impossible during our integration
// tests and the stack will fail to launch with an unvalidated certificate.
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename), {
    env: { account: aws_cdk_lib_1.Aws.ACCOUNT_ID, region: "us-east-1" },
});
stack.templateOptions.description =
    "Integration Test for public HTTP API with a existing function and ALB";
const myVpc = defaults.getTestVpc(stack);
const testSg = new aws_ec2_1.SecurityGroup(stack, "lambda-sg", {
    vpc: myVpc,
    allowAllOutbound: false,
});
const lambdaFunction = defaults.buildLambdaFunction(stack, {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
        vpc: myVpc,
        securityGroups: [testSg],
    },
});
const loadBalancer = defaults.ObtainAlb(stack, "existing-alb", myVpc, false, undefined, {
    internetFacing: false,
    vpc: myVpc,
}, true, {
    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
    autoDeleteObjects: true,
});
const props = {
    existingLambdaObj: lambdaFunction,
    existingLoadBalancerObj: loadBalancer,
    existingVpc: myVpc,
    listenerProps: {
        protocol: "HTTP",
    },
    publicApi: true,
};
const albToLambda = new lib_1.AlbToLambda(stack, "test-one", props);
defaults.addCfnSuppressRules(albToLambda.listener, [
    {
        id: "W56",
        reason: "All integration tests must be HTTP because of certificate limitations.",
    },
]);
const newSecurityGroup = albToLambda.loadBalancer.connections.securityGroups[0]
    .node.defaultChild;
defaults.addCfnSuppressRules(newSecurityGroup, [
    { id: "W29", reason: "CDK created rule that blocks all traffic." },
    { id: "W2", reason: "Rule does not apply for ELB." },
    { id: "W9", reason: "Rule does not apply for ELB." },
]);
defaults.addCfnSuppressRules(testSg, [
    { id: "W29", reason: "CDK created rule that blocks all traffic." },
]);
defaults.suppressAutoDeleteHandlerWarnings(stack);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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