"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsEventSource = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
/**
 * Use an Amazon SQS queue as an event source for AWS Lambda.
 *
 * @stability stable
 */
class SqsEventSource {
    /**
     * @stability stable
     */
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        this._eventSourceMappingId = undefined;
        if (this.props.maxBatchingWindow !== undefined) {
            if (queue.fifo) {
                throw new Error('Batching window is not supported for FIFO queues');
            }
            if (!this.props.maxBatchingWindow.isUnresolved() && this.props.maxBatchingWindow.toSeconds() > 300) {
                throw new Error(`Maximum batching window must be 300 seconds or less (given ${this.props.maxBatchingWindow.toHumanString()})`);
            }
        }
        if (this.props.batchSize !== undefined && !core_1.Token.isUnresolved(this.props.batchSize)) {
            if (this.props.maxBatchingWindow !== undefined && (this.props.batchSize < 1 || this.props.batchSize > 10000)) {
                throw new Error(`Maximum batch size must be between 1 and 10000 inclusive (given ${this.props.batchSize}) when batching window is specified.`);
            }
            if (this.props.maxBatchingWindow === undefined && (this.props.batchSize < 1 || this.props.batchSize > 10)) {
                throw new Error(`Maximum batch size must be between 1 and 10 inclusive (given ${this.props.batchSize}) when batching window is not specified.`);
            }
        }
    }
    /**
     * Called by `lambda.addEventSource` to allow the event source to bind to this function.
     *
     * @stability stable
     */
    bind(target) {
        const eventSourceMapping = target.addEventSourceMapping(`SqsEventSource:${core_1.Names.nodeUniqueId(this.queue.node)}`, {
            batchSize: this.props.batchSize,
            maxBatchingWindow: this.props.maxBatchingWindow,
            enabled: this.props.enabled,
            eventSourceArn: this.queue.queueArn,
        });
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.queue.grantConsumeMessages(target);
    }
    /**
     * The identifier for this EventSourceMapping.
     *
     * @stability stable
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('SqsEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.SqsEventSource = SqsEventSource;
_a = JSII_RTTI_SYMBOL_1;
SqsEventSource[_a] = { fqn: "@aws-cdk/aws-lambda-event-sources.SqsEventSource", version: "1.107.0" };
//# sourceMappingURL=data:application/json;base64,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