"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Utils = void 0;
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const fs = require("fs");
const yaml = require("js-yaml");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
/**
 * Utilities class used across the different resources
 */
class Utils {
    /**
     * Sanitize a string by removing upper case and replacing special characters except underscore
     * @param {string} toSanitize the string to sanitize
     */
    static stringSanitizer(toSanitize) {
        return toSanitize.toLowerCase().replace(/[^\w\s]/gi, '');
    }
    /**
     * Create a random string to be used as a seed for IAM User password
     * @param {string} name the string to which to append a random string
     */
    static randomize(name) {
        return `${name}-${Math.random().toString(36).substring(2, 9).toUpperCase()}`;
    }
    /**
     * Import the default IAM role used by CDK
     * @param {Construct} scope the scope to import the role into
     * @param {string} id the ID of the role in the stack
     */
    static getCdkExecRole(scope, id) {
        const cdkExecutionRoleArn = aws_cdk_lib_1.Fn.sub(aws_cdk_lib_1.DefaultStackSynthesizer.DEFAULT_CLOUDFORMATION_ROLE_ARN, {
            // eslint-disable-next-line @typescript-eslint/naming-convention
            Qualifier: aws_cdk_lib_1.DefaultStackSynthesizer.DEFAULT_QUALIFIER,
        });
        // Makes the CDK execution role LF admin so it can create databases
        return aws_iam_1.Role.fromRoleArn(aws_cdk_lib_1.Stack.of(scope), `${id}Role`, cdkExecutionRoleArn);
    }
    /**
     * Import the default IAM role used for CDK deploy
     * @param {Construct} scope the scope to import the role into
     * @param {string} id the ID of the role in the stack
     */
    static getCdkDeployRole(scope, id) {
        const cdkDeployRoleArn = aws_cdk_lib_1.Fn.sub(aws_cdk_lib_1.DefaultStackSynthesizer.DEFAULT_DEPLOY_ROLE_ARN, {
            // eslint-disable-next-line @typescript-eslint/naming-convention
            Qualifier: aws_cdk_lib_1.DefaultStackSynthesizer.DEFAULT_QUALIFIER,
        });
        // Makes the CDK execution role LF admin so it can create databases
        return aws_iam_1.Role.fromRoleArn(aws_cdk_lib_1.Stack.of(scope), `${id}Role`, cdkDeployRoleArn);
    }
    /**
     * Import the default IAM role used for CDK file publishing
     * @param {Construct} scope the scope to import the role into
     * @param {string} id the ID of the role in the stack
     */
    static getCdkFilePublishRole(scope, id) {
        const cdkDeployRoleArn = aws_cdk_lib_1.Fn.sub(aws_cdk_lib_1.DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PUBLISHING_ROLE_ARN, {
            // eslint-disable-next-line @typescript-eslint/naming-convention
            Qualifier: aws_cdk_lib_1.DefaultStackSynthesizer.DEFAULT_QUALIFIER,
        });
        // Makes the CDK execution role LF admin so it can create databases
        return aws_iam_1.Role.fromRoleArn(aws_cdk_lib_1.Stack.of(scope), `${id}Role`, cdkDeployRoleArn);
    }
    /*
       * Read a YAML file from the path provided and return it
       * @param {string} path the path to the file
       */
    static readYamlDocument(path) {
        try {
            const doc = fs.readFileSync(path, 'utf8');
            return doc;
        }
        catch (e) {
            console.log(e + ' for path: ' + path);
            throw e;
        }
    }
    /**
     * Take a document stored as string and load it as YAML
     * @param {string} document the document stored as string
     */
    static loadYaml(document) {
        return yaml.load(document);
    }
}
exports.Utils = Utils;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXRpbHMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvdXRpbHMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEscUVBQXFFO0FBQ3JFLGlDQUFpQztBQUNqQyx5QkFBeUI7QUFDekIsZ0NBQWdDO0FBRWhDLDZDQUFpRTtBQUNqRSxpREFBMkM7QUFHM0M7O0dBRUc7QUFDSCxNQUFhLEtBQUs7SUFFaEI7OztPQUdHO0lBQ0ksTUFBTSxDQUFDLGVBQWUsQ0FBQyxVQUFrQjtRQUM5QyxPQUFPLFVBQVUsQ0FBQyxXQUFXLEVBQUUsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQzNELENBQUM7SUFFRDs7O09BR0c7SUFDSSxNQUFNLENBQUMsU0FBUyxDQUFDLElBQVk7UUFDbEMsT0FBTyxHQUFHLElBQUksSUFBSSxJQUFJLENBQUMsTUFBTSxFQUFFLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUUsQ0FBQztJQUMvRSxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxjQUFjLENBQUMsS0FBZ0IsRUFBRSxFQUFVO1FBQ3ZELE1BQU0sbUJBQW1CLEdBQUcsZ0JBQUUsQ0FBQyxHQUFHLENBQ2hDLHFDQUF1QixDQUFDLCtCQUErQixFQUN2RDtZQUNFLGdFQUFnRTtZQUNoRSxTQUFTLEVBQUUscUNBQXVCLENBQUMsaUJBQWlCO1NBQ3JELENBQ0YsQ0FBQztRQUNGLG1FQUFtRTtRQUNuRSxPQUFPLGNBQUksQ0FBQyxXQUFXLENBQUMsbUJBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLEVBQUUsR0FBRyxFQUFFLE1BQU0sRUFBRSxtQkFBbUIsQ0FBQyxDQUFDO0lBQzdFLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLGdCQUFnQixDQUFDLEtBQWdCLEVBQUUsRUFBVTtRQUN6RCxNQUFNLGdCQUFnQixHQUFHLGdCQUFFLENBQUMsR0FBRyxDQUM3QixxQ0FBdUIsQ0FBQyx1QkFBdUIsRUFDL0M7WUFDRSxnRUFBZ0U7WUFDaEUsU0FBUyxFQUFFLHFDQUF1QixDQUFDLGlCQUFpQjtTQUNyRCxDQUNGLENBQUM7UUFDRixtRUFBbUU7UUFDbkUsT0FBTyxjQUFJLENBQUMsV0FBVyxDQUFDLG1CQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxFQUFFLEdBQUcsRUFBRSxNQUFNLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztJQUMxRSxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxLQUFnQixFQUFFLEVBQVU7UUFDOUQsTUFBTSxnQkFBZ0IsR0FBRyxnQkFBRSxDQUFDLEdBQUcsQ0FDN0IscUNBQXVCLENBQUMsc0NBQXNDLEVBQzlEO1lBQ0UsZ0VBQWdFO1lBQ2hFLFNBQVMsRUFBRSxxQ0FBdUIsQ0FBQyxpQkFBaUI7U0FDckQsQ0FDRixDQUFDO1FBQ0YsbUVBQW1FO1FBQ25FLE9BQU8sY0FBSSxDQUFDLFdBQVcsQ0FBQyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsRUFBRSxHQUFHLEVBQUUsTUFBTSxFQUFFLGdCQUFnQixDQUFDLENBQUM7SUFDMUUsQ0FBQztJQUNEOzs7U0FHSztJQUNFLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFZO1FBQ3pDLElBQUk7WUFDRixNQUFNLEdBQUcsR0FBRyxFQUFFLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxNQUFNLENBQUMsQ0FBQztZQUMxQyxPQUFPLEdBQUcsQ0FBQztTQUNaO1FBQUMsT0FBTyxDQUFDLEVBQUU7WUFDVixPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxhQUFhLEdBQUcsSUFBSSxDQUFDLENBQUM7WUFDdEMsTUFBTSxDQUFDLENBQUM7U0FDVDtJQUNILENBQUM7SUFFRDs7O09BR0c7SUFDSSxNQUFNLENBQUMsUUFBUSxDQUFDLFFBQWdCO1FBQ3JDLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUM3QixDQUFDO0NBRUY7QUExRkQsc0JBMEZDIiwic291cmNlc0NvbnRlbnQiOlsiLy8gQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4vLyBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogTUlULTBcbmltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzJztcbmltcG9ydCAqIGFzIHlhbWwgZnJvbSAnanMteWFtbCc7XG5cbmltcG9ydCB7IERlZmF1bHRTdGFja1N5bnRoZXNpemVyLCBGbiwgU3RhY2sgfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcbmltcG9ydCB7IFJvbGUgfSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWlhbVwiO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSBcImNvbnN0cnVjdHNcIjtcblxuLyoqXG4gKiBVdGlsaXRpZXMgY2xhc3MgdXNlZCBhY3Jvc3MgdGhlIGRpZmZlcmVudCByZXNvdXJjZXNcbiAqL1xuZXhwb3J0IGNsYXNzIFV0aWxzIHtcblxuICAvKipcbiAgICogU2FuaXRpemUgYSBzdHJpbmcgYnkgcmVtb3ZpbmcgdXBwZXIgY2FzZSBhbmQgcmVwbGFjaW5nIHNwZWNpYWwgY2hhcmFjdGVycyBleGNlcHQgdW5kZXJzY29yZVxuICAgKiBAcGFyYW0ge3N0cmluZ30gdG9TYW5pdGl6ZSB0aGUgc3RyaW5nIHRvIHNhbml0aXplXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHN0cmluZ1Nhbml0aXplcih0b1Nhbml0aXplOiBzdHJpbmcpOiBzdHJpbmcge1xuICAgIHJldHVybiB0b1Nhbml0aXplLnRvTG93ZXJDYXNlKCkucmVwbGFjZSgvW15cXHdcXHNdL2dpLCAnJyk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgcmFuZG9tIHN0cmluZyB0byBiZSB1c2VkIGFzIGEgc2VlZCBmb3IgSUFNIFVzZXIgcGFzc3dvcmRcbiAgICogQHBhcmFtIHtzdHJpbmd9IG5hbWUgdGhlIHN0cmluZyB0byB3aGljaCB0byBhcHBlbmQgYSByYW5kb20gc3RyaW5nXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJhbmRvbWl6ZShuYW1lOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gYCR7bmFtZX0tJHtNYXRoLnJhbmRvbSgpLnRvU3RyaW5nKDM2KS5zdWJzdHJpbmcoMiwgOSkudG9VcHBlckNhc2UoKX1gO1xuICB9XG5cbiAgLyoqXG4gICAqIEltcG9ydCB0aGUgZGVmYXVsdCBJQU0gcm9sZSB1c2VkIGJ5IENESyBcbiAgICogQHBhcmFtIHtDb25zdHJ1Y3R9IHNjb3BlIHRoZSBzY29wZSB0byBpbXBvcnQgdGhlIHJvbGUgaW50b1xuICAgKiBAcGFyYW0ge3N0cmluZ30gaWQgdGhlIElEIG9mIHRoZSByb2xlIGluIHRoZSBzdGFjayBcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZ2V0Q2RrRXhlY1JvbGUoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZykge1xuICAgIGNvbnN0IGNka0V4ZWN1dGlvblJvbGVBcm4gPSBGbi5zdWIoXG4gICAgICBEZWZhdWx0U3RhY2tTeW50aGVzaXplci5ERUZBVUxUX0NMT1VERk9STUFUSU9OX1JPTEVfQVJOLFxuICAgICAge1xuICAgICAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgQHR5cGVzY3JpcHQtZXNsaW50L25hbWluZy1jb252ZW50aW9uXG4gICAgICAgIFF1YWxpZmllcjogRGVmYXVsdFN0YWNrU3ludGhlc2l6ZXIuREVGQVVMVF9RVUFMSUZJRVIsXG4gICAgICB9LFxuICAgICk7XG4gICAgLy8gTWFrZXMgdGhlIENESyBleGVjdXRpb24gcm9sZSBMRiBhZG1pbiBzbyBpdCBjYW4gY3JlYXRlIGRhdGFiYXNlc1xuICAgIHJldHVybiBSb2xlLmZyb21Sb2xlQXJuKFN0YWNrLm9mKHNjb3BlKSwgYCR7aWR9Um9sZWAsIGNka0V4ZWN1dGlvblJvbGVBcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEltcG9ydCB0aGUgZGVmYXVsdCBJQU0gcm9sZSB1c2VkIGZvciBDREsgZGVwbG95IFxuICAgKiBAcGFyYW0ge0NvbnN0cnVjdH0gc2NvcGUgdGhlIHNjb3BlIHRvIGltcG9ydCB0aGUgcm9sZSBpbnRvXG4gICAqIEBwYXJhbSB7c3RyaW5nfSBpZCB0aGUgSUQgb2YgdGhlIHJvbGUgaW4gdGhlIHN0YWNrIFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBnZXRDZGtEZXBsb3lSb2xlKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcpIHtcbiAgICBjb25zdCBjZGtEZXBsb3lSb2xlQXJuID0gRm4uc3ViKFxuICAgICAgRGVmYXVsdFN0YWNrU3ludGhlc2l6ZXIuREVGQVVMVF9ERVBMT1lfUk9MRV9BUk4sXG4gICAgICB7XG4gICAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBAdHlwZXNjcmlwdC1lc2xpbnQvbmFtaW5nLWNvbnZlbnRpb25cbiAgICAgICAgUXVhbGlmaWVyOiBEZWZhdWx0U3RhY2tTeW50aGVzaXplci5ERUZBVUxUX1FVQUxJRklFUixcbiAgICAgIH0sXG4gICAgKTtcbiAgICAvLyBNYWtlcyB0aGUgQ0RLIGV4ZWN1dGlvbiByb2xlIExGIGFkbWluIHNvIGl0IGNhbiBjcmVhdGUgZGF0YWJhc2VzXG4gICAgcmV0dXJuIFJvbGUuZnJvbVJvbGVBcm4oU3RhY2sub2Yoc2NvcGUpLCBgJHtpZH1Sb2xlYCwgY2RrRGVwbG95Um9sZUFybik7XG4gIH1cblxuICAvKipcbiAgICogSW1wb3J0IHRoZSBkZWZhdWx0IElBTSByb2xlIHVzZWQgZm9yIENESyBmaWxlIHB1Ymxpc2hpbmcgXG4gICAqIEBwYXJhbSB7Q29uc3RydWN0fSBzY29wZSB0aGUgc2NvcGUgdG8gaW1wb3J0IHRoZSByb2xlIGludG9cbiAgICogQHBhcmFtIHtzdHJpbmd9IGlkIHRoZSBJRCBvZiB0aGUgcm9sZSBpbiB0aGUgc3RhY2sgXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGdldENka0ZpbGVQdWJsaXNoUm9sZShzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nKSB7XG4gICAgY29uc3QgY2RrRGVwbG95Um9sZUFybiA9IEZuLnN1YihcbiAgICAgIERlZmF1bHRTdGFja1N5bnRoZXNpemVyLkRFRkFVTFRfRklMRV9BU1NFVF9QVUJMSVNISU5HX1JPTEVfQVJOLFxuICAgICAge1xuICAgICAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgQHR5cGVzY3JpcHQtZXNsaW50L25hbWluZy1jb252ZW50aW9uXG4gICAgICAgIFF1YWxpZmllcjogRGVmYXVsdFN0YWNrU3ludGhlc2l6ZXIuREVGQVVMVF9RVUFMSUZJRVIsXG4gICAgICB9LFxuICAgICk7XG4gICAgLy8gTWFrZXMgdGhlIENESyBleGVjdXRpb24gcm9sZSBMRiBhZG1pbiBzbyBpdCBjYW4gY3JlYXRlIGRhdGFiYXNlc1xuICAgIHJldHVybiBSb2xlLmZyb21Sb2xlQXJuKFN0YWNrLm9mKHNjb3BlKSwgYCR7aWR9Um9sZWAsIGNka0RlcGxveVJvbGVBcm4pO1xuICB9XG4gIC8qXG4gICAgICogUmVhZCBhIFlBTUwgZmlsZSBmcm9tIHRoZSBwYXRoIHByb3ZpZGVkIGFuZCByZXR1cm4gaXQgXG4gICAgICogQHBhcmFtIHtzdHJpbmd9IHBhdGggdGhlIHBhdGggdG8gdGhlIGZpbGVcbiAgICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkWWFtbERvY3VtZW50KHBhdGg6IHN0cmluZyk6IHN0cmluZyB7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IGRvYyA9IGZzLnJlYWRGaWxlU3luYyhwYXRoLCAndXRmOCcpO1xuICAgICAgcmV0dXJuIGRvYztcbiAgICB9IGNhdGNoIChlKSB7XG4gICAgICBjb25zb2xlLmxvZyhlICsgJyBmb3IgcGF0aDogJyArIHBhdGgpO1xuICAgICAgdGhyb3cgZTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogVGFrZSBhIGRvY3VtZW50IHN0b3JlZCBhcyBzdHJpbmcgYW5kIGxvYWQgaXQgYXMgWUFNTCBcbiAgICogQHBhcmFtIHtzdHJpbmd9IGRvY3VtZW50IHRoZSBkb2N1bWVudCBzdG9yZWQgYXMgc3RyaW5nXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGxvYWRZYW1sKGRvY3VtZW50OiBzdHJpbmcpOiBhbnkge1xuICAgIHJldHVybiB5YW1sLmxvYWQoZG9jdW1lbnQpO1xuICB9XG5cbn1cbiJdfQ==