"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.createIAMFederatedRole = exports.createIAMRolePolicy = exports.createStudioServiceRolePolicy = exports.addServiceRoleInlinePolicy = exports.createStudioUserRolePolicy = exports.createUserSessionPolicy = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const utils_1 = require("../utils");
const studioS3Policy = require("./resources/studio/emr-studio-s3-policy.json");
const studioServiceRolePolicy = require("./resources/studio/studio-service-role-policy.json");
const studioUserRolePolicy = require("./resources/studio/studio-user-iam-role-policy.json");
const studioSessionPolicy = require("./resources/studio/studio-user-session-policy.json");
const studioUserPolicy = require("./resources/studio/studio-user-sso-role-policy.json");
/**
 * @internal
 * Create a session policy for each user scoped down to the managed endpoint
 * @returns Return the ARN of the policy created
 */
function createUserSessionPolicy(scope, user, studioServiceRoleName, managedEndpointArns, studioId) {
    let policy = JSON.parse(JSON.stringify(studioSessionPolicy));
    //replace the <your-emr-studio-service-role> with the service role created above
    policy.Statement[5].Resource[0] = policy.Statement[5].Resource[0].replace(/<your-emr-studio-service-role>/gi, studioServiceRoleName);
    //replace the region and account for log bucket
    policy.Statement[7].Resource[0] = policy.Statement[7].Resource[0].replace(/<aws-account-id>/gi, aws_cdk_lib_1.Aws.ACCOUNT_ID);
    policy.Statement[7].Resource[0] = policy.Statement[7].Resource[0].replace(/<region>/gi, aws_cdk_lib_1.Aws.REGION);
    //replace the region and account for list virtual cluster
    policy.Statement[8].Resource[0] = policy.Statement[8].Resource[0].replace(/<aws-account-id>/gi, aws_cdk_lib_1.Aws.ACCOUNT_ID);
    policy.Statement[8].Resource[0] = policy.Statement[8].Resource[0].replace(/<region>/gi, aws_cdk_lib_1.Aws.REGION);
    //add restrictions on the managedEndpoint that user of group is allowed to attach to
    for (let managedEndpointArn of managedEndpointArns) {
        policy.Statement[9].Resource[managedEndpointArns.indexOf(managedEndpointArn)] = managedEndpointArn;
        policy.Statement[10].Resource[managedEndpointArns.indexOf(managedEndpointArn)] = managedEndpointArn;
    }
    //create the policy
    let userSessionPolicy = new aws_iam_1.ManagedPolicy(scope, 'studioSessionPolicy' + utils_1.Utils.stringSanitizer(user.identityName), {
        document: aws_iam_1.PolicyDocument.fromJson(policy),
        managedPolicyName: 'studioSessionPolicy' + utils_1.Utils.stringSanitizer(user.identityName) + studioId,
    });
    return userSessionPolicy.managedPolicyArn;
}
exports.createUserSessionPolicy = createUserSessionPolicy;
/**
 * @internal
 * Create a policy for the EMR USER Role
 * @returns Return the ARN of the policy created
 */
function createStudioUserRolePolicy(scope, studioName, studioServiceRoleName) {
    let policyTemplate = JSON.stringify(studioUserPolicy);
    let policy = JSON.parse(policyTemplate);
    //replace the <your-emr-studio-service-role> with the service role created above
    policy.Statement[5].Resource[0] = policy.Statement[5].Resource[0].replace(/<your-emr-studio-service-role>/gi, studioServiceRoleName);
    //replace the log bucket
    policy.Statement[7].Resource[0] = policy.Statement[7].Resource[0].replace(/<aws-account-id>/gi, aws_cdk_lib_1.Aws.ACCOUNT_ID);
    policy.Statement[7].Resource[0] = policy.Statement[7].Resource[0].replace(/<region>/gi, aws_cdk_lib_1.Aws.REGION);
    let userRolePolicy = new aws_iam_1.ManagedPolicy(scope, 'studioUserPolicy' + studioName, {
        document: aws_iam_1.PolicyDocument.fromJson(policy),
        managedPolicyName: 'studioUserPolicy' + studioName,
    });
    return userRolePolicy.managedPolicyArn;
}
exports.createStudioUserRolePolicy = createStudioUserRolePolicy;
/**
 * @internal
 * Add an inline policy to the role passed by the user
 */
function addServiceRoleInlinePolicy(scope, studioServiceRoleArn, bucketName) {
    //Get policy from a JSON template
    let policy = JSON.parse(JSON.stringify(studioS3Policy));
    //Update the service role provided by the user with an inline policy
    //to access the S3 bucket and store notebooks
    policy.Statement[0].Resource[0] = policy.Statement[0].Resource[0].replace(/<your-amazon-s3-bucket>/gi, bucketName);
    policy.Statement[0].Resource[1] = policy.Statement[0].Resource[1].replace(/<your-amazon-s3-bucket>/gi, bucketName);
    let studioServiceRole = aws_iam_1.Role.fromRoleArn(scope, 'studioServiceRoleInlinePolicy', studioServiceRoleArn);
    studioServiceRole.attachInlinePolicy(new aws_iam_1.Policy(scope, 'studioServiceInlinePolicy', {
        document: aws_iam_1.PolicyDocument.fromJson(policy),
    }));
    return studioServiceRole;
}
exports.addServiceRoleInlinePolicy = addServiceRoleInlinePolicy;
/**
 * @internal
 * Create a policy for the EMR Service Role
 * The policy allow access only to a single bucket to store notebooks
 * @returns Return the ARN of the policy created
 */
function createStudioServiceRolePolicy(scope, keyArn, bucketName, studioName) {
    //Get policy from a JSON template
    let policy = JSON.parse(JSON.stringify(studioServiceRolePolicy));
    //Update the policy with the bucketname to scope it down
    policy.Statement[11].Resource[0] = policy.Statement[11].Resource[0].replace(/<your-amazon-s3-bucket>/gi, bucketName);
    policy.Statement[11].Resource[1] = policy.Statement[11].Resource[1].replace(/<your-amazon-s3-bucket>/gi, bucketName);
    //Update with KMS key ARN encrypting the bucket
    policy.Statement[12].Resource[0] = keyArn;
    //Create the policy of service role
    let serviceRolePolicy = new aws_iam_1.ManagedPolicy(scope, 'studioServicePolicy' + studioName, {
        document: aws_iam_1.PolicyDocument.fromJson(policy),
        managedPolicyName: 'studioServicePolicy' + studioName,
    });
    return serviceRolePolicy.managedPolicyArn;
}
exports.createStudioServiceRolePolicy = createStudioServiceRolePolicy;
/**
 * @internal
 * Create a policy for the role to which a user federate
 * Called when working in IAM auth mode with Federated IdP
 * @returns Return the ARN of the policy created
 */
function createIAMRolePolicy(scope, user, studioServiceRoleName, managedEndpointArns, studioId) {
    let policy = JSON.parse(JSON.stringify(studioUserRolePolicy));
    //replace the <your-emr-studio-service-role> with the service role created above
    policy.Statement[5].Resource[0] = policy.Statement[5].Resource[0].replace(/<your-emr-studio-service-role>/gi, studioServiceRoleName);
    //replace the region and account for log bucket
    policy.Statement[7].Resource[0] = policy.Statement[7].Resource[0].replace(/<aws-account-id>/gi, aws_cdk_lib_1.Aws.ACCOUNT_ID);
    policy.Statement[7].Resource[0] = policy.Statement[7].Resource[0].replace(/<region>/gi, aws_cdk_lib_1.Aws.REGION);
    //replace the region and account for list virtual cluster
    policy.Statement[8].Resource[0] = policy.Statement[8].Resource[0].replace(/<aws-account-id>/gi, aws_cdk_lib_1.Aws.ACCOUNT_ID);
    policy.Statement[8].Resource[0] = policy.Statement[8].Resource[0].replace(/<region>/gi, aws_cdk_lib_1.Aws.REGION);
    //add restrictions on the managedEndpoint that user of group is allowed to attach to
    for (let managedEndpointArn of managedEndpointArns) {
        policy.Statement[9].Resource[managedEndpointArns.indexOf(managedEndpointArn)] = managedEndpointArn;
        policy.Statement[10].Resource[managedEndpointArns.indexOf(managedEndpointArn)] = managedEndpointArn;
    }
    //Restrict the studio to which a federated user or iam user can access
    policy.Statement[12].Resource[0] = policy.Statement[12].Resource[0].replace(/<aws-account-id>/gi, aws_cdk_lib_1.Aws.ACCOUNT_ID);
    policy.Statement[12].Resource[0] = policy.Statement[12].Resource[0].replace(/<region>/gi, aws_cdk_lib_1.Aws.REGION);
    policy.Statement[12].Resource[0] = policy.Statement[12].Resource[0].replace(/<your-studio-id>/gi, studioId);
    let identityName = user.identityName ? user.identityName : user.iamUser?.userName;
    //create the policy
    return new aws_iam_1.ManagedPolicy(scope, 'studioSessionPolicy' + utils_1.Utils.stringSanitizer(identityName), {
        document: aws_iam_1.PolicyDocument.fromJson(policy),
        managedPolicyName: 'studioIAMRolePolicy-' + utils_1.Utils.stringSanitizer(identityName) + '-' + studioId,
    });
}
exports.createIAMRolePolicy = createIAMRolePolicy;
/**
 * @internal
 * Create the role to which a user federate
 * Called when working in IAM auth mode with Federated IdP
 * @returns Return the ARN of the policy created
 */
function createIAMFederatedRole(scope, iamRolePolicy, federatedIdPArn, identityName, studioId) {
    return new aws_iam_1.Role(scope, identityName.replace(/[^\w\s]/gi, '') + studioId.replace(/[^\w\s]/gi, ''), {
        assumedBy: new aws_iam_1.FederatedPrincipal(federatedIdPArn, {
            StringEquals: {
                'SAML:aud': 'https://signin.aws.amazon.com/saml',
            },
        }, 'sts:AssumeRoleWithSAML'),
        roleName: 'Role-' + identityName + studioId,
        managedPolicies: [iamRolePolicy],
    });
}
exports.createIAMFederatedRole = createIAMFederatedRole;
//# sourceMappingURL=data:application/json;base64,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