"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrEksNodegroupAsgTagProvider = void 0;
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const constructs_1 = require("constructs");
const pre_bundled_layer_1 = require("../common/pre-bundled-layer");
/**
 * EmrEksNodegroupAsgTagsProvider Construct implementing a custom resource provider for tagging EC2 Auto Scaling Group of EmrEksNodegroup.
 * By default EKS Managed Nodegroups are using EC2 Auto Scaling Group that are not tagged for Kubernetes Cluster Autoscaler usage.
 * If minimum number of node is 0, the Cluster Autoscaler is [not able to scale the nodegroup](https://github.com/aws/containers-roadmap/issues/724)
 */
class EmrEksNodegroupAsgTagProvider extends constructs_1.Construct {
    /**
     * Constructs a new instance of the ManageEnEmrEksNodegroupAsgTagsProviderdpointProvider.
     * The provider can then be used to create custom resources for tagging EC2 Auto Scaling group
     * @param { Construct} scope the Scope of the CDK Construct
     * @param id the ID of the CDK Construct
     */
    constructor(scope, id, props) {
        super(scope, id);
        //The policy allowing asg Tag custom resource call autoscaling api
        const lambdaPolicy = [
            new aws_iam_1.PolicyStatement({
                resources: ['*'],
                actions: [
                    'autoscaling:DescribeAutoScalingGroups',
                ],
            }),
            new aws_iam_1.PolicyStatement({
                resources: ['*'],
                actions: [
                    'autoscaling:CreateOrUpdateTags',
                    'autoscaling:DeleteTags',
                ],
                conditions: {
                    'ForAnyValue:StringEquals': {
                        'aws:ResourceTag/eks:cluster-name': props.eksClusterName,
                    },
                },
            }),
        ];
        // AWS Lambda function supporting the create, update, delete operations on Amazon EMR on EKS managed endpoints
        const onEvent = new pre_bundled_function_1.PreBundledFunction(this, 'Tag', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            codePath: 'emr-eks-platform/resources/lambdas/nodegroup-asg-tag',
            handler: 'lambda.on_event',
            lambdaPolicyStatements: lambdaPolicy,
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            timeout: aws_cdk_lib_1.Duration.seconds(45),
            environment: {
                EKS_CLUSTER_NAME: props.eksClusterName,
            },
        });
        this.provider = new custom_resources_1.Provider(this, 'CustomResourceProvider', {
            onEventHandler: onEvent,
        });
    }
}
exports.EmrEksNodegroupAsgTagProvider = EmrEksNodegroupAsgTagProvider;
//# sourceMappingURL=data:application/json;base64,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