"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.emrOnEksImageMap = exports.EmrEksImageBuilderCRProviderSetup = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const pre_bundled_layer_1 = require("../common/pre-bundled-layer");
/**
 * @internal
 * Create a Customer Resource to start a codebuild project
 * The policy allow access only to a single bucket to store notebooks
 * @returns Return the token to the Provider for CR
 */
function EmrEksImageBuilderCRProviderSetup(scope, codeBuildProjectArn) {
    //The policy allowing the creatio of the job template
    const lambdaPolicy = [
        new aws_iam_1.PolicyStatement({
            resources: [codeBuildProjectArn],
            actions: ['codebuild:BatchGetBuilds', 'codebuild:StartBuild'],
        })
    ];
    // AWS Lambda function supporting the start a codebuild project
    const onEvent = new pre_bundled_function_1.PreBundledFunction(scope, 'OnEvent', {
        codePath: 'docker-builder/resources/lambdas',
        runtime: aws_lambda_1.Runtime.PYTHON_3_9,
        handler: 'lambda.on_event',
        layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
        lambdaPolicyStatements: lambdaPolicy,
        logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        timeout: aws_cdk_lib_1.Duration.seconds(120),
    });
    // AWS Lambda function that check the status of codebuild porject
    const isComplete = new pre_bundled_function_1.PreBundledFunction(scope, 'IsComplete', {
        codePath: 'docker-builder/resources/lambdas',
        handler: 'lambda.is_complete',
        layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
        lambdaPolicyStatements: lambdaPolicy,
        runtime: aws_lambda_1.Runtime.PYTHON_3_9,
        logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        timeout: aws_cdk_lib_1.Duration.seconds(120),
    });
    const provider = new custom_resources_1.Provider(scope, 'CustomResourceProvider', {
        onEventHandler: onEvent,
        isCompleteHandler: isComplete,
        totalTimeout: aws_cdk_lib_1.Duration.minutes(15),
        queryInterval: aws_cdk_lib_1.Duration.seconds(20),
        providerFunctionName: 'dockerBuildPublishFn',
    });
    return provider.serviceToken;
}
exports.EmrEksImageBuilderCRProviderSetup = EmrEksImageBuilderCRProviderSetup;
/**
 * @internal
 * a Map containing the account for each region where the docker image are stored
 * The list is maintained in this AWS documentation (link)[https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/docker-custom-images-tag.html]
 */
exports.emrOnEksImageMap = new Map([
    ['ap-northeast-1', "059004520145"],
    ['ap-northeast-2', "996579266876"],
    ['ap-south-1', "235914868574"],
    ['ap-southeast-1', "671219180197"],
    ['ap-southeast-2', "038297999601"],
    ['ca-central-1', "351826393999"],
    ['eu-central-1', "107292555468"],
    ['eu-north-1', "830386416364"],
    ['eu-west-1', "483788554619"],
    ['eu-west-2', "118780647275"],
    ['eu-west-3', "307523725174"],
    ['sa-east-1', "052806832358"],
    ['us-east-1', "755674844232"],
    ['us-east-2', "711395599931"],
    ['us-west-1', "608033475327"],
    ['us-west-2', "895885662937"],
]);
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZG9ja2VyLWJ1aWxkZXItdXRpbC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9kb2NrZXItYnVpbGRlci9kb2NrZXItYnVpbGRlci11dGlsLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLDZDQUF1QztBQUN2QyxpREFBc0Q7QUFDdEQsdURBQWlEO0FBQ2pELG1EQUFxRDtBQUNyRCxtRUFBd0Q7QUFFeEQseUVBQW9FO0FBQ3BFLG1FQUE4RDtBQUU5RDs7Ozs7R0FLRztBQUNILFNBQWdCLGlDQUFpQyxDQUFFLEtBQWdCLEVBQUUsbUJBQTJCO0lBQzVGLHFEQUFxRDtJQUNyRCxNQUFNLFlBQVksR0FBRztRQUNqQixJQUFJLHlCQUFlLENBQUM7WUFDbEIsU0FBUyxFQUFFLENBQUMsbUJBQW1CLENBQUM7WUFDaEMsT0FBTyxFQUFFLENBQUMsMEJBQTBCLEVBQUUsc0JBQXNCLENBQUM7U0FDOUQsQ0FBQztLQUNILENBQUM7SUFFRiwrREFBK0Q7SUFDL0QsTUFBTSxPQUFPLEdBQUcsSUFBSSx5Q0FBa0IsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO1FBQ3ZELFFBQVEsRUFBRSxrQ0FBa0M7UUFDNUMsT0FBTyxFQUFFLG9CQUFPLENBQUMsVUFBVTtRQUMzQixPQUFPLEVBQUUsaUJBQWlCO1FBQzFCLE1BQU0sRUFBRSxDQUFDLG1DQUFlLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRSw0Q0FBNEMsQ0FBQyxDQUFDO1FBQzFGLHNCQUFzQixFQUFFLFlBQVk7UUFDcEMsWUFBWSxFQUFFLHdCQUFhLENBQUMsUUFBUTtRQUNwQyxPQUFPLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDO0tBQy9CLENBQUMsQ0FBQztJQUVILGlFQUFpRTtJQUNqRSxNQUFNLFVBQVUsR0FBRyxJQUFJLHlDQUFrQixDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7UUFDM0QsUUFBUSxFQUFFLGtDQUFrQztRQUM1QyxPQUFPLEVBQUUsb0JBQW9CO1FBQzdCLE1BQU0sRUFBRSxDQUFDLG1DQUFlLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRSw0Q0FBNEMsQ0FBQyxDQUFDO1FBQzFGLHNCQUFzQixFQUFFLFlBQVk7UUFDcEMsT0FBTyxFQUFFLG9CQUFPLENBQUMsVUFBVTtRQUMzQixZQUFZLEVBQUUsd0JBQWEsQ0FBQyxRQUFRO1FBQ3BDLE9BQU8sRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUM7S0FDL0IsQ0FBQyxDQUFDO0lBRUwsTUFBTSxRQUFRLEdBQUcsSUFBSSwyQkFBUSxDQUFDLEtBQUssRUFBRSx3QkFBd0IsRUFBRTtRQUM3RCxjQUFjLEVBQUUsT0FBTztRQUN2QixpQkFBaUIsRUFBRSxVQUFVO1FBQzdCLFlBQVksRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7UUFDbEMsYUFBYSxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztRQUNuQyxvQkFBb0IsRUFBRSxzQkFBc0I7S0FDN0MsQ0FBQyxDQUFDO0lBRUgsT0FBTyxRQUFRLENBQUMsWUFBWSxDQUFDO0FBQ25DLENBQUM7QUF4Q0QsOEVBd0NDO0FBRUQ7Ozs7R0FJRztBQUNVLFFBQUEsZ0JBQWdCLEdBQUcsSUFBSSxHQUFHLENBQUM7SUFDdEMsQ0FBQyxnQkFBZ0IsRUFBRSxjQUFjLENBQUM7SUFDbEMsQ0FBQyxnQkFBZ0IsRUFBRSxjQUFjLENBQUM7SUFDbEMsQ0FBQyxZQUFZLEVBQUUsY0FBYyxDQUFDO0lBQzlCLENBQUMsZ0JBQWdCLEVBQUUsY0FBYyxDQUFDO0lBQ2xDLENBQUMsZ0JBQWdCLEVBQUUsY0FBYyxDQUFDO0lBQ2xDLENBQUMsY0FBYyxFQUFFLGNBQWMsQ0FBQztJQUNoQyxDQUFDLGNBQWMsRUFBRSxjQUFjLENBQUM7SUFDaEMsQ0FBQyxZQUFZLEVBQUUsY0FBYyxDQUFDO0lBQzlCLENBQUMsV0FBVyxFQUFFLGNBQWMsQ0FBQztJQUM3QixDQUFDLFdBQVcsRUFBRSxjQUFjLENBQUM7SUFDN0IsQ0FBQyxXQUFXLEVBQUUsY0FBYyxDQUFDO0lBQzdCLENBQUMsV0FBVyxFQUFFLGNBQWMsQ0FBQztJQUM3QixDQUFDLFdBQVcsRUFBRSxjQUFjLENBQUM7SUFDN0IsQ0FBQyxXQUFXLEVBQUUsY0FBYyxDQUFDO0lBQzdCLENBQUMsV0FBVyxFQUFFLGNBQWMsQ0FBQztJQUM3QixDQUFDLFdBQVcsRUFBRSxjQUFjLENBQUM7Q0FDOUIsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgRHVyYXRpb24gfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtaWFtXCI7XG5pbXBvcnQgeyBSdW50aW1lIH0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1sYW1iZGFcIjtcbmltcG9ydCB7IFJldGVudGlvbkRheXMgfSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWxvZ3NcIjtcbmltcG9ydCB7IFByb3ZpZGVyIH0gZnJvbSBcImF3cy1jZGstbGliL2N1c3RvbS1yZXNvdXJjZXNcIjtcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gXCJjb25zdHJ1Y3RzXCI7XG5pbXBvcnQgeyBQcmVCdW5kbGVkRnVuY3Rpb24gfSBmcm9tIFwiLi4vY29tbW9uL3ByZS1idW5kbGVkLWZ1bmN0aW9uXCI7XG5pbXBvcnQgeyBQcmVCdW5kbGVkTGF5ZXIgfSBmcm9tIFwiLi4vY29tbW9uL3ByZS1idW5kbGVkLWxheWVyXCI7XG5cbi8qKlxuICogQGludGVybmFsXG4gKiBDcmVhdGUgYSBDdXN0b21lciBSZXNvdXJjZSB0byBzdGFydCBhIGNvZGVidWlsZCBwcm9qZWN0XG4gKiBUaGUgcG9saWN5IGFsbG93IGFjY2VzcyBvbmx5IHRvIGEgc2luZ2xlIGJ1Y2tldCB0byBzdG9yZSBub3RlYm9va3NcbiAqIEByZXR1cm5zIFJldHVybiB0aGUgdG9rZW4gdG8gdGhlIFByb3ZpZGVyIGZvciBDUlxuICovXG5leHBvcnQgZnVuY3Rpb24gRW1yRWtzSW1hZ2VCdWlsZGVyQ1JQcm92aWRlclNldHVwIChzY29wZTogQ29uc3RydWN0LCBjb2RlQnVpbGRQcm9qZWN0QXJuOiBzdHJpbmcpIDogc3RyaW5nIHtcbiAgICAvL1RoZSBwb2xpY3kgYWxsb3dpbmcgdGhlIGNyZWF0aW8gb2YgdGhlIGpvYiB0ZW1wbGF0ZVxuICAgIGNvbnN0IGxhbWJkYVBvbGljeSA9IFtcbiAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgcmVzb3VyY2VzOiBbY29kZUJ1aWxkUHJvamVjdEFybl0sXG4gICAgICAgICAgYWN0aW9uczogWydjb2RlYnVpbGQ6QmF0Y2hHZXRCdWlsZHMnLCAnY29kZWJ1aWxkOlN0YXJ0QnVpbGQnXSxcbiAgICAgICAgfSlcbiAgICAgIF07XG4gIFxuICAgICAgLy8gQVdTIExhbWJkYSBmdW5jdGlvbiBzdXBwb3J0aW5nIHRoZSBzdGFydCBhIGNvZGVidWlsZCBwcm9qZWN0XG4gICAgICBjb25zdCBvbkV2ZW50ID0gbmV3IFByZUJ1bmRsZWRGdW5jdGlvbihzY29wZSwgJ09uRXZlbnQnLCB7XG4gICAgICAgIGNvZGVQYXRoOiAnZG9ja2VyLWJ1aWxkZXIvcmVzb3VyY2VzL2xhbWJkYXMnLFxuICAgICAgICBydW50aW1lOiBSdW50aW1lLlBZVEhPTl8zXzksXG4gICAgICAgIGhhbmRsZXI6ICdsYW1iZGEub25fZXZlbnQnLFxuICAgICAgICBsYXllcnM6IFtQcmVCdW5kbGVkTGF5ZXIuZ2V0T3JDcmVhdGUoc2NvcGUsICdjb21tb24vcmVzb3VyY2VzL2xhbWJkYXMvcHJlLWJ1bmRsZWQtbGF5ZXInKV0sXG4gICAgICAgIGxhbWJkYVBvbGljeVN0YXRlbWVudHM6IGxhbWJkYVBvbGljeSxcbiAgICAgICAgbG9nUmV0ZW50aW9uOiBSZXRlbnRpb25EYXlzLk9ORV9XRUVLLFxuICAgICAgICB0aW1lb3V0OiBEdXJhdGlvbi5zZWNvbmRzKDEyMCksXG4gICAgICB9KTtcbiAgXG4gICAgICAvLyBBV1MgTGFtYmRhIGZ1bmN0aW9uIHRoYXQgY2hlY2sgdGhlIHN0YXR1cyBvZiBjb2RlYnVpbGQgcG9yamVjdFxuICAgICAgY29uc3QgaXNDb21wbGV0ZSA9IG5ldyBQcmVCdW5kbGVkRnVuY3Rpb24oc2NvcGUsICdJc0NvbXBsZXRlJywge1xuICAgICAgICAgIGNvZGVQYXRoOiAnZG9ja2VyLWJ1aWxkZXIvcmVzb3VyY2VzL2xhbWJkYXMnLFxuICAgICAgICAgIGhhbmRsZXI6ICdsYW1iZGEuaXNfY29tcGxldGUnLFxuICAgICAgICAgIGxheWVyczogW1ByZUJ1bmRsZWRMYXllci5nZXRPckNyZWF0ZShzY29wZSwgJ2NvbW1vbi9yZXNvdXJjZXMvbGFtYmRhcy9wcmUtYnVuZGxlZC1sYXllcicpXSxcbiAgICAgICAgICBsYW1iZGFQb2xpY3lTdGF0ZW1lbnRzOiBsYW1iZGFQb2xpY3ksXG4gICAgICAgICAgcnVudGltZTogUnVudGltZS5QWVRIT05fM185LFxuICAgICAgICAgIGxvZ1JldGVudGlvbjogUmV0ZW50aW9uRGF5cy5PTkVfV0VFSyxcbiAgICAgICAgICB0aW1lb3V0OiBEdXJhdGlvbi5zZWNvbmRzKDEyMCksXG4gICAgICAgIH0pO1xuICBcbiAgICAgIGNvbnN0IHByb3ZpZGVyID0gbmV3IFByb3ZpZGVyKHNjb3BlLCAnQ3VzdG9tUmVzb3VyY2VQcm92aWRlcicsIHtcbiAgICAgICAgb25FdmVudEhhbmRsZXI6IG9uRXZlbnQsXG4gICAgICAgIGlzQ29tcGxldGVIYW5kbGVyOiBpc0NvbXBsZXRlLFxuICAgICAgICB0b3RhbFRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMTUpLFxuICAgICAgICBxdWVyeUludGVydmFsOiBEdXJhdGlvbi5zZWNvbmRzKDIwKSxcbiAgICAgICAgcHJvdmlkZXJGdW5jdGlvbk5hbWU6ICdkb2NrZXJCdWlsZFB1Ymxpc2hGbicsXG4gICAgICB9KTtcblxuICAgICAgcmV0dXJuIHByb3ZpZGVyLnNlcnZpY2VUb2tlbjtcbn1cblxuLyoqXG4gKiBAaW50ZXJuYWxcbiAqIGEgTWFwIGNvbnRhaW5pbmcgdGhlIGFjY291bnQgZm9yIGVhY2ggcmVnaW9uIHdoZXJlIHRoZSBkb2NrZXIgaW1hZ2UgYXJlIHN0b3JlZFxuICogVGhlIGxpc3QgaXMgbWFpbnRhaW5lZCBpbiB0aGlzIEFXUyBkb2N1bWVudGF0aW9uIChsaW5rKVtodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9FTVItb24tRUtTLURldmVsb3BtZW50R3VpZGUvZG9ja2VyLWN1c3RvbS1pbWFnZXMtdGFnLmh0bWxdXG4gKi9cbmV4cG9ydCBjb25zdCBlbXJPbkVrc0ltYWdlTWFwID0gbmV3IE1hcChbXG4gIFsnYXAtbm9ydGhlYXN0LTEnLCBcIjA1OTAwNDUyMDE0NVwiXSwgIFxuICBbJ2FwLW5vcnRoZWFzdC0yJywgXCI5OTY1NzkyNjY4NzZcIl0sXG4gIFsnYXAtc291dGgtMScsIFwiMjM1OTE0ODY4NTc0XCJdLFxuICBbJ2FwLXNvdXRoZWFzdC0xJywgXCI2NzEyMTkxODAxOTdcIl0sXG4gIFsnYXAtc291dGhlYXN0LTInLCBcIjAzODI5Nzk5OTYwMVwiXSxcbiAgWydjYS1jZW50cmFsLTEnLCBcIjM1MTgyNjM5Mzk5OVwiXSxcbiAgWydldS1jZW50cmFsLTEnLCBcIjEwNzI5MjU1NTQ2OFwiXSxcbiAgWydldS1ub3J0aC0xJywgXCI4MzAzODY0MTYzNjRcIl0sXG4gIFsnZXUtd2VzdC0xJywgXCI0ODM3ODg1NTQ2MTlcIl0sXG4gIFsnZXUtd2VzdC0yJywgXCIxMTg3ODA2NDcyNzVcIl0sXG4gIFsnZXUtd2VzdC0zJywgXCIzMDc1MjM3MjUxNzRcIl0sXG4gIFsnc2EtZWFzdC0xJywgXCIwNTI4MDY4MzIzNThcIl0sXG4gIFsndXMtZWFzdC0xJywgXCI3NTU2NzQ4NDQyMzJcIl0sXG4gIFsndXMtZWFzdC0yJywgXCI3MTEzOTU1OTk5MzFcIl0sXG4gIFsndXMtd2VzdC0xJywgXCI2MDgwMzM0NzUzMjdcIl0sXG4gIFsndXMtd2VzdC0yJywgXCI4OTU4ODU2NjI5MzdcIl0sXG5dKTsiXX0=