"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataDomainCrawler = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const central_governance_1 = require("./central-governance");
/**
 * This CDK Construct creates a AWS Glue Crawler workflow in Data Domain account.
 * It uses AWS Step Functions state machine to orchestrate the workflow:
 * * creates and destroys the crawler upon execution
 *
 * It is triggered via Amazon EventBridge Rule upon successful execution of DataDomainWorkflow state machine {@link DataDomainWorkflow}.
 * This construct is initiatied in {@link DataDomain} but can be used as a standalone construct.
 *
 * Usage example:
 * ```typescript
 * import { App, Stack } from 'aws-cdk-lib';
 * import { DataDomainCrawler } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = cdk.App();
 * const stack = new Stack(exampleApp, 'DataProductStack');
 *
 * # See {@link DataDomain}
 *
 * new DataDomainCrawler(this, 'DataDomainCrawler', {
 *  workflowRole: workflowRole,
 *  dataProductsBucket: dataProductsBucket,
 *  domainName: 'domainName'
 * });
 * ```
 *
 */
class DataDomainCrawler extends constructs_1.Construct {
    /**
     * Construct a new instance of DataDomainCrawler.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {DataDomainCrawlerProps} props the DataDomainCrawlerProps properties
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.crawlerRole = new aws_iam_1.Role(this, 'CrawlerRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com'),
        });
        // Grant Workflow role to pass crawlerRole
        this.crawlerRole.grantPassRole(props.workflowRole);
        const baseStatements = [
            new aws_iam_1.PolicyStatement({
                actions: [
                    "s3:GetObject*",
                    "s3:GetBucket*",
                    "s3:List*",
                ],
                resources: [
                    props.dataProductsBucket.bucketArn,
                    `${props.dataProductsBucket.bucketArn}/${props.dataProductsPrefix}/*`
                ],
                effect: aws_iam_1.Effect.ALLOW,
            }),
            new aws_iam_1.PolicyStatement({
                actions: ["glue:*"],
                resources: ["*"],
                effect: aws_iam_1.Effect.ALLOW,
            }),
            new aws_iam_1.PolicyStatement({
                actions: [
                    "logs:CreateLogGroup",
                    "logs:CreateLogStream",
                    "logs:PutLogEvents"
                ],
                resources: ["arn:aws:logs:*:*:/aws-glue/*"],
                effect: aws_iam_1.Effect.ALLOW,
            }),
        ];
        var statements;
        if (props.dataProductsBucket.encryptionKey) {
            statements = baseStatements.concat([
                new aws_iam_1.PolicyStatement({
                    actions: [
                        'kms:Decrypt*',
                        'kms:Describe*',
                    ],
                    resources: [props.dataProductsBucket.encryptionKey.keyArn],
                    effect: aws_iam_1.Effect.ALLOW,
                })
            ]);
        }
        else {
            statements = baseStatements;
        }
        new aws_iam_1.ManagedPolicy(this, 'S3AccessPolicy', {
            statements: statements,
            roles: [this.crawlerRole],
        });
        // Task to grant on Db resource link to crawler role
        const grantOnDbResourceLink = new aws_stepfunctions_tasks_1.CallAwsService(this, 'grantOnDbResourceLink', {
            service: 'lakeformation',
            action: 'grantPermissions',
            iamResources: ['*'],
            parameters: {
                'Permissions': [
                    'DESCRIBE'
                ],
                'Principal': {
                    'DataLakePrincipalIdentifier': this.crawlerRole.roleArn
                },
                'Resource': {
                    'Database': {
                        'CatalogId.$': '$.account',
                        'Name.$': "States.Format('rl-{}', $.detail.central_database_name)"
                    },
                }
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD
        });
        const transformInput = new aws_stepfunctions_1.Pass(this, 'transformInput', {
            parameters: {
                'crawlerTables.$': '$.detail.table_names',
                'centralTables.$': '$.detail.table_names',
                'databaseName.$': "States.Format('rl-{}', $.detail.central_database_name)",
                'centralDatabaseName.$': '$.detail.central_database_name',
            }
        });
        grantOnDbResourceLink.next(transformInput);
        const grantOnTarget = new aws_stepfunctions_tasks_1.CallAwsService(this, 'grantOnTarget', {
            service: 'lakeformation',
            action: 'batchGrantPermissions',
            iamResources: ['*'],
            parameters: {
                'Entries': [
                    {
                        'Permissions': ['SELECT', 'INSERT', 'ALTER'],
                        'Principal': {
                            'DataLakePrincipalIdentifier': this.crawlerRole.roleArn
                        },
                        'Resource': {
                            'Table': {
                                'DatabaseName.$': '$.centralDatabaseName',
                                'Name.$': '$.targetTableName',
                                'CatalogId.$': '$.centralAccountId'
                            }
                        }
                    }
                ],
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD
        });
        const grantOnResourceLink = new aws_stepfunctions_tasks_1.CallAwsService(this, 'grantOnResourceLink', {
            service: 'lakeformation',
            action: 'grantPermissions',
            iamResources: ['*'],
            parameters: {
                'Permissions': [
                    'ALL'
                ],
                'Principal': {
                    'DataLakePrincipalIdentifier': this.crawlerRole.roleArn
                },
                'Resource': {
                    'Table': {
                        'DatabaseName.$': '$.databaseName',
                        'Name.$': '$.rlName'
                    },
                }
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD
        });
        const nracGrantsForEachTable = new aws_stepfunctions_1.Map(this, 'nracGrantsForEachTable', {
            itemsPath: '$.detail.table_names',
            maxConcurrency: 2,
            parameters: {
                'targetTableName.$': "$$.Map.Item.Value",
                'rlName.$': "States.Format('rl-{}', $$.Map.Item.Value)",
                'databaseName.$': '$.detail.database_name',
                'centralDatabaseName.$': '$.detail.central_database_name',
                'centralAccountId.$': '$.detail.central_account_id',
            },
            resultSelector: {
                'crawlerTables.$': '$[*].rlName',
                'databaseName.$': '$[0].databaseName',
                'centralTables.$': '$[*].targetTableName',
                'centralDatabaseName.$': '$[0].centralDatabaseName'
            }
        });
        // Task to check LF Access mode (TBAC or NRAC)
        const checkLfAccessMode = new aws_stepfunctions_1.Choice(this, 'checkLfAccessMode')
            .when(aws_stepfunctions_1.Condition.stringEquals('$.detail.lf_access_mode', central_governance_1.LfAccessControlMode.TBAC), grantOnDbResourceLink)
            .otherwise(nracGrantsForEachTable);
        const createCrawlersForEachTable = new aws_stepfunctions_1.Map(this, 'createCrawlersForEachTable', {
            itemsPath: '$.crawlerTables',
            maxConcurrency: 2,
            parameters: {
                'tableName.$': '$$.Map.Item.Value',
                'centralTableName.$': "States.ArrayGetItem($.centralTables, $$.Map.Item.Index)",
                'databaseName.$': '$.databaseName',
                'centralDatabaseName.$': '$.centralDatabaseName',
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD
        });
        grantOnResourceLink.next(new aws_stepfunctions_1.Wait(this, 'waitRlGrant', {
            time: aws_stepfunctions_1.WaitTime.duration(aws_cdk_lib_1.Duration.seconds(15))
        })).next(grantOnTarget);
        const createCrawler = new aws_stepfunctions_tasks_1.CallAwsService(this, 'createCrawler', {
            service: 'glue',
            action: 'createCrawler',
            iamResources: ['*'],
            parameters: {
                'Name.$': "States.Format('{}_{}_{}', $$.Execution.Id, $.databaseName, $.tableName)",
                'Role': this.crawlerRole.roleArn,
                'Targets': {
                    'CatalogTargets': [
                        {
                            'DatabaseName.$': '$.databaseName',
                            'Tables.$': 'States.Array($.tableName)'
                        }
                    ]
                },
                'SchemaChangePolicy': {
                    'DeleteBehavior': 'LOG',
                    'UpdateBehavior': 'UPDATE_IN_DATABASE'
                }
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD
        });
        const startCrawler = new aws_stepfunctions_tasks_1.CallAwsService(this, 'StartCrawler', {
            service: 'glue',
            action: 'startCrawler',
            iamResources: ['*'],
            parameters: {
                'Name.$': "States.Format('{}_{}_{}', $$.Execution.Id, $.databaseName, $.tableName)"
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD
        });
        const waitForCrawler = new aws_stepfunctions_1.Wait(this, 'WaitForCrawler', {
            time: aws_stepfunctions_1.WaitTime.duration(aws_cdk_lib_1.Duration.seconds(15))
        });
        const getCrawler = new aws_stepfunctions_tasks_1.CallAwsService(this, 'GetCrawler', {
            service: 'glue',
            action: 'getCrawler',
            iamResources: ['*'],
            parameters: {
                'Name.$': "States.Format('{}_{}_{}', $$.Execution.Id, $.databaseName, $.tableName)"
            },
            resultPath: '$.crawlerInfo'
        });
        const deleteCrawler = new aws_stepfunctions_tasks_1.CallAwsService(this, 'DeleteCrawler', {
            service: 'glue',
            action: 'deleteCrawler',
            iamResources: ['*'],
            parameters: {
                'Name.$': "States.Format('{}_{}_{}', $$.Execution.Id, $.databaseName, $.tableName)"
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD
        });
        // Forward crawler state to Central account
        const crawlerStatusEvent = new aws_stepfunctions_tasks_1.EventBridgePutEvents(this, 'crawlerStatusEvent', {
            entries: [{
                    detail: aws_stepfunctions_1.TaskInput.fromObject({
                        'dbName': aws_stepfunctions_1.JsonPath.stringAt("$.centralDatabaseName"),
                        'tableName': aws_stepfunctions_1.JsonPath.stringAt("$.centralTableName"),
                        'state': aws_stepfunctions_1.JsonPath.stringAt("$.crawlerInfo.Crawler.State"),
                        'crawlerInfo': aws_stepfunctions_1.JsonPath.stringAt("$.crawlerInfo.Crawler"),
                    }),
                    detailType: 'data-domain-crawler-update',
                    eventBus: props.eventBus,
                    source: 'data-domain-state-change',
                }],
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        deleteCrawler.endStates;
        const checkCrawlerStatusChoice = new aws_stepfunctions_1.Choice(this, 'CheckCrawlerStatusChoice')
            .when(aws_stepfunctions_1.Condition.stringEquals("$.crawlerInfo.Crawler.State", "READY"), deleteCrawler)
            .otherwise(waitForCrawler);
        createCrawler
            .next(startCrawler)
            .next(waitForCrawler)
            .next(getCrawler)
            .next(crawlerStatusEvent)
            .next(checkCrawlerStatusChoice);
        const initState = new aws_stepfunctions_1.Wait(this, 'WaitForMetadata', {
            time: aws_stepfunctions_1.WaitTime.duration(aws_cdk_lib_1.Duration.seconds(15))
        });
        createCrawlersForEachTable.iterator(new aws_stepfunctions_1.Wait(this, 'waitForGrants', {
            time: aws_stepfunctions_1.WaitTime.duration(aws_cdk_lib_1.Duration.seconds(15))
        }).next(createCrawler)).endStates;
        nracGrantsForEachTable.iterator(grantOnResourceLink).next(createCrawlersForEachTable);
        transformInput.next(createCrawlersForEachTable);
        initState.next(checkLfAccessMode);
        // Create Log group for this state machine
        const logGroup = new aws_logs_1.LogGroup(this, 'Crawler', {
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
            logGroupName: '/aws/vendedlogs/data-mesh/crawler',
        });
        logGroup.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
        const stateMachine = new aws_stepfunctions_1.StateMachine(this, 'UpdateTableSchemas', {
            definition: initState,
            role: props.workflowRole,
            logs: {
                destination: logGroup,
                level: aws_stepfunctions_1.LogLevel.ALL,
            },
        });
        this.stateMachine = new aws_events_targets_1.SfnStateMachine(stateMachine);
    }
}
exports.DataDomainCrawler = DataDomainCrawler;
//# sourceMappingURL=data:application/json;base64,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