import { Bucket } from 'aws-cdk-lib/aws-s3';
import { Construct } from 'constructs';
import { TrackedConstruct } from './common/tracked-construct';
/**
 * Properties for the DataLakeStorage Construct
 */
export interface DataLakeStorageProps {
    /**
     * Delay (in days) before moving RAW data to cold storage (Infrequent Access storage class)
     * @default -  Move objects to Infrequent Access after 30 days
     */
    readonly rawInfrequentAccessDelay?: number;
    /**
     * Delay (in days) before archiving RAW data to frozen storage (Glacier storage class)
     * @default -  Move objects to Glacier after 90 days
     */
    readonly rawArchiveDelay?: number;
    /**
     * Delay (in days) before moving CLEAN data to cold storage (Infrequent Access storage class)
     * @default -  Move objects to Infrequent Access after 90 days
     */
    readonly cleanInfrequentAccessDelay?: number;
    /**
     *
     * Delay (in days) before archiving CLEAN data to frozen storage (Glacier storage class)
     * @default -  Objects are not archived to Glacier
     */
    readonly cleanArchiveDelay?: number;
    /**
     * Delay (in days) before moving TRANSFORM data to cold storage (Infrequent Access storage class)
     * @default -  Move objects to Infrequent Access after 90 days
     */
    readonly transformInfrequentAccessDelay?: number;
    /**
     * Delay (in days) before archiving TRANSFORM data to frozen storage (Glacier storage class)
     * @default -  Objects are not archived to Glacier
     */
    readonly transformArchiveDelay?: number;
}
/**
 * A CDK Construct that creates the storage layers of a data lake composed of Amazon S3 Buckets.
 *
 * This construct is based on 3 Amazon S3 buckets configured with AWS best practices:
 *  * S3 buckets for Raw/Cleaned/Transformed data,
 *  * data lifecycle optimization/transitioning to different Amazon S3 storage classes
 *  * server side buckets encryption managed by KMS customer key
 *  * Default single KMS key
 *  * SSL communication enforcement
 *  * access logged to an S3 bucket
 *  * All public access blocked
 *
 * By default the transitioning rules to Amazon S3 storage classes are configured as following:
 *  * Raw data is moved to Infrequent Access after 30 days and archived to Glacier after 90 days
 *  * Clean and Transformed data is moved to Infrequent Access after 90 days and is not archived
 *
 * Objects and buckets are automatically deleted when the CDK application is detroyed.
 *
 * For custom requirements, consider using {@link AraBucket}.
 *
 * Usage example:
 * ```typescript
 * import * as cdk from 'aws-cdk-lib';
 * import { DataLakeStorage } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new cdk.App();
 * const stack = new cdk.Stack(exampleApp, 'DataLakeStorageStack');
 *
 * new DataLakeStorage(stack, 'MyDataLakeStorage', {
 *  rawInfrequentAccessDelay: 90,
 *  rawArchiveDelay: 180,
 *  cleanInfrequentAccessDelay: 180,
 *  cleanArchiveDelay: 360,
 *  transformInfrequentAccessDelay: 180,
 *  transformArchiveDelay: 360,
 * });
 * ```
 */
export declare class DataLakeStorage extends TrackedConstruct {
    readonly rawBucket: Bucket;
    readonly cleanBucket: Bucket;
    readonly transformBucket: Bucket;
    /**
     * Construct a new instance of DataLakeStorage based on Amazon S3 buckets with best practices configuration
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {DataLakeStorageProps} props the DataLakeStorageProps properties
     * @access public
     */
    constructor(scope: Construct, id: string, props?: DataLakeStorageProps);
}
