"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.PreBundledPysparkJobExecutable = void 0;
const path = require("path");
const aws_glue_alpha_1 = require("@aws-cdk/aws-glue-alpha");
/**
 * A Glue JobExecutable with prebundled code that can be used to create Glue Jobs.
 *
 * It changes of the code path by based on the environment that `cdk synth` is running on.
 *
 * This class is used together with a Projen custom task "copy-resources".
 * The task will ensure that the Glue script is available in "lib" folder, with the same relative path.
 *
 * When this construct is being run in JSII, this file will be in `node_modules` folder
 * (as it's installed as a 3rd party library.) So we need to change reference based on __dirname.
 *
 *  * Usage example:
 * ```typescript
 * import { PreBundledPysparkJobExecutable } from 'aws-analytics-reference-architecture';
 * import { Job } from '@aws-cdk/aws-glue-alpha';
 *
 * new glue.Job(this, 'PythonShellJob', {
 *   executable: PreBundledPysparkJobExecutable.pythonEtl({
 *     glueVersion: glue.GlueVersion.V3_0,
 *     pythonVersion: glue.PythonVersion.THREE,
 *     codePath: 'construct-dir/resources/glue/script.py',
 *   }),
 *   description: 'an example PySpark job with bundled script',
 * });
 * ```
 */
class PreBundledPysparkJobExecutable {
    static pythonEtl(props) {
        if (props.script) {
            throw new Error('Use "codePath" prop instead of "script" to ensure the code is bundled into the Glue job');
        }
        let jobProps = { ...props };
        // __dirname is where this file is. In JSII, it is <jsii_tmp_path>/lib/common.
        // When running unit tests, it is ./src/common). In both case, we need to go up one level.
        let assetPath = path.join(__dirname, `../${props.codePath}`);
        jobProps.script = aws_glue_alpha_1.Code.fromAsset(assetPath);
        delete jobProps.codePath;
        return aws_glue_alpha_1.JobExecutable.pythonEtl(jobProps);
    }
}
exports.PreBundledPysparkJobExecutable = PreBundledPysparkJobExecutable;
//# sourceMappingURL=data:application/json;base64,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