"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.directorySnapshot = exports.synthSnapshot = void 0;
const os = require("os");
const path = require("path");
const JSONC = require("comment-json");
const comment_json_1 = require("comment-json");
const fs = require("fs-extra");
const glob_1 = require("glob");
const json_1 = require("../json");
/**
 * Creates a snapshot of the files generated by a project. Ignores any non-text
 * files so that the snapshots are human readable.
 */
function synthSnapshot(project, options = {}) {
    // defensive: verify that "outdir" is actually in a temporary directory
    if (!path.resolve(project.outdir).startsWith(os.tmpdir()) &&
        !project.outdir.includes("project-temp-dir")) {
        throw new Error("Trying to capture a snapshot of a project outside of tmpdir, which implies this test might corrupt an existing project");
    }
    const synthed = Symbol.for("synthed");
    if (synthed in project) {
        throw new Error("duplicate synth()");
    }
    project[synthed] = true;
    const ENV_PROJEN_DISABLE_POST = process.env.PROJEN_DISABLE_POST;
    try {
        process.env.PROJEN_DISABLE_POST = "true";
        project.synth();
        const ignoreExts = ["png", "ico"];
        return directorySnapshot(project.outdir, {
            ...options,
            excludeGlobs: ignoreExts.map((ext) => `**/*.${ext}`),
            supportJsonComments: project.files.some(
            // At least one json file in project supports comments
            (file) => file instanceof json_1.JsonFile && file.supportsComments),
        });
    }
    finally {
        fs.removeSync(project.outdir);
        // values assigned to process.env.XYZ are automatically converted to strings
        if (ENV_PROJEN_DISABLE_POST === undefined) {
            delete process.env.PROJEN_DISABLE_POST;
        }
        else {
            process.env.PROJEN_DISABLE_POST = ENV_PROJEN_DISABLE_POST;
        }
    }
}
exports.synthSnapshot = synthSnapshot;
function isJsonLikeFile(filePath) {
    const file = filePath.toLowerCase();
    return (file.endsWith(".json") || file.endsWith(".json5") || file.endsWith(".jsonc"));
}
function directorySnapshot(root, options = {}) {
    const output = {};
    const files = glob_1.glob.sync("**", {
        ignore: [".git/**", ...(options.excludeGlobs ?? [])],
        cwd: root,
        nodir: true,
        dot: true,
    }); // returns relative file paths with POSIX separators
    const parseJson = options.parseJson ?? true;
    for (const file of files) {
        const filePath = path.join(root, file);
        let content;
        if (!options.onlyFileNames) {
            content = fs.readFileSync(filePath, "utf-8");
            if (parseJson && isJsonLikeFile(filePath)) {
                content = cleanCommentArrays(JSONC.parse(content, undefined, !options.supportJsonComments));
            }
        }
        else {
            content = true;
        }
        output[file] = content;
    }
    return output;
}
exports.directorySnapshot = directorySnapshot;
/**
 * Converts type "CommentArray" back to regular JS "Array"
 * if there are no comments stored in it.
 * Prevents strict checks from failing.
 */
function cleanCommentArrays(obj) {
    if (Array.isArray(obj) || isCommentArrayWithoutComments(obj)) {
        return Array.from(obj).map(cleanCommentArrays);
    }
    if (obj instanceof Object) {
        for (const p of Object.keys(obj)) {
            if (isCommentArrayWithoutComments(obj[p])) {
                obj[p] = Array.from(obj[p]).map(cleanCommentArrays);
            }
            else if (obj[p] instanceof Object) {
                obj[p] = cleanCommentArrays(obj[p]);
            }
        }
    }
    return obj;
}
/**
 * Checks if a "CommentArray" has no comments stored in it.
 */
function isCommentArrayWithoutComments(obj) {
    return (obj instanceof comment_json_1.CommentArray &&
        Object.getOwnPropertySymbols(obj).length === 0);
}
//# sourceMappingURL=data:application/json;base64,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