"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeProject = exports.AutoRelease = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const bundler_1 = require("./bundler");
const jest_1 = require("./jest");
const node_package_1 = require("./node-package");
const projenrc_1 = require("./projenrc");
const build_1 = require("../build");
const common_1 = require("../common");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const util_1 = require("../github/util");
const workflows_model_1 = require("../github/workflows-model");
const ignore_file_1 = require("../ignore-file");
const javascript_1 = require("../javascript");
const license_1 = require("../license");
const release_1 = require("../release");
const util_2 = require("../util");
const version_1 = require("../version");
const PROJEN_SCRIPT = "projen";
/**
 * Automatic bump modes
 */
var AutoRelease;
(function (AutoRelease) {
    /**
     * Automatically bump & release a new version for every commit to "main"
     */
    AutoRelease[AutoRelease["EVERY_COMMIT"] = 0] = "EVERY_COMMIT";
    /**
     * Automatically bump & release a new version on a daily basis.
     */
    AutoRelease[AutoRelease["DAILY"] = 1] = "DAILY";
})(AutoRelease = exports.AutoRelease || (exports.AutoRelease = {}));
/**
 * Node.js project
 *
 * @pjid node
 */
class NodeProject extends github_1.GitHubProject {
    constructor(options) {
        super(options);
        this.package = new node_package_1.NodePackage(this, options);
        this.workflowBootstrapSteps = options.workflowBootstrapSteps ?? [];
        this.workflowGitIdentity =
            options.workflowGitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.artifactsDirectory = options.artifactsDirectory ?? "dist";
        this.artifactsJavascriptDirectory = path_1.join(this.artifactsDirectory, "js");
        this.runScriptCommand = (() => {
            switch (this.packageManager) {
                case node_package_1.NodePackageManager.NPM:
                    return "npm run";
                case node_package_1.NodePackageManager.YARN:
                case node_package_1.NodePackageManager.YARN2:
                    return "yarn run";
                case node_package_1.NodePackageManager.PNPM:
                    return "pnpm run";
                default:
                    throw new Error(`unexpected package manager ${this.packageManager}`);
            }
        })();
        this.nodeVersion =
            options.workflowNodeVersion ?? this.package.minNodeVersion;
        // add PATH for all tasks which includes the project's npm .bin list
        this.tasks.addEnvironment("PATH", '$(npx -c "node -e \\"console.log(process.env.PATH)\\"")');
        this.addLicense(options);
        if (options.npmignoreEnabled ?? true) {
            this.npmignore = new ignore_file_1.IgnoreFile(this, ".npmignore");
        }
        this.addDefaultGitIgnore();
        if (options.gitignore?.length) {
            for (const i of options.gitignore) {
                this.gitignore.exclude(i);
            }
        }
        if (options.npmignore?.length) {
            if (!this.npmignore) {
                throw new Error('.npmignore is not defined for an APP project type. Add "npmIgnore: true" to override this');
            }
            for (const i of options.npmignore) {
                this.npmignore.exclude(i);
            }
        }
        if (!this.ejected) {
            this.setScript(PROJEN_SCRIPT, this.package.projenCommand);
        }
        this.npmignore?.exclude(`/${common_1.PROJEN_RC}`);
        this.npmignore?.exclude(`/${common_1.PROJEN_DIR}/`);
        this.gitignore.include(`/${common_1.PROJEN_RC}`);
        const projen = options.projenDevDependency ?? true;
        if (projen && !this.ejected) {
            const postfix = options.projenVersion ? `@${options.projenVersion}` : "";
            this.addDevDeps(`projen${postfix}`);
        }
        if (!options.defaultReleaseBranch) {
            throw new Error('"defaultReleaseBranch" is temporarily a required option while we migrate its default value from "master" to "main"');
        }
        const buildEnabled = options.buildWorkflow ?? (this.parent ? false : true);
        // configure jest if enabled
        // must be before the build/release workflows
        if (options.jest ?? true) {
            this.jest = new jest_1.Jest(this, options.jestOptions);
        }
        const requiresIdTokenPermission = (options.scopedPackagesOptions ?? []).length > 0 &&
            options.codeArtifactOptions?.authProvider ===
                release_1.CodeArtifactAuthProvider.GITHUB_OIDC;
        const workflowPermissions = {
            idToken: requiresIdTokenPermission ? workflows_model_1.JobPermission.WRITE : undefined,
        };
        if (buildEnabled && this.github) {
            this.buildWorkflow = new build_1.BuildWorkflow(this, {
                buildTask: this.buildTask,
                artifactsDirectory: this.artifactsDirectory,
                containerImage: options.workflowContainerImage,
                gitIdentity: this.workflowGitIdentity,
                mutableBuild: options.mutableBuild,
                preBuildSteps: this.renderWorkflowSetup({
                    mutable: options.mutableBuild ?? true,
                }),
                postBuildSteps: options.postBuildSteps,
                runsOn: options.workflowRunsOn,
                workflowTriggers: options.buildWorkflowTriggers,
                permissions: workflowPermissions,
            });
            this.buildWorkflow.addPostBuildSteps(...this.renderUploadCoverageJobStep(options));
        }
        const release = options.release ??
            options.releaseWorkflow ??
            (this.parent ? false : true);
        if (release) {
            this.addDevDeps(version_1.Version.STANDARD_VERSION);
            this.release = new release_1.Release(this, {
                versionFile: "package.json",
                task: this.buildTask,
                branch: options.defaultReleaseBranch ?? "main",
                artifactsDirectory: this.artifactsDirectory,
                ...options,
                releaseWorkflowSetupSteps: [
                    ...this.renderWorkflowSetup({ mutable: false }),
                    ...(options.releaseWorkflowSetupSteps ?? []),
                ],
                postBuildSteps: [
                    ...(options.postBuildSteps ?? []),
                    ...this.renderUploadCoverageJobStep(options),
                ],
                workflowNodeVersion: this.nodeVersion,
                workflowPermissions,
            });
            this.publisher = this.release.publisher;
            const nodePackageToReleaseCodeArtifactAuthProviderMapping = {
                [node_package_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR]: release_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR,
                [node_package_1.CodeArtifactAuthProvider.GITHUB_OIDC]: release_1.CodeArtifactAuthProvider.GITHUB_OIDC,
            };
            if (options.releaseToNpm ?? false) {
                const codeArtifactOptions = release_1.isAwsCodeArtifactRegistry(this.package.npmRegistry)
                    ? {
                        accessKeyIdSecret: options.codeArtifactOptions?.accessKeyIdSecret,
                        secretAccessKeySecret: options.codeArtifactOptions?.secretAccessKeySecret,
                        roleToAssume: options.codeArtifactOptions?.roleToAssume,
                        authProvider: options.codeArtifactOptions?.authProvider
                            ? nodePackageToReleaseCodeArtifactAuthProviderMapping[options.codeArtifactOptions.authProvider]
                            : release_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR,
                    }
                    : {};
                this.release.publisher.publishToNpm({
                    registry: this.package.npmRegistry,
                    npmTokenSecret: this.package.npmTokenSecret,
                    codeArtifactOptions,
                });
            }
        }
        else {
            // validate that no release options are selected if the release workflow is disabled.
            if (options.releaseToNpm) {
                throw new Error('"releaseToNpm" is not supported if "release" is not set');
            }
            if (options.releaseEveryCommit) {
                throw new Error('"releaseEveryCommit" is not supported if "release" is not set');
            }
            if (options.releaseSchedule) {
                throw new Error('"releaseSchedule" is not supported if "release" is not set');
            }
        }
        if ((options.autoMerge ?? true) &&
            this.github?.mergify &&
            this.buildWorkflow?.buildJobIds) {
            this.autoMerge = new github_1.AutoMerge(this.github, options.autoMergeOptions);
            this.autoMerge.addConditionsLater({
                render: () => this.buildWorkflow?.buildJobIds.map((id) => `status-success=${id}`) ??
                    [],
            });
        }
        const dependabot = options.dependabot ?? false;
        const depsUpgrade = options.depsUpgrade ?? !dependabot;
        if (dependabot && depsUpgrade) {
            throw new Error("'dependabot' cannot be configured together with 'depsUpgrade'");
        }
        const depsAutoApprove = options.autoApproveUpgrades ?? false;
        if (depsAutoApprove && !this.autoApprove && this.github) {
            throw new Error("Automatic approval of dependencies upgrades requires configuring `autoApproveOptions`");
        }
        const autoApproveLabel = (condition) => condition && this.autoApprove?.label
            ? [this.autoApprove.label]
            : undefined;
        if (dependabot) {
            const defaultOptions = {
                labels: autoApproveLabel(depsAutoApprove),
            };
            this.github?.addDependabot(util_2.deepMerge([defaultOptions, options.dependabotOptions ?? {}]));
        }
        if (depsUpgrade) {
            const defaultOptions = {
                workflowOptions: {
                    container: options.workflowContainerImage
                        ? {
                            image: options.workflowContainerImage,
                        }
                        : undefined,
                    labels: autoApproveLabel(depsAutoApprove),
                    gitIdentity: this.workflowGitIdentity,
                },
            };
            this.upgradeWorkflow = new javascript_1.UpgradeDependencies(this, util_2.deepMerge([defaultOptions, options.depsUpgradeOptions ?? {}]));
        }
        if (options.pullRequestTemplate ?? true) {
            this.github?.addPullRequestTemplate(...(options.pullRequestTemplateContents ?? []));
        }
        const projenrcJs = options.projenrcJs ?? !options.projenrcJson;
        if (projenrcJs) {
            new projenrc_1.Projenrc(this, options.projenrcJsOptions);
        }
        // add a bundler component - this enables things like Lambda bundling and in the future web bundling.
        this.bundler = new bundler_1.Bundler(this, options.bundlerOptions);
        if (options.package ?? true) {
            this.packageTask.exec(`mkdir -p ${this.artifactsJavascriptDirectory}`);
            // always use npm here - yarn doesn't add much value
            // sadly we cannot use --pack-destination because it is not supported by older npm
            this.packageTask.exec(`mv $(npm pack) ${this.artifactsJavascriptDirectory}/`);
        }
        if (options.prettier ?? false) {
            this.prettier = new javascript_1.Prettier(this, { ...options.prettierOptions });
        }
    }
    /**
     * @deprecated use `package.allowLibraryDependencies`
     */
    get allowLibraryDependencies() {
        return this.package.allowLibraryDependencies;
    }
    /**
     * @deprecated use `package.entrypoint`
     */
    get entrypoint() {
        return this.package.entrypoint;
    }
    /**
     * Minimum node.js version required by this package.
     */
    get minNodeVersion() {
        return this.package.minNodeVersion;
    }
    /**
     * Maximum node version required by this package.
     */
    get maxNodeVersion() {
        return this.package.maxNodeVersion;
    }
    /**
     * The package manager to use.
     *
     * @deprecated use `package.packageManager`
     */
    get packageManager() {
        return this.package.packageManager;
    }
    /**
     * @deprecated use `package.addField(x, y)`
     */
    get manifest() {
        return this.package.manifest;
    }
    renderUploadCoverageJobStep(options) {
        // run codecov if enabled or a secret token name is passed in
        // AND jest must be configured
        if ((options.codeCov || options.codeCovTokenSecret) && this.jest?.config) {
            return [
                {
                    name: "Upload coverage to Codecov",
                    uses: "codecov/codecov-action@v3",
                    with: options.codeCovTokenSecret
                        ? {
                            token: `\${{ secrets.${options.codeCovTokenSecret} }}`,
                            directory: this.jest.config.coverageDirectory,
                        }
                        : {
                            directory: this.jest.config.coverageDirectory,
                        },
                },
            ];
        }
        else {
            return [];
        }
    }
    addBins(bins) {
        this.package.addBin(bins);
    }
    /**
     * Replaces the contents of an npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name, command) {
        this.package.setScript(name, command);
    }
    /**
     * Removes the npm script (always successful).
     * @param name The name of the script.
     */
    removeScript(name) {
        this.package.removeScript(name);
    }
    /**
     * Indicates if a script by the name name is defined.
     * @param name The name of the script
     */
    hasScript(name) {
        return this.package.hasScript(name);
    }
    /**
     * DEPRECATED
     * @deprecated use `project.compileTask.exec()`
     */
    addCompileCommand(...commands) {
        for (const c of commands) {
            this.compileTask.exec(c);
        }
    }
    /**
     * DEPRECATED
     * @deprecated use `project.testTask.exec()`
     */
    addTestCommand(...commands) {
        for (const c of commands) {
            this.testTask.exec(c);
        }
    }
    /**
     * Directly set fields in `package.json`.
     * @param fields The fields to set
     */
    addFields(fields) {
        for (const [name, value] of Object.entries(fields)) {
            this.package.addField(name, value);
        }
    }
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords) {
        this.package.addKeywords(...keywords);
    }
    /**
     * Get steps for scoped package access
     *
     * @param codeArtifactOptions Details of logging in to AWS
     * @returns array of job steps required for each private scoped packages
     */
    getScopedPackageSteps(codeArtifactOptions) {
        const parsedCodeArtifactOptions = {
            accessKeyIdSecret: codeArtifactOptions?.accessKeyIdSecret ?? "AWS_ACCESS_KEY_ID",
            secretAccessKeySecret: codeArtifactOptions?.secretAccessKeySecret ?? "AWS_SECRET_ACCESS_KEY",
            roleToAssume: codeArtifactOptions?.roleToAssume,
            authProvider: codeArtifactOptions?.authProvider,
        };
        if (parsedCodeArtifactOptions.authProvider ===
            node_package_1.CodeArtifactAuthProvider.GITHUB_OIDC) {
            return [
                {
                    name: "Configure AWS Credentials",
                    uses: "aws-actions/configure-aws-credentials@v1",
                    with: {
                        "aws-region": "us-east-2",
                        "role-to-assume": parsedCodeArtifactOptions.roleToAssume,
                        "role-duration-seconds": 900,
                    },
                },
                {
                    name: "AWS CodeArtifact Login",
                    run: `${this.runScriptCommand} ca:login`,
                },
            ];
        }
        if (parsedCodeArtifactOptions.roleToAssume) {
            return [
                {
                    name: "Configure AWS Credentials",
                    uses: "aws-actions/configure-aws-credentials@v1",
                    with: {
                        "aws-access-key-id": util_1.secretToString(parsedCodeArtifactOptions.accessKeyIdSecret),
                        "aws-secret-access-key": util_1.secretToString(parsedCodeArtifactOptions.secretAccessKeySecret),
                        "aws-region": "us-east-2",
                        "role-to-assume": parsedCodeArtifactOptions.roleToAssume,
                        "role-duration-seconds": 900,
                    },
                },
                {
                    name: "AWS CodeArtifact Login",
                    run: `${this.runScriptCommand} ca:login`,
                },
            ];
        }
        return [
            {
                name: "AWS CodeArtifact Login",
                run: `${this.runScriptCommand} ca:login`,
                env: {
                    AWS_ACCESS_KEY_ID: util_1.secretToString(parsedCodeArtifactOptions.accessKeyIdSecret),
                    AWS_SECRET_ACCESS_KEY: util_1.secretToString(parsedCodeArtifactOptions.secretAccessKeySecret),
                },
            },
        ];
    }
    /**
     * Returns the set of workflow steps which should be executed to bootstrap a
     * workflow.
     *
     * @param options Options.
     * @returns Job steps
     */
    renderWorkflowSetup(options = {}) {
        const install = new Array();
        // first run the workflow bootstrap steps
        install.push(...this.workflowBootstrapSteps);
        if (this.nodeVersion) {
            install.push({
                name: "Setup Node.js",
                uses: "actions/setup-node@v3",
                with: { "node-version": this.nodeVersion },
            });
        }
        if (this.package.packageManager === node_package_1.NodePackageManager.PNPM) {
            install.push({
                name: "Setup pnpm",
                uses: "pnpm/action-setup@v2.2.4",
                with: { version: this.package.pnpmVersion },
            });
        }
        const mutable = options.mutable ?? false;
        if (this.package.scopedPackagesOptions) {
            install.push(...this.getScopedPackageSteps(this.package.codeArtifactOptions));
        }
        install.push({
            name: "Install dependencies",
            run: mutable
                ? this.package.installAndUpdateLockfileCommand
                : this.package.installCommand,
        });
        return install;
    }
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps) {
        return this.package.addDeps(...deps);
    }
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps) {
        return this.package.addDevDeps(...deps);
    }
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps) {
        return this.package.addPeerDeps(...deps);
    }
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps) {
        return this.package.addBundledDeps(...deps);
    }
    addPackageIgnore(pattern) {
        this.npmignore?.addPatterns(pattern);
    }
    addLicense(options) {
        if (this.package.license) {
            new license_1.License(this, {
                spdx: this.package.license,
                copyrightOwner: options.copyrightOwner ?? options.authorName,
                copyrightPeriod: options.copyrightPeriod,
            });
        }
    }
    addDefaultGitIgnore() {
        this.gitignore.exclude("# Logs", "logs", "*.log", "npm-debug.log*", "yarn-debug.log*", "yarn-error.log*", "lerna-debug.log*", "# Diagnostic reports (https://nodejs.org/api/report.html)", "report.[0-9]*.[0-9]*.[0-9]*.[0-9]*.json", "# Runtime data", "pids", "*.pid", "*.seed", "*.pid.lock", "# Directory for instrumented libs generated by jscoverage/JSCover", "lib-cov", "# Coverage directory used by tools like istanbul", "coverage", "*.lcov", "# nyc test coverage", ".nyc_output", "# Compiled binary addons (https://nodejs.org/api/addons.html)", "build/Release", "# Dependency directories", "node_modules/", "jspm_packages/", "# TypeScript cache", "*.tsbuildinfo", "# Optional eslint cache", ".eslintcache", "# Output of 'npm pack'", "*.tgz", "# Yarn Integrity file", ".yarn-integrity", "# parcel-bundler cache (https://parceljs.org/)", ".cache");
    }
    /**
     * Returns the shell command to execute in order to run a task. This will
     * typically be `npx projen TASK`.
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        return `${this.package.projenCommand} ${task.name}`;
    }
    /**
     * The job ID of the build workflow.
     */
    get buildWorkflowJobId() {
        return this.buildWorkflow?.buildJobIds[0];
    }
}
exports.NodeProject = NodeProject;
_a = JSII_RTTI_SYMBOL_1;
NodeProject[_a] = { fqn: "projen.javascript.NodeProject", version: "0.67.87" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibm9kZS1wcm9qZWN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2phdmFzY3JpcHQvbm9kZS1wcm9qZWN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsK0JBQTRCO0FBQzVCLHVDQUFvRDtBQUNwRCxpQ0FBMkM7QUFDM0MsaURBTXdCO0FBQ3hCLHlDQUF1RDtBQUN2RCxvQ0FBeUM7QUFDekMsc0NBQWtEO0FBQ2xELHNDQU1tQjtBQUNuQixtREFBa0U7QUFDbEUseUNBQWdEO0FBQ2hELCtEQUttQztBQUNuQyxnREFBNEM7QUFDNUMsOENBS3VCO0FBQ3ZCLHdDQUFxQztBQUNyQyx3Q0FRb0I7QUFFcEIsa0NBQW9DO0FBQ3BDLHdDQUFxQztBQUVyQyxNQUFNLGFBQWEsR0FBRyxRQUFRLENBQUM7QUE4UC9COztHQUVHO0FBQ0gsSUFBWSxXQVVYO0FBVkQsV0FBWSxXQUFXO0lBQ3JCOztPQUVHO0lBQ0gsNkRBQVksQ0FBQTtJQUVaOztPQUVHO0lBQ0gsK0NBQUssQ0FBQTtBQUNQLENBQUMsRUFWVyxXQUFXLEdBQVgsbUJBQVcsS0FBWCxtQkFBVyxRQVV0QjtBQUVEOzs7O0dBSUc7QUFDSCxNQUFhLFdBQVksU0FBUSxzQkFBYTtJQWlINUMsWUFBWSxPQUEyQjtRQUNyQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFZixJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksMEJBQVcsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDOUMsSUFBSSxDQUFDLHNCQUFzQixHQUFHLE9BQU8sQ0FBQyxzQkFBc0IsSUFBSSxFQUFFLENBQUM7UUFDbkUsSUFBSSxDQUFDLG1CQUFtQjtZQUN0QixPQUFPLENBQUMsbUJBQW1CLElBQUksdUNBQTJCLENBQUM7UUFDN0QsSUFBSSxDQUFDLGtCQUFrQixHQUFHLE9BQU8sQ0FBQyxrQkFBa0IsSUFBSSxNQUFNLENBQUM7UUFDL0QsSUFBSSxDQUFDLDRCQUE0QixHQUFHLFdBQUksQ0FBQyxJQUFJLENBQUMsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFFeEUsSUFBSSxDQUFDLGdCQUFnQixHQUFHLENBQUMsR0FBRyxFQUFFO1lBQzVCLFFBQVEsSUFBSSxDQUFDLGNBQWMsRUFBRTtnQkFDM0IsS0FBSyxpQ0FBa0IsQ0FBQyxHQUFHO29CQUN6QixPQUFPLFNBQVMsQ0FBQztnQkFDbkIsS0FBSyxpQ0FBa0IsQ0FBQyxJQUFJLENBQUM7Z0JBQzdCLEtBQUssaUNBQWtCLENBQUMsS0FBSztvQkFDM0IsT0FBTyxVQUFVLENBQUM7Z0JBQ3BCLEtBQUssaUNBQWtCLENBQUMsSUFBSTtvQkFDMUIsT0FBTyxVQUFVLENBQUM7Z0JBQ3BCO29CQUNFLE1BQU0sSUFBSSxLQUFLLENBQUMsOEJBQThCLElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDO2FBQ3hFO1FBQ0gsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUVMLElBQUksQ0FBQyxXQUFXO1lBQ2QsT0FBTyxDQUFDLG1CQUFtQixJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDO1FBRTdELG9FQUFvRTtRQUNwRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FDdkIsTUFBTSxFQUNOLHlEQUF5RCxDQUMxRCxDQUFDO1FBRUYsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUV6QixJQUFJLE9BQU8sQ0FBQyxnQkFBZ0IsSUFBSSxJQUFJLEVBQUU7WUFDcEMsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLHdCQUFVLENBQUMsSUFBSSxFQUFFLFlBQVksQ0FBQyxDQUFDO1NBQ3JEO1FBRUQsSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUM7UUFFM0IsSUFBSSxPQUFPLENBQUMsU0FBUyxFQUFFLE1BQU0sRUFBRTtZQUM3QixLQUFLLE1BQU0sQ0FBQyxJQUFJLE9BQU8sQ0FBQyxTQUFTLEVBQUU7Z0JBQ2pDLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQzNCO1NBQ0Y7UUFFRCxJQUFJLE9BQU8sQ0FBQyxTQUFTLEVBQUUsTUFBTSxFQUFFO1lBQzdCLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFO2dCQUNuQixNQUFNLElBQUksS0FBSyxDQUNiLDJGQUEyRixDQUM1RixDQUFDO2FBQ0g7WUFFRCxLQUFLLE1BQU0sQ0FBQyxJQUFJLE9BQU8sQ0FBQyxTQUFTLEVBQUU7Z0JBQ2pDLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQzNCO1NBQ0Y7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUNqQixJQUFJLENBQUMsU0FBUyxDQUFDLGFBQWEsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1NBQzNEO1FBRUQsSUFBSSxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsSUFBSSxrQkFBUyxFQUFFLENBQUMsQ0FBQztRQUN6QyxJQUFJLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxJQUFJLG1CQUFVLEdBQUcsQ0FBQyxDQUFDO1FBQzNDLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLElBQUksa0JBQVMsRUFBRSxDQUFDLENBQUM7UUFFeEMsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLG1CQUFtQixJQUFJLElBQUksQ0FBQztRQUNuRCxJQUFJLE1BQU0sSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDM0IsTUFBTSxPQUFPLEdBQUcsT0FBTyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsSUFBSSxPQUFPLENBQUMsYUFBYSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztZQUN6RSxJQUFJLENBQUMsVUFBVSxDQUFDLFNBQVMsT0FBTyxFQUFFLENBQUMsQ0FBQztTQUNyQztRQUVELElBQUksQ0FBQyxPQUFPLENBQUMsb0JBQW9CLEVBQUU7WUFDakMsTUFBTSxJQUFJLEtBQUssQ0FDYixvSEFBb0gsQ0FDckgsQ0FBQztTQUNIO1FBRUQsTUFBTSxZQUFZLEdBQUcsT0FBTyxDQUFDLGFBQWEsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFM0UsNEJBQTRCO1FBQzVCLDZDQUE2QztRQUM3QyxJQUFJLE9BQU8sQ0FBQyxJQUFJLElBQUksSUFBSSxFQUFFO1lBQ3hCLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxXQUFJLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUNqRDtRQUVELE1BQU0seUJBQXlCLEdBQzdCLENBQUMsT0FBTyxDQUFDLHFCQUFxQixJQUFJLEVBQUUsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDO1lBQ2hELE9BQU8sQ0FBQyxtQkFBbUIsRUFBRSxZQUFZO2dCQUN2QyxrQ0FBd0IsQ0FBQyxXQUFXLENBQUM7UUFFekMsTUFBTSxtQkFBbUIsR0FBbUI7WUFDMUMsT0FBTyxFQUFFLHlCQUF5QixDQUFDLENBQUMsQ0FBQywrQkFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsU0FBUztTQUNyRSxDQUFDO1FBRUYsSUFBSSxZQUFZLElBQUksSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUMvQixJQUFJLENBQUMsYUFBYSxHQUFHLElBQUkscUJBQWEsQ0FBQyxJQUFJLEVBQUU7Z0JBQzNDLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUztnQkFDekIsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLGtCQUFrQjtnQkFDM0MsY0FBYyxFQUFFLE9BQU8sQ0FBQyxzQkFBc0I7Z0JBQzlDLFdBQVcsRUFBRSxJQUFJLENBQUMsbUJBQW1CO2dCQUNyQyxZQUFZLEVBQUUsT0FBTyxDQUFDLFlBQVk7Z0JBQ2xDLGFBQWEsRUFBRSxJQUFJLENBQUMsbUJBQW1CLENBQUM7b0JBQ3RDLE9BQU8sRUFBRSxPQUFPLENBQUMsWUFBWSxJQUFJLElBQUk7aUJBQ3RDLENBQUM7Z0JBQ0YsY0FBYyxFQUFFLE9BQU8sQ0FBQyxjQUFjO2dCQUN0QyxNQUFNLEVBQUUsT0FBTyxDQUFDLGNBQWM7Z0JBQzlCLGdCQUFnQixFQUFFLE9BQU8sQ0FBQyxxQkFBcUI7Z0JBQy9DLFdBQVcsRUFBRSxtQkFBbUI7YUFDakMsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLGFBQWEsQ0FBQyxpQkFBaUIsQ0FDbEMsR0FBRyxJQUFJLENBQUMsMkJBQTJCLENBQUMsT0FBTyxDQUFDLENBQzdDLENBQUM7U0FDSDtRQUVELE1BQU0sT0FBTyxHQUNYLE9BQU8sQ0FBQyxPQUFPO1lBQ2YsT0FBTyxDQUFDLGVBQWU7WUFDdkIsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQy9CLElBQUksT0FBTyxFQUFFO1lBQ1gsSUFBSSxDQUFDLFVBQVUsQ0FBQyxpQkFBTyxDQUFDLGdCQUFnQixDQUFDLENBQUM7WUFFMUMsSUFBSSxDQUFDLE9BQU8sR0FBRyxJQUFJLGlCQUFPLENBQUMsSUFBSSxFQUFFO2dCQUMvQixXQUFXLEVBQUUsY0FBYztnQkFDM0IsSUFBSSxFQUFFLElBQUksQ0FBQyxTQUFTO2dCQUNwQixNQUFNLEVBQUUsT0FBTyxDQUFDLG9CQUFvQixJQUFJLE1BQU07Z0JBQzlDLGtCQUFrQixFQUFFLElBQUksQ0FBQyxrQkFBa0I7Z0JBQzNDLEdBQUcsT0FBTztnQkFFVix5QkFBeUIsRUFBRTtvQkFDekIsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLENBQUM7b0JBQy9DLEdBQUcsQ0FBQyxPQUFPLENBQUMseUJBQXlCLElBQUksRUFBRSxDQUFDO2lCQUM3QztnQkFDRCxjQUFjLEVBQUU7b0JBQ2QsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLElBQUksRUFBRSxDQUFDO29CQUNqQyxHQUFHLElBQUksQ0FBQywyQkFBMkIsQ0FBQyxPQUFPLENBQUM7aUJBQzdDO2dCQUVELG1CQUFtQixFQUFFLElBQUksQ0FBQyxXQUFXO2dCQUNyQyxtQkFBbUI7YUFDcEIsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQztZQUV4QyxNQUFNLG1EQUFtRCxHQUdyRDtnQkFDRixDQUFDLHVDQUFtQyxDQUFDLDBCQUEwQixDQUFDLEVBQzlELGtDQUErQixDQUFDLDBCQUEwQjtnQkFDNUQsQ0FBQyx1Q0FBbUMsQ0FBQyxXQUFXLENBQUMsRUFDL0Msa0NBQStCLENBQUMsV0FBVzthQUM5QyxDQUFDO1lBRUYsSUFBSSxPQUFPLENBQUMsWUFBWSxJQUFJLEtBQUssRUFBRTtnQkFDakMsTUFBTSxtQkFBbUIsR0FDdkIsbUNBQXlCLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUM7b0JBQ2pELENBQUMsQ0FBQzt3QkFDRSxpQkFBaUIsRUFDZixPQUFPLENBQUMsbUJBQW1CLEVBQUUsaUJBQWlCO3dCQUNoRCxxQkFBcUIsRUFDbkIsT0FBTyxDQUFDLG1CQUFtQixFQUFFLHFCQUFxQjt3QkFDcEQsWUFBWSxFQUFFLE9BQU8sQ0FBQyxtQkFBbUIsRUFBRSxZQUFZO3dCQUN2RCxZQUFZLEVBQUUsT0FBTyxDQUFDLG1CQUFtQixFQUFFLFlBQVk7NEJBQ3JELENBQUMsQ0FBQyxtREFBbUQsQ0FDakQsT0FBTyxDQUFDLG1CQUFtQixDQUFDLFlBQVksQ0FDekM7NEJBQ0gsQ0FBQyxDQUFDLGtDQUErQixDQUFDLDBCQUEwQjtxQkFDL0Q7b0JBQ0gsQ0FBQyxDQUFDLEVBQUUsQ0FBQztnQkFDVCxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxZQUFZLENBQUM7b0JBQ2xDLFFBQVEsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVc7b0JBQ2xDLGNBQWMsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWM7b0JBQzNDLG1CQUFtQjtpQkFDcEIsQ0FBQyxDQUFDO2FBQ0o7U0FDRjthQUFNO1lBQ0wscUZBQXFGO1lBQ3JGLElBQUksT0FBTyxDQUFDLFlBQVksRUFBRTtnQkFDeEIsTUFBTSxJQUFJLEtBQUssQ0FDYix5REFBeUQsQ0FDMUQsQ0FBQzthQUNIO1lBRUQsSUFBSSxPQUFPLENBQUMsa0JBQWtCLEVBQUU7Z0JBQzlCLE1BQU0sSUFBSSxLQUFLLENBQ2IsK0RBQStELENBQ2hFLENBQUM7YUFDSDtZQUVELElBQUksT0FBTyxDQUFDLGVBQWUsRUFBRTtnQkFDM0IsTUFBTSxJQUFJLEtBQUssQ0FDYiw0REFBNEQsQ0FDN0QsQ0FBQzthQUNIO1NBQ0Y7UUFFRCxJQUNFLENBQUMsT0FBTyxDQUFDLFNBQVMsSUFBSSxJQUFJLENBQUM7WUFDM0IsSUFBSSxDQUFDLE1BQU0sRUFBRSxPQUFPO1lBQ3BCLElBQUksQ0FBQyxhQUFhLEVBQUUsV0FBVyxFQUMvQjtZQUNBLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxrQkFBUyxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLGdCQUFnQixDQUFDLENBQUM7WUFDdEUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxrQkFBa0IsQ0FBQztnQkFDaEMsTUFBTSxFQUFFLEdBQUcsRUFBRSxDQUNYLElBQUksQ0FBQyxhQUFhLEVBQUUsV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUMsa0JBQWtCLEVBQUUsRUFBRSxDQUFDO29CQUNuRSxFQUFFO2FBQ0wsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxNQUFNLFVBQVUsR0FBRyxPQUFPLENBQUMsVUFBVSxJQUFJLEtBQUssQ0FBQztRQUMvQyxNQUFNLFdBQVcsR0FBRyxPQUFPLENBQUMsV0FBVyxJQUFJLENBQUMsVUFBVSxDQUFDO1FBRXZELElBQUksVUFBVSxJQUFJLFdBQVcsRUFBRTtZQUM3QixNQUFNLElBQUksS0FBSyxDQUNiLCtEQUErRCxDQUNoRSxDQUFDO1NBQ0g7UUFFRCxNQUFNLGVBQWUsR0FBRyxPQUFPLENBQUMsbUJBQW1CLElBQUksS0FBSyxDQUFDO1FBRTdELElBQUksZUFBZSxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsSUFBSSxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQ3ZELE1BQU0sSUFBSSxLQUFLLENBQ2IsdUZBQXVGLENBQ3hGLENBQUM7U0FDSDtRQUVELE1BQU0sZ0JBQWdCLEdBQUcsQ0FBQyxTQUFrQixFQUFFLEVBQUUsQ0FDOUMsU0FBUyxJQUFJLElBQUksQ0FBQyxXQUFXLEVBQUUsS0FBSztZQUNsQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLEtBQUssQ0FBQztZQUMxQixDQUFDLENBQUMsU0FBUyxDQUFDO1FBRWhCLElBQUksVUFBVSxFQUFFO1lBQ2QsTUFBTSxjQUFjLEdBQUc7Z0JBQ3JCLE1BQU0sRUFBRSxnQkFBZ0IsQ0FBQyxlQUFlLENBQUM7YUFDMUMsQ0FBQztZQUNGLElBQUksQ0FBQyxNQUFNLEVBQUUsYUFBYSxDQUN4QixnQkFBUyxDQUFDLENBQUMsY0FBYyxFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUM3RCxDQUFDO1NBQ0g7UUFFRCxJQUFJLFdBQVcsRUFBRTtZQUNmLE1BQU0sY0FBYyxHQUErQjtnQkFDakQsZUFBZSxFQUFFO29CQUNmLFNBQVMsRUFBRSxPQUFPLENBQUMsc0JBQXNCO3dCQUN2QyxDQUFDLENBQUM7NEJBQ0UsS0FBSyxFQUFFLE9BQU8sQ0FBQyxzQkFBc0I7eUJBQ3RDO3dCQUNILENBQUMsQ0FBQyxTQUFTO29CQUNiLE1BQU0sRUFBRSxnQkFBZ0IsQ0FBQyxlQUFlLENBQUM7b0JBQ3pDLFdBQVcsRUFBRSxJQUFJLENBQUMsbUJBQW1CO2lCQUN0QzthQUNGLENBQUM7WUFDRixJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksZ0NBQW1CLENBQzVDLElBQUksRUFDSixnQkFBUyxDQUFDLENBQUMsY0FBYyxFQUFFLE9BQU8sQ0FBQyxrQkFBa0IsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUM5RCxDQUFDO1NBQ0g7UUFFRCxJQUFJLE9BQU8sQ0FBQyxtQkFBbUIsSUFBSSxJQUFJLEVBQUU7WUFDdkMsSUFBSSxDQUFDLE1BQU0sRUFBRSxzQkFBc0IsQ0FDakMsR0FBRyxDQUFDLE9BQU8sQ0FBQywyQkFBMkIsSUFBSSxFQUFFLENBQUMsQ0FDL0MsQ0FBQztTQUNIO1FBRUQsTUFBTSxVQUFVLEdBQUcsT0FBTyxDQUFDLFVBQVUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUM7UUFDL0QsSUFBSSxVQUFVLEVBQUU7WUFDZCxJQUFJLG1CQUFRLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1NBQy9DO1FBRUQscUdBQXFHO1FBQ3JHLElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSxpQkFBTyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsY0FBYyxDQUFDLENBQUM7UUFFekQsSUFBSSxPQUFPLENBQUMsT0FBTyxJQUFJLElBQUksRUFBRTtZQUMzQixJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxZQUFZLElBQUksQ0FBQyw0QkFBNEIsRUFBRSxDQUFDLENBQUM7WUFFdkUsb0RBQW9EO1lBQ3BELGtGQUFrRjtZQUNsRixJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FDbkIsa0JBQWtCLElBQUksQ0FBQyw0QkFBNEIsR0FBRyxDQUN2RCxDQUFDO1NBQ0g7UUFFRCxJQUFJLE9BQU8sQ0FBQyxRQUFRLElBQUksS0FBSyxFQUFFO1lBQzdCLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxxQkFBUSxDQUFDLElBQUksRUFBRSxFQUFFLEdBQUcsT0FBTyxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUM7U0FDcEU7SUFDSCxDQUFDO0lBdFlEOztPQUVHO0lBQ0gsSUFBVyx3QkFBd0I7UUFDakMsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLHdCQUF3QixDQUFDO0lBQy9DLENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsVUFBVTtRQUNuQixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDO0lBQ2pDLENBQUM7SUF5QkQ7O09BRUc7SUFDSCxJQUFXLGNBQWM7UUFDdkIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7O09BRUc7SUFDSCxJQUFXLGNBQWM7UUFDdkIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQztJQUNyQyxDQUFDO0lBSUQ7Ozs7T0FJRztJQUNILElBQVcsY0FBYztRQUN2QixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDO0lBQ3JDLENBQUM7SUFZRDs7T0FFRztJQUNILElBQVcsUUFBUTtRQUNqQixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDO0lBQy9CLENBQUM7SUEyVE8sMkJBQTJCLENBQUMsT0FBMkI7UUFDN0QsNkRBQTZEO1FBQzdELDhCQUE4QjtRQUM5QixJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sSUFBSSxPQUFPLENBQUMsa0JBQWtCLENBQUMsSUFBSSxJQUFJLENBQUMsSUFBSSxFQUFFLE1BQU0sRUFBRTtZQUN4RSxPQUFPO2dCQUNMO29CQUNFLElBQUksRUFBRSw0QkFBNEI7b0JBQ2xDLElBQUksRUFBRSwyQkFBMkI7b0JBQ2pDLElBQUksRUFBRSxPQUFPLENBQUMsa0JBQWtCO3dCQUM5QixDQUFDLENBQUM7NEJBQ0UsS0FBSyxFQUFFLGdCQUFnQixPQUFPLENBQUMsa0JBQWtCLEtBQUs7NEJBQ3RELFNBQVMsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxpQkFBaUI7eUJBQzlDO3dCQUNILENBQUMsQ0FBQzs0QkFDRSxTQUFTLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsaUJBQWlCO3lCQUM5QztpQkFDTjthQUNGLENBQUM7U0FDSDthQUFNO1lBQ0wsT0FBTyxFQUFFLENBQUM7U0FDWDtJQUNILENBQUM7SUFFTSxPQUFPLENBQUMsSUFBNEI7UUFDekMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDNUIsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksU0FBUyxDQUFDLElBQVksRUFBRSxPQUFlO1FBQzVDLElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksWUFBWSxDQUFDLElBQVk7UUFDOUIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFNBQVMsQ0FBQyxJQUFZO1FBQzNCLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7T0FHRztJQUNJLGlCQUFpQixDQUFDLEdBQUcsUUFBa0I7UUFDNUMsS0FBSyxNQUFNLENBQUMsSUFBSSxRQUFRLEVBQUU7WUFDeEIsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDMUI7SUFDSCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksY0FBYyxDQUFDLEdBQUcsUUFBa0I7UUFDekMsS0FBSyxNQUFNLENBQUMsSUFBSSxRQUFRLEVBQUU7WUFDeEIsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDdkI7SUFDSCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksU0FBUyxDQUFDLE1BQStCO1FBQzlDLEtBQUssTUFBTSxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQ2xELElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztTQUNwQztJQUNILENBQUM7SUFFRDs7O09BR0c7SUFDSSxXQUFXLENBQUMsR0FBRyxRQUFrQjtRQUN0QyxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxHQUFHLFFBQVEsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNLLHFCQUFxQixDQUMzQixtQkFBb0Q7UUFFcEQsTUFBTSx5QkFBeUIsR0FBRztZQUNoQyxpQkFBaUIsRUFDZixtQkFBbUIsRUFBRSxpQkFBaUIsSUFBSSxtQkFBbUI7WUFDL0QscUJBQXFCLEVBQ25CLG1CQUFtQixFQUFFLHFCQUFxQixJQUFJLHVCQUF1QjtZQUN2RSxZQUFZLEVBQUUsbUJBQW1CLEVBQUUsWUFBWTtZQUMvQyxZQUFZLEVBQUUsbUJBQW1CLEVBQUUsWUFBWTtTQUNoRCxDQUFDO1FBRUYsSUFDRSx5QkFBeUIsQ0FBQyxZQUFZO1lBQ3RDLHVDQUFtQyxDQUFDLFdBQVcsRUFDL0M7WUFDQSxPQUFPO2dCQUNMO29CQUNFLElBQUksRUFBRSwyQkFBMkI7b0JBQ2pDLElBQUksRUFBRSwwQ0FBMEM7b0JBQ2hELElBQUksRUFBRTt3QkFDSixZQUFZLEVBQUUsV0FBVzt3QkFDekIsZ0JBQWdCLEVBQUUseUJBQXlCLENBQUMsWUFBWTt3QkFDeEQsdUJBQXVCLEVBQUUsR0FBRztxQkFDN0I7aUJBQ0Y7Z0JBQ0Q7b0JBQ0UsSUFBSSxFQUFFLHdCQUF3QjtvQkFDOUIsR0FBRyxFQUFFLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixXQUFXO2lCQUN6QzthQUNGLENBQUM7U0FDSDtRQUVELElBQUkseUJBQXlCLENBQUMsWUFBWSxFQUFFO1lBQzFDLE9BQU87Z0JBQ0w7b0JBQ0UsSUFBSSxFQUFFLDJCQUEyQjtvQkFDakMsSUFBSSxFQUFFLDBDQUEwQztvQkFDaEQsSUFBSSxFQUFFO3dCQUNKLG1CQUFtQixFQUFFLHFCQUFjLENBQ2pDLHlCQUF5QixDQUFDLGlCQUFpQixDQUM1Qzt3QkFDRCx1QkFBdUIsRUFBRSxxQkFBYyxDQUNyQyx5QkFBeUIsQ0FBQyxxQkFBcUIsQ0FDaEQ7d0JBQ0QsWUFBWSxFQUFFLFdBQVc7d0JBQ3pCLGdCQUFnQixFQUFFLHlCQUF5QixDQUFDLFlBQVk7d0JBQ3hELHVCQUF1QixFQUFFLEdBQUc7cUJBQzdCO2lCQUNGO2dCQUNEO29CQUNFLElBQUksRUFBRSx3QkFBd0I7b0JBQzlCLEdBQUcsRUFBRSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsV0FBVztpQkFDekM7YUFDRixDQUFDO1NBQ0g7UUFFRCxPQUFPO1lBQ0w7Z0JBQ0UsSUFBSSxFQUFFLHdCQUF3QjtnQkFDOUIsR0FBRyxFQUFFLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixXQUFXO2dCQUN4QyxHQUFHLEVBQUU7b0JBQ0gsaUJBQWlCLEVBQUUscUJBQWMsQ0FDL0IseUJBQXlCLENBQUMsaUJBQWlCLENBQzVDO29CQUNELHFCQUFxQixFQUFFLHFCQUFjLENBQ25DLHlCQUF5QixDQUFDLHFCQUFxQixDQUNoRDtpQkFDRjthQUNGO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUIsQ0FDeEIsVUFBc0MsRUFBRTtRQUV4QyxNQUFNLE9BQU8sR0FBRyxJQUFJLEtBQUssRUFBVyxDQUFDO1FBRXJDLHlDQUF5QztRQUN6QyxPQUFPLENBQUMsSUFBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFFN0MsSUFBSSxJQUFJLENBQUMsV0FBVyxFQUFFO1lBQ3BCLE9BQU8sQ0FBQyxJQUFJLENBQUM7Z0JBQ1gsSUFBSSxFQUFFLGVBQWU7Z0JBQ3JCLElBQUksRUFBRSx1QkFBdUI7Z0JBQzdCLElBQUksRUFBRSxFQUFFLGNBQWMsRUFBRSxJQUFJLENBQUMsV0FBVyxFQUFFO2FBQzNDLENBQUMsQ0FBQztTQUNKO1FBRUQsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsS0FBSyxpQ0FBa0IsQ0FBQyxJQUFJLEVBQUU7WUFDM0QsT0FBTyxDQUFDLElBQUksQ0FBQztnQkFDWCxJQUFJLEVBQUUsWUFBWTtnQkFDbEIsSUFBSSxFQUFFLDBCQUEwQjtnQkFDaEMsSUFBSSxFQUFFLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFO2FBQzVDLENBQUMsQ0FBQztTQUNKO1FBRUQsTUFBTSxPQUFPLEdBQUcsT0FBTyxDQUFDLE9BQU8sSUFBSSxLQUFLLENBQUM7UUFFekMsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLHFCQUFxQixFQUFFO1lBQ3RDLE9BQU8sQ0FBQyxJQUFJLENBQ1YsR0FBRyxJQUFJLENBQUMscUJBQXFCLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxtQkFBbUIsQ0FBQyxDQUNoRSxDQUFDO1NBQ0g7UUFFRCxPQUFPLENBQUMsSUFBSSxDQUFDO1lBQ1gsSUFBSSxFQUFFLHNCQUFzQjtZQUM1QixHQUFHLEVBQUUsT0FBTztnQkFDVixDQUFDLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQywrQkFBK0I7Z0JBQzlDLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWM7U0FDaEMsQ0FBQyxDQUFDO1FBRUgsT0FBTyxPQUFPLENBQUM7SUFDakIsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksT0FBTyxDQUFDLEdBQUcsSUFBYztRQUM5QixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLEdBQUcsSUFBSSxDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksVUFBVSxDQUFDLEdBQUcsSUFBYztRQUNqQyxPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLEdBQUcsSUFBSSxDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFdBQVcsQ0FBQyxHQUFHLElBQWM7UUFDbEMsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxHQUFHLElBQUksQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGNBQWMsQ0FBQyxHQUFHLElBQWM7UUFDckMsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxHQUFHLElBQUksQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFTSxnQkFBZ0IsQ0FBQyxPQUFlO1FBQ3JDLElBQUksQ0FBQyxTQUFTLEVBQUUsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFTyxVQUFVLENBQUMsT0FBMkI7UUFDNUMsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRTtZQUN4QixJQUFJLGlCQUFPLENBQUMsSUFBSSxFQUFFO2dCQUNoQixJQUFJLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPO2dCQUMxQixjQUFjLEVBQUUsT0FBTyxDQUFDLGNBQWMsSUFBSSxPQUFPLENBQUMsVUFBVTtnQkFDNUQsZUFBZSxFQUFFLE9BQU8sQ0FBQyxlQUFlO2FBQ3pDLENBQUMsQ0FBQztTQUNKO0lBQ0gsQ0FBQztJQUVPLG1CQUFtQjtRQUN6QixJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FDcEIsUUFBUSxFQUNSLE1BQU0sRUFDTixPQUFPLEVBQ1AsZ0JBQWdCLEVBQ2hCLGlCQUFpQixFQUNqQixpQkFBaUIsRUFDakIsa0JBQWtCLEVBRWxCLDJEQUEyRCxFQUMzRCx5Q0FBeUMsRUFFekMsZ0JBQWdCLEVBQ2hCLE1BQU0sRUFDTixPQUFPLEVBQ1AsUUFBUSxFQUNSLFlBQVksRUFFWixtRUFBbUUsRUFDbkUsU0FBUyxFQUVULGtEQUFrRCxFQUNsRCxVQUFVLEVBQ1YsUUFBUSxFQUVSLHFCQUFxQixFQUNyQixhQUFhLEVBRWIsK0RBQStELEVBQy9ELGVBQWUsRUFFZiwwQkFBMEIsRUFDMUIsZUFBZSxFQUNmLGdCQUFnQixFQUVoQixvQkFBb0IsRUFDcEIsZUFBZSxFQUVmLHlCQUF5QixFQUN6QixjQUFjLEVBRWQsd0JBQXdCLEVBQ3hCLE9BQU8sRUFFUCx1QkFBdUIsRUFDdkIsaUJBQWlCLEVBRWpCLGdEQUFnRCxFQUNoRCxRQUFRLENBQ1QsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLGNBQWMsQ0FBQyxJQUFVO1FBQzlCLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsSUFBSSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDdEQsQ0FBQztJQUVEOztPQUVHO0lBQ0gsSUFBVyxrQkFBa0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsYUFBYSxFQUFFLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUM1QyxDQUFDOztBQTN2Qkgsa0NBNHZCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGpvaW4gfSBmcm9tIFwicGF0aFwiO1xuaW1wb3J0IHsgQnVuZGxlciwgQnVuZGxlck9wdGlvbnMgfSBmcm9tIFwiLi9idW5kbGVyXCI7XG5pbXBvcnQgeyBKZXN0LCBKZXN0T3B0aW9ucyB9IGZyb20gXCIuL2plc3RcIjtcbmltcG9ydCB7XG4gIENvZGVBcnRpZmFjdEF1dGhQcm92aWRlciBhcyBOb2RlUGFja2FnZUNvZGVBcnRpZmFjdEF1dGhQcm92aWRlcixcbiAgQ29kZUFydGlmYWN0T3B0aW9ucyxcbiAgTm9kZVBhY2thZ2UsXG4gIE5vZGVQYWNrYWdlTWFuYWdlcixcbiAgTm9kZVBhY2thZ2VPcHRpb25zLFxufSBmcm9tIFwiLi9ub2RlLXBhY2thZ2VcIjtcbmltcG9ydCB7IFByb2plbnJjLCBQcm9qZW5yY09wdGlvbnMgfSBmcm9tIFwiLi9wcm9qZW5yY1wiO1xuaW1wb3J0IHsgQnVpbGRXb3JrZmxvdyB9IGZyb20gXCIuLi9idWlsZFwiO1xuaW1wb3J0IHsgUFJPSkVOX0RJUiwgUFJPSkVOX1JDIH0gZnJvbSBcIi4uL2NvbW1vblwiO1xuaW1wb3J0IHtcbiAgQXV0b01lcmdlLFxuICBEZXBlbmRhYm90T3B0aW9ucyxcbiAgR2l0SHViUHJvamVjdCxcbiAgR2l0SHViUHJvamVjdE9wdGlvbnMsXG4gIEdpdElkZW50aXR5LFxufSBmcm9tIFwiLi4vZ2l0aHViXCI7XG5pbXBvcnQgeyBERUZBVUxUX0dJVEhVQl9BQ1RJT05TX1VTRVIgfSBmcm9tIFwiLi4vZ2l0aHViL2NvbnN0YW50c1wiO1xuaW1wb3J0IHsgc2VjcmV0VG9TdHJpbmcgfSBmcm9tIFwiLi4vZ2l0aHViL3V0aWxcIjtcbmltcG9ydCB7XG4gIEpvYlBlcm1pc3Npb24sXG4gIEpvYlBlcm1pc3Npb25zLFxuICBKb2JTdGVwLFxuICBUcmlnZ2Vycyxcbn0gZnJvbSBcIi4uL2dpdGh1Yi93b3JrZmxvd3MtbW9kZWxcIjtcbmltcG9ydCB7IElnbm9yZUZpbGUgfSBmcm9tIFwiLi4vaWdub3JlLWZpbGVcIjtcbmltcG9ydCB7XG4gIFByZXR0aWVyLFxuICBQcmV0dGllck9wdGlvbnMsXG4gIFVwZ3JhZGVEZXBlbmRlbmNpZXMsXG4gIFVwZ3JhZGVEZXBlbmRlbmNpZXNPcHRpb25zLFxufSBmcm9tIFwiLi4vamF2YXNjcmlwdFwiO1xuaW1wb3J0IHsgTGljZW5zZSB9IGZyb20gXCIuLi9saWNlbnNlXCI7XG5pbXBvcnQge1xuICBpc0F3c0NvZGVBcnRpZmFjdFJlZ2lzdHJ5LFxuICBQdWJsaXNoZXIsXG4gIFJlbGVhc2UsXG4gIFJlbGVhc2VQcm9qZWN0T3B0aW9ucyxcbiAgTnBtUHVibGlzaE9wdGlvbnMsXG4gIENvZGVBcnRpZmFjdEF1dGhQcm92aWRlciBhcyBSZWxlYXNlQ29kZUFydGlmYWN0QXV0aFByb3ZpZGVyLFxuICBDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXIsXG59IGZyb20gXCIuLi9yZWxlYXNlXCI7XG5pbXBvcnQgeyBUYXNrIH0gZnJvbSBcIi4uL3Rhc2tcIjtcbmltcG9ydCB7IGRlZXBNZXJnZSB9IGZyb20gXCIuLi91dGlsXCI7XG5pbXBvcnQgeyBWZXJzaW9uIH0gZnJvbSBcIi4uL3ZlcnNpb25cIjtcblxuY29uc3QgUFJPSkVOX1NDUklQVCA9IFwicHJvamVuXCI7XG5cbmV4cG9ydCBpbnRlcmZhY2UgTm9kZVByb2plY3RPcHRpb25zXG4gIGV4dGVuZHMgR2l0SHViUHJvamVjdE9wdGlvbnMsXG4gICAgTm9kZVBhY2thZ2VPcHRpb25zLFxuICAgIFJlbGVhc2VQcm9qZWN0T3B0aW9ucyB7XG4gIC8qKlxuICAgKiBMaWNlbnNlIGNvcHlyaWdodCBvd25lci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0cyB0byB0aGUgdmFsdWUgb2YgYXV0aG9yTmFtZSBvciBcIlwiIGlmIGBhdXRob3JOYW1lYCBpcyB1bmRlZmluZWQuXG4gICAqL1xuICByZWFkb25seSBjb3B5cmlnaHRPd25lcj86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGNvcHlyaWdodCB5ZWFycyB0byBwdXQgaW4gdGhlIExJQ0VOU0UgZmlsZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBjdXJyZW50IHllYXJcbiAgICovXG4gIHJlYWRvbmx5IGNvcHlyaWdodFBlcmlvZD86IHN0cmluZztcblxuICAvKipcbiAgICogVmVyc2lvbiBvZiBwcm9qZW4gdG8gaW5zdGFsbC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBEZWZhdWx0cyB0byB0aGUgbGF0ZXN0IHZlcnNpb24uXG4gICAqL1xuICByZWFkb25seSBwcm9qZW5WZXJzaW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgb2YgXCJwcm9qZW5cIiBzaG91bGQgYmUgaW5zdGFsbGVkIGFzIGEgZGV2RGVwZW5kZW5jeS5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgcHJvamVuRGV2RGVwZW5kZW5jeT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIERlZmluZSBhIEdpdEh1YiB3b3JrZmxvdyBmb3IgYnVpbGRpbmcgUFJzLlxuICAgKiBAZGVmYXVsdCAtIHRydWUgaWYgbm90IGEgc3VicHJvamVjdFxuICAgKi9cbiAgcmVhZG9ubHkgYnVpbGRXb3JrZmxvdz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEF1dG9tYXRpY2FsbHkgdXBkYXRlIGZpbGVzIG1vZGlmaWVkIGR1cmluZyBidWlsZHMgdG8gcHVsbC1yZXF1ZXN0IGJyYW5jaGVzLiBUaGlzIG1lYW5zXG4gICAqIHRoYXQgYW55IGZpbGVzIHN5bnRoZXNpemVkIGJ5IHByb2plbiBvciBlLmcuIHRlc3Qgc25hcHNob3RzIHdpbGwgYWx3YXlzIGJlIHVwLXRvLWRhdGVcbiAgICogYmVmb3JlIGEgUFIgaXMgbWVyZ2VkLlxuICAgKlxuICAgKiBJbXBsaWVzIHRoYXQgUFIgYnVpbGRzIGRvIG5vdCBoYXZlIGFudGktdGFtcGVyIGNoZWNrcy5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgbXV0YWJsZUJ1aWxkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogRGVmaW5lIGEgR2l0SHViIHdvcmtmbG93IHN0ZXAgZm9yIHNlbmRpbmcgY29kZSBjb3ZlcmFnZSBtZXRyaWNzIHRvIGh0dHBzOi8vY29kZWNvdi5pby9cbiAgICogVXNlcyBjb2RlY292L2NvZGVjb3YtYWN0aW9uQHYzXG4gICAqIEEgc2VjcmV0IGlzIHJlcXVpcmVkIGZvciBwcml2YXRlIHJlcG9zLiBDb25maWd1cmVkIHdpdGggQGNvZGVDb3ZUb2tlblNlY3JldFxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgY29kZUNvdj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIERlZmluZSB0aGUgc2VjcmV0IG5hbWUgZm9yIGEgc3BlY2lmaWVkIGh0dHBzOi8vY29kZWNvdi5pby8gdG9rZW5cbiAgICogQSBzZWNyZXQgaXMgcmVxdWlyZWQgdG8gc2VuZCBjb3ZlcmFnZSBmb3IgcHJpdmF0ZSByZXBvc2l0b3JpZXNcbiAgICogQGRlZmF1bHQgLSBpZiB0aGlzIG9wdGlvbiBpcyBub3Qgc3BlY2lmaWVkLCBvbmx5IHB1YmxpYyByZXBvc2l0b3JpZXMgYXJlIHN1cHBvcnRlZFxuICAgKi9cbiAgcmVhZG9ubHkgY29kZUNvdlRva2VuU2VjcmV0Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBERVBSRUNBVEVEOiByZW5hbWVkIHRvIGByZWxlYXNlYC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0cnVlIGlmIG5vdCBhIHN1YnByb2plY3RcbiAgICogQGRlcHJlY2F0ZWQgc2VlIGByZWxlYXNlYC5cbiAgICovXG4gIHJlYWRvbmx5IHJlbGVhc2VXb3JrZmxvdz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEFkZCByZWxlYXNlIG1hbmFnZW1lbnQgdG8gdGhpcyBwcm9qZWN0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWUgKGZhbHNlIGZvciBzdWJwcm9qZWN0cylcbiAgICovXG4gIHJlYWRvbmx5IHJlbGVhc2U/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgbWFpbiByZWxlYXNlIGJyYW5jaC5cbiAgICpcbiAgICogQGRlZmF1bHQgXCJtYWluXCJcbiAgICovXG4gIHJlYWRvbmx5IGRlZmF1bHRSZWxlYXNlQnJhbmNoOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFdvcmtmbG93IHN0ZXBzIHRvIHVzZSBpbiBvcmRlciB0byBib290c3RyYXAgdGhpcyByZXBvLlxuICAgKlxuICAgKiBAZGVmYXVsdCBcInlhcm4gaW5zdGFsbCAtLWZyb3plbi1sb2NrZmlsZSAmJiB5YXJuIHByb2plblwiXG4gICAqL1xuICByZWFkb25seSB3b3JrZmxvd0Jvb3RzdHJhcFN0ZXBzPzogSm9iU3RlcFtdO1xuXG4gIC8qKlxuICAgKiBUaGUgZ2l0IGlkZW50aXR5IHRvIHVzZSBpbiB3b3JrZmxvd3MuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gR2l0SHViIEFjdGlvbnNcbiAgICovXG4gIHJlYWRvbmx5IHdvcmtmbG93R2l0SWRlbnRpdHk/OiBHaXRJZGVudGl0eTtcblxuICAvKipcbiAgICogQXV0b21hdGljYWxseSByZWxlYXNlIHRvIG5wbSB3aGVuIG5ldyB2ZXJzaW9ucyBhcmUgaW50cm9kdWNlZC5cbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHJlbGVhc2VUb05wbT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBub2RlIHZlcnNpb24gdG8gdXNlIGluIEdpdEh1YiB3b3JrZmxvd3MuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gc2FtZSBhcyBgbWluTm9kZVZlcnNpb25gXG4gICAqL1xuICByZWFkb25seSB3b3JrZmxvd05vZGVWZXJzaW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBVc2UgZGVwZW5kYWJvdCB0byBoYW5kbGUgZGVwZW5kZW5jeSB1cGdyYWRlcy5cbiAgICogQ2Fubm90IGJlIHVzZWQgaW4gY29uanVuY3Rpb24gd2l0aCBgZGVwc1VwZ3JhZGVgLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgZGVwZW5kYWJvdD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIE9wdGlvbnMgZm9yIGRlcGVuZGFib3QuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZGVmYXVsdCBvcHRpb25zXG4gICAqL1xuICByZWFkb25seSBkZXBlbmRhYm90T3B0aW9ucz86IERlcGVuZGFib3RPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBVc2UgZ2l0aHViIHdvcmtmbG93cyB0byBoYW5kbGUgZGVwZW5kZW5jeSB1cGdyYWRlcy5cbiAgICogQ2Fubm90IGJlIHVzZWQgaW4gY29uanVuY3Rpb24gd2l0aCBgZGVwZW5kYWJvdGAuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGRlcHNVcGdyYWRlPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogT3B0aW9ucyBmb3IgYFVwZ3JhZGVEZXBlbmRlbmNpZXNgLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGRlZmF1bHQgb3B0aW9uc1xuICAgKi9cbiAgcmVhZG9ubHkgZGVwc1VwZ3JhZGVPcHRpb25zPzogVXBncmFkZURlcGVuZGVuY2llc09wdGlvbnM7XG5cbiAgLyoqXG4gICAqIEF1dG9tYXRpY2FsbHkgYXBwcm92ZSBkZXBzIHVwZ3JhZGUgUFJzLCBhbGxvd2luZyB0aGVtIHRvIGJlXG4gICAqIG1lcmdlZCBieSBtZXJnaWZ5IChpZiBjb25maWd1ZWQpLlxuICAgKlxuICAgKiBUaHJvdyBpZiBzZXQgdG8gdHJ1ZSBidXQgYGF1dG9BcHByb3ZlT3B0aW9uc2AgYXJlIG5vdCBkZWZpbmVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGF1dG9BcHByb3ZlVXBncmFkZXM/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBEZWZpbmVzIGFuIC5ucG1pZ25vcmUgZmlsZS4gTm9ybWFsbHkgdGhpcyBpcyBvbmx5IG5lZWRlZCBmb3IgbGlicmFyaWVzIHRoYXRcbiAgICogYXJlIHBhY2thZ2VkIGFzIHRhcmJhbGxzLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBucG1pZ25vcmVFbmFibGVkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQWRkaXRpb25hbCBlbnRyaWVzIHRvIC5ucG1pZ25vcmUuXG4gICAqIEBkZXByZWNhdGVkIC0gdXNlIGBwcm9qZWN0LmFkZFBhY2thZ2VJZ25vcmVgXG4gICAqL1xuICByZWFkb25seSBucG1pZ25vcmU/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogSW5jbHVkZSBhIEdpdEh1YiBwdWxsIHJlcXVlc3QgdGVtcGxhdGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IHB1bGxSZXF1ZXN0VGVtcGxhdGU/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgY29udGVudHMgb2YgdGhlIHB1bGwgcmVxdWVzdCB0ZW1wbGF0ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0IGNvbnRlbnRcbiAgICovXG4gIHJlYWRvbmx5IHB1bGxSZXF1ZXN0VGVtcGxhdGVDb250ZW50cz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBTZXR1cCBwcmV0dGllci5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHByZXR0aWVyPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogUHJldHRpZXIgb3B0aW9uc1xuICAgKiBAZGVmYXVsdCAtIGRlZmF1bHQgb3B0aW9uc1xuICAgKi9cbiAgcmVhZG9ubHkgcHJldHRpZXJPcHRpb25zPzogUHJldHRpZXJPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBBZGRpdGlvbmFsIGVudHJpZXMgdG8gLmdpdGlnbm9yZVxuICAgKi9cbiAgcmVhZG9ubHkgZ2l0aWdub3JlPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFNldHVwIGplc3QgdW5pdCB0ZXN0c1xuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBqZXN0PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogSmVzdCBvcHRpb25zXG4gICAqIEBkZWZhdWx0IC0gZGVmYXVsdCBvcHRpb25zXG4gICAqL1xuICByZWFkb25seSBqZXN0T3B0aW9ucz86IEplc3RPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZSAob25jZSkgLnByb2plbnJjLmpzIChpbiBKYXZhU2NyaXB0KS4gU2V0IHRvIGBmYWxzZWAgaW4gb3JkZXIgdG8gZGlzYWJsZVxuICAgKiAucHJvamVucmMuanMgZ2VuZXJhdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0cnVlIGlmIHByb2plbnJjSnNvbiBpcyBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcHJvamVucmNKcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIE9wdGlvbnMgZm9yIC5wcm9qZW5yYy5qc1xuICAgKiBAZGVmYXVsdCAtIGRlZmF1bHQgb3B0aW9uc1xuICAgKi9cbiAgcmVhZG9ubHkgcHJvamVucmNKc09wdGlvbnM/OiBQcm9qZW5yY09wdGlvbnM7XG5cbiAgLyoqXG4gICAqIE9wdGlvbnMgZm9yIGBCdW5kbGVyYC5cbiAgICovXG4gIHJlYWRvbmx5IGJ1bmRsZXJPcHRpb25zPzogQnVuZGxlck9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIEEgZGlyZWN0b3J5IHdoaWNoIHdpbGwgY29udGFpbiBidWlsZCBhcnRpZmFjdHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IFwiZGlzdFwiXG4gICAqL1xuICByZWFkb25seSBhcnRpZmFjdHNEaXJlY3Rvcnk/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIERlZmluZXMgYSBgcGFja2FnZWAgdGFzayB0aGF0IHdpbGwgcHJvZHVjZSBhbiBucG0gdGFyYmFsbCB1bmRlciB0aGVcbiAgICogYXJ0aWZhY3RzIGRpcmVjdG9yeSAoZS5nLiBgZGlzdGApLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBwYWNrYWdlPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQnVpbGQgd29ya2Zsb3cgdHJpZ2dlcnNcbiAgICogQGRlZmF1bHQgXCJ7IHB1bGxSZXF1ZXN0OiB7fSwgd29ya2Zsb3dEaXNwYXRjaDoge30gfVwiXG4gICAqL1xuICByZWFkb25seSBidWlsZFdvcmtmbG93VHJpZ2dlcnM/OiBUcmlnZ2Vycztcbn1cblxuLyoqXG4gKiBBdXRvbWF0aWMgYnVtcCBtb2Rlc1xuICovXG5leHBvcnQgZW51bSBBdXRvUmVsZWFzZSB7XG4gIC8qKlxuICAgKiBBdXRvbWF0aWNhbGx5IGJ1bXAgJiByZWxlYXNlIGEgbmV3IHZlcnNpb24gZm9yIGV2ZXJ5IGNvbW1pdCB0byBcIm1haW5cIlxuICAgKi9cbiAgRVZFUllfQ09NTUlULFxuXG4gIC8qKlxuICAgKiBBdXRvbWF0aWNhbGx5IGJ1bXAgJiByZWxlYXNlIGEgbmV3IHZlcnNpb24gb24gYSBkYWlseSBiYXNpcy5cbiAgICovXG4gIERBSUxZLFxufVxuXG4vKipcbiAqIE5vZGUuanMgcHJvamVjdFxuICpcbiAqIEBwamlkIG5vZGVcbiAqL1xuZXhwb3J0IGNsYXNzIE5vZGVQcm9qZWN0IGV4dGVuZHMgR2l0SHViUHJvamVjdCB7XG4gIC8qKlxuICAgKiBBUEkgZm9yIG1hbmFnaW5nIHRoZSBub2RlIHBhY2thZ2UuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcGFja2FnZTogTm9kZVBhY2thZ2U7XG5cbiAgLyoqXG4gICAqIFRoZSAubnBtaWdub3JlIGZpbGUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgbnBtaWdub3JlPzogSWdub3JlRmlsZTtcblxuICAvKipcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBwYWNrYWdlLmFsbG93TGlicmFyeURlcGVuZGVuY2llc2BcbiAgICovXG4gIHB1YmxpYyBnZXQgYWxsb3dMaWJyYXJ5RGVwZW5kZW5jaWVzKCk6IGJvb2xlYW4ge1xuICAgIHJldHVybiB0aGlzLnBhY2thZ2UuYWxsb3dMaWJyYXJ5RGVwZW5kZW5jaWVzO1xuICB9XG5cbiAgLyoqXG4gICAqIEBkZXByZWNhdGVkIHVzZSBgcGFja2FnZS5lbnRyeXBvaW50YFxuICAgKi9cbiAgcHVibGljIGdldCBlbnRyeXBvaW50KCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHRoaXMucGFja2FnZS5lbnRyeXBvaW50O1xuICB9XG5cbiAgLyoqXG4gICAqIENvbXBvbmVudCB0aGF0IHNldHMgdXAgbWVyZ2lmeSBmb3IgbWVyZ2luZyBhcHByb3ZlZCBwdWxsIHJlcXVlc3RzLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGF1dG9NZXJnZT86IEF1dG9NZXJnZTtcblxuICAvKipcbiAgICogVGhlIFBSIGJ1aWxkIEdpdEh1YiB3b3JrZmxvdy4gYHVuZGVmaW5lZGAgaWYgYGJ1aWxkV29ya2Zsb3dgIGlzIGRpc2FibGVkLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGJ1aWxkV29ya2Zsb3c/OiBCdWlsZFdvcmtmbG93O1xuXG4gIC8qKlxuICAgKiBQYWNrYWdlIHB1Ymxpc2hlci4gVGhpcyB3aWxsIGJlIGB1bmRlZmluZWRgIGlmIHRoZSBwcm9qZWN0IGRvZXMgbm90IGhhdmUgYVxuICAgKiByZWxlYXNlIHdvcmtmbG93LlxuICAgKlxuICAgKiBAZGVwcmVjYXRlZCB1c2UgYHJlbGVhc2UucHVibGlzaGVyYC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwdWJsaXNoZXI/OiBQdWJsaXNoZXI7XG5cbiAgLyoqXG4gICAqIFJlbGVhc2UgbWFuYWdlbWVudC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSByZWxlYXNlPzogUmVsZWFzZTtcblxuICAvKipcbiAgICogTWluaW11bSBub2RlLmpzIHZlcnNpb24gcmVxdWlyZWQgYnkgdGhpcyBwYWNrYWdlLlxuICAgKi9cbiAgcHVibGljIGdldCBtaW5Ob2RlVmVyc2lvbigpOiBzdHJpbmcgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzLnBhY2thZ2UubWluTm9kZVZlcnNpb247XG4gIH1cblxuICAvKipcbiAgICogTWF4aW11bSBub2RlIHZlcnNpb24gcmVxdWlyZWQgYnkgdGhpcyBwYWNrYWdlLlxuICAgKi9cbiAgcHVibGljIGdldCBtYXhOb2RlVmVyc2lvbigpOiBzdHJpbmcgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzLnBhY2thZ2UubWF4Tm9kZVZlcnNpb247XG4gIH1cblxuICBwcm90ZWN0ZWQgcmVhZG9ubHkgbm9kZVZlcnNpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBwYWNrYWdlIG1hbmFnZXIgdG8gdXNlLlxuICAgKlxuICAgKiBAZGVwcmVjYXRlZCB1c2UgYHBhY2thZ2UucGFja2FnZU1hbmFnZXJgXG4gICAqL1xuICBwdWJsaWMgZ2V0IHBhY2thZ2VNYW5hZ2VyKCk6IE5vZGVQYWNrYWdlTWFuYWdlciB7XG4gICAgcmV0dXJuIHRoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlcjtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgY29tbWFuZCB0byB1c2UgdG8gcnVuIHNjcmlwdHMgKGUuZy4gYHlhcm4gcnVuYCBvciBgbnBtIHJ1bmAgZGVwZW5kcyBvbiB0aGUgcGFja2FnZSBtYW5hZ2VyKS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBydW5TY3JpcHRDb21tYW5kOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBKZXN0IGNvbmZpZ3VyYXRpb24gKGlmIGVuYWJsZWQpXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgamVzdD86IEplc3Q7XG5cbiAgLyoqXG4gICAqIEBkZXByZWNhdGVkIHVzZSBgcGFja2FnZS5hZGRGaWVsZCh4LCB5KWBcbiAgICovXG4gIHB1YmxpYyBnZXQgbWFuaWZlc3QoKSB7XG4gICAgcmV0dXJuIHRoaXMucGFja2FnZS5tYW5pZmVzdDtcbiAgfVxuXG4gIHB1YmxpYyByZWFkb25seSBidW5kbGVyOiBCdW5kbGVyO1xuXG4gIC8qKlxuICAgKiBUaGUgYnVpbGQgb3V0cHV0IGRpcmVjdG9yeS4gQW4gbnBtIHRhcmJhbGwgd2lsbCBiZSBjcmVhdGVkIHVuZGVyIHRoZSBganNgXG4gICAqIHN1YmRpcmVjdG9yeS4gRm9yIGV4YW1wbGUsIGlmIHRoaXMgaXMgc2V0IHRvIGBkaXN0YCAodGhlIGRlZmF1bHQpLCB0aGUgbnBtXG4gICAqIHRhcmJhbGwgd2lsbCBiZSBwbGFjZWQgdW5kZXIgYGRpc3QvanMvYm9vbS1ib29tLTEuMi4zLnRnYC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhcnRpZmFjdHNEaXJlY3Rvcnk6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGxvY2F0aW9uIG9mIHRoZSBucG0gdGFyYmFsbCBhZnRlciBidWlsZCAoYCR7YXJ0aWZhY3RzRGlyZWN0b3J5fS9qc2ApLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGFydGlmYWN0c0phdmFzY3JpcHREaXJlY3Rvcnk6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHVwZ3JhZGUgd29ya2Zsb3cuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdXBncmFkZVdvcmtmbG93PzogVXBncmFkZURlcGVuZGVuY2llcztcblxuICBwcm90ZWN0ZWQgcmVhZG9ubHkgd29ya2Zsb3dCb290c3RyYXBTdGVwczogSm9iU3RlcFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IHdvcmtmbG93R2l0SWRlbnRpdHk6IEdpdElkZW50aXR5O1xuICBwdWJsaWMgcmVhZG9ubHkgcHJldHRpZXI/OiBQcmV0dGllcjtcblxuICBjb25zdHJ1Y3RvcihvcHRpb25zOiBOb2RlUHJvamVjdE9wdGlvbnMpIHtcbiAgICBzdXBlcihvcHRpb25zKTtcblxuICAgIHRoaXMucGFja2FnZSA9IG5ldyBOb2RlUGFja2FnZSh0aGlzLCBvcHRpb25zKTtcbiAgICB0aGlzLndvcmtmbG93Qm9vdHN0cmFwU3RlcHMgPSBvcHRpb25zLndvcmtmbG93Qm9vdHN0cmFwU3RlcHMgPz8gW107XG4gICAgdGhpcy53b3JrZmxvd0dpdElkZW50aXR5ID1cbiAgICAgIG9wdGlvbnMud29ya2Zsb3dHaXRJZGVudGl0eSA/PyBERUZBVUxUX0dJVEhVQl9BQ1RJT05TX1VTRVI7XG4gICAgdGhpcy5hcnRpZmFjdHNEaXJlY3RvcnkgPSBvcHRpb25zLmFydGlmYWN0c0RpcmVjdG9yeSA/PyBcImRpc3RcIjtcbiAgICB0aGlzLmFydGlmYWN0c0phdmFzY3JpcHREaXJlY3RvcnkgPSBqb2luKHRoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5LCBcImpzXCIpO1xuXG4gICAgdGhpcy5ydW5TY3JpcHRDb21tYW5kID0gKCgpID0+IHtcbiAgICAgIHN3aXRjaCAodGhpcy5wYWNrYWdlTWFuYWdlcikge1xuICAgICAgICBjYXNlIE5vZGVQYWNrYWdlTWFuYWdlci5OUE06XG4gICAgICAgICAgcmV0dXJuIFwibnBtIHJ1blwiO1xuICAgICAgICBjYXNlIE5vZGVQYWNrYWdlTWFuYWdlci5ZQVJOOlxuICAgICAgICBjYXNlIE5vZGVQYWNrYWdlTWFuYWdlci5ZQVJOMjpcbiAgICAgICAgICByZXR1cm4gXCJ5YXJuIHJ1blwiO1xuICAgICAgICBjYXNlIE5vZGVQYWNrYWdlTWFuYWdlci5QTlBNOlxuICAgICAgICAgIHJldHVybiBcInBucG0gcnVuXCI7XG4gICAgICAgIGRlZmF1bHQ6XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGB1bmV4cGVjdGVkIHBhY2thZ2UgbWFuYWdlciAke3RoaXMucGFja2FnZU1hbmFnZXJ9YCk7XG4gICAgICB9XG4gICAgfSkoKTtcblxuICAgIHRoaXMubm9kZVZlcnNpb24gPVxuICAgICAgb3B0aW9ucy53b3JrZmxvd05vZGVWZXJzaW9uID8/IHRoaXMucGFja2FnZS5taW5Ob2RlVmVyc2lvbjtcblxuICAgIC8vIGFkZCBQQVRIIGZvciBhbGwgdGFza3Mgd2hpY2ggaW5jbHVkZXMgdGhlIHByb2plY3QncyBucG0gLmJpbiBsaXN0XG4gICAgdGhpcy50YXNrcy5hZGRFbnZpcm9ubWVudChcbiAgICAgIFwiUEFUSFwiLFxuICAgICAgJyQobnB4IC1jIFwibm9kZSAtZSBcXFxcXCJjb25zb2xlLmxvZyhwcm9jZXNzLmVudi5QQVRIKVxcXFxcIlwiKSdcbiAgICApO1xuXG4gICAgdGhpcy5hZGRMaWNlbnNlKG9wdGlvbnMpO1xuXG4gICAgaWYgKG9wdGlvbnMubnBtaWdub3JlRW5hYmxlZCA/PyB0cnVlKSB7XG4gICAgICB0aGlzLm5wbWlnbm9yZSA9IG5ldyBJZ25vcmVGaWxlKHRoaXMsIFwiLm5wbWlnbm9yZVwiKTtcbiAgICB9XG5cbiAgICB0aGlzLmFkZERlZmF1bHRHaXRJZ25vcmUoKTtcblxuICAgIGlmIChvcHRpb25zLmdpdGlnbm9yZT8ubGVuZ3RoKSB7XG4gICAgICBmb3IgKGNvbnN0IGkgb2Ygb3B0aW9ucy5naXRpZ25vcmUpIHtcbiAgICAgICAgdGhpcy5naXRpZ25vcmUuZXhjbHVkZShpKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBpZiAob3B0aW9ucy5ucG1pZ25vcmU/Lmxlbmd0aCkge1xuICAgICAgaWYgKCF0aGlzLm5wbWlnbm9yZSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgJy5ucG1pZ25vcmUgaXMgbm90IGRlZmluZWQgZm9yIGFuIEFQUCBwcm9qZWN0IHR5cGUuIEFkZCBcIm5wbUlnbm9yZTogdHJ1ZVwiIHRvIG92ZXJyaWRlIHRoaXMnXG4gICAgICAgICk7XG4gICAgICB9XG5cbiAgICAgIGZvciAoY29uc3QgaSBvZiBvcHRpb25zLm5wbWlnbm9yZSkge1xuICAgICAgICB0aGlzLm5wbWlnbm9yZS5leGNsdWRlKGkpO1xuICAgICAgfVxuICAgIH1cblxuICAgIGlmICghdGhpcy5lamVjdGVkKSB7XG4gICAgICB0aGlzLnNldFNjcmlwdChQUk9KRU5fU0NSSVBULCB0aGlzLnBhY2thZ2UucHJvamVuQ29tbWFuZCk7XG4gICAgfVxuXG4gICAgdGhpcy5ucG1pZ25vcmU/LmV4Y2x1ZGUoYC8ke1BST0pFTl9SQ31gKTtcbiAgICB0aGlzLm5wbWlnbm9yZT8uZXhjbHVkZShgLyR7UFJPSkVOX0RJUn0vYCk7XG4gICAgdGhpcy5naXRpZ25vcmUuaW5jbHVkZShgLyR7UFJPSkVOX1JDfWApO1xuXG4gICAgY29uc3QgcHJvamVuID0gb3B0aW9ucy5wcm9qZW5EZXZEZXBlbmRlbmN5ID8/IHRydWU7XG4gICAgaWYgKHByb2plbiAmJiAhdGhpcy5lamVjdGVkKSB7XG4gICAgICBjb25zdCBwb3N0Zml4ID0gb3B0aW9ucy5wcm9qZW5WZXJzaW9uID8gYEAke29wdGlvbnMucHJvamVuVmVyc2lvbn1gIDogXCJcIjtcbiAgICAgIHRoaXMuYWRkRGV2RGVwcyhgcHJvamVuJHtwb3N0Zml4fWApO1xuICAgIH1cblxuICAgIGlmICghb3B0aW9ucy5kZWZhdWx0UmVsZWFzZUJyYW5jaCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAnXCJkZWZhdWx0UmVsZWFzZUJyYW5jaFwiIGlzIHRlbXBvcmFyaWx5IGEgcmVxdWlyZWQgb3B0aW9uIHdoaWxlIHdlIG1pZ3JhdGUgaXRzIGRlZmF1bHQgdmFsdWUgZnJvbSBcIm1hc3RlclwiIHRvIFwibWFpblwiJ1xuICAgICAgKTtcbiAgICB9XG5cbiAgICBjb25zdCBidWlsZEVuYWJsZWQgPSBvcHRpb25zLmJ1aWxkV29ya2Zsb3cgPz8gKHRoaXMucGFyZW50ID8gZmFsc2UgOiB0cnVlKTtcblxuICAgIC8vIGNvbmZpZ3VyZSBqZXN0IGlmIGVuYWJsZWRcbiAgICAvLyBtdXN0IGJlIGJlZm9yZSB0aGUgYnVpbGQvcmVsZWFzZSB3b3JrZmxvd3NcbiAgICBpZiAob3B0aW9ucy5qZXN0ID8/IHRydWUpIHtcbiAgICAgIHRoaXMuamVzdCA9IG5ldyBKZXN0KHRoaXMsIG9wdGlvbnMuamVzdE9wdGlvbnMpO1xuICAgIH1cblxuICAgIGNvbnN0IHJlcXVpcmVzSWRUb2tlblBlcm1pc3Npb24gPVxuICAgICAgKG9wdGlvbnMuc2NvcGVkUGFja2FnZXNPcHRpb25zID8/IFtdKS5sZW5ndGggPiAwICYmXG4gICAgICBvcHRpb25zLmNvZGVBcnRpZmFjdE9wdGlvbnM/LmF1dGhQcm92aWRlciA9PT1cbiAgICAgICAgQ29kZUFydGlmYWN0QXV0aFByb3ZpZGVyLkdJVEhVQl9PSURDO1xuXG4gICAgY29uc3Qgd29ya2Zsb3dQZXJtaXNzaW9uczogSm9iUGVybWlzc2lvbnMgPSB7XG4gICAgICBpZFRva2VuOiByZXF1aXJlc0lkVG9rZW5QZXJtaXNzaW9uID8gSm9iUGVybWlzc2lvbi5XUklURSA6IHVuZGVmaW5lZCxcbiAgICB9O1xuXG4gICAgaWYgKGJ1aWxkRW5hYmxlZCAmJiB0aGlzLmdpdGh1Yikge1xuICAgICAgdGhpcy5idWlsZFdvcmtmbG93ID0gbmV3IEJ1aWxkV29ya2Zsb3codGhpcywge1xuICAgICAgICBidWlsZFRhc2s6IHRoaXMuYnVpbGRUYXNrLFxuICAgICAgICBhcnRpZmFjdHNEaXJlY3Rvcnk6IHRoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5LFxuICAgICAgICBjb250YWluZXJJbWFnZTogb3B0aW9ucy53b3JrZmxvd0NvbnRhaW5lckltYWdlLFxuICAgICAgICBnaXRJZGVudGl0eTogdGhpcy53b3JrZmxvd0dpdElkZW50aXR5LFxuICAgICAgICBtdXRhYmxlQnVpbGQ6IG9wdGlvbnMubXV0YWJsZUJ1aWxkLFxuICAgICAgICBwcmVCdWlsZFN0ZXBzOiB0aGlzLnJlbmRlcldvcmtmbG93U2V0dXAoe1xuICAgICAgICAgIG11dGFibGU6IG9wdGlvbnMubXV0YWJsZUJ1aWxkID8/IHRydWUsXG4gICAgICAgIH0pLFxuICAgICAgICBwb3N0QnVpbGRTdGVwczogb3B0aW9ucy5wb3N0QnVpbGRTdGVwcyxcbiAgICAgICAgcnVuc09uOiBvcHRpb25zLndvcmtmbG93UnVuc09uLFxuICAgICAgICB3b3JrZmxvd1RyaWdnZXJzOiBvcHRpb25zLmJ1aWxkV29ya2Zsb3dUcmlnZ2VycyxcbiAgICAgICAgcGVybWlzc2lvbnM6IHdvcmtmbG93UGVybWlzc2lvbnMsXG4gICAgICB9KTtcblxuICAgICAgdGhpcy5idWlsZFdvcmtmbG93LmFkZFBvc3RCdWlsZFN0ZXBzKFxuICAgICAgICAuLi50aGlzLnJlbmRlclVwbG9hZENvdmVyYWdlSm9iU3RlcChvcHRpb25zKVxuICAgICAgKTtcbiAgICB9XG5cbiAgICBjb25zdCByZWxlYXNlID1cbiAgICAgIG9wdGlvbnMucmVsZWFzZSA/P1xuICAgICAgb3B0aW9ucy5yZWxlYXNlV29ya2Zsb3cgPz9cbiAgICAgICh0aGlzLnBhcmVudCA/IGZhbHNlIDogdHJ1ZSk7XG4gICAgaWYgKHJlbGVhc2UpIHtcbiAgICAgIHRoaXMuYWRkRGV2RGVwcyhWZXJzaW9uLlNUQU5EQVJEX1ZFUlNJT04pO1xuXG4gICAgICB0aGlzLnJlbGVhc2UgPSBuZXcgUmVsZWFzZSh0aGlzLCB7XG4gICAgICAgIHZlcnNpb25GaWxlOiBcInBhY2thZ2UuanNvblwiLCAvLyB0aGlzIGlzIHdoZXJlIFwidmVyc2lvblwiIGlzIHNldCBhZnRlciBidW1wXG4gICAgICAgIHRhc2s6IHRoaXMuYnVpbGRUYXNrLFxuICAgICAgICBicmFuY2g6IG9wdGlvbnMuZGVmYXVsdFJlbGVhc2VCcmFuY2ggPz8gXCJtYWluXCIsXG4gICAgICAgIGFydGlmYWN0c0RpcmVjdG9yeTogdGhpcy5hcnRpZmFjdHNEaXJlY3RvcnksXG4gICAgICAgIC4uLm9wdGlvbnMsXG5cbiAgICAgICAgcmVsZWFzZVdvcmtmbG93U2V0dXBTdGVwczogW1xuICAgICAgICAgIC4uLnRoaXMucmVuZGVyV29ya2Zsb3dTZXR1cCh7IG11dGFibGU6IGZhbHNlIH0pLFxuICAgICAgICAgIC4uLihvcHRpb25zLnJlbGVhc2VXb3JrZmxvd1NldHVwU3RlcHMgPz8gW10pLFxuICAgICAgICBdLFxuICAgICAgICBwb3N0QnVpbGRTdGVwczogW1xuICAgICAgICAgIC4uLihvcHRpb25zLnBvc3RCdWlsZFN0ZXBzID8/IFtdKSxcbiAgICAgICAgICAuLi50aGlzLnJlbmRlclVwbG9hZENvdmVyYWdlSm9iU3RlcChvcHRpb25zKSxcbiAgICAgICAgXSxcblxuICAgICAgICB3b3JrZmxvd05vZGVWZXJzaW9uOiB0aGlzLm5vZGVWZXJzaW9uLFxuICAgICAgICB3b3JrZmxvd1Blcm1pc3Npb25zLFxuICAgICAgfSk7XG5cbiAgICAgIHRoaXMucHVibGlzaGVyID0gdGhpcy5yZWxlYXNlLnB1Ymxpc2hlcjtcblxuICAgICAgY29uc3Qgbm9kZVBhY2thZ2VUb1JlbGVhc2VDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXJNYXBwaW5nOiBSZWNvcmQ8XG4gICAgICAgIE5vZGVQYWNrYWdlQ29kZUFydGlmYWN0QXV0aFByb3ZpZGVyLFxuICAgICAgICBSZWxlYXNlQ29kZUFydGlmYWN0QXV0aFByb3ZpZGVyXG4gICAgICA+ID0ge1xuICAgICAgICBbTm9kZVBhY2thZ2VDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXIuQUNDRVNTX0FORF9TRUNSRVRfS0VZX1BBSVJdOlxuICAgICAgICAgIFJlbGVhc2VDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXIuQUNDRVNTX0FORF9TRUNSRVRfS0VZX1BBSVIsXG4gICAgICAgIFtOb2RlUGFja2FnZUNvZGVBcnRpZmFjdEF1dGhQcm92aWRlci5HSVRIVUJfT0lEQ106XG4gICAgICAgICAgUmVsZWFzZUNvZGVBcnRpZmFjdEF1dGhQcm92aWRlci5HSVRIVUJfT0lEQyxcbiAgICAgIH07XG5cbiAgICAgIGlmIChvcHRpb25zLnJlbGVhc2VUb05wbSA/PyBmYWxzZSkge1xuICAgICAgICBjb25zdCBjb2RlQXJ0aWZhY3RPcHRpb25zOiBOcG1QdWJsaXNoT3B0aW9uc1tcImNvZGVBcnRpZmFjdE9wdGlvbnNcIl0gPVxuICAgICAgICAgIGlzQXdzQ29kZUFydGlmYWN0UmVnaXN0cnkodGhpcy5wYWNrYWdlLm5wbVJlZ2lzdHJ5KVxuICAgICAgICAgICAgPyB7XG4gICAgICAgICAgICAgICAgYWNjZXNzS2V5SWRTZWNyZXQ6XG4gICAgICAgICAgICAgICAgICBvcHRpb25zLmNvZGVBcnRpZmFjdE9wdGlvbnM/LmFjY2Vzc0tleUlkU2VjcmV0LFxuICAgICAgICAgICAgICAgIHNlY3JldEFjY2Vzc0tleVNlY3JldDpcbiAgICAgICAgICAgICAgICAgIG9wdGlvbnMuY29kZUFydGlmYWN0T3B0aW9ucz8uc2VjcmV0QWNjZXNzS2V5U2VjcmV0LFxuICAgICAgICAgICAgICAgIHJvbGVUb0Fzc3VtZTogb3B0aW9ucy5jb2RlQXJ0aWZhY3RPcHRpb25zPy5yb2xlVG9Bc3N1bWUsXG4gICAgICAgICAgICAgICAgYXV0aFByb3ZpZGVyOiBvcHRpb25zLmNvZGVBcnRpZmFjdE9wdGlvbnM/LmF1dGhQcm92aWRlclxuICAgICAgICAgICAgICAgICAgPyBub2RlUGFja2FnZVRvUmVsZWFzZUNvZGVBcnRpZmFjdEF1dGhQcm92aWRlck1hcHBpbmdbXG4gICAgICAgICAgICAgICAgICAgICAgb3B0aW9ucy5jb2RlQXJ0aWZhY3RPcHRpb25zLmF1dGhQcm92aWRlclxuICAgICAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgICA6IFJlbGVhc2VDb2RlQXJ0aWZhY3RBdXRoUHJvdmlkZXIuQUNDRVNTX0FORF9TRUNSRVRfS0VZX1BBSVIsXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIDoge307XG4gICAgICAgIHRoaXMucmVsZWFzZS5wdWJsaXNoZXIucHVibGlzaFRvTnBtKHtcbiAgICAgICAgICByZWdpc3RyeTogdGhpcy5wYWNrYWdlLm5wbVJlZ2lzdHJ5LFxuICAgICAgICAgIG5wbVRva2VuU2VjcmV0OiB0aGlzLnBhY2thZ2UubnBtVG9rZW5TZWNyZXQsXG4gICAgICAgICAgY29kZUFydGlmYWN0T3B0aW9ucyxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIC8vIHZhbGlkYXRlIHRoYXQgbm8gcmVsZWFzZSBvcHRpb25zIGFyZSBzZWxlY3RlZCBpZiB0aGUgcmVsZWFzZSB3b3JrZmxvdyBpcyBkaXNhYmxlZC5cbiAgICAgIGlmIChvcHRpb25zLnJlbGVhc2VUb05wbSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgJ1wicmVsZWFzZVRvTnBtXCIgaXMgbm90IHN1cHBvcnRlZCBpZiBcInJlbGVhc2VcIiBpcyBub3Qgc2V0J1xuICAgICAgICApO1xuICAgICAgfVxuXG4gICAgICBpZiAob3B0aW9ucy5yZWxlYXNlRXZlcnlDb21taXQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICdcInJlbGVhc2VFdmVyeUNvbW1pdFwiIGlzIG5vdCBzdXBwb3J0ZWQgaWYgXCJyZWxlYXNlXCIgaXMgbm90IHNldCdcbiAgICAgICAgKTtcbiAgICAgIH1cblxuICAgICAgaWYgKG9wdGlvbnMucmVsZWFzZVNjaGVkdWxlKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAnXCJyZWxlYXNlU2NoZWR1bGVcIiBpcyBub3Qgc3VwcG9ydGVkIGlmIFwicmVsZWFzZVwiIGlzIG5vdCBzZXQnXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgaWYgKFxuICAgICAgKG9wdGlvbnMuYXV0b01lcmdlID8/IHRydWUpICYmXG4gICAgICB0aGlzLmdpdGh1Yj8ubWVyZ2lmeSAmJlxuICAgICAgdGhpcy5idWlsZFdvcmtmbG93Py5idWlsZEpvYklkc1xuICAgICkge1xuICAgICAgdGhpcy5hdXRvTWVyZ2UgPSBuZXcgQXV0b01lcmdlKHRoaXMuZ2l0aHViLCBvcHRpb25zLmF1dG9NZXJnZU9wdGlvbnMpO1xuICAgICAgdGhpcy5hdXRvTWVyZ2UuYWRkQ29uZGl0aW9uc0xhdGVyKHtcbiAgICAgICAgcmVuZGVyOiAoKSA9PlxuICAgICAgICAgIHRoaXMuYnVpbGRXb3JrZmxvdz8uYnVpbGRKb2JJZHMubWFwKChpZCkgPT4gYHN0YXR1cy1zdWNjZXNzPSR7aWR9YCkgPz9cbiAgICAgICAgICBbXSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIGNvbnN0IGRlcGVuZGFib3QgPSBvcHRpb25zLmRlcGVuZGFib3QgPz8gZmFsc2U7XG4gICAgY29uc3QgZGVwc1VwZ3JhZGUgPSBvcHRpb25zLmRlcHNVcGdyYWRlID8/ICFkZXBlbmRhYm90O1xuXG4gICAgaWYgKGRlcGVuZGFib3QgJiYgZGVwc1VwZ3JhZGUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgXCInZGVwZW5kYWJvdCcgY2Fubm90IGJlIGNvbmZpZ3VyZWQgdG9nZXRoZXIgd2l0aCAnZGVwc1VwZ3JhZGUnXCJcbiAgICAgICk7XG4gICAgfVxuXG4gICAgY29uc3QgZGVwc0F1dG9BcHByb3ZlID0gb3B0aW9ucy5hdXRvQXBwcm92ZVVwZ3JhZGVzID8/IGZhbHNlO1xuXG4gICAgaWYgKGRlcHNBdXRvQXBwcm92ZSAmJiAhdGhpcy5hdXRvQXBwcm92ZSAmJiB0aGlzLmdpdGh1Yikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBcIkF1dG9tYXRpYyBhcHByb3ZhbCBvZiBkZXBlbmRlbmNpZXMgdXBncmFkZXMgcmVxdWlyZXMgY29uZmlndXJpbmcgYGF1dG9BcHByb3ZlT3B0aW9uc2BcIlxuICAgICAgKTtcbiAgICB9XG5cbiAgICBjb25zdCBhdXRvQXBwcm92ZUxhYmVsID0gKGNvbmRpdGlvbjogYm9vbGVhbikgPT5cbiAgICAgIGNvbmRpdGlvbiAmJiB0aGlzLmF1dG9BcHByb3ZlPy5sYWJlbFxuICAgICAgICA/IFt0aGlzLmF1dG9BcHByb3ZlLmxhYmVsXVxuICAgICAgICA6IHVuZGVmaW5lZDtcblxuICAgIGlmIChkZXBlbmRhYm90KSB7XG4gICAgICBjb25zdCBkZWZhdWx0T3B0aW9ucyA9IHtcbiAgICAgICAgbGFiZWxzOiBhdXRvQXBwcm92ZUxhYmVsKGRlcHNBdXRvQXBwcm92ZSksXG4gICAgICB9O1xuICAgICAgdGhpcy5naXRodWI/LmFkZERlcGVuZGFib3QoXG4gICAgICAgIGRlZXBNZXJnZShbZGVmYXVsdE9wdGlvbnMsIG9wdGlvbnMuZGVwZW5kYWJvdE9wdGlvbnMgPz8ge31dKVxuICAgICAgKTtcbiAgICB9XG5cbiAgICBpZiAoZGVwc1VwZ3JhZGUpIHtcbiAgICAgIGNvbnN0IGRlZmF1bHRPcHRpb25zOiBVcGdyYWRlRGVwZW5kZW5jaWVzT3B0aW9ucyA9IHtcbiAgICAgICAgd29ya2Zsb3dPcHRpb25zOiB7XG4gICAgICAgICAgY29udGFpbmVyOiBvcHRpb25zLndvcmtmbG93Q29udGFpbmVySW1hZ2VcbiAgICAgICAgICAgID8ge1xuICAgICAgICAgICAgICAgIGltYWdlOiBvcHRpb25zLndvcmtmbG93Q29udGFpbmVySW1hZ2UsXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgICAgIGxhYmVsczogYXV0b0FwcHJvdmVMYWJlbChkZXBzQXV0b0FwcHJvdmUpLFxuICAgICAgICAgIGdpdElkZW50aXR5OiB0aGlzLndvcmtmbG93R2l0SWRlbnRpdHksXG4gICAgICAgIH0sXG4gICAgICB9O1xuICAgICAgdGhpcy51cGdyYWRlV29ya2Zsb3cgPSBuZXcgVXBncmFkZURlcGVuZGVuY2llcyhcbiAgICAgICAgdGhpcyxcbiAgICAgICAgZGVlcE1lcmdlKFtkZWZhdWx0T3B0aW9ucywgb3B0aW9ucy5kZXBzVXBncmFkZU9wdGlvbnMgPz8ge31dKVxuICAgICAgKTtcbiAgICB9XG5cbiAgICBpZiAob3B0aW9ucy5wdWxsUmVxdWVzdFRlbXBsYXRlID8/IHRydWUpIHtcbiAgICAgIHRoaXMuZ2l0aHViPy5hZGRQdWxsUmVxdWVzdFRlbXBsYXRlKFxuICAgICAgICAuLi4ob3B0aW9ucy5wdWxsUmVxdWVzdFRlbXBsYXRlQ29udGVudHMgPz8gW10pXG4gICAgICApO1xuICAgIH1cblxuICAgIGNvbnN0IHByb2plbnJjSnMgPSBvcHRpb25zLnByb2plbnJjSnMgPz8gIW9wdGlvbnMucHJvamVucmNKc29uO1xuICAgIGlmIChwcm9qZW5yY0pzKSB7XG4gICAgICBuZXcgUHJvamVucmModGhpcywgb3B0aW9ucy5wcm9qZW5yY0pzT3B0aW9ucyk7XG4gICAgfVxuXG4gICAgLy8gYWRkIGEgYnVuZGxlciBjb21wb25lbnQgLSB0aGlzIGVuYWJsZXMgdGhpbmdzIGxpa2UgTGFtYmRhIGJ1bmRsaW5nIGFuZCBpbiB0aGUgZnV0dXJlIHdlYiBidW5kbGluZy5cbiAgICB0aGlzLmJ1bmRsZXIgPSBuZXcgQnVuZGxlcih0aGlzLCBvcHRpb25zLmJ1bmRsZXJPcHRpb25zKTtcblxuICAgIGlmIChvcHRpb25zLnBhY2thZ2UgPz8gdHJ1ZSkge1xuICAgICAgdGhpcy5wYWNrYWdlVGFzay5leGVjKGBta2RpciAtcCAke3RoaXMuYXJ0aWZhY3RzSmF2YXNjcmlwdERpcmVjdG9yeX1gKTtcblxuICAgICAgLy8gYWx3YXlzIHVzZSBucG0gaGVyZSAtIHlhcm4gZG9lc24ndCBhZGQgbXVjaCB2YWx1ZVxuICAgICAgLy8gc2FkbHkgd2UgY2Fubm90IHVzZSAtLXBhY2stZGVzdGluYXRpb24gYmVjYXVzZSBpdCBpcyBub3Qgc3VwcG9ydGVkIGJ5IG9sZGVyIG5wbVxuICAgICAgdGhpcy5wYWNrYWdlVGFzay5leGVjKFxuICAgICAgICBgbXYgJChucG0gcGFjaykgJHt0aGlzLmFydGlmYWN0c0phdmFzY3JpcHREaXJlY3Rvcnl9L2BcbiAgICAgICk7XG4gICAgfVxuXG4gICAgaWYgKG9wdGlvbnMucHJldHRpZXIgPz8gZmFsc2UpIHtcbiAgICAgIHRoaXMucHJldHRpZXIgPSBuZXcgUHJldHRpZXIodGhpcywgeyAuLi5vcHRpb25zLnByZXR0aWVyT3B0aW9ucyB9KTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIHJlbmRlclVwbG9hZENvdmVyYWdlSm9iU3RlcChvcHRpb25zOiBOb2RlUHJvamVjdE9wdGlvbnMpOiBKb2JTdGVwW10ge1xuICAgIC8vIHJ1biBjb2RlY292IGlmIGVuYWJsZWQgb3IgYSBzZWNyZXQgdG9rZW4gbmFtZSBpcyBwYXNzZWQgaW5cbiAgICAvLyBBTkQgamVzdCBtdXN0IGJlIGNvbmZpZ3VyZWRcbiAgICBpZiAoKG9wdGlvbnMuY29kZUNvdiB8fCBvcHRpb25zLmNvZGVDb3ZUb2tlblNlY3JldCkgJiYgdGhpcy5qZXN0Py5jb25maWcpIHtcbiAgICAgIHJldHVybiBbXG4gICAgICAgIHtcbiAgICAgICAgICBuYW1lOiBcIlVwbG9hZCBjb3ZlcmFnZSB0byBDb2RlY292XCIsXG4gICAgICAgICAgdXNlczogXCJjb2RlY292L2NvZGVjb3YtYWN0aW9uQHYzXCIsXG4gICAgICAgICAgd2l0aDogb3B0aW9ucy5jb2RlQ292VG9rZW5TZWNyZXRcbiAgICAgICAgICAgID8ge1xuICAgICAgICAgICAgICAgIHRva2VuOiBgXFwke3sgc2VjcmV0cy4ke29wdGlvbnMuY29kZUNvdlRva2VuU2VjcmV0fSB9fWAsXG4gICAgICAgICAgICAgICAgZGlyZWN0b3J5OiB0aGlzLmplc3QuY29uZmlnLmNvdmVyYWdlRGlyZWN0b3J5LFxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICA6IHtcbiAgICAgICAgICAgICAgICBkaXJlY3Rvcnk6IHRoaXMuamVzdC5jb25maWcuY292ZXJhZ2VEaXJlY3RvcnksXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICBdO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXR1cm4gW107XG4gICAgfVxuICB9XG5cbiAgcHVibGljIGFkZEJpbnMoYmluczogUmVjb3JkPHN0cmluZywgc3RyaW5nPikge1xuICAgIHRoaXMucGFja2FnZS5hZGRCaW4oYmlucyk7XG4gIH1cblxuICAvKipcbiAgICogUmVwbGFjZXMgdGhlIGNvbnRlbnRzIG9mIGFuIG5wbSBwYWNrYWdlLmpzb24gc2NyaXB0LlxuICAgKlxuICAgKiBAcGFyYW0gbmFtZSBUaGUgc2NyaXB0IG5hbWVcbiAgICogQHBhcmFtIGNvbW1hbmQgVGhlIGNvbW1hbmQgdG8gZXhlY3V0ZVxuICAgKi9cbiAgcHVibGljIHNldFNjcmlwdChuYW1lOiBzdHJpbmcsIGNvbW1hbmQ6IHN0cmluZykge1xuICAgIHRoaXMucGFja2FnZS5zZXRTY3JpcHQobmFtZSwgY29tbWFuZCk7XG4gIH1cblxuICAvKipcbiAgICogUmVtb3ZlcyB0aGUgbnBtIHNjcmlwdCAoYWx3YXlzIHN1Y2Nlc3NmdWwpLlxuICAgKiBAcGFyYW0gbmFtZSBUaGUgbmFtZSBvZiB0aGUgc2NyaXB0LlxuICAgKi9cbiAgcHVibGljIHJlbW92ZVNjcmlwdChuYW1lOiBzdHJpbmcpIHtcbiAgICB0aGlzLnBhY2thZ2UucmVtb3ZlU2NyaXB0KG5hbWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyBpZiBhIHNjcmlwdCBieSB0aGUgbmFtZSBuYW1lIGlzIGRlZmluZWQuXG4gICAqIEBwYXJhbSBuYW1lIFRoZSBuYW1lIG9mIHRoZSBzY3JpcHRcbiAgICovXG4gIHB1YmxpYyBoYXNTY3JpcHQobmFtZTogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMucGFja2FnZS5oYXNTY3JpcHQobmFtZSk7XG4gIH1cblxuICAvKipcbiAgICogREVQUkVDQVRFRFxuICAgKiBAZGVwcmVjYXRlZCB1c2UgYHByb2plY3QuY29tcGlsZVRhc2suZXhlYygpYFxuICAgKi9cbiAgcHVibGljIGFkZENvbXBpbGVDb21tYW5kKC4uLmNvbW1hbmRzOiBzdHJpbmdbXSkge1xuICAgIGZvciAoY29uc3QgYyBvZiBjb21tYW5kcykge1xuICAgICAgdGhpcy5jb21waWxlVGFzay5leGVjKGMpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBERVBSRUNBVEVEXG4gICAqIEBkZXByZWNhdGVkIHVzZSBgcHJvamVjdC50ZXN0VGFzay5leGVjKClgXG4gICAqL1xuICBwdWJsaWMgYWRkVGVzdENvbW1hbmQoLi4uY29tbWFuZHM6IHN0cmluZ1tdKSB7XG4gICAgZm9yIChjb25zdCBjIG9mIGNvbW1hbmRzKSB7XG4gICAgICB0aGlzLnRlc3RUYXNrLmV4ZWMoYyk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIERpcmVjdGx5IHNldCBmaWVsZHMgaW4gYHBhY2thZ2UuanNvbmAuXG4gICAqIEBwYXJhbSBmaWVsZHMgVGhlIGZpZWxkcyB0byBzZXRcbiAgICovXG4gIHB1YmxpYyBhZGRGaWVsZHMoZmllbGRzOiB7IFtuYW1lOiBzdHJpbmddOiBhbnkgfSkge1xuICAgIGZvciAoY29uc3QgW25hbWUsIHZhbHVlXSBvZiBPYmplY3QuZW50cmllcyhmaWVsZHMpKSB7XG4gICAgICB0aGlzLnBhY2thZ2UuYWRkRmllbGQobmFtZSwgdmFsdWUpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGtleXdvcmRzIHRvIHBhY2thZ2UuanNvbiAoZGVkdXBsaWNhdGVkKVxuICAgKiBAcGFyYW0ga2V5d29yZHMgVGhlIGtleXdvcmRzIHRvIGFkZFxuICAgKi9cbiAgcHVibGljIGFkZEtleXdvcmRzKC4uLmtleXdvcmRzOiBzdHJpbmdbXSkge1xuICAgIHRoaXMucGFja2FnZS5hZGRLZXl3b3JkcyguLi5rZXl3b3Jkcyk7XG4gIH1cblxuICAvKipcbiAgICogR2V0IHN0ZXBzIGZvciBzY29wZWQgcGFja2FnZSBhY2Nlc3NcbiAgICpcbiAgICogQHBhcmFtIGNvZGVBcnRpZmFjdE9wdGlvbnMgRGV0YWlscyBvZiBsb2dnaW5nIGluIHRvIEFXU1xuICAgKiBAcmV0dXJucyBhcnJheSBvZiBqb2Igc3RlcHMgcmVxdWlyZWQgZm9yIGVhY2ggcHJpdmF0ZSBzY29wZWQgcGFja2FnZXNcbiAgICovXG4gIHByaXZhdGUgZ2V0U2NvcGVkUGFja2FnZVN0ZXBzKFxuICAgIGNvZGVBcnRpZmFjdE9wdGlvbnM6IENvZGVBcnRpZmFjdE9wdGlvbnMgfCB1bmRlZmluZWRcbiAgKTogSm9iU3RlcFtdIHtcbiAgICBjb25zdCBwYXJzZWRDb2RlQXJ0aWZhY3RPcHRpb25zID0ge1xuICAgICAgYWNjZXNzS2V5SWRTZWNyZXQ6XG4gICAgICAgIGNvZGVBcnRpZmFjdE9wdGlvbnM/LmFjY2Vzc0tleUlkU2VjcmV0ID8/IFwiQVdTX0FDQ0VTU19LRVlfSURcIixcbiAgICAgIHNlY3JldEFjY2Vzc0tleVNlY3JldDpcbiAgICAgICAgY29kZUFydGlmYWN0T3B0aW9ucz8uc2VjcmV0QWNjZXNzS2V5U2VjcmV0ID8/IFwiQVdTX1NFQ1JFVF9BQ0NFU1NfS0VZXCIsXG4gICAgICByb2xlVG9Bc3N1bWU6IGNvZGVBcnRpZmFjdE9wdGlvbnM/LnJvbGVUb0Fzc3VtZSxcbiAgICAgIGF1dGhQcm92aWRlcjogY29kZUFydGlmYWN0T3B0aW9ucz8uYXV0aFByb3ZpZGVyLFxuICAgIH07XG5cbiAgICBpZiAoXG4gICAgICBwYXJzZWRDb2RlQXJ0aWZhY3RPcHRpb25zLmF1dGhQcm92aWRlciA9PT1cbiAgICAgIE5vZGVQYWNrYWdlQ29kZUFydGlmYWN0QXV0aFByb3ZpZGVyLkdJVEhVQl9PSURDXG4gICAgKSB7XG4gICAgICByZXR1cm4gW1xuICAgICAgICB7XG4gICAgICAgICAgbmFtZTogXCJDb25maWd1cmUgQVdTIENyZWRlbnRpYWxzXCIsXG4gICAgICAgICAgdXNlczogXCJhd3MtYWN0aW9ucy9jb25maWd1cmUtYXdzLWNyZWRlbnRpYWxzQHYxXCIsXG4gICAgICAgICAgd2l0aDoge1xuICAgICAgICAgICAgXCJhd3MtcmVnaW9uXCI6IFwidXMtZWFzdC0yXCIsXG4gICAgICAgICAgICBcInJvbGUtdG8tYXNzdW1lXCI6IHBhcnNlZENvZGVBcnRpZmFjdE9wdGlvbnMucm9sZVRvQXNzdW1lLFxuICAgICAgICAgICAgXCJyb2xlLWR1cmF0aW9uLXNlY29uZHNcIjogOTAwLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBuYW1lOiBcIkFXUyBDb2RlQXJ0aWZhY3QgTG9naW5cIixcbiAgICAgICAgICBydW46IGAke3RoaXMucnVuU2NyaXB0Q29tbWFuZH0gY2E6bG9naW5gLFxuICAgICAgICB9LFxuICAgICAgXTtcbiAgICB9XG5cbiAgICBpZiAocGFyc2VkQ29kZUFydGlmYWN0T3B0aW9ucy5yb2xlVG9Bc3N1bWUpIHtcbiAgICAgIHJldHVybiBbXG4gICAgICAgIHtcbiAgICAgICAgICBuYW1lOiBcIkNvbmZpZ3VyZSBBV1MgQ3JlZGVudGlhbHNcIixcbiAgICAgICAgICB1c2VzOiBcImF3cy1hY3Rpb25zL2NvbmZpZ3VyZS1hd3MtY3JlZGVudGlhbHNAdjFcIixcbiAgICAgICAgICB3aXRoOiB7XG4gICAgICAgICAgICBcImF3cy1hY2Nlc3Mta2V5LWlkXCI6IHNlY3JldFRvU3RyaW5nKFxuICAgICAgICAgICAgICBwYXJzZWRDb2RlQXJ0aWZhY3RPcHRpb25zLmFjY2Vzc0tleUlkU2VjcmV0XG4gICAgICAgICAgICApLFxuICAgICAgICAgICAgXCJhd3Mtc2VjcmV0LWFjY2Vzcy1rZXlcIjogc2VjcmV0VG9TdHJpbmcoXG4gICAgICAgICAgICAgIHBhcnNlZENvZGVBcnRpZmFjdE9wdGlvbnMuc2VjcmV0QWNjZXNzS2V5U2VjcmV0XG4gICAgICAgICAgICApLFxuICAgICAgICAgICAgXCJhd3MtcmVnaW9uXCI6IFwidXMtZWFzdC0yXCIsXG4gICAgICAgICAgICBcInJvbGUtdG8tYXNzdW1lXCI6IHBhcnNlZENvZGVBcnRpZmFjdE9wdGlvbnMucm9sZVRvQXNzdW1lLFxuICAgICAgICAgICAgXCJyb2xlLWR1cmF0aW9uLXNlY29uZHNcIjogOTAwLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBuYW1lOiBcIkFXUyBDb2RlQXJ0aWZhY3QgTG9naW5cIixcbiAgICAgICAgICBydW46IGAke3RoaXMucnVuU2NyaXB0Q29tbWFuZH0gY2E6bG9naW5gLFxuICAgICAgICB9LFxuICAgICAgXTtcbiAgICB9XG5cbiAgICByZXR1cm4gW1xuICAgICAge1xuICAgICAgICBuYW1lOiBcIkFXUyBDb2RlQXJ0aWZhY3QgTG9naW5cIixcbiAgICAgICAgcnVuOiBgJHt0aGlzLnJ1blNjcmlwdENvbW1hbmR9IGNhOmxvZ2luYCxcbiAgICAgICAgZW52OiB7XG4gICAgICAgICAgQVdTX0FDQ0VTU19LRVlfSUQ6IHNlY3JldFRvU3RyaW5nKFxuICAgICAgICAgICAgcGFyc2VkQ29kZUFydGlmYWN0T3B0aW9ucy5hY2Nlc3NLZXlJZFNlY3JldFxuICAgICAgICAgICksXG4gICAgICAgICAgQVdTX1NFQ1JFVF9BQ0NFU1NfS0VZOiBzZWNyZXRUb1N0cmluZyhcbiAgICAgICAgICAgIHBhcnNlZENvZGVBcnRpZmFjdE9wdGlvbnMuc2VjcmV0QWNjZXNzS2V5U2VjcmV0XG4gICAgICAgICAgKSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBzZXQgb2Ygd29ya2Zsb3cgc3RlcHMgd2hpY2ggc2hvdWxkIGJlIGV4ZWN1dGVkIHRvIGJvb3RzdHJhcCBhXG4gICAqIHdvcmtmbG93LlxuICAgKlxuICAgKiBAcGFyYW0gb3B0aW9ucyBPcHRpb25zLlxuICAgKiBAcmV0dXJucyBKb2Igc3RlcHNcbiAgICovXG4gIHB1YmxpYyByZW5kZXJXb3JrZmxvd1NldHVwKFxuICAgIG9wdGlvbnM6IFJlbmRlcldvcmtmbG93U2V0dXBPcHRpb25zID0ge31cbiAgKTogSm9iU3RlcFtdIHtcbiAgICBjb25zdCBpbnN0YWxsID0gbmV3IEFycmF5PEpvYlN0ZXA+KCk7XG5cbiAgICAvLyBmaXJzdCBydW4gdGhlIHdvcmtmbG93IGJvb3RzdHJhcCBzdGVwc1xuICAgIGluc3RhbGwucHVzaCguLi50aGlzLndvcmtmbG93Qm9vdHN0cmFwU3RlcHMpO1xuXG4gICAgaWYgKHRoaXMubm9kZVZlcnNpb24pIHtcbiAgICAgIGluc3RhbGwucHVzaCh7XG4gICAgICAgIG5hbWU6IFwiU2V0dXAgTm9kZS5qc1wiLFxuICAgICAgICB1c2VzOiBcImFjdGlvbnMvc2V0dXAtbm9kZUB2M1wiLFxuICAgICAgICB3aXRoOiB7IFwibm9kZS12ZXJzaW9uXCI6IHRoaXMubm9kZVZlcnNpb24gfSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIGlmICh0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXIgPT09IE5vZGVQYWNrYWdlTWFuYWdlci5QTlBNKSB7XG4gICAgICBpbnN0YWxsLnB1c2goe1xuICAgICAgICBuYW1lOiBcIlNldHVwIHBucG1cIixcbiAgICAgICAgdXNlczogXCJwbnBtL2FjdGlvbi1zZXR1cEB2Mi4yLjRcIixcbiAgICAgICAgd2l0aDogeyB2ZXJzaW9uOiB0aGlzLnBhY2thZ2UucG5wbVZlcnNpb24gfSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIGNvbnN0IG11dGFibGUgPSBvcHRpb25zLm11dGFibGUgPz8gZmFsc2U7XG5cbiAgICBpZiAodGhpcy5wYWNrYWdlLnNjb3BlZFBhY2thZ2VzT3B0aW9ucykge1xuICAgICAgaW5zdGFsbC5wdXNoKFxuICAgICAgICAuLi50aGlzLmdldFNjb3BlZFBhY2thZ2VTdGVwcyh0aGlzLnBhY2thZ2UuY29kZUFydGlmYWN0T3B0aW9ucylcbiAgICAgICk7XG4gICAgfVxuXG4gICAgaW5zdGFsbC5wdXNoKHtcbiAgICAgIG5hbWU6IFwiSW5zdGFsbCBkZXBlbmRlbmNpZXNcIixcbiAgICAgIHJ1bjogbXV0YWJsZVxuICAgICAgICA/IHRoaXMucGFja2FnZS5pbnN0YWxsQW5kVXBkYXRlTG9ja2ZpbGVDb21tYW5kXG4gICAgICAgIDogdGhpcy5wYWNrYWdlLmluc3RhbGxDb21tYW5kLFxuICAgIH0pO1xuXG4gICAgcmV0dXJuIGluc3RhbGw7XG4gIH1cblxuICAvKipcbiAgICogRGVmaW5lcyBub3JtYWwgZGVwZW5kZW5jaWVzLlxuICAgKlxuICAgKiBAcGFyYW0gZGVwcyBOYW1lcyBtb2R1bGVzIHRvIGluc3RhbGwuIEJ5IGRlZmF1bHQsIHRoZSB0aGUgZGVwZW5kZW5jeSB3aWxsXG4gICAqIGJlIGluc3RhbGxlZCBpbiB0aGUgbmV4dCBgbnB4IHByb2plbmAgcnVuIGFuZCB0aGUgdmVyc2lvbiB3aWxsIGJlIHJlY29yZGVkXG4gICAqIGluIHlvdXIgYHBhY2thZ2UuanNvbmAgZmlsZS4gWW91IGNhbiB1cGdyYWRlIG1hbnVhbGx5IG9yIHVzaW5nIGB5YXJuXG4gICAqIGFkZC91cGdyYWRlYC4gSWYgeW91IHdpc2ggdG8gc3BlY2lmeSBhIHZlcnNpb24gcmFuZ2UgdXNlIHRoaXMgc3ludGF4OlxuICAgKiBgbW9kdWxlQF43YC5cbiAgICovXG4gIHB1YmxpYyBhZGREZXBzKC4uLmRlcHM6IHN0cmluZ1tdKSB7XG4gICAgcmV0dXJuIHRoaXMucGFja2FnZS5hZGREZXBzKC4uLmRlcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIERlZmluZXMgZGV2ZWxvcG1lbnQvdGVzdCBkZXBlbmRlbmNpZXMuXG4gICAqXG4gICAqIEBwYXJhbSBkZXBzIE5hbWVzIG1vZHVsZXMgdG8gaW5zdGFsbC4gQnkgZGVmYXVsdCwgdGhlIHRoZSBkZXBlbmRlbmN5IHdpbGxcbiAgICogYmUgaW5zdGFsbGVkIGluIHRoZSBuZXh0IGBucHggcHJvamVuYCBydW4gYW5kIHRoZSB2ZXJzaW9uIHdpbGwgYmUgcmVjb3JkZWRcbiAgICogaW4geW91ciBgcGFja2FnZS5qc29uYCBmaWxlLiBZb3UgY2FuIHVwZ3JhZGUgbWFudWFsbHkgb3IgdXNpbmcgYHlhcm5cbiAgICogYWRkL3VwZ3JhZGVgLiBJZiB5b3Ugd2lzaCB0byBzcGVjaWZ5IGEgdmVyc2lvbiByYW5nZSB1c2UgdGhpcyBzeW50YXg6XG4gICAqIGBtb2R1bGVAXjdgLlxuICAgKi9cbiAgcHVibGljIGFkZERldkRlcHMoLi4uZGVwczogc3RyaW5nW10pIHtcbiAgICByZXR1cm4gdGhpcy5wYWNrYWdlLmFkZERldkRlcHMoLi4uZGVwcyk7XG4gIH1cblxuICAvKipcbiAgICogRGVmaW5lcyBwZWVyIGRlcGVuZGVuY2llcy5cbiAgICpcbiAgICogV2hlbiBhZGRpbmcgcGVlciBkZXBlbmRlbmNpZXMsIGEgZGV2RGVwZW5kZW5jeSB3aWxsIGFsc28gYmUgYWRkZWQgb24gdGhlXG4gICAqIHBpbm5lZCB2ZXJzaW9uIG9mIHRoZSBkZWNsYXJlZCBwZWVyLiBUaGlzIHdpbGwgZW5zdXJlIHRoYXQgeW91IGFyZSB0ZXN0aW5nXG4gICAqIHlvdXIgY29kZSBhZ2FpbnN0IHRoZSBtaW5pbXVtIHZlcnNpb24gcmVxdWlyZWQgZnJvbSB5b3VyIGNvbnN1bWVycy5cbiAgICpcbiAgICogQHBhcmFtIGRlcHMgTmFtZXMgbW9kdWxlcyB0byBpbnN0YWxsLiBCeSBkZWZhdWx0LCB0aGUgdGhlIGRlcGVuZGVuY3kgd2lsbFxuICAgKiBiZSBpbnN0YWxsZWQgaW4gdGhlIG5leHQgYG5weCBwcm9qZW5gIHJ1biBhbmQgdGhlIHZlcnNpb24gd2lsbCBiZSByZWNvcmRlZFxuICAgKiBpbiB5b3VyIGBwYWNrYWdlLmpzb25gIGZpbGUuIFlvdSBjYW4gdXBncmFkZSBtYW51YWxseSBvciB1c2luZyBgeWFyblxuICAgKiBhZGQvdXBncmFkZWAuIElmIHlvdSB3aXNoIHRvIHNwZWNpZnkgYSB2ZXJzaW9uIHJhbmdlIHVzZSB0aGlzIHN5bnRheDpcbiAgICogYG1vZHVsZUBeN2AuXG4gICAqL1xuICBwdWJsaWMgYWRkUGVlckRlcHMoLi4uZGVwczogc3RyaW5nW10pIHtcbiAgICByZXR1cm4gdGhpcy5wYWNrYWdlLmFkZFBlZXJEZXBzKC4uLmRlcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIERlZmluZXMgYnVuZGxlZCBkZXBlbmRlbmNpZXMuXG4gICAqXG4gICAqIEJ1bmRsZWQgZGVwZW5kZW5jaWVzIHdpbGwgYmUgYWRkZWQgYXMgbm9ybWFsIGRlcGVuZGVuY2llcyBhcyB3ZWxsIGFzIHRvIHRoZVxuICAgKiBgYnVuZGxlZERlcGVuZGVuY2llc2Agc2VjdGlvbiBvZiB5b3VyIGBwYWNrYWdlLmpzb25gLlxuICAgKlxuICAgKiBAcGFyYW0gZGVwcyBOYW1lcyBtb2R1bGVzIHRvIGluc3RhbGwuIEJ5IGRlZmF1bHQsIHRoZSB0aGUgZGVwZW5kZW5jeSB3aWxsXG4gICAqIGJlIGluc3RhbGxlZCBpbiB0aGUgbmV4dCBgbnB4IHByb2plbmAgcnVuIGFuZCB0aGUgdmVyc2lvbiB3aWxsIGJlIHJlY29yZGVkXG4gICAqIGluIHlvdXIgYHBhY2thZ2UuanNvbmAgZmlsZS4gWW91IGNhbiB1cGdyYWRlIG1hbnVhbGx5IG9yIHVzaW5nIGB5YXJuXG4gICAqIGFkZC91cGdyYWRlYC4gSWYgeW91IHdpc2ggdG8gc3BlY2lmeSBhIHZlcnNpb24gcmFuZ2UgdXNlIHRoaXMgc3ludGF4OlxuICAgKiBgbW9kdWxlQF43YC5cbiAgICovXG4gIHB1YmxpYyBhZGRCdW5kbGVkRGVwcyguLi5kZXBzOiBzdHJpbmdbXSkge1xuICAgIHJldHVybiB0aGlzLnBhY2thZ2UuYWRkQnVuZGxlZERlcHMoLi4uZGVwcyk7XG4gIH1cblxuICBwdWJsaWMgYWRkUGFja2FnZUlnbm9yZShwYXR0ZXJuOiBzdHJpbmcpIHtcbiAgICB0aGlzLm5wbWlnbm9yZT8uYWRkUGF0dGVybnMocGF0dGVybik7XG4gIH1cblxuICBwcml2YXRlIGFkZExpY2Vuc2Uob3B0aW9uczogTm9kZVByb2plY3RPcHRpb25zKSB7XG4gICAgaWYgKHRoaXMucGFja2FnZS5saWNlbnNlKSB7XG4gICAgICBuZXcgTGljZW5zZSh0aGlzLCB7XG4gICAgICAgIHNwZHg6IHRoaXMucGFja2FnZS5saWNlbnNlLFxuICAgICAgICBjb3B5cmlnaHRPd25lcjogb3B0aW9ucy5jb3B5cmlnaHRPd25lciA/PyBvcHRpb25zLmF1dGhvck5hbWUsXG4gICAgICAgIGNvcHlyaWdodFBlcmlvZDogb3B0aW9ucy5jb3B5cmlnaHRQZXJpb2QsXG4gICAgICB9KTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGFkZERlZmF1bHRHaXRJZ25vcmUoKSB7XG4gICAgdGhpcy5naXRpZ25vcmUuZXhjbHVkZShcbiAgICAgIFwiIyBMb2dzXCIsXG4gICAgICBcImxvZ3NcIixcbiAgICAgIFwiKi5sb2dcIixcbiAgICAgIFwibnBtLWRlYnVnLmxvZypcIixcbiAgICAgIFwieWFybi1kZWJ1Zy5sb2cqXCIsXG4gICAgICBcInlhcm4tZXJyb3IubG9nKlwiLFxuICAgICAgXCJsZXJuYS1kZWJ1Zy5sb2cqXCIsXG5cbiAgICAgIFwiIyBEaWFnbm9zdGljIHJlcG9ydHMgKGh0dHBzOi8vbm9kZWpzLm9yZy9hcGkvcmVwb3J0Lmh0bWwpXCIsXG4gICAgICBcInJlcG9ydC5bMC05XSouWzAtOV0qLlswLTldKi5bMC05XSouanNvblwiLFxuXG4gICAgICBcIiMgUnVudGltZSBkYXRhXCIsXG4gICAgICBcInBpZHNcIixcbiAgICAgIFwiKi5waWRcIixcbiAgICAgIFwiKi5zZWVkXCIsXG4gICAgICBcIioucGlkLmxvY2tcIixcblxuICAgICAgXCIjIERpcmVjdG9yeSBmb3IgaW5zdHJ1bWVudGVkIGxpYnMgZ2VuZXJhdGVkIGJ5IGpzY292ZXJhZ2UvSlNDb3ZlclwiLFxuICAgICAgXCJsaWItY292XCIsXG5cbiAgICAgIFwiIyBDb3ZlcmFnZSBkaXJlY3RvcnkgdXNlZCBieSB0b29scyBsaWtlIGlzdGFuYnVsXCIsXG4gICAgICBcImNvdmVyYWdlXCIsXG4gICAgICBcIioubGNvdlwiLFxuXG4gICAgICBcIiMgbnljIHRlc3QgY292ZXJhZ2VcIixcbiAgICAgIFwiLm55Y19vdXRwdXRcIixcblxuICAgICAgXCIjIENvbXBpbGVkIGJpbmFyeSBhZGRvbnMgKGh0dHBzOi8vbm9kZWpzLm9yZy9hcGkvYWRkb25zLmh0bWwpXCIsXG4gICAgICBcImJ1aWxkL1JlbGVhc2VcIixcblxuICAgICAgXCIjIERlcGVuZGVuY3kgZGlyZWN0b3JpZXNcIixcbiAgICAgIFwibm9kZV9tb2R1bGVzL1wiLFxuICAgICAgXCJqc3BtX3BhY2thZ2VzL1wiLFxuXG4gICAgICBcIiMgVHlwZVNjcmlwdCBjYWNoZVwiLFxuICAgICAgXCIqLnRzYnVpbGRpbmZvXCIsXG5cbiAgICAgIFwiIyBPcHRpb25hbCBlc2xpbnQgY2FjaGVcIixcbiAgICAgIFwiLmVzbGludGNhY2hlXCIsXG5cbiAgICAgIFwiIyBPdXRwdXQgb2YgJ25wbSBwYWNrJ1wiLFxuICAgICAgXCIqLnRnelwiLFxuXG4gICAgICBcIiMgWWFybiBJbnRlZ3JpdHkgZmlsZVwiLFxuICAgICAgXCIueWFybi1pbnRlZ3JpdHlcIixcblxuICAgICAgXCIjIHBhcmNlbC1idW5kbGVyIGNhY2hlIChodHRwczovL3BhcmNlbGpzLm9yZy8pXCIsXG4gICAgICBcIi5jYWNoZVwiXG4gICAgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBzaGVsbCBjb21tYW5kIHRvIGV4ZWN1dGUgaW4gb3JkZXIgdG8gcnVuIGEgdGFzay4gVGhpcyB3aWxsXG4gICAqIHR5cGljYWxseSBiZSBgbnB4IHByb2plbiBUQVNLYC5cbiAgICpcbiAgICogQHBhcmFtIHRhc2sgVGhlIHRhc2sgZm9yIHdoaWNoIHRoZSBjb21tYW5kIGlzIHJlcXVpcmVkXG4gICAqL1xuICBwdWJsaWMgcnVuVGFza0NvbW1hbmQodGFzazogVGFzaykge1xuICAgIHJldHVybiBgJHt0aGlzLnBhY2thZ2UucHJvamVuQ29tbWFuZH0gJHt0YXNrLm5hbWV9YDtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgam9iIElEIG9mIHRoZSBidWlsZCB3b3JrZmxvdy5cbiAgICovXG4gIHB1YmxpYyBnZXQgYnVpbGRXb3JrZmxvd0pvYklkKCkge1xuICAgIHJldHVybiB0aGlzLmJ1aWxkV29ya2Zsb3c/LmJ1aWxkSm9iSWRzWzBdO1xuICB9XG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgYHJlbmRlckluc3RhbGxTdGVwcygpYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSZW5kZXJXb3JrZmxvd1NldHVwT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBTaG91bGQgdGhlIHBhY2FrZ2UgbG9ja2ZpbGUgYmUgdXBkYXRlZD9cbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IG11dGFibGU/OiBib29sZWFuO1xufVxuIl19