"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const util_1 = require("./util");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
/**
 * Adds support for bundling JavaScript applications and dependencies into a
 * single file. In the future, this will also supports bundling websites.
 */
class Bundler extends component_1.Component {
    /**
     * Creates a `Bundler`.
     */
    constructor(project, options = {}) {
        super(project);
        this.esbuildVersion = options.esbuildVersion;
        this.bundledir = options.assetsDir ?? "assets";
        this.addToPreCompile = options.addToPreCompile ?? true;
        this.loaders = options.loaders;
    }
    /**
     * Returns the `Bundler` instance associated with a project or `undefined` if
     * there is no Bundler.
     * @param project The project
     * @returns A bundler
     */
    static of(project) {
        const isBundler = (o) => o instanceof Bundler;
        return project.components.find(isBundler);
    }
    /**
     * Gets or creates the singleton "bundle" task of the project.
     *
     * If the project doesn't have a "bundle" task, it will be created and spawned
     * during the pre-compile phase.
     */
    get bundleTask() {
        if (!this._task) {
            this.addBundlingSupport();
            this._task = this.project.tasks.addTask("bundle", {
                description: "Prepare assets",
            });
            // install the bundle task into the pre-compile phase.
            if (this.addToPreCompile) {
                this.project.preCompileTask.spawn(this._task);
            }
        }
        return this._task;
    }
    /**
     * Adds a task to the project which bundles a specific entrypoint and all of
     * its dependencies into a single javascript output file.
     *
     * @param entrypoint The relative path of the artifact within the project
     * @param options Bundling options
     */
    addBundle(entrypoint, options) {
        const name = util_1.renderBundleName(entrypoint);
        const outdir = path.posix.join(this.bundledir, name);
        const outfile = path.posix.join(outdir, options.outfile ?? "index.js");
        const args = [
            "esbuild",
            "--bundle",
            entrypoint,
            `--target="${options.target}"`,
            `--platform="${options.platform}"`,
            `--outfile="${outfile}"`,
        ];
        const tsconfig = options.tsconfigPath ?? false;
        if (tsconfig) {
            args.push(`--tsconfig="${tsconfig}"`);
        }
        for (const x of options.externals ?? []) {
            args.push(`--external:${x}`);
        }
        const sourcemap = options.sourcemap ?? false;
        if (sourcemap) {
            args.push("--sourcemap");
        }
        const loaders = options.loaders ?? false ? options.loaders : this.loaders ?? false;
        if (loaders) {
            for (let [extension, loader] of Object.entries(loaders)) {
                args.push(`--loader:.${extension}=${loader}`);
            }
        }
        const bundleTask = this.project.addTask(`bundle:${name}`, {
            description: `Create a JavaScript bundle from ${entrypoint}`,
            exec: args.join(" "),
        });
        this.bundleTask.spawn(bundleTask);
        if (options.executable ?? false) {
            bundleTask.exec(`chmod +x ${outfile}`);
        }
        let watchTask;
        const watch = options.watchTask ?? true;
        if (watch) {
            watchTask = this.project.addTask(`bundle:${name}:watch`, {
                description: `Continuously update the JavaScript bundle from ${entrypoint}`,
                exec: `${args.join(" ")} --watch`,
            });
        }
        return {
            bundleTask: bundleTask,
            watchTask: watchTask,
            outdir: outdir,
            outfile: outfile,
        };
    }
    /**
     * Add bundling support to a project. This is called implicitly when
     * `bundleTask` is referenced first. It adds the dependency on `esbuild`,
     * gitignore/npmignore, etc.
     */
    addBundlingSupport() {
        const ignoreEntry = `/${this.bundledir}/`;
        this.project.addGitIgnore(ignoreEntry);
        this.project.addPackageIgnore(`!${ignoreEntry}`); // include in tarball
        const dep = this.esbuildVersion
            ? `esbuild@${this.esbuildVersion}`
            : "esbuild";
        this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
    }
}
exports.Bundler = Bundler;
_a = JSII_RTTI_SYMBOL_1;
Bundler[_a] = { fqn: "projen.javascript.Bundler", version: "0.67.87" };
//# sourceMappingURL=data:application/json;base64,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