"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.readJsiiManifest = exports.resolveProjectType = exports.readManifest = exports.discover = void 0;
const path = require("path");
const zlib_1 = require("zlib");
const case_1 = require("case");
const fs = require("fs-extra");
const PROJEN_MODULE_ROOT = path.join(__dirname, "..");
const PROJECT_BASE_FQN = "projen.Project";
/**
 * Returns a list of project types exported the modules defined in `moduleDirs`.
 * This list will always also include the built-in projen project types.
 * Modules without a .jsii manifest are skipped.
 *
 * @param moduleDirs A list of npm module directories
 */
function discover(...moduleDirs) {
    const jsii = discoverJsiiTypes(...moduleDirs);
    const result = new Array();
    for (const fqn of Object.keys(jsii)) {
        if (isProjectType(jsii, fqn)) {
            const p = toProjectType(jsii, fqn);
            result.push(p);
        }
    }
    return result.sort((r1, r2) => r1.pjid.localeCompare(r2.pjid));
}
exports.discover = discover;
function readManifest(dir) {
    const jsiiFile = path.join(dir, ".jsii");
    if (!fs.existsSync(jsiiFile)) {
        return undefined;
    } // no jsii manifest
    let manifest = fs.readJsonSync(jsiiFile);
    if (manifest.schema === "jsii/file-redirect") {
        const compressedFile = path.join(dir, manifest.filename);
        if (!fs.existsSync(compressedFile)) {
            throw new Error(`${compressedFile} does not exist.`);
        }
        switch (manifest.compression) {
            case "gzip":
                manifest = JSON.parse(zlib_1.unzipSync(fs.readFileSync(compressedFile)).toString());
                break;
            default:
                throw new Error(`Unsupported compression format: ${manifest.compression}`);
        }
    }
    return manifest;
}
exports.readManifest = readManifest;
/**
 * Resolve all jsii types from @modulesDirs.
 * When a jsii module is found it will recusively list the types from the dependant module as well
 *
 * @param moduleDirs
 * @returns
 */
function discoverJsiiTypes(...moduleDirs) {
    const jsii = {};
    const discoveredManifests = [];
    const discoverJsii = (dir) => {
        const manifest = readManifest(dir);
        if (!manifest) {
            return;
        }
        if (discoveredManifests.includes(manifest.fingerprint)) {
            return;
        }
        discoveredManifests.push(manifest.fingerprint);
        for (const [fqn, type] of Object.entries(manifest.types)) {
            jsii[fqn] = {
                ...type,
            };
        }
        // Also search recursively in nested project dependencies. If the requested module is an external module
        // this will also end-up in the projen module and add the projen types
        if (manifest.dependencies) {
            for (const dependency of Object.keys(manifest.dependencies)) {
                const nestedDependencyFolder = path.dirname(require.resolve(`${dependency}/package.json`, {
                    paths: [dir],
                }));
                if (fs.existsSync(nestedDependencyFolder)) {
                    discoverJsii(nestedDependencyFolder);
                }
            }
        }
    };
    // read all .jsii manifests from all requested modules and merge
    // them all into a single map of fqn->type.
    for (const dir of [...moduleDirs, PROJEN_MODULE_ROOT]) {
        discoverJsii(dir);
        // Read from scoped packages
        if (dir.includes("@") && fs.lstatSync(dir).isDirectory()) {
            const childDirs = fs.readdirSync(dir).map((file) => path.join(dir, file));
            for (const child of childDirs) {
                discoverJsii(child);
            }
        }
    }
    return jsii;
}
function resolveProjectType(projectFqn) {
    let [moduleName] = projectFqn.split(".");
    if (moduleName === "projen") {
        moduleName = PROJEN_MODULE_ROOT;
    }
    // try picking the manifest. We only need the base folder but this is directly a nice check if we request from a valid jsii package
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`, {
        paths: [process.cwd()],
    });
    const moduleFolder = path.dirname(jsiiManifestFile);
    // Read all jsii types that can be loaded from this project type
    const jsii = discoverJsiiTypes(moduleFolder);
    return toProjectType(jsii, projectFqn);
}
exports.resolveProjectType = resolveProjectType;
function toProjectType(jsii, fqn) {
    if (!isProjectType(jsii, fqn)) {
        throw new Error(`Fully qualified name "${fqn}" is not a valid project type.`);
    }
    const typeinfo = jsii[fqn];
    // projen.web.ReactProject -> web.ReactProject
    const typename = fqn.substring(fqn.indexOf(".") + 1);
    const docsurl = `https://projen.io/api/API.html#projen-${typename
        .toLocaleLowerCase()
        .replace(/\./g, "-")}`;
    let pjid = typeinfo.docs?.custom?.pjid ?? case_1.snake(typename).replace(/_project$/, "");
    return {
        moduleName: typeinfo.assembly,
        typename,
        pjid,
        fqn,
        options: discoverOptions(jsii, fqn).sort((o1, o2) => o1.name.localeCompare(o2.name)),
        docs: typeinfo.docs?.summary,
        docsurl,
    };
}
function readJsiiManifest(jsiiFqn) {
    let [moduleName] = jsiiFqn.split(".");
    if (moduleName === "projen") {
        moduleName = PROJEN_MODULE_ROOT;
    }
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`);
    return fs.readJsonSync(jsiiManifestFile);
}
exports.readJsiiManifest = readJsiiManifest;
function discoverOptions(jsii, fqn) {
    const options = {};
    const params = jsii[fqn]?.initializer?.parameters ?? [];
    const optionsParam = params[0];
    const optionsTypeFqn = optionsParam?.type?.fqn;
    if (params.length > 1 ||
        (params.length === 1 && optionsParam?.name !== "options")) {
        throw new Error(`constructor for project ${fqn} must have a single "options" argument of a struct type. got ${JSON.stringify(params)}`);
    }
    addOptions(optionsTypeFqn);
    const opts = Object.values(options);
    return opts.sort((a, b) => a.switch.localeCompare(b.switch));
    function addOptions(ofqn, basePath = [], optional = false) {
        if (!ofqn) {
            return;
        }
        const struct = jsii[ofqn];
        if (!struct) {
            throw new Error(`unable to find options type ${ofqn} for project ${fqn}`);
        }
        for (const prop of struct.properties ?? []) {
            const propPath = [...basePath, prop.name];
            // protect against double-booking
            if (prop.name in options) {
                throw new Error(`duplicate option "${prop.name}" in ${fqn} (already declared in ${options[prop.name].parent})`);
            }
            let jsiiKind;
            if (prop.type?.fqn) {
                jsiiKind = jsii[prop.type?.fqn].kind; // e.g. 'class', 'interface', 'enum'
            }
            const isOptional = optional || prop.optional;
            let defaultValue = prop.docs?.default;
            if (defaultValue === "undefined") {
                defaultValue = undefined;
            }
            // if this is a mandatory option and we have a default value, it has to be JSON-parsable to the correct type
            if (!isOptional && defaultValue) {
                if (!prop.type?.primitive) {
                    throw new Error(`required option "${prop.name}" with a @default must use primitive types (string, number or boolean). type found is: ${JSON.stringify(prop.type)}`);
                }
                checkDefaultIsParsable(prop.name, defaultValue, prop.type?.primitive);
            }
            options[prop.name] = filterUndefined({
                path: propPath,
                parent: struct.name,
                name: prop.name,
                fqn: prop.type?.fqn,
                docs: prop.docs.summary,
                simpleType: prop.type ? getSimpleTypeName(prop.type) : "unknown",
                fullType: prop.type,
                kind: jsiiKind,
                jsonLike: prop.type ? isJsonLike(jsii, prop.type) : undefined,
                switch: propPath.map((p) => case_1.snake(p).replace(/_/g, "-")).join("-"),
                default: defaultValue,
                optional: isOptional,
                featured: prop.docs?.custom?.featured === "true",
                deprecated: prop.docs.stability === "deprecated" ? true : undefined,
            });
        }
        for (const ifc of struct.interfaces ?? []) {
            addOptions(ifc);
        }
    }
}
function getSimpleTypeName(type) {
    if (type?.primitive) {
        return type.primitive; // e.g. 'string', 'boolean', 'number'
    }
    else if (type?.fqn) {
        return type.fqn.split(".").pop(); // projen.NodeProjectOptions -> NodeProjectOptions
    }
    else {
        // any other types such as collection types
        return "unknown";
    }
}
/**
 * Whether a value of this type is serializable into JSON.
 */
function isJsonLike(jsii, type) {
    if (type.primitive) {
        // string, boolean, number, any
        return true;
    }
    else if (type.fqn) {
        const kind = jsii[type.fqn].kind;
        if (["interface", "enum"].includes(kind)) {
            // not 'class'
            return true;
        }
    }
    else if (type.collection) {
        return isJsonLike(jsii, type.collection.elementtype);
    }
    return false;
}
function filterUndefined(obj) {
    const ret = {};
    for (const [k, v] of Object.entries(obj)) {
        if (v !== undefined) {
            ret[k] = v;
        }
    }
    return ret;
}
function isProjectType(jsii, fqn) {
    const type = jsii[fqn];
    if (!type) {
        throw new Error(`Could not find project type with fqn "${fqn}" in  .jsii file.`);
    }
    if (type.kind !== "class") {
        return false;
    }
    if (type.abstract) {
        return false;
    }
    if (type.docs?.deprecated) {
        return false;
    }
    let curr = type;
    while (true) {
        if (curr.fqn === PROJECT_BASE_FQN) {
            return true;
        }
        if (!curr.base) {
            return false;
        }
        curr = jsii[curr.base];
        if (!curr) {
            return false;
        }
    }
}
function checkDefaultIsParsable(prop, value, type) {
    // macros are pass-through
    if (value.startsWith("$")) {
        return;
    }
    try {
        const parsed = JSON.parse(value);
        if (typeof parsed !== type) {
            throw new Error(`cannot parse @default value for mandatory option ${prop} as a ${type}: ${parsed}`);
        }
    }
    catch (e) {
        throw new Error(`unable to JSON.parse() value "${value}" specified as @default for mandatory option "${prop}": ${e.message}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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