"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CiConfiguration = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const case_1 = require("case");
const component_1 = require("../component");
const yaml_1 = require("../yaml");
/**
 * CI for GitLab.
 * A CI is a configurable automated process made up of one or more stages/jobs.
 * @see https://docs.gitlab.com/ee/ci/yaml/
 */
class CiConfiguration extends component_1.Component {
    constructor(project, name, options) {
        super(project);
        /**
         * Defines default scripts that should run *after* all jobs. Can be overriden by the job level `afterScript`.
         */
        this.defaultAfterScript = [];
        /**
         * Defines default scripts that should run *before* all jobs. Can be overriden by the job level `afterScript`.
         */
        this.defaultBeforeScript = [];
        /**
         * A default list of additional Docker images to run scripts in. The service image is linked to the image specified in the  image parameter.
         */
        this.defaultServices = [];
        /**
         * Used to select a specific runner from the list of all runners that are available for the project.
         */
        this.defaultTags = [];
        /**
         * Can be `Include` or `Include[]`. Each `Include` will be a string, or an
         * object with properties for the method if including external YAML file. The external
         * content will be fetched, included and evaluated along the `.gitlab-ci.yml`.
         */
        this.include = [];
        /**
         * Groups jobs into stages. All jobs in one stage must complete before next stage is
         * executed. Defaults to ['build', 'test', 'deploy'].
         */
        this.stages = [];
        /**
         * Global variables that are passed to jobs.
         * If the job already has that variable defined, the job-level variable takes precedence.
         */
        this.variables = {};
        /**
         * The jobs in the CI configuration.
         */
        this.jobs = {};
        this.project = project;
        this.name = path.parse(name).name;
        this.path =
            this.name === "gitlab-ci"
                ? ".gitlab-ci.yml"
                : `.gitlab/ci-templates/${name.toLocaleLowerCase()}.yml`;
        this.file = new yaml_1.YamlFile(this.project, this.path, {
            obj: () => this.renderCI(),
            // GitLab needs to read the file from the repository in order to work.
            committed: true,
        });
        const defaults = options?.default;
        if (defaults) {
            this.defaultAfterScript.push(...(defaults.afterScript ?? []));
            this.defaultArtifacts = defaults.artifacts;
            defaults.beforeScript &&
                this.defaultBeforeScript.push(...defaults.beforeScript);
            this.defaultCache = defaults.cache;
            this.defaultImage = defaults.image;
            this.defaultInterruptible = defaults.interruptible;
            this.defaultRetry = defaults.retry;
            defaults.services && this.addServices(...defaults.services);
            defaults.tags && this.defaultTags.push(...defaults.tags);
            this.defaultTimeout = defaults.timeout;
        }
        this.pages = options?.pages;
        this.workflow = options?.workflow;
        if (options?.stages) {
            this.addStages(...options.stages);
        }
        if (options?.variables) {
            this.addGlobalVariables(options.variables);
        }
        if (options?.jobs) {
            this.addJobs(options.jobs);
        }
    }
    /**
     * Add additional yml/yaml files to the CI includes
     * @param includes The includes to add.
     */
    addIncludes(...includes) {
        for (const additional of includes) {
            this.assertIsValidInclude(additional);
            for (const existing of this.include) {
                if (this.areEqualIncludes(existing, additional)) {
                    throw new Error(`${this.name}: GitLab CI ${existing} already contains one or more templates specified in ${additional}.`);
                }
            }
            this.include.push(additional);
        }
    }
    /**
     * Throw an error if the provided Include is invalid.
     * @see https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/config/external/mapper.rb
     * @param include the Include to validate.
     */
    assertIsValidInclude(include) {
        const combos = [
            include.local,
            include.file && include.project,
            include.remote,
            include.template,
        ];
        const len = combos.filter((x) => Boolean(x)).length;
        if (len !== 1) {
            throw new Error(`${this.name}: GitLab CI include ${include} contains ${len} property combination(s).
        A valid include configuration specifies *one* of the following property combinations.
        * local
        * file, project
        * remote
        * template  
        `);
        }
    }
    /**
     * Check if the equality of Includes.
     * @see https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/config/external/mapper.rb
     * @param x First include to compare.
     * @param y Second include to compare.
     * @returns Whether the includes are equal.
     */
    areEqualIncludes(x, y) {
        if (x.local === y.local && x.local !== undefined) {
            return true;
        }
        else if (x.template === y.template && x.template !== undefined) {
            return true;
        }
        else if (x.remote === y.remote && x.remote !== undefined) {
            return true;
        }
        else if (x.project === y.project && x.ref === y.ref) {
            const xFiles = x.file ? x.file : [];
            const yFiles = y.file ? y.file : [];
            const allFiles = xFiles.concat(yFiles);
            return new Set(allFiles).size !== allFiles.length;
        }
        return false;
    }
    /**
     * Add additional services.
     * @param services The services to add.
     */
    addServices(...services) {
        for (const additional of services) {
            for (const existing of this.defaultServices) {
                if (additional.name === existing.name &&
                    additional.alias === existing.alias) {
                    throw new Error(`${this.name}: GitLab CI already contains service ${additional}.`);
                }
            }
            this.defaultServices.push(additional);
        }
    }
    /**
     * Add a globally defined variable to the CI configuration.
     * @param variables The variables to add.
     */
    addGlobalVariables(variables) {
        for (const [key, value] of Object.entries(variables)) {
            if (this.variables[key] !== undefined) {
                throw new Error(`${this.name}: GitLab CI already contains variable ${key}.`);
            }
            this.variables[key] = value;
        }
    }
    /**
     * Add stages to the CI configuration if not already present.
     * @param stages stages to add.
     */
    addStages(...stages) {
        for (const stage of stages) {
            if (!this.stages.includes(stage)) {
                this.stages.push(stage);
            }
        }
    }
    /**
     * Add jobs and their stages to the CI configuration.
     * @param jobs Jobs to add.
     */
    addJobs(jobs) {
        for (const [key, value] of Object.entries(jobs)) {
            if (this.jobs[key] !== undefined) {
                throw new Error(`${this.name}: GitLab CI already contains job ${key}.`);
            }
            this.jobs[key] = value;
            if (value.stage) {
                this.addStages(value.stage);
            }
        }
    }
    renderCI() {
        return {
            default: this.renderDefault(),
            include: this.include.length > 0 ? snakeCaseKeys(this.include) : undefined,
            pages: snakeCaseKeys(this.pages),
            services: this.defaultServices.length > 0
                ? snakeCaseKeys(this.defaultServices)
                : undefined,
            variables: Object.entries(this.variables).length > 0 ? this.variables : undefined,
            workflow: snakeCaseKeys(this.workflow),
            stages: this.stages.length > 0 ? this.stages : undefined,
            // we do not want to change job names
            // as they can be hidden (https://docs.gitlab.com/ee/ci/jobs/index.html#hide-jobs)
            // or referenced in extends
            ...snakeCaseKeys(this.jobs, true),
        };
    }
    renderDefault() {
        const defaults = {
            afterScript: this.defaultAfterScript.length > 0
                ? this.defaultAfterScript
                : undefined,
            artifacts: this.defaultArtifacts,
            beforeScript: this.defaultBeforeScript.length > 0
                ? this.defaultBeforeScript
                : undefined,
            cache: this.defaultCache,
            image: this.defaultImage,
            interruptible: this.defaultInterruptible,
            retry: this.defaultRetry,
            services: this.defaultServices.length > 0 ? this.defaultServices : undefined,
            tags: this.defaultTags.length > 0 ? this.defaultTags : undefined,
            timeout: this.defaultTimeout,
        };
        return Object.values(defaults).filter((x) => x).length
            ? snakeCaseKeys(defaults)
            : undefined;
    }
}
exports.CiConfiguration = CiConfiguration;
_a = JSII_RTTI_SYMBOL_1;
CiConfiguration[_a] = { fqn: "projen.gitlab.CiConfiguration", version: "0.67.87" };
function snakeCaseKeys(obj, skipTopLevel = false) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map((o) => snakeCaseKeys(o));
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (typeof v === "object" && v != null && k !== "variables") {
            v = snakeCaseKeys(v);
        }
        result[skipTopLevel ? k : case_1.snake(k)] = v;
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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