"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubCredentials = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const util_1 = require("../util");
/**
 * Represents a method of providing GitHub API access for projen workflows.
 */
class GithubCredentials {
    constructor(options) {
        this.options = options;
    }
    /**
     * Provide API access through a GitHub personal access token.
     *
     * The token must be added as a secret to the GitHub repo, and the name of the
     * secret can be specified here.
     *
     * @see https://docs.github.com/en/authentication/keeping-your-account-and-data-secure/creating-a-personal-access-token
     * @default - a secret named "PROJEN_GITHUB_TOKEN"
     */
    static fromPersonalAccessToken(options = {}) {
        return new GithubCredentials({
            setupSteps: [],
            tokenRef: `\${{ secrets.${options.secret ?? "PROJEN_GITHUB_TOKEN"} }}`,
        });
    }
    /**
     * Provide API access through a GitHub App.
     *
     * The GitHub App must be installed on the GitHub repo, its App ID and a
     * private key must be added as secrets to the repo. The name of the secrets
     * can be specified here.
     *
     * @see https://docs.github.com/en/developers/apps/building-github-apps/creating-a-github-app
     * @see https://projen.io/github.html#github-app
     * @default - app id stored in "PROJEN_APP_ID" and private key stored in "PROJEN_APP_PRIVATE_KEY" with all permissions attached to the app
     */
    static fromApp(options = {}) {
        const appIdSecret = options.appIdSecret ?? "PROJEN_APP_ID";
        const privateKeySecret = options.privateKeySecret ?? "PROJEN_APP_PRIVATE_KEY";
        return new GithubCredentials({
            setupSteps: [
                {
                    name: "Generate token",
                    id: "generate_token",
                    uses: "tibdex/github-app-token@021a2405c7f990db57f5eae5397423dcc554159c",
                    with: {
                        app_id: `\${{ secrets.${appIdSecret} }}`,
                        private_key: `\${{ secrets.${privateKeySecret} }}`,
                        permissions: options.permissions
                            ? JSON.stringify(util_1.snakeCaseKeys(options.permissions))
                            : undefined,
                    },
                },
            ],
            tokenRef: "${{ steps.generate_token.outputs.token }}",
        });
    }
    /**
     * Setup steps to obtain GitHub credentials.
     */
    get setupSteps() {
        return [...this.options.setupSteps];
    }
    /**
     * The value to use in a workflow when a GitHub token is expected. This
     * typically looks like "${{ some.path.to.a.value }}".
     */
    get tokenRef() {
        return this.options.tokenRef;
    }
}
exports.GithubCredentials = GithubCredentials;
_a = JSII_RTTI_SYMBOL_1;
GithubCredentials[_a] = { fqn: "projen.github.GithubCredentials", version: "0.67.87" };
//# sourceMappingURL=data:application/json;base64,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