"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerCompose = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const docker_compose_port_1 = require("./docker-compose-port");
const docker_compose_render_1 = require("./docker-compose-render");
const docker_compose_service_1 = require("./docker-compose-service");
const component_1 = require("../component");
const yaml_1 = require("../yaml");
/**
 * Create a docker-compose YAML file.
 */
class DockerCompose extends component_1.Component {
    constructor(project, props) {
        super(project);
        const nameSuffix = props?.nameSuffix ? `${props.nameSuffix}.yml` : "yml";
        new yaml_1.YamlFile(project, `docker-compose.${nameSuffix}`, {
            committed: true,
            readonly: true,
            obj: () => this._synthesizeDockerCompose(),
        });
        if (props?.schemaVersion && !parseFloat(props.schemaVersion)) {
            throw Error("Version tag needs to be a number");
        }
        this.version = props?.schemaVersion ? props.schemaVersion : "3.3";
        this.services = {};
        // Add the services provided via the constructor argument.
        const initialServices = props?.services ?? {};
        for (const [name, serviceDescription] of Object.entries(initialServices)) {
            this.addService(name, serviceDescription);
        }
    }
    /**
     * Depends on a service name.
     */
    static serviceName(serviceName) {
        return {
            serviceName,
        };
    }
    /**
     * Create a port mapping.
     * @param publishedPort Published port number
     * @param targetPort Container's port number
     * @param options Port mapping options
     */
    static portMapping(publishedPort, targetPort, options) {
        const protocol = options?.protocol ?? docker_compose_port_1.DockerComposeProtocol.TCP;
        return {
            target: targetPort,
            published: publishedPort,
            protocol: protocol,
            mode: "host",
        };
    }
    /**
     * Create a bind volume that binds a host path to the target path in the container.
     * @param sourcePath Host path name
     * @param targetPath Target path name
     */
    static bindVolume(sourcePath, targetPath) {
        return {
            bind(_volumeInfo) {
                return {
                    type: "bind",
                    source: sourcePath,
                    target: targetPath,
                };
            },
        };
    }
    /**
     * Create a named volume and mount it to the target path. If you use this
     * named volume in several services, the volume will be shared. In this
     * case, the volume configuration of the first-provided options are used.
     *
     * @param volumeName Name of the volume
     * @param targetPath Target path
     * @param options volume configuration (default: docker compose defaults)
     */
    static namedVolume(volumeName, targetPath, options = {}) {
        return {
            bind(volumeInfo) {
                volumeInfo.addVolumeConfiguration(volumeName, options);
                return {
                    type: "volume",
                    source: volumeName,
                    target: targetPath,
                };
            },
        };
    }
    /**
     * Create a named network and mount it to the target path. If you use this
     * named network in several services, the network will be shared. In this
     * case, the network configuration of the first-provided options are used.
     *
     * @param networkName Name of the network
     * @param options network configuration
     */
    static network(networkName, options = {}) {
        return {
            bind(networkInfo) {
                networkInfo.addNetworkConfiguration(networkName, options);
                return networkName;
            },
        };
    }
    /**
     * Add a service to the docker-compose file.
     * @param serviceName name of the service
     * @param description a service description
     */
    addService(serviceName, description) {
        const service = new docker_compose_service_1.DockerComposeService(serviceName, description);
        this.services[serviceName] = service;
        return service;
    }
    /**
     * @internal
     */
    _synthesizeDockerCompose() {
        if (Object.keys(this.services).length === 0) {
            throw new Error("DockerCompose requires at least one service");
        }
        return docker_compose_render_1.renderDockerComposeFile(this.services, this.version);
    }
}
exports.DockerCompose = DockerCompose;
_a = JSII_RTTI_SYMBOL_1;
DockerCompose[_a] = { fqn: "projen.DockerCompose", version: "0.67.87" };
//# sourceMappingURL=data:application/json;base64,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