"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderDockerComposeFile = void 0;
const util_1 = require("../util");
function renderDockerComposeFile(serviceDescriptions, version) {
    // Record volume configuration
    const volumeConfig = {};
    const volumeInfo = {
        addVolumeConfiguration(volumeName, configuration) {
            if (!volumeConfig[volumeName]) {
                // First volume configuration takes precedence.
                volumeConfig[volumeName] = configuration;
            }
        },
    };
    // Record network configuration
    const networkConfig = {};
    const networkInfo = {
        addNetworkConfiguration(networkName, configuration) {
            if (!networkConfig[networkName]) {
                // First network configuration takes precedence.
                networkConfig[networkName] = configuration;
            }
        },
    };
    // Render service configuration
    const services = {};
    for (const [serviceName, serviceDescription] of Object.entries(serviceDescriptions ?? {})) {
        // Resolve the names of each dependency and check that they exist.
        // Note: They may not exist if the user made a mistake when referencing a
        // service by name via `DockerCompose.serviceName()`.
        // @see DockerCompose.serviceName
        const dependsOn = Array();
        for (const dependsOnServiceName of serviceDescription.dependsOn ?? []) {
            const resolvedServiceName = dependsOnServiceName.serviceName;
            if (resolvedServiceName === serviceName) {
                throw new Error(`Service ${serviceName} cannot depend on itself`);
            }
            if (!serviceDescriptions[resolvedServiceName]) {
                throw new Error(`Unable to resolve service named ${resolvedServiceName} for ${serviceName}`);
            }
            dependsOn.push(resolvedServiceName);
        }
        // Give each volume binding a chance to bind any necessary volume
        // configuration and provide volume mount information for the service.
        const volumes = [];
        for (const volumeBinding of serviceDescription.volumes ?? []) {
            volumes.push(volumeBinding.bind(volumeInfo));
        }
        // Give each network binding a chance to bind any necessary network
        // configuration and provide network mount information for the service.
        const networks = [];
        for (const networkBinding of serviceDescription.networks ?? []) {
            networks.push(networkBinding.bind(networkInfo));
        }
        // Create and store the service configuration, taking care not to create
        // object members with undefined values.
        services[serviceName] = {
            ...getObjectWithKeyAndValueIfValueIsDefined("image", serviceDescription.image),
            ...getObjectWithKeyAndValueIfValueIsDefined("build", serviceDescription.imageBuild),
            ...getObjectWithKeyAndValueIfValueIsDefined("command", serviceDescription.command),
            ...(Object.keys(serviceDescription.environment).length > 0
                ? { environment: serviceDescription.environment }
                : {}),
            ...(serviceDescription.ports.length > 0
                ? { ports: serviceDescription.ports }
                : {}),
            ...(Object.keys(serviceDescription.labels).length > 0
                ? { labels: serviceDescription.labels }
                : {}),
            ...(dependsOn.length > 0 ? { dependsOn } : {}),
            ...(volumes.length > 0 ? { volumes } : {}),
            ...(networks.length > 0 ? { networks } : {}),
        };
    }
    // Explicit with the type here because the decamelize step after this wipes
    // out types.
    const input = {
        version,
        services,
        ...(Object.keys(volumeConfig).length > 0 ? { volumes: volumeConfig } : {}),
        ...(Object.keys(networkConfig).length > 0
            ? { networks: networkConfig }
            : {}),
    };
    // Change most keys to snake case.
    return util_1.decamelizeKeysRecursively(input, {
        shouldDecamelize: shouldDecamelizeDockerComposeKey,
        separator: "_",
    });
}
exports.renderDockerComposeFile = renderDockerComposeFile;
/**
 * Returns `{ [key]: value }` if `value` is defined, otherwise returns `{}` so
 * that object spreading can be used to generate a peculiar interface.
 * @param key
 * @param value
 */
function getObjectWithKeyAndValueIfValueIsDefined(key, value) {
    return value !== undefined ? { [key]: value } : {};
}
/**
 * Determines whether the key at the given path should be decamelized.
 * Largely, all keys should be snake cased. But, there are some
 * exceptions for user-provided names for services, volumes, and
 * environment variables.
 *
 * @param path
 */
function shouldDecamelizeDockerComposeKey(path) {
    const poundPath = path.join("#");
    // Does not decamelize user's names.
    // services.namehere:
    // volumes.namehere:
    // networks.namehere:
    if (/^(services|volumes|networks)#[^#]+$/.test(poundPath)) {
        return false;
    }
    // Does not decamelize environment variables and labels
    // services.namehere.environment.*
    // services.namehere.labels.*
    if (/^services#[^#]+#(environment|labels)#/.test(poundPath)) {
        return false;
    }
    // Does not decamelize build arguments
    // services.namehere.build.args.*
    if (/^services#[^#]+#build#args#/.test(poundPath)) {
        return false;
    }
    // Otherwise, let it all decamelize.
    return true;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZG9ja2VyLWNvbXBvc2UtcmVuZGVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2RvY2tlci1jb21wb3NlL2RvY2tlci1jb21wb3NlLXJlbmRlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFZQSxrQ0FBb0Q7QUE0QnBELFNBQWdCLHVCQUF1QixDQUNyQyxtQkFBeUQsRUFDekQsT0FBZTtJQUVmLDhCQUE4QjtJQUM5QixNQUFNLFlBQVksR0FBOEMsRUFBRSxDQUFDO0lBQ25FLE1BQU0sVUFBVSxHQUErQjtRQUM3QyxzQkFBc0IsQ0FDcEIsVUFBa0IsRUFDbEIsYUFBd0M7WUFFeEMsSUFBSSxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsRUFBRTtnQkFDN0IsK0NBQStDO2dCQUMvQyxZQUFZLENBQUMsVUFBVSxDQUFDLEdBQUcsYUFBYSxDQUFDO2FBQzFDO1FBQ0gsQ0FBQztLQUNGLENBQUM7SUFDRiwrQkFBK0I7SUFDL0IsTUFBTSxhQUFhLEdBQStDLEVBQUUsQ0FBQztJQUNyRSxNQUFNLFdBQVcsR0FBZ0M7UUFDL0MsdUJBQXVCLENBQ3JCLFdBQW1CLEVBQ25CLGFBQXlDO1lBRXpDLElBQUksQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLEVBQUU7Z0JBQy9CLGdEQUFnRDtnQkFDaEQsYUFBYSxDQUFDLFdBQVcsQ0FBQyxHQUFHLGFBQWEsQ0FBQzthQUM1QztRQUNILENBQUM7S0FDRixDQUFDO0lBRUYsK0JBQStCO0lBQy9CLE1BQU0sUUFBUSxHQUFtRCxFQUFFLENBQUM7SUFDcEUsS0FBSyxNQUFNLENBQUMsV0FBVyxFQUFFLGtCQUFrQixDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FDNUQsbUJBQW1CLElBQUksRUFBRSxDQUMxQixFQUFFO1FBQ0Qsa0VBQWtFO1FBQ2xFLHlFQUF5RTtRQUN6RSxxREFBcUQ7UUFDckQsaUNBQWlDO1FBQ2pDLE1BQU0sU0FBUyxHQUFHLEtBQUssRUFBVSxDQUFDO1FBQ2xDLEtBQUssTUFBTSxvQkFBb0IsSUFBSSxrQkFBa0IsQ0FBQyxTQUFTLElBQUksRUFBRSxFQUFFO1lBQ3JFLE1BQU0sbUJBQW1CLEdBQUcsb0JBQW9CLENBQUMsV0FBVyxDQUFDO1lBQzdELElBQUksbUJBQW1CLEtBQUssV0FBVyxFQUFFO2dCQUN2QyxNQUFNLElBQUksS0FBSyxDQUFDLFdBQVcsV0FBVywwQkFBMEIsQ0FBQyxDQUFDO2FBQ25FO1lBQ0QsSUFBSSxDQUFDLG1CQUFtQixDQUFDLG1CQUFtQixDQUFDLEVBQUU7Z0JBQzdDLE1BQU0sSUFBSSxLQUFLLENBQ2IsbUNBQW1DLG1CQUFtQixRQUFRLFdBQVcsRUFBRSxDQUM1RSxDQUFDO2FBQ0g7WUFFRCxTQUFTLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLENBQUM7U0FDckM7UUFFRCxpRUFBaUU7UUFDakUsc0VBQXNFO1FBQ3RFLE1BQU0sT0FBTyxHQUErQixFQUFFLENBQUM7UUFDL0MsS0FBSyxNQUFNLGFBQWEsSUFBSSxrQkFBa0IsQ0FBQyxPQUFPLElBQUksRUFBRSxFQUFFO1lBQzVELE9BQU8sQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1NBQzlDO1FBRUQsbUVBQW1FO1FBQ25FLHVFQUF1RTtRQUN2RSxNQUFNLFFBQVEsR0FBYSxFQUFFLENBQUM7UUFDOUIsS0FBSyxNQUFNLGNBQWMsSUFBSSxrQkFBa0IsQ0FBQyxRQUFRLElBQUksRUFBRSxFQUFFO1lBQzlELFFBQVEsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDO1NBQ2pEO1FBRUQsd0VBQXdFO1FBQ3hFLHdDQUF3QztRQUN4QyxRQUFRLENBQUMsV0FBVyxDQUFDLEdBQUc7WUFDdEIsR0FBRyx3Q0FBd0MsQ0FDekMsT0FBTyxFQUNQLGtCQUFrQixDQUFDLEtBQUssQ0FDekI7WUFDRCxHQUFHLHdDQUF3QyxDQUN6QyxPQUFPLEVBQ1Asa0JBQWtCLENBQUMsVUFBVSxDQUM5QjtZQUNELEdBQUcsd0NBQXdDLENBQ3pDLFNBQVMsRUFDVCxrQkFBa0IsQ0FBQyxPQUFPLENBQzNCO1lBQ0QsR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsV0FBVyxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUM7Z0JBQ3hELENBQUMsQ0FBQyxFQUFFLFdBQVcsRUFBRSxrQkFBa0IsQ0FBQyxXQUFXLEVBQUU7Z0JBQ2pELENBQUMsQ0FBQyxFQUFFLENBQUM7WUFDUCxHQUFHLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLE1BQU0sR0FBRyxDQUFDO2dCQUNyQyxDQUFDLENBQUMsRUFBRSxLQUFLLEVBQUUsa0JBQWtCLENBQUMsS0FBSyxFQUFFO2dCQUNyQyxDQUFDLENBQUMsRUFBRSxDQUFDO1lBQ1AsR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsTUFBTSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUM7Z0JBQ25ELENBQUMsQ0FBQyxFQUFFLE1BQU0sRUFBRSxrQkFBa0IsQ0FBQyxNQUFNLEVBQUU7Z0JBQ3ZDLENBQUMsQ0FBQyxFQUFFLENBQUM7WUFDUCxHQUFHLENBQUMsU0FBUyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsU0FBUyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztZQUM5QyxHQUFHLENBQUMsT0FBTyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztZQUMxQyxHQUFHLENBQUMsUUFBUSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztTQUM3QyxDQUFDO0tBQ0g7SUFFRCwyRUFBMkU7SUFDM0UsYUFBYTtJQUNiLE1BQU0sS0FBSyxHQUE0QjtRQUNyQyxPQUFPO1FBQ1AsUUFBUTtRQUNSLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsT0FBTyxFQUFFLFlBQVksRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7UUFDMUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUM7WUFDdkMsQ0FBQyxDQUFDLEVBQUUsUUFBUSxFQUFFLGFBQWEsRUFBRTtZQUM3QixDQUFDLENBQUMsRUFBRSxDQUFDO0tBQ1IsQ0FBQztJQUVGLGtDQUFrQztJQUNsQyxPQUFPLGdDQUF5QixDQUFDLEtBQUssRUFBRTtRQUN0QyxnQkFBZ0IsRUFBRSxnQ0FBZ0M7UUFDbEQsU0FBUyxFQUFFLEdBQUc7S0FDZixDQUFDLENBQUM7QUFDTCxDQUFDO0FBbkhELDBEQW1IQztBQUVEOzs7OztHQUtHO0FBQ0gsU0FBUyx3Q0FBd0MsQ0FDL0MsR0FBTSxFQUNOLEtBQVE7SUFFUixPQUFPLEtBQUssS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO0FBQ3JELENBQUM7QUFFRDs7Ozs7OztHQU9HO0FBQ0gsU0FBUyxnQ0FBZ0MsQ0FBQyxJQUFjO0lBQ3RELE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7SUFFakMsb0NBQW9DO0lBQ3BDLHFCQUFxQjtJQUNyQixvQkFBb0I7SUFDcEIscUJBQXFCO0lBQ3JCLElBQUkscUNBQXFDLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFO1FBQ3pELE9BQU8sS0FBSyxDQUFDO0tBQ2Q7SUFFRCx1REFBdUQ7SUFDdkQsa0NBQWtDO0lBQ2xDLDZCQUE2QjtJQUM3QixJQUFJLHVDQUF1QyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRTtRQUMzRCxPQUFPLEtBQUssQ0FBQztLQUNkO0lBRUQsc0NBQXNDO0lBQ3RDLGlDQUFpQztJQUNqQyxJQUFJLDZCQUE2QixDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRTtRQUNqRCxPQUFPLEtBQUssQ0FBQztLQUNkO0lBRUQsb0NBQW9DO0lBQ3BDLE9BQU8sSUFBSSxDQUFDO0FBQ2QsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IERvY2tlckNvbXBvc2VCdWlsZCB9IGZyb20gXCIuL2RvY2tlci1jb21wb3NlXCI7XG5pbXBvcnQge1xuICBEb2NrZXJDb21wb3NlTmV0d29ya0NvbmZpZyxcbiAgSURvY2tlckNvbXBvc2VOZXR3b3JrQ29uZmlnLFxufSBmcm9tIFwiLi9kb2NrZXItY29tcG9zZS1uZXR3b3JrXCI7XG5pbXBvcnQgeyBEb2NrZXJDb21wb3NlU2VydmljZVBvcnQgfSBmcm9tIFwiLi9kb2NrZXItY29tcG9zZS1wb3J0XCI7XG5pbXBvcnQgeyBEb2NrZXJDb21wb3NlU2VydmljZSB9IGZyb20gXCIuL2RvY2tlci1jb21wb3NlLXNlcnZpY2VcIjtcbmltcG9ydCB7XG4gIERvY2tlckNvbXBvc2VWb2x1bWVDb25maWcsXG4gIERvY2tlckNvbXBvc2VWb2x1bWVNb3VudCxcbiAgSURvY2tlckNvbXBvc2VWb2x1bWVDb25maWcsXG59IGZyb20gXCIuL2RvY2tlci1jb21wb3NlLXZvbHVtZVwiO1xuaW1wb3J0IHsgZGVjYW1lbGl6ZUtleXNSZWN1cnNpdmVseSB9IGZyb20gXCIuLi91dGlsXCI7XG5cbi8qKlxuICogU3RydWN0dXJlIG9mIGEgZG9ja2VyIGNvbXBvc2UgZmlsZSdzIHNlcnZpY2UgYmVmb3JlIHdlIGRlY2FtZWxpemUuXG4gKiBAaW50ZXJuYWxcbiAqL1xuaW50ZXJmYWNlIERvY2tlckNvbXBvc2VGaWxlU2VydmljZVNjaGVtYSB7XG4gIHJlYWRvbmx5IGRlcGVuZHNPbj86IHN0cmluZ1tdO1xuICByZWFkb25seSBidWlsZD86IERvY2tlckNvbXBvc2VCdWlsZDtcbiAgcmVhZG9ubHkgaW1hZ2U/OiBzdHJpbmc7XG4gIHJlYWRvbmx5IGNvbW1hbmQ/OiBzdHJpbmdbXTtcbiAgcmVhZG9ubHkgdm9sdW1lcz86IERvY2tlckNvbXBvc2VWb2x1bWVNb3VudFtdO1xuICByZWFkb25seSBuZXR3b3Jrcz86IHN0cmluZ1tdO1xuICByZWFkb25seSBwb3J0cz86IERvY2tlckNvbXBvc2VTZXJ2aWNlUG9ydFtdO1xuICByZWFkb25seSBlbnZpcm9ubWVudD86IFJlY29yZDxzdHJpbmcsIHN0cmluZz47XG4gIHJlYWRvbmx5IGxhYmVscz86IFJlY29yZDxzdHJpbmcsIHN0cmluZz47XG59XG5cbi8qKlxuICogU3RydWN0dXJlIG9mIGEgZG9ja2VyIGNvbXBvc2UgZmlsZSBiZWZvcmUgd2UgZGVjYW1lbGl6ZS5cbiAqIEBpbnRlcm5hbFxuICovXG5pbnRlcmZhY2UgRG9ja2VyQ29tcG9zZUZpbGVTY2hlbWEge1xuICB2ZXJzaW9uOiBzdHJpbmc7XG4gIHNlcnZpY2VzOiBSZWNvcmQ8c3RyaW5nLCBEb2NrZXJDb21wb3NlRmlsZVNlcnZpY2VTY2hlbWE+O1xuICB2b2x1bWVzPzogUmVjb3JkPHN0cmluZywgRG9ja2VyQ29tcG9zZVZvbHVtZUNvbmZpZz47XG59XG5cbmV4cG9ydCBmdW5jdGlvbiByZW5kZXJEb2NrZXJDb21wb3NlRmlsZShcbiAgc2VydmljZURlc2NyaXB0aW9uczogUmVjb3JkPHN0cmluZywgRG9ja2VyQ29tcG9zZVNlcnZpY2U+LFxuICB2ZXJzaW9uOiBzdHJpbmdcbik6IG9iamVjdCB7XG4gIC8vIFJlY29yZCB2b2x1bWUgY29uZmlndXJhdGlvblxuICBjb25zdCB2b2x1bWVDb25maWc6IFJlY29yZDxzdHJpbmcsIERvY2tlckNvbXBvc2VWb2x1bWVDb25maWc+ID0ge307XG4gIGNvbnN0IHZvbHVtZUluZm86IElEb2NrZXJDb21wb3NlVm9sdW1lQ29uZmlnID0ge1xuICAgIGFkZFZvbHVtZUNvbmZpZ3VyYXRpb24oXG4gICAgICB2b2x1bWVOYW1lOiBzdHJpbmcsXG4gICAgICBjb25maWd1cmF0aW9uOiBEb2NrZXJDb21wb3NlVm9sdW1lQ29uZmlnXG4gICAgKSB7XG4gICAgICBpZiAoIXZvbHVtZUNvbmZpZ1t2b2x1bWVOYW1lXSkge1xuICAgICAgICAvLyBGaXJzdCB2b2x1bWUgY29uZmlndXJhdGlvbiB0YWtlcyBwcmVjZWRlbmNlLlxuICAgICAgICB2b2x1bWVDb25maWdbdm9sdW1lTmFtZV0gPSBjb25maWd1cmF0aW9uO1xuICAgICAgfVxuICAgIH0sXG4gIH07XG4gIC8vIFJlY29yZCBuZXR3b3JrIGNvbmZpZ3VyYXRpb25cbiAgY29uc3QgbmV0d29ya0NvbmZpZzogUmVjb3JkPHN0cmluZywgRG9ja2VyQ29tcG9zZU5ldHdvcmtDb25maWc+ID0ge307XG4gIGNvbnN0IG5ldHdvcmtJbmZvOiBJRG9ja2VyQ29tcG9zZU5ldHdvcmtDb25maWcgPSB7XG4gICAgYWRkTmV0d29ya0NvbmZpZ3VyYXRpb24oXG4gICAgICBuZXR3b3JrTmFtZTogc3RyaW5nLFxuICAgICAgY29uZmlndXJhdGlvbjogRG9ja2VyQ29tcG9zZU5ldHdvcmtDb25maWdcbiAgICApIHtcbiAgICAgIGlmICghbmV0d29ya0NvbmZpZ1tuZXR3b3JrTmFtZV0pIHtcbiAgICAgICAgLy8gRmlyc3QgbmV0d29yayBjb25maWd1cmF0aW9uIHRha2VzIHByZWNlZGVuY2UuXG4gICAgICAgIG5ldHdvcmtDb25maWdbbmV0d29ya05hbWVdID0gY29uZmlndXJhdGlvbjtcbiAgICAgIH1cbiAgICB9LFxuICB9O1xuXG4gIC8vIFJlbmRlciBzZXJ2aWNlIGNvbmZpZ3VyYXRpb25cbiAgY29uc3Qgc2VydmljZXM6IFJlY29yZDxzdHJpbmcsIERvY2tlckNvbXBvc2VGaWxlU2VydmljZVNjaGVtYT4gPSB7fTtcbiAgZm9yIChjb25zdCBbc2VydmljZU5hbWUsIHNlcnZpY2VEZXNjcmlwdGlvbl0gb2YgT2JqZWN0LmVudHJpZXMoXG4gICAgc2VydmljZURlc2NyaXB0aW9ucyA/PyB7fVxuICApKSB7XG4gICAgLy8gUmVzb2x2ZSB0aGUgbmFtZXMgb2YgZWFjaCBkZXBlbmRlbmN5IGFuZCBjaGVjayB0aGF0IHRoZXkgZXhpc3QuXG4gICAgLy8gTm90ZTogVGhleSBtYXkgbm90IGV4aXN0IGlmIHRoZSB1c2VyIG1hZGUgYSBtaXN0YWtlIHdoZW4gcmVmZXJlbmNpbmcgYVxuICAgIC8vIHNlcnZpY2UgYnkgbmFtZSB2aWEgYERvY2tlckNvbXBvc2Uuc2VydmljZU5hbWUoKWAuXG4gICAgLy8gQHNlZSBEb2NrZXJDb21wb3NlLnNlcnZpY2VOYW1lXG4gICAgY29uc3QgZGVwZW5kc09uID0gQXJyYXk8c3RyaW5nPigpO1xuICAgIGZvciAoY29uc3QgZGVwZW5kc09uU2VydmljZU5hbWUgb2Ygc2VydmljZURlc2NyaXB0aW9uLmRlcGVuZHNPbiA/PyBbXSkge1xuICAgICAgY29uc3QgcmVzb2x2ZWRTZXJ2aWNlTmFtZSA9IGRlcGVuZHNPblNlcnZpY2VOYW1lLnNlcnZpY2VOYW1lO1xuICAgICAgaWYgKHJlc29sdmVkU2VydmljZU5hbWUgPT09IHNlcnZpY2VOYW1lKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgU2VydmljZSAke3NlcnZpY2VOYW1lfSBjYW5ub3QgZGVwZW5kIG9uIGl0c2VsZmApO1xuICAgICAgfVxuICAgICAgaWYgKCFzZXJ2aWNlRGVzY3JpcHRpb25zW3Jlc29sdmVkU2VydmljZU5hbWVdKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICBgVW5hYmxlIHRvIHJlc29sdmUgc2VydmljZSBuYW1lZCAke3Jlc29sdmVkU2VydmljZU5hbWV9IGZvciAke3NlcnZpY2VOYW1lfWBcbiAgICAgICAgKTtcbiAgICAgIH1cblxuICAgICAgZGVwZW5kc09uLnB1c2gocmVzb2x2ZWRTZXJ2aWNlTmFtZSk7XG4gICAgfVxuXG4gICAgLy8gR2l2ZSBlYWNoIHZvbHVtZSBiaW5kaW5nIGEgY2hhbmNlIHRvIGJpbmQgYW55IG5lY2Vzc2FyeSB2b2x1bWVcbiAgICAvLyBjb25maWd1cmF0aW9uIGFuZCBwcm92aWRlIHZvbHVtZSBtb3VudCBpbmZvcm1hdGlvbiBmb3IgdGhlIHNlcnZpY2UuXG4gICAgY29uc3Qgdm9sdW1lczogRG9ja2VyQ29tcG9zZVZvbHVtZU1vdW50W10gPSBbXTtcbiAgICBmb3IgKGNvbnN0IHZvbHVtZUJpbmRpbmcgb2Ygc2VydmljZURlc2NyaXB0aW9uLnZvbHVtZXMgPz8gW10pIHtcbiAgICAgIHZvbHVtZXMucHVzaCh2b2x1bWVCaW5kaW5nLmJpbmQodm9sdW1lSW5mbykpO1xuICAgIH1cblxuICAgIC8vIEdpdmUgZWFjaCBuZXR3b3JrIGJpbmRpbmcgYSBjaGFuY2UgdG8gYmluZCBhbnkgbmVjZXNzYXJ5IG5ldHdvcmtcbiAgICAvLyBjb25maWd1cmF0aW9uIGFuZCBwcm92aWRlIG5ldHdvcmsgbW91bnQgaW5mb3JtYXRpb24gZm9yIHRoZSBzZXJ2aWNlLlxuICAgIGNvbnN0IG5ldHdvcmtzOiBzdHJpbmdbXSA9IFtdO1xuICAgIGZvciAoY29uc3QgbmV0d29ya0JpbmRpbmcgb2Ygc2VydmljZURlc2NyaXB0aW9uLm5ldHdvcmtzID8/IFtdKSB7XG4gICAgICBuZXR3b3Jrcy5wdXNoKG5ldHdvcmtCaW5kaW5nLmJpbmQobmV0d29ya0luZm8pKTtcbiAgICB9XG5cbiAgICAvLyBDcmVhdGUgYW5kIHN0b3JlIHRoZSBzZXJ2aWNlIGNvbmZpZ3VyYXRpb24sIHRha2luZyBjYXJlIG5vdCB0byBjcmVhdGVcbiAgICAvLyBvYmplY3QgbWVtYmVycyB3aXRoIHVuZGVmaW5lZCB2YWx1ZXMuXG4gICAgc2VydmljZXNbc2VydmljZU5hbWVdID0ge1xuICAgICAgLi4uZ2V0T2JqZWN0V2l0aEtleUFuZFZhbHVlSWZWYWx1ZUlzRGVmaW5lZChcbiAgICAgICAgXCJpbWFnZVwiLFxuICAgICAgICBzZXJ2aWNlRGVzY3JpcHRpb24uaW1hZ2VcbiAgICAgICksXG4gICAgICAuLi5nZXRPYmplY3RXaXRoS2V5QW5kVmFsdWVJZlZhbHVlSXNEZWZpbmVkKFxuICAgICAgICBcImJ1aWxkXCIsXG4gICAgICAgIHNlcnZpY2VEZXNjcmlwdGlvbi5pbWFnZUJ1aWxkXG4gICAgICApLFxuICAgICAgLi4uZ2V0T2JqZWN0V2l0aEtleUFuZFZhbHVlSWZWYWx1ZUlzRGVmaW5lZChcbiAgICAgICAgXCJjb21tYW5kXCIsXG4gICAgICAgIHNlcnZpY2VEZXNjcmlwdGlvbi5jb21tYW5kXG4gICAgICApLFxuICAgICAgLi4uKE9iamVjdC5rZXlzKHNlcnZpY2VEZXNjcmlwdGlvbi5lbnZpcm9ubWVudCkubGVuZ3RoID4gMFxuICAgICAgICA/IHsgZW52aXJvbm1lbnQ6IHNlcnZpY2VEZXNjcmlwdGlvbi5lbnZpcm9ubWVudCB9XG4gICAgICAgIDoge30pLFxuICAgICAgLi4uKHNlcnZpY2VEZXNjcmlwdGlvbi5wb3J0cy5sZW5ndGggPiAwXG4gICAgICAgID8geyBwb3J0czogc2VydmljZURlc2NyaXB0aW9uLnBvcnRzIH1cbiAgICAgICAgOiB7fSksXG4gICAgICAuLi4oT2JqZWN0LmtleXMoc2VydmljZURlc2NyaXB0aW9uLmxhYmVscykubGVuZ3RoID4gMFxuICAgICAgICA/IHsgbGFiZWxzOiBzZXJ2aWNlRGVzY3JpcHRpb24ubGFiZWxzIH1cbiAgICAgICAgOiB7fSksXG4gICAgICAuLi4oZGVwZW5kc09uLmxlbmd0aCA+IDAgPyB7IGRlcGVuZHNPbiB9IDoge30pLFxuICAgICAgLi4uKHZvbHVtZXMubGVuZ3RoID4gMCA/IHsgdm9sdW1lcyB9IDoge30pLFxuICAgICAgLi4uKG5ldHdvcmtzLmxlbmd0aCA+IDAgPyB7IG5ldHdvcmtzIH0gOiB7fSksXG4gICAgfTtcbiAgfVxuXG4gIC8vIEV4cGxpY2l0IHdpdGggdGhlIHR5cGUgaGVyZSBiZWNhdXNlIHRoZSBkZWNhbWVsaXplIHN0ZXAgYWZ0ZXIgdGhpcyB3aXBlc1xuICAvLyBvdXQgdHlwZXMuXG4gIGNvbnN0IGlucHV0OiBEb2NrZXJDb21wb3NlRmlsZVNjaGVtYSA9IHtcbiAgICB2ZXJzaW9uLFxuICAgIHNlcnZpY2VzLFxuICAgIC4uLihPYmplY3Qua2V5cyh2b2x1bWVDb25maWcpLmxlbmd0aCA+IDAgPyB7IHZvbHVtZXM6IHZvbHVtZUNvbmZpZyB9IDoge30pLFxuICAgIC4uLihPYmplY3Qua2V5cyhuZXR3b3JrQ29uZmlnKS5sZW5ndGggPiAwXG4gICAgICA/IHsgbmV0d29ya3M6IG5ldHdvcmtDb25maWcgfVxuICAgICAgOiB7fSksXG4gIH07XG5cbiAgLy8gQ2hhbmdlIG1vc3Qga2V5cyB0byBzbmFrZSBjYXNlLlxuICByZXR1cm4gZGVjYW1lbGl6ZUtleXNSZWN1cnNpdmVseShpbnB1dCwge1xuICAgIHNob3VsZERlY2FtZWxpemU6IHNob3VsZERlY2FtZWxpemVEb2NrZXJDb21wb3NlS2V5LFxuICAgIHNlcGFyYXRvcjogXCJfXCIsXG4gIH0pO1xufVxuXG4vKipcbiAqIFJldHVybnMgYHsgW2tleV06IHZhbHVlIH1gIGlmIGB2YWx1ZWAgaXMgZGVmaW5lZCwgb3RoZXJ3aXNlIHJldHVybnMgYHt9YCBzb1xuICogdGhhdCBvYmplY3Qgc3ByZWFkaW5nIGNhbiBiZSB1c2VkIHRvIGdlbmVyYXRlIGEgcGVjdWxpYXIgaW50ZXJmYWNlLlxuICogQHBhcmFtIGtleVxuICogQHBhcmFtIHZhbHVlXG4gKi9cbmZ1bmN0aW9uIGdldE9iamVjdFdpdGhLZXlBbmRWYWx1ZUlmVmFsdWVJc0RlZmluZWQ8SyBleHRlbmRzIHN0cmluZywgVD4oXG4gIGtleTogSyxcbiAgdmFsdWU6IFRcbik6IHsgSzogVCB9IHwge30ge1xuICByZXR1cm4gdmFsdWUgIT09IHVuZGVmaW5lZCA/IHsgW2tleV06IHZhbHVlIH0gOiB7fTtcbn1cblxuLyoqXG4gKiBEZXRlcm1pbmVzIHdoZXRoZXIgdGhlIGtleSBhdCB0aGUgZ2l2ZW4gcGF0aCBzaG91bGQgYmUgZGVjYW1lbGl6ZWQuXG4gKiBMYXJnZWx5LCBhbGwga2V5cyBzaG91bGQgYmUgc25ha2UgY2FzZWQuIEJ1dCwgdGhlcmUgYXJlIHNvbWVcbiAqIGV4Y2VwdGlvbnMgZm9yIHVzZXItcHJvdmlkZWQgbmFtZXMgZm9yIHNlcnZpY2VzLCB2b2x1bWVzLCBhbmRcbiAqIGVudmlyb25tZW50IHZhcmlhYmxlcy5cbiAqXG4gKiBAcGFyYW0gcGF0aFxuICovXG5mdW5jdGlvbiBzaG91bGREZWNhbWVsaXplRG9ja2VyQ29tcG9zZUtleShwYXRoOiBzdHJpbmdbXSkge1xuICBjb25zdCBwb3VuZFBhdGggPSBwYXRoLmpvaW4oXCIjXCIpO1xuXG4gIC8vIERvZXMgbm90IGRlY2FtZWxpemUgdXNlcidzIG5hbWVzLlxuICAvLyBzZXJ2aWNlcy5uYW1laGVyZTpcbiAgLy8gdm9sdW1lcy5uYW1laGVyZTpcbiAgLy8gbmV0d29ya3MubmFtZWhlcmU6XG4gIGlmICgvXihzZXJ2aWNlc3x2b2x1bWVzfG5ldHdvcmtzKSNbXiNdKyQvLnRlc3QocG91bmRQYXRoKSkge1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuXG4gIC8vIERvZXMgbm90IGRlY2FtZWxpemUgZW52aXJvbm1lbnQgdmFyaWFibGVzIGFuZCBsYWJlbHNcbiAgLy8gc2VydmljZXMubmFtZWhlcmUuZW52aXJvbm1lbnQuKlxuICAvLyBzZXJ2aWNlcy5uYW1laGVyZS5sYWJlbHMuKlxuICBpZiAoL15zZXJ2aWNlcyNbXiNdKyMoZW52aXJvbm1lbnR8bGFiZWxzKSMvLnRlc3QocG91bmRQYXRoKSkge1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuXG4gIC8vIERvZXMgbm90IGRlY2FtZWxpemUgYnVpbGQgYXJndW1lbnRzXG4gIC8vIHNlcnZpY2VzLm5hbWVoZXJlLmJ1aWxkLmFyZ3MuKlxuICBpZiAoL15zZXJ2aWNlcyNbXiNdKyNidWlsZCNhcmdzIy8udGVzdChwb3VuZFBhdGgpKSB7XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG5cbiAgLy8gT3RoZXJ3aXNlLCBsZXQgaXQgYWxsIGRlY2FtZWxpemUuXG4gIHJldHVybiB0cnVlO1xufVxuIl19