"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const yargs = require("yargs");
const new_1 = require("./cmds/new");
const synth_1 = require("./synth");
const tasks_1 = require("./tasks");
const common_1 = require("../common");
const logging = require("../logging");
const task_runtime_1 = require("../task-runtime");
const util_1 = require("../util");
const DEFAULT_RC = path_1.resolve(common_1.PROJEN_RC);
async function main() {
    const ya = yargs;
    ya.command(new_1.default);
    const runtime = new task_runtime_1.TaskRuntime(".");
    tasks_1.discoverTaskCommands(runtime, ya);
    ya.recommendCommands();
    ya.strictCommands();
    ya.showHelpOnFail(false);
    ya.wrap(yargs.terminalWidth());
    ya.option("post", {
        type: "boolean",
        default: true,
        desc: "Run post-synthesis steps such as installing dependencies. Use --no-post to skip",
    });
    ya.option("watch", {
        type: "boolean",
        default: false,
        desc: "Keep running and resynthesize when projenrc changes",
        alias: "w",
    });
    ya.options("debug", { type: "boolean", default: false, desc: "Debug logs" });
    ya.options("rc", {
        desc: "path to .projenrc.js file",
        default: DEFAULT_RC,
        type: "string",
    });
    ya.completion();
    ya.help();
    // do not use the default yargs '--version' implementation since it is
    // global by default (it appears on all subcommands)
    ya.version(false);
    ya.option("version", {
        type: "boolean",
        description: "Show version number",
        global: false,
    });
    const args = ya.argv;
    if (args.debug) {
        process.env.DEBUG = "true";
    }
    const nodeVersion = util_1.getNodeMajorVersion();
    if (nodeVersion && nodeVersion < 14) {
        logging.warn(`WARNING: You are using Node v${nodeVersion}, which reaches end of life on April 30, 2022. Support for EOL Node releases may be dropped by projen in the future. Please consider upgrading to Node >= 14 as soon as possible.`);
    }
    // no command means synthesize
    if (args._.length === 0) {
        if (args.version) {
            console.log(common_1.PROJEN_VERSION);
            process.exit(0);
        }
        await synth_1.synth(runtime, {
            post: args.post,
            watch: args.watch,
            rcfile: args.rc,
        });
    }
}
main().catch((e) => {
    console.error(e.stack);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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