"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCdkTypeScriptApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const fs = require("fs-extra");
const auto_discover_1 = require("./auto-discover");
const awscdk_deps_js_1 = require("./awscdk-deps-js");
const cdk_config_1 = require("./cdk-config");
const cdk_tasks_1 = require("./cdk-tasks");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const typescript_1 = require("../typescript");
/**
 * AWS CDK app in TypeScript
 *
 * @pjid awscdk-app-ts
 */
class AwsCdkTypeScriptApp extends typescript_1.TypeScriptAppProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            bundlerOptions: {
                ...options.bundlerOptions,
                // we invoke the "bundle" task as part of the build step in cdk.json so
                // we don't want it to be added to the pre-compile phase.
                addToPreCompile: false,
            },
        });
        this.cdkDeps = new awscdk_deps_js_1.AwsCdkDepsJs(this, {
            dependencyType: dependencies_1.DependencyType.RUNTIME,
            ...options,
        });
        this.appEntrypoint = options.appEntrypoint ?? "main.ts";
        // CLI
        this.addDevDeps(`aws-cdk@${this.cdkDeps.cdkVersion}`);
        // no compile step because we do all of it in typescript directly
        this.compileTask.reset();
        this.cdkTasks = new cdk_tasks_1.CdkTasks(this);
        // add synth to the build
        this.postCompileTask.spawn(this.cdkTasks.synthSilent);
        const tsConfigFile = this.tsconfig?.fileName;
        if (!tsConfigFile) {
            throw new Error("Expecting tsconfig.json");
        }
        this.cdkConfig = new cdk_config_1.CdkConfig(this, {
            app: `npx ts-node -P ${tsConfigFile} --prefer-ts-exts ${path.posix.join(this.srcdir, this.appEntrypoint)}`,
            featureFlags: this.cdkDeps.cdkMajorVersion < 2,
            buildCommand: this.runTaskCommand(this.bundler.bundleTask),
            watchIncludes: [`${this.srcdir}/**/*.ts`, `${this.testdir}/**/*.ts`],
            watchExcludes: [
                "README.md",
                "cdk*.json",
                "**/*.d.ts",
                "**/*.js",
                "tsconfig.json",
                "package*.json",
                "yarn.lock",
                "node_modules",
            ],
            ...options,
        });
        this.gitignore.exclude(".parcel-cache/");
        this.npmignore?.exclude(`${this.cdkConfig.cdkout}/`);
        this.npmignore?.exclude(".cdk.staging/");
        if (this.tsconfig) {
            this.tsconfig.exclude.push(this.cdkConfig.cdkout);
        }
        this.addDevDeps("ts-node");
        if (options.sampleCode ?? true) {
            new SampleCode(this, this.cdkDeps.cdkMajorVersion);
        }
        new auto_discover_1.AutoDiscover(this, {
            srcdir: this.srcdir,
            testdir: this.testdir,
            lambdaOptions: options.lambdaOptions,
            tsconfigPath: this.tsconfigDev.fileName,
            cdkDeps: this.cdkDeps,
            lambdaAutoDiscover: options.lambdaAutoDiscover ?? true,
            edgeLambdaAutoDiscover: options.edgeLambdaAutoDiscover ?? true,
            lambdaExtensionAutoDiscover: options.lambdaExtensionAutoDiscover ?? true,
            integrationTestAutoDiscover: options.integrationTestAutoDiscover ?? true,
        });
    }
    /**
     * The CDK version this app is using.
     */
    get cdkVersion() {
        return this.cdkDeps.cdkVersion;
    }
    /**
     * Adds an AWS CDK module dependencies
     * @param modules The list of modules to depend on
     */
    addCdkDependency(...modules) {
        return this.cdkDeps.addV1Dependencies(...modules);
    }
}
exports.AwsCdkTypeScriptApp = AwsCdkTypeScriptApp;
_a = JSII_RTTI_SYMBOL_1;
AwsCdkTypeScriptApp[_a] = { fqn: "projen.awscdk.AwsCdkTypeScriptApp", version: "0.67.87" };
class SampleCode extends component_1.Component {
    constructor(project, cdkMajorVersion) {
        super(project);
        this.cdkMajorVersion = cdkMajorVersion;
        this.appProject = project;
    }
    synthesize() {
        const outdir = this.project.outdir;
        const srcdir = path.join(outdir, this.appProject.srcdir);
        if (fs.pathExistsSync(srcdir) &&
            fs.readdirSync(srcdir).filter((x) => x.endsWith(".ts"))) {
            return;
        }
        const srcImports = new Array();
        if (this.cdkMajorVersion < 2) {
            srcImports.push("import { App, Construct, Stack, StackProps } from '@aws-cdk/core';");
        }
        else {
            srcImports.push("import { App, Stack, StackProps } from 'aws-cdk-lib';");
            srcImports.push("import { Construct } from 'constructs';");
        }
        const srcCode = `${srcImports.join("\n")}

export class MyStack extends Stack {
  constructor(scope: Construct, id: string, props: StackProps = {}) {
    super(scope, id, props);

    // define resources here...
  }
}

// for development, use account/region from cdk cli
const devEnv = {
  account: process.env.CDK_DEFAULT_ACCOUNT,
  region: process.env.CDK_DEFAULT_REGION,
};

const app = new App();

new MyStack(app, '${this.project.name}-dev', { env: devEnv });
// new MyStack(app, '${this.project.name}-prod', { env: prodEnv });

app.synth();`;
        fs.mkdirpSync(srcdir);
        fs.writeFileSync(path.join(srcdir, this.appProject.appEntrypoint), srcCode);
        const testdir = path.join(outdir, this.appProject.testdir);
        if (fs.pathExistsSync(testdir) &&
            fs.readdirSync(testdir).filter((x) => x.endsWith(".ts"))) {
            return;
        }
        const testImports = new Array();
        if (this.cdkMajorVersion < 2) {
            testImports.push("import { App } from '@aws-cdk/core';");
            testImports.push("import { Template } from '@aws-cdk/assertions';");
        }
        else {
            testImports.push("import { App } from 'aws-cdk-lib';");
            testImports.push("import { Template } from 'aws-cdk-lib/assertions';");
        }
        const appEntrypointName = path.basename(this.appProject.appEntrypoint, ".ts");
        const testCode = `${testImports.join("\n")}
import { MyStack } from '../${this.appProject.srcdir}/${appEntrypointName}';

test('Snapshot', () => {
  const app = new App();
  const stack = new MyStack(app, 'test');

  const template = Template.fromStack(stack);
  expect(template.toJSON()).toMatchSnapshot();
});`;
        fs.mkdirpSync(testdir);
        fs.writeFileSync(path.join(testdir, `${appEntrypointName}.test.ts`), testCode);
    }
}
//# sourceMappingURL=data:application/json;base64,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