import { __rest } from "tslib";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import clsx from 'clsx';
import React, { useCallback, useRef } from 'react';
import { fireCancelableEvent, isPlainLeftClick } from '../internal/events';
import useFocusVisible from '../internal/hooks/focus-visible';
import useForwardFocus from '../internal/hooks/forward-focus';
import styles from './styles.css.js';
import { LeftIcon, RightIcon } from './icon-helper';
import { checkSafeUrl } from '../internal/utils/check-safe-url';
import { useMergeRefs } from '../internal/hooks/use-merge-refs';
import LiveRegion from '../internal/components/live-region';
export const InternalButton = React.forwardRef((_a, ref) => {
    var { children, iconName, __iconClass, onClick, onFollow, iconAlign = 'left', iconUrl, iconSvg, iconAlt, variant = 'normal', loading = false, loadingText, disabled = false, wrapText = true, href, target, rel, download, formAction = 'submit', ariaLabel, ariaExpanded, __hideFocusOutline = false, __nativeAttributes, __internalRootRef = null, __activated = false } = _a, props = __rest(_a, ["children", "iconName", "__iconClass", "onClick", "onFollow", "iconAlign", "iconUrl", "iconSvg", "iconAlt", "variant", "loading", "loadingText", "disabled", "wrapText", "href", "target", "rel", "download", "formAction", "ariaLabel", "ariaExpanded", "__hideFocusOutline", "__nativeAttributes", "__internalRootRef", "__activated"]);
    checkSafeUrl('Button', href);
    const focusVisible = useFocusVisible();
    const isAnchor = Boolean(href);
    const isDisabled = loading || disabled;
    const shouldHaveContent = children && ['icon', 'inline-icon', 'flashbar-icon', 'modal-dismiss'].indexOf(variant) === -1;
    const buttonRef = useRef(null);
    useForwardFocus(ref, buttonRef);
    const handleClick = useCallback((event) => {
        if (isAnchor && isDisabled) {
            return event.preventDefault();
        }
        if (isAnchor && isPlainLeftClick(event)) {
            fireCancelableEvent(onFollow, null, event);
        }
        const { altKey, button, ctrlKey, metaKey, shiftKey } = event;
        fireCancelableEvent(onClick, { altKey, button, ctrlKey, metaKey, shiftKey }, event);
    }, [isAnchor, isDisabled, onClick, onFollow]);
    const buttonClass = clsx(props.className, styles.button, styles[`variant-${variant}`], {
        [styles.disabled]: isDisabled,
        [styles['button-no-wrap']]: !wrapText,
        [styles['button-no-text']]: !shouldHaveContent,
        [styles['is-activated']]: __activated,
    });
    const buttonProps = Object.assign(Object.assign(Object.assign(Object.assign({}, props), (__hideFocusOutline ? undefined : focusVisible)), __nativeAttributes), { 
        // https://github.com/microsoft/TypeScript/issues/36659
        ref: useMergeRefs(buttonRef, __internalRootRef), 'aria-label': ariaLabel, 'aria-expanded': ariaExpanded, className: buttonClass, onClick: handleClick });
    const iconProps = {
        loading,
        iconName,
        iconAlign,
        iconUrl,
        iconSvg,
        iconAlt,
        variant,
        iconClass: __iconClass,
        iconSize: variant === 'modal-dismiss' ? 'medium' : 'normal',
    };
    const buttonContent = (React.createElement(React.Fragment, null,
        React.createElement(LeftIcon, Object.assign({}, iconProps)),
        shouldHaveContent && React.createElement("span", { className: styles.content }, children),
        React.createElement(RightIcon, Object.assign({}, iconProps))));
    if (isAnchor) {
        return (
        // https://github.com/yannickcr/eslint-plugin-react/issues/2962
        // eslint-disable-next-line react/jsx-no-target-blank
        React.createElement(React.Fragment, null,
            React.createElement("a", Object.assign({}, buttonProps, { href: href, target: target, 
                // security recommendation: https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#target
                rel: rel !== null && rel !== void 0 ? rel : (target === '_blank' ? 'noopener noreferrer' : undefined), tabIndex: isDisabled ? -1 : undefined, "aria-disabled": isDisabled ? true : undefined, download: download }), buttonContent),
            loading && loadingText && React.createElement(LiveRegion, null, loadingText)));
    }
    return (React.createElement(React.Fragment, null,
        React.createElement("button", Object.assign({}, buttonProps, { type: formAction === 'none' ? 'button' : 'submit', disabled: isDisabled }), buttonContent),
        loading && loadingText && React.createElement(LiveRegion, null, loadingText)));
});
export default InternalButton;
//# sourceMappingURL=internal.js.map