"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataLakeExporter = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_kinesisfirehose_1 = require("aws-cdk-lib/aws-kinesisfirehose");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const context_options_1 = require("./common/context-options");
const tracked_construct_1 = require("./common/tracked-construct");
/**
 * DataLakeExporter Construct to export data from a stream to the data lake.
 * Source can be an Amazon Kinesis Data Stream.
 * Target can be an Amazon S3 bucket.
 */
class DataLakeExporter extends tracked_construct_1.TrackedConstruct {
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingCode: context_options_1.ContextOptions.DATA_LAKE_ID,
        };
        super(scope, id, trackedConstructProps);
        if (props.deliverySize || 128 > 128) {
            throw 'deliverySize cannot be more than 128MB';
        }
        if (props.deliveryInterval || 900 > 900) {
            throw 'deliveryInterval cannot be more than 900s';
        }
        // const stack = Stack.of(this);
        // Create log group for storing Amazon Kinesis Firehose logs.
        const logGroup = new aws_logs_1.LogGroup(this, 'dataLakeExporterLogGroup', {
            logGroupName: '/aws/data-lake-exporter/',
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        // Create the Kinesis Firehose log stream.
        const firehoseLogStream = new aws_logs_1.LogStream(this, 'dataLakeExporterLogStream', {
            logGroup: logGroup,
            logStreamName: 'firehose-stream',
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        const policyDocumentKinesisFirehose = new aws_iam_1.PolicyDocument({
            statements: [
                new aws_iam_1.PolicyStatement({
                    resources: [
                        `${logGroup.logGroupArn}:log-stream:${firehoseLogStream.logStreamName}`,
                    ],
                    actions: [
                        'logs:PutLogEvents',
                    ],
                }),
            ],
        });
        const managedPolicyKinesisFirehose = new aws_iam_1.ManagedPolicy(this, 'managedPolicyKinesisFirehose', {
            document: policyDocumentKinesisFirehose,
        });
        // Create an Amazon IAM Role used by Amazon Kinesis Firehose delivery stream
        const roleKinesisFirehose = new aws_iam_1.Role(this, 'dataLakeExporterRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('firehose.amazonaws.com'),
            managedPolicies: [managedPolicyKinesisFirehose],
        });
        roleKinesisFirehose.node.addDependency(managedPolicyKinesisFirehose);
        const grantSink = props.sinkBucket.grantWrite(roleKinesisFirehose);
        const grantSource = props.sourceKinesisDataStream.grantRead(roleKinesisFirehose);
        const grantTable = props.sourceGlueTable.grantRead(roleKinesisFirehose);
        const grantGlue = props.sourceGlueTable.grantToUnderlyingResources(roleKinesisFirehose, ['glue:GetTableVersions']);
        // Create the Delivery stream from Cfn because L2 Construct doesn't support conversion to parquet and custom partitioning
        this.cfnIngestionStream = new aws_kinesisfirehose_1.CfnDeliveryStream(this, 'dataLakeExporter', {
            deliveryStreamType: 'KinesisStreamAsSource',
            extendedS3DestinationConfiguration: {
                bucketArn: props.sinkBucket.bucketArn,
                bufferingHints: {
                    intervalInSeconds: props.deliveryInterval || 900,
                    sizeInMBs: props.deliverySize || 128,
                },
                cloudWatchLoggingOptions: {
                    logGroupName: logGroup.logGroupName,
                    logStreamName: firehoseLogStream.logStreamName,
                },
                roleArn: roleKinesisFirehose.roleArn,
                errorOutputPrefix: `${props.sinkObjectKey}-error`,
                prefix: props.sinkObjectKey,
                compressionFormat: 'UNCOMPRESSED',
                s3BackupMode: 'Disabled',
                dataFormatConversionConfiguration: {
                    enabled: true,
                    inputFormatConfiguration: {
                        deserializer: {
                            openXJsonSerDe: {},
                        },
                    },
                    outputFormatConfiguration: {
                        serializer: {
                            parquetSerDe: {},
                        },
                    },
                    schemaConfiguration: {
                        roleArn: roleKinesisFirehose.roleArn,
                        catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        region: aws_cdk_lib_1.Aws.REGION,
                        databaseName: props.sourceGlueDatabase.databaseName,
                        tableName: props.sourceGlueTable.tableName,
                    },
                },
            },
            kinesisStreamSourceConfiguration: {
                kinesisStreamArn: props.sourceKinesisDataStream.streamArn,
                roleArn: roleKinesisFirehose.roleArn,
            },
        });
        // Need to enforce a dependancy because the grant methods generate an IAM Policy without dependency on the Firehose
        this.cfnIngestionStream.node.addDependency(grantSink);
        this.cfnIngestionStream.node.addDependency(grantSource);
        this.cfnIngestionStream.node.addDependency(grantTable);
        this.cfnIngestionStream.node.addDependency(grantGlue);
    }
}
exports.DataLakeExporter = DataLakeExporter;
_a = JSII_RTTI_SYMBOL_1;
DataLakeExporter[_a] = { fqn: "aws-analytics-reference-architecture.DataLakeExporter", version: "2.7.2" };
//# sourceMappingURL=data:application/json;base64,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