import { Duration } from 'aws-cdk-lib';
import { Bucket } from 'aws-cdk-lib/aws-s3';
import { Construct } from 'constructs';
import { PreparedDataset } from './prepared-dataset';
/**
 * The properties for the BatchReplayer construct
 */
export interface BatchReplayerProps {
    /**
     * The [PreparedDataset]{@link PreparedDataset} used to replay data
     */
    readonly dataset: PreparedDataset;
    /**
     * The frequency of the replay
     * @default - The BatchReplayer is triggered every 60 seconds
     */
    readonly frequency?: Duration;
    /**
     * The S3 Bucket sink where the BatchReplayer writes data.
     * :warnning: **If the Bucket is encrypted with KMS, the Key must be managed by this stack.
     */
    readonly sinkBucket: Bucket;
    /**
     * The S3 object key sink where the BatchReplayer writes data.
     * @default - No object key is used and the BatchReplayer writes the dataset in s3://<BUCKET_NAME>/<TABLE_NAME>
     */
    readonly sinkObjectKey?: string;
    /**
     * The maximum file size in Bytes written by the BatchReplayer
     * @default - The BatchReplayer writes 100MB files maximum
     */
    readonly outputFileMaxSizeInBytes?: number;
}
/**
 * Replay the data in the given PartitionedDataset.
 *
 * It will dump files into the `sinkBucket` based on the given `frequency`.
 * The computation is in a Step Function with two Lambda steps.
 *
 * 1. resources/lambdas/find-file-paths
 * Read the manifest file and output a list of S3 file paths within that batch time range
 *
 * 2. resources/lambdas/write-in-batch
 * Take a file path, filter only records within given time range, adjust the the time with offset to
 * make it looks like just being generated. Then write the output to the `sinkBucket`
 *
 * Usage example:
 * ```typescript
 *
 * const myBucket = new Bucket(stack, "MyBucket")
 *
 * new BatchReplayer(stack, "WebSalesReplayer", {
 *   dataset: PreparedDataset.RETAIL_1_GB_WEB_SALE,
 *   s3BucketSink: myBucket
 *   s3ObjectKeySink: 'some-prefix',
 *   frequency: 120,
 *   outputFileMaxSizeInBytes: 10000000,
 * });
 * ```
 *
 * :warnning: **If the Bucket is encrypted with KMS, the Key must be managed by this stack.
 */
export declare class BatchReplayer extends Construct {
    /**
     * Dataset used for replay
     */
    readonly dataset: PreparedDataset;
    /**
     * Frequency (in Seconds) of the replaying. The batch job will start
     * for every given frequency and replay the data in that period
     */
    readonly frequency: number;
    /**
     * Sink bucket where the batch replayer will put data in
     */
    readonly sinkBucket: Bucket;
    /**
     * Sink object key where the batch replayer will put data in
     */
    readonly sinkObjectKey?: string;
    /**
     * Maximum file size for each output file. If the output batch file is,
     * larger than that, it will be splitted into multiple files that fit this size.
     *
     * Default to 100MB (max value)
     */
    readonly outputFileMaxSizeInBytes?: number;
    /**
     * Constructs a new instance of the BatchReplayer construct
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {BatchReplayerProps} props the BatchReplayer [properties]{@link BatchReplayerProps}
     */
    constructor(scope: Construct, id: string, props: BatchReplayerProps);
}
