"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToDynamoDB = void 0;
const dynamodb = require("@aws-cdk/aws-dynamodb");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
class LambdaToDynamoDB extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToDynamoDB class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToDynamoDBProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            deployLambda: props.deployLambda,
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        this.dynamoTable = defaults.buildDynamoDBTable(this, {
            dynamoTableProps: props.dynamoTableProps,
            existingTableObj: props.existingTableObj
        });
        this.lambdaFunction.addEnvironment('DDB_TABLE_NAME', this.dynamoTable.tableName);
        this.dynamoTable.grantReadWriteData(this.lambdaFunction.grantPrincipal);
        // Conditional metadata for cfn_nag
        if (((_a = props.dynamoTableProps) === null || _a === void 0 ? void 0 : _a.billingMode) === dynamodb.BillingMode.PROVISIONED) {
            const cfnTable = this.dynamoTable.node.findChild('Resource');
            cfnTable.cfnOptions.metadata = {
                cfn_nag: {
                    rules_to_suppress: [{
                            id: 'W73',
                            reason: `PROVISIONED billing mode is a default and is not explicitly applied as a setting.`
                        }]
                }
            };
        }
    }
}
exports.LambdaToDynamoDB = LambdaToDynamoDB;
//# sourceMappingURL=data:application/json;base64,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