"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayPythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const client_settings_1 = require("./codegen/components/client-settings");
const docs_project_1 = require("./codegen/docs-project");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const python_2 = require("./samples/python");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "aws_prototyping_sdk.open_api_gateway";
/**
 * Synthesizes a Python Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-py
 */
class OpenApiGatewayPythonProject extends python_1.PythonProject {
    constructor(projectOptions) {
        super({
            ...projectOptions,
            sample: false,
            venv: true,
            venvOptions: {
                envdir: ".env",
                ...projectOptions?.venvOptions,
            },
            pip: true,
            poetry: false,
            // No tests by default, but allow user to override
            pytest: projectOptions.pytest ?? false,
            setuptools: true,
        });
        const options = this.preConstruct(projectOptions);
        if (options.specFile && !path.isAbsolute(options.specFile)) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.forceGenerateCodeAndDocs = options.forceGenerateCodeAndDocs ?? false;
        this.apiSrcDir = options.apiSrcDir ?? "api";
        // Generated project should have a dependency on this project, in order to run the generation scripts
        [OPENAPI_GATEWAY_PDK_PACKAGE_NAME, "constructs", "aws-cdk-lib", "cdk-nag"]
            .filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        // Synthesize the openapi spec early since it's used by the generated python client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join(this.moduleName, this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
            ...(options.specFile && path.isAbsolute(options.specFile)
                ? {
                    overrideSpecPath: options.specFile,
                }
                : {}),
        });
        spec.synth();
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the python client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.PYTHON);
        const clientSettings = new client_settings_1.ClientSettings(this, {
            clientLanguages: [...clientLanguages],
            defaultClientLanguage: languages_1.ClientLanguage.PYTHON,
            documentationFormats: options.documentationFormats ?? [],
            forceGenerateCodeAndDocs: this.forceGenerateCodeAndDocs,
            generatedCodeDir: this.generatedCodeDir,
            specChanged: spec.specChanged,
        });
        // Share the same env between this project and the generated client. Accept a custom venv if part of a monorepo
        const envDir = options.venvOptions?.envdir || ".env";
        // env directory relative to the generated python client
        const clientEnvDir = path.join("..", ...this.generatedCodeDir.split("/").map(() => ".."), envDir);
        this.generatedClients = generate_1.generateClientProjects(clientSettings.clientLanguageConfigs, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: "main",
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: options.authorName ?? "APJ Cope",
                authorEmail: options.authorEmail ?? "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
                // We are more prescriptive about the generated client since we must set up a dependency in the shared env
                pip: true,
                poetry: false,
                venv: true,
                venvOptions: {
                    envdir: clientEnvDir,
                },
                generateLayer: true,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedPythonClient = this.generatedClients[languages_1.ClientLanguage.PYTHON];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedPythonClient.synth();
        // Add a dependency on the generated python client, which should be available since we share the virtual env
        this.addDependency(this.generatedPythonClient.moduleName);
        if (this.hasParent) {
            // Since the generated python client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
            if ("addImplicitDependency" in this.parent) {
                // If we're within a monorepo, add an implicit dependency to ensure the generated python client is built first
                this.parent.addImplicitDependency(this, this.generatedPythonClient);
            }
        }
        // Get the lambda layer dir relative to the root of this project
        const pythonLayerDistDir = path.join(this.generatedCodeDir, languages_1.ClientLanguage.PYTHON, this.generatedPythonClient.layerDistDir);
        // Ensure it's included in the package
        new projen_1.TextFile(this, "MANIFEST.in", {
            lines: [`recursive-include ${pythonLayerDistDir} *`],
        });
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            pythonClientPackageName: this.generatedPythonClient.moduleName,
            sampleCode: options.sample,
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
            moduleName: this.moduleName,
        };
        // Define some helpers for resolving resource paths in spec_utils.py
        new projen_1.SampleFile(this, path.join(this.moduleName, "spec_utils.py"), {
            contents: `import pkgutil, json
from os import path
from pathlib import Path

SPEC_PATH = path.join(str(Path(__file__).absolute().parent), "${this.specDir}/${spec.parsedSpecFileName}")
SPEC = json.loads(pkgutil.get_data(__name__, "${this.specDir}/${spec.parsedSpecFileName}"))

def get_project_root():
    return Path(__file__).absolute().parent.parent

def get_generated_client_layer_directory():
    return path.join(str(get_project_root()), "${pythonLayerDistDir}")
`,
        });
        new projen_1.SampleFile(this, path.join(this.moduleName, "__init__.py"), {
            contents: "#",
        });
        new projen_1.SampleDir(this, path.join(this.moduleName, this.apiSrcDir), {
            files: python_2.getPythonSampleSource(sampleOptions),
        });
        // Generate documentation if needed
        new docs_project_1.DocsProject({
            parent: this,
            outdir: path.join(this.generatedCodeDir, "documentation"),
            name: "docs",
            formatConfigs: clientSettings.documentationFormatConfigs,
            specPath: spec.parsedSpecPath,
        });
    }
    /**
     * This method provides inheritors a chance to synthesize extra resources prior to those created by this project.
     * Return any options you wish to change, other than python project options.
     */
    preConstruct(options) {
        return options;
    }
}
exports.OpenApiGatewayPythonProject = OpenApiGatewayPythonProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayPythonProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayPythonProject", version: "0.12.14" };
//# sourceMappingURL=data:application/json;base64,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