"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPythonSampleSource = void 0;
/**
 * Return a map of file name to contents for the sample python source code
 */
exports.getPythonSampleSource = (options) => ({
    "__init__.py": "#",
    // This file provides a type-safe interface to the exported OpenApiGatewayLambdaApi construct
    "api.py": `from dataclasses import fields
from ${options.openApiGatewayPackageName} import OpenApiGatewayLambdaApi, OpenApiIntegration
from ${options.pythonClientPackageName}.api.default_api_operation_config import OperationLookup, OperationConfig
from ${options.moduleName} import SPEC, SPEC_PATH

class Api(OpenApiGatewayLambdaApi):
    """
    Type-safe construct for the API Gateway resources defined by the spec.
    You will likely not need to modify this file, and can instead extend it and define your integrations.
    """
    def __init__(self, scope, id, integrations: OperationConfig[OpenApiIntegration], **kwargs):
        super().__init__(scope, id,
            **kwargs,
            integrations={ field.name: getattr(integrations, field.name) for field in fields(integrations) },
            spec=SPEC,
            spec_path=SPEC_PATH,
            operation_lookup=OperationLookup,
        )
`,
    ...(options.sampleCode !== false
        ? {
            // Generate an example which instantiates the Api construct
            // TODO: Consider generating this sample from the parsed spec
            "sample_api.py": `from ${options.openApiGatewayPackageName} import Authorizers, OpenApiIntegration
from ${options.pythonClientPackageName}.api.default_api_operation_config import OperationConfig
from aws_cdk.aws_lambda import LayerVersion, Code, Function, Runtime
from .api import Api
from constructs import Construct
from ${options.moduleName} import get_generated_client_layer_directory
from pathlib import Path
from os import path

class SampleApi(Construct):
  """
  An example of how to add integrations to your api
  """
  def __init__(self, scope, id):
      super().__init__(scope, id)

      # Layer which contains the generated client.
      self.generated_client_layer = LayerVersion(self, 'GeneratedClientLayer',
          code=Code.from_asset(get_generated_client_layer_directory())
      )

      self.api = Api(self, 'Api',
          default_authorizer=Authorizers.iam(),
          integrations=OperationConfig(
              say_hello=OpenApiIntegration(
                  function=Function(self, 'SayHello',
                      runtime=Runtime.PYTHON_3_9,
                      code=Code.from_asset(path.join(str(Path(__file__).parent.absolute()), 'handlers')),
                      handler="say_hello_handler_sample.handler",
                      layers=[self.generated_client_layer],
                  ),
              ),
          ),
      )
`,
            // Generate an example lambda handler
            "handlers/say_hello_handler_sample.py": `from ${options.pythonClientPackageName}.api.default_api_operation_config import say_hello_handler, SayHelloRequest, ApiResponse, SayHelloOperationResponses
from ${options.pythonClientPackageName}.model.hello_response import HelloResponse

@say_hello_handler
def handler(input: SayHelloRequest, **kwargs) -> SayHelloOperationResponses:
    """
    An example lambda handler which uses the generated handler wrapper to manage marshalling inputs/outputs
    """
    return ApiResponse(
        status_code=200,
        body=HelloResponse(message="Hello {}!".format(input.request_parameters["name"])),
        headers={}
    )
`,
        }
        : {}),
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHl0aG9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL3Byb2plY3Qvc2FtcGxlcy9weXRob24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7Ozs7Ozt3SEFjd0g7OztBQWdDeEg7O0dBRUc7QUFDVSxRQUFBLHFCQUFxQixHQUFHLENBQ25DLE9BQWdDLEVBQ0EsRUFBRSxDQUFDLENBQUM7SUFDcEMsYUFBYSxFQUFFLEdBQUc7SUFDbEIsNkZBQTZGO0lBQzdGLFFBQVEsRUFBRTtPQUNMLE9BQU8sQ0FBQyx5QkFBeUI7T0FDakMsT0FBTyxDQUFDLHVCQUF1QjtPQUMvQixPQUFPLENBQUMsVUFBVTs7Ozs7Ozs7Ozs7Ozs7O0NBZXhCO0lBQ0MsR0FBRyxDQUFDLE9BQU8sQ0FBQyxVQUFVLEtBQUssS0FBSztRQUM5QixDQUFDLENBQUM7WUFDRSwyREFBMkQ7WUFDM0QsNkRBQTZEO1lBQzdELGVBQWUsRUFBRSxRQUFRLE9BQU8sQ0FBQyx5QkFBeUI7T0FDM0QsT0FBTyxDQUFDLHVCQUF1Qjs7OztPQUkvQixPQUFPLENBQUMsVUFBVTs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Q0E2QnhCO1lBQ08scUNBQXFDO1lBQ3JDLHNDQUFzQyxFQUFFLFFBQVEsT0FBTyxDQUFDLHVCQUF1QjtPQUNoRixPQUFPLENBQUMsdUJBQXVCOzs7Ozs7Ozs7Ozs7Q0FZckM7U0FDTTtRQUNILENBQUMsQ0FBQyxFQUFFLENBQUM7Q0FDUixDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqXG4gQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5cbiBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLlxuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2Ugd2l0aCB0aGUgTGljZW5zZS5cbiBZb3UgbWF5IG9idGFpbiBhIGNvcHkgb2YgdGhlIExpY2Vuc2UgYXRcblxuIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuXG4gVW5sZXNzIHJlcXVpcmVkIGJ5IGFwcGxpY2FibGUgbGF3IG9yIGFncmVlZCB0byBpbiB3cml0aW5nLCBzb2Z0d2FyZVxuIGRpc3RyaWJ1dGVkIHVuZGVyIHRoZSBMaWNlbnNlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuIFwiQVMgSVNcIiBCQVNJUyxcbiBXSVRIT1VUIFdBUlJBTlRJRVMgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZWl0aGVyIGV4cHJlc3Mgb3IgaW1wbGllZC5cbiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnMgYW5kXG4gbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKiogKi9cblxuLyoqXG4gKiBPcHRpb25zIGZvciBweXRob24gc2FtcGxlIGNvZGUgZ2VuZXJhdGlvblxuICovXG5leHBvcnQgaW50ZXJmYWNlIFB5dGhvblNhbXBsZUNvZGVPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBwYWNrYWdlIG5hbWUgd2hpY2ggZXhwb3J0cyB0aGUgT3BlbkFwaUdhdGV3YXlMYW1iZGFBcGkgY29uc3RydWN0IChpZSB0aGlzIHBkayBwYWNrYWdlISlcbiAgICovXG4gIHJlYWRvbmx5IG9wZW5BcGlHYXRld2F5UGFja2FnZU5hbWU6IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBnZW5lcmF0ZWQgcHl0aG9uIGNsaWVudCBwYWNrYWdlXG4gICAqL1xuICByZWFkb25seSBweXRob25DbGllbnRQYWNrYWdlTmFtZTogc3RyaW5nO1xuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdG8gZ2VuZXJhdGUgc2FtcGxlIGNvZGVcbiAgICovXG4gIHJlYWRvbmx5IHNhbXBsZUNvZGU/OiBib29sZWFuO1xuICAvKipcbiAgICogRGlyZWN0b3J5IHdoZXJlIHRoZSBwYXJzZWQgc3BlYyBpcyBvdXRwdXRcbiAgICovXG4gIHJlYWRvbmx5IHNwZWNEaXI6IHN0cmluZztcbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIHBhcnNlZCBzcGVjIGZpbGVcbiAgICovXG4gIHJlYWRvbmx5IHBhcnNlZFNwZWNGaWxlTmFtZTogc3RyaW5nO1xuICAvKipcbiAgICogTmFtZSBvZiB0aGUgbW9kdWxlXG4gICAqL1xuICByZWFkb25seSBtb2R1bGVOYW1lOiBzdHJpbmc7XG59XG5cbi8qKlxuICogUmV0dXJuIGEgbWFwIG9mIGZpbGUgbmFtZSB0byBjb250ZW50cyBmb3IgdGhlIHNhbXBsZSBweXRob24gc291cmNlIGNvZGVcbiAqL1xuZXhwb3J0IGNvbnN0IGdldFB5dGhvblNhbXBsZVNvdXJjZSA9IChcbiAgb3B0aW9uczogUHl0aG9uU2FtcGxlQ29kZU9wdGlvbnNcbik6IHsgW2ZpbGVOYW1lOiBzdHJpbmddOiBzdHJpbmcgfSA9PiAoe1xuICBcIl9faW5pdF9fLnB5XCI6IFwiI1wiLFxuICAvLyBUaGlzIGZpbGUgcHJvdmlkZXMgYSB0eXBlLXNhZmUgaW50ZXJmYWNlIHRvIHRoZSBleHBvcnRlZCBPcGVuQXBpR2F0ZXdheUxhbWJkYUFwaSBjb25zdHJ1Y3RcbiAgXCJhcGkucHlcIjogYGZyb20gZGF0YWNsYXNzZXMgaW1wb3J0IGZpZWxkc1xuZnJvbSAke29wdGlvbnMub3BlbkFwaUdhdGV3YXlQYWNrYWdlTmFtZX0gaW1wb3J0IE9wZW5BcGlHYXRld2F5TGFtYmRhQXBpLCBPcGVuQXBpSW50ZWdyYXRpb25cbmZyb20gJHtvcHRpb25zLnB5dGhvbkNsaWVudFBhY2thZ2VOYW1lfS5hcGkuZGVmYXVsdF9hcGlfb3BlcmF0aW9uX2NvbmZpZyBpbXBvcnQgT3BlcmF0aW9uTG9va3VwLCBPcGVyYXRpb25Db25maWdcbmZyb20gJHtvcHRpb25zLm1vZHVsZU5hbWV9IGltcG9ydCBTUEVDLCBTUEVDX1BBVEhcblxuY2xhc3MgQXBpKE9wZW5BcGlHYXRld2F5TGFtYmRhQXBpKTpcbiAgICBcIlwiXCJcbiAgICBUeXBlLXNhZmUgY29uc3RydWN0IGZvciB0aGUgQVBJIEdhdGV3YXkgcmVzb3VyY2VzIGRlZmluZWQgYnkgdGhlIHNwZWMuXG4gICAgWW91IHdpbGwgbGlrZWx5IG5vdCBuZWVkIHRvIG1vZGlmeSB0aGlzIGZpbGUsIGFuZCBjYW4gaW5zdGVhZCBleHRlbmQgaXQgYW5kIGRlZmluZSB5b3VyIGludGVncmF0aW9ucy5cbiAgICBcIlwiXCJcbiAgICBkZWYgX19pbml0X18oc2VsZiwgc2NvcGUsIGlkLCBpbnRlZ3JhdGlvbnM6IE9wZXJhdGlvbkNvbmZpZ1tPcGVuQXBpSW50ZWdyYXRpb25dLCAqKmt3YXJncyk6XG4gICAgICAgIHN1cGVyKCkuX19pbml0X18oc2NvcGUsIGlkLFxuICAgICAgICAgICAgKiprd2FyZ3MsXG4gICAgICAgICAgICBpbnRlZ3JhdGlvbnM9eyBmaWVsZC5uYW1lOiBnZXRhdHRyKGludGVncmF0aW9ucywgZmllbGQubmFtZSkgZm9yIGZpZWxkIGluIGZpZWxkcyhpbnRlZ3JhdGlvbnMpIH0sXG4gICAgICAgICAgICBzcGVjPVNQRUMsXG4gICAgICAgICAgICBzcGVjX3BhdGg9U1BFQ19QQVRILFxuICAgICAgICAgICAgb3BlcmF0aW9uX2xvb2t1cD1PcGVyYXRpb25Mb29rdXAsXG4gICAgICAgIClcbmAsXG4gIC4uLihvcHRpb25zLnNhbXBsZUNvZGUgIT09IGZhbHNlXG4gICAgPyB7XG4gICAgICAgIC8vIEdlbmVyYXRlIGFuIGV4YW1wbGUgd2hpY2ggaW5zdGFudGlhdGVzIHRoZSBBcGkgY29uc3RydWN0XG4gICAgICAgIC8vIFRPRE86IENvbnNpZGVyIGdlbmVyYXRpbmcgdGhpcyBzYW1wbGUgZnJvbSB0aGUgcGFyc2VkIHNwZWNcbiAgICAgICAgXCJzYW1wbGVfYXBpLnB5XCI6IGBmcm9tICR7b3B0aW9ucy5vcGVuQXBpR2F0ZXdheVBhY2thZ2VOYW1lfSBpbXBvcnQgQXV0aG9yaXplcnMsIE9wZW5BcGlJbnRlZ3JhdGlvblxuZnJvbSAke29wdGlvbnMucHl0aG9uQ2xpZW50UGFja2FnZU5hbWV9LmFwaS5kZWZhdWx0X2FwaV9vcGVyYXRpb25fY29uZmlnIGltcG9ydCBPcGVyYXRpb25Db25maWdcbmZyb20gYXdzX2Nkay5hd3NfbGFtYmRhIGltcG9ydCBMYXllclZlcnNpb24sIENvZGUsIEZ1bmN0aW9uLCBSdW50aW1lXG5mcm9tIC5hcGkgaW1wb3J0IEFwaVxuZnJvbSBjb25zdHJ1Y3RzIGltcG9ydCBDb25zdHJ1Y3RcbmZyb20gJHtvcHRpb25zLm1vZHVsZU5hbWV9IGltcG9ydCBnZXRfZ2VuZXJhdGVkX2NsaWVudF9sYXllcl9kaXJlY3RvcnlcbmZyb20gcGF0aGxpYiBpbXBvcnQgUGF0aFxuZnJvbSBvcyBpbXBvcnQgcGF0aFxuXG5jbGFzcyBTYW1wbGVBcGkoQ29uc3RydWN0KTpcbiAgXCJcIlwiXG4gIEFuIGV4YW1wbGUgb2YgaG93IHRvIGFkZCBpbnRlZ3JhdGlvbnMgdG8geW91ciBhcGlcbiAgXCJcIlwiXG4gIGRlZiBfX2luaXRfXyhzZWxmLCBzY29wZSwgaWQpOlxuICAgICAgc3VwZXIoKS5fX2luaXRfXyhzY29wZSwgaWQpXG5cbiAgICAgICMgTGF5ZXIgd2hpY2ggY29udGFpbnMgdGhlIGdlbmVyYXRlZCBjbGllbnQuXG4gICAgICBzZWxmLmdlbmVyYXRlZF9jbGllbnRfbGF5ZXIgPSBMYXllclZlcnNpb24oc2VsZiwgJ0dlbmVyYXRlZENsaWVudExheWVyJyxcbiAgICAgICAgICBjb2RlPUNvZGUuZnJvbV9hc3NldChnZXRfZ2VuZXJhdGVkX2NsaWVudF9sYXllcl9kaXJlY3RvcnkoKSlcbiAgICAgIClcblxuICAgICAgc2VsZi5hcGkgPSBBcGkoc2VsZiwgJ0FwaScsXG4gICAgICAgICAgZGVmYXVsdF9hdXRob3JpemVyPUF1dGhvcml6ZXJzLmlhbSgpLFxuICAgICAgICAgIGludGVncmF0aW9ucz1PcGVyYXRpb25Db25maWcoXG4gICAgICAgICAgICAgIHNheV9oZWxsbz1PcGVuQXBpSW50ZWdyYXRpb24oXG4gICAgICAgICAgICAgICAgICBmdW5jdGlvbj1GdW5jdGlvbihzZWxmLCAnU2F5SGVsbG8nLFxuICAgICAgICAgICAgICAgICAgICAgIHJ1bnRpbWU9UnVudGltZS5QWVRIT05fM185LFxuICAgICAgICAgICAgICAgICAgICAgIGNvZGU9Q29kZS5mcm9tX2Fzc2V0KHBhdGguam9pbihzdHIoUGF0aChfX2ZpbGVfXykucGFyZW50LmFic29sdXRlKCkpLCAnaGFuZGxlcnMnKSksXG4gICAgICAgICAgICAgICAgICAgICAgaGFuZGxlcj1cInNheV9oZWxsb19oYW5kbGVyX3NhbXBsZS5oYW5kbGVyXCIsXG4gICAgICAgICAgICAgICAgICAgICAgbGF5ZXJzPVtzZWxmLmdlbmVyYXRlZF9jbGllbnRfbGF5ZXJdLFxuICAgICAgICAgICAgICAgICAgKSxcbiAgICAgICAgICAgICAgKSxcbiAgICAgICAgICApLFxuICAgICAgKVxuYCxcbiAgICAgICAgLy8gR2VuZXJhdGUgYW4gZXhhbXBsZSBsYW1iZGEgaGFuZGxlclxuICAgICAgICBcImhhbmRsZXJzL3NheV9oZWxsb19oYW5kbGVyX3NhbXBsZS5weVwiOiBgZnJvbSAke29wdGlvbnMucHl0aG9uQ2xpZW50UGFja2FnZU5hbWV9LmFwaS5kZWZhdWx0X2FwaV9vcGVyYXRpb25fY29uZmlnIGltcG9ydCBzYXlfaGVsbG9faGFuZGxlciwgU2F5SGVsbG9SZXF1ZXN0LCBBcGlSZXNwb25zZSwgU2F5SGVsbG9PcGVyYXRpb25SZXNwb25zZXNcbmZyb20gJHtvcHRpb25zLnB5dGhvbkNsaWVudFBhY2thZ2VOYW1lfS5tb2RlbC5oZWxsb19yZXNwb25zZSBpbXBvcnQgSGVsbG9SZXNwb25zZVxuXG5Ac2F5X2hlbGxvX2hhbmRsZXJcbmRlZiBoYW5kbGVyKGlucHV0OiBTYXlIZWxsb1JlcXVlc3QsICoqa3dhcmdzKSAtPiBTYXlIZWxsb09wZXJhdGlvblJlc3BvbnNlczpcbiAgICBcIlwiXCJcbiAgICBBbiBleGFtcGxlIGxhbWJkYSBoYW5kbGVyIHdoaWNoIHVzZXMgdGhlIGdlbmVyYXRlZCBoYW5kbGVyIHdyYXBwZXIgdG8gbWFuYWdlIG1hcnNoYWxsaW5nIGlucHV0cy9vdXRwdXRzXG4gICAgXCJcIlwiXG4gICAgcmV0dXJuIEFwaVJlc3BvbnNlKFxuICAgICAgICBzdGF0dXNfY29kZT0yMDAsXG4gICAgICAgIGJvZHk9SGVsbG9SZXNwb25zZShtZXNzYWdlPVwiSGVsbG8ge30hXCIuZm9ybWF0KGlucHV0LnJlcXVlc3RfcGFyYW1ldGVyc1tcIm5hbWVcIl0pKSxcbiAgICAgICAgaGVhZGVycz17fVxuICAgIClcbmAsXG4gICAgICB9XG4gICAgOiB7fSksXG59KTtcbiJdfQ==