/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
import { AuthorizationType } from "aws-cdk-lib/aws-apigateway";
import { Construct } from "constructs";
import { OpenAPIV3 } from "openapi-types";
import { Authorizer, CustomAuthorizerType } from "../authorizers";
import { OpenApiIntegrations } from "./api-gateway-integrations-types";
/**
 * Snippet of OpenAPI API Gateway extension for a cognito x-amazon-apigateway-authorizer
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-enable-cognito-user-pool.html
 */
export interface CognitoApiGatewayAuthorizer {
    /**
     * The type of authorizer (always cognito_user_pools)
     */
    readonly type: AuthorizationType.COGNITO;
    /**
     * The arns of the user pools used to authorize the request
     */
    readonly providerARNs: string[];
}
/**
 * Snippet of OpenAPI API Gateway extension for a custom x-amazon-apigateway-authorizer
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-authorizer.html
 */
export interface CustomApiGatewayAuthorizer {
    /**
     * Type of custom authorizer
     */
    readonly type: CustomAuthorizerType;
    /**
     * The part of the request that denotes the identity of the caller
     */
    readonly identitySource?: string;
    /**
     * The lambda invocation uri for the custom authorizer
     */
    readonly authorizerUri: string;
    /**
     * The time in seconds that the authorizer result is cached given the same identity source
     */
    readonly authorizerResultTtlInSeconds: number;
}
/**
 * Open API definition for an api gateway security scheme
 */
export interface ApiGatewaySecurityScheme extends OpenAPIV3.ApiKeySecurityScheme {
    /**
     * The type of api gateway authorizer
     */
    readonly "x-amazon-apigateway-authtype": string;
}
/**
 * Open API definition for an api gateway security scheme with authorizer details
 */
export interface ApiGatewaySecuritySchemeWithAuthorizer<AuthorizerType> extends ApiGatewaySecurityScheme {
    /**
     * Details about the authorizer
     */
    readonly "x-amazon-apigateway-authorizer": AuthorizerType;
}
/**
 * The security scheme for a cognito authorizer
 */
export declare type CognitoSecurityScheme = ApiGatewaySecuritySchemeWithAuthorizer<CognitoApiGatewayAuthorizer>;
/**
 * The security scheme for a custom authorizer
 */
export declare type CustomSecurityScheme = ApiGatewaySecuritySchemeWithAuthorizer<CustomApiGatewayAuthorizer>;
/**
 * The security scheme for an iam authorizer
 */
export declare type IamSecurityScheme = ApiGatewaySecurityScheme;
/**
 * Serialised representation of a method authorizer
 */
export interface SerialisedAuthorizerReference {
    /**
     * The unique identifier of the authorizer to reference
     */
    readonly authorizerId: string;
    /**
     * Scopes to use for this particular authorizer reference
     */
    readonly authorizationScopes?: string[];
}
/**
 * Build a serialized reference to an authorizer for use in an api method
 * @param authorizer the author to serialize
 */
export declare const serializeAsAuthorizerReference: (authorizer: Authorizer) => SerialisedAuthorizerReference | undefined;
/**
 * Return a list of all unique authorizers used in the api
 */
export declare const getAllAuthorizers: (integrations: OpenApiIntegrations, defaultAuthorizer?: Authorizer | undefined) => Authorizer[];
/**
 * Generate the security schemes section of an OpenAPI specification
 */
export declare const prepareSecuritySchemes: (scope: Construct, integrations: OpenApiIntegrations, defaultAuthorizer?: Authorizer | undefined) => {
    [key: string]: OpenAPIV3.SecuritySchemeObject;
};
