"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ParsedSpec = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const util_1 = require("projen/lib/util");
/**
 * Component for parsing the yaml OpenAPI spec as a single json object, resolving references etc.
 */
class ParsedSpec extends projen_1.Component {
    constructor(project, options) {
        super(project);
        this.options = options;
    }
    static parse(specPath, outputPath) {
        // Parse the spec and write to the target output path
        util_1.exec(`./parse-openapi-spec --specPath=${specPath} --outputPath=${outputPath}`, {
            cwd: path.resolve(__dirname, "..", "..", "..", "..", "scripts", "parser"),
        });
    }
    synthesize() {
        super.synthesize();
        ParsedSpec.parse(this.options.specPath, this.options.outputPath);
        const singleSpecFile = util_1.tryReadFileSync(this.options.outputPath);
        if (!singleSpecFile) {
            throw new Error(`Failed to parse specification at ${this.options.specPath}`);
        }
        const parsedSpec = JSON.parse(singleSpecFile);
        // TODO: Remove this validation and update mustache templates as appropriate when the following has been addressed:
        // https://github.com/OpenAPITools/openapi-generator/pull/14568
        // Check that each operation has zero or one tags
        const operationsWithMultipleTags = Object.entries(parsedSpec.paths).flatMap(([urlPath, methods]) => Object.entries(methods ?? {})
            .filter(([, operation]) => typeof operation === "object" &&
            "tags" in operation &&
            operation.tags &&
            operation.tags.length > 1)
            .map(([httpMethod]) => `${httpMethod} ${urlPath}`));
        if (operationsWithMultipleTags.length > 0) {
            throw new Error(`Operations with multiple tags are not yet supported, please tag operations with at most one tag. The following operations have multiple tags: ${operationsWithMultipleTags.join(", ")}`);
        }
    }
}
exports.ParsedSpec = ParsedSpec;
//# sourceMappingURL=data:application/json;base64,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