/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
import type { OpenAPIV3 } from "openapi-types";
import { ApiGatewayIntegration } from "../integrations";
import type { MethodAndPath, OperationLookup } from "../spec";
import { SerialisedAuthorizerReference } from "../spec/api-gateway-auth";
/**
 * Serialise a method and path into a single string
 */
export declare const concatMethodAndPath: ({ method, path }: MethodAndPath) => string;
/**
 * Serialized integration for a method
 */
export interface SerializedMethodIntegration {
    /**
     * The lambda function invocation uri for the api method
     */
    readonly integration: ApiGatewayIntegration;
    /**
     * The authorizer (if any) to apply to the method
     */
    readonly methodAuthorizer?: SerialisedAuthorizerReference;
}
/**
 * Cross-origin resource sharing options
 */
export interface SerializedCorsOptions {
    /**
     * HTTP methods to allow
     */
    readonly allowMethods: string[];
    /**
     * Headers to allow
     */
    readonly allowHeaders: string[];
    /**
     * Origins to allow
     */
    readonly allowOrigins: string[];
    /**
     * HTTP status code to be returned by preflight requests
     */
    readonly statusCode: number;
}
/**
 * Options for preparing an api spec for deployment by api gateway
 */
export interface PrepareApiSpecOptions {
    /**
     * Integrations for api operations
     */
    readonly integrations: {
        [operationId: string]: SerializedMethodIntegration;
    };
    /**
     * Options for cross-origin resource sharing
     */
    readonly corsOptions?: SerializedCorsOptions;
    /**
     * Operation id to method and path mapping
     */
    readonly operationLookup: OperationLookup;
    /**
     * Security schemes to add to the spec
     */
    readonly securitySchemes: {
        [key: string]: OpenAPIV3.SecuritySchemeObject;
    };
    /**
     * The default authorizer to reference
     */
    readonly defaultAuthorizerReference?: SerialisedAuthorizerReference;
}
/**
 * Prepares the api spec for deployment by adding integrations, configuring auth, etc
 */
export declare const prepareApiSpec: (spec: OpenAPIV3.Document, options: PrepareApiSpecOptions) => OpenAPIV3.Document;
