"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const aws_sdk_1 = require("aws-sdk"); // eslint-disable-line
const prepare_spec_1 = require("./prepare-spec");
const s3 = new aws_sdk_1.S3();
/**
 * Prepare the api spec for API Gateway
 * @param inputSpecLocation location of the specification to prepare
 * @param outputSpecLocation location to write the prepared spec to
 * @param options integrations, authorizers etc to apply
 */
const prepare = async ({ inputSpecLocation, outputSpecLocation, ...options }) => {
    // Read the spec from the s3 input location
    const inputSpec = JSON.parse((await s3
        .getObject({
        Bucket: inputSpecLocation.bucket,
        Key: inputSpecLocation.key,
    })
        .promise()).Body.toString("utf-8"));
    // Prepare the spec
    const preparedSpec = prepare_spec_1.prepareApiSpec(inputSpec, options);
    // Write the spec to the s3 output location
    await s3
        .putObject({
        Bucket: outputSpecLocation.bucket,
        Key: outputSpecLocation.key,
        Body: JSON.stringify(preparedSpec),
    })
        .promise();
};
exports.handler = async (event) => {
    const { outputSpecLocation } = event.ResourceProperties;
    switch (event.RequestType) {
        case "Create":
        case "Update":
            // Prepare the spec on create
            await prepare(event.ResourceProperties);
            break;
        case "Delete":
        // Nothing to do for delete
        default:
            break;
    }
    return {
        PhysicalResourceId: outputSpecLocation.key,
        Status: "SUCCESS",
    };
};
//# sourceMappingURL=data:application/json;base64,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