"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CognitoToApiGatewayToLambda = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const api = require("aws-cdk-lib/aws-apigateway");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
class CognitoToApiGatewayToLambda extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the CognitoToApiGatewayToLambda class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {CognitoToApiGatewayToLambdaProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckProps(props);
        // This Construct requires that the auth type be COGNITO regardless of what is specified in the props
        if (props.apiGatewayProps) {
            if (props.apiGatewayProps.defaultMethodOptions === undefined) {
                props.apiGatewayProps.defaultMethodOptions = {
                    authorizationType: api.AuthorizationType.COGNITO,
                };
            }
            else if (props.apiGatewayProps?.defaultMethodOptions.authorizationType === undefined) {
                props.apiGatewayProps.defaultMethodOptions.authorizationType = api.AuthorizationType.COGNITO;
            }
            else if (props.apiGatewayProps?.defaultMethodOptions.authorizationType !== 'COGNITO_USER_POOLS') {
                defaults.printWarning('Overriding Authorization type to be AuthorizationType.COGNITO');
                props.apiGatewayProps.defaultMethodOptions.authorizationType = api.AuthorizationType.COGNITO;
            }
        }
        if (props.apiGatewayProps && (typeof props.apiGatewayProps.proxy !== 'undefined') && (props.apiGatewayProps.proxy === false)) {
            defaults.printWarning('For non-proxy API, addAuthorizers() method must be called after all the resources and methods for API are fuly defined. Not calling addAuthorizers() will result in API methods NOT protected by Cognito.');
        }
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] =
            defaults.GlobalLambdaRestApi(this, this.lambdaFunction, props.apiGatewayProps, props.logGroupProps);
        this.userPool = defaults.buildUserPool(this, props.cognitoUserPoolProps);
        this.userPoolClient = defaults.buildUserPoolClient(this, this.userPool, props.cognitoUserPoolClientProps);
        this.apiGatewayAuthorizer = new api.CfnAuthorizer(this, 'CognitoAuthorizer', {
            restApiId: this.apiGateway.restApiId,
            type: 'COGNITO_USER_POOLS',
            providerArns: [this.userPool.userPoolArn],
            identitySource: "method.request.header.Authorization",
            name: "authorizer"
        });
        this.addAuthorizers();
    }
    addAuthorizers() {
        this.apiGateway.methods.forEach((apiMethod) => {
            // Leave the authorizer NONE for HTTP OPTIONS method to support CORS, for the rest set it to COGNITO
            const child = apiMethod.node.findChild('Resource');
            if (apiMethod.httpMethod === 'OPTIONS') {
                child.addPropertyOverride('AuthorizationType', 'NONE');
            }
            else {
                child.addPropertyOverride('AuthorizationType', 'COGNITO_USER_POOLS');
                child.addPropertyOverride('AuthorizerId', { Ref: this.apiGatewayAuthorizer.logicalId });
            }
        });
    }
}
exports.CognitoToApiGatewayToLambda = CognitoToApiGatewayToLambda;
_a = JSII_RTTI_SYMBOL_1;
CognitoToApiGatewayToLambda[_a] = { fqn: "@aws-solutions-constructs/aws-cognito-apigateway-lambda.CognitoToApiGatewayToLambda", version: "2.26.0" };
//# sourceMappingURL=data:application/json;base64,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