"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.sonarqubeScanner = exports.createSonarqubeProject = exports.generateSonarqubeReports = void 0;
const cube = (path, action = "GET", args) => `curl -X ${action} -u $SONARQUBE_TOKEN: "$SONARQUBE_ENDPOINT${path}" ${args}`;
const setupSonarqubeQualityGates = (defaultProfileOrGateName, specificProfileOrGateName) => [
    `export DEFAULT_GATE=\`${cube(`/api/qualitygates/search?gateName=${defaultProfileOrGateName}`)}\``,
    `export SPECIFIC_GATE=\`${cube(`/api/qualitygates/search?gateName=${specificProfileOrGateName}`)}\``,
    `if [[ "$(echo $SPECIFIC_GATE | jq .errors)" == "null" && "$(echo $SPECIFIC_GATE | jq '.results | length')" -gt 0 ]]; then export GATE_NAME=${specificProfileOrGateName}; else export GATE_NAME=${defaultProfileOrGateName}; fi`,
    `${cube("/api/qualitygates/select?projectKey=$PROJECT_NAME&gateName=$GATE_NAME", "POST")}`,
];
const setupSonarqubeQualityProfiles = (defaultProfileOrGateName, specificProfileOrGateName) => [
    `export DEFAULT_PROFILE=\`${cube(`/api/qualityprofiles/search?qualityProfile=${defaultProfileOrGateName}`)} | jq .profiles\``,
    `export SPECIFIC_PROFILE=\`${cube(`/api/qualityprofiles/search?qualityProfile=${specificProfileOrGateName}`)} | jq .profiles\``,
    `export MERGED_PROFILES=\`jq --argjson arr1 "$DEFAULT_PROFILE" --argjson arr2 "$SPECIFIC_PROFILE" -n '$arr1 + $arr2 | group_by(.language) | map(.[-1])'\``,
    `echo $MERGED_PROFILES | jq -c '.[]' | while read i; do ${cube("/api/qualityprofiles/add_project?project=$PROJECT_NAME&language=`echo $i | jq -r .language`&qualityProfile=`echo $i | jq -r .name`", "POST")}; done`,
];
const setupSonarqubePermissions = (authorizedGroup) => !authorizedGroup
    ? []
    : [
        "admin",
        "codeviewer",
        "issueadmin",
        "securityhotspotadmin",
        "scan",
        "user",
    ].map((p) => `${cube(`/api/permissions/add_group?projectKey=$PROJECT_NAME&groupName=${authorizedGroup}&permission=${p}`, "POST")}`);
const setupSonarqubeProject = ({ sonarqubeTags, sonarqubeAuthorizedGroup, sonarqubeDefaultProfileOrGateName, sonarqubeSpecificProfileOrGateName, }) => {
    return [
        ...setupSonarqubePermissions(sonarqubeAuthorizedGroup),
        `${cube("/api/project_branches/rename?project=$PROJECT_NAME&name=mainline", "POST")}`,
        `${cube(`/api/project_tags/set?project=$PROJECT_NAME&tags=${[
            sonarqubeAuthorizedGroup,
            ...(sonarqubeTags || []),
        ].join(",")}`, "POST")}`,
        ...setupSonarqubeQualityProfiles(sonarqubeDefaultProfileOrGateName, sonarqubeSpecificProfileOrGateName),
        ...setupSonarqubeQualityGates(sonarqubeDefaultProfileOrGateName, sonarqubeSpecificProfileOrGateName),
    ].join(";");
};
exports.generateSonarqubeReports = () => [
    cube("/api/bitegarden/report/pdf_issues_breakdown?resource=$PROJECT_NAME&branch=mainline", "GET", "--output reports/prototype-issues-report.pdf"),
    cube("/api/bitegarden/report/pdf?resource=$PROJECT_NAME&branch=mainline", "GET", "--output reports/prototype-executive-report.pdf"),
    cube("/api/security_reports/download?project=$PROJECT_NAME", "GET", "--output reports/prototype-security-report.pdf"),
];
exports.createSonarqubeProject = (props) => [
    `CREATE_PROJECT_OUTPUT=\`${cube("/api/projects/create?name=$PROJECT_NAME&project=$PROJECT_NAME&visibility=private", "POST")}\``,
    `if [[ "$(echo $CREATE_PROJECT_OUTPUT | jq .errors)" == "null" ]]; then ${setupSonarqubeProject(props)}; fi;`,
    "mkdir -p reports",
];
exports.sonarqubeScanner = () => [
    "npx sonarqube-scanner -Dsonar.login=$SONARQUBE_TOKEN",
    "-Dsonar.projectKey=$PROJECT_NAME",
    "-Dsonar.projectName=$PROJECT_NAME",
    "-Dsonar.projectVersion=`git rev-parse --short HEAD`",
    "-Dsonar.branch.name=mainline",
    "-Dsonar.host.url=$SONARQUBE_ENDPOINT",
    "-Dsonar.cfn.nag.reportFiles=reports/cfn-nag-report.json",
    "-Dsonar.dependencyCheck.htmlReportPath=reports/dependency-check-report.html",
].join(" ");
//# sourceMappingURL=data:application/json;base64,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