#!/usr/bin/env ts-node
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const console = require("console");
const path = require("path");
const process = require("process");
const fs = require("fs-extra");
const cdk_1 = require("projen/lib/cdk");
// The directory where our 'package.json' lives
const MONOPACKAGE_ROOT = process.cwd();
const ROOT_PATH = findWorkspacePath();
const LIBRARIES_ROOT = path.resolve(ROOT_PATH, 'packages');
const UBER_PACKAGE_JSON_PATH = path.join(MONOPACKAGE_ROOT, 'package.json');
const EXCLUDED_PACKAGES = [];
async function main() {
    console.log(`🌴  workspace root path is: ${ROOT_PATH}`);
    const uberPackageJson = await fs.readJson(UBER_PACKAGE_JSON_PATH);
    const libraries = await findLibrariesToPackage(uberPackageJson);
    await verifyDependencies(uberPackageJson, libraries);
    await prepareSourceFiles(libraries, uberPackageJson);
}
main().then(() => process.exit(0), (err) => {
    console.error('❌ An error occurred: ', err.stack);
    process.exit(1);
});
/**
 * Find the workspace root path. Walk up the directory tree until you find nx.json
 */
function findWorkspacePath() {
    return _findRootPath(process.cwd());
    function _findRootPath(part) {
        if (part === path.resolve(part, '..')) {
            throw new Error('couldn\'t find a \'nx.json\' file when walking up the directory tree, are you in a aws-pdk project?');
        }
        if (fs.existsSync(path.resolve(part, 'nx.json'))) {
            return part;
        }
        return _findRootPath(path.resolve(part, '..'));
    }
}
async function findLibrariesToPackage(uberPackageJson) {
    console.log('🔍 Discovering libraries that need packaging...');
    const deprecatedPackages = uberPackageJson.bundle?.deprecatedPackages;
    const result = new Array();
    for (const dir of await fs.readdir(LIBRARIES_ROOT)) {
        const packageJsonPath = path.resolve(LIBRARIES_ROOT, dir, 'package.json');
        if (!fs.pathExistsSync(packageJsonPath)) {
            continue;
        }
        const packageJson = await fs.readJson(packageJsonPath);
        if (packageJson.bundle?.exclude || EXCLUDED_PACKAGES.includes(packageJson.name)) {
            console.log(`\t⚠️ Skipping (bundle excluded):    ${packageJson.name}`);
            continue;
        }
        else if (packageJson.jsii == null) {
            console.log(`\t⚠️ Skipping (not jsii-enabled):   ${packageJson.name}`);
            continue;
        }
        else if (deprecatedPackages?.some(packageName => packageName === packageJson.name)) {
            console.log(`\t⚠️ Skipping (bundle deprecated): ${packageJson.name}`);
            continue;
        }
        else if (packageJson.deprecated) {
            console.log(`\t⚠️ Skipping (deprecated):         ${packageJson.name}`);
            continue;
        }
        else if (packageJson.stability !== cdk_1.Stability.STABLE) {
            console.log(`\t⚠️ Skipping (non stable):         ${packageJson.name}`);
            continue;
        }
        result.push({
            packageJson,
            root: path.join(LIBRARIES_ROOT, dir),
            shortName: packageJson.name === 'aws-prototyping-sdk' ? packageJson.name : packageJson.name.slice('@aws-prototyping-sdk/'.length),
        });
    }
    console.log(`\tℹ️ Found ${result.length} relevant packages!`);
    return result;
}
async function verifyDependencies(packageJson, libraries) {
    console.log('🧐 Verifying dependencies are complete...');
    const toBundle = {};
    for (const library of libraries) {
        for (const depName of library.packageJson.bundleDependencies ?? library.packageJson.bundledDependencies ?? []) {
            const requiredVersion = library.packageJson.devDependencies?.[depName]
                ?? library.packageJson.dependencies?.[depName]
                ?? '*';
            if (toBundle[depName] != null && toBundle[depName] !== requiredVersion) {
                throw new Error(`Required to bundle different versions of ${depName}: ${toBundle[depName]} and ${requiredVersion}.`);
            }
            toBundle[depName] = requiredVersion;
        }
        if (!(library.packageJson.name in packageJson.devDependencies)) {
            throw new Error(`\t⚠️ Missing dev dependency: ${library.packageJson.name}`);
        }
    }
    const workspacePath = path.resolve(ROOT_PATH, 'package.json');
    const workspace = await fs.readJson(workspacePath);
    const spuriousBundledDeps = new Set(packageJson.bundledDependencies ?? []);
    for (const [name, version] of Object.entries(toBundle)) {
        spuriousBundledDeps.delete(name);
        const nohoist = `${packageJson.name}/${name}`;
        if (!workspace.workspaces.nohoist?.includes(nohoist)) {
            throw new Error(`\t⚠️ Missing yarn workspace nohoist: ${nohoist}`);
        }
        if (!(packageJson.bundledDependencies?.includes(name))) {
            throw new Error(`\t⚠️ Missing bundled dependency: ${name} at ${version}`);
        }
        if (packageJson.dependencies?.[name] !== version) {
            throw new Error(`\t⚠️ Missing or incorrect dependency: ${name} at ${version}`);
        }
    }
    packageJson.bundledDependencies = packageJson.bundledDependencies?.filter((dep) => !spuriousBundledDeps.has(dep));
    if (spuriousBundledDeps.size > 0) {
        throw new Error(`\t⚠️ Spurious bundled dependencies detected. Please remove from dependencies: ${spuriousBundledDeps}`);
    }
    console.log('\t✅ Dependencies are correct!');
}
async function prepareSourceFiles(libraries, packageJson) {
    console.log('📝 Preparing source files...');
    const libRoot = resolveLibRoot(packageJson);
    // Should not remove collection directory if we're currently in it. The OS would be unhappy.
    if (libRoot !== process.cwd()) {
        await fs.remove(libRoot);
    }
    const indexStatements = new Array();
    for (const library of libraries) {
        const libDir = path.join(libRoot, library.shortName);
        const copied = await transformPackage(library, packageJson, libDir, libraries);
        if (!copied) {
            continue;
        }
        indexStatements.push(`export * as ${library.shortName.replace(/-/g, '_')} from './${library.shortName}';`);
    }
    await fs.writeFile(path.join(libRoot, 'index.ts'), indexStatements.join('\n'), { encoding: 'utf8' });
    console.log('\t🍺 Success!');
}
async function transformPackage(library, uberPackageJson, destination, allLibraries) {
    await fs.mkdirp(destination);
    await copyOrTransformFiles(library.root, destination, allLibraries, uberPackageJson);
    await fs.writeFile(path.join(destination, 'index.ts'), `export * from './src';\n`, { encoding: 'utf8' });
    const config = uberPackageJson.jsii.targets;
    await fs.writeJson(path.join(destination, '.jsiirc.json'), {
        targets: transformTargets(config, library.packageJson.jsii.targets),
    }, { spaces: 2 });
    // if libRoot is _not_ under the root of the package, generate a file at the
    // root that will refer to the one under lib/ so that users can still import
    // from "monocdk/aws-lambda".
    const relativeLibRoot = uberPackageJson.bundle?.libRoot;
    if (relativeLibRoot && relativeLibRoot !== '.') {
        await fs.writeFile(path.resolve(MONOPACKAGE_ROOT, `${library.shortName}.ts`), `export * from './${relativeLibRoot}/${library.shortName}';\n`, { encoding: 'utf8' });
    }
    return true;
}
function transformTargets(monoConfig, targets) {
    if (targets == null) {
        return targets;
    }
    const result = {};
    for (const [language, config] of Object.entries(targets)) {
        switch (language) {
            case 'dotnet':
                if (monoConfig?.dotnet != null) {
                    result[language] = {
                        namespace: config.namespace,
                    };
                }
                break;
            case 'java':
                if (monoConfig?.java != null) {
                    result[language] = {
                        package: config.package,
                    };
                }
                break;
            case 'python':
                if (monoConfig?.python != null) {
                    result[language] = {
                        module: `${monoConfig.python.module}.${config.module.replace(/^aws_prototyping_sdk\./, '')}`,
                    };
                }
                break;
            default:
                throw new Error(`Unsupported language for submodule configuration translation: ${language}`);
        }
    }
    return result;
}
async function copyOrTransformFiles(from, to, libraries, uberPackageJson) {
    const promises = (await fs.readdir(from)).map(async (name) => {
        if (shouldIgnoreFile(name)) {
            return;
        }
        if (name.endsWith('.d.ts') || name.endsWith('.js')) {
            if (await fs.pathExists(path.join(from, name.replace(/\.(d\.ts|js)$/, '.ts')))) {
                // We won't copy .d.ts and .js files with a corresponding .ts file
                return;
            }
        }
        const source = path.join(from, name);
        const destination = path.join(to, name);
        const stat = await fs.stat(source);
        if (stat.isDirectory()) {
            await fs.mkdirp(destination);
            return copyOrTransformFiles(source, destination, libraries, uberPackageJson);
        }
        if (name.endsWith(".ts")) {
            const sourceCode = fs.readFileSync(source).toString()
                .replace(/(import .* from ["'])@aws-prototyping-sdk(\/.*['"];)/g, `$1${path.relative(path.dirname(destination), path.join(LIBRARIES_ROOT, "aws-prototyping-sdk"))}$2`);
            return fs.writeFile(destination, sourceCode);
        }
        else {
            return fs.copyFile(source, destination);
        }
    });
    await Promise.all(promises);
}
const IGNORED_FILE_NAMES = new Set([
    '.eslintrc.js',
    '.gitignore',
    '.jest.config.js',
    '.jsii',
    '.env',
    'target',
    'dist',
    'lib',
    '.npmignore',
    'node_modules',
    'package.json',
    'tsconfig.json',
    'tsconfig.tsbuildinfo',
    'LICENSE',
    'NOTICE',
]);
function shouldIgnoreFile(name) {
    return IGNORED_FILE_NAMES.has(name);
}
/**
 * Resolves the directory where we're going to collect all the libraries.
 *
 * By default, this is purposely the same as the monopackage root so that our
 * two import styles resolve to the same files but it can be overridden by
 * seeting `bundle.libRoot` in the package.json of the uber package.
 *
 * @param uberPackageJson package.json contents of the uber package
 * @returns The directory where we should collect all the libraries.
 */
function resolveLibRoot(uberPackageJson) {
    return path.resolve(uberPackageJson.bundle?.libRoot ?? MONOPACKAGE_ROOT);
}
//# sourceMappingURL=data:application/json;base64,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