"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const lib_1 = require("../lib");
const cdk = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const api = require("@aws-cdk/aws-apigateway");
require("@aws-cdk/assert/jest");
function deployNewFunc(stack) {
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler'
    };
    return new lib_1.CloudFrontToApiGatewayToLambda(stack, 'test-cloudfront-apigateway-lambda', {
        lambdaFunctionProps
    });
}
function useExistingFunc(stack) {
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    return new lib_1.CloudFrontToApiGatewayToLambda(stack, 'test-cloudfront-apigateway-lambda', {
        existingLambdaObj: new lambda.Function(stack, 'MyExistingFunction', lambdaFunctionProps)
    });
}
test('snapshot test CloudFrontToApiGatewayToLambda default params', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.cloudFrontWebDistribution !== null);
    expect(construct.apiGateway !== null);
    expect(construct.lambdaFunction !== null);
    expect(construct.edgeLambdaFunctionVersion !== null);
    expect(construct.cloudFrontLoggingBucket !== null);
    expect(construct.apiGatewayCloudWatchRole !== null);
    expect(construct.apiGatewayLogGroup !== null);
});
test('check lambda function properties for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                "testcloudfrontapigatewaylambdaLambdaFunctionServiceRoleCB74590F",
                "Arn"
            ]
        },
        Runtime: "nodejs10.x",
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1"
            }
        }
    });
});
test('check lambda function role for deploy: false', () => {
    const stack = new cdk.Stack();
    useExistingFunc(stack);
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "lambda.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        ManagedPolicyArns: [
            {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                    ]
                ]
            }
        ]
    });
});
test('check exception for Missing existingObj from props', () => {
    const stack = new cdk.Stack();
    const props = {};
    try {
        new lib_1.CloudFrontToApiGatewayToLambda(stack, 'test-cloudfront-apigateway-lambda', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('check no prop', () => {
    const stack = new cdk.Stack();
    const props = {};
    try {
        new lib_1.CloudFrontToApiGatewayToLambda(stack, 'test-cloudfront-apigateway-lambda', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('override api gateway properties with existingLambdaObj', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler'
    };
    const fn = new lambda.Function(stack, 'MyExistingFunction', lambdaFunctionProps);
    new lib_1.CloudFrontToApiGatewayToLambda(stack, 'test-cloudfront-apigateway-lambda', {
        existingLambdaObj: fn,
        apiGatewayProps: {
            options: {
                description: "Override description"
            }
        }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Description: "Override description",
        EndpointConfiguration: {
            Types: [
                "REGIONAL"
            ]
        },
        Name: "LambdaRestApi"
    });
});
test('override api gateway properties without existingLambdaObj', () => {
    const stack = new cdk.Stack();
    new lib_1.CloudFrontToApiGatewayToLambda(stack, 'test-cloudfront-apigateway-lambda', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler'
        },
        apiGatewayProps: {
            endpointConfiguration: {
                types: [api.EndpointType.PRIVATE],
            },
            options: {
                description: "Override description"
            }
        }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Description: "Override description",
        EndpointConfiguration: {
            Types: [
                "PRIVATE"
            ]
        },
        Name: "LambdaRestApi"
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5jbG91ZGZyb250LWFwaWdhdGV3YXktbGFtYmRhLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LmNsb3VkZnJvbnQtYXBpZ2F0ZXdheS1sYW1iZGEudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7O0FBRUgsNENBQTZDO0FBQzdDLGdDQUE2RjtBQUM3RixxQ0FBcUM7QUFDckMsOENBQThDO0FBQzlDLCtDQUErQztBQUMvQyxnQ0FBOEI7QUFFOUIsU0FBUyxhQUFhLENBQUMsS0FBZ0I7SUFDckMsTUFBTSxtQkFBbUIsR0FBeUI7UUFDaEQsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7UUFDbEQsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztRQUNuQyxPQUFPLEVBQUUsZUFBZTtLQUN6QixDQUFDO0lBRUYsT0FBTyxJQUFJLG9DQUE4QixDQUFDLEtBQUssRUFBRSxtQ0FBbUMsRUFBRTtRQUNwRixtQkFBbUI7S0FDcEIsQ0FBQyxDQUFDO0FBQ0wsQ0FBQztBQUVELFNBQVMsZUFBZSxDQUFDLEtBQWdCO0lBQ3ZDLE1BQU0sbUJBQW1CLEdBQXlCO1FBQ2hELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7UUFDbkMsT0FBTyxFQUFFLGVBQWU7UUFDeEIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7S0FDbkQsQ0FBQztJQUVGLE9BQU8sSUFBSSxvQ0FBOEIsQ0FBQyxLQUFLLEVBQUUsbUNBQW1DLEVBQUU7UUFDcEYsaUJBQWlCLEVBQUUsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxvQkFBb0IsRUFBRSxtQkFBbUIsQ0FBQztLQUN6RixDQUFDLENBQUM7QUFDTCxDQUFDO0FBRUQsSUFBSSxDQUFDLDZEQUE2RCxFQUFFLEdBQUcsRUFBRTtJQUN2RSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixhQUFhLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDckIsTUFBTSxDQUFDLG1CQUFVLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxlQUFlLEVBQUUsQ0FBQztBQUMvRCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxrQkFBa0IsRUFBRSxHQUFHLEVBQUU7SUFDNUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFFOUIsTUFBTSxTQUFTLEdBQW1DLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUV2RSxNQUFNLENBQUMsU0FBUyxDQUFDLHlCQUF5QixLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ3JELE1BQU0sQ0FBQyxTQUFTLENBQUMsVUFBVSxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ3RDLE1BQU0sQ0FBQyxTQUFTLENBQUMsY0FBYyxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQzFDLE1BQU0sQ0FBQyxTQUFTLENBQUMseUJBQXlCLEtBQUssSUFBSSxDQUFDLENBQUM7SUFDckQsTUFBTSxDQUFDLFNBQVMsQ0FBQyx1QkFBdUIsS0FBSyxJQUFJLENBQUMsQ0FBQztJQUNuRCxNQUFNLENBQUMsU0FBUyxDQUFDLHdCQUF3QixLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ3BELE1BQU0sQ0FBQyxTQUFTLENBQUMsa0JBQWtCLEtBQUssSUFBSSxDQUFDLENBQUM7QUFDaEQsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsbURBQW1ELEVBQUUsR0FBRyxFQUFFO0lBQzdELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUVyQixNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLHVCQUF1QixFQUFFO1FBQ3BELE9BQU8sRUFBRSxlQUFlO1FBQ3hCLElBQUksRUFBRTtZQUNKLFlBQVksRUFBRTtnQkFDWixpRUFBaUU7Z0JBQ2pFLEtBQUs7YUFDTjtTQUNGO1FBQ0QsT0FBTyxFQUFFLFlBQVk7UUFDckIsV0FBVyxFQUFFO1lBQ1gsU0FBUyxFQUFFO2dCQUNULG1DQUFtQyxFQUFFLEdBQUc7YUFDekM7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDhDQUE4QyxFQUFFLEdBQUcsRUFBRTtJQUN4RCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUU5QixlQUFlLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFdkIsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsRUFBRTtRQUM3Qyx3QkFBd0IsRUFBRTtZQUN4QixTQUFTLEVBQUU7Z0JBQ1Q7b0JBQ0UsTUFBTSxFQUFFLGdCQUFnQjtvQkFDeEIsTUFBTSxFQUFFLE9BQU87b0JBQ2YsU0FBUyxFQUFFO3dCQUNULE9BQU8sRUFBRSxzQkFBc0I7cUJBQ2hDO2lCQUNGO2FBQ0Y7WUFDRCxPQUFPLEVBQUUsWUFBWTtTQUN0QjtRQUNELGlCQUFpQixFQUFFO1lBQ2pCO2dCQUNFLFVBQVUsRUFBRTtvQkFDVixFQUFFO29CQUNGO3dCQUNFLE1BQU07d0JBQ047NEJBQ0UsR0FBRyxFQUFFLGdCQUFnQjt5QkFDdEI7d0JBQ0QsMkRBQTJEO3FCQUM1RDtpQkFDRjthQUNGO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxvREFBb0QsRUFBRSxHQUFHLEVBQUU7SUFDOUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFFOUIsTUFBTSxLQUFLLEdBQXdDLEVBQ2xELENBQUM7SUFFRixJQUFJO1FBQ0YsSUFBSSxvQ0FBOEIsQ0FBQyxLQUFLLEVBQUUsbUNBQW1DLEVBQUUsS0FBSyxDQUFDLENBQUM7S0FDdkY7SUFBQyxPQUFPLENBQUMsRUFBRTtRQUNWLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLENBQUM7S0FDakM7QUFDSCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxlQUFlLEVBQUUsR0FBRyxFQUFFO0lBQ3pCLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLE1BQU0sS0FBSyxHQUF3QyxFQUNsRCxDQUFDO0lBRUYsSUFBSTtRQUNGLElBQUksb0NBQThCLENBQUMsS0FBSyxFQUFFLG1DQUFtQyxFQUFFLEtBQUssQ0FBQyxDQUFDO0tBQ3ZGO0lBQUMsT0FBTyxDQUFDLEVBQUU7UUFDVixNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxDQUFDO0tBQ2pDO0FBQ0gsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsd0RBQXdELEVBQUUsR0FBRyxFQUFFO0lBQ2xFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLE1BQU0sbUJBQW1CLEdBQXlCO1FBQ2hELElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1FBQ2xELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7UUFDbkMsT0FBTyxFQUFFLGVBQWU7S0FDekIsQ0FBQztJQUVGLE1BQU0sRUFBRSxHQUFvQixJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLG9CQUFvQixFQUFFLG1CQUFtQixDQUFDLENBQUM7SUFFbEcsSUFBSSxvQ0FBOEIsQ0FBQyxLQUFLLEVBQUUsbUNBQW1DLEVBQUU7UUFDN0UsaUJBQWlCLEVBQUUsRUFBRTtRQUNyQixlQUFlLEVBQUU7WUFDZixPQUFPLEVBQUU7Z0JBQ1AsV0FBVyxFQUFFLHNCQUFzQjthQUNwQztTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQywwQkFBMEIsRUFDdkQ7UUFDRSxXQUFXLEVBQUUsc0JBQXNCO1FBQ25DLHFCQUFxQixFQUFFO1lBQ3JCLEtBQUssRUFBRTtnQkFDTCxVQUFVO2FBQ1g7U0FDRjtRQUNELElBQUksRUFBRSxlQUFlO0tBQ3RCLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDJEQUEyRCxFQUFFLEdBQUcsRUFBRTtJQUNyRSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUU5QixJQUFJLG9DQUE4QixDQUFDLEtBQUssRUFBRSxtQ0FBbUMsRUFBRTtRQUM3RSxtQkFBbUIsRUFBRTtZQUNuQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztZQUNsRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQ25DLE9BQU8sRUFBRSxlQUFlO1NBQ3pCO1FBQ0QsZUFBZSxFQUFFO1lBQ2YscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDO2FBQ2xDO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLFdBQVcsRUFBRSxzQkFBc0I7YUFDcEM7U0FDRjtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsMEJBQTBCLEVBQ3ZEO1FBQ0UsV0FBVyxFQUFFLHNCQUFzQjtRQUNuQyxxQkFBcUIsRUFBRTtZQUNyQixLQUFLLEVBQUU7Z0JBQ0wsU0FBUzthQUNWO1NBQ0Y7UUFDRCxJQUFJLEVBQUUsZUFBZTtLQUN0QixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCAyMDIwIEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuaW1wb3J0IHsgU3ludGhVdGlscyB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgeyBDbG91ZEZyb250VG9BcGlHYXRld2F5VG9MYW1iZGEsIENsb3VkRnJvbnRUb0FwaUdhdGV3YXlUb0xhbWJkYVByb3BzIH0gZnJvbSBcIi4uL2xpYlwiO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gXCJAYXdzLWNkay9jb3JlXCI7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgKiBhcyBhcGkgZnJvbSAnQGF3cy1jZGsvYXdzLWFwaWdhdGV3YXknO1xuaW1wb3J0ICdAYXdzLWNkay9hc3NlcnQvamVzdCc7XG5cbmZ1bmN0aW9uIGRlcGxveU5ld0Z1bmMoc3RhY2s6IGNkay5TdGFjaykge1xuICBjb25zdCBsYW1iZGFGdW5jdGlvblByb3BzOiBsYW1iZGEuRnVuY3Rpb25Qcm9wcyA9IHtcbiAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTBfWCxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgfTtcblxuICByZXR1cm4gbmV3IENsb3VkRnJvbnRUb0FwaUdhdGV3YXlUb0xhbWJkYShzdGFjaywgJ3Rlc3QtY2xvdWRmcm9udC1hcGlnYXRld2F5LWxhbWJkYScsIHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzXG4gIH0pO1xufVxuXG5mdW5jdGlvbiB1c2VFeGlzdGluZ0Z1bmMoc3RhY2s6IGNkay5TdGFjaykge1xuICBjb25zdCBsYW1iZGFGdW5jdGlvblByb3BzOiBsYW1iZGEuRnVuY3Rpb25Qcm9wcyA9IHtcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTBfWCxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYClcbiAgfTtcblxuICByZXR1cm4gbmV3IENsb3VkRnJvbnRUb0FwaUdhdGV3YXlUb0xhbWJkYShzdGFjaywgJ3Rlc3QtY2xvdWRmcm9udC1hcGlnYXRld2F5LWxhbWJkYScsIHtcbiAgICBleGlzdGluZ0xhbWJkYU9iajogbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ015RXhpc3RpbmdGdW5jdGlvbicsIGxhbWJkYUZ1bmN0aW9uUHJvcHMpXG4gIH0pO1xufVxuXG50ZXN0KCdzbmFwc2hvdCB0ZXN0IENsb3VkRnJvbnRUb0FwaUdhdGV3YXlUb0xhbWJkYSBkZWZhdWx0IHBhcmFtcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGRlcGxveU5ld0Z1bmMoc3RhY2spO1xuICBleHBlY3QoU3ludGhVdGlscy50b0Nsb3VkRm9ybWF0aW9uKHN0YWNrKSkudG9NYXRjaFNuYXBzaG90KCk7XG59KTtcblxudGVzdCgnY2hlY2sgcHJvcGVydGllcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgY29uc3QgY29uc3RydWN0OiBDbG91ZEZyb250VG9BcGlHYXRld2F5VG9MYW1iZGEgPSBkZXBsb3lOZXdGdW5jKHN0YWNrKTtcblxuICBleHBlY3QoY29uc3RydWN0LmNsb3VkRnJvbnRXZWJEaXN0cmlidXRpb24gIT09IG51bGwpO1xuICBleHBlY3QoY29uc3RydWN0LmFwaUdhdGV3YXkgIT09IG51bGwpO1xuICBleHBlY3QoY29uc3RydWN0LmxhbWJkYUZ1bmN0aW9uICE9PSBudWxsKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5lZGdlTGFtYmRhRnVuY3Rpb25WZXJzaW9uICE9PSBudWxsKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5jbG91ZEZyb250TG9nZ2luZ0J1Y2tldCAhPT0gbnVsbCk7XG4gIGV4cGVjdChjb25zdHJ1Y3QuYXBpR2F0ZXdheUNsb3VkV2F0Y2hSb2xlICE9PSBudWxsKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5hcGlHYXRld2F5TG9nR3JvdXAgIT09IG51bGwpO1xufSk7XG5cbnRlc3QoJ2NoZWNrIGxhbWJkYSBmdW5jdGlvbiBwcm9wZXJ0aWVzIGZvciBkZXBsb3k6IHRydWUnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gIGRlcGxveU5ld0Z1bmMoc3RhY2spO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbicsIHtcbiAgICBIYW5kbGVyOiBcImluZGV4LmhhbmRsZXJcIixcbiAgICBSb2xlOiB7XG4gICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICBcInRlc3RjbG91ZGZyb250YXBpZ2F0ZXdheWxhbWJkYUxhbWJkYUZ1bmN0aW9uU2VydmljZVJvbGVDQjc0NTkwRlwiLFxuICAgICAgICBcIkFyblwiXG4gICAgICBdXG4gICAgfSxcbiAgICBSdW50aW1lOiBcIm5vZGVqczEwLnhcIixcbiAgICBFbnZpcm9ubWVudDoge1xuICAgICAgVmFyaWFibGVzOiB7XG4gICAgICAgIEFXU19OT0RFSlNfQ09OTkVDVElPTl9SRVVTRV9FTkFCTEVEOiBcIjFcIlxuICAgICAgfVxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgnY2hlY2sgbGFtYmRhIGZ1bmN0aW9uIHJvbGUgZm9yIGRlcGxveTogZmFsc2UnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gIHVzZUV4aXN0aW5nRnVuYyhzdGFjayk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgQXNzdW1lUm9sZVBvbGljeURvY3VtZW50OiB7XG4gICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogXCJzdHM6QXNzdW1lUm9sZVwiLFxuICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgIFByaW5jaXBhbDoge1xuICAgICAgICAgICAgU2VydmljZTogXCJsYW1iZGEuYW1hem9uYXdzLmNvbVwiXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICBdLFxuICAgICAgVmVyc2lvbjogXCIyMDEyLTEwLTE3XCJcbiAgICB9LFxuICAgIE1hbmFnZWRQb2xpY3lBcm5zOiBbXG4gICAgICB7XG4gICAgICAgIFwiRm46OkpvaW5cIjogW1xuICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgW1xuICAgICAgICAgICAgXCJhcm46XCIsXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFJlZjogXCJBV1M6OlBhcnRpdGlvblwiXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgXCI6aWFtOjphd3M6cG9saWN5L3NlcnZpY2Utcm9sZS9BV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGVcIlxuICAgICAgICAgIF1cbiAgICAgICAgXVxuICAgICAgfVxuICAgIF1cbiAgfSk7XG59KTtcblxudGVzdCgnY2hlY2sgZXhjZXB0aW9uIGZvciBNaXNzaW5nIGV4aXN0aW5nT2JqIGZyb20gcHJvcHMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gIGNvbnN0IHByb3BzOiBDbG91ZEZyb250VG9BcGlHYXRld2F5VG9MYW1iZGFQcm9wcyA9IHtcbiAgfTtcblxuICB0cnkge1xuICAgIG5ldyBDbG91ZEZyb250VG9BcGlHYXRld2F5VG9MYW1iZGEoc3RhY2ssICd0ZXN0LWNsb3VkZnJvbnQtYXBpZ2F0ZXdheS1sYW1iZGEnLCBwcm9wcyk7XG4gIH0gY2F0Y2ggKGUpIHtcbiAgICBleHBlY3QoZSkudG9CZUluc3RhbmNlT2YoRXJyb3IpO1xuICB9XG59KTtcblxudGVzdCgnY2hlY2sgbm8gcHJvcCcsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgY29uc3QgcHJvcHM6IENsb3VkRnJvbnRUb0FwaUdhdGV3YXlUb0xhbWJkYVByb3BzID0ge1xuICB9O1xuXG4gIHRyeSB7XG4gICAgbmV3IENsb3VkRnJvbnRUb0FwaUdhdGV3YXlUb0xhbWJkYShzdGFjaywgJ3Rlc3QtY2xvdWRmcm9udC1hcGlnYXRld2F5LWxhbWJkYScsIHByb3BzKTtcbiAgfSBjYXRjaCAoZSkge1xuICAgIGV4cGVjdChlKS50b0JlSW5zdGFuY2VPZihFcnJvcik7XG4gIH1cbn0pO1xuXG50ZXN0KCdvdmVycmlkZSBhcGkgZ2F0ZXdheSBwcm9wZXJ0aWVzIHdpdGggZXhpc3RpbmdMYW1iZGFPYmonLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gIGNvbnN0IGxhbWJkYUZ1bmN0aW9uUHJvcHM6IGxhbWJkYS5GdW5jdGlvblByb3BzID0ge1xuICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU18xMF9YLFxuICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICB9O1xuXG4gIGNvbnN0IGZuOiBsYW1iZGEuRnVuY3Rpb24gPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnTXlFeGlzdGluZ0Z1bmN0aW9uJywgbGFtYmRhRnVuY3Rpb25Qcm9wcyk7XG5cbiAgbmV3IENsb3VkRnJvbnRUb0FwaUdhdGV3YXlUb0xhbWJkYShzdGFjaywgJ3Rlc3QtY2xvdWRmcm9udC1hcGlnYXRld2F5LWxhbWJkYScsIHtcbiAgICBleGlzdGluZ0xhbWJkYU9iajogZm4sXG4gICAgYXBpR2F0ZXdheVByb3BzOiB7XG4gICAgICBvcHRpb25zOiB7XG4gICAgICAgIGRlc2NyaXB0aW9uOiBcIk92ZXJyaWRlIGRlc2NyaXB0aW9uXCJcbiAgICAgIH1cbiAgICB9XG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6QXBpR2F0ZXdheTo6UmVzdEFwaScsXG4gIHtcbiAgICBEZXNjcmlwdGlvbjogXCJPdmVycmlkZSBkZXNjcmlwdGlvblwiLFxuICAgIEVuZHBvaW50Q29uZmlndXJhdGlvbjoge1xuICAgICAgVHlwZXM6IFtcbiAgICAgICAgXCJSRUdJT05BTFwiXG4gICAgICBdXG4gICAgfSxcbiAgICBOYW1lOiBcIkxhbWJkYVJlc3RBcGlcIlxuICB9KTtcbn0pO1xuXG50ZXN0KCdvdmVycmlkZSBhcGkgZ2F0ZXdheSBwcm9wZXJ0aWVzIHdpdGhvdXQgZXhpc3RpbmdMYW1iZGFPYmonLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gIG5ldyBDbG91ZEZyb250VG9BcGlHYXRld2F5VG9MYW1iZGEoc3RhY2ssICd0ZXN0LWNsb3VkZnJvbnQtYXBpZ2F0ZXdheS1sYW1iZGEnLCB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wczoge1xuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTBfWCxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgYXBpR2F0ZXdheVByb3BzOiB7XG4gICAgICBlbmRwb2ludENvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgdHlwZXM6IFthcGkuRW5kcG9pbnRUeXBlLlBSSVZBVEVdLFxuICAgICAgfSxcbiAgICAgIG9wdGlvbnM6IHtcbiAgICAgICAgZGVzY3JpcHRpb246IFwiT3ZlcnJpZGUgZGVzY3JpcHRpb25cIlxuICAgICAgfVxuICAgIH1cbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpBcGlHYXRld2F5OjpSZXN0QXBpJyxcbiAge1xuICAgIERlc2NyaXB0aW9uOiBcIk92ZXJyaWRlIGRlc2NyaXB0aW9uXCIsXG4gICAgRW5kcG9pbnRDb25maWd1cmF0aW9uOiB7XG4gICAgICBUeXBlczogW1xuICAgICAgICBcIlBSSVZBVEVcIlxuICAgICAgXVxuICAgIH0sXG4gICAgTmFtZTogXCJMYW1iZGFSZXN0QXBpXCJcbiAgfSk7XG59KTsiXX0=