"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiSpecProject = void 0;
const path = require("path");
const fs_extra_1 = require("fs-extra");
const lodash_1 = require("lodash");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const util_1 = require("projen/lib/util");
const parsed_spec_1 = require("./components/parsed-spec");
// initialize logger
const logger = log4js_1.getLogger();
/**
 * Project containing the OpenAPI spec, and a parsed spec for use by the CDK construct
 */
class OpenApiSpecProject extends projen_1.Project {
    constructor(options) {
        super(options);
        this.specChanged = true;
        // Store whether we've synthesized the project
        this.synthed = false;
        logger.trace("OpenApiSpecProject constructor");
        // HACK: remove all components but the ones we are registering - removes .gitignore, tasks, etc since these are
        // unused and a distraction for end-users!
        // @ts-ignore
        this._components = [];
        this.specFileName = options.specFileName ?? "spec.yaml";
        this.parsedSpecFileName = options.parsedSpecFileName ?? ".parsed-spec.json";
        if (!this.parsedSpecFileName.endsWith(".json")) {
            throw new Error("Parsed spec file must end with .json");
        }
        this.specPath =
            options.overrideSpecPath ?? path.join(this.outdir, this.specFileName);
        this.parsedSpecPath = path.join(this.outdir, ...(options.parsedSpecOutdir ? [options.parsedSpecOutdir] : []), this.parsedSpecFileName);
        logger.debug(`specPath = "${this.specPath}"`);
        logger.debug(`parsedSpecPath = "${this.parsedSpecPath}"`);
        // Create a sample OpenAPI spec yaml if not defined
        if (!options.overrideSpecPath) {
            new projen_1.SampleFile(this, this.specFileName, {
                sourcePath: path.join(__dirname, "..", "..", "..", "samples", "spec.yaml"),
            });
        }
        // Check if there is already a .parsed-spec.json present
        const existingParsedSpecJson = util_1.tryReadFileSync(this.parsedSpecPath);
        if (existingParsedSpecJson != null) {
            // generate a new temporary parsed-spec
            const tmpParseSpecPath = `${this.parsedSpecPath}.tmp`;
            logger.trace(`Generating temp spec at "${tmpParseSpecPath}"`);
            parsed_spec_1.ParsedSpec.parse(this.specPath, tmpParseSpecPath);
            // load it
            const newParsedSpecJson = util_1.tryReadFileSync(tmpParseSpecPath);
            if (newParsedSpecJson != null) {
                const prevParseSpec = JSON.parse(existingParsedSpecJson);
                const newParsedSpec = JSON.parse(newParsedSpecJson);
                // check if spec changed and keep it
                logger.trace("Comparing previous and newly generated specs.");
                this.specChanged = !lodash_1.isEqual(prevParseSpec, newParsedSpec);
                logger.debug(`Spec Changed :: ${this.specChanged}`);
                // remove tmp parsed-spec file
                logger.trace("Removing temp spec file.");
                fs_extra_1.unlinkSync(tmpParseSpecPath);
            }
        }
        else {
            logger.debug(`No parsedSpec file found at "${this.parsedSpecPath}".`);
        }
        // Parse the spec to produce a consolidated, bundled spec which can be read by cdk constructs or other tooling
        new parsed_spec_1.ParsedSpec(this, {
            specPath: this.specPath,
            outputPath: this.parsedSpecPath,
        });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Save some time by only synthesizing once. We synthesize this project early so that it's available for the parent
        // project's install phase (pre-synth). Projen will call this method again at the usual time to synthesize this,
        // project, at which point we're already done so can skip.
        if (this.synthed) {
            logger.trace("OpenApiSpecProject already synthed. Skipping...");
            return;
        }
        logger.trace("OpenApiSpecProject synth");
        super.synth();
        this.synthed = true;
    }
}
exports.OpenApiSpecProject = OpenApiSpecProject;
//# sourceMappingURL=data:application/json;base64,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