from typing import List, Optional, Callable, NamedTuple
from BaseClasses import CollectionState
from .Options import TimespinnerOptions
from .PreCalculatedWeights import PreCalculatedWeights
from .LogicExtensions import TimespinnerLogic

EventId: Optional[int] = None


class LocationData(NamedTuple):
    region: str
    name: str
    code: Optional[int]
    rule: Optional[Callable[[CollectionState], bool]] = None


def get_location_datas(player: Optional[int], options: Optional[TimespinnerOptions],
                  precalculated_weights: Optional[PreCalculatedWeights]) -> List[LocationData]:
    flooded: Optional[PreCalculatedWeights] = precalculated_weights
    logic = TimespinnerLogic(player, options, precalculated_weights)

    # 1337000 - 1337155 Generic locations
    # 1337171 - 1337175 New Pickup checks
    # 1337246 - 1337249 Ancient Pyramid
    # 1337250 - 1337781 Torch checks
    # 1337782 - 1337999 Reserved
    location_table: List[LocationData] = [
        # Present item locations
        LocationData('Tutorial', 'Tutorial: Yo Momma 1', 1337000),
        LocationData('Tutorial', 'Tutorial: Yo Momma 2', 1337001),
        LocationData('Lake desolation', 'Lake Desolation: Starter chest 2', 1337002),
        LocationData('Lake desolation', 'Lake Desolation: Starter chest 3', 1337003),
        LocationData('Lake desolation', 'Lake Desolation: Starter chest 1', 1337004),
        LocationData('Lake desolation', 'Lake Desolation (Lower): Timespinner Wheel room', 1337005),
        LocationData('Lake desolation', 'Lake Desolation: Forget me not chest', 1337006, lambda state: logic.has_fire(state) and state.can_reach('Upper Lake Serene', 'Region', player)),
        LocationData('Lake desolation', 'Lake Desolation (Lower): Chicken chest', 1337007, logic.has_timestop),
        LocationData('Lower lake desolation', 'Lake Desolation (Lower): Not so secret room', 1337008, logic.can_break_walls),
        LocationData('Lower lake desolation', 'Lake Desolation (Upper): Tank chest', 1337009, logic.has_timestop),
        LocationData('Upper lake desolation', 'Lake Desolation (Upper): Oxygen recovery room', 1337010),
        LocationData('Upper lake desolation', 'Lake Desolation (Upper): Secret room', 1337011, logic.can_break_walls),
        LocationData('Upper lake desolation', 'Lake Desolation (Upper): Double jump cave platform', 1337012, logic.has_doublejump),
        LocationData('Upper lake desolation', 'Lake Desolation (Upper): Double jump cave floor', 1337013),
        LocationData('Upper lake desolation', 'Lake Desolation (Upper): Sparrow chest', 1337014),
        LocationData('Upper lake desolation', 'Lake Desolation (Upper): Crash site pedestal', 1337015),
        LocationData('Upper lake desolation', 'Lake Desolation (Upper): Crash site chest 1', 1337016, lambda state: state.has('Killed Maw', player)),
        LocationData('Upper lake desolation', 'Lake Desolation (Upper): Crash site chest 2', 1337017, lambda state: state.has('Killed Maw', player)),
        LocationData('Eastern lake desolation', 'Lake Desolation: Kitty Boss', 1337018),
        LocationData('Library', 'Library: Basement', 1337019),
        LocationData('Library', 'Library: Warp gate', 1337020),
        LocationData('Library', 'Library: Librarian', 1337021),
        LocationData('Library', 'Library: Reading nook chest', 1337022),
        LocationData('Library', 'Library: Storage room chest 1', 1337023, logic.has_keycard_D),
        LocationData('Library', 'Library: Storage room chest 2', 1337024, logic.has_keycard_D),
        LocationData('Library', 'Library: Storage room chest 3', 1337025, logic.has_keycard_D),
        LocationData('Library top', 'Library: Backer room chest 5', 1337026),
        LocationData('Library top', 'Library: Backer room chest 4', 1337027),
        LocationData('Library top', 'Library: Backer room chest 3', 1337028),
        LocationData('Library top', 'Library: Backer room chest 2', 1337029),
        LocationData('Library top', 'Library: Backer room chest 1', 1337030),
        LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Elevator Key not required', 1337031),
        LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Ye olde Timespinner', 1337032),
        LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Bottom floor', 1337033, logic.has_keycard_C),
        LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Air vents secret', 1337034, logic.can_break_walls),
        LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Elevator chest', 1337035, lambda state: state.has('Elevator Keycard', player)),
        LocationData('Varndagroth tower right (upper)', 'Varndagroth Towers: Bridge', 1337036),
        LocationData('Varndagroth tower right (elevator)', 'Varndagroth Towers (Right): Elevator chest', 1337037),
        LocationData('Varndagroth tower right (upper)', 'Varndagroth Towers (Right): Elevator card chest', 1337038, lambda state: state.has('Elevator Keycard', player) or logic.has_doublejump(state)),
        LocationData('Varndagroth tower right (upper)', 'Varndagroth Towers (Right): Air vents right chest', 1337039, lambda state: state.has('Elevator Keycard', player) or logic.has_doublejump(state)),
        LocationData('Varndagroth tower right (upper)', 'Varndagroth Towers (Right): Air vents left chest', 1337040, lambda state: state.has('Elevator Keycard', player) or logic.has_doublejump(state)),
        LocationData('Varndagroth tower right (lower)', 'Varndagroth Towers (Right): Bottom floor', 1337041),
        LocationData('Varndagroth tower right (elevator)', 'Varndagroth Towers (Right): Varndagroth', 1337042, logic.has_keycard_C),
        LocationData('Varndagroth tower right (elevator)', 'Varndagroth Towers (Right): Spider Hell', 1337043, logic.has_keycard_A),
        LocationData('Skeleton Shaft', 'Sealed Caves (Xarion): Skeleton', 1337044),
        LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Shroom jump room', 1337045, logic.has_timestop),
        LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Double shroom room', 1337046),
        LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Jacksquat room', 1337047, logic.has_forwarddash_doublejump),
        LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Below Jacksquat room', 1337048),
        LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Secret room', 1337049, logic.can_break_walls),
        LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Bottom left room', 1337050),
        LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Last chance before Xarion', 1337051, logic.has_doublejump),
        LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Xarion', 1337052, lambda state: not flooded.flood_xarion or state.has('Water Mask', player)),
        LocationData('Sealed Caves (Sirens)', 'Sealed Caves (Sirens): Water hook', 1337053, lambda state: state.has('Water Mask', player)),
        LocationData('Sealed Caves (Sirens)', 'Sealed Caves (Sirens): Siren room underwater right', 1337054, lambda state: state.has('Water Mask', player)),
        LocationData('Sealed Caves (Sirens)', 'Sealed Caves (Sirens): Siren room underwater left', 1337055, lambda state: state.has('Water Mask', player)),
        LocationData('Sealed Caves (Sirens)', 'Sealed Caves (Sirens): Cave after sirens chest 1', 1337056),
        LocationData('Sealed Caves (Sirens)', 'Sealed Caves (Sirens): Cave after sirens chest 2', 1337057),
        LocationData('Military Fortress', 'Military Fortress: Bomber chest', 1337058, lambda state: state.has('Timespinner Wheel', player) and logic.has_doublejump_of_npc(state)),
        LocationData('Military Fortress', 'Military Fortress: Close combat room', 1337059),
        LocationData('Military Fortress (hangar)', 'Military Fortress: Soldiers bridge', 1337060),
        LocationData('Military Fortress (hangar)', 'Military Fortress: Giantess room', 1337061),
        LocationData('Military Fortress (hangar)', 'Military Fortress: Giantess bridge', 1337062),
        LocationData('Military Fortress (hangar)', 'Military Fortress: B door chest 2', 1337063, lambda state: logic.has_keycard_B(state) and (state.has('Water Mask', player) if flooded.flood_lab else logic.has_doublejump(state))),
        LocationData('Military Fortress (hangar)', 'Military Fortress: B door chest 1', 1337064, lambda state: logic.has_keycard_B(state) and (state.has('Water Mask', player) if flooded.flood_lab else logic.has_doublejump(state))),
        LocationData('Military Fortress (hangar)', 'Military Fortress: Pedestal', 1337065, lambda state: state.has('Water Mask', player) if flooded.flood_lab else (logic.has_doublejump_of_npc(state) or logic.has_forwarddash_doublejump(state))),
        LocationData('Main Lab', 'Lab: Coffee break', 1337066),
        LocationData('Main Lab', 'Lab: Lower trash right', 1337067, logic.has_doublejump),
        LocationData('Main Lab', 'Lab: Lower trash left', 1337068, lambda state: logic.has_doublejump_of_npc(state) if options.lock_key_amadeus else logic.has_upwarddash(state) ),
        LocationData('Main Lab', 'Lab: Below lab entrance', 1337069, logic.has_doublejump),
        LocationData('Main Lab', 'Lab: Trash jump room', 1337070, lambda state: not options.lock_key_amadeus or logic.has_doublejump_of_npc(state) ),
        LocationData('Lab Research', 'Lab: Dynamo Works', 1337071, lambda state: not options.lock_key_amadeus or ( state.has('Lab Access Dynamo', player) and logic.has_upwarddash(state) )),
        LocationData('The lab (upper)', 'Lab: Genza (Blob Mom)', 1337072),
        LocationData('Main Lab', 'Lab: Experiment #13', 1337073, lambda state: not options.lock_key_amadeus or state.has('Lab Access Experiment', player) ),
        LocationData('The lab (upper)', 'Lab: Download and chest room chest', 1337074),
        LocationData('The lab (upper)', 'Lab: Lab secret', 1337075, logic.can_break_walls),
        LocationData('Lab Research', 'Lab: Spider Hell', 1337076, lambda state: logic.has_keycard_A(state)),
        LocationData('Emperors tower (courtyard)', 'Emperor\'s Tower: Courtyard bottom chest', 1337077),
        LocationData('Emperors tower', 'Emperor\'s Tower: Courtyard floor secret', 1337078, lambda state: logic.has_upwarddash(state) and logic.can_break_walls(state)),
        LocationData('Emperors tower', 'Emperor\'s Tower: Courtyard upper chest', 1337079, lambda state: logic.has_upwarddash(state)),
        LocationData('Emperors tower', 'Emperor\'s Tower: Galactic sage room', 1337080),
        LocationData('Emperors tower', 'Emperor\'s Tower: Bottom right tower', 1337081),
        LocationData('Emperors tower', 'Emperor\'s Tower: Wayyyy up there', 1337082, logic.has_doublejump_of_npc),
        LocationData('Emperors tower', 'Emperor\'s Tower: Left tower balcony', 1337083),
        LocationData('Emperors tower', 'Emperor\'s Tower: Emperor\'s Chambers chest', 1337084),
        LocationData('Emperors tower', 'Emperor\'s Tower: Emperor\'s Chambers pedestal', 1337085),
        LocationData('Emperors tower', 'Killed Emperor', EventId),

        # Past item locations
        LocationData('Refugee Camp', 'Refugee Camp: Neliste\'s Bra', 1337086),
        LocationData('Refugee Camp', 'Refugee Camp: Storage chest 3', 1337087),
        LocationData('Refugee Camp', 'Refugee Camp: Storage chest 2', 1337088),
        LocationData('Refugee Camp', 'Refugee Camp: Storage chest 1', 1337089),
        LocationData('Forest', 'Forest: Refugee camp roof', 1337090),
        LocationData('Forest', 'Forest: Bat jump ledge', 1337091, lambda state: logic.has_doublejump_of_npc(state) or logic.has_forwarddash_doublejump(state) or logic.has_fastjump_on_npc(state)),
        LocationData('Forest', 'Forest: Green platform secret', 1337092, logic.can_break_walls),
        LocationData('Forest', 'Forest: Rats guarded chest', 1337093),
        LocationData('Forest', 'Forest: Waterfall chest 1', 1337094, lambda state: state.has('Water Mask', player)),
        LocationData('Forest', 'Forest: Waterfall chest 2', 1337095, lambda state: state.has('Water Mask', player)),
        LocationData('Forest', 'Forest: Batcave', 1337096),
        LocationData('Forest', 'Castle Ramparts: In the moat', 1337097, lambda state: not flooded.flood_moat or state.has('Water Mask', player)),
        LocationData('Left Side forest Caves', 'Forest: Before Serene single bat cave', 1337098),
        LocationData('Upper Lake Serene', 'Lake Serene (Upper): Rat nest', 1337099),
        LocationData('Upper Lake Serene', 'Lake Serene (Upper): Double jump cave platform', 1337100, logic.has_doublejump),
        LocationData('Upper Lake Serene', 'Lake Serene (Upper): Double jump cave floor', 1337101),
        LocationData('Upper Lake Serene', 'Lake Serene (Upper): Cave secret', 1337102, logic.can_break_walls),
        LocationData('Upper Lake Serene', 'Lake Serene: Before Big Bird', 1337175),
        LocationData('Upper Lake Serene', 'Lake Serene: Behind the vines', 1337103),
        LocationData('Upper Lake Serene', 'Lake Serene: Pyramid keys room', 1337104),
        LocationData('Upper Lake Serene', 'Lake Serene (Upper): Chicken ledge', 1337174),
        LocationData('Lower Lake Serene', 'Lake Serene (Lower): Deep dive', 1337105),
        LocationData('Left Side forest Caves', 'Lake Serene (Lower): Under the eels', 1337106, lambda state: state.has('Water Mask', player)),
        LocationData('Lower Lake Serene', 'Lake Serene (Lower): Water spikes room', 1337107),
        LocationData('Lower Lake Serene', 'Lake Serene (Lower): Underwater secret', 1337108, logic.can_break_walls),
        LocationData('Lower Lake Serene', 'Lake Serene (Lower): T chest', 1337109, lambda state: flooded.flood_lake_serene or logic.has_doublejump_of_npc(state)),
        LocationData('Left Side forest Caves', 'Lake Serene (Lower): Past the eels', 1337110, lambda state: state.has('Water Mask', player)),
        LocationData('Lower Lake Serene', 'Lake Serene (Lower): Underwater pedestal', 1337111, lambda state: flooded.flood_lake_serene or logic.has_doublejump(state)),
        LocationData('Caves of Banishment (upper)', 'Caves of Banishment (Maw): Shroom jump room', 1337112, lambda state: flooded.flood_maw or logic.has_doublejump(state)),
        LocationData('Caves of Banishment (upper)', 'Caves of Banishment (Maw): Secret room', 1337113, lambda state: logic.can_break_walls(state) and (not flooded.flood_maw or state.has('Water Mask', player))),
        LocationData('Caves of Banishment (upper)', 'Caves of Banishment (Maw): Bottom left room', 1337114, lambda state: not flooded.flood_maw or state.has('Water Mask', player)),
        LocationData('Caves of Banishment (upper)', 'Caves of Banishment (Maw): Single shroom room', 1337115),
        LocationData('Caves of Banishment (upper)', 'Caves of Banishment (Maw): Jackpot room chest 1', 1337116, lambda state: flooded.flood_maw or logic.has_forwarddash_doublejump(state)),
        LocationData('Caves of Banishment (upper)', 'Caves of Banishment (Maw): Jackpot room chest 2', 1337117, lambda state: flooded.flood_maw or logic.has_forwarddash_doublejump(state)),
        LocationData('Caves of Banishment (upper)', 'Caves of Banishment (Maw): Jackpot room chest 3', 1337118, lambda state: flooded.flood_maw or logic.has_forwarddash_doublejump(state)),
        LocationData('Caves of Banishment (upper)', 'Caves of Banishment (Maw): Jackpot room chest 4', 1337119, lambda state: flooded.flood_maw or logic.has_forwarddash_doublejump(state)),
        LocationData('Caves of Banishment (upper)', 'Caves of Banishment (Maw): Pedestal', 1337120, lambda state: not flooded.flood_maw or state.has('Water Mask', player)),
        LocationData('Caves of Banishment (Maw)', 'Caves of Banishment (Maw): Last chance before Maw', 1337121, lambda state: flooded.flood_maw or logic.has_doublejump(state)),
        LocationData('Caves of Banishment (Maw)', 'Caves of Banishment (Maw): Plasma Crystal', 1337173, lambda state: state.has_any({'Gas Mask', 'Talaria Attachment'}, player)),
        LocationData('Caves of Banishment (Maw)', 'Killed Maw', EventId, lambda state: state.has('Gas Mask', player)),
        LocationData('Caves of Banishment (Maw)', 'Caves of Banishment (Maw): Mineshaft', 1337122, lambda state: state.has_any({'Gas Mask', 'Talaria Attachment'}, player)),
        LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): Wyvern room', 1337123),
        LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): Siren room above water chest', 1337124),
        LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): Siren room underwater left chest', 1337125, lambda state: state.has('Water Mask', player)),
        LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): Siren room underwater right chest', 1337126, lambda state: state.has('Water Mask', player)),
        LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): Siren room underwater right ground', 1337172, lambda state: state.has('Water Mask', player)),
        LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): Water hook', 1337127, lambda state: state.has('Water Mask', player)),
        LocationData('Castle Ramparts', 'Castle Ramparts: Bomber chest', 1337128, logic.has_multiple_small_jumps_of_npc),
        LocationData('Castle Ramparts', 'Castle Ramparts: Freeze the engineer', 1337129, lambda state: state.has('Talaria Attachment', player) or logic.has_timestop(state)),
        LocationData('Castle Ramparts', 'Castle Ramparts: Giantess guarded room', 1337130),
        LocationData('Castle Ramparts', 'Castle Ramparts: Knight and archer guarded room', 1337131),
        LocationData('Castle Ramparts', 'Castle Ramparts: Pedestal', 1337132),
        LocationData('Castle Basement', 'Castle Basement: Secret pedestal', 1337133, logic.can_break_walls),
        LocationData('Castle Basement', 'Castle Basement: Clean the castle basement', 1337134),
        LocationData('Royal towers (lower)', 'Castle Keep: Yas queen room', 1337135, logic.has_pink),
        LocationData('Castle Basement', 'Castle Basement: Giantess guarded chest', 1337136),
        LocationData('Castle Basement', 'Castle Basement: Omelette chest', 1337137),
        LocationData('Castle Basement', 'Castle Basement: Just an egg', 1337138),
        LocationData('Castle Keep', 'Castle Keep: Under the twins', 1337139),
        LocationData('Castle Keep', 'Killed Twins', EventId, logic.has_timestop),
        LocationData('Castle Keep', 'Castle Keep: Advisor jump', 1337171, logic.has_timestop),
        LocationData('Castle Keep', 'Castle Keep: Twins', 1337140, logic.has_timestop),
        LocationData('Castle Keep', 'Castle Keep: Royal guard tiny room', 1337141, lambda state: logic.has_doublejump(state) or logic.has_fastjump_on_npc(state)),
        LocationData('Royal towers (lower)', 'Royal Towers: Floor secret', 1337142, lambda state: logic.has_doublejump(state) and logic.can_break_walls(state)),
        LocationData('Royal towers', 'Royal Towers: Pre-climb gap', 1337143),
        LocationData('Royal towers', 'Royal Towers: Long balcony', 1337144, lambda state: not flooded.flood_courtyard or state.has('Water Mask', player)),
        LocationData('Royal towers', 'Royal Towers: Past bottom struggle juggle', 1337145, lambda state: flooded.flood_courtyard or logic.has_doublejump_of_npc(state)),
        LocationData('Royal towers', 'Royal Towers: Bottom struggle juggle', 1337146, logic.has_doublejump_of_npc),
        LocationData('Royal towers (upper)', 'Royal Towers: Top struggle juggle', 1337147, logic.has_doublejump_of_npc),
        LocationData('Royal towers (upper)', 'Royal Towers: No struggle required', 1337148),
        LocationData('Royal towers', 'Royal Towers: Right tower freebie', 1337149),
        LocationData('Royal towers (upper)', 'Royal Towers: Left tower small balcony', 1337150),
        LocationData('Royal towers (upper)', 'Royal Towers: Left tower royal guard', 1337151),
        LocationData('Royal towers (upper)', 'Royal Towers: Before Aelana', 1337152),
        LocationData('Royal towers (upper)', 'Killed Aelana', EventId),
        LocationData('Royal towers (upper)', 'Royal Towers: Aelana\'s attic', 1337153, logic.has_upwarddash),
        LocationData('Royal towers (upper)', 'Royal Towers: Aelana\'s chest', 1337154),
        LocationData('Royal towers (upper)', 'Royal Towers: Aelana\'s pedestal', 1337155),

        # Ancient pyramid locations
        LocationData('Ancient Pyramid (entrance)', 'Ancient Pyramid: Why not it\'s right there', 1337246),
        LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Conviction guarded room', 1337247),
        LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Pit secret room', 1337248, lambda state: logic.can_break_walls(state) and (not flooded.flood_pyramid_shaft or state.has('Water Mask', player))),
        LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Regret chest', 1337249, lambda state: logic.can_break_walls(state) and (state.has('Water Mask', player) if flooded.flood_pyramid_shaft else logic.has_doublejump(state))),
        LocationData('Ancient Pyramid (right)', 'Ancient Pyramid: Nightmare Door chest', 1337236, lambda state: not flooded.flood_pyramid_back or state.has('Water Mask', player)),
        LocationData('Ancient Pyramid (right)', 'Killed Nightmare', EventId, lambda state: state.has_all({'Timespinner Wheel', 'Timespinner Spindle', 'Timespinner Gear 1', 'Timespinner Gear 2', 'Timespinner Gear 3'}, player) and (not flooded.flood_pyramid_back or state.has('Water Mask', player)))
    ]

    # 1337156 - 1337170 Downloads
    if not options or options.downloadable_items:
        location_table += ( 
            LocationData('Library', 'Library: Terminal 2 (Lachiem)', 1337156, lambda state: state.has('Tablet', player)),
            LocationData('Library', 'Library: Terminal 1 (Windaria)', 1337157, lambda state: state.has('Tablet', player)),
            # 1337158 Is lost in time
            LocationData('Library', 'Library: Terminal 3 (Emperor Nuvius)', 1337159, lambda state: state.has('Tablet', player)),
            LocationData('Library', 'Library: V terminal 1 (War of the Sisters)', 1337160, lambda state: state.has_all({'Tablet', 'Library Keycard V'}, player)),
            LocationData('Library', 'Library: V terminal 2 (Lake Desolation Map)', 1337161, lambda state: state.has_all({'Tablet', 'Library Keycard V'}, player)),
            LocationData('Library', 'Library: V terminal 3 (Vilete)', 1337162, lambda state: state.has_all({'Tablet', 'Library Keycard V'}, player)),
            LocationData('Library top', 'Library: Backer room terminal (Vandagray Metropolis Map)', 1337163, lambda state: state.has('Tablet', player)),
            LocationData('Varndagroth tower right (elevator)', 'Varndagroth Towers (Right): Medbay terminal (Bleakness Research)', 1337164, lambda state: state.has('Tablet', player) and logic.has_keycard_B(state)),
            LocationData('The lab (upper)', 'Lab: Download and chest room terminal (Experiment #13)', 1337165, lambda state: state.has('Tablet', player)),
            LocationData('Lab Research', 'Lab: Middle terminal (Amadeus Laboratory Map)', 1337166, lambda state: state.has('Tablet', player)),
            LocationData('Main Lab', 'Lab: Sentry platform terminal (Origins)', 1337167, lambda state: state.has('Tablet', player) and (not options.lock_key_amadeus or state.has('Lab Access Genza', player) or logic.can_teleport_to(state, "Time", "GateDadsTower"))),
            LocationData('Main Lab', 'Lab: Experiment 13 terminal (W.R.E.C Farewell)', 1337168, lambda state: state.has('Tablet', player)),
            LocationData('Main Lab', 'Lab: Left terminal (Biotechnology)', 1337169, lambda state: state.has('Tablet', player)),
            LocationData('Lab Research', 'Lab: Right terminal (Experiment #11)', 1337170, lambda state: state.has('Tablet', player))
        )

    # 1337176 - 1337176 Cantoran
    if not options or options.cantoran:
        location_table += (
            LocationData('Left Side forest Caves', 'Lake Serene: Cantoran', 1337176),
        )

    # 1337177 - 1337198 Lore Checks
    if not options or options.lore_checks:
        location_table += (
            LocationData('Lower lake desolation', 'Lake Desolation: Memory - Coyote Jump (Time Messenger)', 1337177),
            LocationData('Library', 'Library: Memory - Waterway (A Message)', 1337178),
            LocationData('Library top', 'Library: Memory - Library Gap (Lachiemi Sun)', 1337179),
            LocationData('Library top', 'Library: Memory - Mr. Hat Portrait (Moonlit Night)', 1337180),
            LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Memory - Elevator (Nomads)', 1337181, lambda state: state.has('Elevator Keycard', player)),
            LocationData('Varndagroth tower right (lower)', 'Varndagroth Towers: Memory - Siren Elevator (Childhood)', 1337182, logic.has_keycard_B),
            LocationData('Varndagroth tower right (lower)', 'Varndagroth Towers (Right): Memory - Bottom (Faron)', 1337183),
            LocationData('Military Fortress', 'Military Fortress: Memory - Bomber Climb (A Solution)', 1337184, lambda state: state.has('Timespinner Wheel', player) and logic.has_doublejump_of_npc(state)),
            LocationData('Main Lab', 'Lab: Memory - Genza\'s Secret Stash 1 (An Old Friend)', 1337185, logic.can_break_walls),
            LocationData('Main Lab', 'Lab: Memory - Genza\'s Secret Stash 2 (Twilight Dinner)', 1337186, logic.can_break_walls),
            LocationData('Emperors tower', 'Emperor\'s Tower: Memory - Way Up There (Final Circle)', 1337187, logic.has_doublejump_of_npc),
            LocationData('Forest', 'Forest: Journal - Rats (Lachiem Expedition)', 1337188),
            LocationData('Forest', 'Forest: Journal - Bat Jump Ledge (Peace Treaty)', 1337189, lambda state: logic.has_doublejump_of_npc(state) or logic.has_forwarddash_doublejump(state) or logic.has_fastjump_on_npc(state)),
            LocationData('Forest', 'Forest: Journal - Floating in Moat (Prime Edicts)', 1337190, lambda state: not flooded.flood_moat or state.has('Water Mask', player)),
            LocationData('Castle Ramparts', 'Castle Ramparts: Journal - Archer + Knight (Declaration of Independence)', 1337191),
            LocationData('Castle Keep', 'Castle Keep: Journal - Under the Twins (Letter of Reference)', 1337192),
            LocationData('Castle Basement', 'Castle Basement: Journal - Castle Loop Giantess (Political Advice)', 1337193),
            LocationData('Royal towers (lower)', 'Royal Towers: Journal - Aelana\'s Room (Diplomatic Missive)', 1337194, logic.has_pink),
            LocationData('Royal towers (upper)', 'Royal Towers: Journal - Top Struggle Juggle Base (War of the Sisters)', 1337195),
            LocationData('Royal towers (upper)', 'Royal Towers: Journal - Aelana Boss (Stained Letter)', 1337196),
            LocationData('Royal towers', 'Royal Towers: Journal - Near Bottom Struggle Juggle (Mission Findings)', 1337197, lambda state: flooded.flood_courtyard or logic.has_doublejump_of_npc(state)),
            LocationData('Caves of Banishment (Maw)', 'Caves of Banishment (Maw): Journal - Lower Left Caves (Naivety)', 1337198)
        )

    # 1337199 - 1337232 Reserved for future use

    # 1337233 - 1337235 Pyramid Start checks
    if not options or options.pyramid_start: 
        location_table += (
            LocationData('Ancient Pyramid (entrance)', 'Dark Forest: Training Dummy', 1337233),
            LocationData('Ancient Pyramid (entrance)', 'Temporal Gyre: Forest Entrance', 1337234, lambda state: logic.has_upwarddash(state) or logic.can_teleport_to(state, "Time", "GateGyre")),
            LocationData('Ancient Pyramid (entrance)', 'Ancient Pyramid: Rubble', 1337235),
        )

    # 1337236 Nightmare door

    # 1337237 - 1337245 GyreArchives
    if not options or options.gyre_archives:
        location_table += (
            LocationData('Ravenlord\'s Lair', 'Ravenlord: Post fight (pedestal)', 1337237),
            LocationData('Ifrit\'s Lair', 'Ifrit: Post fight (pedestal)', 1337238),
            LocationData('Temporal Gyre', 'Temporal Gyre: Chest 1', 1337239),
            LocationData('Temporal Gyre', 'Temporal Gyre: Chest 2', 1337240),
            LocationData('Temporal Gyre', 'Temporal Gyre: Chest 3', 1337241),
            LocationData('Ravenlord\'s Lair', 'Ravenlord: Pre fight', 1337242),
            LocationData('Ravenlord\'s Lair', 'Ravenlord: Post fight (chest)', 1337243),
            LocationData('Ifrit\'s Lair', 'Ifrit: Pre fight', 1337244),
            LocationData('Ifrit\'s Lair', 'Ifrit: Post fight (chest)', 1337245),
        )

    # 1337250 - 1337781 Torch checks
    if not options or options.pure_torcher:
        location_table += (
            LocationData('Lower lake desolation', 'Lake Desolation (Lower): Not So Secret Lantern', 1337250, lambda state: logic.can_break_walls(state) and logic.can_break_lanterns(state)),
            LocationData('Lower lake desolation', 'Lake Desolation (Lower): Middle Room Lantern 1', 1337256, logic.can_break_lanterns),
            LocationData('Lower lake desolation', 'Lake Desolation (Lower): Middle Room Lantern 2', 1337257, logic.can_break_lanterns),
            LocationData('Lower lake desolation', 'Lake Desolation (Lower): Timespinner Wheel Lantern 1', 1337258, logic.can_break_lanterns),
            LocationData('Lower lake desolation', 'Lake Desolation (Lower): Timespinner Wheel Lantern 2', 1337259, logic.can_break_lanterns),

            LocationData('Upper lake desolation', 'Lake Desolation (Upper): Upper Left Room Lantern 1', 1337251, logic.can_break_lanterns),
            LocationData('Upper lake desolation', 'Lake Desolation (Upper): Upper Left Room Lantern 2', 1337252, logic.can_break_lanterns),
            LocationData('Upper lake desolation', 'Lake Desolation (Upper): Oxygen Recovery Lantern', 1337253, logic.can_break_lanterns),
            LocationData('Upper lake desolation', 'Lake Desolation (Upper): Upper Right Room Lantern 1', 1337254, logic.can_break_lanterns),
            LocationData('Upper lake desolation', 'Lake Desolation (Upper): Double jump Cave Lantern', 1337255, logic.can_break_lanterns),

            LocationData('Eastern lake desolation', 'Lake Desolation: Metropolis Bridge Lantern 1', 1337773, logic.can_break_lanterns),
            LocationData('Eastern lake desolation', 'Lake Desolation: Metropolis Bridge Lantern 2', 1337774, logic.can_break_lanterns),
            LocationData('Eastern lake desolation', 'Lake Desolation: Metropolis Bridge Lantern 3', 1337775, logic.can_break_lanterns),
            LocationData('Eastern lake desolation', 'Lake Desolation: Metropolis Bridge Lantern 4', 1337776, logic.can_break_lanterns),
            LocationData('Eastern lake desolation', 'Lake Desolation: Metropolis Bridge Lantern 5', 1337777, logic.can_break_lanterns),
            LocationData('Eastern lake desolation', 'Lake Desolation: Metropolis Bridge Lantern 6', 1337778, logic.can_break_lanterns),
            LocationData('Eastern lake desolation', 'Lake Desolation: Metropolis Bridge Lantern 7', 1337779, logic.can_break_lanterns),

            LocationData('Library', 'Library: Sewer Entrance Lantern', 1337489, logic.can_break_lanterns),
            LocationData('Library', 'Library: Left Sewer Lantern 1', 1337422, logic.can_break_lanterns),
            LocationData('Library', 'Library: Left Sewer Lantern 2', 1337423, logic.can_break_lanterns),
            LocationData('Library', 'Library: Right Sewer Lantern 1', 1337424, logic.can_break_lanterns),
            LocationData('Library', 'Library: Right Sewer Lantern 2', 1337425, logic.can_break_lanterns),
            LocationData('Library', 'Library: Right Sewer Lantern 3', 1337426, logic.can_break_lanterns),
            LocationData('Library', 'Library: Right Sewer Lantern 4', 1337427, logic.can_break_lanterns),
            LocationData('Library', 'Library: Right Sewer Lantern 5', 1337428, logic.can_break_lanterns),
            LocationData('Library', 'Library: Right Sewer Lantern 6', 1337429, logic.can_break_lanterns),
            LocationData('Library', 'Library: Sewer Exit Lantern 1', 1337492, logic.can_break_lanterns),
            LocationData('Library', 'Library: Sewer Exit Lantern 2', 1337493, logic.can_break_lanterns),
            LocationData('Library', 'Library: Basement Lantern', 1337494, logic.can_break_lanterns),
            LocationData('Library', 'Library: Exit Lantern', 1337450, logic.can_break_lanterns),
            LocationData('Library', 'Library: Librarian Lantern 1', 1337463, logic.can_break_lanterns),
            LocationData('Library', 'Library: Librarian Lantern 2', 1337464, logic.can_break_lanterns),
            LocationData('Library', 'Library: Left Staircase Lantern 1', 1337465, logic.can_break_lanterns),
            LocationData('Library', 'Library: Left Staircase Lantern 2', 1337466, logic.can_break_lanterns),
            LocationData('Library', 'Library: Left Staircase Lantern 3', 1337467, logic.can_break_lanterns),
            LocationData('Library', 'Library: Left Staircase Lantern 4', 1337468, logic.can_break_lanterns),
            LocationData('Library', 'Library: Lantern 1', 1337471, logic.can_break_lanterns),
            LocationData('Library', 'Library: Lantern 2', 1337472, logic.can_break_lanterns),
            LocationData('Library', 'Library: Lantern 3', 1337473, logic.can_break_lanterns),
            LocationData('Library', 'Library: Lantern 4', 1337474, logic.can_break_lanterns),
            LocationData('Library', 'Library: Lantern 5', 1337475, logic.can_break_lanterns),
            LocationData('Library', 'Library: Lantern 6', 1337476, logic.can_break_lanterns),
            LocationData('Library', 'Library: Lantern 7', 1337477, logic.can_break_lanterns),
            LocationData('Library', 'Library: Storage Room Lantern 1', 1337478, lambda state: logic.has_keycard_D(state) and logic.can_break_lanterns(state)),
            LocationData('Library', 'Library: Storage Room Lantern 2', 1337479, lambda state: logic.has_keycard_D(state) and logic.can_break_lanterns(state)),
            LocationData('Library', 'Library: Waterway Lantern 1', 1337480, logic.can_break_lanterns),
            LocationData('Library', 'Library: Waterway Lantern 2', 1337481, logic.can_break_lanterns),
            LocationData('Library', 'Library: V Room Lantern 1', 1337490, lambda state: state.has('Library Keycard V', player) and logic.can_break_lanterns(state)),
            LocationData('Library', 'Library: V Room Lantern 2', 1337491, lambda state: state.has('Library Keycard V', player) and logic.can_break_lanterns(state)),

            LocationData('Library top', 'Library: Backer Room Lantern 1', 1337484, logic.can_break_lanterns),
            LocationData('Library top', 'Library: Backer Room Lantern 2', 1337485, logic.can_break_lanterns),
            LocationData('Library top', 'Library: Backer Room Lantern 3', 1337486, logic.can_break_lanterns),
            LocationData('Library top', 'Library: Backer Room Lantern 4', 1337487, logic.can_break_lanterns),
            LocationData('Library top', 'Library: Backer Room Lantern 5', 1337488, logic.can_break_lanterns),
            LocationData('Library top', 'Library: Backer Stairs Lantern 1', 1337459, logic.can_break_lanterns),
            LocationData('Library top', 'Library: Backer Stairs Lantern 2', 1337460, logic.can_break_lanterns),
            LocationData('Library top', 'Library: Backer Stairs Lantern 3', 1337461, logic.can_break_lanterns),
            LocationData('Library top', 'Library: Backer Stairs Lantern 4', 1337462, logic.can_break_lanterns),
            LocationData('Library top', 'Library: Mr. Hat Lantern 1', 1337482, logic.can_break_lanterns),
            LocationData('Library top', 'Library: Mr. Hat Lantern 2', 1337483, logic.can_break_lanterns),

            LocationData('Varndagroth tower left', 'Library: Moving Sidewalk Lantern 1', 1337430, logic.can_break_lanterns),
            LocationData('Varndagroth tower left', 'Library: Moving Sidewalk Lantern 2', 1337431, logic.can_break_lanterns),

            LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Entrance Lantern', 1337434, logic.can_break_lanterns),
            LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Bottom Floor Lantern', 1337451, lambda state: logic.has_keycard_C(state) and logic.can_break_lanterns(state)),
            LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Left Elevator Lantern 1', 1337452, lambda state: state.has('Elevator Keycard', player) and logic.can_break_lanterns(state)),
            LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Left Elevator Lantern 2', 1337453, logic.can_break_lanterns),
            LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Left Stairs Base Lantern 1', 1337469, logic.can_break_lanterns),
            LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Left Stairs Base Lantern 2', 1337470, logic.can_break_lanterns),
            LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Left Stairs Floor 2 Lantern 1', 1337432, logic.can_break_lanterns),
            LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Left Stairs Floor 2 Lantern 2', 1337433, logic.can_break_lanterns),
            LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Left Stairs Middle Lantern', 1337454, logic.can_break_lanterns),
            LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Left Stairs Ladder Lantern 1', 1337455, logic.can_break_lanterns),
            LocationData('Varndagroth tower left', 'Varndagroth Towers (Left): Left Stairs Ladder Lantern 2', 1337456, logic.can_break_lanterns),

            LocationData('Varndagroth tower right (upper)', 'Varndagroth Towers: Bridge Entrance Lantern 1', 1337457, logic.can_break_lanterns),
            LocationData('Varndagroth tower right (upper)', 'Varndagroth Towers: Bridge Entrance Lantern 2', 1337458, logic.can_break_lanterns),
            LocationData('Varndagroth tower right (upper)', 'Varndagroth Towers (Right): Bridge Exit Lantern 1', 1337437, logic.can_break_lanterns),
            LocationData('Varndagroth tower right (upper)', 'Varndagroth Towers (Right): Bridge Exit Lantern 2', 1337438, logic.can_break_lanterns),
            LocationData('Varndagroth tower right (upper)', 'Varndagroth Towers (Right): Above Vents Lantern 1', 1337448, lambda state: (state.has('Elevator Keycard', player) or logic.has_doublejump(state)) and logic.can_break_lanterns(state)),
            LocationData('Varndagroth tower right (upper)', 'Varndagroth Towers (Right): Above Vents Lantern 2', 1337449, lambda state: (state.has('Elevator Keycard', player) or logic.has_doublejump(state)) and logic.can_break_lanterns(state)),
            LocationData('Varndagroth tower right (upper)', 'Varndagroth Towers (Right): Vent Lantern 1', 1337445, lambda state: (state.has('Elevator Keycard', player) or logic.has_doublejump(state)) and logic.can_break_lanterns(state)),
            LocationData('Varndagroth tower right (upper)', 'Varndagroth Towers (Right): Vent Lantern 2', 1337446, lambda state: (state.has('Elevator Keycard', player) or logic.has_doublejump(state)) and logic.can_break_lanterns(state)),
            LocationData('Varndagroth tower right (upper)', 'Varndagroth Towers (Right): Vent Lantern 3', 1337447, lambda state: (state.has('Elevator Keycard', player) or logic.has_doublejump(state)) and logic.can_break_lanterns(state)),

            LocationData('Varndagroth tower right (elevator)', 'Varndagroth Towers (Right): Right Elevator Lantern 1', 1337439, logic.can_break_lanterns),
            LocationData('Varndagroth tower right (elevator)', 'Varndagroth Towers (Right): Right Elevator Lantern 2', 1337440, logic.can_break_lanterns),
            LocationData('Varndagroth tower right (elevator)', 'Varndagroth Towers (Right): Right Elevator Lantern 3', 1337441, logic.can_break_lanterns),
            LocationData('Varndagroth tower right (elevator)', 'Varndagroth Towers (Right): Right Elevator Lantern 4', 1337442, logic.can_break_lanterns),
            LocationData('Varndagroth tower right (elevator)', 'Varndagroth Towers (Right): Right Elevator Lantern 5', 1337443, logic.can_break_lanterns),
            LocationData('Varndagroth tower right (elevator)', 'Varndagroth Towers (Right): Right Elevator Lantern 6', 1337444, logic.can_break_lanterns),

            LocationData('Varndagroth tower right (lower)', 'Varndagroth Towers (Right): Right Stairs Lantern 1', 1337435, logic.can_break_lanterns),
            LocationData('Varndagroth tower right (lower)', 'Varndagroth Towers (Right): Right Stairs Lantern 2', 1337436, logic.can_break_lanterns),
            LocationData('Varndagroth tower right (lower)', 'Varndagroth Towers (Right): Base Lantern 1', 1337495, logic.can_break_lanterns),
            LocationData('Varndagroth tower right (lower)', 'Varndagroth Towers (Right): Base Lantern 2', 1337496, logic.can_break_lanterns),

            LocationData('Sealed Caves (Sirens)', 'Sealed Caves (Sirens): Middle Hall Lantern 1', 1337721, logic.can_break_lanterns),
            LocationData('Sealed Caves (Sirens)', 'Sealed Caves (Sirens): Middle Hall Lantern 2', 1337722, logic.can_break_lanterns),
            LocationData('Sealed Caves (Sirens)', 'Sealed Caves (Sirens): Middle Hall Lantern 3', 1337723, logic.can_break_lanterns),
            LocationData('Sealed Caves (Sirens)', 'Sealed Caves (Sirens): Middle Hall Lantern 4', 1337724, logic.can_break_lanterns),
            LocationData('Sealed Caves (Sirens)', 'Sealed Caves (Sirens): First Hall Lantern 1', 1337741, logic.can_break_lanterns),
            LocationData('Sealed Caves (Sirens)', 'Sealed Caves (Sirens): First Hall Lantern 2', 1337742, logic.can_break_lanterns),
            LocationData('Sealed Caves (Sirens)', 'Sealed Caves (Sirens): First Hall Lantern ', 1337743, logic.can_break_lanterns),
            LocationData('Sealed Caves (Sirens)', 'Sealed Caves (Sirens): Condemned Shaft Lantern 1', 1337744, logic.can_break_lanterns),
            LocationData('Sealed Caves (Sirens)', 'Sealed Caves (Sirens): Condemned Shaft Lantern 2', 1337745, logic.can_break_lanterns),

            LocationData('Skeleton Shaft', 'Sealed Caves (Xarion): Skeleton Lantern 1', 1337718, logic.can_break_lanterns),
            LocationData('Skeleton Shaft', 'Sealed Caves (Xarion): Skeleton Lantern 2', 1337719, logic.can_break_lanterns),
            LocationData('Skeleton Shaft', 'Sealed Caves (Xarion): Skeleton Lantern 3', 1337720, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): First Hall Lantern 1', 1337712, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): First Hall Lantern 2', 1337713, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): First Hall Lantern 3', 1337714, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): First Hall Lantern 4', 1337715, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): First Hall Lantern 5', 1337716, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): First Hall Lantern 6', 1337717, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Second Hall Lantern 1', 1337746, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Second Hall Lantern 2', 1337747, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Second Hall Lantern 3', 1337748, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Second Hall Lantern 4', 1337749, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Forked Shaft Lantern 1', 1337750, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Forked Shaft Lantern 2', 1337751, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Forked Shaft Lantern 3', 1337752, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Forked Shaft Lantern 4', 1337753, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Forked Shaft Lantern 5', 1337754, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Shroom Jump Lantern 1', 1337738, lambda state: logic.has_timestop(state) and logic.can_break_lanterns(state)),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Shroom Jump Lantern 2', 1337739, lambda state: logic.has_timestop(state) and logic.can_break_lanterns(state)),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Shroom Jump Lantern 3', 1337740, lambda state: logic.has_timestop(state) and logic.can_break_lanterns(state)),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Shroom Jump Lantern 4', 1337781, lambda state: logic.has_timestop(state) and logic.can_break_lanterns(state)),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Lower Fork Start Lantern 1', 1337761, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Lower Fork Start Lantern 2', 1337762, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Lower Fork Vertical Room Lantern 1', 1337769, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Lower Fork Vertical Room Lantern 2', 1337770, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Lower Fork Vertical Room Lantern 3', 1337771, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Lower Fork Vertical Room Lantern 4', 1337772, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Mini Jackpot Ledge Lantern', 1337733, lambda state: logic.has_forwarddash_doublejump(state) and logic.can_break_lanterns(state)),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Waterfall Lantern 1', 1337731, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Waterfall Lantern 2', 1337732, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Waterfall Lantern 3', 1337734, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Waterfall Lantern 4', 1337735, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Waterfall Lantern 5', 1337736, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Post-Fork Room Lantern 1', 1337763, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Post-Fork Room Lantern 2', 1337764, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Rejoined Hallway Lantern 1', 1337755, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Rejoined Hallway Lantern 2', 1337756, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Rejoined Hallway Lantern 3', 1337757, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Rejoined Hallway Lantern 4', 1337758, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Rejoined Hallway Lantern 5', 1337759, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Rejoined Hallway Lantern 6', 1337760, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Penultimate Hall Lantern 1', 1337725, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Penultimate Hall Lantern 2', 1337726, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Penultimate Hall Lantern 3', 1337727, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Penultimate Hall Lantern 4', 1337728, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Penultimate Hall Lantern 5', 1337729, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Penultimate Hall Lantern 6', 1337730, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Last Chance Room Lantern 1', 1337765, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Last Chance Room Lantern 2', 1337766, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Last Chance Room Lantern 3', 1337767, logic.can_break_lanterns),
            LocationData('Sealed Caves (Xarion)', 'Sealed Caves (Xarion): Last Chance Room Lantern 4', 1337768, lambda state: logic.has_doublejump(state) and logic.can_break_lanterns(state)),

            LocationData('Forest', 'Forest: Rats Lantern', 1337498, logic.can_break_lanterns),
            LocationData('Forest', 'Forest: Ramparts Bridge Lantern 1', 1337499, logic.can_break_lanterns),
            LocationData('Forest', 'Forest: Ramparts Bridge Lantern 2', 1337500, logic.can_break_lanterns),
            LocationData('Forest', 'Forest: Ramparts Bridge Lantern 3', 1337501, logic.can_break_lanterns),
            LocationData('Forest', 'Forest: Batcave Lantern', 1337502, logic.can_break_lanterns),
            LocationData('Forest', 'Forest: Lantern Before Broken Bridge', 1337503, logic.can_break_lanterns),
            LocationData('Left Side forest Caves', 'Forest: Lantern Past Signpost', 1337497, logic.can_break_lanterns),
            LocationData('Left Side forest Caves', 'Forest: Lantern After Broken Bridge', 1337504, logic.can_break_lanterns),
            LocationData('Left Side forest Caves', 'Forest: Left Caves Lantern', 1337505, logic.can_break_lanterns),

            LocationData('Castle Ramparts', 'Castle Ramparts: Giantess Lantern 1', 1337507, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Giantess Lantern 2', 1337508, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Archer + Knight Lantern 1', 1337509, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Archer + Knight Lantern 2', 1337510, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Pedestal Lantern', 1337513, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Big Boulder Room Lantern 1', 1337514, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Big Boulder Room Lantern 2', 1337515, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Left Rooftops Lantern 1', 1337516, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Left Rooftops Lantern 2', 1337517, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Left Rooftops Lantern 3', 1337518, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Middle Hammer Lantern 1', 1337519, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Middle Hammer Lantern 2', 1337520, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Middle Hammer Lantern 3', 1337521, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Exit Lantern 1', 1337522, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Exit Lantern 2', 1337523, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Exit Lantern 3', 1337524, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Exit Lantern 4', 1337525, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Pedestal Stairs Lantern 1', 1337526, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Pedestal Stairs Lantern 2', 1337527, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Pedestal Stairs Lantern 3', 1337528, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Right Rooftops Lantern 1', 1337529, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Right Rooftops Lantern 2', 1337530, logic.can_break_lanterns),
            LocationData('Castle Ramparts', 'Castle Ramparts: Right Rooftops Lantern 3', 1337531, logic.can_break_lanterns),

            LocationData('Castle Basement', 'Castle Basement: Entrance Lantern 1', 1337536, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: Entrance Lantern 2', 1337537, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: Entrance Lantern 3', 1337538, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: Exit Lantern 1', 1337539, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: Exit Lantern 2', 1337540, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: Exit Climb Lantern 1', 1337541, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: Exit Climb Lantern 2', 1337542, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: First Bird Hall Lantern 1', 1337543, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: First Bird Hall Lantern 2', 1337544, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: Single Bird Lantern 1', 1337567, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: Single Bird Lantern 2', 1337568, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: Second Bird Hall Lantern 1', 1337569, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: Second Bird Hall Lantern 2', 1337570, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: Center Shaft Lantern 1', 1337579, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: Center Shaft Lantern 2', 1337580, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: Giantess Lantern 1', 1337581, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: Giantess Lantern 2', 1337582, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: Foyer Low Left Lantern', 1337563, logic.can_break_lanterns),
            LocationData('Castle Basement', 'Castle Basement: Foyer Low Right Lantern', 1337560, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Foyer Mid Left Lantern', 1337562, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Foyer Mid Right Lantern', 1337561, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Foyer High Left Lantern', 1337564, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Foyer High Right Lantern', 1337565, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Beginning Lantern 1', 1337532, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Beginning Lantern 2', 1337533, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Beginning Lantern 3', 1337534, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Beginning Lantern 4', 1337535, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Far-Left Climb Double-Jump Lantern', 1337545, lambda state: logic.has_doublejump(state) and logic.can_break_lanterns(state)),
            LocationData('Castle Keep', 'Castle Keep: Far-Left Climb Lower Lantern 1', 1337546, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Far-Left Climb Lower Lantern 2', 1337547, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Under The Twins Hallway Lantern 1', 1337548, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Under The Twins Hallway Lantern 2', 1337549, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Under The Twins Hallway Lantern 3', 1337550, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Under The Twins Hallway Lantern 4', 1337551, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Right-Middle Hallway Lantern 1', 1337552, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Right-Middle Hallway Lantern 2', 1337553, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Right-Middle Hallway Lantern 3', 1337554, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Right-Middle Hallway Lantern 4', 1337555, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Tiny Royal Guard Room Lantern 1', 1337556, lambda state: logic.can_break_lanterns(state) and (logic.has_doublejump(state) or logic.has_fastjump_on_npc(state))),
            LocationData('Castle Keep', 'Castle Keep: Tiny Royal Guard Room Lantern 2', 1337557, lambda state: logic.can_break_lanterns(state) and (logic.has_doublejump(state) or logic.has_fastjump_on_npc(state))),
            LocationData('Castle Keep', 'Castle Keep: Royal Tower Entrance Lantern 1', 1337558, lambda state: logic.has_doublejump(state) and logic.can_break_lanterns(state)),
            LocationData('Castle Keep', 'Castle Keep: Royal Tower Entrance Lantern 2', 1337559, lambda state: logic.has_doublejump(state) and logic.can_break_lanterns(state)),
            LocationData('Castle Keep', 'Castle Keep: Below Royal Room Lantern', 1337566, lambda state: logic.has_doublejump(state) and logic.can_break_lanterns(state)),
            LocationData('Castle Keep', 'Castle Keep: Outside Aelana\'s Room Lantern 1', 1337571, lambda state: logic.has_doublejump(state) and logic.can_break_lanterns(state)),
            LocationData('Castle Keep', 'Castle Keep: Outside Aelana\'s Room Lantern 2', 1337572, lambda state: logic.has_doublejump(state) and logic.can_break_lanterns(state)),
            LocationData('Castle Keep', 'Castle Keep: Outside Aelana\'s Room Lantern 3', 1337573, lambda state: logic.has_doublejump(state) and logic.can_break_lanterns(state)),
            LocationData('Castle Keep', 'Castle Keep: Outside Aelana\'s Room Lantern 4', 1337574, lambda state: logic.has_doublejump(state) and logic.can_break_lanterns(state)),
            LocationData('Castle Keep', 'Castle Keep: Twins Door Lantern 1', 1337575, lambda state: logic.has_timestop(state) and logic.can_break_lanterns(state)),
            LocationData('Castle Keep', 'Castle Keep: Twins Door Lantern 2', 1337576, lambda state: logic.has_timestop(state) and logic.can_break_lanterns(state)),
            LocationData('Castle Keep', 'Castle Keep: Under The Twins Lantern 1', 1337577, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Under The Twins Lantern 2', 1337578, logic.can_break_lanterns),
            LocationData('Castle Keep', 'Castle Keep: Twins Approach Lantern 1', 1337583, lambda state: logic.has_timestop(state) and logic.can_break_lanterns(state)),
            LocationData('Castle Keep', 'Castle Keep: Twins Approach Lantern 2', 1337584, lambda state: logic.has_timestop(state) and logic.can_break_lanterns(state)),
            LocationData('Castle Keep', 'Castle Keep: Twins Approach Lantern 3', 1337585, lambda state: logic.has_timestop(state) and logic.can_break_lanterns(state)),
            LocationData('Castle Keep', 'Castle Keep: Twins Stairwell Lantern 1', 1337586, lambda state: logic.has_timestop(state) and logic.can_break_lanterns(state)),
            LocationData('Castle Keep', 'Castle Keep: Twins Stairwell Lantern 2', 1337587, lambda state: logic.has_timestop(state) and logic.can_break_lanterns(state)),

            LocationData('Royal towers', 'Royal Towers: Long Balcony Lantern 1', 1337588, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_courtyard or state.has('Water Mask', player))),
            LocationData('Royal towers', 'Royal Towers: Long Balcony Lantern 2', 1337589, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_courtyard or state.has('Water Mask', player))),
            LocationData('Royal towers', 'Royal Towers: Long Balcony Lantern 3', 1337590, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_courtyard or state.has('Water Mask', player))),
            LocationData('Royal towers', 'Royal Towers: Long Balcony Lantern 4', 1337591, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_courtyard or state.has('Water Mask', player))),
            LocationData('Royal towers', 'Royal Towers: Long Balcony Lantern 5', 1337592, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_courtyard or state.has('Water Mask', player))),
            LocationData('Royal towers', 'Royal Towers: Bottom Struggle Base Lantern 1', 1337593, lambda state: logic.can_break_lanterns(state) and (flooded.flood_courtyard or logic.has_doublejump_of_npc(state))),
            LocationData('Royal towers', 'Royal Towers: Bottom Struggle Base Lantern 2', 1337594, lambda state: logic.can_break_lanterns(state) and (flooded.flood_courtyard or logic.has_doublejump_of_npc(state))),
            LocationData('Royal towers', 'Royal Towers: Before Bottom Struggle Lantern 1', 1337605, lambda state: logic.can_break_lanterns(state) and (flooded.flood_courtyard or logic.has_doublejump_of_npc(state))),
            LocationData('Royal towers', 'Royal Towers: Before Bottom Struggle Lantern 2', 1337606, lambda state: logic.can_break_lanterns(state) and (flooded.flood_courtyard or logic.has_doublejump_of_npc(state))),
            LocationData('Royal towers', 'Royal Towers: Past Bottom Struggle Lantern 1', 1337607, lambda state: logic.can_break_lanterns(state) and (flooded.flood_courtyard or logic.has_doublejump_of_npc(state))),
            LocationData('Royal towers', 'Royal Towers: Past Bottom Struggle Lantern 2', 1337608, lambda state: logic.can_break_lanterns(state) and (flooded.flood_courtyard or logic.has_doublejump_of_npc(state))),
            LocationData('Royal towers (upper)', 'Royal Towers: Aelana\'s Attic Lantern 1', 1337780, lambda state: logic.has_upwarddash(state) and logic.can_break_lanterns(state)),
            LocationData('Royal towers (upper)', 'Royal Towers: Aelana\'s Attic Lantern 2', 1337511, lambda state: logic.has_upwarddash(state) and logic.can_break_lanterns(state)),
            LocationData('Royal towers (upper)', 'Royal Towers: Aelana\'s Attic Lantern 3', 1337512, lambda state: logic.has_upwarddash(state) and logic.can_break_lanterns(state)),
            LocationData('Royal towers (upper)', 'Royal Towers: Before Aelana Lantern 1', 1337595, logic.can_break_lanterns),
            LocationData('Royal towers (upper)', 'Royal Towers: Tower Base Entrance Lantern 1', 1337596, logic.can_break_lanterns),
            LocationData('Royal towers (upper)', 'Royal Towers: Tower Base Entrance Lantern 2', 1337597, logic.can_break_lanterns),
            LocationData('Royal towers (upper)', 'Royal Towers: Tower Base Entrance Lantern 3', 1337598, logic.can_break_lanterns),
            LocationData('Royal towers', 'Royal Towers: Lantern Above Time-Stop Demon', 1337599, logic.can_break_lanterns),
            LocationData('Royal towers (lower)', 'Royal Towers: Lantern Below Time-Stop Demon', 1337600, logic.can_break_lanterns),
            LocationData('Royal towers (upper)', 'Royal Towers: Left Tower Base Lantern 1', 1337601, logic.can_break_lanterns),
            LocationData('Royal towers (upper)', 'Royal Towers: Left Tower Base Lantern 2', 1337602, logic.can_break_lanterns),
            LocationData('Royal towers (upper)', 'Royal Towers: Left Royal Guard Lantern 1', 1337603, logic.can_break_lanterns),
            LocationData('Royal towers (upper)', 'Royal Towers: Left Royal Guard Lantern 2', 1337604, logic.can_break_lanterns),
            LocationData('Royal towers', 'Royal Towers: Pre-Climb Lantern 1', 1337609, logic.can_break_lanterns),
            LocationData('Royal towers', 'Royal Towers: Pre-Climb Lantern 2', 1337610, logic.can_break_lanterns),
            LocationData('Royal towers', 'Royal Towers: Bottom Struggle Lantern', 1337611, logic.can_break_lanterns),
            LocationData('Royal towers (upper)', 'Royal Towers: Final Climb Lantern 1', 1337612, logic.can_break_lanterns),
            LocationData('Royal towers (upper)', 'Royal Towers: Final Climb Lantern 2', 1337613, logic.can_break_lanterns),
            LocationData('Royal towers (upper)', 'Royal Towers: Right Tower Base Lantern 1', 1337614, logic.can_break_lanterns),
            LocationData('Royal towers (upper)', 'Royal Towers: Right Tower Base Lantern 2', 1337615, logic.can_break_lanterns),
            LocationData('Royal towers (upper)', 'Royal Towers: Right Tower Base Lantern 3', 1337616, logic.can_break_lanterns),

            LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): First Hall Lantern 1', 1337674, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): First Hall Lantern 2', 1337675, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): First Hall Lantern 3', 1337676, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): Second Hall Lantern 1', 1337700, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): Second Hall Lantern 2', 1337701, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): Second Hall Lantern 3', 1337702, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): Second Hall Lantern 4', 1337703, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): Middle Hall Lantern 1', 1337654, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): Middle Hall Lantern 2', 1337655, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): Middle Hall Lantern 3', 1337656, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Sirens)', 'Caves of Banishment (Sirens): Middle Hall Lantern 4', 1337657, logic.can_break_lanterns),

            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Not Dead Yet Lantern 1', 1337650, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Not Dead Yet Lantern 2', 1337651, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Not Dead Yet Lantern 3', 1337652, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Not Dead Yet Lantern 4', 1337653, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): First Hall Lantern 1', 1337643, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): First Hall Lantern 2', 1337644, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): First Hall Lantern 3', 1337645, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): First Hall Lantern 4', 1337646, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): First Hall Lantern 5', 1337647, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): First Hall Lantern 6', 1337648, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Second Hall Lantern 1', 1337682, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Second Hall Lantern 2', 1337683, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Second Hall Lantern 3', 1337684, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Second Hall Lantern 4', 1337685, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Shroom Jump Lantern 1', 1337670, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Shroom Jump Lantern 2', 1337671, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Shroom Jump Lantern 3', 1337672, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Shroom Jump Lantern 4', 1337673, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Forked Shaft Lantern 1', 1337686, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Forked Shaft Lantern 2', 1337687, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Forked Shaft Lantern 3', 1337688, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Forked Shaft Lantern 4', 1337689, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Forked Shaft Lantern 5', 1337690, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Lower Fork Vertical Room Lantern 1', 1337708, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Lower Fork Vertical Room Lantern 2', 1337709, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Lower Fork Vertical Room Lantern 3', 1337710, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Lower Fork Vertical Room Lantern 4', 1337711, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Penultimate Hall Lantern 1', 1337658, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Penultimate Hall Lantern 2', 1337659, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Penultimate Hall Lantern 3', 1337660, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Penultimate Hall Lantern 4', 1337661, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Penultimate Hall Lantern 5', 1337662, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Penultimate Hall Lantern 6', 1337663, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Jackpot Ledge Lantern', 1337666, lambda state: logic.has_forwarddash_doublejump(state) and logic.can_break_lanterns(state)),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Waterfall Lantern 1', 1337664, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Waterfall Lantern 2', 1337665, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Waterfall Lantern 3', 1337667, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Waterfall Lantern 4', 1337668, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Waterfall Lantern 5', 1337669, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Rejoined Hallway Lantern 1', 1337691, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Rejoined Hallway Lantern 2', 1337692, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Rejoined Hallway Lantern 3', 1337693, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Rejoined Hallway Lantern 4', 1337694, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Rejoined Hallway Lantern 5', 1337695, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Rejoined Hallway Lantern 6', 1337737, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Lower Fork Start Lantern 1', 1337696, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Lower Fork Start Lantern 2', 1337697, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Post-Fork Room Lantern 1', 1337698, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Post-Fork Room Lantern 2', 1337699, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Last Chance Lantern 1', 1337704, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Last Chance Lantern 2', 1337705, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Last Chance Lantern 3', 1337706, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Last Chance Lantern 4', 1337707, logic.can_break_lanterns),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Mineshaft Lantern 1', 1337677, lambda state: logic.can_break_lanterns(state) and state.has_any({'Gas Mask', 'Talaria Attachment'}, player)),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Mineshaft Lantern 2', 1337678, lambda state: logic.can_break_lanterns(state) and state.has_any({'Gas Mask', 'Talaria Attachment'}, player)),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Mineshaft Lantern 3', 1337679, lambda state: logic.can_break_lanterns(state) and state.has_any({'Gas Mask', 'Talaria Attachment'}, player)),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Mineshaft Lantern 4', 1337680, lambda state: logic.can_break_lanterns(state) and state.has_any({'Gas Mask', 'Talaria Attachment'}, player)),
            LocationData('Caves of Banishment (Flooded)', 'Caves of Banishment (Maw): Mineshaft Lantern 5', 1337681, lambda state: logic.can_break_lanterns(state) and state.has_any({'Gas Mask', 'Talaria Attachment'}, player)),

            LocationData('Upper Lake Serene', 'Lake Serene (Upper): Intro Room Lantern', 1337617, logic.can_break_lanterns),
            LocationData('Upper Lake Serene', 'Lake Serene (Upper): Middle Cave Lantern', 1337621, logic.can_break_lanterns),
            LocationData('Upper Lake Serene', 'Lake Serene (Upper): Uncrashed Site Lantern 1', 1337622, logic.can_break_lanterns),
            LocationData('Upper Lake Serene', 'Lake Serene (Upper): Uncrashed Site Lantern 2', 1337623, logic.can_break_lanterns),
            LocationData('Upper Lake Serene', 'Lake Serene (Upper): Uncrashed Site Lantern 3', 1337624, logic.can_break_lanterns),
            LocationData('Upper Lake Serene', 'Lake Serene (Upper): Uncrashed Site Lantern 4', 1337625, logic.can_break_lanterns),
            LocationData('Upper Lake Serene', 'Lake Serene (Upper): Past First Vine Lantern', 1337626, logic.can_break_lanterns),
            LocationData('Left Side forest Caves', 'Lake Serene (Upper): Past Cantoran Lantern', 1337634, logic.can_break_lanterns),
            LocationData('Left Side forest Caves', 'Lake Serene (Upper): Fork Dry Lantern', 1337631, logic.can_break_lanterns),
            LocationData('Left Side forest Caves', 'Lake Serene (Upper): Fork Wet Lantern 1', 1337632, lambda state: logic.can_break_lanterns(state) and state.has('Water Mask', player)),
            LocationData('Left Side forest Caves', 'Lake Serene (Upper): Fork Wet Lantern 2', 1337633, lambda state: logic.can_break_lanterns(state) and state.has('Water Mask', player)),
            LocationData('Left Side forest Caves', 'Lake Serene (Lower): Above The Eels Lantern 1', 1337638, logic.can_break_lanterns),
            LocationData('Left Side forest Caves', 'Lake Serene (Lower): Above The Eels Lantern 2', 1337639, logic.can_break_lanterns),
            LocationData('Left Side forest Caves', 'Lake Serene (Lower): Under The Eels Lantern 1', 1337640, lambda state: logic.can_break_lanterns(state) and state.has('Water Mask', player)),
            LocationData('Left Side forest Caves', 'Lake Serene (Lower): Under The Eels Lantern 2', 1337641, lambda state: logic.can_break_lanterns(state) and state.has('Water Mask', player)),
            LocationData('Left Side forest Caves', 'Lake Serene (Lower): Under The Eels Lantern 3', 1337642, lambda state: logic.can_break_lanterns(state) and state.has('Water Mask', player)),
            LocationData('Left Side forest Caves', 'Lake Serene (Lower): Under The Eels Lantern 4', 1337649, lambda state: logic.can_break_lanterns(state) and state.has('Water Mask', player)),
            LocationData('Left Side forest Caves', 'Lake Serene (Lower): Past the Eels Lantern', 1337630, lambda state: logic.can_break_lanterns(state) and state.has('Water Mask', player)),
            LocationData('Lower Lake Serene', 'Lake Serene (Lower): Underwater Secret Lantern 1', 1337618, lambda state: logic.can_break_lanterns(state) and logic.can_break_walls(state)),
            LocationData('Lower Lake Serene', 'Lake Serene (Lower): Underwater Secret Lantern 2', 1337619, lambda state: logic.can_break_lanterns(state) and logic.can_break_walls(state)),
            LocationData('Lower Lake Serene', 'Lake Serene (Lower): Underwater Secret Lantern 3', 1337620, lambda state: logic.can_break_lanterns(state) and logic.can_break_walls(state)),
            LocationData('Lower Lake Serene', 'Lake Serene (Lower): Water Spikes Lantern 1', 1337635, logic.can_break_lanterns),
            LocationData('Lower Lake Serene', 'Lake Serene (Lower): Water Spikes Lantern 2', 1337636, logic.can_break_lanterns),
            LocationData('Lower Lake Serene', 'Lake Serene (Lower): Water Spikes Lantern 3', 1337637, logic.can_break_lanterns),
            LocationData('Lower Lake Serene', 'Lake Serene (Lower): Underwater Lantern 1', 1337627, logic.can_break_lanterns),
            LocationData('Lower Lake Serene', 'Lake Serene (Lower): Underwater Lantern 2', 1337628, logic.can_break_lanterns),
            LocationData('Lower Lake Serene', 'Lake Serene (Lower): Underwater Lantern 3', 1337629, logic.can_break_lanterns),

            LocationData('Military Fortress', 'Military Fortress: Entrance Lantern 1', 1337260, logic.can_break_lanterns),
            LocationData('Military Fortress', 'Military Fortress: Entrance Lantern 2', 1337261, logic.can_break_lanterns),
            LocationData('Military Fortress', 'Military Fortress: Bombing Room Lower Lantern 1', 1337264, logic.can_break_lanterns),
            LocationData('Military Fortress', 'Military Fortress: Bombing Room Lower Lantern 2', 1337266, logic.can_break_lanterns),
            LocationData('Military Fortress (hangar)', 'Military Fortress: Bombing Room Upper Lantern 1', 1337263, logic.can_break_lanterns),
            LocationData('Military Fortress (hangar)', 'Military Fortress: Bombing Room Upper Lantern 2', 1337265, logic.can_break_lanterns),
            LocationData('Military Fortress (hangar)', 'Military Fortress: Left Bridge Lantern 1', 1337267, logic.can_break_lanterns),
            LocationData('Military Fortress (hangar)', 'Military Fortress: Left Bridge Lantern 2', 1337268, logic.can_break_lanterns),
            LocationData('Military Fortress (hangar)', 'Military Fortress: Left Bridge Lantern 3', 1337269, logic.can_break_lanterns),
            LocationData('Military Fortress (hangar)', 'Military Fortress: Middle Room Lantern 1', 1337270, logic.can_break_lanterns),
            LocationData('Military Fortress (hangar)', 'Military Fortress: Middle Room Lantern 2', 1337271, logic.can_break_lanterns),
            LocationData('Military Fortress (hangar)', 'Military Fortress: Right Bridge Lantern 1', 1337274, logic.can_break_lanterns),
            LocationData('Military Fortress (hangar)', 'Military Fortress: Right Bridge Lantern 2', 1337275, logic.can_break_lanterns),
            LocationData('Military Fortress (hangar)', 'Military Fortress: Right Bridge Lantern 3', 1337276, logic.can_break_lanterns),
            LocationData('Military Fortress (hangar)', 'Military Fortress: Spike Room Lantern 1', 1337262, lambda state: logic.can_break_lanterns(state) and (state.has('Water Mask', player) if flooded.flood_lab else logic.has_doublejump(state))),
            LocationData('Military Fortress (hangar)', 'Military Fortress: Spike Room Lantern 2', 1337272, lambda state: logic.can_break_lanterns(state) and (state.has('Water Mask', player) if flooded.flood_lab else logic.has_doublejump(state))),
            LocationData('Military Fortress (hangar)', 'Military Fortress: Pedestal Lantern', 1337273, lambda state: logic.can_break_lanterns(state) and (state.has('Water Mask', player) if flooded.flood_lab else (logic.has_doublejump_of_npc(state) or logic.has_forwarddash_doublejump(state)))),

            LocationData('Lab Entrance', 'Lab: Intro Hallway Lantern 1', 1337294, logic.can_break_lanterns),
            LocationData('Lab Entrance', 'Lab: Intro Hallway Lantern 2', 1337782, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Coffee Lantern 1', 1337305, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Coffee Lantern 2', 1337306, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Coffee Lantern 3', 1337307, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Lower Trash Entrance Lantern 1', 1337277, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Lower Trash Entrance Lantern 2', 1337278, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Main Shaft Lantern 1', 1337297, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Main Shaft Lantern 2', 1337298, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Main Shaft Lantern 3', 1337299, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Main Shaft Lantern 4', 1337300, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Main Shaft Lantern 5', 1337301, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Main Shaft Lantern 6', 1337302, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Main Shaft Lantern 7', 1337303, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Main Shaft Lantern 8', 1337304, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Trash Stairs Lantern 1', 1337281, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Trash Stairs Lantern 2', 1337506, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Exp. 13 Terminal Lantern 1', 1337295, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Exp. 13 Terminal Lantern 2', 1337296, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Left Terminal Lantern 1', 1337308, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Left Terminal Lantern 2', 1337309, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Trash Jump Lantern 1', 1337282, lambda state: logic.can_break_lanterns(state) and (logic.has_doublejump_of_npc(state) if options.lock_key_amadeus else logic.has_upwarddash(state))),
            LocationData('Main Lab', 'Lab: Trash Jump Lantern 2', 1337283, lambda state: logic.can_break_lanterns(state) and (logic.has_doublejump_of_npc(state) if options.lock_key_amadeus else logic.has_upwarddash(state))),
            LocationData('Lab Research', 'Lab: Spider Hell Entrance Lantern 1', 1337290, logic.can_break_lanterns),
            LocationData('Lab Research', 'Lab: Spider Hell Entrance Lantern 2', 1337291, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Lower Trash Lantern 1', 1337292, lambda state: logic.can_break_lanterns(state) and (logic.has_doublejump_of_npc(state) if options.lock_key_amadeus else logic.has_upwarddash(state))),
            LocationData('Main Lab', 'Lab: Lower Trash Lantern 2', 1337293, lambda state: logic.can_break_lanterns(state) and (logic.has_doublejump_of_npc(state) if options.lock_key_amadeus else logic.has_upwarddash(state))),
            LocationData('The lab (upper)', 'Lab: File Cabinet Lantern 1', 1337310, logic.can_break_lanterns),
            LocationData('The lab (upper)', 'Lab: File Cabinet Lantern 2', 1337311, logic.can_break_lanterns),
            LocationData('The lab (upper)', 'Lab: File Cabinet Staircase Lantern 1', 1337286, logic.can_break_lanterns),
            LocationData('The lab (upper)', 'Lab: File Cabinet Staircase Lantern 2', 1337287, logic.can_break_lanterns),
            LocationData('The lab (upper)', 'Lab: File Cabinet Staircase Lantern 3', 1337288, logic.can_break_lanterns),
            LocationData('The lab (upper)', 'Lab: File Cabinet Staircase Lantern 4', 1337289, logic.can_break_lanterns),
            LocationData('The lab (upper)', 'Lab: Genza Door Lantern 1', 1337284, logic.can_break_lanterns),
            LocationData('The lab (upper)', 'Lab: Genza Door Lantern 2', 1337285, logic.can_break_lanterns),
            LocationData('The lab (upper)', 'Lab: Download and Chest Lantern 1', 1337312, logic.can_break_lanterns),
            LocationData('The lab (upper)', 'Lab: Download and Chest Lantern 2', 1337313, logic.can_break_lanterns),
            LocationData('Main Lab', 'Lab: Sentry Lantern 1', 1337279, lambda state: logic.can_break_lanterns(state) and (not options.lock_key_amadeus or state.has('Lab Access Genza', player) or logic.can_teleport_to(state, "Time", "GateDadsTower"))),
            LocationData('Main Lab', 'Lab: Sentry Lantern 2', 1337280, lambda state: logic.can_break_lanterns(state) and (not options.lock_key_amadeus or state.has('Lab Access Genza', player) or logic.can_teleport_to(state, "Time", "GateDadsTower"))),

            LocationData('Emperors tower (courtyard)', 'Emperor\'s Tower: Courtyard Lantern 1', 1337314, logic.can_break_lanterns),
            LocationData('Emperors tower (courtyard)', 'Emperor\'s Tower: Courtyard Lantern 2', 1337316, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Courtyard Staircase Lantern', 1337315, logic.can_break_lanterns),
            LocationData('Emperors tower (courtyard)', 'Emperor\'s Tower: Courtyard Bottom Lantern 1', 1337342, logic.can_break_lanterns),
            LocationData('Emperors tower (courtyard)', 'Emperor\'s Tower: Courtyard Bottom Lantern 2', 1337343, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Courtyard Floor Secret Lantern 1', 1337336, lambda state: logic.has_upwarddash(state) and logic.can_break_walls(state) and logic.can_break_lanterns(state)),
            LocationData('Emperors tower', 'Emperor\'s Tower: Courtyard Floor Secret Lantern 2', 1337337, lambda state: logic.has_upwarddash(state) and logic.can_break_walls(state) and logic.can_break_lanterns(state)),
            LocationData('Emperors tower', 'Emperor\'s Tower: Courtyard Giantess Lantern 1', 1337331, lambda state: logic.has_upwarddash(state) and logic.can_break_lanterns(state)),
            LocationData('Emperors tower', 'Emperor\'s Tower: Courtyard Giantess Lantern 2', 1337332, lambda state: logic.has_upwarddash(state) and logic.can_break_lanterns(state)),
            LocationData('Emperors tower', 'Emperor\'s Tower: Courtyard Upper Lantern', 1337333, lambda state: logic.has_upwarddash(state) and logic.can_break_lanterns(state)),
            LocationData('Emperors tower', 'Emperor\'s Tower: Lower Hallway Lantern 1', 1337317, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Lower Hallway Lantern 2', 1337318, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Lower Hallway Lantern 3', 1337319, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Lower Hallway Lantern 4', 1337320, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Lower Left Room Lantern 1', 1337321, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Lower Left Room Lantern 2', 1337322, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Outside Way Up There Lantern 1', 1337323, lambda state: logic.has_doublejump_of_npc(state) and logic.can_break_lanterns(state)),
            LocationData('Emperors tower', 'Emperor\'s Tower: Outside Way Up There Lantern 2', 1337324, lambda state: logic.has_doublejump_of_npc(state) and logic.can_break_lanterns(state)),
            LocationData('Emperors tower', 'Emperor\'s Tower: Way Up There Lantern 1', 1337325, lambda state: logic.has_doublejump_of_npc(state) and logic.can_break_lanterns(state)),
            LocationData('Emperors tower', 'Emperor\'s Tower: Way Up There Lantern 2', 1337326, lambda state: logic.has_doublejump_of_npc(state) and logic.can_break_lanterns(state)),
            LocationData('Emperors tower', 'Emperor\'s Tower: Lower Left Tower Lantern 1', 1337327, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Lower Left Tower Lantern 2', 1337328, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Galactic Sage Lantern 1', 1337329, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Galactic Sage Lantern 2', 1337330, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Lower Save Room Lantern 1', 1337334, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Lower Save Room Lantern 2', 1337335, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Climb Past Stairs Lantern 1', 1337338, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Climb Past Stairs Lantern 2', 1337339, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Climb Past Stairs Lantern 3', 1337340, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Climb Past Stairs Lantern 4', 1337341, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Middle Bridge Lantern 1', 1337344, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Middle Bridge Lantern 2', 1337345, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Middle Bridge Lantern 3', 1337346, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Middle Bridge Lantern 4', 1337347, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Middle Bridge Lantern 5', 1337348, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Upper Left Tower Lantern 1', 1337349, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Upper Left Tower Lantern 2', 1337350, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Lower Right Tower Lantern 1', 1337351, logic.can_break_lanterns),
            LocationData('Emperors tower', 'Emperor\'s Tower: Lower Right Tower Lantern 2', 1337352, logic.can_break_lanterns),
			
            LocationData('Ancient Pyramid (entrance)', 'Dark Forest: Pyramid Entrance Lantern 1', 1337369, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (entrance)', 'Dark Forest: Pyramid Entrance Lantern 2', 1337370, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (entrance)', 'Dark Forest: Pyramid Entrance Lantern 3', 1337371, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (entrance)', 'Dark Forest: Pyramid Entrance Lantern 4', 1337372, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (entrance)', 'Dark Forest: Pyramid Entrance Lantern 5', 1337373, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (entrance)', 'Dark Forest: Pyramid Entrance Lantern 6', 1337374, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (entrance)', 'Dark Forest: Training Dummy Lantern 1', 1337375, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (entrance)', 'Dark Forest: Training Dummy Lantern 2', 1337376, logic.can_break_lanterns),

            LocationData('Ancient Pyramid (entrance)', 'Ancient Pyramid: Entrance Lantern 1', 1337377, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (entrance)', 'Ancient Pyramid: Entrance Lantern 2', 1337378, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (entrance)', 'Ancient Pyramid: Rubble Lantern 1', 1337399, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (entrance)', 'Ancient Pyramid: Rubble Lantern 2', 1337400, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (entrance)', 'Ancient Pyramid: Rubble Lantern 3', 1337401, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (entrance)', 'Ancient Pyramid: Entrance Climb Lantern 1', 1337393, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (entrance)', 'Ancient Pyramid: Entrance Climb Lantern 2', 1337394, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (entrance)', 'Ancient Pyramid: Why Not It\'s Right There Lantern', 1337386, logic.can_break_lanterns),

            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: First Enemy Lantern 1', 1337387, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: First Enemy Lantern 2', 1337388, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Upper-Left Stairway Lantern 1', 1337381, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Upper-Left Stairway Lantern 2', 1337382, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Upper-Left Stairway Lantern 3', 1337383, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Regret Shaft Lantern 1', 1337402, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_pyramid_shaft or state.has('Water Mask', player))),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Regret Shaft Lantern 2', 1337403, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_pyramid_shaft or state.has('Water Mask', player))),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Regret Lantern 1', 1337389, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_pyramid_shaft or state.has('Water Mask', player))),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Regret Lantern 2', 1337390, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_pyramid_shaft or state.has('Water Mask', player))),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Regret Lantern 3', 1337391, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_pyramid_shaft or state.has('Water Mask', player))),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Regret Lantern 4', 1337392, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_pyramid_shaft or state.has('Water Mask', player))),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Left Hallway Lantern 1', 1337395, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Left Hallway Lantern 2', 1337396, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Left Hallway Lantern 3', 1337397, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Left Hallway Lantern 4', 1337398, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Pit Secret Lantern 1', 1337404, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_pyramid_shaft or state.has('Water Mask', player))),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Pit Secret Lantern 2', 1337405, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_pyramid_shaft or state.has('Water Mask', player))),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Pit Secret\'s Secret Lantern 1', 1337406, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_pyramid_shaft or state.has('Water Mask', player))),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Pit Secret\'s Secret Lantern 2', 1337407, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_pyramid_shaft or state.has('Water Mask', player))),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Outside Inner Warp Lantern 1', 1337408, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Outside Inner Warp Lantern 2', 1337409, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Nightmare Stairway Entrance Lantern', 1337410, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Conviction Lantern 1', 1337411, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Conviction Lantern 2', 1337412, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: A Long Fall Lantern 1', 1337415, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_pyramid_shaft or state.has('Water Mask', player))),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: A Long Fall Lantern 2', 1337416, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_pyramid_shaft or state.has('Water Mask', player))),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Last Chance Before Shaft Lantern 1', 1337417, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Last Chance Before Shaft Lantern 2', 1337418, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Pit Secret Wall Lantern', 1337419, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Post-Shaft Hallway Lantern 1', 1337420, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (left)', 'Ancient Pyramid: Post-Shaft Hallway Lantern 2', 1337421, logic.can_break_lanterns),

            LocationData('Ancient Pyramid (right)', 'Ancient Pyramid: Upper-Right Stairway Lantern 1', 1337384, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (right)', 'Ancient Pyramid: Upper-Right Stairway Lantern 2', 1337385, logic.can_break_lanterns),
            LocationData('Ancient Pyramid (right)', 'Ancient Pyramid: Nightmare Stairway Lantern 1', 1337379, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_pyramid_back or state.has('Water Mask', player))),
            LocationData('Ancient Pyramid (right)', 'Ancient Pyramid: Nightmare Stairway Lantern 2"', 1337380, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_pyramid_back or state.has('Water Mask', player))),
            LocationData('Ancient Pyramid (right)', 'Ancient Pyramid: Nightmare Door Lantern 1', 1337413, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_pyramid_back or state.has('Water Mask', player))),
            LocationData('Ancient Pyramid (right)', 'Ancient Pyramid: Nightmare Door Lantern 2', 1337414, lambda state: logic.can_break_lanterns(state) and (not flooded.flood_pyramid_back or state.has('Water Mask', player))),
        )
        if not options or options.gyre_archives:
            location_table += (
                LocationData('Temporal Gyre', 'Temporal Gyre: Room 1 Lantern 1', 1337353, logic.can_break_lanterns),
                LocationData('Temporal Gyre', 'Temporal Gyre: Room 1 Lantern 2', 1337354, logic.can_break_lanterns),
                LocationData('Temporal Gyre', 'Temporal Gyre: Room 2 Lantern 1', 1337355, logic.can_break_lanterns),
                LocationData('Temporal Gyre', 'Temporal Gyre: Room 2 Lantern 2', 1337356, logic.can_break_lanterns),
                LocationData('Temporal Gyre', 'Temporal Gyre: Room 3 Lantern 1', 1337357, logic.can_break_lanterns),
                LocationData('Temporal Gyre', 'Temporal Gyre: Room 3 Lantern 2', 1337358, logic.can_break_lanterns),
                LocationData('Temporal Gyre', 'Temporal Gyre: Room 4 Lantern 1', 1337359, logic.can_break_lanterns),
                LocationData('Temporal Gyre', 'Temporal Gyre: Room 4 Lantern 2', 1337360, logic.can_break_lanterns),
                LocationData('Temporal Gyre', 'Temporal Gyre: Room 5 Lantern 1', 1337361, logic.can_break_lanterns),
                LocationData('Temporal Gyre', 'Temporal Gyre: Room 5 Lantern 2', 1337362, logic.can_break_lanterns),
                LocationData('Temporal Gyre', 'Temporal Gyre: Room 6 Lantern 1', 1337363, logic.can_break_lanterns),
                LocationData('Temporal Gyre', 'Temporal Gyre: Room 6 Lantern 2', 1337364, logic.can_break_lanterns),
                LocationData('Temporal Gyre', 'Temporal Gyre: Room 7 Lantern 1', 1337365, logic.can_break_lanterns),
                LocationData('Temporal Gyre', 'Temporal Gyre: Room 7 Lantern 2', 1337366, logic.can_break_lanterns),
                LocationData('Temporal Gyre', 'Temporal Gyre: Room 8 Lantern 1', 1337367, logic.can_break_lanterns),
                LocationData('Temporal Gyre', 'Temporal Gyre: Room 8 Lantern 2', 1337368, logic.can_break_lanterns),
             )
 
    return location_table
