from typing import Dict, Set

from .data import LocationCategory, data


# Item Groups
ITEM_GROUPS: Dict[str, Set[str]] = {}

for item in data.items.values():
    for tag in item.tags:
        if tag not in ITEM_GROUPS:
            ITEM_GROUPS[tag] = set()
        ITEM_GROUPS[tag].add(item.label)

# Location Groups
_LOCATION_GROUP_MAPS = {
    "Abandoned Ship": {
        "MAP_ABANDONED_SHIP_CAPTAINS_OFFICE",
        "MAP_ABANDONED_SHIP_CORRIDORS_1F",
        "MAP_ABANDONED_SHIP_CORRIDORS_B1F",
        "MAP_ABANDONED_SHIP_DECK",
        "MAP_ABANDONED_SHIP_HIDDEN_FLOOR_CORRIDORS",
        "MAP_ABANDONED_SHIP_HIDDEN_FLOOR_ROOMS",
        "MAP_ABANDONED_SHIP_ROOMS2_1F",
        "MAP_ABANDONED_SHIP_ROOMS2_B1F",
        "MAP_ABANDONED_SHIP_ROOMS_1F",
        "MAP_ABANDONED_SHIP_ROOMS_B1F",
        "MAP_ABANDONED_SHIP_ROOM_B1F",
        "MAP_ABANDONED_SHIP_UNDERWATER1",
        "MAP_ABANDONED_SHIP_UNDERWATER2",
    },
    "Aqua Hideout": {
        "MAP_AQUA_HIDEOUT_1F",
        "MAP_AQUA_HIDEOUT_B1F",
        "MAP_AQUA_HIDEOUT_B2F",
    },
    "Battle Frontier": {
        "MAP_ARTISAN_CAVE_1F",
        "MAP_ARTISAN_CAVE_B1F",
        "MAP_BATTLE_FRONTIER_BATTLE_ARENA_BATTLE_ROOM",
        "MAP_BATTLE_FRONTIER_BATTLE_ARENA_CORRIDOR",
        "MAP_BATTLE_FRONTIER_BATTLE_ARENA_LOBBY",
        "MAP_BATTLE_FRONTIER_BATTLE_DOME_BATTLE_ROOM",
        "MAP_BATTLE_FRONTIER_BATTLE_DOME_CORRIDOR",
        "MAP_BATTLE_FRONTIER_BATTLE_DOME_LOBBY",
        "MAP_BATTLE_FRONTIER_BATTLE_DOME_PRE_BATTLE_ROOM",
        "MAP_BATTLE_FRONTIER_BATTLE_FACTORY_BATTLE_ROOM",
        "MAP_BATTLE_FRONTIER_BATTLE_FACTORY_LOBBY",
        "MAP_BATTLE_FRONTIER_BATTLE_FACTORY_PRE_BATTLE_ROOM",
        "MAP_BATTLE_FRONTIER_BATTLE_PALACE_BATTLE_ROOM",
        "MAP_BATTLE_FRONTIER_BATTLE_PALACE_CORRIDOR",
        "MAP_BATTLE_FRONTIER_BATTLE_PALACE_LOBBY",
        "MAP_BATTLE_FRONTIER_BATTLE_PIKE_CORRIDOR",
        "MAP_BATTLE_FRONTIER_BATTLE_PIKE_LOBBY",
        "MAP_BATTLE_FRONTIER_BATTLE_PIKE_ROOM_FINAL",
        "MAP_BATTLE_FRONTIER_BATTLE_PIKE_ROOM_NORMAL",
        "MAP_BATTLE_FRONTIER_BATTLE_PIKE_ROOM_WILD_MONS",
        "MAP_BATTLE_FRONTIER_BATTLE_PIKE_THREE_PATH_ROOM",
        "MAP_BATTLE_FRONTIER_BATTLE_PYRAMID_FLOOR",
        "MAP_BATTLE_FRONTIER_BATTLE_PYRAMID_LOBBY",
        "MAP_BATTLE_FRONTIER_BATTLE_PYRAMID_TOP",
        "MAP_BATTLE_FRONTIER_BATTLE_TOWER_BATTLE_ROOM",
        "MAP_BATTLE_FRONTIER_BATTLE_TOWER_CORRIDOR",
        "MAP_BATTLE_FRONTIER_BATTLE_TOWER_ELEVATOR",
        "MAP_BATTLE_FRONTIER_BATTLE_TOWER_LOBBY",
        "MAP_BATTLE_FRONTIER_BATTLE_TOWER_MULTI_BATTLE_ROOM",
        "MAP_BATTLE_FRONTIER_BATTLE_TOWER_MULTI_CORRIDOR",
        "MAP_BATTLE_FRONTIER_BATTLE_TOWER_MULTI_PARTNER_ROOM",
        "MAP_BATTLE_FRONTIER_EXCHANGE_SERVICE_CORNER",
        "MAP_BATTLE_FRONTIER_LOUNGE1",
        "MAP_BATTLE_FRONTIER_LOUNGE2",
        "MAP_BATTLE_FRONTIER_LOUNGE3",
        "MAP_BATTLE_FRONTIER_LOUNGE4",
        "MAP_BATTLE_FRONTIER_LOUNGE5",
        "MAP_BATTLE_FRONTIER_LOUNGE6",
        "MAP_BATTLE_FRONTIER_LOUNGE7",
        "MAP_BATTLE_FRONTIER_LOUNGE8",
        "MAP_BATTLE_FRONTIER_LOUNGE9",
        "MAP_BATTLE_FRONTIER_MART",
        "MAP_BATTLE_FRONTIER_OUTSIDE_EAST",
        "MAP_BATTLE_FRONTIER_OUTSIDE_WEST",
        "MAP_BATTLE_FRONTIER_POKEMON_CENTER_1F",
        "MAP_BATTLE_FRONTIER_POKEMON_CENTER_2F",
        "MAP_BATTLE_FRONTIER_RANKING_HALL",
        "MAP_BATTLE_FRONTIER_RECEPTION_GATE",
        "MAP_BATTLE_FRONTIER_SCOTTS_HOUSE",
        "MAP_BATTLE_PYRAMID_SQUARE01",
        "MAP_BATTLE_PYRAMID_SQUARE02",
        "MAP_BATTLE_PYRAMID_SQUARE03",
        "MAP_BATTLE_PYRAMID_SQUARE04",
        "MAP_BATTLE_PYRAMID_SQUARE05",
        "MAP_BATTLE_PYRAMID_SQUARE06",
        "MAP_BATTLE_PYRAMID_SQUARE07",
        "MAP_BATTLE_PYRAMID_SQUARE08",
        "MAP_BATTLE_PYRAMID_SQUARE09",
        "MAP_BATTLE_PYRAMID_SQUARE10",
        "MAP_BATTLE_PYRAMID_SQUARE11",
        "MAP_BATTLE_PYRAMID_SQUARE12",
        "MAP_BATTLE_PYRAMID_SQUARE13",
        "MAP_BATTLE_PYRAMID_SQUARE14",
        "MAP_BATTLE_PYRAMID_SQUARE15",
        "MAP_BATTLE_PYRAMID_SQUARE16",
    },
    "Birth Island": {
        "MAP_BIRTH_ISLAND_EXTERIOR",
        "MAP_BIRTH_ISLAND_HARBOR",
    },
    "Contest Hall": {
        "MAP_CONTEST_HALL",
        "MAP_CONTEST_HALL_BEAUTY",
        "MAP_CONTEST_HALL_COOL",
        "MAP_CONTEST_HALL_CUTE",
        "MAP_CONTEST_HALL_SMART",
        "MAP_CONTEST_HALL_TOUGH",
    },
    "Dewford Town": {
        "MAP_DEWFORD_TOWN",
        "MAP_DEWFORD_TOWN_GYM",
        "MAP_DEWFORD_TOWN_HALL",
        "MAP_DEWFORD_TOWN_HOUSE1",
        "MAP_DEWFORD_TOWN_HOUSE2",
        "MAP_DEWFORD_TOWN_POKEMON_CENTER_1F",
        "MAP_DEWFORD_TOWN_POKEMON_CENTER_2F",
    },
    "Ever Grande City": {
        "MAP_EVER_GRANDE_CITY",
        "MAP_EVER_GRANDE_CITY_CHAMPIONS_ROOM",
        "MAP_EVER_GRANDE_CITY_DRAKES_ROOM",
        "MAP_EVER_GRANDE_CITY_GLACIAS_ROOM",
        "MAP_EVER_GRANDE_CITY_HALL1",
        "MAP_EVER_GRANDE_CITY_HALL2",
        "MAP_EVER_GRANDE_CITY_HALL3",
        "MAP_EVER_GRANDE_CITY_HALL4",
        "MAP_EVER_GRANDE_CITY_HALL5",
        "MAP_EVER_GRANDE_CITY_HALL_OF_FAME",
        "MAP_EVER_GRANDE_CITY_PHOEBES_ROOM",
        "MAP_EVER_GRANDE_CITY_POKEMON_CENTER_1F",
        "MAP_EVER_GRANDE_CITY_POKEMON_CENTER_2F",
        "MAP_EVER_GRANDE_CITY_POKEMON_LEAGUE_1F",
        "MAP_EVER_GRANDE_CITY_POKEMON_LEAGUE_2F",
        "MAP_EVER_GRANDE_CITY_SIDNEYS_ROOM",
    },
    "Fallarbor Town": {
        "MAP_FALLARBOR_TOWN",
        "MAP_FALLARBOR_TOWN_BATTLE_TENT_BATTLE_ROOM",
        "MAP_FALLARBOR_TOWN_BATTLE_TENT_CORRIDOR",
        "MAP_FALLARBOR_TOWN_BATTLE_TENT_LOBBY",
        "MAP_FALLARBOR_TOWN_COZMOS_HOUSE",
        "MAP_FALLARBOR_TOWN_MART",
        "MAP_FALLARBOR_TOWN_MOVE_RELEARNERS_HOUSE",
        "MAP_FALLARBOR_TOWN_POKEMON_CENTER_1F",
        "MAP_FALLARBOR_TOWN_POKEMON_CENTER_2F",
    },
    "Faraway Island": {
        "MAP_FARAWAY_ISLAND_ENTRANCE",
        "MAP_FARAWAY_ISLAND_INTERIOR",
    },
    "Fiery Path": {"MAP_FIERY_PATH"},
    "Fortree City": {
        "MAP_FORTREE_CITY",
        "MAP_FORTREE_CITY_DECORATION_SHOP",
        "MAP_FORTREE_CITY_GYM",
        "MAP_FORTREE_CITY_HOUSE1",
        "MAP_FORTREE_CITY_HOUSE2",
        "MAP_FORTREE_CITY_HOUSE3",
        "MAP_FORTREE_CITY_HOUSE4",
        "MAP_FORTREE_CITY_HOUSE5",
        "MAP_FORTREE_CITY_MART",
        "MAP_FORTREE_CITY_POKEMON_CENTER_1F",
        "MAP_FORTREE_CITY_POKEMON_CENTER_2F",
    },
    "Granite Cave": {
        "MAP_GRANITE_CAVE_1F",
        "MAP_GRANITE_CAVE_B1F",
        "MAP_GRANITE_CAVE_B2F",
        "MAP_GRANITE_CAVE_STEVENS_ROOM",
    },
    "Jagged Pass": {"MAP_JAGGED_PASS"},
    "Lavaridge Town": {
        "MAP_LAVARIDGE_TOWN",
        "MAP_LAVARIDGE_TOWN_GYM_1F",
        "MAP_LAVARIDGE_TOWN_GYM_B1F",
        "MAP_LAVARIDGE_TOWN_HERB_SHOP",
        "MAP_LAVARIDGE_TOWN_HOUSE",
        "MAP_LAVARIDGE_TOWN_MART",
        "MAP_LAVARIDGE_TOWN_POKEMON_CENTER_1F",
        "MAP_LAVARIDGE_TOWN_POKEMON_CENTER_2F",
    },
    "Lilycove City": {
        "MAP_LILYCOVE_CITY",
        "MAP_LILYCOVE_CITY_CONTEST_HALL",
        "MAP_LILYCOVE_CITY_CONTEST_LOBBY",
        "MAP_LILYCOVE_CITY_COVE_LILY_MOTEL_1F",
        "MAP_LILYCOVE_CITY_COVE_LILY_MOTEL_2F",
        "MAP_LILYCOVE_CITY_DEPARTMENT_STORE_1F",
        "MAP_LILYCOVE_CITY_DEPARTMENT_STORE_2F",
        "MAP_LILYCOVE_CITY_DEPARTMENT_STORE_3F",
        "MAP_LILYCOVE_CITY_DEPARTMENT_STORE_4F",
        "MAP_LILYCOVE_CITY_DEPARTMENT_STORE_5F",
        "MAP_LILYCOVE_CITY_DEPARTMENT_STORE_ELEVATOR",
        "MAP_LILYCOVE_CITY_DEPARTMENT_STORE_ROOFTOP",
        "MAP_LILYCOVE_CITY_HARBOR",
        "MAP_LILYCOVE_CITY_HOUSE1",
        "MAP_LILYCOVE_CITY_HOUSE2",
        "MAP_LILYCOVE_CITY_HOUSE3",
        "MAP_LILYCOVE_CITY_HOUSE4",
        "MAP_LILYCOVE_CITY_LILYCOVE_MUSEUM_1F",
        "MAP_LILYCOVE_CITY_LILYCOVE_MUSEUM_2F",
        "MAP_LILYCOVE_CITY_MOVE_DELETERS_HOUSE",
        "MAP_LILYCOVE_CITY_POKEMON_CENTER_1F",
        "MAP_LILYCOVE_CITY_POKEMON_CENTER_2F",
        "MAP_LILYCOVE_CITY_POKEMON_TRAINER_FAN_CLUB",
    },
    "Littleroot Town": {
        "MAP_INSIDE_OF_TRUCK",
        "MAP_LITTLEROOT_TOWN",
        "MAP_LITTLEROOT_TOWN_BRENDANS_HOUSE_1F",
        "MAP_LITTLEROOT_TOWN_BRENDANS_HOUSE_2F",
        "MAP_LITTLEROOT_TOWN_MAYS_HOUSE_1F",
        "MAP_LITTLEROOT_TOWN_MAYS_HOUSE_2F",
        "MAP_LITTLEROOT_TOWN_PROFESSOR_BIRCHS_LAB",
    },
    "Magma Hideout": {
        "MAP_MAGMA_HIDEOUT_1F",
        "MAP_MAGMA_HIDEOUT_2F_1R",
        "MAP_MAGMA_HIDEOUT_2F_2R",
        "MAP_MAGMA_HIDEOUT_2F_3R",
        "MAP_MAGMA_HIDEOUT_3F_1R",
        "MAP_MAGMA_HIDEOUT_3F_2R",
        "MAP_MAGMA_HIDEOUT_3F_3R",
        "MAP_MAGMA_HIDEOUT_4F",
    },
    "Marine Cave": {
        "MAP_MARINE_CAVE_END",
        "MAP_MARINE_CAVE_ENTRANCE",
        "MAP_UNDERWATER_MARINE_CAVE",
    },
    "Mauville City": {
        "MAP_MAUVILLE_CITY",
        "MAP_MAUVILLE_CITY_BIKE_SHOP",
        "MAP_MAUVILLE_CITY_GAME_CORNER",
        "MAP_MAUVILLE_CITY_GYM",
        "MAP_MAUVILLE_CITY_HOUSE1",
        "MAP_MAUVILLE_CITY_HOUSE2",
        "MAP_MAUVILLE_CITY_MART",
        "MAP_MAUVILLE_CITY_POKEMON_CENTER_1F",
        "MAP_MAUVILLE_CITY_POKEMON_CENTER_2F",
    },
    "Meteor Falls": {
        "MAP_METEOR_FALLS_1F_1R",
        "MAP_METEOR_FALLS_1F_2R",
        "MAP_METEOR_FALLS_B1F_1R",
        "MAP_METEOR_FALLS_B1F_2R",
        "MAP_METEOR_FALLS_STEVENS_CAVE",
    },
    "Mirage Tower": {
        "MAP_MIRAGE_TOWER_1F",
        "MAP_MIRAGE_TOWER_2F",
        "MAP_MIRAGE_TOWER_3F",
        "MAP_MIRAGE_TOWER_4F",
    },
    "Mossdeep City": {
        "MAP_MOSSDEEP_CITY",
        "MAP_MOSSDEEP_CITY_GAME_CORNER_1F",
        "MAP_MOSSDEEP_CITY_GAME_CORNER_B1F",
        "MAP_MOSSDEEP_CITY_GYM",
        "MAP_MOSSDEEP_CITY_HOUSE1",
        "MAP_MOSSDEEP_CITY_HOUSE2",
        "MAP_MOSSDEEP_CITY_HOUSE3",
        "MAP_MOSSDEEP_CITY_HOUSE4",
        "MAP_MOSSDEEP_CITY_MART",
        "MAP_MOSSDEEP_CITY_POKEMON_CENTER_1F",
        "MAP_MOSSDEEP_CITY_POKEMON_CENTER_2F",
        "MAP_MOSSDEEP_CITY_SPACE_CENTER_1F",
        "MAP_MOSSDEEP_CITY_SPACE_CENTER_2F",
        "MAP_MOSSDEEP_CITY_STEVENS_HOUSE",
    },
    "Mt. Chimney": {
        "MAP_MT_CHIMNEY",
        "MAP_MT_CHIMNEY_CABLE_CAR_STATION",
    },
    "Mt. Pyre": {
        "MAP_MT_PYRE_1F",
        "MAP_MT_PYRE_2F",
        "MAP_MT_PYRE_3F",
        "MAP_MT_PYRE_4F",
        "MAP_MT_PYRE_5F",
        "MAP_MT_PYRE_6F",
        "MAP_MT_PYRE_EXTERIOR",
        "MAP_MT_PYRE_SUMMIT",
    },
    "Navel Rock": {
        "MAP_NAVEL_ROCK_B1F",
        "MAP_NAVEL_ROCK_BOTTOM",
        "MAP_NAVEL_ROCK_DOWN01",
        "MAP_NAVEL_ROCK_DOWN02",
        "MAP_NAVEL_ROCK_DOWN03",
        "MAP_NAVEL_ROCK_DOWN04",
        "MAP_NAVEL_ROCK_DOWN05",
        "MAP_NAVEL_ROCK_DOWN06",
        "MAP_NAVEL_ROCK_DOWN07",
        "MAP_NAVEL_ROCK_DOWN08",
        "MAP_NAVEL_ROCK_DOWN09",
        "MAP_NAVEL_ROCK_DOWN10",
        "MAP_NAVEL_ROCK_DOWN11",
        "MAP_NAVEL_ROCK_ENTRANCE",
        "MAP_NAVEL_ROCK_EXTERIOR",
        "MAP_NAVEL_ROCK_FORK",
        "MAP_NAVEL_ROCK_HARBOR",
        "MAP_NAVEL_ROCK_TOP",
        "MAP_NAVEL_ROCK_UP1",
        "MAP_NAVEL_ROCK_UP2",
        "MAP_NAVEL_ROCK_UP3",
        "MAP_NAVEL_ROCK_UP4",
    },
    "New Mauville": {
        "MAP_NEW_MAUVILLE_ENTRANCE",
        "MAP_NEW_MAUVILLE_INSIDE",
    },
    "Oldale Town": {
        "MAP_OLDALE_TOWN",
        "MAP_OLDALE_TOWN_HOUSE1",
        "MAP_OLDALE_TOWN_HOUSE2",
        "MAP_OLDALE_TOWN_MART",
        "MAP_OLDALE_TOWN_POKEMON_CENTER_1F",
        "MAP_OLDALE_TOWN_POKEMON_CENTER_2F",
    },
    "Pacifidlog Town": {
        "MAP_PACIFIDLOG_TOWN",
        "MAP_PACIFIDLOG_TOWN_HOUSE1",
        "MAP_PACIFIDLOG_TOWN_HOUSE2",
        "MAP_PACIFIDLOG_TOWN_HOUSE3",
        "MAP_PACIFIDLOG_TOWN_HOUSE4",
        "MAP_PACIFIDLOG_TOWN_HOUSE5",
        "MAP_PACIFIDLOG_TOWN_POKEMON_CENTER_1F",
        "MAP_PACIFIDLOG_TOWN_POKEMON_CENTER_2F",
    },
    "Petalburg City": {
        "MAP_PETALBURG_CITY",
        "MAP_PETALBURG_CITY_GYM",
        "MAP_PETALBURG_CITY_HOUSE1",
        "MAP_PETALBURG_CITY_HOUSE2",
        "MAP_PETALBURG_CITY_MART",
        "MAP_PETALBURG_CITY_POKEMON_CENTER_1F",
        "MAP_PETALBURG_CITY_POKEMON_CENTER_2F",
        "MAP_PETALBURG_CITY_WALLYS_HOUSE",
    },
    "Petalburg Woods": {"MAP_PETALBURG_WOODS"},
    "Route 101": {"MAP_ROUTE101"},
    "Route 102": {"MAP_ROUTE102"},
    "Route 103": {"MAP_ROUTE103"},
    "Route 104": {
        "MAP_ROUTE104",
        "MAP_ROUTE104_MR_BRINEYS_HOUSE",
        "MAP_ROUTE104_PRETTY_PETAL_FLOWER_SHOP",
    },
    "Route 105": {
        "MAP_ISLAND_CAVE",
        "MAP_ROUTE105",
        "MAP_UNDERWATER_ROUTE105",
    },
    "Route 106": {"MAP_ROUTE106"},
    "Route 107": {"MAP_ROUTE107"},
    "Route 108": {"MAP_ROUTE108"},
    "Route 109": {
        "MAP_ROUTE109",
        "MAP_ROUTE109_SEASHORE_HOUSE",
    },
    "Route 110": {
        "MAP_ROUTE110",
        "MAP_ROUTE110_SEASIDE_CYCLING_ROAD_NORTH_ENTRANCE",
        "MAP_ROUTE110_SEASIDE_CYCLING_ROAD_SOUTH_ENTRANCE",
    },
    "Trick House": {
        "MAP_ROUTE110_TRICK_HOUSE_CORRIDOR",
        "MAP_ROUTE110_TRICK_HOUSE_END",
        "MAP_ROUTE110_TRICK_HOUSE_ENTRANCE",
        "MAP_ROUTE110_TRICK_HOUSE_PUZZLE1",
        "MAP_ROUTE110_TRICK_HOUSE_PUZZLE2",
        "MAP_ROUTE110_TRICK_HOUSE_PUZZLE3",
        "MAP_ROUTE110_TRICK_HOUSE_PUZZLE4",
        "MAP_ROUTE110_TRICK_HOUSE_PUZZLE5",
        "MAP_ROUTE110_TRICK_HOUSE_PUZZLE6",
        "MAP_ROUTE110_TRICK_HOUSE_PUZZLE7",
        "MAP_ROUTE110_TRICK_HOUSE_PUZZLE8",
    },
    "Route 111": {
        "MAP_DESERT_RUINS",
        "MAP_ROUTE111",
        "MAP_ROUTE111_OLD_LADYS_REST_STOP",
        "MAP_ROUTE111_WINSTRATE_FAMILYS_HOUSE",
    },
    "Route 112": {
        "MAP_ROUTE112",
        "MAP_ROUTE112_CABLE_CAR_STATION",
    },
    "Route 113": {
        "MAP_ROUTE113",
        "MAP_ROUTE113_GLASS_WORKSHOP",
    },
    "Route 114": {
        "MAP_DESERT_UNDERPASS",
        "MAP_ROUTE114",
        "MAP_ROUTE114_FOSSIL_MANIACS_HOUSE",
        "MAP_ROUTE114_FOSSIL_MANIACS_TUNNEL",
        "MAP_ROUTE114_LANETTES_HOUSE",
    },
    "Route 115": {"MAP_ROUTE115"},
    "Route 116": {
        "MAP_ROUTE116",
        "MAP_ROUTE116_TUNNELERS_REST_HOUSE",
    },
    "Route 117": {
        "MAP_ROUTE117",
        "MAP_ROUTE117_POKEMON_DAY_CARE",
    },
    "Route 118": {"MAP_ROUTE118"},
    "Route 119": {
        "MAP_ROUTE119",
        "MAP_ROUTE119_HOUSE",
        "MAP_ROUTE119_WEATHER_INSTITUTE_1F",
        "MAP_ROUTE119_WEATHER_INSTITUTE_2F",
    },
    "Route 120": {
        "MAP_ANCIENT_TOMB",
        "MAP_ROUTE120",
        "MAP_SCORCHED_SLAB",
    },
    "Route 121": {
        "MAP_ROUTE121",
    },
    "Route 122": {"MAP_ROUTE122"},
    "Route 123": {
        "MAP_ROUTE123",
        "MAP_ROUTE123_BERRY_MASTERS_HOUSE",
    },
    "Route 124": {
        "MAP_ROUTE124",
        "MAP_ROUTE124_DIVING_TREASURE_HUNTERS_HOUSE",
        "MAP_UNDERWATER_ROUTE124",
    },
    "Route 125": {
        "MAP_ROUTE125",
        "MAP_UNDERWATER_ROUTE125",
    },
    "Route 126": {
        "MAP_ROUTE126",
        "MAP_UNDERWATER_ROUTE126",
    },
    "Route 127": {
        "MAP_ROUTE127",
        "MAP_UNDERWATER_ROUTE127",
    },
    "Route 128": {
        "MAP_ROUTE128",
        "MAP_UNDERWATER_ROUTE128",
    },
    "Route 129": {
        "MAP_ROUTE129",
        "MAP_UNDERWATER_ROUTE129",
    },
    "Route 130": {"MAP_ROUTE130"},
    "Route 131": {"MAP_ROUTE131"},
    "Route 132": {"MAP_ROUTE132"},
    "Route 133": {"MAP_ROUTE133"},
    "Route 134": {
        "MAP_ROUTE134",
        "MAP_UNDERWATER_ROUTE134",
        "MAP_SEALED_CHAMBER_INNER_ROOM",
        "MAP_SEALED_CHAMBER_OUTER_ROOM",
        "MAP_UNDERWATER_SEALED_CHAMBER",
    },
    "Rustboro City": {
        "MAP_RUSTBORO_CITY",
        "MAP_RUSTBORO_CITY_CUTTERS_HOUSE",
        "MAP_RUSTBORO_CITY_DEVON_CORP_1F",
        "MAP_RUSTBORO_CITY_DEVON_CORP_2F",
        "MAP_RUSTBORO_CITY_DEVON_CORP_3F",
        "MAP_RUSTBORO_CITY_FLAT1_1F",
        "MAP_RUSTBORO_CITY_FLAT1_2F",
        "MAP_RUSTBORO_CITY_FLAT2_1F",
        "MAP_RUSTBORO_CITY_FLAT2_2F",
        "MAP_RUSTBORO_CITY_FLAT2_3F",
        "MAP_RUSTBORO_CITY_GYM",
        "MAP_RUSTBORO_CITY_HOUSE1",
        "MAP_RUSTBORO_CITY_HOUSE2",
        "MAP_RUSTBORO_CITY_HOUSE3",
        "MAP_RUSTBORO_CITY_MART",
        "MAP_RUSTBORO_CITY_POKEMON_CENTER_1F",
        "MAP_RUSTBORO_CITY_POKEMON_CENTER_2F",
        "MAP_RUSTBORO_CITY_POKEMON_SCHOOL",
    },
    "Rusturf Tunnel": {"MAP_RUSTURF_TUNNEL"},
    "Safari Zone": {
        "MAP_ROUTE121_SAFARI_ZONE_ENTRANCE",
        "MAP_SAFARI_ZONE_NORTH",
        "MAP_SAFARI_ZONE_NORTHEAST",
        "MAP_SAFARI_ZONE_NORTHWEST",
        "MAP_SAFARI_ZONE_REST_HOUSE",
        "MAP_SAFARI_ZONE_SOUTH",
        "MAP_SAFARI_ZONE_SOUTHEAST",
        "MAP_SAFARI_ZONE_SOUTHWEST",
    },
    "Seafloor Cavern": {
        "MAP_SEAFLOOR_CAVERN_ENTRANCE",
        "MAP_SEAFLOOR_CAVERN_ROOM1",
        "MAP_SEAFLOOR_CAVERN_ROOM2",
        "MAP_SEAFLOOR_CAVERN_ROOM3",
        "MAP_SEAFLOOR_CAVERN_ROOM4",
        "MAP_SEAFLOOR_CAVERN_ROOM5",
        "MAP_SEAFLOOR_CAVERN_ROOM6",
        "MAP_SEAFLOOR_CAVERN_ROOM7",
        "MAP_SEAFLOOR_CAVERN_ROOM8",
        "MAP_SEAFLOOR_CAVERN_ROOM9",
        "MAP_UNDERWATER_SEAFLOOR_CAVERN",
    },
    "Shoal Cave": {
        "MAP_SHOAL_CAVE_HIGH_TIDE_ENTRANCE_ROOM",
        "MAP_SHOAL_CAVE_HIGH_TIDE_INNER_ROOM",
        "MAP_SHOAL_CAVE_LOW_TIDE_ENTRANCE_ROOM",
        "MAP_SHOAL_CAVE_LOW_TIDE_ICE_ROOM",
        "MAP_SHOAL_CAVE_LOW_TIDE_INNER_ROOM",
        "MAP_SHOAL_CAVE_LOW_TIDE_LOWER_ROOM",
        "MAP_SHOAL_CAVE_LOW_TIDE_STAIRS_ROOM",
    },
    "Sky Pillar": {
        "MAP_SKY_PILLAR_1F",
        "MAP_SKY_PILLAR_2F",
        "MAP_SKY_PILLAR_3F",
        "MAP_SKY_PILLAR_4F",
        "MAP_SKY_PILLAR_5F",
        "MAP_SKY_PILLAR_ENTRANCE",
        "MAP_SKY_PILLAR_OUTSIDE",
        "MAP_SKY_PILLAR_TOP",
    },
    "Slateport City": {
        "MAP_SLATEPORT_CITY",
        "MAP_SLATEPORT_CITY_BATTLE_TENT_BATTLE_ROOM",
        "MAP_SLATEPORT_CITY_BATTLE_TENT_CORRIDOR",
        "MAP_SLATEPORT_CITY_BATTLE_TENT_LOBBY",
        "MAP_SLATEPORT_CITY_HARBOR",
        "MAP_SLATEPORT_CITY_HOUSE",
        "MAP_SLATEPORT_CITY_MART",
        "MAP_SLATEPORT_CITY_NAME_RATERS_HOUSE",
        "MAP_SLATEPORT_CITY_OCEANIC_MUSEUM_1F",
        "MAP_SLATEPORT_CITY_OCEANIC_MUSEUM_2F",
        "MAP_SLATEPORT_CITY_POKEMON_CENTER_1F",
        "MAP_SLATEPORT_CITY_POKEMON_CENTER_2F",
        "MAP_SLATEPORT_CITY_POKEMON_FAN_CLUB",
        "MAP_SLATEPORT_CITY_STERNS_SHIPYARD_1F",
        "MAP_SLATEPORT_CITY_STERNS_SHIPYARD_2F",
    },
    "Sootopolis City": {
        "MAP_CAVE_OF_ORIGIN_1F",
        "MAP_CAVE_OF_ORIGIN_B1F",
        "MAP_CAVE_OF_ORIGIN_ENTRANCE",
        "MAP_SOOTOPOLIS_CITY",
        "MAP_SOOTOPOLIS_CITY_GYM_1F",
        "MAP_SOOTOPOLIS_CITY_GYM_B1F",
        "MAP_SOOTOPOLIS_CITY_HOUSE1",
        "MAP_SOOTOPOLIS_CITY_HOUSE2",
        "MAP_SOOTOPOLIS_CITY_HOUSE3",
        "MAP_SOOTOPOLIS_CITY_HOUSE4",
        "MAP_SOOTOPOLIS_CITY_HOUSE5",
        "MAP_SOOTOPOLIS_CITY_HOUSE6",
        "MAP_SOOTOPOLIS_CITY_HOUSE7",
        "MAP_SOOTOPOLIS_CITY_LOTAD_AND_SEEDOT_HOUSE",
        "MAP_SOOTOPOLIS_CITY_MART",
        "MAP_SOOTOPOLIS_CITY_MYSTERY_EVENTS_HOUSE_1F",
        "MAP_SOOTOPOLIS_CITY_MYSTERY_EVENTS_HOUSE_B1F",
        "MAP_SOOTOPOLIS_CITY_POKEMON_CENTER_1F",
        "MAP_SOOTOPOLIS_CITY_POKEMON_CENTER_2F",
        "MAP_UNDERWATER_SOOTOPOLIS_CITY",
    },
    "Southern Island": {
        "MAP_SOUTHERN_ISLAND_EXTERIOR",
        "MAP_SOUTHERN_ISLAND_INTERIOR",
    },
    "S.S. Tidal": {
        "MAP_SS_TIDAL_CORRIDOR",
        "MAP_SS_TIDAL_LOWER_DECK",
        "MAP_SS_TIDAL_ROOMS",
    },
    "Terra Cave": {
        "MAP_TERRA_CAVE_END",
        "MAP_TERRA_CAVE_ENTRANCE",
    },
    "Trainer Hill": {
        "MAP_TRAINER_HILL_2F",
        "MAP_TRAINER_HILL_3F",
        "MAP_TRAINER_HILL_4F",
        "MAP_TRAINER_HILL_ELEVATOR",
        "MAP_TRAINER_HILL_ENTRANCE",
        "MAP_TRAINER_HILL_ROOF",
    },
    "Verdanturf Town": {
        "MAP_VERDANTURF_TOWN",
        "MAP_VERDANTURF_TOWN_BATTLE_TENT_BATTLE_ROOM",
        "MAP_VERDANTURF_TOWN_BATTLE_TENT_CORRIDOR",
        "MAP_VERDANTURF_TOWN_BATTLE_TENT_LOBBY",
        "MAP_VERDANTURF_TOWN_FRIENDSHIP_RATERS_HOUSE",
        "MAP_VERDANTURF_TOWN_HOUSE",
        "MAP_VERDANTURF_TOWN_MART",
        "MAP_VERDANTURF_TOWN_POKEMON_CENTER_1F",
        "MAP_VERDANTURF_TOWN_POKEMON_CENTER_2F",
        "MAP_VERDANTURF_TOWN_WANDAS_HOUSE",
    },
    "Victory Road": {
        "MAP_VICTORY_ROAD_1F",
        "MAP_VICTORY_ROAD_B1F",
        "MAP_VICTORY_ROAD_B2F",
    },
}

_LOCATION_CATEGORY_TO_GROUP_NAME = {
    LocationCategory.BADGE: "Badges",
    LocationCategory.HM: "HMs",
    LocationCategory.KEY: "Key Items",
    LocationCategory.ROD: "Fishing Rods",
    LocationCategory.BIKE: "Bikes",
    LocationCategory.TICKET: "Tickets",
    LocationCategory.OVERWORLD_ITEM: "Overworld Items",
    LocationCategory.HIDDEN_ITEM: "Hidden Items",
    LocationCategory.GIFT: "NPC Gifts",
    LocationCategory.BERRY_TREE: "Berry Trees",
    LocationCategory.TRAINER: "Trainers",
    LocationCategory.POKEDEX: "Pokedex",
}

LOCATION_GROUPS: Dict[str, Set[str]] = {group_name: set() for group_name in _LOCATION_CATEGORY_TO_GROUP_NAME.values()}
for location in data.locations.values():
    # Category groups
    LOCATION_GROUPS[_LOCATION_CATEGORY_TO_GROUP_NAME[location.category]].add(location.label)

    # Tag groups
    for tag in location.tags:
        if tag not in LOCATION_GROUPS:
            LOCATION_GROUPS[tag] = set()
        LOCATION_GROUPS[tag].add(location.label)

    # Geographic groups
    if location.parent_region != "REGION_POKEDEX":
        map_name = data.regions[location.parent_region].parent_map.name
        for group, maps in _LOCATION_GROUP_MAPS.items():
            if map_name in maps:
                if group not in LOCATION_GROUPS:
                    LOCATION_GROUPS[group] = set()
                LOCATION_GROUPS[group].add(location.label)
                break

# Meta-groups
LOCATION_GROUPS["Cities"] = {
    *LOCATION_GROUPS.get("Littleroot Town", set()),
    *LOCATION_GROUPS.get("Oldale Town", set()),
    *LOCATION_GROUPS.get("Petalburg City", set()),
    *LOCATION_GROUPS.get("Rustboro City", set()),
    *LOCATION_GROUPS.get("Dewford Town", set()),
    *LOCATION_GROUPS.get("Slateport City", set()),
    *LOCATION_GROUPS.get("Mauville City", set()),
    *LOCATION_GROUPS.get("Verdanturf Town", set()),
    *LOCATION_GROUPS.get("Fallarbor Town", set()),
    *LOCATION_GROUPS.get("Lavaridge Town", set()),
    *LOCATION_GROUPS.get("Fortree City", set()),
    *LOCATION_GROUPS.get("Mossdeep City", set()),
    *LOCATION_GROUPS.get("Sootopolis City", set()),
    *LOCATION_GROUPS.get("Pacifidlog Town", set()),
    *LOCATION_GROUPS.get("Ever Grande City", set()),
}

LOCATION_GROUPS["Dungeons"] = {
    *LOCATION_GROUPS.get("Petalburg Woods", set()),
    *LOCATION_GROUPS.get("Rusturf Tunnel", set()),
    *LOCATION_GROUPS.get("Granite Cave", set()),
    *LOCATION_GROUPS.get("Fiery Path", set()),
    *LOCATION_GROUPS.get("Meteor Falls", set()),
    *LOCATION_GROUPS.get("Jagged Pass", set()),
    *LOCATION_GROUPS.get("Mt. Chimney", set()),
    *LOCATION_GROUPS.get("Abandoned Ship", set()),
    *LOCATION_GROUPS.get("New Mauville", set()),
    *LOCATION_GROUPS.get("Mt. Pyre", set()),
    *LOCATION_GROUPS.get("Seafloor Cavern", set()),
    *LOCATION_GROUPS.get("Sky Pillar", set()),
    *LOCATION_GROUPS.get("Victory Road", set()),
}

LOCATION_GROUPS["Routes"] = {
    *LOCATION_GROUPS.get("Route 101", set()),
    *LOCATION_GROUPS.get("Route 102", set()),
    *LOCATION_GROUPS.get("Route 103", set()),
    *LOCATION_GROUPS.get("Route 104", set()),
    *LOCATION_GROUPS.get("Route 105", set()),
    *LOCATION_GROUPS.get("Route 106", set()),
    *LOCATION_GROUPS.get("Route 107", set()),
    *LOCATION_GROUPS.get("Route 108", set()),
    *LOCATION_GROUPS.get("Route 109", set()),
    *LOCATION_GROUPS.get("Route 110", set()),
    *LOCATION_GROUPS.get("Route 111", set()),
    *LOCATION_GROUPS.get("Route 112", set()),
    *LOCATION_GROUPS.get("Route 113", set()),
    *LOCATION_GROUPS.get("Route 114", set()),
    *LOCATION_GROUPS.get("Route 115", set()),
    *LOCATION_GROUPS.get("Route 116", set()),
    *LOCATION_GROUPS.get("Route 117", set()),
    *LOCATION_GROUPS.get("Route 118", set()),
    *LOCATION_GROUPS.get("Route 119", set()),
    *LOCATION_GROUPS.get("Route 120", set()),
    *LOCATION_GROUPS.get("Route 121", set()),
    *LOCATION_GROUPS.get("Route 122", set()),
    *LOCATION_GROUPS.get("Route 123", set()),
    *LOCATION_GROUPS.get("Route 124", set()),
    *LOCATION_GROUPS.get("Route 125", set()),
    *LOCATION_GROUPS.get("Route 126", set()),
    *LOCATION_GROUPS.get("Route 127", set()),
    *LOCATION_GROUPS.get("Route 128", set()),
    *LOCATION_GROUPS.get("Route 129", set()),
    *LOCATION_GROUPS.get("Route 130", set()),
    *LOCATION_GROUPS.get("Route 131", set()),
    *LOCATION_GROUPS.get("Route 132", set()),
    *LOCATION_GROUPS.get("Route 133", set()),
    *LOCATION_GROUPS.get("Route 134", set()),
}
