from typing import TYPE_CHECKING, NamedTuple, Optional

# Entrance Groups #s

# 1 - General
# 2 - Dungeon Entrance
# 3 - Trial Gate
# 4 - Dungeon Exit
# 5 - 
# 6 - Dungeon Completion
# 7 - 
# 8 - 
# 9 - Statue -> Sky
# 10 - Sky -> Statue

class SSAPEntranceData(NamedTuple):
    """
    Class contains data for an AP exit in SS.

    exit_region: Region of the exit
    exit_name: Full name of the exit (Region - Exit)
    entrance_region: Region of the entrance
    entrance_name: Full name of the connecting exit (Region - Exit) or None if one-way
    direction: 1 if one-way, 2 if two-way, 0 if inaccessible
    can_shuffle: True or False
    group: The shuffle group, None for no shuffle
    """

    exit_region: str
    exit_name: str
    entrance_region: str
    entrance_name: Optional[str]
    direction: int
    can_shuffle: bool
    group: Optional[int]

AP_ENTRANCE_TABLE = [
    SSAPEntranceData(
        "Upper Skyloft",
        "Upper Skyloft - Knight Academy",
        "Upper Skyloft - Knight Academy",
        "Upper Skyloft - Knight Academy - Upper Skyloft",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Upper Skyloft",
        "Upper Skyloft - Zelda's Room",
        "Upper Skyloft - Zelda's Room",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Upper Skyloft",
        "Upper Skyloft - Stairs to Central Skyloft",
        "Central Skyloft",
        "Central Skyloft - Stairs to Upper Skyloft",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Upper Skyloft",
        "Upper Skyloft - Loftwing Platforms",
        "Sky",
        "Sky - Dive to Upper Skyloft",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Upper Skyloft - Knight Academy",
        "Upper Skyloft - Knight Academy - Upper Skyloft",
        "Upper Skyloft",
        "Upper Skyloft - Knight Academy",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Upper Skyloft - Zelda's Room",
        "Upper Skyloft - Zelda's Room - Knight Academy",
        "Upper Skyloft - Knight Academy",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Central Skyloft",
        "Central Skyloft - Stairs to Upper Skyloft",
        "Upper Skyloft",
        "Upper Skyloft - Stairs to Central Skyloft",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Central Skyloft",
        "Central Skyloft - Bridge to Skyloft Village",
        "Skyloft Village",
        "Skyloft Village - Bridge to Central Skyloft",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Central Skyloft",
        "Central Skyloft - Ring Beedle's Bell",
        "Beedle's Shop",
        "Beedle's Shop - Leave Shop",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Central Skyloft",
        "Central Skyloft - Loftwing Platforms",
        "Sky",
        "Sky - Dive to Central Skyloft",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Central Skyloft",
        "Central Skyloft - Dungeon Entrance on Skyloft",
        "Sky Keep - First Room",
        "Sky Keep - First Room - Dungeon Exit",
        2, True, 2,
    ),
    SSAPEntranceData(
        "Central Skyloft",
        "Central Skyloft - Trial Gate on Skyloft",
        "Skyloft Silent Realm",
        "Skyloft Silent Realm - Trial Gate",
        2, True, 3,
    ),
    SSAPEntranceData(
        "Skyloft Village",
        "Skyloft Village - Bridge to Central Skyloft",
        "Central Skyloft",
        "Central Skyloft - Bridge to Skyloft Village",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Skyloft Village",
        "Skyloft Village - Shed Door to Batreaux's House",
        "Batreaux's House",
        "Batreaux's House - Exit Shed Door",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Skyloft Village",
        "Skyloft Village - Loftwing Platforms",
        "Sky",
        "Sky - Dive to Skyloft Village",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Batreaux's House",
        "Batreaux's House - Exit Shed Door",
        "Skyloft Village",
        "Skyloft Village - Shed Door to Batreaux's House",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Beedle's Shop",
        "Beedle's Shop - Leave Shop",
        "Central Skyloft",
        "Central Skyloft - Ring Beedle's Bell",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky",
        "Sky - Dive to Upper Skyloft",
        "Upper Skyloft",
        "Upper Skyloft - Loftwing Platforms",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky",
        "Sky - Dive to Central Skyloft",
        "Central Skyloft",
        "Central Skyloft - Loftwing Platforms",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky",
        "Sky - Dive to Skyloft Village",
        "Skyloft Village",
        "Skyloft Village - Loftwing Platforms",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky",
        "Sky - Opening in Clouds",
        "Thunderhead",
        "Thunderhead - Opening in Clouds",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky",
        "Sky - Emerald Pillar",
        "Sealed Grounds - Spiral",
        "Sealed Grounds - Spiral - Fly to Sky",
        2, True, 10,
    ),
    SSAPEntranceData(
        "Sky",
        "Sky - Ruby Pillar",
        "Eldin Volcano - Volcano Entry",
        "Eldin Volcano - Volcano Entry - Fly to Sky",
        2, True, 10,
    ),
    SSAPEntranceData(
        "Sky",
        "Sky - Amber Pillar",
        "Lanayru Mine - Landing",
        "Lanayru Mine - Landing - Fly to Sky",
        2, True, 10,
    ),
    SSAPEntranceData(
        "Thunderhead",
        "Thunderhead - Opening in Clouds",
        "Sky",
        "Sky - Opening in Clouds",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sealed Grounds - Spiral",
        "Sealed Grounds - Spiral - Fly to Sky",
        "Sky",
        "Sky - Emerald Pillar",
        2, True, 9,
    ),
    SSAPEntranceData(
        "Sealed Grounds - Spiral",
        "Sealed Grounds - Spiral - Sealed Temple",
        "Sealed Grounds - Sealed Temple",
        "Sealed Grounds - Sealed Temple - Spiral",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sealed Grounds - Sealed Temple",
        "Sealed Grounds - Sealed Temple - Spiral",
        "Sealed Grounds - Spiral",
        "Sealed Grounds - Spiral - Sealed Temple",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sealed Grounds - Sealed Temple",
        "Sealed Grounds - Sealed Temple - Behind the Temple",
        "Sealed Grounds - Behind the Temple",
        "Sealed Grounds - Behind the Temple - Sealed Temple",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sealed Grounds - Sealed Temple",
        "Sealed Grounds - Sealed Temple - Hylia's Realm",
        "Hylia's Realm",
        "Hylia's Realm - Door to Temple",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sealed Grounds - Behind the Temple",
        "Sealed Grounds - Behind the Temple - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Sealed Grounds - Behind the Temple",
        "Sealed Grounds - Behind the Temple - Sealed Temple",
        "Sealed Grounds - Sealed Temple",
        "Sealed Grounds - Sealed Temple - Behind the Temple",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sealed Grounds - Behind the Temple",
        "Sealed Grounds - Behind the Temple - Path to Forest",
        "Faron Woods - Forest Entry",
        "Faron Woods - Forest Entry - Path out of Forest",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Hylia's Realm",
        "Hylia's Realm - Door to Temple",
        "Sealed Grounds - Sealed Temple",
        "Sealed Grounds - Sealed Temple - Hylia's Realm",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Forest Entry",
        "Faron Woods - Forest Entry - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Faron Woods - Forest Entry",
        "Faron Woods - Forest Entry - Path out of Forest",
        "Sealed Grounds - Behind the Temple",
        "Sealed Grounds - Behind the Temple - Path to Forest",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Forest Entry",
        "Faron Woods - Forest Entry - In the Woods",
        "Faron Woods - In the Woods",
        "Faron Woods - In the Woods - Forest Entry",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - In the Woods",
        "Faron Woods - In the Woods - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Faron Woods - In the Woods",
        "Faron Woods - In the Woods - Forest Entry",
        "Faron Woods - Forest Entry",
        "Faron Woods - Forest Entry - In the Woods",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - In the Woods",
        "Faron Woods - In the Woods - Viewing Platform",
        "Faron Woods - Viewing Platform",
        "Faron Woods - Viewing Platform - In the Woods",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - In the Woods",
        "Faron Woods - In the Woods - Inside Lower Great Tree",
        "Faron Woods - Inside Lower Great Tree - Water",
        "Faron Woods - Inside Lower Great Tree - Water - In the Woods",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - In the Woods",
        "Faron Woods - In the Woods - Great Tree Platforms",
        "Faron Woods - Great Tree Platforms",
        "Faron Woods - Great Tree Platforms - In the Woods",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - In the Woods",
        "Faron Woods - In the Woods - Jump into Lake",
        "Lake Floria - Lake",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Faron Woods - Viewing Platform",
        "Faron Woods - Viewing Platform - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Faron Woods - Viewing Platform",
        "Faron Woods - Viewing Platform - In the Woods",
        "Faron Woods - In the Woods",
        "Faron Woods - In the Woods - Viewing Platform",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Viewing Platform",
        "Faron Woods - Viewing Platform - Deep Woods Entry",
        "Faron Woods - Deep Woods Entry",
        "Faron Woods - Deep Woods Entry - Viewing Platform",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Viewing Platform",
        "Faron Woods - Viewing Platform - Trial Gate in Faron Woods",
        "Faron Silent Realm",
        "Faron Silent Realm - Trial Gate",
        2, True, 3,
    ),
    SSAPEntranceData(
        "Faron Woods - Deep Woods Entry",
        "Faron Woods - Deep Woods Entry - Viewing Platform",
        "Faron Woods - Viewing Platform",
        "Faron Woods - Viewing Platform - Deep Woods Entry",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Deep Woods Entry",
        "Faron Woods - Deep Woods Entry - Deep Woods after Beehive",
        "Faron Woods - Deep Woods after Beehive",
        "Faron Woods - Deep Woods after Beehive - Deep Woods Entry",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Deep Woods after Beehive",
        "Faron Woods - Deep Woods after Beehive - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Faron Woods - Deep Woods after Beehive",
        "Faron Woods - Deep Woods after Beehive - Deep Woods Entry",
        "Faron Woods - Deep Woods Entry",
        "Faron Woods - Deep Woods Entry - Deep Woods after Beehive",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Deep Woods after Beehive",
        "Faron Woods - Deep Woods after Beehive - Dungeon Entrance in Deep Woods",
        "Skyview - First Room",
        "Skyview - First Room - Dungeon Exit",
        2, True, 2,
    ),
    SSAPEntranceData(
        "Faron Woods - Inside Lower Great Tree - Water",
        "Faron Woods - Inside Lower Great Tree - Water - In the Woods",
        "Faron Woods - In the Woods",
        "Faron Woods - In the Woods - Inside Lower Great Tree",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Inside Lower Great Tree - Water",
        "Faron Woods - Inside Lower Great Tree - Water - After Swinging Platforms",
        "Faron Woods - Inside Lower Great Tree - After Swinging Platforms",
        "Faron Woods - Inside Lower Great Tree - After Swinging Platforms - Water",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Inside Lower Great Tree - Water",
        "Faron Woods - Inside Lower Great Tree - Water - Upper Ledge",
        "Faron Woods - Inside Lower Great Tree - Upper Ledge",
        "Faron Woods - Inside Lower Great Tree - Upper Ledge - Water",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Inside Lower Great Tree - After Swinging Platforms",
        "Faron Woods - Inside Lower Great Tree - After Swinging Platforms - Water",
        "Faron Woods - Inside Lower Great Tree - Water",
        "Faron Woods - Inside Lower Great Tree - Water - After Swinging Platforms",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Inside Lower Great Tree - After Swinging Platforms",
        "Faron Woods - Inside Lower Great Tree - After Swinging Platforms - Upper Ledge",
        "Faron Woods - Inside Lower Great Tree - Upper Ledge",
        "Faron Woods - Inside Lower Great Tree - Upper Ledge - After Swinging Platforms",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Inside Lower Great Tree - After Swinging Platforms",
        "Faron Woods - Inside Lower Great Tree - After Swinging Platforms - Great Tree Platforms",
        "Faron Woods - Great Tree Platforms",
        "Faron Woods - Great Tree Platforms - Lower Exit",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Inside Lower Great Tree - Upper Ledge",
        "Faron Woods - Inside Lower Great Tree - Upper Ledge - Water",
        "Faron Woods - Inside Lower Great Tree - Water",
        "Faron Woods - Inside Lower Great Tree - Water - Upper Ledge",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Inside Lower Great Tree - Upper Ledge",
        "Faron Woods - Inside Lower Great Tree - Upper Ledge - After Swinging Platforms",
        "Faron Woods - Inside Lower Great Tree - After Swinging Platforms",
        "Faron Woods - Inside Lower Great Tree - After Swinging Platforms - Upper Ledge",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Inside Upper Great Tree",
        "Faron Woods - Inside Upper Great Tree - Lower Exit",
        "Faron Woods - Great Tree Platforms",
        "Faron Woods - Great Tree Platforms - Upper Exit",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Inside Upper Great Tree",
        "Faron Woods - Inside Upper Great Tree - Upper Exit",
        "Faron Woods - Outside Top of Great Tree",
        "Faron Woods - Outside Top of Great Tree - Inside Upper Great Tree",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Inside Upper Great Tree",
        "Faron Woods - Inside Upper Great Tree - Jump Down to Lower Great Tree",
        "Faron Woods - Inside Lower Great Tree - Upper Ledge",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Faron Woods - Inside Upper Great Tree",
        "Faron Woods - Inside Upper Great Tree - Speak to Water Dragon",
        "Flooded Faron Woods",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Faron Woods - Great Tree Platforms",
        "Faron Woods - Great Tree Platforms - Lower Exit",
        "Faron Woods - Inside Lower Great Tree - After Swinging Platforms",
        "Faron Woods - Inside Lower Great Tree - After Swinging Platforms - Great Tree Platforms",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Great Tree Platforms",
        "Faron Woods - Great Tree Platforms - Upper Exit",
        "Faron Woods - Inside Upper Great Tree",
        "Faron Woods - Inside Upper Great Tree - Lower Exit",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Great Tree Platforms",
        "Faron Woods - Great Tree Platforms - In the Woods",
        "Faron Woods - In the Woods",
        "Faron Woods - In the Woods - Great Tree Platforms",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Faron Woods - Great Tree Platforms",
        "Faron Woods - Great Tree Platforms - Viewing Platform",
        "Faron Woods - Viewing Platform",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Faron Woods - Outside Top of Great Tree",
        "Faron Woods - Outside Top of Great Tree - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Faron Woods - Outside Top of Great Tree",
        "Faron Woods - Outside Top of Great Tree - In the Woods",
        "Faron Woods - In the Woods",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Faron Woods - Outside Top of Great Tree",
        "Faron Woods - Outside Top of Great Tree - Viewing Platform",
        "Faron Woods - Viewing Platform",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Faron Woods - Outside Top of Great Tree",
        "Faron Woods - Outside Top of Great Tree - Inside Upper Great Tree",
        "Faron Woods - Inside Upper Great Tree",
        "Faron Woods - Inside Upper Great Tree - Upper Exit",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lake Floria - Lake",
        "Lake Floria - Lake - Higher Ledge in Lake",
        "Lake Floria - Higher Ledge in Lake",
        "Lake Floria - Higher Ledge in Lake - Lake",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lake Floria - Lake",
        "Lake Floria - Lake - Dragon's Lair",
        "Lake Floria - Dragon's Lair",
        "Lake Floria - Dragon's Lair - Lake",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lake Floria - Higher Ledge in Lake",
        "Lake Floria - Higher Ledge in Lake - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Lake Floria - Higher Ledge in Lake",
        "Lake Floria - Higher Ledge in Lake - Lake",
        "Lake Floria - Lake",
        "Lake Floria - Lake - Higher Ledge in Lake",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lake Floria - Dragon's Lair",
        "Lake Floria - Dragon's Lair - Lake",
        "Lake Floria - Lake",
        "Lake Floria - Lake - Dragon's Lair",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lake Floria - Dragon's Lair",
        "Lake Floria - Dragon's Lair - Waterfall",
        "Lake Floria - Waterfall",
        "Lake Floria - Waterfall - Dragon's Lair",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lake Floria - Waterfall",
        "Lake Floria - Waterfall - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Lake Floria - Waterfall",
        "Lake Floria - Waterfall - Dragon's Lair",
        "Lake Floria - Dragon's Lair",
        "Lake Floria - Dragon's Lair - Waterfall",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lake Floria - Waterfall",
        "Lake Floria - Waterfall - Path into Forest",
        "Faron Woods - In the Woods",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Lake Floria - Waterfall",
        "Lake Floria - Waterfall - Ancient Cistern Ledge",
        "Lake Floria - Ancient Cistern Ledge",
        "Lake Floria - Ancient Cistern Ledge - Waterfall",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lake Floria - Ancient Cistern Ledge",
        "Lake Floria - Ancient Cistern Ledge - Waterfall",
        "Lake Floria - Waterfall",
        "Lake Floria - Waterfall - Ancient Cistern Ledge",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lake Floria - Ancient Cistern Ledge",
        "Lake Floria - Ancient Cistern Ledge - Dungeon Entrance in Lake Floria",
        "Ancient Cistern - Main Room",
        "Ancient Cistern - Main Room - Dungeon Exit",
        2, True, 2,
    ),
    SSAPEntranceData(
        "Flooded Faron Woods",
        "Flooded Faron Woods - Speak to Water Dragon",
        "Faron Woods - Outside Top of Great Tree",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Volcano Entry",
        "Eldin Volcano - Volcano Entry - Fly to Sky",
        "Sky",
        "Sky - Ruby Pillar",
        2, True, 9,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Volcano Entry",
        "Eldin Volcano - Volcano Entry - Volcano East",
        "Eldin Volcano - Volcano East",
        "Eldin Volcano - Volcano East - Volcano Entry",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Volcano Entry",
        "Eldin Volcano - Volcano Entry - Volcano Ascent",
        "Eldin Volcano - Volcano Ascent",
        "Eldin Volcano - Volcano Ascent - Volcano Entry",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Volcano Entry",
        "Eldin Volcano - Volcano Entry - Speak to Gossip Stone",
        "Bokoblin Base - Prison",
        "Bokoblin Base - Prison - Speak to Gossip Stone",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Volcano East",
        "Eldin Volcano - Volcano East - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Volcano East",
        "Eldin Volcano - Volcano East - Volcano Entry",
        "Eldin Volcano - Volcano Entry",
        "Eldin Volcano - Volcano Entry - Volcano East",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Volcano East",
        "Eldin Volcano - Volcano East - Volcano Ascent",
        "Eldin Volcano - Volcano Ascent",
        "Eldin Volcano - Volcano Ascent - Volcano East",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Volcano East",
        "Eldin Volcano - Volcano East - Mogma Cave",
        "Mogma Turf - Entry",
        "Mogma Turf - Entry - Use First Air Vent",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Volcano Ascent",
        "Eldin Volcano - Volcano Ascent - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Volcano Ascent",
        "Eldin Volcano - Volcano Ascent - Volcano Entry",
        "Eldin Volcano - Volcano Entry",
        "Eldin Volcano - Volcano Entry - Volcano Ascent",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Volcano Ascent",
        "Eldin Volcano - Volcano Ascent - Volcano East",
        "Eldin Volcano - Volcano East",
        "Eldin Volcano - Volcano East - Volcano Ascent",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Volcano Ascent",
        "Eldin Volcano - Volcano Ascent - Thrill Digger Cliff",
        "Eldin Volcano - Thrill Digger Cliff",
        "Eldin Volcano - Thrill Digger Cliff - Volcano Ascent",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Volcano Ascent",
        "Eldin Volcano - Volcano Ascent - Bottom of Sand Slide",
        "Eldin Volcano - Bottom of Sand Slide",
        "Eldin Volcano - Bottom of Sand Slide - Volcano Ascent",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Volcano Ascent",
        "Eldin Volcano - Volcano Ascent - Trial Gate in Eldin Volcano",
        "Eldin Silent Realm",
        "Eldin Silent Realm - Trial Gate",
        2, True, 3,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Thrill Digger Cliff",
        "Eldin Volcano - Thrill Digger Cliff - Volcano Ascent",
        "Eldin Volcano - Volcano Ascent",
        "Eldin Volcano - Volcano Ascent - Thrill Digger Cliff",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Thrill Digger Cliff",
        "Eldin Volcano - Thrill Digger Cliff - Thrill Digger Cave",
        "Eldin Volcano - Thrill Digger Cave",
        "Eldin Volcano - Thrill Digger Cave - Exit Thrill Digger Cave",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Thrill Digger Cliff",
        "Eldin Volcano - Thrill Digger Cliff - Near Temple Entrance",
        "Eldin Volcano - Near Temple Entrance",
        "Eldin Volcano - Near Temple Entrance - Thrill Digger Cliff",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Thrill Digger Cave",
        "Eldin Volcano - Thrill Digger Cave - Exit Thrill Digger Cave",
        "Eldin Volcano - Thrill Digger Cliff",
        "Eldin Volcano - Thrill Digger Cliff - Thrill Digger Cave",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Near Temple Entrance",
        "Eldin Volcano - Near Temple Entrance - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Near Temple Entrance",
        "Eldin Volcano - Near Temple Entrance - Thrill Digger Cliff",
        "Eldin Volcano - Thrill Digger Cliff",
        "Eldin Volcano - Thrill Digger Cliff - Near Temple Entrance",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Near Temple Entrance",
        "Eldin Volcano - Near Temple Entrance - Hot Cave",
        "Eldin Volcano - Hot Cave",
        "Eldin Volcano - Hot Cave - Near Temple Entrance",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Near Temple Entrance",
        "Eldin Volcano - Near Temple Entrance - Dungeon Entrance in Eldin Volcano",
        "Earth Temple - First Room",
        "Earth Temple - First Room - Dungeon Exit",
        2, True, 2,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Hot Cave",
        "Eldin Volcano - Hot Cave - Near Temple Entrance",
        "Eldin Volcano - Near Temple Entrance",
        "Eldin Volcano - Near Temple Entrance - Hot Cave",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Hot Cave",
        "Eldin Volcano - Hot Cave - Sand Slide",
        "Eldin Volcano - Sand Slide",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Hot Cave",
        "Eldin Volcano - Hot Cave - Upper Path out of Hot Cave",
        "Volcano Summit - Summit",
        "Volcano Summit - Summit - Path out of Summit before Sandy Slope",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Sand Slide",
        "Eldin Volcano - Sand Slide - Bottom of Sand Slide",
        "Eldin Volcano - Bottom of Sand Slide",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Bottom of Sand Slide",
        "Eldin Volcano - Bottom of Sand Slide - Volcano Ascent",
        "Eldin Volcano - Volcano Ascent",
        "Eldin Volcano - Volcano Ascent - Bottom of Sand Slide",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Eldin Volcano - Bottom of Sand Slide",
        "Eldin Volcano - Bottom of Sand Slide - Hot Cave",
        "Eldin Volcano - Hot Cave",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Mogma Turf - Entry",
        "Mogma Turf - Entry - Use First Air Vent",
        "Eldin Volcano - Volcano East",
        "Eldin Volcano - Volcano East - Mogma Cave",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Mogma Turf - Entry",
        "Mogma Turf - Entry - Use Digging Air Vent",
        "Mogma Turf - After Digging Air Vent",
        "Mogma Turf - After Digging Air Vent - Jump off Ledge",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Mogma Turf - After Digging Air Vent",
        "Mogma Turf - After Digging Air Vent - Jump off Ledge",
        "Mogma Turf - Entry",
        "Mogma Turf - Entry - Use Digging Air Vent",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Mogma Turf - After Digging Air Vent",
        "Mogma Turf - After Digging Air Vent - Use Last Air Vent",
        "Eldin Volcano - Volcano Ascent",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Volcano Summit - Summit",
        "Volcano Summit - Summit - Path out of Summit before Sandy Slope",
        "Eldin Volcano - Hot Cave",
        "Eldin Volcano - Hot Cave - Upper Path out of Hot Cave",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Volcano Summit - Summit",
        "Volcano Summit - Summit - Path out of Summit after Lava Platforms",
        "Volcano Summit - Waterfall",
        "Volcano Summit - Waterfall - Path out of Waterfall",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Volcano Summit - Summit",
        "Volcano Summit - Summit - Upper Path out of Summit after Lava Platforms",
        "Volcano Summit - Before First Frog",
        "Volcano Summit - Before First Frog - Path across from First Frog",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Volcano Summit - Waterfall",
        "Volcano Summit - Waterfall - Path out of Waterfall",
        "Volcano Summit - Summit",
        "Volcano Summit - Summit - Path out of Summit after Lava Platforms",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Volcano Summit - Before First Frog",
        "Volcano Summit - Before First Frog - Path across from First Frog",
        "Volcano Summit - Summit",
        "Volcano Summit - Summit - Upper Path out of Summit after Lava Platforms",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Volcano Summit - Before First Frog",
        "Volcano Summit - Before First Frog - Pass First Frog",
        "Volcano Summit - Between Frogs",
        "Volcano Summit - Between Frogs - Pass First Frog",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Volcano Summit - Between Frogs",
        "Volcano Summit - Between Frogs - Pass First Frog",
        "Volcano Summit - Before First Frog",
        "Volcano Summit - Before First Frog - Pass First Frog",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Volcano Summit - Between Frogs",
        "Volcano Summit - Between Frogs - Pass Second Frog",
        "Volcano Summit - After Second Frog",
        "Volcano Summit - After Second Frog - Pass Second Frog",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Volcano Summit - After Second Frog",
        "Volcano Summit - After Second Frog - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Volcano Summit - After Second Frog",
        "Volcano Summit - After Second Frog - Pass Second Frog",
        "Volcano Summit - Between Frogs",
        "Volcano Summit - Between Frogs - Pass Second Frog",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Volcano Summit - After Second Frog",
        "Volcano Summit - After Second Frog - Dungeon Entrance in Volcano Summit",
        "Fire Sanctuary - First Room",
        "Fire Sanctuary - First Room - Dungeon Exit",
        2, True, 2,
    ),
    SSAPEntranceData(
        "Bokoblin Base - Prison",
        "Bokoblin Base - Prison - Speak to Gossip Stone",
        "Eldin Volcano - Volcano Entry",
        "Eldin Volcano - Volcano Entry - Speak to Gossip Stone",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Bokoblin Base - Prison",
        "Bokoblin Base - Prison - Dig to Volcano",
        "Bokoblin Base - Volcano",
        "Bokoblin Base - Volcano - Dig to Prison",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Bokoblin Base - Volcano",
        "Bokoblin Base - Volcano - Dig to Prison",
        "Bokoblin Base - Prison",
        "Bokoblin Base - Prison - Dig to Volcano",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Bokoblin Base - Volcano",
        "Bokoblin Base - Volcano - Use Air Vent",
        "Bokoblin Base - Top of Volcano",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Bokoblin Base - Top of Volcano",
        "Bokoblin Base - Top of Volcano - Path through Hot Cave",
        "Bokoblin Base - Summit",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Bokoblin Base - Summit",
        "Bokoblin Base - Summit - Dragon's Lair",
        "Bokoblin Base - Dragon's Lair",
        "Bokoblin Base - Dragon's Lair - Exit Dragon's Lair",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Bokoblin Base - Dragon's Lair",
        "Bokoblin Base - Dragon's Lair - Exit Dragon's Lair",
        "Bokoblin Base - Summit",
        "Bokoblin Base - Summit - Dragon's Lair",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Bokoblin Base - Dragon's Lair",
        "Bokoblin Base - Dragon's Lair - Exit Bokoblin Base",
        "Volcano Summit - Summit",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Lanayru Mine - Landing",
        "Lanayru Mine - Landing - Fly to Sky",
        "Sky",
        "Sky - Amber Pillar",
        2, True, 9,
    ),
    SSAPEntranceData(
        "Lanayru Mine - Landing",
        "Lanayru Mine - Landing - Clawshot Path to Caves",
        "Lanayru Mine - Door to Caves",
        "Lanayru Mine - Door to Caves - Clawshot Path to Mine",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mine - Landing",
        "Lanayru Mine - Landing - Pass Statues in Mine",
        "Lanayru Mine - Ending",
        "Lanayru Mine - Ending - Pass Statues in Mine",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mine - Door to Caves",
        "Lanayru Mine - Door to Caves - Clawshot Path to Mine",
        "Lanayru Mine - Landing",
        "Lanayru Mine - Landing - Clawshot Path to Caves",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mine - Door to Caves",
        "Lanayru Mine - Door to Caves - Door to Caves",
        "Lanayru Caves - Caves",
        "Lanayru Caves - Caves - East Exit",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mine - Ending",
        "Lanayru Mine - Ending - Pass Statues in Mine",
        "Lanayru Mine - Landing",
        "Lanayru Mine - Landing - Pass Statues in Mine",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mine - Ending",
        "Lanayru Mine - Ending - Minecart Ride out of Mine",
        "Lanayru Desert - Desert Entry",
        "Lanayru Desert - Desert Entry - Minecart Ride out of Desert",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Desert Entry",
        "Lanayru Desert - Desert Entry - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Desert Entry",
        "Lanayru Desert - Desert Entry - Minecart Ride out of Desert",
        "Lanayru Mine - Ending",
        "Lanayru Mine - Ending - Minecart Ride out of Mine",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Desert Entry",
        "Lanayru Desert - Desert Entry - Sand Oasis",
        "Lanayru Desert - Sand Oasis",
        "Lanayru Desert - Sand Oasis - Desert Entry",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Desert Entry",
        "Lanayru Desert - Desert Entry - Stone Cache",
        "Lanayru Desert - Stone Cache",
        "Lanayru Desert - Stone Cache - Desert Entry",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Sand Oasis",
        "Lanayru Desert - Sand Oasis - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Sand Oasis",
        "Lanayru Desert - Sand Oasis - Desert Entry",
        "Lanayru Desert - Desert Entry",
        "Lanayru Desert - Desert Entry - Sand Oasis",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Sand Oasis",
        "Lanayru Desert - Sand Oasis - Path to Temple of Time",
        "Lanayru Desert - Temple of Time - Desert Gorge",
        "Lanayru Desert - Temple of Time - Desert Gorge - Path to Sand Oasis",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Sand Oasis",
        "Lanayru Desert - Sand Oasis - Raised Entrance to Caves",
        "Lanayru Desert - Higher Ledge in Sand Oasis",
        "Lanayru Desert - Higher Ledge in Sand Oasis - Sand Oasis",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Higher Ledge in Sand Oasis",
        "Lanayru Desert - Higher Ledge in Sand Oasis - Sand Oasis",
        "Lanayru Desert - Sand Oasis",
        "Lanayru Desert - Sand Oasis - Raised Entrance to Caves",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Higher Ledge in Sand Oasis",
        "Lanayru Desert - Higher Ledge in Sand Oasis - Path to Caves",
        "Lanayru Caves - Caves",
        "Lanayru Caves - Caves - North Exit",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Temple of Time - Desert Gorge",
        "Lanayru Desert - Temple of Time - Desert Gorge - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Temple of Time - Desert Gorge",
        "Lanayru Desert - Temple of Time - Desert Gorge - Path to Sand Oasis",
        "Lanayru Desert - Sand Oasis",
        "Lanayru Desert - Sand Oasis - Path to Temple of Time",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Temple of Time - Desert Gorge",
        "Lanayru Desert - Temple of Time - Desert Gorge - Minecart through Tree Root",
        "Lanayru Desert - Temple of Time - Temple",
        "Lanayru Desert - Temple of Time - Temple - Minecart to Desert Gorge",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Temple of Time - Temple",
        "Lanayru Desert - Temple of Time - Temple - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Temple of Time - Temple",
        "Lanayru Desert - Temple of Time - Temple - Minecart to Desert Gorge",
        "Lanayru Desert - Temple of Time - Desert Gorge",
        "Lanayru Desert - Temple of Time - Desert Gorge - Minecart through Tree Root",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Temple of Time - Temple",
        "Lanayru Desert - Temple of Time - Temple - Path to North Desert",
        "Lanayru Desert - North Desert",
        "Lanayru Desert - North Desert - Path to Temple of Time",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - North Desert",
        "Lanayru Desert - North Desert - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Lanayru Desert - North Desert",
        "Lanayru Desert - North Desert - Path to Temple of Time",
        "Lanayru Desert - Temple of Time - Temple",
        "Lanayru Desert - Temple of Time - Temple - Path to North Desert",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - North Desert",
        "Lanayru Desert - North Desert - Stone Cache",
        "Lanayru Desert - Stone Cache",
        "Lanayru Desert - Stone Cache - North Desert",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - North Desert",
        "Lanayru Desert - North Desert - Door to Lightning Node",
        "Lanayru Desert - Lightning Node",
        "Lanayru Desert - Lightning Node - Exit Lightning Node",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - North Desert",
        "Lanayru Desert - North Desert - Dungeon Entrance in Lanayru Desert",
        "Lanayru Mining Facility - First Room",
        "Lanayru Mining Facility - First Room - Dungeon Exit",
        2, True, 2,
    ),
    SSAPEntranceData(
        "Lanayru Desert - North Desert",
        "Lanayru Desert - North Desert - Trial Gate in Lanayru Desert",
        "Lanayru Silent Realm",
        "Lanayru Silent Realm - Trial Gate",
        2, True, 3,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Stone Cache",
        "Lanayru Desert - Stone Cache - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Stone Cache",
        "Lanayru Desert - Stone Cache - Desert Entry",
        "Lanayru Desert - Desert Entry",
        "Lanayru Desert - Desert Entry - Stone Cache",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Stone Cache",
        "Lanayru Desert - Stone Cache - North Desert",
        "Lanayru Desert - North Desert",
        "Lanayru Desert - North Desert - Stone Cache",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Stone Cache",
        "Lanayru Desert - Stone Cache - Door to Fire Node",
        "Lanayru Desert - Fire Node",
        "Lanayru Desert - Fire Node - Exit Fire Node",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Fire Node",
        "Lanayru Desert - Fire Node - Exit Fire Node",
        "Lanayru Desert - Stone Cache",
        "Lanayru Desert - Stone Cache - Door to Fire Node",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Desert - Lightning Node",
        "Lanayru Desert - Lightning Node - Exit Lightning Node",
        "Lanayru Desert - North Desert",
        "Lanayru Desert - North Desert - Door to Lightning Node",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Caves - Caves",
        "Lanayru Caves - Caves - North Exit",
        "Lanayru Desert - Higher Ledge in Sand Oasis",
        "Lanayru Desert - Higher Ledge in Sand Oasis - Path to Caves",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Caves - Caves",
        "Lanayru Caves - Caves - East Exit",
        "Lanayru Mine - Door to Caves",
        "Lanayru Mine - Door to Caves - Door to Caves",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Caves - Caves",
        "Lanayru Caves - Caves - South Exit past Crawlspace",
        "Lanayru Caves - Past Crawlspace",
        "Lanayru Caves - Past Crawlspace - Through Crawlspace",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Caves - Caves",
        "Lanayru Caves - Caves - West Exit by Clawshot Target",
        "Lanayru Caves - Past Locked Door",
        "Lanayru Caves - Past Locked Door - Through Door",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Caves - Past Locked Door",
        "Lanayru Caves - Past Locked Door - Path away from Door",
        "Lanayru Sand Sea - Door to Caves",
        "Lanayru Sand Sea - Door to Caves - Door to Caves",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Caves - Past Locked Door",
        "Lanayru Caves - Past Locked Door - Through Door",
        "Lanayru Caves - Caves",
        "Lanayru Caves - Caves - West Exit by Clawshot Target",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Caves - Past Crawlspace",
        "Lanayru Caves - Past Crawlspace - Path away from Crawlspace",
        "Lanayru Gorge - Gorge",
        "Lanayru Gorge - Gorge - Path to Caves",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Caves - Past Crawlspace",
        "Lanayru Caves - Past Crawlspace - Through Crawlspace",
        "Lanayru Caves - Caves",
        "Lanayru Caves - Caves - South Exit past Crawlspace",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Gorge - Gorge",
        "Lanayru Gorge - Gorge - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Lanayru Gorge - Gorge",
        "Lanayru Gorge - Gorge - Path to Caves",
        "Lanayru Caves - Past Crawlspace",
        "Lanayru Caves - Past Crawlspace - Path away from Crawlspace",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Door to Caves",
        "Lanayru Sand Sea - Door to Caves - Clawshot Path to Harbor",
        "Lanayru Sand Sea - Ancient Harbor",
        "Lanayru Sand Sea - Ancient Harbor - Clawshot Path to Caves",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Door to Caves",
        "Lanayru Sand Sea - Door to Caves - Door to Caves",
        "Lanayru Caves - Past Locked Door",
        "Lanayru Caves - Past Locked Door - Path away from Door",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Ancient Harbor",
        "Lanayru Sand Sea - Ancient Harbor - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Ancient Harbor",
        "Lanayru Sand Sea - Ancient Harbor - Clawshot Path to Caves",
        "Lanayru Sand Sea - Door to Caves",
        "Lanayru Sand Sea - Door to Caves - Clawshot Path to Harbor",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Ancient Harbor",
        "Lanayru Sand Sea - Ancient Harbor - Boat to Sea",
        "Lanayru Sand Sea - Sea",
        "Lanayru Sand Sea - Sea - Dock at Ancient Harbor",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Sea",
        "Lanayru Sand Sea - Sea - Dock at Ancient Harbor",
        "Lanayru Sand Sea - Ancient Harbor",
        "Lanayru Sand Sea - Ancient Harbor - Boat to Sea",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Sea",
        "Lanayru Sand Sea - Sea - Dock at Skipper's Retreat",
        "Lanayru Sand Sea - Skipper's Retreat - Dock",
        "Lanayru Sand Sea - Skipper's Retreat - Dock - Boat to Sea",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Sea",
        "Lanayru Sand Sea - Sea - Dock at Shipyard",
        "Lanayru Sand Sea - Shipyard",
        "Lanayru Sand Sea - Shipyard - Boat to Sea",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Sea",
        "Lanayru Sand Sea - Sea - Dock at Pirate Stronghold",
        "Lanayru Sand Sea - Pirate Stronghold",
        "Lanayru Sand Sea - Pirate Stronghold - Boat to Sea",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Sea",
        "Lanayru Sand Sea - Sea - Dungeon Entrance in Lanayru Sand Sea",
        "Sandship - Deck",
        "Sandship - Deck - Dungeon Exit",
        2, True, 2,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Skipper's Retreat - Dock",
        "Lanayru Sand Sea - Skipper's Retreat - Dock - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Skipper's Retreat - Dock",
        "Lanayru Sand Sea - Skipper's Retreat - Dock - Boat to Sea",
        "Lanayru Sand Sea - Sea",
        "Lanayru Sand Sea - Sea - Dock at Skipper's Retreat",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Skipper's Retreat - Dock",
        "Lanayru Sand Sea - Skipper's Retreat - Dock - After Rock",
        "Lanayru Sand Sea - Skipper's Retreat - After Rock",
        "Lanayru Sand Sea - Skipper's Retreat - After Rock - Dock",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Skipper's Retreat - After Rock",
        "Lanayru Sand Sea - Skipper's Retreat - After Rock - Dock",
        "Lanayru Sand Sea - Skipper's Retreat - Dock",
        "Lanayru Sand Sea - Skipper's Retreat - Dock - After Rock",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Skipper's Retreat - After Rock",
        "Lanayru Sand Sea - Skipper's Retreat - After Rock - Top of Skipper's Retreat",
        "Lanayru Sand Sea - Skipper's Retreat - Top of Skipper's Retreat",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Skipper's Retreat - Top of Skipper's Retreat",
        "Lanayru Sand Sea - Skipper's Retreat - Top of Skipper's Retreat - Shack",
        "Lanayru Sand Sea - Skipper's Retreat - Shack",
        "Lanayru Sand Sea - Skipper's Retreat - Shack - Top of Skipper's Retreat",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Skipper's Retreat - Shack",
        "Lanayru Sand Sea - Skipper's Retreat - Shack - Top of Skipper's Retreat",
        "Lanayru Sand Sea - Skipper's Retreat - Top of Skipper's Retreat",
        "Lanayru Sand Sea - Skipper's Retreat - Top of Skipper's Retreat - Shack",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Skipper's Retreat - Shack",
        "Lanayru Sand Sea - Skipper's Retreat - Shack - After Rock",
        "Lanayru Sand Sea - Skipper's Retreat - After Rock",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Skipper's Retreat - Shack",
        "Lanayru Sand Sea - Skipper's Retreat - Shack - Skydive",
        "Lanayru Sand Sea - Skipper's Retreat - Skydive",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Skipper's Retreat - Skydive",
        "Lanayru Sand Sea - Skipper's Retreat - Skydive - Dock",
        "Lanayru Sand Sea - Skipper's Retreat - Dock",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Shipyard",
        "Lanayru Sand Sea - Shipyard - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Shipyard",
        "Lanayru Sand Sea - Shipyard - Boat to Sea",
        "Lanayru Sand Sea - Sea",
        "Lanayru Sand Sea - Sea - Dock at Shipyard",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Pirate Stronghold",
        "Lanayru Sand Sea - Pirate Stronghold - Fly to Sky",
        "Sky",
        None,
        1, False, 9,
    ),
    SSAPEntranceData(
        "Lanayru Sand Sea - Pirate Stronghold",
        "Lanayru Sand Sea - Pirate Stronghold - Boat to Sea",
        "Lanayru Sand Sea - Sea",
        "Lanayru Sand Sea - Sea - Dock at Pirate Stronghold",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Skyview - First Room",
        "Skyview - First Room - Dungeon Exit",
        "Faron Woods - Deep Woods after Beehive",
        "Faron Woods - Deep Woods after Beehive - Dungeon Entrance in Deep Woods",
        2, True, 4,
    ),
    SSAPEntranceData(
        "Skyview - First Room",
        "Skyview - First Room - Second Room",
        "Skyview - Second Room",
        "Skyview - Second Room - First Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Skyview - Second Room",
        "Skyview - Second Room - Dungeon Exit",
        "Faron Woods - Deep Woods after Beehive",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Skyview - Second Room",
        "Skyview - Second Room - First Room",
        "Skyview - First Room",
        "Skyview - First Room - Second Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Skyview - Second Room",
        "Skyview - Second Room - Main Room",
        "Skyview - Main Room",
        "Skyview - Main Room - Second Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Skyview - Main Room",
        "Skyview - Main Room - Dungeon Exit",
        "Faron Woods - Deep Woods after Beehive",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Skyview - Main Room",
        "Skyview - Main Room - Second Room",
        "Skyview - Second Room",
        "Skyview - Second Room - Main Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Skyview - Main Room",
        "Skyview - Main Room - Boss Door Area",
        "Skyview - Boss Door Area",
        "Skyview - Boss Door Area - Main Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Skyview - Boss Door Area",
        "Skyview - Boss Door Area - Dungeon Exit",
        "Faron Woods - Deep Woods after Beehive",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Skyview - Boss Door Area",
        "Skyview - Boss Door Area - Main Room",
        "Skyview - Main Room",
        "Skyview - Main Room - Boss Door Area",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Skyview - Boss Door Area",
        "Skyview - Boss Door Area - Boss Room",
        "Skyview - Boss Room",
        "Skyview - Boss Room - Boss Door Area",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Skyview - Boss Room",
        "Skyview - Boss Room - Boss Door Area",
        "Skyview - Boss Door Area",
        "Skyview - Boss Door Area - Boss Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Skyview - Boss Room",
        "Skyview - Boss Room - Skyview Spring",
        "Skyview - Skyview Spring",
        "Skyview - Skyview Spring - Boss Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Skyview - Skyview Spring",
        "Skyview - Skyview Spring - Boss Room",
        "Skyview - Boss Room",
        "Skyview - Boss Room - Skyview Spring",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Skyview - Skyview Spring",
        "Skyview - Skyview Spring - Strike Crest",
        "Faron Woods - Deep Woods after Beehive",
        None,
        1, True, 6,
    ),
    SSAPEntranceData(
        "Earth Temple - First Room",
        "Earth Temple - First Room - Dungeon Exit",
        "Eldin Volcano - Near Temple Entrance",
        "Eldin Volcano - Near Temple Entrance - Dungeon Entrance in Eldin Volcano",
        2, True, 4,
    ),
    SSAPEntranceData(
        "Earth Temple - First Room",
        "Earth Temple - First Room - Main Room",
        "Earth Temple - Main Room",
        "Earth Temple - Main Room - First Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Earth Temple - Main Room",
        "Earth Temple - Main Room - Dungeon Exit",
        "Eldin Volcano - Near Temple Entrance",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Earth Temple - Main Room",
        "Earth Temple - Main Room - First Room",
        "Earth Temple - First Room",
        "Earth Temple - First Room - Main Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Earth Temple - Main Room",
        "Earth Temple - Main Room - Room with Slopes",
        "Earth Temple - Room with Slopes",
        "Earth Temple - Room with Slopes - Main Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Earth Temple - Room with Slopes",
        "Earth Temple - Room with Slopes - Main Room",
        "Earth Temple - Main Room",
        "Earth Temple - Main Room - Room with Slopes",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Earth Temple - Room with Slopes",
        "Earth Temple - Room with Slopes - Boss Door Area",
        "Earth Temple - Boss Door Area",
        "Earth Temple - Boss Door Area - Room with Slopes",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Earth Temple - Boss Door Area",
        "Earth Temple - Boss Door Area - Dungeon Exit",
        "Eldin Volcano - Near Temple Entrance",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Earth Temple - Boss Door Area",
        "Earth Temple - Boss Door Area - Room with Slopes",
        "Earth Temple - Room with Slopes",
        "Earth Temple - Room with Slopes - Boss Door Area",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Earth Temple - Boss Door Area",
        "Earth Temple - Boss Door Area - Boss Room",
        "Earth Temple - Boss Room",
        "Earth Temple - Boss Room - Boss Door Area",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Earth Temple - Boss Room",
        "Earth Temple - Boss Room - Boss Door Area",
        "Earth Temple - Boss Door Area",
        "Earth Temple - Boss Door Area - Boss Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Earth Temple - Boss Room",
        "Earth Temple - Boss Room - Earth Spring",
        "Earth Temple - Earth Spring",
        "Earth Temple - Earth Spring - Boss Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Earth Temple - Earth Spring",
        "Earth Temple - Earth Spring - Boss Room",
        "Earth Temple - Boss Room",
        "Earth Temple - Boss Room - Earth Spring",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Earth Temple - Earth Spring",
        "Earth Temple - Earth Spring - Strike Crest",
        "Eldin Volcano - Near Temple Entrance",
        None,
        1, True, 6,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - First Room",
        "Lanayru Mining Facility - First Room - Dungeon Exit",
        "Lanayru Desert - North Desert",
        "Lanayru Desert - North Desert - Dungeon Entrance in Lanayru Desert",
        2, True, 4,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - First Room",
        "Lanayru Mining Facility - First Room - Second Room",
        "Lanayru Mining Facility - Second Room",
        "Lanayru Mining Facility - Second Room - First Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Second Room",
        "Lanayru Mining Facility - Second Room - First Room",
        "Lanayru Mining Facility - First Room",
        "Lanayru Mining Facility - First Room - Second Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Second Room",
        "Lanayru Mining Facility - Second Room - Front Main Room",
        "Lanayru Mining Facility - Front Main Room",
        "Lanayru Mining Facility - Front Main Room - Second Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Second Room",
        "Lanayru Mining Facility - Second Room - Key Locked Room",
        "Lanayru Mining Facility - Key Locked Room",
        "Lanayru Mining Facility - Key Locked Room - Second Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Second Room",
        "Lanayru Mining Facility - Second Room - West Room",
        "Lanayru Mining Facility - West Room",
        "Lanayru Mining Facility - West Room - Second Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Key Locked Room",
        "Lanayru Mining Facility - Key Locked Room - Second Room",
        "Lanayru Mining Facility - Second Room",
        "Lanayru Mining Facility - Second Room - Key Locked Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Key Locked Room",
        "Lanayru Mining Facility - Key Locked Room - Front Main Room",
        "Lanayru Mining Facility - Front Main Room",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - West Room",
        "Lanayru Mining Facility - West Room - Second Room",
        "Lanayru Mining Facility - Second Room",
        "Lanayru Mining Facility - Second Room - West Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - West Room",
        "Lanayru Mining Facility - West Room - West Main Room",
        "Lanayru Mining Facility - West Main Room",
        "Lanayru Mining Facility - West Main Room - West Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Front Main Room",
        "Lanayru Mining Facility - Front Main Room - Dungeon Exit",
        "Lanayru Desert - North Desert",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Front Main Room",
        "Lanayru Mining Facility - Front Main Room - Second Room",
        "Lanayru Mining Facility - Second Room",
        "Lanayru Mining Facility - Second Room - Front Main Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - West Main Room",
        "Lanayru Mining Facility - West Main Room - West Room",
        "Lanayru Mining Facility - West Room",
        "Lanayru Mining Facility - West Room - West Main Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - West Main Room",
        "Lanayru Mining Facility - West Main Room - Front Main Room",
        "Lanayru Mining Facility - Front Main Room",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - West Main Room",
        "Lanayru Mining Facility - West Main Room - Spike Maze",
        "Lanayru Mining Facility - Spike Maze",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Spike Maze",
        "Lanayru Mining Facility - Spike Maze - Boss Door Area",
        "Lanayru Mining Facility - Boss Door Area",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Boss Door Area",
        "Lanayru Mining Facility - Boss Door Area - Dungeon Exit",
        "Lanayru Desert - North Desert",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Boss Door Area",
        "Lanayru Mining Facility - Boss Door Area - Boss Key Room",
        "Lanayru Mining Facility - Boss Key Room",
        "Lanayru Mining Facility - Boss Key Room - Boss Door Area",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Boss Door Area",
        "Lanayru Mining Facility - Boss Door Area - Boss Room",
        "Lanayru Mining Facility - Boss Room",
        "Lanayru Mining Facility - Boss Room - Boss Door Area",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Boss Key Room",
        "Lanayru Mining Facility - Boss Key Room - Boss Door Area",
        "Lanayru Mining Facility - Boss Door Area",
        "Lanayru Mining Facility - Boss Door Area - Boss Key Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Boss Room",
        "Lanayru Mining Facility - Boss Room - Boss Door Area",
        "Lanayru Mining Facility - Boss Door Area",
        "Lanayru Mining Facility - Boss Door Area - Boss Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Boss Room",
        "Lanayru Mining Facility - Boss Room - Hall of Ancient Robots",
        "Lanayru Mining Facility - Hall of Ancient Robots",
        "Lanayru Mining Facility - Hall of Ancient Robots - Boss Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Hall of Ancient Robots",
        "Lanayru Mining Facility - Hall of Ancient Robots - Boss Room",
        "Lanayru Mining Facility - Boss Room",
        "Lanayru Mining Facility - Boss Room - Hall of Ancient Robots",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Lanayru Mining Facility - Hall of Ancient Robots",
        "Lanayru Mining Facility - Hall of Ancient Robots - Exit Hall of Ancient Robots",
        "Lanayru Desert - Temple of Time - Temple",
        None,
        1, True, 6,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Main Room",
        "Ancient Cistern - Main Room - Dungeon Exit",
        "Lake Floria - Ancient Cistern Ledge",
        "Lake Floria - Ancient Cistern Ledge - Dungeon Entrance in Lake Floria",
        2, True, 4,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Main Room",
        "Ancient Cistern - Main Room - East Room",
        "Ancient Cistern - East Room",
        "Ancient Cistern - East Room - Main Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Main Room",
        "Ancient Cistern - Main Room - Past Waterfall",
        "Ancient Cistern - Past Waterfall",
        "Ancient Cistern - Past Waterfall - Main Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Main Room",
        "Ancient Cistern - Main Room - Statue",
        "Ancient Cistern - Statue",
        "Ancient Cistern - Statue - Main Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Statue",
        "Ancient Cistern - Statue - Main Room",
        "Ancient Cistern - Main Room",
        "Ancient Cistern - Main Room - Statue",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Statue",
        "Ancient Cistern - Statue - Basement",
        "Ancient Cistern - Basement",
        "Ancient Cistern - Basement - Statue",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Statue",
        "Ancient Cistern - Statue - Boss Room",
        "Ancient Cistern - Boss Room",
        "Ancient Cistern - Boss Room - Statue",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Ancient Cistern - East Room",
        "Ancient Cistern - East Room - Main Room",
        "Ancient Cistern - Main Room",
        "Ancient Cistern - Main Room - East Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Past Waterfall",
        "Ancient Cistern - Past Waterfall - Main Room",
        "Ancient Cistern - Main Room",
        "Ancient Cistern - Main Room - Past Waterfall",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Past Waterfall",
        "Ancient Cistern - Past Waterfall - Gutters",
        "Ancient Cistern - Gutters",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Gutters",
        "Ancient Cistern - Gutters - Past Gutters",
        "Ancient Cistern - Past Gutters",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Past Gutters",
        "Ancient Cistern - Past Gutters - Dungeon Exit",
        "Lake Floria - Ancient Cistern Ledge",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Past Gutters",
        "Ancient Cistern - Past Gutters - Main Room",
        "Ancient Cistern - Main Room",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Basement",
        "Ancient Cistern - Basement - Statue",
        "Ancient Cistern - Statue",
        "Ancient Cistern - Statue - Basement",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Boss Room",
        "Ancient Cistern - Boss Room - Statue",
        "Ancient Cistern - Statue",
        "Ancient Cistern - Statue - Boss Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Boss Room",
        "Ancient Cistern - Boss Room - Flame Room",
        "Ancient Cistern - Flame Room",
        "Ancient Cistern - Flame Room - Boss Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Flame Room",
        "Ancient Cistern - Flame Room - Boss Room",
        "Ancient Cistern - Boss Room",
        "Ancient Cistern - Boss Room - Flame Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Ancient Cistern - Flame Room",
        "Ancient Cistern - Flame Room - Strike Crest",
        "Lake Floria - Ancient Cistern Ledge",
        None,
        1, True, 6,
    ),
    SSAPEntranceData(
        "Sandship - Deck",
        "Sandship - Deck - Dungeon Exit",
        "Lanayru Sand Sea - Sea",
        "Lanayru Sand Sea - Sea - Dungeon Entrance in Lanayru Sand Sea",
        2, True, 4,
    ),
    SSAPEntranceData(
        "Sandship - Deck",
        "Sandship - Deck - Corridor",
        "Sandship - Corridor",
        "Sandship - Corridor - Deck",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sandship - Deck",
        "Sandship - Deck - Boss Key Room",
        "Sandship - Boss Key Room",
        "Sandship - Boss Key Room - Deck",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sandship - Corridor",
        "Sandship - Corridor - Dungeon Exit",
        "Lanayru Sand Sea - Sea",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Sandship - Corridor",
        "Sandship - Corridor - Deck",
        "Sandship - Deck",
        "Sandship - Deck - Corridor",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sandship - Corridor",
        "Sandship - Corridor - Near Boss Door",
        "Sandship - Near Boss Door",
        "Sandship - Near Boss Door - Corridor",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sandship - Corridor",
        "Sandship - Corridor - Basement",
        "Sandship - Basement",
        "Sandship - Basement - Corridor",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sandship - Corridor",
        "Sandship - Corridor - Bow",
        "Sandship - Bow",
        "Sandship - Bow - Corridor",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sandship - Near Boss Door",
        "Sandship - Near Boss Door - Dungeon Exit",
        "Lanayru Sand Sea - Sea",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Sandship - Near Boss Door",
        "Sandship - Near Boss Door - Corridor",
        "Sandship - Corridor",
        "Sandship - Corridor - Near Boss Door",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sandship - Near Boss Door",
        "Sandship - Near Boss Door - Boss Room",
        "Sandship - Boss Room",
        "Sandship - Boss Room - Near Boss Door",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sandship - Basement",
        "Sandship - Basement - Corridor",
        "Sandship - Corridor",
        "Sandship - Corridor - Basement",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sandship - Bow",
        "Sandship - Bow - Corridor",
        "Sandship - Corridor",
        "Sandship - Corridor - Bow",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sandship - Boss Key Room",
        "Sandship - Boss Key Room - Deck",
        "Sandship - Deck",
        "Sandship - Deck - Boss Key Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sandship - Boss Room",
        "Sandship - Boss Room - Near Boss Door",
        "Sandship - Near Boss Door",
        "Sandship - Near Boss Door - Boss Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sandship - Boss Room",
        "Sandship - Boss Room - Strike Crest",
        "Lanayru Sand Sea - Ancient Harbor",
        None,
        1, True, 6,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - First Room",
        "Fire Sanctuary - First Room - Dungeon Exit",
        "Volcano Summit - After Second Frog",
        "Volcano Summit - After Second Frog - Dungeon Entrance in Volcano Summit",
        2, True, 4,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - First Room",
        "Fire Sanctuary - First Room - Second Room",
        "Fire Sanctuary - Second Room",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Second Room",
        "Fire Sanctuary - Second Room - Dungeon Exit",
        "Volcano Summit - After Second Frog",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Second Room",
        "Fire Sanctuary - Second Room - First Bridge",
        "Fire Sanctuary - First Bridge",
        "Fire Sanctuary - First Bridge - Second Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - First Bridge",
        "Fire Sanctuary - First Bridge - Second Room",
        "Fire Sanctuary - Second Room",
        "Fire Sanctuary - Second Room - First Bridge",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - First Bridge",
        "Fire Sanctuary - First Bridge - Flowing Lava Room",
        "Fire Sanctuary - Flowing Lava Room",
        "Fire Sanctuary - Flowing Lava Room - First Bridge",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Flowing Lava Room",
        "Fire Sanctuary - Flowing Lava Room - First Bridge",
        "Fire Sanctuary - First Bridge",
        "Fire Sanctuary - First Bridge - Flowing Lava Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Flowing Lava Room",
        "Fire Sanctuary - Flowing Lava Room - Upper First Trapped Mogma Room",
        "Fire Sanctuary - Upper First Trapped Mogma Room",
        "Fire Sanctuary - Upper First Trapped Mogma Room - Flowing Lava Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Flowing Lava Room",
        "Fire Sanctuary - Flowing Lava Room - Water Fruit Room",
        "Fire Sanctuary - Water Fruit Room",
        "Fire Sanctuary - Water Fruit Room - Flowing Lava Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Water Fruit Room",
        "Fire Sanctuary - Water Fruit Room - Dungeon Exit",
        "Volcano Summit - After Second Frog",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Water Fruit Room",
        "Fire Sanctuary - Water Fruit Room - Flowing Lava Room",
        "Fire Sanctuary - Flowing Lava Room",
        "Fire Sanctuary - Flowing Lava Room - Water Fruit Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Water Fruit Room",
        "Fire Sanctuary - Water Fruit Room - Lower First Trapped Mogma Room",
        "Fire Sanctuary - Lower First Trapped Mogma Room",
        "Fire Sanctuary - Lower First Trapped Mogma Room - Water Fruit Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Upper First Trapped Mogma Room",
        "Fire Sanctuary - Upper First Trapped Mogma Room - Flowing Lava Room",
        "Fire Sanctuary - Flowing Lava Room",
        "Fire Sanctuary - Flowing Lava Room - Upper First Trapped Mogma Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Lower First Trapped Mogma Room",
        "Fire Sanctuary - Lower First Trapped Mogma Room - Water Fruit Room",
        "Fire Sanctuary - Water Fruit Room",
        "Fire Sanctuary - Water Fruit Room - Lower First Trapped Mogma Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Lower First Trapped Mogma Room",
        "Fire Sanctuary - Lower First Trapped Mogma Room - Under Double Magmanos Fight",
        "Fire Sanctuary - Under Double Magmanos Fight",
        "Fire Sanctuary - Under Double Magmanos Fight - Lower First Trapped Mogma Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Under Double Magmanos Fight",
        "Fire Sanctuary - Under Double Magmanos Fight - Dungeon Exit",
        "Volcano Summit - After Second Frog",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Under Double Magmanos Fight",
        "Fire Sanctuary - Under Double Magmanos Fight - Lower First Trapped Mogma Room",
        "Fire Sanctuary - Lower First Trapped Mogma Room",
        "Fire Sanctuary - Lower First Trapped Mogma Room - Under Double Magmanos Fight",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Under Double Magmanos Fight",
        "Fire Sanctuary - Under Double Magmanos Fight - Second Bridge",
        "Fire Sanctuary - Second Bridge",
        "Fire Sanctuary - Second Bridge - Under Double Magmanos Fight",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Under Double Magmanos Fight",
        "Fire Sanctuary - Under Double Magmanos Fight - Boss Door Area",
        "Fire Sanctuary - Boss Door Area",
        "Fire Sanctuary - Boss Door Area - Under Double Magmanos Fight",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Second Bridge",
        "Fire Sanctuary - Second Bridge - Under Double Magmanos Fight",
        "Fire Sanctuary - Under Double Magmanos Fight",
        "Fire Sanctuary - Under Double Magmanos Fight - Second Bridge",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Second Bridge",
        "Fire Sanctuary - Second Bridge - Second Trapped Mogma Room",
        "Fire Sanctuary - Second Trapped Mogma Room",
        "Fire Sanctuary - Second Trapped Mogma Room - Second Bridge",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Second Trapped Mogma Room",
        "Fire Sanctuary - Second Trapped Mogma Room - Second Bridge",
        "Fire Sanctuary - Second Bridge",
        "Fire Sanctuary - Second Bridge - Second Trapped Mogma Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Boss Door Area",
        "Fire Sanctuary - Boss Door Area - Dungeon Exit",
        "Volcano Summit - After Second Frog",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Boss Door Area",
        "Fire Sanctuary - Boss Door Area - Under Double Magmanos Fight",
        "Fire Sanctuary - Under Double Magmanos Fight",
        "Fire Sanctuary - Under Double Magmanos Fight - Boss Door Area",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Boss Door Area",
        "Fire Sanctuary - Boss Door Area - Staircase Room",
        "Fire Sanctuary - Staircase Room",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Boss Door Area",
        "Fire Sanctuary - Boss Door Area - Boss Room",
        "Fire Sanctuary - Boss Room",
        "Fire Sanctuary - Boss Room - Boss Door Area",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Staircase Room",
        "Fire Sanctuary - Staircase Room - Boss Key Room",
        "Fire Sanctuary - Boss Key Room",
        "Fire Sanctuary - Boss Key Room - Staircase Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Boss Key Room",
        "Fire Sanctuary - Boss Key Room - Staircase Room",
        "Fire Sanctuary - Staircase Room",
        "Fire Sanctuary - Staircase Room - Boss Key Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Boss Key Room",
        "Fire Sanctuary - Boss Key Room - Boss Door Area",
        "Fire Sanctuary - Boss Door Area",
        None,
        1, False, None,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Boss Room",
        "Fire Sanctuary - Boss Room - Boss Door Area",
        "Fire Sanctuary - Boss Door Area",
        "Fire Sanctuary - Boss Door Area - Boss Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Boss Room",
        "Fire Sanctuary - Boss Room - Flame Room",
        "Fire Sanctuary - Flame Room",
        "Fire Sanctuary - Flame Room - Boss Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Flame Room",
        "Fire Sanctuary - Flame Room - Boss Room",
        "Fire Sanctuary - Boss Room",
        "Fire Sanctuary - Boss Room - Flame Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Fire Sanctuary - Flame Room",
        "Fire Sanctuary - Flame Room - Strike Crest",
        "Volcano Summit - After Second Frog",
        None,
        1, True, 6,
    ),
    SSAPEntranceData(
        "Sky Keep - First Room",
        "Sky Keep - First Room - Dungeon Exit",
        "Central Skyloft",
        "Central Skyloft - Dungeon Entrance on Skyloft",
        2, True, 4,
    ),
    SSAPEntranceData(
        "Sky Keep - First Room",
        "Sky Keep - First Room - Skyview Room",
        "Sky Keep - Skyview Room",
        "Sky Keep - Skyview Room - First Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky Keep - Skyview Room",
        "Sky Keep - Skyview Room - Dungeon Exit",
        "Central Skyloft",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Sky Keep - Skyview Room",
        "Sky Keep - Skyview Room - First Room",
        "Sky Keep - First Room",
        "Sky Keep - First Room - Skyview Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky Keep - Skyview Room",
        "Sky Keep - Skyview Room - Lanayru Mining Facility Room",
        "Sky Keep - Lanayru Mining Facility Room",
        "Sky Keep - Lanayru Mining Facility Room - Skyview Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky Keep - Lanayru Mining Facility Room",
        "Sky Keep - Lanayru Mining Facility Room - Dungeon Exit",
        "Central Skyloft",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Sky Keep - Lanayru Mining Facility Room",
        "Sky Keep - Lanayru Mining Facility Room - Skyview Room",
        "Sky Keep - Skyview Room",
        "Sky Keep - Skyview Room - Lanayru Mining Facility Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky Keep - Lanayru Mining Facility Room",
        "Sky Keep - Lanayru Mining Facility Room - Earth Temple Room",
        "Sky Keep - Earth Temple Room",
        "Sky Keep - Earth Temple Room - Lanayru Mining Facility Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky Keep - Lanayru Mining Facility Room",
        "Sky Keep - Lanayru Mining Facility Room - Ancient Cistern Room",
        "Sky Keep - Ancient Cistern Room",
        "Sky Keep - Ancient Cistern Room - Lanayru Mining Facility Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky Keep - Earth Temple Room",
        "Sky Keep - Earth Temple Room - Dungeon Exit",
        "Central Skyloft",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Sky Keep - Earth Temple Room",
        "Sky Keep - Earth Temple Room - Lanayru Mining Facility Room",
        "Sky Keep - Lanayru Mining Facility Room",
        "Sky Keep - Lanayru Mining Facility Room - Earth Temple Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky Keep - Earth Temple Room",
        "Sky Keep - Earth Temple Room - Miniboss Room",
        "Sky Keep - Miniboss Room",
        "Sky Keep - Miniboss Room - Earth Temple Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky Keep - Ancient Cistern Room",
        "Sky Keep - Ancient Cistern Room - Lanayru Mining Facility Room",
        "Sky Keep - Lanayru Mining Facility Room",
        "Sky Keep - Lanayru Mining Facility Room - Ancient Cistern Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky Keep - Ancient Cistern Room",
        "Sky Keep - Ancient Cistern Room - Fire Sanctuary Room",
        "Sky Keep - Fire Sanctuary Room",
        "Sky Keep - Fire Sanctuary Room - Ancient Cistern Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky Keep - Miniboss Room",
        "Sky Keep - Miniboss Room - Earth Temple Room",
        "Sky Keep - Earth Temple Room",
        "Sky Keep - Earth Temple Room - Miniboss Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky Keep - Miniboss Room",
        "Sky Keep - Miniboss Room - Fire Sanctuary Room",
        "Sky Keep - Fire Sanctuary Room",
        "Sky Keep - Fire Sanctuary Room - Miniboss Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky Keep - Fire Sanctuary Room",
        "Sky Keep - Fire Sanctuary Room - Dungeon Exit",
        "Central Skyloft",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Sky Keep - Fire Sanctuary Room",
        "Sky Keep - Fire Sanctuary Room - Ancient Cistern Room",
        "Sky Keep - Ancient Cistern Room",
        "Sky Keep - Ancient Cistern Room - Fire Sanctuary Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky Keep - Fire Sanctuary Room",
        "Sky Keep - Fire Sanctuary Room - Miniboss Room",
        "Sky Keep - Miniboss Room",
        "Sky Keep - Miniboss Room - Fire Sanctuary Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky Keep - Fire Sanctuary Room",
        "Sky Keep - Fire Sanctuary Room - Sandship Room",
        "Sky Keep - Sandship Room",
        "Sky Keep - Sandship Room - Fire Sanctuary Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Sky Keep - Sandship Room",
        "Sky Keep - Sandship Room - Dungeon Exit",
        "Central Skyloft",
        None,
        1, True, 4,
    ),
    SSAPEntranceData(
        "Sky Keep - Sandship Room",
        "Sky Keep - Sandship Room - Fire Sanctuary Room",
        "Sky Keep - Fire Sanctuary Room",
        "Sky Keep - Fire Sanctuary Room - Sandship Room",
        2, True, 1,
    ),
    SSAPEntranceData(
        "Skyloft Silent Realm",
        "Skyloft Silent Realm - Trial Gate",
        "Central Skyloft",
        "Central Skyloft - Trial Gate on Skyloft",
        2, True, 3,
    ),
    SSAPEntranceData(
        "Faron Silent Realm",
        "Faron Silent Realm - Trial Gate",
        "Faron Woods - Viewing Platform",
        "Faron Woods - Viewing Platform - Trial Gate in Faron Woods",
        2, True, 3,
    ),
    SSAPEntranceData(
        "Eldin Silent Realm",
        "Eldin Silent Realm - Trial Gate",
        "Eldin Volcano - Volcano Ascent",
        "Eldin Volcano - Volcano Ascent - Trial Gate in Eldin Volcano",
        2, True, 3,
    ),
    SSAPEntranceData(
        "Lanayru Silent Realm",
        "Lanayru Silent Realm - Trial Gate",
        "Lanayru Desert - North Desert",
        "Lanayru Desert - North Desert - Trial Gate in Lanayru Desert",
        2, True, 3,
    ),
]

class SSEntranceData(NamedTuple):
    """
    Class contains data for a game entrance in SS.

    name: Name of the entrance, as recognized by the patcher.
    statue_name: Statue name or `Link's Room`, as recognized by the patcher.
    province: The province that the entrance leads to.
    stage: The game stage name that the entrance leads to.
    room: The game room int that the entrance leads to.
    layer: The game layer int that the entrance leads to.
    entrance: The game entrance int that the entrance leads to.
    tod: The time of day that the entrance must be forced to.
    type: The type of the entrance. `Entrance`, `Statue`, or `Vanilla`.
    apregion: The name of the Archipelago region that the entrance leads to.
    """

    name: str
    statue_name: Optional[str]
    province: str
    stage: str
    room: int
    layer: int
    entrance: int
    tod: int
    type: str
    apregion: str
    flag_space: Optional[str] = None
    flag: Optional[int] = None
    vanilla_statue: Optional[bool] = None

GAME_ENTRANCE_TABLE = [
    SSEntranceData(
        "Bamboo Island - Entrance from Inside",
        None,
        "The Sky",
        "F020", 0, 0, 11, 2,
        "Entrance", "Sky",
    ),
    SSEntranceData(
        "Bamboo Island - Inside - Entrance",
        None,
        "The Sky",
        "F019r", 0, 1, 0, 2,
        "Entrance", "Sky",
    ),
    SSEntranceData(
        "Batreaux's House - Entrance",
        None,
        "The Sky",
        "F012r", 0, 0, 0, 2,
        "Entrance", "Batreaux's House",
    ),
    SSEntranceData(
        "Batreaux's House Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 29, 2,
        "Entrance", "Skyloft Village",
    ),
    SSEntranceData(
        "Bazaar - North Entrance",
        None,
        "The Sky",
        "F004r", 0, 0, 0, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Bazaar - South Entrance",
        None,
        "The Sky",
        "F004r", 0, 0, 1, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Bazaar - West Entrance",
        None,
        "The Sky",
        "F004r", 0, 0, 2, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Bazaar North Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 2, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Bazaar South Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 3, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Bazaar West Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 20, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Beedle's Shop - Day Entrance",
        None,
        "The Sky",
        "F002r", 0, 0, 0, 2,
        "Entrance", "Beedle's Shop",
    ),
    SSEntranceData(
        "Beedle's Shop - Night Entrance",
        None,
        "The Sky",
        "F002r", 0, 0, 0, 1,
        "Entrance", "Beedle's Shop",
    ),
    SSEntranceData(
        "Behind the Temple - Door Entrance",
        None,
        "Faron Province",
        "F400", 0, 0, 1, 2,
        "Entrance", "Sealed Grounds - Behind the Temple",
    ),
    SSEntranceData(
        "Behind the Temple - Entrance from Faron Woods",
        None,
        "Faron Province",
        "F400", 0, 0, 3, 2,
        "Entrance", "Sealed Grounds - Behind the Temple",
    ),
    SSEntranceData(
        "Behind the Temple - Statue Entrance",
        "Behind the Temple",
        "Faron Province",
        "F400", 0, 0, 10, 2,
        "Statue", "Sealed Grounds - Behind the Temple",
        "Sealed Grounds", 31, False,
    ),
    SSEntranceData(
        "Bertie's House - Entrance",
        None,
        "The Sky",
        "F014r", 0, 0, 1, 2,
        "Entrance", "Skyloft Village",
    ),
    SSEntranceData(
        "Bertie's House Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 34, 2,
        "Entrance", "Skyloft Village",
    ),
    SSEntranceData(
        "Bokoblin Base - Entrance",
        None,
        "Eldin Province",
        "F202", 1, 1, 0, 2,
        "Entrance", "Bokoblin Base - Prison",
    ),
    SSEntranceData(
        "Deep Woods - Deep Woods Statue Entrance",
        "Deep Woods",
        "Faron Province",
        "F101", 0, 0, 14, 2,
        "Statue", "Faron Woods - Deep Woods after Beehive",
        "Faron Woods", 103, False,
    ),
    SSEntranceData(
        "Deep Woods - Entrance from Faron Woods",
        None,
        "Faron Province",
        "F101", 0, 0, 0, 2,
        "Entrance", "Faron Woods - Deep Woods Entry",
    ),
    SSEntranceData(
        "Deep Woods - Forest Temple Statue Entrance",
        "Forest Temple",
        "Faron Province",
        "F101", 0, 0, 13, 2,
        "Statue", "Faron Woods - Deep Woods after Beehive",
        "Faron Woods", 104, False,
    ),
    SSEntranceData(
        "Eldin Volcano - Entrance from Bokoblin Base",
        None,
        "Eldin Province",
        "F200", 1, 1, 5, 2,
        "Entrance", "Eldin Volcano - Volcano Entry",
    ),
    SSEntranceData(
        "Eldin Volcano - Entrance from Earth Temple",
        None,
        "Eldin Province",
        "F200", 4, 0, 1, 2,
        "Entrance", "Eldin Volcano - Near Temple Entrance",
    ),
    SSEntranceData(
        "Eldin Volcano - First Time Entrance",
        None,
        "Eldin Province",
        "F200", 0, 0, 0, 2,
        "Entrance", "Eldin Volcano - Volcano Entry",
    ),
    SSEntranceData(
        "Eldin Volcano - First Vent from Mogma Turf",
        None,
        "Eldin Province",
        "F200", 2, 0, 1, 2,
        "Entrance", "Eldin Volcano - Volcano East",
    ),
    SSEntranceData(
        "Eldin Volcano - Second Vent from Mogma Turf",
        None,
        "Eldin Province",
        "F200", 2, 0, 2, 2,
        "Entrance", "Eldin Volcano - Volcano Ascent",
    ),
    SSEntranceData(
        "Eldin Volcano - Temple Entrance Statue Entrance",
        "Temple Entrance",
        "Eldin Province",
        "F200", 4, 0, 7, 2,
        "Statue", "Eldin Volcano - Near Temple Entrance",
        "Story", 807, False,
    ),
    SSEntranceData(
        "Eldin Volcano - Thrill Digger Cave Entrance",
        None,
        "Eldin Province",
        "F200", 4, 0, 2, 2,
        "Entrance", "Eldin Volcano - Thrill Digger Cliff",
    ),
    SSEntranceData(
        "Eldin Volcano - Volcano Ascent Statue Entrance",
        "Volcano Ascent",
        "Eldin Province",
        "F200", 2, 0, 7, 2,
        "Statue", "Eldin Volcano - Volcano Ascent",
        "Story", 806, False,
    ),
    SSEntranceData(
        "Eldin Volcano - Volcano East Statue Entrance",
        "Volcano East",
        "Eldin Province",
        "F200", 2, 0, 6, 2,
        "Statue", "Eldin Volcano - Volcano East",
        "Story", 805, False,
    ),
    SSEntranceData(
        "Eldin Volcano - Volcano Entrance Statue Entrance",
        "Volcano Entry",
        "Eldin Province",
        "F200", 0, 0, 2, 2,
        "Statue", "Eldin Volcano - Volcano Ascent",
        "Story", 804, True,
    ),
    SSEntranceData(
        "Faron Woods - Entrance from Behind the Temple",
        None,
        "Faron Province",
        "F100", 0, 0, 0, 2,
        "Entrance", "Faron Woods - Forest Entry",
    ),
    SSEntranceData(
        "Faron Woods - Entrance from Deep Woods",
        None,
        "Faron Province",
        "F100", 0, 0, 45, 2,
        "Entrance", "Faron Woods - Viewing Platform",
    ),
    SSEntranceData(
        "Faron Woods - Faron Woods Entry Statue Entrance",
        "Faron Woods Entry",
        "Faron Province",
        "F100", 0, 0, 50, 2,
        "Statue", "Faron Woods - Forest Entry",
        "Story", 800, False,
    ),
    SSEntranceData(
        "Faron Woods - In the Woods Statue Entrance",
        "In the Woods",
        "Faron Province",
        "F100", 0, 0, 51, 2,
        "Statue", "Faron Woods - In the Woods",
        "Story", 801, False,
    ),
    SSEntranceData(
        "Faron Woods - Shortcut Entrance from Floria Waterfall",
        None,
        "Faron Province",
        "F100", 0, 0, 17, 2,
        "Entrance", "Faron Woods - In the Woods",
    ),
    SSEntranceData(
        "Faron Woods - Viewing Platform Statue Entrance",
        "Viewing Platform",
        "Faron Province",
        "F100", 0, 0, 52, 2,
        "Statue", "Faron Woods - Viewing Platform",
        "Story", 802, False,
    ),
    SSEntranceData(
        "Farore's Lair - Entrance from Waterfall",
        None,
        "Faron Province",
        "F102_2", 0, 0, 1, 2,
        "Entrance", "Lake Floria - Dragon's Lair",
    ),
    SSEntranceData(
        "Fire Node - Entrance",
        None,
        "Lanayru Province",
        "F300_3", 0, 0, 0, 2,
        "Entrance", "Lanayru Desert - Fire Node",
    ),
    SSEntranceData(
        "Flooded Faron Woods - Entrance from Upper Flooded Great Tree",
        None,
        "Faron Province",
        "F103", 0, 0, 1, 2,
        "Entrance", "Flooded Faron Woods",
    ),
    SSEntranceData(
        "Flooded Great Tree - Entrance from Upper Flooded Faron Woods",
        None,
        "Faron Province",
        "F103_1", 0, 0, 0, 2,
        "Entrance", "Flooded Faron Woods",
    ),
    SSEntranceData(
        "Floria Waterfall - Entrance from Ancient Cistern",
        "Floria Waterfall",
        "Faron Province",
        "F102_1", 0, 0, 1, 2,
        "Entrance", "Lake Floria - Ancient Cistern Ledge",
    ),
    SSEntranceData(
        "Floria Waterfall - Entrance from Faron Woods",
        None,
        "Faron Province",
        "F102_1", 0, 0, 2, 2,
        "Entrance", "Lake Floria - Waterfall",
    ),
    SSEntranceData(
        "Floria Waterfall - Entrance from Farore's Lair",
        None,
        "Faron Province",
        "F102_1", 0, 0, 0, 2,
        "Entrance", "Lake Floria - Waterfall",
    ),
    SSEntranceData(
        "Floria Waterfall - Statue Entrance",
        "Floria Waterfall",
        "Faron Province",
        "F102_1", 0, 0, 5, 2,
        "Statue", "Lake Floria - Waterfall",
        "Lake Floria", 33, False,
    ),
    SSEntranceData(
        "Goddess Statue - Entrance",
        None,
        "The Sky",
        "F008r", 0, 0, 0, 2,
        "Entrance", "Upper Skyloft",
    ),
    SSEntranceData(
        "Goddess Statue Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 6, 2,
        "Entrance", "Upper Skyloft",
    ),
    SSEntranceData(
        "Gondo's House - Entrance",
        None,
        "The Sky",
        "F015r", 0, 0, 1, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Gondo's House Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 35, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Great Tree - Entrance from Top",
        None,
        "Faron Province",
        "F100_1", 0, 0, 3, 2,
        "Entrance", "Faron Woods - Inside Upper Great Tree",
    ),
    SSEntranceData(
        "Great Tree - Lower Entrance from Platforms",
        None,
        "Faron Province",
        "F100_1", 0, 0, 1, 2,
        "Entrance", "Faron Woods - Inside Lower Great Tree - After Swinging Platforms",
    ),
    SSEntranceData(
        "Great Tree - Platforms - Lower Entrance from Great Tree",
        None,
        "Faron Province",
        "F100", 0, 0, 4, 2,
        "Entrance", "Faron Woods - Great Tree Platforms",
    ),
    SSEntranceData(
        "Great Tree - Platforms - Upper Entrance from Great Tree",
        None,
        "Faron Province",
        "F100", 0, 0, 5, 2,
        "Entrance", "Faron Woods - Great Tree Platforms",
    ),
    SSEntranceData(
        "Great Tree - The Great Tree Statue Entrance",
        "Great Tree Top",
        "Faron Province",
        "F100", 0, 0, 53, 2,
        "Statue", "Faron Woods - Outside Top of Great Tree",
        "Story", 803, False,
    ),
    SSEntranceData(
        "Great Tree - Top Entrance from Great Tree",
        None,
        "Faron Province",
        "F100", 0, 0, 6, 2,
        "Entrance", "Faron Woods - Outside Top of Great Tree",
    ),
    SSEntranceData(
        "Great Tree - Upper Entrance from Platforms",
        None,
        "Faron Province",
        "F100_1", 0, 0, 2, 2,
        "Entrance", "Faron Woods - Inside Upper Great Tree",
    ),
    SSEntranceData(
        "Lake Floria - Below Rock - Statue Entrance",
        "Lake Floria",
        "Faron Province",
        "F102", 3, 0, 2, 2,
        "Statue", "Lake Floria - Higher Ledge in Lake",
        "Lake Floria", 32, False,
    ),
    SSEntranceData(
        "Lanayru Caves - Entrance from Desert",
        None,
        "Lanayru Province",
        "F303", 0, 0, 1, 2,
        "Entrance", "Lanayru Caves - Caves",
    ),
    SSEntranceData(
        "Lanayru Caves - Entrance from Mine",
        None,
        "Lanayru Province",
        "F303", 0, 0, 0, 2,
        "Entrance", "Lanayru Caves - Caves",
    ),
    SSEntranceData(
        "Lanayru Desert - Desert Entrance Statue Entrance",
        "Desert Entrance",
        "Lanayru Province",
        "F300", 0, 0, 15, 2,
        "Statue", "Lanayru Desert - Desert Entry",
        "Lanayru Desert", 66, False,
    ),
    SSEntranceData(
        "Lanayru Desert - Entrance from Caves",
        None,
        "Lanayru Province",
        "F300", 0, 0, 6, 2,
        "Entrance", "Lanayru Desert - Higher Ledge in Sand Oasis",
    ),
    SSEntranceData(
        "Lanayru Desert - Entrance from Mine",
        None,
        "Lanayru Province",
        "F300", 0, 0, 2, 2,
        "Entrance", "Lanayru Desert - Desert Entry",
    ),
    SSEntranceData(
        "Lanayru Desert - Fire Node Entrance",
        None,
        "Lanayru Province",
        "F300", 0, 0, 3, 2,
        "Entrance", "Lanayru Desert - Stone Cache",
    ),
    SSEntranceData(
        "Lanayru Desert - Lightning Node Entrance",
        None,
        "Lanayru Province",
        "F300", 0, 0, 7, 2,
        "Entrance", "Lanayru Desert - North Desert",
    ),
    SSEntranceData(
        "Lanayru Desert - North Desert Statue Entrance",
        "North Desert",
        "Lanayru Province",
        "F300", 0, 0, 17, 2,
        "Statue", "Lanayru Desert - North Desert",
        "Lanayru Desert", 67, False,
    ),
    SSEntranceData(
        "Lanayru Desert - North Entrance from Temple of Time",
        None,
        "Lanayru Province",
        "F300", 0, 0, 1, 2,
        "Entrance", "Lanayru Desert - North Desert",
    ),
    SSEntranceData(
        "Lanayru Desert - South Entrance from Temple of Time",
        None,
        "Lanayru Province",
        "F300", 0, 0, 0, 2,
        "Entrance", "Lanayru Desert - Sand Oasis",
    ),
    SSEntranceData(
        "Lanayru Desert - Stone Cache Statue Entrance",
        "Stone Cache",
        "Lanayru Province",
        "F300", 0, 0, 18, 2,
        "Statue", "Lanayru Desert - Stone Cache",
        "Lanayru Desert", 2, False,
    ),
    SSEntranceData(
        "Lanayru Desert - West Desert Statue Entrance",
        "West Desert",
        "Lanayru Province",
        "F300", 0, 0, 16, 2,
        "Statue", "Lanayru Desert - Sand Oasis",
        "Lanayru Desert", 51, False,
    ),
    SSEntranceData(
        "Lanayru Gorge - Entrance",
        None,
        "Lanayru Province",
        "F302", 0, 2, 0, 2,
        "Entrance", "Lanayru Gorge - Gorge",
    ),
    SSEntranceData(
        "Lanayru Gorge - Statue Entrance",
        "Lanayru Gorge",
        "Lanayru Province",
        "F302", 0, 0, 12, 2,
        "Statue", "Lanayru Gorge - Gorge",
        "Lanayru Gorge", 12, False,
    ),
    SSEntranceData(
        "Lanayru Mine - Entrance from Caves",
        None,
        "Lanayru Province",
        "F300_1", 0, 0, 1, 2,
        "Entrance", "Lanayru Mine - Door to Caves",
    ),
    SSEntranceData(
        "Lanayru Mine - Entrance from Desert",
        None,
        "Lanayru Province",
        "F300_1", 2, 0, 1, 2,
        "Entrance", "Lanayru Mine - Ending",
    ),
    SSEntranceData(
        "Lanayru Mine - First Time Entrance",
        None,
        "Lanayru Province",
        "F300_1", 0, 0, 0, 2,
        "Entrance", "Lanayru Mine - Landing",
    ),
    SSEntranceData(
        "Lanayru Mine - Statue Entrance",
        "Lanayru Mine Entry",
        "Lanayru Province",
        "F300_1", 0, 0, 2, 2,
        "Statue", "Lanayru Mine - Landing",
        "Lanayru Desert", 68, True,
    ),
    SSEntranceData(
        "Lanayru Sand Sea - Ancient Harbour Dock Entrance",
        None,
        "Lanayru Province",
        "F301_1", 0, 0, 0, 2,
        "Entrance", "Lanayru Sand Sea - Sea",
    ),
    SSEntranceData(
        "Lanayru Sand Sea - Pirate Stronghold Dock Entrance",
        None,
        "Lanayru Province",
        "F301_1", 0, 0, 3, 2,
        "Entrance", "Lanayru Sand Sea - Sea",
    ),
    SSEntranceData(
        "Lanayru Sand Sea - Shipyard Dock Entrance",
        None,
        "Lanayru Province",
        "F301_1", 0, 0, 2, 2,
        "Entrance", "Lanayru Sand Sea - Sea",
    ),
    SSEntranceData(
        "Lanayru Sand Sea - Skipper's Retreat Dock Entrance",
        None,
        "Lanayru Province",
        "F301_1", 0, 0, 1, 2,
        "Entrance", "Lanayru Sand Sea - Sea",
    ),
    SSEntranceData(
        "Lanayru Sand Sea Docks - Dock Entrance",
        None,
        "Lanayru Province",
        "F301", 0, 0, 1, 2,
        "Entrance", "Lanayru Sand Sea - Ancient Harbor",
    ),
    SSEntranceData(
        "Lanayru Sand Sea Docks - Statue Entrance",
        "Ancient Harbour",
        "Lanayru Province",
        "F301", 0, 0, 10, 2,
        "Statue", "Lanayru Sand Sea - Ancient Harbor",
        "Lanayru Sand Sea", 10, False,
    ),
    SSEntranceData(
        "Lightning Node - Entrance",
        None,
        "Lanayru Province",
        "F300_2", 0, 0, 0, 2,
        "Entrance", "Lanayru Desert - Lightning Node",
    ),
    SSEntranceData(
        "Lumpy Pumpkin - Back Door Entrance",
        None,
        "The Sky",
        "F020", 0, 0, 24, 2,
        "Entrance", "Sky",
    ),
    SSEntranceData(
        "Lumpy Pumpkin - Main Left Door Entrance",
        None,
        "The Sky",
        "F020", 0, 0, 22, 2,
        "Entrance", "Sky",
    ),
    SSEntranceData(
        "Lumpy Pumpkin - Main Right Door Entrance",
        None,
        "The Sky",
        "F020", 0, 0, 23, 2,
        "Entrance", "Sky",
    ),
    SSEntranceData(
        "Lumpy Pumpkin Building - Back Door Entrance",
        None,
        "The Sky",
        "F011r", 0, 0, 3, 2,
        "Entrance", "Sky",
    ),
    SSEntranceData(
        "Lumpy Pumpkin Building - Main Left Door Entrance",
        None,
        "The Sky",
        "F011r", 0, 0, 2, 2,
        "Entrance", "Sky",
    ),
    SSEntranceData(
        "Lumpy Pumpkin Building - Main Right Door Entrance",
        None,
        "The Sky",
        "F011r", 0, 0, 1, 2,
        "Entrance", "Sky",
    ),
    SSEntranceData(
        "Mallara's House - Entrance",
        None,
        "The Sky",
        "F016r", 0, 0, 1, 2,
        "Entrance", "Skyloft Village",
    ),
    SSEntranceData(
        "Mogma Turf - Entrance",
        None,
        "Eldin Province",
        "F210", 0, 0, 0, 2,
        "Entrance", "Mogma Turf - Entry",
    ),
    SSEntranceData(
        "Orielle and Parrow's House - Entrance",
        None,
        "The Sky",
        "F005r", 0, 0, 1, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Orielle and Parrow's House Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 30, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Peatrice's House - Entrance",
        None,
        "The Sky",
        "F018r", 0, 0, 1, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Peatrice's House Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 38, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Piper's House - Entrance",
        None,
        "The Sky",
        "F007r", 0, 0, 1, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Piper's House Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 32, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Pirate Stronghold - Dock Entrance",
        None,
        "Lanayru Province",
        "F301_6", 0, 0, 0, 2,
        "Entrance", "Lanayru Sand Sea - Pirate Stronghold",
    ),
    SSEntranceData(
        "Pirate Stronghold - Inside - First Door Entrance",
        None,
        "Lanayru Province",
        "F301_2", 1, 0, 0, 2,
        "Entrance", "Lanayru Sand Sea - Pirate Stronghold",
    ),
    SSEntranceData(
        "Pirate Stronghold - Side Entrance",
        None,
        "Lanayru Province",
        "F301_6", 0, 0, 1, 2,
        "Entrance", "Lanayru Sand Sea - Pirate Stronghold",
    ),
    SSEntranceData(
        "Pirate Stronghold - Statue Entrance",
        "Pirate Stronghold",
        "Lanayru Province",
        "F301_6", 0, 0, 10, 2,
        "Statue", "Lanayru Sand Sea - Pirate Stronghold",
        "Lanayru Sand Sea", 84, False,
    ),
    SSEntranceData(
        "Rupin's House - Entrance",
        None,
        "The Sky",
        "F017r", 0, 0, 1, 2,
        "Entrance", "Skyloft Village",
    ),
    SSEntranceData(
        "Rupin's House Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 37, 2,
        "Entrance", "Skyloft Village",
    ),
    SSEntranceData(
        "Sealed Grounds Spiral - Door Entrance",
        None,
        "Faron Province",
        "F401", 1, 0, 2, 2,
        "Entrance", "Sealed Grounds - Spiral",
    ),
    SSEntranceData(
        "Sealed Grounds Spiral - Shortcut Entrance from Behind the Temple",
        None,
        "Faron Province",
        "F401", 1, 0, 0, 2,
        "Entrance", "Sealed Grounds - Spiral",
    ),
    SSEntranceData(
        "Sealed Grounds Spiral - Statue Entrance",
        "Sealed Grounds",
        "Faron Province",
        "F401", 1, 0, 8, 2,
        "Statue", "Sealed Grounds - Spiral",
        "Sealed Grounds", 35, True,
    ),
    SSEntranceData(
        "Sealed Temple - Main Entrance",
        None,
        "Faron Province",
        "F402", 2, 0, 0, 2,
        "Entrance", "Sealed Grounds - Sealed Temple",
    ),
    SSEntranceData(
        "Sealed Temple - Side Entrance",
        None,
        "Faron Province",
        "F402", 2, 0, 1, 2,
        "Entrance", "Sealed Grounds - Sealed Temple",
    ),
    SSEntranceData(
        "Shipyard - Dock Entrance",
        None,
        "Lanayru Province",
        "F301_4", 0, 0, 0, 2,
        "Entrance", "Lanayru Sand Sea - Shipyard",
    ),
    SSEntranceData(
        "Shipyard - Statue Entrance",
        "Lanayru Shipyard",
        "Lanayru Province",
        "F301_4", 0, 0, 10, 2,
        "Statue", "Lanayru Sand Sea - Shipyard",
        "Lanayru Sand Sea", 85, False,
    ),
    SSEntranceData(
        "Skipper's Retreat - Dock Entrance",
        None,
        "Lanayru Province",
        "F301_3", 0, 0, 0, 2,
        "Entrance", "Lanayru Sand Sea - Skipper's Retreat - Dock",
    ),
    SSEntranceData(
        "Skipper's Retreat - Shack - Entrance",
        None,
        "Lanayru Province",
        "F301_5", 0, 0, 0, 2,
        "Entrance", "Lanayru Sand Sea - Skipper's Retreat - Shack",
    ),
    SSEntranceData(
        "Skipper's Retreat - Shack Entrance",
        None,
        "Lanayru Province",
        "F301_3", 0, 0, 1, 2,
        "Entrance", "Lanayru Sand Sea - Skipper's Retreat - Shack",
    ),
    SSEntranceData(
        "Skipper's Retreat - Statue Entrance",
        "Skipper's Retreat",
        "Lanayru Province",
        "F301_3", 0, 0, 10, 2,
        "Statue", "Lanayru Sand Sea - Skipper's Retreat - Dock",
        "Lanayru Sand Sea", 28, False,
    ),
    SSEntranceData(
        "Sky - Beedle's Island - Beedle's Ship Entrance",
        None,
        "The Sky",
        "F020", 0, 4, 0, 1,
        "Entrance", "Beedle's Shop",
    ),
    SSEntranceData(
        "Sky - Eldin Pillar - Entrance",
        "Volcano Entry",
        "The Sky",
        "F020", 0, 0, 13, 2,
        "Entrance", "Sky",
    ),
    SSEntranceData(
        "Sky - Faron Pillar - Entrance",
        "Sealed Grounds",
        "The Sky",
        "F020", 0, 0, 12, 2,
        "Entrance", "Sky",
    ),
    SSEntranceData(
        "Sky - Lanayru Pillar - Entrance",
        None,
        "The Sky",
        "F020", 0, 0, 14, 2,
        "Entrance", "Sky",
    ),
    SSEntranceData(
        "Skyloft - Entrance from Beedle's Shop",
        None,
        "The Sky",
        "F000", 0, 0, 0, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Skyloft - Knight Academy - Chimney",
        None,
        "The Sky",
        "F001r", 6, 0, 1, 2,
        "Entrance", "Upper Skyloft - Zelda's Room",
    ),
    SSEntranceData(
        "Skyloft - Knight Academy - Lower Left Door Entrance",
        None,
        "The Sky",
        "F001r", 0, 0, 2, 2,
        "Entrance", "Upper Skyloft - Knight Academy",
    ),
    SSEntranceData(
        "Skyloft - Knight Academy - Lower Right Door Entrance",
        None,
        "The Sky",
        "F001r", 0, 0, 1, 2,
        "Entrance", "Upper Skyloft - Knight Academy",
    ),
    SSEntranceData(
        "Skyloft - Knight Academy - Start Entrance",
        "Link's Room",
        "The Sky",
        "F001r", 1, 0, 5, 2,
        "Vanilla", "Upper Skyloft - Knight Academy",
    ),
    SSEntranceData(
        "Skyloft - Knight Academy - Upper Left Door Entrance",
        None,
        "The Sky",
        "F001r", 0, 0, 4, 2,
        "Entrance", "Upper Skyloft - Knight Academy",
    ),
    SSEntranceData(
        "Skyloft - Knight Academy - Upper Right Door Entrance",
        None,
        "The Sky",
        "F001r", 0, 0, 3, 2,
        "Entrance", "Upper Skyloft - Knight Academy",
    ),
    SSEntranceData(
        "Skyloft - Knight Academy Lower Left Door Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 5, 2,
        "Entrance", "Upper Skyloft",
    ),
    SSEntranceData(
        "Skyloft - Knight Academy Lower Right Door Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 4, 2,
        "Entrance", "Upper Skyloft",
    ),
    SSEntranceData(
        "Skyloft - Knight Academy Upper Left Door Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 23, 2,
        "Entrance", "Upper Skyloft",
    ),
    SSEntranceData(
        "Skyloft - Knight Academy Upper Right Door Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 24, 2,
        "Entrance", "Upper Skyloft",
    ),
    SSEntranceData(
        "Skyloft - Mallara's House Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 36, 2,
        "Entrance", "Skyloft Village",
    ),
    SSEntranceData(
        "Skyloft - Sparring Hall Left Door Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 21, 2,
        "Entrance", "Upper Skyloft",
    ),
    SSEntranceData(
        "Skyloft - Sparring Hall Right Door Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 26, 2,
        "Entrance", "Upper Skyloft",
    ),
    SSEntranceData(
        "Sparring Hall - Left Door Entrance",
        None,
        "The Sky",
        "F009r", 0, 0, 2, 2,
        "Entrance", "Upper Skyloft",
    ),
    SSEntranceData(
        "Sparring Hall - Right Door Entrance",
        None,
        "The Sky",
        "F009r", 0, 0, 1, 2,
        "Entrance", "Upper Skyloft",
    ),
    SSEntranceData(
        "Sparrot's House - Entrance",
        None,
        "The Sky",
        "F013r", 0, 0, 1, 2,
        "Entrance", "Skyloft Village",
    ),
    SSEntranceData(
        "Sparrot's House Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 33, 2,
        "Entrance", "Skyloft Village",
    ),
    SSEntranceData(
        "Temple of Time - Desert Gorge Statue Entrance",
        "Desert Gorge",
        "Lanayru Province",
        "F300_4", 0, 0, 16, 2,
        "Statue", "Lanayru Desert - Temple of Time - Desert Gorge",
        "Lanayru Desert", 77, False,
    ),
    SSEntranceData(
        "Temple of Time - North Entrance from Desert",
        None,
        "Lanayru Province",
        "F300_4", 0, 0, 1, 2,
        "Entrance", "Lanayru Desert - Temple of Time - Temple",
    ),
    SSEntranceData(
        "Temple of Time - South Entrance from Desert",
        None,
        "Lanayru Province",
        "F300_4", 0, 0, 0, 2,
        "Entrance", "Lanayru Desert - Temple of Time - Desert Gorge",
    ),
    SSEntranceData(
        "Temple of Time - Temple of Time Statue Entrance",
        "Temple of Time",
        "Lanayru Province",
        "F300_4", 0, 0, 17, 2,
        "Statue", "Lanayru Desert - Temple of Time - Temple",
        "Lanayru Desert", 78, False,
    ),
    SSEntranceData(
        "Thrill Digger Cave - Entrance",
        None,
        "Eldin Province",
        "F211", 0, 1, 0, 2,
        "Entrance", "Eldin Volcano - Thrill Digger Cave",
    ),
    SSEntranceData(
        "Waterfall Cave - Lower Entrance",
        None,
        "The Sky",
        "D000", 0, 0, 1, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Waterfall Cave - Upper Entrance",
        None,
        "The Sky",
        "D000", 0, 0, 0, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Waterfall Cave Lower Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 8, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Waterfall Cave Upper Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 7, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Wryna's House - Entrance",
        None,
        "The Sky",
        "F006r", 0, 0, 1, 2,
        "Entrance", "Central Skyloft",
    ),
    SSEntranceData(
        "Wryna's House Entrance",
        None,
        "The Sky",
        "F000", 0, 0, 31, 2,
        "Entrance", "Central Skyloft",
    ),
]

AP_EXIT_GRAPH = {
    "Upper Skyloft": {
        "Knight Academy": "Upper Skyloft - Knight Academy",
        "Zelda's Room": "Upper Skyloft - Zelda's Room",
        "Stairs to Central Skyloft": "Central Skyloft",
        "Loftwing Platforms": "Sky",
    },
    "Upper Skyloft - Knight Academy": {
        "Upper Skyloft": "Upper Skyloft",
    },
    "Upper Skyloft - Zelda's Room": {
        "Knight Academy": "Upper Skyloft - Knight Academy",
    },
    "Central Skyloft": {
        "Stairs to Upper Skyloft": "Upper Skyloft",
        "Bridge to Skyloft Village": "Skyloft Village",
        "Ring Beedle's Bell": "Beedle's Shop",
        "Loftwing Platforms": "Sky",
        "Dungeon Entrance on Skyloft": "Sky Keep - First Room",
        "Trial Gate on Skyloft": "Skyloft Silent Realm",
    },
    "Skyloft Village": {
        "Bridge to Central Skyloft": "Central Skyloft",
        "Shed Door to Batreaux's House": "Batreaux's House",
        "Loftwing Platforms": "Sky",
    },
    "Batreaux's House": {
        "Exit Shed Door": "Skyloft Village",
    },
    "Beedle's Shop": {
        "Leave Shop": "Central Skyloft",
    },
    "Sky": {
        "Dive to Upper Skyloft": "Upper Skyloft",
        "Dive to Central Skyloft": "Central Skyloft",
        "Dive to Skyloft Village": "Skyloft Village",
        "Opening in Clouds": "Thunderhead",
        "Emerald Pillar": "Sealed Grounds - Spiral",
        "Ruby Pillar": "Eldin Volcano - Volcano Entry",
        "Amber Pillar": "Lanayru Mine - Landing",
    },
    "Thunderhead": {
        "Opening in Clouds": "Sky",
    },
    "Sealed Grounds - Spiral": {
        "Fly to Sky": "Sky",
        "Sealed Temple": "Sealed Grounds - Sealed Temple",
    },
    "Sealed Grounds - Sealed Temple": {
        "Spiral": "Sealed Grounds - Spiral",
        "Behind the Temple": "Sealed Grounds - Behind the Temple",
        "Hylia's Realm": "Hylia's Realm",
    },
    "Sealed Grounds - Behind the Temple": {
        "Fly to Sky": "Sky",
        "Sealed Temple": "Sealed Grounds - Sealed Temple",
        "Path to Forest": "Faron Woods - Forest Entry",
    },
    "Hylia's Realm": {
        "Door to Temple": "Sealed Grounds - Sealed Temple",
    },
    "Faron Woods - Forest Entry": {
        "Fly to Sky": "Sky",
        "Path out of Forest": "Sealed Grounds - Behind the Temple",
        "In the Woods": "Faron Woods - In the Woods",
    },
    "Faron Woods - In the Woods": {
        "Fly to Sky": "Sky",
        "Forest Entry": "Faron Woods - Forest Entry",
        "Viewing Platform": "Faron Woods - Viewing Platform",
        "Inside Lower Great Tree": "Faron Woods - Inside Lower Great Tree - Water",
        "Great Tree Platforms": "Faron Woods - Great Tree Platforms",
        "Jump into Lake": "Lake Floria - Lake",
    },
    "Faron Woods - Viewing Platform": {
        "Fly to Sky": "Sky",
        "In the Woods": "Faron Woods - In the Woods",
        "Deep Woods Entry": "Faron Woods - Deep Woods Entry",
        "Trial Gate in Faron Woods": "Faron Silent Realm",
    },
    "Faron Woods - Deep Woods Entry": {
        "Viewing Platform": "Faron Woods - Viewing Platform",
        "Deep Woods after Beehive": "Faron Woods - Deep Woods after Beehive",
    },
    "Faron Woods - Deep Woods after Beehive": {
        "Fly to Sky": "Sky",
        "Deep Woods Entry": "Faron Woods - Deep Woods Entry",
        "Dungeon Entrance in Deep Woods": "Skyview - First Room",
    },
    "Faron Woods - Inside Lower Great Tree - Water": {
        "In the Woods": "Faron Woods - In the Woods",
        "After Swinging Platforms": "Faron Woods - Inside Lower Great Tree - After Swinging Platforms",
        "Upper Ledge": "Faron Woods - Inside Lower Great Tree - Upper Ledge",
    },
    "Faron Woods - Inside Lower Great Tree - After Swinging Platforms": {
        "Water": "Faron Woods - Inside Lower Great Tree - Water",
        "Upper Ledge": "Faron Woods - Inside Lower Great Tree - Upper Ledge",
        "Great Tree Platforms": "Faron Woods - Great Tree Platforms",
    },
    "Faron Woods - Inside Lower Great Tree - Upper Ledge": {
        "Water": "Faron Woods - Inside Lower Great Tree - Water",
        "After Swinging Platforms": "Faron Woods - Inside Lower Great Tree - After Swinging Platforms",
    },
    "Faron Woods - Inside Upper Great Tree": {
        "Lower Exit": "Faron Woods - Great Tree Platforms",
        "Upper Exit": "Faron Woods - Outside Top of Great Tree",
        "Jump Down to Lower Great Tree": "Faron Woods - Inside Lower Great Tree - Upper Ledge",
        "Speak to Water Dragon": "Flooded Faron Woods",
    },
    "Faron Woods - Great Tree Platforms": {
        "Lower Exit": "Faron Woods - Inside Lower Great Tree - After Swinging Platforms",
        "Upper Exit": "Faron Woods - Inside Upper Great Tree",
        "In the Woods": "Faron Woods - In the Woods",
        "Viewing Platform": "Faron Woods - Viewing Platform",
    },
    "Faron Woods - Outside Top of Great Tree": {
        "Fly to Sky": "Sky",
        "In the Woods": "Faron Woods - In the Woods",
        "Viewing Platform": "Faron Woods - Viewing Platform",
        "Inside Upper Great Tree": "Faron Woods - Inside Upper Great Tree",
    },
    "Lake Floria - Lake": {
        "Higher Ledge in Lake": "Lake Floria - Higher Ledge in Lake",
        "Dragon's Lair": "Lake Floria - Dragon's Lair",
    },
    "Lake Floria - Higher Ledge in Lake": {
        "Fly to Sky": "Sky",
        "Lake": "Lake Floria - Lake",
    },
    "Lake Floria - Dragon's Lair": {
        "Lake": "Lake Floria - Lake",
        "Waterfall": "Lake Floria - Waterfall",
    },
    "Lake Floria - Waterfall": {
        "Fly to Sky": "Sky",
        "Dragon's Lair": "Lake Floria - Dragon's Lair",
        "Path into Forest": "Faron Woods - In the Woods",
        "Ancient Cistern Ledge": "Lake Floria - Ancient Cistern Ledge",
    },
    "Lake Floria - Ancient Cistern Ledge": {
        "Waterfall": "Lake Floria - Waterfall",
        "Dungeon Entrance in Lake Floria": "Ancient Cistern - Main Room",
    },
    "Flooded Faron Woods": {
        "Speak to Water Dragon": "Faron Woods - Outside Top of Great Tree",
    },

    "Eldin Volcano - Volcano Entry": {
        "Fly to Sky": "Sky",
        "Volcano East": "Eldin Volcano - Volcano East",
        "Volcano Ascent": "Eldin Volcano - Volcano Ascent",
        "Speak to Gossip Stone": "Bokoblin Base - Prison",
    },
    "Eldin Volcano - Volcano East": {
        "Fly to Sky": "Sky",
        "Volcano Entry": "Eldin Volcano - Volcano Entry",
        "Volcano Ascent": "Eldin Volcano - Volcano Ascent",
        "Mogma Cave": "Mogma Turf - Entry",
    },
    "Eldin Volcano - Volcano Ascent": {
        "Fly to Sky": "Sky",
        "Volcano Entry": "Eldin Volcano - Volcano Entry",
        "Volcano East": "Eldin Volcano - Volcano East",
        "Thrill Digger Cliff": "Eldin Volcano - Thrill Digger Cliff",
        "Bottom of Sand Slide": "Eldin Volcano - Bottom of Sand Slide",
        "Trial Gate in Eldin Volcano": "Eldin Silent Realm",
    },
    "Eldin Volcano - Thrill Digger Cliff": {
        "Volcano Ascent": "Eldin Volcano - Volcano Ascent",
        "Thrill Digger Cave": "Eldin Volcano - Thrill Digger Cave",
        "Near Temple Entrance": "Eldin Volcano - Near Temple Entrance",
    },
    "Eldin Volcano - Thrill Digger Cave": {
        "Exit Thrill Digger Cave": "Eldin Volcano - Thrill Digger Cliff",
    },
    "Eldin Volcano - Near Temple Entrance": {
        "Fly to Sky": "Sky",
        "Thrill Digger Cliff": "Eldin Volcano - Thrill Digger Cliff",
        "Hot Cave": "Eldin Volcano - Hot Cave",
        "Dungeon Entrance in Eldin Volcano": "Earth Temple - First Room",
    },
    "Eldin Volcano - Hot Cave": {
        "Near Temple Entrance": "Eldin Volcano - Near Temple Entrance",
        "Sand Slide": "Eldin Volcano - Sand Slide",
        "Upper Path out of Hot Cave": "Volcano Summit - Summit",
    },
    "Eldin Volcano - Sand Slide": {
        "Bottom of Sand Slide": "Eldin Volcano - Bottom of Sand Slide",
    },
    "Eldin Volcano - Bottom of Sand Slide": {
        "Volcano Ascent": "Eldin Volcano - Volcano Ascent",
        "Hot Cave": "Eldin Volcano - Hot Cave",
    },
    "Mogma Turf - Entry": {
        "Use First Air Vent": "Eldin Volcano - Volcano East",
        "Use Digging Air Vent": "Mogma Turf - After Digging Air Vent",
    },
    "Mogma Turf - After Digging Air Vent": {
        "Jump off Ledge": "Mogma Turf - Entry",
        "Use Last Air Vent": "Eldin Volcano - Volcano Ascent",
    },
    "Volcano Summit - Summit": {
        "Path out of Summit before Sandy Slope": "Eldin Volcano - Hot Cave",
        "Path out of Summit after Lava Platforms": "Volcano Summit - Waterfall",
        "Upper Path out of Summit after Lava Platforms": "Volcano Summit - Before First Frog",
    },
    "Volcano Summit - Waterfall": {
        "Path out of Waterfall": "Volcano Summit - Summit",
    },
    "Volcano Summit - Before First Frog": {
        "Path across from First Frog": "Volcano Summit - Summit",
        "Pass First Frog": "Volcano Summit - Between Frogs",
    },
    "Volcano Summit - Between Frogs": {
        "Pass First Frog": "Volcano Summit - Before First Frog",
        "Pass Second Frog": "Volcano Summit - After Second Frog",
    },
    "Volcano Summit - After Second Frog": {
        "Fly to Sky": "Sky",
        "Pass Second Frog": "Volcano Summit - Between Frogs",
        "Dungeon Entrance in Volcano Summit": "Fire Sanctuary - First Room",
    },
    "Bokoblin Base - Prison": {
        "Speak to Gossip Stone": "Eldin Volcano - Volcano Entry",
        "Dig to Volcano": "Bokoblin Base - Volcano",
    },
    "Bokoblin Base - Volcano": {
        "Dig to Prison": "Bokoblin Base - Prison",
        "Use Air Vent": "Bokoblin Base - Top of Volcano",
    },
    "Bokoblin Base - Top of Volcano": {
        "Path through Hot Cave": "Bokoblin Base - Summit",
    },
    "Bokoblin Base - Summit": {
        "Dragon's Lair": "Bokoblin Base - Dragon's Lair",
    },
    "Bokoblin Base - Dragon's Lair": {
        "Exit Dragon's Lair": "Bokoblin Base - Summit",
        "Exit Bokoblin Base": "Volcano Summit - Summit",
    },

    "Lanayru Mine - Landing": {
        "Fly to Sky": "Sky",
        "Clawshot Path to Caves": "Lanayru Mine - Door to Caves",
        "Pass Statues in Mine": "Lanayru Mine - Ending",
    },
    "Lanayru Mine - Door to Caves": {
        "Clawshot Path to Mine": "Lanayru Mine - Landing",
        "Door to Caves": "Lanayru Caves - Caves",
    },
    "Lanayru Mine - Ending": {
        "Pass Statues in Mine": "Lanayru Mine - Landing",
        "Minecart Ride out of Mine": "Lanayru Desert - Desert Entry",
    },
    "Lanayru Desert - Desert Entry": {
        "Fly to Sky": "Sky",
        "Minecart Ride out of Desert": "Lanayru Mine - Ending",
        "Sand Oasis": "Lanayru Desert - Sand Oasis",
        "Stone Cache": "Lanayru Desert - Stone Cache",
    },
    "Lanayru Desert - Sand Oasis": {
        "Fly to Sky": "Sky",
        "Desert Entry": "Lanayru Desert - Desert Entry",
        "Path to Temple of Time": "Lanayru Desert - Temple of Time - Desert Gorge",
        "Raised Entrance to Caves": "Lanayru Desert - Higher Ledge in Sand Oasis",
    },
    "Lanayru Desert - Higher Ledge in Sand Oasis": {
        "Sand Oasis": "Lanayru Desert - Sand Oasis",
        "Path to Caves": "Lanayru Caves - Caves",
    },
    "Lanayru Desert - Temple of Time - Desert Gorge": {
        "Fly to Sky": "Sky",
        "Path to Sand Oasis": "Lanayru Desert - Sand Oasis",
        "Minecart through Tree Root": "Lanayru Desert - Temple of Time - Temple",
    },
    "Lanayru Desert - Temple of Time - Temple": {
        "Fly to Sky": "Sky",
        "Minecart to Desert Gorge": "Lanayru Desert - Temple of Time - Desert Gorge",
        "Path to North Desert": "Lanayru Desert - North Desert",
    },
    "Lanayru Desert - North Desert": {
        "Fly to Sky": "Sky",
        "Path to Temple of Time": "Lanayru Desert - Temple of Time - Temple",
        "Stone Cache": "Lanayru Desert - Stone Cache",
        "Door to Lightning Node": "Lanayru Desert - Lightning Node",
        "Dungeon Entrance in Lanayru Desert": "Lanayru Mining Facility - First Room",
        "Trial Gate in Lanayru Desert": "Lanayru Silent Realm",
    },
    "Lanayru Desert - Stone Cache": {
        "Fly to Sky": "Sky",
        "Desert Entry": "Lanayru Desert - Desert Entry",
        "North Desert": "Lanayru Desert - North Desert",
        "Door to Fire Node": "Lanayru Desert - Fire Node",
    },
    "Lanayru Desert - Fire Node": {
        "Exit Fire Node": "Lanayru Desert - Stone Cache",
    },
    "Lanayru Desert - Lightning Node": {
        "Exit Lightning Node": "Lanayru Desert - North Desert",
    },
    "Lanayru Caves - Caves": {
        "North Exit": "Lanayru Desert - Higher Ledge in Sand Oasis",
        "East Exit": "Lanayru Mine - Door to Caves",
        "South Exit past Crawlspace": "Lanayru Caves - Past Crawlspace",
        "West Exit by Clawshot Target": "Lanayru Caves - Past Locked Door",
    },
    "Lanayru Caves - Past Locked Door": {
        "Path away from Door": "Lanayru Sand Sea - Door to Caves",
        "Through Door": "Lanayru Caves - Caves",
    },
    "Lanayru Caves - Past Crawlspace": {
        "Path away from Crawlspace": "Lanayru Gorge - Gorge",
        "Through Crawlspace": "Lanayru Caves - Caves",
    },
    "Lanayru Gorge - Gorge": {
        "Fly to Sky": "Sky",
        "Path to Caves": "Lanayru Caves - Past Crawlspace",
    },
    "Lanayru Sand Sea - Door to Caves": {
        "Clawshot Path to Harbor": "Lanayru Sand Sea - Ancient Harbor",
        "Door to Caves": "Lanayru Caves - Past Locked Door",
    },
    "Lanayru Sand Sea - Ancient Harbor": {
        "Fly to Sky": "Sky",
        "Clawshot Path to Caves": "Lanayru Sand Sea - Door to Caves",
        "Boat to Sea": "Lanayru Sand Sea - Sea",
    },
    "Lanayru Sand Sea - Sea": {
        "Dock at Ancient Harbor": "Lanayru Sand Sea - Ancient Harbor",
        "Dock at Skipper's Retreat": "Lanayru Sand Sea - Skipper's Retreat - Dock",
        "Dock at Shipyard": "Lanayru Sand Sea - Shipyard",
        "Dock at Pirate Stronghold": "Lanayru Sand Sea - Pirate Stronghold",
        "Dungeon Entrance in Lanayru Sand Sea": "Sandship - Deck",
    },
    "Lanayru Sand Sea - Skipper's Retreat - Dock": {
        "Fly to Sky": "Sky",
        "Boat to Sea": "Lanayru Sand Sea - Sea",
        "After Rock": "Lanayru Sand Sea - Skipper's Retreat - After Rock",
    },
    "Lanayru Sand Sea - Skipper's Retreat - After Rock": {
        "Dock": "Lanayru Sand Sea - Skipper's Retreat - Dock",
        "Top of Skipper's Retreat": "Lanayru Sand Sea - Skipper's Retreat - Top of Skipper's Retreat",
    },
    "Lanayru Sand Sea - Skipper's Retreat - Top of Skipper's Retreat": {
        "Shack": "Lanayru Sand Sea - Skipper's Retreat - Shack",
    },
    "Lanayru Sand Sea - Skipper's Retreat - Shack": {
        "Top of Skipper's Retreat": "Lanayru Sand Sea - Skipper's Retreat - Top of Skipper's Retreat",
        "After Rock": "Lanayru Sand Sea - Skipper's Retreat - After Rock",
        "Skydive": "Lanayru Sand Sea - Skipper's Retreat - Skydive",
    },
    "Lanayru Sand Sea - Skipper's Retreat - Skydive": {
        "Dock": "Lanayru Sand Sea - Skipper's Retreat - Dock",
    },
    "Lanayru Sand Sea - Shipyard": {
        "Fly to Sky": "Sky",
        "Boat to Sea": "Lanayru Sand Sea - Sea",
    },
    "Lanayru Sand Sea - Pirate Stronghold": {
        "Fly to Sky": "Sky",
        "Boat to Sea": "Lanayru Sand Sea - Sea",
    },

    "Skyview - First Room": {
        "Dungeon Exit": "Faron Woods - Deep Woods after Beehive",
        "Second Room": "Skyview - Second Room",
    },
    "Skyview - Second Room": {
        "Dungeon Exit": "Faron Woods - Deep Woods after Beehive",
        "First Room": "Skyview - First Room",
        "Main Room": "Skyview - Main Room",
    },
    "Skyview - Main Room": {
        "Dungeon Exit": "Faron Woods - Deep Woods after Beehive",
        "Second Room": "Skyview - Second Room",
        "Boss Door Area": "Skyview - Boss Door Area",
    },
    "Skyview - Boss Door Area": {
        "Dungeon Exit": "Faron Woods - Deep Woods after Beehive",
        "Main Room": "Skyview - Main Room",
        "Boss Room": "Skyview - Boss Room",
    },
    "Skyview - Boss Room": {
        "Boss Door Area": "Skyview - Boss Door Area",
        "Skyview Spring": "Skyview - Skyview Spring",
    },
    "Skyview - Skyview Spring": {
        "Boss Room": "Skyview - Boss Room",
        "Strike Crest": "Faron Woods - Deep Woods after Beehive",
    },

    "Earth Temple - First Room": {
        "Dungeon Exit": "Eldin Volcano - Near Temple Entrance",
        "Main Room": "Earth Temple - Main Room",
    },
    "Earth Temple - Main Room": {
        "Dungeon Exit": "Eldin Volcano - Near Temple Entrance",
        "First Room": "Earth Temple - First Room",
        "Room with Slopes": "Earth Temple - Room with Slopes",
    },
    "Earth Temple - Room with Slopes": {
        "Main Room": "Earth Temple - Main Room",
        "Boss Door Area": "Earth Temple - Boss Door Area",
    },
    "Earth Temple - Boss Door Area": {
        "Dungeon Exit": "Eldin Volcano - Near Temple Entrance",
        "Room with Slopes": "Earth Temple - Room with Slopes",
        "Boss Room": "Earth Temple - Boss Room",
    },
    "Earth Temple - Boss Room": {
        "Boss Door Area": "Earth Temple - Boss Door Area",
        "Earth Spring": "Earth Temple - Earth Spring",
    },
    "Earth Temple - Earth Spring": {
        "Boss Room": "Earth Temple - Boss Room",
        "Strike Crest": "Eldin Volcano - Near Temple Entrance",
    },

    "Lanayru Mining Facility - First Room": {
        "Dungeon Exit": "Lanayru Desert - North Desert",
        "Second Room": "Lanayru Mining Facility - Second Room",
    },
    "Lanayru Mining Facility - Second Room": {
        "First Room": "Lanayru Mining Facility - First Room",
        "Front Main Room": "Lanayru Mining Facility - Front Main Room",
        "Key Locked Room": "Lanayru Mining Facility - Key Locked Room",
        "West Room": "Lanayru Mining Facility - West Room",
    },
    "Lanayru Mining Facility - Key Locked Room": {
        "Second Room": "Lanayru Mining Facility - Second Room",
        "Front Main Room": "Lanayru Mining Facility - Front Main Room",
    },
    "Lanayru Mining Facility - West Room": {
        "Second Room": "Lanayru Mining Facility - Second Room",
        "West Main Room": "Lanayru Mining Facility - West Main Room",
    },
    "Lanayru Mining Facility - Front Main Room": {
        "Dungeon Exit": "Lanayru Desert - North Desert",
        "Second Room": "Lanayru Mining Facility - Second Room",
    },
    "Lanayru Mining Facility - West Main Room": {
        "West Room": "Lanayru Mining Facility - West Room",
        "Front Main Room": "Lanayru Mining Facility - Front Main Room",
        "Spike Maze": "Lanayru Mining Facility - Spike Maze",
    },
    "Lanayru Mining Facility - Spike Maze": {
        "Boss Door Area": "Lanayru Mining Facility - Boss Door Area",
    },
    "Lanayru Mining Facility - Boss Door Area": {
        "Dungeon Exit": "Lanayru Desert - North Desert",
        "Boss Key Room": "Lanayru Mining Facility - Boss Key Room",
        "Boss Room": "Lanayru Mining Facility - Boss Room",
    },
    "Lanayru Mining Facility - Boss Key Room": {
        "Boss Door Area": "Lanayru Mining Facility - Boss Door Area",
    },
    "Lanayru Mining Facility - Boss Room": {
        "Boss Door Area": "Lanayru Mining Facility - Boss Door Area",
        "Hall of Ancient Robots": "Lanayru Mining Facility - Hall of Ancient Robots",
    },
    "Lanayru Mining Facility - Hall of Ancient Robots": {
        "Boss Room": "Lanayru Mining Facility - Boss Room",
        "Exit Hall of Ancient Robots": "Lanayru Desert - Temple of Time - Temple",
    },

    "Ancient Cistern - Main Room": {
        "Dungeon Exit": "Lake Floria - Ancient Cistern Ledge",
        "East Room": "Ancient Cistern - East Room",
        "Past Waterfall": "Ancient Cistern - Past Waterfall",
        "Statue": "Ancient Cistern - Statue",
    },
    "Ancient Cistern - Statue": {
        "Main Room": "Ancient Cistern - Main Room",
        "Basement": "Ancient Cistern - Basement",
        "Boss Room": "Ancient Cistern - Boss Room",
    },
    "Ancient Cistern - East Room": {
        "Main Room": "Ancient Cistern - Main Room",
    },
    "Ancient Cistern - Past Waterfall": {
        "Main Room": "Ancient Cistern - Main Room",
        "Gutters": "Ancient Cistern - Gutters",
    },
    "Ancient Cistern - Gutters": {
        "Past Gutters": "Ancient Cistern - Past Gutters",
    },
    "Ancient Cistern - Past Gutters": {
        "Dungeon Exit": "Lake Floria - Ancient Cistern Ledge",
        "Main Room": "Ancient Cistern - Main Room",
    },
    "Ancient Cistern - Basement": {
        "Statue": "Ancient Cistern - Statue",
    },
    "Ancient Cistern - Boss Room": {
        "Statue": "Ancient Cistern - Statue",
        "Flame Room": "Ancient Cistern - Flame Room",
    },
    "Ancient Cistern - Flame Room": {
        "Boss Room": "Ancient Cistern - Boss Room",
        "Strike Crest": "Lake Floria - Ancient Cistern Ledge",
    },

    "Sandship - Deck": {
        "Dungeon Exit": "Lanayru Sand Sea - Sea",
        "Corridor": "Sandship - Corridor",
        "Boss Key Room": "Sandship - Boss Key Room",
    },
    "Sandship - Corridor": {
        "Dungeon Exit": "Lanayru Sand Sea - Sea",
        "Deck": "Sandship - Deck",
        "Near Boss Door": "Sandship - Near Boss Door",
        "Basement": "Sandship - Basement",
        "Bow": "Sandship - Bow",
    },
    "Sandship - Near Boss Door": {
        "Dungeon Exit": "Lanayru Sand Sea - Sea",
        "Corridor": "Sandship - Corridor",
        "Boss Room": "Sandship - Boss Room",
    },
    "Sandship - Basement": {
        "Corridor": "Sandship - Corridor",
    },
    "Sandship - Bow": {
        "Corridor": "Sandship - Corridor",
    },
    "Sandship - Boss Key Room": {
        "Deck": "Sandship - Deck",
    },
    "Sandship - Boss Room": {
        "Near Boss Door": "Sandship - Near Boss Door",
        "Strike Crest": "Lanayru Sand Sea - Ancient Harbor",
    },

    "Fire Sanctuary - First Room": {
        "Dungeon Exit": "Volcano Summit - After Second Frog",
        "Second Room": "Fire Sanctuary - Second Room",
    },
    "Fire Sanctuary - Second Room": {
        "Dungeon Exit": "Volcano Summit - After Second Frog",
        "First Bridge": "Fire Sanctuary - First Bridge",
    },
    "Fire Sanctuary - First Bridge": {
        "Second Room": "Fire Sanctuary - Second Room",
        "Flowing Lava Room": "Fire Sanctuary - Flowing Lava Room",
    },
    "Fire Sanctuary - Flowing Lava Room": {
        "First Bridge": "Fire Sanctuary - First Bridge",
        "Upper First Trapped Mogma Room": "Fire Sanctuary - Upper First Trapped Mogma Room",
        "Water Fruit Room": "Fire Sanctuary - Water Fruit Room",
    },
    "Fire Sanctuary - Water Fruit Room": {
        "Dungeon Exit": "Volcano Summit - After Second Frog",
        "Flowing Lava Room": "Fire Sanctuary - Flowing Lava Room",
        "Lower First Trapped Mogma Room": "Fire Sanctuary - Lower First Trapped Mogma Room",
    },
    "Fire Sanctuary - Upper First Trapped Mogma Room": {
        "Flowing Lava Room": "Fire Sanctuary - Flowing Lava Room",
    },
    "Fire Sanctuary - Lower First Trapped Mogma Room": {
        "Water Fruit Room": "Fire Sanctuary - Water Fruit Room",
        "Under Double Magmanos Fight": "Fire Sanctuary - Under Double Magmanos Fight",
    },
    "Fire Sanctuary - Under Double Magmanos Fight": {
        "Dungeon Exit": "Volcano Summit - After Second Frog",
        "Lower First Trapped Mogma Room": "Fire Sanctuary - Lower First Trapped Mogma Room",
        "Second Bridge": "Fire Sanctuary - Second Bridge",
        "Boss Door Area": "Fire Sanctuary - Boss Door Area",
    },
    "Fire Sanctuary - Second Bridge": {
        "Under Double Magmanos Fight": "Fire Sanctuary - Under Double Magmanos Fight",
        "Second Trapped Mogma Room": "Fire Sanctuary - Second Trapped Mogma Room",
    },
    "Fire Sanctuary - Second Trapped Mogma Room": {
        "Second Bridge": "Fire Sanctuary - Second Bridge",
    },
    "Fire Sanctuary - Boss Door Area": {
        "Dungeon Exit": "Volcano Summit - After Second Frog",
        "Under Double Magmanos Fight": "Fire Sanctuary - Under Double Magmanos Fight",
        "Staircase Room": "Fire Sanctuary - Staircase Room",
        "Boss Room": "Fire Sanctuary - Boss Room",
    },
    "Fire Sanctuary - Staircase Room": {
        "Boss Key Room": "Fire Sanctuary - Boss Key Room",
    },
    "Fire Sanctuary - Boss Key Room": {
        "Staircase Room": "Fire Sanctuary - Staircase Room",
        "Boss Door Area": "Fire Sanctuary - Boss Door Area",
    },
    "Fire Sanctuary - Boss Room": {
        "Boss Door Area": "Fire Sanctuary - Boss Door Area",
        "Flame Room": "Fire Sanctuary - Flame Room",
    },
    "Fire Sanctuary - Flame Room": {
        "Boss Room": "Fire Sanctuary - Boss Room",
        "Strike Crest": "Volcano Summit - After Second Frog",
    },

    "Sky Keep - First Room": {
        "Dungeon Exit": "Central Skyloft",
        "Skyview Room": "Sky Keep - Skyview Room",
    },
    "Sky Keep - Skyview Room": {
        "Dungeon Exit": "Central Skyloft",
        "First Room": "Sky Keep - First Room",
        "Lanayru Mining Facility Room": "Sky Keep - Lanayru Mining Facility Room",
    },
    "Sky Keep - Lanayru Mining Facility Room": {
        "Dungeon Exit": "Central Skyloft",
        "Skyview Room": "Sky Keep - Skyview Room",
        "Earth Temple Room": "Sky Keep - Earth Temple Room",
        "Ancient Cistern Room": "Sky Keep - Ancient Cistern Room",
    },
    "Sky Keep - Earth Temple Room": {
        "Dungeon Exit": "Central Skyloft",
        "Lanayru Mining Facility Room": "Sky Keep - Lanayru Mining Facility Room",
        "Miniboss Room": "Sky Keep - Miniboss Room",
    },
    "Sky Keep - Ancient Cistern Room": {
        "Lanayru Mining Facility Room": "Sky Keep - Lanayru Mining Facility Room",
        "Fire Sanctuary Room": "Sky Keep - Fire Sanctuary Room",
    },
    "Sky Keep - Miniboss Room": {
        "Earth Temple Room": "Sky Keep - Earth Temple Room",
        "Fire Sanctuary Room": "Sky Keep - Fire Sanctuary Room",
    },
    "Sky Keep - Fire Sanctuary Room": {
        "Dungeon Exit": "Central Skyloft",
        "Ancient Cistern Room": "Sky Keep - Ancient Cistern Room",
        "Miniboss Room": "Sky Keep - Miniboss Room",
        "Sandship Room": "Sky Keep - Sandship Room",
    },
    "Sky Keep - Sandship Room": {
        "Dungeon Exit": "Central Skyloft",
        "Fire Sanctuary Room": "Sky Keep - Fire Sanctuary Room",
    },

    "Skyloft Silent Realm": {
        "Trial Gate": "Central Skyloft",
    },
    "Faron Silent Realm": {
        "Trial Gate": "Faron Woods - Viewing Platform",
    },
    "Eldin Silent Realm": {
        "Trial Gate": "Eldin Volcano - Volcano Ascent",
    },
    "Lanayru Silent Realm": {
        "Trial Gate": "Lanayru Desert - North Desert",
    },
}
