import logging
from typing import Mapping, Any, Union, Dict, Optional

from BaseClasses import ItemClassification, Tutorial
from worlds.AutoWorld import World, WebWorld
from .ItemNames import ItemName, long_macguffins, short_macguffins
from .Items import APGOItem, item_table, APGOItemData, create_items
from .Locations import APGOLocation, location_table, create_locations
from .Options import APGOOptions, Goal
from .Regions import create_regions, area_number
from .Trips import generate_trips, Trip
from .rules import set_rules
from ..generic.Rules import set_rule

logger = logging.getLogger(__name__)

GAME_NAME = "Archipela-Go!"


class APGOWebWorld(WebWorld):
    setup_en = Tutorial(
        "Multiworld Setup Guide",
        "A guide to setting up an Archipela-Go! game on your device",
        "English",
        "setup_en.md",
        "setup/en",
        ["Kaito Kid"]
    )
    tutorials = [setup_en]


class APGOWorld(World):
    """
    Archipela-Go is an exercise game designed around walking or jogging outside to unlock progression
    """
    game = GAME_NAME
    author: str = "Kaito Kid & aki665"
    web = APGOWebWorld()

    item_name_to_id = {name: data.id for name, data in item_table.items()}
    location_name_to_id = {name: id for name, id in location_table.items()}

    item_name_groups = {
        "MacGuffins": set(long_macguffins),
        "Letters": set(long_macguffins),
    }

    data_version = 0

    options_dataclass = APGOOptions
    options: APGOOptions

    trips: Dict[str, Trip]
    number_distance_reductions: int
    number_keys: int

    def generate_early(self):
        self.force_change_options_if_incompatible(self.options, self.player, self.player_name)
        generated_trips = generate_trips(self.options.speed_requirement, self.options.number_of_locks, self.options.number_of_trips, self.random)
        self.trips = {trip.location_name: trip for trip in generated_trips}
        self.number_distance_reductions = 0
        self.number_keys = 0
        for trip in self.trips.values():
            if trip.template.key_needed > self.number_keys:
                self.number_keys = trip.template.key_needed

    @staticmethod
    def force_change_options_if_incompatible(options: APGOOptions, player, player_name):
        max_trips = 1210
        max_trips -= (10 - options.number_of_locks) * 100
        if options.speed_requirement.value <= 0:
            max_trips = max_trips // 10
        if options.number_of_trips > max_trips:
            options.number_of_trips.value = max_trips
            logger.warning(
                f"Too many trips requested by {player} ({player_name}) with their settings. Forced down the number of trips to {max_trips}")
        if options.number_of_trips < options.number_of_locks * 2:
            options.number_of_locks.value = options.number_of_trips // 2
            logger.warning(
                f"Too many keys requested by {player} ({player_name}) with their number of trips. Forced down the number of keys to {options.number_of_locks.value}")

    def create_regions(self) -> None:
        world_regions = create_regions(self)

        def create_location(name: str, code: Optional[int], region: str):
            region = world_regions[region]
            location = APGOLocation(self.player, name, code, region)
            location.access_rule = lambda _: True
            region.locations.append(location)

        create_locations(create_location, self.trips)

    def create_items(self) -> None:
        created_items = create_items(self.create_item, self.trips, self.options, self.random)
        self.multiworld.itempool += created_items

        # This is a weird way to count but it works...
        self.number_distance_reductions += sum(item.name == ItemName.distance_reduction for item in created_items)
        self.setup_victory()

    def set_rules(self) -> None:
        set_rules(self, self.player, self.options)

    def create_item(self, item: Union[str, APGOItemData]) -> APGOItem:
        if isinstance(item, str):
            item = item_table[item]

        return APGOItem(item.name, item.classification, item.id, self.player)

    def setup_victory(self):
        last_region = self.multiworld.get_region(area_number(self.number_keys), self.player)
        victory_location = APGOLocation(self.player, "Goal", None, last_region)
        last_region.locations.append(victory_location)
        if self.options.goal == Goal.option_one_hard_travel or self.options.goal == Goal.option_allsanity:
            set_rule(victory_location, lambda state: state.has(ItemName.distance_reduction, self.player, self.number_distance_reductions))
        elif self.options.goal == Goal.option_long_macguffin:
            set_rule(victory_location, lambda state: all([state.has(macguffin, self.player) for macguffin in long_macguffins]))
        elif self.options.goal == Goal.option_short_macguffin:
            set_rule(victory_location, lambda state: all([state.has(macguffin, self.player) for macguffin in short_macguffins]))

        victory_location.place_locked_item(APGOItem("Victory", ItemClassification.progression, None, self.player))
        self.multiworld.completion_condition[self.player] = lambda state: state.has("Victory", self.player)

    def fill_slot_data(self) -> Mapping[str, Any]:
        trips_dictionary = {location_name: trip.as_dict() for location_name, trip in self.trips.items()}
        slot_data = {
            self.options.goal.internal_name: self.options.goal.value,
            self.options.minimum_distance.internal_name: self.options.minimum_distance.value,
            self.options.maximum_distance.internal_name: self.options.maximum_distance.value,
            self.options.speed_requirement.internal_name: self.options.speed_requirement.value,
            "trips": trips_dictionary,
        }
        return slot_data
