from typing import Any, TYPE_CHECKING

from .er_data import portal_mapping
from .options import EntranceLayout, LaurelsLocation

if TYPE_CHECKING:
    from . import TunicWorld


def setup_options_from_slot_data(world: "TunicWorld") -> None:
    if hasattr(world.multiworld, "re_gen_passthrough"):
        if "TUNIC" in world.multiworld.re_gen_passthrough:
            world.using_ut = True
            world.passthrough = world.multiworld.re_gen_passthrough["TUNIC"]
            world.options.start_with_sword.value = world.passthrough["start_with_sword"]
            world.options.keys_behind_bosses.value = world.passthrough["keys_behind_bosses"]
            world.options.sword_progression.value = world.passthrough["sword_progression"]
            world.options.ability_shuffling.value = world.passthrough["ability_shuffling"]
            world.options.laurels_zips.value = world.passthrough["laurels_zips"]
            world.options.ice_grappling.value = world.passthrough["ice_grappling"]
            world.options.ladder_storage.value = world.passthrough["ladder_storage"]
            world.options.ladder_storage_without_items = world.passthrough["ladder_storage_without_items"]
            world.options.lanternless.value = world.passthrough["lanternless"]
            world.options.maskless.value = world.passthrough["maskless"]
            world.options.hexagon_quest.value = world.passthrough["hexagon_quest"]
            world.options.hexagon_quest_ability_type.value = world.passthrough.get("hexagon_quest_ability_type", 0)
            world.options.entrance_rando.value = world.passthrough["entrance_rando"]
            world.options.shuffle_ladders.value = world.passthrough["shuffle_ladders"]
            # world.options.shuffle_fuses.value = world.passthrough.get("shuffle_fuses", 0)
            # world.options.shuffle_bells.value = world.passthrough.get("shuffle_bells", 0)
            world.options.grass_randomizer.value = world.passthrough.get("grass_randomizer", 0)
            world.options.breakable_shuffle.value = world.passthrough.get("breakable_shuffle", 0)
            world.options.entrance_layout.value = EntranceLayout.option_standard
            if ("ziggurat2020_3, ziggurat2020_1_zig2_skip" in world.passthrough["Entrance Rando"].keys()
                    or "ziggurat2020_3, ziggurat2020_1_zig2_skip" in world.passthrough["Entrance Rando"].values()):
                world.options.entrance_layout.value = EntranceLayout.option_fixed_shop
            world.options.decoupled = world.passthrough.get("decoupled", 0)
            world.options.laurels_location.value = LaurelsLocation.option_anywhere
            world.options.combat_logic.value = world.passthrough.get("combat_logic", 0)
            world.options.true_end.value = world.passthrough["true_end"]
        else:
            world.using_ut = False
    else:
        world.using_ut = False


def modify_shop_key(portal_sdt: str) -> str:
    if portal_sdt.startswith("Shop, ") and not portal_sdt.startswith("Shop, P"):
        # the mod internally uses Shop, Previous Region instead of just having the number
        return portal_sdt.replace("Shop, ", "Shop, Previous Region ")
    return portal_sdt


def setup_found_entrances_datastorage(world: "TunicWorld") -> None:
    world.found_entrances_datastorage_key = []
    for portal_sdt1, portal_sdt2 in world.passthrough["Entrance Rando"].items():
        world.found_entrances_datastorage_key.append("Slot:{player}:" + modify_shop_key(portal_sdt1))
        if not world.options.decoupled:
            world.found_entrances_datastorage_key.append("Slot:{player}:" + modify_shop_key(portal_sdt2))


def disconnect_entrances(world: "TunicWorld") -> None:
    world.disconnected_entrances = {}
    for entrance in world.get_entrances():
        if entrance.name.startswith("Shop Portal"):
            world.disconnected_entrances[entrance] = entrance.connected_region
            entrance.connected_region = None
            continue
        # LS entrances start with the same name as the non-LS portal entrance
        name = entrance.name.split(" (LS) ", 1)[0]
        for portal in portal_mapping:
            if portal.name == name:
                world.disconnected_entrances[entrance] = entrance.connected_region
                entrance.connected_region = None


def get_entrance_name_from_sdt(portal_sdt: str) -> str:
    if portal_sdt.startswith("Shop, P"):
        shop_number = portal_sdt.replace("Shop, Previous Region ", "").replace("_", "")
        return f"Shop Portal {shop_number}"
    for portal in portal_mapping:
        if portal.scene_destination() == portal_sdt:
            return portal.name
    else:
        raise Exception("Portal SDT not found for some reason.")


def reconnect_found_entrance(world: "TunicWorld", portal_sdt: str) -> None:
    portal_name = get_entrance_name_from_sdt(portal_sdt)
    entrance_connected = False
    for entrance, region in world.disconnected_entrances.items():
        if entrance.name.startswith(portal_name):
            entrance.connect(region)
            entrance_connected = True
    if not entrance_connected:
        raise Exception("Portal name not found in reconnect_found_entrance")


# for UT poptracker integration map tab switching
def map_page_index(data: Any) -> int:
    mapping: dict[str, int] = {
        "Beneath the Earth": 1,
        "Beneath the Well": 2,
        "The Cathedral": 3,
        "Dark Tomb": 4,
        "Eastern Vault": 5,
        "Frog's Domain": 6,
        "Swamp": 7,
        "Overworld": 8,
        "The Quarry": 9,
        "Ruined Atoll": 10,
        "West Gardens": 11,
        "The Grand Library": 12,
        "East Forest": 13,
        "The Far Shore": 14,
        "The Rooted Ziggurat": 15,
    }
    return mapping.get(data, 0)


# mapping of everything after the second to last slash and the location id
# lua used for the name: string.match(full_name, "[^/]*/[^/]*$")
poptracker_data: dict[str, int] = {
    "Library Lab Chest by Shrine 1/Chest": 509342493,
    "Library Lab Chest by Shrine 3/Chest": 509342494,
    "Library Lab by Fuse/Behind Chalkboard": 509342495,
    "Library Lab Page 3/Page": 509342496,
    "Library Lab Page 1/Page": 509342497,
    "Library Lab Page 2/Page": 509342498,
    "Hero's Grave/Mushroom Relic": 509342499,
    "Mountain Door/Lower Mountain - Page Before Door": 509342500,
    "Changing Room/Normal Chest": 509342501,
    "Fortress Courtyard - Chest Near Cave/Next to the Obelisk": 509342502,
    "Fortress Courtyard - Near Fuse/Pray": 509342503,
    "Fortress Courtyard - Below Walkway/Under the Stairs": 509342504,
    "Fortress Courtyard - Page Near Cave/Heir-To-The-Heir": 509342505,
    "West Furnace/Lantern Pickup": 509342506,
    "Maze Cave/Maze Room Chest": 509342507,
    "Inside the Old House/Normal Chest": 509342508,
    "Inside the Old House/Shield Pickup": 509342509,
    "[West] Obscured Behind Windmill/Behind the Trees": 509342510,
    "[South] Beach Chest/Beside the Bridge": 509342511,
    "[West] Obscured Near Well/Hidden by Trees": 509342512,
    "[Central] Bombable Wall/Let the flowers guide you": 509342513,
    "[Northwest] Chest Near Turret/Mind the Autobolt...": 509342514,
    "[East] Chest Near Pots/Chest": 509342515,
    "[Northwest] Chest Near Golden Obelisk/Underneath the Staff": 509342516,
    "[Southwest] South Chest Near Guard/End of the Bridge": 509342517,
    "[Southwest] West Beach Guarded By Turret/Chest": 509342518,
    "[Southwest] Chest Guarded By Turret/Behind the Trees": 509342519,
    "[Northwest] Shadowy Corner Chest/Dark Ramps Chest": 509342520,
    "[Southwest] Obscured In Tunnel To Beach/Deep in the Wall": 509342521,
    "[Southwest] Obscured In Tunnel To Beach/Deep between the Trees": 509342521,
    "[Southwest] Grapple Chest Over Walkway/Jeffry": 509342522,
    "[Northwest] Chest Beneath Quarry Gate/Across the Bridge": 509342523,
    "[Southeast] Chest Near Swamp/Under the Bridge": 509342524,
    "[Southwest] From West Garden/Dash Across": 509342525,
    "[East] Grapple Chest/Grapple Across": 509342526,
    "[Southwest] West Beach Guarded By Turret 2/Get Across": 509342527,
    "Sand Hook/[Southwest] Beach Chest Near Flowers": 509342528,
    "[Southwest] Bombable Wall Near Fountain/Let the flowers guide you": 509342529,
    "[West] Chest After Bell/Post-Dong!": 509342530,
    "[Southwest] Tunnel Guarded By Turret/Below Jeffry": 509342531,
    "[East] Between ladders near Ruined Passage/Chest": 509342532,
    "[Northeast] Chest Above Patrol Cave/Behind Blue Rudelings": 509342533,
    "[Southwest] Beach Chest Beneath Guard/Under Bridge": 509342534,
    "[Central] Chest Across From Well/Across the Bridge": 509342535,
    "[Northwest] Chest Near Quarry Gate/Rudeling Camp": 509342536,
    "[East] Chest In Trees/Above Locked House": 509342537,
    "[West] Chest Behind Moss Wall/Around the Corner": 509342538,
    "[South] Beach Page/Page": 509342539,
    "[Southeast] Page on Pillar by Swamp/Dash Across": 509342540,
    "[Southwest] Key Pickup/Old House Key": 509342541,
    "[West] Key Pickup/Hero's Path Key": 509342542,
    "[East] Page Near Secret Shop/Page": 509342543,
    "Fountain/[Southwest] Fountain Page": 509342544,
    "[Northwest] Page on Pillar by Dark Tomb/A Terrible Power Rises": 509342545,
    "Magic Staff/[Northwest] Fire Wand Pickup": 509342546,
    "[West] Page on Teleporter/Treasures and Tools": 509342547,
    "[Northwest] Page By Well/If you seek to increase your power...": 509342548,
    "Patrol Cave/Normal Chest": 509342549,
    "Ruined Shop/Chest 1": 509342550,
    "Ruined Shop/Chest 2": 509342551,
    "Ruined Shop/Chest 3": 509342552,
    "Ruined Passage/Page Pickup": 509342553,
    "Shop/Potion 1": 509342554,
    "Shop/Potion 2": 509342555,
    "Shop/Coin 1": 509342556,
    "Shop/Coin 2": 509342557,
    "Special Shop/Secret Page Pickup": 509342558,
    "Stick House/Stick Chest": 509342559,
    "Sealed Temple/Page Pickup": 509342560,
    "Inside Hourglass Cave/Hourglass Chest": 509342561,
    "Secret Chest/Dash Across": 509342562,
    "Page Pickup/A Long, Long Time Ago...": 509342563,
    "Coins in the Well/10 Coins": 509342564,
    "Coins in the Well/15 Coins": 509342565,
    "Coins in the Well/3 Coins": 509342566,
    "Coins in the Well/6 Coins": 509342567,
    "Secret Gathering Place/20 Fairy Reward": 509342568,
    "Secret Gathering Place/10 Fairy Reward": 509342569,
    "[West] Moss Wall Holy Cross/Use the Holy Cross": 509342570,
    "[Southwest] Flowers Holy Cross/Use the Holy Cross": 509342571,
    "Fountain/[Southwest] Fountain Holy Cross": 509342572,
    "[Northeast] Flowers Holy Cross/Use the Holy Cross": 509342573,
    "[East] Weathervane Holy Cross/Use the Holy Cross": 509342574,
    "[West] Windmill Holy Cross/Sacred Geometry": 509342575,
    "Sand Hook/[Southwest] Haiku Holy Cross": 509342576,
    "[West] Windchimes Holy Cross/Power Up!": 509342577,
    "[South] Starting Platform Holy Cross/Back to Work": 509342578,
    "Magic Staff/[Northwest] Golden Obelisk Page": 509342579,
    "Inside the Old House/Holy Cross Door Page": 509342580,
    "Cube Cave/Holy Cross Chest": 509342581,
    "Southeast Cross Door/Chest 3": 509342582,
    "Southeast Cross Door/Chest 2": 509342583,
    "Southeast Cross Door/Chest 1": 509342584,
    "Maze Cave/Maze Room Holy Cross": 509342585,
    "Caustic Light Cave/Holy Cross Chest": 509342586,
    "Inside the Old House/Holy Cross Chest": 509342587,
    "Patrol Cave/Holy Cross Chest": 509342588,
    "Ruined Passage/Holy Cross Chest": 509342589,
    "Inside Hourglass Cave/Holy Cross Chest": 509342590,
    "Sealed Temple/Holy Cross Chest": 509342591,
    "Fountain Cross Door/Page Pickup": 509342592,
    "Secret Gathering Place/Holy Cross Chest": 509342593,
    "Mountain Door/Top of the Mountain - Page At The Peak": 509342594,
    "Monastery/Monastery Chest": 509342595,
    "[Back Entrance] Bushes Holy Cross/Use the Holy Cross": 509342596,
    "[Back Entrance] Chest/Peaceful Chest": 509342597,
    "[Central] Near Shortcut Ladder/By the Boxes": 509342598,
    "[East] Near Telescope/Spoopy": 509342599,
    "[East] Upper Floor/Reminds me of Blighttown": 509342600,
    "[Central] Below Entry Walkway/Even more Stairs!": 509342601,
    "[East] Obscured Near Winding Staircase/At the Bottom": 509342602,
    "[East] Obscured Beneath Scaffolding/In the Miasma Mound": 509342603,
    "[East] Obscured Near Telescope/Weird path?": 509342604,
    "[Back Entrance] Obscured Behind Wall/Happy Water!": 509342605,
    "[Central] Obscured Below Entry Walkway/Down the Stairs": 509342606,
    "[Central] Top Floor Overhang/End of the ruined bridge": 509342607,
    "[East] Near Bridge/Drop that Bridge!": 509342608,
    "[Central] Above Ladder/Climb Ladder": 509342609,
    "[Central] Obscured Behind Staircase/At the Bottom": 509342610,
    "[Central] Above Ladder Dash Chest/Dash Across": 509342611,
    "[West] Upper Area Bombable Wall/Boomy": 509342612,
    "[East] Bombable Wall/Flowers Guide Thee": 509342613,
    "Monastery/Hero's Grave - Ash Relic": 509342614,
    "[West] Shooting Range Secret Path/Obscured Path": 509342615,
    "[West] Near Shooting Range/End of bridge": 509342616,
    "[West] Below Shooting Range/Clever little sneak!": 509342617,
    "[Lowlands] Below Broken Ladder/Miasma Pits": 509342618,
    "[West] Upper Area Near Waterfall/Yummy Polygons": 509342619,
    "[Lowlands] Upper Walkway/Hate them Snipers": 509342620,
    "[West] Lower Area Below Bridge/Go Around": 509342621,
    "[West] Lower Area Isolated Chest/Burn Pots": 509342622,
    "[Lowlands] Near Elevator/End of the Tracks": 509342623,
    "[West] Lower Area After Bridge/Drop that Bridge!": 509342624,
    "Upper - Near Bridge Switch/You can shoot it": 509342625,
    "Upper - Beneath Bridge To Administrator/End of the First Floor": 509342626,
    "Tower - Inside Tower/I'm Scared": 509342627,
    "Lower - Near Corpses/They are Dead": 509342628,
    "Lower - Spider Ambush/Use the Gun": 509342629,
    "Lower - Left Of Checkpoint Before Fuse/Moment of Reprieve": 509342630,
    "Lower - After Guarded Fuse/Defeat those Mechs": 509342631,
    "Lower - Guarded By Double Turrets/Help": 509342632,
    "Lower - After 2nd Double Turret Chest/Haircut Time!": 509342633,
    "Lower - Guarded By Double Turrets 2/Oh god they're everywhere": 509342634,
    "Lower - Hexagon Blue/Scavenger Queen": 509342635,
    "[West] Near Kevin Block/Phonomath": 509342636,
    "[South] Upper Floor On Power Line/Hidden Ladder Chest": 509342637,
    "[South] Chest Near Big Crabs/His Name is Tom": 509342638,
    "[North] Guarded By Bird/Skraw!": 509342639,
    "[Northeast] Chest Beneath Brick Walkway/Mind the Crabbits": 509342640,
    "[Northwest] Bombable Wall/Flowers Guide Thee": 509342641,
    "[North] Obscured Beneath Bridge/In the shallow water": 509342642,
    "[South] Upper Floor On Bricks/Up the Ladder": 509342643,
    "[South] Near Birds/Danlarry and Thranmire ate Jerry!": 509342644,
    "[Northwest] Behind Envoy/Mind the Fairies": 509342645,
    "[Southwest] Obscured Behind Fuse/Saved by the Prayer": 509342646,
    "Locked Brick House/[East] Locked Room Upper Chest": 509342647,
    "[North] From Lower Overworld Entrance/Come from the Overworld": 509342648,
    "Locked Brick House/[East] Locked Room Lower Chest": 509342649,
    "[Northeast] Chest On Brick Walkway/Near Domain": 509342650,
    "[Southeast] Chest Near Fuse/Around the Tower": 509342651,
    "[Northeast] Key Pickup/Around the Hill": 509342652,
    "Cathedral Gauntlet/Gauntlet Reward": 509342653,
    "Secret Legend Trophy Chest/You can use the Holy Cross from the outside": 509342654,
    "[Upper Graveyard] Obscured Behind Hill/Between Two Hills": 509342655,
    "[South Graveyard] 4 Orange Skulls/DJ Khaled - Let's go Golfing!": 509342656,
    "[Central] Near Ramps Up/Up them Ramps": 509342657,
    "[Upper Graveyard] Near Shield Fleemers/Alternatively, Before the Cathedral": 509342658,
    "[South Graveyard] Obscured Behind Ridge/Hidden passage by ladder": 509342659,
    "[South Graveyard] Obscured Beneath Telescope/Through the Nook": 509342660,
    "[Entrance] Above Entryway/Dash Across": 509342661,
    "[Central] South Secret Passage/Wall Man Approves these Vibes": 509342662,
    "[South Graveyard] Upper Walkway On Pedestal/Gazing out over the Graves": 509342663,
    "[South Graveyard] Guarded By Tentacles/Isolated Island": 509342664,
    "[Upper Graveyard] Near Telescope/Overlooking the Graves": 509342665,
    "[Outside Cathedral] Near Moonlight Bridge Door/Down the Hidden Ladder": 509342666,
    "[Entrance] Obscured Inside Watchtower/Go Inside": 509342667,
    "[Entrance] South Near Fence/DAGGER STRAP!!!!!": 509342668,
    "[South Graveyard] Guarded By Big Skeleton/Super Clipping": 509342669,
    "[South Graveyard] Chest Near Graves/The Rest of Our Entire Life is Death": 509342670,
    "[Entrance] North Small Island/Mildly Hidden": 509342671,
    "First Hero's Grave/[Outside Cathedral] Obscured Behind Memorial": 509342672,
    "[Central] Obscured Behind Northern Mountain/Hug the Wall": 509342673,
    "[South Graveyard] Upper Walkway Dash Chest/Around the Hill": 509342674,
    "[South Graveyard] Above Big Skeleton/End of Ledge": 509342675,
    "[Central] Beneath Memorial/Do You Even Live?": 509342676,
    "First Hero's Grave/Hero's Grave - Feathers Relic": 509342677,
    "West Furnace/Chest": 509342678,
    "[West] Near Gardens Entrance/Effigy Skip": 509342679,
    "[Central Highlands] Holy Cross (Blue Lines)/Use the Holy Cross": 509342680,
    "[West Lowlands] Tree Holy Cross Chest/Use the Holy Cross": 509342681,
    "[Southeast Lowlands] Outside Cave/Mind the Chompignoms!": 509342682,
    "[Central Lowlands] Chest Beneath Faeries/As you walk by": 509342683,
    "[North] Behind Holy Cross Door/Extra Sword!": 509342684,
    "[Central Highlands] Top of Ladder Before Boss/Try to be This Strong": 509342685,
    "[Central Lowlands] Passage Beneath Bridge/Take the lower path": 509342686,
    "[North] Across From Page Pickup/I Love Fish!": 509342687,
    "[Central Lowlands] Below Left Walkway/Dash Across": 509342688,
    "[West] In Flooded Walkway/Dash through the water": 509342689,
    "[West] Past Flooded Walkway/Through the Shallow Water": 509342690,
    "[North] Obscured Beneath Hero's Memorial/Take the Long Way Around": 509342691,
    "[Central Lowlands] Chest Near Shortcut Bridge/Between a Rope and a Bridge Place": 509342692,
    "[West Highlands] Upper Left Walkway/By the Rudeling": 509342693,
    "[Central Lowlands] Chest Beneath Save Point/Behind the Way": 509342694,
    "[Central Highlands] Behind Guard Captain/Under Boss Ladder": 509342695,
    "[Central Highlands] After Garden Knight/Did Not Kill You": 509342696,
    "[South Highlands] Secret Chest Beneath Fuse/Pray to the Wall Man": 509342697,
    "[East Lowlands] Page Behind Ice Dagger House/Come from the Far Shore": 509342698,
    "[North] Page Pickup/Survival Tips": 509342699,
    "[Southeast Lowlands] Ice Dagger Pickup/Ice Dagger Cave": 509342700,
    "Hero's Grave/Effigy Relic": 509342701,
    "[East] Bombable Wall/Break Bombable Wall": 509350705,
    "[West] Upper Area Bombable Wall/Break Bombable Wall": 509350704,
    "[East Wing] Bombable Wall/Break Bombable Wall": 509350703,
    "Bombable Wall/Break Bombable Wall": 509350702,
    "[Northwest] Bombable Wall/Break Bombable Wall": 509350701,
    "[Southwest] Bombable Wall Near Fountain/Break Bombable Wall": 509350700,
    "Cube Cave/Break Bombable Wall": 509350699,
    "[Central] Bombable Wall/Break Bombable Wall": 509350698,
    "Purgatory Pots/Pot 33": 509350697,
    "Purgatory Pots/Pot 32": 509350696,
    "Purgatory Pots/Pot 31": 509350695,
    "Purgatory Pots/Pot 30": 509350694,
    "Purgatory Pots/Pot 29": 509350693,
    "Purgatory Pots/Pot 28": 509350692,
    "Purgatory Pots/Pot 27": 509350691,
    "Purgatory Pots/Pot 26": 509350690,
    "Purgatory Pots/Pot 25": 509350689,
    "Purgatory Pots/Pot 24": 509350688,
    "Purgatory Pots/Pot 23": 509350687,
    "Purgatory Pots/Pot 22": 509350686,
    "Purgatory Pots/Pot 21": 509350685,
    "Purgatory Pots/Pot 20": 509350684,
    "Purgatory Pots/Pot 19": 509350683,
    "Purgatory Pots/Pot 18": 509350682,
    "Purgatory Pots/Pot 17": 509350681,
    "Purgatory Pots/Pot 16": 509350680,
    "Purgatory Pots/Pot 15": 509350679,
    "Purgatory Pots/Pot 14": 509350678,
    "Purgatory Pots/Pot 13": 509350677,
    "Purgatory Pots/Pot 12": 509350676,
    "Purgatory Pots/Pot 11": 509350675,
    "Purgatory Pots/Pot 10": 509350674,
    "Purgatory Pots/Pot 9": 509350673,
    "Purgatory Pots/Pot 8": 509350672,
    "Purgatory Pots/Pot 7": 509350671,
    "Purgatory Pots/Pot 6": 509350670,
    "Purgatory Pots/Pot 5": 509350669,
    "Purgatory Pots/Pot 4": 509350668,
    "Purgatory Pots/Pot 3": 509350667,
    "Purgatory Pots/Pot 2": 509350666,
    "Purgatory Pots/Pot 1": 509350665,
    "[1F] Pots by Stairs/Pot 2": 509350664,
    "[1F] Pots by Stairs/Pot 1": 509350663,
    "Crates/Crate 9": 509350662,
    "Crates/Crate 8": 509350661,
    "Crates/Crate 7": 509350660,
    "Crates/Crate 6": 509350659,
    "Crates/Crate 5": 509350658,
    "Crates/Crate 4": 509350657,
    "Crates/Crate 3": 509350656,
    "Crates/Crate 2": 509350655,
    "Crates/Crate 1": 509350654,
    "[Lowlands] Crates/Crate 2": 509350653,
    "[Lowlands] Crates/Crate 1": 509350652,
    "[West] Near Isolated Chest/Crate 5": 509350651,
    "[West] Near Isolated Chest/Crate 4": 509350650,
    "[West] Near Isolated Chest/Crate 3": 509350649,
    "[West] Near Isolated Chest/Crate 2": 509350648,
    "[West] Near Isolated Chest/Crate 1": 509350647,
    "[West] Crates by Shooting Range/Crate 5": 509350646,
    "[West] Crates by Shooting Range/Crate 4": 509350645,
    "[West] Crates by Shooting Range/Crate 3": 509350644,
    "[West] Crates by Shooting Range/Crate 2": 509350643,
    "[West] Crates by Shooting Range/Crate 1": 509350642,
    "[West] Near Isolated Chest/Explosive Pot 2": 509350641,
    "[West] Near Isolated Chest/Explosive Pot 1": 509350640,
    "[West] Explosive Pot above Shooting Range/Explosive Pot": 509350639,
    "[West] Explosive Pots near Bombable Wall/Explosive Pot 2": 509350638,
    "[West] Explosive Pots near Bombable Wall/Explosive Pot 1": 509350637,
    "[Central] Crates near Shortcut Ladder/Crate 5": 509350636,
    "[Central] Crates near Shortcut Ladder/Crate 4": 509350635,
    "[Central] Crates near Shortcut Ladder/Crate 3": 509350634,
    "[Central] Crates near Shortcut Ladder/Crate 2": 509350633,
    "[Central] Crates near Shortcut Ladder/Crate 1": 509350632,
    "[Central] Explosive Pots near Shortcut Ladder/Explosive Pot 2": 509350631,
    "[Central] Explosive Pots near Shortcut Ladder/Explosive Pot 1": 509350630,
    "[Back Entrance] Pots/Pot 5": 509350629,
    "[Back Entrance] Pots/Pot 4": 509350628,
    "[Back Entrance] Pots/Pot 3": 509350627,
    "[Back Entrance] Pots/Pot 2": 509350626,
    "[Back Entrance] Pots/Pot 1": 509350625,
    "[Central] Explosive Pots near Monastery/Explosive Pot 2": 509350624,
    "[Central] Explosive Pots near Monastery/Explosive Pot 1": 509350623,
    "[East] Explosive Pot beneath Scaffolding/Explosive Pot": 509350622,
    "[East] Explosive Pots/Explosive Pot 3": 509350621,
    "[East] Explosive Pots/Explosive Pot 2": 509350620,
    "[East] Explosive Pots/Explosive Pot 1": 509350619,
    "Display Cases/Display Case 3": 509350618,
    "Display Cases/Display Case 2": 509350617,
    "Display Cases/Display Case 1": 509350616,
    "Orb Room Explosive Pots/Explosive Pot 2": 509350615,
    "Orb Room Explosive Pots/Explosive Pot 1": 509350614,
    "Pots after Gate/Pot 2": 509350613,
    "Pots after Gate/Pot 1": 509350612,
    "Slorm Room/Pot": 509350611,
    "Main Room Pots/Pot 2": 509350610,
    "Main Room Pots/Pot 1": 509350609,
    "Side Room Pots/Pot 3": 509350608,
    "Side Room Pots/Pot 2": 509350607,
    "Side Room Pots/Pot 1": 509350606,
    "Pots above Orb Altar/Pot 2": 509350605,
    "Pots above Orb Altar/Pot 1": 509350604,
    "[Upper] Pots/Pot 6": 509350603,
    "[Upper] Pots/Pot 5": 509350602,
    "[Upper] Pots/Pot 4": 509350601,
    "[Upper] Pots/Pot 3": 509350600,
    "[Upper] Pots/Pot 2": 509350599,
    "[Upper] Pots/Pot 1": 509350598,
    "[South] Explosive Pot near Birds/Explosive Pot": 509350597,
    "[West] Broken House/Table": 509350596,
    "[West] Broken House/Pot 2": 509350595,
    "[West] Broken House/Pot 1": 509350594,
    "Fortress Arena/Pot 2": 509350593,
    "Fortress Arena/Pot 1": 509350592,
    "Fortress Leaf Piles - Secret Chest/Leaf Pile 4": 509350591,
    "Fortress Leaf Piles - Secret Chest/Leaf Pile 3": 509350590,
    "Fortress Leaf Piles - Secret Chest/Leaf Pile 2": 509350589,
    "Fortress Leaf Piles - Secret Chest/Leaf Pile 1": 509350588,
    "Barrels/Back Room Barrel 7": 509350587,
    "Barrels/Back Room Barrel 6": 509350586,
    "Barrels/Back Room Barrel 5": 509350585,
    "[Northwest] Sign by Quarry Gate/Sign": 509350400,
    "[Central] Sign South of Checkpoint/Sign": 509350401,
    "[Central] Sign by Ruined Passage/Sign": 509350402,
    "[East] Pots near Slimes/Pot 1": 509350403,
    "[East] Pots near Slimes/Pot 2": 509350404,
    "[East] Pots near Slimes/Pot 3": 509350405,
    "[East] Pots near Slimes/Pot 4": 509350406,
    "[East] Pots near Slimes/Pot 5": 509350407,
    "[East] Forest Sign/Sign": 509350408,
    "[East] Fortress Sign/Sign": 509350409,
    "[North] Pots/Pot 1": 509350410,
    "[North] Pots/Pot 2": 509350411,
    "[North] Pots/Pot 3": 509350412,
    "[North] Pots/Pot 4": 509350413,
    "[West] Sign Near West Garden Entrance/Sign": 509350414,
    "Stick House/Pot 1": 509350415,
    "Stick House/Pot 2": 509350416,
    "Stick House/Pot 3": 509350417,
    "Stick House/Pot 4": 509350418,
    "Stick House/Pot 5": 509350419,
    "Stick House/Pot 6": 509350420,
    "Stick House/Pot 7": 509350421,
    "Ruined Shop/Pot 1": 509350422,
    "Ruined Shop/Pot 2": 509350423,
    "Ruined Shop/Pot 3": 509350424,
    "Ruined Shop/Pot 4": 509350425,
    "Ruined Shop/Pot 5": 509350426,
    "Inside Hourglass Cave/Sign": 509350427,
    "Pots by Slimes/Pot 1": 509350428,
    "Pots by Slimes/Pot 2": 509350429,
    "Pots by Slimes/Pot 3": 509350430,
    "Pots by Slimes/Pot 4": 509350431,
    "Pots by Slimes/Pot 5": 509350432,
    "Pots by Slimes/Pot 6": 509350433,
    "[Upper] Barrels/Barrel 1": 509350434,
    "[Upper] Barrels/Barrel 2": 509350435,
    "[Upper] Barrels/Barrel 3": 509350436,
    "Pots after Guard Captain/Pot 1": 509350437,
    "Pots after Guard Captain/Pot 2": 509350438,
    "Pots after Guard Captain/Pot 3": 509350439,
    "Pots after Guard Captain/Pot 4": 509350440,
    "Pots after Guard Captain/Pot 5": 509350441,
    "Pots after Guard Captain/Pot 6": 509350442,
    "Pots after Guard Captain/Pot 7": 509350443,
    "Pots after Guard Captain/Pot 8": 509350444,
    "Pots after Guard Captain/Pot 9": 509350445,
    "Pots/Pot 1": 509350446,
    "Pots/Pot 2": 509350447,
    "Pots/Pot 3": 509350448,
    "Pots/Pot 4": 509350449,
    "Pots/Pot 5": 509350450,
    "Sign by Grave Path/Sign": 509350451,
    "Sign by Guardhouse 1/Sign": 509350452,
    "Pots by Grave Path/Pot 1": 509350453,
    "Pots by Grave Path/Pot 2": 509350454,
    "Pots by Grave Path/Pot 3": 509350455,
    "Pots by Envoy/Pot 1": 509350456,
    "Pots by Envoy/Pot 2": 509350457,
    "Pots by Envoy/Pot 3": 509350458,
    "Bottom Floor Pots/Pot 1": 509350459,
    "Bottom Floor Pots/Pot 2": 509350460,
    "Bottom Floor Pots/Pot 3": 509350461,
    "Bottom Floor Pots/Pot 4": 509350462,
    "Bottom Floor Pots/Pot 5": 509350463,
    "[Side Room] Pots by Chest/Pot 1": 509350464,
    "[Side Room] Pots by Chest/Pot 2": 509350465,
    "[Side Room] Pots by Chest/Pot 3": 509350466,
    "[Third Room] Barrels by Bridge/Barrel 1": 509350467,
    "[Third Room] Barrels by Bridge/Barrel 2": 509350468,
    "[Third Room] Barrels by Bridge/Barrel 3": 509350469,
    "[Third Room] Barrels after Back Corridor/Barrel 1": 509350470,
    "[Third Room] Barrels after Back Corridor/Barrel 2": 509350471,
    "[Third Room] Barrels after Back Corridor/Barrel 3": 509350472,
    "[Third Room] Barrels after Back Corridor/Barrel 4": 509350473,
    "[Third Room] Barrels after Back Corridor/Barrel 5": 509350474,
    "[Third Room] Barrels by West Turret/Barrel 1": 509350475,
    "[Third Room] Barrels by West Turret/Barrel 2": 509350476,
    "[Third Room] Barrels by West Turret/Barrel 3": 509350477,
    "[Third Room] Pots by East Turret/Pot 1": 509350478,
    "[Third Room] Pots by East Turret/Pot 2": 509350479,
    "[Third Room] Pots by East Turret/Pot 3": 509350480,
    "[Third Room] Pots by East Turret/Pot 4": 509350481,
    "[Third Room] Pots by East Turret/Pot 5": 509350482,
    "[Third Room] Pots by East Turret/Pot 6": 509350483,
    "[Third Room] Pots by East Turret/Pot 7": 509350484,
    "Barrels/Barrel 1": 509350485,
    "Barrels/Barrel 2": 509350486,
    "Pot Hallway Pots/Pot 1": 509350487,
    "Pot Hallway Pots/Pot 2": 509350488,
    "Pot Hallway Pots/Pot 3": 509350489,
    "Pot Hallway Pots/Pot 4": 509350490,
    "Pot Hallway Pots/Pot 5": 509350491,
    "Pot Hallway Pots/Pot 6": 509350492,
    "Pot Hallway Pots/Pot 7": 509350493,
    "Pot Hallway Pots/Pot 8": 509350494,
    "Pot Hallway Pots/Pot 9": 509350495,
    "Pot Hallway Pots/Pot 10": 509350496,
    "Pot Hallway Pots/Pot 11": 509350497,
    "Pot Hallway Pots/Pot 12": 509350498,
    "Pot Hallway Pots/Pot 13": 509350499,
    "Pot Hallway Pots/Pot 14": 509350500,
    "2nd Laser Room Pots/Pot 1": 509350501,
    "2nd Laser Room Pots/Pot 2": 509350502,
    "2nd Laser Room Pots/Pot 3": 509350503,
    "2nd Laser Room Pots/Pot 4": 509350504,
    "2nd Laser Room Pots/Pot 5": 509350505,
    "[Southeast Lowlands] Ice Dagger Pickup/Pot 1": 509350506,
    "[Southeast Lowlands] Ice Dagger Pickup/Pot 2": 509350507,
    "[Southeast Lowlands] Ice Dagger Pickup/Pot 3": 509350508,
    "Fire Pots/Fire Pot 1": 509350509,
    "Fire Pots/Fire Pot 2": 509350510,
    "Fire Pots/Fire Pot 3": 509350511,
    "Fire Pots/Fire Pot 4": 509350512,
    "Fire Pots/Fire Pot 5": 509350513,
    "Fire Pots/Fire Pot 6": 509350514,
    "Fire Pots/Fire Pot 7": 509350515,
    "Fire Pots/Fire Pot 8": 509350516,
    "Upper Fire Pot/Fire Pot": 509350517,
    "[Entry] Pots/Pot 1": 509350518,
    "[Entry] Pots/Pot 2": 509350519,
    "[By Grave] Pots/Pot 1": 509350520,
    "[By Grave] Pots/Pot 2": 509350521,
    "[By Grave] Pots/Pot 3": 509350522,
    "[By Grave] Pots/Pot 4": 509350523,
    "[By Grave] Pots/Pot 5": 509350524,
    "[By Grave] Pots/Pot 6": 509350525,
    "[Central] Fire Pots/Fire Pot 1": 509350526,
    "[Central] Fire Pots/Fire Pot 2": 509350527,
    "[Central] Pots by Door/Pot 1": 509350528,
    "[Central] Pots by Door/Pot 2": 509350529,
    "[Central] Pots by Door/Pot 3": 509350530,
    "[Central] Pots by Door/Pot 4": 509350531,
    "[Central] Pots by Door/Pot 5": 509350532,
    "[Central] Pots by Door/Pot 6": 509350533,
    "[Central] Pots by Door/Pot 7": 509350534,
    "[Central] Pots by Door/Pot 8": 509350535,
    "[Central] Pots by Door/Pot 9": 509350536,
    "[Central] Pots by Door/Pot 10": 509350537,
    "[Central] Pots by Door/Pot 11": 509350538,
    "[East Wing] Pots by Broken Checkpoint/Pot 1": 509350539,
    "[East Wing] Pots by Broken Checkpoint/Pot 2": 509350540,
    "[East Wing] Pots by Broken Checkpoint/Pot 3": 509350541,
    "[West Wing] Pots by Checkpoint/Pot 1": 509350542,
    "[West Wing] Pots by Checkpoint/Pot 2": 509350543,
    "[West Wing] Pots by Checkpoint/Pot 3": 509350544,
    "[West Wing] Pots by Overlook/Pot 1": 509350545,
    "[West Wing] Pots by Overlook/Pot 2": 509350546,
    "[West Wing] Slorm Room Pots/Pot 1": 509350547,
    "[West Wing] Slorm Room Pots/Pot 2": 509350548,
    "[West Wing] Slorm Room Pots/Pot 3": 509350549,
    "[West Wing] Chest Room Pots/Pot 1": 509350550,
    "[West Wing] Chest Room Pots/Pot 2": 509350551,
    "[West Wing] Pots by Stairs to Basement/Pot 1": 509350552,
    "[West Wing] Pots by Stairs to Basement/Pot 2": 509350553,
    "[West Wing] Pots by Stairs to Basement/Pot 3": 509350554,
    "Entry Spot/Pot 1": 509350555,
    "Entry Spot/Pot 2": 509350556,
    "Entry Spot/Crate 1": 509350557,
    "Entry Spot/Crate 2": 509350558,
    "Entry Spot/Crate 3": 509350559,
    "Entry Spot/Crate 4": 509350560,
    "Entry Spot/Crate 5": 509350561,
    "Entry Spot/Crate 6": 509350562,
    "Entry Spot/Crate 7": 509350563,
    "Slorm Room Crates/Crate 1": 509350564,
    "Slorm Room Crates/Crate 2": 509350565,
    "Crates under Rope/Crate 1": 509350566,
    "Crates under Rope/Crate 2": 509350567,
    "Crates under Rope/Crate 3": 509350568,
    "Crates under Rope/Crate 4": 509350569,
    "Crates under Rope/Crate 5": 509350570,
    "Crates under Rope/Crate 6": 509350571,
    "Fuse Room Fire Pots/Fire Pot 1": 509350572,
    "Fuse Room Fire Pots/Fire Pot 2": 509350573,
    "Fuse Room Fire Pots/Fire Pot 3": 509350574,
    "Barrels/Barrel by Back Room 1": 509350575,
    "Barrels/Barrel by Back Room 2": 509350576,
    "Barrels/Barrel by Back Room 3": 509350577,
    "Barrels/Barrel by Back Room 4": 509350578,
    "Barrels/Barrel by Back Room 5": 509350579,
    "Barrels/Barrel by Back Room 6": 509350580,
    "Barrels/Back Room Barrel 1": 509350581,
    "Barrels/Back Room Barrel 2": 509350582,
    "Barrels/Back Room Barrel 3": 509350583,
    "[Powered Secret Room] Chest/Follow the Purple Energy Road": 509342400,
    "[Entryway] Chest/Mind the Slorms": 509342401,
    "[Third Room] Beneath Platform Chest/Run from the tentacles!": 509342402,
    "[Third Room] Tentacle Chest/Water Sucks": 509342403,
    "[Entryway] Obscured Behind Waterfall/You can just go in there": 509342404,
    "[Save Room] Upper Floor Chest 1/Through the Power of Prayer": 509342405,
    "[Save Room] Upper Floor Chest 2/Above the Fox Shrine": 509342406,
    "[Second Room] Underwater Chest/Hidden Passage": 509342407,
    "[Back Corridor] Right Secret/Hidden Path": 509342408,
    "[Back Corridor] Left Secret/Behind the Slorms": 509342409,
    "[Second Room] Obscured Behind Waterfall/Just go in there": 509342410,
    "[Side Room] Chest By Pots/Just Climb up There": 509342411,
    "[Side Room] Chest By Phrends/So Many Phrends!": 509342412,
    "[Second Room] Page/Ruined Atoll Map": 509342413,
    "[Passage To Dark Tomb] Page Pickup/Siege Engine": 509342414,
    "[1F] Guarded By Lasers/Beside 3 Miasma Seekers": 509342415,
    "[1F] Near Spikes/Mind the Miasma Seeker": 509342416,
    "Birdcage Room/[2F] Bird Room": 509342417,
    "[2F] Entryway Upper Walkway/Overlooking Miasma": 509342418,
    "[1F] Library/By the Books": 509342419,
    "[2F] Library/Behind the Ladder": 509342420,
    "[2F] Guarded By Lasers/Before the big reveal...": 509342421,
    "Birdcage Room/[2F] Bird Room Secret": 509342422,
    "[1F] Library Secret/Pray to the Wallman": 509342423,
    "Spike Maze Near Exit/Watch out!": 509342424,
    "2nd Laser Room/Can you roll?": 509342425,
    "1st Laser Room/Use a bomb?": 509342426,
    "Spike Maze Upper Walkway/Just walk right!": 509342427,
    "Skulls Chest/Move the Grave": 509342428,
    "Spike Maze Near Stairs/In the Corner": 509342429,
    "1st Laser Room Obscured/Follow the red laser of death": 509342430,
    "Guardhouse 2 - Upper Floor/In the Mound": 509342431,
    "Guardhouse 2 - Bottom Floor Secret/Hidden Hallway": 509342432,
    "Guardhouse 1 Obscured/Upper Floor Obscured": 509342433,
    "Guardhouse 1/Upper Floor": 509342434,
    "Guardhouse 1 Ledge HC/Dancing Fox Spirit Holy Cross": 509342435,
    "Golden Obelisk Holy Cross/Use the Holy Cross": 509342436,
    "Ice Rod Grapple Chest/Freeze the Blob and ascend With Orb": 509342437,
    "Above Save Point/Chest": 509342438,
    "Above Save Point Obscured/Hidden Path": 509342439,
    "Guardhouse 1 Ledge/From Guardhouse 1 Chest": 509342440,
    "Near Save Point/Chest": 509342441,
    "Ambushed by Spiders/Beneath Spider Chest": 509342442,
    "Near Telescope/Up on the Wall": 509342443,
    "Ambushed by Spiders/Spider Chest": 509342444,
    "Lower Dash Chest/Dash Across": 509342445,
    "Lower Grapple Chest/Grapple Across": 509342446,
    "Bombable Wall/Follow the Flowers": 509342447,
    "Page On Teleporter/Page": 509342448,
    "Forest Belltower Save Point/Near Save Point": 509342449,
    "Forest Belltower - After Guard Captain/Chest": 509342450,
    "East Bell/Forest Belltower - Obscured Near Bell Top Floor": 509342451,
    "Forest Belltower Obscured/Obscured Beneath Bell Bottom Floor": 509342452,
    "Forest Belltower Page/Page Pickup": 509342453,
    "Forest Grave Path - Holy Cross Code by Grave/Single Money Chest": 509342454,
    "Forest Grave Path - Above Gate/Chest": 509342455,
    "Forest Grave Path - Obscured Chest/Behind the Trees": 509342456,
    "Forest Grave Path - Upper Walkway/From the top of the Guardhouse": 509342457,
    "The Hero's Sword/Forest Grave Path - Sword Pickup": 509342458,
    "The Hero's Sword/Hero's Grave - Tooth Relic": 509342459,
    "Fortress Courtyard - From East Belltower/Crack in the Wall": 509342460,
    "Fortress Leaf Piles - Secret Chest/Dusty": 509342461,
    "Fortress Arena/Hexagon Red": 509342462,
    "Fortress Arena/Siege Engine|Vault Key Pickup": 509342463,
    "Fortress East Shortcut - Chest Near Slimes/Mind the Custodians": 509342464,
    "[West Wing] Candles Holy Cross/Use the Holy Cross": 509342465,
    "Westmost Upper Room/[West Wing] Dark Room Chest 1": 509342466,
    "Westmost Upper Room/[West Wing] Dark Room Chest 2": 509342467,
    "[East Wing] Bombable Wall/Bomb the Wall": 509342468,
    "[West Wing] Page Pickup/He will never visit the Far Shore": 509342469,
    "Fortress Grave Path - Upper Walkway/Go Around the East Wing": 509342470,
    "Vault Hero's Grave/Fortress Grave Path - Chest Right of Grave": 509342471,
    "Vault Hero's Grave/Fortress Grave Path - Obscured Chest Left of Grave": 509342472,
    "Vault Hero's Grave/Hero's Grave - Flowers Relic": 509342473,
    "Bridge/Chest": 509342474,
    "Cell Chest 1/Drop the Shortcut Rope": 509342475,
    "Obscured Behind Waterfall/Muffling Bell": 509342476,
    "Back Room Chest/Lose the Lure or take 2 Damage": 509342477,
    "Cell Chest 2/Mind the Custodian": 509342478,
    "Near Vault/Already Stolen": 509342479,
    "Slorm Room/Tobias was Trapped Here Once...": 509342480,
    "Escape Chest/Don't Kick Fimbleton!": 509342481,
    "Grapple Above Hot Tub/Look Up": 509342482,
    "Above Vault/Obscured Doorway Ledge": 509342483,
    "Main Room Top Floor/Mind the Adult Frog": 509342484,
    "Main Room Bottom Floor/Altar Chest": 509342485,
    "Side Room Secret Passage/Upper Right Corner": 509342486,
    "Side Room Chest/Oh No! Our Frogs! They're Dead!": 509342487,
    "Side Room Grapple Secret/Grapple on Over": 509342488,
    "Magic Orb Pickup/Frult Meeting": 509342489,
    "The Librarian/Hexagon Green": 509342490,
    "Library Hall/Holy Cross Chest": 509342491,
    "Library Lab Chest by Shrine 2/Chest": 509342492,
    "Barrels/Back Room Barrel 4": 509350584,
}


# for setting up the poptracker integration
tracker_world = {
    "map_page_maps": ["maps/maps_pop.json"],
    "map_page_locations": ["locations/locations_pop_er.json", "locations/locations_breakables.json"],
    "map_page_setting_key": "Slot:{player}:Current Map",
    "map_page_index": map_page_index,
    "external_pack_key": "ut_poptracker_path",
    "poptracker_name_mapping": poptracker_data
}
