import typing

from enum import Enum
from .duelists import Duelist

id_to_duelist: typing.Dict[int, Duelist] = {duelist.id: duelist for duelist in Duelist}


class DuelRank(Enum):
    SAPOW = 1
    BCD = 2
    SATEC = 3


class Drop:
    card_id: int
    duelist: Duelist
    probability: int  # value / 2048
    duel_rank: DuelRank

    def __init__(self, card_id: int, duelist_id: int, probability: int, pool: DuelRank):
        self.card_id = card_id
        self.duelist = id_to_duelist[duelist_id]
        self.probability = probability
        self.duel_rank = pool

    def __str__(self) -> str:
        return f"Card {self.card_id}: {self.probability} from {self.duelist} {self.duel_rank}"


drop_table: typing.Tuple[Drop, ...] = (
    # SAPOW drops
    Drop(2, 1, 27, DuelRank.SAPOW),
    Drop(9, 1, 90, DuelRank.SAPOW),
    Drop(10, 1, 30, DuelRank.SAPOW),
    Drop(16, 1, 60, DuelRank.SAPOW),
    Drop(19, 1, 6, DuelRank.SAPOW),
    Drop(24, 1, 27, DuelRank.SAPOW),
    Drop(25, 1, 27, DuelRank.SAPOW),
    Drop(30, 1, 27, DuelRank.SAPOW),
    Drop(41, 1, 27, DuelRank.SAPOW),
    Drop(46, 1, 27, DuelRank.SAPOW),
    Drop(47, 1, 27, DuelRank.SAPOW),
    Drop(48, 1, 27, DuelRank.SAPOW),
    Drop(58, 1, 30, DuelRank.SAPOW),
    Drop(59, 1, 27, DuelRank.SAPOW),
    Drop(65, 1, 27, DuelRank.SAPOW),
    Drop(105, 1, 52, DuelRank.SAPOW),
    Drop(123, 1, 52, DuelRank.SAPOW),
    Drop(130, 1, 52, DuelRank.SAPOW),
    Drop(137, 1, 52, DuelRank.SAPOW),
    Drop(167, 1, 52, DuelRank.SAPOW),
    Drop(192, 1, 52, DuelRank.SAPOW),
    Drop(197, 1, 52, DuelRank.SAPOW),
    Drop(202, 1, 52, DuelRank.SAPOW),
    Drop(237, 1, 52, DuelRank.SAPOW),
    Drop(238, 1, 12, DuelRank.SAPOW),
    Drop(289, 1, 52, DuelRank.SAPOW),
    Drop(333, 1, 40, DuelRank.SAPOW),
    Drop(381, 1, 2, DuelRank.SAPOW),
    Drop(387, 1, 52, DuelRank.SAPOW),
    Drop(397, 1, 52, DuelRank.SAPOW),
    Drop(402, 1, 52, DuelRank.SAPOW),
    Drop(410, 1, 52, DuelRank.SAPOW),
    Drop(411, 1, 52, DuelRank.SAPOW),
    Drop(422, 1, 52, DuelRank.SAPOW),
    Drop(436, 1, 52, DuelRank.SAPOW),
    Drop(444, 1, 52, DuelRank.SAPOW),
    Drop(469, 1, 52, DuelRank.SAPOW),
    Drop(484, 1, 52, DuelRank.SAPOW),
    Drop(485, 1, 52, DuelRank.SAPOW),
    Drop(488, 1, 52, DuelRank.SAPOW),
    Drop(504, 1, 52, DuelRank.SAPOW),
    Drop(516, 1, 52, DuelRank.SAPOW),
    Drop(547, 1, 52, DuelRank.SAPOW),
    Drop(548, 1, 52, DuelRank.SAPOW),
    Drop(558, 1, 52, DuelRank.SAPOW),
    Drop(563, 1, 52, DuelRank.SAPOW),
    Drop(635, 1, 52, DuelRank.SAPOW),
    Drop(19, 2, 6, DuelRank.SAPOW),
    Drop(24, 2, 180, DuelRank.SAPOW),
    Drop(58, 2, 180, DuelRank.SAPOW),
    Drop(102, 2, 12, DuelRank.SAPOW),
    Drop(167, 2, 180, DuelRank.SAPOW),
    Drop(302, 2, 16, DuelRank.SAPOW),
    Drop(330, 2, 30, DuelRank.SAPOW),
    Drop(393, 2, 176, DuelRank.SAPOW),
    Drop(395, 2, 176, DuelRank.SAPOW),
    Drop(398, 2, 176, DuelRank.SAPOW),
    Drop(399, 2, 176, DuelRank.SAPOW),
    Drop(402, 2, 176, DuelRank.SAPOW),
    Drop(469, 2, 176, DuelRank.SAPOW),
    Drop(475, 2, 176, DuelRank.SAPOW),
    Drop(527, 2, 176, DuelRank.SAPOW),
    Drop(543, 2, 12, DuelRank.SAPOW),
    Drop(566, 2, 12, DuelRank.SAPOW),
    Drop(570, 2, 6, DuelRank.SAPOW),
    Drop(580, 2, 6, DuelRank.SAPOW),
    Drop(4, 3, 20, DuelRank.SAPOW),
    Drop(16, 3, 64, DuelRank.SAPOW),
    Drop(20, 3, 6, DuelRank.SAPOW),
    Drop(29, 3, 20, DuelRank.SAPOW),
    Drop(50, 3, 44, DuelRank.SAPOW),
    Drop(61, 3, 20, DuelRank.SAPOW),
    Drop(100, 3, 20, DuelRank.SAPOW),
    Drop(104, 3, 44, DuelRank.SAPOW),
    Drop(105, 3, 44, DuelRank.SAPOW),
    Drop(122, 3, 44, DuelRank.SAPOW),
    Drop(130, 3, 44, DuelRank.SAPOW),
    Drop(135, 3, 44, DuelRank.SAPOW),
    Drop(137, 3, 44, DuelRank.SAPOW),
    Drop(152, 3, 44, DuelRank.SAPOW),
    Drop(154, 3, 44, DuelRank.SAPOW),
    Drop(174, 3, 44, DuelRank.SAPOW),
    Drop(182, 3, 44, DuelRank.SAPOW),
    Drop(185, 3, 44, DuelRank.SAPOW),
    Drop(191, 3, 44, DuelRank.SAPOW),
    Drop(197, 3, 44, DuelRank.SAPOW),
    Drop(202, 3, 44, DuelRank.SAPOW),
    Drop(207, 3, 44, DuelRank.SAPOW),
    Drop(210, 3, 44, DuelRank.SAPOW),
    Drop(214, 3, 44, DuelRank.SAPOW),
    Drop(237, 3, 44, DuelRank.SAPOW),
    Drop(242, 3, 44, DuelRank.SAPOW),
    Drop(245, 3, 44, DuelRank.SAPOW),
    Drop(268, 3, 44, DuelRank.SAPOW),
    Drop(333, 3, 25, DuelRank.SAPOW),
    Drop(410, 3, 44, DuelRank.SAPOW),
    Drop(420, 3, 44, DuelRank.SAPOW),
    Drop(422, 3, 44, DuelRank.SAPOW),
    Drop(436, 3, 44, DuelRank.SAPOW),
    Drop(444, 3, 44, DuelRank.SAPOW),
    Drop(485, 3, 44, DuelRank.SAPOW),
    Drop(486, 3, 44, DuelRank.SAPOW),
    Drop(488, 3, 44, DuelRank.SAPOW),
    Drop(492, 3, 44, DuelRank.SAPOW),
    Drop(501, 3, 44, DuelRank.SAPOW),
    Drop(516, 3, 44, DuelRank.SAPOW),
    Drop(524, 3, 44, DuelRank.SAPOW),
    Drop(549, 3, 44, DuelRank.SAPOW),
    Drop(558, 3, 44, DuelRank.SAPOW),
    Drop(563, 3, 44, DuelRank.SAPOW),
    Drop(570, 3, 6, DuelRank.SAPOW),
    Drop(579, 3, 44, DuelRank.SAPOW),
    Drop(580, 3, 6, DuelRank.SAPOW),
    Drop(581, 3, 2, DuelRank.SAPOW),
    Drop(588, 3, 2, DuelRank.SAPOW),
    Drop(589, 3, 44, DuelRank.SAPOW),
    Drop(598, 3, 44, DuelRank.SAPOW),
    Drop(607, 3, 2, DuelRank.SAPOW),
    Drop(611, 3, 44, DuelRank.SAPOW),
    Drop(616, 3, 2, DuelRank.SAPOW),
    Drop(651, 3, 33, DuelRank.SAPOW),
    Drop(681, 3, 16, DuelRank.SAPOW),
    Drop(9, 4, 74, DuelRank.SAPOW),
    Drop(20, 4, 6, DuelRank.SAPOW),
    Drop(24, 4, 74, DuelRank.SAPOW),
    Drop(58, 4, 74, DuelRank.SAPOW),
    Drop(112, 4, 12, DuelRank.SAPOW),
    Drop(123, 4, 74, DuelRank.SAPOW),
    Drop(146, 4, 12, DuelRank.SAPOW),
    Drop(153, 4, 12, DuelRank.SAPOW),
    Drop(165, 4, 12, DuelRank.SAPOW),
    Drop(167, 4, 74, DuelRank.SAPOW),
    Drop(192, 4, 74, DuelRank.SAPOW),
    Drop(234, 4, 12, DuelRank.SAPOW),
    Drop(241, 4, 12, DuelRank.SAPOW),
    Drop(289, 4, 74, DuelRank.SAPOW),
    Drop(307, 4, 16, DuelRank.SAPOW),
    Drop(332, 4, 32, DuelRank.SAPOW),
    Drop(338, 4, 32, DuelRank.SAPOW),
    Drop(387, 4, 74, DuelRank.SAPOW),
    Drop(393, 4, 74, DuelRank.SAPOW),
    Drop(395, 4, 74, DuelRank.SAPOW),
    Drop(397, 4, 74, DuelRank.SAPOW),
    Drop(398, 4, 74, DuelRank.SAPOW),
    Drop(399, 4, 74, DuelRank.SAPOW),
    Drop(402, 4, 74, DuelRank.SAPOW),
    Drop(411, 4, 74, DuelRank.SAPOW),
    Drop(435, 4, 12, DuelRank.SAPOW),
    Drop(450, 4, 12, DuelRank.SAPOW),
    Drop(469, 4, 74, DuelRank.SAPOW),
    Drop(475, 4, 74, DuelRank.SAPOW),
    Drop(480, 4, 12, DuelRank.SAPOW),
    Drop(484, 4, 74, DuelRank.SAPOW),
    Drop(502, 4, 12, DuelRank.SAPOW),
    Drop(504, 4, 74, DuelRank.SAPOW),
    Drop(527, 4, 74, DuelRank.SAPOW),
    Drop(547, 4, 74, DuelRank.SAPOW),
    Drop(548, 4, 74, DuelRank.SAPOW),
    Drop(558, 4, 74, DuelRank.SAPOW),
    Drop(570, 4, 10, DuelRank.SAPOW),
    Drop(580, 4, 8, DuelRank.SAPOW),
    Drop(635, 4, 74, DuelRank.SAPOW),
    Drop(655, 4, 16, DuelRank.SAPOW),
    Drop(681, 4, 16, DuelRank.SAPOW),
    Drop(690, 4, 16, DuelRank.SAPOW),
    Drop(9, 5, 15, DuelRank.SAPOW),
    Drop(10, 5, 15, DuelRank.SAPOW),
    Drop(19, 5, 6, DuelRank.SAPOW),
    Drop(23, 5, 15, DuelRank.SAPOW),
    Drop(24, 5, 15, DuelRank.SAPOW),
    Drop(25, 5, 15, DuelRank.SAPOW),
    Drop(30, 5, 30, DuelRank.SAPOW),
    Drop(34, 5, 15, DuelRank.SAPOW),
    Drop(36, 5, 15, DuelRank.SAPOW),
    Drop(40, 5, 15, DuelRank.SAPOW),
    Drop(45, 5, 15, DuelRank.SAPOW),
    Drop(53, 5, 15, DuelRank.SAPOW),
    Drop(59, 5, 15, DuelRank.SAPOW),
    Drop(61, 5, 15, DuelRank.SAPOW),
    Drop(65, 5, 15, DuelRank.SAPOW),
    Drop(80, 5, 15, DuelRank.SAPOW),
    Drop(96, 5, 15, DuelRank.SAPOW),
    Drop(97, 5, 15, DuelRank.SAPOW),
    Drop(98, 5, 15, DuelRank.SAPOW),
    Drop(102, 5, 10, DuelRank.SAPOW),
    Drop(103, 5, 15, DuelRank.SAPOW),
    Drop(108, 5, 15, DuelRank.SAPOW),
    Drop(109, 5, 15, DuelRank.SAPOW),
    Drop(110, 5, 15, DuelRank.SAPOW),
    Drop(115, 5, 15, DuelRank.SAPOW),
    Drop(118, 5, 15, DuelRank.SAPOW),
    Drop(119, 5, 15, DuelRank.SAPOW),
    Drop(120, 5, 15, DuelRank.SAPOW),
    Drop(121, 5, 15, DuelRank.SAPOW),
    Drop(132, 5, 30, DuelRank.SAPOW),
    Drop(135, 5, 15, DuelRank.SAPOW),
    Drop(138, 5, 15, DuelRank.SAPOW),
    Drop(139, 5, 15, DuelRank.SAPOW),
    Drop(140, 5, 30, DuelRank.SAPOW),
    Drop(141, 5, 15, DuelRank.SAPOW),
    Drop(152, 5, 15, DuelRank.SAPOW),
    Drop(154, 5, 15, DuelRank.SAPOW),
    Drop(162, 5, 15, DuelRank.SAPOW),
    Drop(164, 5, 15, DuelRank.SAPOW),
    Drop(167, 5, 15, DuelRank.SAPOW),
    Drop(169, 5, 15, DuelRank.SAPOW),
    Drop(171, 5, 15, DuelRank.SAPOW),
    Drop(172, 5, 15, DuelRank.SAPOW),
    Drop(177, 5, 15, DuelRank.SAPOW),
    Drop(181, 5, 15, DuelRank.SAPOW),
    Drop(191, 5, 15, DuelRank.SAPOW),
    Drop(196, 5, 30, DuelRank.SAPOW),
    Drop(197, 5, 15, DuelRank.SAPOW),
    Drop(199, 5, 15, DuelRank.SAPOW),
    Drop(201, 5, 15, DuelRank.SAPOW),
    Drop(203, 5, 15, DuelRank.SAPOW),
    Drop(205, 5, 15, DuelRank.SAPOW),
    Drop(206, 5, 15, DuelRank.SAPOW),
    Drop(211, 5, 15, DuelRank.SAPOW),
    Drop(215, 5, 15, DuelRank.SAPOW),
    Drop(219, 5, 15, DuelRank.SAPOW),
    Drop(220, 5, 15, DuelRank.SAPOW),
    Drop(221, 5, 15, DuelRank.SAPOW),
    Drop(225, 5, 15, DuelRank.SAPOW),
    Drop(227, 5, 15, DuelRank.SAPOW),
    Drop(228, 5, 15, DuelRank.SAPOW),
    Drop(231, 5, 15, DuelRank.SAPOW),
    Drop(233, 5, 15, DuelRank.SAPOW),
    Drop(236, 5, 15, DuelRank.SAPOW),
    Drop(237, 5, 15, DuelRank.SAPOW),
    Drop(238, 5, 10, DuelRank.SAPOW),
    Drop(243, 5, 15, DuelRank.SAPOW),
    Drop(244, 5, 15, DuelRank.SAPOW),
    Drop(246, 5, 15, DuelRank.SAPOW),
    Drop(248, 5, 15, DuelRank.SAPOW),
    Drop(250, 5, 15, DuelRank.SAPOW),
    Drop(251, 5, 15, DuelRank.SAPOW),
    Drop(256, 5, 15, DuelRank.SAPOW),
    Drop(257, 5, 30, DuelRank.SAPOW),
    Drop(258, 5, 30, DuelRank.SAPOW),
    Drop(259, 5, 15, DuelRank.SAPOW),
    Drop(262, 5, 15, DuelRank.SAPOW),
    Drop(263, 5, 30, DuelRank.SAPOW),
    Drop(265, 5, 15, DuelRank.SAPOW),
    Drop(269, 5, 15, DuelRank.SAPOW),
    Drop(270, 5, 15, DuelRank.SAPOW),
    Drop(272, 5, 15, DuelRank.SAPOW),
    Drop(273, 5, 15, DuelRank.SAPOW),
    Drop(274, 5, 15, DuelRank.SAPOW),
    Drop(276, 5, 15, DuelRank.SAPOW),
    Drop(280, 5, 15, DuelRank.SAPOW),
    Drop(289, 5, 15, DuelRank.SAPOW),
    Drop(290, 5, 15, DuelRank.SAPOW),
    Drop(291, 5, 15, DuelRank.SAPOW),
    Drop(292, 5, 15, DuelRank.SAPOW),
    Drop(293, 5, 15, DuelRank.SAPOW),
    Drop(294, 5, 15, DuelRank.SAPOW),
    Drop(296, 5, 15, DuelRank.SAPOW),
    Drop(335, 5, 24, DuelRank.SAPOW),
    Drop(393, 5, 14, DuelRank.SAPOW),
    Drop(394, 5, 10, DuelRank.SAPOW),
    Drop(398, 5, 15, DuelRank.SAPOW),
    Drop(406, 5, 15, DuelRank.SAPOW),
    Drop(414, 5, 15, DuelRank.SAPOW),
    Drop(432, 5, 15, DuelRank.SAPOW),
    Drop(444, 5, 15, DuelRank.SAPOW),
    Drop(446, 5, 15, DuelRank.SAPOW),
    Drop(451, 5, 15, DuelRank.SAPOW),
    Drop(452, 5, 15, DuelRank.SAPOW),
    Drop(461, 5, 15, DuelRank.SAPOW),
    Drop(463, 5, 15, DuelRank.SAPOW),
    Drop(484, 5, 15, DuelRank.SAPOW),
    Drop(496, 5, 15, DuelRank.SAPOW),
    Drop(503, 5, 15, DuelRank.SAPOW),
    Drop(514, 5, 15, DuelRank.SAPOW),
    Drop(516, 5, 15, DuelRank.SAPOW),
    Drop(524, 5, 15, DuelRank.SAPOW),
    Drop(548, 5, 15, DuelRank.SAPOW),
    Drop(549, 5, 15, DuelRank.SAPOW),
    Drop(552, 5, 15, DuelRank.SAPOW),
    Drop(556, 5, 15, DuelRank.SAPOW),
    Drop(558, 5, 15, DuelRank.SAPOW),
    Drop(567, 5, 14, DuelRank.SAPOW),
    Drop(576, 5, 14, DuelRank.SAPOW),
    Drop(584, 5, 14, DuelRank.SAPOW),
    Drop(586, 5, 10, DuelRank.SAPOW),
    Drop(591, 5, 14, DuelRank.SAPOW),
    Drop(592, 5, 10, DuelRank.SAPOW),
    Drop(601, 5, 14, DuelRank.SAPOW),
    Drop(602, 5, 14, DuelRank.SAPOW),
    Drop(605, 5, 14, DuelRank.SAPOW),
    Drop(606, 5, 14, DuelRank.SAPOW),
    Drop(608, 5, 14, DuelRank.SAPOW),
    Drop(610, 5, 2, DuelRank.SAPOW),
    Drop(620, 5, 14, DuelRank.SAPOW),
    Drop(629, 5, 14, DuelRank.SAPOW),
    Drop(642, 5, 14, DuelRank.SAPOW),
    Drop(643, 5, 14, DuelRank.SAPOW),
    Drop(4, 6, 14, DuelRank.SAPOW),
    Drop(9, 6, 14, DuelRank.SAPOW),
    Drop(20, 6, 6, DuelRank.SAPOW),
    Drop(23, 6, 14, DuelRank.SAPOW),
    Drop(24, 6, 14, DuelRank.SAPOW),
    Drop(30, 6, 28, DuelRank.SAPOW),
    Drop(34, 6, 14, DuelRank.SAPOW),
    Drop(40, 6, 14, DuelRank.SAPOW),
    Drop(45, 6, 14, DuelRank.SAPOW),
    Drop(47, 6, 14, DuelRank.SAPOW),
    Drop(53, 6, 14, DuelRank.SAPOW),
    Drop(59, 6, 14, DuelRank.SAPOW),
    Drop(61, 6, 14, DuelRank.SAPOW),
    Drop(65, 6, 14, DuelRank.SAPOW),
    Drop(96, 6, 14, DuelRank.SAPOW),
    Drop(97, 6, 14, DuelRank.SAPOW),
    Drop(98, 6, 14, DuelRank.SAPOW),
    Drop(103, 6, 14, DuelRank.SAPOW),
    Drop(108, 6, 14, DuelRank.SAPOW),
    Drop(109, 6, 14, DuelRank.SAPOW),
    Drop(110, 6, 14, DuelRank.SAPOW),
    Drop(118, 6, 14, DuelRank.SAPOW),
    Drop(119, 6, 14, DuelRank.SAPOW),
    Drop(120, 6, 14, DuelRank.SAPOW),
    Drop(132, 6, 28, DuelRank.SAPOW),
    Drop(133, 6, 14, DuelRank.SAPOW),
    Drop(135, 6, 14, DuelRank.SAPOW),
    Drop(138, 6, 14, DuelRank.SAPOW),
    Drop(139, 6, 14, DuelRank.SAPOW),
    Drop(140, 6, 26, DuelRank.SAPOW),
    Drop(141, 6, 14, DuelRank.SAPOW),
    Drop(142, 6, 14, DuelRank.SAPOW),
    Drop(152, 6, 14, DuelRank.SAPOW),
    Drop(154, 6, 14, DuelRank.SAPOW),
    Drop(155, 6, 14, DuelRank.SAPOW),
    Drop(162, 6, 14, DuelRank.SAPOW),
    Drop(167, 6, 14, DuelRank.SAPOW),
    Drop(169, 6, 14, DuelRank.SAPOW),
    Drop(171, 6, 14, DuelRank.SAPOW),
    Drop(172, 6, 13, DuelRank.SAPOW),
    Drop(177, 6, 28, DuelRank.SAPOW),
    Drop(178, 6, 13, DuelRank.SAPOW),
    Drop(180, 6, 13, DuelRank.SAPOW),
    Drop(181, 6, 13, DuelRank.SAPOW),
    Drop(190, 6, 13, DuelRank.SAPOW),
    Drop(191, 6, 13, DuelRank.SAPOW),
    Drop(196, 6, 28, DuelRank.SAPOW),
    Drop(197, 6, 13, DuelRank.SAPOW),
    Drop(199, 6, 13, DuelRank.SAPOW),
    Drop(201, 6, 13, DuelRank.SAPOW),
    Drop(203, 6, 13, DuelRank.SAPOW),
    Drop(205, 6, 13, DuelRank.SAPOW),
    Drop(206, 6, 13, DuelRank.SAPOW),
    Drop(211, 6, 13, DuelRank.SAPOW),
    Drop(215, 6, 28, DuelRank.SAPOW),
    Drop(219, 6, 13, DuelRank.SAPOW),
    Drop(221, 6, 13, DuelRank.SAPOW),
    Drop(225, 6, 13, DuelRank.SAPOW),
    Drop(227, 6, 13, DuelRank.SAPOW),
    Drop(228, 6, 28, DuelRank.SAPOW),
    Drop(231, 6, 13, DuelRank.SAPOW),
    Drop(236, 6, 13, DuelRank.SAPOW),
    Drop(237, 6, 13, DuelRank.SAPOW),
    Drop(243, 6, 13, DuelRank.SAPOW),
    Drop(244, 6, 28, DuelRank.SAPOW),
    Drop(246, 6, 13, DuelRank.SAPOW),
    Drop(248, 6, 13, DuelRank.SAPOW),
    Drop(250, 6, 13, DuelRank.SAPOW),
    Drop(251, 6, 13, DuelRank.SAPOW),
    Drop(253, 6, 13, DuelRank.SAPOW),
    Drop(256, 6, 13, DuelRank.SAPOW),
    Drop(257, 6, 28, DuelRank.SAPOW),
    Drop(258, 6, 28, DuelRank.SAPOW),
    Drop(260, 6, 13, DuelRank.SAPOW),
    Drop(262, 6, 13, DuelRank.SAPOW),
    Drop(263, 6, 28, DuelRank.SAPOW),
    Drop(264, 6, 13, DuelRank.SAPOW),
    Drop(265, 6, 13, DuelRank.SAPOW),
    Drop(269, 6, 13, DuelRank.SAPOW),
    Drop(270, 6, 28, DuelRank.SAPOW),
    Drop(272, 6, 13, DuelRank.SAPOW),
    Drop(273, 6, 13, DuelRank.SAPOW),
    Drop(274, 6, 13, DuelRank.SAPOW),
    Drop(276, 6, 13, DuelRank.SAPOW),
    Drop(279, 6, 13, DuelRank.SAPOW),
    Drop(282, 6, 13, DuelRank.SAPOW),
    Drop(283, 6, 13, DuelRank.SAPOW),
    Drop(289, 6, 13, DuelRank.SAPOW),
    Drop(290, 6, 13, DuelRank.SAPOW),
    Drop(292, 6, 13, DuelRank.SAPOW),
    Drop(293, 6, 13, DuelRank.SAPOW),
    Drop(294, 6, 13, DuelRank.SAPOW),
    Drop(295, 6, 13, DuelRank.SAPOW),
    Drop(296, 6, 13, DuelRank.SAPOW),
    Drop(298, 6, 13, DuelRank.SAPOW),
    Drop(335, 6, 36, DuelRank.SAPOW),
    Drop(393, 6, 13, DuelRank.SAPOW),
    Drop(394, 6, 12, DuelRank.SAPOW),
    Drop(398, 6, 13, DuelRank.SAPOW),
    Drop(414, 6, 13, DuelRank.SAPOW),
    Drop(432, 6, 28, DuelRank.SAPOW),
    Drop(444, 6, 13, DuelRank.SAPOW),
    Drop(446, 6, 28, DuelRank.SAPOW),
    Drop(451, 6, 13, DuelRank.SAPOW),
    Drop(452, 6, 13, DuelRank.SAPOW),
    Drop(461, 6, 26, DuelRank.SAPOW),
    Drop(463, 6, 13, DuelRank.SAPOW),
    Drop(478, 6, 13, DuelRank.SAPOW),
    Drop(484, 6, 13, DuelRank.SAPOW),
    Drop(496, 6, 13, DuelRank.SAPOW),
    Drop(503, 6, 13, DuelRank.SAPOW),
    Drop(516, 6, 13, DuelRank.SAPOW),
    Drop(524, 6, 13, DuelRank.SAPOW),
    Drop(548, 6, 13, DuelRank.SAPOW),
    Drop(549, 6, 13, DuelRank.SAPOW),
    Drop(552, 6, 13, DuelRank.SAPOW),
    Drop(556, 6, 13, DuelRank.SAPOW),
    Drop(558, 6, 13, DuelRank.SAPOW),
    Drop(567, 6, 14, DuelRank.SAPOW),
    Drop(568, 6, 13, DuelRank.SAPOW),
    Drop(573, 6, 13, DuelRank.SAPOW),
    Drop(576, 6, 14, DuelRank.SAPOW),
    Drop(584, 6, 14, DuelRank.SAPOW),
    Drop(585, 6, 13, DuelRank.SAPOW),
    Drop(588, 6, 12, DuelRank.SAPOW),
    Drop(591, 6, 13, DuelRank.SAPOW),
    Drop(592, 6, 12, DuelRank.SAPOW),
    Drop(601, 6, 13, DuelRank.SAPOW),
    Drop(602, 6, 13, DuelRank.SAPOW),
    Drop(605, 6, 13, DuelRank.SAPOW),
    Drop(606, 6, 13, DuelRank.SAPOW),
    Drop(612, 6, 13, DuelRank.SAPOW),
    Drop(620, 6, 14, DuelRank.SAPOW),
    Drop(629, 6, 14, DuelRank.SAPOW),
    Drop(642, 6, 14, DuelRank.SAPOW),
    Drop(652, 6, 18, DuelRank.SAPOW),
    Drop(8, 7, 44, DuelRank.SAPOW),
    Drop(19, 7, 5, DuelRank.SAPOW),
    Drop(40, 7, 21, DuelRank.SAPOW),
    Drop(50, 7, 20, DuelRank.SAPOW),
    Drop(75, 7, 116, DuelRank.SAPOW),
    Drop(76, 7, 44, DuelRank.SAPOW),
    Drop(98, 7, 20, DuelRank.SAPOW),
    Drop(101, 7, 44, DuelRank.SAPOW),
    Drop(104, 7, 20, DuelRank.SAPOW),
    Drop(107, 7, 20, DuelRank.SAPOW),
    Drop(122, 7, 20, DuelRank.SAPOW),
    Drop(123, 7, 20, DuelRank.SAPOW),
    Drop(130, 7, 20, DuelRank.SAPOW),
    Drop(133, 7, 20, DuelRank.SAPOW),
    Drop(134, 7, 20, DuelRank.SAPOW),
    Drop(135, 7, 20, DuelRank.SAPOW),
    Drop(137, 7, 20, DuelRank.SAPOW),
    Drop(145, 7, 20, DuelRank.SAPOW),
    Drop(152, 7, 20, DuelRank.SAPOW),
    Drop(154, 7, 20, DuelRank.SAPOW),
    Drop(157, 7, 44, DuelRank.SAPOW),
    Drop(158, 7, 44, DuelRank.SAPOW),
    Drop(159, 7, 20, DuelRank.SAPOW),
    Drop(174, 7, 20, DuelRank.SAPOW),
    Drop(176, 7, 44, DuelRank.SAPOW),
    Drop(179, 7, 20, DuelRank.SAPOW),
    Drop(180, 7, 20, DuelRank.SAPOW),
    Drop(182, 7, 20, DuelRank.SAPOW),
    Drop(183, 7, 20, DuelRank.SAPOW),
    Drop(185, 7, 20, DuelRank.SAPOW),
    Drop(191, 7, 20, DuelRank.SAPOW),
    Drop(195, 7, 20, DuelRank.SAPOW),
    Drop(198, 7, 20, DuelRank.SAPOW),
    Drop(200, 7, 20, DuelRank.SAPOW),
    Drop(202, 7, 20, DuelRank.SAPOW),
    Drop(203, 7, 20, DuelRank.SAPOW),
    Drop(207, 7, 21, DuelRank.SAPOW),
    Drop(209, 7, 21, DuelRank.SAPOW),
    Drop(210, 7, 38, DuelRank.SAPOW),
    Drop(211, 7, 21, DuelRank.SAPOW),
    Drop(212, 7, 21, DuelRank.SAPOW),
    Drop(214, 7, 21, DuelRank.SAPOW),
    Drop(222, 7, 21, DuelRank.SAPOW),
    Drop(227, 7, 21, DuelRank.SAPOW),
    Drop(229, 7, 21, DuelRank.SAPOW),
    Drop(232, 7, 21, DuelRank.SAPOW),
    Drop(237, 7, 21, DuelRank.SAPOW),
    Drop(242, 7, 21, DuelRank.SAPOW),
    Drop(245, 7, 21, DuelRank.SAPOW),
    Drop(254, 7, 21, DuelRank.SAPOW),
    Drop(261, 7, 21, DuelRank.SAPOW),
    Drop(268, 7, 21, DuelRank.SAPOW),
    Drop(273, 7, 21, DuelRank.SAPOW),
    Drop(274, 7, 21, DuelRank.SAPOW),
    Drop(285, 7, 21, DuelRank.SAPOW),
    Drop(291, 7, 21, DuelRank.SAPOW),
    Drop(292, 7, 21, DuelRank.SAPOW),
    Drop(330, 7, 34, DuelRank.SAPOW),
    Drop(343, 7, 16, DuelRank.SAPOW),
    Drop(420, 7, 21, DuelRank.SAPOW),
    Drop(452, 7, 21, DuelRank.SAPOW),
    Drop(476, 7, 21, DuelRank.SAPOW),
    Drop(486, 7, 21, DuelRank.SAPOW),
    Drop(488, 7, 21, DuelRank.SAPOW),
    Drop(492, 7, 21, DuelRank.SAPOW),
    Drop(501, 7, 21, DuelRank.SAPOW),
    Drop(506, 7, 21, DuelRank.SAPOW),
    Drop(510, 7, 44, DuelRank.SAPOW),
    Drop(524, 7, 21, DuelRank.SAPOW),
    Drop(536, 7, 21, DuelRank.SAPOW),
    Drop(540, 7, 21, DuelRank.SAPOW),
    Drop(544, 7, 21, DuelRank.SAPOW),
    Drop(547, 7, 21, DuelRank.SAPOW),
    Drop(549, 7, 21, DuelRank.SAPOW),
    Drop(556, 7, 21, DuelRank.SAPOW),
    Drop(567, 7, 21, DuelRank.SAPOW),
    Drop(579, 7, 44, DuelRank.SAPOW),
    Drop(589, 7, 44, DuelRank.SAPOW),
    Drop(591, 7, 21, DuelRank.SAPOW),
    Drop(598, 7, 21, DuelRank.SAPOW),
    Drop(602, 7, 21, DuelRank.SAPOW),
    Drop(609, 7, 21, DuelRank.SAPOW),
    Drop(611, 7, 21, DuelRank.SAPOW),
    Drop(620, 7, 21, DuelRank.SAPOW),
    Drop(2, 8, 22, DuelRank.SAPOW),
    Drop(19, 8, 6, DuelRank.SAPOW),
    Drop(20, 8, 4, DuelRank.SAPOW),
    Drop(21, 8, 4, DuelRank.SAPOW),
    Drop(30, 8, 22, DuelRank.SAPOW),
    Drop(34, 8, 22, DuelRank.SAPOW),
    Drop(35, 8, 4, DuelRank.SAPOW),
    Drop(36, 8, 22, DuelRank.SAPOW),
    Drop(38, 8, 28, DuelRank.SAPOW),
    Drop(42, 8, 22, DuelRank.SAPOW),
    Drop(44, 8, 22, DuelRank.SAPOW),
    Drop(74, 8, 22, DuelRank.SAPOW),
    Drop(79, 8, 24, DuelRank.SAPOW),
    Drop(90, 8, 24, DuelRank.SAPOW),
    Drop(96, 8, 22, DuelRank.SAPOW),
    Drop(97, 8, 22, DuelRank.SAPOW),
    Drop(98, 8, 22, DuelRank.SAPOW),
    Drop(99, 8, 48, DuelRank.SAPOW),
    Drop(104, 8, 22, DuelRank.SAPOW),
    Drop(106, 8, 22, DuelRank.SAPOW),
    Drop(108, 8, 22, DuelRank.SAPOW),
    Drop(114, 8, 22, DuelRank.SAPOW),
    Drop(115, 8, 22, DuelRank.SAPOW),
    Drop(129, 8, 22, DuelRank.SAPOW),
    Drop(132, 8, 22, DuelRank.SAPOW),
    Drop(135, 8, 22, DuelRank.SAPOW),
    Drop(139, 8, 22, DuelRank.SAPOW),
    Drop(142, 8, 22, DuelRank.SAPOW),
    Drop(143, 8, 22, DuelRank.SAPOW),
    Drop(144, 8, 22, DuelRank.SAPOW),
    Drop(145, 8, 22, DuelRank.SAPOW),
    Drop(154, 8, 22, DuelRank.SAPOW),
    Drop(174, 8, 20, DuelRank.SAPOW),
    Drop(179, 8, 20, DuelRank.SAPOW),
    Drop(183, 8, 20, DuelRank.SAPOW),
    Drop(184, 8, 20, DuelRank.SAPOW),
    Drop(190, 8, 20, DuelRank.SAPOW),
    Drop(203, 8, 20, DuelRank.SAPOW),
    Drop(213, 8, 20, DuelRank.SAPOW),
    Drop(215, 8, 20, DuelRank.SAPOW),
    Drop(220, 8, 20, DuelRank.SAPOW),
    Drop(223, 8, 24, DuelRank.SAPOW),
    Drop(228, 8, 22, DuelRank.SAPOW),
    Drop(237, 8, 22, DuelRank.SAPOW),
    Drop(244, 8, 22, DuelRank.SAPOW),
    Drop(253, 8, 22, DuelRank.SAPOW),
    Drop(257, 8, 22, DuelRank.SAPOW),
    Drop(259, 8, 22, DuelRank.SAPOW),
    Drop(263, 8, 22, DuelRank.SAPOW),
    Drop(268, 8, 22, DuelRank.SAPOW),
    Drop(366, 8, 22, DuelRank.SAPOW),
    Drop(368, 8, 22, DuelRank.SAPOW),
    Drop(370, 8, 4, DuelRank.SAPOW),
    Drop(371, 8, 4, DuelRank.SAPOW),
    Drop(372, 8, 4, DuelRank.SAPOW),
    Drop(373, 8, 4, DuelRank.SAPOW),
    Drop(385, 8, 4, DuelRank.SAPOW),
    Drop(386, 8, 4, DuelRank.SAPOW),
    Drop(390, 8, 4, DuelRank.SAPOW),
    Drop(391, 8, 20, DuelRank.SAPOW),
    Drop(401, 8, 20, DuelRank.SAPOW),
    Drop(407, 8, 20, DuelRank.SAPOW),
    Drop(426, 8, 48, DuelRank.SAPOW),
    Drop(427, 8, 4, DuelRank.SAPOW),
    Drop(442, 8, 4, DuelRank.SAPOW),
    Drop(453, 8, 46, DuelRank.SAPOW),
    Drop(455, 8, 48, DuelRank.SAPOW),
    Drop(457, 8, 48, DuelRank.SAPOW),
    Drop(465, 8, 26, DuelRank.SAPOW),
    Drop(467, 8, 20, DuelRank.SAPOW),
    Drop(470, 8, 24, DuelRank.SAPOW),
    Drop(471, 8, 20, DuelRank.SAPOW),
    Drop(500, 8, 20, DuelRank.SAPOW),
    Drop(505, 8, 46, DuelRank.SAPOW),
    Drop(509, 8, 4, DuelRank.SAPOW),
    Drop(518, 8, 12, DuelRank.SAPOW),
    Drop(522, 8, 26, DuelRank.SAPOW),
    Drop(525, 8, 12, DuelRank.SAPOW),
    Drop(529, 8, 12, DuelRank.SAPOW),
    Drop(531, 8, 46, DuelRank.SAPOW),
    Drop(532, 8, 24, DuelRank.SAPOW),
    Drop(564, 8, 48, DuelRank.SAPOW),
    Drop(571, 8, 26, DuelRank.SAPOW),
    Drop(572, 8, 26, DuelRank.SAPOW),
    Drop(593, 8, 12, DuelRank.SAPOW),
    Drop(594, 8, 26, DuelRank.SAPOW),
    Drop(599, 8, 46, DuelRank.SAPOW),
    Drop(613, 8, 20, DuelRank.SAPOW),
    Drop(623, 8, 24, DuelRank.SAPOW),
    Drop(627, 8, 26, DuelRank.SAPOW),
    Drop(631, 8, 24, DuelRank.SAPOW),
    Drop(632, 8, 12, DuelRank.SAPOW),
    Drop(634, 8, 24, DuelRank.SAPOW),
    Drop(645, 8, 4, DuelRank.SAPOW),
    Drop(646, 8, 46, DuelRank.SAPOW),
    Drop(707, 8, 4, DuelRank.SAPOW),
    Drop(713, 8, 4, DuelRank.SAPOW),
    Drop(714, 8, 4, DuelRank.SAPOW),
    Drop(5, 9, 18, DuelRank.SAPOW),
    Drop(8, 9, 18, DuelRank.SAPOW),
    Drop(19, 9, 6, DuelRank.SAPOW),
    Drop(29, 9, 18, DuelRank.SAPOW),
    Drop(32, 9, 70, DuelRank.SAPOW),
    Drop(45, 9, 18, DuelRank.SAPOW),
    Drop(47, 9, 18, DuelRank.SAPOW),
    Drop(48, 9, 18, DuelRank.SAPOW),
    Drop(59, 9, 36, DuelRank.SAPOW),
    Drop(80, 9, 72, DuelRank.SAPOW),
    Drop(81, 9, 72, DuelRank.SAPOW),
    Drop(96, 9, 16, DuelRank.SAPOW),
    Drop(97, 9, 16, DuelRank.SAPOW),
    Drop(100, 9, 17, DuelRank.SAPOW),
    Drop(105, 9, 36, DuelRank.SAPOW),
    Drop(107, 9, 16, DuelRank.SAPOW),
    Drop(108, 9, 16, DuelRank.SAPOW),
    Drop(113, 9, 17, DuelRank.SAPOW),
    Drop(114, 9, 17, DuelRank.SAPOW),
    Drop(116, 9, 17, DuelRank.SAPOW),
    Drop(129, 9, 17, DuelRank.SAPOW),
    Drop(134, 9, 17, DuelRank.SAPOW),
    Drop(139, 9, 17, DuelRank.SAPOW),
    Drop(143, 9, 17, DuelRank.SAPOW),
    Drop(144, 9, 17, DuelRank.SAPOW),
    Drop(145, 9, 17, DuelRank.SAPOW),
    Drop(147, 9, 17, DuelRank.SAPOW),
    Drop(148, 9, 17, DuelRank.SAPOW),
    Drop(155, 9, 17, DuelRank.SAPOW),
    Drop(156, 9, 17, DuelRank.SAPOW),
    Drop(158, 9, 17, DuelRank.SAPOW),
    Drop(160, 9, 17, DuelRank.SAPOW),
    Drop(161, 9, 17, DuelRank.SAPOW),
    Drop(173, 9, 17, DuelRank.SAPOW),
    Drop(175, 9, 17, DuelRank.SAPOW),
    Drop(176, 9, 17, DuelRank.SAPOW),
    Drop(177, 9, 17, DuelRank.SAPOW),
    Drop(178, 9, 17, DuelRank.SAPOW),
    Drop(180, 9, 17, DuelRank.SAPOW),
    Drop(183, 9, 17, DuelRank.SAPOW),
    Drop(184, 9, 17, DuelRank.SAPOW),
    Drop(187, 9, 17, DuelRank.SAPOW),
    Drop(188, 9, 17, DuelRank.SAPOW),
    Drop(189, 9, 17, DuelRank.SAPOW),
    Drop(190, 9, 17, DuelRank.SAPOW),
    Drop(195, 9, 17, DuelRank.SAPOW),
    Drop(199, 9, 17, DuelRank.SAPOW),
    Drop(203, 9, 17, DuelRank.SAPOW),
    Drop(205, 9, 17, DuelRank.SAPOW),
    Drop(206, 9, 17, DuelRank.SAPOW),
    Drop(208, 9, 17, DuelRank.SAPOW),
    Drop(209, 9, 17, DuelRank.SAPOW),
    Drop(215, 9, 17, DuelRank.SAPOW),
    Drop(218, 9, 52, DuelRank.SAPOW),
    Drop(222, 9, 17, DuelRank.SAPOW),
    Drop(224, 9, 17, DuelRank.SAPOW),
    Drop(226, 9, 17, DuelRank.SAPOW),
    Drop(232, 9, 17, DuelRank.SAPOW),
    Drop(239, 9, 17, DuelRank.SAPOW),
    Drop(240, 9, 17, DuelRank.SAPOW),
    Drop(243, 9, 17, DuelRank.SAPOW),
    Drop(247, 9, 17, DuelRank.SAPOW),
    Drop(253, 9, 17, DuelRank.SAPOW),
    Drop(264, 9, 17, DuelRank.SAPOW),
    Drop(265, 9, 17, DuelRank.SAPOW),
    Drop(266, 9, 17, DuelRank.SAPOW),
    Drop(267, 9, 17, DuelRank.SAPOW),
    Drop(271, 9, 17, DuelRank.SAPOW),
    Drop(277, 9, 17, DuelRank.SAPOW),
    Drop(282, 9, 17, DuelRank.SAPOW),
    Drop(283, 9, 17, DuelRank.SAPOW),
    Drop(292, 9, 17, DuelRank.SAPOW),
    Drop(298, 9, 17, DuelRank.SAPOW),
    Drop(300, 9, 17, DuelRank.SAPOW),
    Drop(331, 9, 34, DuelRank.SAPOW),
    Drop(332, 9, 34, DuelRank.SAPOW),
    Drop(417, 9, 17, DuelRank.SAPOW),
    Drop(421, 9, 17, DuelRank.SAPOW),
    Drop(432, 9, 17, DuelRank.SAPOW),
    Drop(451, 9, 17, DuelRank.SAPOW),
    Drop(463, 9, 17, DuelRank.SAPOW),
    Drop(476, 9, 17, DuelRank.SAPOW),
    Drop(477, 9, 17, DuelRank.SAPOW),
    Drop(478, 9, 17, DuelRank.SAPOW),
    Drop(479, 9, 17, DuelRank.SAPOW),
    Drop(481, 9, 17, DuelRank.SAPOW),
    Drop(490, 9, 17, DuelRank.SAPOW),
    Drop(503, 9, 17, DuelRank.SAPOW),
    Drop(510, 9, 17, DuelRank.SAPOW),
    Drop(534, 9, 17, DuelRank.SAPOW),
    Drop(536, 9, 17, DuelRank.SAPOW),
    Drop(538, 9, 17, DuelRank.SAPOW),
    Drop(544, 9, 17, DuelRank.SAPOW),
    Drop(556, 9, 17, DuelRank.SAPOW),
    Drop(568, 9, 35, DuelRank.SAPOW),
    Drop(569, 9, 17, DuelRank.SAPOW),
    Drop(573, 9, 52, DuelRank.SAPOW),
    Drop(585, 9, 17, DuelRank.SAPOW),
    Drop(590, 9, 17, DuelRank.SAPOW),
    Drop(604, 9, 17, DuelRank.SAPOW),
    Drop(605, 9, 17, DuelRank.SAPOW),
    Drop(606, 9, 17, DuelRank.SAPOW),
    Drop(3, 10, 20, DuelRank.SAPOW),
    Drop(8, 10, 21, DuelRank.SAPOW),
    Drop(10, 10, 20, DuelRank.SAPOW),
    Drop(13, 10, 26, DuelRank.SAPOW),
    Drop(19, 10, 6, DuelRank.SAPOW),
    Drop(25, 10, 20, DuelRank.SAPOW),
    Drop(41, 10, 26, DuelRank.SAPOW),
    Drop(44, 10, 26, DuelRank.SAPOW),
    Drop(49, 10, 72, DuelRank.SAPOW),
    Drop(50, 10, 68, DuelRank.SAPOW),
    Drop(53, 10, 95, DuelRank.SAPOW),
    Drop(54, 10, 98, DuelRank.SAPOW),
    Drop(55, 10, 72, DuelRank.SAPOW),
    Drop(70, 10, 26, DuelRank.SAPOW),
    Drop(77, 10, 22, DuelRank.SAPOW),
    Drop(80, 10, 22, DuelRank.SAPOW),
    Drop(109, 10, 22, DuelRank.SAPOW),
    Drop(115, 10, 22, DuelRank.SAPOW),
    Drop(116, 10, 21, DuelRank.SAPOW),
    Drop(119, 10, 22, DuelRank.SAPOW),
    Drop(121, 10, 22, DuelRank.SAPOW),
    Drop(123, 10, 22, DuelRank.SAPOW),
    Drop(125, 10, 26, DuelRank.SAPOW),
    Drop(141, 10, 20, DuelRank.SAPOW),
    Drop(149, 10, 26, DuelRank.SAPOW),
    Drop(157, 10, 20, DuelRank.SAPOW),
    Drop(158, 10, 20, DuelRank.SAPOW),
    Drop(164, 10, 22, DuelRank.SAPOW),
    Drop(166, 10, 22, DuelRank.SAPOW),
    Drop(171, 10, 22, DuelRank.SAPOW),
    Drop(180, 10, 20, DuelRank.SAPOW),
    Drop(186, 10, 22, DuelRank.SAPOW),
    Drop(193, 10, 22, DuelRank.SAPOW),
    Drop(194, 10, 22, DuelRank.SAPOW),
    Drop(196, 10, 22, DuelRank.SAPOW),
    Drop(201, 10, 22, DuelRank.SAPOW),
    Drop(209, 10, 20, DuelRank.SAPOW),
    Drop(220, 10, 22, DuelRank.SAPOW),
    Drop(221, 10, 22, DuelRank.SAPOW),
    Drop(225, 10, 22, DuelRank.SAPOW),
    Drop(233, 10, 22, DuelRank.SAPOW),
    Drop(236, 10, 22, DuelRank.SAPOW),
    Drop(248, 10, 22, DuelRank.SAPOW),
    Drop(255, 10, 22, DuelRank.SAPOW),
    Drop(256, 10, 22, DuelRank.SAPOW),
    Drop(259, 10, 22, DuelRank.SAPOW),
    Drop(272, 10, 22, DuelRank.SAPOW),
    Drop(273, 10, 22, DuelRank.SAPOW),
    Drop(274, 10, 22, DuelRank.SAPOW),
    Drop(280, 10, 22, DuelRank.SAPOW),
    Drop(291, 10, 22, DuelRank.SAPOW),
    Drop(293, 10, 22, DuelRank.SAPOW),
    Drop(330, 10, 24, DuelRank.SAPOW),
    Drop(367, 10, 5, DuelRank.SAPOW),
    Drop(397, 10, 20, DuelRank.SAPOW),
    Drop(406, 10, 22, DuelRank.SAPOW),
    Drop(412, 10, 24, DuelRank.SAPOW),
    Drop(459, 10, 22, DuelRank.SAPOW),
    Drop(476, 10, 20, DuelRank.SAPOW),
    Drop(477, 10, 20, DuelRank.SAPOW),
    Drop(478, 10, 20, DuelRank.SAPOW),
    Drop(479, 10, 20, DuelRank.SAPOW),
    Drop(485, 10, 20, DuelRank.SAPOW),
    Drop(488, 10, 20, DuelRank.SAPOW),
    Drop(496, 10, 22, DuelRank.SAPOW),
    Drop(501, 10, 20, DuelRank.SAPOW),
    Drop(506, 10, 20, DuelRank.SAPOW),
    Drop(510, 10, 20, DuelRank.SAPOW),
    Drop(511, 10, 24, DuelRank.SAPOW),
    Drop(514, 10, 22, DuelRank.SAPOW),
    Drop(534, 10, 20, DuelRank.SAPOW),
    Drop(547, 10, 20, DuelRank.SAPOW),
    Drop(552, 10, 22, DuelRank.SAPOW),
    Drop(567, 10, 22, DuelRank.SAPOW),
    Drop(576, 10, 22, DuelRank.SAPOW),
    Drop(579, 10, 20, DuelRank.SAPOW),
    Drop(589, 10, 20, DuelRank.SAPOW),
    Drop(608, 10, 22, DuelRank.SAPOW),
    Drop(609, 10, 20, DuelRank.SAPOW),
    Drop(620, 10, 44, DuelRank.SAPOW),
    Drop(643, 10, 22, DuelRank.SAPOW),
    Drop(2, 11, 32, DuelRank.SAPOW),
    Drop(6, 11, 32, DuelRank.SAPOW),
    Drop(13, 11, 94, DuelRank.SAPOW),
    Drop(20, 11, 8, DuelRank.SAPOW),
    Drop(26, 11, 32, DuelRank.SAPOW),
    Drop(27, 11, 32, DuelRank.SAPOW),
    Drop(31, 11, 32, DuelRank.SAPOW),
    Drop(32, 11, 32, DuelRank.SAPOW),
    Drop(36, 11, 32, DuelRank.SAPOW),
    Drop(41, 11, 32, DuelRank.SAPOW),
    Drop(43, 11, 32, DuelRank.SAPOW),
    Drop(44, 11, 32, DuelRank.SAPOW),
    Drop(46, 11, 32, DuelRank.SAPOW),
    Drop(49, 11, 32, DuelRank.SAPOW),
    Drop(54, 11, 32, DuelRank.SAPOW),
    Drop(55, 11, 32, DuelRank.SAPOW),
    Drop(70, 11, 32, DuelRank.SAPOW),
    Drop(71, 11, 32, DuelRank.SAPOW),
    Drop(78, 11, 32, DuelRank.SAPOW),
    Drop(81, 11, 30, DuelRank.SAPOW),
    Drop(83, 11, 30, DuelRank.SAPOW),
    Drop(86, 11, 30, DuelRank.SAPOW),
    Drop(91, 11, 30, DuelRank.SAPOW),
    Drop(95, 11, 30, DuelRank.SAPOW),
    Drop(125, 11, 94, DuelRank.SAPOW),
    Drop(127, 11, 30, DuelRank.SAPOW),
    Drop(136, 11, 30, DuelRank.SAPOW),
    Drop(149, 11, 30, DuelRank.SAPOW),
    Drop(168, 11, 30, DuelRank.SAPOW),
    Drop(170, 11, 30, DuelRank.SAPOW),
    Drop(186, 11, 62, DuelRank.SAPOW),
    Drop(207, 11, 62, DuelRank.SAPOW),
    Drop(230, 11, 30, DuelRank.SAPOW),
    Drop(272, 11, 62, DuelRank.SAPOW),
    Drop(300, 11, 62, DuelRank.SAPOW),
    Drop(307, 11, 18, DuelRank.SAPOW),
    Drop(332, 11, 40, DuelRank.SAPOW),
    Drop(368, 11, 30, DuelRank.SAPOW),
    Drop(379, 11, 30, DuelRank.SAPOW),
    Drop(405, 11, 30, DuelRank.SAPOW),
    Drop(412, 11, 30, DuelRank.SAPOW),
    Drop(415, 11, 30, DuelRank.SAPOW),
    Drop(416, 11, 30, DuelRank.SAPOW),
    Drop(460, 11, 30, DuelRank.SAPOW),
    Drop(462, 11, 30, DuelRank.SAPOW),
    Drop(466, 11, 94, DuelRank.SAPOW),
    Drop(511, 11, 30, DuelRank.SAPOW),
    Drop(512, 11, 30, DuelRank.SAPOW),
    Drop(538, 11, 62, DuelRank.SAPOW),
    Drop(552, 11, 62, DuelRank.SAPOW),
    Drop(597, 11, 30, DuelRank.SAPOW),
    Drop(636, 11, 94, DuelRank.SAPOW),
    Drop(648, 11, 30, DuelRank.SAPOW),
    Drop(2, 12, 30, DuelRank.SAPOW),
    Drop(9, 12, 28, DuelRank.SAPOW),
    Drop(19, 12, 8, DuelRank.SAPOW),
    Drop(24, 12, 28, DuelRank.SAPOW),
    Drop(30, 12, 28, DuelRank.SAPOW),
    Drop(32, 12, 30, DuelRank.SAPOW),
    Drop(36, 12, 92, DuelRank.SAPOW),
    Drop(78, 12, 30, DuelRank.SAPOW),
    Drop(81, 12, 30, DuelRank.SAPOW),
    Drop(83, 12, 30, DuelRank.SAPOW),
    Drop(86, 12, 30, DuelRank.SAPOW),
    Drop(91, 12, 30, DuelRank.SAPOW),
    Drop(95, 12, 54, DuelRank.SAPOW),
    Drop(96, 12, 92, DuelRank.SAPOW),
    Drop(97, 12, 92, DuelRank.SAPOW),
    Drop(98, 12, 92, DuelRank.SAPOW),
    Drop(108, 12, 28, DuelRank.SAPOW),
    Drop(127, 12, 30, DuelRank.SAPOW),
    Drop(132, 12, 28, DuelRank.SAPOW),
    Drop(135, 12, 28, DuelRank.SAPOW),
    Drop(139, 12, 28, DuelRank.SAPOW),
    Drop(154, 12, 28, DuelRank.SAPOW),
    Drop(170, 12, 30, DuelRank.SAPOW),
    Drop(197, 12, 28, DuelRank.SAPOW),
    Drop(203, 12, 28, DuelRank.SAPOW),
    Drop(215, 12, 28, DuelRank.SAPOW),
    Drop(228, 12, 28, DuelRank.SAPOW),
    Drop(275, 12, 8, DuelRank.SAPOW),
    Drop(283, 12, 28, DuelRank.SAPOW),
    Drop(297, 12, 8, DuelRank.SAPOW),
    Drop(331, 12, 48, DuelRank.SAPOW),
    Drop(335, 12, 48, DuelRank.SAPOW),
    Drop(368, 12, 54, DuelRank.SAPOW),
    Drop(379, 12, 30, DuelRank.SAPOW),
    Drop(405, 12, 54, DuelRank.SAPOW),
    Drop(406, 12, 28, DuelRank.SAPOW),
    Drop(410, 12, 28, DuelRank.SAPOW),
    Drop(411, 12, 28, DuelRank.SAPOW),
    Drop(412, 12, 28, DuelRank.SAPOW),
    Drop(414, 12, 28, DuelRank.SAPOW),
    Drop(415, 12, 28, DuelRank.SAPOW),
    Drop(416, 12, 28, DuelRank.SAPOW),
    Drop(417, 12, 28, DuelRank.SAPOW),
    Drop(418, 12, 8, DuelRank.SAPOW),
    Drop(420, 12, 28, DuelRank.SAPOW),
    Drop(421, 12, 28, DuelRank.SAPOW),
    Drop(422, 12, 28, DuelRank.SAPOW),
    Drop(460, 12, 30, DuelRank.SAPOW),
    Drop(462, 12, 30, DuelRank.SAPOW),
    Drop(498, 12, 8, DuelRank.SAPOW),
    Drop(512, 12, 30, DuelRank.SAPOW),
    Drop(514, 12, 30, DuelRank.SAPOW),
    Drop(544, 12, 30, DuelRank.SAPOW),
    Drop(548, 12, 30, DuelRank.SAPOW),
    Drop(556, 12, 30, DuelRank.SAPOW),
    Drop(585, 12, 30, DuelRank.SAPOW),
    Drop(614, 12, 8, DuelRank.SAPOW),
    Drop(636, 12, 30, DuelRank.SAPOW),
    Drop(643, 12, 30, DuelRank.SAPOW),
    Drop(648, 12, 54, DuelRank.SAPOW),
    Drop(658, 12, 48, DuelRank.SAPOW),
    Drop(5, 13, 10, DuelRank.SAPOW),
    Drop(8, 13, 12, DuelRank.SAPOW),
    Drop(20, 13, 8, DuelRank.SAPOW),
    Drop(29, 13, 10, DuelRank.SAPOW),
    Drop(45, 13, 10, DuelRank.SAPOW),
    Drop(48, 13, 10, DuelRank.SAPOW),
    Drop(50, 13, 10, DuelRank.SAPOW),
    Drop(70, 13, 54, DuelRank.SAPOW),
    Drop(71, 13, 54, DuelRank.SAPOW),
    Drop(77, 13, 24, DuelRank.SAPOW),
    Drop(96, 13, 10, DuelRank.SAPOW),
    Drop(97, 13, 10, DuelRank.SAPOW),
    Drop(98, 13, 10, DuelRank.SAPOW),
    Drop(101, 13, 10, DuelRank.SAPOW),
    Drop(107, 13, 12, DuelRank.SAPOW),
    Drop(108, 13, 12, DuelRank.SAPOW),
    Drop(122, 13, 12, DuelRank.SAPOW),
    Drop(129, 13, 12, DuelRank.SAPOW),
    Drop(134, 13, 12, DuelRank.SAPOW),
    Drop(135, 13, 12, DuelRank.SAPOW),
    Drop(139, 13, 12, DuelRank.SAPOW),
    Drop(140, 13, 32, DuelRank.SAPOW),
    Drop(144, 13, 12, DuelRank.SAPOW),
    Drop(145, 13, 12, DuelRank.SAPOW),
    Drop(147, 13, 12, DuelRank.SAPOW),
    Drop(148, 13, 12, DuelRank.SAPOW),
    Drop(152, 13, 42, DuelRank.SAPOW),
    Drop(154, 13, 12, DuelRank.SAPOW),
    Drop(156, 13, 12, DuelRank.SAPOW),
    Drop(157, 13, 12, DuelRank.SAPOW),
    Drop(158, 13, 12, DuelRank.SAPOW),
    Drop(159, 13, 12, DuelRank.SAPOW),
    Drop(160, 13, 12, DuelRank.SAPOW),
    Drop(161, 13, 12, DuelRank.SAPOW),
    Drop(173, 13, 12, DuelRank.SAPOW),
    Drop(176, 13, 12, DuelRank.SAPOW),
    Drop(177, 13, 24, DuelRank.SAPOW),
    Drop(179, 13, 12, DuelRank.SAPOW),
    Drop(182, 13, 12, DuelRank.SAPOW),
    Drop(183, 13, 12, DuelRank.SAPOW),
    Drop(184, 13, 12, DuelRank.SAPOW),
    Drop(187, 13, 12, DuelRank.SAPOW),
    Drop(189, 13, 12, DuelRank.SAPOW),
    Drop(191, 13, 12, DuelRank.SAPOW),
    Drop(193, 13, 24, DuelRank.SAPOW),
    Drop(195, 13, 12, DuelRank.SAPOW),
    Drop(196, 13, 24, DuelRank.SAPOW),
    Drop(198, 13, 12, DuelRank.SAPOW),
    Drop(199, 13, 20, DuelRank.SAPOW),
    Drop(200, 13, 12, DuelRank.SAPOW),
    Drop(203, 13, 12, DuelRank.SAPOW),
    Drop(205, 13, 20, DuelRank.SAPOW),
    Drop(206, 13, 32, DuelRank.SAPOW),
    Drop(208, 13, 12, DuelRank.SAPOW),
    Drop(209, 13, 12, DuelRank.SAPOW),
    Drop(211, 13, 12, DuelRank.SAPOW),
    Drop(212, 13, 12, DuelRank.SAPOW),
    Drop(214, 13, 12, DuelRank.SAPOW),
    Drop(218, 13, 12, DuelRank.SAPOW),
    Drop(222, 13, 12, DuelRank.SAPOW),
    Drop(224, 13, 12, DuelRank.SAPOW),
    Drop(227, 13, 32, DuelRank.SAPOW),
    Drop(229, 13, 12, DuelRank.SAPOW),
    Drop(230, 13, 24, DuelRank.SAPOW),
    Drop(232, 13, 12, DuelRank.SAPOW),
    Drop(239, 13, 12, DuelRank.SAPOW),
    Drop(240, 13, 12, DuelRank.SAPOW),
    Drop(242, 13, 12, DuelRank.SAPOW),
    Drop(243, 13, 32, DuelRank.SAPOW),
    Drop(245, 13, 12, DuelRank.SAPOW),
    Drop(247, 13, 20, DuelRank.SAPOW),
    Drop(251, 13, 24, DuelRank.SAPOW),
    Drop(254, 13, 12, DuelRank.SAPOW),
    Drop(258, 13, 24, DuelRank.SAPOW),
    Drop(261, 13, 12, DuelRank.SAPOW),
    Drop(265, 13, 32, DuelRank.SAPOW),
    Drop(266, 13, 12, DuelRank.SAPOW),
    Drop(267, 13, 12, DuelRank.SAPOW),
    Drop(268, 13, 12, DuelRank.SAPOW),
    Drop(270, 13, 20, DuelRank.SAPOW),
    Drop(271, 13, 12, DuelRank.SAPOW),
    Drop(277, 13, 12, DuelRank.SAPOW),
    Drop(285, 13, 32, DuelRank.SAPOW),
    Drop(289, 13, 20, DuelRank.SAPOW),
    Drop(292, 13, 32, DuelRank.SAPOW),
    Drop(300, 13, 12, DuelRank.SAPOW),
    Drop(334, 13, 28, DuelRank.SAPOW),
    Drop(393, 13, 20, DuelRank.SAPOW),
    Drop(398, 13, 20, DuelRank.SAPOW),
    Drop(420, 13, 12, DuelRank.SAPOW),
    Drop(421, 13, 12, DuelRank.SAPOW),
    Drop(432, 13, 20, DuelRank.SAPOW),
    Drop(436, 13, 20, DuelRank.SAPOW),
    Drop(444, 13, 20, DuelRank.SAPOW),
    Drop(446, 13, 20, DuelRank.SAPOW),
    Drop(451, 13, 32, DuelRank.SAPOW),
    Drop(452, 13, 32, DuelRank.SAPOW),
    Drop(475, 13, 20, DuelRank.SAPOW),
    Drop(476, 13, 12, DuelRank.SAPOW),
    Drop(484, 13, 20, DuelRank.SAPOW),
    Drop(490, 13, 12, DuelRank.SAPOW),
    Drop(492, 13, 12, DuelRank.SAPOW),
    Drop(503, 13, 32, DuelRank.SAPOW),
    Drop(506, 13, 12, DuelRank.SAPOW),
    Drop(510, 13, 12, DuelRank.SAPOW),
    Drop(524, 13, 20, DuelRank.SAPOW),
    Drop(534, 13, 12, DuelRank.SAPOW),
    Drop(536, 13, 32, DuelRank.SAPOW),
    Drop(540, 13, 12, DuelRank.SAPOW),
    Drop(544, 13, 12, DuelRank.SAPOW),
    Drop(549, 13, 32, DuelRank.SAPOW),
    Drop(556, 13, 12, DuelRank.SAPOW),
    Drop(569, 13, 20, DuelRank.SAPOW),
    Drop(590, 13, 20, DuelRank.SAPOW),
    Drop(591, 13, 12, DuelRank.SAPOW),
    Drop(602, 13, 32, DuelRank.SAPOW),
    Drop(604, 13, 12, DuelRank.SAPOW),
    Drop(605, 13, 20, DuelRank.SAPOW),
    Drop(606, 13, 32, DuelRank.SAPOW),
    Drop(609, 13, 12, DuelRank.SAPOW),
    Drop(2, 14, 50, DuelRank.SAPOW),
    Drop(16, 14, 46, DuelRank.SAPOW),
    Drop(20, 14, 10, DuelRank.SAPOW),
    Drop(34, 14, 48, DuelRank.SAPOW),
    Drop(44, 14, 50, DuelRank.SAPOW),
    Drop(104, 14, 46, DuelRank.SAPOW),
    Drop(109, 14, 48, DuelRank.SAPOW),
    Drop(114, 14, 46, DuelRank.SAPOW),
    Drop(115, 14, 46, DuelRank.SAPOW),
    Drop(128, 14, 10, DuelRank.SAPOW),
    Drop(129, 14, 46, DuelRank.SAPOW),
    Drop(130, 14, 46, DuelRank.SAPOW),
    Drop(134, 14, 46, DuelRank.SAPOW),
    Drop(142, 14, 46, DuelRank.SAPOW),
    Drop(143, 14, 46, DuelRank.SAPOW),
    Drop(144, 14, 46, DuelRank.SAPOW),
    Drop(145, 14, 46, DuelRank.SAPOW),
    Drop(170, 14, 50, DuelRank.SAPOW),
    Drop(174, 14, 46, DuelRank.SAPOW),
    Drop(179, 14, 46, DuelRank.SAPOW),
    Drop(183, 14, 46, DuelRank.SAPOW),
    Drop(184, 14, 46, DuelRank.SAPOW),
    Drop(190, 14, 46, DuelRank.SAPOW),
    Drop(192, 14, 46, DuelRank.SAPOW),
    Drop(198, 14, 46, DuelRank.SAPOW),
    Drop(208, 14, 46, DuelRank.SAPOW),
    Drop(216, 14, 10, DuelRank.SAPOW),
    Drop(220, 14, 48, DuelRank.SAPOW),
    Drop(229, 14, 46, DuelRank.SAPOW),
    Drop(253, 14, 46, DuelRank.SAPOW),
    Drop(259, 14, 46, DuelRank.SAPOW),
    Drop(260, 14, 46, DuelRank.SAPOW),
    Drop(264, 14, 46, DuelRank.SAPOW),
    Drop(268, 14, 46, DuelRank.SAPOW),
    Drop(276, 14, 46, DuelRank.SAPOW),
    Drop(287, 14, 10, DuelRank.SAPOW),
    Drop(334, 14, 58, DuelRank.SAPOW),
    Drop(387, 14, 46, DuelRank.SAPOW),
    Drop(395, 14, 46, DuelRank.SAPOW),
    Drop(433, 14, 10, DuelRank.SAPOW),
    Drop(486, 14, 46, DuelRank.SAPOW),
    Drop(492, 14, 46, DuelRank.SAPOW),
    Drop(540, 14, 46, DuelRank.SAPOW),
    Drop(578, 14, 10, DuelRank.SAPOW),
    Drop(584, 14, 48, DuelRank.SAPOW),
    Drop(587, 14, 10, DuelRank.SAPOW),
    Drop(601, 14, 48, DuelRank.SAPOW),
    Drop(608, 14, 48, DuelRank.SAPOW),
    Drop(612, 14, 46, DuelRank.SAPOW),
    Drop(630, 14, 10, DuelRank.SAPOW),
    Drop(3, 15, 34, DuelRank.SAPOW),
    Drop(4, 15, 34, DuelRank.SAPOW),
    Drop(10, 15, 34, DuelRank.SAPOW),
    Drop(13, 15, 34, DuelRank.SAPOW),
    Drop(19, 15, 8, DuelRank.SAPOW),
    Drop(23, 15, 34, DuelRank.SAPOW),
    Drop(26, 15, 34, DuelRank.SAPOW),
    Drop(27, 15, 34, DuelRank.SAPOW),
    Drop(29, 15, 34, DuelRank.SAPOW),
    Drop(31, 15, 34, DuelRank.SAPOW),
    Drop(40, 15, 66, DuelRank.SAPOW),
    Drop(44, 15, 36, DuelRank.SAPOW),
    Drop(45, 15, 36, DuelRank.SAPOW),
    Drop(46, 15, 36, DuelRank.SAPOW),
    Drop(47, 15, 36, DuelRank.SAPOW),
    Drop(61, 15, 36, DuelRank.SAPOW),
    Drop(65, 15, 36, DuelRank.SAPOW),
    Drop(91, 15, 36, DuelRank.SAPOW),
    Drop(113, 15, 36, DuelRank.SAPOW),
    Drop(117, 15, 8, DuelRank.SAPOW),
    Drop(121, 15, 36, DuelRank.SAPOW),
    Drop(122, 15, 36, DuelRank.SAPOW),
    Drop(125, 15, 36, DuelRank.SAPOW),
    Drop(155, 15, 36, DuelRank.SAPOW),
    Drop(159, 15, 36, DuelRank.SAPOW),
    Drop(168, 15, 36, DuelRank.SAPOW),
    Drop(186, 15, 36, DuelRank.SAPOW),
    Drop(187, 15, 36, DuelRank.SAPOW),
    Drop(188, 15, 36, DuelRank.SAPOW),
    Drop(189, 15, 36, DuelRank.SAPOW),
    Drop(200, 15, 36, DuelRank.SAPOW),
    Drop(201, 15, 36, DuelRank.SAPOW),
    Drop(202, 15, 36, DuelRank.SAPOW),
    Drop(207, 15, 36, DuelRank.SAPOW),
    Drop(212, 15, 36, DuelRank.SAPOW),
    Drop(219, 15, 36, DuelRank.SAPOW),
    Drop(246, 15, 36, DuelRank.SAPOW),
    Drop(248, 15, 36, DuelRank.SAPOW),
    Drop(255, 15, 36, DuelRank.SAPOW),
    Drop(272, 15, 36, DuelRank.SAPOW),
    Drop(282, 15, 36, DuelRank.SAPOW),
    Drop(296, 15, 36, DuelRank.SAPOW),
    Drop(298, 15, 36, DuelRank.SAPOW),
    Drop(300, 15, 36, DuelRank.SAPOW),
    Drop(330, 15, 30, DuelRank.SAPOW),
    Drop(334, 15, 30, DuelRank.SAPOW),
    Drop(404, 15, 8, DuelRank.SAPOW),
    Drop(466, 15, 36, DuelRank.SAPOW),
    Drop(481, 15, 36, DuelRank.SAPOW),
    Drop(496, 15, 36, DuelRank.SAPOW),
    Drop(527, 15, 36, DuelRank.SAPOW),
    Drop(538, 15, 36, DuelRank.SAPOW),
    Drop(552, 15, 36, DuelRank.SAPOW),
    Drop(595, 15, 8, DuelRank.SAPOW),
    Drop(597, 15, 36, DuelRank.SAPOW),
    Drop(598, 15, 36, DuelRank.SAPOW),
    Drop(604, 15, 36, DuelRank.SAPOW),
    Drop(629, 15, 36, DuelRank.SAPOW),
    Drop(636, 15, 36, DuelRank.SAPOW),
    Drop(642, 15, 36, DuelRank.SAPOW),
    Drop(4, 16, 44, DuelRank.SAPOW),
    Drop(10, 16, 44, DuelRank.SAPOW),
    Drop(20, 16, 10, DuelRank.SAPOW),
    Drop(31, 16, 50, DuelRank.SAPOW),
    Drop(70, 16, 50, DuelRank.SAPOW),
    Drop(71, 16, 50, DuelRank.SAPOW),
    Drop(122, 16, 44, DuelRank.SAPOW),
    Drop(140, 16, 44, DuelRank.SAPOW),
    Drop(152, 16, 44, DuelRank.SAPOW),
    Drop(168, 16, 50, DuelRank.SAPOW),
    Drop(177, 16, 44, DuelRank.SAPOW),
    Drop(193, 16, 44, DuelRank.SAPOW),
    Drop(196, 16, 44, DuelRank.SAPOW),
    Drop(199, 16, 44, DuelRank.SAPOW),
    Drop(200, 16, 44, DuelRank.SAPOW),
    Drop(205, 16, 50, DuelRank.SAPOW),
    Drop(206, 16, 50, DuelRank.SAPOW),
    Drop(227, 16, 50, DuelRank.SAPOW),
    Drop(230, 16, 50, DuelRank.SAPOW),
    Drop(243, 16, 50, DuelRank.SAPOW),
    Drop(247, 16, 50, DuelRank.SAPOW),
    Drop(249, 16, 10, DuelRank.SAPOW),
    Drop(251, 16, 50, DuelRank.SAPOW),
    Drop(258, 16, 50, DuelRank.SAPOW),
    Drop(265, 16, 50, DuelRank.SAPOW),
    Drop(270, 16, 50, DuelRank.SAPOW),
    Drop(289, 16, 50, DuelRank.SAPOW),
    Drop(292, 16, 50, DuelRank.SAPOW),
    Drop(296, 16, 50, DuelRank.SAPOW),
    Drop(298, 16, 50, DuelRank.SAPOW),
    Drop(331, 16, 30, DuelRank.SAPOW),
    Drop(334, 16, 30, DuelRank.SAPOW),
    Drop(393, 16, 50, DuelRank.SAPOW),
    Drop(430, 16, 72, DuelRank.SAPOW),
    Drop(431, 16, 72, DuelRank.SAPOW),
    Drop(440, 16, 50, DuelRank.SAPOW),
    Drop(448, 16, 10, DuelRank.SAPOW),
    Drop(454, 16, 10, DuelRank.SAPOW),
    Drop(456, 16, 10, DuelRank.SAPOW),
    Drop(517, 16, 10, DuelRank.SAPOW),
    Drop(530, 16, 44, DuelRank.SAPOW),
    Drop(550, 16, 46, DuelRank.SAPOW),
    Drop(596, 16, 50, DuelRank.SAPOW),
    Drop(615, 16, 46, DuelRank.SAPOW),
    Drop(621, 16, 10, DuelRank.SAPOW),
    Drop(625, 16, 50, DuelRank.SAPOW),
    Drop(626, 16, 50, DuelRank.SAPOW),
    Drop(649, 16, 48, DuelRank.SAPOW),
    Drop(3, 17, 140, DuelRank.SAPOW),
    Drop(4, 17, 70, DuelRank.SAPOW),
    Drop(5, 17, 140, DuelRank.SAPOW),
    Drop(10, 17, 70, DuelRank.SAPOW),
    Drop(19, 17, 12, DuelRank.SAPOW),
    Drop(23, 17, 140, DuelRank.SAPOW),
    Drop(26, 17, 152, DuelRank.SAPOW),
    Drop(31, 17, 80, DuelRank.SAPOW),
    Drop(34, 17, 142, DuelRank.SAPOW),
    Drop(66, 17, 12, DuelRank.SAPOW),
    Drop(77, 17, 130, DuelRank.SAPOW),
    Drop(91, 17, 148, DuelRank.SAPOW),
    Drop(117, 17, 12, DuelRank.SAPOW),
    Drop(122, 17, 64, DuelRank.SAPOW),
    Drop(128, 17, 12, DuelRank.SAPOW),
    Drop(151, 17, 12, DuelRank.SAPOW),
    Drop(163, 17, 12, DuelRank.SAPOW),
    Drop(168, 17, 80, DuelRank.SAPOW),
    Drop(200, 17, 64, DuelRank.SAPOW),
    Drop(216, 17, 12, DuelRank.SAPOW),
    Drop(249, 17, 12, DuelRank.SAPOW),
    Drop(275, 17, 12, DuelRank.SAPOW),
    Drop(281, 17, 12, DuelRank.SAPOW),
    Drop(287, 17, 12, DuelRank.SAPOW),
    Drop(296, 17, 64, DuelRank.SAPOW),
    Drop(297, 17, 12, DuelRank.SAPOW),
    Drop(298, 17, 64, DuelRank.SAPOW),
    Drop(333, 17, 60, DuelRank.SAPOW),
    Drop(379, 17, 140, DuelRank.SAPOW),
    Drop(404, 17, 12, DuelRank.SAPOW),
    Drop(418, 17, 12, DuelRank.SAPOW),
    Drop(448, 17, 12, DuelRank.SAPOW),
    Drop(454, 17, 12, DuelRank.SAPOW),
    Drop(456, 17, 12, DuelRank.SAPOW),
    Drop(498, 17, 12, DuelRank.SAPOW),
    Drop(517, 17, 12, DuelRank.SAPOW),
    Drop(578, 17, 12, DuelRank.SAPOW),
    Drop(587, 17, 12, DuelRank.SAPOW),
    Drop(595, 17, 12, DuelRank.SAPOW),
    Drop(614, 17, 12, DuelRank.SAPOW),
    Drop(621, 17, 12, DuelRank.SAPOW),
    Drop(630, 17, 12, DuelRank.SAPOW),
    Drop(5, 18, 20, DuelRank.SAPOW),
    Drop(6, 18, 45, DuelRank.SAPOW),
    Drop(13, 18, 22, DuelRank.SAPOW),
    Drop(20, 18, 6, DuelRank.SAPOW),
    Drop(25, 18, 20, DuelRank.SAPOW),
    Drop(26, 18, 22, DuelRank.SAPOW),
    Drop(27, 18, 22, DuelRank.SAPOW),
    Drop(31, 18, 22, DuelRank.SAPOW),
    Drop(32, 18, 24, DuelRank.SAPOW),
    Drop(36, 18, 22, DuelRank.SAPOW),
    Drop(41, 18, 22, DuelRank.SAPOW),
    Drop(43, 18, 22, DuelRank.SAPOW),
    Drop(44, 18, 22, DuelRank.SAPOW),
    Drop(46, 18, 22, DuelRank.SAPOW),
    Drop(48, 18, 20, DuelRank.SAPOW),
    Drop(49, 18, 22, DuelRank.SAPOW),
    Drop(54, 18, 22, DuelRank.SAPOW),
    Drop(55, 18, 22, DuelRank.SAPOW),
    Drop(58, 18, 20, DuelRank.SAPOW),
    Drop(70, 18, 22, DuelRank.SAPOW),
    Drop(71, 18, 22, DuelRank.SAPOW),
    Drop(77, 18, 22, DuelRank.SAPOW),
    Drop(81, 18, 24, DuelRank.SAPOW),
    Drop(83, 18, 72, DuelRank.SAPOW),
    Drop(86, 18, 72, DuelRank.SAPOW),
    Drop(95, 18, 45, DuelRank.SAPOW),
    Drop(103, 18, 20, DuelRank.SAPOW),
    Drop(115, 18, 20, DuelRank.SAPOW),
    Drop(119, 18, 20, DuelRank.SAPOW),
    Drop(121, 18, 20, DuelRank.SAPOW),
    Drop(125, 18, 22, DuelRank.SAPOW),
    Drop(136, 18, 44, DuelRank.SAPOW),
    Drop(137, 18, 20, DuelRank.SAPOW),
    Drop(148, 18, 20, DuelRank.SAPOW),
    Drop(149, 18, 44, DuelRank.SAPOW),
    Drop(162, 18, 22, DuelRank.SAPOW),
    Drop(164, 18, 22, DuelRank.SAPOW),
    Drop(166, 18, 22, DuelRank.SAPOW),
    Drop(168, 18, 22, DuelRank.SAPOW),
    Drop(169, 18, 22, DuelRank.SAPOW),
    Drop(170, 18, 24, DuelRank.SAPOW),
    Drop(171, 18, 22, DuelRank.SAPOW),
    Drop(173, 18, 22, DuelRank.SAPOW),
    Drop(175, 18, 22, DuelRank.SAPOW),
    Drop(178, 18, 22, DuelRank.SAPOW),
    Drop(181, 18, 22, DuelRank.SAPOW),
    Drop(186, 18, 22, DuelRank.SAPOW),
    Drop(193, 18, 22, DuelRank.SAPOW),
    Drop(194, 18, 44, DuelRank.SAPOW),
    Drop(222, 18, 22, DuelRank.SAPOW),
    Drop(230, 18, 22, DuelRank.SAPOW),
    Drop(232, 18, 22, DuelRank.SAPOW),
    Drop(233, 18, 44, DuelRank.SAPOW),
    Drop(240, 18, 22, DuelRank.SAPOW),
    Drop(242, 18, 22, DuelRank.SAPOW),
    Drop(245, 18, 22, DuelRank.SAPOW),
    Drop(254, 18, 22, DuelRank.SAPOW),
    Drop(255, 18, 22, DuelRank.SAPOW),
    Drop(261, 18, 22, DuelRank.SAPOW),
    Drop(269, 18, 22, DuelRank.SAPOW),
    Drop(271, 18, 22, DuelRank.SAPOW),
    Drop(277, 18, 22, DuelRank.SAPOW),
    Drop(279, 18, 22, DuelRank.SAPOW),
    Drop(281, 18, 6, DuelRank.SAPOW),
    Drop(295, 18, 22, DuelRank.SAPOW),
    Drop(335, 18, 30, DuelRank.SAPOW),
    Drop(379, 18, 44, DuelRank.SAPOW),
    Drop(402, 18, 22, DuelRank.SAPOW),
    Drop(412, 18, 22, DuelRank.SAPOW),
    Drop(415, 18, 22, DuelRank.SAPOW),
    Drop(416, 18, 22, DuelRank.SAPOW),
    Drop(459, 18, 22, DuelRank.SAPOW),
    Drop(462, 18, 24, DuelRank.SAPOW),
    Drop(466, 18, 22, DuelRank.SAPOW),
    Drop(490, 18, 22, DuelRank.SAPOW),
    Drop(504, 18, 22, DuelRank.SAPOW),
    Drop(511, 18, 22, DuelRank.SAPOW),
    Drop(512, 18, 22, DuelRank.SAPOW),
    Drop(563, 18, 22, DuelRank.SAPOW),
    Drop(597, 18, 22, DuelRank.SAPOW),
    Drop(608, 18, 22, DuelRank.SAPOW),
    Drop(611, 18, 22, DuelRank.SAPOW),
    Drop(636, 18, 24, DuelRank.SAPOW),
    Drop(4, 19, 22, DuelRank.SAPOW),
    Drop(12, 19, 24, DuelRank.SAPOW),
    Drop(14, 19, 24, DuelRank.SAPOW),
    Drop(15, 19, 24, DuelRank.SAPOW),
    Drop(20, 19, 8, DuelRank.SAPOW),
    Drop(29, 19, 22, DuelRank.SAPOW),
    Drop(43, 19, 22, DuelRank.SAPOW),
    Drop(50, 19, 44, DuelRank.SAPOW),
    Drop(61, 19, 22, DuelRank.SAPOW),
    Drop(68, 19, 24, DuelRank.SAPOW),
    Drop(78, 19, 22, DuelRank.SAPOW),
    Drop(82, 19, 24, DuelRank.SAPOW),
    Drop(100, 19, 22, DuelRank.SAPOW),
    Drop(104, 19, 44, DuelRank.SAPOW),
    Drop(105, 19, 44, DuelRank.SAPOW),
    Drop(122, 19, 44, DuelRank.SAPOW),
    Drop(130, 19, 44, DuelRank.SAPOW),
    Drop(135, 19, 44, DuelRank.SAPOW),
    Drop(137, 19, 44, DuelRank.SAPOW),
    Drop(152, 19, 44, DuelRank.SAPOW),
    Drop(154, 19, 44, DuelRank.SAPOW),
    Drop(174, 19, 44, DuelRank.SAPOW),
    Drop(182, 19, 44, DuelRank.SAPOW),
    Drop(185, 19, 44, DuelRank.SAPOW),
    Drop(191, 19, 44, DuelRank.SAPOW),
    Drop(197, 19, 44, DuelRank.SAPOW),
    Drop(202, 19, 44, DuelRank.SAPOW),
    Drop(207, 19, 44, DuelRank.SAPOW),
    Drop(210, 19, 44, DuelRank.SAPOW),
    Drop(214, 19, 44, DuelRank.SAPOW),
    Drop(237, 19, 44, DuelRank.SAPOW),
    Drop(242, 19, 42, DuelRank.SAPOW),
    Drop(245, 19, 42, DuelRank.SAPOW),
    Drop(268, 19, 42, DuelRank.SAPOW),
    Drop(333, 19, 16, DuelRank.SAPOW),
    Drop(389, 19, 4, DuelRank.SAPOW),
    Drop(410, 19, 42, DuelRank.SAPOW),
    Drop(420, 19, 42, DuelRank.SAPOW),
    Drop(422, 19, 42, DuelRank.SAPOW),
    Drop(436, 19, 42, DuelRank.SAPOW),
    Drop(444, 19, 42, DuelRank.SAPOW),
    Drop(485, 19, 42, DuelRank.SAPOW),
    Drop(486, 19, 42, DuelRank.SAPOW),
    Drop(488, 19, 42, DuelRank.SAPOW),
    Drop(492, 19, 42, DuelRank.SAPOW),
    Drop(501, 19, 42, DuelRank.SAPOW),
    Drop(516, 19, 42, DuelRank.SAPOW),
    Drop(524, 19, 42, DuelRank.SAPOW),
    Drop(549, 19, 42, DuelRank.SAPOW),
    Drop(558, 19, 42, DuelRank.SAPOW),
    Drop(563, 19, 42, DuelRank.SAPOW),
    Drop(579, 19, 42, DuelRank.SAPOW),
    Drop(589, 19, 42, DuelRank.SAPOW),
    Drop(598, 19, 42, DuelRank.SAPOW),
    Drop(611, 19, 42, DuelRank.SAPOW),
    Drop(713, 19, 8, DuelRank.SAPOW),
    Drop(19, 20, 8, DuelRank.SAPOW),
    Drop(24, 20, 182, DuelRank.SAPOW),
    Drop(58, 20, 182, DuelRank.SAPOW),
    Drop(167, 20, 182, DuelRank.SAPOW),
    Drop(330, 20, 38, DuelRank.SAPOW),
    Drop(393, 20, 182, DuelRank.SAPOW),
    Drop(395, 20, 182, DuelRank.SAPOW),
    Drop(398, 20, 182, DuelRank.SAPOW),
    Drop(399, 20, 182, DuelRank.SAPOW),
    Drop(402, 20, 182, DuelRank.SAPOW),
    Drop(469, 20, 182, DuelRank.SAPOW),
    Drop(475, 20, 182, DuelRank.SAPOW),
    Drop(527, 20, 182, DuelRank.SAPOW),
    Drop(5, 21, 12, DuelRank.SAPOW),
    Drop(8, 21, 12, DuelRank.SAPOW),
    Drop(20, 21, 6, DuelRank.SAPOW),
    Drop(29, 21, 12, DuelRank.SAPOW),
    Drop(45, 21, 12, DuelRank.SAPOW),
    Drop(48, 21, 12, DuelRank.SAPOW),
    Drop(50, 21, 12, DuelRank.SAPOW),
    Drop(70, 21, 52, DuelRank.SAPOW),
    Drop(71, 21, 52, DuelRank.SAPOW),
    Drop(73, 21, 52, DuelRank.SAPOW),
    Drop(77, 21, 24, DuelRank.SAPOW),
    Drop(89, 21, 24, DuelRank.SAPOW),
    Drop(96, 21, 12, DuelRank.SAPOW),
    Drop(97, 21, 12, DuelRank.SAPOW),
    Drop(98, 21, 12, DuelRank.SAPOW),
    Drop(101, 21, 12, DuelRank.SAPOW),
    Drop(107, 21, 12, DuelRank.SAPOW),
    Drop(108, 21, 12, DuelRank.SAPOW),
    Drop(122, 21, 12, DuelRank.SAPOW),
    Drop(129, 21, 12, DuelRank.SAPOW),
    Drop(131, 21, 26, DuelRank.SAPOW),
    Drop(134, 21, 12, DuelRank.SAPOW),
    Drop(135, 21, 12, DuelRank.SAPOW),
    Drop(139, 21, 12, DuelRank.SAPOW),
    Drop(140, 21, 24, DuelRank.SAPOW),
    Drop(144, 21, 12, DuelRank.SAPOW),
    Drop(145, 21, 12, DuelRank.SAPOW),
    Drop(147, 21, 12, DuelRank.SAPOW),
    Drop(148, 21, 12, DuelRank.SAPOW),
    Drop(150, 21, 26, DuelRank.SAPOW),
    Drop(152, 21, 34, DuelRank.SAPOW),
    Drop(154, 21, 12, DuelRank.SAPOW),
    Drop(156, 21, 12, DuelRank.SAPOW),
    Drop(157, 21, 12, DuelRank.SAPOW),
    Drop(158, 21, 12, DuelRank.SAPOW),
    Drop(159, 21, 12, DuelRank.SAPOW),
    Drop(160, 21, 12, DuelRank.SAPOW),
    Drop(161, 21, 12, DuelRank.SAPOW),
    Drop(173, 21, 12, DuelRank.SAPOW),
    Drop(176, 21, 12, DuelRank.SAPOW),
    Drop(177, 21, 24, DuelRank.SAPOW),
    Drop(179, 21, 12, DuelRank.SAPOW),
    Drop(182, 21, 12, DuelRank.SAPOW),
    Drop(183, 21, 12, DuelRank.SAPOW),
    Drop(184, 21, 12, DuelRank.SAPOW),
    Drop(187, 21, 10, DuelRank.SAPOW),
    Drop(189, 21, 10, DuelRank.SAPOW),
    Drop(191, 21, 10, DuelRank.SAPOW),
    Drop(193, 21, 22, DuelRank.SAPOW),
    Drop(195, 21, 10, DuelRank.SAPOW),
    Drop(196, 21, 22, DuelRank.SAPOW),
    Drop(198, 21, 10, DuelRank.SAPOW),
    Drop(199, 21, 22, DuelRank.SAPOW),
    Drop(200, 21, 10, DuelRank.SAPOW),
    Drop(203, 21, 10, DuelRank.SAPOW),
    Drop(205, 21, 22, DuelRank.SAPOW),
    Drop(206, 21, 34, DuelRank.SAPOW),
    Drop(208, 21, 10, DuelRank.SAPOW),
    Drop(209, 21, 10, DuelRank.SAPOW),
    Drop(211, 21, 10, DuelRank.SAPOW),
    Drop(212, 21, 10, DuelRank.SAPOW),
    Drop(214, 21, 10, DuelRank.SAPOW),
    Drop(218, 21, 10, DuelRank.SAPOW),
    Drop(222, 21, 10, DuelRank.SAPOW),
    Drop(223, 21, 24, DuelRank.SAPOW),
    Drop(224, 21, 10, DuelRank.SAPOW),
    Drop(227, 21, 34, DuelRank.SAPOW),
    Drop(229, 21, 10, DuelRank.SAPOW),
    Drop(230, 21, 22, DuelRank.SAPOW),
    Drop(232, 21, 10, DuelRank.SAPOW),
    Drop(239, 21, 10, DuelRank.SAPOW),
    Drop(240, 21, 10, DuelRank.SAPOW),
    Drop(242, 21, 10, DuelRank.SAPOW),
    Drop(243, 21, 34, DuelRank.SAPOW),
    Drop(245, 21, 10, DuelRank.SAPOW),
    Drop(247, 21, 22, DuelRank.SAPOW),
    Drop(251, 21, 22, DuelRank.SAPOW),
    Drop(254, 21, 10, DuelRank.SAPOW),
    Drop(258, 21, 22, DuelRank.SAPOW),
    Drop(261, 21, 10, DuelRank.SAPOW),
    Drop(265, 21, 34, DuelRank.SAPOW),
    Drop(266, 21, 10, DuelRank.SAPOW),
    Drop(267, 21, 10, DuelRank.SAPOW),
    Drop(268, 21, 10, DuelRank.SAPOW),
    Drop(270, 21, 22, DuelRank.SAPOW),
    Drop(271, 21, 10, DuelRank.SAPOW),
    Drop(277, 21, 10, DuelRank.SAPOW),
    Drop(285, 21, 34, DuelRank.SAPOW),
    Drop(292, 21, 34, DuelRank.SAPOW),
    Drop(334, 21, 30, DuelRank.SAPOW),
    Drop(400, 21, 24, DuelRank.SAPOW),
    Drop(420, 21, 10, DuelRank.SAPOW),
    Drop(421, 21, 10, DuelRank.SAPOW),
    Drop(432, 21, 22, DuelRank.SAPOW),
    Drop(442, 21, 2, DuelRank.SAPOW),
    Drop(443, 21, 24, DuelRank.SAPOW),
    Drop(446, 21, 22, DuelRank.SAPOW),
    Drop(451, 21, 34, DuelRank.SAPOW),
    Drop(452, 21, 34, DuelRank.SAPOW),
    Drop(476, 21, 10, DuelRank.SAPOW),
    Drop(490, 21, 10, DuelRank.SAPOW),
    Drop(492, 21, 10, DuelRank.SAPOW),
    Drop(503, 21, 34, DuelRank.SAPOW),
    Drop(506, 21, 10, DuelRank.SAPOW),
    Drop(510, 21, 10, DuelRank.SAPOW),
    Drop(524, 21, 22, DuelRank.SAPOW),
    Drop(534, 21, 10, DuelRank.SAPOW),
    Drop(536, 21, 34, DuelRank.SAPOW),
    Drop(540, 21, 10, DuelRank.SAPOW),
    Drop(544, 21, 10, DuelRank.SAPOW),
    Drop(549, 21, 34, DuelRank.SAPOW),
    Drop(556, 21, 10, DuelRank.SAPOW),
    Drop(569, 21, 22, DuelRank.SAPOW),
    Drop(590, 21, 22, DuelRank.SAPOW),
    Drop(591, 21, 10, DuelRank.SAPOW),
    Drop(602, 21, 34, DuelRank.SAPOW),
    Drop(604, 21, 10, DuelRank.SAPOW),
    Drop(605, 21, 22, DuelRank.SAPOW),
    Drop(606, 21, 34, DuelRank.SAPOW),
    Drop(609, 21, 10, DuelRank.SAPOW),
    Drop(2, 22, 25, DuelRank.SAPOW),
    Drop(12, 22, 25, DuelRank.SAPOW),
    Drop(14, 22, 25, DuelRank.SAPOW),
    Drop(15, 22, 25, DuelRank.SAPOW),
    Drop(20, 22, 6, DuelRank.SAPOW),
    Drop(34, 22, 25, DuelRank.SAPOW),
    Drop(42, 22, 52, DuelRank.SAPOW),
    Drop(44, 22, 25, DuelRank.SAPOW),
    Drop(68, 22, 25, DuelRank.SAPOW),
    Drop(73, 22, 25, DuelRank.SAPOW),
    Drop(74, 22, 25, DuelRank.SAPOW),
    Drop(84, 22, 25, DuelRank.SAPOW),
    Drop(87, 22, 25, DuelRank.SAPOW),
    Drop(88, 22, 25, DuelRank.SAPOW),
    Drop(89, 22, 25, DuelRank.SAPOW),
    Drop(90, 22, 25, DuelRank.SAPOW),
    Drop(93, 22, 25, DuelRank.SAPOW),
    Drop(104, 22, 25, DuelRank.SAPOW),
    Drop(106, 22, 52, DuelRank.SAPOW),
    Drop(109, 22, 25, DuelRank.SAPOW),
    Drop(114, 22, 25, DuelRank.SAPOW),
    Drop(115, 22, 25, DuelRank.SAPOW),
    Drop(124, 22, 25, DuelRank.SAPOW),
    Drop(126, 22, 52, DuelRank.SAPOW),
    Drop(129, 22, 25, DuelRank.SAPOW),
    Drop(130, 22, 25, DuelRank.SAPOW),
    Drop(131, 22, 25, DuelRank.SAPOW),
    Drop(134, 22, 25, DuelRank.SAPOW),
    Drop(142, 22, 25, DuelRank.SAPOW),
    Drop(143, 22, 25, DuelRank.SAPOW),
    Drop(144, 22, 25, DuelRank.SAPOW),
    Drop(145, 22, 25, DuelRank.SAPOW),
    Drop(150, 22, 25, DuelRank.SAPOW),
    Drop(170, 22, 25, DuelRank.SAPOW),
    Drop(174, 22, 25, DuelRank.SAPOW),
    Drop(179, 22, 25, DuelRank.SAPOW),
    Drop(183, 22, 25, DuelRank.SAPOW),
    Drop(184, 22, 25, DuelRank.SAPOW),
    Drop(190, 22, 25, DuelRank.SAPOW),
    Drop(198, 22, 25, DuelRank.SAPOW),
    Drop(204, 22, 25, DuelRank.SAPOW),
    Drop(208, 22, 25, DuelRank.SAPOW),
    Drop(213, 22, 52, DuelRank.SAPOW),
    Drop(220, 22, 25, DuelRank.SAPOW),
    Drop(229, 22, 25, DuelRank.SAPOW),
    Drop(253, 22, 25, DuelRank.SAPOW),
    Drop(259, 22, 25, DuelRank.SAPOW),
    Drop(260, 22, 25, DuelRank.SAPOW),
    Drop(264, 22, 25, DuelRank.SAPOW),
    Drop(268, 22, 25, DuelRank.SAPOW),
    Drop(276, 22, 25, DuelRank.SAPOW),
    Drop(286, 22, 25, DuelRank.SAPOW),
    Drop(334, 22, 30, DuelRank.SAPOW),
    Drop(366, 22, 25, DuelRank.SAPOW),
    Drop(376, 22, 25, DuelRank.SAPOW),
    Drop(383, 22, 2, DuelRank.SAPOW),
    Drop(400, 22, 25, DuelRank.SAPOW),
    Drop(403, 22, 25, DuelRank.SAPOW),
    Drop(413, 22, 25, DuelRank.SAPOW),
    Drop(425, 22, 25, DuelRank.SAPOW),
    Drop(441, 22, 25, DuelRank.SAPOW),
    Drop(464, 22, 25, DuelRank.SAPOW),
    Drop(468, 22, 25, DuelRank.SAPOW),
    Drop(486, 22, 25, DuelRank.SAPOW),
    Drop(491, 22, 25, DuelRank.SAPOW),
    Drop(492, 22, 25, DuelRank.SAPOW),
    Drop(508, 22, 25, DuelRank.SAPOW),
    Drop(513, 22, 25, DuelRank.SAPOW),
    Drop(528, 22, 25, DuelRank.SAPOW),
    Drop(540, 22, 25, DuelRank.SAPOW),
    Drop(582, 22, 25, DuelRank.SAPOW),
    Drop(584, 22, 25, DuelRank.SAPOW),
    Drop(601, 22, 25, DuelRank.SAPOW),
    Drop(608, 22, 25, DuelRank.SAPOW),
    Drop(612, 22, 25, DuelRank.SAPOW),
    Drop(633, 22, 52, DuelRank.SAPOW),
    Drop(638, 22, 25, DuelRank.SAPOW),
    Drop(650, 22, 25, DuelRank.SAPOW),
    Drop(8, 23, 46, DuelRank.SAPOW),
    Drop(19, 23, 6, DuelRank.SAPOW),
    Drop(40, 23, 24, DuelRank.SAPOW),
    Drop(50, 23, 24, DuelRank.SAPOW),
    Drop(75, 23, 120, DuelRank.SAPOW),
    Drop(76, 23, 46, DuelRank.SAPOW),
    Drop(98, 23, 24, DuelRank.SAPOW),
    Drop(101, 23, 46, DuelRank.SAPOW),
    Drop(104, 23, 24, DuelRank.SAPOW),
    Drop(107, 23, 24, DuelRank.SAPOW),
    Drop(122, 23, 24, DuelRank.SAPOW),
    Drop(130, 23, 24, DuelRank.SAPOW),
    Drop(133, 23, 24, DuelRank.SAPOW),
    Drop(134, 23, 24, DuelRank.SAPOW),
    Drop(135, 23, 24, DuelRank.SAPOW),
    Drop(145, 23, 24, DuelRank.SAPOW),
    Drop(152, 23, 24, DuelRank.SAPOW),
    Drop(154, 23, 24, DuelRank.SAPOW),
    Drop(157, 23, 44, DuelRank.SAPOW),
    Drop(158, 23, 44, DuelRank.SAPOW),
    Drop(159, 23, 20, DuelRank.SAPOW),
    Drop(174, 23, 20, DuelRank.SAPOW),
    Drop(176, 23, 44, DuelRank.SAPOW),
    Drop(179, 23, 20, DuelRank.SAPOW),
    Drop(180, 23, 20, DuelRank.SAPOW),
    Drop(182, 23, 20, DuelRank.SAPOW),
    Drop(183, 23, 20, DuelRank.SAPOW),
    Drop(185, 23, 20, DuelRank.SAPOW),
    Drop(191, 23, 20, DuelRank.SAPOW),
    Drop(195, 23, 20, DuelRank.SAPOW),
    Drop(198, 23, 20, DuelRank.SAPOW),
    Drop(200, 23, 20, DuelRank.SAPOW),
    Drop(202, 23, 20, DuelRank.SAPOW),
    Drop(203, 23, 20, DuelRank.SAPOW),
    Drop(207, 23, 20, DuelRank.SAPOW),
    Drop(209, 23, 20, DuelRank.SAPOW),
    Drop(210, 23, 44, DuelRank.SAPOW),
    Drop(211, 23, 20, DuelRank.SAPOW),
    Drop(212, 23, 20, DuelRank.SAPOW),
    Drop(214, 23, 20, DuelRank.SAPOW),
    Drop(222, 23, 20, DuelRank.SAPOW),
    Drop(227, 23, 20, DuelRank.SAPOW),
    Drop(229, 23, 20, DuelRank.SAPOW),
    Drop(232, 23, 20, DuelRank.SAPOW),
    Drop(237, 23, 20, DuelRank.SAPOW),
    Drop(242, 23, 20, DuelRank.SAPOW),
    Drop(245, 23, 20, DuelRank.SAPOW),
    Drop(254, 23, 20, DuelRank.SAPOW),
    Drop(261, 23, 20, DuelRank.SAPOW),
    Drop(268, 23, 20, DuelRank.SAPOW),
    Drop(273, 23, 20, DuelRank.SAPOW),
    Drop(274, 23, 20, DuelRank.SAPOW),
    Drop(285, 23, 20, DuelRank.SAPOW),
    Drop(291, 23, 20, DuelRank.SAPOW),
    Drop(292, 23, 20, DuelRank.SAPOW),
    Drop(330, 23, 30, DuelRank.SAPOW),
    Drop(377, 23, 2, DuelRank.SAPOW),
    Drop(420, 23, 20, DuelRank.SAPOW),
    Drop(452, 23, 20, DuelRank.SAPOW),
    Drop(476, 23, 20, DuelRank.SAPOW),
    Drop(486, 23, 20, DuelRank.SAPOW),
    Drop(492, 23, 20, DuelRank.SAPOW),
    Drop(501, 23, 20, DuelRank.SAPOW),
    Drop(506, 23, 20, DuelRank.SAPOW),
    Drop(510, 23, 48, DuelRank.SAPOW),
    Drop(511, 23, 20, DuelRank.SAPOW),
    Drop(524, 23, 20, DuelRank.SAPOW),
    Drop(536, 23, 20, DuelRank.SAPOW),
    Drop(540, 23, 20, DuelRank.SAPOW),
    Drop(544, 23, 20, DuelRank.SAPOW),
    Drop(549, 23, 20, DuelRank.SAPOW),
    Drop(556, 23, 20, DuelRank.SAPOW),
    Drop(567, 23, 20, DuelRank.SAPOW),
    Drop(579, 23, 44, DuelRank.SAPOW),
    Drop(589, 23, 44, DuelRank.SAPOW),
    Drop(591, 23, 20, DuelRank.SAPOW),
    Drop(594, 23, 24, DuelRank.SAPOW),
    Drop(598, 23, 20, DuelRank.SAPOW),
    Drop(602, 23, 20, DuelRank.SAPOW),
    Drop(609, 23, 20, DuelRank.SAPOW),
    Drop(611, 23, 20, DuelRank.SAPOW),
    Drop(620, 23, 20, DuelRank.SAPOW),
    Drop(638, 23, 24, DuelRank.SAPOW),
    Drop(3, 24, 24, DuelRank.SAPOW),
    Drop(8, 24, 24, DuelRank.SAPOW),
    Drop(10, 24, 24, DuelRank.SAPOW),
    Drop(13, 24, 24, DuelRank.SAPOW),
    Drop(19, 24, 6, DuelRank.SAPOW),
    Drop(25, 24, 24, DuelRank.SAPOW),
    Drop(41, 24, 24, DuelRank.SAPOW),
    Drop(44, 24, 24, DuelRank.SAPOW),
    Drop(49, 24, 72, DuelRank.SAPOW),
    Drop(50, 24, 72, DuelRank.SAPOW),
    Drop(53, 24, 102, DuelRank.SAPOW),
    Drop(54, 24, 102, DuelRank.SAPOW),
    Drop(55, 24, 72, DuelRank.SAPOW),
    Drop(70, 24, 24, DuelRank.SAPOW),
    Drop(77, 24, 24, DuelRank.SAPOW),
    Drop(80, 24, 24, DuelRank.SAPOW),
    Drop(109, 24, 24, DuelRank.SAPOW),
    Drop(115, 24, 24, DuelRank.SAPOW),
    Drop(116, 24, 24, DuelRank.SAPOW),
    Drop(119, 24, 24, DuelRank.SAPOW),
    Drop(121, 24, 24, DuelRank.SAPOW),
    Drop(125, 24, 22, DuelRank.SAPOW),
    Drop(141, 24, 22, DuelRank.SAPOW),
    Drop(149, 24, 22, DuelRank.SAPOW),
    Drop(157, 24, 22, DuelRank.SAPOW),
    Drop(158, 24, 22, DuelRank.SAPOW),
    Drop(164, 24, 22, DuelRank.SAPOW),
    Drop(166, 24, 22, DuelRank.SAPOW),
    Drop(171, 24, 22, DuelRank.SAPOW),
    Drop(180, 24, 22, DuelRank.SAPOW),
    Drop(186, 24, 22, DuelRank.SAPOW),
    Drop(193, 24, 22, DuelRank.SAPOW),
    Drop(194, 24, 22, DuelRank.SAPOW),
    Drop(196, 24, 22, DuelRank.SAPOW),
    Drop(201, 24, 22, DuelRank.SAPOW),
    Drop(209, 24, 22, DuelRank.SAPOW),
    Drop(220, 24, 22, DuelRank.SAPOW),
    Drop(221, 24, 22, DuelRank.SAPOW),
    Drop(225, 24, 22, DuelRank.SAPOW),
    Drop(233, 24, 22, DuelRank.SAPOW),
    Drop(236, 24, 22, DuelRank.SAPOW),
    Drop(248, 24, 22, DuelRank.SAPOW),
    Drop(255, 24, 22, DuelRank.SAPOW),
    Drop(256, 24, 22, DuelRank.SAPOW),
    Drop(259, 24, 22, DuelRank.SAPOW),
    Drop(272, 24, 22, DuelRank.SAPOW),
    Drop(273, 24, 22, DuelRank.SAPOW),
    Drop(274, 24, 22, DuelRank.SAPOW),
    Drop(280, 24, 22, DuelRank.SAPOW),
    Drop(291, 24, 22, DuelRank.SAPOW),
    Drop(293, 24, 22, DuelRank.SAPOW),
    Drop(330, 24, 30, DuelRank.SAPOW),
    Drop(367, 24, 2, DuelRank.SAPOW),
    Drop(406, 24, 22, DuelRank.SAPOW),
    Drop(412, 24, 22, DuelRank.SAPOW),
    Drop(459, 24, 22, DuelRank.SAPOW),
    Drop(476, 24, 22, DuelRank.SAPOW),
    Drop(477, 24, 22, DuelRank.SAPOW),
    Drop(478, 24, 22, DuelRank.SAPOW),
    Drop(479, 24, 22, DuelRank.SAPOW),
    Drop(496, 24, 22, DuelRank.SAPOW),
    Drop(501, 24, 22, DuelRank.SAPOW),
    Drop(506, 24, 22, DuelRank.SAPOW),
    Drop(510, 24, 22, DuelRank.SAPOW),
    Drop(511, 24, 22, DuelRank.SAPOW),
    Drop(514, 24, 22, DuelRank.SAPOW),
    Drop(534, 24, 22, DuelRank.SAPOW),
    Drop(552, 24, 22, DuelRank.SAPOW),
    Drop(567, 24, 22, DuelRank.SAPOW),
    Drop(576, 24, 22, DuelRank.SAPOW),
    Drop(579, 24, 22, DuelRank.SAPOW),
    Drop(589, 24, 22, DuelRank.SAPOW),
    Drop(594, 24, 22, DuelRank.SAPOW),
    Drop(608, 24, 22, DuelRank.SAPOW),
    Drop(609, 24, 22, DuelRank.SAPOW),
    Drop(620, 24, 42, DuelRank.SAPOW),
    Drop(638, 24, 22, DuelRank.SAPOW),
    Drop(643, 24, 22, DuelRank.SAPOW),
    Drop(9, 25, 72, DuelRank.SAPOW),
    Drop(20, 25, 8, DuelRank.SAPOW),
    Drop(24, 25, 72, DuelRank.SAPOW),
    Drop(58, 25, 72, DuelRank.SAPOW),
    Drop(94, 25, 12, DuelRank.SAPOW),
    Drop(111, 25, 12, DuelRank.SAPOW),
    Drop(112, 25, 11, DuelRank.SAPOW),
    Drop(123, 25, 72, DuelRank.SAPOW),
    Drop(146, 25, 11, DuelRank.SAPOW),
    Drop(153, 25, 11, DuelRank.SAPOW),
    Drop(165, 25, 11, DuelRank.SAPOW),
    Drop(167, 25, 72, DuelRank.SAPOW),
    Drop(192, 25, 72, DuelRank.SAPOW),
    Drop(234, 25, 11, DuelRank.SAPOW),
    Drop(241, 25, 11, DuelRank.SAPOW),
    Drop(289, 25, 72, DuelRank.SAPOW),
    Drop(332, 25, 20, DuelRank.SAPOW),
    Drop(384, 25, 11, DuelRank.SAPOW),
    Drop(387, 25, 72, DuelRank.SAPOW),
    Drop(393, 25, 72, DuelRank.SAPOW),
    Drop(396, 25, 12, DuelRank.SAPOW),
    Drop(397, 25, 72, DuelRank.SAPOW),
    Drop(398, 25, 72, DuelRank.SAPOW),
    Drop(399, 25, 72, DuelRank.SAPOW),
    Drop(402, 25, 72, DuelRank.SAPOW),
    Drop(408, 25, 12, DuelRank.SAPOW),
    Drop(409, 25, 12, DuelRank.SAPOW),
    Drop(411, 25, 72, DuelRank.SAPOW),
    Drop(435, 25, 11, DuelRank.SAPOW),
    Drop(437, 25, 12, DuelRank.SAPOW),
    Drop(439, 25, 12, DuelRank.SAPOW),
    Drop(450, 25, 11, DuelRank.SAPOW),
    Drop(458, 25, 12, DuelRank.SAPOW),
    Drop(469, 25, 72, DuelRank.SAPOW),
    Drop(473, 25, 12, DuelRank.SAPOW),
    Drop(475, 25, 72, DuelRank.SAPOW),
    Drop(480, 25, 11, DuelRank.SAPOW),
    Drop(482, 25, 12, DuelRank.SAPOW),
    Drop(484, 25, 70, DuelRank.SAPOW),
    Drop(487, 25, 12, DuelRank.SAPOW),
    Drop(502, 25, 11, DuelRank.SAPOW),
    Drop(504, 25, 70, DuelRank.SAPOW),
    Drop(509, 25, 6, DuelRank.SAPOW),
    Drop(521, 25, 12, DuelRank.SAPOW),
    Drop(527, 25, 70, DuelRank.SAPOW),
    Drop(533, 25, 12, DuelRank.SAPOW),
    Drop(542, 25, 12, DuelRank.SAPOW),
    Drop(547, 25, 70, DuelRank.SAPOW),
    Drop(548, 25, 70, DuelRank.SAPOW),
    Drop(557, 25, 12, DuelRank.SAPOW),
    Drop(558, 25, 70, DuelRank.SAPOW),
    Drop(570, 25, 11, DuelRank.SAPOW),
    Drop(580, 25, 11, DuelRank.SAPOW),
    Drop(618, 25, 11, DuelRank.SAPOW),
    Drop(635, 25, 70, DuelRank.SAPOW),
    Drop(637, 25, 12, DuelRank.SAPOW),
    Drop(707, 25, 6, DuelRank.SAPOW),
    Drop(712, 25, 12, DuelRank.SAPOW),
    Drop(713, 25, 2, DuelRank.SAPOW),
    Drop(714, 25, 6, DuelRank.SAPOW),
    Drop(2, 26, 25, DuelRank.SAPOW),
    Drop(6, 26, 25, DuelRank.SAPOW),
    Drop(13, 26, 75, DuelRank.SAPOW),
    Drop(20, 26, 6, DuelRank.SAPOW),
    Drop(26, 26, 25, DuelRank.SAPOW),
    Drop(27, 26, 25, DuelRank.SAPOW),
    Drop(31, 26, 25, DuelRank.SAPOW),
    Drop(32, 26, 25, DuelRank.SAPOW),
    Drop(36, 26, 25, DuelRank.SAPOW),
    Drop(41, 26, 25, DuelRank.SAPOW),
    Drop(43, 26, 25, DuelRank.SAPOW),
    Drop(44, 26, 25, DuelRank.SAPOW),
    Drop(46, 26, 25, DuelRank.SAPOW),
    Drop(49, 26, 25, DuelRank.SAPOW),
    Drop(54, 26, 25, DuelRank.SAPOW),
    Drop(55, 26, 25, DuelRank.SAPOW),
    Drop(70, 26, 25, DuelRank.SAPOW),
    Drop(71, 26, 25, DuelRank.SAPOW),
    Drop(78, 26, 25, DuelRank.SAPOW),
    Drop(81, 26, 25, DuelRank.SAPOW),
    Drop(83, 26, 25, DuelRank.SAPOW),
    Drop(86, 26, 25, DuelRank.SAPOW),
    Drop(89, 26, 25, DuelRank.SAPOW),
    Drop(91, 26, 25, DuelRank.SAPOW),
    Drop(95, 26, 25, DuelRank.SAPOW),
    Drop(125, 26, 75, DuelRank.SAPOW),
    Drop(127, 26, 25, DuelRank.SAPOW),
    Drop(131, 26, 25, DuelRank.SAPOW),
    Drop(136, 26, 25, DuelRank.SAPOW),
    Drop(149, 26, 25, DuelRank.SAPOW),
    Drop(168, 26, 25, DuelRank.SAPOW),
    Drop(170, 26, 25, DuelRank.SAPOW),
    Drop(186, 26, 55, DuelRank.SAPOW),
    Drop(207, 26, 55, DuelRank.SAPOW),
    Drop(230, 26, 25, DuelRank.SAPOW),
    Drop(272, 26, 55, DuelRank.SAPOW),
    Drop(332, 26, 30, DuelRank.SAPOW),
    Drop(366, 26, 25, DuelRank.SAPOW),
    Drop(368, 26, 25, DuelRank.SAPOW),
    Drop(379, 26, 25, DuelRank.SAPOW),
    Drop(386, 26, 2, DuelRank.SAPOW),
    Drop(405, 26, 25, DuelRank.SAPOW),
    Drop(412, 26, 25, DuelRank.SAPOW),
    Drop(415, 26, 25, DuelRank.SAPOW),
    Drop(416, 26, 25, DuelRank.SAPOW),
    Drop(460, 26, 25, DuelRank.SAPOW),
    Drop(462, 26, 25, DuelRank.SAPOW),
    Drop(464, 26, 50, DuelRank.SAPOW),
    Drop(465, 26, 55, DuelRank.SAPOW),
    Drop(466, 26, 75, DuelRank.SAPOW),
    Drop(467, 26, 55, DuelRank.SAPOW),
    Drop(468, 26, 50, DuelRank.SAPOW),
    Drop(491, 26, 55, DuelRank.SAPOW),
    Drop(511, 26, 25, DuelRank.SAPOW),
    Drop(512, 26, 25, DuelRank.SAPOW),
    Drop(522, 26, 50, DuelRank.SAPOW),
    Drop(538, 26, 50, DuelRank.SAPOW),
    Drop(552, 26, 50, DuelRank.SAPOW),
    Drop(577, 26, 55, DuelRank.SAPOW),
    Drop(597, 26, 25, DuelRank.SAPOW),
    Drop(636, 26, 75, DuelRank.SAPOW),
    Drop(648, 26, 25, DuelRank.SAPOW),
    Drop(5, 27, 18, DuelRank.SAPOW),
    Drop(8, 27, 18, DuelRank.SAPOW),
    Drop(11, 27, 80, DuelRank.SAPOW),
    Drop(19, 27, 6, DuelRank.SAPOW),
    Drop(29, 27, 18, DuelRank.SAPOW),
    Drop(32, 27, 66, DuelRank.SAPOW),
    Drop(45, 27, 18, DuelRank.SAPOW),
    Drop(47, 27, 18, DuelRank.SAPOW),
    Drop(48, 27, 18, DuelRank.SAPOW),
    Drop(59, 27, 34, DuelRank.SAPOW),
    Drop(79, 27, 80, DuelRank.SAPOW),
    Drop(80, 27, 64, DuelRank.SAPOW),
    Drop(81, 27, 64, DuelRank.SAPOW),
    Drop(96, 27, 16, DuelRank.SAPOW),
    Drop(97, 27, 16, DuelRank.SAPOW),
    Drop(100, 27, 16, DuelRank.SAPOW),
    Drop(107, 27, 16, DuelRank.SAPOW),
    Drop(108, 27, 16, DuelRank.SAPOW),
    Drop(113, 27, 16, DuelRank.SAPOW),
    Drop(114, 27, 16, DuelRank.SAPOW),
    Drop(116, 27, 16, DuelRank.SAPOW),
    Drop(129, 27, 16, DuelRank.SAPOW),
    Drop(134, 27, 16, DuelRank.SAPOW),
    Drop(139, 27, 16, DuelRank.SAPOW),
    Drop(143, 27, 16, DuelRank.SAPOW),
    Drop(144, 27, 16, DuelRank.SAPOW),
    Drop(145, 27, 16, DuelRank.SAPOW),
    Drop(147, 27, 16, DuelRank.SAPOW),
    Drop(148, 27, 16, DuelRank.SAPOW),
    Drop(155, 27, 16, DuelRank.SAPOW),
    Drop(156, 27, 16, DuelRank.SAPOW),
    Drop(158, 27, 16, DuelRank.SAPOW),
    Drop(160, 27, 16, DuelRank.SAPOW),
    Drop(161, 27, 16, DuelRank.SAPOW),
    Drop(173, 27, 16, DuelRank.SAPOW),
    Drop(175, 27, 16, DuelRank.SAPOW),
    Drop(176, 27, 16, DuelRank.SAPOW),
    Drop(177, 27, 16, DuelRank.SAPOW),
    Drop(178, 27, 16, DuelRank.SAPOW),
    Drop(180, 27, 16, DuelRank.SAPOW),
    Drop(183, 27, 16, DuelRank.SAPOW),
    Drop(184, 27, 16, DuelRank.SAPOW),
    Drop(187, 27, 16, DuelRank.SAPOW),
    Drop(188, 27, 16, DuelRank.SAPOW),
    Drop(189, 27, 16, DuelRank.SAPOW),
    Drop(190, 27, 16, DuelRank.SAPOW),
    Drop(195, 27, 16, DuelRank.SAPOW),
    Drop(199, 27, 16, DuelRank.SAPOW),
    Drop(203, 27, 16, DuelRank.SAPOW),
    Drop(205, 27, 16, DuelRank.SAPOW),
    Drop(206, 27, 16, DuelRank.SAPOW),
    Drop(208, 27, 16, DuelRank.SAPOW),
    Drop(209, 27, 16, DuelRank.SAPOW),
    Drop(215, 27, 16, DuelRank.SAPOW),
    Drop(218, 27, 52, DuelRank.SAPOW),
    Drop(222, 27, 16, DuelRank.SAPOW),
    Drop(224, 27, 16, DuelRank.SAPOW),
    Drop(226, 27, 16, DuelRank.SAPOW),
    Drop(232, 27, 16, DuelRank.SAPOW),
    Drop(239, 27, 16, DuelRank.SAPOW),
    Drop(240, 27, 16, DuelRank.SAPOW),
    Drop(243, 27, 16, DuelRank.SAPOW),
    Drop(247, 27, 16, DuelRank.SAPOW),
    Drop(253, 27, 16, DuelRank.SAPOW),
    Drop(264, 27, 16, DuelRank.SAPOW),
    Drop(265, 27, 16, DuelRank.SAPOW),
    Drop(266, 27, 16, DuelRank.SAPOW),
    Drop(267, 27, 16, DuelRank.SAPOW),
    Drop(271, 27, 16, DuelRank.SAPOW),
    Drop(277, 27, 16, DuelRank.SAPOW),
    Drop(282, 27, 16, DuelRank.SAPOW),
    Drop(283, 27, 16, DuelRank.SAPOW),
    Drop(292, 27, 16, DuelRank.SAPOW),
    Drop(298, 27, 16, DuelRank.SAPOW),
    Drop(331, 27, 32, DuelRank.SAPOW),
    Drop(332, 27, 32, DuelRank.SAPOW),
    Drop(382, 27, 2, DuelRank.SAPOW),
    Drop(417, 27, 16, DuelRank.SAPOW),
    Drop(421, 27, 16, DuelRank.SAPOW),
    Drop(432, 27, 16, DuelRank.SAPOW),
    Drop(451, 27, 16, DuelRank.SAPOW),
    Drop(463, 27, 16, DuelRank.SAPOW),
    Drop(476, 27, 16, DuelRank.SAPOW),
    Drop(477, 27, 16, DuelRank.SAPOW),
    Drop(478, 27, 16, DuelRank.SAPOW),
    Drop(479, 27, 16, DuelRank.SAPOW),
    Drop(481, 27, 16, DuelRank.SAPOW),
    Drop(490, 27, 16, DuelRank.SAPOW),
    Drop(503, 27, 16, DuelRank.SAPOW),
    Drop(510, 27, 16, DuelRank.SAPOW),
    Drop(534, 27, 16, DuelRank.SAPOW),
    Drop(536, 27, 16, DuelRank.SAPOW),
    Drop(538, 27, 16, DuelRank.SAPOW),
    Drop(544, 27, 16, DuelRank.SAPOW),
    Drop(556, 27, 16, DuelRank.SAPOW),
    Drop(568, 27, 32, DuelRank.SAPOW),
    Drop(569, 27, 16, DuelRank.SAPOW),
    Drop(573, 27, 52, DuelRank.SAPOW),
    Drop(585, 27, 16, DuelRank.SAPOW),
    Drop(590, 27, 16, DuelRank.SAPOW),
    Drop(604, 27, 16, DuelRank.SAPOW),
    Drop(605, 27, 16, DuelRank.SAPOW),
    Drop(606, 27, 16, DuelRank.SAPOW),
    Drop(3, 28, 12, DuelRank.SAPOW),
    Drop(4, 28, 12, DuelRank.SAPOW),
    Drop(5, 28, 26, DuelRank.SAPOW),
    Drop(6, 28, 12, DuelRank.SAPOW),
    Drop(20, 28, 6, DuelRank.SAPOW),
    Drop(23, 28, 12, DuelRank.SAPOW),
    Drop(25, 28, 12, DuelRank.SAPOW),
    Drop(26, 28, 12, DuelRank.SAPOW),
    Drop(29, 28, 12, DuelRank.SAPOW),
    Drop(33, 28, 12, DuelRank.SAPOW),
    Drop(34, 28, 25, DuelRank.SAPOW),
    Drop(40, 28, 12, DuelRank.SAPOW),
    Drop(42, 28, 12, DuelRank.SAPOW),
    Drop(44, 28, 12, DuelRank.SAPOW),
    Drop(47, 28, 12, DuelRank.SAPOW),
    Drop(48, 28, 25, DuelRank.SAPOW),
    Drop(59, 28, 12, DuelRank.SAPOW),
    Drop(61, 28, 12, DuelRank.SAPOW),
    Drop(65, 28, 12, DuelRank.SAPOW),
    Drop(77, 28, 12, DuelRank.SAPOW),
    Drop(83, 28, 12, DuelRank.SAPOW),
    Drop(84, 28, 12, DuelRank.SAPOW),
    Drop(85, 28, 12, DuelRank.SAPOW),
    Drop(86, 28, 12, DuelRank.SAPOW),
    Drop(87, 28, 12, DuelRank.SAPOW),
    Drop(88, 28, 12, DuelRank.SAPOW),
    Drop(90, 28, 12, DuelRank.SAPOW),
    Drop(91, 28, 12, DuelRank.SAPOW),
    Drop(95, 28, 12, DuelRank.SAPOW),
    Drop(97, 28, 12, DuelRank.SAPOW),
    Drop(100, 28, 12, DuelRank.SAPOW),
    Drop(103, 28, 12, DuelRank.SAPOW),
    Drop(104, 28, 12, DuelRank.SAPOW),
    Drop(106, 28, 12, DuelRank.SAPOW),
    Drop(108, 28, 12, DuelRank.SAPOW),
    Drop(113, 28, 12, DuelRank.SAPOW),
    Drop(114, 28, 25, DuelRank.SAPOW),
    Drop(115, 28, 12, DuelRank.SAPOW),
    Drop(116, 28, 12, DuelRank.SAPOW),
    Drop(118, 28, 12, DuelRank.SAPOW),
    Drop(119, 28, 12, DuelRank.SAPOW),
    Drop(129, 28, 12, DuelRank.SAPOW),
    Drop(133, 28, 12, DuelRank.SAPOW),
    Drop(136, 28, 12, DuelRank.SAPOW),
    Drop(138, 28, 12, DuelRank.SAPOW),
    Drop(139, 28, 12, DuelRank.SAPOW),
    Drop(140, 28, 12, DuelRank.SAPOW),
    Drop(141, 28, 12, DuelRank.SAPOW),
    Drop(142, 28, 25, DuelRank.SAPOW),
    Drop(143, 28, 25, DuelRank.SAPOW),
    Drop(144, 28, 25, DuelRank.SAPOW),
    Drop(145, 28, 12, DuelRank.SAPOW),
    Drop(148, 28, 12, DuelRank.SAPOW),
    Drop(149, 28, 12, DuelRank.SAPOW),
    Drop(155, 28, 12, DuelRank.SAPOW),
    Drop(162, 28, 12, DuelRank.SAPOW),
    Drop(164, 28, 12, DuelRank.SAPOW),
    Drop(169, 28, 25, DuelRank.SAPOW),
    Drop(171, 28, 12, DuelRank.SAPOW),
    Drop(172, 28, 12, DuelRank.SAPOW),
    Drop(173, 28, 25, DuelRank.SAPOW),
    Drop(174, 28, 12, DuelRank.SAPOW),
    Drop(175, 28, 25, DuelRank.SAPOW),
    Drop(177, 28, 12, DuelRank.SAPOW),
    Drop(178, 28, 25, DuelRank.SAPOW),
    Drop(179, 28, 12, DuelRank.SAPOW),
    Drop(180, 28, 12, DuelRank.SAPOW),
    Drop(181, 28, 25, DuelRank.SAPOW),
    Drop(183, 28, 12, DuelRank.SAPOW),
    Drop(184, 28, 12, DuelRank.SAPOW),
    Drop(188, 28, 12, DuelRank.SAPOW),
    Drop(189, 28, 12, DuelRank.SAPOW),
    Drop(190, 28, 25, DuelRank.SAPOW),
    Drop(194, 28, 12, DuelRank.SAPOW),
    Drop(199, 28, 12, DuelRank.SAPOW),
    Drop(204, 28, 12, DuelRank.SAPOW),
    Drop(205, 28, 12, DuelRank.SAPOW),
    Drop(206, 28, 12, DuelRank.SAPOW),
    Drop(213, 28, 12, DuelRank.SAPOW),
    Drop(215, 28, 12, DuelRank.SAPOW),
    Drop(218, 28, 12, DuelRank.SAPOW),
    Drop(220, 28, 12, DuelRank.SAPOW),
    Drop(222, 28, 12, DuelRank.SAPOW),
    Drop(224, 28, 12, DuelRank.SAPOW),
    Drop(226, 28, 12, DuelRank.SAPOW),
    Drop(228, 28, 12, DuelRank.SAPOW),
    Drop(231, 28, 12, DuelRank.SAPOW),
    Drop(232, 28, 12, DuelRank.SAPOW),
    Drop(233, 28, 12, DuelRank.SAPOW),
    Drop(239, 28, 12, DuelRank.SAPOW),
    Drop(240, 28, 12, DuelRank.SAPOW),
    Drop(242, 28, 12, DuelRank.SAPOW),
    Drop(243, 28, 12, DuelRank.SAPOW),
    Drop(244, 28, 12, DuelRank.SAPOW),
    Drop(245, 28, 12, DuelRank.SAPOW),
    Drop(247, 28, 12, DuelRank.SAPOW),
    Drop(250, 28, 12, DuelRank.SAPOW),
    Drop(253, 28, 24, DuelRank.SAPOW),
    Drop(254, 28, 12, DuelRank.SAPOW),
    Drop(257, 28, 12, DuelRank.SAPOW),
    Drop(259, 28, 12, DuelRank.SAPOW),
    Drop(260, 28, 12, DuelRank.SAPOW),
    Drop(261, 28, 12, DuelRank.SAPOW),
    Drop(264, 28, 12, DuelRank.SAPOW),
    Drop(266, 28, 12, DuelRank.SAPOW),
    Drop(268, 28, 12, DuelRank.SAPOW),
    Drop(269, 28, 24, DuelRank.SAPOW),
    Drop(270, 28, 12, DuelRank.SAPOW),
    Drop(271, 28, 12, DuelRank.SAPOW),
    Drop(276, 28, 12, DuelRank.SAPOW),
    Drop(277, 28, 24, DuelRank.SAPOW),
    Drop(279, 28, 24, DuelRank.SAPOW),
    Drop(282, 28, 12, DuelRank.SAPOW),
    Drop(283, 28, 12, DuelRank.SAPOW),
    Drop(295, 28, 24, DuelRank.SAPOW),
    Drop(296, 28, 12, DuelRank.SAPOW),
    Drop(298, 28, 12, DuelRank.SAPOW),
    Drop(331, 28, 30, DuelRank.SAPOW),
    Drop(378, 28, 6, DuelRank.SAPOW),
    Drop(379, 28, 24, DuelRank.SAPOW),
    Drop(391, 28, 12, DuelRank.SAPOW),
    Drop(401, 28, 12, DuelRank.SAPOW),
    Drop(417, 28, 12, DuelRank.SAPOW),
    Drop(432, 28, 12, DuelRank.SAPOW),
    Drop(446, 28, 12, DuelRank.SAPOW),
    Drop(461, 28, 12, DuelRank.SAPOW),
    Drop(463, 28, 12, DuelRank.SAPOW),
    Drop(471, 28, 12, DuelRank.SAPOW),
    Drop(472, 28, 12, DuelRank.SAPOW),
    Drop(477, 28, 12, DuelRank.SAPOW),
    Drop(478, 28, 12, DuelRank.SAPOW),
    Drop(479, 28, 12, DuelRank.SAPOW),
    Drop(481, 28, 12, DuelRank.SAPOW),
    Drop(486, 28, 12, DuelRank.SAPOW),
    Drop(490, 28, 12, DuelRank.SAPOW),
    Drop(500, 28, 12, DuelRank.SAPOW),
    Drop(503, 28, 12, DuelRank.SAPOW),
    Drop(526, 28, 12, DuelRank.SAPOW),
    Drop(534, 28, 12, DuelRank.SAPOW),
    Drop(538, 28, 12, DuelRank.SAPOW),
    Drop(568, 28, 12, DuelRank.SAPOW),
    Drop(569, 28, 12, DuelRank.SAPOW),
    Drop(573, 28, 12, DuelRank.SAPOW),
    Drop(585, 28, 12, DuelRank.SAPOW),
    Drop(590, 28, 12, DuelRank.SAPOW),
    Drop(605, 28, 12, DuelRank.SAPOW),
    Drop(611, 28, 12, DuelRank.SAPOW),
    Drop(612, 28, 24, DuelRank.SAPOW),
    Drop(650, 28, 12, DuelRank.SAPOW),
    Drop(2, 29, 44, DuelRank.SAPOW),
    Drop(6, 29, 44, DuelRank.SAPOW),
    Drop(9, 29, 150, DuelRank.SAPOW),
    Drop(10, 29, 44, DuelRank.SAPOW),
    Drop(19, 29, 10, DuelRank.SAPOW),
    Drop(25, 29, 44, DuelRank.SAPOW),
    Drop(27, 29, 44, DuelRank.SAPOW),
    Drop(30, 29, 44, DuelRank.SAPOW),
    Drop(31, 29, 46, DuelRank.SAPOW),
    Drop(35, 29, 46, DuelRank.SAPOW),
    Drop(38, 29, 46, DuelRank.SAPOW),
    Drop(39, 29, 46, DuelRank.SAPOW),
    Drop(41, 29, 46, DuelRank.SAPOW),
    Drop(46, 29, 46, DuelRank.SAPOW),
    Drop(47, 29, 46, DuelRank.SAPOW),
    Drop(48, 29, 46, DuelRank.SAPOW),
    Drop(59, 29, 46, DuelRank.SAPOW),
    Drop(65, 29, 46, DuelRank.SAPOW),
    Drop(74, 29, 46, DuelRank.SAPOW),
    Drop(89, 29, 46, DuelRank.SAPOW),
    Drop(94, 29, 20, DuelRank.SAPOW),
    Drop(111, 29, 20, DuelRank.SAPOW),
    Drop(130, 29, 90, DuelRank.SAPOW),
    Drop(238, 29, 20, DuelRank.SAPOW),
    Drop(333, 29, 60, DuelRank.SAPOW),
    Drop(381, 29, 2, DuelRank.SAPOW),
    Drop(396, 29, 20, DuelRank.SAPOW),
    Drop(408, 29, 20, DuelRank.SAPOW),
    Drop(409, 29, 20, DuelRank.SAPOW),
    Drop(422, 29, 88, DuelRank.SAPOW),
    Drop(436, 29, 88, DuelRank.SAPOW),
    Drop(437, 29, 20, DuelRank.SAPOW),
    Drop(439, 29, 20, DuelRank.SAPOW),
    Drop(458, 29, 20, DuelRank.SAPOW),
    Drop(473, 29, 20, DuelRank.SAPOW),
    Drop(482, 29, 20, DuelRank.SAPOW),
    Drop(485, 29, 88, DuelRank.SAPOW),
    Drop(487, 29, 20, DuelRank.SAPOW),
    Drop(509, 29, 20, DuelRank.SAPOW),
    Drop(516, 29, 88, DuelRank.SAPOW),
    Drop(521, 29, 20, DuelRank.SAPOW),
    Drop(533, 29, 20, DuelRank.SAPOW),
    Drop(542, 29, 20, DuelRank.SAPOW),
    Drop(557, 29, 20, DuelRank.SAPOW),
    Drop(563, 29, 88, DuelRank.SAPOW),
    Drop(618, 29, 20, DuelRank.SAPOW),
    Drop(637, 29, 20, DuelRank.SAPOW),
    Drop(659, 29, 20, DuelRank.SAPOW),
    Drop(707, 29, 20, DuelRank.SAPOW),
    Drop(712, 29, 20, DuelRank.SAPOW),
    Drop(713, 29, 20, DuelRank.SAPOW),
    Drop(714, 29, 20, DuelRank.SAPOW),
    Drop(3, 30, 60, DuelRank.SAPOW),
    Drop(4, 30, 30, DuelRank.SAPOW),
    Drop(5, 30, 60, DuelRank.SAPOW),
    Drop(10, 30, 30, DuelRank.SAPOW),
    Drop(12, 30, 36, DuelRank.SAPOW),
    Drop(15, 30, 36, DuelRank.SAPOW),
    Drop(19, 30, 5, DuelRank.SAPOW),
    Drop(23, 30, 64, DuelRank.SAPOW),
    Drop(26, 30, 64, DuelRank.SAPOW),
    Drop(31, 30, 32, DuelRank.SAPOW),
    Drop(33, 30, 72, DuelRank.SAPOW),
    Drop(34, 30, 64, DuelRank.SAPOW),
    Drop(39, 30, 72, DuelRank.SAPOW),
    Drop(42, 30, 36, DuelRank.SAPOW),
    Drop(68, 30, 36, DuelRank.SAPOW),
    Drop(73, 30, 32, DuelRank.SAPOW),
    Drop(74, 30, 32, DuelRank.SAPOW),
    Drop(77, 30, 63, DuelRank.SAPOW),
    Drop(84, 30, 32, DuelRank.SAPOW),
    Drop(85, 30, 40, DuelRank.SAPOW),
    Drop(88, 30, 32, DuelRank.SAPOW),
    Drop(90, 30, 72, DuelRank.SAPOW),
    Drop(91, 30, 60, DuelRank.SAPOW),
    Drop(93, 30, 32, DuelRank.SAPOW),
    Drop(122, 30, 30, DuelRank.SAPOW),
    Drop(126, 30, 32, DuelRank.SAPOW),
    Drop(168, 30, 30, DuelRank.SAPOW),
    Drop(200, 30, 30, DuelRank.SAPOW),
    Drop(213, 30, 32, DuelRank.SAPOW),
    Drop(286, 30, 32, DuelRank.SAPOW),
    Drop(296, 30, 30, DuelRank.SAPOW),
    Drop(298, 30, 30, DuelRank.SAPOW),
    Drop(333, 30, 30, DuelRank.SAPOW),
    Drop(376, 30, 32, DuelRank.SAPOW),
    Drop(379, 30, 60, DuelRank.SAPOW),
    Drop(400, 30, 32, DuelRank.SAPOW),
    Drop(403, 30, 32, DuelRank.SAPOW),
    Drop(413, 30, 32, DuelRank.SAPOW),
    Drop(419, 30, 36, DuelRank.SAPOW),
    Drop(423, 30, 32, DuelRank.SAPOW),
    Drop(438, 30, 36, DuelRank.SAPOW),
    Drop(441, 30, 32, DuelRank.SAPOW),
    Drop(464, 30, 32, DuelRank.SAPOW),
    Drop(468, 30, 32, DuelRank.SAPOW),
    Drop(491, 30, 32, DuelRank.SAPOW),
    Drop(508, 30, 32, DuelRank.SAPOW),
    Drop(513, 30, 32, DuelRank.SAPOW),
    Drop(528, 30, 32, DuelRank.SAPOW),
    Drop(571, 30, 36, DuelRank.SAPOW),
    Drop(577, 30, 32, DuelRank.SAPOW),
    Drop(582, 30, 32, DuelRank.SAPOW),
    Drop(638, 30, 32, DuelRank.SAPOW),
    Drop(650, 30, 32, DuelRank.SAPOW),
    Drop(10, 31, 15, DuelRank.SAPOW),
    Drop(19, 31, 6, DuelRank.SAPOW),
    Drop(23, 31, 15, DuelRank.SAPOW),
    Drop(25, 31, 15, DuelRank.SAPOW),
    Drop(30, 31, 30, DuelRank.SAPOW),
    Drop(34, 31, 15, DuelRank.SAPOW),
    Drop(36, 31, 15, DuelRank.SAPOW),
    Drop(40, 31, 15, DuelRank.SAPOW),
    Drop(45, 31, 14, DuelRank.SAPOW),
    Drop(53, 31, 15, DuelRank.SAPOW),
    Drop(59, 31, 15, DuelRank.SAPOW),
    Drop(61, 31, 15, DuelRank.SAPOW),
    Drop(65, 31, 15, DuelRank.SAPOW),
    Drop(70, 31, 15, DuelRank.SAPOW),
    Drop(71, 31, 15, DuelRank.SAPOW),
    Drop(74, 31, 16, DuelRank.SAPOW),
    Drop(80, 31, 14, DuelRank.SAPOW),
    Drop(89, 31, 16, DuelRank.SAPOW),
    Drop(96, 31, 14, DuelRank.SAPOW),
    Drop(97, 31, 14, DuelRank.SAPOW),
    Drop(98, 31, 14, DuelRank.SAPOW),
    Drop(99, 31, 18, DuelRank.SAPOW),
    Drop(103, 31, 15, DuelRank.SAPOW),
    Drop(108, 31, 15, DuelRank.SAPOW),
    Drop(109, 31, 15, DuelRank.SAPOW),
    Drop(110, 31, 15, DuelRank.SAPOW),
    Drop(115, 31, 15, DuelRank.SAPOW),
    Drop(118, 31, 15, DuelRank.SAPOW),
    Drop(119, 31, 15, DuelRank.SAPOW),
    Drop(120, 31, 15, DuelRank.SAPOW),
    Drop(121, 31, 15, DuelRank.SAPOW),
    Drop(131, 31, 16, DuelRank.SAPOW),
    Drop(132, 31, 30, DuelRank.SAPOW),
    Drop(135, 31, 14, DuelRank.SAPOW),
    Drop(138, 31, 15, DuelRank.SAPOW),
    Drop(139, 31, 15, DuelRank.SAPOW),
    Drop(140, 31, 30, DuelRank.SAPOW),
    Drop(141, 31, 14, DuelRank.SAPOW),
    Drop(150, 31, 14, DuelRank.SAPOW),
    Drop(152, 31, 14, DuelRank.SAPOW),
    Drop(154, 31, 14, DuelRank.SAPOW),
    Drop(162, 31, 14, DuelRank.SAPOW),
    Drop(164, 31, 14, DuelRank.SAPOW),
    Drop(169, 31, 14, DuelRank.SAPOW),
    Drop(171, 31, 14, DuelRank.SAPOW),
    Drop(172, 31, 14, DuelRank.SAPOW),
    Drop(177, 31, 14, DuelRank.SAPOW),
    Drop(181, 31, 14, DuelRank.SAPOW),
    Drop(191, 31, 14, DuelRank.SAPOW),
    Drop(193, 31, 14, DuelRank.SAPOW),
    Drop(196, 31, 28, DuelRank.SAPOW),
    Drop(199, 31, 14, DuelRank.SAPOW),
    Drop(201, 31, 14, DuelRank.SAPOW),
    Drop(203, 31, 14, DuelRank.SAPOW),
    Drop(205, 31, 14, DuelRank.SAPOW),
    Drop(206, 31, 14, DuelRank.SAPOW),
    Drop(211, 31, 14, DuelRank.SAPOW),
    Drop(215, 31, 14, DuelRank.SAPOW),
    Drop(219, 31, 14, DuelRank.SAPOW),
    Drop(220, 31, 14, DuelRank.SAPOW),
    Drop(221, 31, 14, DuelRank.SAPOW),
    Drop(223, 31, 16, DuelRank.SAPOW),
    Drop(225, 31, 14, DuelRank.SAPOW),
    Drop(227, 31, 13, DuelRank.SAPOW),
    Drop(228, 31, 13, DuelRank.SAPOW),
    Drop(231, 31, 14, DuelRank.SAPOW),
    Drop(233, 31, 14, DuelRank.SAPOW),
    Drop(236, 31, 14, DuelRank.SAPOW),
    Drop(237, 31, 14, DuelRank.SAPOW),
    Drop(243, 31, 14, DuelRank.SAPOW),
    Drop(244, 31, 14, DuelRank.SAPOW),
    Drop(246, 31, 14, DuelRank.SAPOW),
    Drop(248, 31, 14, DuelRank.SAPOW),
    Drop(250, 31, 14, DuelRank.SAPOW),
    Drop(251, 31, 14, DuelRank.SAPOW),
    Drop(256, 31, 14, DuelRank.SAPOW),
    Drop(257, 31, 28, DuelRank.SAPOW),
    Drop(258, 31, 28, DuelRank.SAPOW),
    Drop(259, 31, 14, DuelRank.SAPOW),
    Drop(262, 31, 14, DuelRank.SAPOW),
    Drop(263, 31, 28, DuelRank.SAPOW),
    Drop(265, 31, 14, DuelRank.SAPOW),
    Drop(269, 31, 14, DuelRank.SAPOW),
    Drop(270, 31, 13, DuelRank.SAPOW),
    Drop(272, 31, 13, DuelRank.SAPOW),
    Drop(273, 31, 13, DuelRank.SAPOW),
    Drop(274, 31, 13, DuelRank.SAPOW),
    Drop(276, 31, 13, DuelRank.SAPOW),
    Drop(280, 31, 13, DuelRank.SAPOW),
    Drop(290, 31, 13, DuelRank.SAPOW),
    Drop(291, 31, 13, DuelRank.SAPOW),
    Drop(292, 31, 13, DuelRank.SAPOW),
    Drop(293, 31, 13, DuelRank.SAPOW),
    Drop(294, 31, 13, DuelRank.SAPOW),
    Drop(296, 31, 13, DuelRank.SAPOW),
    Drop(335, 31, 30, DuelRank.SAPOW),
    Drop(366, 31, 28, DuelRank.SAPOW),
    Drop(368, 31, 28, DuelRank.SAPOW),
    Drop(370, 31, 1, DuelRank.SAPOW),
    Drop(375, 31, 1, DuelRank.SAPOW),
    Drop(376, 31, 14, DuelRank.SAPOW),
    Drop(406, 31, 13, DuelRank.SAPOW),
    Drop(414, 31, 13, DuelRank.SAPOW),
    Drop(425, 31, 14, DuelRank.SAPOW),
    Drop(426, 31, 16, DuelRank.SAPOW),
    Drop(432, 31, 13, DuelRank.SAPOW),
    Drop(446, 31, 13, DuelRank.SAPOW),
    Drop(451, 31, 13, DuelRank.SAPOW),
    Drop(452, 31, 13, DuelRank.SAPOW),
    Drop(453, 31, 16, DuelRank.SAPOW),
    Drop(459, 31, 13, DuelRank.SAPOW),
    Drop(460, 31, 13, DuelRank.SAPOW),
    Drop(461, 31, 13, DuelRank.SAPOW),
    Drop(462, 31, 13, DuelRank.SAPOW),
    Drop(463, 31, 13, DuelRank.SAPOW),
    Drop(496, 31, 13, DuelRank.SAPOW),
    Drop(503, 31, 13, DuelRank.SAPOW),
    Drop(514, 31, 13, DuelRank.SAPOW),
    Drop(524, 31, 13, DuelRank.SAPOW),
    Drop(531, 31, 16, DuelRank.SAPOW),
    Drop(549, 31, 13, DuelRank.SAPOW),
    Drop(552, 31, 13, DuelRank.SAPOW),
    Drop(556, 31, 13, DuelRank.SAPOW),
    Drop(564, 31, 16, DuelRank.SAPOW),
    Drop(567, 31, 13, DuelRank.SAPOW),
    Drop(576, 31, 13, DuelRank.SAPOW),
    Drop(584, 31, 13, DuelRank.SAPOW),
    Drop(591, 31, 13, DuelRank.SAPOW),
    Drop(601, 31, 13, DuelRank.SAPOW),
    Drop(602, 31, 13, DuelRank.SAPOW),
    Drop(605, 31, 13, DuelRank.SAPOW),
    Drop(606, 31, 13, DuelRank.SAPOW),
    Drop(608, 31, 13, DuelRank.SAPOW),
    Drop(613, 31, 16, DuelRank.SAPOW),
    Drop(620, 31, 13, DuelRank.SAPOW),
    Drop(629, 31, 13, DuelRank.SAPOW),
    Drop(642, 31, 14, DuelRank.SAPOW),
    Drop(643, 31, 14, DuelRank.SAPOW),
    Drop(2, 32, 32, DuelRank.SAPOW),
    Drop(6, 32, 32, DuelRank.SAPOW),
    Drop(10, 32, 32, DuelRank.SAPOW),
    Drop(12, 32, 16, DuelRank.SAPOW),
    Drop(15, 32, 16, DuelRank.SAPOW),
    Drop(21, 32, 8, DuelRank.SAPOW),
    Drop(25, 32, 32, DuelRank.SAPOW),
    Drop(27, 32, 32, DuelRank.SAPOW),
    Drop(30, 32, 32, DuelRank.SAPOW),
    Drop(31, 32, 32, DuelRank.SAPOW),
    Drop(33, 32, 16, DuelRank.SAPOW),
    Drop(35, 32, 48, DuelRank.SAPOW),
    Drop(38, 32, 60, DuelRank.SAPOW),
    Drop(39, 32, 32, DuelRank.SAPOW),
    Drop(41, 32, 50, DuelRank.SAPOW),
    Drop(43, 32, 16, DuelRank.SAPOW),
    Drop(46, 32, 32, DuelRank.SAPOW),
    Drop(47, 32, 32, DuelRank.SAPOW),
    Drop(48, 32, 32, DuelRank.SAPOW),
    Drop(59, 32, 32, DuelRank.SAPOW),
    Drop(65, 32, 32, DuelRank.SAPOW),
    Drop(74, 32, 32, DuelRank.SAPOW),
    Drop(78, 32, 16, DuelRank.SAPOW),
    Drop(79, 32, 16, DuelRank.SAPOW),
    Drop(89, 32, 32, DuelRank.SAPOW),
    Drop(90, 32, 16, DuelRank.SAPOW),
    Drop(93, 32, 16, DuelRank.SAPOW),
    Drop(99, 32, 16, DuelRank.SAPOW),
    Drop(100, 32, 16, DuelRank.SAPOW),
    Drop(107, 32, 16, DuelRank.SAPOW),
    Drop(110, 32, 16, DuelRank.SAPOW),
    Drop(118, 32, 16, DuelRank.SAPOW),
    Drop(120, 32, 16, DuelRank.SAPOW),
    Drop(127, 32, 16, DuelRank.SAPOW),
    Drop(138, 32, 16, DuelRank.SAPOW),
    Drop(147, 32, 16, DuelRank.SAPOW),
    Drop(156, 32, 16, DuelRank.SAPOW),
    Drop(160, 32, 16, DuelRank.SAPOW),
    Drop(161, 32, 16, DuelRank.SAPOW),
    Drop(166, 32, 16, DuelRank.SAPOW),
    Drop(172, 32, 16, DuelRank.SAPOW),
    Drop(182, 32, 16, DuelRank.SAPOW),
    Drop(185, 32, 16, DuelRank.SAPOW),
    Drop(195, 32, 16, DuelRank.SAPOW),
    Drop(214, 32, 16, DuelRank.SAPOW),
    Drop(223, 32, 16, DuelRank.SAPOW),
    Drop(224, 32, 16, DuelRank.SAPOW),
    Drop(225, 32, 16, DuelRank.SAPOW),
    Drop(226, 32, 16, DuelRank.SAPOW),
    Drop(231, 32, 16, DuelRank.SAPOW),
    Drop(236, 32, 16, DuelRank.SAPOW),
    Drop(239, 32, 16, DuelRank.SAPOW),
    Drop(250, 32, 16, DuelRank.SAPOW),
    Drop(256, 32, 16, DuelRank.SAPOW),
    Drop(262, 32, 16, DuelRank.SAPOW),
    Drop(266, 32, 16, DuelRank.SAPOW),
    Drop(267, 32, 16, DuelRank.SAPOW),
    Drop(280, 32, 16, DuelRank.SAPOW),
    Drop(290, 32, 16, DuelRank.SAPOW),
    Drop(293, 32, 16, DuelRank.SAPOW),
    Drop(294, 32, 16, DuelRank.SAPOW),
    Drop(376, 32, 16, DuelRank.SAPOW),
    Drop(391, 32, 18, DuelRank.SAPOW),
    Drop(401, 32, 18, DuelRank.SAPOW),
    Drop(407, 32, 18, DuelRank.SAPOW),
    Drop(426, 32, 18, DuelRank.SAPOW),
    Drop(427, 32, 2, DuelRank.SAPOW),
    Drop(445, 32, 44, DuelRank.SAPOW),
    Drop(447, 32, 26, DuelRank.SAPOW),
    Drop(453, 32, 18, DuelRank.SAPOW),
    Drop(465, 32, 16, DuelRank.SAPOW),
    Drop(467, 32, 18, DuelRank.SAPOW),
    Drop(471, 32, 18, DuelRank.SAPOW),
    Drop(474, 32, 44, DuelRank.SAPOW),
    Drop(493, 32, 26, DuelRank.SAPOW),
    Drop(495, 32, 12, DuelRank.SAPOW),
    Drop(500, 32, 18, DuelRank.SAPOW),
    Drop(507, 32, 26, DuelRank.SAPOW),
    Drop(515, 32, 26, DuelRank.SAPOW),
    Drop(522, 32, 16, DuelRank.SAPOW),
    Drop(531, 32, 16, DuelRank.SAPOW),
    Drop(546, 32, 44, DuelRank.SAPOW),
    Drop(559, 32, 44, DuelRank.SAPOW),
    Drop(561, 32, 44, DuelRank.SAPOW),
    Drop(564, 32, 18, DuelRank.SAPOW),
    Drop(565, 32, 12, DuelRank.SAPOW),
    Drop(571, 32, 16, DuelRank.SAPOW),
    Drop(572, 32, 32, DuelRank.SAPOW),
    Drop(583, 32, 26, DuelRank.SAPOW),
    Drop(594, 32, 16, DuelRank.SAPOW),
    Drop(613, 32, 12, DuelRank.SAPOW),
    Drop(622, 32, 12, DuelRank.SAPOW),
    Drop(624, 32, 12, DuelRank.SAPOW),
    Drop(627, 32, 16, DuelRank.SAPOW),
    Drop(645, 32, 12, DuelRank.SAPOW),
    Drop(2, 33, 26, DuelRank.SAPOW),
    Drop(14, 33, 24, DuelRank.SAPOW),
    Drop(19, 33, 8, DuelRank.SAPOW),
    Drop(30, 33, 24, DuelRank.SAPOW),
    Drop(32, 33, 24, DuelRank.SAPOW),
    Drop(36, 33, 74, DuelRank.SAPOW),
    Drop(78, 33, 24, DuelRank.SAPOW),
    Drop(81, 33, 24, DuelRank.SAPOW),
    Drop(83, 33, 24, DuelRank.SAPOW),
    Drop(86, 33, 24, DuelRank.SAPOW),
    Drop(87, 33, 24, DuelRank.SAPOW),
    Drop(89, 33, 24, DuelRank.SAPOW),
    Drop(91, 33, 24, DuelRank.SAPOW),
    Drop(93, 33, 74, DuelRank.SAPOW),
    Drop(95, 33, 50, DuelRank.SAPOW),
    Drop(97, 33, 74, DuelRank.SAPOW),
    Drop(99, 33, 74, DuelRank.SAPOW),
    Drop(106, 33, 24, DuelRank.SAPOW),
    Drop(108, 33, 24, DuelRank.SAPOW),
    Drop(124, 33, 50, DuelRank.SAPOW),
    Drop(127, 33, 24, DuelRank.SAPOW),
    Drop(131, 33, 24, DuelRank.SAPOW),
    Drop(132, 33, 24, DuelRank.SAPOW),
    Drop(139, 33, 24, DuelRank.SAPOW),
    Drop(150, 33, 24, DuelRank.SAPOW),
    Drop(170, 33, 24, DuelRank.SAPOW),
    Drop(204, 33, 24, DuelRank.SAPOW),
    Drop(213, 33, 24, DuelRank.SAPOW),
    Drop(215, 33, 24, DuelRank.SAPOW),
    Drop(228, 33, 24, DuelRank.SAPOW),
    Drop(283, 33, 24, DuelRank.SAPOW),
    Drop(286, 33, 24, DuelRank.SAPOW),
    Drop(366, 33, 24, DuelRank.SAPOW),
    Drop(368, 33, 50, DuelRank.SAPOW),
    Drop(379, 33, 24, DuelRank.SAPOW),
    Drop(388, 33, 74, DuelRank.SAPOW),
    Drop(391, 33, 52, DuelRank.SAPOW),
    Drop(392, 33, 4, DuelRank.SAPOW),
    Drop(405, 33, 50, DuelRank.SAPOW),
    Drop(406, 33, 24, DuelRank.SAPOW),
    Drop(407, 33, 26, DuelRank.SAPOW),
    Drop(408, 33, 8, DuelRank.SAPOW),
    Drop(409, 33, 8, DuelRank.SAPOW),
    Drop(412, 33, 24, DuelRank.SAPOW),
    Drop(413, 33, 50, DuelRank.SAPOW),
    Drop(414, 33, 24, DuelRank.SAPOW),
    Drop(415, 33, 24, DuelRank.SAPOW),
    Drop(416, 33, 24, DuelRank.SAPOW),
    Drop(417, 33, 24, DuelRank.SAPOW),
    Drop(419, 33, 24, DuelRank.SAPOW),
    Drop(423, 33, 24, DuelRank.SAPOW),
    Drop(425, 33, 24, DuelRank.SAPOW),
    Drop(438, 33, 24, DuelRank.SAPOW),
    Drop(441, 33, 24, DuelRank.SAPOW),
    Drop(460, 33, 24, DuelRank.SAPOW),
    Drop(462, 33, 24, DuelRank.SAPOW),
    Drop(468, 33, 24, DuelRank.SAPOW),
    Drop(491, 33, 24, DuelRank.SAPOW),
    Drop(508, 33, 50, DuelRank.SAPOW),
    Drop(509, 33, 10, DuelRank.SAPOW),
    Drop(512, 33, 24, DuelRank.SAPOW),
    Drop(513, 33, 24, DuelRank.SAPOW),
    Drop(514, 33, 24, DuelRank.SAPOW),
    Drop(557, 33, 8, DuelRank.SAPOW),
    Drop(564, 33, 26, DuelRank.SAPOW),
    Drop(585, 33, 24, DuelRank.SAPOW),
    Drop(633, 33, 24, DuelRank.SAPOW),
    Drop(636, 33, 24, DuelRank.SAPOW),
    Drop(643, 33, 24, DuelRank.SAPOW),
    Drop(648, 33, 50, DuelRank.SAPOW),
    Drop(6, 34, 44, DuelRank.SAPOW),
    Drop(13, 34, 22, DuelRank.SAPOW),
    Drop(20, 34, 8, DuelRank.SAPOW),
    Drop(25, 34, 21, DuelRank.SAPOW),
    Drop(26, 34, 21, DuelRank.SAPOW),
    Drop(27, 34, 21, DuelRank.SAPOW),
    Drop(31, 34, 21, DuelRank.SAPOW),
    Drop(32, 34, 21, DuelRank.SAPOW),
    Drop(36, 34, 21, DuelRank.SAPOW),
    Drop(41, 34, 21, DuelRank.SAPOW),
    Drop(43, 34, 21, DuelRank.SAPOW),
    Drop(44, 34, 21, DuelRank.SAPOW),
    Drop(46, 34, 21, DuelRank.SAPOW),
    Drop(48, 34, 21, DuelRank.SAPOW),
    Drop(49, 34, 21, DuelRank.SAPOW),
    Drop(54, 34, 21, DuelRank.SAPOW),
    Drop(55, 34, 21, DuelRank.SAPOW),
    Drop(70, 34, 21, DuelRank.SAPOW),
    Drop(71, 34, 21, DuelRank.SAPOW),
    Drop(77, 34, 21, DuelRank.SAPOW),
    Drop(81, 34, 21, DuelRank.SAPOW),
    Drop(83, 34, 67, DuelRank.SAPOW),
    Drop(84, 34, 67, DuelRank.SAPOW),
    Drop(85, 34, 67, DuelRank.SAPOW),
    Drop(86, 34, 67, DuelRank.SAPOW),
    Drop(87, 34, 67, DuelRank.SAPOW),
    Drop(88, 34, 67, DuelRank.SAPOW),
    Drop(95, 34, 43, DuelRank.SAPOW),
    Drop(103, 34, 21, DuelRank.SAPOW),
    Drop(115, 34, 21, DuelRank.SAPOW),
    Drop(119, 34, 21, DuelRank.SAPOW),
    Drop(121, 34, 21, DuelRank.SAPOW),
    Drop(125, 34, 21, DuelRank.SAPOW),
    Drop(136, 34, 42, DuelRank.SAPOW),
    Drop(149, 34, 42, DuelRank.SAPOW),
    Drop(162, 34, 20, DuelRank.SAPOW),
    Drop(164, 34, 20, DuelRank.SAPOW),
    Drop(166, 34, 20, DuelRank.SAPOW),
    Drop(168, 34, 20, DuelRank.SAPOW),
    Drop(169, 34, 20, DuelRank.SAPOW),
    Drop(170, 34, 20, DuelRank.SAPOW),
    Drop(171, 34, 20, DuelRank.SAPOW),
    Drop(173, 34, 20, DuelRank.SAPOW),
    Drop(175, 34, 20, DuelRank.SAPOW),
    Drop(178, 34, 20, DuelRank.SAPOW),
    Drop(181, 34, 20, DuelRank.SAPOW),
    Drop(186, 34, 20, DuelRank.SAPOW),
    Drop(193, 34, 20, DuelRank.SAPOW),
    Drop(194, 34, 42, DuelRank.SAPOW),
    Drop(204, 34, 20, DuelRank.SAPOW),
    Drop(230, 34, 20, DuelRank.SAPOW),
    Drop(233, 34, 42, DuelRank.SAPOW),
    Drop(255, 34, 20, DuelRank.SAPOW),
    Drop(269, 34, 20, DuelRank.SAPOW),
    Drop(277, 34, 20, DuelRank.SAPOW),
    Drop(279, 34, 20, DuelRank.SAPOW),
    Drop(295, 34, 20, DuelRank.SAPOW),
    Drop(379, 34, 42, DuelRank.SAPOW),
    Drop(390, 34, 4, DuelRank.SAPOW),
    Drop(391, 34, 36, DuelRank.SAPOW),
    Drop(401, 34, 36, DuelRank.SAPOW),
    Drop(412, 34, 20, DuelRank.SAPOW),
    Drop(415, 34, 20, DuelRank.SAPOW),
    Drop(416, 34, 20, DuelRank.SAPOW),
    Drop(459, 34, 20, DuelRank.SAPOW),
    Drop(462, 34, 20, DuelRank.SAPOW),
    Drop(466, 34, 20, DuelRank.SAPOW),
    Drop(471, 34, 36, DuelRank.SAPOW),
    Drop(472, 34, 24, DuelRank.SAPOW),
    Drop(500, 34, 36, DuelRank.SAPOW),
    Drop(511, 34, 20, DuelRank.SAPOW),
    Drop(512, 34, 20, DuelRank.SAPOW),
    Drop(526, 34, 24, DuelRank.SAPOW),
    Drop(597, 34, 20, DuelRank.SAPOW),
    Drop(608, 34, 20, DuelRank.SAPOW),
    Drop(636, 34, 20, DuelRank.SAPOW),
    Drop(650, 34, 20, DuelRank.SAPOW),
    Drop(2, 35, 22, DuelRank.SAPOW),
    Drop(19, 35, 6, DuelRank.SAPOW),
    Drop(20, 35, 4, DuelRank.SAPOW),
    Drop(21, 35, 4, DuelRank.SAPOW),
    Drop(30, 35, 22, DuelRank.SAPOW),
    Drop(34, 35, 22, DuelRank.SAPOW),
    Drop(35, 35, 4, DuelRank.SAPOW),
    Drop(36, 35, 22, DuelRank.SAPOW),
    Drop(38, 35, 28, DuelRank.SAPOW),
    Drop(42, 35, 22, DuelRank.SAPOW),
    Drop(44, 35, 22, DuelRank.SAPOW),
    Drop(74, 35, 22, DuelRank.SAPOW),
    Drop(79, 35, 24, DuelRank.SAPOW),
    Drop(90, 35, 24, DuelRank.SAPOW),
    Drop(96, 35, 22, DuelRank.SAPOW),
    Drop(97, 35, 22, DuelRank.SAPOW),
    Drop(98, 35, 22, DuelRank.SAPOW),
    Drop(99, 35, 48, DuelRank.SAPOW),
    Drop(104, 35, 22, DuelRank.SAPOW),
    Drop(106, 35, 22, DuelRank.SAPOW),
    Drop(108, 35, 22, DuelRank.SAPOW),
    Drop(114, 35, 22, DuelRank.SAPOW),
    Drop(115, 35, 22, DuelRank.SAPOW),
    Drop(129, 35, 22, DuelRank.SAPOW),
    Drop(132, 35, 22, DuelRank.SAPOW),
    Drop(135, 35, 22, DuelRank.SAPOW),
    Drop(139, 35, 22, DuelRank.SAPOW),
    Drop(142, 35, 22, DuelRank.SAPOW),
    Drop(143, 35, 22, DuelRank.SAPOW),
    Drop(144, 35, 22, DuelRank.SAPOW),
    Drop(145, 35, 22, DuelRank.SAPOW),
    Drop(154, 35, 22, DuelRank.SAPOW),
    Drop(174, 35, 20, DuelRank.SAPOW),
    Drop(179, 35, 20, DuelRank.SAPOW),
    Drop(183, 35, 20, DuelRank.SAPOW),
    Drop(184, 35, 20, DuelRank.SAPOW),
    Drop(190, 35, 20, DuelRank.SAPOW),
    Drop(203, 35, 20, DuelRank.SAPOW),
    Drop(213, 35, 20, DuelRank.SAPOW),
    Drop(215, 35, 20, DuelRank.SAPOW),
    Drop(220, 35, 20, DuelRank.SAPOW),
    Drop(223, 35, 24, DuelRank.SAPOW),
    Drop(228, 35, 22, DuelRank.SAPOW),
    Drop(237, 35, 22, DuelRank.SAPOW),
    Drop(244, 35, 22, DuelRank.SAPOW),
    Drop(253, 35, 22, DuelRank.SAPOW),
    Drop(257, 35, 22, DuelRank.SAPOW),
    Drop(259, 35, 22, DuelRank.SAPOW),
    Drop(263, 35, 22, DuelRank.SAPOW),
    Drop(268, 35, 22, DuelRank.SAPOW),
    Drop(366, 35, 22, DuelRank.SAPOW),
    Drop(368, 35, 22, DuelRank.SAPOW),
    Drop(370, 35, 4, DuelRank.SAPOW),
    Drop(371, 35, 4, DuelRank.SAPOW),
    Drop(372, 35, 4, DuelRank.SAPOW),
    Drop(373, 35, 4, DuelRank.SAPOW),
    Drop(385, 35, 4, DuelRank.SAPOW),
    Drop(386, 35, 4, DuelRank.SAPOW),
    Drop(390, 35, 4, DuelRank.SAPOW),
    Drop(391, 35, 20, DuelRank.SAPOW),
    Drop(401, 35, 20, DuelRank.SAPOW),
    Drop(407, 35, 20, DuelRank.SAPOW),
    Drop(426, 35, 48, DuelRank.SAPOW),
    Drop(427, 35, 4, DuelRank.SAPOW),
    Drop(442, 35, 4, DuelRank.SAPOW),
    Drop(453, 35, 46, DuelRank.SAPOW),
    Drop(455, 35, 48, DuelRank.SAPOW),
    Drop(457, 35, 48, DuelRank.SAPOW),
    Drop(465, 35, 26, DuelRank.SAPOW),
    Drop(467, 35, 20, DuelRank.SAPOW),
    Drop(470, 35, 24, DuelRank.SAPOW),
    Drop(471, 35, 20, DuelRank.SAPOW),
    Drop(500, 35, 20, DuelRank.SAPOW),
    Drop(505, 35, 46, DuelRank.SAPOW),
    Drop(509, 35, 4, DuelRank.SAPOW),
    Drop(518, 35, 12, DuelRank.SAPOW),
    Drop(522, 35, 26, DuelRank.SAPOW),
    Drop(525, 35, 12, DuelRank.SAPOW),
    Drop(529, 35, 12, DuelRank.SAPOW),
    Drop(531, 35, 46, DuelRank.SAPOW),
    Drop(532, 35, 24, DuelRank.SAPOW),
    Drop(564, 35, 48, DuelRank.SAPOW),
    Drop(571, 35, 26, DuelRank.SAPOW),
    Drop(572, 35, 26, DuelRank.SAPOW),
    Drop(593, 35, 12, DuelRank.SAPOW),
    Drop(594, 35, 26, DuelRank.SAPOW),
    Drop(599, 35, 46, DuelRank.SAPOW),
    Drop(613, 35, 20, DuelRank.SAPOW),
    Drop(623, 35, 24, DuelRank.SAPOW),
    Drop(627, 35, 26, DuelRank.SAPOW),
    Drop(631, 35, 24, DuelRank.SAPOW),
    Drop(632, 35, 12, DuelRank.SAPOW),
    Drop(634, 35, 24, DuelRank.SAPOW),
    Drop(645, 35, 4, DuelRank.SAPOW),
    Drop(646, 35, 46, DuelRank.SAPOW),
    Drop(707, 35, 4, DuelRank.SAPOW),
    Drop(713, 35, 4, DuelRank.SAPOW),
    Drop(714, 35, 4, DuelRank.SAPOW),
    Drop(1, 36, 26, DuelRank.SAPOW),
    Drop(6, 36, 24, DuelRank.SAPOW),
    Drop(20, 36, 6, DuelRank.SAPOW),
    Drop(25, 36, 24, DuelRank.SAPOW),
    Drop(38, 36, 26, DuelRank.SAPOW),
    Drop(48, 36, 22, DuelRank.SAPOW),
    Drop(79, 36, 24, DuelRank.SAPOW),
    Drop(83, 36, 22, DuelRank.SAPOW),
    Drop(84, 36, 22, DuelRank.SAPOW),
    Drop(85, 36, 22, DuelRank.SAPOW),
    Drop(86, 36, 22, DuelRank.SAPOW),
    Drop(87, 36, 22, DuelRank.SAPOW),
    Drop(88, 36, 22, DuelRank.SAPOW),
    Drop(90, 36, 24, DuelRank.SAPOW),
    Drop(95, 36, 22, DuelRank.SAPOW),
    Drop(99, 36, 24, DuelRank.SAPOW),
    Drop(103, 36, 22, DuelRank.SAPOW),
    Drop(119, 36, 22, DuelRank.SAPOW),
    Drop(136, 36, 22, DuelRank.SAPOW),
    Drop(137, 36, 22, DuelRank.SAPOW),
    Drop(148, 36, 22, DuelRank.SAPOW),
    Drop(149, 36, 22, DuelRank.SAPOW),
    Drop(162, 36, 22, DuelRank.SAPOW),
    Drop(164, 36, 22, DuelRank.SAPOW),
    Drop(169, 36, 22, DuelRank.SAPOW),
    Drop(171, 36, 22, DuelRank.SAPOW),
    Drop(173, 36, 22, DuelRank.SAPOW),
    Drop(175, 36, 22, DuelRank.SAPOW),
    Drop(178, 36, 22, DuelRank.SAPOW),
    Drop(181, 36, 22, DuelRank.SAPOW),
    Drop(194, 36, 22, DuelRank.SAPOW),
    Drop(204, 36, 22, DuelRank.SAPOW),
    Drop(222, 36, 22, DuelRank.SAPOW),
    Drop(223, 36, 24, DuelRank.SAPOW),
    Drop(232, 36, 22, DuelRank.SAPOW),
    Drop(233, 36, 22, DuelRank.SAPOW),
    Drop(240, 36, 22, DuelRank.SAPOW),
    Drop(242, 36, 22, DuelRank.SAPOW),
    Drop(245, 36, 22, DuelRank.SAPOW),
    Drop(254, 36, 22, DuelRank.SAPOW),
    Drop(261, 36, 22, DuelRank.SAPOW),
    Drop(269, 36, 22, DuelRank.SAPOW),
    Drop(271, 36, 22, DuelRank.SAPOW),
    Drop(277, 36, 22, DuelRank.SAPOW),
    Drop(279, 36, 22, DuelRank.SAPOW),
    Drop(295, 36, 22, DuelRank.SAPOW),
    Drop(371, 36, 2, DuelRank.SAPOW),
    Drop(379, 36, 22, DuelRank.SAPOW),
    Drop(391, 36, 48, DuelRank.SAPOW),
    Drop(401, 36, 48, DuelRank.SAPOW),
    Drop(407, 36, 26, DuelRank.SAPOW),
    Drop(424, 36, 16, DuelRank.SAPOW),
    Drop(426, 36, 26, DuelRank.SAPOW),
    Drop(449, 36, 10, DuelRank.SAPOW),
    Drop(453, 36, 26, DuelRank.SAPOW),
    Drop(465, 36, 24, DuelRank.SAPOW),
    Drop(467, 36, 26, DuelRank.SAPOW),
    Drop(471, 36, 48, DuelRank.SAPOW),
    Drop(472, 36, 24, DuelRank.SAPOW),
    Drop(497, 36, 26, DuelRank.SAPOW),
    Drop(500, 36, 48, DuelRank.SAPOW),
    Drop(522, 36, 24, DuelRank.SAPOW),
    Drop(523, 36, 26, DuelRank.SAPOW),
    Drop(526, 36, 24, DuelRank.SAPOW),
    Drop(531, 36, 24, DuelRank.SAPOW),
    Drop(535, 36, 26, DuelRank.SAPOW),
    Drop(539, 36, 44, DuelRank.SAPOW),
    Drop(551, 36, 26, DuelRank.SAPOW),
    Drop(553, 36, 44, DuelRank.SAPOW),
    Drop(560, 36, 44, DuelRank.SAPOW),
    Drop(564, 36, 26, DuelRank.SAPOW),
    Drop(571, 36, 24, DuelRank.SAPOW),
    Drop(572, 36, 24, DuelRank.SAPOW),
    Drop(574, 36, 44, DuelRank.SAPOW),
    Drop(594, 36, 24, DuelRank.SAPOW),
    Drop(600, 36, 44, DuelRank.SAPOW),
    Drop(613, 36, 26, DuelRank.SAPOW),
    Drop(619, 36, 16, DuelRank.SAPOW),
    Drop(627, 36, 24, DuelRank.SAPOW),
    Drop(639, 36, 14, DuelRank.SAPOW),
    Drop(641, 36, 14, DuelRank.SAPOW),
    Drop(647, 36, 26, DuelRank.SAPOW),
    Drop(650, 36, 24, DuelRank.SAPOW),
    Drop(3, 37, 22, DuelRank.SAPOW),
    Drop(4, 37, 22, DuelRank.SAPOW),
    Drop(10, 37, 22, DuelRank.SAPOW),
    Drop(11, 37, 26, DuelRank.SAPOW),
    Drop(13, 37, 22, DuelRank.SAPOW),
    Drop(14, 37, 22, DuelRank.SAPOW),
    Drop(15, 37, 22, DuelRank.SAPOW),
    Drop(19, 37, 8, DuelRank.SAPOW),
    Drop(23, 37, 22, DuelRank.SAPOW),
    Drop(26, 37, 24, DuelRank.SAPOW),
    Drop(27, 37, 24, DuelRank.SAPOW),
    Drop(29, 37, 24, DuelRank.SAPOW),
    Drop(31, 37, 24, DuelRank.SAPOW),
    Drop(33, 37, 26, DuelRank.SAPOW),
    Drop(39, 37, 58, DuelRank.SAPOW),
    Drop(40, 37, 56, DuelRank.SAPOW),
    Drop(42, 37, 70, DuelRank.SAPOW),
    Drop(44, 37, 46, DuelRank.SAPOW),
    Drop(46, 37, 22, DuelRank.SAPOW),
    Drop(47, 37, 22, DuelRank.SAPOW),
    Drop(61, 37, 22, DuelRank.SAPOW),
    Drop(65, 37, 22, DuelRank.SAPOW),
    Drop(68, 37, 22, DuelRank.SAPOW),
    Drop(79, 37, 24, DuelRank.SAPOW),
    Drop(85, 37, 24, DuelRank.SAPOW),
    Drop(90, 37, 24, DuelRank.SAPOW),
    Drop(91, 37, 22, DuelRank.SAPOW),
    Drop(99, 37, 24, DuelRank.SAPOW),
    Drop(113, 37, 22, DuelRank.SAPOW),
    Drop(121, 37, 22, DuelRank.SAPOW),
    Drop(125, 37, 22, DuelRank.SAPOW),
    Drop(155, 37, 22, DuelRank.SAPOW),
    Drop(168, 37, 22, DuelRank.SAPOW),
    Drop(186, 37, 22, DuelRank.SAPOW),
    Drop(188, 37, 22, DuelRank.SAPOW),
    Drop(189, 37, 22, DuelRank.SAPOW),
    Drop(200, 37, 22, DuelRank.SAPOW),
    Drop(201, 37, 22, DuelRank.SAPOW),
    Drop(219, 37, 22, DuelRank.SAPOW),
    Drop(246, 37, 22, DuelRank.SAPOW),
    Drop(248, 37, 22, DuelRank.SAPOW),
    Drop(255, 37, 22, DuelRank.SAPOW),
    Drop(272, 37, 22, DuelRank.SAPOW),
    Drop(282, 37, 22, DuelRank.SAPOW),
    Drop(296, 37, 22, DuelRank.SAPOW),
    Drop(298, 37, 22, DuelRank.SAPOW),
    Drop(376, 37, 22, DuelRank.SAPOW),
    Drop(385, 37, 4, DuelRank.SAPOW),
    Drop(388, 37, 24, DuelRank.SAPOW),
    Drop(403, 37, 22, DuelRank.SAPOW),
    Drop(419, 37, 24, DuelRank.SAPOW),
    Drop(423, 37, 22, DuelRank.SAPOW),
    Drop(438, 37, 24, DuelRank.SAPOW),
    Drop(443, 37, 24, DuelRank.SAPOW),
    Drop(464, 37, 22, DuelRank.SAPOW),
    Drop(465, 37, 24, DuelRank.SAPOW),
    Drop(466, 37, 22, DuelRank.SAPOW),
    Drop(467, 37, 26, DuelRank.SAPOW),
    Drop(468, 37, 22, DuelRank.SAPOW),
    Drop(472, 37, 24, DuelRank.SAPOW),
    Drop(481, 37, 22, DuelRank.SAPOW),
    Drop(483, 37, 48, DuelRank.SAPOW),
    Drop(491, 37, 22, DuelRank.SAPOW),
    Drop(496, 37, 22, DuelRank.SAPOW),
    Drop(522, 37, 24, DuelRank.SAPOW),
    Drop(526, 37, 24, DuelRank.SAPOW),
    Drop(528, 37, 46, DuelRank.SAPOW),
    Drop(531, 37, 24, DuelRank.SAPOW),
    Drop(538, 37, 22, DuelRank.SAPOW),
    Drop(552, 37, 22, DuelRank.SAPOW),
    Drop(571, 37, 24, DuelRank.SAPOW),
    Drop(572, 37, 24, DuelRank.SAPOW),
    Drop(575, 37, 48, DuelRank.SAPOW),
    Drop(577, 37, 46, DuelRank.SAPOW),
    Drop(582, 37, 24, DuelRank.SAPOW),
    Drop(594, 37, 24, DuelRank.SAPOW),
    Drop(597, 37, 22, DuelRank.SAPOW),
    Drop(627, 37, 24, DuelRank.SAPOW),
    Drop(629, 37, 22, DuelRank.SAPOW),
    Drop(636, 37, 22, DuelRank.SAPOW),
    Drop(642, 37, 22, DuelRank.SAPOW),
    Drop(4, 38, 30, DuelRank.SAPOW),
    Drop(10, 38, 30, DuelRank.SAPOW),
    Drop(20, 38, 6, DuelRank.SAPOW),
    Drop(31, 38, 30, DuelRank.SAPOW),
    Drop(38, 38, 34, DuelRank.SAPOW),
    Drop(39, 38, 34, DuelRank.SAPOW),
    Drop(70, 38, 30, DuelRank.SAPOW),
    Drop(71, 38, 32, DuelRank.SAPOW),
    Drop(79, 38, 34, DuelRank.SAPOW),
    Drop(89, 38, 32, DuelRank.SAPOW),
    Drop(90, 38, 34, DuelRank.SAPOW),
    Drop(99, 38, 34, DuelRank.SAPOW),
    Drop(131, 38, 32, DuelRank.SAPOW),
    Drop(140, 38, 32, DuelRank.SAPOW),
    Drop(150, 38, 32, DuelRank.SAPOW),
    Drop(168, 38, 32, DuelRank.SAPOW),
    Drop(177, 38, 32, DuelRank.SAPOW),
    Drop(193, 38, 32, DuelRank.SAPOW),
    Drop(196, 38, 32, DuelRank.SAPOW),
    Drop(199, 38, 32, DuelRank.SAPOW),
    Drop(200, 38, 32, DuelRank.SAPOW),
    Drop(205, 38, 30, DuelRank.SAPOW),
    Drop(206, 38, 30, DuelRank.SAPOW),
    Drop(223, 38, 66, DuelRank.SAPOW),
    Drop(230, 38, 30, DuelRank.SAPOW),
    Drop(243, 38, 30, DuelRank.SAPOW),
    Drop(247, 38, 30, DuelRank.SAPOW),
    Drop(251, 38, 30, DuelRank.SAPOW),
    Drop(258, 38, 30, DuelRank.SAPOW),
    Drop(270, 38, 30, DuelRank.SAPOW),
    Drop(296, 38, 30, DuelRank.SAPOW),
    Drop(298, 38, 30, DuelRank.SAPOW),
    Drop(373, 38, 4, DuelRank.SAPOW),
    Drop(391, 38, 32, DuelRank.SAPOW),
    Drop(401, 38, 32, DuelRank.SAPOW),
    Drop(407, 38, 32, DuelRank.SAPOW),
    Drop(426, 38, 32, DuelRank.SAPOW),
    Drop(430, 38, 48, DuelRank.SAPOW),
    Drop(431, 38, 48, DuelRank.SAPOW),
    Drop(434, 38, 12, DuelRank.SAPOW),
    Drop(440, 38, 30, DuelRank.SAPOW),
    Drop(453, 38, 32, DuelRank.SAPOW),
    Drop(465, 38, 32, DuelRank.SAPOW),
    Drop(467, 38, 32, DuelRank.SAPOW),
    Drop(471, 38, 32, DuelRank.SAPOW),
    Drop(494, 38, 12, DuelRank.SAPOW),
    Drop(500, 38, 32, DuelRank.SAPOW),
    Drop(519, 38, 12, DuelRank.SAPOW),
    Drop(520, 38, 14, DuelRank.SAPOW),
    Drop(522, 38, 32, DuelRank.SAPOW),
    Drop(530, 38, 48, DuelRank.SAPOW),
    Drop(531, 38, 32, DuelRank.SAPOW),
    Drop(550, 38, 48, DuelRank.SAPOW),
    Drop(564, 38, 32, DuelRank.SAPOW),
    Drop(571, 38, 66, DuelRank.SAPOW),
    Drop(572, 38, 32, DuelRank.SAPOW),
    Drop(594, 38, 32, DuelRank.SAPOW),
    Drop(596, 38, 30, DuelRank.SAPOW),
    Drop(613, 38, 32, DuelRank.SAPOW),
    Drop(615, 38, 48, DuelRank.SAPOW),
    Drop(617, 38, 12, DuelRank.SAPOW),
    Drop(625, 38, 30, DuelRank.SAPOW),
    Drop(626, 38, 30, DuelRank.SAPOW),
    Drop(627, 38, 32, DuelRank.SAPOW),
    Drop(649, 38, 30, DuelRank.SAPOW),
    Drop(4, 39, 14, DuelRank.SAPOW),
    Drop(9, 39, 14, DuelRank.SAPOW),
    Drop(20, 39, 6, DuelRank.SAPOW),
    Drop(23, 39, 14, DuelRank.SAPOW),
    Drop(24, 39, 14, DuelRank.SAPOW),
    Drop(30, 39, 28, DuelRank.SAPOW),
    Drop(34, 39, 14, DuelRank.SAPOW),
    Drop(40, 39, 14, DuelRank.SAPOW),
    Drop(45, 39, 14, DuelRank.SAPOW),
    Drop(47, 39, 14, DuelRank.SAPOW),
    Drop(53, 39, 14, DuelRank.SAPOW),
    Drop(59, 39, 14, DuelRank.SAPOW),
    Drop(61, 39, 14, DuelRank.SAPOW),
    Drop(65, 39, 14, DuelRank.SAPOW),
    Drop(96, 39, 14, DuelRank.SAPOW),
    Drop(97, 39, 14, DuelRank.SAPOW),
    Drop(98, 39, 14, DuelRank.SAPOW),
    Drop(103, 39, 14, DuelRank.SAPOW),
    Drop(108, 39, 14, DuelRank.SAPOW),
    Drop(109, 39, 14, DuelRank.SAPOW),
    Drop(110, 39, 14, DuelRank.SAPOW),
    Drop(118, 39, 14, DuelRank.SAPOW),
    Drop(119, 39, 14, DuelRank.SAPOW),
    Drop(120, 39, 14, DuelRank.SAPOW),
    Drop(132, 39, 28, DuelRank.SAPOW),
    Drop(133, 39, 14, DuelRank.SAPOW),
    Drop(135, 39, 14, DuelRank.SAPOW),
    Drop(138, 39, 14, DuelRank.SAPOW),
    Drop(139, 39, 14, DuelRank.SAPOW),
    Drop(140, 39, 26, DuelRank.SAPOW),
    Drop(141, 39, 14, DuelRank.SAPOW),
    Drop(142, 39, 14, DuelRank.SAPOW),
    Drop(152, 39, 14, DuelRank.SAPOW),
    Drop(154, 39, 14, DuelRank.SAPOW),
    Drop(155, 39, 14, DuelRank.SAPOW),
    Drop(162, 39, 14, DuelRank.SAPOW),
    Drop(167, 39, 14, DuelRank.SAPOW),
    Drop(169, 39, 14, DuelRank.SAPOW),
    Drop(171, 39, 14, DuelRank.SAPOW),
    Drop(172, 39, 13, DuelRank.SAPOW),
    Drop(177, 39, 28, DuelRank.SAPOW),
    Drop(178, 39, 13, DuelRank.SAPOW),
    Drop(180, 39, 13, DuelRank.SAPOW),
    Drop(181, 39, 13, DuelRank.SAPOW),
    Drop(190, 39, 13, DuelRank.SAPOW),
    Drop(191, 39, 13, DuelRank.SAPOW),
    Drop(196, 39, 28, DuelRank.SAPOW),
    Drop(197, 39, 13, DuelRank.SAPOW),
    Drop(199, 39, 13, DuelRank.SAPOW),
    Drop(201, 39, 13, DuelRank.SAPOW),
    Drop(203, 39, 13, DuelRank.SAPOW),
    Drop(205, 39, 13, DuelRank.SAPOW),
    Drop(206, 39, 13, DuelRank.SAPOW),
    Drop(211, 39, 13, DuelRank.SAPOW),
    Drop(215, 39, 28, DuelRank.SAPOW),
    Drop(219, 39, 13, DuelRank.SAPOW),
    Drop(221, 39, 13, DuelRank.SAPOW),
    Drop(225, 39, 13, DuelRank.SAPOW),
    Drop(227, 39, 13, DuelRank.SAPOW),
    Drop(228, 39, 28, DuelRank.SAPOW),
    Drop(231, 39, 13, DuelRank.SAPOW),
    Drop(236, 39, 13, DuelRank.SAPOW),
    Drop(237, 39, 13, DuelRank.SAPOW),
    Drop(243, 39, 13, DuelRank.SAPOW),
    Drop(244, 39, 28, DuelRank.SAPOW),
    Drop(246, 39, 13, DuelRank.SAPOW),
    Drop(248, 39, 13, DuelRank.SAPOW),
    Drop(250, 39, 13, DuelRank.SAPOW),
    Drop(251, 39, 13, DuelRank.SAPOW),
    Drop(253, 39, 13, DuelRank.SAPOW),
    Drop(256, 39, 13, DuelRank.SAPOW),
    Drop(257, 39, 28, DuelRank.SAPOW),
    Drop(258, 39, 28, DuelRank.SAPOW),
    Drop(260, 39, 13, DuelRank.SAPOW),
    Drop(262, 39, 13, DuelRank.SAPOW),
    Drop(263, 39, 28, DuelRank.SAPOW),
    Drop(264, 39, 13, DuelRank.SAPOW),
    Drop(265, 39, 13, DuelRank.SAPOW),
    Drop(269, 39, 13, DuelRank.SAPOW),
    Drop(270, 39, 28, DuelRank.SAPOW),
    Drop(272, 39, 13, DuelRank.SAPOW),
    Drop(273, 39, 13, DuelRank.SAPOW),
    Drop(274, 39, 13, DuelRank.SAPOW),
    Drop(276, 39, 13, DuelRank.SAPOW),
    Drop(279, 39, 13, DuelRank.SAPOW),
    Drop(282, 39, 13, DuelRank.SAPOW),
    Drop(283, 39, 13, DuelRank.SAPOW),
    Drop(289, 39, 13, DuelRank.SAPOW),
    Drop(290, 39, 13, DuelRank.SAPOW),
    Drop(292, 39, 13, DuelRank.SAPOW),
    Drop(293, 39, 13, DuelRank.SAPOW),
    Drop(294, 39, 13, DuelRank.SAPOW),
    Drop(295, 39, 13, DuelRank.SAPOW),
    Drop(296, 39, 13, DuelRank.SAPOW),
    Drop(298, 39, 13, DuelRank.SAPOW),
    Drop(335, 39, 36, DuelRank.SAPOW),
    Drop(393, 39, 13, DuelRank.SAPOW),
    Drop(394, 39, 12, DuelRank.SAPOW),
    Drop(398, 39, 13, DuelRank.SAPOW),
    Drop(414, 39, 13, DuelRank.SAPOW),
    Drop(432, 39, 28, DuelRank.SAPOW),
    Drop(444, 39, 13, DuelRank.SAPOW),
    Drop(446, 39, 28, DuelRank.SAPOW),
    Drop(451, 39, 13, DuelRank.SAPOW),
    Drop(452, 39, 13, DuelRank.SAPOW),
    Drop(461, 39, 26, DuelRank.SAPOW),
    Drop(463, 39, 13, DuelRank.SAPOW),
    Drop(478, 39, 13, DuelRank.SAPOW),
    Drop(484, 39, 13, DuelRank.SAPOW),
    Drop(496, 39, 13, DuelRank.SAPOW),
    Drop(503, 39, 13, DuelRank.SAPOW),
    Drop(516, 39, 13, DuelRank.SAPOW),
    Drop(524, 39, 13, DuelRank.SAPOW),
    Drop(548, 39, 13, DuelRank.SAPOW),
    Drop(549, 39, 13, DuelRank.SAPOW),
    Drop(552, 39, 13, DuelRank.SAPOW),
    Drop(556, 39, 13, DuelRank.SAPOW),
    Drop(558, 39, 13, DuelRank.SAPOW),
    Drop(567, 39, 14, DuelRank.SAPOW),
    Drop(568, 39, 13, DuelRank.SAPOW),
    Drop(573, 39, 13, DuelRank.SAPOW),
    Drop(576, 39, 14, DuelRank.SAPOW),
    Drop(584, 39, 14, DuelRank.SAPOW),
    Drop(585, 39, 13, DuelRank.SAPOW),
    Drop(588, 39, 12, DuelRank.SAPOW),
    Drop(591, 39, 13, DuelRank.SAPOW),
    Drop(592, 39, 12, DuelRank.SAPOW),
    Drop(601, 39, 13, DuelRank.SAPOW),
    Drop(602, 39, 13, DuelRank.SAPOW),
    Drop(605, 39, 13, DuelRank.SAPOW),
    Drop(606, 39, 13, DuelRank.SAPOW),
    Drop(612, 39, 13, DuelRank.SAPOW),
    Drop(620, 39, 14, DuelRank.SAPOW),
    Drop(629, 39, 14, DuelRank.SAPOW),
    Drop(642, 39, 14, DuelRank.SAPOW),
    Drop(652, 39, 18, DuelRank.SAPOW),
    # BCD drops
    Drop(9, 1, 98, DuelRank.BCD),
    Drop(16, 1, 57, DuelRank.BCD),
    Drop(19, 1, 6, DuelRank.BCD),
    Drop(24, 1, 29, DuelRank.BCD),
    Drop(47, 1, 29, DuelRank.BCD),
    Drop(48, 1, 29, DuelRank.BCD),
    Drop(58, 1, 30, DuelRank.BCD),
    Drop(102, 1, 14, DuelRank.BCD),
    Drop(105, 1, 57, DuelRank.BCD),
    Drop(123, 1, 57, DuelRank.BCD),
    Drop(130, 1, 57, DuelRank.BCD),
    Drop(137, 1, 57, DuelRank.BCD),
    Drop(167, 1, 56, DuelRank.BCD),
    Drop(192, 1, 56, DuelRank.BCD),
    Drop(197, 1, 56, DuelRank.BCD),
    Drop(202, 1, 56, DuelRank.BCD),
    Drop(237, 1, 56, DuelRank.BCD),
    Drop(238, 1, 14, DuelRank.BCD),
    Drop(289, 1, 56, DuelRank.BCD),
    Drop(301, 1, 20, DuelRank.BCD),
    Drop(333, 1, 40, DuelRank.BCD),
    Drop(387, 1, 56, DuelRank.BCD),
    Drop(397, 1, 56, DuelRank.BCD),
    Drop(402, 1, 56, DuelRank.BCD),
    Drop(410, 1, 56, DuelRank.BCD),
    Drop(411, 1, 56, DuelRank.BCD),
    Drop(422, 1, 56, DuelRank.BCD),
    Drop(436, 1, 56, DuelRank.BCD),
    Drop(444, 1, 56, DuelRank.BCD),
    Drop(469, 1, 56, DuelRank.BCD),
    Drop(484, 1, 56, DuelRank.BCD),
    Drop(485, 1, 56, DuelRank.BCD),
    Drop(488, 1, 56, DuelRank.BCD),
    Drop(504, 1, 56, DuelRank.BCD),
    Drop(516, 1, 56, DuelRank.BCD),
    Drop(547, 1, 56, DuelRank.BCD),
    Drop(548, 1, 56, DuelRank.BCD),
    Drop(558, 1, 56, DuelRank.BCD),
    Drop(563, 1, 56, DuelRank.BCD),
    Drop(635, 1, 56, DuelRank.BCD),
    Drop(677, 1, 18, DuelRank.BCD),
    Drop(681, 1, 18, DuelRank.BCD),
    Drop(690, 1, 18, DuelRank.BCD),
    Drop(19, 2, 6, DuelRank.BCD),
    Drop(24, 2, 180, DuelRank.BCD),
    Drop(58, 2, 180, DuelRank.BCD),
    Drop(167, 2, 180, DuelRank.BCD),
    Drop(302, 2, 16, DuelRank.BCD),
    Drop(330, 2, 30, DuelRank.BCD),
    Drop(393, 2, 180, DuelRank.BCD),
    Drop(395, 2, 178, DuelRank.BCD),
    Drop(398, 2, 178, DuelRank.BCD),
    Drop(399, 2, 178, DuelRank.BCD),
    Drop(402, 2, 178, DuelRank.BCD),
    Drop(469, 2, 178, DuelRank.BCD),
    Drop(475, 2, 178, DuelRank.BCD),
    Drop(527, 2, 178, DuelRank.BCD),
    Drop(681, 2, 14, DuelRank.BCD),
    Drop(690, 2, 15, DuelRank.BCD),
    Drop(699, 2, 1, DuelRank.BCD),
    Drop(16, 3, 64, DuelRank.BCD),
    Drop(20, 3, 6, DuelRank.BCD),
    Drop(50, 3, 44, DuelRank.BCD),
    Drop(104, 3, 44, DuelRank.BCD),
    Drop(105, 3, 44, DuelRank.BCD),
    Drop(122, 3, 44, DuelRank.BCD),
    Drop(130, 3, 44, DuelRank.BCD),
    Drop(135, 3, 44, DuelRank.BCD),
    Drop(137, 3, 44, DuelRank.BCD),
    Drop(152, 3, 44, DuelRank.BCD),
    Drop(154, 3, 44, DuelRank.BCD),
    Drop(174, 3, 44, DuelRank.BCD),
    Drop(182, 3, 44, DuelRank.BCD),
    Drop(185, 3, 44, DuelRank.BCD),
    Drop(191, 3, 44, DuelRank.BCD),
    Drop(197, 3, 44, DuelRank.BCD),
    Drop(202, 3, 44, DuelRank.BCD),
    Drop(207, 3, 44, DuelRank.BCD),
    Drop(210, 3, 44, DuelRank.BCD),
    Drop(214, 3, 44, DuelRank.BCD),
    Drop(237, 3, 44, DuelRank.BCD),
    Drop(242, 3, 44, DuelRank.BCD),
    Drop(245, 3, 44, DuelRank.BCD),
    Drop(268, 3, 44, DuelRank.BCD),
    Drop(301, 3, 16, DuelRank.BCD),
    Drop(302, 3, 16, DuelRank.BCD),
    Drop(333, 3, 36, DuelRank.BCD),
    Drop(343, 3, 16, DuelRank.BCD),
    Drop(410, 3, 44, DuelRank.BCD),
    Drop(420, 3, 44, DuelRank.BCD),
    Drop(422, 3, 44, DuelRank.BCD),
    Drop(436, 3, 44, DuelRank.BCD),
    Drop(444, 3, 44, DuelRank.BCD),
    Drop(485, 3, 44, DuelRank.BCD),
    Drop(486, 3, 44, DuelRank.BCD),
    Drop(488, 3, 44, DuelRank.BCD),
    Drop(492, 3, 44, DuelRank.BCD),
    Drop(501, 3, 44, DuelRank.BCD),
    Drop(516, 3, 44, DuelRank.BCD),
    Drop(524, 3, 44, DuelRank.BCD),
    Drop(549, 3, 44, DuelRank.BCD),
    Drop(558, 3, 44, DuelRank.BCD),
    Drop(563, 3, 44, DuelRank.BCD),
    Drop(579, 3, 44, DuelRank.BCD),
    Drop(588, 3, 2, DuelRank.BCD),
    Drop(589, 3, 44, DuelRank.BCD),
    Drop(598, 3, 44, DuelRank.BCD),
    Drop(611, 3, 44, DuelRank.BCD),
    Drop(651, 3, 34, DuelRank.BCD),
    Drop(677, 3, 18, DuelRank.BCD),
    Drop(681, 3, 18, DuelRank.BCD),
    Drop(690, 3, 18, DuelRank.BCD),
    Drop(9, 4, 76, DuelRank.BCD),
    Drop(20, 4, 6, DuelRank.BCD),
    Drop(24, 4, 76, DuelRank.BCD),
    Drop(58, 4, 76, DuelRank.BCD),
    Drop(123, 4, 76, DuelRank.BCD),
    Drop(167, 4, 76, DuelRank.BCD),
    Drop(192, 4, 76, DuelRank.BCD),
    Drop(289, 4, 76, DuelRank.BCD),
    Drop(307, 4, 18, DuelRank.BCD),
    Drop(323, 4, 18, DuelRank.BCD),
    Drop(332, 4, 35, DuelRank.BCD),
    Drop(338, 4, 35, DuelRank.BCD),
    Drop(339, 4, 35, DuelRank.BCD),
    Drop(345, 4, 18, DuelRank.BCD),
    Drop(350, 4, 35, DuelRank.BCD),
    Drop(387, 4, 76, DuelRank.BCD),
    Drop(393, 4, 76, DuelRank.BCD),
    Drop(395, 4, 75, DuelRank.BCD),
    Drop(397, 4, 75, DuelRank.BCD),
    Drop(398, 4, 75, DuelRank.BCD),
    Drop(399, 4, 75, DuelRank.BCD),
    Drop(402, 4, 75, DuelRank.BCD),
    Drop(411, 4, 75, DuelRank.BCD),
    Drop(469, 4, 75, DuelRank.BCD),
    Drop(475, 4, 75, DuelRank.BCD),
    Drop(484, 4, 75, DuelRank.BCD),
    Drop(504, 4, 72, DuelRank.BCD),
    Drop(527, 4, 72, DuelRank.BCD),
    Drop(547, 4, 72, DuelRank.BCD),
    Drop(548, 4, 72, DuelRank.BCD),
    Drop(558, 4, 72, DuelRank.BCD),
    Drop(635, 4, 75, DuelRank.BCD),
    Drop(655, 4, 18, DuelRank.BCD),
    Drop(681, 4, 18, DuelRank.BCD),
    Drop(690, 4, 18, DuelRank.BCD),
    Drop(9, 5, 26, DuelRank.BCD),
    Drop(19, 5, 12, DuelRank.BCD),
    Drop(24, 5, 26, DuelRank.BCD),
    Drop(40, 5, 26, DuelRank.BCD),
    Drop(45, 5, 26, DuelRank.BCD),
    Drop(61, 5, 26, DuelRank.BCD),
    Drop(65, 5, 26, DuelRank.BCD),
    Drop(96, 5, 26, DuelRank.BCD),
    Drop(97, 5, 26, DuelRank.BCD),
    Drop(98, 5, 26, DuelRank.BCD),
    Drop(102, 5, 12, DuelRank.BCD),
    Drop(108, 5, 26, DuelRank.BCD),
    Drop(118, 5, 26, DuelRank.BCD),
    Drop(135, 5, 26, DuelRank.BCD),
    Drop(138, 5, 26, DuelRank.BCD),
    Drop(139, 5, 26, DuelRank.BCD),
    Drop(140, 5, 26, DuelRank.BCD),
    Drop(141, 5, 26, DuelRank.BCD),
    Drop(152, 5, 26, DuelRank.BCD),
    Drop(154, 5, 26, DuelRank.BCD),
    Drop(162, 5, 26, DuelRank.BCD),
    Drop(167, 5, 26, DuelRank.BCD),
    Drop(169, 5, 26, DuelRank.BCD),
    Drop(172, 5, 26, DuelRank.BCD),
    Drop(177, 5, 26, DuelRank.BCD),
    Drop(181, 5, 26, DuelRank.BCD),
    Drop(191, 5, 26, DuelRank.BCD),
    Drop(197, 5, 26, DuelRank.BCD),
    Drop(199, 5, 26, DuelRank.BCD),
    Drop(203, 5, 26, DuelRank.BCD),
    Drop(205, 5, 26, DuelRank.BCD),
    Drop(206, 5, 26, DuelRank.BCD),
    Drop(211, 5, 26, DuelRank.BCD),
    Drop(215, 5, 26, DuelRank.BCD),
    Drop(219, 5, 26, DuelRank.BCD),
    Drop(227, 5, 26, DuelRank.BCD),
    Drop(228, 5, 26, DuelRank.BCD),
    Drop(231, 5, 26, DuelRank.BCD),
    Drop(237, 5, 26, DuelRank.BCD),
    Drop(238, 5, 12, DuelRank.BCD),
    Drop(243, 5, 26, DuelRank.BCD),
    Drop(244, 5, 26, DuelRank.BCD),
    Drop(250, 5, 26, DuelRank.BCD),
    Drop(257, 5, 26, DuelRank.BCD),
    Drop(259, 5, 26, DuelRank.BCD),
    Drop(262, 5, 26, DuelRank.BCD),
    Drop(265, 5, 26, DuelRank.BCD),
    Drop(269, 5, 26, DuelRank.BCD),
    Drop(270, 5, 26, DuelRank.BCD),
    Drop(276, 5, 26, DuelRank.BCD),
    Drop(289, 5, 26, DuelRank.BCD),
    Drop(292, 5, 26, DuelRank.BCD),
    Drop(296, 5, 26, DuelRank.BCD),
    Drop(304, 5, 30, DuelRank.BCD),
    Drop(335, 5, 60, DuelRank.BCD),
    Drop(393, 5, 26, DuelRank.BCD),
    Drop(394, 5, 12, DuelRank.BCD),
    Drop(398, 5, 26, DuelRank.BCD),
    Drop(432, 5, 26, DuelRank.BCD),
    Drop(444, 5, 26, DuelRank.BCD),
    Drop(446, 5, 26, DuelRank.BCD),
    Drop(451, 5, 26, DuelRank.BCD),
    Drop(452, 5, 26, DuelRank.BCD),
    Drop(461, 5, 26, DuelRank.BCD),
    Drop(463, 5, 26, DuelRank.BCD),
    Drop(484, 5, 25, DuelRank.BCD),
    Drop(503, 5, 25, DuelRank.BCD),
    Drop(516, 5, 25, DuelRank.BCD),
    Drop(524, 5, 25, DuelRank.BCD),
    Drop(548, 5, 25, DuelRank.BCD),
    Drop(549, 5, 25, DuelRank.BCD),
    Drop(556, 5, 25, DuelRank.BCD),
    Drop(558, 5, 25, DuelRank.BCD),
    Drop(586, 5, 12, DuelRank.BCD),
    Drop(591, 5, 25, DuelRank.BCD),
    Drop(592, 5, 12, DuelRank.BCD),
    Drop(602, 5, 25, DuelRank.BCD),
    Drop(605, 5, 25, DuelRank.BCD),
    Drop(606, 5, 25, DuelRank.BCD),
    Drop(610, 5, 2, DuelRank.BCD),
    Drop(629, 5, 25, DuelRank.BCD),
    Drop(642, 5, 25, DuelRank.BCD),
    Drop(4, 6, 18, DuelRank.BCD),
    Drop(9, 6, 18, DuelRank.BCD),
    Drop(20, 6, 8, DuelRank.BCD),
    Drop(24, 6, 18, DuelRank.BCD),
    Drop(40, 6, 18, DuelRank.BCD),
    Drop(45, 6, 18, DuelRank.BCD),
    Drop(47, 6, 18, DuelRank.BCD),
    Drop(59, 6, 18, DuelRank.BCD),
    Drop(61, 6, 18, DuelRank.BCD),
    Drop(65, 6, 18, DuelRank.BCD),
    Drop(96, 6, 18, DuelRank.BCD),
    Drop(97, 6, 18, DuelRank.BCD),
    Drop(98, 6, 18, DuelRank.BCD),
    Drop(108, 6, 18, DuelRank.BCD),
    Drop(118, 6, 18, DuelRank.BCD),
    Drop(133, 6, 18, DuelRank.BCD),
    Drop(135, 6, 18, DuelRank.BCD),
    Drop(138, 6, 18, DuelRank.BCD),
    Drop(139, 6, 18, DuelRank.BCD),
    Drop(140, 6, 36, DuelRank.BCD),
    Drop(141, 6, 18, DuelRank.BCD),
    Drop(142, 6, 18, DuelRank.BCD),
    Drop(152, 6, 18, DuelRank.BCD),
    Drop(154, 6, 18, DuelRank.BCD),
    Drop(155, 6, 18, DuelRank.BCD),
    Drop(162, 6, 18, DuelRank.BCD),
    Drop(167, 6, 18, DuelRank.BCD),
    Drop(169, 6, 18, DuelRank.BCD),
    Drop(172, 6, 18, DuelRank.BCD),
    Drop(177, 6, 24, DuelRank.BCD),
    Drop(178, 6, 18, DuelRank.BCD),
    Drop(180, 6, 18, DuelRank.BCD),
    Drop(181, 6, 18, DuelRank.BCD),
    Drop(190, 6, 18, DuelRank.BCD),
    Drop(191, 6, 18, DuelRank.BCD),
    Drop(197, 6, 18, DuelRank.BCD),
    Drop(199, 6, 18, DuelRank.BCD),
    Drop(203, 6, 18, DuelRank.BCD),
    Drop(205, 6, 18, DuelRank.BCD),
    Drop(206, 6, 18, DuelRank.BCD),
    Drop(211, 6, 18, DuelRank.BCD),
    Drop(215, 6, 36, DuelRank.BCD),
    Drop(219, 6, 18, DuelRank.BCD),
    Drop(227, 6, 18, DuelRank.BCD),
    Drop(228, 6, 36, DuelRank.BCD),
    Drop(231, 6, 18, DuelRank.BCD),
    Drop(237, 6, 18, DuelRank.BCD),
    Drop(243, 6, 18, DuelRank.BCD),
    Drop(244, 6, 36, DuelRank.BCD),
    Drop(250, 6, 18, DuelRank.BCD),
    Drop(253, 6, 18, DuelRank.BCD),
    Drop(257, 6, 36, DuelRank.BCD),
    Drop(260, 6, 18, DuelRank.BCD),
    Drop(262, 6, 18, DuelRank.BCD),
    Drop(263, 6, 36, DuelRank.BCD),
    Drop(264, 6, 18, DuelRank.BCD),
    Drop(265, 6, 18, DuelRank.BCD),
    Drop(269, 6, 18, DuelRank.BCD),
    Drop(270, 6, 36, DuelRank.BCD),
    Drop(276, 6, 18, DuelRank.BCD),
    Drop(279, 6, 18, DuelRank.BCD),
    Drop(282, 6, 18, DuelRank.BCD),
    Drop(283, 6, 18, DuelRank.BCD),
    Drop(289, 6, 18, DuelRank.BCD),
    Drop(292, 6, 18, DuelRank.BCD),
    Drop(295, 6, 18, DuelRank.BCD),
    Drop(296, 6, 18, DuelRank.BCD),
    Drop(298, 6, 18, DuelRank.BCD),
    Drop(304, 6, 24, DuelRank.BCD),
    Drop(335, 6, 48, DuelRank.BCD),
    Drop(393, 6, 16, DuelRank.BCD),
    Drop(394, 6, 16, DuelRank.BCD),
    Drop(398, 6, 18, DuelRank.BCD),
    Drop(432, 6, 36, DuelRank.BCD),
    Drop(444, 6, 18, DuelRank.BCD),
    Drop(446, 6, 36, DuelRank.BCD),
    Drop(451, 6, 18, DuelRank.BCD),
    Drop(452, 6, 18, DuelRank.BCD),
    Drop(461, 6, 36, DuelRank.BCD),
    Drop(463, 6, 18, DuelRank.BCD),
    Drop(478, 6, 18, DuelRank.BCD),
    Drop(484, 6, 18, DuelRank.BCD),
    Drop(503, 6, 18, DuelRank.BCD),
    Drop(516, 6, 18, DuelRank.BCD),
    Drop(524, 6, 18, DuelRank.BCD),
    Drop(548, 6, 18, DuelRank.BCD),
    Drop(549, 6, 18, DuelRank.BCD),
    Drop(556, 6, 18, DuelRank.BCD),
    Drop(558, 6, 18, DuelRank.BCD),
    Drop(568, 6, 18, DuelRank.BCD),
    Drop(573, 6, 18, DuelRank.BCD),
    Drop(585, 6, 18, DuelRank.BCD),
    Drop(588, 6, 16, DuelRank.BCD),
    Drop(591, 6, 18, DuelRank.BCD),
    Drop(592, 6, 16, DuelRank.BCD),
    Drop(602, 6, 18, DuelRank.BCD),
    Drop(605, 6, 18, DuelRank.BCD),
    Drop(606, 6, 18, DuelRank.BCD),
    Drop(612, 6, 18, DuelRank.BCD),
    Drop(629, 6, 18, DuelRank.BCD),
    Drop(642, 6, 18, DuelRank.BCD),
    Drop(652, 6, 26, DuelRank.BCD),
    Drop(8, 7, 42, DuelRank.BCD),
    Drop(19, 7, 5, DuelRank.BCD),
    Drop(50, 7, 21, DuelRank.BCD),
    Drop(75, 7, 115, DuelRank.BCD),
    Drop(98, 7, 21, DuelRank.BCD),
    Drop(101, 7, 42, DuelRank.BCD),
    Drop(104, 7, 21, DuelRank.BCD),
    Drop(107, 7, 21, DuelRank.BCD),
    Drop(122, 7, 21, DuelRank.BCD),
    Drop(123, 7, 21, DuelRank.BCD),
    Drop(130, 7, 21, DuelRank.BCD),
    Drop(133, 7, 21, DuelRank.BCD),
    Drop(134, 7, 21, DuelRank.BCD),
    Drop(135, 7, 21, DuelRank.BCD),
    Drop(137, 7, 21, DuelRank.BCD),
    Drop(145, 7, 21, DuelRank.BCD),
    Drop(152, 7, 21, DuelRank.BCD),
    Drop(154, 7, 21, DuelRank.BCD),
    Drop(157, 7, 42, DuelRank.BCD),
    Drop(158, 7, 42, DuelRank.BCD),
    Drop(159, 7, 21, DuelRank.BCD),
    Drop(174, 7, 21, DuelRank.BCD),
    Drop(176, 7, 38, DuelRank.BCD),
    Drop(179, 7, 21, DuelRank.BCD),
    Drop(180, 7, 21, DuelRank.BCD),
    Drop(182, 7, 21, DuelRank.BCD),
    Drop(183, 7, 21, DuelRank.BCD),
    Drop(185, 7, 21, DuelRank.BCD),
    Drop(191, 7, 21, DuelRank.BCD),
    Drop(195, 7, 21, DuelRank.BCD),
    Drop(198, 7, 21, DuelRank.BCD),
    Drop(200, 7, 21, DuelRank.BCD),
    Drop(202, 7, 21, DuelRank.BCD),
    Drop(203, 7, 21, DuelRank.BCD),
    Drop(207, 7, 21, DuelRank.BCD),
    Drop(209, 7, 21, DuelRank.BCD),
    Drop(210, 7, 41, DuelRank.BCD),
    Drop(211, 7, 21, DuelRank.BCD),
    Drop(212, 7, 21, DuelRank.BCD),
    Drop(214, 7, 21, DuelRank.BCD),
    Drop(222, 7, 21, DuelRank.BCD),
    Drop(227, 7, 21, DuelRank.BCD),
    Drop(229, 7, 21, DuelRank.BCD),
    Drop(232, 7, 21, DuelRank.BCD),
    Drop(237, 7, 21, DuelRank.BCD),
    Drop(242, 7, 21, DuelRank.BCD),
    Drop(245, 7, 22, DuelRank.BCD),
    Drop(254, 7, 22, DuelRank.BCD),
    Drop(261, 7, 22, DuelRank.BCD),
    Drop(268, 7, 22, DuelRank.BCD),
    Drop(285, 7, 22, DuelRank.BCD),
    Drop(292, 7, 22, DuelRank.BCD),
    Drop(302, 7, 18, DuelRank.BCD),
    Drop(308, 7, 18, DuelRank.BCD),
    Drop(330, 7, 35, DuelRank.BCD),
    Drop(343, 7, 18, DuelRank.BCD),
    Drop(344, 7, 18, DuelRank.BCD),
    Drop(420, 7, 22, DuelRank.BCD),
    Drop(452, 7, 22, DuelRank.BCD),
    Drop(476, 7, 22, DuelRank.BCD),
    Drop(486, 7, 22, DuelRank.BCD),
    Drop(488, 7, 22, DuelRank.BCD),
    Drop(492, 7, 22, DuelRank.BCD),
    Drop(501, 7, 22, DuelRank.BCD),
    Drop(506, 7, 22, DuelRank.BCD),
    Drop(510, 7, 44, DuelRank.BCD),
    Drop(524, 7, 22, DuelRank.BCD),
    Drop(536, 7, 22, DuelRank.BCD),
    Drop(540, 7, 22, DuelRank.BCD),
    Drop(544, 7, 22, DuelRank.BCD),
    Drop(547, 7, 22, DuelRank.BCD),
    Drop(549, 7, 22, DuelRank.BCD),
    Drop(556, 7, 22, DuelRank.BCD),
    Drop(579, 7, 44, DuelRank.BCD),
    Drop(589, 7, 44, DuelRank.BCD),
    Drop(591, 7, 22, DuelRank.BCD),
    Drop(598, 7, 22, DuelRank.BCD),
    Drop(602, 7, 22, DuelRank.BCD),
    Drop(609, 7, 22, DuelRank.BCD),
    Drop(611, 7, 22, DuelRank.BCD),
    Drop(677, 7, 18, DuelRank.BCD),
    Drop(682, 7, 18, DuelRank.BCD),
    Drop(683, 7, 18, DuelRank.BCD),
    Drop(687, 7, 18, DuelRank.BCD),
    Drop(2, 8, 26, DuelRank.BCD),
    Drop(19, 8, 6, DuelRank.BCD),
    Drop(30, 8, 26, DuelRank.BCD),
    Drop(34, 8, 26, DuelRank.BCD),
    Drop(36, 8, 26, DuelRank.BCD),
    Drop(42, 8, 26, DuelRank.BCD),
    Drop(44, 8, 28, DuelRank.BCD),
    Drop(74, 8, 28, DuelRank.BCD),
    Drop(79, 8, 24, DuelRank.BCD),
    Drop(90, 8, 24, DuelRank.BCD),
    Drop(96, 8, 28, DuelRank.BCD),
    Drop(97, 8, 28, DuelRank.BCD),
    Drop(98, 8, 28, DuelRank.BCD),
    Drop(99, 8, 48, DuelRank.BCD),
    Drop(104, 8, 28, DuelRank.BCD),
    Drop(106, 8, 28, DuelRank.BCD),
    Drop(108, 8, 28, DuelRank.BCD),
    Drop(114, 8, 28, DuelRank.BCD),
    Drop(115, 8, 28, DuelRank.BCD),
    Drop(129, 8, 28, DuelRank.BCD),
    Drop(132, 8, 28, DuelRank.BCD),
    Drop(135, 8, 28, DuelRank.BCD),
    Drop(139, 8, 28, DuelRank.BCD),
    Drop(142, 8, 28, DuelRank.BCD),
    Drop(143, 8, 28, DuelRank.BCD),
    Drop(144, 8, 28, DuelRank.BCD),
    Drop(145, 8, 28, DuelRank.BCD),
    Drop(154, 8, 28, DuelRank.BCD),
    Drop(174, 8, 28, DuelRank.BCD),
    Drop(179, 8, 28, DuelRank.BCD),
    Drop(183, 8, 28, DuelRank.BCD),
    Drop(184, 8, 28, DuelRank.BCD),
    Drop(190, 8, 28, DuelRank.BCD),
    Drop(203, 8, 28, DuelRank.BCD),
    Drop(213, 8, 28, DuelRank.BCD),
    Drop(215, 8, 28, DuelRank.BCD),
    Drop(220, 8, 28, DuelRank.BCD),
    Drop(223, 8, 20, DuelRank.BCD),
    Drop(228, 8, 28, DuelRank.BCD),
    Drop(237, 8, 28, DuelRank.BCD),
    Drop(244, 8, 28, DuelRank.BCD),
    Drop(253, 8, 28, DuelRank.BCD),
    Drop(257, 8, 28, DuelRank.BCD),
    Drop(259, 8, 28, DuelRank.BCD),
    Drop(263, 8, 28, DuelRank.BCD),
    Drop(268, 8, 28, DuelRank.BCD),
    Drop(366, 8, 28, DuelRank.BCD),
    Drop(368, 8, 28, DuelRank.BCD),
    Drop(455, 8, 64, DuelRank.BCD),
    Drop(457, 8, 64, DuelRank.BCD),
    Drop(465, 8, 24, DuelRank.BCD),
    Drop(470, 8, 32, DuelRank.BCD),
    Drop(505, 8, 64, DuelRank.BCD),
    Drop(518, 8, 10, DuelRank.BCD),
    Drop(522, 8, 24, DuelRank.BCD),
    Drop(525, 8, 16, DuelRank.BCD),
    Drop(529, 8, 10, DuelRank.BCD),
    Drop(531, 8, 44, DuelRank.BCD),
    Drop(532, 8, 32, DuelRank.BCD),
    Drop(571, 8, 24, DuelRank.BCD),
    Drop(572, 8, 24, DuelRank.BCD),
    Drop(593, 8, 16, DuelRank.BCD),
    Drop(594, 8, 24, DuelRank.BCD),
    Drop(599, 8, 64, DuelRank.BCD),
    Drop(623, 8, 28, DuelRank.BCD),
    Drop(627, 8, 24, DuelRank.BCD),
    Drop(631, 8, 32, DuelRank.BCD),
    Drop(632, 8, 16, DuelRank.BCD),
    Drop(634, 8, 32, DuelRank.BCD),
    Drop(646, 8, 64, DuelRank.BCD),
    Drop(5, 9, 19, DuelRank.BCD),
    Drop(8, 9, 19, DuelRank.BCD),
    Drop(19, 9, 6, DuelRank.BCD),
    Drop(29, 9, 19, DuelRank.BCD),
    Drop(45, 9, 19, DuelRank.BCD),
    Drop(47, 9, 19, DuelRank.BCD),
    Drop(48, 9, 19, DuelRank.BCD),
    Drop(96, 9, 19, DuelRank.BCD),
    Drop(97, 9, 19, DuelRank.BCD),
    Drop(100, 9, 19, DuelRank.BCD),
    Drop(105, 9, 38, DuelRank.BCD),
    Drop(107, 9, 19, DuelRank.BCD),
    Drop(108, 9, 19, DuelRank.BCD),
    Drop(113, 9, 19, DuelRank.BCD),
    Drop(114, 9, 19, DuelRank.BCD),
    Drop(116, 9, 19, DuelRank.BCD),
    Drop(129, 9, 19, DuelRank.BCD),
    Drop(134, 9, 19, DuelRank.BCD),
    Drop(139, 9, 19, DuelRank.BCD),
    Drop(143, 9, 19, DuelRank.BCD),
    Drop(144, 9, 19, DuelRank.BCD),
    Drop(145, 9, 19, DuelRank.BCD),
    Drop(147, 9, 19, DuelRank.BCD),
    Drop(148, 9, 19, DuelRank.BCD),
    Drop(155, 9, 19, DuelRank.BCD),
    Drop(156, 9, 19, DuelRank.BCD),
    Drop(158, 9, 19, DuelRank.BCD),
    Drop(160, 9, 19, DuelRank.BCD),
    Drop(161, 9, 19, DuelRank.BCD),
    Drop(173, 9, 19, DuelRank.BCD),
    Drop(175, 9, 19, DuelRank.BCD),
    Drop(176, 9, 19, DuelRank.BCD),
    Drop(177, 9, 19, DuelRank.BCD),
    Drop(178, 9, 19, DuelRank.BCD),
    Drop(180, 9, 19, DuelRank.BCD),
    Drop(183, 9, 19, DuelRank.BCD),
    Drop(184, 9, 19, DuelRank.BCD),
    Drop(187, 9, 19, DuelRank.BCD),
    Drop(188, 9, 19, DuelRank.BCD),
    Drop(189, 9, 19, DuelRank.BCD),
    Drop(190, 9, 19, DuelRank.BCD),
    Drop(195, 9, 19, DuelRank.BCD),
    Drop(199, 9, 19, DuelRank.BCD),
    Drop(203, 9, 19, DuelRank.BCD),
    Drop(205, 9, 19, DuelRank.BCD),
    Drop(206, 9, 19, DuelRank.BCD),
    Drop(208, 9, 19, DuelRank.BCD),
    Drop(209, 9, 19, DuelRank.BCD),
    Drop(215, 9, 19, DuelRank.BCD),
    Drop(218, 9, 52, DuelRank.BCD),
    Drop(222, 9, 19, DuelRank.BCD),
    Drop(224, 9, 19, DuelRank.BCD),
    Drop(226, 9, 19, DuelRank.BCD),
    Drop(232, 9, 19, DuelRank.BCD),
    Drop(239, 9, 19, DuelRank.BCD),
    Drop(240, 9, 19, DuelRank.BCD),
    Drop(243, 9, 19, DuelRank.BCD),
    Drop(247, 9, 19, DuelRank.BCD),
    Drop(253, 9, 19, DuelRank.BCD),
    Drop(264, 9, 19, DuelRank.BCD),
    Drop(265, 9, 19, DuelRank.BCD),
    Drop(266, 9, 19, DuelRank.BCD),
    Drop(267, 9, 19, DuelRank.BCD),
    Drop(271, 9, 19, DuelRank.BCD),
    Drop(277, 9, 19, DuelRank.BCD),
    Drop(282, 9, 19, DuelRank.BCD),
    Drop(283, 9, 19, DuelRank.BCD),
    Drop(292, 9, 19, DuelRank.BCD),
    Drop(298, 9, 19, DuelRank.BCD),
    Drop(300, 9, 19, DuelRank.BCD),
    Drop(308, 9, 19, DuelRank.BCD),
    Drop(331, 9, 48, DuelRank.BCD),
    Drop(332, 9, 48, DuelRank.BCD),
    Drop(417, 9, 19, DuelRank.BCD),
    Drop(421, 9, 19, DuelRank.BCD),
    Drop(432, 9, 19, DuelRank.BCD),
    Drop(451, 9, 19, DuelRank.BCD),
    Drop(463, 9, 19, DuelRank.BCD),
    Drop(476, 9, 19, DuelRank.BCD),
    Drop(477, 9, 19, DuelRank.BCD),
    Drop(478, 9, 19, DuelRank.BCD),
    Drop(479, 9, 19, DuelRank.BCD),
    Drop(481, 9, 19, DuelRank.BCD),
    Drop(490, 9, 19, DuelRank.BCD),
    Drop(503, 9, 19, DuelRank.BCD),
    Drop(510, 9, 19, DuelRank.BCD),
    Drop(534, 9, 19, DuelRank.BCD),
    Drop(536, 9, 19, DuelRank.BCD),
    Drop(538, 9, 19, DuelRank.BCD),
    Drop(544, 9, 19, DuelRank.BCD),
    Drop(556, 9, 19, DuelRank.BCD),
    Drop(568, 9, 36, DuelRank.BCD),
    Drop(569, 9, 19, DuelRank.BCD),
    Drop(573, 9, 52, DuelRank.BCD),
    Drop(585, 9, 19, DuelRank.BCD),
    Drop(590, 9, 19, DuelRank.BCD),
    Drop(604, 9, 19, DuelRank.BCD),
    Drop(605, 9, 19, DuelRank.BCD),
    Drop(606, 9, 19, DuelRank.BCD),
    Drop(682, 9, 20, DuelRank.BCD),
    Drop(3, 10, 33, DuelRank.BCD),
    Drop(8, 10, 33, DuelRank.BCD),
    Drop(10, 10, 33, DuelRank.BCD),
    Drop(19, 10, 6, DuelRank.BCD),
    Drop(25, 10, 33, DuelRank.BCD),
    Drop(50, 10, 84, DuelRank.BCD),
    Drop(53, 10, 120, DuelRank.BCD),
    Drop(77, 10, 33, DuelRank.BCD),
    Drop(80, 10, 33, DuelRank.BCD),
    Drop(109, 10, 33, DuelRank.BCD),
    Drop(115, 10, 33, DuelRank.BCD),
    Drop(116, 10, 33, DuelRank.BCD),
    Drop(119, 10, 33, DuelRank.BCD),
    Drop(121, 10, 33, DuelRank.BCD),
    Drop(123, 10, 33, DuelRank.BCD),
    Drop(141, 10, 26, DuelRank.BCD),
    Drop(157, 10, 26, DuelRank.BCD),
    Drop(158, 10, 26, DuelRank.BCD),
    Drop(164, 10, 26, DuelRank.BCD),
    Drop(166, 10, 26, DuelRank.BCD),
    Drop(171, 10, 26, DuelRank.BCD),
    Drop(180, 10, 26, DuelRank.BCD),
    Drop(186, 10, 26, DuelRank.BCD),
    Drop(193, 10, 26, DuelRank.BCD),
    Drop(194, 10, 26, DuelRank.BCD),
    Drop(196, 10, 26, DuelRank.BCD),
    Drop(201, 10, 26, DuelRank.BCD),
    Drop(209, 10, 26, DuelRank.BCD),
    Drop(220, 10, 26, DuelRank.BCD),
    Drop(221, 10, 26, DuelRank.BCD),
    Drop(225, 10, 26, DuelRank.BCD),
    Drop(233, 10, 26, DuelRank.BCD),
    Drop(236, 10, 26, DuelRank.BCD),
    Drop(248, 10, 26, DuelRank.BCD),
    Drop(255, 10, 26, DuelRank.BCD),
    Drop(256, 10, 26, DuelRank.BCD),
    Drop(259, 10, 26, DuelRank.BCD),
    Drop(272, 10, 26, DuelRank.BCD),
    Drop(273, 10, 26, DuelRank.BCD),
    Drop(274, 10, 26, DuelRank.BCD),
    Drop(280, 10, 26, DuelRank.BCD),
    Drop(291, 10, 26, DuelRank.BCD),
    Drop(293, 10, 26, DuelRank.BCD),
    Drop(330, 10, 30, DuelRank.BCD),
    Drop(367, 10, 2, DuelRank.BCD),
    Drop(397, 10, 26, DuelRank.BCD),
    Drop(406, 10, 26, DuelRank.BCD),
    Drop(459, 10, 26, DuelRank.BCD),
    Drop(476, 10, 26, DuelRank.BCD),
    Drop(477, 10, 26, DuelRank.BCD),
    Drop(478, 10, 26, DuelRank.BCD),
    Drop(479, 10, 26, DuelRank.BCD),
    Drop(485, 10, 26, DuelRank.BCD),
    Drop(488, 10, 26, DuelRank.BCD),
    Drop(496, 10, 26, DuelRank.BCD),
    Drop(501, 10, 26, DuelRank.BCD),
    Drop(506, 10, 26, DuelRank.BCD),
    Drop(510, 10, 26, DuelRank.BCD),
    Drop(514, 10, 26, DuelRank.BCD),
    Drop(534, 10, 26, DuelRank.BCD),
    Drop(547, 10, 26, DuelRank.BCD),
    Drop(552, 10, 26, DuelRank.BCD),
    Drop(567, 10, 26, DuelRank.BCD),
    Drop(576, 10, 26, DuelRank.BCD),
    Drop(579, 10, 26, DuelRank.BCD),
    Drop(589, 10, 26, DuelRank.BCD),
    Drop(608, 10, 26, DuelRank.BCD),
    Drop(609, 10, 26, DuelRank.BCD),
    Drop(620, 10, 42, DuelRank.BCD),
    Drop(643, 10, 26, DuelRank.BCD),
    Drop(683, 10, 16, DuelRank.BCD),
    Drop(6, 11, 40, DuelRank.BCD),
    Drop(13, 11, 120, DuelRank.BCD),
    Drop(20, 11, 10, DuelRank.BCD),
    Drop(26, 11, 40, DuelRank.BCD),
    Drop(27, 11, 40, DuelRank.BCD),
    Drop(31, 11, 40, DuelRank.BCD),
    Drop(36, 11, 40, DuelRank.BCD),
    Drop(41, 11, 40, DuelRank.BCD),
    Drop(43, 11, 40, DuelRank.BCD),
    Drop(44, 11, 40, DuelRank.BCD),
    Drop(46, 11, 40, DuelRank.BCD),
    Drop(49, 11, 40, DuelRank.BCD),
    Drop(54, 11, 40, DuelRank.BCD),
    Drop(55, 11, 40, DuelRank.BCD),
    Drop(70, 11, 40, DuelRank.BCD),
    Drop(71, 11, 40, DuelRank.BCD),
    Drop(95, 11, 40, DuelRank.BCD),
    Drop(125, 11, 120, DuelRank.BCD),
    Drop(136, 11, 40, DuelRank.BCD),
    Drop(149, 11, 40, DuelRank.BCD),
    Drop(168, 11, 40, DuelRank.BCD),
    Drop(186, 11, 80, DuelRank.BCD),
    Drop(207, 11, 80, DuelRank.BCD),
    Drop(230, 11, 40, DuelRank.BCD),
    Drop(272, 11, 80, DuelRank.BCD),
    Drop(300, 11, 80, DuelRank.BCD),
    Drop(307, 11, 26, DuelRank.BCD),
    Drop(312, 11, 26, DuelRank.BCD),
    Drop(316, 11, 60, DuelRank.BCD),
    Drop(332, 11, 60, DuelRank.BCD),
    Drop(412, 11, 40, DuelRank.BCD),
    Drop(415, 11, 40, DuelRank.BCD),
    Drop(416, 11, 40, DuelRank.BCD),
    Drop(466, 11, 120, DuelRank.BCD),
    Drop(511, 11, 40, DuelRank.BCD),
    Drop(512, 11, 40, DuelRank.BCD),
    Drop(538, 11, 80, DuelRank.BCD),
    Drop(552, 11, 80, DuelRank.BCD),
    Drop(597, 11, 40, DuelRank.BCD),
    Drop(684, 11, 26, DuelRank.BCD),
    Drop(9, 12, 40, DuelRank.BCD),
    Drop(19, 12, 12, DuelRank.BCD),
    Drop(24, 12, 40, DuelRank.BCD),
    Drop(30, 12, 40, DuelRank.BCD),
    Drop(36, 12, 108, DuelRank.BCD),
    Drop(95, 12, 40, DuelRank.BCD),
    Drop(96, 12, 108, DuelRank.BCD),
    Drop(97, 12, 108, DuelRank.BCD),
    Drop(98, 12, 108, DuelRank.BCD),
    Drop(108, 12, 40, DuelRank.BCD),
    Drop(132, 12, 40, DuelRank.BCD),
    Drop(135, 12, 40, DuelRank.BCD),
    Drop(139, 12, 40, DuelRank.BCD),
    Drop(154, 12, 40, DuelRank.BCD),
    Drop(197, 12, 40, DuelRank.BCD),
    Drop(203, 12, 40, DuelRank.BCD),
    Drop(215, 12, 40, DuelRank.BCD),
    Drop(228, 12, 40, DuelRank.BCD),
    Drop(275, 12, 8, DuelRank.BCD),
    Drop(283, 12, 40, DuelRank.BCD),
    Drop(297, 12, 8, DuelRank.BCD),
    Drop(319, 12, 64, DuelRank.BCD),
    Drop(325, 12, 30, DuelRank.BCD),
    Drop(331, 12, 64, DuelRank.BCD),
    Drop(335, 12, 64, DuelRank.BCD),
    Drop(406, 12, 40, DuelRank.BCD),
    Drop(410, 12, 40, DuelRank.BCD),
    Drop(411, 12, 40, DuelRank.BCD),
    Drop(412, 12, 40, DuelRank.BCD),
    Drop(414, 12, 40, DuelRank.BCD),
    Drop(415, 12, 40, DuelRank.BCD),
    Drop(416, 12, 40, DuelRank.BCD),
    Drop(417, 12, 40, DuelRank.BCD),
    Drop(418, 12, 8, DuelRank.BCD),
    Drop(420, 12, 40, DuelRank.BCD),
    Drop(421, 12, 40, DuelRank.BCD),
    Drop(422, 12, 40, DuelRank.BCD),
    Drop(498, 12, 8, DuelRank.BCD),
    Drop(512, 12, 36, DuelRank.BCD),
    Drop(514, 12, 36, DuelRank.BCD),
    Drop(544, 12, 36, DuelRank.BCD),
    Drop(548, 12, 36, DuelRank.BCD),
    Drop(556, 12, 36, DuelRank.BCD),
    Drop(585, 12, 36, DuelRank.BCD),
    Drop(614, 12, 8, DuelRank.BCD),
    Drop(643, 12, 36, DuelRank.BCD),
    Drop(658, 12, 60, DuelRank.BCD),
    Drop(684, 12, 30, DuelRank.BCD),
    Drop(5, 13, 12, DuelRank.BCD),
    Drop(8, 13, 12, DuelRank.BCD),
    Drop(20, 13, 8, DuelRank.BCD),
    Drop(29, 13, 12, DuelRank.BCD),
    Drop(45, 13, 12, DuelRank.BCD),
    Drop(48, 13, 12, DuelRank.BCD),
    Drop(50, 13, 12, DuelRank.BCD),
    Drop(96, 13, 12, DuelRank.BCD),
    Drop(97, 13, 12, DuelRank.BCD),
    Drop(98, 13, 12, DuelRank.BCD),
    Drop(101, 13, 12, DuelRank.BCD),
    Drop(107, 13, 12, DuelRank.BCD),
    Drop(108, 13, 12, DuelRank.BCD),
    Drop(122, 13, 12, DuelRank.BCD),
    Drop(129, 13, 12, DuelRank.BCD),
    Drop(134, 13, 12, DuelRank.BCD),
    Drop(135, 13, 12, DuelRank.BCD),
    Drop(139, 13, 12, DuelRank.BCD),
    Drop(144, 13, 12, DuelRank.BCD),
    Drop(145, 13, 12, DuelRank.BCD),
    Drop(147, 13, 12, DuelRank.BCD),
    Drop(148, 13, 12, DuelRank.BCD),
    Drop(152, 13, 44, DuelRank.BCD),
    Drop(154, 13, 12, DuelRank.BCD),
    Drop(156, 13, 12, DuelRank.BCD),
    Drop(157, 13, 12, DuelRank.BCD),
    Drop(158, 13, 12, DuelRank.BCD),
    Drop(159, 13, 12, DuelRank.BCD),
    Drop(160, 13, 12, DuelRank.BCD),
    Drop(161, 13, 12, DuelRank.BCD),
    Drop(173, 13, 12, DuelRank.BCD),
    Drop(176, 13, 12, DuelRank.BCD),
    Drop(177, 13, 24, DuelRank.BCD),
    Drop(179, 13, 12, DuelRank.BCD),
    Drop(182, 13, 12, DuelRank.BCD),
    Drop(183, 13, 12, DuelRank.BCD),
    Drop(184, 13, 12, DuelRank.BCD),
    Drop(187, 13, 12, DuelRank.BCD),
    Drop(189, 13, 12, DuelRank.BCD),
    Drop(191, 13, 12, DuelRank.BCD),
    Drop(195, 13, 12, DuelRank.BCD),
    Drop(198, 13, 12, DuelRank.BCD),
    Drop(199, 13, 20, DuelRank.BCD),
    Drop(200, 13, 12, DuelRank.BCD),
    Drop(203, 13, 12, DuelRank.BCD),
    Drop(205, 13, 20, DuelRank.BCD),
    Drop(206, 13, 44, DuelRank.BCD),
    Drop(208, 13, 12, DuelRank.BCD),
    Drop(209, 13, 12, DuelRank.BCD),
    Drop(211, 13, 12, DuelRank.BCD),
    Drop(212, 13, 12, DuelRank.BCD),
    Drop(214, 13, 12, DuelRank.BCD),
    Drop(218, 13, 12, DuelRank.BCD),
    Drop(222, 13, 12, DuelRank.BCD),
    Drop(224, 13, 12, DuelRank.BCD),
    Drop(227, 13, 44, DuelRank.BCD),
    Drop(229, 13, 12, DuelRank.BCD),
    Drop(232, 13, 12, DuelRank.BCD),
    Drop(239, 13, 12, DuelRank.BCD),
    Drop(240, 13, 12, DuelRank.BCD),
    Drop(242, 13, 12, DuelRank.BCD),
    Drop(243, 13, 44, DuelRank.BCD),
    Drop(245, 13, 12, DuelRank.BCD),
    Drop(247, 13, 22, DuelRank.BCD),
    Drop(254, 13, 12, DuelRank.BCD),
    Drop(261, 13, 12, DuelRank.BCD),
    Drop(265, 13, 44, DuelRank.BCD),
    Drop(266, 13, 12, DuelRank.BCD),
    Drop(267, 13, 12, DuelRank.BCD),
    Drop(268, 13, 12, DuelRank.BCD),
    Drop(270, 13, 28, DuelRank.BCD),
    Drop(271, 13, 12, DuelRank.BCD),
    Drop(277, 13, 12, DuelRank.BCD),
    Drop(285, 13, 44, DuelRank.BCD),
    Drop(289, 13, 20, DuelRank.BCD),
    Drop(292, 13, 44, DuelRank.BCD),
    Drop(300, 13, 12, DuelRank.BCD),
    Drop(308, 13, 20, DuelRank.BCD),
    Drop(328, 13, 20, DuelRank.BCD),
    Drop(334, 13, 44, DuelRank.BCD),
    Drop(393, 13, 20, DuelRank.BCD),
    Drop(398, 13, 22, DuelRank.BCD),
    Drop(420, 13, 12, DuelRank.BCD),
    Drop(421, 13, 12, DuelRank.BCD),
    Drop(432, 13, 22, DuelRank.BCD),
    Drop(436, 13, 22, DuelRank.BCD),
    Drop(444, 13, 22, DuelRank.BCD),
    Drop(446, 13, 22, DuelRank.BCD),
    Drop(451, 13, 44, DuelRank.BCD),
    Drop(452, 13, 44, DuelRank.BCD),
    Drop(475, 13, 22, DuelRank.BCD),
    Drop(476, 13, 12, DuelRank.BCD),
    Drop(484, 13, 22, DuelRank.BCD),
    Drop(490, 13, 12, DuelRank.BCD),
    Drop(492, 13, 12, DuelRank.BCD),
    Drop(503, 13, 44, DuelRank.BCD),
    Drop(506, 13, 12, DuelRank.BCD),
    Drop(510, 13, 12, DuelRank.BCD),
    Drop(524, 13, 22, DuelRank.BCD),
    Drop(534, 13, 12, DuelRank.BCD),
    Drop(536, 13, 44, DuelRank.BCD),
    Drop(540, 13, 12, DuelRank.BCD),
    Drop(544, 13, 12, DuelRank.BCD),
    Drop(549, 13, 44, DuelRank.BCD),
    Drop(556, 13, 12, DuelRank.BCD),
    Drop(569, 13, 22, DuelRank.BCD),
    Drop(590, 13, 22, DuelRank.BCD),
    Drop(591, 13, 12, DuelRank.BCD),
    Drop(602, 13, 44, DuelRank.BCD),
    Drop(604, 13, 12, DuelRank.BCD),
    Drop(605, 13, 22, DuelRank.BCD),
    Drop(606, 13, 44, DuelRank.BCD),
    Drop(609, 13, 12, DuelRank.BCD),
    Drop(682, 13, 20, DuelRank.BCD),
    Drop(16, 14, 48, DuelRank.BCD),
    Drop(20, 14, 6, DuelRank.BCD),
    Drop(34, 14, 40, DuelRank.BCD),
    Drop(104, 14, 48, DuelRank.BCD),
    Drop(109, 14, 44, DuelRank.BCD),
    Drop(114, 14, 48, DuelRank.BCD),
    Drop(115, 14, 40, DuelRank.BCD),
    Drop(128, 14, 10, DuelRank.BCD),
    Drop(129, 14, 48, DuelRank.BCD),
    Drop(130, 14, 48, DuelRank.BCD),
    Drop(134, 14, 48, DuelRank.BCD),
    Drop(142, 14, 48, DuelRank.BCD),
    Drop(143, 14, 48, DuelRank.BCD),
    Drop(144, 14, 48, DuelRank.BCD),
    Drop(145, 14, 48, DuelRank.BCD),
    Drop(174, 14, 48, DuelRank.BCD),
    Drop(179, 14, 48, DuelRank.BCD),
    Drop(183, 14, 48, DuelRank.BCD),
    Drop(184, 14, 48, DuelRank.BCD),
    Drop(190, 14, 48, DuelRank.BCD),
    Drop(192, 14, 48, DuelRank.BCD),
    Drop(198, 14, 48, DuelRank.BCD),
    Drop(208, 14, 48, DuelRank.BCD),
    Drop(216, 14, 10, DuelRank.BCD),
    Drop(220, 14, 40, DuelRank.BCD),
    Drop(229, 14, 48, DuelRank.BCD),
    Drop(253, 14, 48, DuelRank.BCD),
    Drop(259, 14, 44, DuelRank.BCD),
    Drop(260, 14, 48, DuelRank.BCD),
    Drop(264, 14, 48, DuelRank.BCD),
    Drop(268, 14, 48, DuelRank.BCD),
    Drop(276, 14, 48, DuelRank.BCD),
    Drop(287, 14, 10, DuelRank.BCD),
    Drop(311, 14, 32, DuelRank.BCD),
    Drop(323, 14, 32, DuelRank.BCD),
    Drop(334, 14, 64, DuelRank.BCD),
    Drop(350, 14, 32, DuelRank.BCD),
    Drop(387, 14, 48, DuelRank.BCD),
    Drop(395, 14, 48, DuelRank.BCD),
    Drop(433, 14, 10, DuelRank.BCD),
    Drop(486, 14, 48, DuelRank.BCD),
    Drop(492, 14, 48, DuelRank.BCD),
    Drop(540, 14, 48, DuelRank.BCD),
    Drop(578, 14, 10, DuelRank.BCD),
    Drop(584, 14, 44, DuelRank.BCD),
    Drop(587, 14, 10, DuelRank.BCD),
    Drop(601, 14, 44, DuelRank.BCD),
    Drop(608, 14, 44, DuelRank.BCD),
    Drop(612, 14, 48, DuelRank.BCD),
    Drop(630, 14, 10, DuelRank.BCD),
    Drop(684, 14, 32, DuelRank.BCD),
    Drop(3, 15, 44, DuelRank.BCD),
    Drop(4, 15, 40, DuelRank.BCD),
    Drop(10, 15, 44, DuelRank.BCD),
    Drop(19, 15, 12, DuelRank.BCD),
    Drop(23, 15, 44, DuelRank.BCD),
    Drop(29, 15, 40, DuelRank.BCD),
    Drop(40, 15, 80, DuelRank.BCD),
    Drop(45, 15, 80, DuelRank.BCD),
    Drop(47, 15, 38, DuelRank.BCD),
    Drop(61, 15, 38, DuelRank.BCD),
    Drop(65, 15, 38, DuelRank.BCD),
    Drop(113, 15, 38, DuelRank.BCD),
    Drop(117, 15, 12, DuelRank.BCD),
    Drop(121, 15, 40, DuelRank.BCD),
    Drop(122, 15, 38, DuelRank.BCD),
    Drop(155, 15, 38, DuelRank.BCD),
    Drop(159, 15, 38, DuelRank.BCD),
    Drop(186, 15, 40, DuelRank.BCD),
    Drop(187, 15, 38, DuelRank.BCD),
    Drop(188, 15, 38, DuelRank.BCD),
    Drop(189, 15, 38, DuelRank.BCD),
    Drop(200, 15, 38, DuelRank.BCD),
    Drop(201, 15, 40, DuelRank.BCD),
    Drop(202, 15, 38, DuelRank.BCD),
    Drop(207, 15, 38, DuelRank.BCD),
    Drop(212, 15, 38, DuelRank.BCD),
    Drop(219, 15, 38, DuelRank.BCD),
    Drop(246, 15, 40, DuelRank.BCD),
    Drop(248, 15, 40, DuelRank.BCD),
    Drop(255, 15, 38, DuelRank.BCD),
    Drop(272, 15, 38, DuelRank.BCD),
    Drop(282, 15, 38, DuelRank.BCD),
    Drop(296, 15, 38, DuelRank.BCD),
    Drop(298, 15, 38, DuelRank.BCD),
    Drop(300, 15, 38, DuelRank.BCD),
    Drop(315, 15, 32, DuelRank.BCD),
    Drop(325, 15, 32, DuelRank.BCD),
    Drop(330, 15, 64, DuelRank.BCD),
    Drop(334, 15, 64, DuelRank.BCD),
    Drop(404, 15, 12, DuelRank.BCD),
    Drop(481, 15, 38, DuelRank.BCD),
    Drop(496, 15, 38, DuelRank.BCD),
    Drop(527, 15, 38, DuelRank.BCD),
    Drop(538, 15, 38, DuelRank.BCD),
    Drop(552, 15, 38, DuelRank.BCD),
    Drop(595, 15, 12, DuelRank.BCD),
    Drop(598, 15, 38, DuelRank.BCD),
    Drop(604, 15, 38, DuelRank.BCD),
    Drop(629, 15, 38, DuelRank.BCD),
    Drop(642, 15, 38, DuelRank.BCD),
    Drop(652, 15, 32, DuelRank.BCD),
    Drop(685, 15, 64, DuelRank.BCD),
    Drop(4, 16, 64, DuelRank.BCD),
    Drop(10, 16, 64, DuelRank.BCD),
    Drop(20, 16, 10, DuelRank.BCD),
    Drop(122, 16, 64, DuelRank.BCD),
    Drop(140, 16, 64, DuelRank.BCD),
    Drop(152, 16, 64, DuelRank.BCD),
    Drop(177, 16, 64, DuelRank.BCD),
    Drop(193, 16, 64, DuelRank.BCD),
    Drop(196, 16, 64, DuelRank.BCD),
    Drop(199, 16, 64, DuelRank.BCD),
    Drop(200, 16, 64, DuelRank.BCD),
    Drop(205, 16, 58, DuelRank.BCD),
    Drop(206, 16, 58, DuelRank.BCD),
    Drop(227, 16, 58, DuelRank.BCD),
    Drop(243, 16, 58, DuelRank.BCD),
    Drop(247, 16, 58, DuelRank.BCD),
    Drop(249, 16, 10, DuelRank.BCD),
    Drop(251, 16, 58, DuelRank.BCD),
    Drop(258, 16, 58, DuelRank.BCD),
    Drop(265, 16, 58, DuelRank.BCD),
    Drop(270, 16, 58, DuelRank.BCD),
    Drop(289, 16, 58, DuelRank.BCD),
    Drop(292, 16, 58, DuelRank.BCD),
    Drop(296, 16, 58, DuelRank.BCD),
    Drop(298, 16, 58, DuelRank.BCD),
    Drop(307, 16, 32, DuelRank.BCD),
    Drop(315, 16, 32, DuelRank.BCD),
    Drop(331, 16, 32, DuelRank.BCD),
    Drop(334, 16, 32, DuelRank.BCD),
    Drop(393, 16, 58, DuelRank.BCD),
    Drop(430, 16, 88, DuelRank.BCD),
    Drop(431, 16, 88, DuelRank.BCD),
    Drop(448, 16, 10, DuelRank.BCD),
    Drop(454, 16, 10, DuelRank.BCD),
    Drop(456, 16, 10, DuelRank.BCD),
    Drop(517, 16, 10, DuelRank.BCD),
    Drop(530, 16, 48, DuelRank.BCD),
    Drop(550, 16, 48, DuelRank.BCD),
    Drop(615, 16, 46, DuelRank.BCD),
    Drop(621, 16, 10, DuelRank.BCD),
    Drop(649, 16, 48, DuelRank.BCD),
    Drop(686, 16, 32, DuelRank.BCD),
    Drop(3, 17, 174, DuelRank.BCD),
    Drop(4, 17, 86, DuelRank.BCD),
    Drop(5, 17, 174, DuelRank.BCD),
    Drop(10, 17, 86, DuelRank.BCD),
    Drop(19, 17, 12, DuelRank.BCD),
    Drop(23, 17, 174, DuelRank.BCD),
    Drop(34, 17, 174, DuelRank.BCD),
    Drop(66, 17, 12, DuelRank.BCD),
    Drop(77, 17, 174, DuelRank.BCD),
    Drop(117, 17, 12, DuelRank.BCD),
    Drop(122, 17, 84, DuelRank.BCD),
    Drop(128, 17, 13, DuelRank.BCD),
    Drop(151, 17, 13, DuelRank.BCD),
    Drop(163, 17, 13, DuelRank.BCD),
    Drop(200, 17, 84, DuelRank.BCD),
    Drop(216, 17, 13, DuelRank.BCD),
    Drop(249, 17, 13, DuelRank.BCD),
    Drop(275, 17, 13, DuelRank.BCD),
    Drop(281, 17, 13, DuelRank.BCD),
    Drop(287, 17, 13, DuelRank.BCD),
    Drop(296, 17, 84, DuelRank.BCD),
    Drop(297, 17, 13, DuelRank.BCD),
    Drop(298, 17, 84, DuelRank.BCD),
    Drop(302, 17, 48, DuelRank.BCD),
    Drop(329, 17, 48, DuelRank.BCD),
    Drop(333, 17, 84, DuelRank.BCD),
    Drop(404, 17, 13, DuelRank.BCD),
    Drop(418, 17, 13, DuelRank.BCD),
    Drop(448, 17, 13, DuelRank.BCD),
    Drop(454, 17, 13, DuelRank.BCD),
    Drop(456, 17, 13, DuelRank.BCD),
    Drop(498, 17, 13, DuelRank.BCD),
    Drop(517, 17, 13, DuelRank.BCD),
    Drop(578, 17, 13, DuelRank.BCD),
    Drop(587, 17, 13, DuelRank.BCD),
    Drop(595, 17, 13, DuelRank.BCD),
    Drop(614, 17, 13, DuelRank.BCD),
    Drop(621, 17, 13, DuelRank.BCD),
    Drop(630, 17, 13, DuelRank.BCD),
    Drop(685, 17, 84, DuelRank.BCD),
    Drop(687, 17, 84, DuelRank.BCD),
    Drop(5, 18, 26, DuelRank.BCD),
    Drop(6, 18, 52, DuelRank.BCD),
    Drop(13, 18, 28, DuelRank.BCD),
    Drop(20, 18, 8, DuelRank.BCD),
    Drop(25, 18, 26, DuelRank.BCD),
    Drop(26, 18, 26, DuelRank.BCD),
    Drop(27, 18, 26, DuelRank.BCD),
    Drop(31, 18, 26, DuelRank.BCD),
    Drop(36, 18, 26, DuelRank.BCD),
    Drop(41, 18, 26, DuelRank.BCD),
    Drop(43, 18, 26, DuelRank.BCD),
    Drop(44, 18, 26, DuelRank.BCD),
    Drop(46, 18, 26, DuelRank.BCD),
    Drop(48, 18, 26, DuelRank.BCD),
    Drop(49, 18, 26, DuelRank.BCD),
    Drop(54, 18, 26, DuelRank.BCD),
    Drop(55, 18, 26, DuelRank.BCD),
    Drop(58, 18, 26, DuelRank.BCD),
    Drop(70, 18, 26, DuelRank.BCD),
    Drop(71, 18, 26, DuelRank.BCD),
    Drop(77, 18, 26, DuelRank.BCD),
    Drop(95, 18, 52, DuelRank.BCD),
    Drop(103, 18, 26, DuelRank.BCD),
    Drop(115, 18, 26, DuelRank.BCD),
    Drop(119, 18, 26, DuelRank.BCD),
    Drop(121, 18, 26, DuelRank.BCD),
    Drop(125, 18, 26, DuelRank.BCD),
    Drop(136, 18, 48, DuelRank.BCD),
    Drop(137, 18, 24, DuelRank.BCD),
    Drop(148, 18, 24, DuelRank.BCD),
    Drop(149, 18, 48, DuelRank.BCD),
    Drop(162, 18, 24, DuelRank.BCD),
    Drop(164, 18, 24, DuelRank.BCD),
    Drop(166, 18, 24, DuelRank.BCD),
    Drop(168, 18, 24, DuelRank.BCD),
    Drop(169, 18, 24, DuelRank.BCD),
    Drop(171, 18, 24, DuelRank.BCD),
    Drop(173, 18, 24, DuelRank.BCD),
    Drop(175, 18, 24, DuelRank.BCD),
    Drop(178, 18, 24, DuelRank.BCD),
    Drop(181, 18, 24, DuelRank.BCD),
    Drop(186, 18, 24, DuelRank.BCD),
    Drop(193, 18, 24, DuelRank.BCD),
    Drop(194, 18, 48, DuelRank.BCD),
    Drop(222, 18, 24, DuelRank.BCD),
    Drop(230, 18, 24, DuelRank.BCD),
    Drop(232, 18, 24, DuelRank.BCD),
    Drop(233, 18, 48, DuelRank.BCD),
    Drop(240, 18, 24, DuelRank.BCD),
    Drop(242, 18, 24, DuelRank.BCD),
    Drop(245, 18, 24, DuelRank.BCD),
    Drop(254, 18, 24, DuelRank.BCD),
    Drop(255, 18, 24, DuelRank.BCD),
    Drop(261, 18, 24, DuelRank.BCD),
    Drop(269, 18, 24, DuelRank.BCD),
    Drop(271, 18, 24, DuelRank.BCD),
    Drop(277, 18, 24, DuelRank.BCD),
    Drop(279, 18, 24, DuelRank.BCD),
    Drop(281, 18, 8, DuelRank.BCD),
    Drop(295, 18, 24, DuelRank.BCD),
    Drop(303, 18, 50, DuelRank.BCD),
    Drop(335, 18, 50, DuelRank.BCD),
    Drop(402, 18, 24, DuelRank.BCD),
    Drop(412, 18, 24, DuelRank.BCD),
    Drop(415, 18, 24, DuelRank.BCD),
    Drop(416, 18, 24, DuelRank.BCD),
    Drop(459, 18, 24, DuelRank.BCD),
    Drop(466, 18, 24, DuelRank.BCD),
    Drop(490, 18, 24, DuelRank.BCD),
    Drop(504, 18, 24, DuelRank.BCD),
    Drop(511, 18, 24, DuelRank.BCD),
    Drop(512, 18, 24, DuelRank.BCD),
    Drop(563, 18, 24, DuelRank.BCD),
    Drop(597, 18, 24, DuelRank.BCD),
    Drop(608, 18, 26, DuelRank.BCD),
    Drop(611, 18, 24, DuelRank.BCD),
    Drop(4, 19, 23, DuelRank.BCD),
    Drop(20, 19, 6, DuelRank.BCD),
    Drop(29, 19, 23, DuelRank.BCD),
    Drop(43, 19, 23, DuelRank.BCD),
    Drop(50, 19, 46, DuelRank.BCD),
    Drop(61, 19, 23, DuelRank.BCD),
    Drop(78, 19, 23, DuelRank.BCD),
    Drop(100, 19, 23, DuelRank.BCD),
    Drop(104, 19, 46, DuelRank.BCD),
    Drop(105, 19, 46, DuelRank.BCD),
    Drop(122, 19, 46, DuelRank.BCD),
    Drop(130, 19, 46, DuelRank.BCD),
    Drop(135, 19, 46, DuelRank.BCD),
    Drop(137, 19, 46, DuelRank.BCD),
    Drop(152, 19, 46, DuelRank.BCD),
    Drop(154, 19, 46, DuelRank.BCD),
    Drop(174, 19, 46, DuelRank.BCD),
    Drop(182, 19, 46, DuelRank.BCD),
    Drop(185, 19, 46, DuelRank.BCD),
    Drop(191, 19, 46, DuelRank.BCD),
    Drop(197, 19, 46, DuelRank.BCD),
    Drop(202, 19, 46, DuelRank.BCD),
    Drop(207, 19, 46, DuelRank.BCD),
    Drop(210, 19, 46, DuelRank.BCD),
    Drop(214, 19, 46, DuelRank.BCD),
    Drop(237, 19, 46, DuelRank.BCD),
    Drop(242, 19, 44, DuelRank.BCD),
    Drop(245, 19, 44, DuelRank.BCD),
    Drop(268, 19, 44, DuelRank.BCD),
    Drop(333, 19, 22, DuelRank.BCD),
    Drop(344, 19, 20, DuelRank.BCD),
    Drop(410, 19, 44, DuelRank.BCD),
    Drop(420, 19, 44, DuelRank.BCD),
    Drop(422, 19, 44, DuelRank.BCD),
    Drop(436, 19, 44, DuelRank.BCD),
    Drop(444, 19, 44, DuelRank.BCD),
    Drop(485, 19, 44, DuelRank.BCD),
    Drop(486, 19, 44, DuelRank.BCD),
    Drop(488, 19, 44, DuelRank.BCD),
    Drop(492, 19, 44, DuelRank.BCD),
    Drop(501, 19, 44, DuelRank.BCD),
    Drop(516, 19, 44, DuelRank.BCD),
    Drop(524, 19, 44, DuelRank.BCD),
    Drop(549, 19, 44, DuelRank.BCD),
    Drop(558, 19, 44, DuelRank.BCD),
    Drop(563, 19, 44, DuelRank.BCD),
    Drop(579, 19, 44, DuelRank.BCD),
    Drop(589, 19, 44, DuelRank.BCD),
    Drop(598, 19, 44, DuelRank.BCD),
    Drop(611, 19, 44, DuelRank.BCD),
    Drop(651, 19, 20, DuelRank.BCD),
    Drop(19, 20, 8, DuelRank.BCD),
    Drop(24, 20, 182, DuelRank.BCD),
    Drop(58, 20, 182, DuelRank.BCD),
    Drop(167, 20, 182, DuelRank.BCD),
    Drop(330, 20, 38, DuelRank.BCD),
    Drop(393, 20, 182, DuelRank.BCD),
    Drop(395, 20, 182, DuelRank.BCD),
    Drop(398, 20, 182, DuelRank.BCD),
    Drop(399, 20, 182, DuelRank.BCD),
    Drop(402, 20, 182, DuelRank.BCD),
    Drop(469, 20, 182, DuelRank.BCD),
    Drop(475, 20, 182, DuelRank.BCD),
    Drop(527, 20, 182, DuelRank.BCD),
    Drop(5, 21, 12, DuelRank.BCD),
    Drop(8, 21, 12, DuelRank.BCD),
    Drop(20, 21, 6, DuelRank.BCD),
    Drop(29, 21, 12, DuelRank.BCD),
    Drop(45, 21, 12, DuelRank.BCD),
    Drop(48, 21, 12, DuelRank.BCD),
    Drop(50, 21, 12, DuelRank.BCD),
    Drop(70, 21, 56, DuelRank.BCD),
    Drop(71, 21, 56, DuelRank.BCD),
    Drop(77, 21, 24, DuelRank.BCD),
    Drop(96, 21, 12, DuelRank.BCD),
    Drop(97, 21, 12, DuelRank.BCD),
    Drop(98, 21, 12, DuelRank.BCD),
    Drop(101, 21, 12, DuelRank.BCD),
    Drop(107, 21, 12, DuelRank.BCD),
    Drop(108, 21, 12, DuelRank.BCD),
    Drop(122, 21, 12, DuelRank.BCD),
    Drop(129, 21, 12, DuelRank.BCD),
    Drop(134, 21, 12, DuelRank.BCD),
    Drop(135, 21, 12, DuelRank.BCD),
    Drop(139, 21, 12, DuelRank.BCD),
    Drop(140, 21, 24, DuelRank.BCD),
    Drop(144, 21, 12, DuelRank.BCD),
    Drop(145, 21, 12, DuelRank.BCD),
    Drop(147, 21, 12, DuelRank.BCD),
    Drop(148, 21, 12, DuelRank.BCD),
    Drop(152, 21, 38, DuelRank.BCD),
    Drop(154, 21, 12, DuelRank.BCD),
    Drop(156, 21, 12, DuelRank.BCD),
    Drop(157, 21, 12, DuelRank.BCD),
    Drop(158, 21, 12, DuelRank.BCD),
    Drop(159, 21, 12, DuelRank.BCD),
    Drop(160, 21, 12, DuelRank.BCD),
    Drop(161, 21, 12, DuelRank.BCD),
    Drop(173, 21, 12, DuelRank.BCD),
    Drop(176, 21, 12, DuelRank.BCD),
    Drop(177, 21, 24, DuelRank.BCD),
    Drop(179, 21, 12, DuelRank.BCD),
    Drop(182, 21, 12, DuelRank.BCD),
    Drop(183, 21, 12, DuelRank.BCD),
    Drop(184, 21, 12, DuelRank.BCD),
    Drop(187, 21, 12, DuelRank.BCD),
    Drop(189, 21, 12, DuelRank.BCD),
    Drop(191, 21, 12, DuelRank.BCD),
    Drop(193, 21, 24, DuelRank.BCD),
    Drop(195, 21, 12, DuelRank.BCD),
    Drop(196, 21, 24, DuelRank.BCD),
    Drop(198, 21, 12, DuelRank.BCD),
    Drop(199, 21, 24, DuelRank.BCD),
    Drop(200, 21, 12, DuelRank.BCD),
    Drop(203, 21, 12, DuelRank.BCD),
    Drop(205, 21, 24, DuelRank.BCD),
    Drop(206, 21, 38, DuelRank.BCD),
    Drop(208, 21, 12, DuelRank.BCD),
    Drop(209, 21, 12, DuelRank.BCD),
    Drop(211, 21, 12, DuelRank.BCD),
    Drop(212, 21, 12, DuelRank.BCD),
    Drop(214, 21, 12, DuelRank.BCD),
    Drop(218, 21, 12, DuelRank.BCD),
    Drop(222, 21, 12, DuelRank.BCD),
    Drop(224, 21, 12, DuelRank.BCD),
    Drop(227, 21, 38, DuelRank.BCD),
    Drop(229, 21, 12, DuelRank.BCD),
    Drop(230, 21, 24, DuelRank.BCD),
    Drop(232, 21, 12, DuelRank.BCD),
    Drop(239, 21, 12, DuelRank.BCD),
    Drop(240, 21, 12, DuelRank.BCD),
    Drop(242, 21, 12, DuelRank.BCD),
    Drop(243, 21, 40, DuelRank.BCD),
    Drop(245, 21, 12, DuelRank.BCD),
    Drop(247, 21, 24, DuelRank.BCD),
    Drop(251, 21, 24, DuelRank.BCD),
    Drop(254, 21, 12, DuelRank.BCD),
    Drop(258, 21, 24, DuelRank.BCD),
    Drop(261, 21, 12, DuelRank.BCD),
    Drop(265, 21, 40, DuelRank.BCD),
    Drop(266, 21, 12, DuelRank.BCD),
    Drop(267, 21, 12, DuelRank.BCD),
    Drop(268, 21, 12, DuelRank.BCD),
    Drop(270, 21, 24, DuelRank.BCD),
    Drop(271, 21, 12, DuelRank.BCD),
    Drop(277, 21, 12, DuelRank.BCD),
    Drop(285, 21, 40, DuelRank.BCD),
    Drop(292, 21, 40, DuelRank.BCD),
    Drop(334, 21, 30, DuelRank.BCD),
    Drop(420, 21, 12, DuelRank.BCD),
    Drop(421, 21, 12, DuelRank.BCD),
    Drop(432, 21, 24, DuelRank.BCD),
    Drop(446, 21, 26, DuelRank.BCD),
    Drop(451, 21, 40, DuelRank.BCD),
    Drop(452, 21, 40, DuelRank.BCD),
    Drop(476, 21, 12, DuelRank.BCD),
    Drop(490, 21, 12, DuelRank.BCD),
    Drop(492, 21, 12, DuelRank.BCD),
    Drop(503, 21, 40, DuelRank.BCD),
    Drop(506, 21, 12, DuelRank.BCD),
    Drop(510, 21, 12, DuelRank.BCD),
    Drop(524, 21, 24, DuelRank.BCD),
    Drop(534, 21, 12, DuelRank.BCD),
    Drop(536, 21, 40, DuelRank.BCD),
    Drop(540, 21, 12, DuelRank.BCD),
    Drop(544, 21, 12, DuelRank.BCD),
    Drop(549, 21, 40, DuelRank.BCD),
    Drop(556, 21, 12, DuelRank.BCD),
    Drop(569, 21, 24, DuelRank.BCD),
    Drop(590, 21, 24, DuelRank.BCD),
    Drop(591, 21, 12, DuelRank.BCD),
    Drop(602, 21, 40, DuelRank.BCD),
    Drop(604, 21, 12, DuelRank.BCD),
    Drop(605, 21, 24, DuelRank.BCD),
    Drop(606, 21, 40, DuelRank.BCD),
    Drop(609, 21, 12, DuelRank.BCD),
    Drop(2, 22, 33, DuelRank.BCD),
    Drop(14, 22, 35, DuelRank.BCD),
    Drop(20, 22, 7, DuelRank.BCD),
    Drop(34, 22, 33, DuelRank.BCD),
    Drop(44, 22, 33, DuelRank.BCD),
    Drop(87, 22, 35, DuelRank.BCD),
    Drop(89, 22, 35, DuelRank.BCD),
    Drop(93, 22, 35, DuelRank.BCD),
    Drop(104, 22, 33, DuelRank.BCD),
    Drop(106, 22, 76, DuelRank.BCD),
    Drop(109, 22, 33, DuelRank.BCD),
    Drop(114, 22, 33, DuelRank.BCD),
    Drop(115, 22, 33, DuelRank.BCD),
    Drop(124, 22, 35, DuelRank.BCD),
    Drop(129, 22, 33, DuelRank.BCD),
    Drop(130, 22, 33, DuelRank.BCD),
    Drop(131, 22, 35, DuelRank.BCD),
    Drop(134, 22, 33, DuelRank.BCD),
    Drop(142, 22, 33, DuelRank.BCD),
    Drop(143, 22, 33, DuelRank.BCD),
    Drop(144, 22, 33, DuelRank.BCD),
    Drop(145, 22, 33, DuelRank.BCD),
    Drop(150, 22, 35, DuelRank.BCD),
    Drop(170, 22, 33, DuelRank.BCD),
    Drop(174, 22, 33, DuelRank.BCD),
    Drop(179, 22, 33, DuelRank.BCD),
    Drop(183, 22, 33, DuelRank.BCD),
    Drop(184, 22, 33, DuelRank.BCD),
    Drop(190, 22, 33, DuelRank.BCD),
    Drop(198, 22, 33, DuelRank.BCD),
    Drop(204, 22, 35, DuelRank.BCD),
    Drop(208, 22, 33, DuelRank.BCD),
    Drop(213, 22, 76, DuelRank.BCD),
    Drop(220, 22, 33, DuelRank.BCD),
    Drop(229, 22, 33, DuelRank.BCD),
    Drop(253, 22, 33, DuelRank.BCD),
    Drop(259, 22, 33, DuelRank.BCD),
    Drop(260, 22, 33, DuelRank.BCD),
    Drop(264, 22, 33, DuelRank.BCD),
    Drop(268, 22, 33, DuelRank.BCD),
    Drop(276, 22, 33, DuelRank.BCD),
    Drop(334, 22, 48, DuelRank.BCD),
    Drop(366, 22, 33, DuelRank.BCD),
    Drop(413, 22, 33, DuelRank.BCD),
    Drop(425, 22, 33, DuelRank.BCD),
    Drop(468, 22, 33, DuelRank.BCD),
    Drop(486, 22, 33, DuelRank.BCD),
    Drop(491, 22, 33, DuelRank.BCD),
    Drop(492, 22, 33, DuelRank.BCD),
    Drop(508, 22, 33, DuelRank.BCD),
    Drop(513, 22, 33, DuelRank.BCD),
    Drop(540, 22, 33, DuelRank.BCD),
    Drop(584, 22, 33, DuelRank.BCD),
    Drop(601, 22, 33, DuelRank.BCD),
    Drop(608, 22, 33, DuelRank.BCD),
    Drop(612, 22, 33, DuelRank.BCD),
    Drop(633, 22, 76, DuelRank.BCD),
    Drop(650, 22, 33, DuelRank.BCD),
    Drop(8, 23, 42, DuelRank.BCD),
    Drop(19, 23, 6, DuelRank.BCD),
    Drop(40, 23, 22, DuelRank.BCD),
    Drop(50, 23, 22, DuelRank.BCD),
    Drop(75, 23, 117, DuelRank.BCD),
    Drop(76, 23, 42, DuelRank.BCD),
    Drop(98, 23, 22, DuelRank.BCD),
    Drop(101, 23, 42, DuelRank.BCD),
    Drop(104, 23, 22, DuelRank.BCD),
    Drop(107, 23, 22, DuelRank.BCD),
    Drop(122, 23, 22, DuelRank.BCD),
    Drop(130, 23, 22, DuelRank.BCD),
    Drop(133, 23, 22, DuelRank.BCD),
    Drop(134, 23, 22, DuelRank.BCD),
    Drop(135, 23, 22, DuelRank.BCD),
    Drop(145, 23, 22, DuelRank.BCD),
    Drop(152, 23, 22, DuelRank.BCD),
    Drop(154, 23, 22, DuelRank.BCD),
    Drop(157, 23, 42, DuelRank.BCD),
    Drop(158, 23, 42, DuelRank.BCD),
    Drop(159, 23, 22, DuelRank.BCD),
    Drop(174, 23, 22, DuelRank.BCD),
    Drop(176, 23, 42, DuelRank.BCD),
    Drop(179, 23, 22, DuelRank.BCD),
    Drop(180, 23, 22, DuelRank.BCD),
    Drop(182, 23, 22, DuelRank.BCD),
    Drop(183, 23, 22, DuelRank.BCD),
    Drop(185, 23, 22, DuelRank.BCD),
    Drop(191, 23, 22, DuelRank.BCD),
    Drop(195, 23, 22, DuelRank.BCD),
    Drop(198, 23, 22, DuelRank.BCD),
    Drop(200, 23, 22, DuelRank.BCD),
    Drop(202, 23, 22, DuelRank.BCD),
    Drop(203, 23, 22, DuelRank.BCD),
    Drop(207, 23, 22, DuelRank.BCD),
    Drop(209, 23, 22, DuelRank.BCD),
    Drop(210, 23, 42, DuelRank.BCD),
    Drop(211, 23, 22, DuelRank.BCD),
    Drop(212, 23, 22, DuelRank.BCD),
    Drop(214, 23, 22, DuelRank.BCD),
    Drop(222, 23, 22, DuelRank.BCD),
    Drop(227, 23, 22, DuelRank.BCD),
    Drop(229, 23, 22, DuelRank.BCD),
    Drop(232, 23, 22, DuelRank.BCD),
    Drop(237, 23, 22, DuelRank.BCD),
    Drop(242, 23, 22, DuelRank.BCD),
    Drop(245, 23, 22, DuelRank.BCD),
    Drop(254, 23, 22, DuelRank.BCD),
    Drop(261, 23, 22, DuelRank.BCD),
    Drop(268, 23, 22, DuelRank.BCD),
    Drop(273, 23, 22, DuelRank.BCD),
    Drop(274, 23, 22, DuelRank.BCD),
    Drop(285, 23, 22, DuelRank.BCD),
    Drop(291, 23, 22, DuelRank.BCD),
    Drop(292, 23, 22, DuelRank.BCD),
    Drop(330, 23, 30, DuelRank.BCD),
    Drop(377, 23, 1, DuelRank.BCD),
    Drop(420, 23, 22, DuelRank.BCD),
    Drop(452, 23, 22, DuelRank.BCD),
    Drop(476, 23, 22, DuelRank.BCD),
    Drop(486, 23, 22, DuelRank.BCD),
    Drop(492, 23, 22, DuelRank.BCD),
    Drop(501, 23, 22, DuelRank.BCD),
    Drop(506, 23, 22, DuelRank.BCD),
    Drop(510, 23, 42, DuelRank.BCD),
    Drop(511, 23, 22, DuelRank.BCD),
    Drop(524, 23, 22, DuelRank.BCD),
    Drop(536, 23, 22, DuelRank.BCD),
    Drop(540, 23, 22, DuelRank.BCD),
    Drop(544, 23, 22, DuelRank.BCD),
    Drop(549, 23, 22, DuelRank.BCD),
    Drop(556, 23, 22, DuelRank.BCD),
    Drop(567, 23, 22, DuelRank.BCD),
    Drop(579, 23, 42, DuelRank.BCD),
    Drop(589, 23, 42, DuelRank.BCD),
    Drop(591, 23, 22, DuelRank.BCD),
    Drop(598, 23, 22, DuelRank.BCD),
    Drop(602, 23, 22, DuelRank.BCD),
    Drop(609, 23, 22, DuelRank.BCD),
    Drop(611, 23, 22, DuelRank.BCD),
    Drop(620, 23, 22, DuelRank.BCD),
    Drop(3, 24, 24, DuelRank.BCD),
    Drop(8, 24, 24, DuelRank.BCD),
    Drop(10, 24, 24, DuelRank.BCD),
    Drop(13, 24, 24, DuelRank.BCD),
    Drop(19, 24, 6, DuelRank.BCD),
    Drop(25, 24, 24, DuelRank.BCD),
    Drop(41, 24, 24, DuelRank.BCD),
    Drop(44, 24, 24, DuelRank.BCD),
    Drop(49, 24, 72, DuelRank.BCD),
    Drop(50, 24, 72, DuelRank.BCD),
    Drop(53, 24, 102, DuelRank.BCD),
    Drop(54, 24, 102, DuelRank.BCD),
    Drop(55, 24, 72, DuelRank.BCD),
    Drop(70, 24, 26, DuelRank.BCD),
    Drop(77, 24, 26, DuelRank.BCD),
    Drop(80, 24, 26, DuelRank.BCD),
    Drop(109, 24, 26, DuelRank.BCD),
    Drop(115, 24, 26, DuelRank.BCD),
    Drop(116, 24, 26, DuelRank.BCD),
    Drop(119, 24, 26, DuelRank.BCD),
    Drop(121, 24, 26, DuelRank.BCD),
    Drop(125, 24, 22, DuelRank.BCD),
    Drop(141, 24, 22, DuelRank.BCD),
    Drop(149, 24, 22, DuelRank.BCD),
    Drop(157, 24, 22, DuelRank.BCD),
    Drop(158, 24, 22, DuelRank.BCD),
    Drop(164, 24, 22, DuelRank.BCD),
    Drop(166, 24, 22, DuelRank.BCD),
    Drop(171, 24, 22, DuelRank.BCD),
    Drop(180, 24, 22, DuelRank.BCD),
    Drop(186, 24, 22, DuelRank.BCD),
    Drop(193, 24, 22, DuelRank.BCD),
    Drop(194, 24, 22, DuelRank.BCD),
    Drop(196, 24, 22, DuelRank.BCD),
    Drop(201, 24, 22, DuelRank.BCD),
    Drop(209, 24, 22, DuelRank.BCD),
    Drop(220, 24, 22, DuelRank.BCD),
    Drop(221, 24, 22, DuelRank.BCD),
    Drop(225, 24, 22, DuelRank.BCD),
    Drop(233, 24, 22, DuelRank.BCD),
    Drop(236, 24, 22, DuelRank.BCD),
    Drop(248, 24, 22, DuelRank.BCD),
    Drop(255, 24, 22, DuelRank.BCD),
    Drop(256, 24, 22, DuelRank.BCD),
    Drop(259, 24, 22, DuelRank.BCD),
    Drop(272, 24, 22, DuelRank.BCD),
    Drop(273, 24, 22, DuelRank.BCD),
    Drop(274, 24, 22, DuelRank.BCD),
    Drop(280, 24, 22, DuelRank.BCD),
    Drop(291, 24, 22, DuelRank.BCD),
    Drop(293, 24, 22, DuelRank.BCD),
    Drop(330, 24, 36, DuelRank.BCD),
    Drop(367, 24, 2, DuelRank.BCD),
    Drop(406, 24, 22, DuelRank.BCD),
    Drop(412, 24, 22, DuelRank.BCD),
    Drop(459, 24, 22, DuelRank.BCD),
    Drop(476, 24, 22, DuelRank.BCD),
    Drop(477, 24, 22, DuelRank.BCD),
    Drop(478, 24, 22, DuelRank.BCD),
    Drop(479, 24, 22, DuelRank.BCD),
    Drop(496, 24, 22, DuelRank.BCD),
    Drop(501, 24, 22, DuelRank.BCD),
    Drop(506, 24, 22, DuelRank.BCD),
    Drop(510, 24, 22, DuelRank.BCD),
    Drop(511, 24, 22, DuelRank.BCD),
    Drop(514, 24, 22, DuelRank.BCD),
    Drop(534, 24, 22, DuelRank.BCD),
    Drop(552, 24, 22, DuelRank.BCD),
    Drop(567, 24, 22, DuelRank.BCD),
    Drop(576, 24, 22, DuelRank.BCD),
    Drop(579, 24, 22, DuelRank.BCD),
    Drop(589, 24, 22, DuelRank.BCD),
    Drop(608, 24, 22, DuelRank.BCD),
    Drop(609, 24, 22, DuelRank.BCD),
    Drop(620, 24, 42, DuelRank.BCD),
    Drop(638, 24, 22, DuelRank.BCD),
    Drop(643, 24, 22, DuelRank.BCD),
    Drop(9, 25, 72, DuelRank.BCD),
    Drop(20, 25, 6, DuelRank.BCD),
    Drop(24, 25, 72, DuelRank.BCD),
    Drop(58, 25, 72, DuelRank.BCD),
    Drop(94, 25, 10, DuelRank.BCD),
    Drop(111, 25, 10, DuelRank.BCD),
    Drop(112, 25, 10, DuelRank.BCD),
    Drop(123, 25, 72, DuelRank.BCD),
    Drop(146, 25, 10, DuelRank.BCD),
    Drop(153, 25, 10, DuelRank.BCD),
    Drop(165, 25, 12, DuelRank.BCD),
    Drop(167, 25, 74, DuelRank.BCD),
    Drop(192, 25, 74, DuelRank.BCD),
    Drop(234, 25, 12, DuelRank.BCD),
    Drop(241, 25, 12, DuelRank.BCD),
    Drop(289, 25, 74, DuelRank.BCD),
    Drop(332, 25, 32, DuelRank.BCD),
    Drop(384, 25, 12, DuelRank.BCD),
    Drop(387, 25, 74, DuelRank.BCD),
    Drop(393, 25, 74, DuelRank.BCD),
    Drop(396, 25, 10, DuelRank.BCD),
    Drop(397, 25, 74, DuelRank.BCD),
    Drop(398, 25, 74, DuelRank.BCD),
    Drop(399, 25, 74, DuelRank.BCD),
    Drop(402, 25, 74, DuelRank.BCD),
    Drop(408, 25, 10, DuelRank.BCD),
    Drop(409, 25, 10, DuelRank.BCD),
    Drop(411, 25, 74, DuelRank.BCD),
    Drop(435, 25, 12, DuelRank.BCD),
    Drop(437, 25, 10, DuelRank.BCD),
    Drop(439, 25, 10, DuelRank.BCD),
    Drop(450, 25, 12, DuelRank.BCD),
    Drop(458, 25, 10, DuelRank.BCD),
    Drop(469, 25, 74, DuelRank.BCD),
    Drop(473, 25, 10, DuelRank.BCD),
    Drop(475, 25, 74, DuelRank.BCD),
    Drop(480, 25, 12, DuelRank.BCD),
    Drop(482, 25, 10, DuelRank.BCD),
    Drop(484, 25, 72, DuelRank.BCD),
    Drop(487, 25, 10, DuelRank.BCD),
    Drop(502, 25, 12, DuelRank.BCD),
    Drop(504, 25, 72, DuelRank.BCD),
    Drop(521, 25, 10, DuelRank.BCD),
    Drop(527, 25, 72, DuelRank.BCD),
    Drop(533, 25, 10, DuelRank.BCD),
    Drop(542, 25, 10, DuelRank.BCD),
    Drop(547, 25, 72, DuelRank.BCD),
    Drop(548, 25, 72, DuelRank.BCD),
    Drop(557, 25, 10, DuelRank.BCD),
    Drop(558, 25, 72, DuelRank.BCD),
    Drop(570, 25, 12, DuelRank.BCD),
    Drop(580, 25, 12, DuelRank.BCD),
    Drop(618, 25, 10, DuelRank.BCD),
    Drop(635, 25, 72, DuelRank.BCD),
    Drop(637, 25, 10, DuelRank.BCD),
    Drop(659, 25, 2, DuelRank.BCD),
    Drop(712, 25, 6, DuelRank.BCD),
    Drop(714, 25, 2, DuelRank.BCD),
    Drop(2, 26, 30, DuelRank.BCD),
    Drop(6, 26, 30, DuelRank.BCD),
    Drop(13, 26, 85, DuelRank.BCD),
    Drop(20, 26, 5, DuelRank.BCD),
    Drop(26, 26, 30, DuelRank.BCD),
    Drop(27, 26, 30, DuelRank.BCD),
    Drop(31, 26, 30, DuelRank.BCD),
    Drop(32, 26, 30, DuelRank.BCD),
    Drop(36, 26, 30, DuelRank.BCD),
    Drop(41, 26, 30, DuelRank.BCD),
    Drop(43, 26, 30, DuelRank.BCD),
    Drop(44, 26, 30, DuelRank.BCD),
    Drop(46, 26, 30, DuelRank.BCD),
    Drop(49, 26, 30, DuelRank.BCD),
    Drop(54, 26, 30, DuelRank.BCD),
    Drop(55, 26, 30, DuelRank.BCD),
    Drop(70, 26, 30, DuelRank.BCD),
    Drop(71, 26, 30, DuelRank.BCD),
    Drop(78, 26, 30, DuelRank.BCD),
    Drop(81, 26, 27, DuelRank.BCD),
    Drop(83, 26, 27, DuelRank.BCD),
    Drop(86, 26, 27, DuelRank.BCD),
    Drop(89, 26, 27, DuelRank.BCD),
    Drop(91, 26, 27, DuelRank.BCD),
    Drop(95, 26, 27, DuelRank.BCD),
    Drop(125, 26, 84, DuelRank.BCD),
    Drop(127, 26, 27, DuelRank.BCD),
    Drop(131, 26, 27, DuelRank.BCD),
    Drop(136, 26, 27, DuelRank.BCD),
    Drop(149, 26, 27, DuelRank.BCD),
    Drop(168, 26, 27, DuelRank.BCD),
    Drop(170, 26, 27, DuelRank.BCD),
    Drop(186, 26, 56, DuelRank.BCD),
    Drop(207, 26, 56, DuelRank.BCD),
    Drop(230, 26, 27, DuelRank.BCD),
    Drop(272, 26, 56, DuelRank.BCD),
    Drop(332, 26, 40, DuelRank.BCD),
    Drop(366, 26, 27, DuelRank.BCD),
    Drop(368, 26, 27, DuelRank.BCD),
    Drop(379, 26, 27, DuelRank.BCD),
    Drop(405, 26, 27, DuelRank.BCD),
    Drop(412, 26, 27, DuelRank.BCD),
    Drop(415, 26, 27, DuelRank.BCD),
    Drop(416, 26, 27, DuelRank.BCD),
    Drop(460, 26, 27, DuelRank.BCD),
    Drop(462, 26, 27, DuelRank.BCD),
    Drop(464, 26, 58, DuelRank.BCD),
    Drop(466, 26, 84, DuelRank.BCD),
    Drop(468, 26, 56, DuelRank.BCD),
    Drop(491, 26, 58, DuelRank.BCD),
    Drop(511, 26, 27, DuelRank.BCD),
    Drop(512, 26, 27, DuelRank.BCD),
    Drop(538, 26, 56, DuelRank.BCD),
    Drop(552, 26, 56, DuelRank.BCD),
    Drop(597, 26, 27, DuelRank.BCD),
    Drop(636, 26, 86, DuelRank.BCD),
    Drop(648, 26, 27, DuelRank.BCD),
    Drop(5, 27, 18, DuelRank.BCD),
    Drop(8, 27, 18, DuelRank.BCD),
    Drop(19, 27, 6, DuelRank.BCD),
    Drop(29, 27, 18, DuelRank.BCD),
    Drop(32, 27, 70, DuelRank.BCD),
    Drop(45, 27, 18, DuelRank.BCD),
    Drop(47, 27, 18, DuelRank.BCD),
    Drop(48, 27, 18, DuelRank.BCD),
    Drop(59, 27, 36, DuelRank.BCD),
    Drop(80, 27, 64, DuelRank.BCD),
    Drop(81, 27, 64, DuelRank.BCD),
    Drop(96, 27, 18, DuelRank.BCD),
    Drop(97, 27, 18, DuelRank.BCD),
    Drop(100, 27, 18, DuelRank.BCD),
    Drop(107, 27, 18, DuelRank.BCD),
    Drop(108, 27, 18, DuelRank.BCD),
    Drop(113, 27, 18, DuelRank.BCD),
    Drop(114, 27, 18, DuelRank.BCD),
    Drop(116, 27, 18, DuelRank.BCD),
    Drop(129, 27, 18, DuelRank.BCD),
    Drop(134, 27, 18, DuelRank.BCD),
    Drop(139, 27, 18, DuelRank.BCD),
    Drop(143, 27, 18, DuelRank.BCD),
    Drop(144, 27, 18, DuelRank.BCD),
    Drop(145, 27, 18, DuelRank.BCD),
    Drop(147, 27, 18, DuelRank.BCD),
    Drop(148, 27, 18, DuelRank.BCD),
    Drop(155, 27, 18, DuelRank.BCD),
    Drop(156, 27, 18, DuelRank.BCD),
    Drop(158, 27, 18, DuelRank.BCD),
    Drop(160, 27, 18, DuelRank.BCD),
    Drop(161, 27, 18, DuelRank.BCD),
    Drop(173, 27, 18, DuelRank.BCD),
    Drop(175, 27, 18, DuelRank.BCD),
    Drop(176, 27, 18, DuelRank.BCD),
    Drop(177, 27, 18, DuelRank.BCD),
    Drop(178, 27, 18, DuelRank.BCD),
    Drop(180, 27, 18, DuelRank.BCD),
    Drop(183, 27, 18, DuelRank.BCD),
    Drop(184, 27, 18, DuelRank.BCD),
    Drop(187, 27, 18, DuelRank.BCD),
    Drop(188, 27, 18, DuelRank.BCD),
    Drop(189, 27, 18, DuelRank.BCD),
    Drop(190, 27, 18, DuelRank.BCD),
    Drop(195, 27, 18, DuelRank.BCD),
    Drop(199, 27, 18, DuelRank.BCD),
    Drop(203, 27, 18, DuelRank.BCD),
    Drop(205, 27, 18, DuelRank.BCD),
    Drop(206, 27, 18, DuelRank.BCD),
    Drop(208, 27, 18, DuelRank.BCD),
    Drop(209, 27, 18, DuelRank.BCD),
    Drop(215, 27, 18, DuelRank.BCD),
    Drop(218, 27, 48, DuelRank.BCD),
    Drop(222, 27, 18, DuelRank.BCD),
    Drop(224, 27, 18, DuelRank.BCD),
    Drop(226, 27, 18, DuelRank.BCD),
    Drop(232, 27, 18, DuelRank.BCD),
    Drop(239, 27, 18, DuelRank.BCD),
    Drop(240, 27, 18, DuelRank.BCD),
    Drop(243, 27, 18, DuelRank.BCD),
    Drop(247, 27, 18, DuelRank.BCD),
    Drop(253, 27, 18, DuelRank.BCD),
    Drop(264, 27, 18, DuelRank.BCD),
    Drop(265, 27, 18, DuelRank.BCD),
    Drop(266, 27, 18, DuelRank.BCD),
    Drop(267, 27, 18, DuelRank.BCD),
    Drop(271, 27, 18, DuelRank.BCD),
    Drop(277, 27, 18, DuelRank.BCD),
    Drop(282, 27, 18, DuelRank.BCD),
    Drop(283, 27, 18, DuelRank.BCD),
    Drop(292, 27, 18, DuelRank.BCD),
    Drop(298, 27, 18, DuelRank.BCD),
    Drop(331, 27, 30, DuelRank.BCD),
    Drop(332, 27, 30, DuelRank.BCD),
    Drop(417, 27, 18, DuelRank.BCD),
    Drop(421, 27, 18, DuelRank.BCD),
    Drop(432, 27, 18, DuelRank.BCD),
    Drop(451, 27, 18, DuelRank.BCD),
    Drop(463, 27, 18, DuelRank.BCD),
    Drop(476, 27, 18, DuelRank.BCD),
    Drop(477, 27, 18, DuelRank.BCD),
    Drop(478, 27, 18, DuelRank.BCD),
    Drop(479, 27, 18, DuelRank.BCD),
    Drop(481, 27, 18, DuelRank.BCD),
    Drop(490, 27, 18, DuelRank.BCD),
    Drop(503, 27, 18, DuelRank.BCD),
    Drop(510, 27, 18, DuelRank.BCD),
    Drop(534, 27, 18, DuelRank.BCD),
    Drop(536, 27, 18, DuelRank.BCD),
    Drop(538, 27, 18, DuelRank.BCD),
    Drop(544, 27, 18, DuelRank.BCD),
    Drop(556, 27, 18, DuelRank.BCD),
    Drop(568, 27, 32, DuelRank.BCD),
    Drop(569, 27, 18, DuelRank.BCD),
    Drop(573, 27, 48, DuelRank.BCD),
    Drop(585, 27, 18, DuelRank.BCD),
    Drop(590, 27, 18, DuelRank.BCD),
    Drop(604, 27, 18, DuelRank.BCD),
    Drop(605, 27, 18, DuelRank.BCD),
    Drop(606, 27, 18, DuelRank.BCD),
    Drop(3, 28, 12, DuelRank.BCD),
    Drop(4, 28, 12, DuelRank.BCD),
    Drop(5, 28, 26, DuelRank.BCD),
    Drop(6, 28, 12, DuelRank.BCD),
    Drop(20, 28, 6, DuelRank.BCD),
    Drop(23, 28, 12, DuelRank.BCD),
    Drop(25, 28, 12, DuelRank.BCD),
    Drop(26, 28, 12, DuelRank.BCD),
    Drop(29, 28, 12, DuelRank.BCD),
    Drop(34, 28, 26, DuelRank.BCD),
    Drop(40, 28, 12, DuelRank.BCD),
    Drop(42, 28, 12, DuelRank.BCD),
    Drop(44, 28, 12, DuelRank.BCD),
    Drop(47, 28, 12, DuelRank.BCD),
    Drop(48, 28, 26, DuelRank.BCD),
    Drop(59, 28, 12, DuelRank.BCD),
    Drop(61, 28, 12, DuelRank.BCD),
    Drop(65, 28, 12, DuelRank.BCD),
    Drop(77, 28, 12, DuelRank.BCD),
    Drop(83, 28, 12, DuelRank.BCD),
    Drop(86, 28, 12, DuelRank.BCD),
    Drop(87, 28, 12, DuelRank.BCD),
    Drop(91, 28, 12, DuelRank.BCD),
    Drop(95, 28, 12, DuelRank.BCD),
    Drop(97, 28, 12, DuelRank.BCD),
    Drop(100, 28, 12, DuelRank.BCD),
    Drop(103, 28, 12, DuelRank.BCD),
    Drop(104, 28, 12, DuelRank.BCD),
    Drop(106, 28, 12, DuelRank.BCD),
    Drop(108, 28, 12, DuelRank.BCD),
    Drop(113, 28, 12, DuelRank.BCD),
    Drop(114, 28, 26, DuelRank.BCD),
    Drop(115, 28, 12, DuelRank.BCD),
    Drop(116, 28, 12, DuelRank.BCD),
    Drop(118, 28, 12, DuelRank.BCD),
    Drop(119, 28, 12, DuelRank.BCD),
    Drop(129, 28, 12, DuelRank.BCD),
    Drop(133, 28, 12, DuelRank.BCD),
    Drop(136, 28, 12, DuelRank.BCD),
    Drop(138, 28, 12, DuelRank.BCD),
    Drop(139, 28, 12, DuelRank.BCD),
    Drop(140, 28, 12, DuelRank.BCD),
    Drop(141, 28, 12, DuelRank.BCD),
    Drop(142, 28, 26, DuelRank.BCD),
    Drop(143, 28, 26, DuelRank.BCD),
    Drop(144, 28, 26, DuelRank.BCD),
    Drop(145, 28, 12, DuelRank.BCD),
    Drop(148, 28, 12, DuelRank.BCD),
    Drop(149, 28, 12, DuelRank.BCD),
    Drop(155, 28, 12, DuelRank.BCD),
    Drop(162, 28, 12, DuelRank.BCD),
    Drop(164, 28, 12, DuelRank.BCD),
    Drop(169, 28, 26, DuelRank.BCD),
    Drop(171, 28, 12, DuelRank.BCD),
    Drop(172, 28, 12, DuelRank.BCD),
    Drop(173, 28, 26, DuelRank.BCD),
    Drop(174, 28, 12, DuelRank.BCD),
    Drop(175, 28, 26, DuelRank.BCD),
    Drop(177, 28, 12, DuelRank.BCD),
    Drop(178, 28, 26, DuelRank.BCD),
    Drop(179, 28, 12, DuelRank.BCD),
    Drop(180, 28, 12, DuelRank.BCD),
    Drop(181, 28, 26, DuelRank.BCD),
    Drop(183, 28, 12, DuelRank.BCD),
    Drop(184, 28, 12, DuelRank.BCD),
    Drop(188, 28, 12, DuelRank.BCD),
    Drop(189, 28, 12, DuelRank.BCD),
    Drop(190, 28, 26, DuelRank.BCD),
    Drop(194, 28, 12, DuelRank.BCD),
    Drop(199, 28, 12, DuelRank.BCD),
    Drop(204, 28, 12, DuelRank.BCD),
    Drop(205, 28, 12, DuelRank.BCD),
    Drop(206, 28, 12, DuelRank.BCD),
    Drop(213, 28, 12, DuelRank.BCD),
    Drop(215, 28, 12, DuelRank.BCD),
    Drop(218, 28, 12, DuelRank.BCD),
    Drop(220, 28, 12, DuelRank.BCD),
    Drop(222, 28, 12, DuelRank.BCD),
    Drop(224, 28, 12, DuelRank.BCD),
    Drop(226, 28, 12, DuelRank.BCD),
    Drop(228, 28, 12, DuelRank.BCD),
    Drop(231, 28, 12, DuelRank.BCD),
    Drop(232, 28, 12, DuelRank.BCD),
    Drop(233, 28, 12, DuelRank.BCD),
    Drop(239, 28, 12, DuelRank.BCD),
    Drop(240, 28, 12, DuelRank.BCD),
    Drop(242, 28, 12, DuelRank.BCD),
    Drop(243, 28, 12, DuelRank.BCD),
    Drop(244, 28, 12, DuelRank.BCD),
    Drop(245, 28, 12, DuelRank.BCD),
    Drop(247, 28, 12, DuelRank.BCD),
    Drop(250, 28, 12, DuelRank.BCD),
    Drop(253, 28, 24, DuelRank.BCD),
    Drop(254, 28, 12, DuelRank.BCD),
    Drop(257, 28, 12, DuelRank.BCD),
    Drop(259, 28, 12, DuelRank.BCD),
    Drop(260, 28, 12, DuelRank.BCD),
    Drop(261, 28, 12, DuelRank.BCD),
    Drop(264, 28, 12, DuelRank.BCD),
    Drop(266, 28, 12, DuelRank.BCD),
    Drop(268, 28, 12, DuelRank.BCD),
    Drop(269, 28, 24, DuelRank.BCD),
    Drop(270, 28, 12, DuelRank.BCD),
    Drop(271, 28, 12, DuelRank.BCD),
    Drop(276, 28, 12, DuelRank.BCD),
    Drop(277, 28, 24, DuelRank.BCD),
    Drop(279, 28, 24, DuelRank.BCD),
    Drop(282, 28, 12, DuelRank.BCD),
    Drop(283, 28, 12, DuelRank.BCD),
    Drop(295, 28, 24, DuelRank.BCD),
    Drop(296, 28, 12, DuelRank.BCD),
    Drop(298, 28, 12, DuelRank.BCD),
    Drop(331, 28, 34, DuelRank.BCD),
    Drop(379, 28, 24, DuelRank.BCD),
    Drop(417, 28, 12, DuelRank.BCD),
    Drop(432, 28, 12, DuelRank.BCD),
    Drop(446, 28, 12, DuelRank.BCD),
    Drop(461, 28, 12, DuelRank.BCD),
    Drop(463, 28, 12, DuelRank.BCD),
    Drop(477, 28, 12, DuelRank.BCD),
    Drop(478, 28, 12, DuelRank.BCD),
    Drop(479, 28, 12, DuelRank.BCD),
    Drop(481, 28, 12, DuelRank.BCD),
    Drop(486, 28, 12, DuelRank.BCD),
    Drop(490, 28, 12, DuelRank.BCD),
    Drop(503, 28, 12, DuelRank.BCD),
    Drop(534, 28, 12, DuelRank.BCD),
    Drop(538, 28, 12, DuelRank.BCD),
    Drop(568, 28, 12, DuelRank.BCD),
    Drop(569, 28, 12, DuelRank.BCD),
    Drop(573, 28, 12, DuelRank.BCD),
    Drop(585, 28, 12, DuelRank.BCD),
    Drop(590, 28, 12, DuelRank.BCD),
    Drop(605, 28, 12, DuelRank.BCD),
    Drop(611, 28, 12, DuelRank.BCD),
    Drop(612, 28, 24, DuelRank.BCD),
    Drop(650, 28, 12, DuelRank.BCD),
    Drop(682, 28, 18, DuelRank.BCD),
    Drop(683, 28, 18, DuelRank.BCD),
    Drop(687, 28, 18, DuelRank.BCD),
    Drop(693, 28, 34, DuelRank.BCD),
    Drop(695, 28, 34, DuelRank.BCD),
    Drop(2, 29, 50, DuelRank.BCD),
    Drop(6, 29, 50, DuelRank.BCD),
    Drop(9, 29, 178, DuelRank.BCD),
    Drop(10, 29, 50, DuelRank.BCD),
    Drop(19, 29, 12, DuelRank.BCD),
    Drop(25, 29, 50, DuelRank.BCD),
    Drop(27, 29, 50, DuelRank.BCD),
    Drop(30, 29, 50, DuelRank.BCD),
    Drop(31, 29, 50, DuelRank.BCD),
    Drop(41, 29, 50, DuelRank.BCD),
    Drop(46, 29, 50, DuelRank.BCD),
    Drop(47, 29, 50, DuelRank.BCD),
    Drop(48, 29, 50, DuelRank.BCD),
    Drop(59, 29, 50, DuelRank.BCD),
    Drop(65, 29, 50, DuelRank.BCD),
    Drop(89, 29, 50, DuelRank.BCD),
    Drop(94, 29, 22, DuelRank.BCD),
    Drop(102, 29, 22, DuelRank.BCD),
    Drop(111, 29, 22, DuelRank.BCD),
    Drop(130, 29, 100, DuelRank.BCD),
    Drop(238, 29, 22, DuelRank.BCD),
    Drop(333, 29, 66, DuelRank.BCD),
    Drop(381, 29, 2, DuelRank.BCD),
    Drop(396, 29, 22, DuelRank.BCD),
    Drop(408, 29, 22, DuelRank.BCD),
    Drop(409, 29, 22, DuelRank.BCD),
    Drop(422, 29, 96, DuelRank.BCD),
    Drop(436, 29, 96, DuelRank.BCD),
    Drop(437, 29, 22, DuelRank.BCD),
    Drop(439, 29, 22, DuelRank.BCD),
    Drop(444, 29, 96, DuelRank.BCD),
    Drop(458, 29, 22, DuelRank.BCD),
    Drop(473, 29, 22, DuelRank.BCD),
    Drop(482, 29, 22, DuelRank.BCD),
    Drop(485, 29, 96, DuelRank.BCD),
    Drop(487, 29, 22, DuelRank.BCD),
    Drop(516, 29, 96, DuelRank.BCD),
    Drop(521, 29, 12, DuelRank.BCD),
    Drop(533, 29, 12, DuelRank.BCD),
    Drop(542, 29, 22, DuelRank.BCD),
    Drop(557, 29, 22, DuelRank.BCD),
    Drop(563, 29, 96, DuelRank.BCD),
    Drop(618, 29, 12, DuelRank.BCD),
    Drop(637, 29, 22, DuelRank.BCD),
    Drop(659, 29, 12, DuelRank.BCD),
    Drop(712, 29, 12, DuelRank.BCD),
    Drop(714, 29, 2, DuelRank.BCD),
    Drop(3, 30, 80, DuelRank.BCD),
    Drop(4, 30, 40, DuelRank.BCD),
    Drop(5, 30, 80, DuelRank.BCD),
    Drop(10, 30, 40, DuelRank.BCD),
    Drop(12, 30, 40, DuelRank.BCD),
    Drop(15, 30, 40, DuelRank.BCD),
    Drop(19, 30, 6, DuelRank.BCD),
    Drop(23, 30, 80, DuelRank.BCD),
    Drop(26, 30, 80, DuelRank.BCD),
    Drop(31, 30, 40, DuelRank.BCD),
    Drop(34, 30, 80, DuelRank.BCD),
    Drop(42, 30, 40, DuelRank.BCD),
    Drop(68, 30, 40, DuelRank.BCD),
    Drop(73, 30, 39, DuelRank.BCD),
    Drop(74, 30, 39, DuelRank.BCD),
    Drop(77, 30, 78, DuelRank.BCD),
    Drop(84, 30, 39, DuelRank.BCD),
    Drop(88, 30, 39, DuelRank.BCD),
    Drop(91, 30, 78, DuelRank.BCD),
    Drop(93, 30, 40, DuelRank.BCD),
    Drop(122, 30, 40, DuelRank.BCD),
    Drop(126, 30, 39, DuelRank.BCD),
    Drop(168, 30, 40, DuelRank.BCD),
    Drop(200, 30, 40, DuelRank.BCD),
    Drop(213, 30, 39, DuelRank.BCD),
    Drop(286, 30, 39, DuelRank.BCD),
    Drop(296, 30, 40, DuelRank.BCD),
    Drop(298, 30, 40, DuelRank.BCD),
    Drop(333, 30, 30, DuelRank.BCD),
    Drop(376, 30, 39, DuelRank.BCD),
    Drop(379, 30, 78, DuelRank.BCD),
    Drop(400, 30, 39, DuelRank.BCD),
    Drop(403, 30, 39, DuelRank.BCD),
    Drop(413, 30, 39, DuelRank.BCD),
    Drop(423, 30, 39, DuelRank.BCD),
    Drop(441, 30, 39, DuelRank.BCD),
    Drop(464, 30, 39, DuelRank.BCD),
    Drop(468, 30, 39, DuelRank.BCD),
    Drop(491, 30, 39, DuelRank.BCD),
    Drop(508, 30, 39, DuelRank.BCD),
    Drop(513, 30, 39, DuelRank.BCD),
    Drop(528, 30, 39, DuelRank.BCD),
    Drop(577, 30, 39, DuelRank.BCD),
    Drop(638, 30, 39, DuelRank.BCD),
    Drop(650, 30, 39, DuelRank.BCD),
    Drop(10, 31, 16, DuelRank.BCD),
    Drop(19, 31, 6, DuelRank.BCD),
    Drop(23, 31, 16, DuelRank.BCD),
    Drop(25, 31, 16, DuelRank.BCD),
    Drop(30, 31, 40, DuelRank.BCD),
    Drop(34, 31, 16, DuelRank.BCD),
    Drop(36, 31, 16, DuelRank.BCD),
    Drop(40, 31, 16, DuelRank.BCD),
    Drop(45, 31, 16, DuelRank.BCD),
    Drop(53, 31, 16, DuelRank.BCD),
    Drop(59, 31, 16, DuelRank.BCD),
    Drop(61, 31, 16, DuelRank.BCD),
    Drop(65, 31, 16, DuelRank.BCD),
    Drop(70, 31, 16, DuelRank.BCD),
    Drop(71, 31, 16, DuelRank.BCD),
    Drop(74, 31, 14, DuelRank.BCD),
    Drop(80, 31, 16, DuelRank.BCD),
    Drop(89, 31, 14, DuelRank.BCD),
    Drop(96, 31, 16, DuelRank.BCD),
    Drop(97, 31, 16, DuelRank.BCD),
    Drop(98, 31, 16, DuelRank.BCD),
    Drop(103, 31, 16, DuelRank.BCD),
    Drop(108, 31, 16, DuelRank.BCD),
    Drop(109, 31, 16, DuelRank.BCD),
    Drop(110, 31, 16, DuelRank.BCD),
    Drop(115, 31, 16, DuelRank.BCD),
    Drop(118, 31, 16, DuelRank.BCD),
    Drop(119, 31, 16, DuelRank.BCD),
    Drop(120, 31, 16, DuelRank.BCD),
    Drop(121, 31, 16, DuelRank.BCD),
    Drop(131, 31, 14, DuelRank.BCD),
    Drop(132, 31, 38, DuelRank.BCD),
    Drop(135, 31, 16, DuelRank.BCD),
    Drop(138, 31, 16, DuelRank.BCD),
    Drop(139, 31, 16, DuelRank.BCD),
    Drop(140, 31, 38, DuelRank.BCD),
    Drop(141, 31, 14, DuelRank.BCD),
    Drop(150, 31, 12, DuelRank.BCD),
    Drop(152, 31, 14, DuelRank.BCD),
    Drop(154, 31, 14, DuelRank.BCD),
    Drop(162, 31, 14, DuelRank.BCD),
    Drop(164, 31, 14, DuelRank.BCD),
    Drop(169, 31, 14, DuelRank.BCD),
    Drop(171, 31, 14, DuelRank.BCD),
    Drop(172, 31, 14, DuelRank.BCD),
    Drop(177, 31, 14, DuelRank.BCD),
    Drop(181, 31, 14, DuelRank.BCD),
    Drop(191, 31, 14, DuelRank.BCD),
    Drop(193, 31, 14, DuelRank.BCD),
    Drop(196, 31, 30, DuelRank.BCD),
    Drop(199, 31, 14, DuelRank.BCD),
    Drop(201, 31, 14, DuelRank.BCD),
    Drop(203, 31, 14, DuelRank.BCD),
    Drop(205, 31, 14, DuelRank.BCD),
    Drop(206, 31, 14, DuelRank.BCD),
    Drop(211, 31, 14, DuelRank.BCD),
    Drop(215, 31, 14, DuelRank.BCD),
    Drop(219, 31, 14, DuelRank.BCD),
    Drop(220, 31, 14, DuelRank.BCD),
    Drop(221, 31, 14, DuelRank.BCD),
    Drop(225, 31, 14, DuelRank.BCD),
    Drop(227, 31, 14, DuelRank.BCD),
    Drop(228, 31, 14, DuelRank.BCD),
    Drop(231, 31, 14, DuelRank.BCD),
    Drop(233, 31, 14, DuelRank.BCD),
    Drop(236, 31, 14, DuelRank.BCD),
    Drop(237, 31, 14, DuelRank.BCD),
    Drop(243, 31, 14, DuelRank.BCD),
    Drop(244, 31, 14, DuelRank.BCD),
    Drop(246, 31, 14, DuelRank.BCD),
    Drop(248, 31, 14, DuelRank.BCD),
    Drop(250, 31, 14, DuelRank.BCD),
    Drop(251, 31, 14, DuelRank.BCD),
    Drop(256, 31, 14, DuelRank.BCD),
    Drop(257, 31, 30, DuelRank.BCD),
    Drop(258, 31, 30, DuelRank.BCD),
    Drop(259, 31, 14, DuelRank.BCD),
    Drop(262, 31, 14, DuelRank.BCD),
    Drop(263, 31, 30, DuelRank.BCD),
    Drop(265, 31, 14, DuelRank.BCD),
    Drop(269, 31, 14, DuelRank.BCD),
    Drop(270, 31, 14, DuelRank.BCD),
    Drop(272, 31, 14, DuelRank.BCD),
    Drop(273, 31, 14, DuelRank.BCD),
    Drop(274, 31, 14, DuelRank.BCD),
    Drop(276, 31, 14, DuelRank.BCD),
    Drop(280, 31, 14, DuelRank.BCD),
    Drop(290, 31, 14, DuelRank.BCD),
    Drop(291, 31, 14, DuelRank.BCD),
    Drop(292, 31, 14, DuelRank.BCD),
    Drop(293, 31, 14, DuelRank.BCD),
    Drop(294, 31, 14, DuelRank.BCD),
    Drop(296, 31, 14, DuelRank.BCD),
    Drop(335, 31, 36, DuelRank.BCD),
    Drop(366, 31, 30, DuelRank.BCD),
    Drop(368, 31, 30, DuelRank.BCD),
    Drop(375, 31, 2, DuelRank.BCD),
    Drop(376, 31, 12, DuelRank.BCD),
    Drop(406, 31, 16, DuelRank.BCD),
    Drop(414, 31, 16, DuelRank.BCD),
    Drop(425, 31, 12, DuelRank.BCD),
    Drop(432, 31, 14, DuelRank.BCD),
    Drop(446, 31, 14, DuelRank.BCD),
    Drop(451, 31, 14, DuelRank.BCD),
    Drop(452, 31, 14, DuelRank.BCD),
    Drop(459, 31, 14, DuelRank.BCD),
    Drop(460, 31, 14, DuelRank.BCD),
    Drop(461, 31, 14, DuelRank.BCD),
    Drop(462, 31, 14, DuelRank.BCD),
    Drop(463, 31, 14, DuelRank.BCD),
    Drop(496, 31, 14, DuelRank.BCD),
    Drop(503, 31, 14, DuelRank.BCD),
    Drop(514, 31, 14, DuelRank.BCD),
    Drop(524, 31, 14, DuelRank.BCD),
    Drop(549, 31, 14, DuelRank.BCD),
    Drop(552, 31, 14, DuelRank.BCD),
    Drop(556, 31, 14, DuelRank.BCD),
    Drop(567, 31, 14, DuelRank.BCD),
    Drop(576, 31, 14, DuelRank.BCD),
    Drop(584, 31, 14, DuelRank.BCD),
    Drop(591, 31, 14, DuelRank.BCD),
    Drop(601, 31, 14, DuelRank.BCD),
    Drop(602, 31, 14, DuelRank.BCD),
    Drop(605, 31, 14, DuelRank.BCD),
    Drop(606, 31, 14, DuelRank.BCD),
    Drop(608, 31, 14, DuelRank.BCD),
    Drop(620, 31, 14, DuelRank.BCD),
    Drop(629, 31, 14, DuelRank.BCD),
    Drop(642, 31, 14, DuelRank.BCD),
    Drop(643, 31, 14, DuelRank.BCD),
    Drop(2, 32, 40, DuelRank.BCD),
    Drop(6, 32, 40, DuelRank.BCD),
    Drop(10, 32, 40, DuelRank.BCD),
    Drop(12, 32, 20, DuelRank.BCD),
    Drop(15, 32, 20, DuelRank.BCD),
    Drop(21, 32, 6, DuelRank.BCD),
    Drop(25, 32, 40, DuelRank.BCD),
    Drop(27, 32, 40, DuelRank.BCD),
    Drop(30, 32, 40, DuelRank.BCD),
    Drop(31, 32, 40, DuelRank.BCD),
    Drop(33, 32, 20, DuelRank.BCD),
    Drop(39, 32, 40, DuelRank.BCD),
    Drop(41, 32, 58, DuelRank.BCD),
    Drop(43, 32, 20, DuelRank.BCD),
    Drop(46, 32, 40, DuelRank.BCD),
    Drop(47, 32, 40, DuelRank.BCD),
    Drop(48, 32, 40, DuelRank.BCD),
    Drop(59, 32, 40, DuelRank.BCD),
    Drop(65, 32, 40, DuelRank.BCD),
    Drop(74, 32, 40, DuelRank.BCD),
    Drop(78, 32, 18, DuelRank.BCD),
    Drop(79, 32, 18, DuelRank.BCD),
    Drop(89, 32, 38, DuelRank.BCD),
    Drop(90, 32, 18, DuelRank.BCD),
    Drop(93, 32, 18, DuelRank.BCD),
    Drop(99, 32, 18, DuelRank.BCD),
    Drop(100, 32, 18, DuelRank.BCD),
    Drop(107, 32, 18, DuelRank.BCD),
    Drop(110, 32, 18, DuelRank.BCD),
    Drop(118, 32, 18, DuelRank.BCD),
    Drop(120, 32, 18, DuelRank.BCD),
    Drop(127, 32, 18, DuelRank.BCD),
    Drop(138, 32, 18, DuelRank.BCD),
    Drop(147, 32, 18, DuelRank.BCD),
    Drop(156, 32, 18, DuelRank.BCD),
    Drop(160, 32, 18, DuelRank.BCD),
    Drop(161, 32, 18, DuelRank.BCD),
    Drop(166, 32, 18, DuelRank.BCD),
    Drop(172, 32, 18, DuelRank.BCD),
    Drop(182, 32, 18, DuelRank.BCD),
    Drop(185, 32, 18, DuelRank.BCD),
    Drop(195, 32, 18, DuelRank.BCD),
    Drop(214, 32, 18, DuelRank.BCD),
    Drop(223, 32, 18, DuelRank.BCD),
    Drop(224, 32, 18, DuelRank.BCD),
    Drop(225, 32, 18, DuelRank.BCD),
    Drop(226, 32, 18, DuelRank.BCD),
    Drop(231, 32, 18, DuelRank.BCD),
    Drop(236, 32, 18, DuelRank.BCD),
    Drop(239, 32, 18, DuelRank.BCD),
    Drop(250, 32, 18, DuelRank.BCD),
    Drop(256, 32, 18, DuelRank.BCD),
    Drop(262, 32, 18, DuelRank.BCD),
    Drop(266, 32, 18, DuelRank.BCD),
    Drop(267, 32, 18, DuelRank.BCD),
    Drop(280, 32, 18, DuelRank.BCD),
    Drop(290, 32, 18, DuelRank.BCD),
    Drop(293, 32, 18, DuelRank.BCD),
    Drop(294, 32, 18, DuelRank.BCD),
    Drop(376, 32, 18, DuelRank.BCD),
    Drop(445, 32, 54, DuelRank.BCD),
    Drop(447, 32, 32, DuelRank.BCD),
    Drop(465, 32, 18, DuelRank.BCD),
    Drop(474, 32, 54, DuelRank.BCD),
    Drop(493, 32, 32, DuelRank.BCD),
    Drop(495, 32, 10, DuelRank.BCD),
    Drop(507, 32, 32, DuelRank.BCD),
    Drop(515, 32, 32, DuelRank.BCD),
    Drop(522, 32, 16, DuelRank.BCD),
    Drop(531, 32, 16, DuelRank.BCD),
    Drop(546, 32, 54, DuelRank.BCD),
    Drop(559, 32, 54, DuelRank.BCD),
    Drop(561, 32, 54, DuelRank.BCD),
    Drop(565, 32, 12, DuelRank.BCD),
    Drop(571, 32, 16, DuelRank.BCD),
    Drop(572, 32, 32, DuelRank.BCD),
    Drop(583, 32, 32, DuelRank.BCD),
    Drop(594, 32, 16, DuelRank.BCD),
    Drop(622, 32, 12, DuelRank.BCD),
    Drop(624, 32, 10, DuelRank.BCD),
    Drop(627, 32, 16, DuelRank.BCD),
    Drop(2, 33, 26, DuelRank.BCD),
    Drop(14, 33, 26, DuelRank.BCD),
    Drop(19, 33, 6, DuelRank.BCD),
    Drop(30, 33, 26, DuelRank.BCD),
    Drop(32, 33, 26, DuelRank.BCD),
    Drop(36, 33, 76, DuelRank.BCD),
    Drop(78, 33, 26, DuelRank.BCD),
    Drop(81, 33, 26, DuelRank.BCD),
    Drop(83, 33, 26, DuelRank.BCD),
    Drop(86, 33, 26, DuelRank.BCD),
    Drop(87, 33, 26, DuelRank.BCD),
    Drop(89, 33, 26, DuelRank.BCD),
    Drop(91, 33, 26, DuelRank.BCD),
    Drop(93, 33, 76, DuelRank.BCD),
    Drop(95, 33, 76, DuelRank.BCD),
    Drop(97, 33, 76, DuelRank.BCD),
    Drop(99, 33, 76, DuelRank.BCD),
    Drop(106, 33, 26, DuelRank.BCD),
    Drop(108, 33, 26, DuelRank.BCD),
    Drop(124, 33, 50, DuelRank.BCD),
    Drop(127, 33, 26, DuelRank.BCD),
    Drop(131, 33, 26, DuelRank.BCD),
    Drop(132, 33, 26, DuelRank.BCD),
    Drop(139, 33, 26, DuelRank.BCD),
    Drop(150, 33, 26, DuelRank.BCD),
    Drop(170, 33, 26, DuelRank.BCD),
    Drop(204, 33, 26, DuelRank.BCD),
    Drop(213, 33, 26, DuelRank.BCD),
    Drop(215, 33, 26, DuelRank.BCD),
    Drop(228, 33, 26, DuelRank.BCD),
    Drop(283, 33, 26, DuelRank.BCD),
    Drop(286, 33, 26, DuelRank.BCD),
    Drop(366, 33, 26, DuelRank.BCD),
    Drop(368, 33, 50, DuelRank.BCD),
    Drop(379, 33, 26, DuelRank.BCD),
    Drop(388, 33, 70, DuelRank.BCD),
    Drop(405, 33, 50, DuelRank.BCD),
    Drop(406, 33, 26, DuelRank.BCD),
    Drop(408, 33, 6, DuelRank.BCD),
    Drop(409, 33, 6, DuelRank.BCD),
    Drop(412, 33, 26, DuelRank.BCD),
    Drop(413, 33, 50, DuelRank.BCD),
    Drop(414, 33, 26, DuelRank.BCD),
    Drop(415, 33, 26, DuelRank.BCD),
    Drop(416, 33, 26, DuelRank.BCD),
    Drop(417, 33, 26, DuelRank.BCD),
    Drop(419, 33, 26, DuelRank.BCD),
    Drop(423, 33, 26, DuelRank.BCD),
    Drop(425, 33, 26, DuelRank.BCD),
    Drop(438, 33, 26, DuelRank.BCD),
    Drop(441, 33, 26, DuelRank.BCD),
    Drop(460, 33, 26, DuelRank.BCD),
    Drop(462, 33, 26, DuelRank.BCD),
    Drop(468, 33, 26, DuelRank.BCD),
    Drop(491, 33, 26, DuelRank.BCD),
    Drop(508, 33, 50, DuelRank.BCD),
    Drop(512, 33, 26, DuelRank.BCD),
    Drop(513, 33, 26, DuelRank.BCD),
    Drop(514, 33, 26, DuelRank.BCD),
    Drop(557, 33, 6, DuelRank.BCD),
    Drop(585, 33, 26, DuelRank.BCD),
    Drop(633, 33, 26, DuelRank.BCD),
    Drop(636, 33, 26, DuelRank.BCD),
    Drop(643, 33, 26, DuelRank.BCD),
    Drop(648, 33, 50, DuelRank.BCD),
    Drop(6, 34, 46, DuelRank.BCD),
    Drop(13, 34, 26, DuelRank.BCD),
    Drop(20, 34, 6, DuelRank.BCD),
    Drop(25, 34, 26, DuelRank.BCD),
    Drop(26, 34, 26, DuelRank.BCD),
    Drop(27, 34, 26, DuelRank.BCD),
    Drop(31, 34, 26, DuelRank.BCD),
    Drop(32, 34, 26, DuelRank.BCD),
    Drop(36, 34, 26, DuelRank.BCD),
    Drop(41, 34, 26, DuelRank.BCD),
    Drop(43, 34, 26, DuelRank.BCD),
    Drop(44, 34, 26, DuelRank.BCD),
    Drop(46, 34, 26, DuelRank.BCD),
    Drop(48, 34, 26, DuelRank.BCD),
    Drop(49, 34, 26, DuelRank.BCD),
    Drop(54, 34, 26, DuelRank.BCD),
    Drop(55, 34, 26, DuelRank.BCD),
    Drop(70, 34, 26, DuelRank.BCD),
    Drop(71, 34, 26, DuelRank.BCD),
    Drop(77, 34, 26, DuelRank.BCD),
    Drop(81, 34, 26, DuelRank.BCD),
    Drop(83, 34, 67, DuelRank.BCD),
    Drop(84, 34, 67, DuelRank.BCD),
    Drop(85, 34, 67, DuelRank.BCD),
    Drop(86, 34, 67, DuelRank.BCD),
    Drop(87, 34, 67, DuelRank.BCD),
    Drop(88, 34, 67, DuelRank.BCD),
    Drop(95, 34, 46, DuelRank.BCD),
    Drop(103, 34, 26, DuelRank.BCD),
    Drop(115, 34, 26, DuelRank.BCD),
    Drop(119, 34, 26, DuelRank.BCD),
    Drop(121, 34, 26, DuelRank.BCD),
    Drop(125, 34, 26, DuelRank.BCD),
    Drop(136, 34, 42, DuelRank.BCD),
    Drop(149, 34, 42, DuelRank.BCD),
    Drop(162, 34, 21, DuelRank.BCD),
    Drop(164, 34, 21, DuelRank.BCD),
    Drop(166, 34, 21, DuelRank.BCD),
    Drop(168, 34, 21, DuelRank.BCD),
    Drop(169, 34, 21, DuelRank.BCD),
    Drop(170, 34, 21, DuelRank.BCD),
    Drop(171, 34, 21, DuelRank.BCD),
    Drop(173, 34, 21, DuelRank.BCD),
    Drop(175, 34, 21, DuelRank.BCD),
    Drop(178, 34, 21, DuelRank.BCD),
    Drop(181, 34, 21, DuelRank.BCD),
    Drop(186, 34, 21, DuelRank.BCD),
    Drop(193, 34, 21, DuelRank.BCD),
    Drop(194, 34, 42, DuelRank.BCD),
    Drop(204, 34, 21, DuelRank.BCD),
    Drop(230, 34, 21, DuelRank.BCD),
    Drop(233, 34, 42, DuelRank.BCD),
    Drop(255, 34, 21, DuelRank.BCD),
    Drop(269, 34, 21, DuelRank.BCD),
    Drop(277, 34, 21, DuelRank.BCD),
    Drop(279, 34, 21, DuelRank.BCD),
    Drop(295, 34, 21, DuelRank.BCD),
    Drop(379, 34, 42, DuelRank.BCD),
    Drop(412, 34, 21, DuelRank.BCD),
    Drop(415, 34, 21, DuelRank.BCD),
    Drop(416, 34, 21, DuelRank.BCD),
    Drop(459, 34, 21, DuelRank.BCD),
    Drop(462, 34, 21, DuelRank.BCD),
    Drop(466, 34, 21, DuelRank.BCD),
    Drop(472, 34, 21, DuelRank.BCD),
    Drop(511, 34, 21, DuelRank.BCD),
    Drop(512, 34, 21, DuelRank.BCD),
    Drop(526, 34, 21, DuelRank.BCD),
    Drop(597, 34, 21, DuelRank.BCD),
    Drop(608, 34, 21, DuelRank.BCD),
    Drop(636, 34, 21, DuelRank.BCD),
    Drop(650, 34, 21, DuelRank.BCD),
    Drop(2, 35, 26, DuelRank.BCD),
    Drop(19, 35, 6, DuelRank.BCD),
    Drop(30, 35, 26, DuelRank.BCD),
    Drop(34, 35, 26, DuelRank.BCD),
    Drop(36, 35, 26, DuelRank.BCD),
    Drop(42, 35, 26, DuelRank.BCD),
    Drop(44, 35, 28, DuelRank.BCD),
    Drop(74, 35, 28, DuelRank.BCD),
    Drop(79, 35, 24, DuelRank.BCD),
    Drop(90, 35, 24, DuelRank.BCD),
    Drop(96, 35, 28, DuelRank.BCD),
    Drop(97, 35, 28, DuelRank.BCD),
    Drop(98, 35, 28, DuelRank.BCD),
    Drop(99, 35, 48, DuelRank.BCD),
    Drop(104, 35, 28, DuelRank.BCD),
    Drop(106, 35, 28, DuelRank.BCD),
    Drop(108, 35, 28, DuelRank.BCD),
    Drop(114, 35, 28, DuelRank.BCD),
    Drop(115, 35, 28, DuelRank.BCD),
    Drop(129, 35, 28, DuelRank.BCD),
    Drop(132, 35, 28, DuelRank.BCD),
    Drop(135, 35, 28, DuelRank.BCD),
    Drop(139, 35, 28, DuelRank.BCD),
    Drop(142, 35, 28, DuelRank.BCD),
    Drop(143, 35, 28, DuelRank.BCD),
    Drop(144, 35, 28, DuelRank.BCD),
    Drop(145, 35, 28, DuelRank.BCD),
    Drop(154, 35, 28, DuelRank.BCD),
    Drop(174, 35, 28, DuelRank.BCD),
    Drop(179, 35, 28, DuelRank.BCD),
    Drop(183, 35, 28, DuelRank.BCD),
    Drop(184, 35, 28, DuelRank.BCD),
    Drop(190, 35, 28, DuelRank.BCD),
    Drop(203, 35, 28, DuelRank.BCD),
    Drop(213, 35, 28, DuelRank.BCD),
    Drop(215, 35, 28, DuelRank.BCD),
    Drop(220, 35, 28, DuelRank.BCD),
    Drop(223, 35, 20, DuelRank.BCD),
    Drop(228, 35, 28, DuelRank.BCD),
    Drop(237, 35, 28, DuelRank.BCD),
    Drop(244, 35, 28, DuelRank.BCD),
    Drop(253, 35, 28, DuelRank.BCD),
    Drop(257, 35, 28, DuelRank.BCD),
    Drop(259, 35, 28, DuelRank.BCD),
    Drop(263, 35, 28, DuelRank.BCD),
    Drop(268, 35, 28, DuelRank.BCD),
    Drop(366, 35, 28, DuelRank.BCD),
    Drop(368, 35, 28, DuelRank.BCD),
    Drop(455, 35, 64, DuelRank.BCD),
    Drop(457, 35, 64, DuelRank.BCD),
    Drop(465, 35, 24, DuelRank.BCD),
    Drop(470, 35, 32, DuelRank.BCD),
    Drop(505, 35, 64, DuelRank.BCD),
    Drop(518, 35, 10, DuelRank.BCD),
    Drop(522, 35, 24, DuelRank.BCD),
    Drop(525, 35, 16, DuelRank.BCD),
    Drop(529, 35, 10, DuelRank.BCD),
    Drop(531, 35, 44, DuelRank.BCD),
    Drop(532, 35, 32, DuelRank.BCD),
    Drop(571, 35, 24, DuelRank.BCD),
    Drop(572, 35, 24, DuelRank.BCD),
    Drop(593, 35, 16, DuelRank.BCD),
    Drop(594, 35, 24, DuelRank.BCD),
    Drop(599, 35, 64, DuelRank.BCD),
    Drop(623, 35, 28, DuelRank.BCD),
    Drop(627, 35, 24, DuelRank.BCD),
    Drop(631, 35, 32, DuelRank.BCD),
    Drop(632, 35, 16, DuelRank.BCD),
    Drop(634, 35, 32, DuelRank.BCD),
    Drop(646, 35, 64, DuelRank.BCD),
    Drop(6, 36, 32, DuelRank.BCD),
    Drop(20, 36, 6, DuelRank.BCD),
    Drop(25, 36, 32, DuelRank.BCD),
    Drop(48, 36, 32, DuelRank.BCD),
    Drop(79, 36, 24, DuelRank.BCD),
    Drop(83, 36, 32, DuelRank.BCD),
    Drop(84, 36, 32, DuelRank.BCD),
    Drop(85, 36, 22, DuelRank.BCD),
    Drop(86, 36, 32, DuelRank.BCD),
    Drop(87, 36, 32, DuelRank.BCD),
    Drop(88, 36, 32, DuelRank.BCD),
    Drop(90, 36, 22, DuelRank.BCD),
    Drop(95, 36, 32, DuelRank.BCD),
    Drop(99, 36, 24, DuelRank.BCD),
    Drop(103, 36, 32, DuelRank.BCD),
    Drop(119, 36, 32, DuelRank.BCD),
    Drop(136, 36, 32, DuelRank.BCD),
    Drop(137, 36, 32, DuelRank.BCD),
    Drop(148, 36, 32, DuelRank.BCD),
    Drop(149, 36, 32, DuelRank.BCD),
    Drop(162, 36, 32, DuelRank.BCD),
    Drop(164, 36, 32, DuelRank.BCD),
    Drop(169, 36, 32, DuelRank.BCD),
    Drop(171, 36, 32, DuelRank.BCD),
    Drop(173, 36, 32, DuelRank.BCD),
    Drop(175, 36, 32, DuelRank.BCD),
    Drop(178, 36, 32, DuelRank.BCD),
    Drop(181, 36, 28, DuelRank.BCD),
    Drop(194, 36, 28, DuelRank.BCD),
    Drop(204, 36, 28, DuelRank.BCD),
    Drop(222, 36, 28, DuelRank.BCD),
    Drop(223, 36, 24, DuelRank.BCD),
    Drop(232, 36, 28, DuelRank.BCD),
    Drop(233, 36, 28, DuelRank.BCD),
    Drop(240, 36, 28, DuelRank.BCD),
    Drop(242, 36, 28, DuelRank.BCD),
    Drop(245, 36, 28, DuelRank.BCD),
    Drop(254, 36, 28, DuelRank.BCD),
    Drop(261, 36, 28, DuelRank.BCD),
    Drop(269, 36, 28, DuelRank.BCD),
    Drop(271, 36, 28, DuelRank.BCD),
    Drop(277, 36, 28, DuelRank.BCD),
    Drop(279, 36, 28, DuelRank.BCD),
    Drop(295, 36, 28, DuelRank.BCD),
    Drop(379, 36, 28, DuelRank.BCD),
    Drop(424, 36, 12, DuelRank.BCD),
    Drop(449, 36, 12, DuelRank.BCD),
    Drop(465, 36, 24, DuelRank.BCD),
    Drop(472, 36, 24, DuelRank.BCD),
    Drop(497, 36, 32, DuelRank.BCD),
    Drop(522, 36, 24, DuelRank.BCD),
    Drop(523, 36, 32, DuelRank.BCD),
    Drop(526, 36, 24, DuelRank.BCD),
    Drop(531, 36, 24, DuelRank.BCD),
    Drop(535, 36, 32, DuelRank.BCD),
    Drop(539, 36, 54, DuelRank.BCD),
    Drop(551, 36, 32, DuelRank.BCD),
    Drop(553, 36, 54, DuelRank.BCD),
    Drop(560, 36, 54, DuelRank.BCD),
    Drop(571, 36, 24, DuelRank.BCD),
    Drop(572, 36, 24, DuelRank.BCD),
    Drop(574, 36, 54, DuelRank.BCD),
    Drop(594, 36, 24, DuelRank.BCD),
    Drop(600, 36, 54, DuelRank.BCD),
    Drop(619, 36, 12, DuelRank.BCD),
    Drop(627, 36, 24, DuelRank.BCD),
    Drop(639, 36, 16, DuelRank.BCD),
    Drop(641, 36, 16, DuelRank.BCD),
    Drop(647, 36, 32, DuelRank.BCD),
    Drop(650, 36, 32, DuelRank.BCD),
    Drop(3, 37, 26, DuelRank.BCD),
    Drop(4, 37, 26, DuelRank.BCD),
    Drop(10, 37, 26, DuelRank.BCD),
    Drop(11, 37, 24, DuelRank.BCD),
    Drop(13, 37, 26, DuelRank.BCD),
    Drop(14, 37, 26, DuelRank.BCD),
    Drop(15, 37, 26, DuelRank.BCD),
    Drop(19, 37, 6, DuelRank.BCD),
    Drop(23, 37, 26, DuelRank.BCD),
    Drop(26, 37, 26, DuelRank.BCD),
    Drop(27, 37, 26, DuelRank.BCD),
    Drop(29, 37, 26, DuelRank.BCD),
    Drop(31, 37, 26, DuelRank.BCD),
    Drop(33, 37, 24, DuelRank.BCD),
    Drop(39, 37, 56, DuelRank.BCD),
    Drop(40, 37, 58, DuelRank.BCD),
    Drop(42, 37, 72, DuelRank.BCD),
    Drop(44, 37, 48, DuelRank.BCD),
    Drop(46, 37, 24, DuelRank.BCD),
    Drop(47, 37, 24, DuelRank.BCD),
    Drop(61, 37, 24, DuelRank.BCD),
    Drop(65, 37, 24, DuelRank.BCD),
    Drop(68, 37, 24, DuelRank.BCD),
    Drop(79, 37, 22, DuelRank.BCD),
    Drop(85, 37, 22, DuelRank.BCD),
    Drop(90, 37, 22, DuelRank.BCD),
    Drop(91, 37, 24, DuelRank.BCD),
    Drop(99, 37, 22, DuelRank.BCD),
    Drop(113, 37, 24, DuelRank.BCD),
    Drop(121, 37, 24, DuelRank.BCD),
    Drop(125, 37, 24, DuelRank.BCD),
    Drop(155, 37, 24, DuelRank.BCD),
    Drop(168, 37, 24, DuelRank.BCD),
    Drop(186, 37, 24, DuelRank.BCD),
    Drop(188, 37, 24, DuelRank.BCD),
    Drop(189, 37, 24, DuelRank.BCD),
    Drop(200, 37, 24, DuelRank.BCD),
    Drop(201, 37, 24, DuelRank.BCD),
    Drop(219, 37, 24, DuelRank.BCD),
    Drop(246, 37, 24, DuelRank.BCD),
    Drop(248, 37, 22, DuelRank.BCD),
    Drop(255, 37, 22, DuelRank.BCD),
    Drop(272, 37, 22, DuelRank.BCD),
    Drop(282, 37, 22, DuelRank.BCD),
    Drop(296, 37, 22, DuelRank.BCD),
    Drop(298, 37, 22, DuelRank.BCD),
    Drop(376, 37, 22, DuelRank.BCD),
    Drop(388, 37, 22, DuelRank.BCD),
    Drop(403, 37, 22, DuelRank.BCD),
    Drop(419, 37, 22, DuelRank.BCD),
    Drop(423, 37, 22, DuelRank.BCD),
    Drop(438, 37, 22, DuelRank.BCD),
    Drop(443, 37, 22, DuelRank.BCD),
    Drop(464, 37, 22, DuelRank.BCD),
    Drop(465, 37, 22, DuelRank.BCD),
    Drop(466, 37, 22, DuelRank.BCD),
    Drop(468, 37, 22, DuelRank.BCD),
    Drop(472, 37, 22, DuelRank.BCD),
    Drop(481, 37, 22, DuelRank.BCD),
    Drop(483, 37, 46, DuelRank.BCD),
    Drop(491, 37, 22, DuelRank.BCD),
    Drop(496, 37, 22, DuelRank.BCD),
    Drop(522, 37, 22, DuelRank.BCD),
    Drop(526, 37, 22, DuelRank.BCD),
    Drop(528, 37, 46, DuelRank.BCD),
    Drop(531, 37, 22, DuelRank.BCD),
    Drop(538, 37, 22, DuelRank.BCD),
    Drop(552, 37, 22, DuelRank.BCD),
    Drop(571, 37, 22, DuelRank.BCD),
    Drop(572, 37, 22, DuelRank.BCD),
    Drop(575, 37, 46, DuelRank.BCD),
    Drop(577, 37, 46, DuelRank.BCD),
    Drop(582, 37, 22, DuelRank.BCD),
    Drop(594, 37, 22, DuelRank.BCD),
    Drop(597, 37, 22, DuelRank.BCD),
    Drop(627, 37, 22, DuelRank.BCD),
    Drop(629, 37, 22, DuelRank.BCD),
    Drop(636, 37, 22, DuelRank.BCD),
    Drop(642, 37, 22, DuelRank.BCD),
    Drop(4, 38, 48, DuelRank.BCD),
    Drop(10, 38, 48, DuelRank.BCD),
    Drop(20, 38, 6, DuelRank.BCD),
    Drop(31, 38, 48, DuelRank.BCD),
    Drop(39, 38, 34, DuelRank.BCD),
    Drop(70, 38, 48, DuelRank.BCD),
    Drop(71, 38, 48, DuelRank.BCD),
    Drop(79, 38, 34, DuelRank.BCD),
    Drop(89, 38, 48, DuelRank.BCD),
    Drop(90, 38, 34, DuelRank.BCD),
    Drop(99, 38, 34, DuelRank.BCD),
    Drop(131, 38, 48, DuelRank.BCD),
    Drop(140, 38, 48, DuelRank.BCD),
    Drop(150, 38, 48, DuelRank.BCD),
    Drop(168, 38, 48, DuelRank.BCD),
    Drop(177, 38, 40, DuelRank.BCD),
    Drop(193, 38, 40, DuelRank.BCD),
    Drop(196, 38, 40, DuelRank.BCD),
    Drop(199, 38, 40, DuelRank.BCD),
    Drop(200, 38, 40, DuelRank.BCD),
    Drop(205, 38, 38, DuelRank.BCD),
    Drop(206, 38, 38, DuelRank.BCD),
    Drop(223, 38, 66, DuelRank.BCD),
    Drop(230, 38, 38, DuelRank.BCD),
    Drop(243, 38, 38, DuelRank.BCD),
    Drop(247, 38, 38, DuelRank.BCD),
    Drop(251, 38, 38, DuelRank.BCD),
    Drop(258, 38, 38, DuelRank.BCD),
    Drop(270, 38, 38, DuelRank.BCD),
    Drop(296, 38, 38, DuelRank.BCD),
    Drop(298, 38, 38, DuelRank.BCD),
    Drop(430, 38, 58, DuelRank.BCD),
    Drop(431, 38, 58, DuelRank.BCD),
    Drop(434, 38, 14, DuelRank.BCD),
    Drop(440, 38, 32, DuelRank.BCD),
    Drop(465, 38, 32, DuelRank.BCD),
    Drop(494, 38, 14, DuelRank.BCD),
    Drop(519, 38, 14, DuelRank.BCD),
    Drop(520, 38, 14, DuelRank.BCD),
    Drop(522, 38, 32, DuelRank.BCD),
    Drop(530, 38, 58, DuelRank.BCD),
    Drop(531, 38, 32, DuelRank.BCD),
    Drop(550, 38, 58, DuelRank.BCD),
    Drop(571, 38, 66, DuelRank.BCD),
    Drop(572, 38, 32, DuelRank.BCD),
    Drop(594, 38, 32, DuelRank.BCD),
    Drop(596, 38, 35, DuelRank.BCD),
    Drop(615, 38, 48, DuelRank.BCD),
    Drop(617, 38, 14, DuelRank.BCD),
    Drop(625, 38, 35, DuelRank.BCD),
    Drop(626, 38, 35, DuelRank.BCD),
    Drop(627, 38, 32, DuelRank.BCD),
    Drop(649, 38, 35, DuelRank.BCD),
    Drop(4, 39, 18, DuelRank.BCD),
    Drop(9, 39, 18, DuelRank.BCD),
    Drop(20, 39, 8, DuelRank.BCD),
    Drop(24, 39, 18, DuelRank.BCD),
    Drop(40, 39, 18, DuelRank.BCD),
    Drop(45, 39, 18, DuelRank.BCD),
    Drop(47, 39, 18, DuelRank.BCD),
    Drop(59, 39, 18, DuelRank.BCD),
    Drop(61, 39, 18, DuelRank.BCD),
    Drop(65, 39, 18, DuelRank.BCD),
    Drop(96, 39, 18, DuelRank.BCD),
    Drop(97, 39, 18, DuelRank.BCD),
    Drop(98, 39, 18, DuelRank.BCD),
    Drop(108, 39, 18, DuelRank.BCD),
    Drop(118, 39, 18, DuelRank.BCD),
    Drop(133, 39, 18, DuelRank.BCD),
    Drop(135, 39, 18, DuelRank.BCD),
    Drop(138, 39, 18, DuelRank.BCD),
    Drop(139, 39, 18, DuelRank.BCD),
    Drop(140, 39, 36, DuelRank.BCD),
    Drop(141, 39, 18, DuelRank.BCD),
    Drop(142, 39, 18, DuelRank.BCD),
    Drop(152, 39, 18, DuelRank.BCD),
    Drop(154, 39, 18, DuelRank.BCD),
    Drop(155, 39, 18, DuelRank.BCD),
    Drop(162, 39, 18, DuelRank.BCD),
    Drop(167, 39, 18, DuelRank.BCD),
    Drop(169, 39, 18, DuelRank.BCD),
    Drop(172, 39, 18, DuelRank.BCD),
    Drop(177, 39, 24, DuelRank.BCD),
    Drop(178, 39, 18, DuelRank.BCD),
    Drop(180, 39, 18, DuelRank.BCD),
    Drop(181, 39, 18, DuelRank.BCD),
    Drop(190, 39, 18, DuelRank.BCD),
    Drop(191, 39, 18, DuelRank.BCD),
    Drop(197, 39, 18, DuelRank.BCD),
    Drop(199, 39, 18, DuelRank.BCD),
    Drop(203, 39, 18, DuelRank.BCD),
    Drop(205, 39, 18, DuelRank.BCD),
    Drop(206, 39, 18, DuelRank.BCD),
    Drop(211, 39, 18, DuelRank.BCD),
    Drop(215, 39, 36, DuelRank.BCD),
    Drop(219, 39, 18, DuelRank.BCD),
    Drop(227, 39, 18, DuelRank.BCD),
    Drop(228, 39, 36, DuelRank.BCD),
    Drop(231, 39, 18, DuelRank.BCD),
    Drop(237, 39, 18, DuelRank.BCD),
    Drop(243, 39, 18, DuelRank.BCD),
    Drop(244, 39, 36, DuelRank.BCD),
    Drop(250, 39, 18, DuelRank.BCD),
    Drop(253, 39, 18, DuelRank.BCD),
    Drop(257, 39, 36, DuelRank.BCD),
    Drop(260, 39, 18, DuelRank.BCD),
    Drop(262, 39, 18, DuelRank.BCD),
    Drop(263, 39, 36, DuelRank.BCD),
    Drop(264, 39, 18, DuelRank.BCD),
    Drop(265, 39, 18, DuelRank.BCD),
    Drop(269, 39, 18, DuelRank.BCD),
    Drop(270, 39, 36, DuelRank.BCD),
    Drop(276, 39, 18, DuelRank.BCD),
    Drop(279, 39, 18, DuelRank.BCD),
    Drop(282, 39, 18, DuelRank.BCD),
    Drop(283, 39, 18, DuelRank.BCD),
    Drop(289, 39, 18, DuelRank.BCD),
    Drop(292, 39, 18, DuelRank.BCD),
    Drop(295, 39, 18, DuelRank.BCD),
    Drop(296, 39, 18, DuelRank.BCD),
    Drop(298, 39, 18, DuelRank.BCD),
    Drop(304, 39, 24, DuelRank.BCD),
    Drop(335, 39, 48, DuelRank.BCD),
    Drop(393, 39, 16, DuelRank.BCD),
    Drop(394, 39, 16, DuelRank.BCD),
    Drop(398, 39, 18, DuelRank.BCD),
    Drop(432, 39, 36, DuelRank.BCD),
    Drop(444, 39, 18, DuelRank.BCD),
    Drop(446, 39, 36, DuelRank.BCD),
    Drop(451, 39, 18, DuelRank.BCD),
    Drop(452, 39, 18, DuelRank.BCD),
    Drop(461, 39, 36, DuelRank.BCD),
    Drop(463, 39, 18, DuelRank.BCD),
    Drop(478, 39, 18, DuelRank.BCD),
    Drop(484, 39, 18, DuelRank.BCD),
    Drop(503, 39, 18, DuelRank.BCD),
    Drop(516, 39, 18, DuelRank.BCD),
    Drop(524, 39, 18, DuelRank.BCD),
    Drop(548, 39, 18, DuelRank.BCD),
    Drop(549, 39, 18, DuelRank.BCD),
    Drop(556, 39, 18, DuelRank.BCD),
    Drop(558, 39, 18, DuelRank.BCD),
    Drop(568, 39, 18, DuelRank.BCD),
    Drop(573, 39, 18, DuelRank.BCD),
    Drop(585, 39, 18, DuelRank.BCD),
    Drop(588, 39, 16, DuelRank.BCD),
    Drop(591, 39, 18, DuelRank.BCD),
    Drop(592, 39, 16, DuelRank.BCD),
    Drop(602, 39, 18, DuelRank.BCD),
    Drop(605, 39, 18, DuelRank.BCD),
    Drop(606, 39, 18, DuelRank.BCD),
    Drop(612, 39, 18, DuelRank.BCD),
    Drop(629, 39, 18, DuelRank.BCD),
    Drop(642, 39, 18, DuelRank.BCD),
    Drop(652, 39, 26, DuelRank.BCD),
    # SATec drops
    Drop(9, 1, 92, DuelRank.SATEC),
    Drop(16, 1, 54, DuelRank.SATEC),
    Drop(19, 1, 6, DuelRank.SATEC),
    Drop(24, 1, 27, DuelRank.SATEC),
    Drop(47, 1, 27, DuelRank.SATEC),
    Drop(48, 1, 27, DuelRank.SATEC),
    Drop(58, 1, 27, DuelRank.SATEC),
    Drop(102, 1, 12, DuelRank.SATEC),
    Drop(105, 1, 53, DuelRank.SATEC),
    Drop(123, 1, 53, DuelRank.SATEC),
    Drop(130, 1, 53, DuelRank.SATEC),
    Drop(137, 1, 53, DuelRank.SATEC),
    Drop(167, 1, 52, DuelRank.SATEC),
    Drop(192, 1, 52, DuelRank.SATEC),
    Drop(197, 1, 52, DuelRank.SATEC),
    Drop(202, 1, 52, DuelRank.SATEC),
    Drop(237, 1, 52, DuelRank.SATEC),
    Drop(238, 1, 12, DuelRank.SATEC),
    Drop(289, 1, 52, DuelRank.SATEC),
    Drop(301, 1, 20, DuelRank.SATEC),
    Drop(313, 1, 20, DuelRank.SATEC),
    Drop(314, 1, 20, DuelRank.SATEC),
    Drop(333, 1, 36, DuelRank.SATEC),
    Drop(336, 1, 20, DuelRank.SATEC),
    Drop(349, 1, 36, DuelRank.SATEC),
    Drop(387, 1, 52, DuelRank.SATEC),
    Drop(397, 1, 52, DuelRank.SATEC),
    Drop(402, 1, 52, DuelRank.SATEC),
    Drop(410, 1, 52, DuelRank.SATEC),
    Drop(411, 1, 52, DuelRank.SATEC),
    Drop(422, 1, 52, DuelRank.SATEC),
    Drop(436, 1, 52, DuelRank.SATEC),
    Drop(444, 1, 52, DuelRank.SATEC),
    Drop(469, 1, 52, DuelRank.SATEC),
    Drop(484, 1, 52, DuelRank.SATEC),
    Drop(485, 1, 52, DuelRank.SATEC),
    Drop(488, 1, 52, DuelRank.SATEC),
    Drop(504, 1, 52, DuelRank.SATEC),
    Drop(516, 1, 52, DuelRank.SATEC),
    Drop(547, 1, 52, DuelRank.SATEC),
    Drop(548, 1, 52, DuelRank.SATEC),
    Drop(558, 1, 52, DuelRank.SATEC),
    Drop(563, 1, 52, DuelRank.SATEC),
    Drop(635, 1, 52, DuelRank.SATEC),
    Drop(676, 1, 20, DuelRank.SATEC),
    Drop(677, 1, 20, DuelRank.SATEC),
    Drop(681, 1, 20, DuelRank.SATEC),
    Drop(682, 1, 20, DuelRank.SATEC),
    Drop(690, 1, 20, DuelRank.SATEC),
    Drop(19, 2, 6, DuelRank.SATEC),
    Drop(24, 2, 170, DuelRank.SATEC),
    Drop(58, 2, 170, DuelRank.SATEC),
    Drop(167, 2, 168, DuelRank.SATEC),
    Drop(302, 2, 16, DuelRank.SATEC),
    Drop(312, 2, 15, DuelRank.SATEC),
    Drop(313, 2, 15, DuelRank.SATEC),
    Drop(330, 2, 30, DuelRank.SATEC),
    Drop(336, 2, 15, DuelRank.SATEC),
    Drop(345, 2, 15, DuelRank.SATEC),
    Drop(350, 2, 30, DuelRank.SATEC),
    Drop(393, 2, 170, DuelRank.SATEC),
    Drop(395, 2, 170, DuelRank.SATEC),
    Drop(398, 2, 170, DuelRank.SATEC),
    Drop(399, 2, 170, DuelRank.SATEC),
    Drop(402, 2, 168, DuelRank.SATEC),
    Drop(469, 2, 168, DuelRank.SATEC),
    Drop(475, 2, 168, DuelRank.SATEC),
    Drop(527, 2, 168, DuelRank.SATEC),
    Drop(681, 2, 15, DuelRank.SATEC),
    Drop(682, 2, 15, DuelRank.SATEC),
    Drop(690, 2, 15, DuelRank.SATEC),
    Drop(699, 2, 1, DuelRank.SATEC),
    Drop(16, 3, 60, DuelRank.SATEC),
    Drop(20, 3, 4, DuelRank.SATEC),
    Drop(50, 3, 40, DuelRank.SATEC),
    Drop(104, 3, 40, DuelRank.SATEC),
    Drop(105, 3, 40, DuelRank.SATEC),
    Drop(122, 3, 40, DuelRank.SATEC),
    Drop(130, 3, 40, DuelRank.SATEC),
    Drop(135, 3, 40, DuelRank.SATEC),
    Drop(137, 3, 40, DuelRank.SATEC),
    Drop(152, 3, 40, DuelRank.SATEC),
    Drop(154, 3, 40, DuelRank.SATEC),
    Drop(174, 3, 40, DuelRank.SATEC),
    Drop(182, 3, 40, DuelRank.SATEC),
    Drop(185, 3, 40, DuelRank.SATEC),
    Drop(191, 3, 40, DuelRank.SATEC),
    Drop(197, 3, 40, DuelRank.SATEC),
    Drop(202, 3, 40, DuelRank.SATEC),
    Drop(207, 3, 40, DuelRank.SATEC),
    Drop(210, 3, 40, DuelRank.SATEC),
    Drop(214, 3, 40, DuelRank.SATEC),
    Drop(237, 3, 40, DuelRank.SATEC),
    Drop(242, 3, 42, DuelRank.SATEC),
    Drop(245, 3, 42, DuelRank.SATEC),
    Drop(268, 3, 42, DuelRank.SATEC),
    Drop(301, 3, 16, DuelRank.SATEC),
    Drop(302, 3, 16, DuelRank.SATEC),
    Drop(310, 3, 16, DuelRank.SATEC),
    Drop(311, 3, 16, DuelRank.SATEC),
    Drop(333, 3, 36, DuelRank.SATEC),
    Drop(336, 3, 16, DuelRank.SATEC),
    Drop(343, 3, 16, DuelRank.SATEC),
    Drop(344, 3, 18, DuelRank.SATEC),
    Drop(410, 3, 42, DuelRank.SATEC),
    Drop(420, 3, 42, DuelRank.SATEC),
    Drop(422, 3, 42, DuelRank.SATEC),
    Drop(436, 3, 42, DuelRank.SATEC),
    Drop(444, 3, 42, DuelRank.SATEC),
    Drop(485, 3, 42, DuelRank.SATEC),
    Drop(486, 3, 42, DuelRank.SATEC),
    Drop(488, 3, 42, DuelRank.SATEC),
    Drop(492, 3, 42, DuelRank.SATEC),
    Drop(501, 3, 42, DuelRank.SATEC),
    Drop(516, 3, 42, DuelRank.SATEC),
    Drop(524, 3, 42, DuelRank.SATEC),
    Drop(549, 3, 42, DuelRank.SATEC),
    Drop(558, 3, 42, DuelRank.SATEC),
    Drop(563, 3, 42, DuelRank.SATEC),
    Drop(579, 3, 42, DuelRank.SATEC),
    Drop(589, 3, 42, DuelRank.SATEC),
    Drop(598, 3, 42, DuelRank.SATEC),
    Drop(611, 3, 42, DuelRank.SATEC),
    Drop(651, 3, 35, DuelRank.SATEC),
    Drop(654, 3, 35, DuelRank.SATEC),
    Drop(677, 3, 16, DuelRank.SATEC),
    Drop(679, 3, 16, DuelRank.SATEC),
    Drop(681, 3, 16, DuelRank.SATEC),
    Drop(683, 3, 16, DuelRank.SATEC),
    Drop(690, 3, 16, DuelRank.SATEC),
    Drop(9, 4, 60, DuelRank.SATEC),
    Drop(20, 4, 6, DuelRank.SATEC),
    Drop(24, 4, 60, DuelRank.SATEC),
    Drop(58, 4, 60, DuelRank.SATEC),
    Drop(123, 4, 60, DuelRank.SATEC),
    Drop(167, 4, 60, DuelRank.SATEC),
    Drop(192, 4, 60, DuelRank.SATEC),
    Drop(289, 4, 60, DuelRank.SATEC),
    Drop(307, 4, 16, DuelRank.SATEC),
    Drop(308, 4, 16, DuelRank.SATEC),
    Drop(309, 4, 16, DuelRank.SATEC),
    Drop(311, 4, 16, DuelRank.SATEC),
    Drop(312, 4, 16, DuelRank.SATEC),
    Drop(323, 4, 16, DuelRank.SATEC),
    Drop(327, 4, 16, DuelRank.SATEC),
    Drop(332, 4, 32, DuelRank.SATEC),
    Drop(336, 4, 18, DuelRank.SATEC),
    Drop(338, 4, 32, DuelRank.SATEC),
    Drop(339, 4, 32, DuelRank.SATEC),
    Drop(340, 4, 32, DuelRank.SATEC),
    Drop(345, 4, 18, DuelRank.SATEC),
    Drop(350, 4, 32, DuelRank.SATEC),
    Drop(387, 4, 60, DuelRank.SATEC),
    Drop(393, 4, 60, DuelRank.SATEC),
    Drop(395, 4, 60, DuelRank.SATEC),
    Drop(397, 4, 60, DuelRank.SATEC),
    Drop(398, 4, 60, DuelRank.SATEC),
    Drop(399, 4, 60, DuelRank.SATEC),
    Drop(402, 4, 60, DuelRank.SATEC),
    Drop(411, 4, 60, DuelRank.SATEC),
    Drop(469, 4, 60, DuelRank.SATEC),
    Drop(475, 4, 60, DuelRank.SATEC),
    Drop(484, 4, 60, DuelRank.SATEC),
    Drop(504, 4, 60, DuelRank.SATEC),
    Drop(527, 4, 60, DuelRank.SATEC),
    Drop(547, 4, 60, DuelRank.SATEC),
    Drop(548, 4, 60, DuelRank.SATEC),
    Drop(558, 4, 60, DuelRank.SATEC),
    Drop(635, 4, 60, DuelRank.SATEC),
    Drop(655, 4, 20, DuelRank.SATEC),
    Drop(665, 4, 12, DuelRank.SATEC),
    Drop(666, 4, 12, DuelRank.SATEC),
    Drop(671, 4, 12, DuelRank.SATEC),
    Drop(673, 4, 12, DuelRank.SATEC),
    Drop(674, 4, 12, DuelRank.SATEC),
    Drop(676, 4, 12, DuelRank.SATEC),
    Drop(677, 4, 12, DuelRank.SATEC),
    Drop(678, 4, 12, DuelRank.SATEC),
    Drop(679, 4, 12, DuelRank.SATEC),
    Drop(680, 4, 12, DuelRank.SATEC),
    Drop(681, 4, 24, DuelRank.SATEC),
    Drop(690, 4, 22, DuelRank.SATEC),
    Drop(691, 4, 12, DuelRank.SATEC),
    Drop(692, 4, 12, DuelRank.SATEC),
    Drop(693, 4, 12, DuelRank.SATEC),
    Drop(694, 4, 12, DuelRank.SATEC),
    Drop(695, 4, 12, DuelRank.SATEC),
    Drop(697, 4, 12, DuelRank.SATEC),
    Drop(698, 4, 12, DuelRank.SATEC),
    Drop(699, 4, 12, DuelRank.SATEC),
    Drop(700, 4, 12, DuelRank.SATEC),
    Drop(9, 5, 25, DuelRank.SATEC),
    Drop(19, 5, 12, DuelRank.SATEC),
    Drop(24, 5, 25, DuelRank.SATEC),
    Drop(45, 5, 25, DuelRank.SATEC),
    Drop(96, 5, 25, DuelRank.SATEC),
    Drop(97, 5, 25, DuelRank.SATEC),
    Drop(98, 5, 25, DuelRank.SATEC),
    Drop(102, 5, 12, DuelRank.SATEC),
    Drop(108, 5, 25, DuelRank.SATEC),
    Drop(135, 5, 25, DuelRank.SATEC),
    Drop(139, 5, 25, DuelRank.SATEC),
    Drop(152, 5, 23, DuelRank.SATEC),
    Drop(154, 5, 23, DuelRank.SATEC),
    Drop(167, 5, 23, DuelRank.SATEC),
    Drop(177, 5, 23, DuelRank.SATEC),
    Drop(191, 5, 23, DuelRank.SATEC),
    Drop(197, 5, 23, DuelRank.SATEC),
    Drop(199, 5, 23, DuelRank.SATEC),
    Drop(203, 5, 23, DuelRank.SATEC),
    Drop(205, 5, 23, DuelRank.SATEC),
    Drop(206, 5, 23, DuelRank.SATEC),
    Drop(211, 5, 23, DuelRank.SATEC),
    Drop(215, 5, 23, DuelRank.SATEC),
    Drop(227, 5, 23, DuelRank.SATEC),
    Drop(228, 5, 23, DuelRank.SATEC),
    Drop(237, 5, 23, DuelRank.SATEC),
    Drop(238, 5, 12, DuelRank.SATEC),
    Drop(243, 5, 23, DuelRank.SATEC),
    Drop(244, 5, 23, DuelRank.SATEC),
    Drop(259, 5, 23, DuelRank.SATEC),
    Drop(262, 5, 23, DuelRank.SATEC),
    Drop(265, 5, 23, DuelRank.SATEC),
    Drop(270, 5, 23, DuelRank.SATEC),
    Drop(289, 5, 23, DuelRank.SATEC),
    Drop(292, 5, 23, DuelRank.SATEC),
    Drop(304, 5, 32, DuelRank.SATEC),
    Drop(305, 5, 32, DuelRank.SATEC),
    Drop(316, 5, 32, DuelRank.SATEC),
    Drop(319, 5, 32, DuelRank.SATEC),
    Drop(321, 5, 32, DuelRank.SATEC),
    Drop(326, 5, 32, DuelRank.SATEC),
    Drop(330, 5, 33, DuelRank.SATEC),
    Drop(333, 5, 33, DuelRank.SATEC),
    Drop(335, 5, 33, DuelRank.SATEC),
    Drop(336, 5, 32, DuelRank.SATEC),
    Drop(338, 5, 34, DuelRank.SATEC),
    Drop(339, 5, 32, DuelRank.SATEC),
    Drop(340, 5, 32, DuelRank.SATEC),
    Drop(345, 5, 32, DuelRank.SATEC),
    Drop(350, 5, 32, DuelRank.SATEC),
    Drop(393, 5, 23, DuelRank.SATEC),
    Drop(394, 5, 12, DuelRank.SATEC),
    Drop(398, 5, 23, DuelRank.SATEC),
    Drop(432, 5, 23, DuelRank.SATEC),
    Drop(444, 5, 23, DuelRank.SATEC),
    Drop(446, 5, 23, DuelRank.SATEC),
    Drop(451, 5, 23, DuelRank.SATEC),
    Drop(452, 5, 23, DuelRank.SATEC),
    Drop(461, 5, 23, DuelRank.SATEC),
    Drop(463, 5, 23, DuelRank.SATEC),
    Drop(484, 5, 23, DuelRank.SATEC),
    Drop(503, 5, 23, DuelRank.SATEC),
    Drop(516, 5, 23, DuelRank.SATEC),
    Drop(524, 5, 23, DuelRank.SATEC),
    Drop(548, 5, 23, DuelRank.SATEC),
    Drop(549, 5, 23, DuelRank.SATEC),
    Drop(556, 5, 23, DuelRank.SATEC),
    Drop(558, 5, 23, DuelRank.SATEC),
    Drop(586, 5, 10, DuelRank.SATEC),
    Drop(591, 5, 23, DuelRank.SATEC),
    Drop(592, 5, 10, DuelRank.SATEC),
    Drop(602, 5, 23, DuelRank.SATEC),
    Drop(605, 5, 23, DuelRank.SATEC),
    Drop(606, 5, 23, DuelRank.SATEC),
    Drop(610, 5, 2, DuelRank.SATEC),
    Drop(652, 5, 32, DuelRank.SATEC),
    Drop(655, 5, 32, DuelRank.SATEC),
    Drop(683, 5, 32, DuelRank.SATEC),
    Drop(684, 5, 32, DuelRank.SATEC),
    Drop(687, 5, 32, DuelRank.SATEC),
    Drop(688, 5, 32, DuelRank.SATEC),
    Drop(690, 5, 32, DuelRank.SATEC),
    Drop(693, 5, 32, DuelRank.SATEC),
    Drop(4, 6, 15, DuelRank.SATEC),
    Drop(9, 6, 15, DuelRank.SATEC),
    Drop(20, 6, 4, DuelRank.SATEC),
    Drop(24, 6, 15, DuelRank.SATEC),
    Drop(45, 6, 15, DuelRank.SATEC),
    Drop(47, 6, 15, DuelRank.SATEC),
    Drop(96, 6, 15, DuelRank.SATEC),
    Drop(97, 6, 15, DuelRank.SATEC),
    Drop(98, 6, 17, DuelRank.SATEC),
    Drop(108, 6, 17, DuelRank.SATEC),
    Drop(133, 6, 17, DuelRank.SATEC),
    Drop(135, 6, 17, DuelRank.SATEC),
    Drop(139, 6, 17, DuelRank.SATEC),
    Drop(140, 6, 20, DuelRank.SATEC),
    Drop(142, 6, 17, DuelRank.SATEC),
    Drop(152, 6, 17, DuelRank.SATEC),
    Drop(154, 6, 17, DuelRank.SATEC),
    Drop(155, 6, 17, DuelRank.SATEC),
    Drop(167, 6, 17, DuelRank.SATEC),
    Drop(177, 6, 26, DuelRank.SATEC),
    Drop(178, 6, 16, DuelRank.SATEC),
    Drop(180, 6, 16, DuelRank.SATEC),
    Drop(190, 6, 16, DuelRank.SATEC),
    Drop(191, 6, 16, DuelRank.SATEC),
    Drop(197, 6, 16, DuelRank.SATEC),
    Drop(199, 6, 16, DuelRank.SATEC),
    Drop(203, 6, 16, DuelRank.SATEC),
    Drop(205, 6, 16, DuelRank.SATEC),
    Drop(206, 6, 16, DuelRank.SATEC),
    Drop(211, 6, 16, DuelRank.SATEC),
    Drop(215, 6, 26, DuelRank.SATEC),
    Drop(227, 6, 16, DuelRank.SATEC),
    Drop(228, 6, 26, DuelRank.SATEC),
    Drop(237, 6, 16, DuelRank.SATEC),
    Drop(243, 6, 16, DuelRank.SATEC),
    Drop(244, 6, 26, DuelRank.SATEC),
    Drop(253, 6, 16, DuelRank.SATEC),
    Drop(260, 6, 16, DuelRank.SATEC),
    Drop(264, 6, 16, DuelRank.SATEC),
    Drop(265, 6, 16, DuelRank.SATEC),
    Drop(270, 6, 26, DuelRank.SATEC),
    Drop(279, 6, 16, DuelRank.SATEC),
    Drop(282, 6, 16, DuelRank.SATEC),
    Drop(283, 6, 16, DuelRank.SATEC),
    Drop(289, 6, 16, DuelRank.SATEC),
    Drop(290, 6, 16, DuelRank.SATEC),
    Drop(292, 6, 16, DuelRank.SATEC),
    Drop(295, 6, 16, DuelRank.SATEC),
    Drop(298, 6, 16, DuelRank.SATEC),
    Drop(304, 6, 32, DuelRank.SATEC),
    Drop(306, 6, 32, DuelRank.SATEC),
    Drop(317, 6, 32, DuelRank.SATEC),
    Drop(319, 6, 32, DuelRank.SATEC),
    Drop(321, 6, 32, DuelRank.SATEC),
    Drop(326, 6, 32, DuelRank.SATEC),
    Drop(331, 6, 32, DuelRank.SATEC),
    Drop(334, 6, 32, DuelRank.SATEC),
    Drop(335, 6, 48, DuelRank.SATEC),
    Drop(336, 6, 32, DuelRank.SATEC),
    Drop(338, 6, 32, DuelRank.SATEC),
    Drop(339, 6, 32, DuelRank.SATEC),
    Drop(340, 6, 32, DuelRank.SATEC),
    Drop(345, 6, 32, DuelRank.SATEC),
    Drop(350, 6, 32, DuelRank.SATEC),
    Drop(393, 6, 16, DuelRank.SATEC),
    Drop(394, 6, 12, DuelRank.SATEC),
    Drop(398, 6, 16, DuelRank.SATEC),
    Drop(432, 6, 26, DuelRank.SATEC),
    Drop(444, 6, 16, DuelRank.SATEC),
    Drop(446, 6, 26, DuelRank.SATEC),
    Drop(451, 6, 16, DuelRank.SATEC),
    Drop(452, 6, 16, DuelRank.SATEC),
    Drop(461, 6, 26, DuelRank.SATEC),
    Drop(463, 6, 16, DuelRank.SATEC),
    Drop(478, 6, 16, DuelRank.SATEC),
    Drop(484, 6, 16, DuelRank.SATEC),
    Drop(503, 6, 16, DuelRank.SATEC),
    Drop(516, 6, 16, DuelRank.SATEC),
    Drop(524, 6, 16, DuelRank.SATEC),
    Drop(548, 6, 16, DuelRank.SATEC),
    Drop(549, 6, 16, DuelRank.SATEC),
    Drop(556, 6, 16, DuelRank.SATEC),
    Drop(558, 6, 16, DuelRank.SATEC),
    Drop(568, 6, 16, DuelRank.SATEC),
    Drop(573, 6, 16, DuelRank.SATEC),
    Drop(585, 6, 16, DuelRank.SATEC),
    Drop(588, 6, 12, DuelRank.SATEC),
    Drop(591, 6, 16, DuelRank.SATEC),
    Drop(592, 6, 12, DuelRank.SATEC),
    Drop(602, 6, 16, DuelRank.SATEC),
    Drop(605, 6, 16, DuelRank.SATEC),
    Drop(606, 6, 16, DuelRank.SATEC),
    Drop(612, 6, 16, DuelRank.SATEC),
    Drop(652, 6, 26, DuelRank.SATEC),
    Drop(655, 6, 23, DuelRank.SATEC),
    Drop(683, 6, 32, DuelRank.SATEC),
    Drop(684, 6, 32, DuelRank.SATEC),
    Drop(687, 6, 32, DuelRank.SATEC),
    Drop(688, 6, 32, DuelRank.SATEC),
    Drop(690, 6, 32, DuelRank.SATEC),
    Drop(693, 6, 32, DuelRank.SATEC),
    Drop(8, 7, 40, DuelRank.SATEC),
    Drop(19, 7, 5, DuelRank.SATEC),
    Drop(50, 7, 17, DuelRank.SATEC),
    Drop(75, 7, 100, DuelRank.SATEC),
    Drop(98, 7, 19, DuelRank.SATEC),
    Drop(101, 7, 40, DuelRank.SATEC),
    Drop(104, 7, 19, DuelRank.SATEC),
    Drop(107, 7, 19, DuelRank.SATEC),
    Drop(122, 7, 19, DuelRank.SATEC),
    Drop(123, 7, 19, DuelRank.SATEC),
    Drop(130, 7, 19, DuelRank.SATEC),
    Drop(133, 7, 19, DuelRank.SATEC),
    Drop(134, 7, 19, DuelRank.SATEC),
    Drop(135, 7, 19, DuelRank.SATEC),
    Drop(137, 7, 19, DuelRank.SATEC),
    Drop(145, 7, 20, DuelRank.SATEC),
    Drop(152, 7, 19, DuelRank.SATEC),
    Drop(154, 7, 19, DuelRank.SATEC),
    Drop(157, 7, 38, DuelRank.SATEC),
    Drop(158, 7, 38, DuelRank.SATEC),
    Drop(159, 7, 18, DuelRank.SATEC),
    Drop(174, 7, 18, DuelRank.SATEC),
    Drop(176, 7, 38, DuelRank.SATEC),
    Drop(179, 7, 18, DuelRank.SATEC),
    Drop(180, 7, 18, DuelRank.SATEC),
    Drop(182, 7, 18, DuelRank.SATEC),
    Drop(183, 7, 18, DuelRank.SATEC),
    Drop(185, 7, 18, DuelRank.SATEC),
    Drop(191, 7, 18, DuelRank.SATEC),
    Drop(195, 7, 18, DuelRank.SATEC),
    Drop(198, 7, 18, DuelRank.SATEC),
    Drop(200, 7, 18, DuelRank.SATEC),
    Drop(202, 7, 18, DuelRank.SATEC),
    Drop(203, 7, 18, DuelRank.SATEC),
    Drop(207, 7, 18, DuelRank.SATEC),
    Drop(209, 7, 18, DuelRank.SATEC),
    Drop(210, 7, 38, DuelRank.SATEC),
    Drop(211, 7, 18, DuelRank.SATEC),
    Drop(212, 7, 18, DuelRank.SATEC),
    Drop(214, 7, 18, DuelRank.SATEC),
    Drop(222, 7, 18, DuelRank.SATEC),
    Drop(227, 7, 18, DuelRank.SATEC),
    Drop(229, 7, 18, DuelRank.SATEC),
    Drop(232, 7, 18, DuelRank.SATEC),
    Drop(237, 7, 18, DuelRank.SATEC),
    Drop(242, 7, 18, DuelRank.SATEC),
    Drop(245, 7, 18, DuelRank.SATEC),
    Drop(254, 7, 18, DuelRank.SATEC),
    Drop(261, 7, 18, DuelRank.SATEC),
    Drop(268, 7, 18, DuelRank.SATEC),
    Drop(285, 7, 18, DuelRank.SATEC),
    Drop(292, 7, 18, DuelRank.SATEC),
    Drop(302, 7, 20, DuelRank.SATEC),
    Drop(308, 7, 20, DuelRank.SATEC),
    Drop(310, 7, 18, DuelRank.SATEC),
    Drop(324, 7, 18, DuelRank.SATEC),
    Drop(328, 7, 18, DuelRank.SATEC),
    Drop(330, 7, 42, DuelRank.SATEC),
    Drop(336, 7, 32, DuelRank.SATEC),
    Drop(337, 7, 18, DuelRank.SATEC),
    Drop(343, 7, 20, DuelRank.SATEC),
    Drop(344, 7, 20, DuelRank.SATEC),
    Drop(345, 7, 20, DuelRank.SATEC),
    Drop(420, 7, 18, DuelRank.SATEC),
    Drop(452, 7, 18, DuelRank.SATEC),
    Drop(476, 7, 18, DuelRank.SATEC),
    Drop(486, 7, 18, DuelRank.SATEC),
    Drop(488, 7, 18, DuelRank.SATEC),
    Drop(492, 7, 18, DuelRank.SATEC),
    Drop(501, 7, 18, DuelRank.SATEC),
    Drop(506, 7, 18, DuelRank.SATEC),
    Drop(510, 7, 38, DuelRank.SATEC),
    Drop(524, 7, 18, DuelRank.SATEC),
    Drop(536, 7, 18, DuelRank.SATEC),
    Drop(540, 7, 18, DuelRank.SATEC),
    Drop(544, 7, 18, DuelRank.SATEC),
    Drop(547, 7, 18, DuelRank.SATEC),
    Drop(549, 7, 18, DuelRank.SATEC),
    Drop(556, 7, 18, DuelRank.SATEC),
    Drop(579, 7, 38, DuelRank.SATEC),
    Drop(589, 7, 38, DuelRank.SATEC),
    Drop(591, 7, 18, DuelRank.SATEC),
    Drop(598, 7, 18, DuelRank.SATEC),
    Drop(602, 7, 18, DuelRank.SATEC),
    Drop(609, 7, 18, DuelRank.SATEC),
    Drop(611, 7, 18, DuelRank.SATEC),
    Drop(656, 7, 32, DuelRank.SATEC),
    Drop(666, 7, 18, DuelRank.SATEC),
    Drop(677, 7, 18, DuelRank.SATEC),
    Drop(682, 7, 20, DuelRank.SATEC),
    Drop(683, 7, 20, DuelRank.SATEC),
    Drop(684, 7, 20, DuelRank.SATEC),
    Drop(687, 7, 20, DuelRank.SATEC),
    Drop(688, 7, 20, DuelRank.SATEC),
    Drop(690, 7, 18, DuelRank.SATEC),
    Drop(2, 8, 26, DuelRank.SATEC),
    Drop(19, 8, 4, DuelRank.SATEC),
    Drop(30, 8, 26, DuelRank.SATEC),
    Drop(34, 8, 26, DuelRank.SATEC),
    Drop(36, 8, 26, DuelRank.SATEC),
    Drop(42, 8, 26, DuelRank.SATEC),
    Drop(44, 8, 26, DuelRank.SATEC),
    Drop(74, 8, 26, DuelRank.SATEC),
    Drop(96, 8, 26, DuelRank.SATEC),
    Drop(97, 8, 26, DuelRank.SATEC),
    Drop(98, 8, 26, DuelRank.SATEC),
    Drop(104, 8, 26, DuelRank.SATEC),
    Drop(106, 8, 26, DuelRank.SATEC),
    Drop(108, 8, 26, DuelRank.SATEC),
    Drop(114, 8, 26, DuelRank.SATEC),
    Drop(115, 8, 26, DuelRank.SATEC),
    Drop(129, 8, 26, DuelRank.SATEC),
    Drop(132, 8, 26, DuelRank.SATEC),
    Drop(135, 8, 26, DuelRank.SATEC),
    Drop(139, 8, 26, DuelRank.SATEC),
    Drop(142, 8, 26, DuelRank.SATEC),
    Drop(143, 8, 26, DuelRank.SATEC),
    Drop(144, 8, 26, DuelRank.SATEC),
    Drop(145, 8, 26, DuelRank.SATEC),
    Drop(154, 8, 26, DuelRank.SATEC),
    Drop(174, 8, 24, DuelRank.SATEC),
    Drop(179, 8, 24, DuelRank.SATEC),
    Drop(183, 8, 24, DuelRank.SATEC),
    Drop(184, 8, 24, DuelRank.SATEC),
    Drop(190, 8, 24, DuelRank.SATEC),
    Drop(203, 8, 24, DuelRank.SATEC),
    Drop(213, 8, 24, DuelRank.SATEC),
    Drop(215, 8, 24, DuelRank.SATEC),
    Drop(220, 8, 24, DuelRank.SATEC),
    Drop(228, 8, 24, DuelRank.SATEC),
    Drop(237, 8, 24, DuelRank.SATEC),
    Drop(244, 8, 22, DuelRank.SATEC),
    Drop(253, 8, 22, DuelRank.SATEC),
    Drop(257, 8, 22, DuelRank.SATEC),
    Drop(259, 8, 22, DuelRank.SATEC),
    Drop(263, 8, 22, DuelRank.SATEC),
    Drop(268, 8, 22, DuelRank.SATEC),
    Drop(303, 8, 24, DuelRank.SATEC),
    Drop(313, 8, 24, DuelRank.SATEC),
    Drop(315, 8, 24, DuelRank.SATEC),
    Drop(320, 8, 1, DuelRank.SATEC),
    Drop(321, 8, 24, DuelRank.SATEC),
    Drop(322, 8, 24, DuelRank.SATEC),
    Drop(323, 8, 24, DuelRank.SATEC),
    Drop(331, 8, 48, DuelRank.SATEC),
    Drop(336, 8, 48, DuelRank.SATEC),
    Drop(337, 8, 20, DuelRank.SATEC),
    Drop(342, 8, 1, DuelRank.SATEC),
    Drop(343, 8, 24, DuelRank.SATEC),
    Drop(344, 8, 24, DuelRank.SATEC),
    Drop(345, 8, 24, DuelRank.SATEC),
    Drop(347, 8, 1, DuelRank.SATEC),
    Drop(366, 8, 22, DuelRank.SATEC),
    Drop(368, 8, 22, DuelRank.SATEC),
    Drop(455, 8, 46, DuelRank.SATEC),
    Drop(457, 8, 46, DuelRank.SATEC),
    Drop(470, 8, 24, DuelRank.SATEC),
    Drop(505, 8, 46, DuelRank.SATEC),
    Drop(525, 8, 10, DuelRank.SATEC),
    Drop(532, 8, 24, DuelRank.SATEC),
    Drop(593, 8, 10, DuelRank.SATEC),
    Drop(599, 8, 46, DuelRank.SATEC),
    Drop(623, 8, 24, DuelRank.SATEC),
    Drop(631, 8, 24, DuelRank.SATEC),
    Drop(632, 8, 10, DuelRank.SATEC),
    Drop(634, 8, 24, DuelRank.SATEC),
    Drop(646, 8, 46, DuelRank.SATEC),
    Drop(657, 8, 2, DuelRank.SATEC),
    Drop(661, 8, 2, DuelRank.SATEC),
    Drop(665, 8, 24, DuelRank.SATEC),
    Drop(672, 8, 1, DuelRank.SATEC),
    Drop(676, 8, 24, DuelRank.SATEC),
    Drop(682, 8, 24, DuelRank.SATEC),
    Drop(683, 8, 24, DuelRank.SATEC),
    Drop(684, 8, 24, DuelRank.SATEC),
    Drop(686, 8, 20, DuelRank.SATEC),
    Drop(687, 8, 24, DuelRank.SATEC),
    Drop(688, 8, 24, DuelRank.SATEC),
    Drop(690, 8, 24, DuelRank.SATEC),
    Drop(693, 8, 24, DuelRank.SATEC),
    Drop(695, 8, 24, DuelRank.SATEC),
    Drop(5, 9, 18, DuelRank.SATEC),
    Drop(8, 9, 18, DuelRank.SATEC),
    Drop(19, 9, 6, DuelRank.SATEC),
    Drop(29, 9, 18, DuelRank.SATEC),
    Drop(45, 9, 18, DuelRank.SATEC),
    Drop(47, 9, 18, DuelRank.SATEC),
    Drop(48, 9, 18, DuelRank.SATEC),
    Drop(96, 9, 16, DuelRank.SATEC),
    Drop(97, 9, 16, DuelRank.SATEC),
    Drop(100, 9, 16, DuelRank.SATEC),
    Drop(105, 9, 32, DuelRank.SATEC),
    Drop(107, 9, 16, DuelRank.SATEC),
    Drop(108, 9, 16, DuelRank.SATEC),
    Drop(113, 9, 16, DuelRank.SATEC),
    Drop(114, 9, 16, DuelRank.SATEC),
    Drop(116, 9, 16, DuelRank.SATEC),
    Drop(129, 9, 16, DuelRank.SATEC),
    Drop(134, 9, 16, DuelRank.SATEC),
    Drop(139, 9, 16, DuelRank.SATEC),
    Drop(143, 9, 16, DuelRank.SATEC),
    Drop(144, 9, 16, DuelRank.SATEC),
    Drop(145, 9, 16, DuelRank.SATEC),
    Drop(147, 9, 16, DuelRank.SATEC),
    Drop(148, 9, 16, DuelRank.SATEC),
    Drop(155, 9, 16, DuelRank.SATEC),
    Drop(156, 9, 16, DuelRank.SATEC),
    Drop(158, 9, 16, DuelRank.SATEC),
    Drop(160, 9, 16, DuelRank.SATEC),
    Drop(161, 9, 16, DuelRank.SATEC),
    Drop(173, 9, 16, DuelRank.SATEC),
    Drop(175, 9, 16, DuelRank.SATEC),
    Drop(176, 9, 16, DuelRank.SATEC),
    Drop(177, 9, 16, DuelRank.SATEC),
    Drop(178, 9, 16, DuelRank.SATEC),
    Drop(180, 9, 16, DuelRank.SATEC),
    Drop(183, 9, 16, DuelRank.SATEC),
    Drop(184, 9, 16, DuelRank.SATEC),
    Drop(187, 9, 16, DuelRank.SATEC),
    Drop(188, 9, 16, DuelRank.SATEC),
    Drop(189, 9, 16, DuelRank.SATEC),
    Drop(190, 9, 16, DuelRank.SATEC),
    Drop(195, 9, 16, DuelRank.SATEC),
    Drop(199, 9, 16, DuelRank.SATEC),
    Drop(203, 9, 16, DuelRank.SATEC),
    Drop(205, 9, 16, DuelRank.SATEC),
    Drop(206, 9, 16, DuelRank.SATEC),
    Drop(208, 9, 16, DuelRank.SATEC),
    Drop(209, 9, 16, DuelRank.SATEC),
    Drop(215, 9, 16, DuelRank.SATEC),
    Drop(218, 9, 48, DuelRank.SATEC),
    Drop(222, 9, 16, DuelRank.SATEC),
    Drop(224, 9, 16, DuelRank.SATEC),
    Drop(226, 9, 16, DuelRank.SATEC),
    Drop(232, 9, 16, DuelRank.SATEC),
    Drop(239, 9, 16, DuelRank.SATEC),
    Drop(240, 9, 16, DuelRank.SATEC),
    Drop(243, 9, 16, DuelRank.SATEC),
    Drop(247, 9, 16, DuelRank.SATEC),
    Drop(253, 9, 16, DuelRank.SATEC),
    Drop(264, 9, 16, DuelRank.SATEC),
    Drop(265, 9, 16, DuelRank.SATEC),
    Drop(266, 9, 16, DuelRank.SATEC),
    Drop(267, 9, 16, DuelRank.SATEC),
    Drop(271, 9, 16, DuelRank.SATEC),
    Drop(277, 9, 16, DuelRank.SATEC),
    Drop(282, 9, 16, DuelRank.SATEC),
    Drop(283, 9, 16, DuelRank.SATEC),
    Drop(292, 9, 16, DuelRank.SATEC),
    Drop(298, 9, 16, DuelRank.SATEC),
    Drop(300, 9, 16, DuelRank.SATEC),
    Drop(308, 9, 36, DuelRank.SATEC),
    Drop(324, 9, 36, DuelRank.SATEC),
    Drop(326, 9, 36, DuelRank.SATEC),
    Drop(331, 9, 64, DuelRank.SATEC),
    Drop(332, 9, 64, DuelRank.SATEC),
    Drop(417, 9, 16, DuelRank.SATEC),
    Drop(421, 9, 16, DuelRank.SATEC),
    Drop(432, 9, 16, DuelRank.SATEC),
    Drop(451, 9, 16, DuelRank.SATEC),
    Drop(463, 9, 16, DuelRank.SATEC),
    Drop(476, 9, 16, DuelRank.SATEC),
    Drop(477, 9, 16, DuelRank.SATEC),
    Drop(478, 9, 16, DuelRank.SATEC),
    Drop(479, 9, 16, DuelRank.SATEC),
    Drop(481, 9, 16, DuelRank.SATEC),
    Drop(490, 9, 16, DuelRank.SATEC),
    Drop(503, 9, 16, DuelRank.SATEC),
    Drop(510, 9, 16, DuelRank.SATEC),
    Drop(534, 9, 16, DuelRank.SATEC),
    Drop(536, 9, 16, DuelRank.SATEC),
    Drop(538, 9, 16, DuelRank.SATEC),
    Drop(544, 9, 16, DuelRank.SATEC),
    Drop(556, 9, 16, DuelRank.SATEC),
    Drop(568, 9, 26, DuelRank.SATEC),
    Drop(569, 9, 16, DuelRank.SATEC),
    Drop(573, 9, 52, DuelRank.SATEC),
    Drop(585, 9, 16, DuelRank.SATEC),
    Drop(590, 9, 16, DuelRank.SATEC),
    Drop(604, 9, 16, DuelRank.SATEC),
    Drop(605, 9, 16, DuelRank.SATEC),
    Drop(606, 9, 16, DuelRank.SATEC),
    Drop(679, 9, 32, DuelRank.SATEC),
    Drop(682, 9, 36, DuelRank.SATEC),
    Drop(683, 9, 48, DuelRank.SATEC),
    Drop(690, 9, 32, DuelRank.SATEC),
    Drop(699, 9, 32, DuelRank.SATEC),
    Drop(8, 10, 48, DuelRank.SATEC),
    Drop(19, 10, 4, DuelRank.SATEC),
    Drop(50, 10, 250, DuelRank.SATEC),
    Drop(116, 10, 40, DuelRank.SATEC),
    Drop(123, 10, 40, DuelRank.SATEC),
    Drop(141, 10, 42, DuelRank.SATEC),
    Drop(157, 10, 40, DuelRank.SATEC),
    Drop(158, 10, 40, DuelRank.SATEC),
    Drop(180, 10, 40, DuelRank.SATEC),
    Drop(209, 10, 40, DuelRank.SATEC),
    Drop(305, 10, 60, DuelRank.SATEC),
    Drop(306, 10, 60, DuelRank.SATEC),
    Drop(310, 10, 32, DuelRank.SATEC),
    Drop(325, 10, 32, DuelRank.SATEC),
    Drop(330, 10, 60, DuelRank.SATEC),
    Drop(336, 10, 32, DuelRank.SATEC),
    Drop(338, 10, 32, DuelRank.SATEC),
    Drop(339, 10, 32, DuelRank.SATEC),
    Drop(340, 10, 32, DuelRank.SATEC),
    Drop(343, 10, 32, DuelRank.SATEC),
    Drop(344, 10, 32, DuelRank.SATEC),
    Drop(345, 10, 32, DuelRank.SATEC),
    Drop(350, 10, 32, DuelRank.SATEC),
    Drop(397, 10, 24, DuelRank.SATEC),
    Drop(476, 10, 42, DuelRank.SATEC),
    Drop(477, 10, 42, DuelRank.SATEC),
    Drop(478, 10, 42, DuelRank.SATEC),
    Drop(479, 10, 42, DuelRank.SATEC),
    Drop(485, 10, 42, DuelRank.SATEC),
    Drop(488, 10, 42, DuelRank.SATEC),
    Drop(501, 10, 42, DuelRank.SATEC),
    Drop(506, 10, 42, DuelRank.SATEC),
    Drop(510, 10, 42, DuelRank.SATEC),
    Drop(534, 10, 42, DuelRank.SATEC),
    Drop(547, 10, 42, DuelRank.SATEC),
    Drop(579, 10, 42, DuelRank.SATEC),
    Drop(589, 10, 42, DuelRank.SATEC),
    Drop(609, 10, 42, DuelRank.SATEC),
    Drop(662, 10, 64, DuelRank.SATEC),
    Drop(666, 10, 32, DuelRank.SATEC),
    Drop(677, 10, 32, DuelRank.SATEC),
    Drop(683, 10, 32, DuelRank.SATEC),
    Drop(684, 10, 32, DuelRank.SATEC),
    Drop(688, 10, 32, DuelRank.SATEC),
    Drop(690, 10, 32, DuelRank.SATEC),
    Drop(693, 10, 32, DuelRank.SATEC),
    Drop(695, 10, 32, DuelRank.SATEC),
    Drop(699, 10, 32, DuelRank.SATEC),
    Drop(13, 11, 136, DuelRank.SATEC),
    Drop(20, 11, 10, DuelRank.SATEC),
    Drop(41, 11, 46, DuelRank.SATEC),
    Drop(44, 11, 46, DuelRank.SATEC),
    Drop(54, 11, 46, DuelRank.SATEC),
    Drop(70, 11, 46, DuelRank.SATEC),
    Drop(78, 11, 46, DuelRank.SATEC),
    Drop(125, 11, 132, DuelRank.SATEC),
    Drop(149, 11, 45, DuelRank.SATEC),
    Drop(186, 11, 90, DuelRank.SATEC),
    Drop(207, 11, 90, DuelRank.SATEC),
    Drop(272, 11, 90, DuelRank.SATEC),
    Drop(300, 11, 90, DuelRank.SATEC),
    Drop(307, 11, 32, DuelRank.SATEC),
    Drop(312, 11, 32, DuelRank.SATEC),
    Drop(313, 11, 32, DuelRank.SATEC),
    Drop(316, 11, 64, DuelRank.SATEC),
    Drop(317, 11, 72, DuelRank.SATEC),
    Drop(321, 11, 32, DuelRank.SATEC),
    Drop(325, 11, 32, DuelRank.SATEC),
    Drop(327, 11, 32, DuelRank.SATEC),
    Drop(332, 11, 64, DuelRank.SATEC),
    Drop(339, 11, 32, DuelRank.SATEC),
    Drop(340, 11, 32, DuelRank.SATEC),
    Drop(341, 11, 32, DuelRank.SATEC),
    Drop(344, 11, 32, DuelRank.SATEC),
    Drop(345, 11, 32, DuelRank.SATEC),
    Drop(349, 11, 32, DuelRank.SATEC),
    Drop(412, 11, 45, DuelRank.SATEC),
    Drop(415, 11, 45, DuelRank.SATEC),
    Drop(416, 11, 45, DuelRank.SATEC),
    Drop(538, 11, 90, DuelRank.SATEC),
    Drop(552, 11, 90, DuelRank.SATEC),
    Drop(672, 11, 64, DuelRank.SATEC),
    Drop(676, 11, 32, DuelRank.SATEC),
    Drop(684, 11, 36, DuelRank.SATEC),
    Drop(685, 11, 36, DuelRank.SATEC),
    Drop(687, 11, 36, DuelRank.SATEC),
    Drop(690, 11, 32, DuelRank.SATEC),
    Drop(9, 12, 36, DuelRank.SATEC),
    Drop(19, 12, 10, DuelRank.SATEC),
    Drop(24, 12, 36, DuelRank.SATEC),
    Drop(30, 12, 36, DuelRank.SATEC),
    Drop(96, 12, 108, DuelRank.SATEC),
    Drop(97, 12, 108, DuelRank.SATEC),
    Drop(98, 12, 108, DuelRank.SATEC),
    Drop(108, 12, 36, DuelRank.SATEC),
    Drop(132, 12, 36, DuelRank.SATEC),
    Drop(135, 12, 36, DuelRank.SATEC),
    Drop(139, 12, 36, DuelRank.SATEC),
    Drop(154, 12, 36, DuelRank.SATEC),
    Drop(197, 12, 36, DuelRank.SATEC),
    Drop(203, 12, 36, DuelRank.SATEC),
    Drop(215, 12, 36, DuelRank.SATEC),
    Drop(228, 12, 36, DuelRank.SATEC),
    Drop(275, 12, 8, DuelRank.SATEC),
    Drop(283, 12, 36, DuelRank.SATEC),
    Drop(297, 12, 8, DuelRank.SATEC),
    Drop(319, 12, 64, DuelRank.SATEC),
    Drop(320, 12, 64, DuelRank.SATEC),
    Drop(322, 12, 36, DuelRank.SATEC),
    Drop(325, 12, 36, DuelRank.SATEC),
    Drop(331, 12, 32, DuelRank.SATEC),
    Drop(335, 12, 32, DuelRank.SATEC),
    Drop(406, 12, 36, DuelRank.SATEC),
    Drop(410, 12, 36, DuelRank.SATEC),
    Drop(411, 12, 36, DuelRank.SATEC),
    Drop(412, 12, 36, DuelRank.SATEC),
    Drop(414, 12, 36, DuelRank.SATEC),
    Drop(415, 12, 36, DuelRank.SATEC),
    Drop(416, 12, 36, DuelRank.SATEC),
    Drop(417, 12, 36, DuelRank.SATEC),
    Drop(418, 12, 8, DuelRank.SATEC),
    Drop(420, 12, 36, DuelRank.SATEC),
    Drop(421, 12, 36, DuelRank.SATEC),
    Drop(422, 12, 36, DuelRank.SATEC),
    Drop(498, 12, 8, DuelRank.SATEC),
    Drop(514, 12, 36, DuelRank.SATEC),
    Drop(544, 12, 36, DuelRank.SATEC),
    Drop(548, 12, 36, DuelRank.SATEC),
    Drop(556, 12, 36, DuelRank.SATEC),
    Drop(585, 12, 36, DuelRank.SATEC),
    Drop(614, 12, 8, DuelRank.SATEC),
    Drop(643, 12, 36, DuelRank.SATEC),
    Drop(658, 12, 64, DuelRank.SATEC),
    Drop(660, 12, 64, DuelRank.SATEC),
    Drop(679, 12, 30, DuelRank.SATEC),
    Drop(684, 12, 32, DuelRank.SATEC),
    Drop(685, 12, 36, DuelRank.SATEC),
    Drop(688, 12, 36, DuelRank.SATEC),
    Drop(690, 12, 36, DuelRank.SATEC),
    Drop(695, 12, 32, DuelRank.SATEC),
    Drop(5, 13, 10, DuelRank.SATEC),
    Drop(8, 13, 12, DuelRank.SATEC),
    Drop(20, 13, 6, DuelRank.SATEC),
    Drop(29, 13, 12, DuelRank.SATEC),
    Drop(45, 13, 12, DuelRank.SATEC),
    Drop(48, 13, 12, DuelRank.SATEC),
    Drop(50, 13, 12, DuelRank.SATEC),
    Drop(96, 13, 12, DuelRank.SATEC),
    Drop(97, 13, 12, DuelRank.SATEC),
    Drop(98, 13, 12, DuelRank.SATEC),
    Drop(101, 13, 12, DuelRank.SATEC),
    Drop(107, 13, 12, DuelRank.SATEC),
    Drop(108, 13, 12, DuelRank.SATEC),
    Drop(122, 13, 12, DuelRank.SATEC),
    Drop(129, 13, 12, DuelRank.SATEC),
    Drop(134, 13, 12, DuelRank.SATEC),
    Drop(135, 13, 12, DuelRank.SATEC),
    Drop(139, 13, 12, DuelRank.SATEC),
    Drop(144, 13, 12, DuelRank.SATEC),
    Drop(145, 13, 12, DuelRank.SATEC),
    Drop(147, 13, 12, DuelRank.SATEC),
    Drop(148, 13, 12, DuelRank.SATEC),
    Drop(152, 13, 32, DuelRank.SATEC),
    Drop(154, 13, 12, DuelRank.SATEC),
    Drop(156, 13, 12, DuelRank.SATEC),
    Drop(157, 13, 12, DuelRank.SATEC),
    Drop(158, 13, 12, DuelRank.SATEC),
    Drop(159, 13, 12, DuelRank.SATEC),
    Drop(160, 13, 12, DuelRank.SATEC),
    Drop(161, 13, 12, DuelRank.SATEC),
    Drop(173, 13, 12, DuelRank.SATEC),
    Drop(176, 13, 12, DuelRank.SATEC),
    Drop(177, 13, 24, DuelRank.SATEC),
    Drop(179, 13, 12, DuelRank.SATEC),
    Drop(182, 13, 12, DuelRank.SATEC),
    Drop(183, 13, 12, DuelRank.SATEC),
    Drop(184, 13, 12, DuelRank.SATEC),
    Drop(187, 13, 12, DuelRank.SATEC),
    Drop(189, 13, 12, DuelRank.SATEC),
    Drop(191, 13, 12, DuelRank.SATEC),
    Drop(195, 13, 12, DuelRank.SATEC),
    Drop(198, 13, 12, DuelRank.SATEC),
    Drop(199, 13, 20, DuelRank.SATEC),
    Drop(200, 13, 12, DuelRank.SATEC),
    Drop(203, 13, 12, DuelRank.SATEC),
    Drop(205, 13, 20, DuelRank.SATEC),
    Drop(206, 13, 30, DuelRank.SATEC),
    Drop(208, 13, 12, DuelRank.SATEC),
    Drop(209, 13, 12, DuelRank.SATEC),
    Drop(211, 13, 12, DuelRank.SATEC),
    Drop(212, 13, 12, DuelRank.SATEC),
    Drop(214, 13, 12, DuelRank.SATEC),
    Drop(218, 13, 12, DuelRank.SATEC),
    Drop(222, 13, 12, DuelRank.SATEC),
    Drop(224, 13, 12, DuelRank.SATEC),
    Drop(227, 13, 30, DuelRank.SATEC),
    Drop(229, 13, 12, DuelRank.SATEC),
    Drop(232, 13, 12, DuelRank.SATEC),
    Drop(239, 13, 12, DuelRank.SATEC),
    Drop(240, 13, 12, DuelRank.SATEC),
    Drop(242, 13, 12, DuelRank.SATEC),
    Drop(243, 13, 30, DuelRank.SATEC),
    Drop(245, 13, 12, DuelRank.SATEC),
    Drop(247, 13, 20, DuelRank.SATEC),
    Drop(254, 13, 12, DuelRank.SATEC),
    Drop(261, 13, 12, DuelRank.SATEC),
    Drop(265, 13, 32, DuelRank.SATEC),
    Drop(266, 13, 12, DuelRank.SATEC),
    Drop(267, 13, 12, DuelRank.SATEC),
    Drop(268, 13, 12, DuelRank.SATEC),
    Drop(270, 13, 20, DuelRank.SATEC),
    Drop(271, 13, 12, DuelRank.SATEC),
    Drop(277, 13, 12, DuelRank.SATEC),
    Drop(285, 13, 32, DuelRank.SATEC),
    Drop(289, 13, 20, DuelRank.SATEC),
    Drop(292, 13, 32, DuelRank.SATEC),
    Drop(300, 13, 12, DuelRank.SATEC),
    Drop(308, 13, 24, DuelRank.SATEC),
    Drop(309, 13, 24, DuelRank.SATEC),
    Drop(324, 13, 24, DuelRank.SATEC),
    Drop(328, 13, 24, DuelRank.SATEC),
    Drop(334, 13, 36, DuelRank.SATEC),
    Drop(337, 13, 36, DuelRank.SATEC),
    Drop(393, 13, 20, DuelRank.SATEC),
    Drop(398, 13, 20, DuelRank.SATEC),
    Drop(420, 13, 12, DuelRank.SATEC),
    Drop(421, 13, 12, DuelRank.SATEC),
    Drop(432, 13, 20, DuelRank.SATEC),
    Drop(436, 13, 20, DuelRank.SATEC),
    Drop(444, 13, 20, DuelRank.SATEC),
    Drop(446, 13, 20, DuelRank.SATEC),
    Drop(451, 13, 30, DuelRank.SATEC),
    Drop(452, 13, 30, DuelRank.SATEC),
    Drop(475, 13, 20, DuelRank.SATEC),
    Drop(476, 13, 12, DuelRank.SATEC),
    Drop(484, 13, 20, DuelRank.SATEC),
    Drop(490, 13, 12, DuelRank.SATEC),
    Drop(492, 13, 12, DuelRank.SATEC),
    Drop(503, 13, 30, DuelRank.SATEC),
    Drop(506, 13, 12, DuelRank.SATEC),
    Drop(510, 13, 12, DuelRank.SATEC),
    Drop(524, 13, 20, DuelRank.SATEC),
    Drop(534, 13, 12, DuelRank.SATEC),
    Drop(536, 13, 30, DuelRank.SATEC),
    Drop(540, 13, 12, DuelRank.SATEC),
    Drop(544, 13, 12, DuelRank.SATEC),
    Drop(549, 13, 30, DuelRank.SATEC),
    Drop(556, 13, 12, DuelRank.SATEC),
    Drop(569, 13, 20, DuelRank.SATEC),
    Drop(590, 13, 20, DuelRank.SATEC),
    Drop(591, 13, 12, DuelRank.SATEC),
    Drop(602, 13, 30, DuelRank.SATEC),
    Drop(604, 13, 12, DuelRank.SATEC),
    Drop(605, 13, 20, DuelRank.SATEC),
    Drop(606, 13, 30, DuelRank.SATEC),
    Drop(609, 13, 12, DuelRank.SATEC),
    Drop(664, 13, 40, DuelRank.SATEC),
    Drop(682, 13, 24, DuelRank.SATEC),
    Drop(683, 13, 24, DuelRank.SATEC),
    Drop(687, 13, 24, DuelRank.SATEC),
    Drop(690, 13, 24, DuelRank.SATEC),
    Drop(693, 13, 24, DuelRank.SATEC),
    Drop(16, 14, 40, DuelRank.SATEC),
    Drop(20, 14, 6, DuelRank.SATEC),
    Drop(104, 14, 40, DuelRank.SATEC),
    Drop(114, 14, 40, DuelRank.SATEC),
    Drop(128, 14, 6, DuelRank.SATEC),
    Drop(129, 14, 40, DuelRank.SATEC),
    Drop(130, 14, 40, DuelRank.SATEC),
    Drop(134, 14, 40, DuelRank.SATEC),
    Drop(142, 14, 40, DuelRank.SATEC),
    Drop(143, 14, 40, DuelRank.SATEC),
    Drop(144, 14, 40, DuelRank.SATEC),
    Drop(145, 14, 40, DuelRank.SATEC),
    Drop(174, 14, 40, DuelRank.SATEC),
    Drop(179, 14, 40, DuelRank.SATEC),
    Drop(183, 14, 40, DuelRank.SATEC),
    Drop(184, 14, 40, DuelRank.SATEC),
    Drop(190, 14, 40, DuelRank.SATEC),
    Drop(192, 14, 40, DuelRank.SATEC),
    Drop(198, 14, 40, DuelRank.SATEC),
    Drop(208, 14, 40, DuelRank.SATEC),
    Drop(216, 14, 6, DuelRank.SATEC),
    Drop(229, 14, 40, DuelRank.SATEC),
    Drop(253, 14, 40, DuelRank.SATEC),
    Drop(260, 14, 40, DuelRank.SATEC),
    Drop(264, 14, 40, DuelRank.SATEC),
    Drop(268, 14, 40, DuelRank.SATEC),
    Drop(276, 14, 40, DuelRank.SATEC),
    Drop(287, 14, 6, DuelRank.SATEC),
    Drop(311, 14, 36, DuelRank.SATEC),
    Drop(312, 14, 36, DuelRank.SATEC),
    Drop(323, 14, 40, DuelRank.SATEC),
    Drop(334, 14, 64, DuelRank.SATEC),
    Drop(350, 14, 40, DuelRank.SATEC),
    Drop(387, 14, 40, DuelRank.SATEC),
    Drop(395, 14, 40, DuelRank.SATEC),
    Drop(433, 14, 6, DuelRank.SATEC),
    Drop(486, 14, 40, DuelRank.SATEC),
    Drop(492, 14, 40, DuelRank.SATEC),
    Drop(540, 14, 40, DuelRank.SATEC),
    Drop(578, 14, 6, DuelRank.SATEC),
    Drop(587, 14, 6, DuelRank.SATEC),
    Drop(612, 14, 40, DuelRank.SATEC),
    Drop(630, 14, 6, DuelRank.SATEC),
    Drop(653, 14, 32, DuelRank.SATEC),
    Drop(661, 14, 2, DuelRank.SATEC),
    Drop(665, 14, 32, DuelRank.SATEC),
    Drop(666, 14, 74, DuelRank.SATEC),
    Drop(668, 14, 32, DuelRank.SATEC),
    Drop(671, 14, 8, DuelRank.SATEC),
    Drop(672, 14, 32, DuelRank.SATEC),
    Drop(673, 14, 8, DuelRank.SATEC),
    Drop(674, 14, 8, DuelRank.SATEC),
    Drop(676, 14, 8, DuelRank.SATEC),
    Drop(677, 14, 8, DuelRank.SATEC),
    Drop(678, 14, 10, DuelRank.SATEC),
    Drop(679, 14, 32, DuelRank.SATEC),
    Drop(680, 14, 10, DuelRank.SATEC),
    Drop(684, 14, 32, DuelRank.SATEC),
    Drop(685, 14, 34, DuelRank.SATEC),
    Drop(688, 14, 34, DuelRank.SATEC),
    Drop(690, 14, 32, DuelRank.SATEC),
    Drop(691, 14, 10, DuelRank.SATEC),
    Drop(692, 14, 10, DuelRank.SATEC),
    Drop(693, 14, 32, DuelRank.SATEC),
    Drop(694, 14, 10, DuelRank.SATEC),
    Drop(695, 14, 32, DuelRank.SATEC),
    Drop(697, 14, 10, DuelRank.SATEC),
    Drop(698, 14, 10, DuelRank.SATEC),
    Drop(699, 14, 32, DuelRank.SATEC),
    Drop(700, 14, 10, DuelRank.SATEC),
    Drop(4, 15, 32, DuelRank.SATEC),
    Drop(19, 15, 8, DuelRank.SATEC),
    Drop(29, 15, 32, DuelRank.SATEC),
    Drop(40, 15, 64, DuelRank.SATEC),
    Drop(45, 15, 34, DuelRank.SATEC),
    Drop(47, 15, 34, DuelRank.SATEC),
    Drop(61, 15, 34, DuelRank.SATEC),
    Drop(65, 15, 34, DuelRank.SATEC),
    Drop(113, 15, 34, DuelRank.SATEC),
    Drop(117, 15, 6, DuelRank.SATEC),
    Drop(122, 15, 34, DuelRank.SATEC),
    Drop(155, 15, 34, DuelRank.SATEC),
    Drop(159, 15, 34, DuelRank.SATEC),
    Drop(187, 15, 34, DuelRank.SATEC),
    Drop(188, 15, 34, DuelRank.SATEC),
    Drop(189, 15, 34, DuelRank.SATEC),
    Drop(200, 15, 34, DuelRank.SATEC),
    Drop(202, 15, 34, DuelRank.SATEC),
    Drop(207, 15, 34, DuelRank.SATEC),
    Drop(212, 15, 34, DuelRank.SATEC),
    Drop(219, 15, 34, DuelRank.SATEC),
    Drop(282, 15, 36, DuelRank.SATEC),
    Drop(296, 15, 36, DuelRank.SATEC),
    Drop(298, 15, 36, DuelRank.SATEC),
    Drop(300, 15, 36, DuelRank.SATEC),
    Drop(315, 15, 36, DuelRank.SATEC),
    Drop(316, 15, 36, DuelRank.SATEC),
    Drop(325, 15, 36, DuelRank.SATEC),
    Drop(329, 15, 64, DuelRank.SATEC),
    Drop(330, 15, 64, DuelRank.SATEC),
    Drop(334, 15, 64, DuelRank.SATEC),
    Drop(404, 15, 8, DuelRank.SATEC),
    Drop(481, 15, 36, DuelRank.SATEC),
    Drop(496, 15, 36, DuelRank.SATEC),
    Drop(527, 15, 36, DuelRank.SATEC),
    Drop(538, 15, 36, DuelRank.SATEC),
    Drop(595, 15, 8, DuelRank.SATEC),
    Drop(598, 15, 36, DuelRank.SATEC),
    Drop(604, 15, 36, DuelRank.SATEC),
    Drop(629, 15, 36, DuelRank.SATEC),
    Drop(642, 15, 36, DuelRank.SATEC),
    Drop(652, 15, 36, DuelRank.SATEC),
    Drop(657, 15, 64, DuelRank.SATEC),
    Drop(661, 15, 2, DuelRank.SATEC),
    Drop(665, 15, 32, DuelRank.SATEC),
    Drop(666, 15, 32, DuelRank.SATEC),
    Drop(668, 15, 64, DuelRank.SATEC),
    Drop(672, 15, 32, DuelRank.SATEC),
    Drop(676, 15, 32, DuelRank.SATEC),
    Drop(677, 15, 32, DuelRank.SATEC),
    Drop(685, 15, 64, DuelRank.SATEC),
    Drop(686, 15, 64, DuelRank.SATEC),
    Drop(687, 15, 32, DuelRank.SATEC),
    Drop(690, 15, 32, DuelRank.SATEC),
    Drop(693, 15, 32, DuelRank.SATEC),
    Drop(695, 15, 32, DuelRank.SATEC),
    Drop(699, 15, 32, DuelRank.SATEC),
    Drop(4, 16, 56, DuelRank.SATEC),
    Drop(20, 16, 8, DuelRank.SATEC),
    Drop(122, 16, 56, DuelRank.SATEC),
    Drop(140, 16, 56, DuelRank.SATEC),
    Drop(152, 16, 56, DuelRank.SATEC),
    Drop(177, 16, 56, DuelRank.SATEC),
    Drop(199, 16, 56, DuelRank.SATEC),
    Drop(200, 16, 56, DuelRank.SATEC),
    Drop(205, 16, 54, DuelRank.SATEC),
    Drop(206, 16, 54, DuelRank.SATEC),
    Drop(227, 16, 54, DuelRank.SATEC),
    Drop(243, 16, 54, DuelRank.SATEC),
    Drop(247, 16, 54, DuelRank.SATEC),
    Drop(249, 16, 10, DuelRank.SATEC),
    Drop(265, 16, 54, DuelRank.SATEC),
    Drop(270, 16, 54, DuelRank.SATEC),
    Drop(289, 16, 54, DuelRank.SATEC),
    Drop(292, 16, 54, DuelRank.SATEC),
    Drop(296, 16, 54, DuelRank.SATEC),
    Drop(298, 16, 54, DuelRank.SATEC),
    Drop(307, 16, 36, DuelRank.SATEC),
    Drop(311, 16, 32, DuelRank.SATEC),
    Drop(315, 16, 36, DuelRank.SATEC),
    Drop(316, 16, 64, DuelRank.SATEC),
    Drop(317, 16, 64, DuelRank.SATEC),
    Drop(319, 16, 36, DuelRank.SATEC),
    Drop(321, 16, 36, DuelRank.SATEC),
    Drop(322, 16, 36, DuelRank.SATEC),
    Drop(328, 16, 36, DuelRank.SATEC),
    Drop(331, 16, 32, DuelRank.SATEC),
    Drop(334, 16, 32, DuelRank.SATEC),
    Drop(340, 16, 36, DuelRank.SATEC),
    Drop(393, 16, 54, DuelRank.SATEC),
    Drop(431, 16, 84, DuelRank.SATEC),
    Drop(448, 16, 10, DuelRank.SATEC),
    Drop(454, 16, 10, DuelRank.SATEC),
    Drop(456, 16, 10, DuelRank.SATEC),
    Drop(517, 16, 10, DuelRank.SATEC),
    Drop(530, 16, 42, DuelRank.SATEC),
    Drop(621, 16, 10, DuelRank.SATEC),
    Drop(656, 16, 32, DuelRank.SATEC),
    Drop(661, 16, 2, DuelRank.SATEC),
    Drop(663, 16, 64, DuelRank.SATEC),
    Drop(664, 16, 32, DuelRank.SATEC),
    Drop(665, 16, 32, DuelRank.SATEC),
    Drop(672, 16, 36, DuelRank.SATEC),
    Drop(676, 16, 32, DuelRank.SATEC),
    Drop(686, 16, 36, DuelRank.SATEC),
    Drop(687, 16, 36, DuelRank.SATEC),
    Drop(690, 16, 36, DuelRank.SATEC),
    Drop(4, 17, 86, DuelRank.SATEC),
    Drop(5, 17, 174, DuelRank.SATEC),
    Drop(19, 17, 12, DuelRank.SATEC),
    Drop(66, 17, 12, DuelRank.SATEC),
    Drop(117, 17, 12, DuelRank.SATEC),
    Drop(122, 17, 84, DuelRank.SATEC),
    Drop(128, 17, 12, DuelRank.SATEC),
    Drop(151, 17, 12, DuelRank.SATEC),
    Drop(163, 17, 12, DuelRank.SATEC),
    Drop(200, 17, 84, DuelRank.SATEC),
    Drop(216, 17, 12, DuelRank.SATEC),
    Drop(249, 17, 12, DuelRank.SATEC),
    Drop(275, 17, 12, DuelRank.SATEC),
    Drop(281, 17, 12, DuelRank.SATEC),
    Drop(287, 17, 12, DuelRank.SATEC),
    Drop(296, 17, 84, DuelRank.SATEC),
    Drop(297, 17, 12, DuelRank.SATEC),
    Drop(298, 17, 84, DuelRank.SATEC),
    Drop(302, 17, 56, DuelRank.SATEC),
    Drop(303, 17, 56, DuelRank.SATEC),
    Drop(329, 17, 56, DuelRank.SATEC),
    Drop(333, 17, 88, DuelRank.SATEC),
    Drop(404, 17, 12, DuelRank.SATEC),
    Drop(418, 17, 12, DuelRank.SATEC),
    Drop(448, 17, 12, DuelRank.SATEC),
    Drop(454, 17, 12, DuelRank.SATEC),
    Drop(456, 17, 12, DuelRank.SATEC),
    Drop(498, 17, 12, DuelRank.SATEC),
    Drop(517, 17, 12, DuelRank.SATEC),
    Drop(578, 17, 12, DuelRank.SATEC),
    Drop(587, 17, 12, DuelRank.SATEC),
    Drop(595, 17, 12, DuelRank.SATEC),
    Drop(614, 17, 12, DuelRank.SATEC),
    Drop(621, 17, 12, DuelRank.SATEC),
    Drop(630, 17, 12, DuelRank.SATEC),
    Drop(653, 17, 80, DuelRank.SATEC),
    Drop(661, 17, 40, DuelRank.SATEC),
    Drop(665, 17, 40, DuelRank.SATEC),
    Drop(666, 17, 40, DuelRank.SATEC),
    Drop(668, 17, 40, DuelRank.SATEC),
    Drop(672, 17, 40, DuelRank.SATEC),
    Drop(676, 17, 40, DuelRank.SATEC),
    Drop(677, 17, 40, DuelRank.SATEC),
    Drop(679, 17, 40, DuelRank.SATEC),
    Drop(685, 17, 84, DuelRank.SATEC),
    Drop(686, 17, 84, DuelRank.SATEC),
    Drop(687, 17, 84, DuelRank.SATEC),
    Drop(688, 17, 84, DuelRank.SATEC),
    Drop(690, 17, 40, DuelRank.SATEC),
    Drop(693, 17, 40, DuelRank.SATEC),
    Drop(695, 17, 40, DuelRank.SATEC),
    Drop(699, 17, 40, DuelRank.SATEC),
    Drop(5, 18, 40, DuelRank.SATEC),
    Drop(20, 18, 12, DuelRank.SATEC),
    Drop(25, 18, 40, DuelRank.SATEC),
    Drop(48, 18, 40, DuelRank.SATEC),
    Drop(58, 18, 40, DuelRank.SATEC),
    Drop(103, 18, 40, DuelRank.SATEC),
    Drop(115, 18, 40, DuelRank.SATEC),
    Drop(119, 18, 40, DuelRank.SATEC),
    Drop(121, 18, 40, DuelRank.SATEC),
    Drop(137, 18, 38, DuelRank.SATEC),
    Drop(148, 18, 38, DuelRank.SATEC),
    Drop(162, 18, 38, DuelRank.SATEC),
    Drop(164, 18, 38, DuelRank.SATEC),
    Drop(169, 18, 38, DuelRank.SATEC),
    Drop(171, 18, 38, DuelRank.SATEC),
    Drop(173, 18, 38, DuelRank.SATEC),
    Drop(175, 18, 38, DuelRank.SATEC),
    Drop(178, 18, 38, DuelRank.SATEC),
    Drop(181, 18, 38, DuelRank.SATEC),
    Drop(222, 18, 38, DuelRank.SATEC),
    Drop(232, 18, 38, DuelRank.SATEC),
    Drop(233, 18, 32, DuelRank.SATEC),
    Drop(240, 18, 38, DuelRank.SATEC),
    Drop(242, 18, 38, DuelRank.SATEC),
    Drop(245, 18, 38, DuelRank.SATEC),
    Drop(254, 18, 38, DuelRank.SATEC),
    Drop(255, 18, 38, DuelRank.SATEC),
    Drop(261, 18, 38, DuelRank.SATEC),
    Drop(269, 18, 38, DuelRank.SATEC),
    Drop(271, 18, 38, DuelRank.SATEC),
    Drop(277, 18, 38, DuelRank.SATEC),
    Drop(279, 18, 38, DuelRank.SATEC),
    Drop(281, 18, 12, DuelRank.SATEC),
    Drop(295, 18, 38, DuelRank.SATEC),
    Drop(303, 18, 72, DuelRank.SATEC),
    Drop(304, 18, 76, DuelRank.SATEC),
    Drop(321, 18, 40, DuelRank.SATEC),
    Drop(325, 18, 40, DuelRank.SATEC),
    Drop(335, 18, 72, DuelRank.SATEC),
    Drop(402, 18, 38, DuelRank.SATEC),
    Drop(490, 18, 38, DuelRank.SATEC),
    Drop(504, 18, 38, DuelRank.SATEC),
    Drop(563, 18, 38, DuelRank.SATEC),
    Drop(611, 18, 38, DuelRank.SATEC),
    Drop(655, 18, 36, DuelRank.SATEC),
    Drop(683, 18, 40, DuelRank.SATEC),
    Drop(684, 18, 40, DuelRank.SATEC),
    Drop(685, 18, 40, DuelRank.SATEC),
    Drop(686, 18, 40, DuelRank.SATEC),
    Drop(688, 18, 72, DuelRank.SATEC),
    Drop(690, 18, 40, DuelRank.SATEC),
    Drop(4, 19, 18, DuelRank.SATEC),
    Drop(20, 19, 4, DuelRank.SATEC),
    Drop(29, 19, 18, DuelRank.SATEC),
    Drop(43, 19, 18, DuelRank.SATEC),
    Drop(50, 19, 38, DuelRank.SATEC),
    Drop(61, 19, 18, DuelRank.SATEC),
    Drop(78, 19, 18, DuelRank.SATEC),
    Drop(100, 19, 18, DuelRank.SATEC),
    Drop(104, 19, 38, DuelRank.SATEC),
    Drop(105, 19, 38, DuelRank.SATEC),
    Drop(122, 19, 38, DuelRank.SATEC),
    Drop(130, 19, 38, DuelRank.SATEC),
    Drop(135, 19, 38, DuelRank.SATEC),
    Drop(137, 19, 38, DuelRank.SATEC),
    Drop(152, 19, 38, DuelRank.SATEC),
    Drop(154, 19, 38, DuelRank.SATEC),
    Drop(174, 19, 38, DuelRank.SATEC),
    Drop(182, 19, 38, DuelRank.SATEC),
    Drop(185, 19, 38, DuelRank.SATEC),
    Drop(191, 19, 38, DuelRank.SATEC),
    Drop(197, 19, 38, DuelRank.SATEC),
    Drop(202, 19, 38, DuelRank.SATEC),
    Drop(207, 19, 38, DuelRank.SATEC),
    Drop(210, 19, 38, DuelRank.SATEC),
    Drop(214, 19, 38, DuelRank.SATEC),
    Drop(237, 19, 38, DuelRank.SATEC),
    Drop(242, 19, 44, DuelRank.SATEC),
    Drop(245, 19, 44, DuelRank.SATEC),
    Drop(268, 19, 44, DuelRank.SATEC),
    Drop(301, 19, 20, DuelRank.SATEC),
    Drop(302, 19, 20, DuelRank.SATEC),
    Drop(310, 19, 21, DuelRank.SATEC),
    Drop(311, 19, 21, DuelRank.SATEC),
    Drop(333, 19, 20, DuelRank.SATEC),
    Drop(336, 19, 20, DuelRank.SATEC),
    Drop(344, 19, 22, DuelRank.SATEC),
    Drop(345, 19, 22, DuelRank.SATEC),
    Drop(410, 19, 44, DuelRank.SATEC),
    Drop(420, 19, 44, DuelRank.SATEC),
    Drop(422, 19, 44, DuelRank.SATEC),
    Drop(436, 19, 44, DuelRank.SATEC),
    Drop(444, 19, 44, DuelRank.SATEC),
    Drop(485, 19, 44, DuelRank.SATEC),
    Drop(486, 19, 44, DuelRank.SATEC),
    Drop(488, 19, 44, DuelRank.SATEC),
    Drop(492, 19, 44, DuelRank.SATEC),
    Drop(501, 19, 44, DuelRank.SATEC),
    Drop(516, 19, 44, DuelRank.SATEC),
    Drop(524, 19, 44, DuelRank.SATEC),
    Drop(549, 19, 44, DuelRank.SATEC),
    Drop(558, 19, 44, DuelRank.SATEC),
    Drop(563, 19, 44, DuelRank.SATEC),
    Drop(579, 19, 44, DuelRank.SATEC),
    Drop(589, 19, 44, DuelRank.SATEC),
    Drop(598, 19, 44, DuelRank.SATEC),
    Drop(611, 19, 44, DuelRank.SATEC),
    Drop(651, 19, 22, DuelRank.SATEC),
    Drop(654, 19, 22, DuelRank.SATEC),
    Drop(677, 19, 20, DuelRank.SATEC),
    Drop(681, 19, 16, DuelRank.SATEC),
    Drop(19, 20, 6, DuelRank.SATEC),
    Drop(24, 20, 172, DuelRank.SATEC),
    Drop(58, 20, 172, DuelRank.SATEC),
    Drop(167, 20, 172, DuelRank.SATEC),
    Drop(302, 20, 18, DuelRank.SATEC),
    Drop(312, 20, 18, DuelRank.SATEC),
    Drop(330, 20, 30, DuelRank.SATEC),
    Drop(336, 20, 16, DuelRank.SATEC),
    Drop(345, 20, 18, DuelRank.SATEC),
    Drop(350, 20, 32, DuelRank.SATEC),
    Drop(393, 20, 172, DuelRank.SATEC),
    Drop(395, 20, 172, DuelRank.SATEC),
    Drop(398, 20, 172, DuelRank.SATEC),
    Drop(399, 20, 172, DuelRank.SATEC),
    Drop(402, 20, 172, DuelRank.SATEC),
    Drop(469, 20, 172, DuelRank.SATEC),
    Drop(475, 20, 172, DuelRank.SATEC),
    Drop(527, 20, 172, DuelRank.SATEC),
    Drop(681, 20, 16, DuelRank.SATEC),
    Drop(699, 20, 2, DuelRank.SATEC),
    Drop(5, 21, 12, DuelRank.SATEC),
    Drop(8, 21, 12, DuelRank.SATEC),
    Drop(20, 21, 6, DuelRank.SATEC),
    Drop(29, 21, 12, DuelRank.SATEC),
    Drop(45, 21, 12, DuelRank.SATEC),
    Drop(48, 21, 12, DuelRank.SATEC),
    Drop(50, 21, 12, DuelRank.SATEC),
    Drop(70, 21, 50, DuelRank.SATEC),
    Drop(71, 21, 50, DuelRank.SATEC),
    Drop(77, 21, 22, DuelRank.SATEC),
    Drop(96, 21, 12, DuelRank.SATEC),
    Drop(97, 21, 12, DuelRank.SATEC),
    Drop(98, 21, 12, DuelRank.SATEC),
    Drop(101, 21, 12, DuelRank.SATEC),
    Drop(107, 21, 12, DuelRank.SATEC),
    Drop(108, 21, 12, DuelRank.SATEC),
    Drop(122, 21, 12, DuelRank.SATEC),
    Drop(129, 21, 12, DuelRank.SATEC),
    Drop(134, 21, 12, DuelRank.SATEC),
    Drop(135, 21, 12, DuelRank.SATEC),
    Drop(139, 21, 12, DuelRank.SATEC),
    Drop(140, 21, 22, DuelRank.SATEC),
    Drop(144, 21, 12, DuelRank.SATEC),
    Drop(145, 21, 12, DuelRank.SATEC),
    Drop(147, 21, 12, DuelRank.SATEC),
    Drop(148, 21, 12, DuelRank.SATEC),
    Drop(152, 21, 36, DuelRank.SATEC),
    Drop(154, 21, 12, DuelRank.SATEC),
    Drop(156, 21, 12, DuelRank.SATEC),
    Drop(157, 21, 12, DuelRank.SATEC),
    Drop(158, 21, 12, DuelRank.SATEC),
    Drop(159, 21, 12, DuelRank.SATEC),
    Drop(160, 21, 12, DuelRank.SATEC),
    Drop(161, 21, 12, DuelRank.SATEC),
    Drop(173, 21, 12, DuelRank.SATEC),
    Drop(176, 21, 12, DuelRank.SATEC),
    Drop(177, 21, 22, DuelRank.SATEC),
    Drop(179, 21, 12, DuelRank.SATEC),
    Drop(182, 21, 12, DuelRank.SATEC),
    Drop(183, 21, 12, DuelRank.SATEC),
    Drop(184, 21, 12, DuelRank.SATEC),
    Drop(187, 21, 10, DuelRank.SATEC),
    Drop(189, 21, 10, DuelRank.SATEC),
    Drop(191, 21, 10, DuelRank.SATEC),
    Drop(193, 21, 20, DuelRank.SATEC),
    Drop(195, 21, 10, DuelRank.SATEC),
    Drop(196, 21, 20, DuelRank.SATEC),
    Drop(198, 21, 10, DuelRank.SATEC),
    Drop(199, 21, 20, DuelRank.SATEC),
    Drop(200, 21, 10, DuelRank.SATEC),
    Drop(203, 21, 10, DuelRank.SATEC),
    Drop(205, 21, 20, DuelRank.SATEC),
    Drop(206, 21, 34, DuelRank.SATEC),
    Drop(208, 21, 10, DuelRank.SATEC),
    Drop(209, 21, 10, DuelRank.SATEC),
    Drop(211, 21, 10, DuelRank.SATEC),
    Drop(212, 21, 10, DuelRank.SATEC),
    Drop(214, 21, 10, DuelRank.SATEC),
    Drop(218, 21, 10, DuelRank.SATEC),
    Drop(222, 21, 10, DuelRank.SATEC),
    Drop(224, 21, 10, DuelRank.SATEC),
    Drop(227, 21, 34, DuelRank.SATEC),
    Drop(229, 21, 10, DuelRank.SATEC),
    Drop(230, 21, 20, DuelRank.SATEC),
    Drop(232, 21, 10, DuelRank.SATEC),
    Drop(239, 21, 10, DuelRank.SATEC),
    Drop(240, 21, 10, DuelRank.SATEC),
    Drop(242, 21, 10, DuelRank.SATEC),
    Drop(243, 21, 34, DuelRank.SATEC),
    Drop(245, 21, 10, DuelRank.SATEC),
    Drop(247, 21, 20, DuelRank.SATEC),
    Drop(251, 21, 20, DuelRank.SATEC),
    Drop(254, 21, 10, DuelRank.SATEC),
    Drop(258, 21, 20, DuelRank.SATEC),
    Drop(261, 21, 10, DuelRank.SATEC),
    Drop(265, 21, 34, DuelRank.SATEC),
    Drop(266, 21, 10, DuelRank.SATEC),
    Drop(267, 21, 10, DuelRank.SATEC),
    Drop(268, 21, 10, DuelRank.SATEC),
    Drop(270, 21, 20, DuelRank.SATEC),
    Drop(271, 21, 10, DuelRank.SATEC),
    Drop(277, 21, 10, DuelRank.SATEC),
    Drop(285, 21, 34, DuelRank.SATEC),
    Drop(292, 21, 34, DuelRank.SATEC),
    Drop(308, 21, 24, DuelRank.SATEC),
    Drop(309, 21, 24, DuelRank.SATEC),
    Drop(324, 21, 24, DuelRank.SATEC),
    Drop(328, 21, 24, DuelRank.SATEC),
    Drop(334, 21, 30, DuelRank.SATEC),
    Drop(337, 21, 36, DuelRank.SATEC),
    Drop(346, 21, 24, DuelRank.SATEC),
    Drop(420, 21, 10, DuelRank.SATEC),
    Drop(421, 21, 10, DuelRank.SATEC),
    Drop(432, 21, 20, DuelRank.SATEC),
    Drop(446, 21, 20, DuelRank.SATEC),
    Drop(451, 21, 34, DuelRank.SATEC),
    Drop(452, 21, 34, DuelRank.SATEC),
    Drop(476, 21, 10, DuelRank.SATEC),
    Drop(490, 21, 10, DuelRank.SATEC),
    Drop(492, 21, 10, DuelRank.SATEC),
    Drop(503, 21, 34, DuelRank.SATEC),
    Drop(506, 21, 10, DuelRank.SATEC),
    Drop(510, 21, 10, DuelRank.SATEC),
    Drop(524, 21, 20, DuelRank.SATEC),
    Drop(534, 21, 10, DuelRank.SATEC),
    Drop(536, 21, 34, DuelRank.SATEC),
    Drop(540, 21, 10, DuelRank.SATEC),
    Drop(544, 21, 10, DuelRank.SATEC),
    Drop(549, 21, 34, DuelRank.SATEC),
    Drop(556, 21, 10, DuelRank.SATEC),
    Drop(569, 21, 20, DuelRank.SATEC),
    Drop(590, 21, 20, DuelRank.SATEC),
    Drop(591, 21, 10, DuelRank.SATEC),
    Drop(602, 21, 34, DuelRank.SATEC),
    Drop(604, 21, 10, DuelRank.SATEC),
    Drop(605, 21, 20, DuelRank.SATEC),
    Drop(606, 21, 34, DuelRank.SATEC),
    Drop(609, 21, 10, DuelRank.SATEC),
    Drop(664, 21, 24, DuelRank.SATEC),
    Drop(682, 21, 20, DuelRank.SATEC),
    Drop(683, 21, 20, DuelRank.SATEC),
    Drop(687, 21, 20, DuelRank.SATEC),
    Drop(2, 22, 42, DuelRank.SATEC),
    Drop(20, 22, 4, DuelRank.SATEC),
    Drop(34, 22, 42, DuelRank.SATEC),
    Drop(44, 22, 42, DuelRank.SATEC),
    Drop(104, 22, 42, DuelRank.SATEC),
    Drop(109, 22, 42, DuelRank.SATEC),
    Drop(114, 22, 42, DuelRank.SATEC),
    Drop(115, 22, 42, DuelRank.SATEC),
    Drop(129, 22, 42, DuelRank.SATEC),
    Drop(130, 22, 42, DuelRank.SATEC),
    Drop(134, 22, 44, DuelRank.SATEC),
    Drop(142, 22, 44, DuelRank.SATEC),
    Drop(143, 22, 44, DuelRank.SATEC),
    Drop(144, 22, 44, DuelRank.SATEC),
    Drop(145, 22, 44, DuelRank.SATEC),
    Drop(170, 22, 44, DuelRank.SATEC),
    Drop(174, 22, 44, DuelRank.SATEC),
    Drop(179, 22, 44, DuelRank.SATEC),
    Drop(183, 22, 44, DuelRank.SATEC),
    Drop(184, 22, 44, DuelRank.SATEC),
    Drop(190, 22, 44, DuelRank.SATEC),
    Drop(198, 22, 44, DuelRank.SATEC),
    Drop(208, 22, 44, DuelRank.SATEC),
    Drop(220, 22, 44, DuelRank.SATEC),
    Drop(229, 22, 44, DuelRank.SATEC),
    Drop(253, 22, 44, DuelRank.SATEC),
    Drop(259, 22, 44, DuelRank.SATEC),
    Drop(260, 22, 44, DuelRank.SATEC),
    Drop(264, 22, 44, DuelRank.SATEC),
    Drop(268, 22, 44, DuelRank.SATEC),
    Drop(276, 22, 44, DuelRank.SATEC),
    Drop(311, 22, 32, DuelRank.SATEC),
    Drop(312, 22, 32, DuelRank.SATEC),
    Drop(323, 22, 32, DuelRank.SATEC),
    Drop(334, 22, 48, DuelRank.SATEC),
    Drop(350, 22, 32, DuelRank.SATEC),
    Drop(486, 22, 44, DuelRank.SATEC),
    Drop(492, 22, 44, DuelRank.SATEC),
    Drop(540, 22, 44, DuelRank.SATEC),
    Drop(584, 22, 44, DuelRank.SATEC),
    Drop(601, 22, 44, DuelRank.SATEC),
    Drop(608, 22, 44, DuelRank.SATEC),
    Drop(612, 22, 44, DuelRank.SATEC),
    Drop(653, 22, 32, DuelRank.SATEC),
    Drop(661, 22, 2, DuelRank.SATEC),
    Drop(665, 22, 32, DuelRank.SATEC),
    Drop(666, 22, 64, DuelRank.SATEC),
    Drop(672, 22, 32, DuelRank.SATEC),
    Drop(684, 22, 32, DuelRank.SATEC),
    Drop(685, 22, 32, DuelRank.SATEC),
    Drop(688, 22, 32, DuelRank.SATEC),
    Drop(8, 23, 39, DuelRank.SATEC),
    Drop(19, 23, 5, DuelRank.SATEC),
    Drop(40, 23, 19, DuelRank.SATEC),
    Drop(50, 23, 19, DuelRank.SATEC),
    Drop(75, 23, 105, DuelRank.SATEC),
    Drop(76, 23, 39, DuelRank.SATEC),
    Drop(98, 23, 19, DuelRank.SATEC),
    Drop(101, 23, 39, DuelRank.SATEC),
    Drop(104, 23, 19, DuelRank.SATEC),
    Drop(107, 23, 19, DuelRank.SATEC),
    Drop(122, 23, 19, DuelRank.SATEC),
    Drop(130, 23, 19, DuelRank.SATEC),
    Drop(133, 23, 19, DuelRank.SATEC),
    Drop(134, 23, 19, DuelRank.SATEC),
    Drop(135, 23, 19, DuelRank.SATEC),
    Drop(145, 23, 19, DuelRank.SATEC),
    Drop(152, 23, 19, DuelRank.SATEC),
    Drop(154, 23, 19, DuelRank.SATEC),
    Drop(157, 23, 38, DuelRank.SATEC),
    Drop(158, 23, 38, DuelRank.SATEC),
    Drop(159, 23, 18, DuelRank.SATEC),
    Drop(174, 23, 18, DuelRank.SATEC),
    Drop(176, 23, 38, DuelRank.SATEC),
    Drop(179, 23, 18, DuelRank.SATEC),
    Drop(180, 23, 18, DuelRank.SATEC),
    Drop(182, 23, 18, DuelRank.SATEC),
    Drop(183, 23, 18, DuelRank.SATEC),
    Drop(185, 23, 18, DuelRank.SATEC),
    Drop(191, 23, 18, DuelRank.SATEC),
    Drop(195, 23, 18, DuelRank.SATEC),
    Drop(198, 23, 18, DuelRank.SATEC),
    Drop(200, 23, 18, DuelRank.SATEC),
    Drop(202, 23, 18, DuelRank.SATEC),
    Drop(203, 23, 18, DuelRank.SATEC),
    Drop(207, 23, 18, DuelRank.SATEC),
    Drop(209, 23, 18, DuelRank.SATEC),
    Drop(210, 23, 38, DuelRank.SATEC),
    Drop(211, 23, 18, DuelRank.SATEC),
    Drop(212, 23, 18, DuelRank.SATEC),
    Drop(214, 23, 18, DuelRank.SATEC),
    Drop(222, 23, 18, DuelRank.SATEC),
    Drop(227, 23, 18, DuelRank.SATEC),
    Drop(229, 23, 18, DuelRank.SATEC),
    Drop(232, 23, 18, DuelRank.SATEC),
    Drop(237, 23, 18, DuelRank.SATEC),
    Drop(242, 23, 18, DuelRank.SATEC),
    Drop(245, 23, 18, DuelRank.SATEC),
    Drop(254, 23, 18, DuelRank.SATEC),
    Drop(261, 23, 18, DuelRank.SATEC),
    Drop(268, 23, 18, DuelRank.SATEC),
    Drop(273, 23, 18, DuelRank.SATEC),
    Drop(274, 23, 18, DuelRank.SATEC),
    Drop(285, 23, 18, DuelRank.SATEC),
    Drop(291, 23, 18, DuelRank.SATEC),
    Drop(292, 23, 18, DuelRank.SATEC),
    Drop(302, 23, 20, DuelRank.SATEC),
    Drop(308, 23, 20, DuelRank.SATEC),
    Drop(310, 23, 20, DuelRank.SATEC),
    Drop(324, 23, 20, DuelRank.SATEC),
    Drop(328, 23, 20, DuelRank.SATEC),
    Drop(330, 23, 36, DuelRank.SATEC),
    Drop(343, 23, 18, DuelRank.SATEC),
    Drop(344, 23, 18, DuelRank.SATEC),
    Drop(346, 23, 20, DuelRank.SATEC),
    Drop(420, 23, 18, DuelRank.SATEC),
    Drop(452, 23, 18, DuelRank.SATEC),
    Drop(476, 23, 18, DuelRank.SATEC),
    Drop(486, 23, 18, DuelRank.SATEC),
    Drop(492, 23, 18, DuelRank.SATEC),
    Drop(501, 23, 18, DuelRank.SATEC),
    Drop(506, 23, 18, DuelRank.SATEC),
    Drop(510, 23, 38, DuelRank.SATEC),
    Drop(511, 23, 18, DuelRank.SATEC),
    Drop(524, 23, 18, DuelRank.SATEC),
    Drop(536, 23, 18, DuelRank.SATEC),
    Drop(540, 23, 18, DuelRank.SATEC),
    Drop(544, 23, 18, DuelRank.SATEC),
    Drop(549, 23, 18, DuelRank.SATEC),
    Drop(556, 23, 18, DuelRank.SATEC),
    Drop(567, 23, 18, DuelRank.SATEC),
    Drop(579, 23, 38, DuelRank.SATEC),
    Drop(589, 23, 38, DuelRank.SATEC),
    Drop(591, 23, 18, DuelRank.SATEC),
    Drop(598, 23, 18, DuelRank.SATEC),
    Drop(602, 23, 18, DuelRank.SATEC),
    Drop(609, 23, 18, DuelRank.SATEC),
    Drop(611, 23, 18, DuelRank.SATEC),
    Drop(620, 23, 18, DuelRank.SATEC),
    Drop(656, 23, 36, DuelRank.SATEC),
    Drop(677, 23, 18, DuelRank.SATEC),
    Drop(682, 23, 18, DuelRank.SATEC),
    Drop(683, 23, 18, DuelRank.SATEC),
    Drop(684, 23, 18, DuelRank.SATEC),
    Drop(687, 23, 18, DuelRank.SATEC),
    Drop(688, 23, 18, DuelRank.SATEC),
    Drop(3, 24, 21, DuelRank.SATEC),
    Drop(8, 24, 21, DuelRank.SATEC),
    Drop(10, 24, 21, DuelRank.SATEC),
    Drop(13, 24, 21, DuelRank.SATEC),
    Drop(19, 24, 5, DuelRank.SATEC),
    Drop(25, 24, 21, DuelRank.SATEC),
    Drop(41, 24, 21, DuelRank.SATEC),
    Drop(44, 24, 21, DuelRank.SATEC),
    Drop(49, 24, 66, DuelRank.SATEC),
    Drop(50, 24, 66, DuelRank.SATEC),
    Drop(53, 24, 93, DuelRank.SATEC),
    Drop(54, 24, 93, DuelRank.SATEC),
    Drop(55, 24, 66, DuelRank.SATEC),
    Drop(70, 24, 21, DuelRank.SATEC),
    Drop(77, 24, 21, DuelRank.SATEC),
    Drop(80, 24, 21, DuelRank.SATEC),
    Drop(109, 24, 21, DuelRank.SATEC),
    Drop(115, 24, 21, DuelRank.SATEC),
    Drop(116, 24, 21, DuelRank.SATEC),
    Drop(119, 24, 21, DuelRank.SATEC),
    Drop(121, 24, 21, DuelRank.SATEC),
    Drop(125, 24, 20, DuelRank.SATEC),
    Drop(141, 24, 20, DuelRank.SATEC),
    Drop(149, 24, 20, DuelRank.SATEC),
    Drop(157, 24, 20, DuelRank.SATEC),
    Drop(158, 24, 20, DuelRank.SATEC),
    Drop(164, 24, 20, DuelRank.SATEC),
    Drop(166, 24, 20, DuelRank.SATEC),
    Drop(171, 24, 20, DuelRank.SATEC),
    Drop(180, 24, 20, DuelRank.SATEC),
    Drop(186, 24, 20, DuelRank.SATEC),
    Drop(193, 24, 20, DuelRank.SATEC),
    Drop(194, 24, 20, DuelRank.SATEC),
    Drop(196, 24, 20, DuelRank.SATEC),
    Drop(201, 24, 20, DuelRank.SATEC),
    Drop(209, 24, 20, DuelRank.SATEC),
    Drop(220, 24, 20, DuelRank.SATEC),
    Drop(221, 24, 20, DuelRank.SATEC),
    Drop(225, 24, 20, DuelRank.SATEC),
    Drop(233, 24, 20, DuelRank.SATEC),
    Drop(236, 24, 20, DuelRank.SATEC),
    Drop(248, 24, 20, DuelRank.SATEC),
    Drop(255, 24, 20, DuelRank.SATEC),
    Drop(256, 24, 20, DuelRank.SATEC),
    Drop(259, 24, 20, DuelRank.SATEC),
    Drop(272, 24, 20, DuelRank.SATEC),
    Drop(273, 24, 20, DuelRank.SATEC),
    Drop(274, 24, 20, DuelRank.SATEC),
    Drop(280, 24, 20, DuelRank.SATEC),
    Drop(291, 24, 20, DuelRank.SATEC),
    Drop(293, 24, 20, DuelRank.SATEC),
    Drop(305, 24, 36, DuelRank.SATEC),
    Drop(306, 24, 36, DuelRank.SATEC),
    Drop(325, 24, 36, DuelRank.SATEC),
    Drop(330, 24, 36, DuelRank.SATEC),
    Drop(406, 24, 20, DuelRank.SATEC),
    Drop(412, 24, 20, DuelRank.SATEC),
    Drop(459, 24, 20, DuelRank.SATEC),
    Drop(476, 24, 20, DuelRank.SATEC),
    Drop(477, 24, 20, DuelRank.SATEC),
    Drop(478, 24, 20, DuelRank.SATEC),
    Drop(479, 24, 20, DuelRank.SATEC),
    Drop(496, 24, 20, DuelRank.SATEC),
    Drop(501, 24, 20, DuelRank.SATEC),
    Drop(506, 24, 20, DuelRank.SATEC),
    Drop(510, 24, 20, DuelRank.SATEC),
    Drop(511, 24, 20, DuelRank.SATEC),
    Drop(514, 24, 20, DuelRank.SATEC),
    Drop(534, 24, 20, DuelRank.SATEC),
    Drop(552, 24, 20, DuelRank.SATEC),
    Drop(567, 24, 20, DuelRank.SATEC),
    Drop(576, 24, 20, DuelRank.SATEC),
    Drop(579, 24, 20, DuelRank.SATEC),
    Drop(589, 24, 20, DuelRank.SATEC),
    Drop(608, 24, 20, DuelRank.SATEC),
    Drop(609, 24, 20, DuelRank.SATEC),
    Drop(620, 24, 42, DuelRank.SATEC),
    Drop(643, 24, 20, DuelRank.SATEC),
    Drop(662, 24, 40, DuelRank.SATEC),
    Drop(683, 24, 26, DuelRank.SATEC),
    Drop(684, 24, 26, DuelRank.SATEC),
    Drop(688, 24, 26, DuelRank.SATEC),
    Drop(9, 25, 56, DuelRank.SATEC),
    Drop(20, 25, 4, DuelRank.SATEC),
    Drop(24, 25, 56, DuelRank.SATEC),
    Drop(58, 25, 56, DuelRank.SATEC),
    Drop(112, 25, 8, DuelRank.SATEC),
    Drop(123, 25, 56, DuelRank.SATEC),
    Drop(146, 25, 8, DuelRank.SATEC),
    Drop(153, 25, 8, DuelRank.SATEC),
    Drop(165, 25, 8, DuelRank.SATEC),
    Drop(167, 25, 56, DuelRank.SATEC),
    Drop(192, 25, 56, DuelRank.SATEC),
    Drop(234, 25, 8, DuelRank.SATEC),
    Drop(241, 25, 8, DuelRank.SATEC),
    Drop(289, 25, 56, DuelRank.SATEC),
    Drop(307, 25, 18, DuelRank.SATEC),
    Drop(308, 25, 18, DuelRank.SATEC),
    Drop(309, 25, 18, DuelRank.SATEC),
    Drop(311, 25, 18, DuelRank.SATEC),
    Drop(312, 25, 18, DuelRank.SATEC),
    Drop(323, 25, 20, DuelRank.SATEC),
    Drop(327, 25, 20, DuelRank.SATEC),
    Drop(332, 25, 36, DuelRank.SATEC),
    Drop(336, 25, 20, DuelRank.SATEC),
    Drop(338, 25, 36, DuelRank.SATEC),
    Drop(339, 25, 36, DuelRank.SATEC),
    Drop(340, 25, 36, DuelRank.SATEC),
    Drop(341, 25, 24, DuelRank.SATEC),
    Drop(342, 25, 10, DuelRank.SATEC),
    Drop(345, 25, 18, DuelRank.SATEC),
    Drop(350, 25, 36, DuelRank.SATEC),
    Drop(384, 25, 8, DuelRank.SATEC),
    Drop(387, 25, 56, DuelRank.SATEC),
    Drop(393, 25, 56, DuelRank.SATEC),
    Drop(397, 25, 56, DuelRank.SATEC),
    Drop(398, 25, 56, DuelRank.SATEC),
    Drop(399, 25, 60, DuelRank.SATEC),
    Drop(402, 25, 60, DuelRank.SATEC),
    Drop(411, 25, 60, DuelRank.SATEC),
    Drop(435, 25, 8, DuelRank.SATEC),
    Drop(450, 25, 8, DuelRank.SATEC),
    Drop(469, 25, 60, DuelRank.SATEC),
    Drop(475, 25, 60, DuelRank.SATEC),
    Drop(480, 25, 8, DuelRank.SATEC),
    Drop(484, 25, 60, DuelRank.SATEC),
    Drop(502, 25, 8, DuelRank.SATEC),
    Drop(504, 25, 60, DuelRank.SATEC),
    Drop(527, 25, 60, DuelRank.SATEC),
    Drop(547, 25, 60, DuelRank.SATEC),
    Drop(548, 25, 60, DuelRank.SATEC),
    Drop(558, 25, 60, DuelRank.SATEC),
    Drop(570, 25, 8, DuelRank.SATEC),
    Drop(580, 25, 8, DuelRank.SATEC),
    Drop(635, 25, 60, DuelRank.SATEC),
    Drop(655, 25, 18, DuelRank.SATEC),
    Drop(659, 25, 12, DuelRank.SATEC),
    Drop(665, 25, 8, DuelRank.SATEC),
    Drop(666, 25, 8, DuelRank.SATEC),
    Drop(671, 25, 8, DuelRank.SATEC),
    Drop(673, 25, 8, DuelRank.SATEC),
    Drop(674, 25, 8, DuelRank.SATEC),
    Drop(676, 25, 8, DuelRank.SATEC),
    Drop(677, 25, 8, DuelRank.SATEC),
    Drop(678, 25, 8, DuelRank.SATEC),
    Drop(679, 25, 8, DuelRank.SATEC),
    Drop(680, 25, 8, DuelRank.SATEC),
    Drop(681, 25, 12, DuelRank.SATEC),
    Drop(683, 25, 16, DuelRank.SATEC),
    Drop(690, 25, 12, DuelRank.SATEC),
    Drop(691, 25, 8, DuelRank.SATEC),
    Drop(692, 25, 8, DuelRank.SATEC),
    Drop(693, 25, 8, DuelRank.SATEC),
    Drop(694, 25, 8, DuelRank.SATEC),
    Drop(695, 25, 8, DuelRank.SATEC),
    Drop(697, 25, 8, DuelRank.SATEC),
    Drop(698, 25, 8, DuelRank.SATEC),
    Drop(699, 25, 8, DuelRank.SATEC),
    Drop(700, 25, 8, DuelRank.SATEC),
    Drop(2, 26, 28, DuelRank.SATEC),
    Drop(6, 26, 28, DuelRank.SATEC),
    Drop(13, 26, 64, DuelRank.SATEC),
    Drop(20, 26, 2, DuelRank.SATEC),
    Drop(26, 26, 28, DuelRank.SATEC),
    Drop(27, 26, 28, DuelRank.SATEC),
    Drop(31, 26, 28, DuelRank.SATEC),
    Drop(32, 26, 28, DuelRank.SATEC),
    Drop(36, 26, 28, DuelRank.SATEC),
    Drop(41, 26, 28, DuelRank.SATEC),
    Drop(43, 26, 28, DuelRank.SATEC),
    Drop(44, 26, 28, DuelRank.SATEC),
    Drop(46, 26, 28, DuelRank.SATEC),
    Drop(49, 26, 28, DuelRank.SATEC),
    Drop(54, 26, 28, DuelRank.SATEC),
    Drop(55, 26, 28, DuelRank.SATEC),
    Drop(70, 26, 28, DuelRank.SATEC),
    Drop(71, 26, 28, DuelRank.SATEC),
    Drop(78, 26, 28, DuelRank.SATEC),
    Drop(81, 26, 30, DuelRank.SATEC),
    Drop(83, 26, 30, DuelRank.SATEC),
    Drop(86, 26, 30, DuelRank.SATEC),
    Drop(91, 26, 30, DuelRank.SATEC),
    Drop(95, 26, 30, DuelRank.SATEC),
    Drop(125, 26, 64, DuelRank.SATEC),
    Drop(127, 26, 30, DuelRank.SATEC),
    Drop(136, 26, 30, DuelRank.SATEC),
    Drop(149, 26, 30, DuelRank.SATEC),
    Drop(168, 26, 30, DuelRank.SATEC),
    Drop(170, 26, 30, DuelRank.SATEC),
    Drop(186, 26, 40, DuelRank.SATEC),
    Drop(207, 26, 40, DuelRank.SATEC),
    Drop(230, 26, 24, DuelRank.SATEC),
    Drop(272, 26, 40, DuelRank.SATEC),
    Drop(307, 26, 28, DuelRank.SATEC),
    Drop(312, 26, 28, DuelRank.SATEC),
    Drop(313, 26, 28, DuelRank.SATEC),
    Drop(316, 26, 48, DuelRank.SATEC),
    Drop(317, 26, 48, DuelRank.SATEC),
    Drop(321, 26, 32, DuelRank.SATEC),
    Drop(325, 26, 28, DuelRank.SATEC),
    Drop(327, 26, 28, DuelRank.SATEC),
    Drop(332, 26, 40, DuelRank.SATEC),
    Drop(368, 26, 24, DuelRank.SATEC),
    Drop(379, 26, 24, DuelRank.SATEC),
    Drop(405, 26, 24, DuelRank.SATEC),
    Drop(412, 26, 24, DuelRank.SATEC),
    Drop(415, 26, 24, DuelRank.SATEC),
    Drop(416, 26, 24, DuelRank.SATEC),
    Drop(460, 26, 30, DuelRank.SATEC),
    Drop(462, 26, 30, DuelRank.SATEC),
    Drop(466, 26, 64, DuelRank.SATEC),
    Drop(468, 26, 40, DuelRank.SATEC),
    Drop(511, 26, 24, DuelRank.SATEC),
    Drop(512, 26, 24, DuelRank.SATEC),
    Drop(538, 26, 40, DuelRank.SATEC),
    Drop(552, 26, 40, DuelRank.SATEC),
    Drop(597, 26, 30, DuelRank.SATEC),
    Drop(636, 26, 64, DuelRank.SATEC),
    Drop(648, 26, 24, DuelRank.SATEC),
    Drop(672, 26, 40, DuelRank.SATEC),
    Drop(676, 26, 24, DuelRank.SATEC),
    Drop(684, 26, 24, DuelRank.SATEC),
    Drop(685, 26, 24, DuelRank.SATEC),
    Drop(687, 26, 24, DuelRank.SATEC),
    Drop(5, 27, 16, DuelRank.SATEC),
    Drop(8, 27, 16, DuelRank.SATEC),
    Drop(19, 27, 4, DuelRank.SATEC),
    Drop(29, 27, 16, DuelRank.SATEC),
    Drop(32, 27, 64, DuelRank.SATEC),
    Drop(45, 27, 16, DuelRank.SATEC),
    Drop(47, 27, 16, DuelRank.SATEC),
    Drop(48, 27, 16, DuelRank.SATEC),
    Drop(59, 27, 32, DuelRank.SATEC),
    Drop(80, 27, 62, DuelRank.SATEC),
    Drop(81, 27, 62, DuelRank.SATEC),
    Drop(96, 27, 16, DuelRank.SATEC),
    Drop(97, 27, 16, DuelRank.SATEC),
    Drop(100, 27, 16, DuelRank.SATEC),
    Drop(107, 27, 16, DuelRank.SATEC),
    Drop(108, 27, 16, DuelRank.SATEC),
    Drop(113, 27, 16, DuelRank.SATEC),
    Drop(114, 27, 16, DuelRank.SATEC),
    Drop(116, 27, 16, DuelRank.SATEC),
    Drop(129, 27, 16, DuelRank.SATEC),
    Drop(134, 27, 16, DuelRank.SATEC),
    Drop(139, 27, 16, DuelRank.SATEC),
    Drop(143, 27, 16, DuelRank.SATEC),
    Drop(144, 27, 16, DuelRank.SATEC),
    Drop(145, 27, 16, DuelRank.SATEC),
    Drop(147, 27, 16, DuelRank.SATEC),
    Drop(148, 27, 16, DuelRank.SATEC),
    Drop(155, 27, 16, DuelRank.SATEC),
    Drop(156, 27, 16, DuelRank.SATEC),
    Drop(158, 27, 16, DuelRank.SATEC),
    Drop(160, 27, 16, DuelRank.SATEC),
    Drop(161, 27, 16, DuelRank.SATEC),
    Drop(173, 27, 16, DuelRank.SATEC),
    Drop(175, 27, 16, DuelRank.SATEC),
    Drop(176, 27, 16, DuelRank.SATEC),
    Drop(177, 27, 16, DuelRank.SATEC),
    Drop(178, 27, 16, DuelRank.SATEC),
    Drop(180, 27, 16, DuelRank.SATEC),
    Drop(183, 27, 16, DuelRank.SATEC),
    Drop(184, 27, 16, DuelRank.SATEC),
    Drop(187, 27, 16, DuelRank.SATEC),
    Drop(188, 27, 16, DuelRank.SATEC),
    Drop(189, 27, 16, DuelRank.SATEC),
    Drop(190, 27, 16, DuelRank.SATEC),
    Drop(195, 27, 16, DuelRank.SATEC),
    Drop(199, 27, 16, DuelRank.SATEC),
    Drop(203, 27, 16, DuelRank.SATEC),
    Drop(205, 27, 16, DuelRank.SATEC),
    Drop(206, 27, 16, DuelRank.SATEC),
    Drop(208, 27, 16, DuelRank.SATEC),
    Drop(209, 27, 16, DuelRank.SATEC),
    Drop(215, 27, 16, DuelRank.SATEC),
    Drop(218, 27, 52, DuelRank.SATEC),
    Drop(222, 27, 16, DuelRank.SATEC),
    Drop(224, 27, 16, DuelRank.SATEC),
    Drop(226, 27, 16, DuelRank.SATEC),
    Drop(232, 27, 16, DuelRank.SATEC),
    Drop(239, 27, 16, DuelRank.SATEC),
    Drop(240, 27, 16, DuelRank.SATEC),
    Drop(243, 27, 16, DuelRank.SATEC),
    Drop(247, 27, 16, DuelRank.SATEC),
    Drop(253, 27, 16, DuelRank.SATEC),
    Drop(264, 27, 16, DuelRank.SATEC),
    Drop(265, 27, 16, DuelRank.SATEC),
    Drop(266, 27, 16, DuelRank.SATEC),
    Drop(267, 27, 16, DuelRank.SATEC),
    Drop(271, 27, 16, DuelRank.SATEC),
    Drop(277, 27, 16, DuelRank.SATEC),
    Drop(282, 27, 16, DuelRank.SATEC),
    Drop(283, 27, 16, DuelRank.SATEC),
    Drop(292, 27, 16, DuelRank.SATEC),
    Drop(298, 27, 16, DuelRank.SATEC),
    Drop(308, 27, 26, DuelRank.SATEC),
    Drop(324, 27, 26, DuelRank.SATEC),
    Drop(326, 27, 26, DuelRank.SATEC),
    Drop(331, 27, 50, DuelRank.SATEC),
    Drop(332, 27, 50, DuelRank.SATEC),
    Drop(346, 27, 24, DuelRank.SATEC),
    Drop(417, 27, 16, DuelRank.SATEC),
    Drop(421, 27, 16, DuelRank.SATEC),
    Drop(432, 27, 16, DuelRank.SATEC),
    Drop(451, 27, 16, DuelRank.SATEC),
    Drop(463, 27, 16, DuelRank.SATEC),
    Drop(476, 27, 16, DuelRank.SATEC),
    Drop(477, 27, 16, DuelRank.SATEC),
    Drop(478, 27, 16, DuelRank.SATEC),
    Drop(479, 27, 16, DuelRank.SATEC),
    Drop(481, 27, 16, DuelRank.SATEC),
    Drop(490, 27, 16, DuelRank.SATEC),
    Drop(503, 27, 16, DuelRank.SATEC),
    Drop(510, 27, 16, DuelRank.SATEC),
    Drop(534, 27, 16, DuelRank.SATEC),
    Drop(536, 27, 16, DuelRank.SATEC),
    Drop(538, 27, 16, DuelRank.SATEC),
    Drop(544, 27, 16, DuelRank.SATEC),
    Drop(556, 27, 16, DuelRank.SATEC),
    Drop(568, 27, 30, DuelRank.SATEC),
    Drop(569, 27, 16, DuelRank.SATEC),
    Drop(573, 27, 52, DuelRank.SATEC),
    Drop(585, 27, 16, DuelRank.SATEC),
    Drop(590, 27, 16, DuelRank.SATEC),
    Drop(604, 27, 16, DuelRank.SATEC),
    Drop(605, 27, 16, DuelRank.SATEC),
    Drop(606, 27, 16, DuelRank.SATEC),
    Drop(682, 27, 24, DuelRank.SATEC),
    Drop(683, 27, 24, DuelRank.SATEC),
    Drop(3, 28, 12, DuelRank.SATEC),
    Drop(4, 28, 12, DuelRank.SATEC),
    Drop(5, 28, 25, DuelRank.SATEC),
    Drop(6, 28, 12, DuelRank.SATEC),
    Drop(20, 28, 6, DuelRank.SATEC),
    Drop(23, 28, 12, DuelRank.SATEC),
    Drop(25, 28, 12, DuelRank.SATEC),
    Drop(26, 28, 12, DuelRank.SATEC),
    Drop(29, 28, 12, DuelRank.SATEC),
    Drop(34, 28, 25, DuelRank.SATEC),
    Drop(40, 28, 12, DuelRank.SATEC),
    Drop(42, 28, 12, DuelRank.SATEC),
    Drop(44, 28, 12, DuelRank.SATEC),
    Drop(47, 28, 12, DuelRank.SATEC),
    Drop(48, 28, 25, DuelRank.SATEC),
    Drop(59, 28, 12, DuelRank.SATEC),
    Drop(61, 28, 12, DuelRank.SATEC),
    Drop(65, 28, 12, DuelRank.SATEC),
    Drop(77, 28, 12, DuelRank.SATEC),
    Drop(83, 28, 12, DuelRank.SATEC),
    Drop(86, 28, 12, DuelRank.SATEC),
    Drop(91, 28, 12, DuelRank.SATEC),
    Drop(95, 28, 12, DuelRank.SATEC),
    Drop(97, 28, 12, DuelRank.SATEC),
    Drop(100, 28, 12, DuelRank.SATEC),
    Drop(103, 28, 12, DuelRank.SATEC),
    Drop(104, 28, 12, DuelRank.SATEC),
    Drop(108, 28, 12, DuelRank.SATEC),
    Drop(113, 28, 12, DuelRank.SATEC),
    Drop(114, 28, 25, DuelRank.SATEC),
    Drop(115, 28, 12, DuelRank.SATEC),
    Drop(116, 28, 12, DuelRank.SATEC),
    Drop(118, 28, 12, DuelRank.SATEC),
    Drop(119, 28, 12, DuelRank.SATEC),
    Drop(129, 28, 12, DuelRank.SATEC),
    Drop(133, 28, 12, DuelRank.SATEC),
    Drop(136, 28, 12, DuelRank.SATEC),
    Drop(138, 28, 12, DuelRank.SATEC),
    Drop(139, 28, 12, DuelRank.SATEC),
    Drop(140, 28, 12, DuelRank.SATEC),
    Drop(141, 28, 12, DuelRank.SATEC),
    Drop(142, 28, 25, DuelRank.SATEC),
    Drop(143, 28, 25, DuelRank.SATEC),
    Drop(144, 28, 25, DuelRank.SATEC),
    Drop(145, 28, 12, DuelRank.SATEC),
    Drop(148, 28, 12, DuelRank.SATEC),
    Drop(149, 28, 12, DuelRank.SATEC),
    Drop(155, 28, 12, DuelRank.SATEC),
    Drop(162, 28, 12, DuelRank.SATEC),
    Drop(164, 28, 12, DuelRank.SATEC),
    Drop(169, 28, 25, DuelRank.SATEC),
    Drop(171, 28, 12, DuelRank.SATEC),
    Drop(172, 28, 12, DuelRank.SATEC),
    Drop(173, 28, 25, DuelRank.SATEC),
    Drop(174, 28, 12, DuelRank.SATEC),
    Drop(175, 28, 25, DuelRank.SATEC),
    Drop(177, 28, 12, DuelRank.SATEC),
    Drop(178, 28, 25, DuelRank.SATEC),
    Drop(179, 28, 12, DuelRank.SATEC),
    Drop(180, 28, 12, DuelRank.SATEC),
    Drop(181, 28, 25, DuelRank.SATEC),
    Drop(183, 28, 12, DuelRank.SATEC),
    Drop(184, 28, 12, DuelRank.SATEC),
    Drop(188, 28, 12, DuelRank.SATEC),
    Drop(189, 28, 12, DuelRank.SATEC),
    Drop(190, 28, 25, DuelRank.SATEC),
    Drop(194, 28, 12, DuelRank.SATEC),
    Drop(199, 28, 12, DuelRank.SATEC),
    Drop(205, 28, 12, DuelRank.SATEC),
    Drop(206, 28, 12, DuelRank.SATEC),
    Drop(215, 28, 12, DuelRank.SATEC),
    Drop(218, 28, 12, DuelRank.SATEC),
    Drop(220, 28, 12, DuelRank.SATEC),
    Drop(222, 28, 12, DuelRank.SATEC),
    Drop(224, 28, 12, DuelRank.SATEC),
    Drop(226, 28, 12, DuelRank.SATEC),
    Drop(228, 28, 10, DuelRank.SATEC),
    Drop(231, 28, 10, DuelRank.SATEC),
    Drop(232, 28, 10, DuelRank.SATEC),
    Drop(233, 28, 10, DuelRank.SATEC),
    Drop(239, 28, 10, DuelRank.SATEC),
    Drop(240, 28, 10, DuelRank.SATEC),
    Drop(242, 28, 10, DuelRank.SATEC),
    Drop(243, 28, 10, DuelRank.SATEC),
    Drop(244, 28, 10, DuelRank.SATEC),
    Drop(245, 28, 10, DuelRank.SATEC),
    Drop(247, 28, 10, DuelRank.SATEC),
    Drop(250, 28, 10, DuelRank.SATEC),
    Drop(253, 28, 20, DuelRank.SATEC),
    Drop(254, 28, 10, DuelRank.SATEC),
    Drop(257, 28, 10, DuelRank.SATEC),
    Drop(259, 28, 10, DuelRank.SATEC),
    Drop(260, 28, 10, DuelRank.SATEC),
    Drop(261, 28, 10, DuelRank.SATEC),
    Drop(264, 28, 10, DuelRank.SATEC),
    Drop(266, 28, 10, DuelRank.SATEC),
    Drop(268, 28, 10, DuelRank.SATEC),
    Drop(269, 28, 20, DuelRank.SATEC),
    Drop(270, 28, 10, DuelRank.SATEC),
    Drop(271, 28, 10, DuelRank.SATEC),
    Drop(276, 28, 10, DuelRank.SATEC),
    Drop(277, 28, 20, DuelRank.SATEC),
    Drop(279, 28, 20, DuelRank.SATEC),
    Drop(282, 28, 10, DuelRank.SATEC),
    Drop(283, 28, 10, DuelRank.SATEC),
    Drop(295, 28, 20, DuelRank.SATEC),
    Drop(296, 28, 10, DuelRank.SATEC),
    Drop(298, 28, 10, DuelRank.SATEC),
    Drop(303, 28, 24, DuelRank.SATEC),
    Drop(321, 28, 25, DuelRank.SATEC),
    Drop(323, 28, 25, DuelRank.SATEC),
    Drop(331, 28, 32, DuelRank.SATEC),
    Drop(346, 28, 25, DuelRank.SATEC),
    Drop(379, 28, 20, DuelRank.SATEC),
    Drop(417, 28, 10, DuelRank.SATEC),
    Drop(432, 28, 10, DuelRank.SATEC),
    Drop(446, 28, 10, DuelRank.SATEC),
    Drop(461, 28, 10, DuelRank.SATEC),
    Drop(463, 28, 10, DuelRank.SATEC),
    Drop(477, 28, 10, DuelRank.SATEC),
    Drop(478, 28, 10, DuelRank.SATEC),
    Drop(479, 28, 10, DuelRank.SATEC),
    Drop(481, 28, 10, DuelRank.SATEC),
    Drop(486, 28, 10, DuelRank.SATEC),
    Drop(490, 28, 10, DuelRank.SATEC),
    Drop(503, 28, 10, DuelRank.SATEC),
    Drop(534, 28, 10, DuelRank.SATEC),
    Drop(538, 28, 10, DuelRank.SATEC),
    Drop(568, 28, 10, DuelRank.SATEC),
    Drop(569, 28, 10, DuelRank.SATEC),
    Drop(573, 28, 10, DuelRank.SATEC),
    Drop(585, 28, 10, DuelRank.SATEC),
    Drop(590, 28, 10, DuelRank.SATEC),
    Drop(605, 28, 10, DuelRank.SATEC),
    Drop(611, 28, 10, DuelRank.SATEC),
    Drop(612, 28, 20, DuelRank.SATEC),
    Drop(676, 28, 25, DuelRank.SATEC),
    Drop(682, 28, 25, DuelRank.SATEC),
    Drop(683, 28, 25, DuelRank.SATEC),
    Drop(684, 28, 25, DuelRank.SATEC),
    Drop(687, 28, 25, DuelRank.SATEC),
    Drop(688, 28, 25, DuelRank.SATEC),
    Drop(693, 28, 36, DuelRank.SATEC),
    Drop(695, 28, 36, DuelRank.SATEC),
    Drop(2, 29, 46, DuelRank.SATEC),
    Drop(6, 29, 46, DuelRank.SATEC),
    Drop(9, 29, 150, DuelRank.SATEC),
    Drop(10, 29, 46, DuelRank.SATEC),
    Drop(19, 29, 4, DuelRank.SATEC),
    Drop(25, 29, 46, DuelRank.SATEC),
    Drop(27, 29, 46, DuelRank.SATEC),
    Drop(30, 29, 46, DuelRank.SATEC),
    Drop(31, 29, 46, DuelRank.SATEC),
    Drop(41, 29, 46, DuelRank.SATEC),
    Drop(46, 29, 46, DuelRank.SATEC),
    Drop(47, 29, 46, DuelRank.SATEC),
    Drop(48, 29, 46, DuelRank.SATEC),
    Drop(59, 29, 46, DuelRank.SATEC),
    Drop(65, 29, 46, DuelRank.SATEC),
    Drop(102, 29, 20, DuelRank.SATEC),
    Drop(130, 29, 90, DuelRank.SATEC),
    Drop(238, 29, 20, DuelRank.SATEC),
    Drop(301, 29, 64, DuelRank.SATEC),
    Drop(313, 29, 64, DuelRank.SATEC),
    Drop(314, 29, 64, DuelRank.SATEC),
    Drop(333, 29, 64, DuelRank.SATEC),
    Drop(336, 29, 64, DuelRank.SATEC),
    Drop(345, 29, 64, DuelRank.SATEC),
    Drop(349, 29, 64, DuelRank.SATEC),
    Drop(381, 29, 2, DuelRank.SATEC),
    Drop(422, 29, 88, DuelRank.SATEC),
    Drop(436, 29, 80, DuelRank.SATEC),
    Drop(444, 29, 84, DuelRank.SATEC),
    Drop(485, 29, 80, DuelRank.SATEC),
    Drop(516, 29, 80, DuelRank.SATEC),
    Drop(563, 29, 80, DuelRank.SATEC),
    Drop(676, 29, 32, DuelRank.SATEC),
    Drop(677, 29, 32, DuelRank.SATEC),
    Drop(681, 29, 64, DuelRank.SATEC),
    Drop(682, 29, 64, DuelRank.SATEC),
    Drop(690, 29, 32, DuelRank.SATEC),
    Drop(3, 30, 100, DuelRank.SATEC),
    Drop(4, 30, 56, DuelRank.SATEC),
    Drop(5, 30, 100, DuelRank.SATEC),
    Drop(10, 30, 56, DuelRank.SATEC),
    Drop(19, 30, 4, DuelRank.SATEC),
    Drop(23, 30, 100, DuelRank.SATEC),
    Drop(26, 30, 100, DuelRank.SATEC),
    Drop(31, 30, 56, DuelRank.SATEC),
    Drop(34, 30, 100, DuelRank.SATEC),
    Drop(77, 30, 100, DuelRank.SATEC),
    Drop(91, 30, 100, DuelRank.SATEC),
    Drop(122, 30, 48, DuelRank.SATEC),
    Drop(168, 30, 48, DuelRank.SATEC),
    Drop(200, 30, 48, DuelRank.SATEC),
    Drop(296, 30, 46, DuelRank.SATEC),
    Drop(298, 30, 46, DuelRank.SATEC),
    Drop(302, 30, 40, DuelRank.SATEC),
    Drop(303, 30, 40, DuelRank.SATEC),
    Drop(329, 30, 40, DuelRank.SATEC),
    Drop(333, 30, 80, DuelRank.SATEC),
    Drop(379, 30, 100, DuelRank.SATEC),
    Drop(653, 30, 80, DuelRank.SATEC),
    Drop(661, 30, 40, DuelRank.SATEC),
    Drop(665, 30, 40, DuelRank.SATEC),
    Drop(666, 30, 40, DuelRank.SATEC),
    Drop(668, 30, 40, DuelRank.SATEC),
    Drop(672, 30, 40, DuelRank.SATEC),
    Drop(684, 30, 40, DuelRank.SATEC),
    Drop(685, 30, 80, DuelRank.SATEC),
    Drop(686, 30, 80, DuelRank.SATEC),
    Drop(687, 30, 80, DuelRank.SATEC),
    Drop(688, 30, 80, DuelRank.SATEC),
    Drop(10, 31, 14, DuelRank.SATEC),
    Drop(19, 31, 5, DuelRank.SATEC),
    Drop(23, 31, 14, DuelRank.SATEC),
    Drop(25, 31, 14, DuelRank.SATEC),
    Drop(30, 31, 30, DuelRank.SATEC),
    Drop(34, 31, 14, DuelRank.SATEC),
    Drop(36, 31, 14, DuelRank.SATEC),
    Drop(40, 31, 14, DuelRank.SATEC),
    Drop(45, 31, 14, DuelRank.SATEC),
    Drop(53, 31, 14, DuelRank.SATEC),
    Drop(59, 31, 14, DuelRank.SATEC),
    Drop(61, 31, 14, DuelRank.SATEC),
    Drop(65, 31, 14, DuelRank.SATEC),
    Drop(70, 31, 14, DuelRank.SATEC),
    Drop(71, 31, 14, DuelRank.SATEC),
    Drop(80, 31, 14, DuelRank.SATEC),
    Drop(96, 31, 14, DuelRank.SATEC),
    Drop(97, 31, 14, DuelRank.SATEC),
    Drop(98, 31, 14, DuelRank.SATEC),
    Drop(103, 31, 14, DuelRank.SATEC),
    Drop(108, 31, 14, DuelRank.SATEC),
    Drop(109, 31, 14, DuelRank.SATEC),
    Drop(110, 31, 14, DuelRank.SATEC),
    Drop(115, 31, 14, DuelRank.SATEC),
    Drop(118, 31, 14, DuelRank.SATEC),
    Drop(119, 31, 14, DuelRank.SATEC),
    Drop(120, 31, 14, DuelRank.SATEC),
    Drop(121, 31, 14, DuelRank.SATEC),
    Drop(132, 31, 30, DuelRank.SATEC),
    Drop(135, 31, 14, DuelRank.SATEC),
    Drop(138, 31, 14, DuelRank.SATEC),
    Drop(139, 31, 14, DuelRank.SATEC),
    Drop(140, 31, 30, DuelRank.SATEC),
    Drop(141, 31, 13, DuelRank.SATEC),
    Drop(152, 31, 13, DuelRank.SATEC),
    Drop(154, 31, 13, DuelRank.SATEC),
    Drop(162, 31, 13, DuelRank.SATEC),
    Drop(164, 31, 13, DuelRank.SATEC),
    Drop(169, 31, 13, DuelRank.SATEC),
    Drop(171, 31, 13, DuelRank.SATEC),
    Drop(172, 31, 13, DuelRank.SATEC),
    Drop(177, 31, 13, DuelRank.SATEC),
    Drop(181, 31, 13, DuelRank.SATEC),
    Drop(191, 31, 13, DuelRank.SATEC),
    Drop(193, 31, 13, DuelRank.SATEC),
    Drop(196, 31, 28, DuelRank.SATEC),
    Drop(199, 31, 13, DuelRank.SATEC),
    Drop(201, 31, 13, DuelRank.SATEC),
    Drop(203, 31, 13, DuelRank.SATEC),
    Drop(205, 31, 13, DuelRank.SATEC),
    Drop(206, 31, 13, DuelRank.SATEC),
    Drop(211, 31, 13, DuelRank.SATEC),
    Drop(215, 31, 13, DuelRank.SATEC),
    Drop(219, 31, 13, DuelRank.SATEC),
    Drop(220, 31, 13, DuelRank.SATEC),
    Drop(221, 31, 13, DuelRank.SATEC),
    Drop(225, 31, 13, DuelRank.SATEC),
    Drop(227, 31, 13, DuelRank.SATEC),
    Drop(228, 31, 13, DuelRank.SATEC),
    Drop(231, 31, 13, DuelRank.SATEC),
    Drop(233, 31, 13, DuelRank.SATEC),
    Drop(236, 31, 13, DuelRank.SATEC),
    Drop(237, 31, 13, DuelRank.SATEC),
    Drop(243, 31, 13, DuelRank.SATEC),
    Drop(244, 31, 13, DuelRank.SATEC),
    Drop(246, 31, 13, DuelRank.SATEC),
    Drop(248, 31, 13, DuelRank.SATEC),
    Drop(250, 31, 13, DuelRank.SATEC),
    Drop(251, 31, 13, DuelRank.SATEC),
    Drop(256, 31, 13, DuelRank.SATEC),
    Drop(257, 31, 28, DuelRank.SATEC),
    Drop(258, 31, 28, DuelRank.SATEC),
    Drop(259, 31, 13, DuelRank.SATEC),
    Drop(262, 31, 13, DuelRank.SATEC),
    Drop(263, 31, 28, DuelRank.SATEC),
    Drop(265, 31, 13, DuelRank.SATEC),
    Drop(269, 31, 13, DuelRank.SATEC),
    Drop(270, 31, 13, DuelRank.SATEC),
    Drop(272, 31, 13, DuelRank.SATEC),
    Drop(273, 31, 13, DuelRank.SATEC),
    Drop(274, 31, 13, DuelRank.SATEC),
    Drop(276, 31, 13, DuelRank.SATEC),
    Drop(280, 31, 13, DuelRank.SATEC),
    Drop(290, 31, 13, DuelRank.SATEC),
    Drop(291, 31, 13, DuelRank.SATEC),
    Drop(292, 31, 13, DuelRank.SATEC),
    Drop(293, 31, 13, DuelRank.SATEC),
    Drop(294, 31, 13, DuelRank.SATEC),
    Drop(296, 31, 13, DuelRank.SATEC),
    Drop(304, 31, 24, DuelRank.SATEC),
    Drop(305, 31, 24, DuelRank.SATEC),
    Drop(316, 31, 12, DuelRank.SATEC),
    Drop(319, 31, 12, DuelRank.SATEC),
    Drop(321, 31, 24, DuelRank.SATEC),
    Drop(326, 31, 24, DuelRank.SATEC),
    Drop(335, 31, 36, DuelRank.SATEC),
    Drop(338, 31, 12, DuelRank.SATEC),
    Drop(339, 31, 12, DuelRank.SATEC),
    Drop(340, 31, 12, DuelRank.SATEC),
    Drop(345, 31, 12, DuelRank.SATEC),
    Drop(368, 31, 28, DuelRank.SATEC),
    Drop(406, 31, 13, DuelRank.SATEC),
    Drop(414, 31, 13, DuelRank.SATEC),
    Drop(432, 31, 13, DuelRank.SATEC),
    Drop(446, 31, 13, DuelRank.SATEC),
    Drop(451, 31, 13, DuelRank.SATEC),
    Drop(452, 31, 13, DuelRank.SATEC),
    Drop(459, 31, 13, DuelRank.SATEC),
    Drop(460, 31, 13, DuelRank.SATEC),
    Drop(461, 31, 13, DuelRank.SATEC),
    Drop(462, 31, 12, DuelRank.SATEC),
    Drop(463, 31, 12, DuelRank.SATEC),
    Drop(496, 31, 12, DuelRank.SATEC),
    Drop(503, 31, 12, DuelRank.SATEC),
    Drop(514, 31, 12, DuelRank.SATEC),
    Drop(524, 31, 12, DuelRank.SATEC),
    Drop(549, 31, 12, DuelRank.SATEC),
    Drop(552, 31, 12, DuelRank.SATEC),
    Drop(556, 31, 12, DuelRank.SATEC),
    Drop(567, 31, 12, DuelRank.SATEC),
    Drop(576, 31, 12, DuelRank.SATEC),
    Drop(584, 31, 12, DuelRank.SATEC),
    Drop(591, 31, 12, DuelRank.SATEC),
    Drop(601, 31, 12, DuelRank.SATEC),
    Drop(602, 31, 12, DuelRank.SATEC),
    Drop(605, 31, 12, DuelRank.SATEC),
    Drop(606, 31, 12, DuelRank.SATEC),
    Drop(608, 31, 12, DuelRank.SATEC),
    Drop(620, 31, 12, DuelRank.SATEC),
    Drop(629, 31, 12, DuelRank.SATEC),
    Drop(642, 31, 13, DuelRank.SATEC),
    Drop(643, 31, 13, DuelRank.SATEC),
    Drop(652, 31, 24, DuelRank.SATEC),
    Drop(655, 31, 24, DuelRank.SATEC),
    Drop(683, 31, 24, DuelRank.SATEC),
    Drop(684, 31, 24, DuelRank.SATEC),
    Drop(687, 31, 24, DuelRank.SATEC),
    Drop(688, 31, 24, DuelRank.SATEC),
    Drop(2, 32, 32, DuelRank.SATEC),
    Drop(6, 32, 32, DuelRank.SATEC),
    Drop(10, 32, 32, DuelRank.SATEC),
    Drop(12, 32, 12, DuelRank.SATEC),
    Drop(15, 32, 12, DuelRank.SATEC),
    Drop(21, 32, 4, DuelRank.SATEC),
    Drop(25, 32, 32, DuelRank.SATEC),
    Drop(27, 32, 32, DuelRank.SATEC),
    Drop(30, 32, 32, DuelRank.SATEC),
    Drop(31, 32, 32, DuelRank.SATEC),
    Drop(33, 32, 12, DuelRank.SATEC),
    Drop(41, 32, 48, DuelRank.SATEC),
    Drop(43, 32, 12, DuelRank.SATEC),
    Drop(46, 32, 32, DuelRank.SATEC),
    Drop(47, 32, 32, DuelRank.SATEC),
    Drop(48, 32, 32, DuelRank.SATEC),
    Drop(59, 32, 32, DuelRank.SATEC),
    Drop(65, 32, 32, DuelRank.SATEC),
    Drop(74, 32, 32, DuelRank.SATEC),
    Drop(78, 32, 12, DuelRank.SATEC),
    Drop(89, 32, 32, DuelRank.SATEC),
    Drop(93, 32, 12, DuelRank.SATEC),
    Drop(100, 32, 12, DuelRank.SATEC),
    Drop(107, 32, 12, DuelRank.SATEC),
    Drop(110, 32, 12, DuelRank.SATEC),
    Drop(118, 32, 12, DuelRank.SATEC),
    Drop(120, 32, 12, DuelRank.SATEC),
    Drop(127, 32, 12, DuelRank.SATEC),
    Drop(138, 32, 12, DuelRank.SATEC),
    Drop(147, 32, 12, DuelRank.SATEC),
    Drop(156, 32, 12, DuelRank.SATEC),
    Drop(160, 32, 12, DuelRank.SATEC),
    Drop(161, 32, 12, DuelRank.SATEC),
    Drop(166, 32, 12, DuelRank.SATEC),
    Drop(172, 32, 12, DuelRank.SATEC),
    Drop(182, 32, 12, DuelRank.SATEC),
    Drop(185, 32, 12, DuelRank.SATEC),
    Drop(195, 32, 12, DuelRank.SATEC),
    Drop(214, 32, 12, DuelRank.SATEC),
    Drop(224, 32, 12, DuelRank.SATEC),
    Drop(225, 32, 12, DuelRank.SATEC),
    Drop(226, 32, 12, DuelRank.SATEC),
    Drop(231, 32, 12, DuelRank.SATEC),
    Drop(236, 32, 12, DuelRank.SATEC),
    Drop(239, 32, 12, DuelRank.SATEC),
    Drop(250, 32, 12, DuelRank.SATEC),
    Drop(256, 32, 12, DuelRank.SATEC),
    Drop(262, 32, 12, DuelRank.SATEC),
    Drop(266, 32, 12, DuelRank.SATEC),
    Drop(267, 32, 12, DuelRank.SATEC),
    Drop(280, 32, 12, DuelRank.SATEC),
    Drop(290, 32, 12, DuelRank.SATEC),
    Drop(293, 32, 12, DuelRank.SATEC),
    Drop(294, 32, 12, DuelRank.SATEC),
    Drop(302, 32, 24, DuelRank.SATEC),
    Drop(308, 32, 24, DuelRank.SATEC),
    Drop(310, 32, 24, DuelRank.SATEC),
    Drop(314, 32, 24, DuelRank.SATEC),
    Drop(315, 32, 24, DuelRank.SATEC),
    Drop(324, 32, 24, DuelRank.SATEC),
    Drop(328, 32, 24, DuelRank.SATEC),
    Drop(336, 32, 24, DuelRank.SATEC),
    Drop(337, 32, 24, DuelRank.SATEC),
    Drop(343, 32, 24, DuelRank.SATEC),
    Drop(344, 32, 24, DuelRank.SATEC),
    Drop(345, 32, 24, DuelRank.SATEC),
    Drop(349, 32, 48, DuelRank.SATEC),
    Drop(376, 32, 12, DuelRank.SATEC),
    Drop(445, 32, 40, DuelRank.SATEC),
    Drop(447, 32, 24, DuelRank.SATEC),
    Drop(474, 32, 40, DuelRank.SATEC),
    Drop(493, 32, 24, DuelRank.SATEC),
    Drop(495, 32, 10, DuelRank.SATEC),
    Drop(507, 32, 24, DuelRank.SATEC),
    Drop(515, 32, 24, DuelRank.SATEC),
    Drop(546, 32, 40, DuelRank.SATEC),
    Drop(559, 32, 40, DuelRank.SATEC),
    Drop(561, 32, 40, DuelRank.SATEC),
    Drop(565, 32, 10, DuelRank.SATEC),
    Drop(583, 32, 24, DuelRank.SATEC),
    Drop(622, 32, 10, DuelRank.SATEC),
    Drop(624, 32, 10, DuelRank.SATEC),
    Drop(653, 32, 24, DuelRank.SATEC),
    Drop(656, 32, 48, DuelRank.SATEC),
    Drop(661, 32, 24, DuelRank.SATEC),
    Drop(665, 32, 24, DuelRank.SATEC),
    Drop(666, 32, 24, DuelRank.SATEC),
    Drop(668, 32, 24, DuelRank.SATEC),
    Drop(672, 32, 24, DuelRank.SATEC),
    Drop(679, 32, 24, DuelRank.SATEC),
    Drop(682, 32, 24, DuelRank.SATEC),
    Drop(683, 32, 24, DuelRank.SATEC),
    Drop(684, 32, 24, DuelRank.SATEC),
    Drop(685, 32, 24, DuelRank.SATEC),
    Drop(686, 32, 24, DuelRank.SATEC),
    Drop(687, 32, 24, DuelRank.SATEC),
    Drop(688, 32, 24, DuelRank.SATEC),
    Drop(2, 33, 24, DuelRank.SATEC),
    Drop(14, 33, 24, DuelRank.SATEC),
    Drop(19, 33, 4, DuelRank.SATEC),
    Drop(30, 33, 24, DuelRank.SATEC),
    Drop(32, 33, 24, DuelRank.SATEC),
    Drop(36, 33, 70, DuelRank.SATEC),
    Drop(78, 33, 24, DuelRank.SATEC),
    Drop(81, 33, 24, DuelRank.SATEC),
    Drop(83, 33, 24, DuelRank.SATEC),
    Drop(86, 33, 24, DuelRank.SATEC),
    Drop(87, 33, 24, DuelRank.SATEC),
    Drop(89, 33, 24, DuelRank.SATEC),
    Drop(91, 33, 24, DuelRank.SATEC),
    Drop(93, 33, 70, DuelRank.SATEC),
    Drop(95, 33, 46, DuelRank.SATEC),
    Drop(97, 33, 70, DuelRank.SATEC),
    Drop(106, 33, 24, DuelRank.SATEC),
    Drop(108, 33, 24, DuelRank.SATEC),
    Drop(124, 33, 46, DuelRank.SATEC),
    Drop(127, 33, 24, DuelRank.SATEC),
    Drop(131, 33, 24, DuelRank.SATEC),
    Drop(132, 33, 24, DuelRank.SATEC),
    Drop(139, 33, 24, DuelRank.SATEC),
    Drop(150, 33, 24, DuelRank.SATEC),
    Drop(170, 33, 24, DuelRank.SATEC),
    Drop(204, 33, 24, DuelRank.SATEC),
    Drop(213, 33, 24, DuelRank.SATEC),
    Drop(215, 33, 24, DuelRank.SATEC),
    Drop(228, 33, 24, DuelRank.SATEC),
    Drop(283, 33, 24, DuelRank.SATEC),
    Drop(286, 33, 24, DuelRank.SATEC),
    Drop(319, 33, 40, DuelRank.SATEC),
    Drop(320, 33, 40, DuelRank.SATEC),
    Drop(322, 33, 20, DuelRank.SATEC),
    Drop(325, 33, 20, DuelRank.SATEC),
    Drop(331, 33, 40, DuelRank.SATEC),
    Drop(335, 33, 40, DuelRank.SATEC),
    Drop(347, 33, 40, DuelRank.SATEC),
    Drop(366, 33, 24, DuelRank.SATEC),
    Drop(368, 33, 46, DuelRank.SATEC),
    Drop(379, 33, 24, DuelRank.SATEC),
    Drop(405, 33, 46, DuelRank.SATEC),
    Drop(406, 33, 24, DuelRank.SATEC),
    Drop(412, 33, 24, DuelRank.SATEC),
    Drop(413, 33, 46, DuelRank.SATEC),
    Drop(414, 33, 24, DuelRank.SATEC),
    Drop(415, 33, 24, DuelRank.SATEC),
    Drop(416, 33, 24, DuelRank.SATEC),
    Drop(417, 33, 24, DuelRank.SATEC),
    Drop(423, 33, 24, DuelRank.SATEC),
    Drop(425, 33, 24, DuelRank.SATEC),
    Drop(441, 33, 24, DuelRank.SATEC),
    Drop(460, 33, 24, DuelRank.SATEC),
    Drop(462, 33, 24, DuelRank.SATEC),
    Drop(468, 33, 24, DuelRank.SATEC),
    Drop(491, 33, 24, DuelRank.SATEC),
    Drop(508, 33, 46, DuelRank.SATEC),
    Drop(512, 33, 24, DuelRank.SATEC),
    Drop(513, 33, 24, DuelRank.SATEC),
    Drop(514, 33, 24, DuelRank.SATEC),
    Drop(557, 33, 4, DuelRank.SATEC),
    Drop(585, 33, 24, DuelRank.SATEC),
    Drop(633, 33, 24, DuelRank.SATEC),
    Drop(636, 33, 24, DuelRank.SATEC),
    Drop(643, 33, 24, DuelRank.SATEC),
    Drop(648, 33, 46, DuelRank.SATEC),
    Drop(658, 33, 40, DuelRank.SATEC),
    Drop(660, 33, 40, DuelRank.SATEC),
    Drop(684, 33, 20, DuelRank.SATEC),
    Drop(685, 33, 20, DuelRank.SATEC),
    Drop(688, 33, 20, DuelRank.SATEC),
    Drop(6, 34, 42, DuelRank.SATEC),
    Drop(13, 34, 21, DuelRank.SATEC),
    Drop(20, 34, 4, DuelRank.SATEC),
    Drop(25, 34, 21, DuelRank.SATEC),
    Drop(26, 34, 21, DuelRank.SATEC),
    Drop(27, 34, 21, DuelRank.SATEC),
    Drop(31, 34, 21, DuelRank.SATEC),
    Drop(32, 34, 21, DuelRank.SATEC),
    Drop(36, 34, 21, DuelRank.SATEC),
    Drop(41, 34, 21, DuelRank.SATEC),
    Drop(43, 34, 21, DuelRank.SATEC),
    Drop(44, 34, 21, DuelRank.SATEC),
    Drop(46, 34, 21, DuelRank.SATEC),
    Drop(48, 34, 21, DuelRank.SATEC),
    Drop(49, 34, 21, DuelRank.SATEC),
    Drop(54, 34, 21, DuelRank.SATEC),
    Drop(55, 34, 21, DuelRank.SATEC),
    Drop(70, 34, 21, DuelRank.SATEC),
    Drop(71, 34, 21, DuelRank.SATEC),
    Drop(77, 34, 21, DuelRank.SATEC),
    Drop(81, 34, 21, DuelRank.SATEC),
    Drop(83, 34, 67, DuelRank.SATEC),
    Drop(84, 34, 67, DuelRank.SATEC),
    Drop(86, 34, 67, DuelRank.SATEC),
    Drop(87, 34, 67, DuelRank.SATEC),
    Drop(88, 34, 67, DuelRank.SATEC),
    Drop(95, 34, 43, DuelRank.SATEC),
    Drop(103, 34, 21, DuelRank.SATEC),
    Drop(115, 34, 21, DuelRank.SATEC),
    Drop(119, 34, 21, DuelRank.SATEC),
    Drop(121, 34, 21, DuelRank.SATEC),
    Drop(125, 34, 21, DuelRank.SATEC),
    Drop(136, 34, 42, DuelRank.SATEC),
    Drop(149, 34, 42, DuelRank.SATEC),
    Drop(162, 34, 20, DuelRank.SATEC),
    Drop(164, 34, 20, DuelRank.SATEC),
    Drop(166, 34, 20, DuelRank.SATEC),
    Drop(168, 34, 20, DuelRank.SATEC),
    Drop(169, 34, 20, DuelRank.SATEC),
    Drop(170, 34, 20, DuelRank.SATEC),
    Drop(171, 34, 20, DuelRank.SATEC),
    Drop(173, 34, 20, DuelRank.SATEC),
    Drop(175, 34, 20, DuelRank.SATEC),
    Drop(178, 34, 20, DuelRank.SATEC),
    Drop(181, 34, 20, DuelRank.SATEC),
    Drop(186, 34, 20, DuelRank.SATEC),
    Drop(193, 34, 20, DuelRank.SATEC),
    Drop(194, 34, 42, DuelRank.SATEC),
    Drop(204, 34, 20, DuelRank.SATEC),
    Drop(230, 34, 20, DuelRank.SATEC),
    Drop(233, 34, 42, DuelRank.SATEC),
    Drop(255, 34, 20, DuelRank.SATEC),
    Drop(269, 34, 20, DuelRank.SATEC),
    Drop(277, 34, 20, DuelRank.SATEC),
    Drop(279, 34, 20, DuelRank.SATEC),
    Drop(295, 34, 20, DuelRank.SATEC),
    Drop(303, 34, 20, DuelRank.SATEC),
    Drop(304, 34, 20, DuelRank.SATEC),
    Drop(321, 34, 20, DuelRank.SATEC),
    Drop(325, 34, 20, DuelRank.SATEC),
    Drop(335, 34, 36, DuelRank.SATEC),
    Drop(379, 34, 42, DuelRank.SATEC),
    Drop(412, 34, 20, DuelRank.SATEC),
    Drop(415, 34, 20, DuelRank.SATEC),
    Drop(416, 34, 20, DuelRank.SATEC),
    Drop(459, 34, 20, DuelRank.SATEC),
    Drop(462, 34, 20, DuelRank.SATEC),
    Drop(466, 34, 20, DuelRank.SATEC),
    Drop(511, 34, 20, DuelRank.SATEC),
    Drop(512, 34, 20, DuelRank.SATEC),
    Drop(597, 34, 20, DuelRank.SATEC),
    Drop(608, 34, 20, DuelRank.SATEC),
    Drop(636, 34, 20, DuelRank.SATEC),
    Drop(650, 34, 20, DuelRank.SATEC),
    Drop(655, 34, 18, DuelRank.SATEC),
    Drop(683, 34, 20, DuelRank.SATEC),
    Drop(684, 34, 20, DuelRank.SATEC),
    Drop(685, 34, 20, DuelRank.SATEC),
    Drop(686, 34, 20, DuelRank.SATEC),
    Drop(688, 34, 36, DuelRank.SATEC),
    Drop(690, 34, 20, DuelRank.SATEC),
    Drop(2, 35, 26, DuelRank.SATEC),
    Drop(19, 35, 4, DuelRank.SATEC),
    Drop(30, 35, 26, DuelRank.SATEC),
    Drop(34, 35, 26, DuelRank.SATEC),
    Drop(36, 35, 26, DuelRank.SATEC),
    Drop(42, 35, 26, DuelRank.SATEC),
    Drop(44, 35, 26, DuelRank.SATEC),
    Drop(74, 35, 26, DuelRank.SATEC),
    Drop(96, 35, 26, DuelRank.SATEC),
    Drop(97, 35, 26, DuelRank.SATEC),
    Drop(98, 35, 26, DuelRank.SATEC),
    Drop(104, 35, 26, DuelRank.SATEC),
    Drop(106, 35, 26, DuelRank.SATEC),
    Drop(108, 35, 26, DuelRank.SATEC),
    Drop(114, 35, 26, DuelRank.SATEC),
    Drop(115, 35, 26, DuelRank.SATEC),
    Drop(129, 35, 26, DuelRank.SATEC),
    Drop(132, 35, 26, DuelRank.SATEC),
    Drop(135, 35, 26, DuelRank.SATEC),
    Drop(139, 35, 26, DuelRank.SATEC),
    Drop(142, 35, 26, DuelRank.SATEC),
    Drop(143, 35, 26, DuelRank.SATEC),
    Drop(144, 35, 26, DuelRank.SATEC),
    Drop(145, 35, 26, DuelRank.SATEC),
    Drop(154, 35, 26, DuelRank.SATEC),
    Drop(174, 35, 24, DuelRank.SATEC),
    Drop(179, 35, 24, DuelRank.SATEC),
    Drop(183, 35, 24, DuelRank.SATEC),
    Drop(184, 35, 24, DuelRank.SATEC),
    Drop(190, 35, 24, DuelRank.SATEC),
    Drop(203, 35, 24, DuelRank.SATEC),
    Drop(213, 35, 24, DuelRank.SATEC),
    Drop(215, 35, 24, DuelRank.SATEC),
    Drop(220, 35, 24, DuelRank.SATEC),
    Drop(228, 35, 24, DuelRank.SATEC),
    Drop(237, 35, 24, DuelRank.SATEC),
    Drop(244, 35, 22, DuelRank.SATEC),
    Drop(253, 35, 22, DuelRank.SATEC),
    Drop(257, 35, 22, DuelRank.SATEC),
    Drop(259, 35, 22, DuelRank.SATEC),
    Drop(263, 35, 22, DuelRank.SATEC),
    Drop(268, 35, 22, DuelRank.SATEC),
    Drop(303, 35, 24, DuelRank.SATEC),
    Drop(313, 35, 24, DuelRank.SATEC),
    Drop(315, 35, 24, DuelRank.SATEC),
    Drop(320, 35, 1, DuelRank.SATEC),
    Drop(321, 35, 24, DuelRank.SATEC),
    Drop(322, 35, 24, DuelRank.SATEC),
    Drop(323, 35, 24, DuelRank.SATEC),
    Drop(331, 35, 48, DuelRank.SATEC),
    Drop(336, 35, 48, DuelRank.SATEC),
    Drop(337, 35, 20, DuelRank.SATEC),
    Drop(342, 35, 1, DuelRank.SATEC),
    Drop(343, 35, 24, DuelRank.SATEC),
    Drop(344, 35, 24, DuelRank.SATEC),
    Drop(345, 35, 24, DuelRank.SATEC),
    Drop(347, 35, 1, DuelRank.SATEC),
    Drop(366, 35, 22, DuelRank.SATEC),
    Drop(368, 35, 22, DuelRank.SATEC),
    Drop(455, 35, 46, DuelRank.SATEC),
    Drop(457, 35, 46, DuelRank.SATEC),
    Drop(470, 35, 24, DuelRank.SATEC),
    Drop(505, 35, 46, DuelRank.SATEC),
    Drop(525, 35, 10, DuelRank.SATEC),
    Drop(532, 35, 24, DuelRank.SATEC),
    Drop(593, 35, 10, DuelRank.SATEC),
    Drop(599, 35, 46, DuelRank.SATEC),
    Drop(623, 35, 24, DuelRank.SATEC),
    Drop(631, 35, 24, DuelRank.SATEC),
    Drop(632, 35, 10, DuelRank.SATEC),
    Drop(634, 35, 24, DuelRank.SATEC),
    Drop(646, 35, 46, DuelRank.SATEC),
    Drop(657, 35, 2, DuelRank.SATEC),
    Drop(661, 35, 2, DuelRank.SATEC),
    Drop(665, 35, 24, DuelRank.SATEC),
    Drop(672, 35, 1, DuelRank.SATEC),
    Drop(676, 35, 24, DuelRank.SATEC),
    Drop(682, 35, 24, DuelRank.SATEC),
    Drop(683, 35, 24, DuelRank.SATEC),
    Drop(684, 35, 24, DuelRank.SATEC),
    Drop(686, 35, 20, DuelRank.SATEC),
    Drop(687, 35, 24, DuelRank.SATEC),
    Drop(688, 35, 24, DuelRank.SATEC),
    Drop(690, 35, 24, DuelRank.SATEC),
    Drop(693, 35, 24, DuelRank.SATEC),
    Drop(695, 35, 24, DuelRank.SATEC),
    Drop(6, 36, 16, DuelRank.SATEC),
    Drop(20, 36, 2, DuelRank.SATEC),
    Drop(25, 36, 16, DuelRank.SATEC),
    Drop(48, 36, 12, DuelRank.SATEC),
    Drop(83, 36, 20, DuelRank.SATEC),
    Drop(84, 36, 20, DuelRank.SATEC),
    Drop(86, 36, 20, DuelRank.SATEC),
    Drop(87, 36, 20, DuelRank.SATEC),
    Drop(88, 36, 20, DuelRank.SATEC),
    Drop(95, 36, 20, DuelRank.SATEC),
    Drop(103, 36, 16, DuelRank.SATEC),
    Drop(119, 36, 16, DuelRank.SATEC),
    Drop(136, 36, 20, DuelRank.SATEC),
    Drop(137, 36, 16, DuelRank.SATEC),
    Drop(148, 36, 16, DuelRank.SATEC),
    Drop(149, 36, 20, DuelRank.SATEC),
    Drop(162, 36, 20, DuelRank.SATEC),
    Drop(164, 36, 20, DuelRank.SATEC),
    Drop(169, 36, 20, DuelRank.SATEC),
    Drop(171, 36, 20, DuelRank.SATEC),
    Drop(173, 36, 16, DuelRank.SATEC),
    Drop(175, 36, 18, DuelRank.SATEC),
    Drop(178, 36, 20, DuelRank.SATEC),
    Drop(181, 36, 20, DuelRank.SATEC),
    Drop(194, 36, 20, DuelRank.SATEC),
    Drop(204, 36, 20, DuelRank.SATEC),
    Drop(222, 36, 20, DuelRank.SATEC),
    Drop(232, 36, 20, DuelRank.SATEC),
    Drop(233, 36, 20, DuelRank.SATEC),
    Drop(240, 36, 20, DuelRank.SATEC),
    Drop(242, 36, 20, DuelRank.SATEC),
    Drop(245, 36, 20, DuelRank.SATEC),
    Drop(254, 36, 20, DuelRank.SATEC),
    Drop(261, 36, 20, DuelRank.SATEC),
    Drop(269, 36, 20, DuelRank.SATEC),
    Drop(271, 36, 20, DuelRank.SATEC),
    Drop(277, 36, 20, DuelRank.SATEC),
    Drop(279, 36, 20, DuelRank.SATEC),
    Drop(295, 36, 20, DuelRank.SATEC),
    Drop(302, 36, 25, DuelRank.SATEC),
    Drop(303, 36, 25, DuelRank.SATEC),
    Drop(308, 36, 25, DuelRank.SATEC),
    Drop(311, 36, 25, DuelRank.SATEC),
    Drop(314, 36, 25, DuelRank.SATEC),
    Drop(315, 36, 25, DuelRank.SATEC),
    Drop(324, 36, 25, DuelRank.SATEC),
    Drop(328, 36, 25, DuelRank.SATEC),
    Drop(329, 36, 25, DuelRank.SATEC),
    Drop(336, 36, 25, DuelRank.SATEC),
    Drop(337, 36, 25, DuelRank.SATEC),
    Drop(343, 36, 25, DuelRank.SATEC),
    Drop(344, 36, 25, DuelRank.SATEC),
    Drop(345, 36, 25, DuelRank.SATEC),
    Drop(349, 36, 25, DuelRank.SATEC),
    Drop(379, 36, 20, DuelRank.SATEC),
    Drop(449, 36, 8, DuelRank.SATEC),
    Drop(497, 36, 24, DuelRank.SATEC),
    Drop(523, 36, 24, DuelRank.SATEC),
    Drop(535, 36, 24, DuelRank.SATEC),
    Drop(539, 36, 40, DuelRank.SATEC),
    Drop(551, 36, 24, DuelRank.SATEC),
    Drop(553, 36, 40, DuelRank.SATEC),
    Drop(560, 36, 40, DuelRank.SATEC),
    Drop(574, 36, 40, DuelRank.SATEC),
    Drop(600, 36, 40, DuelRank.SATEC),
    Drop(639, 36, 10, DuelRank.SATEC),
    Drop(641, 36, 10, DuelRank.SATEC),
    Drop(647, 36, 24, DuelRank.SATEC),
    Drop(650, 36, 20, DuelRank.SATEC),
    Drop(653, 36, 25, DuelRank.SATEC),
    Drop(656, 36, 24, DuelRank.SATEC),
    Drop(661, 36, 48, DuelRank.SATEC),
    Drop(665, 36, 24, DuelRank.SATEC),
    Drop(666, 36, 8, DuelRank.SATEC),
    Drop(668, 36, 24, DuelRank.SATEC),
    Drop(671, 36, 8, DuelRank.SATEC),
    Drop(672, 36, 24, DuelRank.SATEC),
    Drop(673, 36, 8, DuelRank.SATEC),
    Drop(674, 36, 8, DuelRank.SATEC),
    Drop(675, 36, 16, DuelRank.SATEC),
    Drop(676, 36, 8, DuelRank.SATEC),
    Drop(677, 36, 8, DuelRank.SATEC),
    Drop(678, 36, 8, DuelRank.SATEC),
    Drop(679, 36, 24, DuelRank.SATEC),
    Drop(680, 36, 8, DuelRank.SATEC),
    Drop(684, 36, 24, DuelRank.SATEC),
    Drop(685, 36, 48, DuelRank.SATEC),
    Drop(686, 36, 48, DuelRank.SATEC),
    Drop(687, 36, 48, DuelRank.SATEC),
    Drop(688, 36, 48, DuelRank.SATEC),
    Drop(691, 36, 8, DuelRank.SATEC),
    Drop(692, 36, 8, DuelRank.SATEC),
    Drop(693, 36, 8, DuelRank.SATEC),
    Drop(694, 36, 8, DuelRank.SATEC),
    Drop(695, 36, 8, DuelRank.SATEC),
    Drop(697, 36, 8, DuelRank.SATEC),
    Drop(698, 36, 8, DuelRank.SATEC),
    Drop(699, 36, 8, DuelRank.SATEC),
    Drop(700, 36, 8, DuelRank.SATEC),
    Drop(3, 37, 22, DuelRank.SATEC),
    Drop(4, 37, 22, DuelRank.SATEC),
    Drop(10, 37, 22, DuelRank.SATEC),
    Drop(13, 37, 22, DuelRank.SATEC),
    Drop(14, 37, 22, DuelRank.SATEC),
    Drop(15, 37, 22, DuelRank.SATEC),
    Drop(19, 37, 4, DuelRank.SATEC),
    Drop(23, 37, 22, DuelRank.SATEC),
    Drop(26, 37, 22, DuelRank.SATEC),
    Drop(27, 37, 22, DuelRank.SATEC),
    Drop(29, 37, 22, DuelRank.SATEC),
    Drop(31, 37, 22, DuelRank.SATEC),
    Drop(40, 37, 52, DuelRank.SATEC),
    Drop(42, 37, 70, DuelRank.SATEC),
    Drop(44, 37, 46, DuelRank.SATEC),
    Drop(46, 37, 20, DuelRank.SATEC),
    Drop(47, 37, 20, DuelRank.SATEC),
    Drop(61, 37, 20, DuelRank.SATEC),
    Drop(65, 37, 20, DuelRank.SATEC),
    Drop(68, 37, 20, DuelRank.SATEC),
    Drop(91, 37, 20, DuelRank.SATEC),
    Drop(113, 37, 20, DuelRank.SATEC),
    Drop(121, 37, 20, DuelRank.SATEC),
    Drop(125, 37, 20, DuelRank.SATEC),
    Drop(155, 37, 20, DuelRank.SATEC),
    Drop(168, 37, 20, DuelRank.SATEC),
    Drop(186, 37, 20, DuelRank.SATEC),
    Drop(188, 37, 20, DuelRank.SATEC),
    Drop(189, 37, 20, DuelRank.SATEC),
    Drop(200, 37, 20, DuelRank.SATEC),
    Drop(201, 37, 20, DuelRank.SATEC),
    Drop(219, 37, 20, DuelRank.SATEC),
    Drop(246, 37, 20, DuelRank.SATEC),
    Drop(248, 37, 20, DuelRank.SATEC),
    Drop(255, 37, 20, DuelRank.SATEC),
    Drop(272, 37, 20, DuelRank.SATEC),
    Drop(282, 37, 20, DuelRank.SATEC),
    Drop(296, 37, 22, DuelRank.SATEC),
    Drop(298, 37, 22, DuelRank.SATEC),
    Drop(315, 37, 24, DuelRank.SATEC),
    Drop(316, 37, 24, DuelRank.SATEC),
    Drop(325, 37, 24, DuelRank.SATEC),
    Drop(329, 37, 48, DuelRank.SATEC),
    Drop(330, 37, 48, DuelRank.SATEC),
    Drop(334, 37, 48, DuelRank.SATEC),
    Drop(347, 37, 24, DuelRank.SATEC),
    Drop(376, 37, 22, DuelRank.SATEC),
    Drop(403, 37, 22, DuelRank.SATEC),
    Drop(423, 37, 22, DuelRank.SATEC),
    Drop(464, 37, 22, DuelRank.SATEC),
    Drop(466, 37, 22, DuelRank.SATEC),
    Drop(468, 37, 22, DuelRank.SATEC),
    Drop(481, 37, 22, DuelRank.SATEC),
    Drop(491, 37, 22, DuelRank.SATEC),
    Drop(496, 37, 22, DuelRank.SATEC),
    Drop(528, 37, 46, DuelRank.SATEC),
    Drop(538, 37, 22, DuelRank.SATEC),
    Drop(552, 37, 22, DuelRank.SATEC),
    Drop(577, 37, 46, DuelRank.SATEC),
    Drop(597, 37, 22, DuelRank.SATEC),
    Drop(629, 37, 22, DuelRank.SATEC),
    Drop(636, 37, 22, DuelRank.SATEC),
    Drop(642, 37, 22, DuelRank.SATEC),
    Drop(652, 37, 24, DuelRank.SATEC),
    Drop(657, 37, 48, DuelRank.SATEC),
    Drop(659, 37, 8, DuelRank.SATEC),
    Drop(661, 37, 24, DuelRank.SATEC),
    Drop(665, 37, 24, DuelRank.SATEC),
    Drop(666, 37, 24, DuelRank.SATEC),
    Drop(668, 37, 48, DuelRank.SATEC),
    Drop(672, 37, 24, DuelRank.SATEC),
    Drop(676, 37, 24, DuelRank.SATEC),
    Drop(677, 37, 24, DuelRank.SATEC),
    Drop(685, 37, 48, DuelRank.SATEC),
    Drop(686, 37, 48, DuelRank.SATEC),
    Drop(687, 37, 24, DuelRank.SATEC),
    Drop(690, 37, 24, DuelRank.SATEC),
    Drop(693, 37, 24, DuelRank.SATEC),
    Drop(695, 37, 24, DuelRank.SATEC),
    Drop(699, 37, 24, DuelRank.SATEC),
    Drop(4, 38, 30, DuelRank.SATEC),
    Drop(10, 38, 30, DuelRank.SATEC),
    Drop(20, 38, 4, DuelRank.SATEC),
    Drop(31, 38, 30, DuelRank.SATEC),
    Drop(70, 38, 30, DuelRank.SATEC),
    Drop(71, 38, 30, DuelRank.SATEC),
    Drop(89, 38, 32, DuelRank.SATEC),
    Drop(131, 38, 32, DuelRank.SATEC),
    Drop(140, 38, 32, DuelRank.SATEC),
    Drop(150, 38, 32, DuelRank.SATEC),
    Drop(168, 38, 32, DuelRank.SATEC),
    Drop(177, 38, 32, DuelRank.SATEC),
    Drop(193, 38, 32, DuelRank.SATEC),
    Drop(196, 38, 32, DuelRank.SATEC),
    Drop(199, 38, 32, DuelRank.SATEC),
    Drop(200, 38, 32, DuelRank.SATEC),
    Drop(205, 38, 30, DuelRank.SATEC),
    Drop(206, 38, 30, DuelRank.SATEC),
    Drop(230, 38, 30, DuelRank.SATEC),
    Drop(243, 38, 30, DuelRank.SATEC),
    Drop(247, 38, 30, DuelRank.SATEC),
    Drop(251, 38, 30, DuelRank.SATEC),
    Drop(258, 38, 30, DuelRank.SATEC),
    Drop(270, 38, 30, DuelRank.SATEC),
    Drop(296, 38, 30, DuelRank.SATEC),
    Drop(298, 38, 30, DuelRank.SATEC),
    Drop(307, 38, 30, DuelRank.SATEC),
    Drop(311, 38, 30, DuelRank.SATEC),
    Drop(315, 38, 30, DuelRank.SATEC),
    Drop(316, 38, 60, DuelRank.SATEC),
    Drop(317, 38, 60, DuelRank.SATEC),
    Drop(319, 38, 30, DuelRank.SATEC),
    Drop(321, 38, 30, DuelRank.SATEC),
    Drop(322, 38, 30, DuelRank.SATEC),
    Drop(328, 38, 30, DuelRank.SATEC),
    Drop(331, 38, 30, DuelRank.SATEC),
    Drop(334, 38, 30, DuelRank.SATEC),
    Drop(340, 38, 30, DuelRank.SATEC),
    Drop(341, 38, 30, DuelRank.SATEC),
    Drop(342, 38, 30, DuelRank.SATEC),
    Drop(347, 38, 60, DuelRank.SATEC),
    Drop(349, 38, 30, DuelRank.SATEC),
    Drop(350, 38, 30, DuelRank.SATEC),
    Drop(430, 38, 42, DuelRank.SATEC),
    Drop(431, 38, 42, DuelRank.SATEC),
    Drop(434, 38, 10, DuelRank.SATEC),
    Drop(440, 38, 24, DuelRank.SATEC),
    Drop(494, 38, 10, DuelRank.SATEC),
    Drop(519, 38, 10, DuelRank.SATEC),
    Drop(530, 38, 42, DuelRank.SATEC),
    Drop(550, 38, 42, DuelRank.SATEC),
    Drop(596, 38, 24, DuelRank.SATEC),
    Drop(615, 38, 42, DuelRank.SATEC),
    Drop(617, 38, 10, DuelRank.SATEC),
    Drop(625, 38, 24, DuelRank.SATEC),
    Drop(626, 38, 24, DuelRank.SATEC),
    Drop(649, 38, 24, DuelRank.SATEC),
    Drop(656, 38, 25, DuelRank.SATEC),
    Drop(659, 38, 8, DuelRank.SATEC),
    Drop(661, 38, 25, DuelRank.SATEC),
    Drop(663, 38, 48, DuelRank.SATEC),
    Drop(664, 38, 25, DuelRank.SATEC),
    Drop(665, 38, 25, DuelRank.SATEC),
    Drop(672, 38, 25, DuelRank.SATEC),
    Drop(676, 38, 24, DuelRank.SATEC),
    Drop(685, 38, 25, DuelRank.SATEC),
    Drop(686, 38, 25, DuelRank.SATEC),
    Drop(687, 38, 25, DuelRank.SATEC),
    Drop(688, 38, 24, DuelRank.SATEC),
    Drop(4, 39, 15, DuelRank.SATEC),
    Drop(9, 39, 15, DuelRank.SATEC),
    Drop(20, 39, 4, DuelRank.SATEC),
    Drop(24, 39, 15, DuelRank.SATEC),
    Drop(45, 39, 15, DuelRank.SATEC),
    Drop(47, 39, 15, DuelRank.SATEC),
    Drop(96, 39, 15, DuelRank.SATEC),
    Drop(97, 39, 15, DuelRank.SATEC),
    Drop(98, 39, 17, DuelRank.SATEC),
    Drop(108, 39, 17, DuelRank.SATEC),
    Drop(133, 39, 17, DuelRank.SATEC),
    Drop(135, 39, 17, DuelRank.SATEC),
    Drop(139, 39, 17, DuelRank.SATEC),
    Drop(140, 39, 20, DuelRank.SATEC),
    Drop(142, 39, 17, DuelRank.SATEC),
    Drop(152, 39, 17, DuelRank.SATEC),
    Drop(154, 39, 17, DuelRank.SATEC),
    Drop(155, 39, 17, DuelRank.SATEC),
    Drop(167, 39, 17, DuelRank.SATEC),
    Drop(177, 39, 26, DuelRank.SATEC),
    Drop(178, 39, 16, DuelRank.SATEC),
    Drop(180, 39, 16, DuelRank.SATEC),
    Drop(190, 39, 16, DuelRank.SATEC),
    Drop(191, 39, 16, DuelRank.SATEC),
    Drop(197, 39, 16, DuelRank.SATEC),
    Drop(199, 39, 16, DuelRank.SATEC),
    Drop(203, 39, 16, DuelRank.SATEC),
    Drop(205, 39, 16, DuelRank.SATEC),
    Drop(206, 39, 16, DuelRank.SATEC),
    Drop(211, 39, 16, DuelRank.SATEC),
    Drop(215, 39, 26, DuelRank.SATEC),
    Drop(227, 39, 16, DuelRank.SATEC),
    Drop(228, 39, 26, DuelRank.SATEC),
    Drop(237, 39, 16, DuelRank.SATEC),
    Drop(243, 39, 16, DuelRank.SATEC),
    Drop(244, 39, 26, DuelRank.SATEC),
    Drop(253, 39, 16, DuelRank.SATEC),
    Drop(260, 39, 16, DuelRank.SATEC),
    Drop(264, 39, 16, DuelRank.SATEC),
    Drop(265, 39, 16, DuelRank.SATEC),
    Drop(270, 39, 26, DuelRank.SATEC),
    Drop(279, 39, 16, DuelRank.SATEC),
    Drop(282, 39, 16, DuelRank.SATEC),
    Drop(283, 39, 16, DuelRank.SATEC),
    Drop(289, 39, 16, DuelRank.SATEC),
    Drop(290, 39, 16, DuelRank.SATEC),
    Drop(292, 39, 16, DuelRank.SATEC),
    Drop(295, 39, 16, DuelRank.SATEC),
    Drop(298, 39, 16, DuelRank.SATEC),
    Drop(304, 39, 32, DuelRank.SATEC),
    Drop(306, 39, 32, DuelRank.SATEC),
    Drop(317, 39, 32, DuelRank.SATEC),
    Drop(319, 39, 32, DuelRank.SATEC),
    Drop(321, 39, 32, DuelRank.SATEC),
    Drop(326, 39, 32, DuelRank.SATEC),
    Drop(331, 39, 32, DuelRank.SATEC),
    Drop(334, 39, 32, DuelRank.SATEC),
    Drop(335, 39, 48, DuelRank.SATEC),
    Drop(336, 39, 32, DuelRank.SATEC),
    Drop(338, 39, 32, DuelRank.SATEC),
    Drop(339, 39, 32, DuelRank.SATEC),
    Drop(340, 39, 32, DuelRank.SATEC),
    Drop(345, 39, 32, DuelRank.SATEC),
    Drop(350, 39, 32, DuelRank.SATEC),
    Drop(393, 39, 16, DuelRank.SATEC),
    Drop(394, 39, 12, DuelRank.SATEC),
    Drop(398, 39, 16, DuelRank.SATEC),
    Drop(432, 39, 26, DuelRank.SATEC),
    Drop(444, 39, 16, DuelRank.SATEC),
    Drop(446, 39, 26, DuelRank.SATEC),
    Drop(451, 39, 16, DuelRank.SATEC),
    Drop(452, 39, 16, DuelRank.SATEC),
    Drop(461, 39, 26, DuelRank.SATEC),
    Drop(463, 39, 16, DuelRank.SATEC),
    Drop(478, 39, 16, DuelRank.SATEC),
    Drop(484, 39, 16, DuelRank.SATEC),
    Drop(503, 39, 16, DuelRank.SATEC),
    Drop(516, 39, 16, DuelRank.SATEC),
    Drop(524, 39, 16, DuelRank.SATEC),
    Drop(548, 39, 16, DuelRank.SATEC),
    Drop(549, 39, 16, DuelRank.SATEC),
    Drop(556, 39, 16, DuelRank.SATEC),
    Drop(558, 39, 16, DuelRank.SATEC),
    Drop(568, 39, 16, DuelRank.SATEC),
    Drop(573, 39, 16, DuelRank.SATEC),
    Drop(585, 39, 16, DuelRank.SATEC),
    Drop(588, 39, 12, DuelRank.SATEC),
    Drop(591, 39, 16, DuelRank.SATEC),
    Drop(592, 39, 12, DuelRank.SATEC),
    Drop(602, 39, 16, DuelRank.SATEC),
    Drop(605, 39, 16, DuelRank.SATEC),
    Drop(606, 39, 16, DuelRank.SATEC),
    Drop(612, 39, 16, DuelRank.SATEC),
    Drop(652, 39, 26, DuelRank.SATEC),
    Drop(655, 39, 23, DuelRank.SATEC),
    Drop(683, 39, 32, DuelRank.SATEC),
    Drop(684, 39, 32, DuelRank.SATEC),
    Drop(687, 39, 32, DuelRank.SATEC),
    Drop(688, 39, 32, DuelRank.SATEC),
    Drop(690, 39, 32, DuelRank.SATEC),
    Drop(693, 39, 32, DuelRank.SATEC),
)

# Make useful mappings:
# duelist -> duel rank pool -> drops
duelist_to_pools = {}
# card -> duel rank pool -> drops
card_id_to_pools = {}
for drop in drop_table:
    if drop.duelist not in duelist_to_pools:
        duelist_to_pools[drop.duelist] = {}
    if drop.card_id not in card_id_to_pools:
        card_id_to_pools[drop.card_id] = {}
    if drop.duel_rank not in duelist_to_pools[drop.duelist]:
        duelist_to_pools[drop.duelist][drop.duel_rank] = []
    if drop.duel_rank not in card_id_to_pools[drop.card_id]:
        card_id_to_pools[drop.card_id][drop.duel_rank] = []
    duelist_to_pools[drop.duelist][drop.duel_rank].append(drop)
    card_id_to_pools[drop.card_id][drop.duel_rank].append(drop)
