import typing

from itertools import chain
from typing import Optional
from enum import Enum
from .drop_pools import card_id_to_pools, Drop
from .utils import Constants


class GuardianStar(Enum):
    SUN = 1
    MOON = 2
    VENUS = 3
    MERCURY = 4

    MARS = 5
    JUPITER = 6
    SATURN = 7
    URANUS = 8
    PLUTO = 9
    NEPTUNE = 10


class Type(Enum):
    DRAGON = 1
    SPELLCASTER = 2
    ZOMBIE = 3
    WARRIOR = 4
    BEAST_WARRIOR = 5
    BEAST = 6
    WINGED_BEAST = 7
    FIEND = 8
    FAIRY = 9
    INSECT = 10
    DINOSAUR = 11
    REPTILE = 12
    FISH = 13
    SEA_SERPENT = 14
    AQUA = 15
    PYRO = 16
    THUNDER = 17
    ROCK = 18
    PLANT = 19
    MACHINE = 20
    EQUIP = 21
    MAGIC = 22
    TRAP = 23
    RITUAL = 24


class Attribute(Enum):
    DARK = 1
    LIGHT = 2
    WATER = 3
    FIRE = 4
    EARTH = 5
    WIND = 6


class Card:
    id: int
    level: int
    attack: int
    defense: int
    starchips: int
    name: str
    first_guardian_star: Optional[GuardianStar]
    second_guardian_star: Optional[GuardianStar]
    type: Type
    attribute: Optional[Attribute]
    password: Optional[str]
    drop_pool: typing.Tuple[Drop, ...]

    def __init__(self, _id: int, level: int, attack: int, defense: int, starchips: int, name: str,
                 first_guardian_star: Optional[GuardianStar], second_guardian_star: Optional[GuardianStar],
                 type: Type, attribute: Optional[Attribute], password: Optional[str]):
        self.id = _id
        self.level = level
        self.attack = attack
        self.defense = defense
        self.starchips = starchips
        self.name = name
        self.first_guardian_star = first_guardian_star
        self.second_guardian_star = second_guardian_star
        self.type = type
        self.attribute = attribute
        self.password = password
        # FM-TODO: add ways to obtain card besides drops (e.g fusions, rituals, starchips)
        if self.id in card_id_to_pools:
            self.drop_pool = tuple(chain.from_iterable(card_id_to_pools[self.id].values()))
        else:
            self.drop_pool = tuple()

    def __str__(self) -> str:
        return (
            f"{self.name} "
            f"dropped from {', '.join(str(e) for e in self.drop_pool)} "
        )

    @property
    def location_id(self) -> int:
        return self.id + Constants.CARD_ID_OFFSET


all_cards: typing.Tuple[Card, ...] = (
    Card(1, 8, 3000, 2500, 999999, "Blue-eyes White Dragon",
         GuardianStar.SUN, GuardianStar.MARS, Type.DRAGON, Attribute.LIGHT, "89631139"),
    Card(2, 4, 800, 2000, 160, "Mystical Elf",
         GuardianStar.SUN, GuardianStar.JUPITER, Type.SPELLCASTER, Attribute.LIGHT, "15025844"),
    Card(3, 4, 1200, 1000, 70, "Hitotsu-me Giant",
         GuardianStar.MOON, GuardianStar.MARS, Type.BEAST_WARRIOR, Attribute.EARTH, "76184692"),
    Card(4, 3, 1200, 700, 999999, "Baby Dragon",
         GuardianStar.URANUS, GuardianStar.SUN, Type.DRAGON, Attribute.WIND, "88819587"),
    Card(5, 3, 1000, 500, 30, "Ryu-kishin",
         GuardianStar.SATURN, GuardianStar.SUN, Type.FIEND, Attribute.DARK, "15303296"),
    Card(6, 4, 1300, 1400, 140, "Feral Imp",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.FIEND, Attribute.DARK, "41392891"),
    Card(7, 4, 1400, 1200, 999999, "Winged Dragon #1",
         GuardianStar.SATURN, GuardianStar.MOON, Type.DRAGON, Attribute.WIND, "87796900"),
    Card(8, 2, 800, 600, 30, "Mushroom Man",
         GuardianStar.JUPITER, GuardianStar.SUN, Type.PLANT, Attribute.EARTH, "14181608"),
    Card(9, 1, 500, 200, 15, "Shadow Specter",
         GuardianStar.MOON, GuardianStar.SATURN, Type.ZOMBIE, Attribute.DARK, "40575313"),
    Card(10, 4, 1500, 800, 80, "Blackland Fire Dragon",
         GuardianStar.MERCURY, GuardianStar.MOON, Type.DRAGON, Attribute.DARK, "87564352"),
    Card(11, 6, 1750, 2030, 1000, "Sword Arm of Dragon",
         GuardianStar.URANUS, GuardianStar.MOON, Type.DINOSAUR, Attribute.EARTH, "13069066"),
    Card(12, 5, 1800, 1500, 290, "Swamp Battleguard",
         GuardianStar.MERCURY, GuardianStar.MARS, Type.WARRIOR, Attribute.EARTH, "40453765"),
    Card(13, 4, 1200, 1400, 120, "Tyhone",
         GuardianStar.SATURN, GuardianStar.VENUS, Type.WINGED_BEAST, Attribute.WIND, "72842870"),
    Card(14, 5, 1800, 1300, 230, "Battle Steer",
         GuardianStar.MOON, GuardianStar.URANUS, Type.BEAST_WARRIOR, Attribute.EARTH, "18246479"),
    Card(15, 5, 1800, 1600, 320, "Flame Swordsman",
         GuardianStar.MARS, GuardianStar.SUN, Type.WARRIOR, Attribute.FIRE, "45231177"),
    Card(16, 2, 500, 400, 999999, "Time Wizard",
         GuardianStar.VENUS, GuardianStar.SUN, Type.SPELLCASTER, Attribute.LIGHT, "71625222"),
    Card(17, 1, 200, 300, 999999, "Right Leg of the Forbidden One",
         GuardianStar.MERCURY, GuardianStar.JUPITER, Type.SPELLCASTER, Attribute.DARK, "8124921"),
    Card(18, 1, 200, 300, 999999, "Left Leg of the Forbidden One",
         GuardianStar.MERCURY, GuardianStar.SATURN, Type.SPELLCASTER, Attribute.DARK, "44519536"),
    Card(19, 1, 200, 300, 999999, "Right Arm of the Forbidden One",
         GuardianStar.MERCURY, GuardianStar.URANUS, Type.SPELLCASTER, Attribute.DARK, "70903634"),
    Card(20, 1, 200, 300, 999999, "Left Arm of the Forbidden One",
         GuardianStar.MERCURY, GuardianStar.PLUTO, Type.SPELLCASTER, Attribute.DARK, "7902349"),
    Card(21, 3, 1000, 1000, 999999, "Exodia the Forbidden",
         GuardianStar.MERCURY, GuardianStar.NEPTUNE, Type.SPELLCASTER, Attribute.DARK, "33396948"),
    Card(22, 6, 2500, 1200, 999999, "Summoned Skull",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.FIEND, Attribute.DARK, "70781052"),
    Card(23, 3, 1400, 700, 60, "The Wicked Worm Beast",
         GuardianStar.URANUS, GuardianStar.VENUS, Type.BEAST, Attribute.EARTH, "6285791"),
    Card(24, 1, 300, 200, 10, "Skull Servant",
         GuardianStar.MOON, GuardianStar.NEPTUNE, Type.ZOMBIE, Attribute.DARK, "32274490"),
    Card(25, 4, 1300, 1000, 80, "Horn Imp",
         GuardianStar.MOON, GuardianStar.MERCURY, Type.FIEND, Attribute.DARK, "69669405"),
    Card(26, 4, 1700, 1000, 999999, "Battle Ox",
         GuardianStar.JUPITER, GuardianStar.MOON, Type.BEAST_WARRIOR, Attribute.EARTH, "5053103"),
    Card(27, 4, 1200, 1500, 140, "Beaver Warrior",
         GuardianStar.JUPITER, GuardianStar.VENUS, Type.BEAST_WARRIOR, Attribute.EARTH, "32452818"),
    Card(28, 3, 800, 1200, 999999, "Rock Ogre Grotto #1",
         GuardianStar.URANUS, GuardianStar.MERCURY, Type.ROCK, Attribute.EARTH, "68846917"),
    Card(29, 3, 600, 1000, 35, "Mountain Warrior",
         GuardianStar.URANUS, GuardianStar.MARS, Type.BEAST_WARRIOR, Attribute.EARTH, "4931562"),
    Card(30, 3, 1200, 900, 60, "Zombie Warrior",
         GuardianStar.MOON, GuardianStar.MARS, Type.ZOMBIE, Attribute.DARK, "31339260"),
    Card(31, 4, 1500, 1200, 140, "Koumori Dragon",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.DRAGON, Attribute.DARK, "67724379"),
    Card(32, 4, 1600, 1200, 160, "Two-headed King Rex",
         GuardianStar.URANUS, GuardianStar.NEPTUNE, Type.DINOSAUR, Attribute.EARTH, "94119974"),
    Card(33, 6, 2200, 1500, 1000, "Judge Man",
         GuardianStar.SUN, GuardianStar.SATURN, Type.WARRIOR, Attribute.EARTH, "30113682"),
    Card(34, 3, 600, 1500, 60, "Saggi the Dark Clown",
         GuardianStar.MERCURY, GuardianStar.MOON, Type.SPELLCASTER, Attribute.DARK, "66602787"),
    Card(35, 7, 2500, 2100, 999999, "Dark Magician",
         GuardianStar.MERCURY, GuardianStar.MARS, Type.SPELLCASTER, Attribute.DARK, "46986414"),
    Card(36, 4, 1500, 1200, 140, "The Snake Hair",
         GuardianStar.MOON, GuardianStar.SATURN, Type.ZOMBIE, Attribute.DARK, "29491031"),
    Card(37, 7, 2600, 2100, 999999, "Gaia the Dragon Champion",
         GuardianStar.SATURN, GuardianStar.MERCURY, Type.DRAGON, Attribute.WIND, "66889139"),
    Card(38, 7, 2300, 2100, 999999, "Gaia the Fierce Knight",
         GuardianStar.MERCURY, GuardianStar.JUPITER, Type.WARRIOR, Attribute.EARTH, "6368038"),
    Card(39, 5, 2000, 1500, 999999, "Curse of Dragon",
         GuardianStar.SATURN, GuardianStar.PLUTO, Type.DRAGON, Attribute.DARK, "28279543"),
    Card(40, 3, 200, 1800, 55, "Dragon Piper",
         GuardianStar.MARS, GuardianStar.MOON, Type.PYRO, Attribute.FIRE, "55763552"),
    Card(41, 4, 1400, 1200, 120, "Celtic Guardian",
         GuardianStar.SUN, GuardianStar.URANUS, Type.WARRIOR, Attribute.EARTH, "91152256"),
    Card(42, 5, 1200, 2200, 320, "Illusionist Faceless Mage",
         GuardianStar.VENUS, GuardianStar.MARS, Type.SPELLCASTER, Attribute.DARK, "28546905"),
    Card(43, 4, 1500, 1200, 140, "Karbonala Warrior",
         GuardianStar.URANUS, GuardianStar.JUPITER, Type.WARRIOR, Attribute.EARTH, "54541900"),
    Card(44, 4, 1600, 1000, 120, "Rogue Doll",
         GuardianStar.SUN, GuardianStar.PLUTO, Type.SPELLCASTER, Attribute.LIGHT, "91939608"),
    Card(45, 3, 1000, 500, 30, "Oscillo Hero #2",
         GuardianStar.PLUTO, GuardianStar.SUN, Type.THUNDER, Attribute.LIGHT, "27324313"),
    Card(46, 4, 1200, 1500, 140, "Griffore",
         GuardianStar.JUPITER, GuardianStar.MERCURY, Type.BEAST, Attribute.EARTH, "53829412"),
    Card(47, 3, 1200, 600, 45, "Torike",
         GuardianStar.URANUS, GuardianStar.SUN, Type.BEAST, Attribute.EARTH, "80813021"),
    Card(48, 3, 1000, 600, 35, "Sangan",
         GuardianStar.MOON, GuardianStar.URANUS, Type.FIEND, Attribute.DARK, "26202165"),
    Card(49, 4, 1200, 1500, 140, "Big Insect",
         GuardianStar.JUPITER, GuardianStar.URANUS, Type.INSECT, Attribute.EARTH, "53606874"),
    Card(50, 2, 500, 700, 25, "Basic Insect",
         GuardianStar.JUPITER, GuardianStar.PLUTO, Type.INSECT, Attribute.EARTH, "89091579"),
    Card(51, 4, 1500, 1200, 999999, "Armored Lizard",
         GuardianStar.NEPTUNE, GuardianStar.SUN, Type.REPTILE, Attribute.EARTH, "15480588"),
    Card(52, 5, 1500, 2000, 999999, "Hercules Beetle",
         GuardianStar.JUPITER, GuardianStar.NEPTUNE, Type.INSECT, Attribute.EARTH, "52584282"),
    Card(53, 4, 1200, 1000, 70, "Killer Needle",
         GuardianStar.JUPITER, GuardianStar.SUN, Type.INSECT, Attribute.WIND, "88979991"),
    Card(54, 4, 1200, 1400, 999999, "Gokibore",
         GuardianStar.JUPITER, GuardianStar.MOON, Type.INSECT, Attribute.EARTH, "15367030"),
    Card(55, 4, 1500, 1200, 140, "Giant Flea",
         GuardianStar.JUPITER, GuardianStar.VENUS, Type.INSECT, Attribute.EARTH, "41762634"),
    Card(56, 2, 500, 400, 999999, "Larvae Moth",
         GuardianStar.JUPITER, GuardianStar.MERCURY, Type.INSECT, Attribute.EARTH, "87756343"),
    Card(57, 8, 2600, 2500, 999999, "Great Moth",
         GuardianStar.JUPITER, GuardianStar.URANUS, Type.INSECT, Attribute.EARTH, "14141448"),
    Card(58, 1, 300, 200, 10, "Kuriboh",
         GuardianStar.MERCURY, GuardianStar.SATURN, Type.FIEND, Attribute.DARK, "40640057"),
    Card(59, 3, 1200, 800, 55, "Mammoth Graveyard",
         GuardianStar.MERCURY, GuardianStar.PLUTO, Type.DINOSAUR, Attribute.EARTH, "40374923"),
    Card(60, 4, 1600, 800, 999999, "Great White",
         GuardianStar.NEPTUNE, GuardianStar.MOON, Type.FISH, Attribute.WATER, "13429800"),
    Card(61, 3, 1200, 800, 55, "Wolf",
         GuardianStar.JUPITER, GuardianStar.PLUTO, Type.BEAST, Attribute.EARTH, "49417509"),
    Card(62, 4, 1300, 1400, 999999, "Harpie Lady",
         GuardianStar.SATURN, GuardianStar.NEPTUNE, Type.WINGED_BEAST, Attribute.WIND, "76812113"),
    Card(63, 6, 1950, 2100, 999999, "Harpie Lady Sisters",
         GuardianStar.SATURN, GuardianStar.MARS, Type.WINGED_BEAST, Attribute.WIND, "12206212"),
    Card(64, 4, 1300, 1100, 999999, "Tiger Axe",
         GuardianStar.JUPITER, GuardianStar.NEPTUNE, Type.BEAST_WARRIOR, Attribute.EARTH, "49791927"),
    Card(65, 3, 1200, 800, 55, "Silver Fang",
         GuardianStar.JUPITER, GuardianStar.SUN, Type.BEAST, Attribute.EARTH, "90357090"),
    Card(66, 4, 1500, 1200, 140, "Kojikocy",
         GuardianStar.URANUS, GuardianStar.MOON, Type.WARRIOR, Attribute.EARTH, "1184620"),
    Card(67, 8, 3500, 3000, 999999, "Perfectly Ultimate Great Moth",
         GuardianStar.JUPITER, GuardianStar.MOON, Type.INSECT, Attribute.EARTH, "48579379"),
    Card(68, 5, 1800, 1500, 290, "Garoozis",
         GuardianStar.MARS, GuardianStar.VENUS, Type.BEAST_WARRIOR, Attribute.FIRE, "14977074"),
    Card(69, 7, 2400, 2000, 999999, "Thousand Dragon",
         GuardianStar.MARS, GuardianStar.MERCURY, Type.DRAGON, Attribute.WIND, "41462083"),
    Card(70, 4, 1200, 1400, 120, "Fiend Kraken",
         GuardianStar.NEPTUNE, GuardianStar.VENUS, Type.AQUA, Attribute.WATER, "77456781"),
    Card(71, 4, 1200, 1500, 140, "Jellyfish",
         GuardianStar.NEPTUNE, GuardianStar.MERCURY, Type.AQUA, Attribute.WATER, "14851496"),
    Card(72, 3, 0, 2000, 999999, "Cocoon of Evolution",
         GuardianStar.JUPITER, GuardianStar.VENUS, Type.INSECT, Attribute.EARTH, "40240595"),
    Card(73, 5, 1800, 1500, 290, "Kairyu-shin",
         GuardianStar.NEPTUNE, GuardianStar.MARS, Type.SEA_SERPENT, Attribute.WATER, "76634149"),
    Card(74, 3, 1300, 2000, 290, "Giant Soldier of Stone",
         GuardianStar.URANUS, GuardianStar.VENUS, Type.ROCK, Attribute.EARTH, "13039848"),
    Card(75, 2, 800, 600, 30, "Man-eating Plant",
         GuardianStar.MERCURY, GuardianStar.JUPITER, Type.PLANT, Attribute.EARTH, "49127943"),
    Card(76, 4, 1100, 1200, 80, "Krokodilus",
         GuardianStar.NEPTUNE, GuardianStar.JUPITER, Type.REPTILE, Attribute.WATER, "76512652"),
    Card(77, 4, 1300, 1200, 100, "Grappler",
         GuardianStar.NEPTUNE, GuardianStar.SATURN, Type.REPTILE, Attribute.WATER, "2906250"),
    Card(78, 4, 1700, 1150, 160, "Axe Raider",
         GuardianStar.URANUS, GuardianStar.MERCURY, Type.WARRIOR, Attribute.EARTH, "48305365"),
    Card(79, 6, 1800, 2000, 500, "Megazowler",
         GuardianStar.URANUS, GuardianStar.NEPTUNE, Type.DINOSAUR, Attribute.EARTH, "75390004"),
    Card(80, 4, 1500, 800, 80, "Uraby",
         GuardianStar.URANUS, GuardianStar.MARS, Type.DINOSAUR, Attribute.EARTH, "1784619"),
    Card(81, 4, 1600, 1200, 160, "Crawling Dragon #2",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.DINOSAUR, Attribute.EARTH, "38289717"),
    Card(82, 7, 2400, 2000, 999999, "Red-eyes B. Dragon",
         GuardianStar.MOON, GuardianStar.NEPTUNE, Type.DRAGON, Attribute.DARK, "74677422"),
    Card(83, 4, 920, 1930, 160, "Castle of Dark Illusions",
         GuardianStar.MERCURY, GuardianStar.URANUS, Type.FIEND, Attribute.DARK, "62121"),
    Card(84, 5, 1380, 1930, 290, "Reaper of the Cards",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.FIEND, Attribute.DARK, "33066139"),
    Card(85, 5, 2000, 1530, 350, "King of Yamimakai",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.FIEND, Attribute.DARK, "69455834"),
    Card(86, 5, 1380, 1530, 180, "Barox",
         GuardianStar.JUPITER, GuardianStar.MERCURY, Type.FIEND, Attribute.DARK, "6840573"),
    Card(87, 5, 1610, 1460, 200, "Dark Chimera",
         GuardianStar.MERCURY, GuardianStar.NEPTUNE, Type.FIEND, Attribute.DARK, "32344688"),
    Card(88, 5, 1150, 2150, 290, "Metal Guardian",
         GuardianStar.MOON, GuardianStar.MERCURY, Type.FIEND, Attribute.DARK, "68339286"),
    Card(89, 5, 1000, 2000, 200, "Catapult Turtle",
         GuardianStar.NEPTUNE, GuardianStar.SUN, Type.AQUA, Attribute.WATER, "95727991"),
    Card(90, 6, 1800, 2000, 500, "Gyakutenno Megami",
         GuardianStar.SUN, GuardianStar.NEPTUNE, Type.FAIRY, Attribute.LIGHT, "31122090"),
    Card(91, 4, 1300, 1550, 999999, "Mystic Horseman",
         GuardianStar.JUPITER, GuardianStar.URANUS, Type.BEAST, Attribute.EARTH, "68516705"),
    Card(92, 6, 2000, 1700, 999999, "Rabid Horseman",
         GuardianStar.JUPITER, GuardianStar.PLUTO, Type.BEAST_WARRIOR, Attribute.EARTH, "94905343"),
    Card(93, 5, 1500, 1700, 260, "Zanki",
         GuardianStar.URANUS, GuardianStar.JUPITER, Type.WARRIOR, Attribute.EARTH, "30090452"),
    Card(94, 5, 1600, 1400, 200, "Crawling Dragon",
         GuardianStar.URANUS, GuardianStar.SUN, Type.DRAGON, Attribute.EARTH, "67494157"),
    Card(95, 4, 1350, 1400, 140, "Crass Clown",
         GuardianStar.VENUS, GuardianStar.JUPITER, Type.FIEND, Attribute.DARK, "93889755"),
    Card(96, 3, 1500, 0, 260, "Armored Zombie",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "20277860"),
    Card(97, 3, 1600, 0, 200, "Dragon Zombie",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "66672569"),
    Card(98, 2, 1350, 0, 140, "Clown Zombie",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "92667214"),
    Card(99, 6, 1800, 2000, 500, "Pumpking the King of Ghosts",
         GuardianStar.JUPITER, GuardianStar.NEPTUNE, Type.ZOMBIE, Attribute.DARK, "29155212"),
    Card(100, 3, 700, 1000, 40, "Battle Warrior",
         GuardianStar.URANUS, GuardianStar.MOON, Type.WARRIOR, Attribute.EARTH, "55550921"),
    Card(101, 2, 700, 600, 30, "Wings of Wicked Flame",
         GuardianStar.MARS, GuardianStar.SATURN, Type.PYRO, Attribute.FIRE, "92944626"),
    Card(102, 2, 900, 400, 30, "Mask of Darkness",
         GuardianStar.MERCURY, GuardianStar.MOON, Type.FIEND, Attribute.DARK, "28933734"),
    Card(103, 3, 800, 1300, 999999, "Job-change Mirror",
         GuardianStar.MERCURY, GuardianStar.MARS, Type.FIEND, Attribute.DARK, "55337339"),
    Card(104, 2, 600, 500, 25, "Curtain of the Dark Ones",
         GuardianStar.MERCURY, GuardianStar.JUPITER, Type.SPELLCASTER, Attribute.DARK, "22026707"),
    Card(105, 2, 500, 400, 20, "Tomozaurus",
         GuardianStar.URANUS, GuardianStar.VENUS, Type.DINOSAUR, Attribute.EARTH, "46457856"),
    Card(106, 5, 1700, 1400, 230, "Spirit of the Winds",
         GuardianStar.SATURN, GuardianStar.SUN, Type.SPELLCASTER, Attribute.WIND, "54615781"),
    Card(107, 2, 800, 600, 30, "Kageningen",
         GuardianStar.MOON, GuardianStar.MARS, Type.WARRIOR, Attribute.DARK, "80600490"),
    Card(108, 3, 700, 900, 35, "Graveyard and the Hand of Invitation",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "27094595"),
    Card(109, 4, 1200, 1000, 70, "Goddess with the Third Eye",
         GuardianStar.SUN, GuardianStar.VENUS, Type.FAIRY, Attribute.LIGHT, "53493204"),
    Card(110, 3, 1100, 1000, 60, "Hero of the East",
         GuardianStar.URANUS, GuardianStar.MERCURY, Type.WARRIOR, Attribute.EARTH, "89987208"),
    Card(111, 5, 1600, 1400, 200, "Doma The Angel of Silence",
         GuardianStar.MERCURY, GuardianStar.SATURN, Type.FAIRY, Attribute.DARK, "16972957"),
    Card(112, 4, 1200, 1000, 70, "That Which Feeds on Life",
         GuardianStar.MERCURY, GuardianStar.URANUS, Type.FIEND, Attribute.DARK, "52367652"),
    Card(113, 3, 800, 900, 40, "Dark Gray",
         GuardianStar.JUPITER, GuardianStar.MOON, Type.BEAST, Attribute.EARTH, "9159938"),
    Card(114, 3, 1000, 700, 40, "White Magical Hat",
         GuardianStar.SUN, GuardianStar.MARS, Type.SPELLCASTER, Attribute.LIGHT, "15150365"),
    Card(115, 4, 1300, 1100, 120, "Kamion Wizard",
         GuardianStar.MERCURY, GuardianStar.PLUTO, Type.SPELLCASTER, Attribute.DARK, "41544074"),
    Card(116, 3, 900, 800, 40, "Nightmare Scorpion",
         GuardianStar.URANUS, GuardianStar.NEPTUNE, Type.INSECT, Attribute.EARTH, "88643173"),
    Card(117, 4, 1400, 1200, 120, "Spirit of the Books",
         GuardianStar.SATURN, GuardianStar.MOON, Type.WINGED_BEAST, Attribute.WIND, "14037717"),
    Card(118, 3, 1000, 1000, 55, "Supporter in the Shadows",
         GuardianStar.JUPITER, GuardianStar.VENUS, Type.WARRIOR, Attribute.EARTH, "41422426"),
    Card(119, 4, 1300, 900, 70, "Trial of Nightmares",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.FIEND, Attribute.DARK, "77827521"),
    Card(120, 3, 1200, 900, 60, "Dream Clown",
         GuardianStar.VENUS, GuardianStar.SATURN, Type.WARRIOR, Attribute.EARTH, "13215230"),
    Card(121, 4, 700, 1700, 120, "Sleeping Lion",
         GuardianStar.JUPITER, GuardianStar.SUN, Type.BEAST, Attribute.EARTH, "40200834"),
    Card(122, 2, 900, 300, 25, "Yamatano Dragon Scroll",
         GuardianStar.SATURN, GuardianStar.VENUS, Type.DRAGON, Attribute.WIND, "76704943"),
    Card(123, 1, 300, 400, 15, "Dark Plant",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.PLANT, Attribute.DARK, "13193642"),
    Card(124, 5, 1700, 1400, 230, "Ancient Tool",
         GuardianStar.MOON, GuardianStar.SATURN, Type.MACHINE, Attribute.DARK, "49587396"),
    Card(125, 4, 1500, 1100, 120, "Faith Bird",
         GuardianStar.SUN, GuardianStar.JUPITER, Type.WINGED_BEAST, Attribute.WIND, "75582395"),
    Card(126, 5, 1800, 1500, 290, "Orion the Battle King",
         GuardianStar.SUN, GuardianStar.SATURN, Type.FAIRY, Attribute.LIGHT, "2971090"),
    Card(127, 5, 1700, 1200, 180, "Ansatsu",
         GuardianStar.MERCURY, GuardianStar.NEPTUNE, Type.WARRIOR, Attribute.EARTH, "48365709"),
    Card(128, 5, 1200, 1700, 180, "LaMoon",
         GuardianStar.SUN, GuardianStar.URANUS, Type.SPELLCASTER, Attribute.LIGHT, "75850803"),
    Card(129, 3, 800, 700, 30, "Nemuriko",
         GuardianStar.MERCURY, GuardianStar.MOON, Type.SPELLCASTER, Attribute.DARK, "90963488"),
    Card(130, 2, 600, 400, 20, "Weather Control",
         GuardianStar.SUN, GuardianStar.PLUTO, Type.FAIRY, Attribute.LIGHT, "37243151"),
    Card(131, 5, 1600, 1400, 200, "Octoberser",
         GuardianStar.NEPTUNE, GuardianStar.MOON, Type.AQUA, Attribute.WATER, "74637266"),
    Card(132, 3, 1200, 900, 60, "The 13th Grave",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "32864"),
    Card(133, 3, 1100, 800, 50, "Charubin the Fire Knight",
         GuardianStar.MARS, GuardianStar.URANUS, Type.PYRO, Attribute.FIRE, "37421579"),
    Card(134, 2, 700, 700, 30, "Mystical Capture Chain",
         GuardianStar.SUN, GuardianStar.NEPTUNE, Type.FAIRY, Attribute.LIGHT, "63515678"),
    Card(135, 2, 600, 600, 25, "Fiend's Hand",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "52800428"),
    Card(136, 4, 1400, 1300, 140, "Witty Phantom",
         GuardianStar.MERCURY, GuardianStar.MARS, Type.FIEND, Attribute.DARK, "36304921"),
    Card(137, 2, 500, 500, 20, "Mystery Hand",
         GuardianStar.MOON, GuardianStar.URANUS, Type.FIEND, Attribute.DARK, "62793020"),
    Card(138, 3, 1100, 900, 55, "Dragon Statue",
         GuardianStar.URANUS, GuardianStar.MARS, Type.WARRIOR, Attribute.EARTH, "9197735"),
    Card(139, 3, 900, 700, 35, "Blue-eyed Silver Zombie",
         GuardianStar.SUN, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "35282433"),
    Card(140, 3, 1000, 1000, 55, "Toad Master",
         GuardianStar.NEPTUNE, GuardianStar.VENUS, Type.AQUA, Attribute.WATER, "62671448"),
    Card(141, 3, 700, 1300, 55, "Spiked Snail",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.INSECT, Attribute.DARK, "98075147"),
    Card(142, 3, 900, 1000, 50, "Flame Manipulator",
         GuardianStar.MARS, GuardianStar.PLUTO, Type.SPELLCASTER, Attribute.FIRE, "34460851"),
    Card(143, 3, 800, 900, 40, "Necrolancer the Timelord",
         GuardianStar.VENUS, GuardianStar.URANUS, Type.SPELLCASTER, Attribute.DARK, "61454890"),
    Card(144, 3, 700, 900, 35, "Djinn the Watcher of the Wind",
         GuardianStar.SATURN, GuardianStar.MERCURY, Type.SPELLCASTER, Attribute.WIND, "97843505"),
    Card(145, 2, 700, 700, 30, "The Bewitching Phantom Thief",
         GuardianStar.VENUS, GuardianStar.PLUTO, Type.SPELLCASTER, Attribute.DARK, "24348204"),
    Card(146, 4, 900, 1300, 70, "Temple of Skulls",
         GuardianStar.MERCURY, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "732302"),
    Card(147, 3, 600, 900, 999999, "Monster Egg",
         GuardianStar.URANUS, GuardianStar.JUPITER, Type.WARRIOR, Attribute.EARTH, "36121917"),
    Card(148, 3, 800, 700, 30, "The Shadow Who Controls the Dark",
         GuardianStar.MERCURY, GuardianStar.JUPITER, Type.FIEND, Attribute.DARK, "63125616"),
    Card(149, 4, 1400, 1200, 120, "Lord of the Lamp",
         GuardianStar.MERCURY, GuardianStar.SATURN, Type.FIEND, Attribute.DARK, "99510761"),
    Card(150, 5, 1700, 1400, 230, "Akihiron",
         GuardianStar.NEPTUNE, GuardianStar.JUPITER, Type.AQUA, Attribute.WATER, "36904469"),
    Card(151, 4, 1200, 1300, 100, "Rhaimundos of the Red Sword",
         GuardianStar.NEPTUNE, GuardianStar.MARS, Type.WARRIOR, Attribute.EARTH, "62403074"),
    Card(152, 2, 500, 700, 25, "The Melting Red Shadow",
         GuardianStar.MOON, GuardianStar.NEPTUNE, Type.AQUA, Attribute.WATER, "98898173"),
    Card(153, 3, 900, 1200, 60, "Dokuroizo the Grim Reaper",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "25882881"),
    Card(154, 2, 700, 500, 25, "Fire Reaper",
         GuardianStar.MARS, GuardianStar.SUN, Type.ZOMBIE, Attribute.DARK, "53581214"),
    Card(155, 3, 800, 1000, 45, "Larvas",
         GuardianStar.JUPITER, GuardianStar.MERCURY, Type.BEAST, Attribute.EARTH, "94675535"),
    Card(156, 3, 300, 1200, 30, "Hard Armor",
         GuardianStar.URANUS, GuardianStar.SUN, Type.WARRIOR, Attribute.EARTH, "20060230"),
    Card(157, 2, 700, 600, 30, "Firegrass",
         GuardianStar.MARS, GuardianStar.MOON, Type.PLANT, Attribute.EARTH, "53293545"),
    Card(158, 2, 800, 600, 30, "Man Eater",
         GuardianStar.JUPITER, GuardianStar.URANUS, Type.PLANT, Attribute.EARTH, "93553943"),
    Card(159, 2, 500, 800, 30, "Dig Beak",
         GuardianStar.JUPITER, GuardianStar.PLUTO, Type.BEAST, Attribute.EARTH, "29948642"),
    Card(160, 3, 1000, 500, 30, "M-warrior #1",
         GuardianStar.URANUS, GuardianStar.MOON, Type.WARRIOR, Attribute.EARTH, "56342351"),
    Card(161, 3, 500, 1000, 30, "M-warrior #2",
         GuardianStar.URANUS, GuardianStar.VENUS, Type.WARRIOR, Attribute.EARTH, "92731455"),
    Card(162, 3, 1250, 800, 55, "Tainted Wisdom",
         GuardianStar.MOON, GuardianStar.MERCURY, Type.FIEND, Attribute.DARK, "28725004"),
    Card(163, 4, 1300, 1300, 120, "Lisark",
         GuardianStar.SATURN, GuardianStar.PLUTO, Type.BEAST, Attribute.EARTH, "55210709"),
    Card(164, 4, 1300, 1000, 80, "Lord of Zemia",
         GuardianStar.MERCURY, GuardianStar.URANUS, Type.FIEND, Attribute.DARK, "81618817"),
    Card(165, 3, 1400, 700, 60, "The Judgement Hand",
         GuardianStar.SUN, GuardianStar.VENUS, Type.WARRIOR, Attribute.EARTH, "28003512"),
    Card(166, 4, 1000, 1500, 100, "Mysterious Puppeteer",
         GuardianStar.URANUS, GuardianStar.MERCURY, Type.WARRIOR, Attribute.EARTH, "54098121"),
    Card(167, 1, 400, 200, 15, "Ancient Jar",
         GuardianStar.URANUS, GuardianStar.NEPTUNE, Type.ROCK, Attribute.EARTH, "81492226"),
    Card(168, 4, 1500, 1250, 140, "Darkfire Dragon",
         GuardianStar.MARS, GuardianStar.VENUS, Type.DRAGON, Attribute.DARK, "17881964"),
    Card(169, 3, 1200, 800, 55, "Dark King of the Abyss",
         GuardianStar.MERCURY, GuardianStar.PLUTO, Type.FIEND, Attribute.DARK, "53375573"),
    Card(170, 4, 800, 2000, 160, "Spirit of the Harp",
         GuardianStar.SUN, GuardianStar.MARS, Type.FAIRY, Attribute.LIGHT, "80770678"),
    Card(171, 4, 1200, 1000, 70, "Big Eye",
         GuardianStar.VENUS, GuardianStar.NEPTUNE, Type.FIEND, Attribute.DARK, "16768387"),
    Card(172, 3, 700, 1300, 55, "Armaill",
         GuardianStar.MARS, GuardianStar.MERCURY, Type.WARRIOR, Attribute.EARTH, "53153481"),
    Card(173, 3, 600, 1000, 35, "Dark Prisoner",
         GuardianStar.MERCURY, GuardianStar.NEPTUNE, Type.FIEND, Attribute.DARK, "89558090"),
    Card(174, 2, 900, 200, 25, "Hurricail",
         GuardianStar.SATURN, GuardianStar.NEPTUNE, Type.SPELLCASTER, Attribute.WIND, "15042735"),
    Card(175, 3, 1000, 700, 40, "Ancient Brain",
         GuardianStar.MERCURY, GuardianStar.MOON, Type.FIEND, Attribute.DARK, "42431843"),
    Card(176, 2, 800, 600, 30, "Fire Eye",
         GuardianStar.MARS, GuardianStar.SATURN, Type.PYRO, Attribute.FIRE, "88435542"),
    Card(177, 3, 800, 1000, 45, "Monsturtle",
         GuardianStar.NEPTUNE, GuardianStar.MERCURY, Type.AQUA, Attribute.WATER, "15820147"),
    Card(178, 3, 1000, 800, 45, "Claw Reacher",
         GuardianStar.MERCURY, GuardianStar.MARS, Type.FIEND, Attribute.DARK, "41218256"),
    Card(179, 2, 700, 600, 30, "Phantom Dewan",
         GuardianStar.MERCURY, GuardianStar.JUPITER, Type.SPELLCASTER, Attribute.DARK, "77603950"),
    Card(180, 3, 800, 1000, 45, "Arlownay",
         GuardianStar.JUPITER, GuardianStar.NEPTUNE, Type.PLANT, Attribute.EARTH, "14708569"),
    Card(181, 3, 1000, 1000, 55, "Dark Shade",
         GuardianStar.SATURN, GuardianStar.MARS, Type.FIEND, Attribute.WIND, "40196604"),
    Card(182, 2, 500, 700, 25, "Masked Clown",
         GuardianStar.MERCURY, GuardianStar.SATURN, Type.WARRIOR, Attribute.DARK, "77581312"),
    Card(183, 2, 600, 800, 30, "Lucky Trinket",
         GuardianStar.SUN, GuardianStar.JUPITER, Type.SPELLCASTER, Attribute.LIGHT, "3985011"),
    Card(184, 3, 600, 900, 30, "Genin",
         GuardianStar.VENUS, GuardianStar.SUN, Type.SPELLCASTER, Attribute.LIGHT, "49370026"),
    Card(185, 2, 600, 500, 25, "Eyearmor",
         GuardianStar.URANUS, GuardianStar.MARS, Type.WARRIOR, Attribute.EARTH, "64511793"),
    Card(186, 4, 1100, 1400, 100, "Fiend Refrection #2",
         GuardianStar.VENUS, GuardianStar.MARS, Type.WINGED_BEAST, Attribute.LIGHT, "2863439"),
    Card(187, 3, 700, 800, 30, "Gate Deeg",
         GuardianStar.MOON, GuardianStar.MARS, Type.BEAST, Attribute.DARK, "49258578"),
    Card(188, 3, 800, 900, 40, "Synchar",
         GuardianStar.JUPITER, GuardianStar.SUN, Type.BEAST, Attribute.EARTH, "75646173"),
    Card(189, 3, 900, 700, 35, "Fusionist",
         GuardianStar.JUPITER, GuardianStar.MOON, Type.BEAST, Attribute.EARTH, "1641882"),
    Card(190, 3, 1000, 800, 45, "Akakieisu",
         GuardianStar.MERCURY, GuardianStar.URANUS, Type.SPELLCASTER, Attribute.DARK, "38035986"),
    Card(191, 2, 600, 600, 25, "LaLa Li-oon",
         GuardianStar.PLUTO, GuardianStar.MOON, Type.THUNDER, Attribute.WIND, "9430387"),
    Card(192, 1, 400, 300, 15, "Key Mace",
         GuardianStar.SUN, GuardianStar.SATURN, Type.FAIRY, Attribute.LIGHT, "1929294"),
    Card(193, 4, 1000, 1500, 100, "Turtle Tiger",
         GuardianStar.NEPTUNE, GuardianStar.SATURN, Type.AQUA, Attribute.WATER, "37313348"),
    Card(194, 4, 1200, 1300, 100, "Terra the Terrible",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.FIEND, Attribute.DARK, "63308047"),
    Card(195, 2, 900, 500, 30, "Doron",
         GuardianStar.NEPTUNE, GuardianStar.MOON, Type.WARRIOR, Attribute.EARTH, "756652"),
    Card(196, 4, 1000, 1200, 70, "Arma Knight",
         GuardianStar.NEPTUNE, GuardianStar.VENUS, Type.AQUA, Attribute.WATER, "36151751"),
    Card(197, 2, 500, 400, 20, "Mech Mole Zombie",
         GuardianStar.MOON, GuardianStar.SATURN, Type.ZOMBIE, Attribute.DARK, "63545455"),
    Card(198, 2, 800, 500, 30, "Happy Lover",
         GuardianStar.SUN, GuardianStar.URANUS, Type.FAIRY, Attribute.LIGHT, "99030164"),
    Card(199, 3, 900, 800, 40, "Penguin Knight",
         GuardianStar.URANUS, GuardianStar.MERCURY, Type.AQUA, Attribute.WATER, "36039163"),
    Card(200, 2, 600, 700, 30, "Petit Dragon",
         GuardianStar.SATURN, GuardianStar.SUN, Type.DRAGON, Attribute.WIND, "75356564"),
    Card(201, 4, 1200, 1000, 70, "Frenzied Panda",
         GuardianStar.JUPITER, GuardianStar.VENUS, Type.BEAST, Attribute.EARTH, "98818516"),
    Card(202, 2, 400, 600, 20, "Air Marmot of Nefariousness",
         GuardianStar.MOON, GuardianStar.URANUS, Type.BEAST, Attribute.EARTH, "75889523"),
    Card(203, 2, 600, 800, 30, "Phantom Ghost",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "61201220"),
    Card(204, 5, 1700, 1400, 230, "Mabarrel",
         GuardianStar.MOON, GuardianStar.NEPTUNE, Type.FIEND, Attribute.DARK, "98795934"),
    Card(205, 3, 900, 800, 40, "Dorover",
         GuardianStar.NEPTUNE, GuardianStar.SUN, Type.AQUA, Attribute.WATER, "24194033"),
    Card(206, 3, 900, 700, 35, "Twin Long Rods #1",
         GuardianStar.MOON, GuardianStar.MERCURY, Type.AQUA, Attribute.WATER, "60589682"),
    Card(207, 2, 600, 500, 25, "Droll Bird",
         GuardianStar.SATURN, GuardianStar.MOON, Type.WINGED_BEAST, Attribute.WIND, "97973387"),
    Card(208, 3, 600, 900, 30, "Petit Angel",
         GuardianStar.SUN, GuardianStar.PLUTO, Type.FAIRY, Attribute.LIGHT, "38142739"),
    Card(209, 2, 700, 700, 30, "Winged Cleaver",
         GuardianStar.JUPITER, GuardianStar.MERCURY, Type.INSECT, Attribute.EARTH, "39175982"),
    Card(210, 2, 600, 500, 25, "Hinotama Soul",
         GuardianStar.MARS, GuardianStar.URANUS, Type.PYRO, Attribute.FIRE, "96851799"),
    Card(211, 2, 700, 600, 30, "Kaminarikozou",
         GuardianStar.PLUTO, GuardianStar.VENUS, Type.THUNDER, Attribute.WIND, "15510988"),
    Card(212, 2, 700, 600, 30, "Meotoko",
         GuardianStar.MOON, GuardianStar.MARS, Type.BEAST, Attribute.EARTH, "53832650"),
    Card(213, 4, 1200, 2000, 260, "Aqua Madoor",
         GuardianStar.NEPTUNE, GuardianStar.MERCURY, Type.SPELLCASTER, Attribute.WATER, "85639257"),
    Card(214, 2, 800, 400, 25, "Kagemusha of the Blue Flame",
         GuardianStar.MARS, GuardianStar.SUN, Type.WARRIOR, Attribute.EARTH, "15401633"),
    Card(215, 3, 1000, 800, 45, "Flame Ghost",
         GuardianStar.MARS, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "58528964"),
    Card(216, 4, 1200, 1400, 120, "Dryad",
         GuardianStar.JUPITER, GuardianStar.URANUS, Type.SPELLCASTER, Attribute.EARTH, "84916669"),
    Card(217, 9, 3200, 2500, 999999, "B. Skull Dragon",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.DRAGON, Attribute.DARK, "11901678"),
    Card(218, 3, 900, 700, 35, "Two-mouth Darkruler",
         GuardianStar.MOON, GuardianStar.SATURN, Type.DINOSAUR, Attribute.EARTH, "57305373"),
    Card(219, 3, 1050, 1000, 55, "Solitude",
         GuardianStar.MOON, GuardianStar.URANUS, Type.BEAST_WARRIOR, Attribute.EARTH, "84794011"),
    Card(220, 4, 900, 1400, 80, "Masked Sorcerer",
         GuardianStar.MERCURY, GuardianStar.NEPTUNE, Type.SPELLCASTER, Attribute.DARK, "10189126"),
    Card(221, 3, 700, 1400, 60, "Kumootoko",
         GuardianStar.JUPITER, GuardianStar.PLUTO, Type.INSECT, Attribute.EARTH, "56283725"),
    Card(222, 2, 800, 600, 30, "Midnight Fiend",
         GuardianStar.MOON, GuardianStar.NEPTUNE, Type.FIEND, Attribute.DARK, "83678433"),
    Card(223, 6, 2100, 1800, 550, "Roaring Ocean Snake",
         GuardianStar.NEPTUNE, GuardianStar.MARS, Type.AQUA, Attribute.WATER, "19066538"),
    Card(224, 3, 500, 1100, 35, "Trap Master",
         GuardianStar.MERCURY, GuardianStar.MOON, Type.WARRIOR, Attribute.EARTH, "46461247"),
    Card(225, 4, 1400, 800, 70, "Fiend Sword",
         GuardianStar.MOON, GuardianStar.MERCURY, Type.WARRIOR, Attribute.DARK, "22855882"),
    Card(226, 3, 900, 800, 40, "Skull Stalker",
         GuardianStar.MOON, GuardianStar.MARS, Type.WARRIOR, Attribute.DARK, "54844990"),
    Card(227, 2, 600, 700, 30, "Hitodenchak",
         GuardianStar.NEPTUNE, GuardianStar.MARS, Type.AQUA, Attribute.WATER, "46718686"),
    Card(228, 3, 1000, 900, 50, "Wood Remains",
         GuardianStar.JUPITER, GuardianStar.NEPTUNE, Type.ZOMBIE, Attribute.DARK, "17733394"),
    Card(229, 2, 700, 600, 30, "Hourglass of Life",
         GuardianStar.SUN, GuardianStar.NEPTUNE, Type.FAIRY, Attribute.LIGHT, "8783685"),
    Card(230, 4, 1500, 1200, 140, "Rare Fish",
         GuardianStar.NEPTUNE, GuardianStar.JUPITER, Type.FISH, Attribute.WATER, "80516007"),
    Card(231, 3, 800, 1200, 55, "Wood Clown",
         GuardianStar.MERCURY, GuardianStar.MARS, Type.WARRIOR, Attribute.EARTH, "17511156"),
    Card(232, 2, 600, 800, 30, "Madjinn Gunn",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.FIEND, Attribute.DARK, "43905751"),
    Card(233, 4, 1300, 1100, 120, "Dark Titan of Terror",
         GuardianStar.MERCURY, GuardianStar.JUPITER, Type.FIEND, Attribute.DARK, "89494469"),
    Card(234, 4, 1600, 800, 120, "Beautiful Headhuntress",
         GuardianStar.MERCURY, GuardianStar.SATURN, Type.WARRIOR, Attribute.EARTH, "16899564"),
    Card(235, 3, 900, 1200, 999999, "Wodan the Resident of the Forest",
         GuardianStar.JUPITER, GuardianStar.SUN, Type.WARRIOR, Attribute.EARTH, "42883273"),
    Card(236, 4, 1000, 1200, 70, "Guardian of the Labyrinth",
         GuardianStar.PLUTO, GuardianStar.MERCURY, Type.WARRIOR, Attribute.EARTH, "89272878"),
    Card(237, 2, 500, 500, 20, "Haniwa",
         GuardianStar.URANUS, GuardianStar.JUPITER, Type.ROCK, Attribute.EARTH, "84285623"),
    Card(238, 2, 800, 600, 30, "Yashinoki",
         GuardianStar.JUPITER, GuardianStar.MOON, Type.PLANT, Attribute.EARTH, "41061625"),
    Card(239, 3, 900, 700, 35, "Vishwar Randi",
         GuardianStar.MERCURY, GuardianStar.URANUS, Type.WARRIOR, Attribute.DARK, "78556320"),
    Card(240, 3, 700, 800, 30, "The Drdek",
         GuardianStar.MOON, GuardianStar.SATURN, Type.FIEND, Attribute.DARK, "8944575"),
    Card(241, 4, 1200, 1200, 120, "Dark Assailant",
         GuardianStar.MOON, GuardianStar.URANUS, Type.ZOMBIE, Attribute.DARK, "41949033"),
    Card(242, 2, 600, 600, 25, "Candle of Fate",
         GuardianStar.MARS, GuardianStar.MOON, Type.FIEND, Attribute.DARK, "47695416"),
    Card(243, 3, 900, 700, 35, "Water Element",
         GuardianStar.NEPTUNE, GuardianStar.SATURN, Type.AQUA, Attribute.WATER, "3732747"),
    Card(244, 3, 900, 1000, 50, "Dissolverock",
         GuardianStar.MARS, GuardianStar.VENUS, Type.ROCK, Attribute.EARTH, "40826495"),
    Card(245, 2, 800, 400, 25, "Meda Bat",
         GuardianStar.MOON, GuardianStar.NEPTUNE, Type.FIEND, Attribute.DARK, "76211194"),
    Card(246, 4, 1100, 1000, 60, "One Who Hunts Souls",
         GuardianStar.MOON, GuardianStar.MERCURY, Type.BEAST_WARRIOR, Attribute.EARTH, "3606209"),
    Card(247, 3, 900, 800, 40, "Root Water",
         GuardianStar.NEPTUNE, GuardianStar.SUN, Type.FISH, Attribute.WATER, "39004808"),
    Card(248, 4, 1200, 1000, 70, "Master & Expert",
         GuardianStar.JUPITER, GuardianStar.VENUS, Type.BEAST, Attribute.EARTH, "75499502"),
    Card(249, 4, 1400, 1200, 120, "Water Omotics",
         GuardianStar.NEPTUNE, GuardianStar.MOON, Type.AQUA, Attribute.WATER, "2483611"),
    Card(250, 3, 800, 1200, 55, "Hyo",
         GuardianStar.NEPTUNE, GuardianStar.MERCURY, Type.WARRIOR, Attribute.WATER, "38982356"),
    Card(251, 3, 1200, 900, 60, "Enchanting Mermaid",
         GuardianStar.NEPTUNE, GuardianStar.VENUS, Type.FISH, Attribute.WATER, "75376965"),
    Card(252, 3, 1100, 900, 999999, "Nekogal #1",
         GuardianStar.JUPITER, GuardianStar.MERCURY, Type.BEAST, Attribute.EARTH, "1761063"),
    Card(253, 3, 800, 1000, 45, "Angelwitch",
         GuardianStar.MERCURY, GuardianStar.NEPTUNE, Type.SPELLCASTER, Attribute.DARK, "37160778"),
    Card(254, 2, 500, 750, 25, "Embryonic Beast",
         GuardianStar.MOON, GuardianStar.MARS, Type.FIEND, Attribute.DARK, "64154377"),
    Card(255, 4, 500, 2000, 100, "Prevent Rat",
         GuardianStar.JUPITER, GuardianStar.URANUS, Type.BEAST, Attribute.EARTH, "549481"),
    Card(256, 4, 1200, 1000, 70, "Dimensional Warrior",
         GuardianStar.VENUS, GuardianStar.MARS, Type.WARRIOR, Attribute.EARTH, "37043180"),
    Card(257, 3, 800, 1200, 55, "Stone Armadiller",
         GuardianStar.URANUS, GuardianStar.SUN, Type.ROCK, Attribute.EARTH, "63432835"),
    Card(258, 4, 1000, 1100, 60, "Beastking of the Swamps",
         GuardianStar.URANUS, GuardianStar.MERCURY, Type.AQUA, Attribute.WATER, "99426834"),
    Card(259, 4, 1000, 1300, 80, "Ancient Sorcerer",
         GuardianStar.MERCURY, GuardianStar.MOON, Type.SPELLCASTER, Attribute.DARK, "36821538"),
    Card(260, 3, 750, 1100, 45, "Lunar Queen Elzaim",
         GuardianStar.MOON, GuardianStar.VENUS, Type.FAIRY, Attribute.LIGHT, "62210247"),
    Card(261, 2, 700, 600, 30, "Wicked Mirror",
         GuardianStar.VENUS, GuardianStar.JUPITER, Type.FIEND, Attribute.DARK, "15150371"),
    Card(262, 3, 800, 1300, 60, "The Little Swordsman of Aile",
         GuardianStar.NEPTUNE, GuardianStar.MARS, Type.WARRIOR, Attribute.WATER, "25109950"),
    Card(263, 3, 700, 1400, 60, "Rock Ogre Grotto #2",
         GuardianStar.URANUS, GuardianStar.SUN, Type.ROCK, Attribute.EARTH, "62193699"),
    Card(264, 3, 500, 1300, 45, "Wing Egg Elf",
         GuardianStar.SUN, GuardianStar.MARS, Type.FAIRY, Attribute.LIGHT, "98582704"),
    Card(265, 3, 800, 700, 30, "The Furious Sea King",
         GuardianStar.NEPTUNE, GuardianStar.JUPITER, Type.AQUA, Attribute.WATER, "18710707"),
    Card(266, 3, 900, 700, 35, "Princess of Tsurugi",
         GuardianStar.SATURN, GuardianStar.VENUS, Type.WARRIOR, Attribute.WIND, "51371017"),
    Card(267, 3, 1000, 500, 30, "Unknown Warrior of Fiend",
         GuardianStar.MERCURY, GuardianStar.MARS, Type.WARRIOR, Attribute.DARK, "97360116"),
    Card(268, 2, 700, 500, 25, "Sectarian of Secrets",
         GuardianStar.MERCURY, GuardianStar.JUPITER, Type.SPELLCASTER, Attribute.DARK, "15507080"),
    Card(269, 3, 1100, 900, 55, "Versago the Destroyer",
         GuardianStar.MERCURY, GuardianStar.SATURN, Type.FIEND, Attribute.DARK, "50259460"),
    Card(270, 3, 1000, 900, 50, "Wetha",
         GuardianStar.NEPTUNE, GuardianStar.SATURN, Type.AQUA, Attribute.WATER, "96643568"),
    Card(271, 3, 900, 600, 30, "Megirus Light",
         GuardianStar.MERCURY, GuardianStar.URANUS, Type.FIEND, Attribute.DARK, "23032273"),
    Card(272, 4, 1300, 900, 70, "Mavelus",
         GuardianStar.MARS, GuardianStar.MERCURY, Type.WINGED_BEAST, Attribute.WIND, "59036972"),
    Card(273, 3, 600, 1500, 60, "Ancient Tree of Enlightenment",
         GuardianStar.JUPITER, GuardianStar.PLUTO, Type.PLANT, Attribute.EARTH, "86421986"),
    Card(274, 3, 500, 1600, 60, "Green Phantom King",
         GuardianStar.JUPITER, GuardianStar.NEPTUNE, Type.PLANT, Attribute.EARTH, "22910685"),
    Card(275, 4, 1500, 1000, 100, "Ground Attacker Bugroth",
         GuardianStar.URANUS, GuardianStar.MOON, Type.MACHINE, Attribute.EARTH, "58314394"),
    Card(276, 3, 1000, 1000, 55, "Ray & Temperature",
         GuardianStar.SUN, GuardianStar.SATURN, Type.FAIRY, Attribute.LIGHT, "85309439"),
    Card(277, 3, 300, 1300, 35, "Gorgon Egg",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.FIEND, Attribute.DARK, "11793047"),
    Card(278, 1, 300, 200, 10, "Petit Moth",
         GuardianStar.JUPITER, GuardianStar.SUN, Type.INSECT, Attribute.EARTH, "58192742"),
    Card(279, 3, 1000, 900, 50, "King Fog",
         GuardianStar.MERCURY, GuardianStar.NEPTUNE, Type.FIEND, Attribute.DARK, "84686841"),
    Card(280, 4, 800, 1500, 80, "Protector of the Throne",
         GuardianStar.URANUS, GuardianStar.VENUS, Type.WARRIOR, Attribute.EARTH, "10071456"),
    Card(281, 4, 1500, 1000, 100, "Mystic Clown",
         GuardianStar.MERCURY, GuardianStar.MOON, Type.FIEND, Attribute.DARK, "47060154"),
    Card(282, 3, 800, 1000, 45, "Mystical Sheep #2",
         GuardianStar.VENUS, GuardianStar.SATURN, Type.BEAST, Attribute.EARTH, "83464209"),
    Card(283, 3, 1100, 700, 45, "Holograh",
         GuardianStar.SUN, GuardianStar.JUPITER, Type.MACHINE, Attribute.EARTH, "10859908"),
    Card(284, 3, 1200, 900, 999999, "Tao the Chanter",
         GuardianStar.VENUS, GuardianStar.URANUS, Type.SPELLCASTER, Attribute.EARTH, "46247516"),
    Card(285, 2, 700, 600, 30, "Serpent Marauder",
         GuardianStar.NEPTUNE, GuardianStar.SUN, Type.REPTILE, Attribute.EARTH, "82742611"),
    Card(286, 5, 1500, 1800, 290, "Gatekeeper",
         GuardianStar.MERCURY, GuardianStar.MARS, Type.MACHINE, Attribute.DARK, "19737320"),
    Card(287, 4, 1200, 1400, 120, "Ogre of the Black Shadow",
         GuardianStar.MOON, GuardianStar.SATURN, Type.BEAST_WARRIOR, Attribute.EARTH, "45121025"),
    Card(288, 3, 600, 1400, 999999, "Dark Artist",
         GuardianStar.MERCURY, GuardianStar.JUPITER, Type.FIEND, Attribute.DARK, "72520073"),
    Card(289, 1, 400, 300, 15, "Change Slime",
         GuardianStar.NEPTUNE, GuardianStar.MOON, Type.AQUA, Attribute.WATER, "18914778"),
    Card(290, 4, 1100, 1000, 60, "Moon Envoy",
         GuardianStar.MOON, GuardianStar.SATURN, Type.WARRIOR, Attribute.LIGHT, "45909477"),
    Card(291, 4, 1300, 1000, 80, "Fireyarou",
         GuardianStar.MARS, GuardianStar.SATURN, Type.PYRO, Attribute.FIRE, "71407486"),
    Card(292, 2, 400, 1000, 30, "Psychic Kappa",
         GuardianStar.NEPTUNE, GuardianStar.VENUS, Type.AQUA, Attribute.WATER, "7892180"),
    Card(293, 4, 1100, 1100, 70, "Masaki the Legendary Swordsman",
         GuardianStar.SUN, GuardianStar.URANUS, Type.WARRIOR, Attribute.EARTH, "44287299"),
    Card(294, 3, 1200, 900, 60, "Dragoness the Wicked Knight",
         GuardianStar.SATURN, GuardianStar.MERCURY, Type.WARRIOR, Attribute.WIND, "70681994"),
    Card(295, 3, 600, 1300, 50, "Bio Plant",
         GuardianStar.JUPITER, GuardianStar.MOON, Type.FIEND, Attribute.DARK, "7670542"),
    Card(296, 3, 700, 1300, 55, "One-eyed Shield Dragon",
         GuardianStar.SATURN, GuardianStar.PLUTO, Type.DRAGON, Attribute.WIND, "33064647"),
    Card(297, 4, 1400, 1200, 120, "Cyber Soldier of Darkworld",
         GuardianStar.MOON, GuardianStar.URANUS, Type.MACHINE, Attribute.DARK, "75559356"),
    Card(298, 3, 900, 900, 45, "Wicked Dragon with the Ersatz Head",
         GuardianStar.MOON, GuardianStar.NEPTUNE, Type.DRAGON, Attribute.WIND, "2957055"),
    Card(299, 3, 1200, 900, 999999, "Sonic Maid",
         GuardianStar.SUN, GuardianStar.PLUTO, Type.WARRIOR, Attribute.EARTH, "38942059"),
    Card(300, 3, 800, 800, 35, "Kurama",
         GuardianStar.SATURN, GuardianStar.NEPTUNE, Type.WINGED_BEAST, Attribute.WIND, "85705804"),
    Card(301, 0, 0, 0, 800, "Legendary Sword",
         None, None, Type.EQUIP, None, "61854111"),
    Card(302, 0, 0, 0, 800, "Sword of Dark Destruction",
         None, None, Type.EQUIP, None, "37120512"),
    Card(303, 0, 0, 0, 800, "Dark Energy",
         None, None, Type.EQUIP, None, "4614116"),
    Card(304, 0, 0, 0, 800, "Axe of Despair",
         None, None, Type.EQUIP, None, "40619825"),
    Card(305, 0, 0, 0, 800, "Laser Cannon Armor",
         None, None, Type.EQUIP, None, "77007920"),
    Card(306, 0, 0, 0, 800, "Insect Armor with Laser Cannon",
         None, None, Type.EQUIP, None, "3492538"),
    Card(307, 0, 0, 0, 800, "Elf's Light",
         None, None, Type.EQUIP, None, "39897277"),
    Card(308, 0, 0, 0, 800, "Beast Fangs",
         None, None, Type.EQUIP, None, "46009906"),
    Card(309, 0, 0, 0, 800, "Steel Shell",
         None, None, Type.EQUIP, None, "2370081"),
    Card(310, 0, 0, 0, 800, "Vile Germs",
         None, None, Type.EQUIP, None, "39774685"),
    Card(311, 0, 0, 0, 800, "Black Pendant",
         None, None, Type.EQUIP, None, "65169794"),
    Card(312, 0, 0, 0, 800, "Silver Bow and Arrow",
         None, None, Type.EQUIP, None, "1557499"),
    Card(313, 0, 0, 0, 800, "Horn of Light",
         None, None, Type.EQUIP, None, "38552107"),
    Card(314, 0, 0, 0, 800, "Horn of the Unicorn",
         None, None, Type.EQUIP, None, "64047146"),
    Card(315, 0, 0, 0, 800, "Dragon Treasure",
         None, None, Type.EQUIP, None, "1435851"),
    Card(316, 0, 0, 0, 800, "Electro-whip",
         None, None, Type.EQUIP, None, "37820550"),
    Card(317, 0, 0, 0, 800, "Cyber Shield",
         None, None, Type.EQUIP, None, "63224564"),
    Card(318, 0, 0, 0, 999999, "Elegant Egotist",
         None, None, Type.EQUIP, None, "90219263"),
    Card(319, 0, 0, 0, 800, "Mystical Moon",
         None, None, Type.EQUIP, None, "36607978"),
    Card(320, 0, 0, 0, 800, "Stop Defense",
         None, None, Type.MAGIC, None, "63102017"),
    Card(321, 0, 0, 0, 800, "Malevolent Nuzzler",
         None, None, Type.EQUIP, None, "99597615"),
    Card(322, 0, 0, 0, 800, "Violet Crystal",
         None, None, Type.EQUIP, None, "15052462"),
    Card(323, 0, 0, 0, 800, "Book of Secret Arts",
         None, None, Type.EQUIP, None, "91595718"),
    Card(324, 0, 0, 0, 800, "Invigoration",
         None, None, Type.EQUIP, None, "98374133"),
    Card(325, 0, 0, 0, 999999, "Machine Conversion Factory",
         None, None, Type.EQUIP, None, "25769732"),
    Card(326, 0, 0, 0, 800, "Raise Body Heat",
         None, None, Type.EQUIP, None, "51267887"),
    Card(327, 0, 0, 0, 800, "Follow Wind",
         None, None, Type.EQUIP, None, "98252586"),
    Card(328, 0, 0, 0, 800, "Power of Kaishin",
         None, None, Type.EQUIP, None, "77027445"),
    Card(329, 0, 0, 0, 999999, "Dragon Capture Jar",
         None, None, Type.MAGIC, None, "50045299"),
    Card(330, 0, 0, 0, 55, "Forest",
         None, None, Type.MAGIC, None, "87430998"),
    Card(331, 0, 0, 0, 55, "Wasteland",
         None, None, Type.MAGIC, None, "23424603"),
    Card(332, 0, 0, 0, 55, "Mountain",
         None, None, Type.MAGIC, None, "50913601"),
    Card(333, 0, 0, 0, 55, "Sogen",
         None, None, Type.MAGIC, None, "86318356"),
    Card(334, 0, 0, 0, 55, "Umi",
         None, None, Type.MAGIC, None, "22702055"),
    Card(335, 0, 0, 0, 55, "Yami",
         None, None, Type.MAGIC, None, "59197169"),
    Card(336, 0, 0, 0, 20000, "Dark Hole",
         None, None, Type.MAGIC, None, "53129443"),
    Card(337, 0, 0, 0, 999999, "Raigeki",
         None, None, Type.MAGIC, None, "12580477"),
    Card(338, 0, 0, 0, 999999, "Mooyan Curry",
         None, None, Type.MAGIC, None, "58074572"),
    Card(339, 0, 0, 0, 55, "Red Medicine",
         None, None, Type.MAGIC, None, "38199696"),
    Card(340, 0, 0, 0, 20000, "Goblin's Secret Remedy",
         None, None, Type.MAGIC, None, "11868825"),
    Card(341, 0, 0, 0, 35000, "Soul of the Pure",
         None, None, Type.MAGIC, None, "47852924"),
    Card(342, 0, 0, 0, 999999, "Dian Keto the Cure Master",
         None, None, Type.MAGIC, None, "84257639"),
    Card(343, 0, 0, 0, 10, "Sparks",
         None, None, Type.MAGIC, None, "76103675"),
    Card(344, 0, 0, 0, 55, "Hinotama",
         None, None, Type.MAGIC, None, "46130346"),
    Card(345, 0, 0, 0, 20000, "Final Flame",
         None, None, Type.MAGIC, None, "73134081"),
    Card(346, 0, 0, 0, 35000, "Ookazi",
         None, None, Type.MAGIC, None, "19523799"),
    Card(347, 0, 0, 0, 999999, "Tremendous Fire",
         None, None, Type.MAGIC, None, "46918794"),
    Card(348, 0, 0, 0, 999999, "Swords of Revealing Light",
         None, None, Type.MAGIC, None, "72302403"),
    Card(349, 0, 0, 0, 999999, "Spellbinding Circle",
         None, None, Type.MAGIC, None, "18807108"),
    Card(350, 0, 0, 0, 999999, "Dark-piercing Light",
         None, None, Type.MAGIC, None, "45895206"),
    Card(351, 4, 1300, 1500, 160, "Yaranzo",
         GuardianStar.MOON, GuardianStar.MERCURY, Type.ZOMBIE, Attribute.DARK, "71280811"),
    Card(352, 4, 1400, 1400, 160, "Kanan the Swordmistress",
         GuardianStar.URANUS, GuardianStar.NEPTUNE, Type.WARRIOR, Attribute.EARTH, "12829151"),
    Card(353, 4, 1500, 1200, 140, "Takriminos",
         GuardianStar.NEPTUNE, GuardianStar.MERCURY, Type.SEA_SERPENT, Attribute.WATER, "44073668"),
    Card(354, 3, 1200, 900, 60, "Stuffed Animal",
         GuardianStar.MOON, GuardianStar.MARS, Type.WARRIOR, Attribute.EARTH, "71068263"),
    Card(355, 5, 1500, 1800, 290, "Megasonic Eye",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.MACHINE, Attribute.DARK, "7562372"),
    Card(356, 7, 2300, 2100, 0, "Super War-lion",
         GuardianStar.JUPITER, GuardianStar.VENUS, Type.BEAST, Attribute.EARTH, None),
    Card(357, 5, 1600, 1800, 320, "Yamadron",
         GuardianStar.MARS, GuardianStar.URANUS, Type.DRAGON, Attribute.FIRE, "70345785"),
    Card(358, 7, 2500, 2300, 1700, "Seiyaryu",
         GuardianStar.SUN, GuardianStar.NEPTUNE, Type.DRAGON, Attribute.LIGHT, "6740720"),
    Card(359, 3, 1100, 800, 50, "Three-legged Zombies",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "33734439"),
    Card(360, 8, 2800, 2300, 0, "Zera The Mant",
         GuardianStar.MOON, GuardianStar.SATURN, Type.FIEND, Attribute.DARK, None),
    Card(361, 4, 1200, 1000, 70, "Flying Penguin",
         GuardianStar.NEPTUNE, GuardianStar.JUPITER, Type.AQUA, Attribute.WATER, "5628232"),
    Card(362, 5, 0, 3000, 200, "Millennium Shield",
         GuardianStar.URANUS, GuardianStar.MARS, Type.WARRIOR, Attribute.EARTH, "32012841"),
    Card(363, 4, 1400, 1000, 120, "Fairy's Gift",
         GuardianStar.JUPITER, GuardianStar.MERCURY, Type.SPELLCASTER, Attribute.LIGHT, "68401546"),
    Card(364, 8, 3000, 2500, 0, "Black Luster Soldier",
         GuardianStar.MOON, GuardianStar.URANUS, Type.WARRIOR, Attribute.DARK, None),
    Card(365, 6, 2100, 1800, 0, "Fiend's Mirror",
         GuardianStar.VENUS, GuardianStar.PLUTO, Type.FIEND, Attribute.DARK, None),
    Card(366, 5, 0, 3000, 200, "Labyrinth Wall",
         GuardianStar.URANUS, GuardianStar.JUPITER, Type.ROCK, Attribute.EARTH, "67284908"),
    Card(367, 4, 2200, 100, 80, "Jirai Gumo",
         GuardianStar.JUPITER, GuardianStar.URANUS, Type.INSECT, Attribute.EARTH, "94773007"),
    Card(368, 5, 1600, 1300, 180, "Shadow Ghoul",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "30778711"),
    Card(369, 7, 1600, 3000, 999999, "Wall Shadow",
         GuardianStar.MOON, GuardianStar.NEPTUNE, Type.WARRIOR, Attribute.DARK, "63162310"),
    Card(370, 7, 2400, 2400, 999999, "Labyrinth Tank",
         GuardianStar.MOON, GuardianStar.MERCURY, Type.MACHINE, Attribute.DARK, "99551425"),
    Card(371, 7, 2600, 2200, 999999, "Sanga of the Thunder",
         GuardianStar.PLUTO, GuardianStar.MARS, Type.THUNDER, Attribute.LIGHT, "25955164"),
    Card(372, 7, 2400, 2200, 999999, "Kazejin",
         GuardianStar.SATURN, GuardianStar.MARS, Type.SPELLCASTER, Attribute.WIND, "62340868"),
    Card(373, 7, 2500, 2400, 999999, "Suijin",
         GuardianStar.NEPTUNE, GuardianStar.SATURN, Type.AQUA, Attribute.WATER, "98434877"),
    Card(374, 11, 3750, 3400, 0, "Gate Guardian",
         GuardianStar.MOON, GuardianStar.MARS, Type.WARRIOR, Attribute.DARK, None),
    Card(375, 5, 1800, 1500, 290, "Dungeon Worm",
         GuardianStar.URANUS, GuardianStar.MOON, Type.INSECT, Attribute.EARTH, "51228280"),
    Card(376, 5, 1800, 1600, 320, "Monster Tamer",
         GuardianStar.JUPITER, GuardianStar.PLUTO, Type.WARRIOR, Attribute.EARTH, "97612389"),
    Card(377, 4, 1600, 1200, 160, "Ryu-kishin Powered",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.FIEND, Attribute.DARK, "24611934"),
    Card(378, 6, 2000, 1600, 400, "Swordstalker",
         GuardianStar.MOON, GuardianStar.SATURN, Type.WARRIOR, Attribute.DARK, "50005633"),
    Card(379, 4, 1800, 1000, 160, "La Jinn the Mystical Genie",
         GuardianStar.MERCURY, GuardianStar.SATURN, Type.FIEND, Attribute.DARK, "97590747"),
    Card(380, 12, 4500, 3800, 0, "Blue-eyes Ultimate Dragon",
         GuardianStar.SUN, GuardianStar.VENUS, Type.DRAGON, Attribute.LIGHT, None),
    Card(381, 4, 800, 1600, 120, "Toon Alligator",
         GuardianStar.NEPTUNE, GuardianStar.MARS, Type.REPTILE, Attribute.WATER, "59383041"),
    Card(382, 5, 1800, 1600, 320, "Rude Kaiser",
         GuardianStar.URANUS, GuardianStar.SUN, Type.BEAST_WARRIOR, Attribute.EARTH, "26378150"),
    Card(383, 5, 2000, 1300, 290, "Parrot Dragon",
         GuardianStar.SATURN, GuardianStar.SUN, Type.DRAGON, Attribute.WIND, "62762898"),
    Card(384, 4, 1100, 1500, 120, "Dark Rabbit",
         GuardianStar.MOON, GuardianStar.URANUS, Type.BEAST, Attribute.DARK, "99261403"),
    Card(385, 7, 2300, 2000, 900, "Bickuribox",
         GuardianStar.MOON, GuardianStar.NEPTUNE, Type.FIEND, Attribute.DARK, "25655502"),
    Card(386, 7, 2000, 2500, 999999, "Harpie's Pet Dragon",
         GuardianStar.SATURN, GuardianStar.MOON, Type.DRAGON, Attribute.WIND, "52040216"),
    Card(387, 1, 400, 300, 15, "Mystic Lamp",
         GuardianStar.MERCURY, GuardianStar.URANUS, Type.SPELLCASTER, Attribute.DARK, "98049915"),
    Card(388, 6, 1750, 2000, 1000, "Pendulum Machine",
         GuardianStar.MERCURY, GuardianStar.NEPTUNE, Type.MACHINE, Attribute.DARK, "24433920"),
    Card(389, 5, 1850, 1500, 290, "Giltia the D. Knight",
         GuardianStar.SUN, GuardianStar.MARS, Type.WARRIOR, Attribute.LIGHT, "51828629"),
    Card(390, 7, 2200, 2500, 999999, "Launcher Spider",
         GuardianStar.MARS, GuardianStar.PLUTO, Type.MACHINE, Attribute.FIRE, "87322377"),
    Card(391, 7, 2600, 1900, 999999, "Zoa",
         GuardianStar.MOON, GuardianStar.MERCURY, Type.FIEND, Attribute.DARK, "24311372"),
    Card(392, 8, 3000, 2300, 999999, "Metalzoa",
         GuardianStar.MOON, GuardianStar.MARS, Type.MACHINE, Attribute.DARK, "50705071"),
    Card(393, 1, 250, 200, 10, "Zone Eater",
         GuardianStar.NEPTUNE, GuardianStar.SUN, Type.AQUA, Attribute.WATER, "86100785"),
    Card(394, 1, 250, 300, 10, "Steel Scorpion",
         GuardianStar.JUPITER, GuardianStar.NEPTUNE, Type.MACHINE, Attribute.EARTH, "13599884"),
    Card(395, 1, 300, 200, 10, "Dancing Elf",
         GuardianStar.JUPITER, GuardianStar.SUN, Type.FAIRY, Attribute.WIND, "59983499"),
    Card(396, 5, 1550, 1650, 260, "Ocubeam",
         GuardianStar.SATURN, GuardianStar.VENUS, Type.FAIRY, Attribute.LIGHT, "86088138"),
    Card(397, 1, 300, 350, 15, "Leghul",
         GuardianStar.JUPITER, GuardianStar.MOON, Type.INSECT, Attribute.EARTH, "12472242"),
    Card(398, 1, 300, 250, 10, "Ooguchi",
         GuardianStar.NEPTUNE, GuardianStar.MOON, Type.AQUA, Attribute.WATER, "58861941"),
    Card(399, 1, 250, 250, 10, "Swordsman from a Foreign Land",
         GuardianStar.URANUS, GuardianStar.VENUS, Type.WARRIOR, Attribute.EARTH, "85255550"),
    Card(400, 5, 1800, 1500, 290, "Emperor of the Land and Sea",
         GuardianStar.NEPTUNE, GuardianStar.VENUS, Type.REPTILE, Attribute.WATER, "11250655"),
    Card(401, 6, 2150, 1950, 700, "Ushi Oni",
         GuardianStar.JUPITER, GuardianStar.VENUS, Type.FIEND, Attribute.DARK, "48649353"),
    Card(402, 1, 250, 350, 15, "Monster Eye",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.FIEND, Attribute.DARK, "84133008"),
    Card(403, 5, 1750, 1550, 290, "Leogun",
         GuardianStar.URANUS, GuardianStar.MERCURY, Type.BEAST, Attribute.EARTH, "10538007"),
    Card(404, 5, 1350, 1600, 180, "Tatsunootoshigo",
         GuardianStar.NEPTUNE, GuardianStar.MERCURY, Type.BEAST, Attribute.EARTH, "47922711"),
    Card(405, 5, 1450, 1500, 180, "Saber Slasher",
         GuardianStar.MOON, GuardianStar.SATURN, Type.MACHINE, Attribute.DARK, "73911410"),
    Card(406, 4, 1000, 1300, 80, "Yaiba Robo",
         GuardianStar.MOON, GuardianStar.URANUS, Type.MACHINE, Attribute.DARK, "10315429"),
    Card(407, 6, 2200, 2000, 800, "Machine King",
         GuardianStar.MARS, GuardianStar.SUN, Type.MACHINE, Attribute.EARTH, "46700124"),
    Card(408, 6, 1750, 1900, 400, "Giant Mech-soldier",
         GuardianStar.MOON, GuardianStar.NEPTUNE, Type.MACHINE, Attribute.EARTH, "72299832"),
    Card(409, 6, 1850, 1700, 350, "Metal Dragon",
         GuardianStar.MARS, GuardianStar.MOON, Type.MACHINE, Attribute.WIND, "9293977"),
    Card(410, 2, 400, 500, 20, "Mechanical Spider",
         GuardianStar.JUPITER, GuardianStar.MERCURY, Type.MACHINE, Attribute.EARTH, "45688586"),
    Card(411, 1, 300, 350, 15, "Bat",
         GuardianStar.MOON, GuardianStar.MERCURY, Type.MACHINE, Attribute.WIND, "72076281"),
    Card(412, 4, 1200, 1400, 120, "Giga-tech Wolf",
         GuardianStar.JUPITER, GuardianStar.URANUS, Type.MACHINE, Attribute.EARTH, "8471389"),
    Card(413, 5, 1500, 1700, 260, "Cyber Soldier",
         GuardianStar.MOON, GuardianStar.MARS, Type.MACHINE, Attribute.DARK, "44865098"),
    Card(414, 3, 900, 1200, 60, "Shovel Crusher",
         GuardianStar.URANUS, GuardianStar.NEPTUNE, Type.MACHINE, Attribute.EARTH, "71950093"),
    Card(415, 4, 1850, 800, 120, "Mechanicalchacer",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.MACHINE, Attribute.DARK, "7359741"),
    Card(416, 4, 850, 1800, 120, "Blocker",
         GuardianStar.MOON, GuardianStar.SATURN, Type.MACHINE, Attribute.DARK, "34743446"),
    Card(417, 3, 800, 900, 40, "Blast Juggler",
         GuardianStar.MARS, GuardianStar.VENUS, Type.MACHINE, Attribute.FIRE, "70138455"),
    Card(418, 4, 900, 1600, 100, "Golgoil",
         GuardianStar.VENUS, GuardianStar.NEPTUNE, Type.MACHINE, Attribute.EARTH, "7526150"),
    Card(419, 5, 1700, 1800, 350, "Giganto",
         GuardianStar.MOON, GuardianStar.URANUS, Type.MACHINE, Attribute.DARK, "33621868"),
    Card(420, 2, 700, 500, 25, "Cyber-Stein",
         GuardianStar.MOON, GuardianStar.NEPTUNE, Type.MACHINE, Attribute.DARK, "69015963"),
    Card(421, 2, 750, 700, 30, "Cyber Commander",
         GuardianStar.MOON, GuardianStar.MERCURY, Type.MACHINE, Attribute.DARK, "6400512"),
    Card(422, 2, 500, 400, 20, "Jinzo #7",
         GuardianStar.MOON, GuardianStar.MARS, Type.MACHINE, Attribute.DARK, "32809211"),
    Card(423, 5, 1650, 1800, 320, "Dice Armadillo",
         GuardianStar.JUPITER, GuardianStar.PLUTO, Type.MACHINE, Attribute.EARTH, "69893315"),
    Card(424, 6, 1900, 1800, 1000, "Sky Dragon",
         GuardianStar.SATURN, GuardianStar.URANUS, Type.DRAGON, Attribute.WIND, "95288024"),
    Card(425, 5, 1600, 1500, 230, "Thunder Dragon",
         GuardianStar.PLUTO, GuardianStar.JUPITER, Type.THUNDER, Attribute.LIGHT, "31786629"),
    Card(426, 7, 2000, 2300, 900, "Stone D.",
         GuardianStar.URANUS, GuardianStar.MARS, Type.ROCK, Attribute.EARTH, "68171737"),
    Card(427, 7, 2300, 2000, 900, "Kaiser Dragon",
         GuardianStar.SUN, GuardianStar.JUPITER, Type.DRAGON, Attribute.LIGHT, "94566432"),
    Card(428, 1, 300, 400, 999999, "Magician of Faith",
         GuardianStar.SUN, GuardianStar.SATURN, Type.SPELLCASTER, Attribute.LIGHT, "31560081"),
    Card(429, 3, 950, 700, 999999, "Goddess of Whim",
         GuardianStar.SUN, GuardianStar.URANUS, Type.FAIRY, Attribute.LIGHT, "67959180"),
    Card(430, 4, 1400, 1000, 120, "Water Magician",
         GuardianStar.NEPTUNE, GuardianStar.MERCURY, Type.AQUA, Attribute.WATER, "93343894"),
    Card(431, 3, 1150, 900, 55, "Ice Water",
         GuardianStar.NEPTUNE, GuardianStar.MOON, Type.AQUA, Attribute.WATER, "20848593"),
    Card(432, 3, 1100, 700, 45, "Waterdragon Fairy",
         GuardianStar.NEPTUNE, GuardianStar.VENUS, Type.AQUA, Attribute.WATER, "66836598"),
    Card(433, 4, 1450, 1200, 120, "Ancient Elf",
         GuardianStar.SUN, GuardianStar.PLUTO, Type.SPELLCASTER, Attribute.LIGHT, "93221206"),
    Card(434, 5, 1750, 1500, 260, "Beautiful Beast Trainer",
         GuardianStar.JUPITER, GuardianStar.NEPTUNE, Type.WARRIOR, Attribute.EARTH, "29616941"),
    Card(435, 4, 1250, 1000, 70, "Water Girl",
         GuardianStar.NEPTUNE, GuardianStar.VENUS, Type.AQUA, Attribute.WATER, "55014050"),
    Card(436, 2, 500, 400, 20, "White Dolphin",
         GuardianStar.NEPTUNE, GuardianStar.JUPITER, Type.FISH, Attribute.WATER, "92409659"),
    Card(437, 5, 1900, 1600, 350, "Deepsea Shark",
         GuardianStar.NEPTUNE, GuardianStar.MARS, Type.FISH, Attribute.WATER, "28593363"),
    Card(438, 5, 1600, 1900, 350, "Metal Fish",
         GuardianStar.NEPTUNE, GuardianStar.SATURN, Type.MACHINE, Attribute.WATER, "55998462"),
    Card(439, 5, 1650, 1700, 290, "Bottom Dweller",
         GuardianStar.NEPTUNE, GuardianStar.SUN, Type.FISH, Attribute.WATER, "81386177"),
    Card(440, 4, 1800, 800, 120, "7 Colored Fish",
         GuardianStar.NEPTUNE, GuardianStar.SUN, Type.FISH, Attribute.WATER, "23771716"),
    Card(441, 5, 1800, 1500, 290, "Mech Bass",
         GuardianStar.NEPTUNE, GuardianStar.VENUS, Type.MACHINE, Attribute.WATER, "50176820"),
    Card(442, 6, 2250, 1900, 700, "Aqua Dragon",
         GuardianStar.NEPTUNE, GuardianStar.MOON, Type.SEA_SERPENT, Attribute.WATER, "86164529"),
    Card(443, 6, 2000, 1700, 1000, "Sea King Dragon",
         GuardianStar.NEPTUNE, GuardianStar.MERCURY, Type.SEA_SERPENT, Attribute.WATER, "23659124"),
    Card(444, 2, 450, 500, 20, "Turu-Purun",
         GuardianStar.NEPTUNE, GuardianStar.JUPITER, Type.AQUA, Attribute.WATER, "59053232"),
    Card(445, 4, 1300, 1000, 80, "Guardian of the Sea",
         GuardianStar.NEPTUNE, GuardianStar.MARS, Type.AQUA, Attribute.WATER, "85448931"),
    Card(446, 3, 1050, 900, 50, "Aqua Snake",
         GuardianStar.NEPTUNE, GuardianStar.SATURN, Type.AQUA, Attribute.WATER, "12436646"),
    Card(447, 4, 1800, 800, 120, "Giant Red Seasnake",
         GuardianStar.NEPTUNE, GuardianStar.SUN, Type.AQUA, Attribute.WATER, "58831685"),
    Card(448, 5, 1600, 1300, 180, "Spike Seadra",
         GuardianStar.PLUTO, GuardianStar.MERCURY, Type.SEA_SERPENT, Attribute.WATER, "85326399"),
    Card(449, 5, 1250, 2100, 290, "30,000-Year White Turtle",
         GuardianStar.NEPTUNE, GuardianStar.MOON, Type.AQUA, Attribute.WATER, "11714098"),
    Card(450, 3, 1200, 900, 60, "Kappa Avenger",
         GuardianStar.NEPTUNE, GuardianStar.VENUS, Type.AQUA, Attribute.WATER, "48109103"),
    Card(451, 3, 650, 900, 30, "Kanikabuto",
         GuardianStar.NEPTUNE, GuardianStar.SATURN, Type.AQUA, Attribute.WATER, "84103702"),
    Card(452, 2, 600, 700, 30, "Zarigun",
         GuardianStar.NEPTUNE, GuardianStar.MERCURY, Type.AQUA, Attribute.WATER, "10598400"),
    Card(453, 6, 2000, 2200, 800, "Millennium Golem",
         GuardianStar.URANUS, GuardianStar.JUPITER, Type.ROCK, Attribute.EARTH, "47986555"),
    Card(454, 4, 1500, 1000, 100, "Destroyer Golem",
         GuardianStar.URANUS, GuardianStar.SUN, Type.ROCK, Attribute.EARTH, "73481154"),
    Card(455, 4, 1000, 1300, 80, "Barrel Rock",
         GuardianStar.URANUS, GuardianStar.MOON, Type.ROCK, Attribute.EARTH, "10476868"),
    Card(456, 4, 1300, 1200, 100, "Minomushi Warrior",
         GuardianStar.URANUS, GuardianStar.VENUS, Type.ROCK, Attribute.EARTH, "46864967"),
    Card(457, 4, 1200, 1000, 70, "Stone Ghost",
         GuardianStar.URANUS, GuardianStar.MERCURY, Type.ROCK, Attribute.EARTH, "72269672"),
    Card(458, 5, 1900, 1400, 290, "Kaminari Attack",
         GuardianStar.PLUTO, GuardianStar.VENUS, Type.THUNDER, Attribute.WIND, "9653271"),
    Card(459, 4, 1200, 1300, 100, "Tripwire Beast",
         GuardianStar.PLUTO, GuardianStar.SATURN, Type.THUNDER, Attribute.EARTH, "45042329"),
    Card(460, 5, 1400, 1500, 180, "Bolt Escargot",
         GuardianStar.PLUTO, GuardianStar.JUPITER, Type.THUNDER, Attribute.WATER, "12146024"),
    Card(461, 3, 1100, 800, 50, "Bolt Penguin",
         GuardianStar.PLUTO, GuardianStar.MOON, Type.THUNDER, Attribute.WATER, "48531733"),
    Card(462, 4, 1500, 1300, 160, "The Immortal of Thunder",
         GuardianStar.PLUTO, GuardianStar.SUN, Type.THUNDER, Attribute.LIGHT, "84926738"),
    Card(463, 3, 800, 900, 40, "Electric Snake",
         GuardianStar.PLUTO, GuardianStar.MARS, Type.THUNDER, Attribute.LIGHT, "11324436"),
    Card(464, 5, 1800, 1500, 290, "Wing Eagle",
         GuardianStar.SATURN, GuardianStar.MERCURY, Type.WINGED_BEAST, Attribute.WIND, "47319141"),
    Card(465, 6, 2100, 1800, 550, "Punished Eagle",
         GuardianStar.SATURN, GuardianStar.PLUTO, Type.WINGED_BEAST, Attribute.WIND, "74703140"),
    Card(466, 4, 1550, 1200, 140, "Skull Red Bird",
         GuardianStar.SATURN, GuardianStar.SUN, Type.WINGED_BEAST, Attribute.WIND, "10202894"),
    Card(467, 6, 2300, 1800, 700, "Crimson Sunbird",
         GuardianStar.MARS, GuardianStar.MERCURY, Type.WINGED_BEAST, Attribute.FIRE, "46696593"),
    Card(468, 5, 1200, 2000, 260, "Queen Bird",
         GuardianStar.SATURN, GuardianStar.MOON, Type.WINGED_BEAST, Attribute.WIND, "73081602"),
    Card(469, 1, 300, 300, 15, "Armed Ninja",
         GuardianStar.NEPTUNE, GuardianStar.MARS, Type.WARRIOR, Attribute.EARTH, "9076207"),
    Card(470, 4, 1300, 1400, 140, "Magical Ghost",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "46474915"),
    Card(471, 6, 2200, 1800, 800, "Soul Hunter",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.FIEND, Attribute.DARK, "72869010"),
    Card(472, 6, 2100, 1600, 1000, "Air Eater",
         GuardianStar.MOON, GuardianStar.SATURN, Type.FIEND, Attribute.WIND, "8353769"),
    Card(473, 5, 1900, 1500, 320, "Vermillion Sparrow",
         GuardianStar.MARS, GuardianStar.VENUS, Type.PYRO, Attribute.FIRE, "35752363"),
    Card(474, 4, 1100, 1300, 120, "Sea Kamen",
         GuardianStar.NEPTUNE, GuardianStar.SUN, Type.AQUA, Attribute.WATER, "71746462"),
    Card(475, 1, 300, 250, 10, "Sinister Serpent",
         GuardianStar.NEPTUNE, GuardianStar.JUPITER, Type.REPTILE, Attribute.WATER, "8131171"),
    Card(476, 2, 600, 800, 30, "Ganigumo",
         GuardianStar.JUPITER, GuardianStar.SUN, Type.INSECT, Attribute.EARTH, "34536276"),
    Card(477, 3, 950, 700, 35, "Alinsection",
         GuardianStar.JUPITER, GuardianStar.MOON, Type.INSECT, Attribute.EARTH, "70924884"),
    Card(478, 3, 1000, 800, 45, "Insect Soldiers of the Sky",
         GuardianStar.SATURN, GuardianStar.NEPTUNE, Type.INSECT, Attribute.WIND, "7019529"),
    Card(479, 3, 800, 900, 999999, "Cockroach Knight",
         GuardianStar.JUPITER, GuardianStar.VENUS, Type.INSECT, Attribute.EARTH, "33413638"),
    Card(480, 4, 1250, 1000, 70, "Kuwagata a",
         GuardianStar.JUPITER, GuardianStar.MERCURY, Type.INSECT, Attribute.EARTH, "60802233"),
    Card(481, 3, 850, 800, 35, "Burglar",
         GuardianStar.JUPITER, GuardianStar.URANUS, Type.BEAST, Attribute.EARTH, "6297941"),
    Card(482, 5, 1900, 1500, 320, "Pragtical",
         GuardianStar.URANUS, GuardianStar.NEPTUNE, Type.DINOSAUR, Attribute.EARTH, "33691040"),
    Card(483, 6, 2000, 1700, 1000, "Garvas",
         GuardianStar.JUPITER, GuardianStar.PLUTO, Type.BEAST, Attribute.EARTH, "69780745"),
    Card(484, 1, 300, 350, 15, "Ameba",
         GuardianStar.NEPTUNE, GuardianStar.MOON, Type.AQUA, Attribute.WATER, "95174353"),
    Card(485, 2, 550, 400, 20, "Korogashi",
         GuardianStar.JUPITER, GuardianStar.NEPTUNE, Type.INSECT, Attribute.EARTH, "32569498"),
    Card(486, 2, 650, 500, 25, "Boo Koo",
         GuardianStar.SUN, GuardianStar.JUPITER, Type.SPELLCASTER, Attribute.DARK, "68963107"),
    Card(487, 5, 1800, 1400, 260, "Flower Wolf",
         GuardianStar.JUPITER, GuardianStar.SUN, Type.BEAST, Attribute.EARTH, "95952802"),
    Card(488, 2, 400, 500, 20, "Rainbow Flower",
         GuardianStar.JUPITER, GuardianStar.MOON, Type.PLANT, Attribute.EARTH, "21347810"),
    Card(489, 3, 1100, 600, 40, "Barrel Lily",
         GuardianStar.JUPITER, GuardianStar.VENUS, Type.PLANT, Attribute.EARTH, "67841515"),
    Card(490, 2, 750, 700, 30, "Needle Ball",
         GuardianStar.MOON, GuardianStar.URANUS, Type.FIEND, Attribute.DARK, "94230224"),
    Card(491, 5, 1700, 1500, 260, "Peacock",
         GuardianStar.JUPITER, GuardianStar.MERCURY, Type.WINGED_BEAST, Attribute.WIND, "20624263"),
    Card(492, 2, 500, 700, 25, "Hoshiningen",
         GuardianStar.SUN, GuardianStar.NEPTUNE, Type.FAIRY, Attribute.LIGHT, "67629977"),
    Card(493, 5, 1550, 1400, 180, "Maha Vailo",
         GuardianStar.SUN, GuardianStar.VENUS, Type.SPELLCASTER, Attribute.LIGHT, "93013676"),
    Card(494, 5, 1550, 1700, 260, "Rainbow Marine Mermaid",
         GuardianStar.NEPTUNE, GuardianStar.VENUS, Type.FISH, Attribute.WATER, "29402771"),
    Card(495, 5, 1750, 1500, 260, "Musician King",
         GuardianStar.SUN, GuardianStar.MARS, Type.SPELLCASTER, Attribute.LIGHT, "56907389"),
    Card(496, 4, 1000, 1200, 70, "Wilmee",
         GuardianStar.JUPITER, GuardianStar.URANUS, Type.BEAST, Attribute.EARTH, "92391084"),
    Card(497, 4, 900, 1700, 120, "Yado Karu",
         GuardianStar.NEPTUNE, GuardianStar.MERCURY, Type.AQUA, Attribute.WATER, "29380133"),
    Card(498, 5, 1550, 1300, 160, "Morinphen",
         GuardianStar.MOON, GuardianStar.NEPTUNE, Type.FIEND, Attribute.DARK, "55784832"),
    Card(499, 1, 250, 300, 999999, "Kattapillar",
         GuardianStar.JUPITER, GuardianStar.NEPTUNE, Type.INSECT, Attribute.EARTH, "81179446"),
    Card(500, 6, 2000, 2100, 700, "Dragon Seeker",
         GuardianStar.MOON, GuardianStar.MERCURY, Type.FIEND, Attribute.DARK, "28563545"),
    Card(501, 2, 450, 600, 20, "Man-eater Bug",
         GuardianStar.JUPITER, GuardianStar.PLUTO, Type.INSECT, Attribute.EARTH, "54652250"),
    Card(502, 4, 1300, 1100, 120, "D. Human",
         GuardianStar.URANUS, GuardianStar.MARS, Type.WARRIOR, Attribute.EARTH, "81057959"),
    Card(503, 3, 700, 900, 35, "Turtle Raccoon",
         GuardianStar.JUPITER, GuardianStar.SUN, Type.AQUA, Attribute.WATER, "17441953"),
    Card(504, 1, 400, 300, 15, "Fungi of the Musk",
         GuardianStar.JUPITER, GuardianStar.MOON, Type.FIEND, Attribute.DARK, "53830602"),
    Card(505, 3, 800, 1000, 45, "Prisman",
         GuardianStar.URANUS, GuardianStar.JUPITER, Type.ROCK, Attribute.LIGHT, "80234301"),
    Card(506, 2, 650, 600, 25, "Gale Dogra",
         GuardianStar.JUPITER, GuardianStar.VENUS, Type.INSECT, Attribute.EARTH, "16229315"),
    Card(507, 4, 1600, 1200, 160, "Crazy Fish",
         GuardianStar.NEPTUNE, GuardianStar.JUPITER, Type.FISH, Attribute.WATER, "53713014"),
    Card(508, 5, 1800, 1400, 260, "Cyber Saurus",
         GuardianStar.URANUS, GuardianStar.SUN, Type.MACHINE, Attribute.EARTH, "89112729"),
    Card(509, 6, 2200, 2000, 800, "Bracchio-raidus",
         GuardianStar.URANUS, GuardianStar.MOON, Type.DINOSAUR, Attribute.WATER, "16507828"),
    Card(510, 2, 900, 500, 30, "Laughing Flower",
         GuardianStar.JUPITER, GuardianStar.MERCURY, Type.PLANT, Attribute.EARTH, "42591472"),
    Card(511, 4, 1400, 1300, 140, "Bean Soldier",
         GuardianStar.JUPITER, GuardianStar.URANUS, Type.PLANT, Attribute.EARTH, "84990171"),
    Card(512, 4, 1400, 1300, 140, "Cannon Soldier",
         GuardianStar.MOON, GuardianStar.MARS, Type.MACHINE, Attribute.DARK, "11384280"),
    Card(513, 5, 1650, 1600, 260, "Guardian of the Throne Room",
         GuardianStar.SUN, GuardianStar.SATURN, Type.MACHINE, Attribute.LIGHT, "47879985"),
    Card(514, 4, 1300, 1000, 80, "Brave Scizzar",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.MACHINE, Attribute.DARK, "74277583"),
    Card(515, 4, 1100, 1400, 100, "The Statue of Easter Island",
         GuardianStar.URANUS, GuardianStar.VENUS, Type.ROCK, Attribute.EARTH, "10262698"),
    Card(516, 2, 600, 300, 20, "Muka Muka",
         GuardianStar.URANUS, GuardianStar.MERCURY, Type.ROCK, Attribute.EARTH, "46657337"),
    Card(517, 5, 1300, 1600, 180, "Sand Stone",
         GuardianStar.URANUS, GuardianStar.MARS, Type.ROCK, Attribute.EARTH, "73051941"),
    Card(518, 6, 1450, 2200, 400, "Boulder Tortoise",
         GuardianStar.URANUS, GuardianStar.NEPTUNE, Type.AQUA, Attribute.EARTH, "9540040"),
    Card(519, 5, 1600, 1500, 230, "Fire Kraken",
         GuardianStar.MARS, GuardianStar.SATURN, Type.AQUA, Attribute.FIRE, "46534755"),
    Card(520, 6, 1900, 1700, 400, "Turtle Bird",
         GuardianStar.SATURN, GuardianStar.MARS, Type.AQUA, Attribute.WATER, "72929454"),
    Card(521, 6, 1900, 1700, 400, "Skullbird",
         GuardianStar.SATURN, GuardianStar.VENUS, Type.WINGED_BEAST, Attribute.WIND, "8327462"),
    Card(522, 6, 2000, 1900, 550, "Monstrous Bird",
         GuardianStar.SATURN, GuardianStar.MERCURY, Type.WINGED_BEAST, Attribute.WIND, "35712107"),
    Card(523, 4, 1800, 1000, 160, "The Bistro Butcher",
         GuardianStar.MOON, GuardianStar.SATURN, Type.FIEND, Attribute.DARK, "71107816"),
    Card(524, 2, 550, 500, 20, "Star Boy",
         GuardianStar.NEPTUNE, GuardianStar.MARS, Type.AQUA, Attribute.WATER, "8201910"),
    Card(525, 5, 1300, 1800, 230, "Spirit of the Mountain",
         GuardianStar.URANUS, GuardianStar.JUPITER, Type.SPELLCASTER, Attribute.EARTH, "34690519"),
    Card(526, 6, 1750, 1900, 400, "Neck Hunter",
         GuardianStar.MOON, GuardianStar.URANUS, Type.FIEND, Attribute.DARK, "70084224"),
    Card(527, 1, 300, 250, 10, "Milus Radiant",
         GuardianStar.JUPITER, GuardianStar.NEPTUNE, Type.BEAST, Attribute.EARTH, "7489323"),
    Card(528, 5, 1600, 1800, 320, "Togex",
         GuardianStar.URANUS, GuardianStar.SUN, Type.BEAST, Attribute.EARTH, "33878931"),
    Card(529, 6, 2100, 1800, 550, "Flame Cerebrus",
         GuardianStar.MARS, GuardianStar.PLUTO, Type.PYRO, Attribute.FIRE, "60862676"),
    Card(530, 3, 950, 1000, 50, "Eldeen",
         GuardianStar.MERCURY, GuardianStar.MOON, Type.SPELLCASTER, Attribute.LIGHT, "6367785"),
    Card(531, 6, 2100, 1700, 500, "Mystical Sand",
         GuardianStar.MERCURY, GuardianStar.SATURN, Type.ROCK, Attribute.EARTH, "32751480"),
    Card(532, 4, 1900, 900, 160, "Gemini Elf",
         GuardianStar.SUN, GuardianStar.URANUS, Type.SPELLCASTER, Attribute.EARTH, "69140098"),
    Card(533, 6, 1900, 1700, 400, "Kwagar Hercules",
         GuardianStar.JUPITER, GuardianStar.SUN, Type.INSECT, Attribute.EARTH, "95144193"),
    Card(534, 3, 850, 750, 35, "Minar",
         GuardianStar.MOON, GuardianStar.NEPTUNE, Type.INSECT, Attribute.EARTH, "32539892"),
    Card(535, 4, 1150, 1400, 100, "Kamakiriman",
         GuardianStar.JUPITER, GuardianStar.MOON, Type.INSECT, Attribute.EARTH, "68928540"),
    Card(536, 2, 800, 600, 30, "Mechaleon",
         GuardianStar.NEPTUNE, GuardianStar.SATURN, Type.REPTILE, Attribute.WATER, "94412545"),
    Card(537, 2, 750, 600, 30, "Mega Thunderball",
         GuardianStar.PLUTO, GuardianStar.MERCURY, Type.THUNDER, Attribute.WIND, "21817254"),
    Card(538, 3, 900, 800, 40, "Niwatori",
         GuardianStar.JUPITER, GuardianStar.VENUS, Type.WINGED_BEAST, Attribute.EARTH, "7805359"),
    Card(539, 3, 1100, 700, 45, "Corroding Shark",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "34290067"),
    Card(540, 2, 900, 400, 30, "Skelengel",
         GuardianStar.SUN, GuardianStar.PLUTO, Type.FAIRY, Attribute.LIGHT, "60694662"),
    Card(541, 2, 450, 500, 999999, "Hane-Hane",
         GuardianStar.SATURN, GuardianStar.PLUTO, Type.BEAST, Attribute.EARTH, "7089711"),
    Card(542, 5, 1400, 1600, 200, "Misairuzame",
         GuardianStar.NEPTUNE, GuardianStar.SUN, Type.FISH, Attribute.WATER, "33178416"),
    Card(543, 4, 1350, 800, 60, "Tongyo",
         GuardianStar.NEPTUNE, GuardianStar.MOON, Type.FISH, Attribute.WATER, "69572024"),
    Card(544, 2, 900, 500, 30, "Dharma Cannon",
         GuardianStar.MOON, GuardianStar.MERCURY, Type.MACHINE, Attribute.DARK, "96967123"),
    Card(545, 6, 1700, 1900, 999999, "Skelgon",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "32355828"),
    Card(546, 4, 1250, 900, 60, "Wow Warrior",
         GuardianStar.NEPTUNE, GuardianStar.MARS, Type.FISH, Attribute.WATER, "69750536"),
    Card(547, 1, 350, 300, 15, "Griggle",
         GuardianStar.JUPITER, GuardianStar.MERCURY, Type.PLANT, Attribute.EARTH, "95744531"),
    Card(548, 1, 400, 300, 15, "Bone Mouse",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "21239280"),
    Card(549, 2, 700, 500, 25, "Frog The Jam",
         GuardianStar.NEPTUNE, GuardianStar.JUPITER, Type.AQUA, Attribute.WATER, "68638985"),
    Card(550, 4, 1350, 1000, 80, "Behegon",
         GuardianStar.NEPTUNE, GuardianStar.SATURN, Type.AQUA, Attribute.WATER, "94022093"),
    Card(551, 4, 2000, 800, 160, "Dark Elf",
         GuardianStar.MERCURY, GuardianStar.JUPITER, Type.SPELLCASTER, Attribute.DARK, "21417692"),
    Card(552, 4, 1200, 1000, 70, "Winged Dragon #2",
         GuardianStar.SATURN, GuardianStar.NEPTUNE, Type.WINGED_BEAST, Attribute.WIND, "57405307"),
    Card(553, 3, 1250, 800, 55, "Mushroom Man #2",
         GuardianStar.JUPITER, GuardianStar.URANUS, Type.WARRIOR, Attribute.EARTH, "93900406"),
    Card(554, 5, 1550, 1800, 999999, "Lava Battleguard",
         GuardianStar.MERCURY, GuardianStar.MARS, Type.WARRIOR, Attribute.EARTH, "20394040"),
    Card(555, 6, 1700, 1900, 999999, "Tyhone #2",
         GuardianStar.MARS, GuardianStar.URANUS, Type.DRAGON, Attribute.FIRE, "56789759"),
    Card(556, 2, 800, 600, 30, "The Wandering Doomed",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "93788854"),
    Card(557, 5, 1400, 1800, 260, "Steel Ogre Grotto #1",
         GuardianStar.MOON, GuardianStar.MARS, Type.MACHINE, Attribute.EARTH, "29172562"),
    Card(558, 2, 400, 400, 20, "Pot the Trick",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.ROCK, Attribute.EARTH, "55567161"),
    Card(559, 3, 1250, 700, 50, "Oscillo Hero",
         GuardianStar.VENUS, GuardianStar.SUN, Type.WARRIOR, Attribute.EARTH, "82065276"),
    Card(560, 4, 950, 1400, 80, "Invader from Another Dimension",
         GuardianStar.VENUS, GuardianStar.MARS, Type.FIEND, Attribute.DARK, "28450915"),
    Card(561, 4, 1200, 1000, 70, "Lesser Dragon",
         GuardianStar.MOON, GuardianStar.SATURN, Type.DRAGON, Attribute.WIND, "55444629"),
    Card(562, 2, 750, 600, 999999, "Needle Worm",
         GuardianStar.JUPITER, GuardianStar.PLUTO, Type.INSECT, Attribute.EARTH, "81843628"),
    Card(563, 2, 550, 400, 20, "Wretched Ghost of the Attic",
         GuardianStar.MERCURY, GuardianStar.URANUS, Type.FIEND, Attribute.DARK, "17238333"),
    Card(564, 6, 2200, 1800, 800, "Great Mammoth of Goldfine",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "54622031"),
    Card(565, 5, 2100, 1300, 320, "Man-eating Black Shark",
         GuardianStar.NEPTUNE, GuardianStar.SUN, Type.FISH, Attribute.WATER, "80727036"),
    Card(566, 3, 1200, 900, 60, "Yormungarde",
         GuardianStar.NEPTUNE, GuardianStar.MOON, Type.REPTILE, Attribute.EARTH, "17115745"),
    Card(567, 3, 1200, 900, 60, "Darkworld Thorns",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.PLANT, Attribute.EARTH, "43500484"),
    Card(568, 3, 1000, 850, 45, "Anthrosaurus",
         GuardianStar.URANUS, GuardianStar.MOON, Type.DINOSAUR, Attribute.EARTH, "89904598"),
    Card(569, 3, 900, 800, 40, "Drooling Lizard",
         GuardianStar.NEPTUNE, GuardianStar.VENUS, Type.REPTILE, Attribute.EARTH, "16353197"),
    Card(570, 3, 1300, 800, 60, "Trakadon",
         GuardianStar.URANUS, GuardianStar.VENUS, Type.DINOSAUR, Attribute.EARTH, "42348802"),
    Card(571, 6, 2100, 1800, 999999, "B. Dragon Jungle King",
         GuardianStar.JUPITER, GuardianStar.NEPTUNE, Type.DRAGON, Attribute.EARTH, "89832901"),
    Card(572, 6, 2100, 1700, 500, "Empress Judge",
         GuardianStar.URANUS, GuardianStar.VENUS, Type.WARRIOR, Attribute.EARTH, "15237615"),
    Card(573, 3, 1100, 700, 45, "Little D",
         GuardianStar.URANUS, GuardianStar.MERCURY, Type.DINOSAUR, Attribute.EARTH, "42625254"),
    Card(574, 4, 1100, 1200, 80, "Witch of the Black Forest",
         GuardianStar.MERCURY, GuardianStar.PLUTO, Type.SPELLCASTER, Attribute.DARK, "78010363"),
    Card(575, 6, 1800, 1900, 1000, "Ancient One of the Deep Forest",
         GuardianStar.JUPITER, GuardianStar.SUN, Type.BEAST, Attribute.EARTH, "14015067"),
    Card(576, 3, 1100, 1000, 60, "Giant Scorpion of the Tundra",
         GuardianStar.JUPITER, GuardianStar.MOON, Type.INSECT, Attribute.EARTH, "41403766"),
    Card(577, 5, 1850, 1600, 320, "Crow Goblin",
         GuardianStar.SATURN, GuardianStar.MARS, Type.WINGED_BEAST, Attribute.WIND, "77998771"),
    Card(578, 5, 1350, 1200, 100, "Leo Wizard",
         GuardianStar.MERCURY, GuardianStar.NEPTUNE, Type.SPELLCASTER, Attribute.EARTH, "4392470"),
    Card(579, 2, 750, 400, 25, "Abyss Flower",
         GuardianStar.JUPITER, GuardianStar.VENUS, Type.PLANT, Attribute.EARTH, "40387124"),
    Card(580, 3, 1100, 900, 55, "Patrol Robo",
         GuardianStar.MOON, GuardianStar.URANUS, Type.MACHINE, Attribute.EARTH, "76775123"),
    Card(581, 4, 1450, 1000, 120, "Takuhee",
         GuardianStar.SATURN, GuardianStar.SUN, Type.WINGED_BEAST, Attribute.WIND, "3170832"),
    Card(582, 5, 1800, 1700, 350, "Dark Witch",
         GuardianStar.SUN, GuardianStar.NEPTUNE, Type.FAIRY, Attribute.LIGHT, "35565537"),
    Card(583, 4, 950, 1500, 120, "Weather Report",
         GuardianStar.NEPTUNE, GuardianStar.MERCURY, Type.AQUA, Attribute.WATER, "72053645"),
    Card(584, 3, 1000, 1100, 60, "Binding Chain",
         GuardianStar.SUN, GuardianStar.VENUS, Type.FAIRY, Attribute.LIGHT, "8058240"),
    Card(585, 3, 800, 1000, 45, "Mechanical Snail",
         GuardianStar.MOON, GuardianStar.NEPTUNE, Type.MACHINE, Attribute.DARK, "34442949"),
    Card(586, 3, 650, 900, 30, "Greenkappa",
         GuardianStar.MOON, GuardianStar.MERCURY, Type.WARRIOR, Attribute.DARK, "61831093"),
    Card(587, 4, 1300, 1400, 140, "Mon Larvas",
         GuardianStar.JUPITER, GuardianStar.MERCURY, Type.BEAST, Attribute.EARTH, "7225792"),
    Card(588, 3, 900, 1100, 55, "Living Vase",
         GuardianStar.JUPITER, GuardianStar.URANUS, Type.PLANT, Attribute.EARTH, "34320307"),
    Card(589, 2, 500, 600, 25, "Tentacle Plant",
         GuardianStar.JUPITER, GuardianStar.PLUTO, Type.PLANT, Attribute.WATER, "60715406"),
    Card(590, 3, 800, 900, 40, "Beaked Snake",
         GuardianStar.NEPTUNE, GuardianStar.JUPITER, Type.REPTILE, Attribute.EARTH, "6103114"),
    Card(591, 2, 700, 600, 30, "Morphing Jar",
         GuardianStar.URANUS, GuardianStar.NEPTUNE, Type.ROCK, Attribute.EARTH, "33508719"),
    Card(592, 3, 850, 900, 40, "Muse-A",
         GuardianStar.SUN, GuardianStar.MARS, Type.FAIRY, Attribute.LIGHT, "69992868"),
    Card(593, 5, 1400, 1800, 260, "Giant Turtle Who Feeds on Flames",
         GuardianStar.NEPTUNE, GuardianStar.MARS, Type.AQUA, Attribute.WATER, "96981563"),
    Card(594, 6, 2000, 1800, 500, "Rose Spectre of Dunn",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.PLANT, Attribute.DARK, "32485271"),
    Card(595, 4, 1300, 1400, 140, "Fiend Refrection #1",
         GuardianStar.VENUS, GuardianStar.JUPITER, Type.WINGED_BEAST, Attribute.WIND, "68870276"),
    Card(596, 4, 1600, 1200, 160, "Ghoul with an Appetite",
         GuardianStar.MOON, GuardianStar.PLUTO, Type.ZOMBIE, Attribute.DARK, "95265975"),
    Card(597, 4, 1500, 1200, 140, "Pale Beast",
         GuardianStar.JUPITER, GuardianStar.NEPTUNE, Type.BEAST, Attribute.EARTH, "21263083"),
    Card(598, 2, 600, 550, 25, "Little Chimera",
         GuardianStar.SATURN, GuardianStar.MOON, Type.BEAST, Attribute.FIRE, "68658728"),
    Card(599, 4, 1550, 800, 80, "Violent Rain",
         GuardianStar.NEPTUNE, GuardianStar.MERCURY, Type.AQUA, Attribute.WATER, "94042337"),
    Card(600, 4, 1050, 1200, 70, "Key Mace #2",
         GuardianStar.MERCURY, GuardianStar.MOON, Type.FIEND, Attribute.DARK, "20541432"),
    Card(601, 3, 700, 1400, 60, "Tenderness",
         GuardianStar.SUN, GuardianStar.JUPITER, Type.FAIRY, Attribute.LIGHT, "57935140"),
    Card(602, 2, 750, 500, 25, "Penguin Soldier",
         GuardianStar.NEPTUNE, GuardianStar.MARS, Type.AQUA, Attribute.WATER, "93920745"),
    Card(603, 4, 1100, 1200, 999999, "Fairy Dragon",
         GuardianStar.SUN, GuardianStar.SATURN, Type.DRAGON, Attribute.WIND, "20315854"),
    Card(604, 3, 750, 800, 30, "Obese Marmot of Nefariousness",
         GuardianStar.MOON, GuardianStar.MARS, Type.BEAST, Attribute.EARTH, "56713552"),
    Card(605, 3, 950, 800, 40, "Liquid Beast",
         GuardianStar.NEPTUNE, GuardianStar.SATURN, Type.AQUA, Attribute.WATER, "93108297"),
    Card(606, 3, 850, 700, 30, "Twin Long Rods #2",
         GuardianStar.NEPTUNE, GuardianStar.SUN, Type.AQUA, Attribute.WATER, "29692206"),
    Card(607, 4, 1250, 1300, 100, "Great Bill",
         GuardianStar.JUPITER, GuardianStar.SUN, Type.BEAST, Attribute.EARTH, "55691901"),
    Card(608, 4, 1300, 1100, 120, "Shining Friendship",
         GuardianStar.SUN, GuardianStar.URANUS, Type.FAIRY, Attribute.LIGHT, "82085619"),
    Card(609, 2, 600, 700, 30, "Bladefly",
         GuardianStar.JUPITER, GuardianStar.MOON, Type.INSECT, Attribute.WIND, "28470714"),
    Card(610, 3, 850, 800, 35, "Electric Lizard",
         GuardianStar.PLUTO, GuardianStar.SUN, Type.THUNDER, Attribute.EARTH, "55875323"),
    Card(611, 2, 650, 500, 25, "Hiro's Shadow Scout",
         GuardianStar.MOON, GuardianStar.SATURN, Type.FIEND, Attribute.DARK, "81863068"),
    Card(612, 3, 1100, 800, 50, "Lady of Faith",
         GuardianStar.SUN, GuardianStar.PLUTO, Type.SPELLCASTER, Attribute.LIGHT, "17358176"),
    Card(613, 7, 2800, 2100, 1900, "Twin-headed Thunder Dragon",
         GuardianStar.PLUTO, GuardianStar.MOON, Type.THUNDER, Attribute.LIGHT, "54752875"),
    Card(614, 5, 1600, 1400, 200, "Hunter Spider",
         GuardianStar.JUPITER, GuardianStar.VENUS, Type.INSECT, Attribute.EARTH, "80141480"),
    Card(615, 4, 850, 1400, 70, "Armored Starfish",
         GuardianStar.NEPTUNE, GuardianStar.MOON, Type.AQUA, Attribute.WATER, "17535588"),
    Card(616, 4, 1100, 1200, 80, "Hourglass of Courage",
         GuardianStar.SUN, GuardianStar.NEPTUNE, Type.FAIRY, Attribute.LIGHT, "43530283"),
    Card(617, 5, 1700, 1600, 290, "Marine Beast",
         GuardianStar.NEPTUNE, GuardianStar.MERCURY, Type.FISH, Attribute.WATER, "29929832"),
    Card(618, 6, 1900, 1700, 400, "Warrior of Tradition",
         GuardianStar.MERCURY, GuardianStar.MARS, Type.WARRIOR, Attribute.EARTH, "56413937"),
    Card(619, 5, 1650, 1900, 350, "Rock Spirit",
         GuardianStar.URANUS, GuardianStar.MARS, Type.SPELLCASTER, Attribute.EARTH, "82818645"),
    Card(620, 4, 1000, 1200, 70, "Snakeyashi",
         GuardianStar.JUPITER, GuardianStar.MERCURY, Type.PLANT, Attribute.EARTH, "29802344"),
    Card(621, 5, 1650, 1300, 180, "Succubus Knight",
         GuardianStar.MERCURY, GuardianStar.JUPITER, Type.WARRIOR, Attribute.DARK, "55291359"),
    Card(622, 5, 1600, 1500, 230, "Ill Witch",
         GuardianStar.SUN, GuardianStar.VENUS, Type.SPELLCASTER, Attribute.LIGHT, "81686058"),
    Card(623, 4, 1200, 1300, 100, "The Thing That Hides in the Mud",
         GuardianStar.URANUS, GuardianStar.JUPITER, Type.ROCK, Attribute.EARTH, "18180762"),
    Card(624, 5, 1650, 1300, 180, "High Tide Gyojin",
         GuardianStar.NEPTUNE, GuardianStar.MERCURY, Type.AQUA, Attribute.WATER, "54579801"),
    Card(625, 4, 1600, 1100, 140, "Fairy of the Fountain",
         GuardianStar.NEPTUNE, GuardianStar.VENUS, Type.AQUA, Attribute.WATER, "81563416"),
    Card(626, 4, 1300, 1400, 140, "Amazon of the Seas",
         GuardianStar.NEPTUNE, GuardianStar.MARS, Type.FISH, Attribute.WATER, "17968114"),
    Card(627, 6, 1900, 2000, 550, "Nekogal #2",
         GuardianStar.JUPITER, GuardianStar.URANUS, Type.BEAST_WARRIOR, Attribute.EARTH, "43352213"),
    Card(628, 2, 550, 500, 999999, "Witch's Apprentice",
         GuardianStar.MERCURY, GuardianStar.SATURN, Type.SPELLCASTER, Attribute.DARK, "80741828"),
    Card(629, 3, 950, 1100, 55, "Armored Rat",
         GuardianStar.JUPITER, GuardianStar.PLUTO, Type.BEAST, Attribute.EARTH, "16246527"),
    Card(630, 4, 1400, 1100, 100, "Ancient Lizard Warrior",
         GuardianStar.NEPTUNE, GuardianStar.SATURN, Type.REPTILE, Attribute.EARTH, "43230671"),
    Card(631, 4, 1500, 1300, 160, "Maiden of the Moonlight",
         GuardianStar.MOON, GuardianStar.VENUS, Type.SPELLCASTER, Attribute.LIGHT, "79629370"),
    Card(632, 5, 1600, 1500, 230, "Stone Ogre Grotto",
         GuardianStar.URANUS, GuardianStar.SUN, Type.ROCK, Attribute.EARTH, "15023985"),
    Card(633, 5, 1400, 1700, 230, "Winged Egg of New Life",
         GuardianStar.SUN, GuardianStar.MARS, Type.FAIRY, Attribute.LIGHT, "42418084"),
    Card(634, 4, 1150, 1300, 120, "Night Lizard",
         GuardianStar.NEPTUNE, GuardianStar.JUPITER, Type.AQUA, Attribute.WATER, "78402798"),
    Card(635, 1, 350, 300, 15, "Queen's Double",
         GuardianStar.URANUS, GuardianStar.MOON, Type.WARRIOR, Attribute.EARTH, "5901497"),
    Card(636, 4, 1600, 1200, 160, "Blue-winged Crown",
         GuardianStar.SATURN, GuardianStar.VENUS, Type.WINGED_BEAST, Attribute.WIND, "41396436"),
    Card(637, 5, 1500, 1800, 290, "Trent",
         GuardianStar.JUPITER, GuardianStar.SUN, Type.PLANT, Attribute.EARTH, "78780140"),
    Card(638, 5, 1800, 1500, 290, "Queen of Autumn Leaves",
         GuardianStar.JUPITER, GuardianStar.MOON, Type.PLANT, Attribute.EARTH, "4179849"),
    Card(639, 5, 1850, 1300, 230, "Amphibious Bugroth",
         GuardianStar.NEPTUNE, GuardianStar.SUN, Type.AQUA, Attribute.WATER, "40173854"),
    Card(640, 3, 900, 700, 999999, "Acid Crawler",
         GuardianStar.JUPITER, GuardianStar.VENUS, Type.INSECT, Attribute.EARTH, "77568553"),
    Card(641, 5, 1350, 1700, 200, "Invader of the Throne",
         GuardianStar.URANUS, GuardianStar.VENUS, Type.WARRIOR, Attribute.EARTH, "3056267"),
    Card(642, 3, 1150, 900, 55, "Mystical Sheep #1",
         GuardianStar.VENUS, GuardianStar.SATURN, Type.BEAST, Attribute.EARTH, "30451366"),
    Card(643, 4, 1350, 1000, 80, "Disk Magician",
         GuardianStar.MOON, GuardianStar.URANUS, Type.MACHINE, Attribute.DARK, "76446915"),
    Card(644, 2, 400, 450, 20, "Flame Viper",
         GuardianStar.MARS, GuardianStar.PLUTO, Type.PYRO, Attribute.EARTH, "2830619"),
    Card(645, 6, 1900, 2200, 700, "Royal Guard",
         GuardianStar.URANUS, GuardianStar.MERCURY, Type.MACHINE, Attribute.EARTH, "39239728"),
    Card(646, 3, 1300, 700, 55, "Gruesome Goo",
         GuardianStar.NEPTUNE, GuardianStar.MOON, Type.AQUA, Attribute.WATER, "65623423"),
    Card(647, 4, 1500, 900, 120, "Hyosube",
         GuardianStar.NEPTUNE, GuardianStar.VENUS, Type.AQUA, Attribute.WATER, "2118022"),
    Card(648, 5, 1600, 1300, 180, "Machine Attacker",
         GuardianStar.MOON, GuardianStar.NEPTUNE, Type.MACHINE, Attribute.EARTH, "38116136"),
    Card(649, 4, 1450, 1000, 120, "Hibikime",
         GuardianStar.MERCURY, GuardianStar.URANUS, Type.WARRIOR, Attribute.EARTH, "64501875"),
    Card(650, 5, 1650, 1600, 260, "Whiptail Crow",
         GuardianStar.MOON, GuardianStar.MERCURY, Type.FIEND, Attribute.DARK, "91996584"),
    Card(651, 0, 0, 0, 800, "Kunai with Chain",
         None, None, Type.EQUIP, None, "37390589"),
    Card(652, 0, 0, 0, 10, "Magical Labyrinth",
         None, None, Type.EQUIP, None, "64389297"),
    Card(653, 0, 0, 0, 999999, "Warrior Elimination",
         None, None, Type.MAGIC, None, "90873992"),
    Card(654, 0, 0, 0, 800, "Salamandra",
         None, None, Type.EQUIP, None, "32268901"),
    Card(655, 0, 0, 0, 10, "Cursebreaker",
         None, None, Type.MAGIC, None, "69666645"),
    Card(656, 0, 0, 0, 999999, "Eternal Rest",
         None, None, Type.MAGIC, None, "95051344"),
    Card(657, 0, 0, 0, 50000, "Megamorph",
         None, None, Type.EQUIP, None, "22046459"),
    Card(658, 0, 0, 0, 999999, "Metalmorph",
         None, None, Type.EQUIP, None, "68540058"),
    Card(659, 0, 0, 0, 800, "Winged Trumpeter",
         None, None, Type.EQUIP, None, "94939166"),
    Card(660, 0, 0, 0, 999999, "Stain Storm",
         None, None, Type.MAGIC, None, "21323861"),
    Card(661, 0, 0, 0, 999999, "Crush Card",
         None, None, Type.MAGIC, None, "57728570"),
    Card(662, 0, 0, 0, 999999, "Eradicating Aerosol",
         None, None, Type.MAGIC, None, "94716515"),
    Card(663, 0, 0, 0, 999999, "Breath of Light",
         None, None, Type.MAGIC, None, "20101223"),
    Card(664, 0, 0, 0, 999999, "Eternal Draught",
         None, None, Type.MAGIC, None, "56606928"),
    Card(665, 0, 0, 0, 10, "Curse of Millennium Shield",
         None, None, Type.RITUAL, None, "83094937"),
    Card(666, 0, 0, 0, 10, "Yamadron Ritual",
         None, None, Type.RITUAL, None, "29089635"),
    Card(667, 0, 0, 0, 999999, "Gate Guardian Ritual",
         None, None, Type.RITUAL, None, "56483330"),
    Card(668, 0, 0, 0, 800, "Bright Castle",
         None, None, Type.EQUIP, None, "82878489"),
    Card(669, 0, 0, 0, 999999, "Shadow Spell",
         None, None, Type.MAGIC, None, "29267084"),
    Card(670, 0, 0, 0, 100, "Black Luster Ritual",
         None, None, Type.RITUAL, None, "55761792"),
    Card(671, 0, 0, 0, 100, "Zera Ritual",
         None, None, Type.RITUAL, None, "81756897"),
    Card(672, 0, 0, 0, 999999, "Harpie's Feather Duster",
         None, None, Type.MAGIC, None, "18144506"),
    Card(673, 0, 0, 0, 100, "War-lion Ritual",
         None, None, Type.RITUAL, None, "54539105"),
    Card(674, 0, 0, 0, 100, "Beastry Mirror Ritual",
         None, None, Type.RITUAL, None, "81933259"),
    Card(675, 0, 0, 0, 999999, "Ultimate Dragon",
         None, None, Type.RITUAL, None, "17928958"),
    Card(676, 0, 0, 0, 50, "Commencement Dance",
         None, None, Type.RITUAL, None, "43417563"),
    Card(677, 0, 0, 0, 50, "Hamburger Recipe",
         None, None, Type.RITUAL, None, "80811661"),
    Card(678, 0, 0, 0, 100, "Revival of Sennen Genjin",
         None, None, Type.RITUAL, None, "16206366"),
    Card(679, 0, 0, 0, 50, "Novox's Prayer",
         None, None, Type.RITUAL, None, "43694075"),
    Card(680, 0, 0, 0, 50, "Curse of Tri-Horned Dragon",
         None, None, Type.RITUAL, None, "79699070"),
    Card(681, 0, 0, 0, 999999, "House of Adhesive Tape",
         None, None, Type.TRAP, None, "15083728"),
    Card(682, 0, 0, 0, 999999, "Eatgaboon",
         None, None, Type.TRAP, None, "42578427"),
    Card(683, 0, 0, 0, 999999, "Bear Trap",
         None, None, Type.TRAP, None, "78977532"),
    Card(684, 0, 0, 0, 999999, "Invisible Wire",
         None, None, Type.TRAP, None, "15361130"),
    Card(685, 0, 0, 0, 999999, "Acid Trap Hole",
         None, None, Type.TRAP, None, "41356845"),
    Card(686, 0, 0, 0, 999999, "Widespread Ruin",
         None, None, Type.TRAP, None, "77754944"),
    Card(687, 0, 0, 0, 999999, "Goblin Fan",
         None, None, Type.TRAP, None, "4149689"),
    Card(688, 0, 0, 0, 999999, "Bad Reaction to Simochi",
         None, None, Type.TRAP, None, "40633297"),
    Card(689, 0, 0, 0, 999999, "Reverse Trap",
         None, None, Type.TRAP, None, "77622396"),
    Card(690, 0, 0, 0, 999999, "Fake Trap",
         None, None, Type.TRAP, None, "3027001"),
    Card(691, 0, 0, 0, 50, "Revived of Serpent Night Dragon",
         None, None, Type.RITUAL, None, "39411600"),
    Card(692, 0, 0, 0, 100, "Turtle Oath",
         None, None, Type.RITUAL, None, "76806714"),
    Card(693, 0, 0, 0, 100, "Contruct of Mask",
         None, None, Type.RITUAL, None, "2304453"),
    Card(694, 0, 0, 0, 100, "Resurrection of Chakra",
         None, None, Type.RITUAL, None, "39399168"),
    Card(695, 0, 0, 0, 50, "Puppet Ritual",
         None, None, Type.RITUAL, None, "5783166"),
    Card(696, 0, 0, 0, 999999, "Javelin Beetle Pact",
         None, None, Type.RITUAL, None, "41182875"),
    Card(697, 0, 0, 0, 50, "Garma Sword Oath",
         None, None, Type.RITUAL, None, "78577570"),
    Card(698, 0, 0, 0, 50, "Cosmo Queen's Prayer",
         None, None, Type.RITUAL, None, "4561679"),
    Card(699, 0, 0, 0, 10, "Revival of Skeleton Rider",
         None, None, Type.RITUAL, None, "31066283"),
    Card(700, 0, 0, 0, 10, "Fortress Whale's Oath",
         None, None, Type.RITUAL, None, "77454922"),
    Card(701, 6, 1950, 1850, 0, "Performance of Sword",
         GuardianStar.SUN, GuardianStar.SATURN, Type.WARRIOR, Attribute.EARTH, None),
    Card(702, 6, 2000, 1850, 0, "Hungry Burger",
         GuardianStar.MOON, GuardianStar.MARS, Type.WARRIOR, Attribute.DARK, None),
    Card(703, 8, 2750, 2500, 0, "Sengenjin",
         GuardianStar.JUPITER, GuardianStar.MERCURY, Type.BEAST_WARRIOR, Attribute.EARTH, None),
    Card(704, 7, 2050, 2500, 0, "Skull Guardian",
         GuardianStar.SUN, GuardianStar.JUPITER, Type.WARRIOR, Attribute.LIGHT, None),
    Card(705, 8, 2850, 2350, 0, "Tri-horned Dragon",
         GuardianStar.MOON, GuardianStar.JUPITER, Type.DRAGON, Attribute.DARK, None),
    Card(706, 7, 2350, 2400, 0, "Serpent Night Dragon",
         GuardianStar.MERCURY, GuardianStar.PLUTO, Type.DRAGON, Attribute.DARK, None),
    Card(707, 7, 2650, 2250, 999999, "Skull Knight",
         GuardianStar.MERCURY, GuardianStar.NEPTUNE, Type.SPELLCASTER, Attribute.DARK, "2504891"),
    Card(708, 8, 2900, 2450, 0, "Cosmo Queen",
         GuardianStar.VENUS, GuardianStar.URANUS, Type.SPELLCASTER, Attribute.DARK, None),
    Card(709, 7, 2450, 2000, 0, "Chakra",
         GuardianStar.MERCURY, GuardianStar.MOON, Type.FIEND, Attribute.DARK, None),
    Card(710, 8, 2550, 2500, 0, "Crab Turtle",
         GuardianStar.NEPTUNE, GuardianStar.JUPITER, Type.AQUA, Attribute.WATER, None),
    Card(711, 7, 2200, 2350, 999999, "Mikazukinoyaiba",
         GuardianStar.SUN, GuardianStar.URANUS, Type.DRAGON, Attribute.DARK, "38277918"),
    Card(712, 6, 1800, 2000, 999999, "Meteor Dragon",
         GuardianStar.URANUS, GuardianStar.NEPTUNE, Type.DRAGON, Attribute.EARTH, "64271667"),
    Card(713, 8, 3500, 2000, 0, "Meteor B. Dragon",
         GuardianStar.MARS, GuardianStar.SUN, Type.DRAGON, Attribute.FIRE, None),
    Card(714, 6, 2250, 1800, 999999, "Firewing Pegasus",
         GuardianStar.MARS, GuardianStar.MOON, Type.BEAST, Attribute.FIRE, "27054370"),
    Card(715, 7, 2000, 2350, 0, "Psycho-Puppet",
         GuardianStar.MOON, GuardianStar.SATURN, Type.FIEND, Attribute.DARK, None),
    Card(716, 7, 2550, 2150, 0, "Garma Sword",
         GuardianStar.MOON, GuardianStar.URANUS, Type.WARRIOR, Attribute.DARK, None),
    Card(717, 8, 2450, 2550, 0, "Javelin Beetle",
         GuardianStar.JUPITER, GuardianStar.URANUS, Type.INSECT, Attribute.EARTH, None),
    Card(718, 7, 2350, 2150, 0, "Fortress Whale",
         GuardianStar.NEPTUNE, GuardianStar.SATURN, Type.FISH, Attribute.WATER, None),
    Card(719, 6, 1900, 1850, 0, "Dokurorider",
         GuardianStar.URANUS, GuardianStar.MARS, Type.ZOMBIE, Attribute.DARK, None),
    Card(720, 6, 2000, 1800, 0, "Mask of Shine & Dark",
         GuardianStar.VENUS, GuardianStar.PLUTO, Type.SPELLCASTER, Attribute.DARK, None),
    Card(721, 0, 0, 0, 0, "Dark Magic Ritual",
         None, None, Type.RITUAL, None, None),
    Card(722, 8, 2800, 2600, 0, "Magician of Black Chaos",
         GuardianStar.MERCURY, GuardianStar.MARS, Type.SPELLCASTER, Attribute.DARK, None)
)

id_to_card = {card.id: card for card in all_cards}
name_to_card = {card.name: card for card in all_cards}
