from typing import Dict, List, NamedTuple, Optional

from BaseClasses import MultiWorld, Region, Entrance
from .Locations import KH1Location, location_table


class KH1RegionData(NamedTuple):
    locations: List[str]
    region_exits: Optional[List[str]]


def create_regions(kh1world):
    multiworld = kh1world.multiworld
    player     = kh1world.player
    options    = kh1world.options
    
    regions: Dict[str, KH1RegionData] = {
        "Menu":             KH1RegionData([], ["Awakening", "Levels", "World Map"]),
        "Awakening":        KH1RegionData([],   []),
        "Destiny Islands":  KH1RegionData([],   []),
        "Traverse Town":    KH1RegionData([],   []),
        "Wonderland":       KH1RegionData([],   []),
        "Olympus Coliseum": KH1RegionData([],   []),
        "Deep Jungle":      KH1RegionData([],   []),
        "Agrabah":          KH1RegionData([],   []),
        "Monstro":          KH1RegionData([],   []),
        "Atlantica":        KH1RegionData([],   []),
        "Halloween Town":   KH1RegionData([],   []),
        "Neverland":        KH1RegionData([],   []),
        "Hollow Bastion":   KH1RegionData([],   []),
        "End of the World": KH1RegionData([],   []),
        "100 Acre Wood":    KH1RegionData([],   []),
        "Levels":           KH1RegionData([],   []),
        "Homecoming":       KH1RegionData([],   []),
        "World Map":        KH1RegionData([],   ["Destiny Islands", "Traverse Town", 
                                         "Wonderland", "Olympus Coliseum", "Deep Jungle",
                                         "Agrabah", "Monstro", "Atlantica",
                                         "Halloween Town", "Neverland", "Hollow Bastion",
                                         "End of the World", "100 Acre Wood", "Homecoming"])
    }
    
    if not options.atlantica:
        del regions["Atlantica"]
        regions["World Map"].region_exits.remove("Atlantica")
    if not options.destiny_islands:
        del regions["Destiny Islands"]
        regions["World Map"].region_exits.remove("Destiny Islands")

    # Set up locations
    regions["Agrabah"].locations.append("Agrabah Aladdin's House Main Street Entrance Chest")
    regions["Agrabah"].locations.append("Agrabah Aladdin's House Plaza Entrance Chest")
    regions["Agrabah"].locations.append("Agrabah Alley Chest")
    regions["Agrabah"].locations.append("Agrabah Bazaar Across Windows Chest")
    regions["Agrabah"].locations.append("Agrabah Bazaar Blue Trinity")
    regions["Agrabah"].locations.append("Agrabah Bazaar High Corner Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Bottomless Hall Across Chasm Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Bottomless Hall Pillar Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Bottomless Hall Raised Platform Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Dark Chamber Abu Gem Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Dark Chamber Across from Relic Chamber Entrance Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Dark Chamber Bridge Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Dark Chamber Near Save Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Entrance Left Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Entrance Tall Tower Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Entrance White Trinity Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Hall High Left Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Hall Near Bottomless Hall Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Hidden Room Left Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Hidden Room Right Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Relic Chamber Jump from Stairs Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Relic Chamber Stairs Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Silent Chamber Blue Trinity Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Treasure Room Above Fire Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Treasure Room Across Platforms Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Treasure Room Large Treasure Pile Chest")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Treasure Room Red Trinity")
    regions["Agrabah"].locations.append("Agrabah Cave of Wonders Treasure Room Small Treasure Pile Chest")
    regions["Agrabah"].locations.append("Agrabah Defeat Jafar Blizzard Event")
    regions["Agrabah"].locations.append("Agrabah Defeat Jafar Genie Ansem's Report 1")
    regions["Agrabah"].locations.append("Agrabah Defeat Jafar Genie Fire Event")
    regions["Agrabah"].locations.append("Agrabah Defeat Pot Centipede Ray of Light Event")
    regions["Agrabah"].locations.append("Agrabah Main Street High Above Alley Entrance Chest")
    regions["Agrabah"].locations.append("Agrabah Main Street High Above Palace Gates Entrance Chest")
    regions["Agrabah"].locations.append("Agrabah Main Street Right Palace Entrance Chest")
    regions["Agrabah"].locations.append("Agrabah Palace Gates High Close to Palace Chest")
    regions["Agrabah"].locations.append("Agrabah Palace Gates High Opposite Palace Chest")
    regions["Agrabah"].locations.append("Agrabah Palace Gates Low Chest")
    regions["Agrabah"].locations.append("Agrabah Plaza By Storage Chest")
    regions["Agrabah"].locations.append("Agrabah Plaza Raised Terrace Chest")
    regions["Agrabah"].locations.append("Agrabah Plaza Top Corner Chest")
    regions["Agrabah"].locations.append("Agrabah Seal Keyhole Genie Event")
    regions["Agrabah"].locations.append("Agrabah Seal Keyhole Green Trinity Event")
    regions["Agrabah"].locations.append("Agrabah Seal Keyhole Three Wishes Event")
    regions["Agrabah"].locations.append("Agrabah Storage Behind Barrel Chest")
    regions["Agrabah"].locations.append("Agrabah Storage Green Trinity Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Bamboo Thicket Save Gorillas")
    regions["Deep Jungle"].locations.append("Deep Jungle Camp Blue Trinity Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Camp Ether Experiment")
    regions["Deep Jungle"].locations.append("Deep Jungle Camp Hi-Potion Experiment")
    regions["Deep Jungle"].locations.append("Deep Jungle Camp Replication Experiment")
    regions["Deep Jungle"].locations.append("Deep Jungle Camp Save Gorillas")
    regions["Deep Jungle"].locations.append("Deep Jungle Cavern of Hearts Navi-G Piece Event")
    regions["Deep Jungle"].locations.append("Deep Jungle Cavern of Hearts White Trinity Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Cliff Right Cliff Left Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Cliff Right Cliff Right Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Cliff Save Gorillas")
    regions["Deep Jungle"].locations.append("Deep Jungle Climbing Trees Blue Trinity Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Climbing Trees Save Gorillas")
    regions["Deep Jungle"].locations.append("Deep Jungle Defeat Clayton Cure Event")
    regions["Deep Jungle"].locations.append("Deep Jungle Defeat Sabor White Fang Event")
    regions["Deep Jungle"].locations.append("Deep Jungle Hippo's Lagoon Center Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Hippo's Lagoon Left Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Hippo's Lagoon Right Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Seal Keyhole Jungle King Event")
    regions["Deep Jungle"].locations.append("Deep Jungle Seal Keyhole Red Trinity Event")
    regions["Deep Jungle"].locations.append("Deep Jungle Tent Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Tent Protect-G Event")
    regions["Deep Jungle"].locations.append("Deep Jungle Treetop Green Trinity")
    regions["Deep Jungle"].locations.append("Deep Jungle Tree House Beneath Tree House Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Tree House Rooftop Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Tree House Save Gorillas")
    regions["Deep Jungle"].locations.append("Deep Jungle Tree House Suspended Boat Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Tunnel Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Vines 2 Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Vines Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Waterfall Cavern High Middle Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Waterfall Cavern High Wall Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Waterfall Cavern Low Chest")
    regions["Deep Jungle"].locations.append("Deep Jungle Waterfall Cavern Middle Chest")
    regions["End of the World"].locations.append("End of the World Defeat Chernabog Superglide Event")
    regions["End of the World"].locations.append("End of the World Final Dimension 10th Chest")
    regions["End of the World"].locations.append("End of the World Final Dimension 1st Chest")
    regions["End of the World"].locations.append("End of the World Final Dimension 2nd Chest")
    regions["End of the World"].locations.append("End of the World Final Dimension 3rd Chest")
    regions["End of the World"].locations.append("End of the World Final Dimension 4th Chest")
    regions["End of the World"].locations.append("End of the World Final Dimension 5th Chest")
    regions["End of the World"].locations.append("End of the World Final Dimension 6th Chest")
    regions["End of the World"].locations.append("End of the World Final Dimension 7th Chest")
    regions["End of the World"].locations.append("End of the World Final Dimension 8th Chest")
    regions["End of the World"].locations.append("End of the World Final Dimension 9th Chest")
    regions["End of the World"].locations.append("End of the World Final Rest Chest")
    regions["End of the World"].locations.append("End of the World Giant Crevasse 1st Chest")
    regions["End of the World"].locations.append("End of the World Giant Crevasse 2nd Chest")
    regions["End of the World"].locations.append("End of the World Giant Crevasse 3rd Chest")
    regions["End of the World"].locations.append("End of the World Giant Crevasse 4th Chest")
    regions["End of the World"].locations.append("End of the World Giant Crevasse 5th Chest")
    regions["End of the World"].locations.append("End of the World World Terminus 100 Acre Wood Chest")
    regions["End of the World"].locations.append("End of the World World Terminus Agrabah Chest")
    regions["End of the World"].locations.append("End of the World World Terminus Atlantica Chest")
    regions["End of the World"].locations.append("End of the World World Terminus Deep Jungle Chest")
    regions["End of the World"].locations.append("End of the World World Terminus Halloween Town Chest")
    regions["End of the World"].locations.append("End of the World World Terminus Hollow Bastion Chest")
    regions["End of the World"].locations.append("End of the World World Terminus Neverland Chest")
    regions["End of the World"].locations.append("End of the World World Terminus Olympus Coliseum Chest")
    regions["End of the World"].locations.append("End of the World World Terminus Traverse Town Chest")
    regions["End of the World"].locations.append("End of the World World Terminus Wonderland Chest")
    regions["Halloween Town"].locations.append("Halloween Town Boneyard Tombstone Puzzle Chest")
    regions["Halloween Town"].locations.append("Halloween Town Bridge Left of Gate Chest")
    regions["Halloween Town"].locations.append("Halloween Town Bridge Right of Gate Chest")
    regions["Halloween Town"].locations.append("Halloween Town Bridge Under Bridge")
    regions["Halloween Town"].locations.append("Halloween Town Cemetery Behind Grave Chest")
    regions["Halloween Town"].locations.append("Halloween Town Cemetery Between Graves Chest")
    regions["Halloween Town"].locations.append("Halloween Town Cemetery By Cat Shape Chest")
    regions["Halloween Town"].locations.append("Halloween Town Cemetery By Striped Grave Chest")
    regions["Halloween Town"].locations.append("Halloween Town Defeat Oogie Boogie Ansem's Report 7")
    regions["Halloween Town"].locations.append("Halloween Town Defeat Oogie Boogie Holy Circlet Event")
    regions["Halloween Town"].locations.append("Halloween Town Defeat Oogie's Manor Gravity Event")
    regions["Halloween Town"].locations.append("Halloween Town Graveyard Forget-Me-Not Event")
    regions["Halloween Town"].locations.append("Halloween Town Guillotine Square High Tower Chest")
    regions["Halloween Town"].locations.append("Halloween Town Guillotine Square Pumpkin Structure Left Chest")
    regions["Halloween Town"].locations.append("Halloween Town Guillotine Square Pumpkin Structure Right Chest")
    regions["Halloween Town"].locations.append("Halloween Town Guillotine Square Ring Jack's Doorbell 3 Times")
    regions["Halloween Town"].locations.append("Halloween Town Guillotine Square Under Jack's House Stairs Chest")
    regions["Halloween Town"].locations.append("Halloween Town Lab Torn Page")
    regions["Halloween Town"].locations.append("Halloween Town Moonlight Hill White Trinity Chest")
    regions["Halloween Town"].locations.append("Halloween Town Oogie's Manor Entrance Steps Chest")
    regions["Halloween Town"].locations.append("Halloween Town Oogie's Manor Grounds Red Trinity Chest")
    regions["Halloween Town"].locations.append("Halloween Town Oogie's Manor Hollow Chest")
    regions["Halloween Town"].locations.append("Halloween Town Oogie's Manor Inside Entrance Chest")
    regions["Halloween Town"].locations.append("Halloween Town Oogie's Manor Lower Iron Cage Chest")
    regions["Halloween Town"].locations.append("Halloween Town Oogie's Manor Upper Iron Cage Chest")
    regions["Halloween Town"].locations.append("Halloween Town Seal Keyhole Pumpkinhead Event")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Base Level Bubble Under the Wall Platform Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Base Level Near Crystal Switch Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Base Level Platform Near Entrance Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Castle Gates Freestanding Pillar Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Castle Gates Gravity Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Castle Gates High Pillar Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Defeat Behemoth Omega Arts Event")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Defeat Dragon Maleficent Fireglow Event")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Defeat Maleficent Ansem's Report 5")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Defeat Maleficent Donald Cheer Event")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Defeat Riku I White Trinity Event")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Defeat Riku II Ragnarok Event")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Dungeon Blue Trinity")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Dungeon By Candles Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Dungeon Corner Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Entrance Hall Emblem Piece (Chest)")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Entrance Hall Emblem Piece (Flame)")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Entrance Hall Emblem Piece (Fountain)")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Entrance Hall Emblem Piece (Statue)")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Entrance Hall Left of Emblem Door Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Grand Hall Left of Gate Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Grand Hall Oblivion Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Grand Hall Steps Right Side Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Great Crest After Battle Platform Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Great Crest Blue Trinity")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Great Crest Lower Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion High Tower 1st Gravity Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion High Tower 2nd Gravity Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion High Tower Above Sliding Blocks Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Library 1st Floor Turn the Carousel Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Library 2nd Floor Turn the Carousel 1st Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Library 2nd Floor Turn the Carousel 2nd Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Library Speak to Aerith Cure")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Library Speak to Belle Divine Rose")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Library Top of Bookshelf Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Library Top of Bookshelf Turn the Carousel Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Lift Stop from Waterway Examine Node")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Lift Stop Heartless Sigil Door Gravity Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Lift Stop Library Node After High Tower Switch Gravity Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Lift Stop Library Node Gravity Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Lift Stop Outside Library Gravity Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Lift Stop Under High Tower Sliding Blocks Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Rising Falls Floating Platform Near Bubble Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Rising Falls Floating Platform Near Save Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Rising Falls High Platform Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Rising Falls Under Water 1st Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Rising Falls Under Water 2nd Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Rising Falls Water's Surface Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Rising Falls White Trinity Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Speak to Princesses Fire Event")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Speak with Aerith Ansem's Report 10")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Speak with Aerith Ansem's Report 2")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Speak with Aerith Ansem's Report 4")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Speak with Aerith Ansem's Report 6")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Waterway Blizzard on Bubble Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Waterway Near Save Chest")
    regions["Hollow Bastion"].locations.append("Hollow Bastion Waterway Unlock Passage from Base Level Chest")
    regions["Monstro"].locations.append("Monstro Chamber 2 Ground Chest")
    regions["Monstro"].locations.append("Monstro Chamber 2 Platform Chest")
    regions["Monstro"].locations.append("Monstro Chamber 3 Ground Chest")
    regions["Monstro"].locations.append("Monstro Chamber 3 Near Chamber 6 Entrance Chest")
    regions["Monstro"].locations.append("Monstro Chamber 3 Platform Above Chamber 2 Entrance Chest")
    regions["Monstro"].locations.append("Monstro Chamber 3 Platform Near Chamber 6 Entrance Chest")
    regions["Monstro"].locations.append("Monstro Chamber 5 Atop Barrel Chest")
    regions["Monstro"].locations.append("Monstro Chamber 5 Blue Trinity")
    regions["Monstro"].locations.append("Monstro Chamber 5 Low 1st Chest")
    regions["Monstro"].locations.append("Monstro Chamber 5 Low 2nd Chest")
    regions["Monstro"].locations.append("Monstro Chamber 5 Platform Chest")
    regions["Monstro"].locations.append("Monstro Chamber 6 Low Chest")
    regions["Monstro"].locations.append("Monstro Chamber 6 Other Platform Chest")
    regions["Monstro"].locations.append("Monstro Chamber 6 Platform Near Chamber 5 Entrance Chest")
    regions["Monstro"].locations.append("Monstro Chamber 6 Raised Area Near Chamber 1 Entrance Chest")
    regions["Monstro"].locations.append("Monstro Chamber 6 White Trinity Chest")
    regions["Monstro"].locations.append("Monstro Defeat Parasite Cage I Goofy Cheer Event")
    regions["Monstro"].locations.append("Monstro Defeat Parasite Cage II Stop Event")
    regions["Monstro"].locations.append("Monstro Mouth Blue Trinity")
    regions["Monstro"].locations.append("Monstro Mouth Boat Deck Chest")
    regions["Monstro"].locations.append("Monstro Mouth Green Trinity Top of Boat Chest")
    regions["Monstro"].locations.append("Monstro Mouth High Platform Across from Boat Chest")
    regions["Monstro"].locations.append("Monstro Mouth High Platform Boat Side Chest")
    regions["Monstro"].locations.append("Monstro Mouth High Platform Near Teeth Chest")
    regions["Monstro"].locations.append("Monstro Mouth Near Ship Chest")
    regions["Monstro"].locations.append("Monstro Throat Blue Trinity")
    regions["Neverland"].locations.append("Neverland Cabin Chest")
    regions["Neverland"].locations.append("Neverland Captain's Cabin Chest")
    regions["Neverland"].locations.append("Neverland Clock Tower 01:00 Door")
    regions["Neverland"].locations.append("Neverland Clock Tower 02:00 Door")
    regions["Neverland"].locations.append("Neverland Clock Tower 03:00 Door")
    regions["Neverland"].locations.append("Neverland Clock Tower 04:00 Door")
    regions["Neverland"].locations.append("Neverland Clock Tower 05:00 Door")
    regions["Neverland"].locations.append("Neverland Clock Tower 06:00 Door")
    regions["Neverland"].locations.append("Neverland Clock Tower 07:00 Door")
    regions["Neverland"].locations.append("Neverland Clock Tower 08:00 Door")
    regions["Neverland"].locations.append("Neverland Clock Tower 09:00 Door")
    regions["Neverland"].locations.append("Neverland Clock Tower 10:00 Door")
    regions["Neverland"].locations.append("Neverland Clock Tower 11:00 Door")
    regions["Neverland"].locations.append("Neverland Clock Tower 12:00 Door")
    regions["Neverland"].locations.append("Neverland Clock Tower Chest")
    regions["Neverland"].locations.append("Neverland Defeat Anti Sora Raven's Claw Event")
    regions["Neverland"].locations.append("Neverland Defeat Captain Hook Ars Arcanum Event")
    regions["Neverland"].locations.append("Neverland Defeat Hook Ansem's Report 9")
    regions["Neverland"].locations.append("Neverland Encounter Hook Cure Event")
    regions["Neverland"].locations.append("Neverland Galley Chest")
    regions["Neverland"].locations.append("Neverland Hold Aero Chest")
    regions["Neverland"].locations.append("Neverland Hold Flight 1st Chest")
    regions["Neverland"].locations.append("Neverland Hold Flight 2nd Chest")
    regions["Neverland"].locations.append("Neverland Hold Yellow Trinity Green Chest")
    regions["Neverland"].locations.append("Neverland Hold Yellow Trinity Left Blue Chest")
    regions["Neverland"].locations.append("Neverland Hold Yellow Trinity Right Blue Chest")
    regions["Neverland"].locations.append("Neverland Pirate Ship Crows Nest Chest")
    regions["Neverland"].locations.append("Neverland Pirate Ship Deck White Trinity Chest")
    regions["Neverland"].locations.append("Neverland Seal Keyhole Fairy Harp Event")
    regions["Neverland"].locations.append("Neverland Seal Keyhole Glide Event")
    regions["Neverland"].locations.append("Neverland Seal Keyhole Navi-G Piece Event")
    regions["Neverland"].locations.append("Neverland Seal Keyhole Tinker Bell Event")
    regions["Olympus Coliseum"].locations.append("Olympus Coliseum Clear Phil's Training Thunder Event")
    regions["Olympus Coliseum"].locations.append("Olympus Coliseum Cloud Sonic Blade Event")
    regions["Olympus Coliseum"].locations.append("Olympus Coliseum Coliseum Gates Blizzaga Chest")
    regions["Olympus Coliseum"].locations.append("Olympus Coliseum Coliseum Gates Blizzara Chest")
    regions["Olympus Coliseum"].locations.append("Olympus Coliseum Coliseum Gates Entry Pass Event")
    regions["Olympus Coliseum"].locations.append("Olympus Coliseum Coliseum Gates Green Trinity")
    regions["Olympus Coliseum"].locations.append("Olympus Coliseum Coliseum Gates Hero's License Event")
    regions["Olympus Coliseum"].locations.append("Olympus Coliseum Coliseum Gates Left Behind Columns Chest")
    regions["Olympus Coliseum"].locations.append("Olympus Coliseum Coliseum Gates Left Blue Trinity Chest")
    regions["Olympus Coliseum"].locations.append("Olympus Coliseum Coliseum Gates Right Blue Trinity Chest")
    regions["Olympus Coliseum"].locations.append("Olympus Coliseum Coliseum Gates White Trinity Chest")
    regions["Olympus Coliseum"].locations.append("Olympus Coliseum Defeat Cerberus Inferno Band Event")
    regions["Traverse Town"].locations.append("Traverse Town 1st District Accessory Shop Roof Chest")
    #regions["Traverse Town"].locations.append("Traverse Town 1st District Aerith Gift")
    regions["Traverse Town"].locations.append("Traverse Town 1st District Blue Trinity Balcony Chest")
    regions["Traverse Town"].locations.append("Traverse Town 1st District Blue Trinity by Exit Door")
    regions["Traverse Town"].locations.append("Traverse Town 1st District Candle Puzzle Chest")
    regions["Traverse Town"].locations.append("Traverse Town 1st District Leon Gift")
    regions["Traverse Town"].locations.append("Traverse Town 1st District Safe Postcard")
   #regions["Traverse Town"].locations.append("Traverse Town 1st District Speak with Cid Event")
    regions["Traverse Town"].locations.append("Traverse Town 2nd District Boots and Shoes Awning Chest")
    regions["Traverse Town"].locations.append("Traverse Town 2nd District Gizmo Shop Facade Chest")
    regions["Traverse Town"].locations.append("Traverse Town 2nd District Rooftop Chest")
    regions["Traverse Town"].locations.append("Traverse Town 3rd District Balcony Postcard")
    regions["Traverse Town"].locations.append("Traverse Town 3rd District Blue Trinity")
    regions["Traverse Town"].locations.append("Traverse Town Accessory Shop Chest")
    regions["Traverse Town"].locations.append("Traverse Town Alleyway Balcony Chest")
    regions["Traverse Town"].locations.append("Traverse Town Alleyway Behind Crates Chest")
    regions["Traverse Town"].locations.append("Traverse Town Alleyway Blue Room Awning Chest")
    regions["Traverse Town"].locations.append("Traverse Town Alleyway Corner Chest")
    regions["Traverse Town"].locations.append("Traverse Town Defeat Guard Armor Blue Trinity Event")
    regions["Traverse Town"].locations.append("Traverse Town Defeat Guard Armor Brave Warrior Event")
    regions["Traverse Town"].locations.append("Traverse Town Defeat Guard Armor Dodge Roll Event")
    regions["Traverse Town"].locations.append("Traverse Town Defeat Guard Armor Fire Event")
    regions["Traverse Town"].locations.append("Traverse Town Defeat Opposite Armor Aero Event")
    regions["Traverse Town"].locations.append("Traverse Town Defeat Opposite Armor Navi-G Piece Event")
    regions["Traverse Town"].locations.append("Traverse Town Geppetto's House Chest")
    regions["Traverse Town"].locations.append("Traverse Town Geppetto's House Geppetto All Summons Reward")
    regions["Traverse Town"].locations.append("Traverse Town Geppetto's House Geppetto Reward 1")
    regions["Traverse Town"].locations.append("Traverse Town Geppetto's House Geppetto Reward 2")
    regions["Traverse Town"].locations.append("Traverse Town Geppetto's House Geppetto Reward 3")
    regions["Traverse Town"].locations.append("Traverse Town Geppetto's House Geppetto Reward 4")
    regions["Traverse Town"].locations.append("Traverse Town Geppetto's House Geppetto Reward 5")
    regions["Traverse Town"].locations.append("Traverse Town Geppetto's House Postcard")
    regions["Traverse Town"].locations.append("Traverse Town Geppetto's House Talk to Pinocchio")
    regions["Traverse Town"].locations.append("Traverse Town Gizmo Shop Postcard 1")
    regions["Traverse Town"].locations.append("Traverse Town Gizmo Shop Postcard 2")
    regions["Traverse Town"].locations.append("Traverse Town Green Room Clock Puzzle Chest")
    regions["Traverse Town"].locations.append("Traverse Town Green Room Table Chest")
    regions["Traverse Town"].locations.append("Traverse Town Item Shop Postcard")
    regions["Traverse Town"].locations.append("Traverse Town Item Workshop Left Chest")
    regions["Traverse Town"].locations.append("Traverse Town Item Workshop Postcard")
    regions["Traverse Town"].locations.append("Traverse Town Item Workshop Right Chest")
    regions["Traverse Town"].locations.append("Traverse Town Kairi Secret Waterway Oathkeeper Event")
    regions["Traverse Town"].locations.append("Traverse Town Leon Secret Waterway Earthshine Event")
    regions["Traverse Town"].locations.append("Traverse Town Magician's Study Blue Trinity")
    regions["Traverse Town"].locations.append("Traverse Town Magician's Study Obtained All Arts Items")
    regions["Traverse Town"].locations.append("Traverse Town Magician's Study Obtained All LV1 Magic")
    regions["Traverse Town"].locations.append("Traverse Town Magician's Study Obtained All LV3 Magic")
    regions["Traverse Town"].locations.append("Traverse Town Mail Postcard 01 Event")
    regions["Traverse Town"].locations.append("Traverse Town Mail Postcard 02 Event")
    regions["Traverse Town"].locations.append("Traverse Town Mail Postcard 03 Event")
    regions["Traverse Town"].locations.append("Traverse Town Mail Postcard 04 Event")
    regions["Traverse Town"].locations.append("Traverse Town Mail Postcard 05 Event")
    regions["Traverse Town"].locations.append("Traverse Town Mail Postcard 06 Event")
    regions["Traverse Town"].locations.append("Traverse Town Mail Postcard 07 Event")
    regions["Traverse Town"].locations.append("Traverse Town Mail Postcard 08 Event")
    regions["Traverse Town"].locations.append("Traverse Town Mail Postcard 09 Event")
    regions["Traverse Town"].locations.append("Traverse Town Mail Postcard 10 Event")
    regions["Traverse Town"].locations.append("Traverse Town Mystical House Glide Chest")
    regions["Traverse Town"].locations.append("Traverse Town Mystical House Yellow Trinity Chest")
    regions["Traverse Town"].locations.append("Traverse Town Piano Room Return 10 Puppies")
    regions["Traverse Town"].locations.append("Traverse Town Piano Room Return 20 Puppies")
    regions["Traverse Town"].locations.append("Traverse Town Piano Room Return 30 Puppies")
    regions["Traverse Town"].locations.append("Traverse Town Piano Room Return 40 Puppies")
    regions["Traverse Town"].locations.append("Traverse Town Piano Room Return 50 Puppies Reward 1")
    regions["Traverse Town"].locations.append("Traverse Town Piano Room Return 50 Puppies Reward 2")
    regions["Traverse Town"].locations.append("Traverse Town Piano Room Return 60 Puppies")
    regions["Traverse Town"].locations.append("Traverse Town Piano Room Return 70 Puppies")
    regions["Traverse Town"].locations.append("Traverse Town Piano Room Return 80 Puppies")
    regions["Traverse Town"].locations.append("Traverse Town Piano Room Return 90 Puppies")
    regions["Traverse Town"].locations.append("Traverse Town Piano Room Return 99 Puppies Reward 1")
    regions["Traverse Town"].locations.append("Traverse Town Piano Room Return 99 Puppies Reward 2")
    regions["Traverse Town"].locations.append("Traverse Town Red Room Chest")
    regions["Traverse Town"].locations.append("Traverse Town Secret Waterway Navi Gummi Event")
    regions["Traverse Town"].locations.append("Traverse Town Secret Waterway Near Stairs Chest")
    regions["Traverse Town"].locations.append("Traverse Town Secret Waterway White Trinity Chest")
    regions["Traverse Town"].locations.append("Traverse Town Synth 15 Items")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 01")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 02")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 03")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 04")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 05")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 06")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 07")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 08")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 09")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 10")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 11")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 12")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 13")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 14")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 15")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 16")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 17")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 18")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 19")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 20")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 21")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 22")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 23")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 24")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 25")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 26")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 27")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 28")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 29")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 30")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 31")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 32")
    regions["Traverse Town"].locations.append("Traverse Town Synth Item 33")
    regions["Wonderland"].locations.append("Wonderland Bizarre Room Examine Flower Pot")
    regions["Wonderland"].locations.append("Wonderland Bizarre Room Green Trinity Chest")
    regions["Wonderland"].locations.append("Wonderland Bizarre Room Lamp Chest")
    regions["Wonderland"].locations.append("Wonderland Bizarre Room Navi-G Piece Event")
    regions["Wonderland"].locations.append("Wonderland Bizarre Room Read Book")
    regions["Wonderland"].locations.append("Wonderland Defeat Trickmaster Blizzard Event")
    regions["Wonderland"].locations.append("Wonderland Defeat Trickmaster Ifrit's Horn Event")
    regions["Wonderland"].locations.append("Wonderland Lotus Forest Blue Trinity in Alcove")
    regions["Wonderland"].locations.append("Wonderland Lotus Forest Blue Trinity by Moving Boulder")
    regions["Wonderland"].locations.append("Wonderland Lotus Forest Corner Chest")
    regions["Wonderland"].locations.append("Wonderland Lotus Forest Glide Chest")
    regions["Wonderland"].locations.append("Wonderland Lotus Forest Nut Chest")
    regions["Wonderland"].locations.append("Wonderland Lotus Forest Red Flower Raise Lily Pads")
    regions["Wonderland"].locations.append("Wonderland Lotus Forest Red Flowers on the Main Path")
    regions["Wonderland"].locations.append("Wonderland Lotus Forest Through the Painting Thunder Plant Chest")
    regions["Wonderland"].locations.append("Wonderland Lotus Forest Through the Painting White Trinity Chest")
    regions["Wonderland"].locations.append("Wonderland Lotus Forest Thunder Plant Chest")
    regions["Wonderland"].locations.append("Wonderland Lotus Forest Yellow Elixir Flower Through Painting")
    regions["Wonderland"].locations.append("Wonderland Lotus Forest Yellow Flowers in Middle Clearing and Through Painting")
    regions["Wonderland"].locations.append("Wonderland Queen's Castle Hedge Left Red Chest")
    regions["Wonderland"].locations.append("Wonderland Queen's Castle Hedge Right Blue Chest")
    regions["Wonderland"].locations.append("Wonderland Queen's Castle Hedge Right Red Chest")
    regions["Wonderland"].locations.append("Wonderland Rabbit Hole Defeat Heartless 1 Chest")
    regions["Wonderland"].locations.append("Wonderland Rabbit Hole Defeat Heartless 2 Chest")
    regions["Wonderland"].locations.append("Wonderland Rabbit Hole Defeat Heartless 3 Chest")
    regions["Wonderland"].locations.append("Wonderland Rabbit Hole Green Trinity Chest")
    regions["Wonderland"].locations.append("Wonderland Tea Party Garden Above Lotus Forest Entrance 1st Chest")
    regions["Wonderland"].locations.append("Wonderland Tea Party Garden Above Lotus Forest Entrance 2nd Chest")
    regions["Wonderland"].locations.append("Wonderland Tea Party Garden Across From Bizarre Room Entrance Chest")
    regions["Wonderland"].locations.append("Wonderland Tea Party Garden Bear and Clock Puzzle Chest")
    regions["Wonderland"].locations.append("Wonderland Tea Party Garden Left Cushioned Chair")
    regions["Wonderland"].locations.append("Wonderland Tea Party Garden Left Gray Chair")
    regions["Wonderland"].locations.append("Wonderland Tea Party Garden Left Pink Chair")
    regions["Wonderland"].locations.append("Wonderland Tea Party Garden Right Brown Chair")
    regions["Wonderland"].locations.append("Wonderland Tea Party Garden Right Yellow Chair")
    if options.hundred_acre_wood:
        regions["100 Acre Wood"].locations.append("100 Acre Wood Meadow Inside Log Chest")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Bouncing Spot Left Cliff Chest")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Bouncing Spot Right Tree Alcove Chest")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Bouncing Spot Under Giant Pot Chest")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Bouncing Spot Turn in Rare Nut 1")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Bouncing Spot Turn in Rare Nut 2")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Bouncing Spot Turn in Rare Nut 3")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Bouncing Spot Turn in Rare Nut 4")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Bouncing Spot Turn in Rare Nut 5")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Pooh's House Owl Cheer")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Convert Torn Page 1")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Convert Torn Page 2")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Convert Torn Page 3")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Convert Torn Page 4")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Convert Torn Page 5")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Pooh's House Start Fire")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Pooh's Room Cabinet")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Pooh's Room Chimney")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Bouncing Spot Break Log")
        regions["100 Acre Wood"].locations.append("100 Acre Wood Bouncing Spot Fall Through Top of Tree Next to Pooh")
    if options.atlantica:
        regions["Atlantica"].locations.append("Atlantica Sunken Ship In Flipped Boat Chest")
        regions["Atlantica"].locations.append("Atlantica Sunken Ship Seabed Chest")
        regions["Atlantica"].locations.append("Atlantica Sunken Ship Inside Ship Chest")
        regions["Atlantica"].locations.append("Atlantica Ariel's Grotto High Chest")
        regions["Atlantica"].locations.append("Atlantica Ariel's Grotto Middle Chest")
        regions["Atlantica"].locations.append("Atlantica Ariel's Grotto Low Chest")
        regions["Atlantica"].locations.append("Atlantica Ursula's Lair Use Fire on Urchin Chest")
        regions["Atlantica"].locations.append("Atlantica Undersea Gorge Jammed by Ariel's Grotto Chest")
        regions["Atlantica"].locations.append("Atlantica Triton's Palace White Trinity Chest")
        regions["Atlantica"].locations.append("Atlantica Defeat Ursula I Mermaid Kick Event")
        regions["Atlantica"].locations.append("Atlantica Defeat Ursula II Thunder Event")
        regions["Atlantica"].locations.append("Atlantica Seal Keyhole Crabclaw Event")
        regions["Atlantica"].locations.append("Atlantica Undersea Gorge Blizzard Clam")
        regions["Atlantica"].locations.append("Atlantica Undersea Gorge Ocean Floor Clam")
        regions["Atlantica"].locations.append("Atlantica Undersea Valley Higher Cave Clam")
        regions["Atlantica"].locations.append("Atlantica Undersea Valley Lower Cave Clam")
        regions["Atlantica"].locations.append("Atlantica Undersea Valley Fire Clam")
        regions["Atlantica"].locations.append("Atlantica Undersea Valley Wall Clam")
        regions["Atlantica"].locations.append("Atlantica Undersea Valley Pillar Clam")
        regions["Atlantica"].locations.append("Atlantica Undersea Valley Ocean Floor Clam")
        regions["Atlantica"].locations.append("Atlantica Triton's Palace Thunder Clam")
        regions["Atlantica"].locations.append("Atlantica Triton's Palace Wall Right Clam")
        regions["Atlantica"].locations.append("Atlantica Triton's Palace Near Path Clam")
        regions["Atlantica"].locations.append("Atlantica Triton's Palace Wall Left Clam")
        regions["Atlantica"].locations.append("Atlantica Cavern Nook Clam")
        regions["Atlantica"].locations.append("Atlantica Below Deck Clam")
        regions["Atlantica"].locations.append("Atlantica Undersea Garden Clam")
        regions["Atlantica"].locations.append("Atlantica Undersea Cave Clam")
        regions["Atlantica"].locations.append("Atlantica Sunken Ship Crystal Trident Event")
        regions["Atlantica"].locations.append("Atlantica Defeat Ursula II Ansem's Report 3")
    if options.cups.current_key != "off":
        regions["Olympus Coliseum"].locations.append("Complete Phil Cup")
        regions["Olympus Coliseum"].locations.append("Complete Phil Cup Solo")
        regions["Olympus Coliseum"].locations.append("Complete Phil Cup Time Trial")
        regions["Olympus Coliseum"].locations.append("Complete Pegasus Cup")
        regions["Olympus Coliseum"].locations.append("Complete Pegasus Cup Solo")
        regions["Olympus Coliseum"].locations.append("Complete Pegasus Cup Time Trial")
        regions["Olympus Coliseum"].locations.append("Complete Hercules Cup")
        regions["Olympus Coliseum"].locations.append("Complete Hercules Cup Solo")
        regions["Olympus Coliseum"].locations.append("Complete Hercules Cup Time Trial")
        regions["Olympus Coliseum"].locations.append("Hercules Cup Defeat Cloud Event")
        regions["Olympus Coliseum"].locations.append("Hercules Cup Yellow Trinity Event")
        regions["Olympus Coliseum"].locations.append("Olympus Coliseum Olympia Chest")
        if options.cups.current_key == "hades_cup":
            regions["Olympus Coliseum"].locations.append("Complete Hades Cup")
            regions["Olympus Coliseum"].locations.append("Complete Hades Cup Solo")
            regions["Olympus Coliseum"].locations.append("Complete Hades Cup Time Trial")
            regions["Olympus Coliseum"].locations.append("Hades Cup Defeat Cloud and Leon Event")
            regions["Olympus Coliseum"].locations.append("Hades Cup Defeat Yuffie Event")
            regions["Olympus Coliseum"].locations.append("Hades Cup Defeat Cerberus Event")
            regions["Olympus Coliseum"].locations.append("Hades Cup Defeat Behemoth Event")
            regions["Olympus Coliseum"].locations.append("Hades Cup Defeat Hades Event")
            regions["Olympus Coliseum"].locations.append("Olympus Coliseum Defeat Hades Ansem's Report 8")
            regions["Olympus Coliseum"].locations.append("Olympus Coliseum Gates Purple Jar After Defeating Hades")
        if options.cups.current_key == "hades_cup" and options.super_bosses:
            regions["Olympus Coliseum"].locations.append("Olympus Coliseum Defeat Ice Titan Diamond Dust Event")
    if options.super_bosses:
        regions["Neverland"].locations.append("Neverland Defeat Phantom Stop Event")
        regions["Agrabah"].locations.append("Agrabah Defeat Kurt Zisa Zantetsuken Event")
        regions["Agrabah"].locations.append("Agrabah Defeat Kurt Zisa Ansem's Report 11")
    if options.super_bosses or options.final_rest_door_key.current_key == "sephiroth":
        regions["Olympus Coliseum"].locations.append("Olympus Coliseum Defeat Sephiroth Ansem's Report 12")
        regions["Olympus Coliseum"].locations.append("Olympus Coliseum Defeat Sephiroth One-Winged Angel Event")
    if options.super_bosses or options.final_rest_door_key.current_key == "unknown":
        regions["Hollow Bastion"].locations.append("Hollow Bastion Defeat Unknown Ansem's Report 13")
        regions["Hollow Bastion"].locations.append("Hollow Bastion Defeat Unknown EXP Necklace Event")
    if options.jungle_slider:
        regions["Deep Jungle"].locations.append("Deep Jungle Jungle Slider 10 Fruits")
        regions["Deep Jungle"].locations.append("Deep Jungle Jungle Slider 20 Fruits")
        regions["Deep Jungle"].locations.append("Deep Jungle Jungle Slider 30 Fruits")
        regions["Deep Jungle"].locations.append("Deep Jungle Jungle Slider 40 Fruits")
        regions["Deep Jungle"].locations.append("Deep Jungle Jungle Slider 50 Fruits")
    for i in range(1,options.level_checks+1):
        regions["Levels"].locations.append("Level " + str(i+1).rjust(3, '0') + " (Slot 1)")
        if i+1 in kh1world.get_slot_2_levels():
            regions["Levels"].locations.append("Level " + str(i+1).rjust(3, '0') + " (Slot 2)")
    if options.destiny_islands:
        regions["Destiny Islands"].locations.append("Destiny Islands Seashore Capture Fish 1 (Day 2)")
        regions["Destiny Islands"].locations.append("Destiny Islands Seashore Capture Fish 2 (Day 2)")
        regions["Destiny Islands"].locations.append("Destiny Islands Seashore Capture Fish 3 (Day 2)")
        regions["Destiny Islands"].locations.append("Destiny Islands Seashore Gather Seagull Egg (Day 2)")
        regions["Destiny Islands"].locations.append("Destiny Islands Seashore Log on Riku's Island (Day 1)")
        regions["Destiny Islands"].locations.append("Destiny Islands Seashore Log under Bridge (Day 1)")
        regions["Destiny Islands"].locations.append("Destiny Islands Seashore Gather Cloth (Day 1)")
        regions["Destiny Islands"].locations.append("Destiny Islands Seashore Gather Rope (Day 1)")
       #regions["Destiny Islands"].locations.append("Destiny Islands Seashore Deliver Kairi Items (Day 1)")
        regions["Destiny Islands"].locations.append("Destiny Islands Secret Place Gather Mushroom (Day 2)")
        regions["Destiny Islands"].locations.append("Destiny Islands Cove Gather Mushroom Near Zip Line (Day 2)")
        regions["Destiny Islands"].locations.append("Destiny Islands Cove Gather Mushroom in Small Cave (Day 2)")
        regions["Destiny Islands"].locations.append("Destiny Islands Cove Talk to Kairi (Day 2)")
        regions["Destiny Islands"].locations.append("Destiny Islands Gather Drinking Water (Day 2)")
       #regions["Destiny Islands"].locations.append("Destiny Islands Cove Deliver Kairi Items (Day 2)")
        regions["Destiny Islands"].locations.append("Destiny Islands Chest")
    regions["Homecoming"].locations.append("Final Ansem")
    
    for location in kh1world.get_starting_accessory_locations():
        regions[location_table[location].category].locations.append(location)

    # Set up the regions correctly.
    for name, data in regions.items():
        multiworld.regions.append(create_region(multiworld, player, name, data))

def connect_entrances(kh1world):
    multiworld = kh1world.multiworld
    player     = kh1world.player
    options    = kh1world.options
    
    multiworld.get_entrance("Awakening", player).connect(multiworld.get_region("Awakening", player))
    if options.destiny_islands:
        multiworld.get_entrance("Destiny Islands", player).connect(multiworld.get_region("Destiny Islands", player))
    multiworld.get_entrance("Traverse Town", player).connect(multiworld.get_region("Traverse Town", player))
    multiworld.get_entrance("Wonderland", player).connect(multiworld.get_region("Wonderland", player))
    multiworld.get_entrance("Olympus Coliseum", player).connect(multiworld.get_region("Olympus Coliseum", player))
    multiworld.get_entrance("Deep Jungle", player).connect(multiworld.get_region("Deep Jungle", player))
    multiworld.get_entrance("Agrabah", player).connect(multiworld.get_region("Agrabah", player))
    multiworld.get_entrance("Monstro", player).connect(multiworld.get_region("Monstro", player))
    if options.atlantica:
        multiworld.get_entrance("Atlantica", player).connect(multiworld.get_region("Atlantica", player))
    multiworld.get_entrance("Halloween Town", player).connect(multiworld.get_region("Halloween Town", player))
    multiworld.get_entrance("Neverland", player).connect(multiworld.get_region("Neverland", player))
    multiworld.get_entrance("Hollow Bastion", player).connect(multiworld.get_region("Hollow Bastion", player))
    multiworld.get_entrance("End of the World", player).connect(multiworld.get_region("End of the World", player))
    multiworld.get_entrance("100 Acre Wood", player).connect(multiworld.get_region("100 Acre Wood", player))
    multiworld.get_entrance("World Map", player).connect(multiworld.get_region("World Map", player))
    multiworld.get_entrance("Levels", player).connect(multiworld.get_region("Levels", player))
    multiworld.get_entrance("Homecoming", player).connect(multiworld.get_region("Homecoming", player))

def create_region(multiworld: MultiWorld, player: int, name: str, data: KH1RegionData):
    region = Region(name, player, multiworld)
    if data.locations:
        for loc_name in data.locations:
            loc_data = location_table.get(loc_name)
            location = KH1Location(player, loc_name, loc_data.code if loc_data else None, region)
            region.locations.append(location)

    if data.region_exits:
        for exit in data.region_exits:
            entrance = Entrance(player, exit, region)
            region.exits.append(entrance)

    return region
