from enum import IntEnum
from typing import Optional, NamedTuple, Dict

from BaseClasses import Location, Region
from .Items import DSRItem

class DSRLocationCategory(IntEnum):
    SKIP = 0,
    EVENT = 1,
    BOSS = 2,
    BONFIRE = 3,
    DOOR = 4,
    ITEM_LOT = 5,
    ENEMY_DROP = 6


class DSRLocationData(NamedTuple):
    id: int
    name: str
    default_item: str
    category: DSRLocationCategory


class DSRLocation(Location):
    game: str = "Dark Souls Remastered"
    category: DSRLocationCategory
    default_item_name: str

    def __init__(
            self,
            player: int,
            name: str,
            category: DSRLocationCategory,
            default_item_name: str,
            address: Optional[int] = None,
            parent: Optional[Region] = None):
        super().__init__(player, name, address, parent)
        self.default_item_name = default_item_name
        self.category = category
        self.id = id

    @staticmethod
    def get_name_to_id() -> dict:
        base_id = 11110000
        table_offset = 1000

        table_order = [
            "Undead Asylum Cell",
            "Undead Asylum Cell Door",
            "Northern Undead Asylum F2 East Door",
            "Northern Undead Asylum", 
            "Northern Undead Asylum - After F2 East Door",
            "Undead Asylum Big Pilgrim Door",
            "Firelink Shrine", 
            "Upper Undead Burg", 
            "Upper Undead Burg - Pine Resin Chest",
            "Undead Parish", 
            "Firelink Shrine - After Undead Parish Elevator",
            "Northern Undead Asylum - Second Visit F2 West Door",
            "Northern Undead Asylum - Second Visit Snuggly Trades",
            "Northern Undead Asylum - Second Visit Behind F2 West Door",
            "Undead Burg Basement Door",
            "Lower Undead Burg", 
            "Lower Undead Burg - After Residence Key",
            "Watchtower Basement",
            "Depths", 
            "Depths - After Sewer Chamber Key",
            "Depths to Blighttown Door",
            "Blighttown", 
            "Valley of the Drakes", 
            "Valley of the Drakes - After Defeating Four Kings", 
            "Door between Upper New Londo and Valley of the Drakes",
            "Darkroot Basin", 
            "Darkroot Garden", 
            "Darkroot Garden - Behind Artorias Door", 
            "The Great Hollow", 
            "Ash Lake", 
            "Sen's Fortress",
            "Sen's Fortress - After Cage Key",
            "Anor Londo", 
            "Painted World of Ariamis",
            "Painted World of Ariamis - After Annex Key",
            "Upper New Londo Ruins",
            "New Londo Ruins Door to the Seal",
            "Lower New Londo Ruins", 
            "The Abyss", 
            "The Duke's Archives", 
            "The Duke's Archives Cell Door",
            "The Duke's Archives - Getting out of Cell",
            "The Duke's Archives - After Archive Prison Extra Key",
            "The Duke's Archives - After Archive Tower Giant Door Key", 
            "The Duke's Archives - Giant Cell",
            "Crystal Cave", 
            "The Duke's Archives - First Arena after Seath's Death", 
            "Demon Ruins",
            "Demon Ruins - Behind Golden Fog Wall",
            "Demon Ruins Shortcut",
            "Lost Izalith", 
            "The Catacombs", 
            "The Catacombs - Door 1",
            "The Catacombs - After Door 1",
            "Tomb of the Giants", 
            "Tomb of the Giants - Behind Golden Fog Wall",
            "Kiln of the First Flame",
            # "Sanctuary Garden", 
            # "Oolacile Sanctuary", 
            # "Royal Wood", 
            # "Royal Wood - After Hawkeye Gough",
            # "Oolacile Township", 
            # "Oolacile Township - After Crest Key",
            # "Chasm of the Abyss", 
         ]

        output = {}
        for i, region_name in enumerate(table_order):
            if len(location_tables[region_name]) > table_offset:
                raise Exception("A location table has {} entries, that is more than {} entries (table #{})".format(len(location_tables[region_name]), table_offset, i))
            output.update({location_data.name: location_data.id for location_data in location_tables[region_name]})
        return output

    def place_locked_item(self, item: DSRItem):
        self.item = item
        self.locked = True
        item.location = self
location_skip_categories = {
DSRLocationCategory.EVENT, DSRLocationCategory.SKIP, DSRLocationCategory.BOSS, DSRLocationCategory.BONFIRE
}
# Last id used = 714
location_tables = {
"Undead Asylum Cell": [
    DSRLocationData(11110000, f"UA: Dungeon Cell Key", f"Dungeon Cell Key", DSRLocationCategory.SKIP),
],
"Undead Asylum Cell Door": [
    DSRLocationData(11110001, f"UA: Undead Asylum Cell Door opened", f"Firebomb", DSRLocationCategory.DOOR),
],
"Northern Undead Asylum": [
    DSRLocationData(11110002, f"UA: Undead Asylum - Courtyard lit", f"Undead Asylum - Courtyard lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110003, f"UA: Undead Asylum - Boss Door opened", f"Firebomb", DSRLocationCategory.DOOR),
    DSRLocationData(11110005, f"UA: Undead Asylum - Interior lit", f"Undead Asylum - Interior lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110006, f"UA: Estus Flask", f"Estus Flask", DSRLocationCategory.SKIP),
    DSRLocationData(11110007, f"UA: Undead Asylum F2 East Key", f"Undead Asylum F2 East Key", DSRLocationCategory.SKIP),
    DSRLocationData(11110554, f"UA: Undead Asylum Shortcut Door opened", f"Firebomb", DSRLocationCategory.DOOR),
],     
"Northern Undead Asylum F2 East Door": [
    DSRLocationData(11110008, f"UA: Undead Asylum F2 East Door opened", f"Firebomb", DSRLocationCategory.DOOR),
],   
"Northern Undead Asylum - After F2 East Door": [
    DSRLocationData(11110009, f"UA: Asylum Demon Defeated", f"Asylum Demon Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110010, f"UA: Big Pilgrim's Key", f"Big Pilgrim's Key", DSRLocationCategory.SKIP),
    DSRLocationData(11110012, f"UA: Soul of a Lost Undead - Outside Asylum", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Undead Asylum Big Pilgrim Door": [
    DSRLocationData(11110011, f"UA: Undead Asylum Big Pilgrim Door opened", f"Firebomb", DSRLocationCategory.DOOR),  
],
"Firelink Shrine": [
    DSRLocationData(11110013, f"FS: Firelink Shrine lit", f"Firelink Shrine lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110014, f"FS: Firebomb - Outside Petrus", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110015, f"FS: Humanity - Firelink Well", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110016, f"FS: Soul of a Lost Undead - Outside Firelink Elevator", f"Firebomb",  DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110017, f"FS: Cracked Red Eye Orb - Firelink Elevator Drop", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110018, f"FS: Morning Star", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110019, f"FS: Talisman", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110020, f"FS: Homeward Bone - Firelink Elevator Drop", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110021, f"FS: Lloyd's Talisman", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110022, f"FS: Soul of a Lost Undead - Outside Frampt", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110023, f"FS: Soul of a Lost Undead - Above Frampt", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110024, f"FS: Large Soul of a Lost Undead - Graveyard 1", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110025, f"FS: Large Soul of a Lost Undead - Graveyard 2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110026, f"FS: Zweihander", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110027, f"FS: Winged Spear", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110028, f"FS: Caduceus Round Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110029, f"FS: Binoculars", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110030, f"FS: Ring of Sacrifice - Jump onto Aqueduct", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110031, f"FS: Soul of a Lost Undead - Aqueduct Cliff", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110032, f"FS: Soul of a Lost Undead - Aqueduct Entrance", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110033, f"FS: Soul of a Lost Undead - Sewer Rat", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],  
"Upper Undead Burg": [
    DSRLocationData(11110568, f"UB: Soul of a Lost Undead - Window Corpse", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110034, f"UB: Large Soul of a Lost Undead - Barrel Drop", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110035, f"UB: Rubbish - Undead Burg", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110036, f"UB: Humanity - Undead Burg Entrance Wall", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110037, f"UB: Soul of a Lost Undead - Upper Burg Fogwall House", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110038, f"UB: Wooden Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110039, f"UB: Undead Burg lit", f"Undead Burg lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110040, f"UB: Uchigatana", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110041, f"UB: Orange Guidance Soapstone", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110042, f"UB: Residence Key", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110043, f"UB: Throwing Knife", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110044, f"UB: Light Crossbow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110617, f"UB: Standard Bolt", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110045, f"UB: Black Firebomb - Side House Chest", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110046, f"UB: Soul of a Lost Undead - Firebomb Throwers", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110047, f"UB: Black Knight guarding Blue Tearstone Ring", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110048, f"UB: Blue Tearstone Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110050, f"UB: Crystal Lizard hidden in Barrel", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110053, f"UB: Taurus Demon Defeated", f"Taurus Demon Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110054, f"UB: Large Soul of a Lost Undead - After Taurus", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110055, f"UB: White Sign Soapstone", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110056, f"UB: Claymore", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110057, f"UB: Soul of a Nameless Soldier - Hellkite Bridge", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110058, f"UB: Undead Burg - Sunlight Altar lit", f"Undead Burg - Sunlight Altar lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110059, f"UB: Undead Burg Sunlight Altar opened", f"Firebomb", DSRLocationCategory.DOOR),
],
"Upper Undead Burg - Pine Resin Chest": [
    DSRLocationData(11110060, f"UB: Gold Pine Resin - 3x Gold Pine Resin in Chest behind Locked Door", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Undead Parish": [
    DSRLocationData(11110061, f"UP: Black Knight in Tower", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110062, f"UP: Alluring Skull", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110063, f"UP: Mystery Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110064, f"UP: Large Soul of a Lost Undead - Hollow Room Rafters", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110065, f"UP: Knight Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110066, f"UP: Undead Parish lit", f"Undead Parish lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110067, f"UP: Halberd", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110068, f"UP: Basement Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110069, f"UP: Berenike Knight", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110070, f"UP: Fire Keeper Soul - Undead Parish", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110071, f"UP: Large Soul of a Nameless Soldier - Parish Window", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110072, f"UP: Humanity - Parish Barrel", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110073, f"UP: Bell Gargoyles Defeated", f"Bell Gargoyles Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110074, f"UP: Bell of Awakening #1 rung", f"Bell of Awakening #1", DSRLocationCategory.EVENT),
],
"Firelink Shrine - After Undead Parish Elevator": [
    DSRLocationData(11110075, f"FS: Undead Asylum F2 West Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Northern Undead Asylum - Second Visit F2 West Door": [
    DSRLocationData(11110076, f"UA2: Undead Asylum F2 West Door opened", f"Firebomb", DSRLocationCategory.DOOR),    
],
"Northern Undead Asylum - Second Visit Behind F2 West Door": [    
    DSRLocationData(11110078, f"UA2: Rusted Iron Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Northern Undead Asylum - Second Visit Snuggly Trades": [
    DSRLocationData(11110077, f"UA2: Crest Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110079, f"UA2: Peculiar Doll", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110080, f"UA2: Stray Demon Defeated", f"Stray Demon Defeated", DSRLocationCategory.BOSS),
    # DSRLocationData(11110081, f"Snuggly: Pendant -> Souvenir of Reprisal", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110082, f"Snuggly: Rubbish -> Titanite Chunk", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110083, f"Snuggly: Sunlight Medal -> White Titanite Chunk", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110084, f"Snuggly: Bloodred Moss Clump -> Twinkling Titanite", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110085, f"Snuggly: Purple Moss Clump -> Twinkling Titanite", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110086, f"Snuggly: Blooming Purple Moss Clump -> Twinkling Titanite x2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110087, f"Snuggly: Cracked Red Eye Orb -> Purging Stone x2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110088, f"Snuggly: Humanity -> Ring of Sacrifice", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110089, f"Snuggly: Twin Humanities -> Rare Ring of Sacrifice", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110090, f"Snuggly: Prism Stone -> Demon Titanite", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110091, f"Snuggly: Dung Pie -> Demon Titanite", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110092, f"Snuggly: Pyromancy Flame -> Red Titanite Chunk", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110093, f"Snuggly: Pyromancy Flame (Ascended) -> Red Titanite Slab", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110094, f"Snuggly: Egg Vermifuge -> Dragon Scale", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110095, f"Snuggly: Sunlight Maggot -> Old Witch's Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110096, f"Snuggly: Sack -> Demon's Great Hammer", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110097, f"Snuggly: Skull Lantern -> Ring of Fog", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110098, f"Snuggly: Ring of the Sun Princess -> Divine Blessing x2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110099, f"Snuggly: Xanthous Crown -> Ring of Favor and Protection", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(11110100, f"Snuggly: Soul of Manus -> Sorcery: Pursuers", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],  
"Undead Burg Basement Door": [
    DSRLocationData(11110101, f"UB: Undead Burg Basement opened", f"Firebomb", DSRLocationCategory.DOOR),
],
"Lower Undead Burg": [
    DSRLocationData(11110102, f"UB: Undead Burg -> Lower Undead Burg opened", f"Firebomb", DSRLocationCategory.DOOR),
    DSRLocationData(11110103, f"UB: Twin Humanities - Lower Burg Torch Hollows", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110104, f"UB: Mail Breaker", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110105, f"UB: Large Soul of a Lost Undead - Barrel outside Capra", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110106, f"UB: Large Soul of a Lost Undead - Hallway to Depths", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110565, f"UB: Thief Mask", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110618, f"UB: Black Leather Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110619, f"UB: Black Leather Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110620, f"UB: Black Leather Boots", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110566, f"UB: Target Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110107, f"UB: Undead Burg Female Merchant Shortcut opened", f"Firebomb", DSRLocationCategory.DOOR),
    DSRLocationData(11110108, f"UB: Capra Demon Defeated", f"Capra Demon Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110109, f"UB: Key to Depths", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Lower Undead Burg - After Residence Key": [
    DSRLocationData(11110110, f"UB: Sorcerer Hat", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110621, f"UB: Sorcerer Cloak", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110622, f"UB: Sorcerer Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110623, f"UB: Sorcerer Boots", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110111, f"UB: Sorcerer's Catalyst", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Watchtower Basement": [
    DSRLocationData(11110051, f"UB: Undead Burg Watchtower Upper opened", f"Firebomb", DSRLocationCategory.DOOR),
    DSRLocationData(11110049, f"UB: Havel's Ring", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110052, f"UB: Undead Burg Watchtower Lower opened", f"Firebomb", DSRLocationCategory.DOOR),
],
"Depths": [
    DSRLocationData(11110112, f"DE: Large Soul of a Nameless Soldier - Depths Entrance", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110113, f"DE: Large Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110114, f"DE: Spider Shield - Ledge near Giant Rat", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110115, f"DE: Soul of a Nameless Soldier - Ooze Trap", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110116, f"DE: Greataxe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110117, f"DE: Sewer Chamber Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110118, f"DE: Soul of a Nameless Soldier - Near Channeler", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110119, f"DE: Soul of a Nameless Soldier - Giant Rat", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110564, f"DE: Soul of a Nameless Soldier - Giant Rat Room Entrance", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110120, f"DE: Large Titanite Shard - Depths Channeler", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110121, f"DE: Large Soul of a Nameless Soldier - Depths Basalisks", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110122, f"DE: Soul of a Nameless Soldier - Basilisk Sewer Maze", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110123, f"DE: Ring of the Evil Eye", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110124, f"DE: Humanity - Depths Pitfall Jump", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110125, f"DE: Heavy Crossbow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110624, f"DE: Heavy Bolt", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110126, f"DE: Depths Shortcut opened", f"Firebomb", DSRLocationCategory.DOOR),
    DSRLocationData(11110127, f"DE: Gaping Dragon Defeated", f"Gaping Dragon Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110128, f"DE: Blighttown Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110129, f"DE: Standard Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110711, f"DE: Hard Leather Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110712, f"DE: Hard Leather Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110713, f"DE: Hard Leather Boots", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Depths - After Sewer Chamber Key": [
    DSRLocationData(11110131, f"DE: Depths Bonfire Room opened", f"Firebomb", DSRLocationCategory.DOOR),
    DSRLocationData(11110132, f"DE: Depths lit", f"Depths lit", DSRLocationCategory.BONFIRE),
],
"Depths to Blighttown Door": [
    DSRLocationData(11110130, f"DE: Depths -> Blighttown opened", f"Firebomb", DSRLocationCategory.DOOR),
],
"Blighttown": [
    DSRLocationData(11110133, f"BT: Soul of a Proud Knight - Blighttown Entrance Dropoff", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110134, f"BT: Large Soul of a Nameless Soldier - Corpse at Left Ramp", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110135, f"BT: Blooming Purple Moss Clump - Suspended Walkway", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110136, f"BT: Humanity - Blighttown Swaying Bridge", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110137, f"BT: Iaito", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110138, f"BT: Blighttown Catwalk lit", f"Blighttown Catwalk lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110139, f"BT: Soul of a Proud Knight - Below Shadow Set", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110140, f"BT: Soul of a Proud Knight - Corpse on Wooden Structure Ledge", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110141, f"BT: Shadow Mask", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110625, f"BT: Shadow Garb", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110626, f"BT: Shadow Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110627, f"BT: Shadow Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110142, f"BT: Soul of a Proud Knight - Corpse in Wooden Structure near Third Catwalk", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110143, f"BT: Eagle Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110144, f"BT: Pyromancy: Power Within", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110145, f"BT: Large Soul of a Nameless Soldier - Wall Hugger", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110146, f"BT: Wanderer Hood", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110628, f"BT: Wanderer Coat", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110629, f"BT: Wanderer Manchette", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110630, f"BT: Wanderer Boots", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110147, f"BT: Falchion", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110148, f"BT: Large Soul of a Proud Knight - Corpse below Ramp", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110595, f"BT: Whip", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110149, f"BT: Dragon Scale - Blighttown", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110150, f"BT: Butcher Knife", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110151, f"BT: Humanity - Maneater Mildred", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110152, f"BT: Blighttown Swamp lit", f"Blighttown Swamp lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110153, f"BT: Large Soul of a Proud Knight - Quelana", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110154, f"BT: Great Club", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110155, f"BT: Tattered Cloth Hood", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110631, f"BT: Tattered Cloth Robe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110632, f"BT: Tattered Cloth Manchette", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110633, f"BT: Heavy Boots", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110156, f"BT: Pyromancy: Poison Mist", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110157, f"BT: Large Soul of a Proud Knight - Corpse between Bonfire and Great Hollow Entrance", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110158, f"BT: Large Titanite Shard - Near Server", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110159, f"BT: Server", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110160, f"BT: Green Titanite Shard - Blighttown", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110161, f"BT: Plank Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110162, f"BT: Twin Humanities - Great Hallow Entrance", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110163, f"BT: Large Titanite Shard - Near Quelana", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110164, f"BT: Chaos Witch Quelaag Defeated", f"Chaos Witch Quelaag Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110560, f"BT: Soul of Quelaag", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110165, f"BT: Quelaag's Domain lit", f"Quelaag's Domain lit", DSRLocationCategory.BONFIRE),    
    DSRLocationData(11110556, f"BT: Bell of Awakening #2 rung", f"Bell of Awakening #2", DSRLocationCategory.EVENT),
    DSRLocationData(11110166, f"BT: Soul of a Proud Knight - Blighttown Fire Keeper Soul", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110167, f"BT: Fire Keeper Soul - Blighttown", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110168, f"BT: Mask of the Sealer", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110634, f"BT: Crimson Robe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110635, f"BT: Crimson Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110636, f"BT: Crimson Waistcloth", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110169, f"BT: Tin Banishment Catalyst", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110170, f"BT: Sorcery: Remedy", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110171, f"BT: Key to New Londo Ruins", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Valley of the Drakes": [
    DSRLocationData(11110173, f"VotD: Large Soul of a Nameless Soldier - New Londo Door", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110174, f"VotD: Astora's Straight Sword", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110175, f"VotD: Dragon Crest Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110176, f"VotD: Soul of a Proud Knight - Undead Dragon", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110177, f"VotD: Undead Dragon in Valley of the Drakes", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110178, f"VotD: Humanity - Corpse in Small Cave", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110179, f"VotD: Brigand Hood", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110637, f"VotD: Brigand Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110638, f"VotD: Brigand Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110639, f"VotD: Brigand Trousers", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110180, f"VotD: Spider Shield - Brigand Armor Set", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110181, f"VotD: Red Tearstone Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Valley of the Drakes - After Defeating Four Kings": [
    DSRLocationData(11110182, f"VotD: Witch Hat", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110640, f"VotD: Witch Cloak", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110641, f"VotD: Witch Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110642, f"VotD: Witch Skirt", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110183, f"VotD: Beatrice's Catalyst", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Door between Upper New Londo and Valley of the Drakes": [
    DSRLocationData(11110331, f"VotD: New Londo Ruins -> Valley of the Drakes opened", f"Firebomb", DSRLocationCategory.DOOR),
],
"Darkroot Basin": [
    DSRLocationData(11110184, f"DB: Crystal Lizard near Hunter Set", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110185, f"DB: Leather Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110705, f"DB: Leather Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110706, f"DB: Leather Boots", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110577, f"DB: Large Soul of a Nameless Soldier - Darkroot Hydra", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110186, f"DB: Longbow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110707, f"DB: Feather Arrow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110187, f"DB: Grass Crest Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110188, f"DB: Black Knight near Bonfire", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110189, f"DB: Darkroot Basin lit", f"Darkroot Basin lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110190, f"DB: Knight Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110643, f"DB: Knight Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110644, f"DB: Knight Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110645, f"DB: Knight Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    #DSRLocationData(11110191, f"DB: Crown of Dusk", f"Firebomb", DSRLocationCategory.ITEM_LOT),    #Commented out for now, since it's a missable NPC Interaction
    #DSRLocationData(11110646, f"DB: Antiquated Dress", f"Firebomb", DSRLocationCategory.ITEM_LOT), #Commented out for now, since it's a missable NPC Interaction
    #DSRLocationData(11110647, f"DB: Antiquated Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),#Commented out for now, since it's a missable NPC Interaction
    #DSRLocationData(11110648, f"DB: Antiquated Skirt", f"Firebomb", DSRLocationCategory.ITEM_LOT), #Commented out for now, since it's a missable NPC Interaction
],
"Darkroot Garden": [
    DSRLocationData(11110192, f"DG: Titanite Demon below Andre", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110193, f"DG: Large Soul of a Nameless Soldier - Darkroot Cliff", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110194, f"DG: Darkroot Garden lit", f"Darkroot Garden lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110195, f"DG: Large Soul of a Nameless Soldier - Darkroot Ambush", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110196, f"DG: Elite Knight Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110649, f"DG: Elite Knight Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110650, f"DG: Elite Knight Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110651, f"DG: Elite Knight Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110197, f"DG: Wolf Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110198, f"DG: Soul of a Proud Knight - Past Fake Trees", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110199, f"DG: Partizan", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110200, f"DG: Moonlight Butterfly Defeated", f"Moonlight Butterfly Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110557, f"DG: Soul of the Moonlight Butterfly", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110201, f"DG: Homeward Bone - Frozen Blacksmith", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110202, f"DG: Watchtower Basement Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110203, f"DG: Divine Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Darkroot Garden - Behind Artorias Door": [
    DSRLocationData(11110553, f"DG: Eastern Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110652, f"DG: Eastern Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110653, f"DG: Eastern Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110654, f"DG: Eastern Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110204, f"DG: Stone Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110655, f"DG: Stone Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110656, f"DG: Stone Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110657, f"DG: Stone Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110578, f"DG: Large Soul of a Brave Warrior - Darkroot Garden Cats", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    #DSRLocationData(11110205, f"DG: Shiva Loot", f"Firebomb", DSRLocationCategory.ENEMY_DROP), #Commented out for now, since it's a NPC
    #DSRLocationData(11110206, f"DG: Humanity - Shiva", f"Firebomb", DSRLocationCategory.ENEMY_DROP), #Commented out for now, since it's a NPC
    #DSRLocationData(11110207, f"DG: Pharis Loot", f"Firebomb", DSRLocationCategory.ENEMY_DROP), #Commented out for now, since it's a NPC
    DSRLocationData(11110208, f"DG: Twin Humanities - Pharis", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110209, f"DG: Soul of a Brave Warrior - Down Ladder before Stone Bridge", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110210, f"DG: Enchanted Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110211, f"DG: Great Grey Wolf Sif Defeated", f"Great Grey Wolf Sif Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110212, f"DG: Best Boi Loot", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110559, f"DG: Soul of Sif", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110213, f"DG: Hornet Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"The Great Hollow": [
    DSRLocationData(11110214, f"GH: The Great Hollow lit", f"The Great Hollow lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110586, f"GH: Titanite Chunk - Drop on Top Branch", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110587, f"GH: Blue Titanite Chunk - Second Last Drop before Floor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110588, f"GH: White Titanite Chunk - Third Last Drop before Floor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110589, f"GH: Red Titanite Chunk - Last Drop before Floor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110590, f"GH: Blue Titanite Chunk - Drop on Branch into Hole", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110591, f"GH: White Titanite Chunk - Drop below Floor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110215, f"GH: Twinkling Titanite - Drop from Crystal Lizard (Random Spawn 1)", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110216, f"GH: Twinkling Titanite - Drop from Crystal Lizard (Random Spawn 2)", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110217, f"GH: Twinkling Titanite - Drop from Crystal Lizard (Random Spawn 3)", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110218, f"GH: Twinkling Titanite - Drop from Crystal Lizard (Random Spawn 4)", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110219, f"GH: Twinkling Titanite - Drop from Crystal Lizard (Random Spawn 5)", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110220, f"GH: Twinkling Titanite - Drop from Crystal Lizard (Random Spawn 6)", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110221, f"GH: Twinkling Titanite - Drop from Crystal Lizard (Random Spawn 7)", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110222, f"GH: Twinkling Titanite - Drop from Crystal Lizard (Random Spawn 8)", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110223, f"GH: Twinkling Titanite - Drop from Crystal Lizard (Random Spawn 9)", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110224, f"GH: Chloranthy Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110225, f"GH: Large Soul of a Nameless Soldier - Corpse Entry to Basilisks", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110592, f"GH: Large Soul of a Nameless Soldier - Corpse on Lower Floor with Basilisks", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110593, f"GH: Titanite Chunk - Corpse on Mushroom", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110594, f"GH: Red Titanite Chunk - Near Top Branch Drop", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Ash Lake": [
    DSRLocationData(11110226, f"ASH: Ash Lake lit", f"Ash Lake lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110227, f"ASH: Dragon Scale - Ash Lake Start", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110228, f"ASH: Dragon Scale - Ash Lake Water", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110229, f"ASH: Dragon Scale - Ash Lake Log", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110230, f"ASH: Miracle: Great Magic Barrier", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110231, f"ASH: Ash Lake - Stone Dragon lit", f"Ash Lake - Stone Dragon lit", DSRLocationCategory.BONFIRE),
],
"Sen's Fortress": [
    DSRLocationData(11110611, f"SF: Divine Blessing - Sen's Fortress", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110613, f"SF: Rare Ring of Sacrifice - Sen's Fortress", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110614, f"SF: Large Soul of a Proud Knight - Corpse after Ladder out of Pit", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110232, f"SF: Sen's Fortress Main Gate opened", f"Firebomb", DSRLocationCategory.DOOR),
    DSRLocationData(11110233, f"SF: Soul of a Brave Warrior - Sen's Fortress Entrance", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110234, f"SF: Demon Titanite - Drop from Titanite Demon in Pit 1", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110235, f"SF: Demon Titanite - Drop from Titanite Demon in Pit 2", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110236, f"SF: Demon Titanite - Drop from Titanite Demon in Pit 3", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110237, f"SF: Demon Titanite - Drop from Titanite Demon in Pit 4", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110238, f"SF: Soul of a Brave Warrior - Sen's Fortress Pit", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110239, f"SF: Scythe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110240, f"SF: Large Titanite Shard - Sen's Fortress Trap Chest", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110241, f"SF: Ring of Steel Protection", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110242, f"SF: Shotel", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110243, f"SF: Covetous Gold Serpent Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110244, f"SF: Lightning Spear", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110246, f"SF: Large Soul of a Proud Knight - Sen's Fortress Ball Dropdown", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110247, f"SF: Black Sorcerer Hat", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110658, f"SF: Black Sorcerer Cloak", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110659, f"SF: Black Sorcerer Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110660, f"SF: Black Sorcerer Boots", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110248, f"SF: Sorcery: Hush", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110249, f"SF: Slumbering Dragoncrest Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110250, f"SF: Large Titanite Shard - Guarded by Serpent Soldier", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110251, f"SF: Large Soul of a Brave Warrior - Sen's Fortress above Bonfire", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110252, f"SF: Sen's Fortress lit", f"Sen's Fortress lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110253, f"SF: Flame Stoneplate Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),    
    DSRLocationData(11110254, f"SF: Large Titanite Shard - Corpse bombed by Giant", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110255, f"SF: Ricard's Rapier", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110256, f"SF: Sniper Crossbow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110661, f"SF: Sniper Bolt", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110257, f"SF: Cage Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110258, f"SF: Iron Golem Defeated", f"Iron Golem Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110259, f"SF: Core of an Iron Golem", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
],
"Sen's Fortress - After Cage Key": [
    DSRLocationData(11110260, f"SF: Soul of a Hero - Sen's Fortress Cage", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Anor Londo": [
    DSRLocationData(11110261, f"AL: Anor Londo lit", f"Anor Londo lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110262, f"AL: Demon Titanite - Anor Londo Giants", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110263, f"AL: Crystal Halberd", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110264, f"AL: Twinkling Titanite - Anor Londo", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110265, f"AL: Demon Titanite - Anor Londo Elevator", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110266, f"AL: Divine Blessing - Anor Londo Rafters", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110267, f"AL: Anor Londo - Darkmoon Tomb lit", f"Anor Londo - Darkmoon Tomb lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110268, f"AL: Ring of the Sun's Firstborn", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110269, f"AL: Sorcery: Great Magic Weapon", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110270, f"AL: Black Iron Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110662, f"AL: Black Iron Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110663, f"AL: Black Iron Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110664, f"AL: Black Iron Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110271, f"AL: Greatsword", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110272, f"AL: Black Iron Greatshield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110273, f"AL: Anor Londo - Residence lit", f"Anor Londo - Residence lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110274, f"AL: Soul of a Hero - Left Silver Archer", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110275, f"AL: Sunlight Medal - Anor Londo", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110276, f"AL: Havel's Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110665, f"AL: Havel's Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110616, f"AL: Havel's Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110666, f"AL: Havel's Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110277, f"AL: Dragon Tooth", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110278, f"AL: Havel's Greatshield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110279, f"AL: Occult Club", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110280, f"AL: Soul of a Hero - Anor Londo Rooms", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110281, f"AL: Titanite Demon below Balcony", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110282, f"AL: Gold Coin - Anor Londo Mimic", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110283, f"AL: Silver Coin - Anor Londo Mimic", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110284, f"AL: Demon Titanite - Anor Londo Bedroom", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110285, f"AL: Silver Knight Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110667, f"AL: Silver Knight Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110286, f"AL: Silver Knight Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110668, f"AL: Silver Knight Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110287, f"AL: Anor Londo Bonfire Shortcut opened", f"Firebomb", DSRLocationCategory.DOOR),
    DSRLocationData(11110288, f"AL: Dragonslayer Greatbow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110714, f"AL: Dragonslayer Arrow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110289, f"AL: Titanite Chunk - near Giant Blacksmith", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110290, f"AL: Hawk Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110291, f"AL: Anor Londo Giant Blacksmith Shortcut opened", f"Firebomb", DSRLocationCategory.DOOR),
    DSRLocationData(11110292, f"AL: Anor Londo Main Hall Door opened", f"Firebomb", DSRLocationCategory.DOOR),
    DSRLocationData(11110293, f"AL: Ornstein and Smough Defeated", f"Ornstein and Smough Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110561, f"AL: Soul of Ornstein", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110562, f"AL: Soul of Smough", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110294, f"AL: Lordvessel", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110295, f"AL: Gwyndolin Defeated", f"Gwyndolin Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110296, f"AL: Soul of Gwyndolin", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110297, f"AL: Brass Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110669, f"AL: Brass Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110670, f"AL: Brass Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110671, f"AL: Brass Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110298, f"AL: Miracle: Sunlight Blade", f"Firebomb", DSRLocationCategory.ITEM_LOT),    
    DSRLocationData(11110555, f"AL: Anor Londo Chamber of the Princess lit", f"Anor Londo Chamber of the Princess lit", DSRLocationCategory.BONFIRE),
],
"Painted World of Ariamis": [
    DSRLocationData(11110299, f"PW: Painted World lit", f"Painted World lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110300, f"PW: Humanity - Painted World Hanging Body", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110571, f"PW: Soul of a Proud Knight - Corpse up First Staircase", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110572, f"PW: Soul of a Proud Knight - Corpse near Bonewheels", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110575, f"PW: Soul of a Proud Knight - By Hanging Corpse", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110573, f"PW: Ring of Sacrifice - Painted World", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110574, f"PW: Soul of a Brave Warrior - Next to Stairs", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110576, f"PW: Soul of a Brave Warrior - Corpse at Backside Tower Corner", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110301, f"PW: Soul of a Proud Knight - Painted World Corvid Ambush", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110302, f"PW: Twin Humanities - Painted World", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110303, f"PW: Soul of a Brave Warrior - Corpse Hanging out of Hole", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110304, f"PW: Soul of a Brave Warrior - Painted World Albino Rats", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110305, f"PW: Painting Guardian Hood", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110708, f"PW: Painting Guardian Robe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110709, f"PW: Painting Guardian Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110710, f"PW: Painting Guardian Waistcloth", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110306, f"PW: Egg Vermifuge - Painted World", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110307, f"PW: Velka's Rapier", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110308, f"PW: Large Soul of a Proud Knight - Ambush after Building", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110310, f"PW: Red Sign Soapstone", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110311, f"PW: Soul of a Brave Warrior - Painted World Tower", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110312, f"PW: Undead Dragon in Painted World", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110313, f"PW: Bloodshield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110314, f"PW: Large Soul of a Proud Knight - Painted World Undead Dragon", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110315, f"PW: Soul of a Proud Knight - Jeremiah 1", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110316, f"PW: Soul of a Proud Knight - Jeremiah 2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110317, f"PW: Large Soul of a Proud Knight - Jeremiah", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110318, f"PW: Pyromancy: Acid Surge", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110319, f"PW: Notched Whip", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110320, f"PW: Soul of a Brave Warrior - Painted World Bonewheels", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110321, f"PW: Pyromancy: Fire Surge", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110322, f"PW: Annex Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110323, f"PW: Humanity - Painted World Courtyard", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110324, f"PW: Large Titanite Shard", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110325, f"PW: Crossbreed Priscilla Defeated", f"Crossbreed Priscilla Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110326, f"PW: Soul of Priscilla", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110327, f"PW: Xanthous Crown", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110672, f"PW: Xanthous Overcoat", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110673, f"PW: Xanthous Gloves", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110674, f"PW: Xanthous Waistcloth", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
],
"Painted World of Ariamis - After Annex Key": [
    DSRLocationData(11110309, f"PW: Dark Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110328, f"PW: Miracle: Vow of Silence", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110329, f"PW: Mask of Velka", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110675, f"PW: Black Cleric Robe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110676, f"PW: Black Manchette", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110677, f"PW: Black Tights", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110330, f"PW: Gold Coin - Painted World", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Upper New Londo Ruins": [
    DSRLocationData(11110332, f"NL: Soul of a Nameless Soldier - New Londo Entrance", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110333, f"NL: Estoc", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110334, f"NL: Transient Curse - Before Ghosts", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110335, f"NL: Fire Keeper Soul - New Londo", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110336, f"NL: Transient Curse - 2x Transient Curses from Corpse inside Pot behind Wall", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110337, f"NL: Parrying Dagger", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110338, f"NL: Transient Curse - Banshee", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110339, f"NL: Large Soul of a Nameless Soldier - Near Ladder Shortcut", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110340, f"NL: Key to the Seal", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110341, f"NL: Soul of a Proud Knight - New Londo Back Corridor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110342, f"NL: Cursebite Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110343, f"NL: Green Titanite Shard - New Londo", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110344, f"NL: Large Soul of a Nameless Soldier - Corpse before Seal", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110345, f"NL: Humanity - New Londo Ledge", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110346, f"NL: Rare Ring of Sacrifice - New Londo Roof", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110347, f"NL: Composite Bow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110678, f"NL: Large Arrow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"New Londo Ruins Door to the Seal": [
    DSRLocationData(11110348, f"NL: New Londo Ruins Door to the Seal opened", f"Firebomb", DSRLocationCategory.DOOR),
],
"Lower New Londo Ruins": [
    DSRLocationData(11110703, f"NL: Humanity - Lower New Londo Ruins", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110704, f"NL: Large Soul of a Proud Knight - Right Path after Stairs", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110349, f"NL: Large Soul of a Proud Knight - Corpse in Shed", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110350, f"NL: Soul of a Brave Warrior - New Londo Stairs", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110351, f"NL: Very Large Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110352, f"NL: Large Soul of a Proud Knight - Corpse before Illusory Wall", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110353, f"NL: Titanite Chunk - Chest before Four Kings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110354, f"NL: Cracked Red Eye Orb - New Londo", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110355, f"NL: Large Soul of a Proud Knight - Corpse on Stairs", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110356, f"NL: Titanite Chunk - Chest behind Illusory Wall", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"The Abyss": [
    DSRLocationData(11110357, f"TA: Four Kings Defeated", f"Four Kings Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110358, f"TA: Bequeathed Lord Soul Shard (Four Kings)", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110359, f"TA: The Abyss lit", f"The Abyss lit", DSRLocationCategory.BONFIRE),
],
"The Duke's Archives": [
    DSRLocationData(11110360, f"DA: Duke's Archives - Entrance lit", f"Duke's Archives - Entrance lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110361, f"DA: Broken Pendant", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110362, f"DA: Soul of a Brave Warrior - Archives Under Stairs", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110363, f"DA: Twinkling Titanite - Archives Chest", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110364, f"DA: Twinkling Titanite - Archives Balcony", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110365, f"DA: Crystal Knight Shield", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110366, f"DA: Twinkling Titanite - Drop from Crystal Lizard in Tunnel", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110367, f"DA: Duke's Archives Cell lit", f"Duke's Archives Cell lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110369, f"DA: Archive Tower Cell Key", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
],
"The Duke's Archives Cell Door": [
    DSRLocationData(11110368, f"DA: Duke's Archives Cell Door opened", f"Firebomb", DSRLocationCategory.DOOR),
],
"The Duke's Archives - Getting out of Cell": [
    DSRLocationData(11110370, f"DA: Archive Prison Extra Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"The Duke's Archives - After Archive Prison Extra Key": [    
    DSRLocationData(11110371, f"DA: Large Soul of a Brave Warrior - In Reah's Cell", f"Firebomb", DSRLocationCategory.ITEM_LOT), 
    DSRLocationData(11110372, f"DA: Maiden Hood", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110679, f"DA: Maiden Robe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110680, f"DA: Maiden Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110681, f"DA: Maiden Skirt", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110373, f"DA: White Seance Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110374, f"DA: Large Soul of a Brave Warrior - Archives Cell", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110375, f"DA: Soul of a Brave Warrior - Archives Cell Scafolding", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110376, f"DA: Archive Tower Giant Door Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110377, f"DA: Miracle: Soothing Sunlight", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110378, f"DA: Miracle: Bountiful Sunlight", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
],
"The Duke's Archives - After Archive Tower Giant Door Key": [    
    DSRLocationData(11110379, f"DA: Avelyn", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110380, f"DA: Twinkling Titanite - Archives Chest 2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110381, f"DA: Sorcery: Strong Magic Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110382, f"DA: Duke's Archives - Balcony lit", f"Duke's Archives - Balcony lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110383, f"DA: Duke's Archives Bookshelf Door opened", f"Firebomb", DSRLocationCategory.DOOR),
    DSRLocationData(11110384, f"DA: Blue Titanite Chunk - Archives Chest", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110385, f"DA: Enchanted Falchion", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110386, f"DA: Archive Tower Giant Cell Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110387, f"DA: Crystal Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110388, f"DA: Six-Eyed Helm of the Channelers", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110682, f"DA: Robe of the Channelers", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110683, f"DA: Gauntlets of the Channelers", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110684, f"DA: Waistcloth of the Channelers", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110389, f"DA: Prism Stone - Duke's Archives", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110390, f"DA: Crystalline Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110685, f"DA: Crystalline Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110686, f"DA: Crystalline Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110687, f"DA: Crystalline Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110391, f"DA: Blue Titanite Chunk - Archives Courtyard", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"The Duke's Archives - Giant Cell": [
    DSRLocationData(11110392, f"DA: Fire Keeper Soul - Archives Giant Cell", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Crystal Cave": [
    DSRLocationData(11110393, f"CC: Humanity - Crystal Caverns", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110394, f"CC: Blue Titanite Chunk - Crystal Caverns", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110395, f"CC: Twinkling Titanite - Drop from first Crystal Lizard", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110396, f"CC: Twinkling Titanite - Drop from second Crystal Lizard", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110397, f"CC: Twinkling Titanite - Drop from third Crystal Lizard", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110398, f"CC: Blue Titanite Slab - Crystal Caverns", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110399, f"CC: Soul of a Hero - Crystal Caverns", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110400, f"CC: Seath the Scaleless Defeated", f"Seath the Scaleless Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110401, f"CC: Bequeathed Lord Soul Shard (Seath)", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110402, f"CC: Crystal Cave lit", f"Crystal Cave lit", DSRLocationCategory.BONFIRE),
],
"The Duke's Archives - First Arena after Seath's Death": [
    DSRLocationData(11110403, f"DA: Large Magic Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110404, f"DA: Soul of a Great Hero - Seath Room", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Demon Ruins": [
    DSRLocationData(11110405, f"DR: Demon Ruins - Entrance lit", f"Demon Ruins - Entrance lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110406, f"DR: Gold-Hemmed Black Hood", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110688, f"DR: Gold-Hemmed Black Cloak", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110689, f"DR: Gold-Hemmed Black Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110690, f"DR: Gold-Hemmed Black Skirt", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110407, f"DR: Ceaseless Discharge Defeated", f"Ceaseless Discharge Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110408, f"DR: Large Soul of a Proud Knight - First Jump over the Lava", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110409, f"DR: Chaos Flame Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110410, f"DR: Soul of a Proud Knight - Ceased Discharge", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110412, f"DR: Demon Ruins - Staircase lit", f"Demon Ruins - Staircase lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110413, f"DR: Green Titanite Shard - 2x Green Titanite Shards under the Staircase", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110609, f"DR: Soul of a Proud Knight - Ledge Corpse Top of Stairs", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110414, f"DR: Soul of a Proud Knight - Demon Ruins Bonfire", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110415, f"DR: Large Flame Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110416, f"DR: Soul of a Brave Warrior - Capra Outside Firesage", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Demon Ruins - Behind Golden Fog Wall": [
    DSRLocationData(11110602, f"DR: Soul of a Brave Warrior - On Platform Below Roots to Centipede", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110417, f"DR: Demon Firesage Defeated", f"Demon Firesage Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110418, f"DR: Demon's Catalyst", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110419, f"DR: Soul of a Brave Warrior - Ruins/Domain shortcut", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110420, f"DR: Soul of a Brave Warrior - Chaos Door", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110421, f"DR: Demon Ruins - Catacombs lit", f"Demon Ruins - Catacombs lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110422, f"DR: Centipede Demon Defeated", f"Centipede Demon Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110423, f"DR: Orange Charred Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110424, f"DR: Green Titanite Shard - Centipede Demon", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Demon Ruins Shortcut": [
    DSRLocationData(11110439, f"DR: Demon Ruins Shortcut opened", f"Firebomb", DSRLocationCategory.DOOR),
],
"Lost Izalith": [
    DSRLocationData(11110411, f"LI: Helm of Thorns", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110691, f"LI: Armor of Thorns", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110692, f"LI: Gauntlets of Thorns", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110693, f"LI: Leggings of Thorns", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110426, f"LI: Lost Izalith - Lava Pits lit", f"Lost Izalith - Lava Pits lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110427, f"LI: Soul of a Brave Warrior - First Platform after Entrance", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110428, f"LI: Soul of a Brave Warrior - Near Soul of a Great Hero Chest", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110429, f"LI: Soul of a Great Hero - Lost Izalith", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110430, f"LI: Twin Humanities - Lost Izalith", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110431, f"LI: Divine Blessing - Izalith Lava 1", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110432, f"LI: Divine Blessing - Izalith Lava 2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110433, f"LI: Lost Izalith - Past Illusory Wall lit", f"Lost Izalith - Past Illusory Wall lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110434, f"LI: Large Soul of a Brave Warrior - Izalith City", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110435, f"LI: Soul of a Hero - Izalith City", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110436, f"LI: Rare Ring of Sacrifice - Izalith City", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110437, f"LI: Titanite Demon near Shortcut", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110438, f"LI: Twinkling Titanite - Drop from Twinkling Titanite near Titanite Demon", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110440, f"LI: Red Titanite Chunk - Inside Chaos Eater Pit", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110441, f"LI: Green Titanite Shard - Chaos Eater Pit", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110442, f"LI: Soul of a Brave Warrior - Inside Chaos Eater Pit", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110443, f"LI: Red Titanite Slab - Chaos Eater Pit", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110596, f"LI: Red Titanite Chunk - After Chaos Eater Pit", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110444, f"LI: Pyromancy: Chaos Fire Whip", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110563, f"LI: Izalith Catalyst", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110445, f"LI: Bed of Chaos Defeated", f"Bed of Chaos Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110446, f"LI: Lord Soul (Bed of Chaos)", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110447, f"LI: Lost Izalith - Heart of Chaos lit", f"Lost Izalith - Heart of Chaos lit", DSRLocationCategory.BONFIRE),
],
"The Catacombs": [
    DSRLocationData(11110448, f"TC: Catacombs - Necromancer Cave lit", f"Catacombs - Necromancer Cave lit", DSRLocationCategory.BONFIRE),
],
"The Catacombs - Door 1": [
    DSRLocationData(11110449, f"TC: Catacombs Door 1 opened", f"Firebomb", DSRLocationCategory.DOOR),
],
"The Catacombs - After Door 1": [  
    DSRLocationData(11110579, f"TC: Darkmoon Seance Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),  
    DSRLocationData(11110450, f"TC: Lucerne", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110451, f"TC: Large Soul of a Nameless Soldier - Room before First Rotating Bridge", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110452, f"TC: Soul of a Proud Knight - After Second Switch", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110453, f"TC: Twinkling Titanite - Drop from Crystal Lizard further up the Staircase", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110454, f"TC: Twinkling Titanite - Drop from Crystal Lizard further down the Staircase", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110455, f"TC: Soul of a Proud Knight - First Spiral Stairway Upper", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110456, f"TC: Soul of a Proud Knight - First Spiral Stairway Lower", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110457, f"TC: Great Scythe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110458, f"TC: Catacombs - Past Illusory Wall lit", f"Catacombs - Past Illusory Wall lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110459, f"TC: Catacombs Door 2 opened", f"Firebomb", DSRLocationCategory.DOOR),
    DSRLocationData(11110460, f"TC: Green Titanite Shard - On Platform near Vamos", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110461, f"TC: Catacombs - Vamos lit", f"Catacombs - Vamos lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110462, f"TC: Miracle: Tranquil Walk of Peace", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110463, f"TC: Titanite Demon near Tomb", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110464, f"TC: Eye of Death - Catacombs Titanite Demon", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110465, f"TC: Large Soul of a Nameless Soldier - Above Black Knight", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110466, f"TC: Soul of a Proud Knight - Black Knight Ledge", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110467, f"TC: White Titanite Chunk - Drop from Axe-wielding Black Knight", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110468, f"TC: Large Soul of a Nameless Soldier - Catacombs Above Bonewheels", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110469, f"TC: Priest's Hat", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110694, f"TC: Holy Robe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110695, f"TC: Traveling Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110696, f"TC: Holy Trousers", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110552, f"TC: Mace", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110470, f"TC: Pinwheel Defeated", f"Pinwheel Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110471, f"TC: Rite of Kindling", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110472, f"TC: Soul of a Proud Knight - Above Pinwheel", f"Firebomb", DSRLocationCategory.ITEM_LOT),
],
"Tomb of the Giants": [
    DSRLocationData(11110580, f"TotG: Large Soul of a Proud Knight - Right after Green Light Slide", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110581, f"TotG: Large Soul of a Proud Knight - Ledge Overlooking Pinwheel", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110486, f"TotG: Soul of a Brave Warrior - Behind Bone Archer", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110585, f"TotG: Large Soul of a Proud Knight - Giant Coffin Drop", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110473, f"TotG: Large Soul of a Proud Knight - Upper Giant Coffin", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110474, f"TotG: Humanity - Corpse on Ledge", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110475, f"TotG: Tomb of the Giants - Patches lit", f"Tomb of the Giants - Patches lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110476, f"TotG: Large Soul of a Proud Knight - Patches Pit 1", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110477, f"TotG: Large Soul of a Proud Knight - Patches Pit 2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110491, f"TotG: White Titanite Chunk - Patches Kick", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110492, f"TotG: White Titanite Chunk - Lower Tomb of the Giants", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110478, f"TotG: Skull Lantern", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110479, f"TotG: Eye of Death - Upper Giant Coffin", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110480, f"TotG: Large Divine Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110481, f"TotG: Soul of a Brave Warrior - Lower Giant Coffin", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110482, f"TotG: Effigy Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110483, f"TotG: Tomb of the Giants lit", f"Tomb of the Giants lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110484, f"TotG: Covetous Silver Serpent Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110485, f"TotG: White Titanite Chunk - Drop from Halberd-wielding Black Knight", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110582, f"TotG: Soul of a Brave Warrior - Lower Tomb of the Giants", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110487, f"TotG: Twinkling Titanite - Drop from Crystal Lizard near Paladin", f"Firebomb", DSRLocationCategory.ENEMY_DROP), 
    DSRLocationData(11110488, f"TotG: Paladin Leeroy Loot", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110489, f"TotG: Humanity - Paladin Leeroy", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
],                    
"Tomb of the Giants - Behind Golden Fog Wall": [
    DSRLocationData(11110490, f"TotG: White Titanite Slab - Outside Nito", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110583, f"TotG: Soul of a Hero - Pinwheel Clone Room Entryway", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110493, f"TotG: Soul of a Hero - Outside Nito", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110494, f"TotG: Gravelord Nito Defeated", f"Gravelord Nito Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110584, f"TotG: White Titanite Chunk - Outside Nito", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110495, f"TotG: Lord Soul (Nito)", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110496, f"TotG: Tomb of the Giants - Altar of the Gravelord lit", f"Tomb of the Giants - Altar of the Gravelord lit", DSRLocationCategory.BONFIRE),
    DSRLocationData(11110497, f"TotG: Paladin Helm", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110697, f"TotG: Paladin Armor", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110698, f"TotG: Paladin Gauntlets", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    DSRLocationData(11110699, f"TotG: Paladin Leggings", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
],
"Kiln of the First Flame": [ 
    DSRLocationData(11110498, f"KoFF: Black Knight Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110700, f"KoFF: Black Knight Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110701, f"KoFF: Black Knight Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110702, f"KoFF: Black Knight Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    DSRLocationData(11110499, f"KoFF: Gwyn, Lord of Cinder Defeated", f"Gwyn, Lord of Cinder Defeated", DSRLocationCategory.BOSS),
    DSRLocationData(11110500, f"KoFF: Soul of Gwyn, Lord of Cinder", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
]                     
#"Sanctuary Garden": [     
  #  DSRLocationData(11110501, f"Oolacile - Sanctuary Garden lit", f"Oolacile - Sanctuary Garden lit", DSRLocationCategory.BONFIRE),
  #  DSRLocationData(11110502, f"Sanctuary Guardian Defeated", f"Sanctuary Guardian Defeated", DSRLocationCategory.BOSS),
  #  DSRLocationData(11110503, f"Guardian Soul", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
#],                         
#"Oolacile Sanctuary": [    
  #  DSRLocationData(11110504, f"Humanity - Corpse", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110505, f"Oolacile - Sanctuary lit", f"Oolacile - Sanctuary lit", DSRLocationCategory.BONFIRE),
#],                         
#"Royal Wood": [            
  #  DSRLocationData(11110506, f"Guardian Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110507, f"Blue Titanite Slab - In Chest at Pool", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110508, f"Guardian Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110509, f"Twinkling Titanite - Drop from Crystal Lizard past Stone Guardian", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
  #  DSRLocationData(11110510, f"Guardian Helmet", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110511, f"Gough's Great Arrow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110512, f"Guardian Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110513, f"Soul of a Proud Knight - Corpse in Tunnel", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110514, f"Twinkling Titanite - Drop from Crystal Lizard on Right Side", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
  #  DSRLocationData(11110515, f"Gold Coin - Down the Slope", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110516, f"Artorias the Abysswalker Defeated", f"Artorias the Abysswalker Defeated", DSRLocationCategory.BOSS),
  #  DSRLocationData(11110517, f"Soul of Artorias", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
  #  DSRLocationData(11110518, f"Soul of a Brave Warrior - Outside of Tower", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110519, f"Twinkling Titanite - Drop from Crystal Lizard on Small Platform", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
#],                        
#"Royal Wood - After Hawkeye Gough": [
  #  DSRLocationData(11110520, f"Black Dragon Kalameet Defeated", f"Black Dragon Kalameet Defeated", DSRLocationCategory.BOSS),
  #  DSRLocationData(11110521, f"Calamity Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
#],
#"Oolacile Township": [    
  #  DSRLocationData(11110522, f"Oolacile - Township lit", f"Oolacile - Township lit", DSRLocationCategory.BONFIRE),
  #  DSRLocationData(11110523, f"Soul of a Hero - Behind left Stairs", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110524, f"Large Soul of a Proud Knight - Roof", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110525, f"Soul of a Brave Warrior - Room with Oolacile Sorceress", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110526, f"Silver Pendant", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110527, f"Sorcery: Dark Orb", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110528, f"I'm Sorry Carving", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110529, f"Soul of a Brave Warrior - Wooden Platform", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110530, f"Very Good! Carving", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
  #  DSRLocationData(11110531, f"Rubbish - Outside of Door", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110532, f"Soul of a Hero - Before Doorway", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110533, f"Red Titanite Chunk - In Chest behind Illusory Wall", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110534, f"Crest Key", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
  #  DSRLocationData(11110535, f"Sorcery: Dark Fog", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110536, f"Twin Humanities - Drop onto Roof", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110537, f"Soul of a Brave Warrior - Corpse in left Corner", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110538, f"Oolacile - Township Dungeon lit", f"Oolacile - Township Dungeon lit", DSRLocationCategory.BONFIRE),
#],                        
#"Oolacile Township - After Crest Key": [
  #  DSRLocationData(11110539, f"Oolacile Crest Key Door opened", f"Firebomb", DSRLocationCategory.DOOR),
#],
#"Chasm of the Abyss": [   
  #  DSRLocationData(11110540, f"Twinkling Titanite - Drop from Crystal Lizard near Entrance", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
  #  DSRLocationData(11110541, f"Sorcery: Dark Bead", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110542, f"Cleansing Greatshield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110543, f"Humanity - Corpse after Prism Stone", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110544, f"Pyromancy: Black Flame", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110545, f"Help Me! Carving", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110546, f"Twin Humanities - Under Pillar", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110547, f"White Titanite Slab - On Ledge", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110548, f"Soul of a Hero - Left Ledge", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110549, f"Manus, Father of the Abyss Defeated", f"Manus, Father of the Abyss Defeated", DSRLocationCategory.BOSS),
  #  DSRLocationData(11110550, f"Soul of Manus", f"Firebomb", DSRLocationCategory.ITEM_LOT),
  #  DSRLocationData(11110551, f"Chasm of the Abyss lit", f"Chasm of the Abyss lit", DSRLocationCategory.BONFIRE),
#],
# "Bosses": [
    # DSRLocationData(f"Asylum Demon Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Taurus Demon Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Bell Gargoyles Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Capra Demon Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Ceaseless Discharge Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Centipede Demon Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Chaos Witch Quelaag Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Crossbreed Priscilla Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Demon Firesage Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Ornstein and Smough Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Four Kings Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Gaping Dragon Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Gravelord Nito Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Great Grey Wolf Sif Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Gwyn, Lord of Cinder Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Iron Golem Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Moonlight Butterfly Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Pinwheel Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Seath the Scaleless Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Black Dragon Kalameet Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Bed of Chaos Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Manus, Father of the Abyss Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Artorias the Abysswalker Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Sanctuary Guardian Defeated", f"Firebomb", DSRLocationCategory.BOSS),
    # DSRLocationData(f"Gwyndolin Defeated", f"Firebomb", DSRLocationCategory.BOSS),    
    # DSRLocationData(f"Stray Demon Defeated", f"Firebomb", DSRLocationCategory.BOSS),
# ],
# "Bonfires": [
    # DSRLocationData(f"Firelink Shrine lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Undead Parish lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Depths lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Undead Burg - Sunfire Altar lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Quelaag's Domain lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Anor Londo lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Anor Londo Chamber of the Princess lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Undead Asylum - Courtyard lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Undead Asylum - Interior lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Undead Burg lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Darkroot Garden lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Darkroot Basin lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Blighttown Catwalk lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Blighttown Swamp lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"The Great Hollow lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Ash Lake lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Ash Lake - Stone Dragon lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Demon Ruins - Entrance lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Demon Ruins - Staircase lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Demon Ruins - Catacombs lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Lost Izalith - Lava Pits lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Lost Izalith - Past Illusory Wall lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Lost Izalith - Heart of Chaos lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Sen's Fortress lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Anor Londo - Darkmoon Tomb lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Anor Londo - Residence lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Painted World lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Duke's Archives - Entrance lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Duke's Archives - Cell lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Duke's Archives - Balchony lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Crystal Cave lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Catacombs - Necromancer Cave lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Catacombs - Vamos lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Catacombs - Past Illusory Wall lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Tomb of the Giants - Patches lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Tomb of the Giants lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Tomb of the Giants - Altar of the Gravelord lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"The Abyss lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Oolacile - Sanctuary Garden lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Oolacile - Sanctuary lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Oolacile - Township Dungeon lit", f"Firebomb", DSRLocationCategory.BONFIRE),
    # DSRLocationData(f"Chasm of the Abyss lit", f"Firebomb", DSRLocationCategory.BONFIRE),
# ],
# "Doors": [
    # DSRLocationData(f"Depths Shortcut opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Depths -> Blighttown opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Depths Bonfire Room opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Undead Burg Female Merchant Shortcut opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Undead Burg -> Lower Undead Burg opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Undead Burg Basement opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Undead Burg Watchtower Upper opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Undead Burg Watchtower Lower opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Undead Burg Sunlight Altar opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Oolacile Crest Key Door opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Catacombs Door 1 opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Catacombs Door 2 opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Demon Ruins Shortcut opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Sen's Fortress Main Gate opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Anor Londo Main Hall Door opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Anor Londo Giant Blacksmith Shortcut opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Anor Londo Bonfire Shortcut opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"New Londo Ruins Door to the Seal opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"New Londo Ruins -> Valley of the Drakes opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Duke's Archives Bookshelf Door opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Duke's Archives Cell Door opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Undead Asylum Cell Door opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Undead Asylum F2 West Door opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Undead Asylum Shortcut Door opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Undead Asylum F2 East Door opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Undead Asylum Big Pilgrim Door opened", f"Firebomb", DSRLocationCategory.DOOR),
    # DSRLocationData(f"Undead Asylum Boss Door opened", f"Firebomb", DSRLocationCategory.DOOR),
# ],
# "ItemLots": [
    # DSRLocationData(f"White Sign Soapstone", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Tiny Being's Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Pierce Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Purging Stone", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Pierce Shield_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Speckled Stoneplate Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Miracle: Emit Force", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Speckled Stoneplate Ring_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Undead Asylum F2 East Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Lordvessel", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Key to the Seal", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Pyromancy Flame", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Miracle: Lightning Spear", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Miracle: Great Lightning Spear", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Copper Coin", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sunlight Medal", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Cat Covenant Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Divine Blessing", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Ring of Fog", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Replenishment", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Gravelord Sword", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Miracle: Gravelord Sword Dance", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Miracle: Gravelord Greatsword Dance", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Twin Humanities", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Titanite Slab", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dragon Eye", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dragon Head Stone", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dragon Torso Stone", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Pyromancy Flame_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Pyromancy: Fire Tempest", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Pyromancy: Great Chaos Fireball", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Pyromancy: Chaos Fire Whip", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Pyromancy: Chaos Storm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Hero", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Ring of the Sun Princess", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Blue Eye Orb", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Darkmoon Blade Covenant Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Miracle: Darkmoon Blade", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Darkmoon Talisman", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dark Hand", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Red Eye Orb", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Darksword", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dark Mask", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dark Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dark Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dark Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Eye Orb", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Archive Tower Giant Cell Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dingy Hood", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dingy Robe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dingy Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Blood-Stained Skirt", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Eye Orb_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Fire Keeper Soul", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Pendant", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Blighttown Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Key to Depths", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of Priscilla", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of the Moonlight Butterfly", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Covenant of Artorias", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of Sif", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Rite of Kindling", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Lord Soul", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of Quelaag", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Lord Soul_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Core of an Iron Golem", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of Gwyndolin", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of Ornstein", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of Smough", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Bequeathed Lord Soul Shard", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Bequeathed Lord Soul Shard_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of Gwyn, Lord of Cinder", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Big Pilgrim's Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Orange Charred Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Chaos Blade", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Chaos Blade_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Chaos Blade_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Purging Stone_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Twinkling Titanite", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Twinkling Titanite_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Twinkling Titanite_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dragon Scale", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Demon Titanite", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Ring of Sacrifice", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Demon Titanite_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"White Titanite Chunk", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Souvenir of Reprisal", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Titanite Chunk", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Blue Titanite Chunk", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Rare Ring of Sacrifice", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Red Titanite Chunk", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Red Titanite Slab", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Ring of Fog_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Old Witch's Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Ring of Favor and Protection", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Demon's Great Hammer", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Divine Blessing_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sorcery: Pursuers", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Iron Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Armor of the Sun", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Iron Bracelet", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Iron Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sunlight Straight Sword", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sunlight Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sunlight Talisman", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"White Sign Soapstone_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Fire Keeper Soul_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Estus Flask", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sorcery: Hush", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Slumbering Dragoncrest Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Ivory Talisman", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Servant Roster", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Pyromancy: Fire Tempest_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Key to the Seal_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Crest of Artorias", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_7", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Orange Guidance Soapstone", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Residence Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_8", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Speckled Stoneplate Ring_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_9", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Ring of Favor and Protection_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_10", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Murakumo", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Iron Round Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_11", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Crescent Axe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_12", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Book of the Guilty", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Twin Humanities_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dark Wood Grain Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_13", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Man-serpent Greatsword", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Butcher Knife", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_14", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Grant", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    # DSRLocationData(f"Sanctus", f"Firebomb", DSRLocationCategory.ENEMY_DROP),
    # DSRLocationData(f"Humanity_15", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Notched Whip", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Havel's Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Ricard's Rapier", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Izalith Catalyst", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Crest Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Big Hat", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Tin Crystallization Ctlyst.", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sorcery: White Dragon Breath", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Bow of Pharis", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Pharis's Hat", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Twin Humanities_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_16", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Nameless Soldier", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Greataxe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Nameless Soldier_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Nameless Soldier", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Nameless Soldier_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Spider Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Titanite Shard", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Nameless Soldier_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Nameless Soldier_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Ring of the Evil Eye", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Nameless Soldier_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Heavy Crossbow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Heavy Bolt", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Standard Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Hard Leather Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Hard Leather Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Hard Leather Boots", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sewer Chamber Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Mystery Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Mail Breaker", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_17", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Fire Keeper Soul_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Lost Undead", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Rubbish", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Light Crossbow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Standard Bolt", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_18", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Throwing Knife", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Wooden Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Basement Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Blue Tearstone Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Claymore", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Nameless Soldier_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Alluring Skull", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Lost Undead_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Knight Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Lost Undead_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Thief Mask", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Leather Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Leather Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Leather Boots", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Target Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Lost Undead_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Firebomb", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Gold Pine Resin", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_7", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Lost Undead_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Twin Humanities_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Nameless Soldier_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sorcerer Hat", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sorcerer Cloak", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sorcerer Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sorcerer Boots", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sorcerer's Catalyst", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Halberd", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_19", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Lost Undead_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Caduceus Round Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Lost Undead_7", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_8", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_9", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_10", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Lloyd's Talisman", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_11", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Firebomb", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_12", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Ring of Sacrifice_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_13", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Zweihander", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Winged Spear", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Binoculars", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Morning Star", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Talisman", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Homeward Bone", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Cracked Red Eye Orb", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Undead Asylum F2 West Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dried Finger", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Mask of Velka", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Cleric Robe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Manchette", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Tights", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Miracle: Vow of Silence", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Gold Coin", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Annex Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Xanthous Crown", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Xanthous Overcoat", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Xanthous Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Xanthous Waistcloth", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Ring of Sacrifice_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Pyromancy: Acid Surge", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Bloodshield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_20", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_21", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Egg Vermifuge", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Velka's Rapier", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Red Sign Soapstone", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dark Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Painting Guardian Hood", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Painting Guardian Robe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Painting Guardian Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Painting Guardian Waistcloth", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Partizan", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Leather Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Leather Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Leather Boots", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Longbow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Feather Arrow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Nameless Soldier_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Nameless Soldier_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Wolf Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Elite Knight Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Elite Knight Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Elite Knight Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Elite Knight Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Eastern Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Eastern Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Eastern Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Eastern Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Brave Warrior", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Watchtower Basement Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Divine Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Homeward Bone_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Crown of Dusk", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Antiquated Dress", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Antiquated Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Antiquated Skirt", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Hornet Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Nameless Soldier_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Enchanted Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Stone Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Stone Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Stone Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Stone Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_7", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_7", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Grass Crest Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Knight Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Knight Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Knight Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Knight Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_14", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_22", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_8", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Gold Coin_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_15", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_16", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_17", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_18", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_19", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_8", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_20", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_21", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Hero_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_22", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Rubbish_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Hero_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_9", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_10", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Gough's Great Arrow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_11", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"White Titanite Slab", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Hero_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_23", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_24", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_23", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_25", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_26", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_27", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_12", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_13", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_28", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_29", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_14", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Twin Humanities_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Hero_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Twin Humanities_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Brave Warrior_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Blue Titanite Slab", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Red Titanite Chunk_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_30", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Titanite Slab_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Nameless Soldier_7", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_9", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Darkmoon Seance Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Eye of Death", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Lucerne ", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Green Titanite Shard", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Great Scythe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Nameless Soldier_8", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_10", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Priest's Hat", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Holy Robe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Traveling Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Holy Trousers", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Mace", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_11", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Nameless Soldier_9", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Miracle: Tranquil Walk of Peace", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_12", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_13", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_7", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_24", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_8", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_15", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_9", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_10", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Eye of Death_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Skull Lantern", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"White Titanite Chunk_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_16", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Effigy Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Covetous Silver Serpent Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"White Titanite Chunk_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_17", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Hero_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"White Titanite Slab_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Hero_7", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Paladin Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Paladin Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Paladin Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Paladin Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"White Titanite Chunk_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_11", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Divine Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Plank Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Cloranthy Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Titanite Chunk_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Nameless Soldier_10", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Blue Titanite Chunk_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"White Titanite Chunk_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Red Titanite Chunk_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Blue Titanite Chunk_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"White Titanite Chunk_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Nameless Soldier_11", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Titanite Chunk_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dragon Scale_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dragon Scale_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dragon Scale_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Miracle: Great Magic Barrier", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Twin Humanities_7", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Red Titanite Chunk_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_14", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Shadow Mask", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Shadow Garb", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Shadow Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Shadow Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Nameless Soldier_12", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_25", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Blooming Purple Moss Clump", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_15", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_16", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Pyromancy: Power Within", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Nameless Soldier_13", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_17", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Eagle Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Whip", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Mask of the Sealer", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Crimson Robe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Crimson Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Crimson Waistcloth", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Tin Banishment Catalyst", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_18", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Titanite Shard_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Great Club", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_12", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Tattered Cloth Hood", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Tattered Cloth Robe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Tattered Cloth Manchette", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Heavy Boots", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Pyromancy: Poison Mist", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_13", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Green Titanite Shard_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Titanite Shard_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Wanderer Hood", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Wanderer Coat", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Wanderer Manchette", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Wanderer Boots", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Falchion", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_14", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Iaito", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Fire Keeper Soul_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Server", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Helm of Thorns", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Armor of Thorns", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Gauntlets of Thorns", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Leggings of Thorns", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Key to New Londo Ruins", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dragon Scale_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sorcery: Remedy", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Red Titanite Chunk_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Red Titanite Chunk_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_18", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Green Titanite Shard_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_19", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_15", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_20", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Green Titanite Shard_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Flame Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_19", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_20", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Gold-Hemmed Black Hood", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Gold-Hemmed Black Cloak", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Gold-Hemmed Black Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Gold-Hemmed Black Skirt", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Brave Warrior_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Brave Warrior_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Brave Warrior_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Brave Warrior_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_21", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Brave Warrior_7", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_22", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Brave Warrior_8", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Green Titanite Shard_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Brave Warrior_9", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Brave Warrior_10", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Brave Warrior_11", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_23", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_24", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Divine Blessing_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Divine Blessing_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Brave Warrior_12", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Twin Humanities_8", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Brave Warrior_13", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Rare Ring of Sacrifice_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Brave Warrior_14", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Hero_8", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Red Titanite Slab_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Great Hero", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_21", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Pyromancy: Chaos Fire Whip_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Chaos Flame Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Ring of Steel Protection", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_25", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Titanite Shard_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_16", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Divine Blessing_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_17", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Sorcerer Hat", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Sorcerer Cloak", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Sorcerer Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Sorcerer Boots", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sorcery: Hush_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Covetous Gold Serpent Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Titanite Shard_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Flame Stoneplate Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Rare Ring of Sacrifice_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Cage Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Shotel", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_18", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Slumbering Dragoncrest Ring_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_19", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_26", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Scythe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Brave Warrior_15", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Titanite Shard_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Hero_9", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_20", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sniper Crossbow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sniper Bolt", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sorcery: Great Magic Weapon", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Ring of the Sun's Firstborn", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Titanite Chunk_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Hero_10", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dragonslayer Greatbow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dragonslayer Arrow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Iron Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Iron Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Iron Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Iron Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Greatsword", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Iron Greatshield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Helm of Favor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Embraced Armor of Favor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Gauntlets of Favor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Leggings of Favor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Demon Titanite_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Divine Blessing_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Havel's Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Havel's Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Havel's Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Havel's Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Havel's Greatshield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dragon Tooth", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Silver Knight Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Silver Knight Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Silver Knight Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Silver Knight Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Demon Titanite_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Hawk Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Miracle: Sunlight Blade", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Brass Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Brass Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Brass Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Brass Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Twinkling Titanite_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Demon Titanite_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sunlight Medal_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Hero_11", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Nameless Soldier_7", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Estoc", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Fire Keeper Soul_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Transient Curse", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Parrying Dagger", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Transient Curse_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Nameless Soldier_14", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Green Titanite Shard_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_22", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Nameless Soldier_15", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Rare Ring of Sacrifice_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Cursebite Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_26", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Composite Bow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Arrow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Nameless Soldier_16", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Astora's Straight Sword", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Proud Knight_23", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dragon Crest Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_27", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Brigand Hood", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Brigand Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Brigand Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Brigand Trousers", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Spider Shield_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_21", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_28", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_27", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_22", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Titanite Chunk_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Cracked Red Eye Orb_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_23", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Witch Hat", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Witch Cloak", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Witch Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Witch Skirt", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Beatrice's Catalyst", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Proud Knight_24", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Red Tearstone Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Very Large Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Titanite Chunk_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Transient Curse_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_28", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Twinkling Titanite_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Avelyn", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Great Hero_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Prism Stone", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Brave Warrior_29", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Maiden Hood", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Maiden Robe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Maiden Gloves", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Maiden Skirt", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"White Seance Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Soul of a Brave Warrior_16", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Blue Titanite Chunk_4", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Humanity_29", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Blue Titanite Slab_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Blue Titanite Chunk_5", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Hero_12", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Fire Keeper Soul_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Archive Prison Extra Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Twinkling Titanite_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Twinkling Titanite_7", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Magic Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sorcery: Strong Magic Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Blue Titanite Chunk_6", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Six-Eyed Helm of the Channelers", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Robe of the Channelers", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Gauntlets of the Channelers", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Waistcloth of the Channelers", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Archive Tower Giant Cell Key_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Crystal Ember", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Archive Tower Giant Door Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sage Robe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Traveling Gloves_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Traveling Boots", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Logan's Catalyst", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Crystalline Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Crystalline Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Crystalline Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Crystalline Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Knight Helm", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Knight Armor", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Knight Gauntlets", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Black Knight Leggings", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dungeon Cell Key", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Rusted Iron Ring", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Soul of a Lost Undead_31", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Peculiar Doll", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Longsword", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Heater Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Broadsword", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Tower Kite Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Scimitar", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Leather Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Bandit's Knife", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Target Shield_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Battle Axe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Spider Shield_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Shortsword", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Large Leather Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Short Bow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Standard Arrow", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Dagger", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Small Leather Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Sorcerer's Catalyst_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Hand Axe", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Cracked Round Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Pyromancy Flame_3", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Mace_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"East-West Shield", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Canvas Talisman", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Club", f"Firebomb", DSRLocationCategory.ITEM_LOT),
    # DSRLocationData(f"Plank Shield_2", f"Firebomb", DSRLocationCategory.ITEM_LOT),
# ]




}

location_dictionary: Dict[str, DSRLocationData] = {}
for location_table in location_tables.values():
    location_dictionary.update({location_data.name: location_data for location_data in location_table})
