from __future__ import annotations
__all__ = ['GameListPanel', 
           'GameListItem', 
           'GameListItemLongText', 
           'GameListItemShortText', 
           'GameTrailingPressedIconButton',
           'SlotListItemHeader',
           'SlotListItem',
           'HintListItem',
           'HintListDropdown',
           'IconBadge',
           ]
from textwrap import wrap

from kivy.animation import Animation
from kivy.metrics import dp
from kivy.properties import StringProperty, DictProperty, ObjectProperty, NumericProperty
from kivy.uix.behaviors import ButtonBehavior
from kivymd.uix.behaviors import RotateBehavior, CommonElevationBehavior
from kivymd.theming import ThemableBehavior
from kivymd.uix.menu import MDDropdownMenu
from kivymd.uix.boxlayout import MDBoxLayout
from kivymd.uix.badge import MDBadge

from kivymd.uix.list import *
from kivymd.uix.expansionpanel import *

from kivymd.uix.tooltip import MDTooltip

from kivy.lang import Builder
import os
from kivy.clock import Clock
from kivymd.app import MDApp
from kivymd.icon_definitions import md_icons
from typing import TYPE_CHECKING, Any, Callable, Tuple

if TYPE_CHECKING:
    from NetUtils import HintStatus, MWGGUIHintStatus
    from CommonClient import CommonContext

from mwgg_gui.components.guidataclasses import UIPlayerData, UIHint, HintStatus

with open(os.path.join(os.path.dirname(__file__), "expansionlist.kv"), encoding="utf-8") as kv_file:
    Builder.load_string(kv_file.read())

class IconBadge(MDBadge):
    """
    A custom badge widget for displaying icons.
    """
    pass

class SlotListItemHeader(MDBoxLayout, CommonElevationBehavior):
    """
    Header widget for displaying slot item information, it 
    contains slot name and game information.
    
    Attributes:
        slot_name (StringProperty): The name of the slot
        game (StringProperty): The name of the game
        panel (ObjectProperty): Reference to the parent panel
    """
    slot_name: StringProperty
    game: StringProperty
    panel: ObjectProperty

    def __init__(self, item_data, panel, **kwargs):
        """
        Initialize the SlotListItemHeader.
        
        Args:
            game_data (dict): Dictionary containing slot and game information
            panel: Reference to the parent panel
        """
        self.panel = panel
        self.item_data = item_data
        if self.item_data.pronouns:
            self.slot_name = self.item_data.slot_name + " (" + self.item_data.pronouns + ")"
        else:
            self.slot_name = self.item_data.slot_name
        self.game = self.item_data.game

        super().__init__(**kwargs)


class GameListItemHeader(MDBoxLayout, ButtonBehavior, ThemableBehavior):
    """
    Header widget for displaying game item information in the game list.
    
    Attributes:
        game_tag (StringProperty): The tag identifier for the game
        game_data (DictProperty): Dictionary containing game information
        panel (ObjectProperty): Reference to the parent panel
        on_game_select (ObjectProperty): Callback function for game selection
    """
    game_module: StringProperty
    game_data: DictProperty
    panel: ObjectProperty
    on_game_select: ObjectProperty = None

    def __init__(self, game_module, game_data, panel, on_game_select=None, **kwargs):
        """
        Initialize the GameListItemHeader.
        
        Args:
            game_tag (str): The tag identifier for the game
            game_data (dict): Dictionary containing game information
            panel: Reference to the parent panel
            on_game_select: Callback function for game selection
        """
        self.game_module = game_module
        self.game_data = game_data
        self.panel = panel
        self.on_game_select = on_game_select
        self.style = "Tonal"
        super().__init__(**kwargs)

    def on_release(self, *args):
        """Handle press event for game selection"""
        if self.on_game_select:
            self.on_game_select(self.game_module)

    def list_tooltip(self, item_list: list[str]) -> dict[str, str]:
        """
        Create tooltip text for a list of items.
        
        Wraps the text to fit within specified width constraints and creates
        both a label (shortened) and tooltip (full) version.
        
        Args:
            item_list (list[str]): List of items to create tooltip for
            
        Returns:
            dict[str, str]: Dictionary with 'label' (shortened text) and 
                           'tooltip' (full text) keys
        """
        full_list = ", ".join(item_list).rstrip(", ")
        wrapped_list = wrap(full_list, width=17, break_on_hyphens=False, max_lines=3)
        item_dict = {
            "label": "\n".join(wrapped_list).rstrip("\n"),
            "tooltip": "\n".join(wrap(full_list, width=40, break_on_hyphens=False)).rstrip("\n")
        }
        return item_dict

class MWBaseListItem(MDBoxLayout, CommonElevationBehavior):
    """
    Base class for list items.
    Widget for displaying individual hint items in the hint player list.
    
    This class is used to display a hint item in the hint player list.
    Displays entrance, location, item, and goal information.
    
    Attributes:
        slot_icon_entrance (ObjectProperty): Icon widget for entrance
        slot_text_entrance (ObjectProperty): Text widget for entrance name
        slot_icon_location (ObjectProperty): Icon widget for location
        slot_text_location (ObjectProperty): Text widget for location name
        slot_icon_item (ObjectProperty): Icon widget for item
        slot_text_item (ObjectProperty): Text widget for item name
        slot_icon_goal (ObjectProperty): Icon widget for goal
        item_name (StringProperty): Name of the item
        location_name (StringProperty): Name of the location
        entrance_name (StringProperty): Name of the entrance
        game_status (StringProperty): Current status of the game
        for_bk_mode (StringProperty): Indicates if item is for BK mode
        for_goal (StringProperty): Indicates if item is for their goal
        from_shop (StringProperty): Indicates if item is from a shop
        classification (StringProperty): Classification of the item
        assigned_level (StringProperty): Assigned level
    """
    entrance_texture: Tuple[NumericProperty, NumericProperty]
    slot_icon_entrance: ObjectProperty
    slot_text_entrance: ObjectProperty
    slot_icon_location: ObjectProperty
    slot_text_location: ObjectProperty
    slot_icon_item: ObjectProperty
    slot_text_item: ObjectProperty
    slot_icon_goal: ObjectProperty
    item_name: StringProperty
    location_name: StringProperty
    entrance_name: StringProperty
    game_status: StringProperty
    for_bk_mode: StringProperty
    for_goal: StringProperty
    from_shop: StringProperty
    classification: StringProperty
    assigned_level: StringProperty
    found: StringProperty
    badge_text: StringProperty

    def __init__(self, hint_data: UIHint, game_status: str, shadow_colors: dict, **kwargs):
        """
        Initialize the SlotListItem.
        
        Args:
            game_status (str): Current status of the game
            game_data (dict): Dictionary containing stored data
        """
        from NetUtils import HintStatus, MWGGUIHintStatus
        self.hint_data = hint_data
        self.game_status = game_status
        self.entrance_name = self.hint_data.entrance if self.hint_data.entrance else "Vanilla"
        self.location_name = self.hint_data.location
        self.item_name = self.hint_data.item
        self.classification = self.hint_data.classification
        self.found = self.hint_data.found
        self.status = self.hint_data.hint_status
        self.mwgg_status = self.hint_data.mwgg_hint_status

        super().__init__(**kwargs)

        self.badge_text = ""
        if self.mwgg_status == MWGGUIHintStatus.HINT_BK_MODE:
            badge_text += md_icons["food"] + " "
        if self.mwgg_status == MWGGUIHintStatus.HINT_GOAL:
            badge_text += md_icons["flag_checkered"] + " "
        if self.mwgg_status == MWGGUIHintStatus.HINT_SHOP:
            badge_text += md_icons["shop"] + " "

    def populate_slot_item(self):
        pass

    def set_prio_behavior(self, item_colors: dict[str, list[str]]):
        if self.hint_data.assigned_classification:
            self.classification = self.hint_data.assigned_classification
        if self.classification == "Trap":
            self.elevation_level = 1
            self.shadow_color = item_colors["trap"]
        if self.classification == "Filler":
            self.elevation_level = 2
            self.shadow_color = item_colors["regular"]
        if self.classification == "Useful":
            self.elevation_level = 3
            self.shadow_color = item_colors["useful"]
        if self.classification == "Progression - Logically Relevant":
            self.elevation_level = 4
            self.shadow_color = item_colors["logically_required"]
        if self.classification == "Progression":
            self.elevation_level = 5
            self.shadow_color = item_colors["progression"]
        if self.classification == "Progression - Requried for Goal":
            self.elevation_level = 6
            self.shadow_color = item_colors["goal"]
        if self.found == "Found":
            self.elevation_level = 0

    def list_tooltip(self, item_list: list[str]) -> dict[str, str]:
        """
        Create tooltip text for a list of items.
        
        Wraps the text to fit within specified width constraints and creates
        both a label (shortened) and tooltip (full) version.
        
        Args:
            item_list (list[str]): List of items to create tooltip for
            
        Returns:
            dict[str, str]: Dictionary with 'label' (shortened text) and 
                           'tooltip' (full text) keys
        """
        full_list = ", ".join(item_list).rstrip(", ")
        wrapped_list = wrap(full_list, width=17, break_on_hyphens=False, max_lines=3)
        item_dict = {
            "label": "\n".join(wrapped_list).rstrip("\n"),
            "tooltip": "\n".join(wrap(full_list, width=40, break_on_hyphens=False)).rstrip("\n")
        }
        return item_dict

class SlotListItem(MWBaseListItem):
    """
    Widget for displaying individual slot items in the slot list.
    """
    def __init__(self, hint_data: UIHint, game_status: str, shadow_colors: dict, **kwargs):
        super().__init__(hint_data, game_status, shadow_colors, **kwargs)
        self.slot_icon_location = self.ids.slot_icon_location
        self.slot_text_location = self.ids.slot_text_location
        self.slot_icon_item = self.ids.slot_icon_item
        self.slot_text_item = self.ids.slot_text_item
        self.slot_icon_goal = self.ids.slot_icon_goal

        if self.badge_text != "":
            self.slot_icon_item.add_widget(IconBadge(text=self.badge_text.rstrip()))

        Clock.schedule_once(lambda x: self.populate_slot_item())
        Clock.schedule_once(lambda x: self.set_prio_behavior(shadow_colors), .5)

        self.height = self.estimate_height()

    def populate_slot_item(self):
        """
        Populate the slot item with entrance, location, item, and goal information.
        
        This method sets up the visual elements of the slot item including
        entrance information, location text, item text, and goal icon.
        """
        if self.entrance_name == "Vanilla":
            # I hate it, but somehow != didn't work.
            if not self.entrance_name:
                pass
        else:
            self.slot_text_entrance = (MDListItemSupportingText(text=self.entrance_name, do_wrap=True))
            self.slot_icon_entrance = (MDListItemLeadingIcon(icon="door-open", pos_hint={"center_y": 0.55}))
            self.slot_item_middle_container = (MDBoxLayout(orientation="horizontal", spacing=dp(4), size_hint_y=.5, pos_hint={"center_y": 0.5}))
            self.slot_item_middle_container.add_widget(self.slot_icon_entrance)
            self.slot_item_middle_container.add_widget(self.slot_text_entrance)
            self.add_widget(self.slot_item_middle_container, 1)
        self.slot_text_location.text = self.location_name
        self.slot_text_item.text = self.item_name
        self.slot_icon_goal.icon = "flag_checkered" if self.game_status == "GOAL" else "blank"

    def estimate_height(self):
        """
        Estimate the height of the slot item.

        Funky workaround because the scrollview height of the list item is set before the
        text is set, so any wrapping will cause the height to be wrong.
        """
        nheight = dp(36) + (dp(4) * len(self.children))
        if self.entrance_name == "Vanilla":
            pass
        else:
            eheight = self.slot_text_entrance.font_size * self.slot_text_entrance.line_height
            if len(self.entrance_name) < 20:
                nheight += eheight
            else:
                nheight = nheight + (eheight * 2)
        lheight = self.slot_text_location.font_size * self.slot_text_location.line_height
        if len(self.location_name) < 20:
            nheight += lheight
        else:
            nheight = nheight + (lheight * 2)
        iheight = self.slot_text_item.font_size * self.slot_text_item.line_height
        if len(self.item_name) < 20:
            nheight += iheight
        else:
            nheight = nheight + (iheight * 2)
        return nheight

class HintListItem(MWBaseListItem):
    hint_icon_status: StringProperty
    hint_status_text: StringProperty

    def __init__(self, hint_data: UIHint, game_status: str, shadow_colors: dict, hint_icon_status: str, hint_status_text: str, **kwargs):
        self.hint_icon_status = hint_icon_status
        self.hint_status_text = hint_status_text
        self.dropdown = None  # Will be set by the parent
        super().__init__(hint_data, game_status, shadow_colors, **kwargs)
        Clock.schedule_once(lambda x: self.populate_slot_item())
        Clock.schedule_once(lambda x: self.set_prio_behavior(shadow_colors), .5)

    def populate_slot_item(self):
        """
        Populate the hint item with entrance, location, item, and goal information.
        
        This method sets up the visual elements of the hint item including
        entrance information, location text, item text, and goal icon.
        """
        if self.hint_data.entrance == "Vanilla" or not self.hint_data.entrance:
            self.remove_widget(self.ids.hint_item_entrance_container)

    def open_dropdown(self):
        """Open the status dropdown menu"""
        if self.dropdown:
            self.dropdown.open()

    @staticmethod
    def on_hide(hint_instance, value):
        hint_instance.hint_data.hide = value
    
    def on_bkmode(self, instance):
        """Handle BK mode button activation"""
        self.dispatch('on_bkmode', instance)
    
    def on_goal(self, instance):
        """Handle goal button activation"""
        self.dispatch('on_goal', instance)
    
    def on_shop(self, instance):
        """Handle shop button activation"""
        self.dispatch('on_shop', instance)

class HintListDropdown(MDDropdownMenu):
    def __init__(self, *args, status_names: dict[HintStatus, str], status_icons: dict[HintStatus, str], 
                 dropdown_callback: Callable[[HintStatus], None], **kwargs):
        # Create items before calling super().__init__
        items = []
        for status in (HintStatus.HINT_NO_PRIORITY, HintStatus.HINT_PRIORITY, HintStatus.HINT_AVOID):
            name = status_names[status]
            items.append({
                "text": name,
                "leading_icon": status_icons[status],
                "on_release": lambda x=status: self._on_item_release(dropdown_callback, x)
            })
        
        # Pass items to parent constructor
        super().__init__(*args, items=items, **kwargs)
    
    def _on_item_release(self, callback: Callable[[HintStatus], None], status: HintStatus):
        """Handle item release and dismiss dropdown"""
        callback(status)
        self.dismiss()

class ListItemTooltip(MDTooltip):
    """
    Base class for tooltip behavior.
    
    Provides tooltip functionality for game list items.
    """
    pass

class GameListItemLongText(ListItemTooltip, MDListItemSupportingText):
    """
    List item with tooltip behavior for long text.
    
    Implements a list item with tooltip behavior for text that may be
    truncated and needs a tooltip to show the full content.
    
    Attributes:
        text (StringProperty): The display text
        tooltip_text (StringProperty): The full text shown in tooltip
    """
    text = StringProperty("")
    tooltip_text = StringProperty("")

    def __init__(self, text, tooltip_text, **kwargs):
        """
        Initialize the GameListItemLongText.
        
        Args:
            text (str): The display text
            tooltip_text (str): The tooltip text for long items
        """
        self.text = text
        self.tooltip_text = tooltip_text
        super().__init__(**kwargs)


class GameListItemShortText(MDListItemSupportingText):
    """
    List item with no tooltip behavior for short text.

    Implements a list item without tooltip behavior for text that
    fits within the display area without truncation.
    
    Attributes:
        text (StringProperty): The display text
    """
    text = StringProperty("")
    
    def __init__(self, text, **kwargs):
        """
        Initialize the GameListItemShortText.
        
        Args:
            text (str): The display text
        """
        self.text = text
        super().__init__(**kwargs)

class GameListItem(MDListItem, CommonElevationBehavior):
    """
    Widget for displaying individual game items in the game list.
    
    This displays a single item from a dictionary (genre, theme, etc).
    Supports both long text with tooltips and short text without tooltips.
    
    Attributes:
        text (StringProperty): The display text
        icon (StringProperty): The icon to display
        tooltip_text (StringProperty): The tooltip text for long items
    """
    text = StringProperty("")
    icon = StringProperty("")
    tooltip_text = StringProperty("")
    
    def __init__(self, text="", icon="blank", tooltip_text="", **kwargs):
        """
        Initialize the GameListItem.
        
        Args:
            text (str): The display text
            icon (str): The icon to display (default: "blank")
            tooltip_text (str): The tooltip text for long items (default: "")
        """
        super().__init__(**kwargs)
        self.text = text
        self.icon = icon
        self.tooltip_text = tooltip_text
        self.width = 256
        self.pos_hint = {"center_y": 0.5}

        Clock.schedule_once(lambda x: self.remove_extra_container())
            # Create and add the text widget
        if "..." in self.text:
            text_widget = GameListItemLongText(text, tooltip_text)
        else:
            text_widget = GameListItemShortText(text)
        self.add_widget(text_widget)

    def remove_extra_container(self):
        """
        Remove the extra trailing container from the list item.
        
        This method cleans up the widget structure by removing
        unnecessary container elements.
        """
        try:
            self.remove_widget(self.ids.trailing_container)
        except:
            pass

class GameListPanel(MDExpansionPanel):
    """
    Expansion panel for displaying game information in the game or hintlist.
    
    This class is used to display a game item in the game list.
    It is a subclass of MDExpansionPanel and can display either
    slot items (if hints are present) or game metadata.
    
    Attributes:
        item_name (StringProperty): The name of the item
        item_data (DictProperty): Dictionary containing item information
        icon (StringProperty): The icon to display (default: "game-controller")
        leading_avatar (MDListItemLeadingAvatar): Avatar widget for the item
        panel_header (MDExpansionPanelHeader): Header widget for the panel
        panel_content (MDExpansionPanelContent): Content widget for the panel
        panel_header_layout (ObjectProperty): Layout for the panel header
        on_game_select (ObjectProperty): Callback function for game selection
    """
    item_name: StringProperty
    item_data: Any
    icon = StringProperty("game-controller")
    leading_avatar: MDListItemLeadingAvatar
    panel_header: MDExpansionPanelHeader
    panel_content: MDExpansionPanelContent
    panel_header_layout: ObjectProperty
    on_game_select: ObjectProperty = None
    app: MDApp
    
    def __init__(self, item_name, item_data, on_game_select=None, **kwargs):
        """
        Initialize the GameListPanel.
        
        Args:
            item_name (str): The name of the item
            item_data (dict): Dictionary containing item information
            on_game_select: Callback function for game selection
            **kwargs: Additional keyword arguments for MDExpansionPanel
        """
        super().__init__(**kwargs)
        self.app = MDApp.get_running_app()
        self.item_name = item_name
        self.item_data = item_data
        self.on_game_select = on_game_select
        self.width = 256
        self.pos_hint = {"center_y": 0.5}
        if isinstance(self.item_data, UIPlayerData):
            Clock.schedule_once(lambda x: self.populate_slot_item(ctx=self.app.ctx))
            Clock.schedule_once(lambda x: self.set_self_height(), 1)
        else:
            Clock.schedule_once(lambda x: self.populate_game_item())

    def set_self_height(self):
        self.panel_content.height = self.panel_content.minimum_height

    def populate_slot_item(self, ctx: "CommonContext"):
        """
        Populate the panel with slot items when hints are present.
        
        This method sets up the panel to display slot information
        including the header with avatar and slot items for each hint.
        """
        self.panel_header = self.ids.panel_header
        self.panel_content = self.ids.panel_content
        self.panel_header_layout = SlotListItemHeader(item_data=self.item_data, panel=self)
        self.leading_avatar = self.panel_header_layout.ids.leading_avatar
        self.panel_header.add_widget(self.panel_header_layout)
        self.leading_avatar.source = "" #self.item_data['avatar']
        if self.item_data.bk_mode:
            self.panel_header_layout.ids.slot_item_container.add_widget(BaseListItemIcon(icon="food", theme_font_size="Custom", font_size=dp(14), pos_hint={"center_y": 0.5}),1)
        if self.item_data.in_call:
            self.panel_header_layout.ids.slot_item_container.add_widget(BaseListItemIcon(icon="headphones", theme_font_size="Custom", font_size=dp(14), pos_hint={"center_y": 0.5}),1)
        if self.item_data.game_status == "GOAL":
            self.panel_header_layout.ids.game_item_container.add_widget(BaseListItemIcon(icon="flag_checkered", theme_font_size="Custom", font_size=dp(14), pos_hint={"center_y": 0.5}),1)
        i = 1 if self.app.theme_cls.theme_style == "Dark" else 0
        item_colors = {
            "trap": self.app.theme_mw.markup_tags_theme.trap_item_color[i],
            "regular": self.app.theme_mw.markup_tags_theme.regular_item_color[i],
            "useful": self.app.theme_mw.markup_tags_theme.useful_item_color[i],
            "progression_deprioritized": self.app.theme_mw.markup_tags_theme.progression_deprioritized_item_color[i],
            "progression": self.app.theme_mw.markup_tags_theme.progression_item_color[i],
            "progression_goal": self.app.theme_mw.markup_tags_theme.progression_goal_item_color[i],
            }
        for hint in self.item_data.hints.values():
            if not hint.hide or self.app.show_all_hints:
                item_widget = SlotListItem(hint_data=hint, game_status=self.item_data.game_status, shadow_colors=item_colors)
                self.panel_content.add_widget(item_widget)

    def populate_game_item(self):
        """
        Populate the panel with game metadata when no hints are present.
        
        This method sets up the panel to display game information
        including genres, themes, keywords, player perspectives, ratings,
        and release dates.
        """
        self.panel_header = self.ids.panel_header
        self.panel_content = self.ids.panel_content
        self.panel_header_layout = GameListItemHeader(
            game_module=self.item_name, 
            game_data=self.item_data, 
            panel=self,
            on_game_select=self.on_game_select
        )
        self.leading_avatar = self.panel_header_layout.ids.leading_avatar
        self.panel_header.add_widget(self.panel_header_layout)
        self.leading_avatar.source = self.item_data['cover_url']
        for item in self.item_data:
            if item == "genres" and self.item_data['genres']:
                list_tooltip = self.list_tooltip(self.item_data['genres'])
                self.panel_content.add_widget(GameListItem(text=list_tooltip['label'], icon="dice-multiple", tooltip_text=list_tooltip['tooltip']))
            elif item == "themes" and self.item_data['themes']:
                list_tooltip = self.list_tooltip(self.item_data['themes'])
                self.panel_content.add_widget(GameListItem(text=list_tooltip['label'], icon="sword", tooltip_text=list_tooltip['tooltip']))
            # elif item == "keywords" and self.item_data['keywords']:
            #     list_tooltip = self.list_tooltip(self.item_data['keywords'])
            #     self.panel_content.add_widget(GameListItem(text=list_tooltip['label'], icon="tag-outline", tooltip_text=list_tooltip['tooltip']))
            elif item == "player_perspectives" and self.item_data['player_perspectives']:
                list_tooltip = self.list_tooltip(self.item_data['player_perspectives'])
                self.panel_content.add_widget(GameListItem(text=list_tooltip['label'], icon="eye-outline", tooltip_text=list_tooltip['tooltip']))
            elif item == "rating" and self.item_data['rating']:
                list_tooltip = self.list_tooltip(self.item_data['rating'])
                self.panel_content.add_widget(GameListItem(text=list_tooltip['label'], icon="alert-box-outline", tooltip_text=list_tooltip['tooltip']))
            elif item == "release_date" and self.item_data['release_date']:
                self.panel_content.add_widget(GameListItem(text=str(self.item_data['release_date']), icon="calendar-month", tooltip_text=str(self.item_data['release_date'])))
                
    def list_tooltip(self, item_list: list[str]) -> dict[str, str]:
        """
        Create tooltip text for a list of items.
        
        Wraps the text to fit within specified width constraints and creates
        both a label (shortened) and tooltip (full) version.
        
        Args:
            item_list (list[str]): List of items to create tooltip for
            
        Returns:
            dict[str, str]: Dictionary with 'label' (shortened text) and 
                           'tooltip' (full text) keys
        """
        full_list = ", ".join(item_list).rstrip(", ")
        wrapped_list = wrap(full_list, width=18, break_on_hyphens=False, max_lines=3)
        item_dict = {
            "label": "\n".join(wrapped_list).rstrip("\n"),
            "tooltip": "\n".join(wrap(full_list, width=40, break_on_hyphens=False)).rstrip("\n")
        }
        return item_dict

    def toggle_expansion(self, instance):
        """
        Toggle the expansion state of the panel.
        
        Animates the padding change and opens/closes the panel
        with appropriate chevron icon updates.
        
        Args:
            instance: The widget instance that triggered the toggle
        """
        Animation(
            padding=[dp(4), dp(12), dp(4), dp(12)]
            if not self.is_open
            else [dp(8),dp(4),dp(8),dp(4)],
            d=0.2,
        ).start(self)
        self.open() if not self.is_open else self.close()
        self.set_chevron_up(instance) if self.is_open else self.set_chevron_down(instance)

class GameTrailingPressedIconButton(
    ButtonBehavior, RotateBehavior, MDListItemTrailingIcon
):
    """
    A button widget that combines button behavior, rotation behavior,
    and trailing icon functionality for game list items.
    
    This class provides an interactive icon button that can be pressed
    and rotated, typically used for trailing icons in list items.
    """
    ...