import copy
from typing import TYPE_CHECKING, Dict, List, Optional

from Options import Option
from .data import data, NATIONAL_ID_TO_SPECIES_ID, EncounterType, EventData
from .options import ShuffleBuildingEntrances, ShuffleDungeonEntrances

if TYPE_CHECKING:
    from . import PokemonFRLGWorld


def ut_set_options(world: "PokemonFRLGWorld") -> None:
    for key, value in world.ut_slot_data.items():
        opt: Optional[Option] = getattr(world.options, key, None)
        if opt is not None:
            setattr(world.options, key, opt.from_any(value))
    world.tracker_world["map_page_maps"] = copy.deepcopy(MAP_PAGE_MAPS)
    world.tracker_world["map_page_locations"] = copy.deepcopy(MAP_PAGE_LOCATIONS)


def ut_set_maps(world: "PokemonFRLGWorld") -> None:
    extra_key_items = world.options.extra_key_items
    route_10_modified = "Modify Route 10" in world.options.modify_world_state.value
    route_12_boulders = "Route 12 Boulders" in world.options.modify_world_state.value
    route_12_modified = "Modify Route 12" in world.options.modify_world_state.value
    route_23_trees = "Route 23 Trees" in world.options.modify_world_state.value
    route_23_modified = "Modify Route 23" in world.options.modify_world_state.value
    if "Modify Route 2" in world.options.modify_world_state.value:
        world.tracker_world["map_page_maps"].append("maps/maps_route_2_modified.json")
    else:
        world.tracker_world["map_page_maps"].append("maps/maps_route_2_vanilla.json")
    if "Block Tunnels" in world.options.modify_world_state.value:
        world.tracker_world["map_page_maps"].append("maps/maps_tunnels_blocked.json")
    else:
        world.tracker_world["map_page_maps"].append("maps/maps_tunnels_vanilla.json")
    if "Modify Route 9" in world.options.modify_world_state.value:
        world.tracker_world["map_page_maps"].append("maps/maps_route_9_modified.json")
    else:
        world.tracker_world["map_page_maps"].append("maps/maps_route_9_vanilla.json")
    if extra_key_items and route_10_modified:
        world.tracker_world["map_page_maps"].append("maps/maps_route_10_all.json")
    elif extra_key_items:
        world.tracker_world["map_page_maps"].append("maps/maps_route_10_extra_items.json")
    elif route_10_modified:
        world.tracker_world["map_page_maps"].append("maps/maps_route_10_modified.json")
    else:
        world.tracker_world["map_page_maps"].append("maps/maps_route_10_vanilla.json")
    if route_12_boulders and route_12_modified:
        world.tracker_world["map_page_maps"].append("maps/maps_route_12_all.json")
    elif route_12_boulders:
        world.tracker_world["map_page_maps"].append("maps/maps_route_12_boulders.json")
    elif route_12_modified:
        world.tracker_world["map_page_maps"].append("maps/maps_route_12_modified.json")
    else:
        world.tracker_world["map_page_maps"].append("maps/maps_route_12_vanilla.json")
    if "Modify Route 16" in world.options.modify_world_state.value:
        world.tracker_world["map_page_maps"].append("maps/maps_route_16_modified.json")
    else:
        world.tracker_world["map_page_maps"].append("maps/maps_route_16_vanilla.json")
    if route_23_trees and route_23_modified:
        world.tracker_world["map_page_maps"].append("maps/maps_route_23_all.json")
    elif route_23_trees:
        world.tracker_world["map_page_maps"].append("maps/maps_route_23_trees.json")
    elif route_23_modified:
        world.tracker_world["map_page_maps"].append("maps/maps_route_23_modified.json")
    else:
        world.tracker_world["map_page_maps"].append("maps/maps_route_23_vanilla.json")
    if "Victory Road Rocks" in world.options.modify_world_state.value:
        world.tracker_world["map_page_maps"].append("maps/maps_victory_road_rocks.json")
    else:
        world.tracker_world["map_page_maps"].append("maps/maps_victory_road_vanilla.json")
    if extra_key_items:
        world.tracker_world["map_page_maps"].append("maps/maps_cinnabar_extra_items.json")
    else:
        world.tracker_world["map_page_maps"].append("maps/maps_cinnabar_vanilla.json")
    if world.options.split_teas:
        world.tracker_world["map_page_maps"].append("maps/maps_split_teas.json")
    else:
        world.tracker_world["map_page_maps"].append("maps/maps_vanilla_teas.json")


def ut_set_locations(world: "PokemonFRLGWorld") -> None:
    for starting_town, locations in STARTING_TOWN_PC_LOCATIONS.items():
        if world.ut_slot_data["starting_town"] != data.constants[starting_town]:
            for location in locations:
                try:
                    world.tracker_world["poptracker_name_mapping"].pop(location)
                except KeyError:
                    pass
    if world.options.shuffle_pokemon_centers:
        world.tracker_world["map_page_locations"].append("ut_locations/entrances/pokemon_center_entrances.json")
    else:
        world.tracker_world["map_page_locations"].append("ut_locations/entrances/pokemon_center_overview.json")
    if world.options.shuffle_gyms:
        world.tracker_world["map_page_locations"].append("ut_locations/entrances/gym_entrances.json")
    else:
        world.tracker_world["map_page_locations"].append("ut_locations/entrances/gym_overview.json")
    if world.options.shuffle_marts:
        world.tracker_world["map_page_locations"].append("ut_locations/entrances/mart_entrances.json")
    else:
        world.tracker_world["map_page_locations"].append("ut_locations/entrances/mart_overview.json")
    if world.options.shuffle_harbors:
        world.tracker_world["map_page_locations"].append("ut_locations/entrances/harbor_entrances.json")
    if world.options.shuffle_buildings != ShuffleBuildingEntrances.option_off:
        world.tracker_world["map_page_locations"].append("ut_locations/entrances/building_entrances.json")
    else:
        world.tracker_world["map_page_locations"].append("ut_locations/entrances/building_overview.json")
    if (world.options.shuffle_dungeons != ShuffleDungeonEntrances.option_off
            and world.options.shuffle_dungeons != ShuffleDungeonEntrances.option_seafoam):
        world.tracker_world["map_page_locations"].append("ut_locations/entrances/dungeon_entrances.json")
    else:
        world.tracker_world["map_page_locations"].append("ut_locations/entrances/dungeon_overview.json")
    if world.options.shuffle_interiors:
        world.tracker_world["map_page_locations"].append("ut_locations/entrances/interior_entrances.json")
    else:
        world.tracker_world["map_page_locations"].append("ut_locations/entrances/interior_overview.json")
    if world.options.shuffle_warp_tiles:
        world.tracker_world["map_page_locations"].append("ut_locations/entrances/warp_tile_entrances.json")
    if world.options.shuffle_dropdowns:
        world.tracker_world["map_page_locations"].append("ut_locations/entrances/dropdown_entrances.json")


def ut_set_wild_pokemon(world: "PokemonFRLGWorld") -> None:
    game_version = world.options.game_version.current_key
    wild_encounters = world.ut_slot_data["wild_encounters"]

    for encounter_key, ids in wild_encounters.items():
        already_placed: Dict[int, int] = {}
        index = 0

        if encounter_key.startswith("LAND") or encounter_key.startswith("WATER"):
            encounter_type, map_id = encounter_key.split("_", 1)
        else:
            encounter_type, _, map_id = encounter_key.split("_", 2)

        map_data = world.modified_maps[map_id]
        encounters = (map_data.encounters[EncounterType.LAND] if encounter_type == "LAND" else
                      map_data.encounters[EncounterType.WATER] if encounter_type == "WATER" else
                      map_data.encounters[EncounterType.FISHING])
        slots = encounters.slots[game_version]
        if encounter_type == "OLD":
            slots = [e for i, e in enumerate(slots) if i < 2]
        elif encounter_type == "GOOD":
            slots = [e for i, e in enumerate(slots) if 2 <= i < 5]
        elif encounter_type == "SUPER":
            slots = [e for i, e in enumerate(slots) if i >= 5]

        for slot in slots:
            if slot.species_id in already_placed:
                slot.species_id = already_placed[slot.species_id]
            else:
                new_species_id = NATIONAL_ID_TO_SPECIES_ID[ids[index]]
                already_placed[slot.species_id] = new_species_id
                slot.species_id = new_species_id
                index += 1


def ut_set_legendary_pokemon(world: "PokemonFRLGWorld") -> None:
    for id, legendary_pokemon in world.modified_legendary_pokemon.items():
        if id not in world.modified_events or id not in world.ut_slot_data["static_encounters"]:
            continue

        species_id = NATIONAL_ID_TO_SPECIES_ID[world.ut_slot_data["static_encounters"][id]]
        item = world.modified_events[id].item

        if item.startswith("Static"):
            item = f"Static {world.modified_species[species_id].name}"
        else:
            item = world.modified_species[species_id].name

        new_event = EventData(
            world.modified_events[id].id,
            world.modified_events[id].name,
            item,
            world.modified_events[id].parent_region_id,
            world.modified_events[id].category
        )

        world.modified_events[id] = new_event


def ut_set_misc_pokemon(world: "PokemonFRLGWorld") -> None:
    for id, misc_pokemon in world.modified_misc_pokemon.items():
        if id not in world.modified_events or id not in world.ut_slot_data["static_encounters"]:
            continue

        species_id = NATIONAL_ID_TO_SPECIES_ID[world.ut_slot_data["static_encounters"][id]]
        item = world.modified_events[id].item

        if item.startswith("Static"):
            item = f"Static {world.modified_species[species_id].name}"
        else:
            item = world.modified_species[species_id].name

        new_event = EventData(
            world.modified_events[id].id,
            world.modified_events[id].name,
            item,
            world.modified_events[id].parent_region_id,
            world.modified_events[id].category
        )

        world.modified_events[id] = new_event

    for id, trade_pokemon in world.modified_trade_pokemon.items():
        if id not in world.modified_events or id not in world.ut_slot_data["static_encounters"]:
            continue

        species_id = NATIONAL_ID_TO_SPECIES_ID[world.ut_slot_data["static_encounters"][id]]
        item = world.modified_events[id].item

        if item.startswith("Static"):
            item = f"Static {world.modified_species[species_id].name}"
        else:
            item = world.modified_species[species_id].name

        new_event = EventData(
            world.modified_events[id].id,
            world.modified_events[id].name,
            item,
            world.modified_events[id].parent_region_id,
            world.modified_events[id].category
        )

        world.modified_events[id] = new_event


def ut_set_tm_hm_compatibility(world: "PokemonFRLGWorld") -> None:
    for species_id, species in world.modified_species.items():
        species.tm_hm_compatibility = world.ut_slot_data["tm_hm_compatibility"][species.name]


def ut_set_requested_trade_pokemon(world: "PokemonFRLGWorld") -> None:
    game_version = world.options.game_version.current_key

    for id, trade_pokemon in world.modified_trade_pokemon.items():
        species_id = world.ut_slot_data["requested_trade_pokemon"][id]
        trade_pokemon.requested_species_id[game_version] = species_id
        world.logic.required_trade_pokemon[data.events[id].name] = data.species[species_id].name


def ut_set_entrances(world: "PokemonFRLGWorld") -> None:
    deferred_entrances = (hasattr(world.multiworld, "enforce_deferred_connections")
                          and world.multiworld.enforce_deferred_connections in ("on", "default"))
    datastorage_key_prefix = "pokemon_frlg_{player}_"

    if "entrances" in world.ut_slot_data:
        for entrance_name, region_name in world.ut_slot_data["entrances"].items():
            entrance = world.get_entrance(entrance_name)
            region = world.get_region(region_name)
            entrance.connected_region.entrances.remove(entrance)
            if deferred_entrances:
                entrance.connected_region = None
                datastorage_key = datastorage_key_prefix + entrance_name
                world.found_entrances_datastorage_key.append(datastorage_key)
            else:
                entrance.connect(region)


def ut_reconnect_found_entrances(world: "PokemonFRLGWorld", found_key: str) -> None:
    entrance_name = found_key.split("_")[-1]
    region_name = world.ut_slot_data["entrances"][entrance_name]
    entrance = world.get_entrance(entrance_name)
    if entrance.connected_region is None:
        region = world.get_region(region_name)
        entrance.connect(region)


def map_page_index(data: List[int]) -> int:
    if type(data) is list:
        if data[0] in MAP_PAGE_INDEX and data[1] in MAP_PAGE_INDEX[data[0]]:
            map = MAP_PAGE_INDEX[data[0]]
            map_name = map[data[1]]
            if map_name in MAP_NAME_TO_INDEX:
                return MAP_NAME_TO_INDEX[map_name]
    return 0


MAP_PAGE_INDEX: Dict[int, Dict[int, str]] = {
    768: {0: "pallet_town"},
    1024: {0: "players_house_1f"},
    1025: {0: "players_house_2f"},
    1026: {0: "rivals_house"},
    1027: {0: "oaks_lab"},
    769: {0: "viridian_city"},
    1280: {0: "viridian_house"},
    1281: {0: "viridian_gym"},
    1282: {0: "viridian_school"},
    1283: {0: "viridian_mart"},
    1284: {0: "viridian_pokemon_center_1f"},
    1285: {0: "viridian_pokemon_center_2f"},
    770: {0: "pewter_city"},
    1536: {0: "pewter_museum_1f"},
    1537: {0: "pewter_museum_2f"},
    1538: {0: "pewter_gym"},
    1539: {0: "pewter_mart"},
    1540: {0: "pewter_nidoran_house"},
    1541: {0: "pewter_pokemon_center_1f"},
    1542: {0: "pewter_pokemon_center_2f"},
    1543: {0: "pewter_info_house"},
    771: {0: "cerulean_city"},
    1792: {0: "badge_guys_house"},
    1793: {0: "robbed_house"},
    1794: {0: "cerulean_trade_house"},
    1795: {0: "cerulean_pokemon_center_1f"},
    1796: {0: "cerulean_pokemon_center_2f"},
    1797: {0: "cerulean_gym"},
    1798: {0: "bike_shop"},
    1799: {0: "cerulean_mart"},
    1800: {0: "wonder_news_house"},
    1801: {0: "berry_powder_mans_house"},
    773: {0: "vermilion_city"},
    2304: {0: "vermilion_fishing_house"},
    2305: {0: "vermilion_pokemon_center_1f"},
    2306: {0: "vermilion_pokemon_center_2f"},
    2307: {0: "pokemon_fan_club"},
    2308: {0: "vermilion_trade_house"},
    2309: {0: "vermilion_mart"},
    2310: {0: "vermilion_gym"},
    2311: {0: "vermilion_pidgey_house"},
    772: {0: "lavender_town"},
    2048: {0: "lavender_pokemon_center_1f"},
    2049: {0: "lavender_pokemon_center_2f"},
    2050: {0: "volunteer_pokemon_house"},
    2051: {0: "lavender_cubone_house"},
    2052: {0: "name_raters_house"},
    2053: {0: "lavender_mart"},
    774: {0: "celadon_city"},
    2560: {0: "celadon_department_store_1f"},
    2561: {0: "celadon_department_store_2f"},
    2562: {0: "celadon_department_store_3f"},
    2563: {0: "celadon_department_store_4f"},
    2564: {0: "celadon_department_store_5f"},
    2565: {0: "celadon_department_store_roof"},
    2566: {0: "celadon_department_store_elevator"},
    2567: {0: "celadon_condominiums_1f"},
    2568: {0: "celadon_condominiums_2f"},
    2569: {0: "celadon_condominiums_3f"},
    2570: {0: "celadon_condominiums_roof"},
    2571: {0: "celadon_condominiums_roof_room"},
    2572: {0: "celadon_pokemon_center_1f"},
    2573: {0: "celadon_pokemon_center_2f"},
    2574: {0: "celadon_game_corner"},
    2575: {0: "celadon_game_corner_prize_room"},
    2576: {0: "celadon_gym"},
    2577: {0: "celadon_restaurant"},
    2578: {0: "celadon_rocket_house"},
    2579: {0: "celadon_hotel"},
    775: {0: "fuchsia_city"},
    2817: {0: "fuchsia_mart"},
    2818: {0: "safari_zone_office"},
    2819: {0: "fuchsia_gym"},
    2820: {0: "bills_grandpas_house"},
    2821: {0: "fuchsia_pokemon_center_1f"},
    2822: {0: "fuchsia_pokemon_center_2f"},
    2823: {0: "safari_zone_wardens_house"},
    2824: {0: "fuchsia_fishing_house"},
    2825: {0: "move_deleters_house"},
    778: {0: "saffron_city"},
    3584: {0: "copycats_house_1f"},
    3585: {0: "copycats_house_2f"},
    3586: {0: "saffron_dojo"},
    3587: {0: "saffron_gym"},
    3588: {0: "saffron_pidgey_house"},
    3589: {0: "saffron_mart"},
    3590: {0: "saffron_pokemon_center_1f"},
    3591: {0: "saffron_pokemon_center_2f"},
    3592: {0: "mr_psychics_house"},
    3593: {0: "pokemon_trainer_fan_club"},
    776: {0: "cinnabar_island"},
    3072: {0: "cinnabar_gym"},
    3073: {0: "pokemon_lab_entrance"},
    3074: {0: "pokemon_lab_lounge"},
    3075: {0: "pokemon_lab_research_room"},
    3076: {0: "pokemon_lab_experiment_room"},
    3077: {0: "cinnabar_pokemon_center_1f"},
    3078: {0: "cinnabar_pokemon_center_2f"},
    3079: {0: "cinnabar_mart"},
    777: {0: "indigo_plateau"},
    3328: {0: "indigo_plateau_pokemon_center_1f"},
    3329: {0: "indigo_plateau_pokemon_center_2f"},
    331: {0: "loreleis_room"},
    332: {0: "brunos_room"},
    333: {0: "agathas_room"},
    334: {0: "lances_room"},
    335: {0: "champions_room"},
    336: {0: "hall_of_fame"},
    3840: {0: "viridian_forest_south_entrance"},
    256: {0: "viridian_forest_north",
          1: "viridian_forest_south"},
    3843: {0: "viridian_forest_north_entrance"},
    257: {0: "mt_moon_1f"},
    258: {0: "mt_moon_b1f"},
    259: {0: "mt_moon_b2f"},
    286: {0: "underground_path_north_entrance"},
    287: {0: "underground_path_north_south_tunnel_north",
          1: "underground_path_north_south_tunnel_south"},
    288: {0: "underground_path_south_entrance"},
    260: {0: "ss_anne_exterior"},
    261: {0: "ss_anne_1f_corridor"},
    262: {0: "ss_anne_2f_corridor"},
    263: {0: "ss_anne_3f_corridor"},
    264: {0: "ss_anne_b1f_corridor"},
    265: {0: "ss_anne_deck"},
    266: {0: "ss_anne_kitchen"},
    267: {0: "ss_anne_captains_office"},
    268: {0: "ss_anne_1f_room_1"},
    269: {0: "ss_anne_1f_room_2"},
    270: {0: "ss_anne_1f_room_3"},
    271: {0: "ss_anne_1f_room_4"},
    272: {0: "ss_anne_1f_room_5"},
    273: {0: "ss_anne_1f_room_7"},
    274: {0: "ss_anne_2f_room_1"},
    275: {0: "ss_anne_2f_room_2"},
    276: {0: "ss_anne_2f_room_3"},
    277: {0: "ss_anne_2f_room_4"},
    278: {0: "ss_anne_2f_room_5"},
    279: {0: "ss_anne_2f_room_6"},
    280: {0: "ss_anne_b1f_room_1"},
    281: {0: "ss_anne_b1f_room_2"},
    282: {0: "ss_anne_b1f_room_3"},
    283: {0: "ss_anne_b1f_room_4"},
    284: {0: "ss_anne_b1f_room_5"},
    285: {0: "ss_anne_1f_room_6"},
    292: {0: "digletts_cave_north_entrance"},
    293: {0: "digletts_cave_b1f_north",
          1: "digletts_cave_b1f_south"},
    294: {0: "digletts_cave_south_entrance"},
    337: {0: "rock_tunnel_1f"},
    338: {0: "rock_tunnel_b1f"},
    351: {0: "power_plant"},
    344: {0: "pokemon_tower_1f"},
    345: {0: "pokemon_tower_2f"},
    346: {0: "pokemon_tower_3f"},
    347: {0: "pokemon_tower_4f"},
    348: {0: "pokemon_tower_5f"},
    349: {0: "pokemon_tower_6f"},
    350: {0: "pokemon_tower_7f"},
    289: {0: "underground_path_west_entrance"},
    290: {0: "underground_path_east_west_tunnel_west",
          1: "underground_path_east_west_tunnel_east"},
    291: {0: "underground_path_east_entrance"},
    298: {0: "rocket_hideout_b1f"},
    299: {0: "rocket_hideout_b2f"},
    300: {0: "rocket_hideout_b3f"},
    301: {0: "rocket_hideout_b4f"},
    2816: {0: "safari_zone_entrance"},
    319: {0: "safari_zone_center"},
    320: {0: "safari_zone_east"},
    321: {0: "safari_zone_north"},
    322: {0: "safari_zone_west"},
    323: {0: "safari_zone_center_rest_house"},
    324: {0: "safari_zone_east_rest_house"},
    325: {0: "safari_zone_north_rest_house"},
    326: {0: "safari_zone_west_rest_house"},
    327: {0: "safari_zone_secret_house"},
    303: {0: "silph_co_1f"},
    304: {0: "silph_co_2f"},
    305: {0: "silph_co_3f"},
    306: {0: "silph_co_4f"},
    307: {0: "silph_co_5f"},
    308: {0: "silph_co_6f"},
    309: {0: "silph_co_7f"},
    310: {0: "silph_co_8f"},
    311: {0: "silph_co_9f"},
    312: {0: "silph_co_10f"},
    313: {0: "silph_co_11f"},
    339: {0: "seafoam_islands_1f"},
    340: {0: "seafoam_islands_b1f"},
    341: {0: "seafoam_islands_b2f"},
    342: {0: "seafoam_islands_b3f"},
    343: {0: "seafoam_islands_b4f"},
    315: {0: "pokemon_mansion_1f"},
    316: {0: "pokemon_mansion_2f"},
    317: {0: "pokemon_mansion_3f"},
    318: {0: "pokemon_mansion_b1f"},
    295: {0: "victory_road_1f"},
    296: {0: "victory_road_2f"},
    297: {0: "victory_road_3f"},
    328: {0: "cerulean_cave_1f"},
    329: {0: "cerulean_cave_2f"},
    330: {0: "cerulean_cave_b1f"},
    780: {0: "one_island"},
    8192: {0: "one_island_pokemon_center_1f"},
    8193: {0: "one_island_pokemon_center_2f"},
    8194: {0: "one_island_old_couples_house"},
    8195: {0: "one_island_lass_house"},
    8196: {0: "one_island_harbor"},
    814: {0: "treasure_beach"},
    813: {0: "kindle_road_north",
          1: "kindle_road_center_top",
          2: "kindle_road_center_bottom",
          3: "kindle_road_south"},
    569: {0: "ember_spa"},
    352: {0: "mt_ember_ruby_path_b4f"},
    353: {0: "mt_ember_exterior"},
    354: {0: "mt_ember_summit_path_1f"},
    355: {0: "mt_ember_summit_path_2f"},
    356: {0: "mt_ember_summit_path_3f"},
    357: {0: "mt_ember_summit"},
    358: {0: "mt_ember_ruby_path_b5f"},
    359: {0: "mt_ember_ruby_path_1f"},
    360: {0: "mt_ember_ruby_path_b1f"},
    361: {0: "mt_ember_ruby_path_b2f"},
    362: {0: "mt_ember_ruby_path_b3f"},
    364: {0: "mt_ember_ruby_path_b1f_return"},
    363: {0: "mt_ember_ruby_path_b2f_return"},
    781: {0: "two_island"},
    8448: {0: "two_island_game_corner"},
    8449: {0: "two_island_move_maniacs_house"},
    8450: {0: "two_island_pokemon_center_1f"},
    8451: {0: "two_island_pokemon_center_2f"},
    8452: {0: "two_island_harbor"},
    815: {0: "cape_brink"},
    10240: {0: "starter_tutors_house"},
    782: {0: "three_island"},
    8704: {0: "three_island_lostelles_house"},
    8705: {0: "three_island_pokemon_center_1f"},
    8706: {0: "three_island_pokemon_center_2f"},
    8707: {0: "three_island_mart"},
    8708: {0: "three_island_sabrina_fans_house"},
    8709: {0: "three_island_beautys_house"},
    8710: {0: "three_island_worried_fathers_house"},
    8711: {0: "three_island_lostelles_friends_house"},
    817: {0: "three_isle_port"},
    9728: {0: "three_isle_port_harbor"},
    546: {0: "three_isle_path"},
    816: {0: "bond_bridge_west",
          1: "bond_bridge_east"},
    365: {0: "berry_forest"},
    783: {0: "four_island"},
    8960: {0: "four_island_pokemon_day_care"},
    8961: {0: "four_island_pokemon_center_1f"},
    8962: {0: "four_island_pokemon_center_2f"},
    8963: {0: "four_island_move_tutors_house"},
    8964: {0: "loreleis_house"},
    8965: {0: "four_island_harbor"},
    8966: {0: "sticker_mans_house"},
    8967: {0: "four_island_mart"},
    366: {0: "icefall_cave_front"},
    367: {0: "icefall_cave_1f"},
    368: {0: "icefall_cave_b1f"},
    369: {0: "icefall_cave_back"},
    784: {0: "five_island"},
    9216: {0: "five_island_pokemon_center_1f"},
    9217: {0: "five_island_pokemon_center_2f"},
    9218: {0: "five_island_harbor"},
    9219: {0: "five_island_couples_house"},
    9220: {0: "five_island_old_mans_house"},
    824: {0: "five_isle_meadow"},
    370: {0: "rocket_warehouse"},
    825: {0: "memorial_pillar_north",
          1: "memorial_pillar_south"},
    823: {0: "water_labyrinth_west",
          1: "water_labyrinth_east"},
    822: {0: "resort_gorgeous_west",
          1: "resort_gorgeous_east"},
    9984: {0: "resort_gorgeous_house"},
    524: {0: "lost_cave_entrance"},
    525: {0: "lost_cave_room_1"},
    526: {0: "lost_cave_room_2"},
    527: {0: "lost_cave_room_3"},
    528: {0: "lost_cave_room_4"},
    529: {0: "lost_cave_room_5"},
    530: {0: "lost_cave_room_6"},
    531: {0: "lost_cave_room_7"},
    532: {0: "lost_cave_room_8"},
    533: {0: "lost_cave_room_9"},
    534: {0: "lost_cave_room_10"},
    535: {0: "lost_cave_room_11"},
    536: {0: "lost_cave_room_12"},
    537: {0: "lost_cave_room_13"},
    538: {0: "lost_cave_room_14"},
    786: {0: "six_island"},
    9472: {0: "six_island_pokemon_center_1f"},
    9473: {0: "six_island_pokemon_center_2f"},
    9474: {0: "six_island_harbor"},
    9475: {0: "six_island_old_mans_house"},
    9476: {0: "six_island_mart"},
    828: {0: "water_path_north",
          1: "water_path_center",
          2: "water_path_south"},
    10496: {0: "water_path_heracross_womans_house"},
    10497: {0: "water_path_mans_house"},
    829: {0: "ruin_valley"},
    371: {0: "dotted_hole_1f"},
    372: {0: "dotted_hole_b1f"},
    373: {0: "dotted_hole_b2f"},
    374: {0: "dotted_hole_b3f"},
    375: {0: "dotted_hole_b4f"},
    376: {0: "dotted_hole_sapphire_room"},
    827: {0: "green_path_west",
          1: "green_path_east"},
    377: {0: "pattern_bush"},
    826: {0: "outcast_island_north",
          1: "outcast_island_south"},
    378: {0: "altering_cave"},
    785: {0: "seven_island"},
    7936: {0: "seven_island_battle_house"},
    7937: {0: "seven_island_battle_house"},
    7938: {0: "seven_island_mart"},
    7939: {0: "seven_island_pokemon_center_1f"},
    7940: {0: "seven_island_pokemon_center_2f"},
    7942: {0: "seven_island_harbor"},
    831: {0: "canyon_entrance"},
    832: {0: "sevault_canyon_north",
          1: "sevault_canyon_south"},
    10752: {0: "sevault_canyon_chansey_house"},
    547: {0: "tanoby_key"},
    833: {0: "tanoby_ruins_west",
          1: "tanoby_ruins_center_left",
          2: "tanoby_ruins_center_right",
          3: "tanoby_ruins_east"},
    539: {0: "monean_chamber"},
    540: {0: "liptoo_chamber"},
    541: {0: "weepth_chamber"},
    542: {0: "dilford_chamber"},
    543: {0: "scufib_chamber"},
    544: {0: "rixy_chamber"},
    545: {0: "viapois_chamber"},
    830: {0: "trainer_tower_exterior"},
    522: {0: "trainer_tower_lobby"},
    512: {0: "navel_rock_exterior"},
    548: {0: "navel_rock_1f"},
    549: {0: "navel_rock_summit"},
    550: {0: "navel_rock_base"},
    551: {0: "navel_rock_summit_path_2f"},
    552: {0: "navel_rock_summit_path_3f"},
    553: {0: "navel_rock_summit_path_4f"},
    554: {0: "navel_rock_summit_path_5f"},
    555: {0: "navel_rock_base_path_b1f"},
    556: {0: "navel_rock_base_path_b2f"},
    557: {0: "navel_rock_base_path_b3f"},
    558: {0: "navel_rock_base_path_b4f"},
    559: {0: "navel_rock_base_path_b5f"},
    560: {0: "navel_rock_base_path_b6f"},
    561: {0: "navel_rock_base_path_b7f"},
    562: {0: "navel_rock_base_path_b8f"},
    563: {0: "navel_rock_base_path_b9f"},
    564: {0: "navel_rock_base_path_b10f"},
    565: {0: "navel_rock_base_path_b11f"},
    566: {0: "navel_rock_b1f"},
    567: {0: "navel_rock_fork_north",
          1: "navel_rock_fork_center",
          2: "navel_rock_fork_south"},
    571: {0: "navel_rock_harbor"},
    568: {0: "birth_island"},
    570: {0: "birth_island_harbor"},
    787: {0: "route_1"},
    788: {0: "route_2_north",
          1: "route_2_south"},
    3841: {0: "route_2_trade_house"},
    3842: {0: "route_2_gate"},
    789: {0: "route_3_west",
          1: "route_3_east"},
    790: {0: "route_4_west",
          1: "route_4_east"},
    4096: {0: "route_4_pokemon_center_1f",
           1: "route_4_pokemon_center_2f"},
    791: {0: "route_5"},
    4352: {0: "route_5_pokemon_day_care"},
    4353: {0: "route_5_gate"},
    792: {0: "route_6"},
    4608: {0: "route_6_gate"},
    793: {0: "route_7"},
    4864: {0: "route_7_gate"},
    794: {0: "route_8_west",
          1: "route_8_east"},
    5120: {0: "route_8_gate"},
    795: {0: "route_9_west",
          1: "route_9_east"},
    796: {0: "route_10_north",
          1: "route_10_south"},
    5376: {0: "route_10_pokemon_center_1f"},
    5377: {0: "route_10_pokemon_center_2f"},
    797: {0: "route_11_west",
          1: "route_11_east"},
    5632: {0: "route_11_gate_1f"},
    5633: {0: "route_11_gate_2f"},
    798: {0: "route_12_north",
          1: "route_12_center",
          2: "route_12_south"},
    5888: {0: "route_12_gate_1f"},
    5889: {0: "route_12_gate_2f"},
    5890: {0: "route_12_fishing_house"},
    799: {0: "route_13_west",
          1: "route_13_east"},
    800: {0: "route_14_north",
          1: "route_14_south"},
    801: {0: "route_15_west",
          1: "route_15_east"},
    6144: {0: "route_15_gate_1f"},
    6145: {0: "route_15_gate_2f"},
    802: {0: "route_16"},
    6400: {0: "route_16_house"},
    6401: {0: "route_16_gate_1f"},
    6402: {0: "route_16_gate_2f"},
    803: {0: "route_17_north",
          1: "route_17_center_top",
          2: "route_17_center_bottom",
          3: "route_17_south"},
    804: {0: "route_18_west",
          1: "route_18_east"},
    6656: {0: "route_18_gate_1f"},
    6657: {0: "route_18_gate_2f"},
    805: {0: "route_19_north",
          1: "route_19_south"},
    806: {0: "route_20_west",
          1: "route_20_center",
          2: "route_20_east"},
    807: {0: "route_21_north"},
    808: {0: "route_21_south"},
    809: {0: "route_22"},
    7168: {0: "route_22_gate"},
    810: {0: "route_23_north",
          1: "route_23_center_top",
          2: "route_23_center_bottom",
          3: "route_23_south"},
    811: {0: "route_24"},
    812: {0: "route_25_west",
          1: "route_25_east"},
    7680: {0: "sea_cottage"}
}

MAP_NAME_TO_INDEX: Dict[str, int] = {
    "kanto_sevii": 0,
    "kanto": 1,
    "sevii_123": 2,
    "sevii_45": 3,
    "sevii_67": 4,
    "kanto_sevii_er": 5,
    "kanto_er": 6,
    "sevii_123_er": 7,
    "sevii_45_er": 8,
    "sevii_67_er": 9,
    "pallet_town": 10,
    "players_house_1f": 11,
    "players_house_2f": 12,
    "rivals_house": 13,
    "oaks_lab": 14,
    "viridian_city": 15,
    "viridian_house": 16,
    "viridian_gym": 17,
    "viridian_school": 18,
    "viridian_mart": 19,
    "viridian_pokemon_center_1f": 20,
    "viridian_pokemon_center_2f": 21,
    "pewter_city": 22,
    "pewter_museum_1f": 23,
    "pewter_museum_2f": 24,
    "pewter_gym": 25,
    "pewter_mart": 26,
    "pewter_nidoran_house": 27,
    "pewter_pokemon_center_1f": 28,
    "pewter_pokemon_center_2f": 29,
    "pewter_info_house": 30,
    "cerulean_city": 31,
    "badge_guys_house": 32,
    "robbed_house": 33,
    "cerulean_trade_house": 34,
    "cerulean_pokemon_center_1f": 35,
    "cerulean_pokemon_center_2f": 36,
    "cerulean_gym": 37,
    "bike_shop": 38,
    "cerulean_mart": 39,
    "wonder_news_house": 40,
    "berry_powder_mans_house": 41,
    "vermilion_city": 42,
    "vermilion_fishing_house": 43,
    "vermilion_pokemon_center_1f": 44,
    "vermilion_pokemon_center_2f": 45,
    "pokemon_fan_club": 46,
    "vermilion_trade_house": 47,
    "vermilion_mart": 48,
    "vermilion_gym": 49,
    "vermilion_pidgey_house": 50,
    "lavender_town": 51,
    "lavender_pokemon_center_1f": 52,
    "lavender_pokemon_center_2f": 53,
    "volunteer_pokemon_house": 54,
    "lavender_cubone_house": 55,
    "name_raters_house": 56,
    "lavender_mart": 57,
    "celadon_city": 58,
    "celadon_department_store_1f": 59,
    "celadon_department_store_2f": 60,
    "celadon_department_store_3f": 61,
    "celadon_department_store_4f": 62,
    "celadon_department_store_5f": 63,
    "celadon_department_store_roof": 64,
    "celadon_department_store_elevator": 65,
    "celadon_condominiums_1f": 451,
    "celadon_condominiums_2f": 66,
    "celadon_condominiums_3f": 67,
    "celadon_condominiums_roof": 68,
    "celadon_condominiums_roof_room": 69,
    "celadon_pokemon_center_1f": 70,
    "celadon_pokemon_center_2f": 71,
    "celadon_game_corner": 72,
    "celadon_game_corner_prize_room": 73,
    "celadon_gym": 74,
    "celadon_restaurant": 75,
    "celadon_rocket_house": 76,
    "celadon_hotel": 77,
    "fuchsia_city": 78,
    "fuchsia_mart": 79,
    "safari_zone_office": 80,
    "fuchsia_gym": 81,
    "bills_grandpas_house": 82,
    "fuchsia_pokemon_center_1f": 83,
    "fuchsia_pokemon_center_2f": 84,
    "safari_zone_wardens_house": 85,
    "fuchsia_fishing_house": 86,
    "move_deleters_house": 87,
    "saffron_city": 88,
    "copycats_house_1f": 89,
    "copycats_house_2f": 90,
    "saffron_dojo": 91,
    "saffron_gym": 92,
    "saffron_pidgey_house": 93,
    "saffron_mart": 94,
    "saffron_pokemon_center_1f": 95,
    "saffron_pokemon_center_2f": 96,
    "mr_psychics_house": 97,
    "pokemon_trainer_fan_club": 98,
    "cinnabar_island": 450,
    "cinnabar_gym": 99,
    "pokemon_lab_entrance": 100,
    "pokemon_lab_lounge": 101,
    "pokemon_lab_research_room": 102,
    "pokemon_lab_experiment_room": 103,
    "cinnabar_pokemon_center_1f": 104,
    "cinnabar_pokemon_center_2f": 105,
    "cinnabar_mart": 106,
    "indigo_plateau": 107,
    "indigo_plateau_pokemon_center_1f": 108,
    "indigo_plateau_pokemon_center_2f": 109,
    "loreleis_room": 110,
    "brunos_room": 111,
    "agathas_room": 112,
    "lances_room": 113,
    "champions_room": 114,
    "hall_of_fame": 115,
    "viridian_forest_north": 116,
    "viridian_forest_south": 117,
    "viridian_forest_south_entrance": 118,
    "viridian_forest_north_entrance": 119,
    "mt_moon_1f": 120,
    "mt_moon_b1f": 121,
    "mt_moon_b2f": 122,
    "underground_path_north_entrance": 123,
    "underground_path_south_entrance": 124,
    "underground_path_north_south_tunnel_north": 125,
    "underground_path_north_south_tunnel_south": 126,
    "ss_anne_exterior": 127,
    "ss_anne_1f_corridor": 128,
    "ss_anne_kitchen": 129,
    "ss_anne_1f_room_1": 130,
    "ss_anne_1f_room_2": 131,
    "ss_anne_1f_room_3": 132,
    "ss_anne_1f_room_4": 133,
    "ss_anne_1f_room_5": 134,
    "ss_anne_1f_room_6": 135,
    "ss_anne_1f_room_7": 136,
    "ss_anne_b1f_corridor": 137,
    "ss_anne_b1f_room_1": 138,
    "ss_anne_b1f_room_2": 139,
    "ss_anne_b1f_room_3": 140,
    "ss_anne_b1f_room_4": 141,
    "ss_anne_b1f_room_5": 142,
    "ss_anne_2f_corridor": 143,
    "ss_anne_2f_room_1": 144,
    "ss_anne_2f_room_2": 145,
    "ss_anne_2f_room_3": 146,
    "ss_anne_2f_room_4": 147,
    "ss_anne_2f_room_5": 148,
    "ss_anne_2f_room_6": 149,
    "ss_anne_3f_corridor": 150,
    "ss_anne_deck": 151,
    "ss_anne_captains_office": 152,
    "digletts_cave_north_entrance": 153,
    "digletts_cave_south_entrance": 154,
    "digletts_cave_b1f_north": 155,
    "digletts_cave_b1f_south": 156,
    "rock_tunnel_1f": 157,
    "rock_tunnel_b1f": 158,
    "power_plant": 159,
    "pokemon_tower_1f": 160,
    "pokemon_tower_2f": 161,
    "pokemon_tower_3f": 162,
    "pokemon_tower_4f": 163,
    "pokemon_tower_5f": 164,
    "pokemon_tower_6f": 165,
    "pokemon_tower_7f": 166,
    "underground_path_east_entrance": 167,
    "underground_path_west_entrance": 168,
    "underground_path_east_west_tunnel_west": 169,
    "underground_path_east_west_tunnel_east": 170,
    "rocket_hideout_b1f": 171,
    "rocket_hideout_b2f": 172,
    "rocket_hideout_b3f": 173,
    "rocket_hideout_b4f": 174,
    "safari_zone_entrance": 175,
    "safari_zone_center": 176,
    "safari_zone_center_rest_house": 177,
    "safari_zone_east": 178,
    "safari_zone_east_rest_house": 179,
    "safari_zone_north": 180,
    "safari_zone_north_rest_house": 181,
    "safari_zone_west": 182,
    "safari_zone_west_rest_house": 183,
    "safari_zone_secret_house": 184,
    "silph_co_1f": 185,
    "silph_co_2f": 186,
    "silph_co_3f": 187,
    "silph_co_4f": 188,
    "silph_co_5f": 189,
    "silph_co_6f": 190,
    "silph_co_7f": 191,
    "silph_co_8f": 192,
    "silph_co_9f": 193,
    "silph_co_10f": 194,
    "silph_co_11f": 195,
    "seafoam_islands_1f": 196,
    "seafoam_islands_b1f": 197,
    "seafoam_islands_b2f": 198,
    "seafoam_islands_b3f": 199,
    "seafoam_islands_b4f": 200,
    "pokemon_mansion_1f": 201,
    "pokemon_mansion_2f": 202,
    "pokemon_mansion_3f": 203,
    "pokemon_mansion_b1f": 204,
    "victory_road_1f": 447,
    "victory_road_2f": 448,
    "victory_road_3f": 449,
    "mt_ember_exterior": 205,
    "mt_ember_summit_path_1f": 206,
    "mt_ember_summit_path_2f": 207,
    "mt_ember_summit_path_3f": 208,
    "mt_ember_summit": 209,
    "mt_ember_ruby_path_1f": 210,
    "mt_ember_ruby_path_b1f": 211,
    "mt_ember_ruby_path_b1f_return": 212,
    "mt_ember_ruby_path_b2f": 213,
    "mt_ember_ruby_path_b2f_return": 214,
    "mt_ember_ruby_path_b3f": 215,
    "mt_ember_ruby_path_b4f": 216,
    "mt_ember_ruby_path_b5f": 217,
    "berry_forest": 218,
    "icefall_cave_front": 219,
    "icefall_cave_1f": 220,
    "icefall_cave_b1f": 221,
    "icefall_cave_back": 222,
    "rocket_warehouse": 223,
    "lost_cave_entrance": 224,
    "lost_cave_room_1": 225,
    "lost_cave_room_2": 226,
    "lost_cave_room_3": 227,
    "lost_cave_room_4": 228,
    "lost_cave_room_5": 229,
    "lost_cave_room_6": 230,
    "lost_cave_room_7": 231,
    "lost_cave_room_8": 232,
    "lost_cave_room_9": 233,
    "lost_cave_room_10": 234,
    "lost_cave_room_11": 235,
    "lost_cave_room_12": 236,
    "lost_cave_room_13": 237,
    "lost_cave_room_14": 238,
    "dotted_hole_1f": 239,
    "dotted_hole_b1f": 240,
    "dotted_hole_b2f": 241,
    "dotted_hole_b3f": 242,
    "dotted_hole_b4f": 243,
    "dotted_hole_sapphire_room": 244,
    "pattern_bush": 245,
    "altering_cave": 246,
    "tanoby_key": 247,
    "monean_chamber": 248,
    "liptoo_chamber": 249,
    "weepth_chamber": 250,
    "dilford_chamber": 251,
    "scufib_chamber": 252,
    "rixy_chamber": 253,
    "viapois_chamber": 254,
    "cerulean_cave_1f": 255,
    "cerulean_cave_2f": 256,
    "cerulean_cave_b1f": 257,
    "one_island": 258,
    "one_island_pokemon_center_1f": 259,
    "one_island_pokemon_center_2f": 260,
    "one_island_old_couples_house": 261,
    "one_island_lass_house": 262,
    "one_island_harbor": 263,
    "treasure_beach": 264,
    "kindle_road_north": 265,
    "kindle_road_center_top": 266,
    "kindle_road_center_bottom": 267,
    "kindle_road_south": 268,
    "ember_spa": 269,
    "two_island": 270,
    "two_island_game_corner": 271,
    "two_island_move_maniacs_house": 272,
    "two_island_pokemon_center_1f": 273,
    "two_island_pokemon_center_2f": 274,
    "two_island_harbor": 275,
    "cape_brink": 276,
    "starter_tutors_house": 277,
    "three_isle_port": 278,
    "three_isle_port_harbor": 279,
    "three_isle_path": 280,
    "three_island": 281,
    "three_island_lostelles_house": 282,
    "three_island_pokemon_center_1f": 283,
    "three_island_pokemon_center_2f": 284,
    "three_island_mart": 285,
    "three_island_sabrina_fans_house": 286,
    "three_island_beautys_house": 287,
    "three_island_worried_fathers_house": 288,
    "three_island_lostelles_friends_house": 289,
    "bond_bridge_west": 290,
    "bond_bridge_east": 291,
    "four_island": 292,
    "four_island_pokemon_day_care": 293,
    "four_island_pokemon_center_1f": 294,
    "four_island_pokemon_center_2f": 295,
    "four_island_move_tutors_house": 296,
    "loreleis_house": 297,
    "four_island_harbor": 298,
    "sticker_mans_house": 299,
    "four_island_mart": 300,
    "five_island": 301,
    "five_island_pokemon_center_1f": 302,
    "five_island_pokemon_center_2f": 303,
    "five_island_harbor": 304,
    "five_island_couples_house": 305,
    "five_island_old_mans_house": 306,
    "five_isle_meadow": 307,
    "memorial_pillar_north": 308,
    "memorial_pillar_south": 309,
    "water_labyrinth_west": 310,
    "water_labyrinth_east": 311,
    "resort_gorgeous_west": 312,
    "resort_gorgeous_east": 313,
    "resort_gorgeous_house": 314,
    "six_island": 315,
    "six_island_pokemon_center_1f": 316,
    "six_island_pokemon_center_2f": 317,
    "six_island_harbor": 318,
    "six_island_old_mans_house": 319,
    "six_island_mart": 320,
    "water_path_north": 321,
    "water_path_center": 322,
    "water_path_south": 323,
    "water_path_heracross_womans_house": 324,
    "water_path_mans_house": 325,
    "ruin_valley": 326,
    "green_path_west": 327,
    "green_path_east": 328,
    "outcast_island_north": 329,
    "outcast_island_south": 330,
    "seven_island": 331,
    "seven_island_battle_house": 332,
    "seven_island_mart": 333,
    "seven_island_pokemon_center_1f": 334,
    "seven_island_pokemon_center_2f": 335,
    "seven_island_harbor": 336,
    "canyon_entrance": 337,
    "sevault_canyon_north": 338,
    "sevault_canyon_south": 339,
    "sevault_canyon_chansey_house": 340,
    "tanoby_ruins_west": 341,
    "tanoby_ruins_center_left": 342,
    "tanoby_ruins_center_right": 343,
    "tanoby_ruins_east": 344,
    "trainer_tower_exterior": 345,
    "trainer_tower_lobby": 346,
    "navel_rock_exterior": 347,
    "navel_rock_1f": 348,
    "navel_rock_b1f": 349,
    "navel_rock_fork_north": 350,
    "navel_rock_fork_center": 351,
    "navel_rock_fork_south": 352,
    "navel_rock_summit_path_2f": 353,
    "navel_rock_summit_path_3f": 354,
    "navel_rock_summit_path_4f": 355,
    "navel_rock_summit_path_5f": 356,
    "navel_rock_summit": 357,
    "navel_rock_base_path_b1f": 358,
    "navel_rock_base_path_b2f": 359,
    "navel_rock_base_path_b3f": 360,
    "navel_rock_base_path_b4f": 361,
    "navel_rock_base_path_b5f": 362,
    "navel_rock_base_path_b6f": 363,
    "navel_rock_base_path_b7f": 364,
    "navel_rock_base_path_b8f": 365,
    "navel_rock_base_path_b9f": 366,
    "navel_rock_base_path_b10f": 367,
    "navel_rock_base_path_b11f": 368,
    "navel_rock_base": 369,
    "navel_rock_harbor": 370,
    "birth_island": 371,
    "birth_island_harbor": 372,
    "route_1": 373,
    "route_2_north": 434,
    "route_2_south": 374,
    "route_2_gate": 375,
    "route_2_trade_house": 376,
    "route_3_west": 377,
    "route_3_east": 378,
    "route_4_west": 379,
    "route_4_east": 380,
    "route_4_pokemon_center_1f": 381,
    "route_4_pokemon_center_2f": 382,
    "route_5": 435,
    "route_5_pokemon_day_care": 383,
    "route_5_gate": 384,
    "route_6": 436,
    "route_6_gate": 385,
    "route_7": 437,
    "route_7_gate": 386,
    "route_8_west": 438,
    "route_8_east": 387,
    "route_8_gate": 388,
    "route_9_west": 439,
    "route_9_east": 389,
    "route_10_north": 390,
    "route_10_south": 440,
    "route_10_pokemon_center_1f": 391,
    "route_10_pokemon_center_2f": 392,
    "route_11_west": 393,
    "route_11_east": 394,
    "route_11_gate_1f": 395,
    "route_11_gate_2f": 396,
    "route_12_north": 441,
    "route_12_center": 442,
    "route_12_south": 443,
    "route_12_gate_1f": 397,
    "route_12_gate_2f": 398,
    "route_12_fishing_house": 399,
    "route_13_west": 400,
    "route_13_east": 401,
    "route_14_north": 402,
    "route_14_south": 403,
    "route_15_west": 404,
    "route_15_east": 405,
    "route_15_gate_1f": 406,
    "route_15_gate_2f": 407,
    "route_16": 444,
    "route_16_house": 408,
    "route_16_gate_1f": 409,
    "route_16_gate_2f": 410,
    "route_17_north": 411,
    "route_17_center_top": 412,
    "route_17_center_bottom": 413,
    "route_17_south": 414,
    "route_18_west": 415,
    "route_18_east": 416,
    "route_18_gate_1f": 417,
    "route_18_gate_2f": 418,
    "route_19_north": 419,
    "route_19_south": 420,
    "route_20_west": 421,
    "route_20_center": 422,
    "route_20_east": 423,
    "route_21_north": 424,
    "route_21_south": 425,
    "route_22": 426,
    "route_22_gate": 427,
    "route_23_north": 428,
    "route_23_center_top": 445,
    "route_23_center_bottom": 446,
    "route_23_south": 429,
    "route_24": 430,
    "route_25_west": 431,
    "route_25_east": 432,
    "sea_cottage": 433
}

MAP_PAGE_MAPS: List[str] = [
    "maps/maps.json"
]

MAP_PAGE_LOCATIONS: List[str] = [
    "ut_locations/cities/cerulean_city.json",
    "ut_locations/cities/celadon_city.json",
    "ut_locations/cities/cinnabar_island.json",
    "ut_locations/cities/fuchsia_city.json",
    "ut_locations/cities/indigo_plateau.json",
    "ut_locations/cities/lavender_town.json",
    "ut_locations/cities/pallet_town.json",
    "ut_locations/cities/pewter_city.json",
    "ut_locations/cities/saffron_city.json",
    "ut_locations/cities/vermilion_city.json",
    "ut_locations/cities/viridian_city.json",
    "ut_locations/dungeons/berry_forest.json",
    "ut_locations/dungeons/cerulean_cave.json",
    "ut_locations/dungeons/digletts_cave.json",
    "ut_locations/dungeons/dotted_hole.json",
    "ut_locations/dungeons/icefall_cave.json",
    "ut_locations/dungeons/lost_cave.json",
    "ut_locations/dungeons/mt_ember.json",
    "ut_locations/dungeons/mt_moon.json",
    "ut_locations/dungeons/pattern_bush.json",
    "ut_locations/dungeons/pokemon_mansion.json",
    "ut_locations/dungeons/pokemon_tower.json",
    "ut_locations/dungeons/power_plant.json",
    "ut_locations/dungeons/rock_tunnel.json",
    "ut_locations/dungeons/rocket_hideout.json",
    "ut_locations/dungeons/rocket_warehouse.json",
    "ut_locations/dungeons/safari_zone.json",
    "ut_locations/dungeons/seafoam_islands.json",
    "ut_locations/dungeons/silph_co.json",
    "ut_locations/dungeons/ss_anne.json",
    "ut_locations/dungeons/underground_path_ew.json",
    "ut_locations/dungeons/underground_path_ns.json",
    "ut_locations/dungeons/victory_road.json",
    "ut_locations/dungeons/viridian_forest.json",
    "ut_locations/islands/bond_bridge.json",
    "ut_locations/islands/canyon_entrance.json",
    "ut_locations/islands/cape_brink.json",
    "ut_locations/islands/ember_spa.json",
    "ut_locations/islands/five_island.json",
    "ut_locations/islands/five_isle_meadow.json",
    "ut_locations/islands/four_island.json",
    "ut_locations/islands/green_path.json",
    "ut_locations/islands/kindle_road.json",
    "ut_locations/islands/memorial_pillar.json",
    "ut_locations/islands/navel_rock.json",
    "ut_locations/islands/one_island.json",
    "ut_locations/islands/outcast_island.json",
    "ut_locations/islands/resort_gorgeous.json",
    "ut_locations/islands/ruin_valley.json",
    "ut_locations/islands/sevault_canyon.json",
    "ut_locations/islands/seven_island.json",
    "ut_locations/islands/six_island.json",
    "ut_locations/islands/tanoby_ruins.json",
    "ut_locations/islands/three_island.json",
    "ut_locations/islands/three_isle_path.json",
    "ut_locations/islands/trainer_tower.json",
    "ut_locations/islands/treasure_beach.json",
    "ut_locations/islands/two_island.json",
    "ut_locations/islands/water_labyrinth.json",
    "ut_locations/islands/water_path.json",
    "ut_locations/overworld/cities.json",
    "ut_locations/overworld/entrances.json",
    "ut_locations/overworld/dungeons.json",
    "ut_locations/overworld/islands.json",
    "ut_locations/overworld/routes.json",
    "ut_locations/routes/route_1.json",
    "ut_locations/routes/route_2.json",
    "ut_locations/routes/route_3.json",
    "ut_locations/routes/route_4.json",
    "ut_locations/routes/route_6.json",
    "ut_locations/routes/route_7.json",
    "ut_locations/routes/route_8.json",
    "ut_locations/routes/route_9.json",
    "ut_locations/routes/route_10.json",
    "ut_locations/routes/route_11.json",
    "ut_locations/routes/route_12.json",
    "ut_locations/routes/route_13.json",
    "ut_locations/routes/route_14.json",
    "ut_locations/routes/route_15.json",
    "ut_locations/routes/route_16.json",
    "ut_locations/routes/route_17.json",
    "ut_locations/routes/route_18.json",
    "ut_locations/routes/route_19.json",
    "ut_locations/routes/route_20.json",
    "ut_locations/routes/route_21.json",
    "ut_locations/routes/route_22.json",
    "ut_locations/routes/route_23.json",
    "ut_locations/routes/route_24.json",
    "ut_locations/routes/route_25.json"
]

POPTRACKER_LOCATIONS: Dict[str, int] = {
    "Pallet Town/Unlock Fly Destination": "FLY_UNLOCK_VISITED_PALLET_TOWN",
    "Visit Pallet Town/Unlock Fly Destination": "FLY_UNLOCK_VISITED_PALLET_TOWN",
    "Pallet Town/PC Item": "PC_ITEM_POTION",
    "Player's House/PC Item": "PC_ITEM_POTION",
    "Player's House 2F PC/Item": "PC_ITEM_POTION",
    "Viridian City/Unlock Fly Destination": "FLY_UNLOCK_VISITED_VIRIDIAN_CITY",
    "Visit Viridian City/Unlock Fly Destination": "FLY_UNLOCK_VISITED_VIRIDIAN_CITY",
    "Viridian City/PC Item": "PC_ITEM_POTION",
    "Viridian Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Viridian Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Viridian City/Shop Item 1": "SHOP_VIRIDIAN_CITY_1",
    "Viridian City/Shop Item 2": "SHOP_VIRIDIAN_CITY_2",
    "Viridian City/Shop Item 3": "SHOP_VIRIDIAN_CITY_3",
    "Viridian City/Shop Item 4": "SHOP_VIRIDIAN_CITY_4",
    "Viridian Poke Mart/Shop Item 1": "SHOP_VIRIDIAN_CITY_1",
    "Viridian Poke Mart/Shop Item 2": "SHOP_VIRIDIAN_CITY_2",
    "Viridian Poke Mart/Shop Item 3": "SHOP_VIRIDIAN_CITY_3",
    "Viridian Poke Mart/Shop Item 4": "SHOP_VIRIDIAN_CITY_4",
    "Viridian Poke Mart Clerk/Shop Item 1": "SHOP_VIRIDIAN_CITY_1",
    "Viridian Poke Mart Clerk/Shop Item 2": "SHOP_VIRIDIAN_CITY_2",
    "Viridian Poke Mart Clerk/Shop Item 3": "SHOP_VIRIDIAN_CITY_3",
    "Viridian Poke Mart Clerk/Shop Item 4": "SHOP_VIRIDIAN_CITY_4",
    "Pewter City/Unlock Fly Destination": "FLY_UNLOCK_VISITED_PEWTER_CITY",
    "Visit Pewter City/Unlock Fly Destination": "FLY_UNLOCK_VISITED_PEWTER_CITY",
    "Pewter City/PC Item": "PC_ITEM_POTION",
    "Pewter Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Pewter Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Pewter City/Shop Item 1": "SHOP_PEWTER_CITY_1",
    "Pewter City/Shop Item 2": "SHOP_PEWTER_CITY_2",
    "Pewter City/Shop Item 3": "SHOP_PEWTER_CITY_3",
    "Pewter City/Shop Item 4": "SHOP_PEWTER_CITY_4",
    "Pewter City/Shop Item 5": "SHOP_PEWTER_CITY_5",
    "Pewter City/Shop Item 6": "SHOP_PEWTER_CITY_6",
    "Pewter City/Shop Item 7": "SHOP_PEWTER_CITY_7",
    "Pewter City/Shop Item 8": "SHOP_PEWTER_CITY_8",
    "Pewter Poke Mart/Shop Item 1": "SHOP_PEWTER_CITY_1",
    "Pewter Poke Mart/Shop Item 2": "SHOP_PEWTER_CITY_2",
    "Pewter Poke Mart/Shop Item 3": "SHOP_PEWTER_CITY_3",
    "Pewter Poke Mart/Shop Item 4": "SHOP_PEWTER_CITY_4",
    "Pewter Poke Mart/Shop Item 5": "SHOP_PEWTER_CITY_5",
    "Pewter Poke Mart/Shop Item 6": "SHOP_PEWTER_CITY_6",
    "Pewter Poke Mart/Shop Item 7": "SHOP_PEWTER_CITY_7",
    "Pewter Poke Mart/Shop Item 8": "SHOP_PEWTER_CITY_8",
    "Pewter Poke Mart Clerk/Shop Item 1": "SHOP_PEWTER_CITY_1",
    "Pewter Poke Mart Clerk/Shop Item 2": "SHOP_PEWTER_CITY_2",
    "Pewter Poke Mart Clerk/Shop Item 3": "SHOP_PEWTER_CITY_3",
    "Pewter Poke Mart Clerk/Shop Item 4": "SHOP_PEWTER_CITY_4",
    "Pewter Poke Mart Clerk/Shop Item 5": "SHOP_PEWTER_CITY_5",
    "Pewter Poke Mart Clerk/Shop Item 6": "SHOP_PEWTER_CITY_6",
    "Pewter Poke Mart Clerk/Shop Item 7": "SHOP_PEWTER_CITY_7",
    "Pewter Poke Mart Clerk/Shop Item 8": "SHOP_PEWTER_CITY_8",
    "Cerulean City/Unlock Fly Destination": "FLY_UNLOCK_VISITED_CERULEAN_CITY",
    "Cerulean City Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_CERULEAN_CITY",
    "Cerulean City/PC Item": "PC_ITEM_POTION",
    "Cerulean Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Cerulean Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Cerulean City/Shop Item 1": "SHOP_CERULEAN_CITY_1",
    "Cerulean City/Shop Item 2": "SHOP_CERULEAN_CITY_2",
    "Cerulean City/Shop Item 3": "SHOP_CERULEAN_CITY_3",
    "Cerulean City/Shop Item 4": "SHOP_CERULEAN_CITY_4",
    "Cerulean City/Shop Item 5": "SHOP_CERULEAN_CITY_5",
    "Cerulean City/Shop Item 6": "SHOP_CERULEAN_CITY_6",
    "Cerulean City/Shop Item 7": "SHOP_CERULEAN_CITY_7",
    "Cerulean City/Shop Item 8": "SHOP_CERULEAN_CITY_8",
    "Cerulean City/Shop Item 9": "SHOP_CERULEAN_CITY_9",
    "Cerulean Poke Mart/Shop Item 1": "SHOP_CERULEAN_CITY_1",
    "Cerulean Poke Mart/Shop Item 2": "SHOP_CERULEAN_CITY_2",
    "Cerulean Poke Mart/Shop Item 3": "SHOP_CERULEAN_CITY_3",
    "Cerulean Poke Mart/Shop Item 4": "SHOP_CERULEAN_CITY_4",
    "Cerulean Poke Mart/Shop Item 5": "SHOP_CERULEAN_CITY_5",
    "Cerulean Poke Mart/Shop Item 6": "SHOP_CERULEAN_CITY_6",
    "Cerulean Poke Mart/Shop Item 7": "SHOP_CERULEAN_CITY_7",
    "Cerulean Poke Mart/Shop Item 8": "SHOP_CERULEAN_CITY_8",
    "Cerulean Poke Mart/Shop Item 9": "SHOP_CERULEAN_CITY_9",
    "Cerulean Poke Mart Clerk/Shop Item 1": "SHOP_CERULEAN_CITY_1",
    "Cerulean Poke Mart Clerk/Shop Item 2": "SHOP_CERULEAN_CITY_2",
    "Cerulean Poke Mart Clerk/Shop Item 3": "SHOP_CERULEAN_CITY_3",
    "Cerulean Poke Mart Clerk/Shop Item 4": "SHOP_CERULEAN_CITY_4",
    "Cerulean Poke Mart Clerk/Shop Item 5": "SHOP_CERULEAN_CITY_5",
    "Cerulean Poke Mart Clerk/Shop Item 6": "SHOP_CERULEAN_CITY_6",
    "Cerulean Poke Mart Clerk/Shop Item 7": "SHOP_CERULEAN_CITY_7",
    "Cerulean Poke Mart Clerk/Shop Item 8": "SHOP_CERULEAN_CITY_8",
    "Cerulean Poke Mart Clerk/Shop Item 9": "SHOP_CERULEAN_CITY_9",
    "Vermilion City/Unlock Fly Destination": "FLY_UNLOCK_VISITED_VERMILION_CITY",
    "Vermilion City Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_VERMILION_CITY",
    "Vermilion City/PC Item": "PC_ITEM_POTION",
    "Vermilion Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Vermilion Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Vermilion City/Shop Item 1": "SHOP_VERMILION_CITY_1",
    "Vermilion City/Shop Item 2": "SHOP_VERMILION_CITY_2",
    "Vermilion City/Shop Item 3": "SHOP_VERMILION_CITY_3",
    "Vermilion City/Shop Item 4": "SHOP_VERMILION_CITY_4",
    "Vermilion City/Shop Item 5": "SHOP_VERMILION_CITY_5",
    "Vermilion City/Shop Item 6": "SHOP_VERMILION_CITY_6",
    "Vermilion City/Shop Item 7": "SHOP_VERMILION_CITY_7",
    "Vermilion Poke Mart/Shop Item 1": "SHOP_VERMILION_CITY_1",
    "Vermilion Poke Mart/Shop Item 2": "SHOP_VERMILION_CITY_2",
    "Vermilion Poke Mart/Shop Item 3": "SHOP_VERMILION_CITY_3",
    "Vermilion Poke Mart/Shop Item 4": "SHOP_VERMILION_CITY_4",
    "Vermilion Poke Mart/Shop Item 5": "SHOP_VERMILION_CITY_5",
    "Vermilion Poke Mart/Shop Item 6": "SHOP_VERMILION_CITY_6",
    "Vermilion Poke Mart/Shop Item 7": "SHOP_VERMILION_CITY_7",
    "Vermilion Poke Mart Clerk/Shop Item 1": "SHOP_VERMILION_CITY_1",
    "Vermilion Poke Mart Clerk/Shop Item 2": "SHOP_VERMILION_CITY_2",
    "Vermilion Poke Mart Clerk/Shop Item 3": "SHOP_VERMILION_CITY_3",
    "Vermilion Poke Mart Clerk/Shop Item 4": "SHOP_VERMILION_CITY_4",
    "Vermilion Poke Mart Clerk/Shop Item 5": "SHOP_VERMILION_CITY_5",
    "Vermilion Poke Mart Clerk/Shop Item 6": "SHOP_VERMILION_CITY_6",
    "Vermilion Poke Mart Clerk/Shop Item 7": "SHOP_VERMILION_CITY_7",
    "Lavender Town/Unlock Fly Destination": "FLY_UNLOCK_VISITED_LAVENDER_TOWN",
    "Lavender Town Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_LAVENDER_TOWN",
    "Lavender Town/PC Item": "PC_ITEM_POTION",
    "Lavender Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Lavender Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Lavender Town/Shop Item 1": "SHOP_LAVENDER_TOWN_1",
    "Lavender Town/Shop Item 2": "SHOP_LAVENDER_TOWN_2",
    "Lavender Town/Shop Item 3": "SHOP_LAVENDER_TOWN_3",
    "Lavender Town/Shop Item 4": "SHOP_LAVENDER_TOWN_4",
    "Lavender Town/Shop Item 5": "SHOP_LAVENDER_TOWN_5",
    "Lavender Town/Shop Item 6": "SHOP_LAVENDER_TOWN_6",
    "Lavender Town/Shop Item 7": "SHOP_LAVENDER_TOWN_7",
    "Lavender Town/Shop Item 8": "SHOP_LAVENDER_TOWN_8",
    "Lavender Town/Shop Item 9": "SHOP_LAVENDER_TOWN_9",
    "Lavender Poke Mart/Shop Item 1": "SHOP_LAVENDER_TOWN_1",
    "Lavender Poke Mart/Shop Item 2": "SHOP_LAVENDER_TOWN_2",
    "Lavender Poke Mart/Shop Item 3": "SHOP_LAVENDER_TOWN_3",
    "Lavender Poke Mart/Shop Item 4": "SHOP_LAVENDER_TOWN_4",
    "Lavender Poke Mart/Shop Item 5": "SHOP_LAVENDER_TOWN_5",
    "Lavender Poke Mart/Shop Item 6": "SHOP_LAVENDER_TOWN_6",
    "Lavender Poke Mart/Shop Item 7": "SHOP_LAVENDER_TOWN_7",
    "Lavender Poke Mart/Shop Item 8": "SHOP_LAVENDER_TOWN_8",
    "Lavender Poke Mart/Shop Item 9": "SHOP_LAVENDER_TOWN_9",
    "Lavender Poke Mart Clerk/Shop Item 1": "SHOP_LAVENDER_TOWN_1",
    "Lavender Poke Mart Clerk/Shop Item 2": "SHOP_LAVENDER_TOWN_2",
    "Lavender Poke Mart Clerk/Shop Item 3": "SHOP_LAVENDER_TOWN_3",
    "Lavender Poke Mart Clerk/Shop Item 4": "SHOP_LAVENDER_TOWN_4",
    "Lavender Poke Mart Clerk/Shop Item 5": "SHOP_LAVENDER_TOWN_5",
    "Lavender Poke Mart Clerk/Shop Item 6": "SHOP_LAVENDER_TOWN_6",
    "Lavender Poke Mart Clerk/Shop Item 7": "SHOP_LAVENDER_TOWN_7",
    "Lavender Poke Mart Clerk/Shop Item 8": "SHOP_LAVENDER_TOWN_8",
    "Lavender Poke Mart Clerk/Shop Item 9": "SHOP_LAVENDER_TOWN_9",
    "Celadon City/Unlock Fly Destination": "FLY_UNLOCK_VISITED_CELADON_CITY",
    "Celadon City Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_CELADON_CITY",
    "Celadon City/PC Item": "PC_ITEM_POTION",
    "Celadon Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Celadon Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Celadon City/TM Shop Item 1": "SHOP_CELADON_CITY_DEPT_TM_1",
    "Celadon City/TM Shop Item 2": "SHOP_CELADON_CITY_DEPT_TM_2",
    "Celadon City/TM Shop Item 3": "SHOP_CELADON_CITY_DEPT_TM_3",
    "Celadon City/TM Shop Item 4": "SHOP_CELADON_CITY_DEPT_TM_4",
    "Celadon City/TM Shop Item 5": "SHOP_CELADON_CITY_DEPT_TM_5",
    "Celadon City/TM Shop Item 6": "SHOP_CELADON_CITY_DEPT_TM_6",
    "Celadon Department Store/TM Shop Item 1": "SHOP_CELADON_CITY_DEPT_TM_1",
    "Celadon Department Store/TM Shop Item 2": "SHOP_CELADON_CITY_DEPT_TM_2",
    "Celadon Department Store/TM Shop Item 3": "SHOP_CELADON_CITY_DEPT_TM_3",
    "Celadon Department Store/TM Shop Item 4": "SHOP_CELADON_CITY_DEPT_TM_4",
    "Celadon Department Store/TM Shop Item 5": "SHOP_CELADON_CITY_DEPT_TM_5",
    "Celadon Department Store/TM Shop Item 6": "SHOP_CELADON_CITY_DEPT_TM_6",
    "Celadon Department Store TM Shop Clerk/Shop Item 1": "SHOP_CELADON_CITY_DEPT_TM_1",
    "Celadon Department Store TM Shop Clerk/Shop Item 2": "SHOP_CELADON_CITY_DEPT_TM_2",
    "Celadon Department Store TM Shop Clerk/Shop Item 3": "SHOP_CELADON_CITY_DEPT_TM_3",
    "Celadon Department Store TM Shop Clerk/Shop Item 4": "SHOP_CELADON_CITY_DEPT_TM_4",
    "Celadon Department Store TM Shop Clerk/Shop Item 5": "SHOP_CELADON_CITY_DEPT_TM_5",
    "Celadon Department Store TM Shop Clerk/Shop Item 6": "SHOP_CELADON_CITY_DEPT_TM_6",
    "Celadon City/Shop Item 1": "SHOP_CELADON_CITY_DEPT_ITEM_1",
    "Celadon City/Shop Item 2": "SHOP_CELADON_CITY_DEPT_ITEM_2",
    "Celadon City/Shop Item 3": "SHOP_CELADON_CITY_DEPT_ITEM_3",
    "Celadon City/Shop Item 4": "SHOP_CELADON_CITY_DEPT_ITEM_4",
    "Celadon City/Shop Item 5": "SHOP_CELADON_CITY_DEPT_ITEM_5",
    "Celadon City/Shop Item 6": "SHOP_CELADON_CITY_DEPT_ITEM_6",
    "Celadon City/Shop Item 7": "SHOP_CELADON_CITY_DEPT_ITEM_7",
    "Celadon City/Shop Item 8": "SHOP_CELADON_CITY_DEPT_ITEM_8",
    "Celadon City/Shop Item 9": "SHOP_CELADON_CITY_DEPT_ITEM_9",
    "Celadon Department Store/Shop Item 1": "SHOP_CELADON_CITY_DEPT_ITEM_1",
    "Celadon Department Store/Shop Item 2": "SHOP_CELADON_CITY_DEPT_ITEM_2",
    "Celadon Department Store/Shop Item 3": "SHOP_CELADON_CITY_DEPT_ITEM_3",
    "Celadon Department Store/Shop Item 4": "SHOP_CELADON_CITY_DEPT_ITEM_4",
    "Celadon Department Store/Shop Item 5": "SHOP_CELADON_CITY_DEPT_ITEM_5",
    "Celadon Department Store/Shop Item 6": "SHOP_CELADON_CITY_DEPT_ITEM_6",
    "Celadon Department Store/Shop Item 7": "SHOP_CELADON_CITY_DEPT_ITEM_7",
    "Celadon Department Store/Shop Item 8": "SHOP_CELADON_CITY_DEPT_ITEM_8",
    "Celadon Department Store/Shop Item 9": "SHOP_CELADON_CITY_DEPT_ITEM_9",
    "Celadon Department Store Item Shop Clerk/Shop Item 1": "SHOP_CELADON_CITY_DEPT_ITEM_1",
    "Celadon Department Store Item Shop Clerk/Shop Item 2": "SHOP_CELADON_CITY_DEPT_ITEM_2",
    "Celadon Department Store Item Shop Clerk/Shop Item 3": "SHOP_CELADON_CITY_DEPT_ITEM_3",
    "Celadon Department Store Item Shop Clerk/Shop Item 4": "SHOP_CELADON_CITY_DEPT_ITEM_4",
    "Celadon Department Store Item Shop Clerk/Shop Item 5": "SHOP_CELADON_CITY_DEPT_ITEM_5",
    "Celadon Department Store Item Shop Clerk/Shop Item 6": "SHOP_CELADON_CITY_DEPT_ITEM_6",
    "Celadon Department Store Item Shop Clerk/Shop Item 7": "SHOP_CELADON_CITY_DEPT_ITEM_7",
    "Celadon Department Store Item Shop Clerk/Shop Item 8": "SHOP_CELADON_CITY_DEPT_ITEM_8",
    "Celadon Department Store Item Shop Clerk/Shop Item 9": "SHOP_CELADON_CITY_DEPT_ITEM_9",
    "Celadon City/Evo Shop Item 1": "SHOP_CELADON_CITY_DEPT_EVO_1",
    "Celadon City/Evo Shop Item 2": "SHOP_CELADON_CITY_DEPT_EVO_2",
    "Celadon City/Evo Shop Item 3": "SHOP_CELADON_CITY_DEPT_EVO_3",
    "Celadon City/Evo Shop Item 4": "SHOP_CELADON_CITY_DEPT_EVO_4",
    "Celadon City/Evo Shop Item 5": "SHOP_CELADON_CITY_DEPT_EVO_5",
    "Celadon City/Evo Shop Item 6": "SHOP_CELADON_CITY_DEPT_EVO_6",
    "Celadon Department Store/Evo Shop Item 1": "SHOP_CELADON_CITY_DEPT_EVO_1",
    "Celadon Department Store/Evo Shop Item 2": "SHOP_CELADON_CITY_DEPT_EVO_2",
    "Celadon Department Store/Evo Shop Item 3": "SHOP_CELADON_CITY_DEPT_EVO_3",
    "Celadon Department Store/Evo Shop Item 4": "SHOP_CELADON_CITY_DEPT_EVO_4",
    "Celadon Department Store/Evo Shop Item 5": "SHOP_CELADON_CITY_DEPT_EVO_5",
    "Celadon Department Store/Evo Shop Item 6": "SHOP_CELADON_CITY_DEPT_EVO_6",
    "Celadon Department Store Evo Shop Clerk/Shop Item 1": "SHOP_CELADON_CITY_DEPT_EVO_1",
    "Celadon Department Store Evo Shop Clerk/Shop Item 2": "SHOP_CELADON_CITY_DEPT_EVO_2",
    "Celadon Department Store Evo Shop Clerk/Shop Item 3": "SHOP_CELADON_CITY_DEPT_EVO_3",
    "Celadon Department Store Evo Shop Clerk/Shop Item 4": "SHOP_CELADON_CITY_DEPT_EVO_4",
    "Celadon Department Store Evo Shop Clerk/Shop Item 5": "SHOP_CELADON_CITY_DEPT_EVO_5",
    "Celadon Department Store Evo Shop Clerk/Shop Item 6": "SHOP_CELADON_CITY_DEPT_EVO_6",
    "Celadon City/Vitamin Shop Item 1": "SHOP_CELADON_CITY_DEPT_VITAMIN_1",
    "Celadon City/Vitamin Shop Item 2": "SHOP_CELADON_CITY_DEPT_VITAMIN_2",
    "Celadon City/Vitamin Shop Item 3": "SHOP_CELADON_CITY_DEPT_VITAMIN_3",
    "Celadon City/Vitamin Shop Item 4": "SHOP_CELADON_CITY_DEPT_VITAMIN_4",
    "Celadon City/Vitamin Shop Item 5": "SHOP_CELADON_CITY_DEPT_VITAMIN_5",
    "Celadon City/Vitamin Shop Item 6": "SHOP_CELADON_CITY_DEPT_VITAMIN_6",
    "Celadon Department Store/Vitamin Shop Item 1": "SHOP_CELADON_CITY_DEPT_VITAMIN_1",
    "Celadon Department Store/Vitamin Shop Item 2": "SHOP_CELADON_CITY_DEPT_VITAMIN_2",
    "Celadon Department Store/Vitamin Shop Item 3": "SHOP_CELADON_CITY_DEPT_VITAMIN_3",
    "Celadon Department Store/Vitamin Shop Item 4": "SHOP_CELADON_CITY_DEPT_VITAMIN_4",
    "Celadon Department Store/Vitamin Shop Item 5": "SHOP_CELADON_CITY_DEPT_VITAMIN_5",
    "Celadon Department Store/Vitamin Shop Item 6": "SHOP_CELADON_CITY_DEPT_VITAMIN_6",
    "Celadon Department Store Vitamin Shop Clerk/Shop Item 1": "SHOP_CELADON_CITY_DEPT_VITAMIN_1",
    "Celadon Department Store Vitamin Shop Clerk/Shop Item 2": "SHOP_CELADON_CITY_DEPT_VITAMIN_2",
    "Celadon Department Store Vitamin Shop Clerk/Shop Item 3": "SHOP_CELADON_CITY_DEPT_VITAMIN_3",
    "Celadon Department Store Vitamin Shop Clerk/Shop Item 4": "SHOP_CELADON_CITY_DEPT_VITAMIN_4",
    "Celadon Department Store Vitamin Shop Clerk/Shop Item 5": "SHOP_CELADON_CITY_DEPT_VITAMIN_5",
    "Celadon Department Store Vitamin Shop Clerk/Shop Item 6": "SHOP_CELADON_CITY_DEPT_VITAMIN_6",
    "Celadon City/Battle Shop Item 1": "SHOP_CELADON_CITY_DEPT_BATTLE_1",
    "Celadon City/Battle Shop Item 2": "SHOP_CELADON_CITY_DEPT_BATTLE_2",
    "Celadon City/Battle Shop Item 3": "SHOP_CELADON_CITY_DEPT_BATTLE_3",
    "Celadon City/Battle Shop Item 4": "SHOP_CELADON_CITY_DEPT_BATTLE_4",
    "Celadon City/Battle Shop Item 5": "SHOP_CELADON_CITY_DEPT_BATTLE_5",
    "Celadon City/Battle Shop Item 6": "SHOP_CELADON_CITY_DEPT_BATTLE_6",
    "Celadon City/Battle Shop Item 7": "SHOP_CELADON_CITY_DEPT_BATTLE_7",
    "Celadon Department Store/Battle Shop Item 1": "SHOP_CELADON_CITY_DEPT_BATTLE_1",
    "Celadon Department Store/Battle Shop Item 2": "SHOP_CELADON_CITY_DEPT_BATTLE_2",
    "Celadon Department Store/Battle Shop Item 3": "SHOP_CELADON_CITY_DEPT_BATTLE_3",
    "Celadon Department Store/Battle Shop Item 4": "SHOP_CELADON_CITY_DEPT_BATTLE_4",
    "Celadon Department Store/Battle Shop Item 5": "SHOP_CELADON_CITY_DEPT_BATTLE_5",
    "Celadon Department Store/Battle Shop Item 6": "SHOP_CELADON_CITY_DEPT_BATTLE_6",
    "Celadon Department Store/Battle Shop Item 7": "SHOP_CELADON_CITY_DEPT_BATTLE_7",
    "Celadon Department Store Battle Shop Clerk/Shop Item 1": "SHOP_CELADON_CITY_DEPT_BATTLE_1",
    "Celadon Department Store Battle Shop Clerk/Shop Item 2": "SHOP_CELADON_CITY_DEPT_BATTLE_2",
    "Celadon Department Store Battle Shop Clerk/Shop Item 3": "SHOP_CELADON_CITY_DEPT_BATTLE_3",
    "Celadon Department Store Battle Shop Clerk/Shop Item 4": "SHOP_CELADON_CITY_DEPT_BATTLE_4",
    "Celadon Department Store Battle Shop Clerk/Shop Item 5": "SHOP_CELADON_CITY_DEPT_BATTLE_5",
    "Celadon Department Store Battle Shop Clerk/Shop Item 6": "SHOP_CELADON_CITY_DEPT_BATTLE_6",
    "Celadon Department Store Battle Shop Clerk/Shop Item 7": "SHOP_CELADON_CITY_DEPT_BATTLE_7",
    "Celadon City/Vending Machine Item 1": "SHOP_CELADON_CITY_DEPT_VENDING_MACHINES_1",
    "Celadon City/Vending Machine Item 2": "SHOP_CELADON_CITY_DEPT_VENDING_MACHINES_2",
    "Celadon City/Vending Machine Item 3": "SHOP_CELADON_CITY_DEPT_VENDING_MACHINES_3",
    "Celadon Department Store/Vending Machine Item 1": "SHOP_CELADON_CITY_DEPT_VENDING_MACHINES_1",
    "Celadon Department Store/Vending Machine Item 2": "SHOP_CELADON_CITY_DEPT_VENDING_MACHINES_2",
    "Celadon Department Store/Vending Machine Item 3": "SHOP_CELADON_CITY_DEPT_VENDING_MACHINES_3",
    "Celadon Department Store Vending Machines/Vending Machine Item 1": "SHOP_CELADON_CITY_DEPT_VENDING_MACHINES_1",
    "Celadon Department Store Vending Machines/Vending Machine Item 2": "SHOP_CELADON_CITY_DEPT_VENDING_MACHINES_2",
    "Celadon Department Store Vending Machines/Vending Machine Item 3": "SHOP_CELADON_CITY_DEPT_VENDING_MACHINES_3",
    "Celadon City/Game Corner Prize Item 1": "SHOP_CELADON_CITY_GAME_CORNER_PRIZE_1",
    "Celadon City/Game Corner Prize Item 2": "SHOP_CELADON_CITY_GAME_CORNER_PRIZE_2",
    "Celadon City/Game Corner Prize Item 3": "SHOP_CELADON_CITY_GAME_CORNER_PRIZE_3",
    "Celadon City/Game Corner Prize Item 4": "SHOP_CELADON_CITY_GAME_CORNER_PRIZE_4",
    "Celadon City/Game Corner Prize Item 5": "SHOP_CELADON_CITY_GAME_CORNER_PRIZE_5",
    "Celadon Game Corner Prize Room/Prize Item 1": "SHOP_CELADON_CITY_GAME_CORNER_PRIZE_1",
    "Celadon Game Corner Prize Room/Prize Item 2": "SHOP_CELADON_CITY_GAME_CORNER_PRIZE_2",
    "Celadon Game Corner Prize Room/Prize Item 3": "SHOP_CELADON_CITY_GAME_CORNER_PRIZE_3",
    "Celadon Game Corner Prize Room/Prize Item 4": "SHOP_CELADON_CITY_GAME_CORNER_PRIZE_4",
    "Celadon Game Corner Prize Room/Prize Item 5": "SHOP_CELADON_CITY_GAME_CORNER_PRIZE_5",
    "Celadon Game Corner Prize Room Worker (Left)/Prize Item 1": "SHOP_CELADON_CITY_GAME_CORNER_PRIZE_1",
    "Celadon Game Corner Prize Room Worker (Left)/Prize Item 2": "SHOP_CELADON_CITY_GAME_CORNER_PRIZE_2",
    "Celadon Game Corner Prize Room Worker (Left)/Prize Item 3": "SHOP_CELADON_CITY_GAME_CORNER_PRIZE_3",
    "Celadon Game Corner Prize Room Worker (Left)/Prize Item 4": "SHOP_CELADON_CITY_GAME_CORNER_PRIZE_4",
    "Celadon Game Corner Prize Room Worker (Left)/Prize Item 5": "SHOP_CELADON_CITY_GAME_CORNER_PRIZE_5",
    "Celadon City/Game Corner Prize TM 1": "SHOP_CELADON_CITY_GAME_CORNER_TM_PRIZE_1",
    "Celadon City/Game Corner Prize TM 2": "SHOP_CELADON_CITY_GAME_CORNER_TM_PRIZE_2",
    "Celadon City/Game Corner Prize TM 3": "SHOP_CELADON_CITY_GAME_CORNER_TM_PRIZE_3",
    "Celadon City/Game Corner Prize TM 4": "SHOP_CELADON_CITY_GAME_CORNER_TM_PRIZE_4",
    "Celadon City/Game Corner Prize TM 5": "SHOP_CELADON_CITY_GAME_CORNER_TM_PRIZE_5",
    "Celadon Game Corner Prize Room/Prize TM 1": "SHOP_CELADON_CITY_GAME_CORNER_TM_PRIZE_1",
    "Celadon Game Corner Prize Room/Prize TM 2": "SHOP_CELADON_CITY_GAME_CORNER_TM_PRIZE_2",
    "Celadon Game Corner Prize Room/Prize TM 3": "SHOP_CELADON_CITY_GAME_CORNER_TM_PRIZE_3",
    "Celadon Game Corner Prize Room/Prize TM 4": "SHOP_CELADON_CITY_GAME_CORNER_TM_PRIZE_4",
    "Celadon Game Corner Prize Room/Prize TM 5": "SHOP_CELADON_CITY_GAME_CORNER_TM_PRIZE_5",
    "Celadon Game Corner Prize Room Worker (Right)/Prize TM 1": "SHOP_CELADON_CITY_GAME_CORNER_TM_PRIZE_1",
    "Celadon Game Corner Prize Room Worker (Right)/Prize TM 2": "SHOP_CELADON_CITY_GAME_CORNER_TM_PRIZE_2",
    "Celadon Game Corner Prize Room Worker (Right)/Prize TM 3": "SHOP_CELADON_CITY_GAME_CORNER_TM_PRIZE_3",
    "Celadon Game Corner Prize Room Worker (Right)/Prize TM 4": "SHOP_CELADON_CITY_GAME_CORNER_TM_PRIZE_4",
    "Celadon Game Corner Prize Room Worker (Right)/Prize TM 5": "SHOP_CELADON_CITY_GAME_CORNER_TM_PRIZE_5",
    "Fuchsia City/Unlock Fly Destination": "FLY_UNLOCK_VISITED_FUCHSIA_CITY",
    "Fuchsia City Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_FUCHSIA_CITY",
    "Fuchsia City/PC Item": "PC_ITEM_POTION",
    "Fuchsia Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Fuchsia Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Fuchsia City/Shop Item 1": "SHOP_FUCHSIA_CITY_1",
    "Fuchsia City/Shop Item 2": "SHOP_FUCHSIA_CITY_2",
    "Fuchsia City/Shop Item 3": "SHOP_FUCHSIA_CITY_3",
    "Fuchsia City/Shop Item 4": "SHOP_FUCHSIA_CITY_4",
    "Fuchsia City/Shop Item 5": "SHOP_FUCHSIA_CITY_5",
    "Fuchsia City/Shop Item 6": "SHOP_FUCHSIA_CITY_6",
    "Fuchsia Poke Mart/Shop Item 1": "SHOP_FUCHSIA_CITY_1",
    "Fuchsia Poke Mart/Shop Item 2": "SHOP_FUCHSIA_CITY_2",
    "Fuchsia Poke Mart/Shop Item 3": "SHOP_FUCHSIA_CITY_3",
    "Fuchsia Poke Mart/Shop Item 4": "SHOP_FUCHSIA_CITY_4",
    "Fuchsia Poke Mart/Shop Item 5": "SHOP_FUCHSIA_CITY_5",
    "Fuchsia Poke Mart/Shop Item 6": "SHOP_FUCHSIA_CITY_6",
    "Fuchsia Poke Mart Clerk/Shop Item 1": "SHOP_FUCHSIA_CITY_1",
    "Fuchsia Poke Mart Clerk/Shop Item 2": "SHOP_FUCHSIA_CITY_2",
    "Fuchsia Poke Mart Clerk/Shop Item 3": "SHOP_FUCHSIA_CITY_3",
    "Fuchsia Poke Mart Clerk/Shop Item 4": "SHOP_FUCHSIA_CITY_4",
    "Fuchsia Poke Mart Clerk/Shop Item 5": "SHOP_FUCHSIA_CITY_5",
    "Fuchsia Poke Mart Clerk/Shop Item 6": "SHOP_FUCHSIA_CITY_6",
    "Saffron City/Unlock Fly Destination": "FLY_UNLOCK_VISITED_SAFFRON_CITY",
    "Saffron City Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_SAFFRON_CITY",
    "Saffron City/PC Item": "PC_ITEM_POTION",
    "Saffron Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Saffron Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Saffron City/Shop Item 1": "SHOP_SAFFRON_CITY_1",
    "Saffron City/Shop Item 2": "SHOP_SAFFRON_CITY_2",
    "Saffron City/Shop Item 3": "SHOP_SAFFRON_CITY_3",
    "Saffron City/Shop Item 4": "SHOP_SAFFRON_CITY_4",
    "Saffron City/Shop Item 5": "SHOP_SAFFRON_CITY_5",
    "Saffron City/Shop Item 6": "SHOP_SAFFRON_CITY_6",
    "Saffron Poke Mart/Shop Item 1": "SHOP_SAFFRON_CITY_1",
    "Saffron Poke Mart/Shop Item 2": "SHOP_SAFFRON_CITY_2",
    "Saffron Poke Mart/Shop Item 3": "SHOP_SAFFRON_CITY_3",
    "Saffron Poke Mart/Shop Item 4": "SHOP_SAFFRON_CITY_4",
    "Saffron Poke Mart/Shop Item 5": "SHOP_SAFFRON_CITY_5",
    "Saffron Poke Mart/Shop Item 6": "SHOP_SAFFRON_CITY_6",
    "Saffron Poke Mart Clerk/Shop Item 1": "SHOP_SAFFRON_CITY_1",
    "Saffron Poke Mart Clerk/Shop Item 2": "SHOP_SAFFRON_CITY_2",
    "Saffron Poke Mart Clerk/Shop Item 3": "SHOP_SAFFRON_CITY_3",
    "Saffron Poke Mart Clerk/Shop Item 4": "SHOP_SAFFRON_CITY_4",
    "Saffron Poke Mart Clerk/Shop Item 5": "SHOP_SAFFRON_CITY_5",
    "Saffron Poke Mart Clerk/Shop Item 6": "SHOP_SAFFRON_CITY_6",
    "Cinnabar Island/Unlock Fly Destination": "FLY_UNLOCK_VISITED_CINNABAR_ISLAND",
    "Cinnabar Island Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_CINNABAR_ISLAND",
    "Cinnabar Island/PC Item": "PC_ITEM_POTION",
    "Cinnabar Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Cinnabar Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Cinnabar Island/Shop Item 1": "SHOP_CINNABAR_ISLAND_1",
    "Cinnabar Island/Shop Item 2": "SHOP_CINNABAR_ISLAND_2",
    "Cinnabar Island/Shop Item 3": "SHOP_CINNABAR_ISLAND_3",
    "Cinnabar Island/Shop Item 4": "SHOP_CINNABAR_ISLAND_4",
    "Cinnabar Island/Shop Item 5": "SHOP_CINNABAR_ISLAND_5",
    "Cinnabar Island/Shop Item 6": "SHOP_CINNABAR_ISLAND_6",
    "Cinnabar Island/Shop Item 7": "SHOP_CINNABAR_ISLAND_7",
    "Cinnabar Poke Mart/Shop Item 1": "SHOP_CINNABAR_ISLAND_1",
    "Cinnabar Poke Mart/Shop Item 2": "SHOP_CINNABAR_ISLAND_2",
    "Cinnabar Poke Mart/Shop Item 3": "SHOP_CINNABAR_ISLAND_3",
    "Cinnabar Poke Mart/Shop Item 4": "SHOP_CINNABAR_ISLAND_4",
    "Cinnabar Poke Mart/Shop Item 5": "SHOP_CINNABAR_ISLAND_5",
    "Cinnabar Poke Mart/Shop Item 6": "SHOP_CINNABAR_ISLAND_6",
    "Cinnabar Poke Mart/Shop Item 7": "SHOP_CINNABAR_ISLAND_7",
    "Cinnabar Poke Mart Clerk/Shop Item 1": "SHOP_CINNABAR_ISLAND_1",
    "Cinnabar Poke Mart Clerk/Shop Item 2": "SHOP_CINNABAR_ISLAND_2",
    "Cinnabar Poke Mart Clerk/Shop Item 3": "SHOP_CINNABAR_ISLAND_3",
    "Cinnabar Poke Mart Clerk/Shop Item 4": "SHOP_CINNABAR_ISLAND_4",
    "Cinnabar Poke Mart Clerk/Shop Item 5": "SHOP_CINNABAR_ISLAND_5",
    "Cinnabar Poke Mart Clerk/Shop Item 6": "SHOP_CINNABAR_ISLAND_6",
    "Cinnabar Poke Mart Clerk/Shop Item 7": "SHOP_CINNABAR_ISLAND_7",
    "Indigo Plateau/Unlock Fly Destination": "FLY_UNLOCK_VISITED_INDIGO_PLATEAU",
    "Indigo Plateau Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_INDIGO_PLATEAU",
    "Indigo Plateau/PC Item": "PC_ITEM_POTION",
    "Indigo Plateau Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Indigo Plateau Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Indigo Plateau/Shop Item 1": "SHOP_INDIGO_PLATEAU_1",
    "Indigo Plateau/Shop Item 2": "SHOP_INDIGO_PLATEAU_2",
    "Indigo Plateau/Shop Item 3": "SHOP_INDIGO_PLATEAU_3",
    "Indigo Plateau/Shop Item 4": "SHOP_INDIGO_PLATEAU_4",
    "Indigo Plateau/Shop Item 5": "SHOP_INDIGO_PLATEAU_5",
    "Indigo Plateau/Shop Item 6": "SHOP_INDIGO_PLATEAU_6",
    "Indigo Plateau/Shop Item 7": "SHOP_INDIGO_PLATEAU_7",
    "Indigo Plateau Pokemon Center/Shop Item 1": "SHOP_INDIGO_PLATEAU_1",
    "Indigo Plateau Pokemon Center/Shop Item 2": "SHOP_INDIGO_PLATEAU_2",
    "Indigo Plateau Pokemon Center/Shop Item 3": "SHOP_INDIGO_PLATEAU_3",
    "Indigo Plateau Pokemon Center/Shop Item 4": "SHOP_INDIGO_PLATEAU_4",
    "Indigo Plateau Pokemon Center/Shop Item 5": "SHOP_INDIGO_PLATEAU_5",
    "Indigo Plateau Pokemon Center/Shop Item 6": "SHOP_INDIGO_PLATEAU_6",
    "Indigo Plateau Pokemon Center/Shop Item 7": "SHOP_INDIGO_PLATEAU_7",
    "Indigo Plateau Pokemon Center Shop Clerk/Shop Item 1": "SHOP_INDIGO_PLATEAU_1",
    "Indigo Plateau Pokemon Center Shop Clerk/Shop Item 2": "SHOP_INDIGO_PLATEAU_2",
    "Indigo Plateau Pokemon Center Shop Clerk/Shop Item 3": "SHOP_INDIGO_PLATEAU_3",
    "Indigo Plateau Pokemon Center Shop Clerk/Shop Item 4": "SHOP_INDIGO_PLATEAU_4",
    "Indigo Plateau Pokemon Center Shop Clerk/Shop Item 5": "SHOP_INDIGO_PLATEAU_5",
    "Indigo Plateau Pokemon Center Shop Clerk/Shop Item 6": "SHOP_INDIGO_PLATEAU_6",
    "Indigo Plateau Pokemon Center Shop Clerk/Shop Item 7": "SHOP_INDIGO_PLATEAU_7",
    "One Island Town/Unlock Fly Destination": "FLY_UNLOCK_VISITED_ONE_ISLAND",
    "One Island Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_ONE_ISLAND",
    "One Island Town/PC Item": "PC_ITEM_POTION",
    "One Island Pokemon Center/PC Item": "PC_ITEM_POTION",
    "One Island Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Two Island Town/Unlock Fly Destination": "FLY_UNLOCK_VISITED_TWO_ISLAND",
    "Two Island Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_TWO_ISLAND",
    "Two Island Town/PC Item": "PC_ITEM_POTION",
    "Two Island Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Two Island Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Two Island Town/Market Stall Item 1": "SHOP_TWO_ISLAND_1",
    "Two Island Town/Market Stall Item 2": "SHOP_TWO_ISLAND_2",
    "Two Island Town/Market Stall Item 3": "SHOP_TWO_ISLAND_3",
    "Two Island Town/Market Stall Item 4": "SHOP_TWO_ISLAND_4",
    "Two Island Town/Market Stall Item 5": "SHOP_TWO_ISLAND_5",
    "Two Island Town/Market Stall Item 6": "SHOP_TWO_ISLAND_6",
    "Two Island Town/Market Stall Item 7": "SHOP_TWO_ISLAND_7",
    "Two Island Town/Market Stall Item 8": "SHOP_TWO_ISLAND_8",
    "Two Island Town/Market Stall Item 9": "SHOP_TWO_ISLAND_9",
    "Market Stall Clerk/Market Stall Item 1": "SHOP_TWO_ISLAND_1",
    "Market Stall Clerk/Market Stall Item 2": "SHOP_TWO_ISLAND_2",
    "Market Stall Clerk/Market Stall Item 3": "SHOP_TWO_ISLAND_3",
    "Market Stall Clerk/Market Stall Item 4": "SHOP_TWO_ISLAND_4",
    "Market Stall Clerk/Market Stall Item 5": "SHOP_TWO_ISLAND_5",
    "Market Stall Clerk/Market Stall Item 6": "SHOP_TWO_ISLAND_6",
    "Market Stall Clerk/Market Stall Item 7": "SHOP_TWO_ISLAND_7",
    "Market Stall Clerk/Market Stall Item 8": "SHOP_TWO_ISLAND_8",
    "Market Stall Clerk/Market Stall Item 9": "SHOP_TWO_ISLAND_9",
    "Three Island Town/Unlock Fly Destination": "FLY_UNLOCK_VISITED_THREE_ISLAND",
    "Three Island Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_THREE_ISLAND",
    "Three Island Town/PC Item": "PC_ITEM_POTION",
    "Three Island Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Three Island Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Three Island Town/Shop Item 1": "SHOP_THREE_ISLAND_1",
    "Three Island Town/Shop Item 2": "SHOP_THREE_ISLAND_2",
    "Three Island Town/Shop Item 3": "SHOP_THREE_ISLAND_3",
    "Three Island Town/Shop Item 4": "SHOP_THREE_ISLAND_4",
    "Three Island Town/Shop Item 5": "SHOP_THREE_ISLAND_5",
    "Three Island Town/Shop Item 6": "SHOP_THREE_ISLAND_6",
    "Three Island Poke Mart/Shop Item 1": "SHOP_THREE_ISLAND_1",
    "Three Island Poke Mart/Shop Item 2": "SHOP_THREE_ISLAND_2",
    "Three Island Poke Mart/Shop Item 3": "SHOP_THREE_ISLAND_3",
    "Three Island Poke Mart/Shop Item 4": "SHOP_THREE_ISLAND_4",
    "Three Island Poke Mart/Shop Item 5": "SHOP_THREE_ISLAND_5",
    "Three Island Poke Mart/Shop Item 6": "SHOP_THREE_ISLAND_6",
    "Three Island Poke Mart Clerk/Shop Item 1": "SHOP_THREE_ISLAND_1",
    "Three Island Poke Mart Clerk/Shop Item 2": "SHOP_THREE_ISLAND_2",
    "Three Island Poke Mart Clerk/Shop Item 3": "SHOP_THREE_ISLAND_3",
    "Three Island Poke Mart Clerk/Shop Item 4": "SHOP_THREE_ISLAND_4",
    "Three Island Poke Mart Clerk/Shop Item 5": "SHOP_THREE_ISLAND_5",
    "Three Island Poke Mart Clerk/Shop Item 6": "SHOP_THREE_ISLAND_6",
    "Four Island Town/Unlock Fly Destination": "FLY_UNLOCK_VISITED_FOUR_ISLAND",
    "Four Island Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_FOUR_ISLAND",
    "Four Island Town/PC Item": "PC_ITEM_POTION",
    "Four Island Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Four Island Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Four Island Town/Shop Item 1": "SHOP_FOUR_ISLAND_1",
    "Four Island Town/Shop Item 2": "SHOP_FOUR_ISLAND_2",
    "Four Island Town/Shop Item 3": "SHOP_FOUR_ISLAND_3",
    "Four Island Town/Shop Item 4": "SHOP_FOUR_ISLAND_4",
    "Four Island Town/Shop Item 5": "SHOP_FOUR_ISLAND_5",
    "Four Island Town/Shop Item 6": "SHOP_FOUR_ISLAND_6",
    "Four Island Town/Shop Item 7": "SHOP_FOUR_ISLAND_7",
    "Four Island Town/Shop Item 8": "SHOP_FOUR_ISLAND_8",
    "Four Island Poke Mart/Shop Item 1": "SHOP_FOUR_ISLAND_1",
    "Four Island Poke Mart/Shop Item 2": "SHOP_FOUR_ISLAND_2",
    "Four Island Poke Mart/Shop Item 3": "SHOP_FOUR_ISLAND_3",
    "Four Island Poke Mart/Shop Item 4": "SHOP_FOUR_ISLAND_4",
    "Four Island Poke Mart/Shop Item 5": "SHOP_FOUR_ISLAND_5",
    "Four Island Poke Mart/Shop Item 6": "SHOP_FOUR_ISLAND_6",
    "Four Island Poke Mart/Shop Item 7": "SHOP_FOUR_ISLAND_7",
    "Four Island Poke Mart/Shop Item 8": "SHOP_FOUR_ISLAND_8",
    "Four Island Poke Mart Clerk/Shop Item 1": "SHOP_FOUR_ISLAND_1",
    "Four Island Poke Mart Clerk/Shop Item 2": "SHOP_FOUR_ISLAND_2",
    "Four Island Poke Mart Clerk/Shop Item 3": "SHOP_FOUR_ISLAND_3",
    "Four Island Poke Mart Clerk/Shop Item 4": "SHOP_FOUR_ISLAND_4",
    "Four Island Poke Mart Clerk/Shop Item 5": "SHOP_FOUR_ISLAND_5",
    "Four Island Poke Mart Clerk/Shop Item 6": "SHOP_FOUR_ISLAND_6",
    "Four Island Poke Mart Clerk/Shop Item 7": "SHOP_FOUR_ISLAND_7",
    "Four Island Poke Mart Clerk/Shop Item 8": "SHOP_FOUR_ISLAND_8",
    "Five Island Town/Unlock Fly Destination": "FLY_UNLOCK_VISITED_FIVE_ISLAND",
    "Five Island Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_FIVE_ISLAND",
    "Five Island Town/PC Item": "PC_ITEM_POTION",
    "Five Island Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Five Island Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Six Island Town/Unlock Fly Destination": "FLY_UNLOCK_VISITED_SIX_ISLAND",
    "Six Island Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_SIX_ISLAND",
    "Six Island Town/PC Item": "PC_ITEM_POTION",
    "Six Island Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Six Island Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Six Island Town/Shop Item 1": "SHOP_SIX_ISLAND_1",
    "Six Island Town/Shop Item 2": "SHOP_SIX_ISLAND_2",
    "Six Island Town/Shop Item 3": "SHOP_SIX_ISLAND_3",
    "Six Island Town/Shop Item 4": "SHOP_SIX_ISLAND_4",
    "Six Island Town/Shop Item 5": "SHOP_SIX_ISLAND_5",
    "Six Island Town/Shop Item 6": "SHOP_SIX_ISLAND_6",
    "Six Island Town/Shop Item 7": "SHOP_SIX_ISLAND_7",
    "Six Island Town/Shop Item 8": "SHOP_SIX_ISLAND_8",
    "Six Island Poke Mart/Shop Item 1": "SHOP_SIX_ISLAND_1",
    "Six Island Poke Mart/Shop Item 2": "SHOP_SIX_ISLAND_2",
    "Six Island Poke Mart/Shop Item 3": "SHOP_SIX_ISLAND_3",
    "Six Island Poke Mart/Shop Item 4": "SHOP_SIX_ISLAND_4",
    "Six Island Poke Mart/Shop Item 5": "SHOP_SIX_ISLAND_5",
    "Six Island Poke Mart/Shop Item 6": "SHOP_SIX_ISLAND_6",
    "Six Island Poke Mart/Shop Item 7": "SHOP_SIX_ISLAND_7",
    "Six Island Poke Mart/Shop Item 8": "SHOP_SIX_ISLAND_8",
    "Six Island Poke Mart Clerk/Shop Item 1": "SHOP_SIX_ISLAND_1",
    "Six Island Poke Mart Clerk/Shop Item 2": "SHOP_SIX_ISLAND_2",
    "Six Island Poke Mart Clerk/Shop Item 3": "SHOP_SIX_ISLAND_3",
    "Six Island Poke Mart Clerk/Shop Item 4": "SHOP_SIX_ISLAND_4",
    "Six Island Poke Mart Clerk/Shop Item 5": "SHOP_SIX_ISLAND_5",
    "Six Island Poke Mart Clerk/Shop Item 6": "SHOP_SIX_ISLAND_6",
    "Six Island Poke Mart Clerk/Shop Item 7": "SHOP_SIX_ISLAND_7",
    "Six Island Poke Mart Clerk/Shop Item 8": "SHOP_SIX_ISLAND_8",
    "Seven Island Town/Unlock Fly Destination": "FLY_UNLOCK_VISITED_SEVEN_ISLAND",
    "Seven Island Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_SEVEN_ISLAND",
    "Seven Island Town/PC Item": "PC_ITEM_POTION",
    "Seven Island Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Seven Island Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Seven Island Town/Shop Item 1": "SHOP_SEVEN_ISLAND_1",
    "Seven Island Town/Shop Item 2": "SHOP_SEVEN_ISLAND_2",
    "Seven Island Town/Shop Item 3": "SHOP_SEVEN_ISLAND_3",
    "Seven Island Town/Shop Item 4": "SHOP_SEVEN_ISLAND_4",
    "Seven Island Town/Shop Item 5": "SHOP_SEVEN_ISLAND_5",
    "Seven Island Town/Shop Item 6": "SHOP_SEVEN_ISLAND_6",
    "Seven Island Town/Shop Item 7": "SHOP_SEVEN_ISLAND_7",
    "Seven Island Town/Shop Item 8": "SHOP_SEVEN_ISLAND_8",
    "Seven Island Town/Shop Item 9": "SHOP_SEVEN_ISLAND_9",
    "Seven Island Poke Mart/Shop Item 1": "SHOP_SEVEN_ISLAND_1",
    "Seven Island Poke Mart/Shop Item 2": "SHOP_SEVEN_ISLAND_2",
    "Seven Island Poke Mart/Shop Item 3": "SHOP_SEVEN_ISLAND_3",
    "Seven Island Poke Mart/Shop Item 4": "SHOP_SEVEN_ISLAND_4",
    "Seven Island Poke Mart/Shop Item 5": "SHOP_SEVEN_ISLAND_5",
    "Seven Island Poke Mart/Shop Item 6": "SHOP_SEVEN_ISLAND_6",
    "Seven Island Poke Mart/Shop Item 7": "SHOP_SEVEN_ISLAND_7",
    "Seven Island Poke Mart/Shop Item 8": "SHOP_SEVEN_ISLAND_8",
    "Seven Island Poke Mart/Shop Item 9": "SHOP_SEVEN_ISLAND_9",
    "Seven Island Poke Mart Clerk/Shop Item 1": "SHOP_SEVEN_ISLAND_1",
    "Seven Island Poke Mart Clerk/Shop Item 2": "SHOP_SEVEN_ISLAND_2",
    "Seven Island Poke Mart Clerk/Shop Item 3": "SHOP_SEVEN_ISLAND_3",
    "Seven Island Poke Mart Clerk/Shop Item 4": "SHOP_SEVEN_ISLAND_4",
    "Seven Island Poke Mart Clerk/Shop Item 5": "SHOP_SEVEN_ISLAND_5",
    "Seven Island Poke Mart Clerk/Shop Item 6": "SHOP_SEVEN_ISLAND_6",
    "Seven Island Poke Mart Clerk/Shop Item 7": "SHOP_SEVEN_ISLAND_7",
    "Seven Island Poke Mart Clerk/Shop Item 8": "SHOP_SEVEN_ISLAND_8",
    "Seven Island Poke Mart Clerk/Shop Item 9": "SHOP_SEVEN_ISLAND_9",
    "Trainer Tower/Shop Item 1": "SHOP_TRAINER_TOWER_1",
    "Trainer Tower/Shop Item 2": "SHOP_TRAINER_TOWER_2",
    "Trainer Tower/Shop Item 3": "SHOP_TRAINER_TOWER_3",
    "Trainer Tower/Shop Item 4": "SHOP_TRAINER_TOWER_4",
    "Trainer Tower/Shop Item 5": "SHOP_TRAINER_TOWER_5",
    "Trainer Tower/Shop Item 6": "SHOP_TRAINER_TOWER_6",
    "Trainer Tower/Shop Item 7": "SHOP_TRAINER_TOWER_7",
    "Trainer Tower/Shop Item 8": "SHOP_TRAINER_TOWER_8",
    "Trainer Tower/Shop Item 9": "SHOP_TRAINER_TOWER_9",
    "Trainer Tower Lobby/Shop Item 1": "SHOP_TRAINER_TOWER_1",
    "Trainer Tower Lobby/Shop Item 2": "SHOP_TRAINER_TOWER_2",
    "Trainer Tower Lobby/Shop Item 3": "SHOP_TRAINER_TOWER_3",
    "Trainer Tower Lobby/Shop Item 4": "SHOP_TRAINER_TOWER_4",
    "Trainer Tower Lobby/Shop Item 5": "SHOP_TRAINER_TOWER_5",
    "Trainer Tower Lobby/Shop Item 6": "SHOP_TRAINER_TOWER_6",
    "Trainer Tower Lobby/Shop Item 7": "SHOP_TRAINER_TOWER_7",
    "Trainer Tower Lobby/Shop Item 8": "SHOP_TRAINER_TOWER_8",
    "Trainer Tower Lobby/Shop Item 9": "SHOP_TRAINER_TOWER_9",
    "Trainer Tower Lobby Shop Clerk/Shop Item 1": "SHOP_TRAINER_TOWER_1",
    "Trainer Tower Lobby Shop Clerk/Shop Item 2": "SHOP_TRAINER_TOWER_2",
    "Trainer Tower Lobby Shop Clerk/Shop Item 3": "SHOP_TRAINER_TOWER_3",
    "Trainer Tower Lobby Shop Clerk/Shop Item 4": "SHOP_TRAINER_TOWER_4",
    "Trainer Tower Lobby Shop Clerk/Shop Item 5": "SHOP_TRAINER_TOWER_5",
    "Trainer Tower Lobby Shop Clerk/Shop Item 6": "SHOP_TRAINER_TOWER_6",
    "Trainer Tower Lobby Shop Clerk/Shop Item 7": "SHOP_TRAINER_TOWER_7",
    "Trainer Tower Lobby Shop Clerk/Shop Item 8": "SHOP_TRAINER_TOWER_8",
    "Trainer Tower Lobby Shop Clerk/Shop Item 9": "SHOP_TRAINER_TOWER_9",
    "Route 4 West/Unlock Fly Destination": "FLY_UNLOCK_VISITED_ROUTE4",
    "Route 4 Pokemon Center/Unlock Fly Destination": "FLY_UNLOCK_VISITED_ROUTE4",
    "Route 4 Pokemon Center Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_ROUTE4",
    "Route 4 West/PC Item": "PC_ITEM_POTION",
    "Route 4 Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Route 4 Pokemon Center PC/Item": "PC_ITEM_POTION",
    "Route 10 North/Unlock Fly Destination": "FLY_UNLOCK_VISITED_ROUTE10",
    "Route 10 Pokemon Center/Unlock Fly Destination": "FLY_UNLOCK_VISITED_ROUTE10",
    "Route 10 Pokemon Center Visit/Unlock Fly Destination": "FLY_UNLOCK_VISITED_ROUTE10",
    "Route 10 North/PC Item": "PC_ITEM_POTION",
    "Route 10 Pokemon Center/PC Item": "PC_ITEM_POTION",
    "Route 10 Pokemon Center PC/Item": "PC_ITEM_POTION"
}

STARTING_TOWN_PC_LOCATIONS = {
    "SPAWN_PALLET_TOWN": ["Pallet Town/PC Item", "Player's House/PC Item", "Player's House 2F PC/Item"],
    "SPAWN_VIRIDIAN_CITY": ["Viridian City/PC Item", "Viridian Pokemon Center/PC Item",
                            "Viridian Pokemon Center PC/Item"],
    "SPAWN_PEWTER_CITY": ["Pewter City/PC Item", "Pewter Pokemon Center/PC Item", "Pewter Pokemon Center PC/Item"],
    "SPAWN_CERULEAN_CITY": ["Cerulean City/PC Item", "Cerulean Pokemon Center/PC Item",
                            "Cerulean Pokemon Center PC/Item"],
    "SPAWN_LAVENDER_TOWN": ["Lavender Town/PC Item", "Lavender Pokemon Center/PC Item",
                            "Lavender Pokemon Center PC/Item"],
    "SPAWN_VERMILION_CITY": ["Vermilion City/PC Item", "Vermilion Pokemon Center/PC Item",
                             "Vermilion Pokemon Center PC/Item"],
    "SPAWN_CELADON_CITY": ["Celadon City/PC Item", "Celadon Pokemon Center/PC Item", "Celadon Pokemon Center PC/Item"],
    "SPAWN_FUCHSIA_CITY": ["Fuchsia City/PC Item", "Fuchsia Pokemon Center/PC Item", "Fuchsia Pokemon Center PC/Item"],
    "SPAWN_CINNABAR_ISLAND": ["Cinnabar Island/PC Item", "Cinnabar Pokemon Center/PC Item",
                              "Cinnabar Pokemon Center PC/Item"],
    "SPAWN_INDIGO_PLATEAU": ["Indigo Plateau/PC Item", "Indigo Plateau Pokemon Center/PC Item",
                             "Indigo Plateau Pokemon Center PC/Item"],
    "SPAWN_SAFFRON_CITY": ["Saffron City/PC Item", "Saffron Pokemon Center/PC Item", "Saffron Pokemon Center PC/Item"],
    "SPAWN_ROUTE4": ["Route 4 West/PC Item", "Route 4 Pokemon Center/PC Item", "Route 4 Pokemon Center PC/Item"],
    "SPAWN_ROUTE10": ["Route 10 North/PC Item", "Route 10 Pokemon Center/PC Item", "Route 10 Pokemon Center PC/Item"],
    "SPAWN_ONE_ISLAND": ["One Island Town/PC Item", "One Island Pokemon Center/PC Item",
                         "One Island Pokemon Center PC/Item"],
    "SPAWN_TWO_ISLAND": ["Two Island Town/PC Item", "Two Island Pokemon Center/PC Item",
                         "Two Island Pokemon Center PC/Item"],
    "SPAWN_THREE_ISLAND": ["Three Island Town/PC Item", "Three Island Pokemon Center/PC Item",
                           "Three Island Pokemon Center PC/Item"],
    "SPAWN_FOUR_ISLAND": ["Four Island Town/PC Item", "Four Island Pokemon Center/PC Item",
                          "Four Island Pokemon Center PC/Item"],
    "SPAWN_FIVE_ISLAND": ["Five Island Town/PC Item", "Five Island Pokemon Center/PC Item",
                          "Five Island Pokemon Center PC/Item"],
    "SPAWN_SEVEN_ISLAND": ["Seven Island Town/PC Item", "Seven Island Pokemon Center/PC Item",
                           "Seven Island Pokemon Center PC/Item"],
    "SPAWN_SIX_ISLAND": ["Six Island Town/PC Item", "Six Island Pokemon Center/PC Item",
                         "Six Island Pokemon Center PC/Item"]
}
