from BaseClasses import MultiWorld, Item
from typing import Optional, List, TYPE_CHECKING, Union, get_args, get_origin, Any
from worlds.AutoWorld import World
from .Data import category_table
from .Items import SimpsonsHitAndRunItem
from .Locations import SimpsonsHitAndRunLocation
from .hooks.Helpers import before_is_category_enabled, before_is_item_enabled, before_is_location_enabled
from types import GenericAlias

from typing import Union

def format_to_valid_identifier(input: str) -> str:
    """Make sure the input is a valid python identifier"""
    input = input.strip()
    if input[:1].isdigit():
        input = "_" + input
    return input.replace(" ", "_")

def is_option_enabled(multiworld: MultiWorld, player: int, name: str) -> bool:
    return get_option_value(multiworld, player, name) > 0

def get_option_value(multiworld: MultiWorld, player: int, name: str) -> Union[int, dict]:
    option = getattr(multiworld.worlds[player].options, name, None)
    if option is None:
        return 0

    return option.value

def clamp(value, min, max):
    """Returns value clamped to the inclusive range of min and max"""
    if value < min:
        return min
    elif value > max:
        return max
    else:
        return value

def is_category_enabled(multiworld: MultiWorld, player: int, category_name: str) -> bool:
    """Check if a category has been disabled by a yaml option."""
    hook_result = before_is_category_enabled(multiworld, player, category_name)
    if hook_result is not None:
        return hook_result

    category_data = category_table.get(category_name, {})
    if "yaml_option" in category_data:
        for option_name in category_data["yaml_option"]:
            required = True
            if option_name.startswith("!"):
                option_name = option_name[1:]
                required = False

            if is_option_enabled(multiworld, player, option_name) != required:
                return False
    return True

def is_item_name_enabled(multiworld: MultiWorld, player: int, item_name: str) -> bool:
    """Check if an item named 'item_name' has been disabled by a yaml option."""
    item = multiworld.worlds[player].item_name_to_item.get(item_name, {})
    if not item:
        return False

    return is_item_enabled(multiworld, player, item)

def is_item_enabled(multiworld: MultiWorld, player: int, item: SimpsonsHitAndRunItem) -> bool:
    """Check if an item has been disabled by a yaml option."""
    hook_result = before_is_item_enabled(multiworld, player, item)
    if hook_result is not None:
        return hook_result

    return _is_manualobject_enabled(multiworld, player, item)

def is_location_name_enabled(multiworld: MultiWorld, player: int, location_name: str) -> bool:
    """Check if a location named 'location_name' has been disabled by a yaml option."""
    location = multiworld.worlds[player].location_name_to_location.get(location_name, {})
    if not location:
        return False

    return is_location_enabled(multiworld, player, location)

def is_location_enabled(multiworld: MultiWorld, player: int, location: SimpsonsHitAndRunLocation) -> bool:
    """Check if a location has been disabled by a yaml option."""
    hook_result = before_is_location_enabled(multiworld, player, location)
    if hook_result is not None:
        return hook_result

    return _is_manualobject_enabled(multiworld, player, location)

def _is_manualobject_enabled(multiworld: MultiWorld, player: int, object: any) -> bool:
    """Internal method: Check if a Manual Object has any category disabled by a yaml option.
    \nPlease use the proper is_'item/location'_enabled or is_'item/location'_name_enabled methods instead.
    """
    enabled = True
    for category in object.get("category", []):
        if not is_category_enabled(multiworld, player, category):
            enabled = False
            break

    return enabled

def get_items_for_player(multiworld: MultiWorld, player: int) -> List[Item]:
    """Return list of items of a player including placed items"""
    return [i for i in multiworld.get_items() if i.player == player]

def get_items_with_value(world: World, multiworld: MultiWorld, value: str, player: Optional[int] = None, force: bool = False) -> dict[str, int]:
    """Return a dict of every items with a specific value type present in their respective 'value' dict\n
    Output in the format 'Item Name': 'value count'\n
    Keep a cache of the result and wont redo unless 'force == True'
    """
    if player is None:
        player = world.player

    player_items = get_items_for_player(multiworld, player)
    # Just a small check to prevent caching {} if items don't exist yet
    if not player_items:
        return {value: -1}

    value = value.lower().strip()

    if not hasattr(world, 'item_values'): #Cache of just the item values
        world.item_values = {}

    if not world.item_values.get(player):
        world.item_values[player] = {}

    if value not in world.item_values.get(player, {}).keys() or force:
        item_with_values = {i.name: world.item_name_to_item[i.name]['value'].get(value, 0)
                            for i in player_items if i.code is not None
                            and i.name in world.item_name_groups.get(f'has_{value}_value', [])}
        world.item_values[player][value] = item_with_values
    return world.item_values[player].get(value)

def convert_string_to_type(input: str, target_type: type) -> Any:
    """Take a string and attempt to convert it to {target_type}
    \ntarget_type can be a single type(ex. str), an union (int|str), an Optional type (Optional[str]) or a combo of any of those (Optional[int|str])
    \nSpecial logic:
    - When target_type is Optional or contains None: it will check if input.lower() is "none"
    - When target_type contains bool: it will check if input.lower() is "true", "1", "false" or "0"
    - If bool is the last type in target_type it also run the input directly through bool(input) if previous fails
    \nif you want this to possibly fail without Exceptions include str in target_type, your input should get returned if all the other conversions fails
    """
    def checktype(target_type, found_types: list):
        if issubclass(type(target_type), type): #is it a single type (str, list, etc)
            if target_type not in found_types:
                found_types.append(target_type)

        elif issubclass(type(target_type), GenericAlias): #is it something like list[str] and dict{str:int}
            if target_type not in found_types and get_origin(target_type) not in found_types: #dont add 'dict[str]' if we already search for 'dict'
                found_types.append(target_type)

        elif issubclass(type(target_type), type(str|int)) \
            or issubclass(type(target_type), type(Union[str|int])): #Support both version of Union, and Optional and other alike
            for arg in get_args(target_type):
                checktype(arg, found_types)

        else:
            raise Exception(f"'{value}' cannot be converted to {target_type} since its not a supported type \nAsk about it in #Manual-support and it might be added.")

    found_types = []
    checktype(target_type, found_types)

    if str in found_types: #do it last
        found_types.remove(str)
        found_types.append(str)

    value = input.strip()
    i = 0
    errors = []
    for value_type in found_types:
        i += 1
        if issubclass(value_type, type(None)):
            if value.lower() == 'none':
                return None
            errors.append(str(value_type) + ": value was not 'none'")

        elif issubclass(value_type, bool):
            if value.lower() in ['true', '1', 'on']:
                return True

            elif value.lower() in ['false', '0', 'off']:
                return False

            else:
                if i == len(found_types):
                    return value_type(value) #if its the last type might as well try and convert to bool
                errors.append(str(value_type) + ": value was not in either ['true', '1', 'on'] or ['false', '0', 'off']")

        elif issubclass(value_type, list) or issubclass(value_type, dict) \
            or issubclass(value_type, set) or issubclass(type(value_type), GenericAlias):
            try:
                try:
                    converted_value = ast.literal_eval(value)
                except ValueError as e:
                    # The ValueError from ast when the string cannot be evaluated as a literal is usually something like
                    # "malformed node or string on line 1: <ast.Name object at 0x000001AEBBCC7590>", which is not
                    # helpful, so re-raise with a better exception message.
                    raise ValueError(f"'{value}' could not be evaluated as a literal") from e

                compareto = get_origin(value_type) if issubclass(type(value_type), GenericAlias) else value_type
                if issubclass(compareto, type(converted_value)):
                    return converted_value
                else:
                    errors.append(str(value_type) + f": value '{value}' was not a valid {str(compareto)}")
            except Exception as e:
                errors.append(str(value_type) + ": " + str(e))
                continue
        else:
            try:
                return value_type(value)

            except Exception as e:
                errors.append(str(value_type) + ": " + str(e))
                continue

    newline = "\n"
    raise Exception(f"'{value}' could not be converted to {target_type}, here's the conversion failure message(s):\n\n{newline.join([' - ' + str(validation_error) for validation_error in errors])}\n\n")