"""Crown Location List."""

from __future__ import annotations

from enum import IntEnum, auto
from typing import TYPE_CHECKING

from randomizer.Enums.Events import Events
from randomizer.Enums.Kongs import Kongs
from randomizer.Enums.Levels import Levels
from randomizer.Enums.Regions import Regions
from randomizer.Enums.Settings import HelmSetting, RemovedBarriersSelected, ShufflePortLocations, KasplatRandoSetting
from randomizer.Enums.Switches import Switches
from randomizer.Enums.Time import Time
from randomizer.Enums.Maps import Maps


class CustomLocation:
    """Class to store information pertaining to a custom location for objects, such as crown pads, dirt patches and melon crates."""

    def __init__(
        self,
        *,
        map=0,
        name="",
        x=0,
        y=0,
        z=0,
        rot_x=0,
        rot_y=0,
        rot_z=0,
        max_size=160,
        logic_region=0,
        logic=None,
        group=0,
        vanilla_crown=False,
        vanilla_crate=False,
        vanilla_patch=False,
        vanilla_port=False,
        is_rotating_room=False,
        is_fungi_hidden_patch=False,
        is_galleon_floating_crate=False,
        banned_types: list = [],
        default_index=0,
        tied_warp_event: Events = None,
    ) -> None:
        """Initialize with given data."""
        self.map = map
        self.name = name
        self.coords = [x, y, z]
        self.rot_x = rot_x
        self.rot_y = rot_y
        self.rot_z = rot_z
        self.max_size = max_size
        self.logic = logic
        self.logic_region = logic_region
        self.group = group
        self.vanilla_crown = vanilla_crown
        self.vanilla_crate = vanilla_crate
        self.vanilla_patch = vanilla_patch
        self.vanilla_port = vanilla_port
        self.selected = vanilla_crown or vanilla_patch or vanilla_port or vanilla_crate
        self.is_rotating_room = is_rotating_room
        self.is_fungi_hidden_patch = is_fungi_hidden_patch
        self.is_galleon_floating_crate = is_galleon_floating_crate
        self.banned_types = banned_types
        self.default_index = default_index
        self.placement_subindex = default_index
        self.tied_warp_event = tied_warp_event
        if logic is None:
            self.has_access_logic = False
            self.logic = lambda _: True
        else:
            self.has_access_logic = True
            self.logic = logic

    def setCustomLocation(self, value: bool) -> None:
        """Set location's state regarding rando."""
        self.selected = value

    def isValidLocation(self, placement_type: LocationTypes) -> bool:
        """Determine whether the location is valid for placement of a certain object."""
        if self.selected:
            return False
        if placement_type in self.banned_types:
            return False
        return True


def getBannedWarps(spoiler) -> list[Events]:
    """Get list of banned warp events based on settings."""
    lst = [
        # All of these float on water, lets make these static
        Events.GalleonW2bTagged,
        Events.GalleonW4bTagged,
        Events.GalleonW5bTagged,
        # Only way to ensure 2 hidden warps don't link to eachother
        Events.CavesW3bTagged,
        Events.CavesW4bTagged,
        Events.CavesW5aTagged,
        # Locations with extra logic
        Events.JapesW5bTagged,
        Events.AztecW5bTagged,
        Events.GalleonW4aTagged,
    ]
    WARP_SHUFFLE_SETTING = spoiler.settings.bananaport_placement_rando
    PLATFORMING_SETTING = spoiler.LogicVariables.monkey_maneuvers
    if WARP_SHUFFLE_SETTING in [ShufflePortLocations.on, ShufflePortLocations.half_vanilla]:
        # Access to the Lanky Kasplat and potential coins or coloured bananas
        lst.append(Events.LlamaW2bTagged)
    if not PLATFORMING_SETTING:
        # Access to Blueprint Pillar or Bonus Cave for not [Diddy or Tiny]
        lst.append(Events.CavesW4aTagged)
    if WARP_SHUFFLE_SETTING == ShufflePortLocations.half_vanilla:
        lst.extend(
            [
                # Japes
                Events.JapesW1aTagged,  # W1 Portal
                Events.JapesW2aTagged,  # W2 Entrance
                Events.JapesW3aTagged,  # W3 Painting
                Events.JapesW4aTagged,  # W4 Tunnel
                # Aztec
                Events.AztecW1aTagged,  # W1 Portal
                Events.AztecW2aTagged,  # W2 Oasis
                Events.AztecW3aTagged,  # W3 Totem
                Events.AztecW4aTagged,  # W4 Totem
                # Llama
                Events.LlamaW1aTagged,  # W1 Near Entrance
                # Factory
                Events.FactoryW1aTagged,  # W1 Lobby
                Events.FactoryW2aTagged,  # W2 Lobby
                Events.FactoryW3aTagged,  # W3 Lobby
                Events.FactoryW4aTagged,  # W4 Prod Bottom
                Events.FactoryW5bTagged,  # W5 Funky
                # Galleon
                Events.GalleonW1aTagged,  # W1 Main Area
                Events.GalleonW3aTagged,  # W3 Main Area
                # Fungi
                Events.ForestW1aTagged,  # W1 Clock
                Events.ForestW2aTagged,  # W2 Clock
                Events.ForestW3aTagged,  # W3 Clock
                Events.ForestW4aTagged,  # W4 Clock
                Events.ForestW5bTagged,  # W5 Low
                # Caves
                Events.CavesW1aTagged,  # W1 Start
                Events.CavesW2aTagged,  # W2 Start
                # Castle
                Events.CastleW1aTagged,  # W1 Start
                Events.CastleW2aTagged,  # W2 Start
                Events.CastleW3aTagged,  # W3 Start
                Events.CastleW4aTagged,  # W4 Start
                Events.CastleW5aTagged,  # W5 Start
                # Crypt
                Events.CryptW1aTagged,  # W1 Start
                Events.CryptW2aTagged,  # W2 Start
                Events.CryptW3aTagged,  # W3 Start
                # Isles
                Events.IslesW1aTagged,  # W1 Ring
                Events.IslesW2aTagged,  # W2 Ring
                Events.IslesW3aTagged,  # W3 Ring
                Events.IslesW4aTagged,  # W4 Ring
                Events.IslesW5aTagged,  # W5 Ring
            ]
        )
    return lst


def resetCustomLocations(spoiler) -> None:
    """Reset all locations to their default selection-state."""
    BANNED_WARPS = getBannedWarps(spoiler)
    for key in CustomLocations.keys():
        for location in CustomLocations[key]:
            location.selected = location.vanilla_crown or location.vanilla_crate or location.vanilla_patch or location.vanilla_port
            if spoiler.settings.crown_placement_rando and location.vanilla_crown and not location.is_rotating_room:
                location.selected = False
            if spoiler.settings.random_patches and location.vanilla_patch and not location.is_fungi_hidden_patch:
                location.selected = False
            if spoiler.settings.random_crates and location.vanilla_crate and not location.is_galleon_floating_crate:
                location.selected = False
            if spoiler.settings.bananaport_placement_rando not in [ShufflePortLocations.off, ShufflePortLocations.vanilla_only] and location.vanilla_port:
                location.selected = False
            if location.tied_warp_event in BANNED_WARPS:
                # Make sure these warps cannot be selected by anything
                location.selected = True


WARP_MAX_SIZE = 56


class LocationTypes(IntEnum):
    """Location type Enum."""

    CrownPad = auto()
    DirtPatch = auto()
    MelonCrate = auto()
    Bananaport = auto()


CustomLocations = {
    Levels.JungleJapes: [
        CustomLocation(
            map=Maps.JungleJapes,
            name="Near Funky",
            x=1989.7,
            y=520,
            z=2086.71,
            max_size=72,
            logic_region=Regions.JapesHill,
            vanilla_crown=True,
            group=4,
        ),
        CustomLocation(
            name="On Painting Hill",
            map=Maps.JungleJapes,
            x=550.814,
            y=370.167,
            z=1873.436,
            rot_y=1070,
            max_size=64,
            vanilla_patch=True,
            logic_region=Regions.JapesPaintingRoomHill,
            group=1,
        ),
        CustomLocation(
            name="Next to level entrance",
            map=Maps.JungleJapes,
            x=754.0,
            y=286.0,
            z=824.0,
            rot_y=796,
            max_size=64,
            logic_region=Regions.JungleJapesStart,
            group=3,
        ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="On Tree (Starting Area)",
            x=1101,
            y=478,
            z=266,
            max_size=40,
            logic_region=Regions.JungleJapesStart,
            logic=lambda l: l.climbing,
            group=3,
        ),
        CustomLocation(
            name="Next to first tunnel entrance",
            map=Maps.JungleJapes,
            x=1357.0,
            y=283.0,
            z=205.0,
            rot_y=3572,
            max_size=64,
            logic_region=Regions.JungleJapesStart,
            group=3,
        ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="Diddy Cavern",
            x=2381,
            y=280,
            z=392,
            max_size=56,
            logic_region=Regions.JapesBeyondPeanutGate,
            group=3,
        ),
        CustomLocation(
            name="Inside Diddy's Cavern",
            map=Maps.JungleJapes,
            x=2475.0,
            y=280.0,
            z=508.0,
            rot_y=2427,
            max_size=64,
            logic_region=Regions.JapesBeyondPeanutGate,
            group=3,
        ),
        CustomLocation(
            name="First tunnel - later half",
            map=Maps.JungleJapes,
            x=1812.0,
            y=280.0,
            z=797.0,
            rot_y=1171,
            max_size=64,
            logic_region=Regions.JungleJapesStart,
            group=3,
        ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="Painting Hill",
            x=554,
            y=370,
            z=1804,
            max_size=56,
            logic_region=Regions.JapesPaintingRoomHill,
            group=1,
            banned_types=[LocationTypes.MelonCrate],
        ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="Hive area near Hunky barrel",
            x=2337,
            y=551,
            z=3156,
            rot_y=1160,
            max_size=96,
            logic_region=Regions.JapesBeyondFeatherGate,
            group=6,
        ),
        # CrownLocation(
        #     name="Under Chunky's Barrel",
        #     map=Maps.JungleJapes,
        #     x=2345.0,
        #     y=551.0,
        #     z=3152.0,
        #     rot_y=1160,
        #     max_size=64,
        #     logic_region=Regions.JapesBeyondFeatherGate,
        #     group=6,
        # ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="Near Stump",
            x=1912,
            y=539,
            z=3289,
            max_size=72,
            logic_region=Regions.JapesBeyondFeatherGate,
            group=6,
        ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="Near Log",
            x=2419,
            y=539,
            z=2834,
            max_size=72,
            logic_region=Regions.JapesBeyondFeatherGate,
            group=6,
        ),
        CustomLocation(
            name="Near the Vine Pit",
            map=Maps.JungleJapes,
            x=796.0,
            y=284.0,
            z=2471.0,
            rot_y=2480,
            max_size=64,
            logic_region=Regions.JapesBeyondCoconutGate2,
            group=2,
        ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="Vine Pit",
            x=1150,
            y=230,
            z=2613,
            max_size=72,
            logic_region=Regions.JapesBeyondCoconutGate2,
            group=2,
        ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="Lanky useless slope - middle",
            x=2209,
            y=339,
            z=3205,
            max_size=64,
            logic_region=Regions.JapesUselessSlope,
            group=2,
        ),
        CustomLocation(
            name="Lanky useless slope - left",
            map=Maps.JungleJapes,
            x=2263.0,
            y=338.0,
            z=3158.0,
            rot_y=3305,
            max_size=64,
            logic_region=Regions.JapesUselessSlope,
            banned_types=[LocationTypes.MelonCrate],
            group=2,
        ),
        CustomLocation(
            name="Cranky-tunnel Crossing",
            map=Maps.JungleJapes,
            x=1633.0,
            y=210.0,
            z=3015.0,
            rot_y=2161,
            max_size=64,
            logic_region=Regions.JapesBeyondCoconutGate2,
            group=2,
        ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="In the Rambi Cave",
            x=715.62884521484,
            y=280,
            z=3758.80859375,
            rot_y=41,
            max_size=64,
            logic_region=Regions.BeyondRambiGate,
            vanilla_crate=True,
            banned_types=[LocationTypes.CrownPad, LocationTypes.Bananaport],
            group=2,
        ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="Fairy Pool",
            x=597,
            y=240,
            z=3123,
            max_size=64,
            logic_region=Regions.BeyondRambiGate,
            group=2,
        ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="Behind Lanky Hut",
            x=2052,
            y=280,
            z=4350,
            max_size=56,
            logic_region=Regions.JapesBeyondCoconutGate2,
            group=2,
        ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="Behind DK Hut",
            x=1307,
            y=280,
            z=4327,
            max_size=56,
            logic_region=Regions.JapesBeyondCoconutGate2,
            group=2,
        ),
        CustomLocation(
            name="Left of DK hut",
            map=Maps.JungleJapes,
            x=1409.0,
            y=280.0,
            z=4367.0,
            rot_y=2142,
            max_size=64,
            logic_region=Regions.JapesBeyondCoconutGate2,
            group=2,
        ),
        # CrownLocation(map=Maps.JungleJapes, name="On Cranky's Lab", x=1696, y=360, z=4002, max_size=40, logic_region=0, group=2), # Doesn't work with shop rando
        CustomLocation(
            map=Maps.JungleJapes,
            name="Near the Rambi cage",
            x=1705,
            y=280,
            z=4233,
            max_size=72,
            logic_region=Regions.JapesBeyondCoconutGate2,
            group=2,
        ),
        CustomLocation(
            name="Behind Stormy Shop",
            map=Maps.JungleJapes,
            x=1697.0,
            y=280.0,
            z=4088.0,
            rot_y=0,
            max_size=64,
            logic_region=Regions.JapesBeyondCoconutGate2,
            group=2,
        ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="Minecart Alcove",
            x=1106,
            y=288,
            z=1967,
            rot_y=2707,
            max_size=48,
            logic_region=Regions.JungleJapesMain,
            group=1,
        ),
        # CrownLocation(
        #     name="Minecart Exit",
        #     map=Maps.JungleJapes,
        #     x=1108.0,
        #     y=288.0,
        #     z=1970.0,
        #     rot_y=2707,
        #     max_size=64,
        #     logic_region=Regions.JungleJapesMain,
        #     group=1,
        # ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="Near highest shop",
            x=2045,
            y=680,
            z=2522,
            max_size=56,
            logic_region=Regions.JapesHillTop,
            group=4,
        ),
        # CrownLocation(map=Maps.JungleJapes, name="On Mountain", x=1616, y=989, z=2439, max_size=80, logic_region=Regions.JapesTopOfMountain, group=4),
        CustomLocation(
            name="Behind Diddy's Mountain",
            map=Maps.JungleJapes,
            x=1542.0,
            y=790.0,
            z=2578.0,
            rot_y=2969,
            max_size=64,
            logic_region=Regions.JapesHillTop,
            group=4,
        ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="Near Kong Cage",
            x=949,
            y=852,
            z=2384,
            max_size=56,
            logic_region=Regions.JapesHillTop,
            group=4,
        ),
        CustomLocation(
            name="Near the cannon",
            map=Maps.JungleJapes,
            x=1274.0,
            y=520.0,
            z=2225.0,
            rot_y=2275,
            max_size=64,
            logic_region=Regions.JapesCannonPlatform,
            group=4,
        ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="Behind the Mountain",
            x=1586.9215087891,
            y=790.5,
            z=2616.7092285156,
            rot_y=349,
            max_size=64,
            logic_region=Regions.JapesHillTop,
            group=4,
            vanilla_crate=True,
        ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="Near Cannon Platform",
            x=1282,
            y=520,
            z=2262,
            max_size=56,
            logic_region=Regions.JapesCannonPlatform,
            group=4,
        ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="In Troff 'n' Scoff Alcove",
            x=770,
            y=538,
            z=2332,
            max_size=56,
            logic_region=Regions.JapesTnSAlcove,
            group=4,
        ),
        # CrownLocation(
        #     map=Maps.JungleJapes,
        #     name="Near Underground Entrance",
        #     x=2446,
        #     y=280,
        #     z=1143,
        #     max_size=64,
        #     logic_region=Regions.JungleJapesStart,
        #     group=1
        # ),
        CustomLocation(
            map=Maps.JungleJapes,
            name="Lower River",
            x=2381,
            y=280,
            z=1685,
            max_size=72,
            logic_region=Regions.JungleJapesStart,
            group=1,
        ),
        # CrownLocation(map=Maps.JungleJapes, name="Starting Area (Low)", x=742, y=286, z=825, max_size=56, logic_region=Regions.JungleJapesStart, group=1),
        CustomLocation(
            name="Vanilla Near Warp 1",
            map=Maps.JungleJapes,
            x=914.366,
            y=280,
            z=796.257,
            rot_y=3994,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.JungleJapesStart,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.JapesW1aTagged,
        ),
        CustomLocation(
            name="Vanilla Hive Warp 5",
            map=Maps.JungleJapes,
            x=2582.048,
            y=539.333,
            z=2948.962,
            rot_y=1343,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.JapesBeyondFeatherGate,
            group=6,
            vanilla_port=True,
            tied_warp_event=Events.JapesW5aTagged,
        ),
        CustomLocation(
            name="Vanilla Near Warp 2",
            map=Maps.JungleJapes,
            x=1492.371,
            y=280,
            z=1424.4,
            rot_y=2048,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.JungleJapesStart,
            group=2,
            vanilla_port=True,
            tied_warp_event=Events.JapesW2aTagged,
        ),
        CustomLocation(
            name="Vanilla Painting Warp 3",
            map=Maps.JungleJapes,
            x=1007.791,
            y=288,
            z=2213.355,
            rot_y=2674,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.JungleJapesMain,
            group=1,
            vanilla_port=True,
            tied_warp_event=Events.JapesW3aTagged,
        ),
        CustomLocation(
            name="Vanilla Far Warp 1",
            map=Maps.JungleJapes,
            x=1515.683,
            y=280,
            z=1202.195,
            rot_y=1957,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.JungleJapesStart,
            group=2,
            vanilla_port=True,
            tied_warp_event=Events.JapesW1bTagged,
        ),
        CustomLocation(
            name="Vanilla Blast Warp 3",
            map=Maps.JungleJapes,
            x=2457.394,
            y=280,
            z=1494.277,
            rot_y=3391,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.JungleJapesStart,
            group=2,
            vanilla_port=True,
            tied_warp_event=Events.JapesW3bTagged,
        ),
        CustomLocation(
            name="Vanilla Mountain Warp 2",
            map=Maps.JungleJapes,
            x=1521.21,
            y=790,
            z=2301.499,
            rot_y=2287,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.JapesHillTop,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.JapesW2bTagged,
        ),
        CustomLocation(
            name="Vanilla Tunnel Warp 4",
            map=Maps.JungleJapes,
            x=1598.821,
            y=280,
            z=2217.09,
            rot_y=2082,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.JapesBeyondCoconutGate2,
            group=2,
            vanilla_port=True,
            tied_warp_event=Events.JapesW4aTagged,
        ),
        CustomLocation(
            name="Vanilla Mountain Warp 5",
            map=Maps.JungleJapes,
            x=1615.459,
            y=989.103,
            z=2440.312,
            rot_y=1889,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.JapesTopOfMountain,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.JapesW5bTagged,
        ),
        CustomLocation(
            name="Vanilla Stormy Warp 4",
            map=Maps.JungleJapes,
            x=1696.499,
            y=280,
            z=3887.24,
            rot_y=2094,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.JapesBeyondCoconutGate2,
            group=2,
            vanilla_port=True,
            tied_warp_event=Events.JapesW4bTagged,
        ),
        CustomLocation(
            name="Behind the boulder",
            map=Maps.JungleJapes,
            x=2433.0,
            y=280.0,
            z=1114.0,
            rot_y=3784,
            max_size=64,
            logic_region=Regions.JungleJapesStart,
            group=2,
        ),
        CustomLocation(
            map=Maps.JapesUnderGround,
            name="Underground: Behind Cannon",
            x=433,
            y=20,
            z=104,
            max_size=40,
            logic_region=Regions.JapesCatacomb,
            group=1,
        ),
        CustomLocation(
            map=Maps.JapesUnderGround,
            name="Underground: Near Vines",
            x=453,
            y=20,
            z=814,
            max_size=56,
            logic_region=Regions.JapesCatacomb,
            group=1,
        ),
        CustomLocation(
            map=Maps.JapesLankyCave,
            name="Painting Room: Near Peg",
            x=100,
            y=80,
            z=346,
            max_size=48,
            logic_region=Regions.JapesLankyCave,
            group=1,
            banned_types=[LocationTypes.CrownPad, LocationTypes.Bananaport],
        ),
        CustomLocation(
            map=Maps.JapesMountain,
            name="Mountain: Near Entrance (Ground)",
            x=328,
            y=40,
            z=471,
            max_size=48,
            logic_region=Regions.Mine,
            group=5,
        ),
        CustomLocation(
            map=Maps.JapesMountain,
            name="Mountain: Near Entrance (High)",
            x=497,
            y=140,
            z=512,
            max_size=56,
            logic_region=Regions.Mine,
            group=5,
        ),
        CustomLocation(
            map=Maps.JapesMountain,
            name="Mountain: On Barrel",
            x=690,
            y=135,
            z=757,
            rot_y=2013,
            max_size=64,
            logic_region=Regions.Mine,
            group=5,
        ),
        # CrownLocation(
        #     name="Mountain: On a Barrel",
        #     map=Maps.JapesMountain,
        #     x=691.0,
        #     y=135.0,
        #     z=753.0,
        #     rot_y=2013,
        #     max_size=64,
        #     logic_region=Regions.Mine,
        #     group=5,
        # ),
        CustomLocation(
            map=Maps.JapesMountain, name="Mountain: Near HiLo Machine", x=326, y=133, z=1510, max_size=32, logic_region=Regions.Mine, logic=lambda l: (l.charge and l.isdiddy) or l.CanPhase(), group=5
        ),
        CustomLocation(
            map=Maps.JapesMountain,
            name="Mountain: Under Conveyor",
            x=42,
            y=220,
            z=1056,
            max_size=48,
            logic_region=Regions.Mine,
            logic=lambda l: (l.CanSlamSwitch(Levels.JungleJapes, 1) and l.isdiddy) or l.CanPhase(),
            banned_types=[LocationTypes.CrownPad, LocationTypes.Bananaport],
            group=5,
        ),
        CustomLocation(
            map=Maps.JapesTinyHive,
            name="Hive: Main Room",
            x=1385,
            y=212,
            z=1381,
            max_size=112,
            logic_region=Regions.TinyHive,
            group=6,
        ),
        CustomLocation(
            map=Maps.JapesTinyHive,
            name="Hive: 1st Room",
            x=610,
            y=130,
            z=1279,
            max_size=96,
            logic_region=Regions.TinyHive,
            group=6,
        ),
        CustomLocation(
            map=Maps.JapesTinyHive,
            name="Hive: 3rd Room",
            x=2547,
            y=254,
            z=1354,
            max_size=96,
            logic_region=Regions.TinyHive,
            logic=lambda l: (l.CanSlamSwitch(Levels.JungleJapes, 1) or l.CanPhase()) and l.tiny,
            group=6,
        ),
    ],
    Levels.AngryAztec: [
        CustomLocation(
            map=Maps.AztecTinyTemple,
            name="Tiny Temple: Vulture Room",
            x=1466.42,
            y=305.33,
            z=2340.39,
            max_size=65.6,
            logic_region=Regions.TempleVultureRoom,
            vanilla_crown=True,
            logic=lambda l: l.CanSlamSwitch(Levels.AngryAztec, 1) and l.grape and l.islanky,
            group=2,
        ),
        CustomLocation(
            name="Oasis",
            map=Maps.AngryAztec,
            x=2426.34,
            y=115.5,
            z=960.642,
            rot_y=2618,
            max_size=64,
            vanilla_patch=True,
            logic_region=Regions.AngryAztecOasis,
            group=1,
        ),
        CustomLocation(
            name="Chunky 5DT",
            map=Maps.AztecChunky5DTemple,
            x=652.778,
            y=85.0,
            z=1544.845,
            rot_y=1036,
            max_size=64,
            vanilla_patch=True,
            logic_region=Regions.ChunkyTemple,
            group=7,
        ),
        CustomLocation(
            name="Tiny Temple: Main room back left",
            map=Maps.AztecTinyTemple,
            x=1727.0,
            y=284.0,
            z=649.0,
            rot_y=3417,
            max_size=64,
            logic_region=Regions.TempleStart,
            group=2,
        ),
        CustomLocation(
            map=Maps.AztecTinyTemple,
            name="Tiny Temple: Starting Room (Low)",
            x=1802,
            y=283,
            z=611,
            max_size=80,
            logic_region=Regions.TempleStart,
            group=2,
        ),
        CustomLocation(
            map=Maps.AztecTinyTemple,
            name="Tiny Temple: Starting Room (High)",
            x=1370,
            y=490,
            z=1126,
            max_size=48,
            logic_region=Regions.TempleStart,
            logic=lambda l: l.CanSlamSwitch(Levels.AngryAztec, 1) and l.diddy,
            group=2,
        ),
        CustomLocation(
            map=Maps.AztecTinyTemple,
            name="Tiny Temple: Kong Free Room",
            x=524,
            y=344,
            z=1468,
            max_size=80,
            logic_region=Regions.TempleKONGRoom,
            group=2,
        ),
        CustomLocation(
            name="Tiny Temple: Next to Kong cage",
            map=Maps.AztecTinyTemple,
            x=565.0,
            y=344.0,
            z=1146.0,
            rot_y=3959,
            max_size=64,
            logic_region=Regions.TempleKONGRoom,
            group=2,
        ),
        CustomLocation(
            name="Entrance tunnel - near DK door",
            map=Maps.AngryAztec,
            x=1372.0,
            y=120.0,
            z=1125.0,
            rot_y=4084,
            max_size=64,
            logic_region=Regions.AztecTunnelBeforeOasis,
            group=1,
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="Blueprint Room",
            x=1224,
            y=120,
            z=740,
            max_size=40,
            logic_region=Regions.AztecTunnelBeforeOasis,
            logic=lambda l: l.CanPhase() or (l.hasMoveSwitchsanity(Switches.AztecBlueprintDoor, False) and ((l.strongKong and l.isdonkey) or (l.twirl and l.istiny))),
            group=1,
            banned_types=[LocationTypes.Bananaport],  # Hard to detect that it's bad to link to Quicksand Cave, in which case it tricks the seed into assuming any kong can use this port
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="Near Oasis Sand",
            x=2151,
            y=120,
            z=983,
            max_size=56,
            logic_region=Regions.AngryAztecOasis,
            group=1,
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="Behind Tiny Temple",
            x=3345,
            y=153,
            z=507,
            max_size=48,
            logic_region=Regions.AngryAztecOasis,
            group=1,
        ),
        CustomLocation(
            name="Next to Tiny Temple - left",
            map=Maps.AngryAztec,
            x=3184.0,
            y=153.0,
            z=343.0,
            rot_y=4009,
            max_size=64,
            logic_region=Regions.AngryAztecOasis,
            group=1,
        ),
        CustomLocation(
            name="Next to Tiny Temple - right",
            map=Maps.AngryAztec,
            x=3489.0,
            y=153.0,
            z=702.0,
            rot_y=3001,
            max_size=64,
            logic_region=Regions.AngryAztecOasis,
            group=1,
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="On Tiny Temple",
            x=3192,
            y=352,
            z=500,
            max_size=48,
            logic_region=Regions.AngryAztecOasis,
            logic=lambda l: ((l.jetpack and l.diddy) or l.CanMoonkick()) and l.climbing,
            group=1,
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="Near Oasis Shop",
            x=2430,
            y=120,
            z=509,
            rot_y=3492,
            max_size=56,
            logic_region=Regions.AngryAztecOasis,
            group=1,
        ),
        # CrownLocation(
        #     name="Near Oasis Shop (2)",
        #     map=Maps.AngryAztec,
        #     x=2421.0,
        #     y=120.0,
        #     z=489.0,
        #     rot_y=3492,
        #     max_size=64,
        #     logic_region=Regions.AngryAztecOasis,
        #     group=1,
        # ),
        CustomLocation(
            name="Behind Llama Cage",
            map=Maps.AngryAztec,
            x=2070.0,
            y=153.0,
            z=1706.0,
            rot_y=3424,
            max_size=64,
            logic_region=Regions.AngryAztecOasis,
            group=1,
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="Near Hunky Chunky Barrel",
            x=3216,
            y=120,
            z=1490,
            max_size=56,
            logic_region=Regions.AngryAztecConnectorTunnel,
            group=5,
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="Left of Hunky Cage",
            x=4276.6,
            y=120,
            z=2266,
            max_size=64,
            logic_region=Regions.AngryAztecConnectorTunnel,
            group=5,
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="Right of Hunky Cage",
            x=4283,
            y=120,
            z=2543.6,
            max_size=64,
            logic_region=Regions.AngryAztecConnectorTunnel,
            group=5,
        ),
        CustomLocation(
            name="Behind Hunky Cage",
            map=Maps.AngryAztec,
            x=4395.0,
            y=120.0,
            z=2409.0,
            rot_y=2992,
            max_size=64,
            logic_region=Regions.AngryAztecConnectorTunnel,
            group=5,
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="Near Llama Temple (Left)",
            x=2781,
            y=160,
            z=3264,
            max_size=56,
            logic_region=Regions.AngryAztecMain,
            group=3,
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="Near Llama Temple (Right)",
            x=3154,
            y=160,
            z=3172,
            max_size=56,
            logic_region=Regions.AngryAztecMain,
            group=3,
        ),
        CustomLocation(
            name="Next to Llama Temple",
            map=Maps.AngryAztec,
            x=3110.0,
            y=160.0,
            z=3193.0,
            rot_y=284,
            max_size=64,
            logic_region=Regions.AngryAztecMain,
            group=3,
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="On Llama Temple - back",
            x=2884,
            y=437,
            z=2903,
            max_size=64,
            logic_region=Regions.AngryAztecMain,
            logic=lambda l: (l.jetpack and l.diddy) or l.CanMoonkick(),
            group=3,
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="On Llama Temple",
            x=2892.8391113281,
            y=437.5,
            z=2952.9016113281,
            rot_y=106,
            max_size=64,
            logic_region=Regions.AngryAztecMain,
            vanilla_crate=True,
            group=3,
            logic=lambda l: (l.jetpack and l.diddy) or l.CanMoonkick(),
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="Near Quicksand Tunnel Shop",
            x=3000,
            y=120,
            z=4532,
            max_size=56,
            logic_region=Regions.AngryAztecMain,
            group=3,
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="On 5-Door Temple",
            x=2056,
            y=420,
            z=3648,
            max_size=48,
            logic_region=Regions.AngryAztecMain,
            logic=lambda l: (l.jetpack and l.diddy) or l.CanMoonkick(),
            group=3,
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="Near Strong Kong Warp",
            x=3422,
            y=120,
            z=4514,
            max_size=56,
            logic_region=Regions.AngryAztecMain,
            group=3,
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="Near Vulture Cage",
            x=4013,
            y=226,
            z=4589,
            max_size=28.8,
            logic_region=Regions.AngryAztecMain,
            logic=lambda l: ((l.can_use_vines or l.CanMoonkick()) and l.climbing) or (l.jetpack and l.diddy),
            group=3,
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="Under Vulture Cage",
            x=4102,
            y=120,
            z=4548,
            max_size=64,
            logic_region=Regions.AngryAztecMain,
            group=3,
        ),
        CustomLocation(
            name="Next to Vulture cage shop",
            map=Maps.AngryAztec,
            x=4028.0,
            y=120.0,
            z=4505.0,
            rot_y=1496,
            max_size=64,
            logic_region=Regions.AngryAztecMain,
            group=3,
        ),
        CustomLocation(
            name="Behind Gong tower",
            map=Maps.AngryAztec,
            x=4524.0,
            y=80.0,
            z=2936.0,
            rot_y=3663,
            max_size=64,
            logic_region=Regions.AngryAztecMain,
            group=3,
        ),
        CustomLocation(
            name="Left of Gong tower",
            map=Maps.AngryAztec,
            x=4190.0,
            y=80.0,
            z=3011.0,
            rot_y=3902,
            max_size=64,
            logic_region=Regions.AngryAztecMain,
            group=3,
        ),
        # CrownLocation(map=Maps.AngryAztec, name="Near Gong Tower", x=4212, y=80, z=2959, max_size=64, logic_region=Regions.AngryAztecMain, group=3),
        CustomLocation(
            map=Maps.AngryAztec,
            name="Near Gong Tower",
            x=4679.9067382813,
            y=80,
            z=3202.2341308594,
            rot_y=47,
            max_size=64,
            logic_region=Regions.AngryAztecMain,
            vanilla_crate=True,
            group=3,
        ),
        CustomLocation(
            map=Maps.AngryAztec,
            name="Sealed Quicksand Tunnel",
            x=2783,
            y=120,
            z=4817,
            max_size=56,
            logic_region=Regions.AztecDonkeyQuicksandCave,
            group=3,
        ),
        CustomLocation(
            name="Vanilla Tunnel Warp 1",
            map=Maps.AngryAztec,
            x=805.376,
            y=120,
            z=785.657,
            rot_y=2048,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.BetweenVinesByPortal,
            group=12,
            vanilla_port=True,
            tied_warp_event=Events.AztecW1aTagged,
        ),
        CustomLocation(
            name="Vanilla Oasis Warp 2",
            map=Maps.AngryAztec,
            x=2891.304,
            y=154.354,
            z=860.214,
            rot_y=3504,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.AngryAztecOasis,
            group=1,
            vanilla_port=True,
            tied_warp_event=Events.AztecW2aTagged,
        ),
        CustomLocation(
            name="Vanilla Totem Warp 2",
            map=Maps.AngryAztec,
            x=3501.093,
            y=120,
            z=3071.741,
            rot_y=2014,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.AngryAztecMain,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.AztecW2bTagged,
        ),
        CustomLocation(
            name="Vanilla Totem Warp 5",
            map=Maps.AngryAztec,
            x=3481.304,
            y=120,
            z=4490.359,
            rot_y=2014,
            max_size=int(WARP_MAX_SIZE * 0.92),
            logic_region=Regions.AngryAztecMain,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.AztecW5aTagged,
        ),
        CustomLocation(
            name="Vanilla Totem Warp 3",
            map=Maps.AngryAztec,
            x=3502.324,
            y=121.082,
            z=3396.902,
            rot_y=1923,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.AngryAztecMain,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.AztecW3aTagged,
        ),
        CustomLocation(
            name="Vanilla Oasis Warp 1",
            map=Maps.AngryAztec,
            x=2365.761,
            y=120,
            z=752.703,
            rot_y=3118,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.AngryAztecOasis,
            group=1,
            vanilla_port=True,
            tied_warp_event=Events.AztecW1bTagged,
        ),
        CustomLocation(
            name="Vanilla Totem Warp 4",
            map=Maps.AngryAztec,
            x=3412.203,
            y=121.381,
            z=3380.916,
            rot_y=1980,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.AngryAztecMain,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.AztecW4aTagged,
        ),
        CustomLocation(
            name="Vanilla Tunnel Warp 3",
            map=Maps.AngryAztec,
            x=2802.317,
            y=120,
            z=2523.051,
            rot_y=1070,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.AngryAztecConnectorTunnel,
            group=5,
            vanilla_port=True,
            tied_warp_event=Events.AztecW3bTagged,
        ),
        CustomLocation(
            name="Vanilla Funky Warp 4",
            map=Maps.AngryAztec,
            x=2918.319,
            y=120,
            z=4420.91,
            rot_y=1911,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.AngryAztecMain,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.AztecW4bTagged,
        ),
        CustomLocation(
            name="Vanilla Tunnel Warp 5",
            map=Maps.AngryAztec,
            x=2778.521,
            y=120,
            z=4752.402,
            rot_y=1081,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.AztecDonkeyQuicksandCave,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.AztecW5bTagged,
        ),
        CustomLocation(
            map=Maps.AztecDonkey5DTemple,
            name="Donkey 5DT: Dead End",
            x=99,
            y=20,
            z=464,
            max_size=64,
            logic_region=Regions.DonkeyTempleDeadEndRight,
            group=7,
        ),
        CustomLocation(
            map=Maps.AztecDiddy5DTemple,
            name="Diddy 5DT: Dead End",
            x=1060,
            y=20,
            z=493,
            max_size=64,
            logic_region=Regions.DiddyTemple,
            logic=lambda l: (l.peanut and l.isdiddy) or l.CanPhase(),
            group=7,
        ),
        CustomLocation(
            map=Maps.AztecLanky5DTemple,
            name="Lanky 5DT: Dead End",
            x=767,
            y=122,
            z=916,
            max_size=48,
            logic_region=Regions.LankyTemple,
            logic=lambda l: (l.grape and l.islanky) or l.CanPhase(),
            group=7,
        ),
        CustomLocation(
            map=Maps.AztecLanky5DTemple,
            name="Lanky 5DT: Right side",
            x=180,
            y=47,
            z=658,
            max_size=72,
            logic_region=Regions.LankyTemple,
            group=7,
        ),
        CustomLocation(
            map=Maps.AztecTiny5DTemple, name="Tiny 5DT: Dead End", x=329, y=123, z=1420, max_size=48, logic_region=Regions.TinyTemple, logic=lambda l: (l.feather and l.istiny) or l.CanPhase(), group=7
        ),
        CustomLocation(
            map=Maps.AztecChunky5DTemple,
            name="Chunky 5DT: Right side",
            x=375,
            y=20,
            z=321,
            max_size=72,
            logic_region=Regions.ChunkyTemple,
            group=7,
        ),
        CustomLocation(
            map=Maps.AztecChunky5DTemple,
            name="Chunky 5DT: Left side",
            x=779,
            y=47,
            z=678,
            max_size=72,
            logic_region=Regions.ChunkyTemple,
            group=7,
        ),
        CustomLocation(
            map=Maps.AztecLlamaTemple,
            name="Llama Temple: Llama Right",
            x=1737,
            y=472,
            z=2548,
            max_size=72,
            rot_y=608,
            logic_region=Regions.LlamaTemple,
            group=4,
        ),
        CustomLocation(
            map=Maps.AztecLlamaTemple,
            name="Llama Temple: Entrance",
            x=2634.7512207031,
            y=371.66665649414,
            z=2420.3410644531,
            rot_y=179,
            max_size=64,
            logic_region=Regions.LlamaTemple,
            vanilla_crate=True,
            group=4,
        ),
        CustomLocation(
            name="Llama Temple: Next to Llama Right",
            map=Maps.AztecLlamaTemple,
            x=1777.0,
            y=472.0,
            z=2592.0,
            rot_y=608,
            max_size=64,
            logic_region=Regions.LlamaTemple,
            group=4,
        ),
        CustomLocation(
            name="Llama Temple: Next to Llama Left",
            map=Maps.AztecLlamaTemple,
            x=1795.0,
            y=472.0,
            z=2158.0,
            rot_y=1649,
            max_size=64,
            logic_region=Regions.LlamaTemple,
            group=4,
        ),
        CustomLocation(
            map=Maps.AztecLlamaTemple,
            name="Llama Temple: Llama Left",
            x=1737,
            y=472,
            z=2200,
            rot_y=1649,
            max_size=72,
            logic_region=Regions.LlamaTemple,
            group=4,
        ),
        CustomLocation(
            map=Maps.AztecLlamaTemple,
            name="Llama Temple: Matching Room",
            x=1082,
            y=641,
            z=2186,
            max_size=72,
            logic_region=Regions.LlamaTempleMatching,
            group=4,
        ),
        CustomLocation(
            name="Llama Temple: Vanilla Close Warp 1",
            map=Maps.AztecLlamaTemple,
            x=2644.013,
            y=371.333,
            z=2185.246,
            rot_y=3186,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.LlamaTemple,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.LlamaW1aTagged,
        ),
        CustomLocation(
            name="Llama Temple: Vanilla Far Warp 1",
            map=Maps.AztecLlamaTemple,
            x=1529.909,
            y=641.333,
            z=2407.096,
            rot_y=1024,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.LlamaTemple,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.LlamaW1bTagged,
        ),
        CustomLocation(
            name="Llama Temple: Vanilla Lava Warp 2",
            map=Maps.AztecLlamaTemple,
            x=1408.806,
            y=420.333,
            z=3754.905,
            rot_y=3254,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.LlamaTempleBack,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.LlamaW2bTagged,
        ),
        CustomLocation(
            name="Llama Temple: Vanilla Close Warp 2",
            map=Maps.AztecLlamaTemple,
            x=2638.672,
            y=371.333,
            z=2632.833,
            rot_y=2776,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.LlamaTemple,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.LlamaW2aTagged,
        ),
        CustomLocation(
            map=Maps.AztecLlamaTemple,
            name="In the Water",
            x=2273,
            y=207,
            z=2385,
            max_size=64,
            logic_region=Regions.LlamaTemple,
            logic=lambda l: Events.AztecLlamaSpit in l.Events and l.HasGun(Kongs.any) and l.swim and l.scope and ((l.istiny and l.isKrushaAdjacent(Kongs.tiny)) or (not l.istiny)),
            group=4,
            banned_types=[LocationTypes.CrownPad, LocationTypes.DirtPatch, LocationTypes.Bananaport],
        ),
        CustomLocation(
            map=Maps.AztecLlamaTemple,
            name="Llama Temple: DK Switch",
            x=1695,
            y=433,
            z=1704,
            max_size=48,
            logic_region=Regions.LlamaTemple,
            group=4,
        ),
        CustomLocation(
            map=Maps.AztecLlamaTemple,
            name="Llama Temple: Lava Room",
            x=1227,
            y=420,
            z=3572,
            max_size=56,
            logic_region=Regions.LlamaTempleBack,
            group=4,
        ),
    ],
    Levels.FranticFactory: [
        CustomLocation(
            map=Maps.FranticFactory,
            name="Under R&D Grate",
            x=4119,
            y=1313,
            z=1165.81,
            max_size=81.6,
            logic_region=Regions.RandDUpper,
            logic=lambda l: (l.grab and l.donkey) or l.CanAccessRNDRoom(),
            vanilla_crown=True,
            group=3,
            banned_types=[LocationTypes.MelonCrate, LocationTypes.Bananaport],
        ),
        CustomLocation(
            name="Dark Room",
            map=Maps.FranticFactory,
            x=1850.584,
            y=6.5,
            z=666.077,
            rot_y=3110,
            max_size=64,
            vanilla_patch=True,
            logic_region=Regions.BeyondHatch,
            logic=lambda l: ((l.punch and l.chunky) or l.CanPhase()),
            group=4,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="On Foyer Carpet",
            x=1265,
            y=830,
            z=2504,
            max_size=96,
            logic_region=Regions.FranticFactoryStart,
            group=1,
        ),
        CustomLocation(
            name="Near Foyer carpet",
            map=Maps.FranticFactory,
            x=1264.0,
            y=830.0,
            z=2550.0,
            rot_y=0,
            max_size=64,
            logic_region=Regions.FranticFactoryStart,
            group=1,
        ),
        # CrownLocation(map=Maps.FranticFactory, name="Foyer far left", x=1106, y=842, z=2106, max_size=64, logic_region=Regions.FranticFactoryStart, group=1),
        CustomLocation(
            name="Clock-in room left",
            map=Maps.FranticFactory,
            x=1130.0,
            y=842.0,
            z=2130.0,
            rot_y=432,
            max_size=64,
            logic_region=Regions.FranticFactoryStart,
            group=1,
        ),
        CustomLocation(
            name="Clock-in room right",
            map=Maps.FranticFactory,
            x=1409.0,
            y=842.0,
            z=2104.0,
            rot_y=3618,
            max_size=64,
            logic_region=Regions.FranticFactoryStart,
            group=1,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Near Hatch",
            x=519,
            y=804,
            z=1958,
            max_size=56,
            logic_region=Regions.FranticFactoryStart,
            group=1,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Hatch Pole Center",
            x=644,
            y=459,
            z=1778,
            rot_y=4073,
            max_size=72,
            logic_region=Regions.AlcoveBeyondHatch,
            group=1,
        ),
        # CrownLocation(
        #     name="Halfway the hatch near entrance - next to the window",
        #     map=Maps.FranticFactory,
        #     x=642.0,
        #     y=459.0,
        #     z=1796.0,
        #     rot_y=4073,
        #     max_size=64,
        #     logic_region=Regions.AlcoveBeyondHatch,
        #     group=1,
        # ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Hatch Pole Bottom",
            x=654,
            y=167,
            z=1988,
            max_size=64,
            logic_region=Regions.LowerCore,
            group=1,
        ),
        CustomLocation(
            name="Tunnel to production room",
            map=Maps.FranticFactory,
            x=70.0,
            y=6.0,
            z=1350.0,
            rot_y=193,
            max_size=64,
            logic_region=Regions.LowerCore,
            group=1,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Production Room Lower Section",
            x=517,
            y=188,
            z=1331,
            max_size=48,
            logic_region=Regions.MiddleCore,
            group=1,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Under High Conveyors",
            x=783,
            y=677,
            z=970,
            max_size=40,
            logic_region=Regions.UpperCore,
            group=1,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Past Tiny Production Bonus",
            x=400,
            y=858.5,
            z=1615,
            max_size=32,
            logic_region=Regions.UpperCore,
            logic=lambda l: l.twirl and l.tiny,
            group=1,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Vent platform near elevators",
            x=988,
            y=322,
            z=1175,
            max_size=40,
            logic_region=Regions.UpperCore,
            group=1,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Storage Room Corner",
            x=974,
            y=66.5,
            z=908,
            max_size=32,
            logic_region=Regions.BeyondHatch,
            group=4,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Near left storage shop",
            x=316,
            y=165,
            z=805,
            max_size=64,
            logic_region=Regions.BeyondHatch,
            group=4,
            banned_types=[LocationTypes.MelonCrate],
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Near right storage shop",
            x=319.03137207031,
            y=165.5,
            z=596.36285400391,
            rot_y=359,
            max_size=64,
            logic_region=Regions.BeyondHatch,
            vanilla_crate=True,
            group=4,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Dark Room Corner",
            x=1850,
            y=6,
            z=525,
            max_size=72,
            logic_region=Regions.BeyondHatch,
            logic=lambda l: (l.punch and l.chunky) or l.phasewalk,
            group=4,
        ),
        CustomLocation(map=Maps.FranticFactory, name="Arcade Room Bench", x=1922, y=1143, z=1515, max_size=40, logic_region=Regions.FactoryArcadeTunnel, group=4),
        CustomLocation(
            name="Next to DK Arcade",
            map=Maps.FranticFactory,
            x=1784.0,
            y=1106.0,
            z=1273.0,
            rot_y=0,
            max_size=64,
            logic_region=Regions.FactoryArcadeTunnel,
            group=4,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Near Snide",
            x=1702,
            y=810,
            z=2240,
            max_size=48,
            logic_region=Regions.Testing,
            group=5,
        ),
        CustomLocation(
            name="Near Snide (hidden)",
            map=Maps.FranticFactory,
            x=1603.0,
            y=810.0,
            z=2210.0,
            rot_y=2163,
            max_size=64,
            logic_region=Regions.Testing,
            group=5,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Right Corridor",
            x=1710,
            y=837,
            z=2328,
            max_size=48,
            logic_region=Regions.FranticFactoryStart,
            group=5,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Number Game",
            x=2666,
            y=1002,
            z=1952,
            max_size=48,
            logic_region=Regions.Testing,
            group=5,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Under Block Tower Stairs",
            x=2014,
            y=1027,
            z=1348,
            rot_y=3026,
            max_size=72,
            logic_region=Regions.Testing,
            group=5,
        ),
        # CrownLocation(
        #     name="Toy Room Under Stairs",
        #     map=Maps.FranticFactory,
        #     x=2015.0,
        #     y=1026.0,
        #     z=1364.0,
        #     rot_y=3026,
        #     max_size=64,
        #     logic_region=Regions.Testing,
        #     group=2,
        # ),
        CustomLocation(
            name="On the Block Tower",
            map=Maps.FranticFactory,
            x=2375.0,
            y=1266.0,
            z=1377.0,
            rot_y=2013,
            max_size=32,
            logic_region=Regions.Testing,
            logic=lambda l: (l.spring or l.CanMoonkick() or l.CanMoontail()),
            group=2,
            banned_types=[LocationTypes.CrownPad],
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Under Testing barrel",
            x=2634,
            y=1026,
            z=1101,
            max_size=80,
            logic_region=Regions.Testing,
            group=5,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Testing shop entrance left",
            x=1595,
            y=1113,
            z=760,
            max_size=64,
            logic_region=Regions.Testing,
            group=5,
            banned_types=[LocationTypes.MelonCrate],
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Testing shop left",
            x=1370,
            y=1131,
            z=551,
            max_size=48,
            logic_region=Regions.Testing,
            group=5,
        ),
        CustomLocation(
            name="Testing shop right",
            map=Maps.FranticFactory,
            x=1656.0,
            y=1113.0,
            z=502.0,
            rot_y=273,
            max_size=64,
            logic_region=Regions.Testing,
            group=2,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Testing shop entrance right",
            x=1735.7963867188,
            y=1113.1666259766,
            z=641.83825683594,
            rot_y=0,
            max_size=64,
            logic_region=Regions.Testing,
            vanilla_crate=True,
            group=2,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Piano Room left corner",
            x=3382,
            y=1264,
            z=641,
            max_size=56,
            logic_region=Regions.RandD,
            logic=lambda l: (l.trombone and l.lanky) or l.CanAccessRNDRoom(),
            group=3,
            banned_types=[LocationTypes.CrownPad, LocationTypes.Bananaport],
        ),
        CustomLocation(
            name="Piano Room right",
            map=Maps.FranticFactory,
            x=3470.0,
            y=1264.0,
            z=303.0,
            rot_y=113,
            max_size=64,
            logic_region=Regions.RandD,
            logic=lambda l: ((l.trombone and l.islanky) or l.CanAccessRNDRoom()),
            group=3,
            banned_types=[LocationTypes.CrownPad, LocationTypes.Bananaport],
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Diddy R&D corner",
            x=4450,
            y=1336,
            z=735,
            max_size=72,
            logic_region=Regions.RandDUpper,
            logic=lambda l: (l.guitar and l.diddy) or l.CanAccessRNDRoom(),
            group=3,
            banned_types=[LocationTypes.CrownPad, LocationTypes.Bananaport],
        ),
        CustomLocation(
            name="Diddy R&D",
            map=Maps.FranticFactory,
            x=4463.0,
            y=1336.0,
            z=551.0,
            rot_y=3528,
            max_size=64,
            logic_region=Regions.RandDUpper,
            logic=lambda l: ((l.guitar and l.isdiddy) or l.CanAccessRNDRoom()),
            group=3,
            banned_types=[LocationTypes.CrownPad, LocationTypes.Bananaport],
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Chunky R&D",
            x=4574,
            y=1336,
            z=1350,
            max_size=64,
            logic_region=Regions.RandDUpper,
            logic=lambda l: (l.triangle and l.climbing and l.chunky and l.punch) or l.CanAccessRNDRoom(),
            group=3,
            banned_types=[
                LocationTypes.CrownPad,
                LocationTypes.Bananaport,
            ],  # Entering a crown battle during the Toy Boss fight would break the fight until level re-entry
        ),
        CustomLocation(
            name="In front of toy monster room",
            map=Maps.FranticFactory,
            x=4345.0,
            y=1416.0,
            z=1354.0,
            rot_y=2654,
            max_size=64,
            logic_region=Regions.RandDUpper,
            logic=lambda l: ((l.punch and l.climbing and l.ischunky) or l.CanAccessRNDRoom()),
            group=3,
            banned_types=[
                LocationTypes.CrownPad,
                LocationTypes.Bananaport,
            ],  # Entering a crown battle during the Toy Boss fight would break the fight until level re-entry
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Car Race room",
            x=3553,
            y=1264,
            z=1383,
            max_size=48,
            logic_region=Regions.RandD,
            logic=lambda l: (l.mini and l.istiny) or l.phasewalk,
        ),
        CustomLocation(
            name="Also car race room",
            map=Maps.FranticFactory,
            x=3544.0,
            y=1264.0,
            z=1301.0,
            rot_y=95,
            max_size=64,
            logic_region=Regions.FactoryTinyRaceLobby,
            logic=lambda l: ((l.mini and l.istiny) or l.CanPhase()),
            group=3,
        ),
        CustomLocation(
            name="R&D lever room",
            map=Maps.FranticFactory,
            x=3693.0,
            y=1263.0,
            z=1412.0,
            rot_y=1604,
            max_size=64,
            logic_region=Regions.RandD,
            group=3,
        ),
        CustomLocation(
            map=Maps.FranticFactory,
            name="Under second R&D Grate",
            x=4054,
            y=1313,
            z=776,
            max_size=80,
            logic_region=Regions.RandDUpper,
            logic=lambda l: (l.grab and l.donkey) or l.CanPhase() or l.generalclips,
            group=3,
            banned_types=[LocationTypes.MelonCrate, LocationTypes.Bananaport],
        ),
        CustomLocation(
            name="Vanilla Foyer Warp 2",
            map=Maps.FranticFactory,
            x=1372.674,
            y=845.667,
            z=2501.402,
            rot_y=0,
            max_size=int(WARP_MAX_SIZE * 0.92),
            logic_region=Regions.FranticFactoryStart,
            group=5,
            vanilla_port=True,
            tied_warp_event=Events.FactoryW1aTagged,
        ),
        CustomLocation(
            name="Vanilla Foyer Warp 3",
            map=Maps.FranticFactory,
            x=1435.392,
            y=842.667,
            z=2173.301,
            rot_y=3060,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.FranticFactoryStart,
            group=5,
            vanilla_port=True,
            tied_warp_event=Events.FactoryW3aTagged,
        ),
        CustomLocation(
            name="Vanilla R&D Warp 2",
            map=Maps.FranticFactory,
            x=3802.578,
            y=1264,
            z=1143.848,
            rot_y=3902,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.RandD,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.FactoryW2bTagged,
        ),
        CustomLocation(
            name="Vanilla Snide Warp 3",
            map=Maps.FranticFactory,
            x=1537.891,
            y=810.667,
            z=2159.589,
            rot_y=68,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.Testing,
            group=5,
            vanilla_port=True,
            tied_warp_event=Events.FactoryW3bTagged,
        ),
        CustomLocation(
            name="Vanilla High Warp 4",
            map=Maps.FranticFactory,
            x=907.894,
            y=422.5,
            z=1535.177,
            rot_y=2628,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.SpinningCore,
            group=1,
            vanilla_port=True,
            tied_warp_event=Events.FactoryW4bTagged,
        ),
        CustomLocation(
            name="Vanilla Low Warp 4",
            map=Maps.FranticFactory,
            x=916.314,
            y=0,
            z=1332.135,
            rot_y=1024,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.BeyondHatch,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.FactoryW4aTagged,
        ),
        CustomLocation(
            name="Vanilla Testing Warp 5",
            map=Maps.FranticFactory,
            x=1624.949,
            y=1113.167,
            z=649.048,
            rot_y=535,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.Testing,
            group=2,
            vanilla_port=True,
            tied_warp_event=Events.FactoryW5bTagged,
        ),
        CustomLocation(
            name="Vanilla Foyer Warp 1",
            map=Maps.FranticFactory,
            x=1153.009,
            y=845.667,
            z=2502.361,
            rot_y=0,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.FranticFactoryStart,
            group=5,
            vanilla_port=True,
            tied_warp_event=Events.FactoryW1aTagged,
        ),
        CustomLocation(
            name="Vanilla Storage Warp 1",
            map=Maps.FranticFactory,
            x=1576.006,
            y=6.5,
            z=521.252,
            rot_y=4050,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.BeyondHatch,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.FactoryW1bTagged,
        ),
        CustomLocation(
            name="Vanilla Arcade Warp 5",
            map=Maps.FranticFactory,
            x=1800.378,
            y=1137.833,
            z=1374.809,
            rot_y=967,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.FactoryArcadeTunnel,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.FactoryW5aTagged,
        ),
        CustomLocation(
            map=Maps.FactoryCrusher,
            name="Crusher: Central Safehaven",
            x=116,
            y=0.5,
            z=468,
            max_size=72,
            logic_region=Regions.InsideCore,
            logic=lambda l: (l.strongKong and l.isdonkey) or l.CanPhase() or l.generalclips,
            group=1,
        ),
        CustomLocation(
            map=Maps.FactoryPowerHut,
            name="Power Hut: Corner",
            x=62,
            y=0,
            z=64,
            max_size=48,
            logic_region=Regions.PowerHut,
            group=4,
        ),
    ],
    Levels.GloomyGalleon: [
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="Under cavern shop",
            x=3296.94,
            y=1670,
            z=2450.29,
            max_size=84.8,
            logic_region=Regions.GloomyGalleonStart,
            logic=lambda l: (l.punch and l.chunky) or l.CanPhase() or l.CanSkew(False, False),
            vanilla_crown=True,
            group=3,
        ),
        CustomLocation(
            name="Lighthouse: Interior Rear",
            map=Maps.GalleonLighthouse,
            x=457.54,
            y=0.0,
            z=716.299,
            rot_y=18,
            max_size=64,
            vanilla_patch=True,
            logic_region=Regions.Lighthouse,
            group=1,
        ),
        CustomLocation(
            name="Near Cactus",
            map=Maps.GloomyGalleon,
            x=3899.9157714844,
            y=1551.6126708984,
            z=1405.4954833984,
            rot_y=0,
            max_size=64,
            is_galleon_floating_crate=True,
            logic_region=Regions.Shipyard,
            vanilla_crate=True,
            banned_types=[LocationTypes.DirtPatch, LocationTypes.CrownPad, LocationTypes.Bananaport],
            group=6,
        ),
        CustomLocation(
            name="Front of cannonball",
            map=Maps.GloomyGalleon,
            x=1366.0,
            y=1610.0,
            z=2586.0,
            rot_y=0,
            max_size=64,
            logic_region=Regions.GalleonBeyondPineappleGate,
            logic=lambda l: l.CanGetOnCannonGamePlatform(),
            group=4,
        ),
        CustomLocation(
            name="Behind the cannonball",
            map=Maps.GloomyGalleon,
            x=1261.0,
            y=1610.0,
            z=2588.0,
            rot_y=0,
            max_size=64,
            logic_region=Regions.GalleonBeyondPineappleGate,
            logic=lambda l: l.CanGetOnCannonGamePlatform(),
            group=4,
        ),
        CustomLocation(
            name="Next to cannonball cannon",
            map=Maps.GloomyGalleon,
            x=1310.0,
            y=1610.0,
            z=3055.0,
            rot_y=2048,
            max_size=64,
            logic_region=Regions.GalleonBeyondPineappleGate,
            logic=lambda l: l.CanGetOnCannonGamePlatform(),
            group=4,
        ),
        CustomLocation(
            name="Entrance tunnel crossing",
            map=Maps.GloomyGalleon,
            x=2534.0,
            y=1610.0,
            z=3231.0,
            rot_y=3094,
            max_size=64,
            logic_region=Regions.GloomyGalleonStart,
            group=3,
        ),
        CustomLocation(
            name="Tunnel to chests far",
            map=Maps.GloomyGalleon,
            x=3072,
            y=1790,
            z=3501,
            max_size=48,
            logic_region=Regions.GloomyGalleonStart,
            group=3,
        ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="Tunnel to chests close",
            x=3072,
            y=1790,
            z=3360,
            rot_y=2048,
            max_size=72,
            logic_region=Regions.GloomyGalleonStart,
            group=3,
        ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="Chests tunnel alcove",
            x=3048,
            y=1670,
            z=3832,
            max_size=64,
            logic_region=Regions.GloomyGalleonStart,
            group=3,
        ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="Chests room",
            x=3506,
            y=1670,
            z=3802,
            max_size=88,
            logic_region=Regions.GloomyGalleonStart,
            group=3,
        ),
        CustomLocation(
            name="Behind Punch GB Chest",
            map=Maps.GloomyGalleon,
            x=3564.0,
            y=1670.0,
            z=3944.0,
            rot_y=2503,
            max_size=64,
            logic_region=Regions.GloomyGalleonStart,
            group=3,
        ),
        CustomLocation(
            name="Behind the cannon wreck",
            map=Maps.GloomyGalleon,
            x=3199.0,
            y=1670.0,
            z=3463.0,
            rot_y=2264,
            max_size=64,
            logic_region=Regions.GloomyGalleonStart,
            group=3,
        ),
        CustomLocation(
            map=Maps.Galleon2DShip,
            name="Lanky 2DS: Inside Chest",
            x=2423,
            y=2.158,
            z=862,
            rot_y=0,
            max_size=160,
            logic_region=Regions.LankyShip,
            group=8,
            banned_types=[LocationTypes.CrownPad, LocationTypes.DirtPatch, LocationTypes.Bananaport],
        ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="Left of cavern shop",
            x=3175,
            y=1670,
            z=2527,
            max_size=56,
            logic_region=Regions.GloomyGalleonStart,
            group=3,
        ),
        # CustomLocation(
        #     name="On the ship near Cranky",
        #     map=Maps.GloomyGalleon,
        #     x=3068.0,
        #     y=1790.0,
        #     z=3386.0,
        #     rot_y=2048,
        #     max_size=64,
        #     logic_region=Regions.GloomyGalleonStart,
        #     group=3,
        # ),
        # CrownLocation(
        #     map=Maps.GloomyGalleon,
        #     name="Front of Cranky",
        #     x=3314,
        #     y=1790,
        #     z=2474,
        #     max_size=64,
        #     logic_region=Regions.GloomyGalleonStart,
        #     group=3,
        # ),
        CustomLocation(
            name="Front of cavern shop",
            map=Maps.GloomyGalleon,
            x=3357.0,
            y=1790.0,
            z=2436.0,
            rot_y=3276,
            max_size=64,
            logic_region=Regions.GloomyGalleonStart,
            group=3,
        ),
        # CrownLocation(map=Maps.GloomyGalleon, name="On Cranky", x=3290, y=1870, z=2372, max_size=32, logic_region=0, group=0),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="Near Bridge Warp 3",
            x=3116,
            y=1890,
            z=2896,
            max_size=40,
            logic_region=Regions.GalleonPastVines,
            group=3,
        ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="Near the Rocketbarrel",
            x=1396,
            y=1610,
            z=4150,
            max_size=56,
            logic_region=Regions.LighthousePlatform,
            group=1,
        ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="Near the Baboon Blast",
            x=1618,
            y=1610,
            z=4175,
            max_size=56,
            logic_region=Regions.LighthousePlatform,
            group=1,
        ),
        CustomLocation(
            name="Next to Lighthouse ladder",
            map=Maps.GloomyGalleon,
            x=1611.0,
            y=1610.0,
            z=3933.0,
            rot_y=3652,
            max_size=64,
            logic_region=Regions.LighthousePlatform,
            group=1,
        ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="Under lighthouse Enguarde box",
            x=1987,
            y=969,
            z=3693,
            max_size=160,
            logic_region=Regions.LighthouseUnderwater,
            group=7,
            logic=lambda l: Events.LighthouseEnguarde in l.Events,
            banned_types=[LocationTypes.CrownPad, LocationTypes.DirtPatch, LocationTypes.Bananaport],
        ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="In Enguarde Alcove",
            x=823.223,
            y=1464.592,
            z=4797.005,
            max_size=160,
            logic_region=Regions.LighthouseEnguardeDoor,
            group=7,
            logic=lambda l: Events.LighthouseEnguarde in l.Events,
            banned_types=[LocationTypes.CrownPad, LocationTypes.DirtPatch, LocationTypes.Bananaport],
        ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="In Front of Mermaid Palace",
            x=964.75,
            y=105,
            z=4371.308,
            max_size=160,
            logic_region=Regions.LighthouseUnderwater,
            group=7,
            logic=lambda l: Events.LighthouseEnguarde in l.Events,
            banned_types=[LocationTypes.CrownPad, LocationTypes.DirtPatch, LocationTypes.Bananaport],
        ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="On Rocketbarrel platform",
            x=1336,
            y=1660,
            z=4071,
            rot_y=910,
            max_size=32,
            logic_region=Regions.LighthousePlatform,
            group=1,
        ),
        # CrownLocation(
        #     name="Under Diddy Barrel",
        #     map=Maps.GloomyGalleon,
        #     x=1340.0,
        #     y=1660.0,
        #     z=4043.0,
        #     rot_y=910,
        #     max_size=43.52,
        #     logic_region=Regions.LighthousePlatform,
        #     group=1,
        # ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="Lighthouse Alcove",
            x=680,
            y=1564,
            z=3940,
            max_size=88,
            logic_region=Regions.LighthouseSurface,
            group=1,
            logic=lambda l: Events.WaterLowered in l.Events,
            banned_types=[LocationTypes.MelonCrate],
        ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="Behind lighthouse shop",
            x=2071,
            y=1610,
            z=4823,
            max_size=48,
            logic_region=Regions.LighthouseSnideAlcove,
            group=1,
            banned_types=[LocationTypes.CrownPad, LocationTypes.Bananaport],
        ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="On the submarine",
            x=3918,
            y=442,
            z=1362,
            max_size=160,
            logic_region=Regions.ShipyardUnderwater,
            group=6,
            logic=lambda l: Events.ShipyardEnguarde in l.Events,
            banned_types=[LocationTypes.CrownPad, LocationTypes.DirtPatch, LocationTypes.Bananaport],
        ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="In the Shipwreck with Replenishables",
            x=2319,
            y=639,
            z=2387,
            logic_region=Regions.ShipyardUnderwater,
            group=6,
            logic=lambda l: Events.ShipyardEnguarde in l.Events,
            banned_types=[LocationTypes.CrownPad, LocationTypes.DirtPatch, LocationTypes.Bananaport],
        ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="Behind Musical Cactus",
            x=4470.597,
            y=1298.062,
            z=974.728,
            logic_region=Regions.ShipyardUnderwater,
            group=6,
            logic=lambda l: Events.ShipyardEnguarde in l.Events,
            banned_types=[LocationTypes.CrownPad, LocationTypes.DirtPatch, LocationTypes.Bananaport],
        ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="On Top of Mechfish Grate",
            x=3065.921,
            y=72.238,
            z=529.702,
            logic_region=Regions.ShipyardUnderwater,
            group=6,
            logic=lambda l: Events.ShipyardEnguarde in l.Events,
            banned_types=[LocationTypes.DirtPatch, LocationTypes.CrownPad, LocationTypes.Bananaport],
        ),
        CustomLocation(
            map=Maps.GloomyGalleon,
            name="On Gold Tower",
            x=1660,
            y=2040,
            z=487,
            max_size=40,
            logic_region=Regions.TreasureRoom,
            logic=lambda l: ((l.balloon and l.islanky) and Events.WaterRaised in l.Events or (Events.ShipyardTreasureRoomOpened in l.Events and l.monkey_maneuvers)) or l.CanMoonkick(),
            group=5,
            banned_types=[LocationTypes.Bananaport],  # Hard to detect that it's bad to link to Diddy's tower, in which case it bricks seed gen
        ),
        CustomLocation(
            name="Treasure Room UnderWater",
            map=Maps.GloomyGalleon,
            x=1548,
            y=1458,
            z=708,
            max_size=160,
            logic_region=Regions.TreasureRoom,
            group=5,
            logic=lambda l: Events.ShipyardTreasureRoomOpened in l.Events,
            banned_types=[LocationTypes.CrownPad, LocationTypes.DirtPatch, LocationTypes.Bananaport],
        ),
        CustomLocation(
            name="Vanilla Floating Warp 5",
            map=Maps.GloomyGalleon,
            x=3175.065,
            y=1543.23,
            z=1844.807,
            rot_y=2913,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.Shipyard,
            group=6,
            vanilla_port=True,
            tied_warp_event=Events.GalleonW5bTagged,
        ),
        CustomLocation(
            name="Vanilla Lighthouse Warp 1",
            map=Maps.GloomyGalleon,
            x=1519.874,
            y=1610.421,
            z=4220.399,
            rot_y=0,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.LighthousePlatform,
            group=1,
            vanilla_port=True,
            tied_warp_event=Events.GalleonW1bTagged,
        ),
        CustomLocation(
            name="Vanilla Alcove Warp 3",
            map=Maps.GloomyGalleon,
            x=1981.99,
            y=1610,
            z=4774.565,
            rot_y=2480,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.LighthouseSnideAlcove,
            group=1,
            vanilla_port=True,
            tied_warp_event=Events.GalleonW3bTagged,
        ),
        CustomLocation(
            name="Vanilla Lighthouse Warp 5",
            map=Maps.GloomyGalleon,
            x=1395.717,
            y=1609.944,
            z=3969.458,
            rot_y=2731,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.LighthousePlatform,
            group=1,
            vanilla_port=True,
            tied_warp_event=Events.GalleonW5aTagged,
        ),
        CustomLocation(
            name="Vanilla Cavern Warp 2",
            map=Maps.GloomyGalleon,
            x=2905.288,
            y=1700,
            z=3367.477,
            rot_y=2094,
            max_size=int(WARP_MAX_SIZE * 0.76),
            logic_region=Regions.GloomyGalleonStart,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.GalleonW2aTagged,
        ),
        CustomLocation(
            name="Vanilla Bridge Warp 3",
            map=Maps.GloomyGalleon,
            x=3067.098,
            y=1890,
            z=2922.948,
            rot_y=0,
            max_size=int(WARP_MAX_SIZE * 1.24),
            logic_region=Regions.GalleonPastVines,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.GalleonW3aTagged,
        ),
        CustomLocation(
            name="Vanilla Floating Warp 2",
            map=Maps.GloomyGalleon,
            x=1556.673,
            y=1548.119,
            z=2008.956,
            rot_y=853,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.Shipyard,
            group=6,
            vanilla_port=True,
            tied_warp_event=Events.GalleonW2bTagged,
        ),
        CustomLocation(
            name="Vanilla Gold Tower Warp 4",
            map=Maps.GloomyGalleon,
            x=2167.65,
            y=1880,
            z=763.183,
            rot_y=2731,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.TreasureRoomDiddyGoldTower,
            group=12,
            vanilla_port=True,
            tied_warp_event=Events.GalleonW4aTagged,
        ),
        CustomLocation(
            name="Vanilla Cavern Warp 1",
            map=Maps.GloomyGalleon,
            x=2844.223,
            y=1673.752,
            z=3235.207,
            rot_x=6,
            rot_y=3049,
            rot_z=2,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.GloomyGalleonStart,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.GalleonW1aTagged,
        ),
        CustomLocation(
            name="Vanilla Floating Warp 4",
            map=Maps.GloomyGalleon,
            x=2762.929,
            y=1550.856,
            z=1431.044,
            rot_y=0,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.Shipyard,
            group=6,
            vanilla_port=True,
            tied_warp_event=Events.GalleonW4bTagged,
        ),
        CustomLocation(
            map=Maps.GalleonSickBay,
            name="Seasick Ship: Left of Cannon",
            x=718,
            y=20,
            z=129,
            max_size=56,
            logic_region=Regions.SickBay,
            group=2,
        ),
        CustomLocation(
            name="Seasick Ship: Entrance",
            map=Maps.GalleonSickBay,
            x=628.0,
            y=20.0,
            z=229.0,
            rot_y=2048,
            max_size=64,
            logic_region=Regions.SickBay,
            group=2,
        ),
        CustomLocation(
            map=Maps.GalleonSickBay,
            name="Seasick Ship: Right of Cannon",
            x=544,
            y=20,
            z=129,
            max_size=56,
            logic_region=Regions.SickBay,
            group=2,
        ),
        # CrownLocation(
        #     map=Maps.GalleonSickBay,
        #     name="Seasick Ship: In Corner",
        #     x=703,
        #     y=20,
        #     z=911,
        #     max_size=64,
        #     logic_region=Regions.SickBay,
        #     group=2,
        # ),
        CustomLocation(
            name="Seasick Ship: Back left corner",
            map=Maps.GalleonSickBay,
            x=701.0,
            y=20.0,
            z=899.0,
            rot_y=2525,
            max_size=64,
            logic_region=Regions.SickBay,
            group=2,
        ),
        CustomLocation(
            name="Seasick Ship: Behind the non-alcoholic tower",
            map=Maps.GalleonSickBay,
            x=159.0,
            y=20.0,
            z=920.0,
            rot_y=978,
            max_size=64,
            logic_region=Regions.SickBay,
            logic=lambda l: ((l.punch and l.ischunky) or l.CanPhase()),
            group=2,
        ),
        CustomLocation(
            map=Maps.GalleonSickBay,
            name="Seasick Ship: Behind Spinning Barrels",
            x=142,
            y=20,
            z=851,
            max_size=64,
            logic_region=Regions.SickBay,
            logic=lambda l: (l.punch and l.ischunky) or l.CanPhase(),
            group=2,
        ),
        CustomLocation(
            map=Maps.GalleonLighthouse,
            name="Lighthouse: Bottom Left",
            x=703,
            y=0,
            z=469,
            max_size=80,
            logic_region=Regions.Lighthouse,
            group=1,
        ),
        CustomLocation(
            map=Maps.GalleonLighthouse,
            name="Lighthouse: Back Right",
            x=282,
            y=0,
            z=670,
            max_size=80,
            logic_region=Regions.Lighthouse,
            group=1,
        ),
        CustomLocation(
            name="Lighthouse: Behind Whomp's Fortress floor 2",
            map=Maps.GalleonLighthouse,
            x=453.0,
            y=200.0,
            z=596.0,
            rot_y=0,
            max_size=64,
            logic_region=Regions.LighthouseAboveLadder,
            group=1,
        ),
        CustomLocation(
            name="Lighthouse: On top of Whomp's Fortress",
            map=Maps.GalleonLighthouse,
            x=418.0,
            y=720.0,
            z=497.0,
            rot_y=3572,
            max_size=64,
            logic_region=Regions.LighthouseAboveLadder,
            group=1,
        ),
    ],
    Levels.FungiForest: [
        CustomLocation(
            map=Maps.FungiForest,
            name="Giant Mushroom High Ladder Platform",
            x=1254.33,
            y=1079.33,
            z=1307.16,
            max_size=78.4,
            logic_region=Regions.MushroomUpperExterior,
            vanilla_crown=True,
            group=4,
        ),
        CustomLocation(
            name="Front of Beanstalk",
            map=Maps.FungiForest,
            x=2279.848,
            y=228.931,
            z=600.56,
            rot_y=1020,
            max_size=64,
            vanilla_patch=True,
            logic_region=Regions.WormArea,
            group=1,
        ),
        CustomLocation(
            name="Mill Grass",
            map=Maps.FungiForest,
            x=4674.706,
            y=149.873,
            z=4165.153,
            rot_y=2584,
            max_size=64,
            vanilla_patch=True,
            logic_region=Regions.MillArea,
            banned_types=[LocationTypes.Bananaport],
            is_fungi_hidden_patch=True,
            group=2,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near Well",
            x=2399,
            y=110,
            z=3186,
            max_size=96,
            logic_region=Regions.FungiForestStart,
            group=1,
            banned_types=[LocationTypes.DirtPatch, LocationTypes.Bananaport],
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Behind Clock",
            x=2300,
            y=603,
            z=2322,
            max_size=56,
            logic_region=Regions.FungiForestStart,
            group=1,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="In front of Clock",
            x=2591,
            y=603,
            z=2237,
            max_size=64,
            logic_region=Regions.FungiForestStart,
            group=1,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near Blue Tunnel",
            x=3210,
            y=167,
            z=2613,
            max_size=56,
            logic_region=Regions.FungiForestStart,
            group=1,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near Mills shop",
            x=3154,
            y=268,
            z=3682,
            max_size=56,
            logic_region=Regions.MillArea,
            group=2,
            logic=lambda l: (l.TimeAccess(Regions.MillArea, Time.Day)),
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Behind rafters barn",
            x=3139,
            y=272,
            z=4343,
            max_size=64,
            logic_region=Regions.MillArea,
            group=2,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Left of rafters barn",
            x=3400,
            y=272,
            z=4652,
            max_size=64,
            logic_region=Regions.MillArea,
            group=2,
        ),
        CustomLocation(
            name="Next to Diddy Pad",
            map=Maps.FungiForest,
            x=3396.0,
            y=272.0,
            z=4551.0,
            rot_y=345,
            max_size=64,
            logic_region=Regions.MillArea,
            group=2,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Behind Dark Attic",
            x=3175.0856933594,
            y=272.33334350586,
            z=4425.095703125,
            rot_y=31,
            max_size=64,
            logic_region=Regions.MillArea,
            vanilla_crate=True,
            group=2,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near Mill Tag",
            x=4706,
            y=139,
            z=4373,
            max_size=64,
            logic_region=Regions.MillArea,
            group=2,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near Well Exit",
            x=5279,
            y=207,
            z=3556,
            max_size=72,
            logic_region=Regions.MillArea,
            group=2,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near Winch",
            x=4533,
            y=162,
            z=3372,
            max_size=72,
            logic_region=Regions.MillArea,
            group=2,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near Mill Punch Door",
            x=4439,
            y=162,
            z=3853,
            max_size=72,
            logic_region=Regions.MillArea,
            group=2,
        ),
        CustomLocation(
            name="Next to Crusher Output",
            map=Maps.FungiForest,
            x=4404.0,
            y=162.0,
            z=3520.0,
            rot_y=1525,
            max_size=64,
            logic_region=Regions.MillArea,
            group=2,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="En route to DK Barn",
            x=4604,
            y=206,
            z=2844,
            max_size=64,
            logic_region=Regions.ThornvineArea,
            group=6,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Right of DK Barn",
            x=4492,
            y=116,
            z=1959,
            max_size=80,
            logic_region=Regions.ThornvineArea,
            group=6,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Far Right of DK Barn",
            x=4147,
            y=115,
            z=1496,
            max_size=64,
            logic_region=Regions.ThornvineArea,
            group=6,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Behind DK Barn",
            x=3486,
            y=115,
            z=1427,
            max_size=56,
            logic_region=Regions.ThornvineArea,
            group=6,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Far Left of DK Barn",
            x=3529,
            y=115,
            z=2035,
            max_size=80,
            logic_region=Regions.ThornvineArea,
            group=6,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near DK Barn",
            x=4151,
            y=115,
            z=1784,
            max_size=64,
            logic_region=Regions.ThornvineArea,
            group=6,
        ),
        # CrownLocation(map=Maps.FungiForest, name="Behind Beanstalk Night Gate", x=3621, y=186, z=936, max_size=48, logic_region=0, group=1),
        CustomLocation(
            name="Alcove near the apple",
            map=Maps.FungiForest,
            x=3635.0,
            y=186.0,
            z=936.0,
            rot_y=2947,
            max_size=64,
            logic_region=Regions.WormArea,
            logic=lambda l: (l.TimeAccess(Regions.WormArea, Time.Night) or l.CanPhase()),
            group=1,
        ),
        CustomLocation(
            name="On the Tomato Field",
            map=Maps.FungiForest,
            x=3158.0,
            y=228.0,
            z=768.0,
            rot_y=989,
            max_size=64,
            logic_region=Regions.WormArea,
            group=1,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near beanstalk shop",
            x=2890,
            y=174,
            z=189,
            max_size=64,
            logic_region=Regions.WormArea,
            group=1,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near Beanstalk Area Entrance",
            x=2721,
            y=200,
            z=982,
            max_size=64,
            logic_region=Regions.WormArea,
            group=1,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near Beanstalk",
            x=1991,
            y=231,
            z=829,
            max_size=72,
            logic_region=Regions.WormArea,
            group=1,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near Beanstalk Mini Monkey",
            x=1902,
            y=227,
            z=369,
            max_size=72,
            logic_region=Regions.WormArea,
            group=1,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near Giant Mushroom",
            x=1642,
            y=234,
            z=867,
            max_size=64,
            logic_region=Regions.GiantMushroomArea,
            group=4,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near Yellow Tunnel",
            x=236,
            y=179,
            z=1307,
            max_size=96,
            logic_region=Regions.GiantMushroomArea,
            group=4,
            banned_types=[LocationTypes.DirtPatch, LocationTypes.Bananaport],
        ),
        # CustomLocation(  # Clips through the floor
        #     map=Maps.FungiForest,
        #     name="Near Cranky",
        #     x=583,
        #     y=182,
        #     z=272,
        #     max_size=72,
        #     logic_region=Regions.GiantMushroomArea,
        #     group=4,
        # ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near Lower Baboon Blast Ladder",
            x=567,
            y=389,
            z=731,
            max_size=64,
            logic_region=Regions.MushroomLowerExterior,
            group=4,
        ),
        CustomLocation(
            name="Near Baboon Blast",
            map=Maps.FungiForest,
            x=752.0,
            y=589.0,
            z=1296.0,
            rot_y=534,
            max_size=64,
            logic_region=Regions.MushroomBlastLevelExterior,
            group=4,
        ),
        # CrownLocation(
        #     map=Maps.FungiForest,
        #     name="Behind Upper Baboon Blast Ladder",
        #     x=751,
        #     y=589,
        #     z=1297,
        #     max_size=64,
        #     logic_region=Regions.MushroomBlastLevelExterior,
        #     group=4,
        # ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Above Upper Baboon Blast Ladder",
            x=671,
            y=779,
            z=1320,
            max_size=56,
            logic_region=Regions.MushroomUpperMidExterior,
            group=4,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Highest Giant Mushroom Platform",
            x=1196,
            y=1250,
            z=1315,
            max_size=64,
            logic_region=Regions.MushroomUpperExterior,
            group=4,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Behind Rabbit",
            x=2408,
            y=142,
            z=3705,
            max_size=80,
            logic_region=Regions.HollowTreeArea,
            group=5,
        ),
        CustomLocation(
            name="Next to Rabbit's house",
            map=Maps.FungiForest,
            x=2297.0,
            y=142.0,
            z=3703.0,
            rot_y=2946,
            max_size=64,
            logic_region=Regions.HollowTreeArea,
            group=5,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Under Owl Tree",
            x=1274,
            y=249,
            z=3750,
            max_size=72,
            logic_region=Regions.HollowTreeArea,
            group=5,
        ),
        CustomLocation(
            name="Under the Owl Tree",
            map=Maps.FungiForest,
            x=1274.0,
            y=249.0,
            z=3686.0,
            rot_y=2048,
            max_size=64,
            logic_region=Regions.HollowTreeArea,
            group=5,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near Owl Tree",
            x=443.95544433594,
            y=189.33332824707,
            z=3946.9885253906,
            rot_y=10,
            max_size=64,
            logic_region=Regions.HollowTreeArea,
            vanilla_crate=True,
            group=5,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near Owl Rocketbarrel",
            x=534,
            y=189,
            z=3948,
            rot_y=1080,
            max_size=72,
            logic_region=Regions.HollowTreeArea,
            group=5,
        ),
        # CrownLocation(
        #     name="near Diddy Barrel",
        #     map=Maps.FungiForest,
        #     x=549.0,
        #     y=189.0,
        #     z=3940.0,
        #     rot_y=1080,
        #     max_size=64,
        #     logic_region=Regions.HollowTreeArea,
        #     group=5,
        # ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Grass Near Owl Rocketbarrel",
            x=278,
            y=190,
            z=3707,
            max_size=96,
            logic_region=Regions.HollowTreeArea,
            group=5,
            banned_types=[LocationTypes.DirtPatch, LocationTypes.Bananaport],
        ),
        CustomLocation(
            name="Top of Owl Tree",
            map=Maps.FungiForest,
            x=1268.0,
            y=575.0,
            z=3840.0,
            rot_y=34,
            max_size=64,
            logic_region=Regions.HollowTreeArea,
            logic=lambda l: ((l.jetpack and l.isdiddy) or l.CanMoonkick()),
            group=5,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="On Mill",
            x=4164,
            y=376,
            z=3526,
            max_size=64,
            logic_region=Regions.MillArea,
            group=5,
        ),
        CustomLocation(
            name="Vanilla Clock Warp 1",
            map=Maps.FungiForest,
            x=2721.163,
            y=275.479,
            z=2346.108,
            rot_x=16,
            rot_y=956,
            rot_z=2,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.FungiForestStart,
            group=1,
            vanilla_port=True,
            tied_warp_event=Events.ForestW1aTagged,
        ),
        CustomLocation(
            name="Vanilla Mill Warp 1",
            map=Maps.FungiForest,
            x=4135.515,
            y=164.246,
            z=3744.737,
            rot_y=3595,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.MillArea,
            group=5,
            vanilla_port=True,
            banned_types=[LocationTypes.MelonCrate],
            tied_warp_event=Events.ForestW1bTagged,
        ),
        CustomLocation(
            name="Vanilla Clock Warp 2",
            map=Maps.FungiForest,
            x=2554.467,
            y=281.944,
            z=2113.114,
            rot_x=6,
            rot_y=1843,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.FungiForestStart,
            group=1,
            vanilla_port=True,
            tied_warp_event=Events.ForestW2aTagged,
        ),
        CustomLocation(
            name="Vanilla Beanstalk Warp 2",
            map=Maps.FungiForest,
            x=3253.749,
            y=182.831,
            z=224.007,
            rot_x=356,
            rot_y=3982,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.WormArea,
            group=1,
            vanilla_port=True,
            banned_types=[LocationTypes.MelonCrate],
            tied_warp_event=Events.ForestW2bTagged,
        ),
        CustomLocation(
            name="Vanilla Clock Warp 3",
            map=Maps.FungiForest,
            x=2234.942,
            y=273.516,
            z=2227.188,
            rot_x=10,
            rot_y=2742,
            max_size=int(WARP_MAX_SIZE * 1.08),
            logic_region=Regions.FungiForestStart,
            group=1,
            vanilla_port=True,
            tied_warp_event=Events.ForestW3aTagged,
        ),
        CustomLocation(
            name="Vanilla Clock Warp 4",
            map=Maps.FungiForest,
            x=2483.208,
            y=256.76,
            z=2555.135,
            rot_x=16,
            rot_y=0,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.FungiForestStart,
            group=1,
            vanilla_port=True,
            tied_warp_event=Events.ForestW4aTagged,
        ),
        CustomLocation(
            name="Vanilla Owl Tree Warp 4",
            map=Maps.FungiForest,
            x=1276.934,
            y=249.333,
            z=3533.974,
            rot_y=2014,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.HollowTreeArea,
            group=5,
            vanilla_port=True,
            tied_warp_event=Events.ForestW4bTagged,
        ),
        CustomLocation(
            name="Vanilla Lower Warp 5",
            map=Maps.FungiForest,
            x=933.914,
            y=219.5,
            z=1724.376,
            rot_y=1798,
            max_size=int(WARP_MAX_SIZE * 1.24),
            logic_region=Regions.GiantMushroomArea,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.ForestW5bTagged,
        ),
        CustomLocation(
            name="Vanilla Upper Warp 5",
            map=Maps.FungiForest,
            x=459.833,
            y=1250,
            z=1200.736,
            rot_y=3334,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.MushroomUpperExterior,
            group=4,
            vanilla_port=True,
            banned_types=[LocationTypes.MelonCrate],
            tied_warp_event=Events.ForestW5aTagged,
        ),
        CustomLocation(
            name="Vanilla Mushroom Warp 3",
            map=Maps.FungiForest,
            x=1179.775,
            y=179.333,
            z=1242.065,
            rot_y=546,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.GiantMushroomArea,
            group=4,
            vanilla_port=True,
            banned_types=[LocationTypes.MelonCrate],
            tied_warp_event=Events.ForestW3bTagged,
        ),
        CustomLocation(
            map=Maps.ForestAnthill,
            name="Anthill: Orange Platform",
            x=768,
            y=205,
            z=421,
            max_size=56,
            logic_region=Regions.Anthill,
            group=5,
        ),
        CustomLocation(
            map=Maps.ForestWinchRoom,
            name="Winch Room: Opposite Entrance",
            x=310,
            y=0,
            z=342,
            max_size=64,
            logic_region=Regions.WinchRoom,
            group=3,
        ),
        CustomLocation(
            map=Maps.FungiForest,
            name="Near Thornvine Barn",
            x=4578.4951171875,
            y=115,
            z=1824.7908935547,
            rot_y=52,
            max_size=64,
            logic_region=Regions.ThornvineArea,
            vanilla_crate=True,
            group=6,
        ),
        CustomLocation(
            map=Maps.ForestThornvineBarn,
            name="DK Barn: Near Entrance",
            x=537,
            y=4,
            z=143,
            max_size=72,
            logic_region=Regions.ThornvineBarn,
            group=6,
        ),
        CustomLocation(
            map=Maps.ForestThornvineBarn,
            name="DK Barn: Near Ladder",
            x=106,
            y=4,
            z=590,
            max_size=64,
            logic_region=Regions.ThornvineBarn,
            group=6,
        ),
        CustomLocation(
            name="DK Barn: Next to ladder",
            map=Maps.ForestThornvineBarn,
            x=80.0,
            y=4.0,
            z=627.0,
            rot_y=2048,
            max_size=64,
            logic_region=Regions.ThornvineBarn,
            group=6,
        ),
        CustomLocation(
            map=Maps.ForestThornvineBarn,
            name="DK Barn: back left",
            x=566.02142333984,
            y=4,
            z=524.61657714844,
            rot_y=0,
            max_size=64,
            logic_region=Regions.ThornvineBarn,
            vanilla_crate=True,
            group=6,
        ),
        CustomLocation(
            map=Maps.ForestMillFront,
            name="Mill Front: Near Conveyor",
            x=63,
            y=0,
            z=400,
            max_size=48,
            logic_region=Regions.GrinderRoom,
            group=3,
        ),
        CustomLocation(
            map=Maps.ForestMillFront,
            name="Mill Front: Near Mini Monkey",
            x=256,
            y=0,
            z=196,
            max_size=80,
            logic_region=Regions.GrinderRoom,
            group=3,
        ),
        CustomLocation(
            name="Mill front: Near levers",
            map=Maps.ForestMillFront,
            x=234.0,
            y=0.0,
            z=229.0,
            rot_y=352,
            max_size=64,
            logic_region=Regions.GrinderRoom,
            group=3,
        ),
        CustomLocation(
            name="Mill Back: near Chunky's coins",
            map=Maps.ForestMillBack,
            x=608.0,
            y=0.0,
            z=585.0,
            rot_y=2707,
            max_size=64,
            logic_region=Regions.MillChunkyTinyArea,
            group=3,
        ),
        CustomLocation(
            map=Maps.ForestGiantMushroom,
            name="Giant Mushroom: Near Tiny Bonus",
            x=550,
            y=409,
            z=200,
            max_size=72,
            logic_region=Regions.MushroomLowerMid,
            group=4,
        ),
        CustomLocation(
            map=Maps.ForestGiantMushroom,
            name="Giant Mushroom: Near Gun Switches",
            x=448,
            y=82,
            z=195,
            max_size=64,
            logic_region=Regions.MushroomLower,
            group=4,
        ),
        CustomLocation(
            map=Maps.ForestGiantMushroom,
            name="Giant Mushroom: Near Bottom Cannon",
            x=596,
            y=0,
            z=680,
            max_size=72,
            logic_region=Regions.MushroomLower,
            group=4,
        ),
        CustomLocation(
            map=Maps.ForestGiantMushroom,
            name="Giant Mushroom: Near vines",
            x=218,
            y=979,
            z=529,
            max_size=64,
            logic_region=Regions.MushroomUpperMid,
            group=4,
        ),
        CustomLocation(
            name="Giant Mushroom: Next to a cannon",
            map=Maps.ForestGiantMushroom,
            x=127.0,
            y=1189.0,
            z=532.0,
            rot_y=1137,
            max_size=64,
            logic_region=Regions.MushroomUpperVineFloor,
            logic=lambda l: ((l.istiny and l.twirl) or (l.isdonkey and (not l.isKrushaAdjacent(Kongs.donkey)))),
            group=4,
        ),
        CustomLocation(
            # With reduced fall damage threshold and OHKO, you have to slide off of the night door platform to avoid fall damage
            name="Giant Mushroom: Cannon below night door",
            map=Maps.ForestGiantMushroom,
            x=763.0,
            y=739.0,
            z=513.0,
            rot_y=3163,
            max_size=64,
            logic_region=Regions.MushroomNightDoor,
            group=4,
        ),
        CustomLocation(
            map=Maps.ForestGiantMushroom,
            name="Giant Mushroom: On Top Viney Platform",
            x=543,
            y=1169,
            z=700,
            max_size=80,
            logic_region=Regions.MushroomUpper,
            group=4,
        ),
        CustomLocation(
            map=Maps.ForestMillAttic,
            name="Mill Attic: Near Box",
            x=138,
            y=0,
            z=310,
            max_size=64,
            logic_region=Regions.MillAttic,
            group=3,
        ),
        CustomLocation(
            map=Maps.ForestLankyZingersRoom,
            name="Mushroom Zingers: Opposite Entrance",
            x=414,
            y=0,
            z=282,
            max_size=64,
            logic_region=Regions.MushroomLankyZingersRoom,
            group=4,
        ),
        CustomLocation(
            map=Maps.ForestLankyMushroomsRoom,
            name="Mushroom Slam: Opposite Entrance",
            x=408,
            y=0,
            z=309,
            max_size=64,
            logic_region=Regions.MushroomLankyMushroomsRoom,
            group=4,
        ),
        CustomLocation(
            map=Maps.ForestChunkyFaceRoom,
            name="Face Puzzle: Near Puzzle",
            x=229,
            y=0,
            z=441,
            max_size=64,
            logic_region=Regions.MushroomChunkyRoom,
            group=4,
        ),
        CustomLocation(
            map=Maps.ForestMillBack,
            name="Mill Back: Near Thatch",
            x=449,
            y=0,
            z=610,
            max_size=80,
            logic_region=Regions.MillChunkyTinyArea,
            group=3,
        ),
        CustomLocation(
            map=Maps.ForestSpider,
            name="Spider: Opposite Entrance",
            x=917,
            y=172,
            z=599,
            max_size=72,
            logic_region=Regions.SpiderRoom,
            group=3,
        ),
    ],
    Levels.CrystalCaves: [
        CustomLocation(
            map=Maps.CavesRotatingCabin,
            name="Rotating cabin: wall",
            x=329,
            y=0,
            z=193,
            max_size=56,
            logic_region=Regions.RotatingCabin,
            logic=lambda l: l.Slam and l.isdonkey,
            vanilla_crown=True,
            is_rotating_room=True,
            group=1,
        ),
        CustomLocation(
            name="Giant Kosha Room",
            map=Maps.CrystalCaves,
            x=1820.313,
            y=231.833,
            z=3596.593,
            rot_y=2006,
            max_size=64,
            vanilla_patch=True,
            logic_region=Regions.GiantKosha,
            group=5,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="In Monkeyport igloo",
            x=311,
            y=48,
            z=1719,
            max_size=32,
            logic_region=Regions.IglooArea,
            logic=lambda l: ((Events.CavesMonkeyportAccess in l.Events or l.CanPhaseswim())),
            group=3,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="In breakable igloo",
            x=755,
            y=48,
            z=818,
            max_size=32,
            logic_region=Regions.IglooArea,
            logic=lambda l: Events.CavesLargeBoulderButton in l.Events or l.CanPhaseswim() or l.generalclips,
            group=3,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="On igloo Pillar",
            x=328,
            y=132.5,
            z=1522,
            max_size=24,
            logic_region=Regions.IglooArea,
            logic=lambda l: (l.jetpack and l.isdiddy) or (l.twirl and l.istiny and (not l.isKrushaAdjacent(Kongs.tiny))),
            group=3,
            banned_types=[LocationTypes.MelonCrate],
        ),
        CustomLocation(
            name="On top of the Igloo",
            map=Maps.CrystalCaves,
            x=576.0,
            y=142.0,
            z=1285.0,
            rot_y=1092,
            max_size=64,
            logic_region=Regions.IglooArea,
            group=3,
            banned_types=[LocationTypes.CrownPad, LocationTypes.MelonCrate],
        ),
        CustomLocation(
            name="Under tag barrel near igloo",
            map=Maps.CrystalCaves,
            x=221.0,
            y=48.0,
            z=1412.0,
            rot_y=3276,
            max_size=64,
            logic_region=Regions.IglooArea,
            group=3,
            banned_types=[LocationTypes.MelonCrate],
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="In front of Igloo",
            x=686,
            y=5,
            z=861,
            max_size=64,
            logic_region=Regions.IglooArea,
            group=3,
            logic=lambda l: (l.HasGun(Kongs.any) and ((l.istiny and l.isKrushaAdjacent(Kongs.tiny)) or (not l.istiny))) or l.Slam,
            banned_types=[LocationTypes.CrownPad, LocationTypes.DirtPatch, LocationTypes.Bananaport],
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="In the mini cave",
            x=453,
            y=180,
            z=2571,
            max_size=64,
            logic_region=Regions.CavesBonusCave,
            group=2,
        ),
        CustomLocation(
            name="Near ice wall to boulder",
            map=Maps.CrystalCaves,
            x=1416.0,
            y=298.0,
            z=2430.0,
            rot_y=2275,
            max_size=64,
            logic_region=Regions.CrystalCavesMain,
            group=4,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="In Giant Boulder Room",
            x=1941,
            y=280,
            z=2338,
            max_size=80,
            logic_region=Regions.BoulderCave,
            group=4,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="In front of Cranky",
            x=1202,
            y=281,
            z=1649,
            max_size=40,
            logic_region=Regions.CrystalCavesMain,
            group=4,
            banned_types=[LocationTypes.MelonCrate],
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Below Ice Castle (yellow)",
            x=2066,
            y=151,
            z=1145,
            max_size=40,
            logic_region=Regions.CrystalCavesMain,
            group=2,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Below Ice castle (blue)",
            x=1952,
            y=172,
            z=1181,
            max_size=40,
            logic_region=Regions.CrystalCavesMain,
            group=2,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Ice Castle Tag barrel",
            x=2164,
            y=280,
            z=1304,
            max_size=48,
            logic_region=Regions.CrystalCavesMain,
            group=2,
        ),
        CustomLocation(
            name="Next to Ice Castle",
            map=Maps.CrystalCaves,
            x=2125.0,
            y=257.0,
            z=1054.0,
            rot_y=3584,
            max_size=64,
            logic_region=Regions.CrystalCavesMain,
            group=2,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="On Ice Castle",
            x=2176,
            y=343.5,
            z=1002,
            max_size=32,
            logic_region=Regions.CrystalCavesMain,
            logic=lambda l: (l.balloon and l.islanky) or (l.jetpack and l.isdiddy) or l.monkey_maneuvers,
            group=2,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Near Small Boulder",
            x=1598,
            y=276,
            z=970,
            max_size=56,
            logic_region=Regions.CrystalCavesMain,
            group=2,
        ),
        CustomLocation(
            name="Near ice wall to shop",
            map=Maps.CrystalCaves,
            x=1473.0,
            y=98.0,
            z=850.0,
            rot_y=375,
            max_size=64,
            logic_region=Regions.CrystalCavesMain,
            group=4,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Near cave shop",
            x=1220,
            y=64,
            z=587,
            max_size=64,
            logic_region=Regions.CavesSnideArea,
            group=4,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Under Small Boulder",
            x=1412,
            y=90,
            z=1013,
            max_size=64,
            logic_region=Regions.CrystalCavesMain,
            group=2,
            banned_types=[LocationTypes.DirtPatch, LocationTypes.MelonCrate],
        ),
        # CrownLocation(map=Maps.CrystalCaves, name="Near Entrance (OoB)", x=1718, y=-29, z=30, max_size=64, logic_region=0, group=0),
        CustomLocation(
            name="Near ice wall to Gone room",
            map=Maps.CrystalCaves,
            x=2264.0,
            y=13.0,
            z=248.0,
            rot_y=3219,
            max_size=64,
            logic_region=Regions.CrystalCavesMain,
            group=4,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Near Gorilla Gone Room",
            x=2149,
            y=13,
            z=152,
            max_size=64,
            logic_region=Regions.CrystalCavesMain,
            group=4,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="In Gorilla Gone Room",
            x=2650,
            y=13,
            z=469,
            max_size=64,
            logic_region=Regions.CavesGGRoom,
            group=4,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Near Kasplat Spire",
            x=2700,
            y=152,
            z=772,
            max_size=64,
            logic_region=Regions.CrystalCavesMain,
            group=4,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Near Funky's",
            x=2543,
            y=172,
            z=1173,
            max_size=40,
            logic_region=Regions.CrystalCavesMain,
            group=2,
        ),
        CustomLocation(
            name="Near Funky under Rocketbarrel",
            map=Maps.CrystalCaves,
            x=3013.0,
            y=253.0,
            z=931.0,
            rot_y=3716,
            max_size=64,
            logic_region=Regions.CrystalCavesMain,
            group=2,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Mini Monkeyport cave",
            x=3554,
            y=286,
            z=619,
            max_size=64,
            logic_region=Regions.CavesBlueprintCave,
            group=2,
        ),
        CustomLocation(
            name="Near headphones",
            map=Maps.CrystalCaves,
            x=2735.0,
            y=162.0,
            z=1795.0,
            rot_y=1103,
            max_size=64,
            logic_region=Regions.CabinArea,
            group=1,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Headphones ledge",
            x=2987,
            y=118,
            z=1615,
            max_size=64,
            logic_region=Regions.CabinArea,
            group=1,
        ),
        # CrownLocation(map=Maps.CrystalCaves, name="Near Ice Tag (Mid-Air)", x=2238, y=394, z=1169, max_size=40, logic_region=0, group=0),
        # CrownLocation(
        #     map=Maps.CrystalCaves,
        #     name="Near 1DC",
        #     x=2412,
        #     y=277,
        #     z=1960,
        #     max_size=64,
        #     logic_region=Regions.CabinArea,
        #     group=1,
        # ),
        CustomLocation(
            name="Next to Lanky cabin",
            map=Maps.CrystalCaves,
            x=2404.0,
            y=276.0,
            z=1947.0,
            rot_y=227,
            max_size=64,
            logic_region=Regions.CabinArea,
            group=1,
        ),
        CustomLocation(
            name="Left of rotating cabin",
            map=Maps.CrystalCaves,
            x=2961.0,
            y=281.0,
            z=2370.0,
            rot_y=2400,
            max_size=64,
            logic_region=Regions.CabinArea,
            group=1,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Next to rotating cabin",
            x=2903.5,
            y=281.8,
            z=2312,
            rot_y=2400,
            max_size=56,
            logic_region=Regions.CabinArea,
            group=1,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Right of rotating cabin",
            x=2672,
            y=281,
            z=2500,
            rot_y=2225,
            max_size=56,
            logic_region=Regions.CabinArea,
            group=1,
        ),
        # CrownLocation(
        #     name="Next to Donkey 1DC - right",
        #     map=Maps.CrystalCaves,
        #     x=2666.0,
        #     y=282.0,
        #     z=2494.0,
        #     rot_y=2225,
        #     max_size=64,
        #     logic_region=Regions.CabinArea,
        #     group=1,
        # ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Cabin Kasplat Pillar",
            x=2984,
            y=379,
            z=1848,
            max_size=64,
            logic_region=Regions.CabinArea,
            group=1,
            banned_types=[LocationTypes.MelonCrate],
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Near cabins Rocketbarrel",
            x=2465,
            y=206.8,
            z=2530,
            max_size=40,
            logic_region=Regions.CabinArea,
            group=1,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Near Tiny cabin",
            x=3551,
            y=260,
            z=1900,
            max_size=48,
            logic_region=Regions.CabinArea,
            group=1,
        ),
        CustomLocation(
            name="Near cabins Tag barrel",
            map=Maps.CrystalCaves,
            x=3631.0,
            y=260.0,
            z=1534.0,
            rot_y=3111,
            max_size=64,
            logic_region=Regions.CabinArea,
            group=1,
        ),
        CustomLocation(
            map=Maps.CrystalCaves,
            name="Near Diddy Upper cabin",
            x=3684,
            y=343,
            z=1886,
            max_size=40,
            logic_region=Regions.CabinArea,
            group=1,
        ),
        CustomLocation(
            name="Near Diddy upper Cabin door",
            map=Maps.CrystalCaves,
            x=3610.0,
            y=343.0,
            z=1761.0,
            rot_y=3072,
            max_size=64,
            logic_region=Regions.CabinArea,
            group=1,
        ),
        CustomLocation(
            name="Vanilla Close Warp 1",
            map=Maps.CrystalCaves,
            x=2170.177,
            y=64.818,
            z=651.922,
            rot_x=2,
            rot_y=2048,
            rot_z=2,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.CrystalCavesMain,
            group=2,
            vanilla_port=True,
            tied_warp_event=Events.CavesW1aTagged,
        ),
        CustomLocation(
            name="Vanilla Close Warp 2",
            map=Maps.CrystalCaves,
            x=2259.137,
            y=71.157,
            z=653.479,
            rot_x=4,
            rot_y=2048,
            rot_z=2,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.CrystalCavesMain,
            group=2,
            vanilla_port=True,
            tied_warp_event=Events.CavesW2aTagged,
        ),
        CustomLocation(
            name="Vanilla Igloo Warp 1",
            map=Maps.CrystalCaves,
            x=386.949,
            y=48.5,
            z=970.654,
            rot_y=319,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.IglooArea,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.CavesW1bTagged,
        ),
        CustomLocation(
            name="Vanilla Cabins Warp 2",
            map=Maps.CrystalCaves,
            x=2652.649,
            y=280,
            z=2182.157,
            rot_y=1411,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.CabinArea,
            group=1,
            vanilla_port=True,
            tied_warp_event=Events.CavesW2bTagged,
        ),
        CustomLocation(
            name="Vanilla Pillar Warp 5",
            map=Maps.CrystalCaves,
            x=2762.267,
            y=366.833,
            z=953.577,
            rot_y=57,
            max_size=int(0.68 * WARP_MAX_SIZE),
            logic_region=Regions.CavesBlueprintPillar,
            group=2,
            vanilla_port=True,
            tied_warp_event=Events.CavesW5aTagged,
        ),
        CustomLocation(
            name="Vanilla Hidden Warp 3",
            map=Maps.CrystalCaves,
            x=539.131,
            y=180.333,
            z=2511.16,
            rot_y=740,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.CavesBonusCave,
            group=2,
            vanilla_port=True,
            tied_warp_event=Events.CavesW3bTagged,
        ),
        CustomLocation(
            name="Vanilla Igloo Warp 3",
            map=Maps.CrystalCaves,
            x=176.505,
            y=50.167,
            z=1089.408,
            rot_y=683,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.IglooArea,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.CavesW3aTagged,
        ),
        CustomLocation(
            name="Vanilla Hidden Warp 4",
            map=Maps.CrystalCaves,
            x=3339.909,
            y=286.667,
            z=569.045,
            rot_y=2697,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.CavesBlueprintCave,
            group=2,
            vanilla_port=True,
            tied_warp_event=Events.CavesW4bTagged,
        ),
        CustomLocation(
            name="Vanilla Spire Warp 4",
            map=Maps.CrystalCaves,
            x=1196.707,
            y=133.5,
            z=1923.537,
            rot_y=3948,
            max_size=int(WARP_MAX_SIZE * 0.68),
            logic_region=Regions.CavesBananaportSpire,
            group=2,
            vanilla_port=True,
            tied_warp_event=Events.CavesW4aTagged,
        ),
        CustomLocation(
            name="Vanilla Cabin Warp 5",
            map=Maps.CrystalCaves,
            x=3614.799,
            y=343,
            z=1838.974,
            rot_y=3174,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.CabinArea,
            group=1,
            vanilla_port=True,
            tied_warp_event=Events.CavesW5bTagged,
        ),
        CustomLocation(
            map=Maps.CavesDiddyIgloo,
            name="Diddy igloo: Center",
            x=286,
            y=0,
            z=295,
            max_size=64,
            logic_region=Regions.DiddyIgloo,
            group=3,
            banned_types=[LocationTypes.CrownPad],
        ),  # crown pad removed due to potential crash
        CustomLocation(
            map=Maps.CavesDonkeyIgloo,
            name="DK igloo: Behind Maze",
            x=469,
            y=0,
            z=177,
            max_size=80,
            logic_region=Regions.DonkeyIgloo,
            group=3,
        ),
        CustomLocation(
            map=Maps.CavesLankyIgloo,
            name="Lanky igloo: High Platform",
            x=273,
            y=123.3,
            z=245,
            max_size=40,
            logic_region=Regions.LankyIgloo,
            logic=lambda l: ((l.balloon or l.monkey_maneuvers) and l.islanky) or (l.monkey_maneuvers and (l.isdiddy or l.istiny)),
            group=3,
        ),
        CustomLocation(
            map=Maps.CavesTinyIgloo,
            name="Tiny igloo: Opposite Entrance",
            x=385,
            y=0,
            z=200,
            max_size=64,
            logic_region=Regions.TinyIgloo,
            group=3,
        ),
        CustomLocation(
            map=Maps.CavesLankyCabin,
            name="Lanky cabin: Carpet",
            x=448,
            y=0,
            z=332,
            max_size=72,
            logic_region=Regions.LankyCabin,
            group=1,
        ),
        CustomLocation(
            map=Maps.CavesChunkyCabin,
            name="Chunky cabin: Back Left Corner",
            x=493,
            y=0,
            z=536,
            max_size=56,
            logic_region=Regions.ChunkyCabin,
            group=1,
            banned_types=[LocationTypes.CrownPad],
        ),
        CustomLocation(
            map=Maps.CavesDiddyUpperCabin,
            name="Diddy Upper cabin: Right",
            x=106,
            y=0,
            z=353,
            max_size=72,
            logic_region=Regions.DiddyUpperCabin,
            group=1,
        ),
        CustomLocation(
            map=Maps.CavesDonkeyCabin,
            name="Donkey cabin: Opposite Entrance",
            x=165,
            y=0,
            z=463,
            max_size=56,
            logic_region=Regions.DonkeyCabin,
            group=1,
        ),
        CustomLocation(
            name="Ice Castle: Left of puzzle",
            map=Maps.CavesFrozenCastle,
            x=311.0,
            y=0.0,
            z=194.0,
            rot_y=3800,
            max_size=64,
            logic_region=Regions.FrozenCastle,
            group=2,
        ),
        CustomLocation(
            name="Ice Castle: Right of puzzle",
            map=Maps.CavesFrozenCastle,
            x=227.0,
            y=0.0,
            z=404.0,
            rot_y=1786,
            max_size=64,
            logic_region=Regions.FrozenCastle,
            group=2,
        ),
        CustomLocation(
            map=Maps.CavesTinyCabin,
            name="Tiny Cabin: Interior",
            x=179,
            y=0,
            z=255,
            max_size=56,
            logic_region=Regions.TinyCabin,
            logic=lambda l: l.oranges,
            group=1,
        ),
    ],
    Levels.CreepyCastle: [
        CustomLocation(
            map=Maps.CastleGreenhouse,
            name="Greenhouse: Center",
            x=503.276,
            y=0,
            z=581.451,
            max_size=52.8,
            logic_region=Regions.Greenhouse,
            logic=lambda l: l.islanky or l.settings.free_trade_items,
            group=5,
            vanilla_crown=True,
            banned_types=[LocationTypes.DirtPatch],
        ),
        CustomLocation(
            name="Top of Castle near shop",
            map=Maps.CreepyCastle,
            x=655.9,
            y=1794.167,
            z=1386.9,
            rot_y=3094,
            max_size=64,
            vanilla_patch=True,
            logic_region=Regions.CreepyCastleMain,
            group=5,
        ),
        # CustomLocation(map=Maps.CastleGreenhouse, name="Greenhouse: Dead End (1)", x=224, y=0, z=494, max_size=32, logic_region=Regions.Greenhouse, group=5, banned_types=[LocationTypes.DirtPatch]),
        # CustomLocation(map=Maps.CastleGreenhouse, name="Greenhouse: Dead End (2)", x=874, y=0, z=522, max_size=32, logic_region=Regions.Greenhouse, group=5, banned_types=[LocationTypes.DirtPatch]),
        CustomLocation(
            map=Maps.CastleGreenhouse,
            name="Greenhouse: GB area",
            x=170,
            y=0,
            z=227,
            max_size=56,
            logic_region=Regions.Greenhouse,
            group=5,
            banned_types=[LocationTypes.DirtPatch],
        ),
        # CustomLocation(map=Maps.CastleGreenhouse, name="Greenhouse: Dead End (3)", x=779, y=0, z=125, max_size=32, logic_region=Regions.Greenhouse, group=5, banned_types=[LocationTypes.DirtPatch]),
        CustomLocation(
            map=Maps.CreepyCastle,
            name="Near the Tree",
            x=1201,
            y=471.5,
            z=105,
            max_size=80,
            logic_region=Regions.CreepyCastleMain,
            group=4,
        ),
        CustomLocation(
            name="Next to lowest tree",
            map=Maps.CreepyCastle,
            x=1691.0,
            y=372.0,
            z=1995.0,
            rot_y=3561,
            max_size=64,
            logic_region=Regions.CastleVeryBottom,
            group=4,
        ),
        CustomLocation(
            map=Maps.CreepyCastle,
            name="Near lower entrance - right",
            x=1361,
            y=366,
            z=2108,
            max_size=64,
            logic_region=Regions.CastleVeryBottom,
            group=4,
        ),
        CustomLocation(
            map=Maps.CreepyCastle,
            name="Lowest ledge near ladder",
            x=420,
            y=366,
            z=1934,
            max_size=64,
            logic_region=Regions.CastleVeryBottom,
            group=4,
        ),
        CustomLocation(
            map=Maps.CreepyCastle,
            name="Graveyard behind stone",
            x=500,
            y=523,
            z=1660,
            max_size=64,
            logic_region=Regions.CastleGraveyardPlatform,
            group=4,
        ),
        CustomLocation(
            name="Lowest ledge towards Tiny Kasplat",
            map=Maps.CreepyCastle,
            x=245.0,
            y=366.0,
            z=1810.0,
            rot_y=3766,
            max_size=64,
            logic_region=Regions.CastleVeryBottom,
            group=4,
        ),
        CustomLocation(
            name="Near the lower door",
            map=Maps.CreepyCastle,
            x=1319.0,
            y=523.0,
            z=1885.0,
            rot_y=3151,
            max_size=64,
            logic_region=Regions.CreepyCastleMain,
            group=4,
        ),
        CustomLocation(
            name="Upper Gravestone",
            map=Maps.CreepyCastle,
            x=746.0,
            y=521.0,
            z=1873.0,
            rot_y=3280,
            max_size=64,
            logic_region=Regions.CastleGraveyardPlatform,
            group=4,
        ),
        CustomLocation(
            map=Maps.CreepyCastle,
            name="Near Dungeon Tunnel Steps",
            x=1298,
            y=523,
            z=1777,
            max_size=64,
            logic_region=Regions.CreepyCastleMain,
            group=4,
        ),
        CustomLocation(
            map=Maps.CreepyCastle,
            name="Near Dungeon Tunnel",
            x=902,
            y=648,
            z=1620,
            max_size=64,
            logic_region=Regions.CreepyCastleMain,
            group=4,
        ),
        CustomLocation(
            map=Maps.CreepyCastle,
            name="On square stone",
            x=204,
            y=628,
            z=1433,
            max_size=72,
            logic_region=Regions.CreepyCastleMain,
            group=4,
        ),
        # CrownLocation(
        #     map=Maps.CreepyCastle,
        #     name="Near Drawbridge Entrance",
        #     x=664,
        #     y=548.8,
        #     z=532,
        #     max_size=80,
        #     logic_region=Regions.CreepyCastleMain,
        #     group=4,
        # ),
        CustomLocation(
            name="Next to the drawbridge",
            map=Maps.CreepyCastle,
            x=662.0,
            y=548.0,
            z=522.0,
            rot_y=2814,
            max_size=64,
            logic_region=Regions.CreepyCastleMain,
            group=4,
        ),
        CustomLocation(
            map=Maps.CreepyCastle,
            name="Near Lower Rocketbarrel",
            x=176,
            y=622,
            z=578,
            max_size=56,
            logic_region=Regions.CreepyCastleMain,
            group=4,
        ),
        CustomLocation(
            map=Maps.CreepyCastle,
            name="Near Lower Tag Barrel",
            x=1623,
            y=673,
            z=655,
            max_size=56,
            logic_region=Regions.CreepyCastleMain,
            group=4,
        ),
        CustomLocation(
            map=Maps.CreepyCastle,
            name="Near Headphones",
            x=1778,
            y=676,
            z=921,
            max_size=48,
            logic_region=Regions.CreepyCastleMain,
            group=4,
        ),
        CustomLocation(
            map=Maps.CreepyCastle,
            name="Near Drawbridge Exit",
            x=763,
            y=673,
            z=1016,
            max_size=64,
            logic_region=Regions.CreepyCastleMain,
            group=4,
        ),
        CustomLocation(
            map=Maps.CreepyCastle,
            name="Near middle shop",
            x=483,
            y=1135,
            z=1379,
            max_size=64,
            logic_region=Regions.CreepyCastleMain,
            group=5,
        ),
        CustomLocation(
            map=Maps.CreepyCastle,
            name="Near Shed",
            x=1688,
            y=1391,
            z=1802,
            max_size=56,
            logic_region=Regions.CreepyCastleMain,
            group=5,
        ),
        CustomLocation(
            name="Next to Greenhouse",
            map=Maps.CreepyCastle,
            x=1588.0,
            y=1391.0,
            z=1870.0,
            rot_y=2309,
            max_size=64,
            logic_region=Regions.CreepyCastleMain,
            group=5,
        ),
        CustomLocation(
            map=Maps.CreepyCastle,
            name="Near Wind Tower",
            x=1707,
            y=1731,
            z=1255,
            max_size=64,
            logic_region=Regions.CreepyCastleMain,
            group=5,
        ),
        # CrownLocation(
        #     map=Maps.CreepyCastle,
        #     name="Near Wind Tower (2)",
        #     x=1707,
        #     y=1731,
        #     z=1375,
        #     max_size=64,
        #     logic_region=Regions.CreepyCastleMain,
        #     group=5,
        # ),
        CustomLocation(
            name="Top of Castle near fence",
            map=Maps.CreepyCastle,
            x=1696.0,
            y=1731.0,
            z=1384.0,
            rot_y=1479,
            max_size=64,
            logic_region=Regions.CreepyCastleMain,
            group=5,
        ),
        CustomLocation(
            map=Maps.CreepyCastle,
            name="Near top shop",
            x=713,
            y=1794,
            z=1243,
            max_size=64,
            logic_region=Regions.CreepyCastleMain,
            group=5,
        ),
        CustomLocation(
            map=Maps.CreepyCastle,
            name="On Wind Tower",
            x=1560,
            y=2023,
            z=1322,
            max_size=80,
            logic_region=Regions.CreepyCastleMain,
            logic=lambda l: l.jetpack and l.isdiddy,
            group=5,
        ),
        CustomLocation(
            name="Vanilla Overhang Warp 2",
            map=Maps.CreepyCastle,
            x=1899.247,
            y=904.136,
            z=950.597,
            rot_y=3664,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.CreepyCastleMain,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.CastleW2bTagged,
        ),
        CustomLocation(
            name="Vanilla Top Warp 5",
            map=Maps.CreepyCastle,
            x=1341.102,
            y=1731.167,
            z=1120.739,
            rot_y=2526,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.CreepyCastleMain,
            group=5,
            vanilla_port=True,
            tied_warp_event=Events.CastleW5bTagged,
        ),
        CustomLocation(
            name="Vanilla Moat Warp 1",
            map=Maps.CreepyCastle,
            x=1532.901,
            y=673.575,
            z=799.607,
            rot_y=1172,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.CreepyCastleMain,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.CastleW1aTagged,
        ),
        CustomLocation(
            name="Vanilla Moat Warp 5",
            map=Maps.CreepyCastle,
            x=938.578,
            y=673.5,
            z=652.779,
            rot_y=2708,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.CreepyCastleMain,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.CastleW5aTagged,
        ),
        CustomLocation(
            name="Vanilla Back Warp 1",
            map=Maps.CreepyCastle,
            x=1023.578,
            y=648.279,
            z=1657.753,
            rot_y=46,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.CreepyCastleMain,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.CastleW1bTagged,
        ),
        CustomLocation(
            name="Vanilla Moat Warp 3",
            map=Maps.CreepyCastle,
            x=1214.247,
            y=673.5,
            z=582.991,
            rot_y=2071,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.CreepyCastleMain,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.CastleW3aTagged,
        ),
        CustomLocation(
            name="Vanilla Moat Warp 2",
            map=Maps.CreepyCastle,
            x=1384.938,
            y=673.742,
            z=633.478,
            rot_y=1820,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.CreepyCastleMain,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.CastleW2aTagged,
        ),
        CustomLocation(
            name="Vanilla Moat Warp 4",
            map=Maps.CreepyCastle,
            x=1082.711,
            y=673.5,
            z=583.805,
            rot_y=2094,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.CreepyCastleMain,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.CastleW4aTagged,
        ),
        CustomLocation(
            name="Vanilla Shop Warp 3",
            map=Maps.CreepyCastle,
            x=389.7,
            y=1135.469,
            z=1394.365,
            rot_y=1138,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.CreepyCastleMain,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.CastleW3bTagged,
        ),
        CustomLocation(
            name="Vanilla Shed Warp 4",
            map=Maps.CreepyCastle,
            x=1536.106,
            y=1381.167,
            z=1770.412,
            rot_y=1991,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.CreepyCastleMain,
            group=5,
            vanilla_port=True,
            tied_warp_event=Events.CastleW4bTagged,
        ),
        CustomLocation(
            map=Maps.CastleBallroom,
            name="Ballroom: Near Left Candle",
            x=410,
            y=40,
            z=221,
            max_size=88,
            logic_region=Regions.Ballroom,
            group=1,
        ),
        CustomLocation(
            map=Maps.CastleBallroom,
            name="Ballroom: Near Right Candle",
            x=847,
            y=40,
            z=454,
            max_size=88,
            logic_region=Regions.Ballroom,
            group=1,
        ),
        CustomLocation(
            name="Ballroom: Back Left",
            map=Maps.CastleBallroom,
            x=261.0,
            y=40.0,
            z=241.0,
            rot_y=546,
            max_size=64,
            logic_region=Regions.Ballroom,
            group=1,
        ),
        CustomLocation(
            name="Ballroom: Back Right",
            map=Maps.CastleBallroom,
            x=825.0,
            y=40.0,
            z=258.0,
            rot_y=3310,
            max_size=64,
            logic_region=Regions.Ballroom,
            group=1,
        ),
        # CrownLocation(map=Maps.CastleDungeon, name="Dungeon: Diddy Room (OoB)", x=403, y=90, z=3307, max_size=72, logic_region=0, group=0),
        CustomLocation(
            map=Maps.CastleDungeon,
            name="Dungeon: Near Diddy Room",
            x=442,
            y=115,
            z=2595,
            max_size=64,
            logic_region=Regions.Dungeon,
            group=3,
        ),
        CustomLocation(
            map=Maps.CastleDungeon,
            name="Dungeon: Donkey Room",
            x=1492,
            y=170,
            z=2000,
            max_size=80,
            logic_region=Regions.Dungeon,
            logic=lambda l: (l.CanSlamSwitch(Levels.CreepyCastle, 3) and l.donkey) or l.CanPhase(),
            group=3,
        ),
        CustomLocation(
            name="Dungeon: Behind punch gate",
            map=Maps.CastleDungeon,
            x=316.0,
            y=115.0,
            z=2525.0,
            rot_y=1024,
            max_size=64,
            logic_region=Regions.Dungeon,
            logic=lambda l: ((l.punch and l.ischunky) or l.CanPhase()),
            group=3,
        ),
        CustomLocation(
            map=Maps.CastleShed,
            name="Shed: Near Entrance",
            x=179,
            y=0,
            z=219,
            max_size=64,
            logic_region=Regions.Shed,
            group=5,
        ),
        CustomLocation(
            map=Maps.CastleLowerCave,
            name="Lower cave: Lower Portion",
            x=559,
            y=90,
            z=1153,
            max_size=64,
            logic_region=Regions.LowerCave,
            group=3,
        ),
        CustomLocation(
            map=Maps.CastleLowerCave,
            name="Lower cave: Also Behind Mausoleum",
            x=1848,
            y=320,
            z=1186,
            max_size=64,
            logic_region=Regions.LowerCave,
            group=3,
        ),
        CustomLocation(
            map=Maps.CastleLowerCave,
            name="Lower cave: Near shop",
            x=1359,
            y=200,
            z=433,
            max_size=64,
            logic_region=Regions.LowerCave,
            group=3,
        ),
        CustomLocation(
            map=Maps.CastleLowerCave,
            name="Lower cave: Behind Mausoleum",
            x=1889.9437255859,
            y=320,
            z=1260.7548828125,
            rot_y=0,
            max_size=64,
            logic_region=Regions.LowerCave,
            vanilla_crate=True,
            group=3,
        ),
        CustomLocation(
            name="Crypt: entrance",
            map=Maps.CastleCrypt,
            x=626.0,
            y=240.0,
            z=1674.0,
            rot_y=3072,
            max_size=64,
            logic_region=Regions.Crypt,
            group=3,
        ),
        CustomLocation(
            map=Maps.CastleCrypt,
            name="Crypt: Near Chunky Coffin",
            x=1280,
            y=160,
            z=2867,
            max_size=72,
            logic_region=Regions.CryptChunkyRoom,
            group=3,
        ),
        CustomLocation(
            map=Maps.CastleCrypt,
            name="Crypt: Near Diddy Coffin",
            x=2069,
            y=0,
            z=593,
            max_size=72,
            logic_region=Regions.CryptDiddyRoom,
            group=3,
        ),
        CustomLocation(
            name="Crypt: Vanilla Far Warp 1",
            map=Maps.CastleCrypt,
            x=1740.353,
            y=0,
            z=379.404,
            rot_y=3038,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.Crypt,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.CryptW1bTagged,
        ),
        CustomLocation(
            name="Crypt: Vanilla Close Warp 1",
            map=Maps.CastleCrypt,
            x=734.276,
            y=240,
            z=1564.207,
            rot_y=3095,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.Crypt,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.CryptW1aTagged,
        ),
        CustomLocation(
            name="Crypt: Vanilla Close Warp 2",
            map=Maps.CastleCrypt,
            x=732.318,
            y=240,
            z=1661.099,
            rot_y=3107,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.Crypt,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.CryptW2aTagged,
        ),
        CustomLocation(
            name="Crypt: Vanilla Close Warp 3",
            map=Maps.CastleCrypt,
            x=731.274,
            y=240,
            z=1768.498,
            rot_y=3083,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.Crypt,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.CryptW3aTagged,
        ),
        CustomLocation(
            name="Crypt: Vanilla Far Warp 3",
            map=Maps.CastleCrypt,
            x=1076.167,
            y=160,
            z=2478.131,
            rot_y=2094,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.Crypt,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.CryptW3bTagged,
        ),
        CustomLocation(
            name="Crypt: Vanilla Far Warp 2",
            map=Maps.CastleCrypt,
            x=1511.406,
            y=80,
            z=2048.463,
            rot_y=2048,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.Crypt,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.CryptW2bTagged,
        ),
        CustomLocation(
            name="Mausoleum: entrance",
            map=Maps.CastleMausoleum,
            x=731.0,
            y=240.0,
            z=1068.0,
            rot_y=3072,
            max_size=64,
            logic_region=Regions.Mausoleum,
            group=3,
        ),
        CustomLocation(
            map=Maps.CastleMausoleum,
            name="Mausoleum: Lanky Tunnel",
            x=1186,
            y=160,
            z=130,
            max_size=64,
            logic_region=Regions.Mausoleum,
            group=3,
        ),
        CustomLocation(
            map=Maps.CastleUpperCave,
            name="Upper cave: Near Pit",
            x=704,
            y=200,
            z=852,
            max_size=64,
            logic_region=Regions.UpperCave,
            group=3,
        ),
        CustomLocation(
            map=Maps.CastleUpperCave,
            name="Upper cave: Near shop",
            x=1104,
            y=300,
            z=2241,
            max_size=64,
            logic_region=Regions.UpperCave,
            group=3,
        ),
        CustomLocation(
            map=Maps.CastleLibrary,
            name="Library: Enemy Gauntlet Room",
            x=289,
            y=190,
            z=530,
            max_size=80,
            logic_region=Regions.Library,
            group=6,
            banned_types=[LocationTypes.CrownPad],
        ),  # Crown Disabled - if you do the crown first the enemies don't spawn, locking you in
        CustomLocation(
            map=Maps.CastleLibrary,
            name="Library: Flying Book Room",
            x=2772,
            y=180,
            z=500,
            max_size=64,
            logic_region=Regions.LibraryPastBooks,
            group=6,
        ),
        CustomLocation(
            map=Maps.CastleMuseum,
            name="Museum: Near Race",
            x=312,
            y=200,
            z=1784,
            max_size=64,
            logic_region=Regions.MuseumBehindGlass,
            group=2,
        ),
        CustomLocation(
            name="Museum: Pillar Front",
            map=Maps.CastleMuseum,
            x=1003.0,
            y=200.0,
            z=1513.0,
            rot_y=2969,
            max_size=64,
            logic_region=Regions.MuseumBehindGlass,
            logic=lambda l: ((l.monkeyport and l.istiny) or l.CanPhase()),
            group=2,
        ),
        CustomLocation(
            name="Museum: Pillar Back Right",
            map=Maps.CastleMuseum,
            x=1238.0,
            y=200.0,
            z=1612.0,
            rot_y=2628,
            max_size=64,
            logic_region=Regions.MuseumBehindGlass,
            logic=lambda l: ((l.monkeyport and l.istiny) or l.CanPhase()),
            group=2,
        ),
        CustomLocation(
            map=Maps.CastleMuseum,
            name="Museum: Behind Pillar",
            x=1265,
            y=200,
            z=1525,
            max_size=64,
            logic_region=Regions.MuseumBehindGlass,
            logic=lambda l: (l.monkeyport and l.istiny) or l.CanPhase(),
            group=2,
        ),
        CustomLocation(
            name="Museum: Pillar Back Left",
            map=Maps.CastleMuseum,
            x=1236.0,
            y=200.0,
            z=1400.0,
            rot_y=3697,
            max_size=64,
            logic_region=Regions.MuseumBehindGlass,
            logic=lambda l: ((l.monkeyport and l.istiny) or l.CanPhase()),
            group=2,
        ),
        CustomLocation(
            map=Maps.CastleMuseum,
            name="Museum: Main Room",
            x=595,
            y=100,
            z=440,
            max_size=96,
            logic_region=Regions.Museum,
            group=2,
        ),
        CustomLocation(
            map=Maps.CastleTrashCan,
            name="Trash Can: Near Cheese",
            x=465,
            y=15,
            z=510,
            max_size=96,
            logic_region=Regions.TrashCan,
            group=5,
        ),
        CustomLocation(
            map=Maps.CastleTree,
            name="Tree: Starting Room",
            x=972,
            y=400,
            z=884,
            max_size=64,
            logic_region=Regions.CastleTree,
            group=4,
        ),
    ],
    Levels.DKIsles: [
        CustomLocation(
            map=Maps.IslesSnideRoom,
            name="Snide's Room: Under Rock",
            x=361.02,
            y=0,
            z=280.06,
            max_size=68.8,
            logic_region=Regions.IslesSnideRoom,
            logic=lambda l: l.chunky and l.barrels,
            group=6,
            vanilla_crown=True,
            default_index=1,
        ),
        CustomLocation(
            map=Maps.FungiForestLobby,
            name="Fungi Lobby: Gorilla Gone Box",
            x=290.99,
            y=20,
            z=637.07,
            max_size=56,
            logic_region=Regions.FungiForestLobby,
            logic=lambda l: l.CanOpenForestLobbyGoneDoor() and l.chunky and l.gorillaGone,
            group=12,
            vanilla_crown=True,
        ),
        CustomLocation(
            name="Front of Fungi Building",
            map=Maps.Isles,
            x=2647.643,
            y=1498.0,
            z=929.797,
            rot_y=748,
            max_size=64,
            vanilla_patch=True,
            logic_region=Regions.CabinIsle,
            group=2,
        ),
        CustomLocation(
            name="On Aztec Building",
            map=Maps.Isles,
            x=3509.673,
            y=1170.0,
            z=1733.509,
            rot_y=1784,
            max_size=64,
            vanilla_patch=True,
            logic_region=Regions.AztecLobbyRoof,
            group=3,
        ),
        CustomLocation(
            name="Under Caves Lobby Entrance",
            map=Maps.Isles,
            x=2401.601,
            y=800.0,
            z=1571.532,
            rot_y=4028,
            max_size=64,
            vanilla_patch=True,
            logic_region=Regions.IslesHill,
            group=3,
        ),
        CustomLocation(
            name="Castle Lobby: Center",
            map=Maps.CreepyCastleLobby,
            x=579.809,
            y=245.5,
            z=681.709,
            rot_y=2074,
            max_size=64,
            vanilla_patch=True,
            logic_region=Regions.CreepyCastleLobby,
            logic=lambda l: ((l.chunky and l.balloon and l.islanky and l.barrels) or l.CanMoonkick() or (l.monkey_maneuvers and l.istiny and l.twirl and (not l.isKrushaAdjacent(Kongs.tiny)))),
            group=14,
        ),
        CustomLocation(
            name="Training Grounds: Banana Hoard",
            map=Maps.TrainingGrounds,
            x=2497.648,
            y=191.0,
            z=1036.583,
            rot_y=0,
            max_size=64,
            vanilla_patch=True,
            logic_region=Regions.TrainingGrounds,
            logic=lambda l: (l.can_use_vines or l.CanMoonkick()) and l.climbing,
            group=1,
        ),
        CustomLocation(
            name="Training Grounds: Rear Tunnel",
            map=Maps.TrainingGrounds,
            x=1223.714,
            y=37.208,
            z=2200.538,
            rot_y=1002,
            max_size=64,
            vanilla_patch=True,
            logic_region=Regions.TrainingGrounds,
            group=1,
        ),
        CustomLocation(
            name="Prison: Back",
            map=Maps.KLumsy,
            x=1499.675,
            y=95.0,
            z=1233.831,
            rot_y=2736,
            max_size=64,
            vanilla_patch=True,
            logic_region=Regions.Prison,
            group=5,
        ),
        CustomLocation(
            name="Behind Fungi Building",
            map=Maps.Isles,
            x=2436.0,
            y=1498.0,
            z=817.0,
            rot_y=637,
            max_size=64,
            logic_region=Regions.CabinIsle,
            group=2,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Fungi floating island",
            x=2683,
            y=1498,
            z=818,
            max_size=64,
            logic_region=Regions.CabinIsle,
            group=2,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Waterfall Platform",
            x=3049,
            y=1490,
            z=1234,
            max_size=64,
            logic_region=Regions.IslesAboveWaterfall,
            group=2,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Down Near Caves Lobby Tree",
            x=2497,
            y=507,
            z=1903,
            max_size=64,
            logic_region=Regions.IslesMain,
            group=3,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Near K. Rool",
            x=3005,
            y=500,
            z=855,
            max_size=80,
            logic_region=Regions.IslesMain,
            group=3,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Near Fungi Cannon",
            x=3440,
            y=495,
            z=1222,
            max_size=64,
            logic_region=Regions.IslesMain,
            group=3,
        ),
        CustomLocation(
            name="In Fungi cannon rock",
            map=Maps.Isles,
            x=3516.0,
            y=500.0,
            z=633.0,
            rot_y=1934,
            max_size=48,
            logic_region=Regions.IslesMain,
            logic=lambda l: (l.settings.open_lobbies or Events.GalleonKeyTurnedIn in l.Events or l.CanPhase()),
            group=2,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Next to Caves Lobby Tree",
            x=2427,
            y=600,
            z=1822,
            max_size=64,
            logic_region=Regions.IslesMain,
            group=3,
        ),
        # CrownLocation(
        #     map=Maps.Isles,
        #     name="Behind Aztec Building",
        #     x=3650,
        #     y=1020,
        #     z=1776,
        #     max_size=64,
        #     logic_region=Regions.IslesMainUpper,
        #     group=3
        # ),
        CustomLocation(
            name="Isles Boulders",
            map=Maps.Isles,
            x=2813.0,
            y=1058.0,
            z=2054.0,
            rot_y=3959,
            max_size=64,
            logic_region=Regions.IslesMainUpper,
            group=3,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Front of Aztec Building",
            x=3225,
            y=1000,
            z=1613,
            max_size=56,
            logic_region=Regions.IslesMainUpper,
            group=3,
        ),
        CustomLocation(
            name="Behind Aztec Building",
            map=Maps.Isles,
            x=3643.0,
            y=1020.0,
            z=1790.0,
            rot_y=2742,
            max_size=64,
            logic_region=Regions.IslesMainUpper,
            group=3,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Outside K. Lumsy",
            x=3107,
            y=520,
            z=3500,
            max_size=48,
            logic_region=Regions.KremIsle,
            group=5,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Near Monkeyport pad",
            x=2519,
            y=550,
            z=4152,
            max_size=80,
            logic_region=Regions.KremIsle,
            group=6,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Back of Krem Isle",
            x=1852,
            y=600,
            z=3920,
            max_size=80,
            logic_region=Regions.KremIsle,
            group=6,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Under DK Caged GB",
            x=2000,
            y=550,
            z=3325,
            max_size=80,
            logic_region=Regions.KremIsle,
            group=6,
        ),
        CustomLocation(
            name="Back of Krem Isle (Lower)",
            map=Maps.Isles,
            x=2019.0,
            y=590.0,
            z=4146.0,
            rot_y=1615,
            max_size=64,
            logic_region=Regions.KremIsle,
            group=6,
        ),
        # CustomLocation(  # Same location as Behind Factory Lobby Entrance
        #     name="Back of Kroc Isle (Middle)",
        #     map=Maps.Isles,
        #     x=2350.0,
        #     y=1199.0,
        #     z=3887.0,
        #     rot_y=1956,
        #     max_size=64,
        #     logic_region=Regions.KremIsleBeyondLift,
        #     group=6,
        # ),
        CustomLocation(
            map=Maps.Isles,
            name="Behind Factory Lobby Entrance",
            x=2395,
            y=1200,
            z=3899,
            max_size=64,
            logic_region=Regions.KremIsleBeyondLift,
            group=6,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Right of Factory Lobby Entrance",
            x=2159,
            y=1200,
            z=3518,
            max_size=64,
            logic_region=Regions.KremIsleBeyondLift,
            group=6,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Behind Helm Lobby Entrance",
            x=2370,
            y=1720,
            z=3809,
            max_size=40,
            logic_region=Regions.KremIsleTopLevel,
            group=6,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Left Krem Isle Arm (side)",
            x=2218,
            y=1620,
            z=3488,
            max_size=80,
            logic_region=Regions.KremIsleTopLevel,
            group=6,
        ),
        CustomLocation(
            name="Left Krem Isle Arm",
            map=Maps.Isles,
            x=2313.0,
            y=1620.0,
            z=3214.0,
            rot_y=3891,
            max_size=64,
            logic_region=Regions.KremIsleTopLevel,
            group=6,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Right Krem Isle Arm",
            x=2700,
            y=1620,
            z=3315,
            max_size=80,
            logic_region=Regions.KremIsleTopLevel,
            group=6,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Fairy Isle",
            x=885,
            y=500,
            z=2307,
            max_size=64,
            logic_region=Regions.OuterIsles,
            group=4,
        ),
        CustomLocation(
            name="Behind Fairy Isle",
            map=Maps.Isles,
            x=754.0,
            y=500.0,
            z=2386.0,
            rot_y=807,
            max_size=64,
            group=4,
            logic_region=Regions.OuterIsles,
            logic=lambda _: True,
        ),
        CustomLocation(
            map=Maps.Isles,
            name="Small Island",
            x=965,
            y=500,
            z=1410,
            max_size=80,
            logic_region=Regions.OuterIsles,
            group=4,
        ),
        CustomLocation(
            name="Vanilla Ring Warp 1",
            map=Maps.Isles,
            x=2802.864,
            y=500,
            z=2400.161,
            rot_y=2048,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.IslesMain,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.IslesW1aTagged,
        ),
        CustomLocation(
            name="Vanilla K. Lumsy Warp 1",
            map=Maps.Isles,
            x=3483.24,
            y=500,
            z=3150.5,
            rot_y=1729,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.KremIsle,
            group=6,
            vanilla_port=True,
            tied_warp_event=Events.IslesW1bTagged,
        ),
        CustomLocation(
            name="Vanilla Ring Warp 2",
            map=Maps.Isles,
            x=2749.802,
            y=500,
            z=2399.598,
            rot_y=2094,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.IslesMain,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.IslesW2aTagged,
        ),
        CustomLocation(
            name="Vanilla Upper Warp 2",
            map=Maps.Isles,
            x=3355.592,
            y=1014.464,
            z=1672.928,
            rot_x=8,
            rot_y=2788,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.IslesMainUpper,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.IslesW2bTagged,
        ),
        CustomLocation(
            name="Vanilla Waterfall Warp 3",
            map=Maps.Isles,
            x=3044.718,
            y=500,
            z=978.782,
            rot_y=1968,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.IslesMain,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.IslesW3bTagged,
        ),
        CustomLocation(
            name="Vanilla Ring Warp 5",
            map=Maps.Isles,
            x=2817.764,
            y=500,
            z=2347.447,
            rot_y=2105,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.IslesMain,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.IslesW5aTagged,
        ),
        CustomLocation(
            name="Vanilla Ring Warp 3",
            map=Maps.Isles,
            x=2733.295,
            y=500,
            z=2348.008,
            rot_y=2025,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.IslesMain,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.IslesW3aTagged,
        ),
        CustomLocation(
            name="Vanilla Ring Warp 4",
            map=Maps.Isles,
            x=2780.518,
            y=500,
            z=2310.291,
            rot_y=2048,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.IslesMain,
            group=3,
            vanilla_port=True,
            tied_warp_event=Events.IslesW4aTagged,
        ),
        CustomLocation(
            name="Vanilla Krem Warp 4",
            map=Maps.Isles,
            x=2494.455,
            y=1200,
            z=3305.243,
            rot_y=1889,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.KremIsleBeyondLift,
            group=6,
            vanilla_port=True,
            tied_warp_event=Events.IslesW4bTagged,
        ),
        CustomLocation(
            name="Vanilla Island Warp 5",
            map=Maps.Isles,
            x=930.92,
            y=500,
            z=2632.983,
            rot_y=0,
            max_size=WARP_MAX_SIZE,
            logic_region=Regions.OuterIsles,
            group=4,
            vanilla_port=True,
            tied_warp_event=Events.IslesW5bTagged,
        ),
        CustomLocation(
            name="Japes Lobby: Near Tag Barrel",
            map=Maps.JungleJapesLobby,
            x=713.0,
            y=4.0,
            z=266.0,
            rot_y=1945,
            max_size=64,
            logic_region=Regions.JungleJapesLobby,
            group=8,
        ),
        CustomLocation(
            map=Maps.JungleJapesLobby,
            name="Japes Lobby: Near Portal",
            x=711,
            y=0,
            z=632,
            max_size=64,
            logic_region=Regions.JungleJapesLobby,
            group=8,
        ),
        CustomLocation(
            map=Maps.AngryAztecLobby,
            name="Aztec Lobby: Front of Door",
            x=680,
            y=0,
            z=439,
            max_size=64,
            logic_region=Regions.AngryAztecLobby,
            group=9,
        ),
        CustomLocation(
            map=Maps.AngryAztecLobby,
            name="Aztec Lobby: back room right",
            x=930,
            y=0,
            z=637,
            max_size=64,
            logic_region=Regions.AngryAztecLobby,
            logic=lambda l: l.hasMoveSwitchsanity(Switches.IslesAztecLobbyFeather, False) or l.CanPhase(),
            group=9,
        ),
        CustomLocation(
            name="Aztec Lobby: back room left",
            map=Maps.AngryAztecLobby,
            x=1128.0,
            y=0.0,
            z=586.0,
            rot_y=694,
            max_size=64,
            logic_region=Regions.AngryAztecLobby,
            logic=lambda l: l.hasMoveSwitchsanity(Switches.IslesAztecLobbyFeather, False) or l.CanPhase(),
            group=9,
        ),
        CustomLocation(
            map=Maps.FranticFactoryLobby,
            name="Factory Lobby: Near Lever",
            x=280,
            y=0,
            z=292,
            max_size=64,
            logic_region=Regions.FranticFactoryLobby,
            group=10,
        ),
        CustomLocation(
            name="Factory Lobby: High Platform",
            map=Maps.FranticFactoryLobby,
            x=664.0,
            y=133.0,
            z=588.0,
            rot_y=2048,
            max_size=64,
            logic_region=Regions.FranticFactoryLobby,
            logic=lambda l: ((l.grab and l.isdonkey) or l.CanMoonkick() or (l.monkey_maneuvers and (l.isdiddy or l.istiny))),
            group=10,
        ),
        CustomLocation(
            map=Maps.FranticFactoryLobby,
            name="Factory Lobby: Above Portal",
            x=677,
            y=134,
            z=367,
            max_size=64,
            logic_region=Regions.FranticFactoryLobby,
            logic=lambda l: ((l.grab and l.donkey) or (l.monkey_maneuvers and (l.isdiddy or l.istiny))),
            group=10,
        ),
        CustomLocation(
            map=Maps.GloomyGalleonLobby,
            name="Galleon Lobby: Right of Portal",
            x=429,
            y=139.6,
            z=942,
            max_size=64,
            logic_region=Regions.GloomyGalleonLobby,
            group=11,
        ),
        CustomLocation(
            map=Maps.GloomyGalleonLobby,
            name="Galleon Lobby: Left of Portal",
            x=855,
            y=119.6,
            z=886,
            max_size=56,
            logic_region=Regions.GloomyGalleonLobby,
            group=11,
        ),
        CustomLocation(
            name="Galleon Lobby: Mini room",
            map=Maps.GloomyGalleonLobby,
            x=838.0,
            y=99.0,
            z=232.0,
            rot_y=978,
            max_size=64,
            logic_region=Regions.GloomyGalleonLobby,
            logic=lambda l: ((l.mini and l.CanSlamSwitch(Levels.GloomyGalleon, 2) and l.istiny and l.chunky and l.swim) or l.CanPhaseswim()) and (not l.IsLavaWater() or l.Melons >= 3),
            group=11,
            banned_types=[LocationTypes.CrownPad],  # Crown pad would need testing if it's possible to get up.
        ),
        CustomLocation(
            name="Forest Lobby: On Tag Crate",
            map=Maps.FungiForestLobby,
            x=436.0,
            y=46.0,
            z=252.0,
            rot_y=1024,
            max_size=64,
            logic_region=Regions.FungiForestLobby,
            group=12,
            banned_types=[LocationTypes.MelonCrate],
        ),
        CustomLocation(
            name="Caves Lobby: On the Lava",
            map=Maps.CrystalCavesLobby,
            x=387.0,
            y=2.0,
            z=207.0,
            rot_y=785,
            max_size=64,
            logic_region=Regions.CrystalCavesLobby,
            logic=lambda l: ((l.punch and l.strongKong and l.isdonkey) or l.CanPhase() or l.ledgeclip),
            group=13,
            banned_types=[LocationTypes.CrownPad],
        ),
        CustomLocation(
            map=Maps.CrystalCavesLobby,
            name="Caves Lobby: Right of Portal",
            x=1091,
            y=118,
            z=541,
            max_size=64,
            logic_region=Regions.CrystalCavesLobby,
            group=13,
        ),
        CustomLocation(
            map=Maps.CrystalCavesLobby,
            name="Caves Lobby: Diddy ledge",
            x=794,
            y=280.4,
            z=739,
            max_size=56,
            logic_region=Regions.CrystalCavesLobby,
            logic=lambda l: (l.jetpack and l.isdiddy) or l.CanMoonkick(),
            group=13,
        ),
        CustomLocation(
            map=Maps.CrystalCavesLobby,
            name="Caves Lobby: boulder room",
            x=1751,
            y=13.5,
            z=532,
            max_size=64,
            logic_region=Regions.CrystalCavesLobby,
            logic=lambda l: (l.punch and l.chunky) or l.CanPhase() or l.ledgeclip,
            group=13,
        ),
        CustomLocation(
            name="Castle Lobby: Behind the entrance",
            map=Maps.CreepyCastleLobby,
            x=577.0,
            y=60.0,
            z=67.0,
            rot_y=773,
            max_size=64,
            logic_region=Regions.CreepyCastleLobby,
            group=14,
        ),
        CustomLocation(
            map=Maps.CreepyCastleLobby,
            name="Castle Lobby: Right of Entrance",
            x=355,
            y=60,
            z=269,
            max_size=64,
            logic_region=Regions.CreepyCastleLobby,
            group=14,
        ),
        CustomLocation(
            map=Maps.CreepyCastleLobby,
            name="Castle Lobby: Left of Portal",
            x=803,
            y=60,
            z=1066,
            max_size=64,
            logic_region=Regions.CreepyCastleLobby,
            group=14,
        ),
        CustomLocation(
            name="Helm Lobby: Next to Tag Barrel",
            map=Maps.HideoutHelmLobby,
            x=300.0,
            y=191.0,
            z=269.0,
            rot_y=2048,
            max_size=64,
            logic_region=Regions.HideoutHelmLobby,
            group=7,
        ),
        CustomLocation(
            map=Maps.HideoutHelmLobby,
            name="Helm Lobby: Bonus Platform",
            x=690,
            y=196.4,
            z=638,
            max_size=48,
            logic_region=Regions.HideoutHelmLobby,
            logic=lambda l: (l.hasMoveSwitchsanity(Switches.IslesHelmLobbyGone, False) and l.can_use_vines) or (l.monkey_maneuvers and l.istiny and l.twirl),
            group=7,
        ),
        CustomLocation(
            name="Helm Lobby: Kasplat island",
            map=Maps.HideoutHelmLobby,
            x=325.0,
            y=191.0,
            z=643.0,
            rot_y=0,
            max_size=64,
            logic_region=Regions.HideoutHelmLobby,
            logic=lambda l: ((l.coconut and l.scope) or (l.twirl and l.istiny and l.monkey_maneuvers)),
            group=7,
        ),
        CustomLocation(
            map=Maps.TrainingGrounds,
            name="Training Grounds: Far Mountain",
            x=1153,
            y=252,
            z=1822,
            max_size=72,
            logic_region=Regions.TrainingGrounds,
            logic=lambda l: l.climbing,
            group=1,
        ),
        CustomLocation(
            name="Training Grounds: On the rear hill",
            map=Maps.TrainingGrounds,
            x=1086.0,
            y=252.0,
            z=1833.0,
            rot_y=489,
            max_size=64,
            logic_region=Regions.TrainingGrounds,
            logic=lambda l: l.climbing,
            group=1,
        ),
        CustomLocation(
            name="Training Grounds: On the entrance hill",
            map=Maps.TrainingGrounds,
            x=1108.0,
            y=220.0,
            z=701.0,
            rot_y=3026,
            max_size=64,
            logic_region=Regions.TrainingGrounds,
            logic=lambda l: l.climbing and ((l.twirl and l.istiny) or (l.monkey_maneuvers and l.isdonkey and (not l.isKrushaAdjacent(Kongs.donkey)))),
            group=1,
        ),
        CustomLocation(
            map=Maps.TrainingGrounds,
            name="Training Grounds: Near Mountain",
            x=1187,
            y=225,
            z=734,
            max_size=72,
            logic_region=Regions.TrainingGrounds,
            logic=lambda l: l.climbing and ((l.twirl and l.istiny) or (l.monkey_maneuvers and l.isdonkey and (not l.isKrushaAdjacent(Kongs.donkey)))),
            group=1,
        ),
        CustomLocation(
            map=Maps.TrainingGrounds,
            name="Training Grounds: Rear Cave",
            x=1196,
            y=36.4,
            z=2119,
            max_size=56,
            logic_region=Regions.TrainingGrounds,
            group=1,
        ),
        CustomLocation(
            map=Maps.TrainingGrounds,
            name="Training Grounds: Banana Hoard (back)",
            x=2500,
            y=211,
            z=920,
            max_size=56,
            logic_region=Regions.TrainingGrounds,
            logic=lambda l: (l.can_use_vines or l.CanMoonkick()) and l.climbing,
            group=1,
        ),
        CustomLocation(
            map=Maps.TrainingGrounds,
            name="Training Grounds: Underwater",
            x=1874,
            y=-176,
            z=1159,
            max_size=64,
            logic_region=Regions.TrainingGrounds,
            logic=lambda l: l.HasGun(Kongs.any) and ((l.istiny and l.isKrushaAdjacent(Kongs.tiny)) or (not l.istiny)) and l.swim and l.scope,
            group=3,
            banned_types=[LocationTypes.CrownPad, LocationTypes.DirtPatch],
        ),
        # CustomLocation(
        #     map=Maps.TrainingGrounds,
        #     name="Training Grounds: Under Water in Corner",
        #     x=1962,
        #     y=-187,
        #     z=1324,
        #     max_size=64,
        #     logic_region=Regions.TrainingGrounds,
        #     logic=lambda l: l.HasGun(Kongs.any) and ((l.istiny and l.isKrushaAdjacent(Kongs.tiny)) or (not l.istiny)) and l.swim and l.scope,
        #     group=2,
        #     banned_types=[LocationTypes.CrownPad, LocationTypes.DirtPatch],
        # ),
        CustomLocation(
            map=Maps.TrainingGrounds,
            name="Training Grounds: Near Pool",
            x=1625,
            y=36,
            z=1585,
            max_size=72,
            logic_region=Regions.TrainingGrounds,
            group=1,
        ),
        CustomLocation(
            name="Treehouse: back",
            map=Maps.Treehouse,
            x=288.0,
            y=85.0,
            z=488.0,
            rot_y=3072,
            max_size=64,
            logic_region=Regions.Treehouse,
            group=1,
        ),
        CustomLocation(
            map=Maps.BananaFairyRoom,
            name="Banana Fairy Room: Right of Queen",
            x=648,
            y=37.5,
            z=133,
            max_size=80,
            logic_region=Regions.BananaFairyRoom,
            group=4,
        ),
        CustomLocation(
            map=Maps.BananaFairyRoom,
            name="Banana Fairy Room: Behind Queen",
            x=1025,
            y=37.5,
            z=508,
            max_size=80,
            logic_region=Regions.BananaFairyRoom,
            group=4,
        ),
        CustomLocation(
            name="Banana Fairy Room: Behind Fairy Chair",
            map=Maps.BananaFairyRoom,
            x=835.0,
            y=37.0,
            z=563.0,
            rot_y=1080,
            max_size=64,
            logic_region=Regions.BananaFairyRoom,
            group=4,
        ),
        CustomLocation(
            name="Banana Fairy Room: Rareware room",
            map=Maps.BananaFairyRoom,
            x=644.0,
            y=37.0,
            z=1085.0,
            rot_y=2048,
            max_size=64,
            logic_region=Regions.RarewareGBRoom,
            group=4,
        ),
        CustomLocation(
            map=Maps.BananaFairyRoom,
            name="Banana Fairy Room: Rareware Room right",
            x=446,
            y=37.5,
            z=1227,
            max_size=96,
            logic_region=Regions.RarewareGBRoom,
            group=4,
        ),
        CustomLocation(
            name="Snide's Room: Next to Snide's",
            map=Maps.IslesSnideRoom,
            x=576.0,
            y=0.0,
            z=450.0,
            rot_y=341,
            max_size=64,
            logic_region=Regions.IslesSnideRoom,
            group=6,
        ),
        CustomLocation(
            map=Maps.KLumsy,
            name="Prison: Back Right",
            x=1580,
            y=95,
            z=868,
            max_size=80,
            logic_region=Regions.Prison,
            group=5,
        ),
        CustomLocation(
            map=Maps.KLumsy,
            name="Prison: Near Left",
            x=560,
            y=95,
            z=1340,
            max_size=80,
            logic_region=Regions.Prison,
            group=5,
        ),
        CustomLocation(
            name="Prison: Under K. Lumsy",
            map=Maps.KLumsy,
            x=1020.0,
            y=50.0,
            z=1001.0,
            rot_y=682,
            max_size=64,
            logic_region=Regions.Prison,
            logic=lambda l: (l.IsKLumsyFree() or l.CanPhase()),
            group=5,
        ),
    ],
    Levels.HideoutHelm: [
        CustomLocation(
            map=Maps.HideoutHelm,
            name="Metal grate by Mini Monkey barrel",
            x=841,
            y=-136,
            z=2545,
            max_size=80,
            logic_region=Regions.HideoutHelmMiniRoom,
            group=1,
            banned_types=[LocationTypes.MelonCrate],  # Temporarily disabled due to a bug causing a key to spawn on the ceiling
        ),
        CustomLocation(
            map=Maps.HideoutHelm,
            name="Mini Monkey room right side",
            x=1010,
            y=-132,
            z=2243,
            max_size=80,
            logic_region=Regions.HideoutHelmMiniRoom,
            group=1,
            banned_types=[LocationTypes.MelonCrate],  # Temporarily disabled due to a bug causing a key to spawn on the ceiling
        ),
        CustomLocation(
            map=Maps.HideoutHelm,
            name="Pineapple switch room in ammo alcove",
            x=1077,
            y=-164,
            z=1261,
            max_size=80,
            logic_region=Regions.HideoutHelmSwitchRoom,
            group=1,
            banned_types=[LocationTypes.MelonCrate],  # Temporarily disabled due to a bug causing a key to spawn on the ceiling
        ),
        CustomLocation(
            map=Maps.HideoutHelm,
            name="First room left of Tag barrel",
            x=2065,
            y=-461,
            z=480,
            max_size=80,
            logic_region=Regions.HideoutHelmStart,
            group=1,
            banned_types=[LocationTypes.MelonCrate],  # Temporarily disabled due to a bug causing a key to spawn on the ceiling
        ),
        CustomLocation(
            map=Maps.HideoutHelm,
            name="Top of Blast-o-Matic",
            x=1047.6,
            y=448.1,
            z=3391.4,
            max_size=40,
            logic_region=Regions.HideoutHelmMain,
            vanilla_crown=True,
            logic=lambda l: l.jetpack and l.isdiddy,
            group=2,
        ),
        # CrownLocation(
        #     map=Maps.HideoutHelm,
        #     name="DK's room",
        #     x=396,
        #     y=-132,
        #     z=3841,
        #     max_size=80,
        #     logic_region=Regions.HideoutHelmMain,  # FIX ME if uncommented
        #     logic=lambda l: Events.HelmDoorsOpened in l.Events and l.punch and l.bongos and l.isdonkey and (
        #             l.settings.helm_setting == HelmSetting.skip_all
        #             or (
        #                 Events.HelmDonkeyDone in l.Events
        #                 and Events.HelmChunkyDone in l.Events
        #                 and Events.HelmTinyDone in l.Events
        #                 and Events.HelmLankyDone in l.Events
        #                 and Events.HelmDiddyDone in l.Events
        #             )
        #         ),
        #     group=2,
        #     banned_types=[LocationTypes.CrownPad],
        # ),
        # CrownLocation(
        #     map=Maps.HideoutHelm,
        #     name="Chunky's room",
        #     x=1518,
        #     y=-72,
        #     z=2761,
        #     max_size=80,
        #     logic_region=Regions.HideoutHelmMain,  # FIX ME if uncommented
        #     logic=lambda l: Events.HelmDoorsOpened in l.Events and l.punch and l.triangle and l.ischunky and (
        #             l.settings.helm_setting == HelmSetting.skip_all
        #             or (
        #                 Events.HelmDonkeyDone in l.Events
        #                 and Events.HelmChunkyDone in l.Events
        #                 and Events.HelmTinyDone in l.Events
        #                 and Events.HelmLankyDone in l.Events
        #                 and Events.HelmDiddyDone in l.Events
        #             )
        #         ),
        #     group=2,
        #     banned_types=[LocationTypes.CrownPad],
        # ),
        # CrownLocation(
        #     map=Maps.HideoutHelm,
        #     name="Tiny's room",
        #     x=281,
        #     y=-32,
        #     z=3281,
        #     max_size=80,
        #     logic_region=Regions.HideoutHelmMain,  # FIX ME if uncommented
        #     logic=lambda l: Events.HelmDoorsOpened in l.Events and l.punch and l.saxophone and l.istiny and (
        #             l.settings.helm_setting == HelmSetting.skip_all
        #             or (
        #                 Events.HelmDonkeyDone in l.Events
        #                 and Events.HelmChunkyDone in l.Events
        #                 and Events.HelmTinyDone in l.Events
        #                 and Events.HelmLankyDone in l.Events
        #                 and Events.HelmDiddyDone in l.Events
        #             )
        #         ),
        #     group=2,
        #     banned_types=[LocationTypes.CrownPad],
        # ),
        # CrownLocation(
        #     map=Maps.HideoutHelm,
        #     name="Lanky's room",
        #     x=1824,
        #     y=48,
        #     z=3272,
        #     max_size=80,
        #     logic_region=Regions.HideoutHelmMain,  # FIX ME if uncommented
        #     logic=lambda l: Events.HelmDoorsOpened in l.Events and l.punch and l.trombone and l.islanky and (
        #             l.settings.helm_setting == HelmSetting.skip_all
        #             or (
        #                 Events.HelmDonkeyDone in l.Events
        #                 and Events.HelmChunkyDone in l.Events
        #                 and Events.HelmTinyDone in l.Events
        #                 and Events.HelmLankyDone in l.Events
        #                 and Events.HelmDiddyDone in l.Events
        #             )
        #         ),
        #     group=2,
        #     banned_types=[LocationTypes.CrownPad],
        # ),
        # CrownLocation(
        #     map=Maps.HideoutHelm,
        #     name="Diddy's room",
        #     x=583,
        #     y=208,
        #     z=2741,
        #     max_size=80,
        #     logic_region=Regions.HideoutHelmMain,  # FIX ME if uncommented
        #     logic=lambda l: Events.HelmDoorsOpened in l.Events and l.punch and l.jetpack and l.guitar and l.isdiddy and (
        #             l.settings.helm_setting == HelmSetting.skip_all
        #             or (
        #                 Events.HelmDonkeyDone in l.Events
        #                 and Events.HelmChunkyDone in l.Events
        #                 and Events.HelmTinyDone in l.Events
        #                 and Events.HelmLankyDone in l.Events
        #                 and Events.HelmDiddyDone in l.Events
        #             )
        #         ),
        #     group=2,
        #     banned_types=[LocationTypes.CrownPad],
        # ),
        CustomLocation(
            map=Maps.HideoutHelm,
            name="Blast-o-Matic platform left side",
            x=1048,
            y=-2,
            z=3266,
            max_size=64,
            logic_region=Regions.HideoutHelmMain,
            group=2,
        ),
        CustomLocation(
            map=Maps.HideoutHelm,
            name="Blast-o-Matic platform right side",
            x=1051,
            y=-2,
            z=3518,
            max_size=64,
            logic_region=Regions.HideoutHelmMain,
            group=2,
        ),
        CustomLocation(
            map=Maps.HideoutHelm,
            name="Under K. Rool door",
            x=1054,
            y=-132,
            z=3721,
            max_size=64,
            logic_region=Regions.HideoutHelmMain,
            group=2,
        ),
        CustomLocation(
            map=Maps.HideoutHelm,
            name="Navigation room near terminals",
            x=1262,
            y=10,
            z=4467,
            max_size=120,
            logic_region=Regions.HideoutHelmAfterBoM,
            group=3,
        ),
        CustomLocation(
            map=Maps.HideoutHelm,
            name="Navigation room near left window",
            x=1584,
            y=10,
            z=4349,
            max_size=80,
            logic_region=Regions.HideoutHelmAfterBoM,
            group=3,
        ),
        CustomLocation(
            map=Maps.HideoutHelm,
            name="Navigation room near right window",
            x=1577,
            y=10,
            z=4593,
            max_size=80,
            logic_region=Regions.HideoutHelmAfterBoM,
            group=3,
        ),
        CustomLocation(
            map=Maps.HideoutHelm,
            name="K. Rool room near kong faces",
            x=523,
            y=74,
            z=5341,
            max_size=96,
            logic_region=Regions.HideoutHelmThroneRoom,
            group=3,
        ),
        CustomLocation(
            map=Maps.HideoutHelm,
            name="K. Rool room in front of chair",
            x=548,
            y=74,
            z=5036,
            max_size=96,
            logic_region=Regions.HideoutHelmThroneRoom,
            group=3,
        ),
    ],
}
